#include "defs.h"
#include "mp.e"
#include "mp.h"

void
mp_chebyshev	WITH_3_ARGS(
	mp_float_array,		c,
	mp_int,			n,
	mp_int,			flag
)
/*
Converts the power series coefficients c[0], ..., c[n - 1] to
Chebyshev series coefficients.  (It is assumed that the constant
term in the chebyshev sum is halved.)

flag = 0 means that c[i] is the coefficient of x^i on input,
	of T(i)(x) on output,
flag = -1 means that c[i] is the coefficient of x^(2i + 1) on input,
	of T(2i + 1)(x) on output,
flag = +1 means that c[i] is the coefficient of x^(2i) on input,
	of T(2i) on output.
*/
{
    mp_round_type	save_round = round;
    mp_int		abs_flag = int_abs(flag), j;


    round = MP_TRUNC;

    if (n < 0 || abs_flag > 1)
	mp_error("mp_chebyshev: illegal arguments");

    for (j = n - 1; j >= 0; j--)
    {
	mp_int		i, j2;
	mp_float	temp;
	

	temp  =  mp_array_element(c, j);

	mp_mul_int_eq(temp, 2);

	if ((j2 = j + 2 - abs_flag) < n)
	    mp_add_eq(temp, mp_array_element(c, j2));

	for (i = j + 1; i < n; i++)
	{
	    mp_int	i1 = i + 2 - abs_flag;
	    mp_float	temp;
	    

	    temp =  mp_array_element(c, i);

	    if (i1 < n)
		mp_add_eq(temp, mp_array_element(c, i1));
	    
	    mp_div_int_eq(temp, 2);
	}

	if (flag == -1 || flag == 1 && j > 0)
	    for (i = j; i < n; i++)
	    {
		mp_float	temp;
		

		temp =  mp_array_element(c, i);

		if (i < n - 1)
		    mp_add_eq(temp, mp_array_element(c, i + 1));
		
		mp_div_int_eq(temp, 2);
	    }
    }

    round = save_round;
}


mp_float
mp_chev		WITH_5_ARGS(
	mp_float_array,		c,
	mp_int,			n,
	mp_int,			flag,
	mp_float,		x,
	mp_float,		y
)
/*
Returns y = chebyshev series evaluated at x.  The n coefficients are in c.
For a description of c, n, and flag, see mp_chebyshev().
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t;
    mp_acc_float	temp1, temp2, temp3, temp4;
    mp_int		i;


    if (n < 0 || int_abs(flag) > 1)
	mp_error("mp_chebyshev: illegal arguments");

    round = MP_TRUNC;

    b = mp_b(xp);
    t = mp_t(xp);

    mp_acc_float_alloc_4(b, t, temp1, temp2, temp3, temp4);

    mp_set_sign(mp_acc_float_ptr(temp1), 0);
    mp_set_sign(mp_acc_float_ptr(temp2), 0);

    mp_mul_int(x, 2, temp4);

    if (flag)
    {
	mp_mul_eq(temp4, temp4);
	mp_add_int_eq(temp4, -2);
    }

    for (i = n - 1; i >= 0; i--)
    {
	/*
	Interchange pointers rather than mp numbers.
	*/

	mp_acc_float	swap_temp;

	swap_temp = temp3;
	temp3 = temp2;
	temp2 = temp1;
	temp1 = swap_temp;

	mp_mul(temp4, temp2, temp1);
	mp_sub_eq(temp1, temp3);
	mp_add_eq(temp1, mp_array_element(c, i));
    }

    if (flag >= 0)
    {
	mp_sub_eq(temp1, temp3);
	mp_div_int(temp1, 2, y);
    }

    else
    {
	mp_sub_eq(temp1, temp2);
	mp_mul(x, temp1, y);
    }

    mp_acc_float_delete(temp4);
    mp_acc_float_delete(temp3);
    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);
    round = save_round;

    return y;
}
