#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_gam_q		WITH_3_ARGS(
	mp_int,	i,
	mp_int, 	j,
	mp_float,	x
)
/*
Returns x = Gamma(i/j) for small integers i and j.  The time taken is O(t^2)
if i/j is not too large.  If i/j > 100 (approximately) it is faster to use
mp_gamma().
*/
{
    /*
    The method used is reduction of the argument to (0, 1) and then a
    direct expansion of the defining integral truncated at a
    sufficiently high limit, using 2t digits to compensate for
    cancellation.  mp_gam_q() is very slow if i/j is very large,
    because the relation Gamma(x+1) = x * Gamma(x) is used repeatedly.
    mp_gam_q() could be speeded up by using the asymptotic series or
    continued fraction for (integral from n to infinity of
    u^(i/j - 1) * exp(-u)du).
    */

    mp_ptr_type		xp = mp_ptr(x);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t, new_t, working_t, t_plus_2;
    mp_int		n, id, in, il, i2, limit;


    if (j == 0)
	mp_error("mp_gam_q: j is zero");

    DEBUG_BEGIN(DEBUG_GAMMA);
    DEBUG_PRINTF_1("+gam_q {\n");
    DEBUG_PRINTF_3("i = %d, j = %d\n\n", i, j);

    round = MP_TRUNC;

    /*
    Make j positive and reduce to lowest terms.
    */

    if (j < 0)
    {
	i = -i;
	j = -j;
    }

    mp_int_gcd(&i, &j);
    i2 = 1;

    if (j == 1)
    {
	if (i <= 0)
	    mp_error("mp_gam_q: i/j is zero or a negative integer");

	/*
	Here i/j is a positive integer.
	*/

	mp_int_to_mp(1, x);
    }

    else if (j == 2)
    {
	/*
	For speed, treat as special case.
	*/

	mp_pi(x);
	mp_sqrt(x, x);
    }

    else
    {

	mp_acc_float	temp1, temp2;
	mp_ptr_type	temp1_ptr, temp2_ptr;

	/*
	j > 2 so reduce to (0, 1).
	*/

	i2 = unsigned_mod(i, j);

	/*
	Now 0 < i2 < j.  Compute upper limit of integral.  n will be greater
	than or equal to t * log(b).
	*/

	b = mp_b(xp);
	t = mp_t(xp);

	n = mp_times_log2_b((7 * t + 9) / 10, b);

	t_plus_2 = t + 2;

	/*
	Use greater precision temporary floats to compensate for cancellation
	with new_t >= n/log(b).
	*/

	new_t = t + mp_change_base(b, 2, (3 * n + 1) / 2);

	mp_acc_float_alloc_2(b, new_t, temp1, temp2);
	mp_change_up();

	mp_int_to_mp(n, temp1);
	mp_copy(temp1, temp2);

	il = 0;
	in = j - i2;
	id = i2;

	working_t = new_t;

	temp1_ptr = mp_acc_float_ptr(temp1);
	temp2_ptr = mp_acc_float_ptr(temp2);

#define fix_pointers()		if (mp_has_changed())			    \
				{					    \
				    temp1_ptr = mp_acc_float_ptr(temp1);    \
				    temp2_ptr = mp_acc_float_ptr(temp2);    \
				}


	do
	{
	    if (++il >= n)
		working_t = mp_expt(temp2_ptr) + t_plus_2;

	    if (working_t < 2)
		working_t = 2;

	    else if (working_t > new_t)
		working_t = new_t;

	    mp_t(temp2_ptr) = working_t;

	    /*
	    Check if in - j or id + j might overflow.
	    */

	    if (int_max(int_abs(in), id) > MAX_INT - j)
		mp_error("mp_gam_q: j too large");
	    
	    in -= j;
	    id += j;

	    mp_mul_double_q(temp2, n, in, il, id);


	    /*
	    Adjust t for add.
	    */

	    if (working_t < t_plus_2)
		working_t = t_plus_2;
	    
	    fix_pointers();

	    mp_t(temp1_ptr) = mp_t(temp2_ptr) = working_t;


	    /*
	    Add in the next term.
	    */

	    mp_add_eq(temp1, temp2);
	    fix_pointers();

	} while (!mp_is_zero(temp2_ptr) && mp_expt(temp2_ptr) >= -t);


	/*
	Restore original t.
	*/

	mp_t(temp1_ptr) = mp_t(temp2_ptr) = t;

	mp_mul_q(temp1, j, i2, x);
	mp_q_power(n, 1, i2 - j, j, temp2);
	mp_mul_eq(x, temp2);

	mp_acc_float_delete(temp2);
	mp_acc_float_delete(temp1);
	mp_change_down();
    }

    /*
    Now x is Gamma(i2/j), so use the recurrence relation repeatedly to get
    Gamma(i/j) (slow if i/j is large).
    */

    limit = MAX_INT / j;

    while (in = id = 1, i != i2)
    {
	if (i < i2)

	    do
	    {
		in *= j;
		id *= i;
		i += j;
	    } while (in <= limit && int_abs(id) <= MAX_INT / int_abs(i) && i != i2);

	else

	    do
	    {
		in *= i2;
		id *= j;
		i2 += j;
	    } while (id <= limit && int_abs(in) <= MAX_INT / int_abs(i2) && i != i2);

	mp_mul_q_eq(x, in, id);
    }

    round = save_round;

    DEBUG_1("-} x = ", xp);
    DEBUG_END();

    return x;

#undef fix_pointers
}


mp_float
mp_log_gam	WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = log(Gamma(x)), for positive x, using Stirling's aymptotic
expansion.  This is slower than mp_gam_q() (unless x is large), and
uses more space, so use mp_gam_q() and mp_log() if x is rational and
not too large (say <= 200).  The space required is O(t^2) (more precisely:
(t * log2(b)/12 + O(1)) * t/2 + O(t)).  The time taken is O(t^3).
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t;
    mp_int		nl, nt, q, nl_2, xl, i;
    mp_acc_float	temp1, temp2, temp3;


    DEBUG_BEGIN(DEBUG_GAMMA);
    DEBUG_PRINTF_1("+log_gam {\n");
    DEBUG_1("x = ", xp);

    mp_check_2("mp_log_gam", xp, yp);

    if (!mp_is_pos(xp))
	mp_error("mp_log_gam: x is not positive");

    b = mp_b(xp);
    t = mp_t(xp);

    /*
    Estimate:
	nl = number of terms needed in Stirling's approximation,
	xl = lower bound on x for sufficient accuracy.
    */

    nl = mp_times_log2_b(t, b) / 12 + 2;
    nl_2 = 2 * nl;

    nt = t + 1;
    q = 1;

    for (i = 2; i <= nl_2; i++)
    {
	while (q > MAX_INT / i)
	{
	    q /= b;
	    q++;
	    nt++;
	}

	q *= i;
    }


    xl = 1;

    do
	xl++;
    while (mp_change_base(6 * xl + xl / 4, b, nt) > nl_2);


    DEBUG_PRINTF_2("xl = %d\n", xl);

    round = MP_TRUNC;
    mp_acc_float_alloc_3(b, t, temp1, temp2, temp3);
    mp_change_up();

    mp_move(x, temp2);
    mp_set_sign(mp_ptr(y), 0);


    /*
    See if x large enough to use asymptotic series.
    */

    if (mp_cmp_int(temp2, xl) < 0)
    {
	/*
	Here x is not large enough, so increase using the identity:

		Gamma(x + 1) = x * Gamma(x).

	to correct the result.
	*/

	mp_bool		first = TRUE;


	do
	{
	    if (first)
	    {
		first = FALSE;
		mp_copy(temp2, y);
	    }

	    else
		mp_mul_eq(y, temp2);

	    mp_add_int_eq(temp2, 1);

	} while (mp_cmp_int(temp2, xl) < 0);

	mp_log(y, y);

	if (mp_has_changed())
	    yp = mp_ptr(y);

	mp_set_sign(yp, -mp_sign(yp));
    }

    /*
    Compute first terms in Stirling's approximation.
    */

    mp_log(temp2, temp3);
    mp_add_q(temp2, -1, 2, temp1);
    mp_mul_eq(temp3, temp1);
    mp_sub_eq(temp3, temp2);
    mp_add_eq(y, temp3);

    mp_pi(temp3);
    mp_mul_int_eq(temp3, 2);
    mp_log(temp3, temp3);
    mp_div_int_eq(temp3, 2);
    mp_add_eq(y, temp3);


    /*
    If x is very large, we can return here.
    */

    if (mp_acc_expt(temp2) < t)
    {
	mp_ptr_type		temp1_ptr = mp_acc_float_ptr(temp1);
	mp_float_array		bern = mp_float_array_alloc(nl, b, t);

	mp_bern(bern, -nl);
	mp_int_power(temp2, -2, temp3);

	for (i = 0; i < nl; i++)
	{
	    mp_copy(mp_array_element(bern, i), temp1);

	    mp_mul_double_q(temp1, 1, 1, 2 * i + 2, 2 * i + 1);
	    mp_mul_eq(temp2, temp3);
	    mp_mul_eq(temp1, temp2);

	    if (mp_has_changed())
		temp1_ptr = mp_acc_float_ptr(temp1);

	    if (mp_is_zero(temp1_ptr) || mp_expt(temp1_ptr) <= -t)
		break;

	    mp_add_eq(y, temp1);
	}

	mp_delete_float_array(bern);
    }

    round = save_round;

    mp_acc_float_delete(temp3);
    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);

    DEBUG_1("-} y = ", yp);
    DEBUG_END();

    return y;
}


mp_float
mp_gamma	WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = Gamma(x) using mp_gam_q() if int_abs(x) <= MAX_INT / 240 - 1 and
240 * x is an integer, otherwise using mp_log_gam().  The space required
is O(t^2), time O(t^3).
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t;
    mp_acc_float	temp1, temp2;
    mp_ptr_type		temp1_ptr, temp2_ptr;


    DEBUG_BEGIN(DEBUG_GAMMA);
    DEBUG_PRINTF_1("+gamma {\n");
    DEBUG_1("x = ", xp);

    mp_check_2("mp_gam", xp, yp);

    b = mp_b(xp);
    t = mp_t(xp);

    mp_change_up();
    mp_acc_float_alloc_2(b, t, temp1, temp2);
    round = MP_TRUNC;

    temp1_ptr = mp_acc_float_ptr(temp1);
    temp2_ptr = mp_acc_float_ptr(temp2);


#define fix_pointers()		if (mp_has_changed())			    \
				{					    \
				    xp = mp_ptr(x);	    		    \
				    temp1_ptr = mp_acc_float_ptr(temp1);    \
				    temp2_ptr = mp_acc_float_ptr(temp2);    \
				}

    mp_abs(x, temp2);

    if (mp_cmp_int(temp2, MAX_INT / 240 - 1) <= 0)
    {
	/*
	See if 240 * x is almost an integer.
	*/

	mp_int		ix, count;
	mp_bool		use_gam_q = FALSE;


	ix = mp_to_int(mp_mul_int(x, 240, temp2));

	/*
	Compare with ix and ix + 1 because temp2 could be just below an
	integer.
	*/

	for (count = 0; count < 1; count++)
	{
	    mp_add_int(temp2, -ix, temp1);

	    fix_pointers();

	    if (mp_is_zero(temp1_ptr) || mp_expt(temp2_ptr) - mp_expt(temp1_ptr)
			>= t - 1 && mp_expt(temp2_ptr) >= mp_expt(temp1_ptr))
		
	    {
		use_gam_q = TRUE;
		break;
	    }
	    
	    ix++;
	}

	if (use_gam_q || mp_expt(xp) <= -t)
	{
	    if (use_gam_q)
	    {
		/*
		x == ix / 240 so use mp_gam_q() unless x is zero or
		a negative integer.
		*/

		if (ix <= 0 && int_abs(ix) % 240 == 0)
		    mp_error("mp_gamma: x is zero or a negative integer");

		mp_gam_q(ix, 240, y);
	    }

	    else
		/*
		int_abs(x) is very small.
		*/

		mp_rec(x, y);

	    round = save_round;

	    mp_acc_float_delete(temp2);
	    mp_acc_float_delete(temp1);
	    mp_change_down();

	    DEBUG_1("-} y = ", yp);
	    return y;
	}

	/*
	int_abs(x) is large or not simple rational; fall through to other case.
	*/
    }


    /*
    x is not close enough to an integer.  Check sign of x.
    */

    fix_pointers();

    if (mp_is_neg(xp))
    {
	/*
	Use reflection formula.  Subtract even integer to avoid errors
	near poles.
	*/

	mp_int		n;

	mp_abs(x, y);

	mp_div_int(y, 2, temp2);
	mp_to_frac(temp2, temp2);
	mp_mul_int_eq(temp2, 2);
	mp_add_q(temp2, 1, 2, temp1);

	n = mp_to_int(temp1);
	mp_add_int_eq(temp2, -n);

	/*
	Now int_abs(temp2) <= 1/2 and the sign is determined by n.
	*/

	fix_pointers();

	if (mp_is_zero(temp2_ptr))
	    mp_error("mp_gamma: x is zero or a negative integer");

	mp_pi(temp1);
	mp_mul_eq(temp2, temp1);
	mp_sin(temp2, temp2);
	mp_mul_eq(temp2, y);

	fix_pointers();
	if (mp_is_zero(temp2_ptr))
	    mp_error("mp_gamma: overflow");

	mp_div_eq(temp1, temp2);

	if (n % 2 == 0)
	{
	    fix_pointers();
	    mp_set_sign(temp1_ptr, -mp_sign(temp1_ptr));
	}

	mp_log_gam(y, y);

	fix_pointers();

	mp_set_sign(yp, -mp_sign(yp));

	mp_exp(y, y);
	mp_mul(y, temp1, y);
    }

    else
    {
	/*
	x is positive so use mp_log_gam() directly.
	*/

	mp_log_gam(x, y);
	mp_div_int(y, MAX_EXPT, temp1);

	/*
	Note that log(2) < 7/10.
	*/

	if (mp_cmp_q(temp1, mp_times_log2_b(7, b), 10) >= 0)
	    mp_error("mp_gamma: overflow");

	/*
	It's usually safe to call mp_exp() here.
	*/

	mp_exp(y, y);
    }

    round = save_round;

    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);
    mp_change_down();

    DEBUG_1("-} y = ", mp_ptr(y));

    return y;
}
