#include "defs.h"
#include "debug.e"
#include "integer.e"
#include "poly.h"
#include "dyn_arr.h"
#include "poly_mat.h"
#include "poly_z_faclst.h"
#include "zm.e"
#include "error.e"

t_handle
poly_u_zm_fact_b4_large WITH_4_ARGS(
    t_handle,       pring,
    integer_big,    pdig,
    t_poly,      apoly,
    t_handle,         vecth
)
/*
 * poly_u_zm_fact_b4_large: modular univariate polynomial determining the
 * factors for a large prime modulus. Random algorithm.
 * pdig : prime beta-integer
 * vecth : matrix of vector entries determined in step B3 of Berlekamp's
 * factorization algorithm as set out in Knuth pp.423
 * Assume that, there are more than one factors., i.e. that the poly
 * is not irreducible.
*/
{
    block_declarations;
    t_handle           aph;           /* t_handle to apoly                      */
    t_poly        vpower;        /* linear combination^((p-1)/2) - 1     */
    t_poly        upoly;         /* a non-trivial factor already found   */
    t_poly        gi;            /* gcd( upoly, vpower )                 */
    t_handle           vkph;          /* t_handle for the kth poly of vecth     */
    t_handle           resmh;         /* result t_handle of 1-D matrix of polys */
    t_int    total_factors; /* total factors to be found            */
    t_int    nfactors;      /* total number of factors found so far */
    t_int    fcount;        /* counter of factors                   */
    t_int    nterms;        /* number of terms in the vectors       */
    t_int    termno;        /* counter of terms                     */
    t_int    k;             /* counter of vectors                   */
    t_int    i;             /* factor counter                       */
    t_int    randmod;       /* random number modulus pdig           */
    t_matrix		phdl;
    integer_big		temp1;
    integer_big		temp2;
    t_int	anterms;
    t_int	atermno;
    integer_big		acoefft;
    t_int	aexpt;
    t_int	j;
    t_int	len;
    t_int	off_a;
    t_int	off_b;
    t_int	off_apoly;
    t_int	off_k;
    t_int	apolydeg;
    t_int	dega;
    t_int	degb;

    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B4_LARGE,
    {
	cay_print( "entering poly_u_zm_fact_b4_large()\nvectors are\n" );
	poly_z_Qmatrix_write( vecth );
    }
    );

    aph = m_poly_poly_to_handle( apoly );

    total_factors = m_poly_mat_row( vecth );    
    /* total number of factors to be found   */

    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B4_LARGE,
    {
	cay_print( "expecting %d factors from vectors\n", total_factors );
	poly_z_Qmatrix_write( vecth );
    }
    );

    nterms = m_poly_mat_col( vecth );

    resmh = m_poly_z_faclst_alloc( total_factors );
    m_poly_z_faclst_factor( resmh, 0 ) = m_modpoly_incref( pring, apoly );
    m_poly_z_faclst_power( resmh, 0 ) = 1;
    m_poly_z_faclst_len_put( resmh, 1 );
    nfactors = 1;

#define ROW_APOLY 1
#define ROW_A 2
#define ROW_B 3
#define ROW_C 4
#define ROW_TOTAL 4

    len = 2 * nterms;
    off_a = (ROW_A - 1) * len;
    off_b = (ROW_B - 1) * len;
    off_apoly = (ROW_APOLY - 1) * len;

    phdl = poly_mat_new( ROW_TOTAL, len );

    for ( j = ROW_TOTAL * len; j >= 1; --j )
    {
	m_poly_mat_entry( phdl, j ) = 0;
    }

    aph = m_poly_poly_to_handle( apoly );
    anterms = m_poly_nterms( aph );
    apolydeg = m_poly_expt( aph, anterms-1 );

    for ( atermno=0; atermno < anterms; atermno++ )
    {
	    aexpt = m_poly_expt( aph, atermno );
	    acoefft = m_poly_coefft( aph, atermno );
	    m_poly_mat_entry( phdl, off_apoly + aexpt+1 ) = integer_incref( acoefft );
    }

    while ( nfactors < total_factors )
    /*
     * no. of factors found < total no. of factors possible.
     * use different vpoly on each iteration.
     */
    {
	for ( j=1; j<=nterms; ++j )
	{
	    temp1 = m_poly_mat_entry( phdl, off_a + j );
	    integer_delref( temp1 );
	    m_poly_mat_entry( phdl, off_a + j ) = 0;
	}

	/* ROW_A := a1*vect[1] + a2*vect[2] + ... + an*vect[n] */
	/* ai are random numbers such that 0 <= ai < pdig      */

	IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B4_LARGE,
	{
	    cay_print( "vpoly =" );
	}
	);

	for ( k = 1; k <= total_factors; k++ )
	{
	    randmod = utl_rand_int( 0, pdig-1 );

	    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B4_LARGE,
	    {
		if ( k != 1 )
		{
		    cay_print( " +" );
		}
		cay_print( " %d * v_%d", randmod, k );
	    }
	    );

	    off_k = (k - 1) * nterms;

	    for ( j=1; j<=nterms; ++j )
	    {
		temp1 = m_poly_mat_entry( vecth, off_k + j );
		temp2 = modint_mult( pdig, temp1, randmod );
		temp1 = m_poly_mat_entry( phdl, off_a + j );
		m_poly_mat_entry( phdl, off_a + j ) = modint_add( pdig, temp1, temp2 );
		integer_delref( temp1 );
		integer_delref( temp2 );
	    }
	}

	IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B4_LARGE,
	{
	    cay_print( "\n" );
	    cay_print( "pdhl = \n" );
	    poly_z_Qmatrix_write( phdl );
	}
	);

	/* vpoly is established         */
	/* vpower = vpoly^((p-1)/2) - 1 */

	for ( j = nterms; ; --j )
	{
		if ( j == 0 || m_poly_mat_entry( phdl, off_a + j ) != 0 )
		{
			dega = j;
			break;
		}
	}

	poly_u_zm_d_power_rem( pdig, (pdig-1)/2, phdl, dega, ROW_A, apolydeg+1, ROW_APOLY, &degb, ROW_B, ROW_C );
	temp1 = m_poly_mat_entry( phdl, off_b + 1 );
	m_poly_mat_entry( phdl, off_b + 1 ) = modint_subtract( pdig, temp1, 1 );
	integer_delref( temp1 );

	m_poly_create_empty(&vkph, m_poly_princvar( aph ),
                                           m_poly_princvar (aph), degb );

	for ( termno = 0; termno < degb; termno++ )
	{
		m_poly_coefft( vkph, termno ) = m_poly_mat_entry( phdl, off_b + termno+1 );
		m_poly_mat_entry( phdl, off_b + termno+1 ) = 0;
		m_poly_expt( vkph, termno ) = termno;
	}

	vpower = poly_z_clean( pring, m_poly_handle_to_poly( vkph ));

	IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT_B4_LARGE,
	{
	    cay_print( "vpower = " );
	    poly_z_write( pring, vpower );
	    cay_print( "\n" );
	}
	);

	/*
	* if current partial factorisation of apoly is
	* u1(x),...,ut(x) then for all i s.t. deg(ui) > 1
	* replace ui(x) by gi(x) and (ui(x)/gi(x) )
	*/

	for ( i=0; i<nfactors; ++i )
	{
	    upoly = m_poly_z_faclst_factor( resmh, i );

	    if ( poly_deg( upoly ) < 2 )
	    {
		continue;
	    }

	    gi = poly_u_zm_gcd( pring, pdig, upoly, vpower );

#if 0
	    vmpower = modpoly_pseudo_rem( pring, pdig, vpower, upoly );
	    gi = poly_u_zm_gcd( pring, pdig, vmpower, upoly );
	    m_modpoly_delref( pring, vmpower );
#endif

	    if ((!poly_z_is_one_poly(pring, gi)) && poly_u_zm_different( pring, pdig, upoly, gi ) )
	    {
		/*
		 * if there is a non-trivial gcd of a factor with vpower then 
		 * it should be inserted. however if nfactors is already
		 * total_factors then something has gone wrong.
		 */

		DENY( nfactors == total_factors );

		/* if gi is non-trivial factor, replace upoly with gi, 
		 * and upoly/gi
		 *
		 * remove upoly from result table;
		 * i.e. move other entries up
		 */

		for ( fcount = i; fcount < nfactors - 1; fcount++ )
		{
		    m_poly_z_faclst_factor( resmh, fcount ) = m_poly_z_faclst_factor( resmh, fcount + 1 );
		}

		/* insert gi into result table */

		m_poly_z_faclst_factor( resmh, nfactors - 1 ) = gi;

		/* insert upoly/gi into result table */

		m_poly_z_faclst_factor( resmh, nfactors ) = modpoly_div( pring, pdig, upoly, gi );
		m_poly_z_faclst_power( resmh, nfactors ) = 1;

		nfactors++;
		m_poly_z_faclst_len_put( resmh, nfactors );

		m_modpoly_delref( pring, upoly );
	    }
	    else
	    {
		m_modpoly_delref( pring, gi );
	    }
	}

	m_modpoly_delref( pring, vpower );
    }

    for ( i = ROW_TOTAL * len; i >= 1; --i )
    {
	temp1 = m_poly_mat_entry( phdl, i );
	integer_delref( temp1 );
    }
    mem_delete_h( phdl );

    return  resmh;
}
