//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : bigint_lattice.h 
// Author      : Werner Backes (WB), Thorsten Lauer (TL) 
// Last change : WB/TL, Feb 10 1995, initial version
//

#ifndef _bigint_lattice_h
#define _bigint_lattice_h

#include <LiDIA/bigfloat.h>
#include <LiDIA/bigint_matrix.h>
#include <LiDIA/timer.h>

#ifndef rint
#define rint(x) (((x)-floor(x))<0.5?floor(x):ceil(x)) 
#endif

class lattice_gensys;
class lattice_basis;

class bigint_lattice 
{
  private :
    friend      class lattice_gensys;
    friend      class lattice_basis;
    long 	rows;				// Number of Zeilen
    long 	cols;				// Number of Spalten
    bigint**	value;				// Values (column by column)

    long 	cols_rank;			// rank of the lattice
						// for internal use

//
// static variables for temporary use to increase speed
//
    static bigint tempvz;
    static bigint tempmz0;
    static bigint tempmz1;
    static bigint tempmz2;
    static bigint tempmz3;
    static bigint tempmz4;
    static bigint ergmz;

    static long vectsize;

    static long tempvectsize;
    static bigint* tempvect0;    
    static bigint* tempvect1;    

    static long last;

  public :

//
// Constructor / Destructors 
//
    bigint_lattice(long, long);
    bigint_lattice(long, long, bigint**);
    bigint_lattice(const bigint_lattice&);
    bigint_lattice(const bigint_matrix&);
    ~bigint_lattice();    

//  
// Input / Output
//
    friend istream& operator >> (istream&, bigint_lattice&);
    friend ostream& operator << (ostream&, const bigint_lattice&);

//
// Assignments
//
    bigint_lattice operator = (const bigint_lattice&);
    bigint_lattice operator = (const bigint_matrix&);
    void assign(const bigint_lattice&);
    void assign(const bigint_matrix&);
    void assign_zero();

//
// Algorithms
//
    void lll(unsigned long&, long, long, timer&);   
//    void lll_rand(unsigned long&, long, long, timer&);   
    void lll_dbl(double, unsigned long&, timer&);
    void lll_bfl(double, unsigned long&, timer&, long);
    void lll_trans_dbl(bigint_lattice&, double, unsigned long&, timer&);
    void lll_trans(bigint_lattice&, unsigned long&, long, long, timer&);   
    void mlll(bigint*, unsigned long&, timer&);
    void lin_gen_system(bigint_lattice&, long, long, 
                        unsigned long&, timer&, long&);
    void compute_basis(const bigint_lattice&, const bigint_lattice&);

  protected :
  
  
    
//
// "Vector" - Operations
//

#ifdef __GNUC__

    friend void assign(bigint* a, bigint* b)
    {
      debug_handler("bigint_lattice", "friend void assign(a, b)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        a[i].assign(b[i]);
    }

    friend void assign_zero(bigint* a)
    {
      debug_handler("bigint_lattice", "friend void assign_zero(a)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        a[i].assign_zero();
    }

    friend void add(bigint* c, bigint* a, bigint* b)
    {
      debug_handler("bigint_lattice", "friend void add(c, a, b)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        add(c[i],a[i],b[i]);
    }

    friend void subtract(bigint* c, bigint* a, bigint* b)
    {
      debug_handler("bigint_lattice", "friend void subtract(c, a, b)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        subtract(c[i],a[i],b[i]);
    }

    friend void scalmul(bigint* b, const bigint& d, bigint* a)
    {
      debug_handler("bigint_lattice", "friend scalmul(b, d, a)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        multiply(b[i], a[i], d);
    }

    friend void scalprod(bigint& res, bigint* a, bigint* b)
    {
      debug_handler("bigint_lattice", "friend scalprod(res, a, b)");
      long i;
      res.assign_zero();
      for (i=bigint_lattice::vectsize-1;i>=0;i--) 
        {    
          multiply(bigint_lattice::tempvz, a[i], b[i]);
          add(res, res, bigint_lattice::tempvz);
        }
    }

    friend void scalquad(bigint& res, bigint* a)
    {
      debug_handler("bigint_lattice", "friend scalquad(res, a)");
      long i;
      res.assign_zero();
      for (i=bigint_lattice::vectsize-1;i>=0;i--) 
        {    
          square(bigint_lattice::tempvz, a[i]);
          add(res, res, bigint_lattice::tempvz);
        }
    }

    friend void l2_norm(bigint& norm, bigint* v)
    {
      debug_handler("bigint_lattice", "l2_norm(norm, v)");
      long i;
      norm.assign_zero();		// Initialisation
      for (i=bigint_lattice::vectsize-1;i>=0;i--) 
        {
          square(bigint_lattice::tempvz, v[i]);
          add(norm, norm, bigint_lattice::tempvz);
        }
    //  sqrt(norm, norm);
    }

    friend void l1_norm(bigint& norm, bigint* v)
    {
      debug_handler("bigint_lattice", "l1_norm(norm, v)");
      long i;
      norm.assign_zero();	 // Initialisation
      for (i=bigint_lattice::vectsize-1;i>=0;i--) 
        {
          bigint_lattice::tempvz.assign(abs(v[i]));
          add(norm, norm, bigint_lattice::tempvz);
        }
    }

    friend void bfl_assign(bigfloat* a, bigfloat* b)
    {
      debug_handler("bigint_lattice", "friend bfl_assign(a, b)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        a[i].assign(b[i]);
    }

    friend void bfl_assign_zero(bigfloat* a)
    {
      debug_handler("bigint_lattice", "friend bfl_assign_zero(a)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        a[i].assign_zero();
    }

    friend void bfl_add(bigfloat* c, bigfloat* a, bigfloat* b)
    {
      debug_handler("bigint_lattice", "friend bfl_add(c, a, b)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        add(c[i],a[i],b[i]);
    }

    friend void bfl_subtract(bigfloat* c, bigfloat* a, bigfloat* b)
    {
      debug_handler("bigint_lattice", "friend bfl_subtract(c, a, b)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        subtract(c[i],a[i],b[i]);
    }

    friend void bfl_scalmul(bigfloat* b, const bigfloat& d, bigfloat* a)
    {
      debug_handler("bigint_lattice", "friend bfl_scalmul(b, d, a)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        multiply(b[i], a[i], d);
    }

    friend void bfl_scalprod(bigfloat& res, bigfloat* a, bigfloat* b)
    {
      debug_handler("bigint_lattice", "friend bfl_scalprod(res, a, b)");
      long i;
      bigfloat temp;
      res.assign_zero();
      for (i=bigint_lattice::vectsize-1;i>=0;i--) 
        {    
          multiply(temp, a[i], b[i]);
          add(res, res, temp);
        }
    }

    friend void bfl_scalquad(bigfloat& res, bigfloat* a)
    {
      debug_handler("bigint_lattice", "friend bfl_scalquad(res, a)");
      long i;
      bigfloat temp;
      res.assign_zero();
      for (i=bigint_lattice::vectsize-1;i>=0;i--) 
        {    
          square(temp, a[i]);
          add(res, res, temp);
        }
    }

    friend void bfl_l2_norm(bigfloat& norm, bigfloat* v)
    {
      debug_handler("bigint_lattice", "bfl_l2_norm(norm, v)");
      long i;
      bigfloat temp;
      norm.assign_zero();		// Initialisation
      for (i=bigint_lattice::vectsize-1;i>=0;i--) 
        {
          square(temp, v[i]);
          add(norm, norm, temp);
        }
      sqrt(norm, norm);
    }

    friend void bfl_l1_norm(bigfloat& norm, bigfloat* v)
    {
      debug_handler("bigint_lattice", "friend bfl_l1_norm(norm, v)");
      long i;
      bigfloat temp;
      norm.assign_zero();	 // Initialisation
      for (i=bigint_lattice::vectsize-1;i>=0;i--) 
        {
          temp.assign(abs(v[i]));
          add(norm, norm, temp);
        }
    }

    friend void dbl_assign(double* a, double* b)
    {
      debug_handler("bigint_lattice","friend void dbl_assign(a, b)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        a[i]=b[i];
    }

    friend void dbl_assign_zero(double* a)
    {
      debug_handler("bigint_lattice","friend void dbl_assign_zero(a)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        a[i]=0;
    }

    friend void dbl_add(double* c, double* a, double* b)
    {
      debug_handler("bigint_lattice","friend void dbl_add(c, a, b)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        c[i]=a[i]+b[i];
    }

    friend void dbl_subtract(double* c, double* a, double* b)
    {
      debug_handler("bigint_lattice","friend void dbl_subtract(c, a, b)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        c[i]=a[i]-b[i];
    }

    friend void dbl_scalmul(double* b, const double& d, double* a)
    {
      debug_handler("bigint_lattice","friend dbl_scalmul(b, d, a)");
      long i;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        b[i]=a[i]*d;
    }

    friend void dbl_scalprod(double& res, double* a, double* b)
    {
      debug_handler("bigint_lattice","friend dbl_scalprod(res, a, b)");
      long i;
      res=0;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        res+=a[i]*b[i];
    }

    friend void dbl_scalquad(double& res, double* a)
    {
      debug_handler("bigint_lattice","friend dbl_scalquad(res, a, b)");
      long i;
      res=0;
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        res+=a[i]*a[i];
    }

    friend void dbl_l2_norm(double& norm, double* v)
    {
      debug_handler("bigint_lattice","friend dbl_l2_norm(norm, v)");
      long i;
      norm=0;		// Initialisation
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        norm+=v[i]*v[i];
      norm=sqrt(norm);
    }

    friend void dbl_l1_norm(double& norm, double* v)
    {
      debug_handler("bigint_lattice","friend dbl_l1_norm(norm, v)");
      long i;
      norm=0;	 // Initialisation
      for (i=bigint_lattice::vectsize-1;i>=0;i--)
        norm+=fabs(v[i]);
    }

#else

    friend void dbl_assign(double*, double*); 
    friend void dbl_assign_zero(double*);
    friend void dbl_add(double*, double*, double*);
    friend void dbl_subtract(double*, double*, double*);

    friend void dbl_scalmul(double*, const double&, double*);
    friend void dbl_scalprod(double&, double*, double*);
    friend void dbl_scalquad(double&, double*);
    
    friend void dbl_l1_norm(double&, double*);
    friend void dbl_l2_norm(double&, double*);

    friend void bfl_assign(bigfloat*, bigfloat*); 
    friend void bfl_assign_zero(bigfloat*);
    friend void bfl_add(bigfloat*, bigfloat*, bigfloat*);
    friend void bfl_subtract(bigfloat*, bigfloat*, bigfloat*);

    friend void bfl_scalmul(bigfloat*, const bigfloat&, bigfloat*);
    friend void bfl_scalprod(bigfloat&, bigfloat*, bigfloat*);
    friend void bfl_scalquad(bigfloat&, bigfloat*);
    
    friend void bfl_l1_norm(bigfloat&, bigfloat*);
    friend void bfl_l2_norm(bigfloat&, bigfloat*);

    friend void assign(bigint*, bigint*); 
    friend void assign_zero(bigint*);
    friend void add(bigint*, bigint*, bigint*);
    friend void subtract(bigint*, bigint*, bigint*);

    friend void scalmul(bigint*, const bigint&, bigint*);
    friend void scalprod(bigint&, bigint*, bigint*);
    friend void scalquad(bigint&, bigint*);
    
    friend void l1_norm(bigint&, bigint*);
    friend void l2_norm(bigint&, bigint*);

#endif

    friend void swap(bigint*& a, bigint*& b)
    {
      register bigint* temp;
      temp=a;
      a=b;
      b=temp;
    }

    friend void dbl_swap(double*& a, double*& b)
    {
      register double* temp;
      temp=a;
      a=b;
      b=temp;
    }

    friend void bfl_swap(bigfloat*& a, bigfloat*& b)
    {
      register bigfloat* temp;
      temp=a;
      a=b;
      b=temp;
    }

//
//  Algorithm - Subroutines
//
    
    long compute_read_precision();
    long compute_precision();
    void alpha_compute(bigfloat&);
    void gamma_compute(bigfloat&, long);
    void zwei_pot_q_compute(bigfloat&, long&, bigfloat&);

};

#endif

