

//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : sort_vector.h 
// Author      : Frank Lehmann (FL), Markus Maurer (MM) 
//               Thomas Papanikolaou (TP)
// Last change : FL/MM, Feb 15 1995, initial version
//               FL/MM, May 10 1995, clean-up
//		 FL/MM, May 15 1995, changed type of size variables to base_vector<T>::size_type
//		 FL/MM, May 27 1995, moved protected members "sort_dir", "el_cmp"
//				     from sort_vector to base_vector; changed member
//				     operator=; removed friend swap()
//		 FL/MM, Jul  6 1995, replaced base_vector<T>::size_type by lidia_size_t
//               FL/MM, Jul 24 1995, added 'const' to T* parameters of constructors


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// *
// *    File        :  sort_vector.h
// *
// *    Description :  -) definition of template class 'sort_vector'
// *                      to allow use of sortable vectors
// *
// *                   -) virtual base class is 'base_vector'
// *
// *                   -) uses template class 'comparator'
// *
// *    Notes       :  *) any class MUST support the operators
// *                                  <, <=, and  == 
// *                      to allow its use in the template
// *
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *



#ifndef LIDIA_SORT_VECTOR_H
#define LIDIA_SORT_VECTOR_H

#include <LiDIA/lidia.h>

#include <LiDIA/base_vector.h>
#include <LiDIA/comparator.h>



const char SORT_VECTOR_DEF  = -1 ;
const char SORT_VECTOR_UP   =  0 ;
const char SORT_VECTOR_DOWN =  1 ;
const char SORT_VECTOR_CMP  =  2 ;



 template <class T> class sort_vector : virtual public base_vector < T >

  {


   /* * * * * * * * *  class description  * * * * * * * *

   public :


   //   *****  constructor  functions  *****

   sort_vector () ;
   sort_vector ( char md ) ;

   sort_vector ( lidia_size_t all ) ;
   sort_vector ( lidia_size_t all,  char md ) ;

   sort_vector ( lidia_size_t all,  lidia_size_t len ) ;
   sort_vector ( lidia_size_t all,  lidia_size_t len,  char md ) ;

   sort_vector ( const base_vector < T > & v ) ;
   sort_vector ( const base_vector < T > & v,  char md ) ;

   sort_vector ( const T *v,  lidia_size_t len ) ;
   sort_vector ( const T *v,  lidia_size_t len,  char md ) ;


   //   ***** destructor *****

   ~sort_vector() ;


   //   ***** assignment *****

   sort_vector<T> & operator= ( const sort_vector<T> & v ) ;


   //   *****  reading & modifying sort-directions  *****

   char sort_direction ( void ) ;

   int set_sort_direction ( char sd ) ; 
   int set_sort_direction ( int (*cmp) ( const T & a, const T & b ) ) ;



   //   *****  sort-functions using Quick-Sort routines  *****

   inline int sort ( int (*cmp) ( const T & a, const T & b ) , lidia_size_t l = -1 , lidia_size_t r = -1  ) ;

   inline int sort ( char sort_direction = SORT_VECTOR_DEF , lidia_size_t l = -1, lidia_size_t r = - 1 ) ;
   inline int sort ( lidia_size_t l, lidia_size_t r ) ;

   //  --- protected functions  ---

   int sort_up   ( lidia_size_t l , lidia_size_t r ) ;
   int sort_down ( lidia_size_t l , lidia_size_t r ) ;


   //    *****  functions for linear search in vectors  *****

   bool linear_search ( const T & x , lidia_size_t & pos )  const ;

   //    *****  functions for binary search in sorted vectors  *****

   bool  bin_search ( const  T & x , lidia_size_t & pos , int (*cmp) ( const T & a, const T & b ), lidia_size_t l = -1 , lidia_size_t r = -1  ) const ; 

   inline bool bin_search ( const T & x , lidia_size_t & pos , char sort_direction = SORT_VECTOR_DEF, lidia_size_t l = -1 , lidia_size_t r = -1  ) const ;
   inline bool bin_search ( const T & x , lidia_size_t & pos , lidia_size_t l , lidia_size_t r ) const ;


   //  --- protected functions  ---

   bool bin_search_up   ( const  T & x , lidia_size_t & pos , lidia_size_t l , lidia_size_t r ) const ;
   bool bin_search_down ( const  T & x , lidia_size_t & pos , lidia_size_t l , lidia_size_t r ) const ;



   //    ***** functions to insert new elts. into a vector  *****

   int insert_at ( const T & x, lidia_size_t pos ) ;

   int insert ( const T & x, int (*cmp) ( const T & a, const T & b ), lidia_size_t l = -1, lidia_size_t r = -1 ) ;

   int insert ( const T & x, char sort_direction = SORT_VECTOR_DEF, lidia_size_t l = -1, lidia_size_t r = -1 ) ;
   inline int insert ( const T & x, lidia_size_t l, lidia_size_t r ) ;


   //    ***** functions to remove elts. from within a vector  *****

   int remove_from ( lidia_size_t pos, lidia_size_t l = 1 ) ;

   int remove ( const T & x, int (*cmp) ( const T & a, const T & b ), lidia_size_t l = -1, lidia_size_t r = -1 ) ;
   
   int remove ( const T & x, char sort_direction = SORT_VECTOR_DEF, lidia_size_t l = -1, lidia_size_t r = -1 ) ;
   inline int remove ( const T & x, lidia_size_t l, lidia_size_t r ) ;

   int lex_compare ( sort_vector< T > & w ) const ;


   * * * * * * * * * * * * * * * * * * * * * * * */




   // *****  IMPLEMENTATION  *****

   public :


   // *****  constructor  functions  *****

   sort_vector () : base_vector< T > ( FIXED )
    {
       debug_handler ( "sort_vector", "constructor ()" ) ;

       sort_dir = SORT_VECTOR_UP ;
       el_cmp   = nil       ;
    }

   sort_vector ( char md ) : base_vector< T > ( md )
    {
       debug_handler ( "sort_vector", "constructor ( char )" ) ;

       sort_dir = SORT_VECTOR_UP ;
       el_cmp   = nil       ;
    }


   sort_vector ( lidia_size_t all ) : base_vector< T > ( all, FIXED )
    {
       debug_handler ( "sort_vector", "constructor ( lidia_size_t )" ) ;

       sort_dir = SORT_VECTOR_UP ;
       el_cmp   = nil       ;
    }

   sort_vector ( lidia_size_t all, char md ) : base_vector< T > ( all, md )
    {
       debug_handler ( "sort_vector", "constructor ( lidia_size_t ,  char )" ) ;

       sort_dir = SORT_VECTOR_UP ;
       el_cmp   = nil       ;
    }


   sort_vector ( lidia_size_t all, lidia_size_t len ) : base_vector< T > ( all, len, FIXED )
    {
       debug_handler ( "sort_vector", "constructor ( lidia_size_t, lidia_size_t )" ) ;

       sort_dir = SORT_VECTOR_UP ;
       el_cmp   = nil       ;
    }

   sort_vector ( lidia_size_t all, lidia_size_t len, char md ) : base_vector< T > ( all, len, md )
    {
       debug_handler ( "sort_vector", "constructor ( lidia_size_t, lidia_size_t, char )" ) ;

       sort_dir = SORT_VECTOR_UP ;
       el_cmp   = nil       ;
    }


   sort_vector ( sort_vector < T > & v ) : base_vector< T > ( v, FIXED )
    {
       debug_handler ( "sort_vector", "constructor ( base_vector < T > & )" ) ;

       sort_dir = v.sort_dir ;
       el_cmp   = v.el_cmp   ;
    }

   sort_vector ( base_vector < T > & v, char md ) : base_vector< T > ( v, md )
    {
       debug_handler ( "sort_vector", "constructor ( base_vector < T > &, char  )" ) ;

       sort_dir = SORT_VECTOR_UP ;
       el_cmp   = nil       ;
    }

   sort_vector ( const T *v, lidia_size_t len ) : base_vector< T > ( v, len, FIXED )
    {
       debug_handler ( "sort_vector", "constructor ( const T *, lidia_size_t )" ) ;

       sort_dir = SORT_VECTOR_UP ;
       el_cmp   = nil       ;
    }

   sort_vector ( const T *v, lidia_size_t len, char md ) : base_vector< T > ( v, len, md )
    {
       debug_handler ( "sort_vector", "constructor ( const T *, lidia_size_t, char )" ) ;

       sort_dir = SORT_VECTOR_UP ;
       el_cmp   = nil       ;
    }


 // ***** destructor *****

 LiDIA_VEC_VIRTUAL ~sort_vector ()
  {
    debug_handler ( "sort_vector", "destructor()") ;

    // ***   memory deallocation is done by
    // ***   base_vector - destructor
  }




 // ***** assignment *****

 sort_vector<T> & operator= ( const sort_vector<T> & v ) 
  {
    debug_handler ( "sort_vector", "operator=" ) ;

    if ( this != & v )
     {
        base_vector<T>::operator= ( v ) ;
     }

    return ( *this ) ;
  }



 // ***** sort - direction *****

 char sort_direction ( void )
  {
     debug_handler ( "sort_vector", "sort_dir ()" ) ;
     return  sort_dir ;
  }

 int set_sort_direction ( char sd )
  {
     debug_handler ( "sort_vector", "set_sort_dir ( char )" ) ;

     int rc ;
 
     if ( ( sd != SORT_VECTOR_UP ) && ( sd != SORT_VECTOR_DOWN ) )
      {
         lidia_error_handler ( "sort_vector", "set_sort_dir ( char )::invalid direction" ) ;
         rc = 1 ;
      }
     else
      {
        sort_dir = sd  ;
        el_cmp   = nil ;
        rc = 0 ;
      }

     return rc ;
  }

 int set_sort_direction ( int (*cmp) ( const T & a, const T & b ) )
  {
     debug_handler ( "sort_vector", "set_sort_dir ( int (*)( )" ) ;

     sort_dir = SORT_VECTOR_CMP ;
     el_cmp   = cmp ;

     return 0 ;
  }



 // *****  sort-function with compare-function as parameter  *****

 int sort ( int (*cmp) ( const T & a, const T & b ) , lidia_size_t l = -1 , lidia_size_t r = -1  )
  {
     debug_handler ( "sort_vector", "sort(base_vector< T >,int (*) (const T&, const T&)), lidia_size_t" ) ;

     lidia_size_t gap, i, j, len ;
     T   temp ;
     T  *ptr  ;

     // sorting complete vector by default

     if ( l == -1 )  l = 0 ;
     if ( r == -1 )  r = length - 1 ;
    
     if ( length == 0 && l == 0 && r == -1 )  // empty vector and default-cut

        return 0 ;

     if ( l < 0 || l >= length || r < 0 || r >= length || l > r )

        lidia_error_handler ( "sort_vector" , "sort( cmp, lidia_size_t, lidia_size_t )::invalid cut" ) ;

     ptr = & data[l] ;
     len = r - l + 1 ;                      // length of selected cut

     for ( gap = len / 2 ; gap > 0 ; gap /= 2 )
      {
        for ( i = gap ; i < len ; i++ )
         {
           for ( j = i - gap ; j >= 0 ; j -= gap )
            {
               if ( cmp ( ptr[j], ptr[j + gap] ) < 0 )  break ;

                temp         = ptr[j]       ;
                ptr[j]       = ptr[j + gap] ;
                ptr[j + gap] = temp         ;
            }
         }
      }

     // sort_dir = SORT_VECTOR_CMP ;
     // el_cmp   = cmp ;

     return 0 ;
   }


   // *****  sort-functions using comparator-functions  *****

   inline int sort ( char sort_direction = SORT_VECTOR_DEF , lidia_size_t l = -1, lidia_size_t r = - 1 )
    {
       debug_handler ( "sort_vector", "sort ( char , lidia_size_t, lidia_size_t )" ) ;

       if ( sort_direction == SORT_VECTOR_DEF )   sort_direction = sort_dir ;

       // sorting complete vector by default

       if ( l == -1 )  l = 0 ;
       if ( r == -1 )  r = length - 1 ;


       if ( length == 0 && l == 0 && r == -1 )  // empty vector and default-cut
          return 0 ;

       if ( l < 0 || l >= length || r < 0 || r >= length || l > r )
          lidia_error_handler ( "sort_vector" , "sort( char, lidia_size_t, lidia_size_t )::invalid cut" ) ;


       // call of corresponding function

       switch ( sort_direction )
        {
           case SORT_VECTOR_UP   : sort_up   ( l, r ) ;
                                   // sort_dir = SORT_VECTOR_UP ;
                                   break ;

           case SORT_VECTOR_DOWN : sort_down ( l, r ) ;
                                   // sort_dir = SORT_VECTOR_DOWN ;
                                   break ;

           case SORT_VECTOR_CMP  : sort      ( el_cmp, l, r ) ;
                                   // sort_dir = SORT_VECTOR_CMP ;
                                   break ;

           default : lidia_error_handler ( "sort_vector" , "sort ( char, lidia_size_t, lidia_size_t )::invalid switch-type" ) ;
        }

       return 0 ;
    }


   inline int sort ( lidia_size_t l, lidia_size_t r )
    {
       debug_handler ( "sort_vector", "sort ( lidia_size_t, lidia_size_t )" ) ;
       return (int) sort ( SORT_VECTOR_DEF , l , r ) ;
    }




   protected :

   int sort_up ( lidia_size_t l , lidia_size_t r )
    {
       debug_handler ( "sort_vector", "sort_up ( lidia_size_t, lidia_size_t )" ) ;

       lidia_size_t gap, i, j ;
       T   temp ;
       T  *ptr  ;

       ptr = & data[l] ;
       r   = r - l + 1 ;                       // length of selected cut

       for ( gap = r / 2 ; gap > 0 ; gap /= 2 )
        {
          for ( i = gap ; i < r ; i++ )
           {
             for ( j = i - gap ; j >= 0 ; j -= gap )
              {
                 if ( comparator< T >::less_equal ( ptr[j], ptr[j + gap] ) > 0 )  break ;

                 temp         = ptr[j]       ;
                 ptr[j]       = ptr[j + gap] ;
                 ptr[j + gap] = temp         ;
              }
           }
        }

       return 0 ;
    }

   int sort_down ( lidia_size_t l , lidia_size_t r )
    {
       debug_handler ( "sort_vector", "sort_down ( lidia_size_t, lidia_size_t )" ) ;

       lidia_size_t gap, i, j ;
       T   temp ;
       T  *ptr  ;

       ptr = & data[l] ;
       r   = r - l + 1 ;                       // length of selected cut

       for ( gap = r / 2 ; gap > 0 ; gap /= 2 )
        {
          for ( i = gap ; i < r ; i++ )
           {
             for ( j = i - gap ; j >= 0 ; j -= gap )
              {
                 if ( comparator< T >::greater_equal ( ptr[j], ptr[j + gap] ) > 0 )  break ;

                 temp         = ptr[j]       ;
                 ptr[j]       = ptr[j + gap] ;
                 ptr[j + gap] = temp         ;
              }
           }
        }

       return 0 ;
     }




   public :

   //  *****  function for linear search in a vector  *****

    bool linear_search ( const  T & x , lidia_size_t & i ) const
     {
        debug_handler ( "sort_vector", "search ( T, lidia_size_t )" ) ;

        i = 0 ;

        while ( ( i < length ) && ! comparator< T >::equal ( x , data[i] ) )
           i++ ;

        if ( i < length )
           return true ;
        else
           return false ;
     }



   //  *****  functions for binary search in sorted vectos  *****

   //  *****  NOTE : binary search can only be applied to a vector
   //  *****         which is sorted in an appropriate order


    bool bin_search ( const  T & x, lidia_size_t & pos, int (*cmp) ( const T & a, const T & b ), lidia_size_t l = -1 ,lidia_size_t r = -1  ) const
     {
        debug_handler ( "sort_vector", "bin_search ( T, lidia_size_t, cmp(), lidia_size_t, lidia_size_t )" ) ;

        int c ;

        // search in complete vector by default

        if ( l == -1 )  l = 0 ;
        if ( r == -1 )  r = length - 1 ;

        if ( length == 0 && l == 0 && r == -1 )  // empty vector and default-cut
         {
             pos = 0 ;
             return false ;
         }

        if ( l < 0 || l >= length || r < 0 || r >= length || l > r )
          lidia_error_handler ( "sort_vector" , "bin_search( T, int, cmp, int, int )::invalid cut" ) ;


        // pick center element

        pos = ( l + r ) / 2 ;

        while ( l <= r )
         {
             c = cmp ( x , data[pos] ) ;

             if ( c == 0 )
              {
                  // search for first occurence of x

                  while ( ( pos > 0 ) &&  ( cmp ( x , data[pos-1] ) == 0 ) )
                       pos-- ;

                  return true ;
              }
             else if ( c < 0 )   // x must occur before data[pos]
              {
                  // cont. search in data[l],...,data[pos-1]
                  r = pos - 1 ;
              }
             else                // x must occur after data[pos]
              {
                  // cont. search in data[pos+1],...,data[r]
                  l = pos + 1 ;
              }

             pos = ( l + r ) / 2 ;
         }

        pos = r + 1 ;

        return false ;
     }



   inline bool bin_search ( const T & x, lidia_size_t & pos, char sort_direction = SORT_VECTOR_DEF, lidia_size_t l = -1, lidia_size_t r = -1  ) const
    {
       debug_handler ( "sort_vector", "bin_search ( T, lidia_size_t, char, lidia_size_t, lidia_size_t )" ) ;

       int rc ;

       if ( sort_direction == SORT_VECTOR_DEF )  sort_direction = sort_dir ;

       // search in complete vector by default

       if ( l == -1 )  l = 0 ;
       if ( r == -1 )  r = length - 1 ;

       if ( length == 0 && l == 0 && r == -1 )  // empty vector and default-cut
        {
            pos = false ;        
            return 0 ;
        }

       if ( l < 0 || l >= length || r < 0 || r >= length || l > r )
          lidia_error_handler ( "sort_vector" , "bin_search( T,lidia_size_t,char,lidia_size_t,lidia_size_t)::invalid cut") ;

       switch ( sort_direction )
        {
           case SORT_VECTOR_UP   : // vector is sorted in ascending orde
                                   rc = bin_search_up   ( x, pos, l, r ) ;
                                   break ;

           case SORT_VECTOR_DOWN : // vector is sorted in descending orde
                                   rc = bin_search_down ( x, pos, l, r ) ;
                                   break ;

           case SORT_VECTOR_CMP  : rc = bin_search ( x, pos, el_cmp, l, r ) ;
                                   break ;

           default : lidia_error_handler ( "sort_vector", 
                                     "bin_search(T,lidia_size_t,char,lidia_size_t,lidia_size_t)::invalid switch-type" ) ;
        }

       return rc  ;
    }


   inline bool bin_search ( const T & x , lidia_size_t & pos , lidia_size_t l , lidia_size_t r ) const
    {
       debug_handler ( "sort_vector", "bin_search ( T, lidia_size_t, lidia_size_t, lidia_size_t )" ) ;
       return (int) bin_search ( x , pos , SORT_VECTOR_DEF , l , r ) ;
    }




    protected :

    bool bin_search_up ( const  T & x , lidia_size_t & pos , lidia_size_t l , lidia_size_t r ) const
     {
        debug_handler ( "sort_vector", "bin_search_up ( T, lidia_size_t, lidia_size_t, lidia_size_t )" ) ;


        // pick center element

        pos = ( l + r ) / 2 ;

        while ( l <= r )
         {
             if ( comparator< T >::less_equal ( x , data[pos] ) )
              {
                if ( comparator< T >::equal ( data[pos] , x ) )
                 {
                     while ( pos > 0 && comparator< T >::equal ( data[pos-1] , x ) )
                          pos-- ;

                     return true ;
                 }
                else // x < data[pos]
                 {
                     // cont. search in data[l],...,data[pos-1]
                     r = pos - 1 ;
                 }
              }
             else    // x > data[pos]
              {
                  // cont. search in data[pos+1],...,data[r]
                  l = pos + 1 ;
              }

             pos = ( l + r ) / 2 ;
         }

        pos = r + 1 ;

        return false ;
     }


    int bin_search_down ( const  T & x , lidia_size_t & pos , lidia_size_t l , lidia_size_t r ) const
     {
        debug_handler ( "sort_vector", "bin_search_down ( T, lidia_size_t, lidia_size_t, lidia_size_t )" ) ;

        pos = ( l + r ) / 2 ;

        while ( l <= r )
         {
             if ( comparator< T >::greater_equal ( x , data[pos] ) )
              {
                if ( comparator< T >::equal ( data[pos] , x ) )
                 {
                     while ( pos > 0 && comparator< T >::equal ( data[pos-1] , x ) )
                          pos-- ;

                     return true ;
                 }
                else // x > data[pos]
                 {
                     // cont. search in data[l],...,data[pos-1]
                     r = pos - 1 ;
                 }
              }
             else    // x < data[pos]
              {
                  // cont. search in data[pos+1],...,data[r]
                  l = pos + 1 ;
              }

             pos = ( l + r ) / 2 ;
         }

        pos = r + 1 ;

        return false ;
     }




   public :

   //  *****  insertion into a sorted vector  *****

   int insert_at ( const T & x , lidia_size_t pos )
    {
        debug_handler ( "sort_vector", "insert_at ( T, lidia_size_t )" ) ;

	int err ;
        lidia_size_t i ;
        lidia_size_t l = length ;

        if ( pos < 0 )
         {
             lidia_error_handler ( "sort_vector" , "insert_at( T , lidia_size_t )::invalid position" ) ;
             return 1 ;
         }

        err = set_size ( (l+1) > (pos+1) ? l+1 : pos+1 ) ;

        // if size has been increased, insert at position 'pos'

        if ( ! err )
         {
             for ( i = length-1 ; i > pos ; i-- )
                data[i] = data[ i-1 ] ;

             data[pos] = x ;
         }
        else
         {
             lidia_error_handler ( "sort_vector" , "insert_at( T , lidia_size_t )::invalid position" ) ;
         }

        return err ;
    }


   int insert ( const T & x , int (*cmp) ( const T & a, const T & b ), lidia_size_t l = -1 , lidia_size_t r = -1 )
    {
        debug_handler ( "sort_vector", "insert ( T, cmp, lidia_size_t, lidia_size_t )" ) ;

        lidia_size_t pos, i, found ;
        int err ;

        // consider complete vector by default

        if ( l == -1 )  l = 0 ;
        if ( r == -1 )  r = length - 1 ;


        if ( length == 0 && l == 0 && r == -1 )  // empty vector and default-cut
         {
             err = set_size ( 1 ) ;

             if ( ! err ) data[0] = x ;

             return err ;
         }

        if ( l < 0 || l >= length || r < 0 || r >= length || l > r )
           lidia_error_handler ( "sort_vector" , "insert( T, cmp, lidia_size_t, lidia_size_t )::invalid cut" ) ;


        // examine, where to put element x

        found = bin_search ( x , pos , cmp , l , r  ) ;
        err   = set_size ( length + 1 ) ;


        // if size has been increased, insert at position 'pos'

        if ( ! err )
         {
           for ( i = length-1 ; i > pos ; i-- )
              data[i] = data[ i-1 ] ;

           data[pos] = x ;
         }
        else
         {
             lidia_error_handler ( "sort_vector" , "insert( T , cmp, lidia_size_t, lidia_size_t )::invalid position" ) ;
         }

        return err ;
    }


   int insert ( const T & x , char sort_direction = SORT_VECTOR_DEF , lidia_size_t l = -1 , lidia_size_t r = -1 )
    {
        debug_handler ( "sort_vector", "insert ( T, char, lidia_size_t, lidia_size_t )" ) ;

        lidia_size_t pos, i, found ;
        int err           ;

        // consider complete vector by default

        if ( l == -1 )  l = 0 ;
        if ( r == -1 )  r = length - 1 ;


        if ( length == 0 && l == 0 && r == -1 )  // empty vector and default-cut
         {
             err = set_size ( 1 ) ;

             if ( ! err ) data[0] = x ;

             return err ;
         }

        if ( l < 0 || l >= length || r < 0 || r >= length || l > r )
          lidia_error_handler ( "sort_vector" , "insert( T, char, lidia_size_t, lidia_size_t )::invalid cut" ) ;


        // examine, where to put element x

        found = bin_search ( x , pos , sort_direction , l , r ) ;
        err   = set_size ( length + 1 ) ;


        // if size has been increased, insert at position 'pos'

        if ( ! err )
         {
           for ( i = length-1 ; i > pos ; i-- )
              data[i] = data[ i-1 ] ;

           data[pos] = x ;
         }
        else
         {
             lidia_error_handler ( "sort_vector" , "insert( T, char, lidia_size_t,  lidia_size_t )::invalid position" ) ;
         }

        return err ;
    }

   inline int insert ( const T & x , lidia_size_t l , lidia_size_t r )
    {
        debug_handler ( "sort_vector", "insert ( T, lidia_size_t, lidia_size_t )" ) ;
        return (int) insert ( x , SORT_VECTOR_DEF , l , r ) ;
    }



   //  *****  deleting  elements from within *****

   int remove_from ( lidia_size_t pos, lidia_size_t l = 1 )
    {
        debug_handler ( "sort_vector", "remove_from ( lidia_size_t, lidia_size_t )" ) ;

        lidia_size_t i ;
	int err ;

        if ( ( l <= 0 ) || ( pos < 0 ) || ( pos + l > length ) )
         {
             lidia_error_handler ("sort_vector",
                            "remove_from(lidia_size_t,lidia_size_t)::invalid position or num. of elements" ) ;
             return 1 ;
         }

        for ( i = pos ; i < length-l ; i++ )
           data[i] = data[ i+l ] ;

        err = set_size ( length - l ) ;

        return err ;
    }


   int remove ( const T & x , int (*cmp) ( const T & a, const T & b ), lidia_size_t l = -1 , lidia_size_t r = -1 )
    {
        debug_handler ( "sort_vector", "remove ( T, cmp, lidia_size_t, lidia_size_t )" ) ;

        lidia_size_t pos, i, found ;
        int err           ;


        if ( l == -1 )  l = 0 ;
        if ( r == -1 )  r = length - 1 ;

        if ( l < 0 || l >= length || r < 0 || r >= length || l > r )
          lidia_error_handler ( "sort_vector",
			  "remove( T, cmp, lidia_size_t, lidia_size_t )::invalid cut or empty vector" ) ;


        // determine first occurence of x in the vector

        found = bin_search ( x , pos , cmp , l , r  ) ;


        // if x could be found, remove it from position 'pos'

        if ( found )
         {
            for ( i = pos ; i < length-1 ; i++ )
               data[i] = data[ i+1 ] ;

            err = set_size ( length - 1 ) ;
         }
        else
         {
            // x is not an element of 'this'
            err = 1 ;
         }

        return err ;
    }


   int remove ( const T & x , char sort_direction = SORT_VECTOR_DEF , lidia_size_t l = -1 , lidia_size_t r = -1 )
    {
        debug_handler ( "sort_vector", "remove ( T, char, lidia_size_t, lidia_size_t )" ) ;

        lidia_size_t pos, i, found ;
        int err           ;


        if ( l == -1 )  l = 0 ;
        if ( r == -1 )  r = length - 1 ;

        if ( l < 0 || l >= length || r < 0 || r >= length || l > r )
          lidia_error_handler ( "sort_vector",
			  "remove( T, char, lidia_size_t, lidia_size_t )::invalid cut or empty vector" ) ;


        // determine first occurence of x in the vector

        found = bin_search ( x , pos , sort_direction , l , r ) ;


        // if x could be found, remove it from position 'pos'

        if ( found )
         {
            for ( i = pos ; i < length-1 ; i++ )
               data[i] = data[ i+1 ] ;

            err = set_size ( length - 1 ) ;
         }
        else
         {
            // x is not an element of 'this'
            err = 1 ;
         }

        return err ;
    }


   inline int  remove ( const T & x, lidia_size_t l , lidia_size_t r )
    {
        debug_handler ( "sort_vector", "remove ( T, lidia_size_t, lidia_size_t )" ) ;
        return (int) remove ( x , SORT_VECTOR_DEF , l , r ) ;
    }

   int lex_compare ( sort_vector< T > & w ) const 
    {
       lidia_size_t i = 0 ;
       lidia_size_t minl = (length < w.length ) ? length : w.length ;

       while ( i < minl && comparator< T >::equal( data[i] , w.data[i] ) ) i++ ;

       if ( i < minl )
        {
            if ( comparator< T >::less_equal( data[i] , w.data[i] ) )
             {
               if ( comparator< T >::equal( data[i] , w.data[i] ) )
                return 0 ;
	       else
                return -1 ;
             }
            else
             {
                return  1 ;
	     }
        }  
       else
        {
           if ( i < w.length ) return -1 ;

           if ( i <   length ) 
              return  1 ;
           else
              return  0 ;
        }          
    }
 } ;


#endif

