//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : bi_lattice.c 
// Author      : Werner Backes (WB), Thorsten Lauer (TL) 
// Last change : WB/TL, Feb 10 1995, initial version
//

#include <LiDIA/bigint_lattice.h>
#include <LiDIA/bigfloat.h>

extern "C" void srand48(long);
extern "C" double drand48();
extern "C" long lrand48();
// 
// variables for temporary use
//

// 
// For storing temporary values
//
bigint bigint_lattice::tempvz;
bigint bigint_lattice::tempmz0;
bigint bigint_lattice::tempmz1;
bigint bigint_lattice::tempmz2;
bigint bigint_lattice::tempmz3;
bigint bigint_lattice::tempmz4;
bigint bigint_lattice::ergmz;

//
// vector - size for arithmetic operations
//
long bigint_lattice::vectsize;

//
// vectors for temporary use
//
long bigint_lattice::tempvectsize;
bigint* bigint_lattice::tempvect0;
bigint* bigint_lattice::tempvect1;

//
// number of class bigint_lattice
// (the last one frees tempvect)
//
long bigint_lattice::last;

// 
// Constructor / Destructor
//

//
// The presentation of the lattice is optimized for exchanging 
// columns. The problem is that it takes more time exchanging 
// rows. Changing of columns is only changing of pointers  
// The constructor creates a static vectors with dynamic size.
// The last destructor that is called frees this vector.
//

//
// Simple constructor with dimension (m, n) 
// rows and columns 
// 
bigint_lattice::bigint_lattice(long m, long n)
{
  debug_handler("bigint_lattice", "bigint_lattice(m, n)");
  long i;
  last++;
  rows=m;
  cols=n;
  cols_rank=n;
  value=new bigint*[cols];
  memory_handler(value, "bigint_lattice","bigint_lattice(m, n) :: "
                        "Not enough memory !");
  for (i=0;i<cols;i++)
    {
      value[i]=new bigint[rows];
      memory_handler(value[i], "bigint_lattice","bigint_lattice(m, n) :: "
                               "Not enough memory !");
    }
  if (tempvectsize < rows+1)
    {
      if (tempvectsize != 0)
        {
          delete[] tempvect0;
          delete[] tempvect1;
        }
      tempvectsize=rows+1;
      tempvect0=new bigint[tempvectsize];
      memory_handler(tempvect0, "bigint_lattice","bigint_lattice(m, n) :: "
                                "Not enough memory !");
      tempvect1=new bigint[tempvectsize];
      memory_handler(tempvect1, "bigint_lattice","bigint_lattice(m, n) :: "
                                "Not enough memory !");
    }
}

//
// Constructor with dimension (m, n) 
// rows and columns and values bigint **abi
// 
bigint_lattice::bigint_lattice(long m, long n, bigint **abi)
{
  debug_handler("bigint_lattice", "bigint_lattice(m, n, abi)");
  long i;
  last++;
  rows=m;
  cols=n;
  cols_rank=n;
  value=new bigint*[cols];
  memory_handler(value, "bigint_lattice","bigint_lattice(m, n) :: "
                        "Not enough memory !");
  for (i=0;i<cols;i++)
    {
      value[i]=new bigint[rows];
      memory_handler(value[i], "bigint_lattice","bigint_lattice(m, n) :: "
                               "Not enough memory !");
      for (register long j=0;j<rows;j++)
        value[i][j].assign(abi[j][i]);
    }
  if (tempvectsize < rows+1)
    {
      if (tempvectsize != 0)
        {
          delete[] tempvect0;
          delete[] tempvect1;
        }
      tempvectsize=rows+1;
      tempvect0=new bigint[tempvectsize];
      memory_handler(tempvect0, "bigint_lattice","bigint_lattice(m, n) :: "
                                "Not enough memory !");
      tempvect1=new bigint[tempvectsize];
      memory_handler(tempvect1, "bigint_lattice","bigint_lattice(m, n) :: "
                                "Not enough memory !");
    }
}

//
// Copy - constructor 
// creating a copy of bigint_lattice A
//
bigint_lattice::bigint_lattice(const bigint_lattice& A)
{
  debug_handler("bigint_lattice", "bigint_lattice(A)");
  long i, j;
  last++;
  rows=A.rows;
  cols=A.cols;
  cols_rank=A.cols_rank;
  value=new bigint*[cols];
  memory_handler(value, "bigint_lattice","bigint_lattice(A) :: "
                        "Not enough memory !");
   for (i=0;i<cols;i++)
    {
      value[i]=new bigint[rows];
      memory_handler(value[i], "bigint_lattice","bigint_lattice(A) :: "
                               "Not enough memory !");
      for (j=0;j<rows;j++)
         value[i][j].assign(A.value[i][j]);
    } 
  if (tempvectsize < rows+1)
    {
      if (tempvectsize != 0)
        {
          delete[] tempvect0;
          delete[] tempvect1;
        }
      tempvectsize=rows+1;
      tempvect0=new bigint[tempvectsize];
      memory_handler(tempvect0, "bigint_lattice","bigint_lattice(A) :: "
                                "Not enough memory !");
      tempvect1=new bigint[tempvectsize];
      memory_handler(tempvect1, "bigint_lattice","bigint_lattice(A) :: "
                                "Not enough memory !");
    }
}

//
// Special copy - constructor
// make a transformated copy of bigint_matrix A
//
bigint_lattice::bigint_lattice(const bigint_matrix& A)
{
  debug_handler("bigint_lattice", "bigint_lattice(A)");
  long i, j;
  last++;
  rows=A.get_no_of_rows();
  cols=A.get_no_of_columns();
  cols_rank=cols;
  value=new bigint*[cols];
  memory_handler(value, "bigint_lattice","bigint_lattice(A) :: "
                        "Not enough memory !");
   for (i=0;i<cols;i++)
    {
      value[i]=new bigint[rows];
      memory_handler(value[i], "bigint_lattice","bigint_lattice(A) :: "
                               "Not enough memory !");
      for (j=0;j<rows;j++)
         value[i][j].assign(A.member(j,i));
    } 
  if (tempvectsize < rows+1)
    {
      if (tempvectsize != 0)
        {
          delete[] tempvect0;
          delete[] tempvect1;
        }
      tempvectsize=rows+1;
      tempvect0=new bigint[tempvectsize];
      memory_handler(tempvect0, "bigint_lattice","bigint_lattice(A) :: "
                                "Not enough memory !");
      tempvect1=new bigint[tempvectsize];
      memory_handler(tempvect1, "bigint_lattice","bigint_lattice(A) :: "
                                "Not enough memory !");
    }
}
    
//
// The destructor that frees allocated storage
//
bigint_lattice::~bigint_lattice()
{
  debug_handler("bigint_lattice", "~bigint_lattice()");
  while (cols--)
    delete[] value[cols];
  delete[] value;
  last--;
  if (last == 0)
   {
     tempvectsize=0;
     delete[] tempvect0;
     delete[] tempvect1;
   }
}

//
// Input / Output
//
// The lattice is read row by row, but the presentation
// of the lattice is column by column
//        
istream& operator >> (istream& in, bigint_lattice& A)
{
  debug_handler("bigint_lattice", "operator >> (in, A)");
  long i, j;

  for (i=0;i<A.rows;i++)
    for (j=0;j<A.cols;j++)
      in >> A.value[j][i];
  return (in);
}

//
// The lattice is written row by row to ostream 
//
ostream& operator << (ostream& out, const bigint_lattice& A)
{
  debug_handler("bigint_lattice", "operator << (out, A)");
  long i, j;
  
  for (i=0;i<A.rows;i++)
    {   
      out << "(";
      for (j=0;j<A.cols-1;j++)
        out << A.value[j][i] << ", ";
      out << A.value[A.cols-1][i] << ")\n";
    }
  return (out);
}

//
// Assignment of two bigint_lattice
//
void bigint_lattice::assign(const bigint_lattice& A)
{
  debug_handler("bigint_lattice", "assign(A)");
  long i, j;
  
  if ((rows == A.rows) && (cols == A.cols))
    {
      cols_rank=A.cols_rank;
      for (i=0;i<cols;i++)
        for (j=0;j<rows;j++)
          value[i][j].assign(A.value[i][j]);
      return;
    }
  lidia_error_handler("bigint_lattice", "assign(A) :: "
                "tryed to assign two different sized lattice");
}

//
// Assignment of bigint_matrix to bigint_lattice
//
void bigint_lattice::assign(const bigint_matrix& A)
{
  debug_handler("bigint_lattice", "assign(A)");
  long i, j;
  
  if ((rows == A.get_no_of_rows()) && (cols == A.get_no_of_columns()))
    {
      cols_rank=cols;
      for (i=0;i<cols;i++)
        for (j=0;j<rows;j++)
          value[i][j].assign(A.member(j,i));
      return;
    }
  lidia_error_handler("bigint_lattice", "assign(A) :: "
                "tryed to assign two different sized lattice");
}

//
// Fills lattice with zero
//
void bigint_lattice::assign_zero()
{
  debug_handler("bigint_lattice", "assign_zero()");
  long i, j;
  
  for (i=0;i<cols;i++)
    for (j=0;j<rows;j++)
      value[i][j].assign_zero();
}

//
// Overloaded operator, returns bigint_lattice
//
bigint_lattice bigint_lattice::operator = (const bigint_lattice& A)
{
  debug_handler("bigint_lattice", "operator = (A)");
  assign(A);
  return (*this);
}

//
// Overloaded operator for bigint_matrix, returns bigint_lattice
//
bigint_lattice bigint_lattice::operator = (const bigint_matrix& A)
{
  debug_handler("bigint_lattice", "operator = (A)");
  assign(A);
  return (*this);
}

#ifndef __GNUC__

//
// Vector Operations
//

//
// This vector operations need the static variable vectsize, 
// where the size of the vectors is stored 
// (operating size can be different from allocated size)
//

void assign(bigint* a, bigint* b)
{
  debug_handler("bigint_lattice", "friend void assign(a, b)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    a[i].assign(b[i]);
}

void assign_zero(bigint* a)
{
  debug_handler("bigint_lattice", "friend void assign_zero(a)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    a[i].assign_zero();
}

void add(bigint* c, bigint* a, bigint* b)
{
  debug_handler("bigint_lattice", "friend void add(c, a, b)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    add(c[i],a[i],b[i]);
}

void subtract(bigint* c, bigint* a, bigint* b)
{
  debug_handler("bigint_lattice", "friend void subtract(c, a, b)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    subtract(c[i],a[i],b[i]);
}

void scalmul(bigint* b, const bigint& d, bigint* a)
{
  debug_handler("bigint_lattice", "friend scalmul(b, d, a)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    multiply(b[i], a[i], d);
}

void scalprod(bigint& res, bigint* a, bigint* b)
{
  debug_handler("bigint_lattice", "friend scalprod(res, a, b)");
  long i;
  res.assign_zero();
  for (i=bigint_lattice::vectsize-1;i>=0;i--) 
    {    
      multiply(bigint_lattice::tempvz, a[i], b[i]);
      add(res, res, bigint_lattice::tempvz);
    }
}

void scalquad(bigint& res, bigint* a)
{
  debug_handler("bigint_lattice", "friend scalquad(res, a)");
  long i;
  res.assign_zero();
  for (i=bigint_lattice::vectsize-1;i>=0;i--) 
    {    
      square(bigint_lattice::tempvz, a[i]);
      add(res, res, bigint_lattice::tempvz);
    }
}

void l2_norm(bigint& norm, bigint* v)
{
  debug_handler("bigint_lattice", "l2_norm(norm, v)");
  long i;
  norm.assign_zero();		// Initialisation
  for (i=bigint_lattice::vectsize-1;i>=0;i--) 
    {
      square(bigint_lattice::tempvz, v[i]);
      add(norm, norm, bigint_lattice::tempvz);
    }
//  sqrt(norm, norm);
}

void l1_norm(bigint& norm, bigint* v)
{
  debug_handler("bigint_lattice", "l1_norm(norm, v)");
  long i;
  norm.assign_zero();	 // Initialisation
  for (i=bigint_lattice::vectsize-1;i>=0;i--) 
    {
      bigint_lattice::tempvz.assign(abs(v[i]));
      add(norm, norm, bigint_lattice::tempvz);
    }
}

void bfl_assign(bigfloat* a, bigfloat* b)
{
  debug_handler("bigint_lattice", "friend bfl_assign(a, b)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    a[i].assign(b[i]);
}

void bfl_assign_zero(bigfloat* a)
{
  debug_handler("bigint_lattice", "friend bfl_assign_zero(a)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    a[i].assign_zero();
}

void bfl_add(bigfloat* c, bigfloat* a, bigfloat* b)
{
  debug_handler("bigint_lattice", "friend bfl_add(c, a, b)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    add(c[i],a[i],b[i]);
}

void bfl_subtract(bigfloat* c, bigfloat* a, bigfloat* b)
{
  debug_handler("bigint_lattice", "bfl_subtract(c, a, b)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    subtract(c[i],a[i],b[i]);
}

void bfl_scalmul(bigfloat* b, const bigfloat& d, bigfloat* a)
{
  debug_handler("bigint_lattice", "friend bfl_scalmul(b, d, a)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    multiply(b[i], a[i], d);
}

void bfl_scalprod(bigfloat& res, bigfloat* a, bigfloat* b)
{
  debug_handler("bigint_lattice", "friend bfl_scalprod(res, a, b)");
  long i;
  bigfloat temp;
  res.assign_zero();
  for (i=bigint_lattice::vectsize-1;i>=0;i--) 
    {    
      multiply(temp, a[i], b[i]);
      add(res, res, temp);
    }
}

void bfl_scalquad(bigfloat& res, bigfloat* a)
{
  debug_handler("bigint_lattice", "friend bfl_scalquad(res, a)");
  long i;
  bigfloat temp;
  res.assign_zero();
  for (i=bigint_lattice::vectsize-1;i>=0;i--) 
    {    
      square(temp, a[i]);
      add(res, res, temp);
    }
}

void bfl_l2_norm(bigfloat& norm, bigfloat* v)
{
  debug_handler("bigint_lattice", "bfl_l2_norm(norm, v)");
  long i;
  bigfloat temp;
  norm.assign_zero();		// Initialisation
  for (i=bigint_lattice::vectsize-1;i>=0;i--) 
    {
      square(temp, v[i]);
      add(norm, norm, temp);
    }
  sqrt(norm, norm);
}

void bfl_l1_norm(bigfloat& norm, bigfloat* v)
{
  debug_handler("bigint_lattice", "friend bfl_l1_norm(norm, v)");
  long i;
  bigfloat temp;
  norm.assign_zero();	 // Initialisation
  for (i=bigint_lattice::vectsize-1;i>=0;i--) 
    {
      temp.assign(abs(v[i]));
      add(norm, norm, temp);
    }
}

void dbl_assign(double* a, double* b)
{
  debug_handler("bigint_lattice","friend void dbl_assign(a, b)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    a[i]=b[i];
}

void dbl_assign_zero(double* a)
{
  debug_handler("bigint_lattice","friend void dbl_assign_zero(a)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    a[i]=0;
}

void dbl_add(double* c, double* a, double* b)
{
  debug_handler("bigint_lattice","friend void dbl_add(c, a, b)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    c[i]=a[i]+b[i];
}

void dbl_subtract(double* c, double* a, double* b)
{
  debug_handler("bigint_lattice","friend void dbl_subtract(c, a, b)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    c[i]=a[i]-b[i];
}

void dbl_scalmul(double* b, const double& d, double* a)
{
  debug_handler("bigint_lattice","friend dbl_scalmul(b, d, a)");
  long i;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    b[i]=a[i]*d;
}


void dbl_scalprod(double& res, double* a, double* b)
{
  debug_handler("bigint_lattice","friend dbl_scalprod(res, a, b)");
  long i;
  res=0;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    res+=a[i]*b[i];
}

void dbl_scalquad(double& res, double* a)
{
  debug_handler("bigint_lattice","friend dbl_scalquad(res, a)");
  long i;
  res=0;
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    res+=a[i]*a[i];
}

void dbl_l2_norm(double& norm, double* v)
{
  debug_handler("bigint_lattice","dbl_l2_norm(norm, v)");
  long i;
  norm=0;		// Initialisation
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    norm+=v[i]*v[i];
  norm=sqrt(norm);
}

void dbl_l1_norm(double& norm, double* v)
{
  debug_handler("bigint_lattice","dbl_l1_norm(norm, v)");
  long i;
  norm=0;	 // Initialisation
  for (i=bigint_lattice::vectsize-1;i>=0;i--)
    norm+=fabs(v[i]);
}

#endif

//
// Algorithm`s
//

//
// Benne de Weger version of lll 
// result : lll - reduced lattice for parameter a / b
//
void bigint_lattice::lll(unsigned long& red_count, long a, long b, timer& timing)
{
  debug_handler("bigint_lattice", "lll(red_count, a, b, timing)");
  bigint_lattice lz(rows, cols), 
                 help_matrix(rows, cols);
  long i, j, l, cx;
  
  red_count=0;
  vectsize=rows;

  timing.start_timer();
  ::assign_zero(tempvect0);

  tempvect0[0].assign_one();

  for (i=0;i<cols;i++) 
    {
      ::assign(help_matrix.value[i], value[i]);
      for (j=0;j<i;j++) 
        {
          scalprod(lz.value[j][i], value[i], help_matrix.value[j]);
          for (cx=0;cx<rows;cx++)
            {
              multiply(help_matrix.value[i][cx],tempvect0[j+1],help_matrix.value[i][cx]);
              multiply(tempvect1[cx],lz.value[j][i],help_matrix.value[j][cx]);
              subtract(help_matrix.value[i][cx], help_matrix.value[i][cx], tempvect1[cx]);
            }
//          scalmul(help_matrix.value[i], tempvect0[j+1], help_matrix.value[i]);
//          scalmul(tempvect1, lz.value[j][i],  help_matrix.value[j]);
//          subtract(help_matrix.value[i], help_matrix.value[i], tempvect1);
          for (l=0;l<help_matrix.rows;l++)
            divide(help_matrix.value[i][l], help_matrix.value[i][l], tempvect0[j]); 
        }
      scalquad(tempvect0[i+1], help_matrix.value[i]);
      divide(tempvect0[i+1], tempvect0[i+1], tempvect0[i]);
    }

  long k=1;
  while(k != cols) 
    {
      l=k-1;
      tempmz0.assign(abs(lz.value[l][k]));
      tempmz0.multiply_by_2();
      if(tempmz0.compare(tempvect0[l+1])>0) 
        {
          red_count++;
//          si = lz.value[l][k].sign() * tempvect0[l+1].sign();
          tempmz1.assign(abs(tempvect0[l+1]));
          add(ergmz, tempmz0, tempmz1);

//          divide(tempmz3, ergmz, tempmz1);
//          tempmz3.divide_by_2();
          divide(ergmz, ergmz, tempmz1);
          ergmz.divide_by_2();
          
          if (lz.value[l][k].sign() != tempvect0[l+1].sign())
            ergmz.negate();
//          multiply(ergmz, tempmz3, si);
          for (cx=0;cx<rows;cx++)
            {
              multiply(tempvect1[cx],ergmz,value[l][cx]);
              subtract(value[k][cx],value[k][cx],tempvect1[cx]);
            }
//          scalmul(tempvect1, ergmz, value[l]);
//          subtract(value[k], value[k], tempvect1);

          for (j=0;j<l;j++) 
            {
              multiply(tempmz3, ergmz, lz.value[j][l]);
	      subtract(lz.value[j][k], lz.value[j][k], tempmz3);
            }
          multiply(tempmz3, ergmz, tempvect0[l+1]);
          subtract(lz.value[l][k], lz.value[l][k], tempmz3);
        }

      square(tempmz0, tempvect0[k]);
      multiply(tempmz1, tempmz0, a);
      square(tempmz0, lz.value[l][k]);
      multiply(tempmz2, tempmz0, b);
      subtract(tempmz0, tempmz1, tempmz2);
      multiply(tempmz1, tempvect0[k-1], tempvect0[k+1]);
      multiply(tempmz2, tempmz1, b);

      if(tempmz2.compare(tempmz0) < 0) 
        {
          swap(value[k-1], value[k]);
          for (j=0;j<=k-2;j++)
	    swap(lz.value[j][k-1], lz.value[j][k]);
          for (i=k+1;i<cols;i++) 
            {
              multiply(ergmz, lz.value[k-1][i], lz.value[k-1][k]);
              multiply(tempmz4, lz.value[k][i], tempvect0[k-1]);
              add(tempmz3, ergmz, tempmz4);
              divide(ergmz, tempmz3, tempvect0[k]);
              multiply(tempmz4, lz.value[k-1][i], tempvect0[k+1]);
              multiply(tempmz3, lz.value[k][i], lz.value[k-1][k]);
              subtract(tempmz1, tempmz4, tempmz3);
              divide(tempmz4, tempmz1, tempvect0[k]);
              lz.value[k-1][i].assign(ergmz);
              lz.value[k][i].assign(tempmz4);
            }
          square(tempmz3, lz.value[k-1][k]);
          multiply(ergmz, tempvect0[k-1], tempvect0[k+1]);
          add(tempmz4, ergmz, tempmz3);
          divide(tempvect0[k], tempmz4, tempvect0[k]);
          if(k > 1)
            k--; 
        }
      else 
        { 
          for (l=k-2;l>=0;l--) 
            {
              red_count++;
	      tempmz3.assign(abs(lz.value[l][k]));
              tempmz3.multiply_by_2();	
	      if(tempmz3.compare(tempvect0[l+1])>0) 
                {
//                  si = (lz.value[l][k]).sign() * (tempvect0[l+1]).sign();
// Was passiert, wenn eines der beiden Vorzeichen Null ist ?
// Kann dies auftreten ?
//
                  ergmz.assign(abs(tempvect0[l+1]));
                  add(tempmz1, tempmz3, ergmz);
                  divide(tempmz4, tempmz1, ergmz);
		  tempmz4.divide_by_2();
                  if ((lz.value[l][k]).sign() != (tempvect0[l+1]).sign())
                    tempmz4.negate();

//	            multiply(tempmz4, ergmz, si);
//                  scalmul(tempvect1, tempmz4, value[l]);
//                  subtract(value[k], value[k], tempvect1);

                  for (cx=0;cx<rows;cx++)
                    {
                       multiply(tempvect1[cx],tempmz4,value[l][cx]);
                       subtract(value[k][cx],value[k][cx],tempvect1[cx]);
                    }
                  for (j=0;j<l;j++) 
                    {
                      multiply(tempmz0, tempmz4, lz.value[j][l]);
	              subtract(lz.value[j][k], lz.value[j][k], tempmz0);
                    }
                  multiply(tempmz0, tempmz4, tempvect0[l+1]);
                  subtract(lz.value[l][k], lz.value[l][k], tempmz0);
	        }
            }
          k++;
        }
   }
  timing.stop_timer();
}

//
// Benne de Weger version of lll on the identic lattice
// result : transformation lattice
//
void bigint_lattice::lll_trans(bigint_lattice &T, unsigned long& red_count, long a, long b, timer& timing)
{
  debug_handler("bigint_lattice", "lll_trans(red_count, a, b, timing)");
  bigint_lattice lz(rows, cols), 
                 help_matrix(rows, cols);
  long i, j, l, cx;

  red_count=0;
  vectsize=rows;

  timing.start_timer();
  ::assign_zero(tempvect0);

  tempvect0[0].assign_one();

  for (i=0;i<cols;i++) 
    {
      ::assign(help_matrix.value[i], value[i]);
      for (j=0;j<i;j++) 
        {
          scalprod(lz.value[j][i], value[i], help_matrix.value[j]);
          for (cx=0;cx<rows;cx++)
            {
              multiply(help_matrix.value[i][cx],tempvect0[j+1],help_matrix.value[i][cx]);
              multiply(tempvect1[cx],lz.value[j][i],help_matrix.value[j][cx]);
              subtract(help_matrix.value[i][cx], help_matrix.value[i][cx], tempvect1[cx]);
            }
//          scalmul(help_matrix.value[i], tempvect0[j+1], help_matrix.value[i]);
//          scalmul(tempvect1, lz.value[j][i],  help_matrix.value[j]);
//          subtract(help_matrix.value[i], help_matrix.value[i], tempvect1);
          for (l=0;l<help_matrix.rows;l++)
            divide(help_matrix.value[i][l], help_matrix.value[i][l], tempvect0[j]); 
        }
      scalquad(tempvect0[i+1], help_matrix.value[i]);
      divide(tempvect0[i+1], tempvect0[i+1], tempvect0[i]);
    }

  long k=1;
  while(k != cols) 
    {
      l=k-1;
      tempmz0.assign(abs(lz.value[l][k]));
      tempmz0.multiply_by_2();
      if(tempmz0.compare(tempvect0[l+1])>0) 
        {
          red_count++;
//          si = lz.value[l][k].sign() * tempvect0[l+1].sign();
          tempmz1.assign(abs(tempvect0[l+1]));
          add(ergmz, tempmz0, tempmz1);

//          divide(tempmz3, ergmz, tempmz1);
//          tempmz3.divide_by_2();
          divide(ergmz, ergmz, tempmz1);
          ergmz.divide_by_2();
          
          if (lz.value[l][k].sign() != tempvect0[l+1].sign())
            ergmz.negate();
//          multiply(ergmz, tempmz3, si);
          for (cx=0;cx<cols;cx++)
            {
              multiply(tempvect1[cx],ergmz,T.value[l][cx]);
              subtract(T.value[k][cx],T.value[k][cx],tempvect1[cx]);
            }
//          scalmul(tempvect1, ergmz, T.value[l]);
//          subtract(T.value[k], T.value[k], tempvect1);

          for (j=0;j<l;j++) 
            {
              multiply(tempmz3, ergmz, lz.value[j][l]);
	      subtract(lz.value[j][k], lz.value[j][k], tempmz3);
            }
          multiply(tempmz3, ergmz, tempvect0[l+1]);
          subtract(lz.value[l][k], lz.value[l][k], tempmz3);
        }

      square(tempmz0, tempvect0[k]);
      multiply(tempmz1, tempmz0, a);
      square(tempmz0, lz.value[l][k]);
      multiply(tempmz2, tempmz0, b);
      subtract(tempmz0, tempmz1, tempmz2);
      multiply(tempmz1, tempvect0[k-1], tempvect0[k+1]);
      multiply(tempmz2, tempmz1, b);

      if(tempmz2.compare(tempmz0) < 0) 
        {
          swap(T.value[k-1], T.value[k]);
          for (j=0;j<=k-2;j++)
	    swap(lz.value[j][k-1], lz.value[j][k]);
          for (i=k+1;i<cols;i++) 
            {
              multiply(ergmz, lz.value[k-1][i], lz.value[k-1][k]);
              multiply(tempmz4, lz.value[k][i], tempvect0[k-1]);
              add(tempmz3, ergmz, tempmz4);
              divide(ergmz, tempmz3, tempvect0[k]);
              multiply(tempmz4, lz.value[k-1][i], tempvect0[k+1]);
              multiply(tempmz3, lz.value[k][i], lz.value[k-1][k]);
              subtract(tempmz1, tempmz4, tempmz3);
              divide(tempmz4, tempmz1, tempvect0[k]);
              lz.value[k-1][i].assign(ergmz);
              lz.value[k][i].assign(tempmz4);
            }
          square(tempmz3, lz.value[k-1][k]);
          multiply(ergmz, tempvect0[k-1], tempvect0[k+1]);
          add(tempmz4, ergmz, tempmz3);
          divide(tempvect0[k], tempmz4, tempvect0[k]);
          if(k > 1)
            k--; 
        }
      else 
        { 
          for (l=k-2;l>=0;l--) 
            {
              red_count++;
	      tempmz3.assign(abs(lz.value[l][k]));
              tempmz3.multiply_by_2();	
	      if(tempmz3.compare(tempvect0[l+1])>0) 
                {
//                  si = (lz.value[l][k]).sign() * (tempvect0[l+1]).sign();
// Was passiert, wenn eines der beiden Vorzeichen Null ist ?
// Kann dies auftreten ?
//
                  ergmz.assign(abs(tempvect0[l+1]));
                  add(tempmz1, tempmz3, ergmz);
                  divide(tempmz4, tempmz1, ergmz);
		  tempmz4.divide_by_2();
                  if ((lz.value[l][k]).sign() != (tempvect0[l+1]).sign())
                    tempmz4.negate();

//	            multiply(tempmz4, ergmz, si);
//                  scalmul(tempvect1, tempmz4, T.value[l]);
//                  subtract(T.value[k], T.value[k], tempvect1);

                  for (cx=0;cx<cols;cx++)
                    {
                       multiply(tempvect1[cx],tempmz4,T.value[l][cx]);
                       subtract(T.value[k][cx],T.value[k][cx],tempvect1[cx]);
                    }
                  for (j=0;j<l;j++) 
                    {
                      multiply(tempmz0, tempmz4, lz.value[j][l]);
	              subtract(lz.value[j][k], lz.value[j][k], tempmz0);
                    }
                  multiply(tempmz0, tempmz4, tempvect0[l+1]);
                  subtract(lz.value[l][k], lz.value[l][k], tempmz0);
	        }
            }
          k++;
        }
   }
  timing.stop_timer();
}

//
// Buchmann - Kessler version for generating systems
// 
void bigint_lattice::lin_gen_system(bigint_lattice& T, long a, long b, 
                                    unsigned long& rc, timer& ti, long& rk)

{
  debug_handler("bigint_lattice", "lin_gen_system(T, a, b, rc, ti, rk) [1]");

  bigint_lattice Atilde(rows+cols,cols),	// The approximate lattice
                 Ahead(rows, cols);	// The weighted lll_matrix_bigfloat

  bigint *help=new bigint[rows];
  bigint *rel=new bigint[cols];
  bigint *temp=new bigint[rows];

  bigfloat vor, rechts;
  bigfloat zweipotq, alpha;
  bigfloat norm1, norm2;

  bigint bi_norm1, bi_norm2;
  bigint zwpq;

  long i,j,n2 = cols;
  long prec;
  long bit_prec;
  
  prec=compute_precision();
  bit_prec=(long )((double )prec*(log(10)/log(2)));
  bit_prec=((bit_prec/200)+1)*52;
  
  zweipotq.precision(prec);
  alpha_compute(alpha);
  zwei_pot_q_compute(zweipotq, n2, alpha);
  zweipotq.bigintify(zwpq);
  for (i = 0; i < cols; ++i)
    for (j = 0; j < rows; ++j)
      multiply(Ahead.value[i][j],zwpq,value[i][j]);
  for (i=0;i<cols;++i)
    {
      Atilde.value[i][i].assign_one();  // 1, wenn i = j, 0 sonst
      for (j=cols;j<(rows+cols);++j)
        Atilde.value[i][j].assign(Ahead.value[i][j-cols]);
    }
    
  debug_handler("bigint_lattice", "lin_gen_system(T, a, b, rc, ti, rk)  [2]");
//  Atilde.lll(rc, a, b, ti);
  Atilde.lll_bfl((double) a/(double) b, rc, ti, bit_prec);
  debug_handler("bigint_lattice", "lin_gen_system(T, a, b, rc, ti, rk)  [3]");  

  long l = 0;
  do
    {
      vectsize=rows;
      ::assign_zero(help);	// Initializes help with the zero - vector
       for (j = 0; j < cols; ++j)
        {
          rel[j].assign(Atilde.value[l][j]);
          scalmul(temp,rel[j], Ahead.value[j]);
          add(help,help,temp);
        }
      l2_norm(bi_norm2,help);
      norm2.assign(bi_norm2);
      vectsize=cols;
      l1_norm(bi_norm1,rel);
      norm1.assign(bi_norm1);
      sqrt(norm1, norm1);
      ++l;
      ::divide(vor, n2, rows);
      vor.multiply_by_2();
      sqrt(vor, vor);
      multiply(vor, vor, norm1);
      ::divide(rechts, (bigfloat) sqrt(n2), (bigfloat )2.0);
      multiply(rechts, rechts, norm1);
    } 
  while ((zweipotq.compare(vor)>0) && (norm2.compare(rechts)<=0) && (l < cols));
  debug_handler("bigint_lattice", "lin_gen_system(T, a, b, rc, ti, rk)  [4]");  
  if (l >= cols)
    warning_handler("bigint_lattice","Tr_lin_gen_system(T, a, b, rc, ti, rk)  :: "
                    "lattice of dimension 1");

  rk=cols-l+1;            // cols is the dimension of the lattice
  T.cols_rank=rk;
  for (i=0;i<cols-T.cols_rank;i++)
    for (j=0;j<cols;j++)
      T.value[i][j].assign_zero();    
  for (i=cols-T.cols_rank;i<cols;++i)
    for (j=0;j<cols;++j)
      T.value[i][j].assign(Atilde.value[i][j]);
//
// Free allocated storage
//
  delete[] rel;
  delete[] help;
  delete[] temp;
} 

//
// Schnorr - Euchner version of lll optimized for bigints usign doubles
// result : lll - reduced lattice for parameter y
//
void bigint_lattice::lll_bfl(double y, unsigned long& red_count, timer& ti, long bit_prec)
{
  debug_handler("bigint_lattice", "lll_bfl(y, red_count) [1]");
  bool Fc;
  bool korr=false;
  long i,j;
  long k = 1;
  long prec=(long )((double )bit_prec*log(2)/log(10));
  bigfloat bound=exp(log(2)*bit_prec/2);
  bigfloat invbound=1/bound;
  bigfloat tempbfl0;
  bigfloat tempbfl1;
  bigfloat tempbfl2;
  bigfloat **my;
  bigfloat **lattice;
  bigfloat *vect;
  bigfloat *vect1;
  bigfloat *vect_sq;
  bigfloat conv;
  bigfloat halb(0.5);
  
  bound.precision(prec);
  vectsize=cols;
  
//
// Allocating Memory for gram schmidt matrix
//
  my=new bigfloat*[cols];
  memory_handler(my,"bigint_lattice","lll_bfl(y, red_count) :: "
                    "not enough memory !");
  lattice=new bigfloat*[cols];
  memory_handler(lattice,"bigint_lattice","lll_bfl(y, red_count) :: "
                         "not enough memory !");
  for (i=0;i<cols;i++)
    {
      my[i]=new bigfloat[cols+rows];
      memory_handler(my[i],"bigint_lattice","lll_bfl(y, red_count) :: "
                           "not enough memory !");
      lattice[i]=&my[i][cols];
    }  

//
// Allocating Memory for vector of squares
//

  vect=new bigfloat[2*cols+rows];
  memory_handler(vect,"bigint_lattice","lll_bfl(y, red_count) :: "
                      "not enough memory !");
  vect_sq=&vect[cols];
  vect1=&vect[2*cols];


  ti.start_timer();
  k=1;

//
// Start
//
// Copy into floating point
//
  for (i=0;i<cols;i++)
    for (j=0;j<rows;j++)
      lattice[i][j].assign(value[i][j]);

  red_count=0;
  vectsize=rows;
  Fc=false;
  bfl_assign_zero(vect);
  bfl_scalprod(vect[0],lattice[0], lattice[0]);

  while (k < cols)
    {
      Fc = true;
      while (Fc)
      {	
//
// begin of orthogonalization
//
        debug_handler("bigint_lattice", "lll_bfl(y, red_count) [2]");
        bfl_scalprod(vect[k], lattice[k], lattice[k]);
        sqrt(vect_sq[k], vect[k]);
        for (j = 0; j < k; ++j)
          {
            bfl_scalprod(tempbfl0, lattice[k], lattice[j]);
            multiply(tempbfl1, invbound, vect_sq[k]);
            multiply(tempbfl1, tempbfl1, vect_sq[j]);
//
// Check Precision, correct if nessesary
//
            if (abs(tempbfl0).compare(tempbfl1) < 0)
              {
                scalprod(tempmz0, value[k], value[j]);
                my[j][k].assign(tempmz0);
              }
            else
              my[j][k].assign(tempbfl0);
              
            for (i = 0; i < j; ++i) 
              {
                multiply(tempbfl1, my[i][j], my[i][k]);
                multiply(tempbfl1, tempbfl1, vect[i]);
                ::subtract(my[j][k], my[j][k], tempbfl1);
//                my[j][k]-=my[i][j]*my[i][k]*vect[i];
              }
                    
            tempbfl0.assign(my[j][k]);
            ::divide(my[j][k], my[j][k], vect[j]);
            multiply(tempbfl1, my[j][k], tempbfl0);
            ::subtract(vect[k], vect[k], tempbfl1);
//            vect[k]-=my[j][k]*tempbfl0;
          }
//
// end of orthogonalization
//
        Fc = false;
//
// begin of reduction
//
        debug_handler("bigint_lattice", "lll_bfl(y, red_count) [3]");
        for (j = k-1; j >= 0; j--)
          if (abs(my[j][k]).compare(halb) > 0)
            {
              ++red_count;
              korr=true;
              round (tempbfl0, my[j][k]);
              if (abs(tempbfl0).compare(bound) > 0)
                Fc=true;

              tempbfl0.bigintify(tempmz0);
              scalmul(tempvect1, tempmz0, value[j]);
              subtract(value[k], value[k], tempvect1);
                
	      for (i = 0; i < j; ++i) 
                {
                  multiply(tempbfl1, tempbfl0, my[i][j]);
                  subtract(my[i][k], my[i][k], tempbfl1);
//                  my[i][k]-=tempbfl0*my[i][j];
                }
              subtract(my[j][k], my[j][k], tempbfl0);
//              my[j][k]-=tempbfl0;
              

            }

//
// correction (not in paper)
//

          if (korr == true)
            {
              for  (j=0;j<rows;j++)
                lattice[k][j].assign(value[k][j]);
              bfl_scalquad(vect_sq[k], lattice[k]);
              sqrt(vect_sq[k], vect_sq[k]);
              korr=false;
            }

  

//
// end of reduction
//
	if (Fc)
	  if (k > 1) 
	    --k;
      }

    debug_handler("bigint_lattice", "lll_bfl(y, red_count) [4]");
    tempbfl0.assign(y);
    ::multiply(tempbfl0, tempbfl0, vect[k-1]);
    square(tempbfl1, my[k-1][k]);
    ::multiply(tempbfl1, tempbfl1, vect[k-1]);
    add(tempbfl1, tempbfl1, vect[k]);
    
//    if (y*vect[k-1] > my[k-1][k]*my[k-1][k]*vect[k-1]+vect[k])       
    if (tempbfl0.compare(tempbfl1) > 0)       
      {
	swap(value[k], value[k-1]);
        bfl_swap(lattice[k], lattice[k-1]);
        tempbfl2.assign(vect_sq[k]);
        vect_sq[k].assign(vect_sq[k-1]);
        vect_sq[k-1].assign(tempbfl2);
        
	if (k > 1)
	  --k;
        if(k == 1) 
          {
            bfl_scalprod(vect[0], lattice[0], lattice[0]);
            sqrt(vect_sq[0], vect[0]);
          }
      }
    else
      ++k;
  }
  ti.stop_timer();
  for (i=0;i<cols;i++)
    delete[] my[i];
  delete[] my;
  delete[] lattice;
  delete[] vect;
}   

//
// Schnorr - Euchner version of lll optimized for bigints usign doubles
// result : lll - reduced lattice for parameter y
//
void bigint_lattice::lll_dbl(double y, unsigned long& red_count, timer& ti)
{
  debug_handler("bigint_lattice", "lll_dbl(y, red_count) [1]");
  bool Fc;
  bool korr=false;
  long i,j;
  long k = 1;
  double bound=exp(log(2)*52/2);
  double invbound=1/bound;
  double tempdbl0;
  double **my;
  double **lattice;
  double *vect;
  double *vect1;
  double *vect_sq;
  bigfloat conv;
  
  vectsize=cols;
  
//
// Allocating Memory for gram schmidt matrix
//
  my=new double*[cols];
  memory_handler(my,"bigint_lattice","lll_dbl(y, red_count) :: "
                    "not enough memory !");
  lattice=new double*[cols];
  memory_handler(lattice,"bigint_lattice","lll_dbl(y, red_count) :: "
                         "not enough memory !");
  for (i=0;i<cols;i++)
    {
      my[i]=new double[cols+rows];
      memory_handler(my[i],"bigint_lattice","lll_dbl(y, red_count) :: "
                           "not enough memory !");
      lattice[i]=&my[i][cols];
    }  

//
// Allocating Memory for vector of squares
//

  vect=new double[2*cols+rows];
  memory_handler(vect,"bigint_lattice","lll_dbl(y, red_count) :: "
                      "not enough memory !");
  vect_sq=&vect[cols];
  vect1=&vect[2*cols];


  ti.start_timer();
  k=1;

//
// Start
//
// Copy into floating point
//
  for (i=0;i<cols;i++)
    for (j=0;j<rows;j++)
      lattice[i][j]=dbl(value[i][j]);      

  red_count=0;
  vectsize=rows;
  Fc=false;
  dbl_assign_zero(vect);
  dbl_scalprod(vect[0],lattice[0], lattice[0]);

  while (k < cols)
    {
      Fc = true;
      while (Fc)
      {	
//
// begin of orthogonalization
//
        debug_handler("bigint_lattice", "lll_dbl(y, red_count) [2]");
        dbl_scalprod(vect[k], lattice[k], lattice[k]);
        vect_sq[k]=sqrt(vect[k]);
        for (j = 0; j < k; ++j)
          {
            dbl_scalprod(tempdbl0, lattice[k], lattice[j]);
//
// Check Precision, correct if nessesary
//
            if (fabs(tempdbl0) < invbound*vect_sq[k]*vect_sq[j])
              {
                scalprod(tempmz0, value[k], value[j]);
                my[j][k]=dbl(tempmz0);
              }
            else
              my[j][k]=tempdbl0;
              
            for (i = 0; i < j; ++i) 
              my[j][k]-=my[i][j]*my[i][k]*vect[i];
     
            tempdbl0=my[j][k];
            my[j][k]/=vect[j];
            vect[k]-=my[j][k]*tempdbl0;
          }
//
// end of orthogonalization
//
        Fc = false;
//
// begin of reduction
//
        debug_handler("bigint_lattice", "lll_dbl(y, red_count) [3]");
        for (j = k-1; j >= 0; j--)
          if (fabs(my[j][k]) > 0.5)
            {
              ++red_count;
              korr=true;
              tempdbl0=rint(my[j][k]);
              if (fabs(tempdbl0) > bound)
                Fc=true;

              conv.assign(tempdbl0);
              conv.bigintify(tempmz0);
              scalmul(tempvect1, tempmz0, value[j]);
              subtract(value[k], value[k], tempvect1);
                
	      for (i = 0; i < j; ++i) 
                {
                  my[i][k]-=tempdbl0*my[i][j];
                }

              my[j][k]-=tempdbl0;
              

            }

//
// correction (not in paper)
//

          if (korr == true)
            {
              for  (j=0;j<rows;j++)
                lattice[k][j]=dbl(value[k][j]);
              dbl_scalquad(vect_sq[k], lattice[k]);
              vect_sq[k]=sqrt(vect_sq[k]);
              korr=false;
            }

  

//
// end of reduction
//
	if (Fc)
	  if (k > 1) 
	    --k;
      }

    debug_handler("bigint_lattice", "lll_dbl(y, red_count) [4]");
    if (y*vect[k-1] > my[k-1][k]*my[k-1][k]*vect[k-1]+vect[k])       
      {
	swap(value[k], value[k-1]);
        dbl_swap(lattice[k], lattice[k-1]);
        tempdbl0=vect_sq[k];
        vect_sq[k]=vect_sq[k-1];
        vect_sq[k-1]=tempdbl0;
        
	if (k > 1)
	  --k;
        if(k == 1) 
          {
            dbl_scalprod(vect[0], lattice[0], lattice[0]);
            vect_sq[0]=sqrt(vect[0]);
          }
      }
    else
      ++k;
  }
  ti.stop_timer();
  for (i=0;i<cols;i++)
    delete[] my[i];
  delete[] my;
  delete[] lattice;
  delete[] vect;
}   

//
// Schnorr - Euchner version of lll optimized for bigints usign doubles
// result : unimodular matrix to get lll - reduced lattice for parameter y
//
void bigint_lattice::lll_trans_dbl(bigint_lattice& Tr, double y, unsigned long& red_count, timer& ti)
{
  debug_handler("bigint_lattice", "lll_dbl(Tr, y, red_count) [1]");
  bool Fc;
  bool korr=false;
  long i,j;
  long k = 1;
  double bound=exp(log(2)*52/2);
  double invbound=1/bound;
  double tempdbl0;
  double **my;
  double **lattice;
  double *vect;
  double *vect1;
  double *vect_sq;
  bigfloat conv;
  
  vectsize=cols;
  
//
// Allocating Memory for gram schmidt matrix
//
  my=new double*[cols];
  memory_handler(my,"bigint_lattice","lll_dbl(y, red_count) :: "
                    "not enough memory !");
  lattice=new double*[cols];
  memory_handler(lattice,"bigint_lattice","lll_dbl(y, red_count) :: "
                         "not enough memory !");
  for (i=0;i<cols;i++)
    {
      my[i]=new double[cols+rows];
      memory_handler(my[i],"bigint_lattice","lll_dbl(y, red_count) :: "
                           "not enough memory !");
      lattice[i]=&my[i][cols];
    }  

//
// Allocating Memory for vector of squares
//

  vect=new double[2*cols+rows];
  memory_handler(vect,"bigint_lattice","lll_dbl(y, red_count) :: "
                      "not enough memory !");
  vect_sq=&vect[cols];
  vect1=&vect[2*cols];


  ti.start_timer();
  k=1;

//
// Start
//
// Copy into floating point
//
  for (i=0;i<cols;i++)
    for (j=0;j<rows;j++)
      lattice[i][j]=dbl(value[i][j]);      

  red_count=0;
  vectsize=rows;
  Fc=false;
  dbl_assign_zero(vect);
  dbl_scalprod(vect[0],lattice[0], lattice[0]);

  while (k < cols)
    {
      Fc = true;
      while (Fc)
      {	
//
// begin of orthogonalization
//
        debug_handler("bigint_lattice", "lll_dbl(y, red_count) [2]");
        dbl_scalprod(vect[k], lattice[k], lattice[k]);
        vect_sq[k]=sqrt(vect[k]);
        for (j = 0; j < k; ++j)
          {
            dbl_scalprod(tempdbl0, lattice[k], lattice[j]);
//
// Check Precision, correct if nessesary
//
            if (fabs(tempdbl0) < invbound*vect_sq[k]*vect_sq[j])
              {
                scalprod(tempmz0, value[k], value[j]);
                my[j][k]=dbl(tempmz0);
              }
            else
              my[j][k]=tempdbl0;
              
            for (i = 0; i < j; ++i) 
              my[j][k]-=my[i][j]*my[i][k]*vect[i];
     
            tempdbl0=my[j][k];
            my[j][k]/=vect[j];
            vect[k]-=my[j][k]*tempdbl0;
          }
//
// end of orthogonalization
//
        Fc = false;
//
// begin of reduction
//
        debug_handler("bigint_lattice", "lll_dbl(y, red_count) [3]");
        for (j = k-1; j >= 0; j--)
          if (fabs(my[j][k]) > 0.5)
            {
              ++red_count;
              korr=true;
              tempdbl0=rint(my[j][k]);
              if (fabs(tempdbl0) > bound)
                Fc=true;
                
              conv.assign(tempdbl0);
              conv.bigintify(tempmz0);
              scalmul(tempvect1, tempmz0, value[j]);
              subtract(value[k], value[k], tempvect1);
              vectsize=cols;
              scalmul(tempvect1, tempmz0, Tr.value[j]);
              subtract(Tr.value[k], Tr.value[k], tempvect1);                  
              vectsize=rows;
	      for (i = 0; i < j; ++i) 
                {
                  my[i][k]-=tempdbl0*my[i][j];
                }

              my[j][k]-=tempdbl0;
              

            }

//
// correction (not in paper)
//

          if (korr == true)
            {
              for  (j=0;j<rows;j++)
                lattice[k][j]=dbl(value[k][j]);
              dbl_scalquad(vect_sq[k], lattice[k]);
              vect_sq[k]=sqrt(vect_sq[k]);
              korr=false;
            }

  

//
// end of reduction
//
	if (Fc)
	  if (k > 1) 
	    --k;
      }

    debug_handler("bigint_lattice", "lll_dbl(y, red_count) [4]");
    if (y*vect[k-1] > my[k-1][k]*my[k-1][k]*vect[k-1]+vect[k])       
      {
	swap(value[k], value[k-1]);
        swap(Tr.value[k], Tr.value[k-1]);
        dbl_swap(lattice[k], lattice[k-1]);
        tempdbl0=vect_sq[k];
        vect_sq[k]=vect_sq[k-1];
        vect_sq[k-1]=tempdbl0;
        
	if (k > 1)
	  --k;
        if(k == 1) 
          {
            dbl_scalprod(vect[0], lattice[0], lattice[0]);
            vect_sq[0]=sqrt(vect[0]);
          }
      }
    else
      ++k;
  }
  ti.stop_timer();
  for (i=0;i<cols;i++)
    delete[] my[i];
  delete[] my;
  delete[] lattice;
  delete[] vect;
}   

//
// Computes lll - reduced basis by multiplikation of A and T 
// T transformation lattice
//
void bigint_lattice::compute_basis(const bigint_lattice& A,
                                   const bigint_lattice& T)
{
  debug_handler("bigint_lattice", "compute_basis(A, T)");
  bigint_lattice temp(A.rows, A.cols);
  long i, j, cx;
// Computing the basis - vectors
  for (i=0;i<A.rows;++i)
    for (j=0;j<A.cols;++j) 
      for (cx=0;cx<A.cols;cx++)
        {
          multiply(tempmz0, A.value[cx][i], T.value[j][cx]);
          add(temp.value[j][i],temp.value[j][i],tempmz0);
        }
  assign(temp);
}

//
// Algorithm - Subroutines
//

void bigint_lattice::gamma_compute(bigfloat& g, long l)
{
  debug_handler("bigint_lattice","gamma_compute(g, l)");
// Computation of the l-th Hermite - Constant gamma,
  bigfloat tempbf;
  switch(l)
    {
      case 1 : {
        g.assign_one(); 
      } break;
      case 2: {
        g.assign(4.0/3.0); 
      } break;
      case 3: { 
        g.assign(2); 
      } break;
      case 4: {
        g.assign(4); 
      } break;
      case 5: {
        g.assign(8); 
      } break;
      case 6: {
        g.assign(64.0/3.0); 
      } break;
      case 7: {
        g.assign(64); 
      } break;
      case 8: { 
        g.assign(256); 
      } break;
      default: {
        tempbf.assign(0.75);
        log(tempbf, tempbf);
        g.assign((long )(l * (l-1)));
        g.divide_by_2();
        multiply(tempbf, tempbf, g);
        exp(g, tempbf); 
      } break;
    }
}

void bigint_lattice::alpha_compute(bigfloat& alpha)
{
// alpha = max{vect[1].l2_norm(),...,vect[rows].l2_norm()}
// norm = vect[i].l2_norm(), 0 <= i < rows
  debug_handler("bigint_lattice","alpha_compute(alpha)");
  long i;
  vectsize=rows;
  l2_norm(tempmz1,value[0]);
  for (i=1;i<cols;++i)
    {
      l2_norm(tempmz0,value[i]);
      if (tempmz1.compare(tempmz0) < 0)
        tempmz1.assign(tempmz0);
    }
  alpha.assign(tempmz1);
  sqrt(alpha, alpha);
}

void bigint_lattice::zwei_pot_q_compute(bigfloat& zweipotq, long& n2, bigfloat& alpha)
{
  debug_handler("bigint_lattice","zwei_pot_q_compute(zwpq, n2, alpha)");
  long beta = cols;
  bigfloat tempbf0;
  bigfloat tempbf1;
  bigfloat tempbf2;

// Computation of beta = min (A.rows, A.cols)
  if (rows < cols)
    beta=rows;
 
  tempbf0.assign(n2);
  sqrt(tempbf0,tempbf0);
  tempbf0.divide_by_2();
  multiply(tempbf0,cols,tempbf0);
  tempbf1.assign(cols);
  sqrt(tempbf1,tempbf1);
  add(tempbf0,tempbf0,tempbf1);
//  add(lambda,tempbf0,tempbf1);
 
  log(tempbf1,alpha);
  multiply(tempbf1,tempbf1,beta);
  exp(tempbf1,tempbf1);
// tempbf1 == potdummy
  gamma_compute(tempbf2 ,beta);
// gamma_compute(gammadummy ,beta);
 
  sqrt(tempbf2, tempbf2);
  multiply(tempbf2, tempbf2, tempbf1);
  multiply(tempbf2, tempbf2, tempbf0);
  tempbf0.assign(n2);
  multiply(tempbf0, tempbf0, cols);
  sqrt(tempbf0, tempbf0);
  add(tempbf0, tempbf0, 2);
  multiply(zweipotq, tempbf0, tempbf2);
  
  tempbf0.assign(beta + cols);
  tempbf0.divide_by_2();
  subtract(tempbf0, tempbf0, 1);
  tempbf0.multiply_by_2();
  exp(tempbf0, tempbf0);
  multiply(tempbf0, zweipotq, tempbf0);
 
  tempbf2.assign(beta + 1);
  tempbf2.divide_by_2();
  tempbf1.assign(rows);
  log(tempbf1,tempbf1);
  multiply(tempbf2, tempbf2, tempbf1);
  exp(tempbf2, tempbf2);
  divide(tempbf0, tempbf0, tempbf2); 
  ceil(zweipotq, tempbf0);
}

long bigint_lattice::compute_read_precision()
{
  debug_handler("bigint_lattice","compute_read_precision()");
  long new_prec;
  long read_prec=0;
  for (register long x=0;x<cols;++x)
    for (register long y=0;y<rows;++y)
      if (read_prec < (new_prec=(long )(value[x][y].bit_length()/log(10)+1)))
        read_prec=new_prec;
  return (read_prec);
}

long bigint_lattice::compute_precision()
{
  debug_handler("bigint_lattice","compute_precision()");
  bigfloat alpha,zweipotq;
  long read_prec=compute_read_precision();
  long n2=cols;
  long prec;
  alpha_compute(alpha);
  zwei_pot_q_compute(zweipotq,n2, alpha);
  read_prec=2*read_prec+cols-1;
  prec=(long )(2*(((zweipotq.bit_length()+1)*
        log(2)/log(10)+1)+read_prec)+(cols+rows)-1);
  return (prec);
}


