//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : lattice_basis.c
// Author      : Werner Backes (WB), Thorsten Lauer (TL) 
// Last change : WB/TL, Feb 13 1995, initial version
//

#include <LiDIA/lattice_basis.h>

//
// Constructors / Destructors
//

lattice_basis::lattice_basis(long n, long m):lattice_gensys(n,m)
{
  if (lattice_cols > lattice_rows)
    lidia_error_handler("lattice_basis","lattice_basis(n, m) :: "
                  "lattice is no basis !");
}

lattice_basis::lattice_basis(long n, long m, double** ado):lattice_gensys(n,m,ado)
{
  if (lattice_cols > lattice_rows)
    lidia_error_handler("lattice_basis","lattice_basis(n, m, ado) :: "
                  "lattice is no basis !");
}

lattice_basis::lattice_basis(long n, long m, bigint** abi):lattice_gensys(n,m,abi)
{
  if (lattice_cols > lattice_rows)
    lidia_error_handler("lattice_basis","lattice_basis(n, m, abi) :: "
                  "lattice is no basis !");
}

lattice_basis::lattice_basis(long n, long m, bigfloat** abf):lattice_gensys(n,m,abf)
{
  if (lattice_cols > lattice_rows)
    lidia_error_handler("lattice_basis","lattice_basis(n, m, abf) :: "
                  "lattice is no basis !");
}

lattice_basis::lattice_basis(const lattice_basis& B):lattice_gensys(B)
{
  if (lattice_cols > lattice_rows)
    lidia_error_handler("lattice_basis","lattice_basis(B) :: "
                  "lattice is no basis !");
}

lattice_basis::lattice_basis(const bigint_matrix& B):lattice_gensys(B)
{
  if (lattice_cols > lattice_rows)
    lidia_error_handler("lattice_basis","lattice_basis(B) :: "
                  "lattice is no basis !");
}

//
// Algorithms
//

void lattice_basis::compute_precision()
{
  debug_handler("lattice_gensys","compute_precision()");
  bigfloat alpha,zweipotq;
  long read_prec=compute_read_precision();
  long n2=lattice_cols;
  lbf->alpha_compute(alpha);
  lbf->zwei_pot_q_compute(zweipotq,n2, alpha);
  read_prec=2*read_prec+lattice_cols-1;
  prec=(long )(2*(((zweipotq.exponent()+zweipotq.bit_length()+1)*
        log(2)/log(10)+1)+read_prec)+(lattice_cols+lattice_rows)-1);
}



void lattice_basis::lll()
{
  debug_handler("lattice_basis","lll()");
  if ((lattice_cols == 1) && (lattice_rows == 1))
    lidia_error_handler("lattice_basis","lll() :: "
                  "lattice of dimension 1 detected !");
  switch (comp_mode) {
    case bigfloat_mode : {
       bigfloat_lattice temp(lattice_cols,lattice_rows);
       lbf->tempmz0.precision(prec);
       lbf->tempmz0.assign(prec);
       lbf->tempmz0.divide_by_2();
       lbf->tempmz1.assign(10);
       log(lbf->tempmz2,lbf->tempmz1);
       multiply(lbf->tempmz1,lbf->tempmz0,lbf->tempmz2);
       exp(lbf->ergmz,lbf->tempmz1);   
       T.start_timer();
       lbf->lll(temp,lbf->ergmz,y_param,red_count);
       T.stop_timer();
    } break;
    case bigint_mode : lbi->lll_dbl((double )y_zaehler/(double )y_nenner, red_count, T); break;
//    case bigint_mode : lbi->lll(red_count, y_zaehler,y_nenner, T); break;
    case double_mode : {
        double_lattice temp(lattice_cols,lattice_rows);
        double schranke=exp(7.5*log(2));
        T.start_timer();
        ldo->lll(temp,schranke,y_param,red_count);
        T.stop_timer();
    } break;
    default : {
      lidia_error_handler("lattice_basis","lll() :: "
                    "illegal computing mode !");
    } break;
  }
}

void lattice_basis::lll(const lattice_basis& B)
{
  debug_handler("lattice_basis","lll(B)");
  if ((B.lattice_cols == 1) && (B.lattice_rows == 1))
    lidia_error_handler("lattice_basis","lll(B) :: "
                  "lattice of dimension 1 detected !");

  if (comp_mode == B.comp_mode)
    {
      if ((lattice_cols != B.lattice_cols) ||
          (lattice_rows != B.lattice_rows))
        lidia_error_handler("lattice_basis","lll(B) :: "
                      "two lattice of different size !");
      switch (comp_mode) {
        case bigfloat_mode : {
           lbf->assign(*B.lbf);
           bigfloat_lattice temp(lattice_cols,lattice_rows);
           lbf->tempmz0.precision(prec);
           lbf->tempmz0.assign(prec);
           lbf->tempmz0.divide_by_2();
           lbf->tempmz1.assign(10);
           log(lbf->tempmz2,lbf->tempmz1);
           multiply(lbf->tempmz1,lbf->tempmz0,lbf->tempmz2);
           exp(lbf->ergmz,lbf->tempmz1);   
           T.start_timer();
           lbf->lll(temp,lbf->ergmz,y_param,red_count);
           T.stop_timer();
        } break;
        case bigint_mode : {
          lbi->assign(*B.lbi);
          lbi->lll_dbl((double )y_zaehler/(double )y_nenner, red_count, T); 
//          lbi->lll(red_count,y_zaehler,y_nenner,T); 
        } break;
        case double_mode : {
          ldo->assign(*B.ldo);
          double_lattice temp(lattice_cols,lattice_rows);
          double schranke=exp(7.5*log(2));
          T.start_timer();
          ldo->lll(temp,schranke,y_param,red_count);
          T.stop_timer();
        } break;
        default : {
          lidia_error_handler("lattice_basis","lll(B) :: "
                        "illegal computing mode !");
        } break;
      }
      return;
    }
  lidia_error_handler("lattice_basis","lll(B) :: "
                "two different computing modes !");
}

void lattice_basis::lll_trans(lattice_basis& Tr)
{
  debug_handler("lattice_basis","lll_trans(Tr)");
  long i, j;
  
  if ((lattice_cols == 1) && (lattice_rows == 1))
    lidia_error_handler("lattice_basis","lll_trans(Tr) :: "
                  "lattice of dimension 1 detected !");

  if ((Tr.lattice_cols == lattice_cols) && 
      (Tr.lattice_rows == lattice_cols))
    {
      switch (comp_mode) {
        case bigfloat_mode : {
          bigfloat_lattice temp(lattice_cols,lattice_cols);
          bigfloat_lattice emt(*lbf);
          lbf->tempmz0.precision(prec);
          lbf->tempmz0.assign(prec);
          lbf->tempmz0.divide_by_2();
          lbf->tempmz1.assign(10);
          log(lbf->tempmz2,lbf->tempmz1);
          multiply(lbf->tempmz1,lbf->tempmz0,lbf->tempmz2);
          exp(lbf->ergmz,lbf->tempmz1);   
          lbf->vectsize=lattice_cols;
          for (i=0;i<lattice_cols;i++)
            {
              for (j=0;j<lattice_cols;j++)
                Tr.lbf->value[i][j].assign_zero();  
              Tr.lbf->value[i][i].assign_one();
            }
          T.start_timer();
          emt.lll_trans(*Tr.lbf,temp,lbf->ergmz,y_param,red_count);
          T.stop_timer();
        } break;
        case bigint_mode : {
          bigint_lattice emt(*lbi);
          lbi->vectsize=lattice_cols;
          for (i=0;i<lattice_cols;i++)
            {      
              for (j=0;j<lattice_cols;j++)
                Tr.lbi->value[i][j].assign_zero();
              Tr.lbi->value[i][i].assign_one(); 
            }
          emt.lll_trans_dbl(*Tr.lbi, (double )y_zaehler/(double )y_nenner, red_count, T); 
//          lbi->lll_trans(*Tr.lbi, red_count, y_zaehler,y_nenner, T); 
        } break;
        case double_mode : {
          double_lattice temp(lattice_rows,lattice_cols);
          double_lattice emt(*ldo);
          double schranke=exp(7.5*log(2));
          ldo->vectsize=lattice_cols;
          for (i=0;i<lattice_cols;i++)
            {
              for (j=0;j<lattice_cols;j++)
                Tr.ldo->value[i][j]=0;
              Tr.ldo->value[i][i]=1;
            } 
          T.start_timer();
          emt.lll_trans(*Tr.ldo,temp,schranke,y_param,red_count);
          T.stop_timer();
        } break;
        default : {
          lidia_error_handler("lattice_basis","lll_trans(Tr) :: "
                        "illegal computing mode !");
        } break;
      }
      return;
    }
  lidia_error_handler("lattice_basis","lll_trans(Tr) :: "
                "illegal transformation lattice size !");
}

//
// Conversion
//

int lattice_basis::make_basis(lattice_gensys L)
{
  debug_handler("lattice_basis","make_basis(L)"); 
  int rtval=0;
  long x, y;
  
  if (comp_mode != L.comp_mode)
    lidia_error_handler("lattice_basis","make_basis(L) :: "
                  "two different computing modes !");
  switch (comp_mode) {
    case double_mode : {
      double_lattice my(L.lattice_cols, L.lattice_cols);
      double_lattice gso(L.lattice_rows, L.lattice_cols);
      if ((L.ldo->gram_schmidt_orth(gso, my) == 0) ||
          (L.lattice_cols > L.lattice_rows))
        {
          long basis_rank;
          L.ldo->lin_gen_system(my, y_param, red_count, T, basis_rank);
          L.ldo->compute_basis(*L.ldo,my); 
          if ((lattice_rows != L.lattice_rows) ||
              (basis_rank != lattice_cols))
            {
              delete ldo;
              lattice_rows=L.lattice_rows;
              lattice_cols=basis_rank;
              lattice_read=true;
              io_mode=L.io_mode;
              red_count=L.red_count;
              T=L.T;
              y_param=L.y_param;
              y_zaehler=L.y_zaehler;
              y_nenner=L.y_nenner;
              ldo=new double_lattice(lattice_rows, basis_rank);
            }
          for (x=0;x<lattice_cols;x++)
            for (y=0;y<lattice_rows;y++)
              ldo->value[x][y]=L.ldo->value[x+(L.lattice_cols-basis_rank)][y];
          rtval=0;
        }
      else
        {
          if ((lattice_cols != L.lattice_cols) ||
              (lattice_rows != L.lattice_rows))
            {
              delete ldo;
              lattice_cols=L.lattice_cols;
              lattice_rows=L.lattice_rows;
              ldo=new double_lattice(lattice_rows, lattice_cols);
            }
          assign(L);
          rtval=1;
        }
    } break;
    case bigint_mode : {
      bigfloat_lattice bfmy(L.lattice_cols, L.lattice_cols);
      L.set_computing_mode(bigfloat_mode);
      bigfloat_lattice gso(L.lattice_rows, L.lattice_cols);
      bigint_lattice my(L.lattice_cols, L.lattice_cols);
      if ((L.lbf->gram_schmidt_orth(gso, bfmy, L.prec) == 0) ||
          (L.lattice_cols > L.lattice_rows))
        {
          long basis_rank;
          L.set_computing_mode(bigint_mode);
          L.lbi->lin_gen_system(my, y_zaehler, y_nenner, red_count, T, basis_rank);
          L.lbi->compute_basis(*L.lbi,my);
          if ((lattice_rows != L.lattice_rows) ||
              (basis_rank != lattice_cols))
            {
              delete lbi;
              lattice_rows=L.lattice_rows;
              lattice_cols=basis_rank;
              lattice_read=true;
              io_mode=L.io_mode;
              red_count=L.red_count;
              T=L.T;
              y_param=L.y_param;
              y_zaehler=L.y_zaehler;
              y_nenner=L.y_nenner;
              lbi=new bigint_lattice(lattice_rows, basis_rank);
            }
          for (x=0;x<lattice_cols;x++)
            for (y=0;y<lattice_rows;y++)
              L.lbi->value[x+(L.lattice_cols-basis_rank)][y].assign(lbi->value[x][y]);
          rtval=0;
        }
      else
        {
          L.set_computing_mode(bigint_mode);
          if ((lattice_cols != L.lattice_cols) ||
              (lattice_rows != L.lattice_rows))
            {
              delete lbi;
              lattice_cols=L.lattice_cols;
              lattice_rows=L.lattice_rows;
              lbi=new bigint_lattice(lattice_rows, lattice_cols);
            }
          assign(L);
          rtval=1;
        }
    } break;
    case bigfloat_mode : {
      bigfloat_lattice my(L.lattice_cols, L.lattice_cols);
      bigfloat_lattice gso(L.lattice_rows, L.lattice_cols);
      if ((L.lbf->gram_schmidt_orth(gso, my, L.prec) == 0) ||
          (L.lattice_cols > L.lattice_rows))
        {
          long basis_rank;
          L.lbf->lin_gen_system(my, L.prec, y_param, red_count, T, basis_rank);
          L.lbf->compute_basis(*L.lbf, my);
          if ((lattice_rows != L.lattice_rows) ||
              (basis_rank != lattice_cols))
            {
              delete lbf;
              lattice_rows=L.lattice_rows;
              lattice_cols=basis_rank;
              lattice_read=true;
              io_mode=L.io_mode;
              red_count=L.red_count;
              T=L.T;
              y_param=L.y_param;
              y_zaehler=L.y_zaehler;
              y_nenner=L.y_nenner;
              lbf=new bigfloat_lattice(lattice_rows, basis_rank);
            }
          for (x=0;x<lattice_cols;x++)
            for (y=0;y<lattice_rows;y++)
              lbf->value[x][y].assign(L.lbf->value[x+(L.lattice_cols-basis_rank)][y]);
          compute_precision();
          rtval=0;
        }
      else
        {
          if ((lattice_cols != L.lattice_cols) ||
              (lattice_rows != L.lattice_rows))
            {
              delete lbf;
              lattice_cols=L.lattice_cols;
              lattice_rows=L.lattice_rows;
              lbf=new bigfloat_lattice(lattice_rows, lattice_cols);
            }
          assign(L);
          compute_precision();
          rtval=1;
        }
    } break;
    default : {
      lidia_error_handler("lattice_basis","make_basis(L) :: "
                    "illegal computing mode !");
    } break;
  }
  return (rtval);
}

//
// lll - checking
//

int lattice_basis::lll_check(double y)
{
  debug_handler("lattice_basis","lll_check(y)");
  switch (comp_mode) {
    case double_mode : {
      return(ldo->lll_check(y));
    } break;
    case bigint_mode : {
      lattice_basis temp(*this);
      temp.set_computing_mode(bigfloat_mode);
      return(temp.lll_check(y));
    } break;
    case bigfloat_mode : {
      return(lbf->lll_check(y,prec));
    } break;
    default : {
      lidia_error_handler("lattice_basis","lll_check(y) :: "
                    "illegal computing mode !");
    } break;

  }
  return 0;
}

double lattice_basis::lll_check()
{
  debug_handler("lattice_basis","lll_check()");
  switch (comp_mode) {
    case double_mode : {
      return(ldo->lll_check());
    } break;
    case bigint_mode : {
      lattice_basis temp(*this);
      temp.set_computing_mode(bigfloat_mode);
      return(temp.lll_check(prec));
    } break;
    case bigfloat_mode : {
      return(lbf->lll_check(prec));
    } break;
    default : {
      lidia_error_handler("lattice_basis","lll_check() :: "
                    "illegal computing mode !");
    } break;

  }
  return 0.0;
}

void lattice_basis::gram_schmidt_orth()
{
  debug_handler("lattice_basis","gram_schmidt_orth()");
  switch (comp_mode) {
    case double_mode : {
      double_lattice my(lattice_cols, lattice_cols);
      double_lattice gso(lattice_rows, lattice_cols);
      ldo->gram_schmidt_orth(gso,my);
      ldo->assign(gso);
    } break;
    case bigint_mode : {
      warning_handler("lattice_basis","gram_schmidt_orth(L) :: "
                      "Sorry not implemented for bigint");
    } break;
    case bigfloat_mode : {
      bigfloat_lattice my(lattice_cols, lattice_cols);
      bigfloat_lattice gso(lattice_rows, lattice_cols);
      lbf->gram_schmidt_orth(gso,my,prec);
      lbf->assign(gso);
    } break;
    default : {
      lidia_error_handler("lattice_basis","gram_schmidt_orth() :: "
                    "illegal computing mode !");
    } break;
  }
}

void lattice_basis::gram_schmidt_orth(const lattice_basis& B)
{
  debug_handler("lattice_basis","gram_schmidt_orth(B)");
  if (comp_mode == B.comp_mode)
    {
      switch (comp_mode) {
        case double_mode : {
          double_lattice my(lattice_cols, lattice_cols);
          double_lattice gso(lattice_rows, lattice_cols);
          ldo->assign(*B.ldo);
          ldo->gram_schmidt_orth(gso,my);
          ldo->assign(gso);
        } break;
        case bigint_mode : {
          warning_handler("lattice_basis","gram_schmidt_orth(B) :: "
                          "Sorry not implemented for bigint");
        } break;
        case bigfloat_mode : {
          bigfloat_lattice my(lattice_cols, lattice_cols);
          bigfloat_lattice gso(lattice_rows, lattice_cols);
          lbf->assign(*B.lbf);
          lbf->gram_schmidt_orth(gso,my,prec);  
          lbf->assign(gso);
        } break;
        default : {
          lidia_error_handler("lattice_basis","gram_schmidt_orth(B) :: "
                        "illegal computing mode !");
        } break;
      }
      return;
    }
  lidia_error_handler("lattice_basis","gram_schmidt_orth(B) :: "
                "two different computing modes !");
} 

void lll(bigint_matrix& A, const bigint_matrix& B)
{
  debug_handler("lattice_basis", "friend lll(A, B)");
  lattice_basis temp(B);
  if ((A.get_no_of_columns() != B.get_no_of_columns()) ||
      (A.get_no_of_rows() != B.get_no_of_rows()))
    lidia_error_handler("lattice_basis","friend lll(A, B) :: "
                  "bigint_matrix A, B with different size !");
  temp.lll_mbi(A);
}

void lll_trans(bigint_matrix& T, const bigint_matrix& A)
{
  debug_handler("lattice_basis", "friend lll_trans(T, A)");
  lattice_basis temp(A);
  if ((A.get_no_of_columns() != T.get_no_of_columns()) ||
      (A.get_no_of_columns() != T.get_no_of_rows()))
    lidia_error_handler("lattice_basis","friend lll_trans(T, A) :: "
                  "bigint_matrix A, T with illegal size !");
  temp.lll_trans_mbi(T);
}

void lattice_basis::lll_mbi(bigint_matrix& A)
{
  debug_handler("lattice_basis", "friend lll(A, B)");
  long x, y;
  lll();
  for (x=0;x<lattice_rows;x++)
    for (y=0;y<lattice_cols;y++)
       A.sto(x,y,lbi->value[y][x]);
}

void lattice_basis::lll_trans_mbi(bigint_matrix& T)
{
  debug_handler("lattice_basis", "friend lll_trans(T, A)");
  lattice_basis Tr(lattice_cols, lattice_cols);
  long i, x, y;
  Tr.set_computing_mode(bigint_mode);
  for (i=0;i<lattice_cols;Tr.lbi->value[i][i++].assign_one());
  lll_trans(Tr);
  for (x=0;x<lattice_cols;x++)
    for (y=0;y<lattice_cols;y++)
      T.sto(x,y,Tr.lbi->value[y][x]);
}

void lattice_basis::check_basis()
{
  debug_handler("lattice_basis","check_basis()");
  if (lattice_cols > lattice_rows)
    lidia_error_handler("lattice_basis","check_basis() :: "
                  "lattice is no basis !");
}


