      SUBROUTINE DSYTQR (D,E,N,X,LDX,WORK,JOB,INFO)
      INTEGER LDX,N,JOB,INFO
      DOUBLE PRECISION D(*),E(*),X(LDX,*),WORK(*)
C
C     DSYTQR, Double precision SYmmetric Tridiagonal QR.
C     DSYTQR computes the eigenvalues and, optionally, the eigenvectors
C     of a real symmetric tridiagonal matrix.
C
C     ON ENTRY
C
C        D       DOUBLE PRECISION (N)
C                The diagonal of a symmetric tridiagonal matrix.
C
C        E       DOUBLE PRECISION (N)
C                The offdiagonal of the matrix, E(2) through E(N).
C
C        N       INTEGER
C                The order of the matrix.
C
C        X       DOUBLE PRECISION (LDX,N)
C                Usually, the output of DSYACM.
C                If the tridiagonal matrix is the primary data,
C                X should be initialized to the identity matrix.
C
C        LDX     INTEGER
C                The leading dimension of the array X .
C
C        WORK    DOUBLE PRECISION (2*N)
C                WORK is a scratch array, used only if JOB > 0,
C                in which case its dimension must be at least 2*N.
C
C        JOB     INTEGER
C                = 0  Eigenvalues only.
C                = 1  Eigenvalues and eigenvectors, usual case.
C                = 2  Eigenvalues and eigenvectors, special case, see below.
C
C     ON RETURN
C
C        D       The eigenvalues, in decreasing order.
C
C        X       The corresponding eigenvectors, if requested.
C
C        INFO    INTEGER
C                = 0, usual case, results are accurate.
C                = nonzero, results may be inaccurate, see below.
C
C     The two special cases, JOB = 2 and INFO nonzero, involve tradeoffs
C     between speed and accuracy that apply only to so-called graded
C     matrices whose elements vary over several orders of magnitude in
C     a uniform way.  Graded matrices should be permuted so that their
C     large elements are in the upper left corner.  JOB = 1 initiates a
C     fast version of an algorithm which leads to vectors whose residuals
C     are small relative to the norm of the input matrix.  JOB = 2
C     initiates a slower version of the algorithm which leads to eigenvectors
C     whose individual components may be more accurate.  The execution times
C     of the two versions may differ by as much as 50 percent.
C     In rare situations associated with underflow, the output value of INFO
C     may be a nonzero value, M, indicating that the iteration for the M-th
C     eigenvalue did not converge satisfactorily.  This is a warning that the
C     computed eigenvalues D(1) through D(M) may be inaccurate.  Better
C     results might be obtained by multiplying the matrix by a huge scale
C     factor, say 1.0E30, computing the eigenvalues again, and then dividing
C     the computed eigenvalues by the scale factor.
C     See the EISPACK User's Guide for more details.
C
C     EISPACK 8X. This version dated 06/12/84 .
C     Cleve Moler, University of New Mexico.
C
C  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
C
C     INTERNAL VARIABLES
C
      DOUBLE PRECISION F,C,S,R,T,SHIFT,TST1,TST2,PYTHAG
      INTEGER I,J,K,L,M,ITER,MAXIT,PHASE
C
C        F          The element being "chased" by the implicit QR iteration.
C        C,S        Cosine and sine of Givens transformation.
C        R,T        Temporary variables in shift calculation and QR step.
C        SHIFT      The QR shift.
C        TST1,TST2  Used in convergence test.
C        PYTHAG     External function.  PYTHAG(A,B) = SQRT(A**2+B**2).
C        I          Index, most likely candidate for vectorization.
C        J,K        Other indices.
C        L          Start of submatrix.
C        M          End of submatrix and index of eigenvalue being found.
C        ITER       Iteration counter.
C        MAXIT      Maximum number of iterations.
C        PHASE      There are two phases.  In phase!1, the implicit tridiagonal
C                   QR algorithm with a shift from the lower 2 by 2 is used to
C                   compute a single eigenvalue, without accumulation of
C                   transformations.  If JOB = 0, only phase 1 is used.
C                   In phase 2, the QR algorithm with a shift obtained from
C                   phase 1 (a "perfect" shift) is used with accumulation of
C                   transformations to obtain eigenvectors.  If JOB = 1,
C                   only one step of phase 2 is used for each eigenvalue.
C                   This is fast, and produces accurate eigenvectors for most
C                   matrices.  If JOB = 2, the phase 2 steps are repeated until
C                   a stringent local convergence test is satisfied.  This takes
C                   more time, but produces more accurate eigenvectors for
C                   graded matrices.
C
      MAXIT = 30*N
      ITER = 0
      INFO = 0
      DO 70 M = N, 2, -1
         PHASE = 1
C
C        Save a copy of the current tridiagonal matrix for phase 2
C
         IF (JOB .GT. 0) THEN
            DO 05 I = 1, M
               WORK(I) = D(I)
               WORK(I+N) = E(I)
   05       CONTINUE
         ENDIF
   10    CONTINUE
C
C           Find L so that E(L) is negligible.
C           If the iteration count is too large, neglect E(M).
C           See the comments below about underflow.
C
            DO 20 L = M, 2, -1
               TST1 = DABS(D(L-1)) + DABS(D(L))
               TST2 = TST1 + DABS(E(L))
               IF (ITER .GT. MAXIT) THEN
                  TST2 = TST1
                  IF (INFO .EQ. 0) INFO = M
               ENDIF
               IF (TST2 .EQ. TST1) GO TO 30
   20       CONTINUE
            L = 1
C
C           If E(M) is negligible, then accept D(M) as an eigenvalue.
C
   30       IF (L .EQ. M) THEN
               IF (JOB .EQ. 0 .OR. PHASE .EQ. 2) GO TO 70
               PHASE = 2
               SHIFT = D(M)
               DO 40 I = 1, M
                  D(I) = WORK(I)
                  E(I) = WORK(I+N)
   40          CONTINUE
               GO TO 10
            ENDIF
            ITER = ITER + 1
C
C           Phase 1 shift calculation.
C
            IF (PHASE .EQ. 1) THEN
               T = (D(M-1) - D(M))/(2.0D0*E(M))
               R = PYTHAG(1.0D0,T)
               IF (T .LT. 0.0D0) R = -R
               SHIFT = D(M) - E(M)/(T + R)
            ENDIF
C
C           Implicit QR iteration, chase nonzero  F  down matrix.
C
            E(L) = D(L) - SHIFT
            F = E(L+1)
            DO 60 J = L, M-1
               R = PYTHAG(E(J),F)
               IF (R .NE. 0.0D0) THEN
                  C = E(J)/R
                  S = F/R
               ENDIF
               T = S*(D(J+1) - D(J)) + 2.0D0*C*E(J+1)
               E(J) = R
               E(J+1) = E(J+1) - C*T
               T = S*T
               D(J) = D(J) + T
               D(J+1) = D(J+1) - T
               IF (J .LT. M-1) THEN
                  F = S*E(J+2)
C                 Underflow in the computation of F jeopardizes convergence,
C                 but it is too late to do anything now, except rescale the
C                 input matrix and start all over.
                  E(J+2) = -C*E(J+2)
               ENDIF
C
C              Accumulate transformations during phase 2.
C
               IF (PHASE .EQ. 2) THEN
                  DO 50 I = 1, N
                     T = X(I,J)
                     X(I,J) = C*T + S*X(I,J+1)
                     X(I,J+1) = S*T - C*X(I,J+1)
   50             CONTINUE
               ENDIF
   60       CONTINUE
            E(L) = 0.0D0
         IF (PHASE .EQ. 1 .OR. JOB .EQ. 2) GO TO 10
   70 CONTINUE
C
C     Sort the eigenvalues, and possibly the eigenvectors.
C
      DO 130 J = 1, N-1
         K = J
         DO 110 I = J+1, N
            IF (D(I) .GT. D(K)) K = I
  110    CONTINUE
         IF (K .EQ. J) GO TO 130
         T = D(J)
         D(J) = D(K)
         D(K) = T
         IF (JOB .EQ. 0) GO TO 130
         DO 120 I = 1, N
            T = X(I,J)
            X(I,J) = X(I,K)
            X(I,K) = T
  120    CONTINUE
  130 CONTINUE
      RETURN
      END
