
#include "xdouble.h"

#include "IsFinite.h"

#include <math.h>
#include <float.h>



long xdouble::oprec = 10;

void xdouble::SetOutputPrecision(long p)
{
   if (p < 1) Error("xdouble: output precision too small");

   oprec = p;
}
   


xdouble::xdouble(double a) 
{
   if (!IsFinite(&a))
      Error("double to xdouble conversion: non finite value");
   x = a;
   e = 0;
   normalize();
}

xdouble::xdouble(float aa) 
{
   double a = aa;
   if (!IsFinite(&a))
      Error("double to xdouble conversion: non finite value");
   x = a;
   e = 0;
   normalize();
}

void operator<<(double& xx, xdouble a)
{
   double x;
   long e;

   x = a.x;
   e = a.e;

   while (e > 0) { x *= XD_BOUND; e--; }
   while (e < 0) { x *= XD_BOUND_INV; e++; }

   xx = x;
}


void xdouble::normalize() 
{
   if (x == 0) 
      e = 0;
   else if (x > 0) {
      while (x < XD_HBOUND_INV) { x *= XD_BOUND; e--; }
      while (x > XD_HBOUND) { x *= XD_BOUND_INV; e++; }
   }
   else {
      while (x > -XD_HBOUND_INV) { x *= XD_BOUND; e--; }
      while (x < -XD_HBOUND) { x *= XD_BOUND_INV; e++; }
   }

   if (e >= (1L << (ZZ_BITS_PER_LONG-2)) - 1)
      Error("xdouble: overflow");

   if (e <= -(1L << (ZZ_BITS_PER_LONG-2)) + 1)
      Error("xdouble: underflow");
}


xdouble operator+(xdouble a, xdouble b)
{
   xdouble z;

   if (a.x == 0) 
      return b;

   if (b.x == 0)
     return a;
      

   if (a.e == b.e) {
      z.x = a.x + b.x;
      z.e = a.e;
      z.normalize();
      return z;
   }
   else if (a.e > b.e) {
      if (a.e > b.e+1)
         return a;

      z.x = a.x + b.x*XD_BOUND_INV;
      z.e = a.e;
      z.normalize();
      return z;
   }
   else {
      if (b.e > a.e+1)
         return b;

      z.x = a.x*XD_BOUND_INV + b.x;
      z.e = b.e;
      z.normalize();
      return z;
   }
}


xdouble operator-(xdouble a, xdouble b)
{
   xdouble z;

   if (a.x == 0)
      return -b;

   if (b.x == 0)
      return a;

   if (a.e == b.e) {
      z.x = a.x - b.x;
      z.e = a.e;
      z.normalize();
      return z;
   }
   else if (a.e > b.e) {
      if (a.e > b.e+1)
         return a;

      z.x = a.x - b.x*XD_BOUND_INV;
      z.e = a.e;
      z.normalize();
      return z;
   }
   else {
      if (b.e > a.e+1)
         return -b;

      z.x = a.x*XD_BOUND_INV - b.x;
      z.e = b.e;
      z.normalize();
      return z;
   }
}

xdouble operator-(xdouble a)
{
   xdouble z;
   z.x = -a.x;
   z.e = a.e;
   return z;
}

xdouble operator*(xdouble a, xdouble b)
{
   xdouble z;

   z.e = a.e + b.e;
   z.x = a.x * b.x;
   z.normalize();
   return z;
}

xdouble operator/(xdouble a, xdouble b)
{
   xdouble z;

   if (b.x == 0) Error("zdouble division by 0");

   z.e = a.e - b.e;
   z.x = a.x / b.x;
   z.normalize();
   return z;
}



long compare(xdouble a, xdouble b)
{
   xdouble z = a - b;

   if (z.x < 0)
      return -1;
   else if (z.x == 0)
      return 0;
   else
      return 1;
}


xdouble trunc(xdouble a)
{
   if (a.x >= 0)
      return floor(a);
   else
      return ceil(a);
}


xdouble floor(xdouble a)
{
   xdouble z;

   if (a.e == 0) {
      z.x = floor(a.x);
      z.e = 0;
      z.normalize();
      return z;
   }
   else if (a.e > 0) {
      return a;
   }
   else {
      if (a.x < 0)
         return -1;
      else
         return 0;
   }
}

xdouble ceil(xdouble a)
{
   xdouble z;

   if (a.e == 0) {
      z.x = ceil(a.x);
      z.e = 0;
      z.normalize();
      return z;
   }
   else if (a.e > 0) {
      return a;
   }
   else {
      if (a.x < 0)
         return 0;
      else
         return 1;
   }
}

xdouble XDouble(const ZZ& a)
{
   long *n = a.rep;
   xdouble res;
   long i;
   static xdouble fradix = ZZ_RADIX;

   if (!n) return 0;

   if ((i = n[0]) < 0)
      i = -i;
   res = (xdouble) (n[i--]);
   for (; i; i--)
      res = res * fradix + n[i];
   if (n[0] < 0)
      res = -res;

   return res;
}



#define MustAlloc(c, len)  (!(c) || ((c)[-1] >> 1) < (len))



void zxdoubtoz(xdouble a, verylong *xx)
{
   verylong x;
   long neg, i, t, sz;
   static xdouble fradix_inv = ZZ_FRADIX_INV;
   static xdouble fradix = ZZ_RADIX;


   a = floor(a);

   ForceToMem(&a.x);

   if (a < 0) {
      a = -a;
      neg = 1;
   }
   else
      neg = 0;

   if (a == 0) {
      zzero(xx);
      return;
   }



   sz = 1;
   a = a*fradix_inv;

   while (a >= 1) {
      a = a*fradix_inv;
      sz++;
   }

   x = *xx;
   if (MustAlloc(x, sz)) {
      zsetlength(&x, sz);
      *xx = x;
   }


   for (i = sz; i > 0; i--) {
      a = a*fradix;
      t << a;
      x[i] = t;
      a = a - t;
   }

   x[0] = (neg ? -sz : sz);
}

xdouble fabs(xdouble a)
{
   xdouble z;

   z.e = a.e;
   z.x = fabs(a.x);
   return z;
}

xdouble sqrt(xdouble a)
{
   if (a == 0)
      return 0;

   if (a < 0)
      Error("xdouble: sqrt of negative number");

   xdouble t;

   if (a.e & 1) {
      t.e = (a.e - 1)/2;
      t.x = sqrt(a.x * XD_BOUND);
   }
   else {
      t.e = a.e/2;
      t.x = sqrt(a.x);
   }

   t.normalize();

   return t;
}
      

void power(xdouble& z, xdouble a, const ZZ& e)
{
   xdouble b, res;

   b = a;

   res = 1;
   long n = NumBits(e);
   long i;

   for (i = n-1; i >= 0; i--) {
      res = res * res;
      if (bit(e, i))
         res = res * b;
   }

   if (sign(e) < 0) 
      z = 1/res;
   else
      z = res;
}




void power(xdouble& z, xdouble a, long e)
{
   static ZZ E;
   E << e;
   power(z, a, E);
}
   

   

ostream& operator<<(ostream& s, xdouble a)
{
   if (a == 0) {
      s << "0";
      return s;
   }

   xdouble b;
   long neg;

   if (a < 0) {
      b = -a;
      neg = 1;
   }
   else {
      b = a;
      neg = 0;
   }

   long k;

   k = long((xdouble::oprec*3.321928095-log(b.x)/log(2.0)-b.e)/3.321928095);


   xdouble c;

   power(c, 10, k);

   b = b*c;

   power(c, 10, xdouble::oprec);

   while (b < c) {
      b = b * 10;
      k++;
   }

   while (b >= c) {
      b = b/10;
      k--;
   }

   b = b + 0.5;
   k = -k;

   ZZ B;
   B << b;

   char *bp = new char[xdouble::oprec+10];

   if (!bp) Error("RR output: out of memory");

   long len, i;

   len = 0;
   do {
      bp[len] = DivRem(B, B, 10) + '0';
      len++;
   } while (B > 0);

   for (i = 0; i < len/2; i++) {
      char tmp;
      tmp = bp[i];
      bp[i] = bp[len-1-i];
      bp[len-1-i] = tmp;
   }

   i = len-1;
   while (bp[i] == '0') i--;

   k += (len-1-i);
   len = i+1;

   bp[len] = '\0';

   if (k > 3 || k < -len - 3) {
      // use scientific notation

      if (neg) s << "-";
      s << "0." << bp << "e" << k + len;
   }
   else if (k >= 0) {
      if (neg) s << "-";
      s << bp;
      for (i = 0; i < k; i++) 
         s << "0";
   }
   else if (k <= -len) {
      if (neg) s << "-";
      s << "0.";
      for (i = 0; i < -len-k; i++)
         s << "0";
      s << bp;
   }
   else {
      if (neg) s << "-";
      for (i = 0; i < len+k; i++)
         s << bp[i];

      s << ".";

      for (i = len+k; i < len; i++)
         s << bp[i];
   }

   delete [] bp;
   return s;
}

istream& operator>>(istream& s, xdouble& x)
{
   long c;
   long sign;
   ZZ a, b;

   if (!s) Error("bad xdouble input");

   c = s.peek();
   while (c == ' ' || c == '\n' || c == '\t') {
      s.get();
      c = s.peek();
   }

   if (c == '-') {
      sign = -1;


      s.get();
      c = s.peek();
      while (c == ' ' || c == '\n' || c == '\t') {
         s.get();
         c = s.peek();
      }
   }
   else
      sign = 1;

   long got1 = 0;
   long got_dot = 0;
   long got2 = 0;

   a << 0;
   b << 1;

   if (c >= '0' && c <= '9') {
      got1 = 1;

      while (c >= '0' && c <= '9') {
         mul(a, a, 10);
         add(a, a, c-'0');
         s.get();
         c = s.peek();
      }
   }

   if (c == '.') {
      got_dot = 1;

      s.get();
      c = s.peek();

      if (c >= '0' && c <= '9') {
         got2 = 1;
   
         while (c >= '0' && c <= '9') {
            mul(a, a, 10);
            add(a, a, c-'0');
            mul(b, b, 10);
            s.get();
            c = s.peek();
         }
      }
   }

   if (got_dot && !got1 && !got2)  Error("bad RR input");

   ZZ e;

   long got_e = 0;
   long e_sign;

   if (c == 'e' || c == 'E') {
      got_e = 1;

      s.get();
      c = s.peek();

      if (c == '-') {
         e_sign = -1;
         s.get();
         c = s.peek();
      }
      else if (c == '+') {
         e_sign = 1;
         s.get();
         c = s.peek();
      }
      else
         e_sign = 1;

      if (c < '0' || c > '9') Error("bad xdouble input");

      e << 0;
      while (c >= '0' && c <= '9') {
         mul(e, e, 10);
         add(e, e, c-'0');
         s.get();
         c = s.peek();
      }
   }

   if (!got1 && !got2 && !got_e) Error("bad xdouble input");

   xdouble t1, t2, v;

   if (got1 || got2) {
      t1 << a;
      t2 << b;
      v = t1/t2;
   }
   else
      v = 1;

   if (sign < 0)
      v = -v;

   if (got_e) {
      if (e_sign < 0) negate(e, e);
      power(t1, 10, e);
      v = v * t1;
   }

   x = v;
   return s;
}
      

