;;; -*- Mode:Lisp; Package:Weyli; Base:10; Lowercase:T; Syntax:Common-Lisp -*-
;;; ===========================================================================
;;;			 Topology and the Basics of Geometry
;;; ===========================================================================
;;; (c) Copyright 1991 Cornell University

;;; $Id: topology.lisp,v 1.7 1992/05/13 23:07:45 rz Exp $

(in-package "WEYLI")

(define-domain-element-classes euclidean-space euclidean-space-element)

;; This is needed to avoid a precidence problem. 
(defmethod make-element ((domain euclidean-space) (value vector) &rest values)
  (declare (ignore values))
  (make-element-free-module-vector domain value))

(defmethod print-object ((elt euclidean-space-element) stream)
  (print-free-module-element elt stream))

(define-domain-creator euclidean-space (dimension)
  (make-instance 'euclidean-space
		 :coefficient-domain *general*
		 :dimension dimension)
  :predicate (lambda (d)
		 (and (eql (class-name (class-of d)) 'euclidean-space)
		      (eql (coefficient-domain d) *general*)
		      (eql (dimension d) dimension))))

(defmethod print-object ((domain euclidean-space) stream)
  (format stream #+Genera "E~D" #-Genera "E^~D"
	  (dimension domain)))

(defmethod print-object ((p point) stream)
  (format stream "<~D>" (point-order-number p)))

(defmethod print-object ((p abstract-point) stream)
  (let ((name (point-name p)))
    (if name (format stream "<~A>" name)
	(format stream "<~D>" (point-order-number p)))))

(defmethod make-element ((space abstract-space) name &rest ignore2)
  (declare (ignore ignore2))
  (make-instance 'abstract-point :domain space :name name))

(defmethod pt-order-predicate ((p1 point) (p2 point)) 
  (lisp:< (point-order-number p1) (point-order-number p2)))

(defmethod-sd binary= ((p1 point) (p2 point))
  (lisp:= (point-order-number p1) (point-order-number p2)))
  

;; Always oriented.  Notice that a simplex is not an element of any
;; particular domain!
(defclass simplex ()
     ((points :reader vertex-set
	      :initarg :points)))

(defun weyl::make-simplex (&rest points)
  (let ((space ()))
    (loop for p in points
	  do (cond ((null space)
		    (setq space (domain-of p)))
		   ((not (eql space (domain-of p)))
		    (error "~S is not an element of ~S, simplices must ~
be homogeneous"
			   p space))))
    (make-instance 'simplex :points (sort #-Genera points
					  #+Genera (copy-list points) 
					  #'pt-order-predicate))))

(defun make-simplex (&rest points)
  (make-instance 'simplex :points (sort #-Genera points 
					#+Genera (copy-list points)
					#'pt-order-predicate)))

(defmethod print-object ((s simplex) stream)
  (format stream "(~S~{, ~S~})"
	  (first (vertex-set s))
	  (rest (vertex-set s))))

(defmethod dimension ((s simplex))
  (- (length (vertex-set s)) 1))

(defmethod simplex-order ((a simplex) (b simplex))
  (loop for av in (vertex-set a)
	for bv in (vertex-set b)
	do (cond ((pt-order-predicate av bv)
		  (return t))
		 ((eql av bv) nil)
		 (t (return nil)))))

;; Notice that this isn't an around method!  --RZ
(defmethod binary= ((a simplex) (b simplex))
  (let ((a-vertices (vertex-set a))
	(b-vertices (vertex-set b)))
    (and (eql (length a-vertices) (length b-vertices))
	 (loop for av in a-vertices
	       for bv in b-vertices
	       do (cond ((pt-order-predicate av bv)
			 (return nil))
			((eql av bv) nil)
			(t (return nil)))
	       finally (return t)))))

;; FIXTHIS:  Chains are not part of any particular group.  They should
;; be though, shouldn't they!!

(defclass chain ()
     ((terms :initarg :terms :reader chain-terms)))

(defmethod print-object ((c chain) stream)
  (flet ((print-term (s coef)
	   (cond ((minus? coef)
		  (princ " - " stream)
		  (setq coef (- coef)))
		 (t (princ " + " stream)))
	   (unless (1? coef)
	     (princ coef stream))
	   (princ s stream)))
    (let* ((terms (chain-terms c))
	   (s (first (first terms)))
	   (coef (rest (first terms))))
      (cond ((null terms) (princ 0 stream))
	    (t (cond ((minus? coef)
		      (princ " - " stream)
		      (setq coef (- coef))))
	       (unless (1? coef)
		 (princ coef stream))
	       (princ s stream)
	       (loop for (simplex . coef) in (rest terms)
		     do (print-term simplex coef)))))))

(defun make-chain (&rest simplices)
  (let ((simps nil))
    (flet ((insert (simp)
	     (loop for sim in simps
		   do (when (= (first simp) (first sim))
			(setf (rest sim) (+ (rest simp) (rest sim)))
			(return t))
		   finally (push simp simps))))
      (loop for simp in simplices
	    do (insert simp))
      (make-instance 'chain
		     :terms (sort (loop for simp in simps
					unless (0? (rest simp))
					  collect simp)
				  (lambda (x y) (simplex-order (first x) (first y))))))))

(defmethod boundary ((s simplex))
  (let ((list nil)
	(vs (vertex-set s)))
    (loop for v in vs
	  for coef = 1 then (minus coef)
	  do (push (cons (%apply #'make-simplex (remove v vs))
			 coef)
		   list)) 
    (%apply #'make-chain list)))

(defun free-group-plus (xt yt)
  (pair-up-terms xt (simp1 c1) yt (simp2 c2) simplex-order
    (if simp1 (if simp2 (let ((c-sum (+ c1 c2)))
			  (if (not (0? c-sum))
			      (collect-term simp1 c-sum)))
		  (collect-term simp1 c1))
	(collect-term simp2 c2))))

(defun free-group-scalar-times (c terms)
  (cond ((0? c) nil)
	(t (loop for (simp . coef) in terms
		 for c1 = (* c coef)		; coefficient ring need not be an 
		 unless (0? c1)			; integral domain!
		   collect (cons simp c1)))))

(defmethod plus ((x chain) (y chain))
  (%apply #'make-chain 
	 (free-group-plus (chain-terms x) (chain-terms y))))

(defmethod times ((x (or integer t)) (y chain))
  (%apply #'make-chain (free-group-scalar-times x (chain-terms y))))

(defmethod times ((x chain) (y (or integer t)))
  (%apply #'make-chain (free-group-scalar-times (chain-terms x) y)))

(defmethod boundary ((c chain))
  (let ((terms (chain-terms c))
	ans)
    (setq ans (* (rest (first terms)) (boundary (first (first terms)))))
    (loop for (simp . coef) in (rest terms)
	  do(setq ans (+ ans (* coef (boundary simp)))))
    ans))

(defmacro map-over-simplices (complex (simplex n) &body body)
  `(%map-over-simplices ,complex ,n (lambda (,simplex) ,@body)))

(defmethod %map-over-simplices ((s simplex) n function)
  (let* ((vertices (vertex-set s))
	 (dim (dimension s)))
    (flet ((map-over-n-simplices (n)
	     (choose vertices (vs (1+ n))
	       (%funcall function (%apply #'make-simplex vs)))))
      (cond ((null n)
	     (loop for i below (1+ dim)
		   do (map-over-n-simplices i)))
	    ((< n dim) 
	     (map-over-n-simplices n))
	    ((= n dim)
	     (funcall function s))))))

(defclass complex ()
    ((simplices :initform (make-hash-table)
		:reader simplex-hash-table)))

(defmethod insert ((s simplex) (c complex) &key (subsimplices? t))
  (let ((simplex-hash (simplex-hash-table c)))
    (flet ((insert-simp (simp)
	     (loop for vertex in (vertex-set simp)
		   do (pushnew simp (gethash vertex simplex-hash) :test #'=))))
      (if (null subsimplices?)
	  (insert-simp s)
	  (map-over-simplices s (sub-simp nil)
	    (insert-simp sub-simp))))))

(defmethod delete ((s simplex) (c complex) &key (subsimplices? t))
  (let ((simplex-hash (simplex-hash-table c)))
    (flet ((delete-simp (simp)
	     (loop for vertex in (vertex-set simp)
		   do (setf (gethash vertex simplex-hash)
			    (delete simp (gethash vertex simplex-hash) :test #'=)))))
      (if (null subsimplices?)
	  (delete-simp s)
	  (map-over-simplices s (simp nil)
	    (delete-simp simp))))))

(defmethod %map-over-simplices ((c complex) n function)
  (let ((simplex-hash (simplex-hash-table c))
	(pts-so-far nil))
    (flet ((usable-simplex (s)
	     (let ((vs (vertex-set s)))
	       (and (or (null n)
			(eql n (dimension s)))
		    (loop for v in vs
			  do (when (member v pts-so-far)
			       (return nil))
			  finally (return t))))))
      (maphash (lambda (pt simplices)
		 (loop for simp in simplices
		       do (when (usable-simplex simp)
			    (%funcall function simp)))
		 (push pt pts-so-far))
	       simplex-hash))))

