;;; -*- Mode:Lisp; Package: Weyli; Base:10; Lowercase:T; Syntax:Common-Lisp -*-
;;; ===========================================================================
;;;			Code Walker Extensions
;;; ===========================================================================
;;; (c) copyright 1992 Cornell University

;;; $id: upolynomial.lisp,v 2.11 1992/05/15 15:55:08 rz exp $

;; These extensions allow the code walker to be used to walk Weyl
;; (general representation) expressions as well as lisp code.  Thus
;; far they don't do anything significant, when extended to include
;; operators like sum, integral etc.  It will do some more non-trivial
;; things.

(in-package "WEYLI")

;; At the moment the code walker only walks over expressions in
;; general representation.

(defun note-universal-quantification (thing env)
  (push (list thing :lexical-var :universal-quantification)
	(cadddr (walker::env-lock env))))

(defvar walk-form-simplify-expressions-p t)

(defmacro possibly-simplify (expr)
  `(setq ,expr
	 (if (and walk-form-simplify-expressions-p
		  (not (symbolp ,expr))
		  (not (consp ,expr)))
	     (simplify ,expr)
	     ,expr)))
	 

(defun walker::non-sexpr-hook (form context env)
  (walk-math-expression form context env))

;; This method takes care of variables and numbers by design.
(defmethod walk-math-expression (form context env)
  (declare (ignore context env))
  form)

(defmethod walk-math-expression ((form ge-1-ary) context env)  
  (let ((new-arg (walker::walk-form-internal (argument-of form) context env)))
    (unless (eql new-arg (argument-of form))
      (setq form (make-instance (class-of form) :domain (domain-of form)
				:arg new-arg))
      (possibly-simplify form))
    form))

(defmethod walk-math-expression ((form ge-nary) context env)
  (let (new-terms changed?)
    (loop for arg in (terms-of form)
	  for new-form = (walker::walk-form-internal arg context env)
	  do (push new-form new-terms)
	     (when (not (eq new-form arg))
	       (setq changed? t)))
    (when changed?
      (setq form (make-instance (class-of form) :domain (domain-of form)
				:terms (nreverse new-terms)))
      (possibly-simplify form))
    form))
	   
(defmethod walk-math-expression ((form ge-expt) context env)
  (let ((new-base (walker::walk-form-internal (base-of form) context env))
	(new-expt (walker::walk-form-internal (exponent-of form) context env)))
    (unless (and (eql new-base (base-of form))
		 (eql new-expt (exponent-of form)))
      (setq form 
	    (make-instance (class-of form) :domain (domain-of form)
			   :base new-base :exp new-expt))
      (possibly-simplify form))
    form))

(defmethod walk-math-expression ((form ge-equation) context env)
  (let ((new-lhs (walker::walk-form-internal (lhs-of form) context env))
	(new-rhs (walker::walk-form-internal (rhs-of form) context env)))
    (unless (and (eql new-lhs (lhs-of form))
		 (eql new-rhs (rhs-of form)))
      (setq form
	    (make-instance (class-of form) :domain (domain-of form)
			   :lhs new-lhs :rhs new-rhs))
      (possibly-simplify form))
    form))

(defmethod walk-math-expression ((form ge-deriv) context env)
  (let ((new-arg (walker::walk-form-internal (argument-of form) context env)))
    (unless (eql new-arg (argument-of form))
      (setq form (make-ge-deriv (domain-of form) new-arg (varlist-of form)))
      (possibly-simplify form))
    form))

(defmethod walk-math-expression
    ((form universal-quantified-set) context oldenv)
  (walker::walker-environment-bind (newenv oldenv)
    (walker::note-lexical-binding (var-of form) newenv)
    (let (new-exprs changed?)
      (loop for arg in (exprs-of form)
	    for new-arg = (walker::walk-form-internal arg context newenv)
	    do (unless (eql new-arg arg)
		 (setq changed? t))
	       (push new-arg new-exprs))
      (when changed?
	(setq form (apply #'make-union (var-of form) (var-domain-of form)
			  (nreverse new-exprs))))
      (possibly-simplify form))
    form))

(defmethod walk-math-expression ((x ge-function) context env)
  (declare (ignore context))
  (let ((args (walker::walk-repeat-eval (args-of x) env)))
    (if (eql args (args-of x)) x
	(apply #'make-ge-funct (domain-of x) (funct-of x) args))))
		  
