BeginPackage["Torelli`", "Global`"]

(*

This package defines some routines for calculation of the Torelli
mapping and using the Nielsen's algorithm (the shorter version).

Version 0.9, March 5, 1992.

Send comments and bug reports to
  Juha Haataja, Centre for Scientific Computing (jhaataja@csc.fi).

*)

(************************************************************************)

(* 
An example of use:

a = {{1001, 999}, {999, 1001}};
b = {{I*1001, 999}, {-999, I*1001}};
p1 = -1/2;
p2 = 4985/5015;
theta = N[999/1001, 16];

level = 4;
torell = Torelli[{p1, p2, theta, level}, {a,b}];
*)


(************************************************************************)

GenVal::usage = "Calculate the image of complex number z using generator gen:
GenVal[gen,z]"

FixedPoints::usage = "Calculate the attracting and repulsing fixed points
of a Moebius transformation: FixedPoints[gen]"

EllipticQ::usage = "Check if Moebius transformation is elliptic"

Nielsen::usage = "Computer a better basic for generators"

Torelli::usage = "Calculate the Torelli mapping"

(************************************************************************)

GenVal[{{a_,b_},{c_, d_}}, z_] := (a*z+b)/(c*z+d)

FixedPoints[gen:{{a_,b_},{c_, d_}}] :=
        Block[{za, zr, z, val = GenVal[gen,z], k},
        {za,zr} = z /. NSolve[val == z, z];
        val = GenVal[gen, 0];
        k = (val - za)/(val - zr);
        If[Abs[k] > 1, {za,zr}, {zr,za}]]

EllipticQ[gen_List] :=
	Block[{az, rz},
	{az, rz} = FixedPoints[gen];	
	If[(Abs[az] < 1 && Abs[rz] > 1) || (Abs[az] > 1 && Abs[rz] < 1),
		True, False]]

CalculateK[gen_List, z_:0] :=
	Block[{az, rz, gz},
	{az, rz} = FixedPoints[gen];
	gz = GenVal[gen, z];
	Abs[(gz - rz*z - az)/(gz -az*z - rz)]]

(************************************************************************)

CombineGenerators[genlist_List, index_Integer] :=
	Block[{glist = Drop[genlist, {index}], g = genlist[[index]], 
		h, k, min, gk = CalculateK[g]},
	glist = Join[glist, Map[Inverse, glist]];
	h = Map[Dot[g,#]&, glist];
	min = g; Scan[If[CalculateK[#] < gk, min = #]&, h];
	min]

Nielsen[genlist_List, maxiter_:10] := 
	If[Apply[Or, Map[EllipticQ, genlist]],
	"There is an elliptic element in the generators",
	Block[{indexes = Table[i, {i, 1, Length[genlist]}],
		gen1 = genlist, gen2},
	For[i = 1, i <= maxiter, i++,
		gen2 = Map[CombineGenerators[gen1, #]&, indexes];
		If[SameQ[gen1, gen2], Break[], gen1 = gen2]];
	gen1]]



Torelli[{p1_, p2_, theta_, n_}, gen_List] :=
	Block[{p3 = I*p1, p4 = I*p2, theta1 = Conjugate[theta],
	theta2 = Conjugate[I*theta], genlist, len,
	param = {{p1,p2,theta1},{p3,p4,theta1},
		{p1,p2,theta2},{p3,p4,theta2}}, 
	omega, indexes, idmat = {{1,0},{0,1}}},
	omega = Map[TorelliOmega[#, idmat]&, param];
	If[n < 1, omega,
		genlist = Join[gen, Map[Inverse,gen]]; 
		len = Length[genlist];
		indexes = Table[{i,n,1}, {i, len}];
		w = Table[GenTorelli[indexes[[i]], genlist[[i]], genlist, 
			param], {i, len}];
		omega = omega + Sum[w[[i]], {i, len}]]]


TorelliOmega[{p1_, p2_, theta_}, gen_List] :=
	Block[{points = {{p2,1},{p1,1}}, t},
	t = Map[GenVal[gen,#]&, {p1,p2}];
	Log[Apply[Divide,(1-theta*t)]]]

GenTorelli[{idx_,n_,level_}, elem_List, genlist_List, param_List] :=
	Block[{omega = Map[TorelliOmega[#, elem]&, param],
	len = Length[genlist], indexes, genl, w},
	If[n <= level, omega,
	indexes = Drop[Table[i, {i, 1, len}], 
		{1+Mod[idx + len/2 - 1, len]}];
	indexes = Select[indexes, (0 != Det[genlist[[#]]]&)];
	genl = Map[Dot[#, elem]&, genlist[[indexes]]]; 
	len = Length[genl];
	indexes = Table[{indexes[[i]], n, level+1}, {i, 1, Length[indexes]}];
	w = Table[GenTorelli[indexes[[i]], genl[[i]], genlist, param], 
		{i, 1, len}];
	omega = omega + Sum[w[[i]], {i, 1, len}]]]


(************************************************************************)


EndPackage[]
Null

