/* ======================================================================= 
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */


package org.jfree.chart;

import java.util.*;
import java.math.BigDecimal;

import java.awt.AWTEvent;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.util.EventListener;
import java.util.ResourceBundle;

import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.ToolTipManager;
import javax.swing.event.EventListenerList;

import org.jfree.chart.editor.ChartEditor;
import org.jfree.chart.editor.ChartEditorManager;
import org.jfree.chart.entity.ChartEntity;
import org.jfree.chart.entity.EntityCollection;
import org.jfree.chart.event.ChartChangeEvent;
import org.jfree.chart.event.ChartChangeListener;
import org.jfree.chart.event.ChartProgressEvent;
import org.jfree.chart.event.ChartProgressListener;
import org.jfree.chart.plot.Plot;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.PlotRenderingInfo;
import org.jfree.chart.plot.ValueAxisPlot;
import org.jfree.chart.plot.Zoomable;
import org.jfree.ui.ExtensionFileFilter;

/**
 * A class for linking two or more ChartPanel instances
 * together for printing and saving.
 * 
 * @author viski-project, Univ. of Helsinki
 */
public class MultiChartPanel extends ChartPanel {
    
    /** List of panels of which contents will be used in save/print */
    private List panels;

    /**
     * Constructs a panel that displays the specified chart.
     *
     * @param chart  the chart.
     * @param panels  the list of chartpanels.
     */
    public MultiChartPanel(JFreeChart chart,
                           List panels) {
        
        super(chart,
              DEFAULT_WIDTH,
              DEFAULT_HEIGHT,
              DEFAULT_MINIMUM_DRAW_WIDTH,
              DEFAULT_MINIMUM_DRAW_HEIGHT,
              DEFAULT_MAXIMUM_DRAW_WIDTH,
              DEFAULT_MAXIMUM_DRAW_HEIGHT,
              DEFAULT_BUFFER_USED,
              true,  // properties
              true,  // save
              true,  // print
              true,  // zoom
              true   // tooltips
              );
        this.panels=panels;

    }
  
    /**
     * Constructs a panel containing a chart.
     *
     * @param chart  the chart.
     * @param useBuffer  a flag controlling whether or not an off-screen buffer
     *                   is used.
     * @param panels  the list of chartpanels.
     */
    public MultiChartPanel(JFreeChart chart,
                           boolean useBuffer,
                           List panels) {
    
        super(chart,
             DEFAULT_WIDTH,
             DEFAULT_HEIGHT,
             DEFAULT_MINIMUM_DRAW_WIDTH,
             DEFAULT_MINIMUM_DRAW_HEIGHT,
             DEFAULT_MAXIMUM_DRAW_WIDTH,
             DEFAULT_MAXIMUM_DRAW_HEIGHT,
             useBuffer,
             true,  // properties
             true,  // save
             true,  // print
             true,  // zoom
             true   // tooltips
             );
        this.panels=panels;
    
    }
    
    /**
     * Constructs a JFreeChart panel.
     *
     * @param chart  the chart.
     * @param properties  a flag indicating whether or not the chart property
     *                    editor should be available via the popup menu.
     * @param save  a flag indicating whether or not save options should be
     *              available via the popup menu.
     * @param print  a flag indicating whether or not the print option
     *               should be available via the popup menu.
     * @param zoom  a flag indicating whether or not zoom options should
     *              be added to the popup menu.
     * @param tooltips  a flag indicating whether or not tooltips should be
     *                  enabled for the chart.
     * @param panels  the list of chartpanels.
     */
    public MultiChartPanel(JFreeChart chart,
                           boolean properties,
                           boolean save,
                           boolean print,
                           boolean zoom,
                           boolean tooltips,
                           List panels) {
   
        super(chart,
              DEFAULT_WIDTH,
              DEFAULT_HEIGHT,
              DEFAULT_MINIMUM_DRAW_WIDTH,
              DEFAULT_MINIMUM_DRAW_HEIGHT,
              DEFAULT_MAXIMUM_DRAW_WIDTH,
              DEFAULT_MAXIMUM_DRAW_HEIGHT,
              DEFAULT_BUFFER_USED,
              properties,
              save,
              print,
              zoom,
              tooltips
              );
        this.panels=panels;
    }
    /**
     * Constructs a JFreeChart panel.
     *
     * @param chart  the chart.
     * @param width  the preferred width of the panel.
     * @param height  the preferred height of the panel.
     * @param minimumDrawWidth  the minimum drawing width.
     * @param minimumDrawHeight  the minimum drawing height.
     * @param maximumDrawWidth  the maximum drawing width.
     * @param maximumDrawHeight  the maximum drawing height.
     * @param useBuffer  a flag that indicates whether to use the 
     *                   buffer to improve performance (at the 
     *                   memory).
     * @param properties  a flag indicating whether or not the chart property
     *                    editor should be available via the popup 
     * @param save  a flag indicating whether or not save options should be
     *              available via the popup menu.
     * @param print  a flag indicating whether or not the print option
     *               should be available via the popup menu.
     * @param zoom  a flag indicating whether or not zoom options should be 
     *              added to the popup menu.
     * @param tooltips  a flag indicating whether or not tooltips should be 
     *                  enabled for the chart.
     * @param panels  the list of chartpanels.
     */
    public MultiChartPanel(JFreeChart chart,
                           int width,
                           int height,
                           int minimumDrawWidth,
                           int minimumDrawHeight,
                           int maximumDrawWidth,
                           int maximumDrawHeight,
                           boolean useBuffer,
                           boolean properties,
                           boolean save,
                           boolean print,
                           boolean zoom,
                           boolean tooltips,
                           List panels) {
	super(chart,
              width,
              height,
              minimumDrawWidth,
              minimumDrawHeight,
              maximumDrawWidth,
              maximumDrawHeight,
              useBuffer,
              properties,
              save,
              print,
              zoom,
              tooltips
              );
	this.panels=panels;
    }

    /**
     * Constructs a JFreeChart panel capable of multiprint/save
     * from existing ChartPanel
     *
     * @param panels  the list of chartpanels.
     * @param chartpanel  the existing ChartPanel.
     */
    public MultiChartPanel(List panels, ChartPanel chartpanel) 
    {
        super(chartpanel.getChart());
        this.panels=panels;
    }
  
    /**
     * Prints the chart on a single page.
     *
     * @param g  the graphics context.
     * @param pf  the page format to use.
     * @param pageIndex  the index of the page. If not <code>0</code>, nothing 
     *                   gets print.
     *
     * @return The result of printing.
     */    
    public int print(Graphics g, PageFormat pf, int pageIndex) {
        
        // if panels is null, size 0 or 1, use ChartPanel's printing method
        if(panels == null || panels.size() == 0 || panels.size() == 1) {
           return super.print(g, pf, pageIndex);
        } 
       
        if (pageIndex != 0) {
            return NO_SUCH_PAGE;
        }
        
        Graphics2D g2 = (Graphics2D) g;
        double x = pf.getImageableX();
        double y = pf.getImageableY();
        double w = pf.getImageableWidth();
        double h = pf.getImageableHeight();

       //number of columns in "printing grid" (a changing method could be added)
        int saveColumns = 2;
        //number of rows in "printing grid" (number of charts / saveColumns, rounded up)
        int saveRows;

        //special case bubblegum fix, so images won't scale "wrong"        
        if (panels.size()==saveColumns && pf.getOrientation() == PageFormat.PORTRAIT) {
            saveRows = 2;
        }
        else {
            saveRows = new BigDecimal((double)panels.size()).divide(
                                      new BigDecimal((double)saveColumns), BigDecimal.ROUND_UP).intValue();
	}

        Iterator i = panels.iterator();
	while(i.hasNext()) {
            for(int j = 0; j < saveColumns; ++j) {
                if (i.hasNext()) {
                    ((MultiChartPanel)i.next()).getChart().draw(
                        g2, new Rectangle2D.Double(x, y, w/saveColumns, h/saveRows), 
                        this.getAnchor(), null
                    );
                    x = x + w/saveColumns;
                }
            }
            x = pf.getImageableX();  
            y = y + h/saveRows; 

        }
        return PAGE_EXISTS;
    }

    /**
     * Opens a file chooser and gives the user an opportunity to save the 
     * charts in same file in PNG format.
     *
     * @throws IOException if there is an I/O error.
     */    
    public void doSaveAs() throws IOException {
        
        //if panels is null, use ChartPanel's saving method 
        if (panels == null || panels.size() == 0 || panels.size() == 1) {
            super.doSaveAs();
        }
        
        else {
            JFileChooser fileChooser = new JFileChooser();
            ExtensionFileFilter filter = new ExtensionFileFilter(
            localizationResources.getString("PNG_Image_Files"), ".png"
            );
            fileChooser.addChoosableFileFilter(filter);

            int option = fileChooser.showSaveDialog(this);
            if (option == JFileChooser.APPROVE_OPTION) {
                String filename = fileChooser.getSelectedFile().getPath();
                if (isEnforceFileExtensions()) {
                    if (!filename.endsWith(".png")) {
                        filename = filename + ".png";
                    }
                }  
                ArrayList charts = new ArrayList();
                Iterator i = panels.iterator();
                while(i.hasNext()) {
                    MultiChartPanel panel = (MultiChartPanel)i.next();
                    JFreeChart chart = panel.getChart();
                    charts.add(chart);
                }
                MultiChartUtilities.saveChartsAsPNG(
                    new File(filename), charts, getWidth(), getHeight()
                );
            }
        }  
    }
}
