/* =======================================================================
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */

package org.jfree.chart.plot;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.jfree.data.hc.HCTreeNode;
import org.jfree.data.hc.DataRange;
import org.jfree.chart.plot.AbstractHCClusteringInfo;

/**
 * A class that stores information about the state of a clustering tree
 * used in {@link HCPlot}. This is organized as a tree with the same topology
 * as the original clustering tree.
 *
 * @author  viski project
 */
public class StandardHCClusteringInfo extends AbstractHCClusteringInfo {

    private HCTreeNodeInfo root;

    /**
     * Returns the visible data range. i.e. number of rows/columns
     * of a heatmap currently seen on the screen.
     *
     * @return  The datarange specifying the visible indices.
     */
    public int getNumberOfVisibleItems() {

	return this.root.getVisibleDataRange().getWidth();

    }

    /**
     * Creates a new HCTreeNodeInfo reflecting data in HCTreeNode.
     * The constructor also creates objects for all the childs
     * of HCTreeNode.
     *
     * @param rootNode the root node of a clustering tree for which
     * HCTreeNodeInfo-objects are created for.
     * @param names  names of rows or columns.
     * @param location  column or leaf tree?
     */
    public StandardHCClusteringInfo(HCTreeNode rootNode, String[] names, int location) {

	if (rootNode == null) throw new NullPointerException(
		"You must create a DummyHCClusteringInfo if you do not have " +
		"a root node."
	);
	if ((location != HCPlot.LEFT) && (location != HCPlot.TOP))
	    throw new IllegalArgumentException(
		    "location must be LEFT or RIGHT.");

        this.location=location;
	this.names = names;
	this.listeners = new LinkedList();

	this.root = new HCTreeNodeInfo(this, rootNode);

	this.root.updateVisibleDataRange();

    }

    /**
     * Return the HCTreeNodeInfo corresponding to the root node
     * of the clustering tree.
     *
     * @return  the root node.
     */
    public HCTreeNodeInfo getRootNode() {

	return this.root;

    }

    /**
     * Return the names of visible rows at subtree rooted at a specified node.
     *
     * @param node  the root of a subtree to generate names for.
     *
     * @return  the names as a List object.
     */
    private List generateNames(HCTreeNodeInfo node) {

	LinkedList list = new LinkedList();
	if ((!node.isNodeOpen()) || ((node.getLeftChild() == null) && (node.getRightChild() == null))) {
	    list.add(new NameObject(node));
	} else {
	    if (node.getLeftChild() != null)
		list.addAll(generateNames(node.getLeftChild()));
	    if (node.getRightChild() != null)
		list.addAll(generateNames(node.getRightChild()));
	}
	return list;

    }

    /**
     * Return the names of visible rows/columns corresponding to
     * this clustering info object.
     *
     * @return  a list object containing the names.
     */
    public List getNames() {

	return generateNames(this.root);

    }

    /**
     * Returns the indices of the dataset that
     * correspond to a specified visible index
     *
     * @param row  the row.
     * @return  The datarange specifying the rows.
     */
    public DataRange getDataRangeForVisibleIndex(int row) {

	return this.root.getNodeByVisibleIndex(row).getNode().getDataRange();
	
    }

    /**
     * A class used to generate name lists.
     */
    public class NameObject implements Comparable {

	private HCTreeNodeInfo node;

	/**
	 * Creates a new name object.
	 *
	 * @param node  the node to create the name object for.
	 */
	public NameObject (HCTreeNodeInfo node) {

	    this.node = node;

	}

	/**
	 * Return the names of visible rows/columns corresponding to
	 * this clustering info object.
	 *
	 * @return  a list object containing the names.
	 */
	public int compareTo(Object o) throws ClassCastException {

	    if (o instanceof NameObject) {

		NameObject n = (NameObject)o;
		try{
		if (n.node.getNode().getDataRange().getLeftBound() >
		    this.node.getNode().getDataRange().getLeftBound()
		) return -1;
		else if (n.node.getNode().getDataRange().getLeftBound() >
		    this.node.getNode().getDataRange().getLeftBound()
		) return 1;
		else return 0;
		} catch (Exception e) { throw new ClassCastException(); }

	    }
	    else throw new ClassCastException();

	}

	/**
	 * Return the name as a string.
	 *
	 * @return  the name as a string.
	 */
	public String toString() {

	    return this.node.toString();

	}

    }

}
