package it.sauronsoftware.cron4j;

import java.util.ArrayList;

/**
 * The cron4j scheduler.
 * 
 * <p>
 * With a Scheduler instance you can schedule as many tasks as you need, whose
 * execution is regulated by UNIX crontab-like patterns.
 * </p>
 * <p>
 * First of all you have to instance a Scheduler object:
 * </p>
 * 
 * <pre>
 * Scheduler myScheduler = new Scheduler();
 * </pre>
 * 
 * <p>
 * Now schedule your tasks, given as plain Runnable objects:
 * </p>
 * 
 * <pre>
 * myScheduler.scheduler(aRunnableObject, aUNIXCrontabLikeSchedulingPattern);
 * </pre>
 * 
 * <p>
 * Start the scheduler:
 * </p>
 * 
 * <pre>
 * myScheduler.start();
 * </pre>
 * 
 * <p>
 * Stop the scheduler when you need it no more:
 * </p>
 * 
 * <pre>
 * myScheduler.stop();
 * </pre>
 * 
 * <p>
 * In every moment of the scheduler life you can schedule another task calling
 * its schedule() method, regardless it is started or not. All you need, once
 * again, is a Runnable object and a scheduling pattern.
 * </p>
 * <p>
 * Note that the schedule() method gives you back an identifier for the task.
 * With its identifier you can retrieve later informations about the task, and
 * you can also reschedule() and deschedule() it.
 * </p>
 * <h2>Scheduling patterns</h2>
 * <p>
 * A UNIX crontab-like pattern is a string splitted in five space separated
 * parts. Each part is intented as:
 * </p>
 * <ol>
 * <li><strong>Minutes sub-pattern</strong>. During which minutes of the hour
 * should the task been launched? The values range is from 0 to 59. </li>
 * <li><strong>Hours sub-pattern</strong>. During which hours of the day
 * should the task been launched? The values range is from 0 to 23.</li>
 * <li><strong>Days of month sub-pattern</strong>. During which days of the
 * month should the task been launched? The values range is from 0 to 31.</li>
 * <li><strong>Months sub-pattern</strong>. During which months of the year
 * should the task been launched? The values range is from 1 (January) to 12
 * (December), otherwise this sub-pattern allows the aliases &quot;jan&quot;,
 * &quot;feb&quot;, &quot;mar&quot;, &quot;apr&quot;, &quot;may&quot;,
 * &quot;jun&quot;, &quot;jul&quot;, &quot;aug&quot;, &quot;sep&quot;,
 * &quot;oct&quot;, &quot;nov&quot; and &quot;dec&quot;.</li>
 * <li><strong>Days of week sub-pattern</strong>. During which days of the
 * week should the task been launched? The values range is from 0 (Sunday) to 6
 * (Saturday), otherwise this sub-pattern allows the aliases &quot;sun&quot;,
 * &quot;mon&quot;, &quot;tue&quot;, &quot;wed&quot;, &quot;thu&quot;,
 * &quot;fri&quot; and &quot;sat&quot;.</li>
 * </ol>
 * <p>
 * The star wildcard character is also admitted, indicating &quot;every minute
 * of the hour&quot;, &quot;every hour of the day&quot;, &quot;every day of the
 * month&quot;, &quot;every month of the year&quot; and &quot;every day of the
 * week&quot;, according to the sub-pattern in which it is used.
 * </p>
 * <p>
 * Once the scheduler is started, a task will be launched when the five parts in
 * its scheduling pattern will be true at the same time.
 * </p>
 * <p>
 * Some examples:
 * </p>
 * <p>
 * <strong>5 * * * *</strong><br>
 * This pattern causes a task to be launched once every hour, at the begin of
 * the fifth minute (00:05, 01:05, 02:05 etc.).
 * </p>
 * <p>
 * <strong>* * * * *</strong><br>
 * This pattern causes a task to be launched every minute.
 * </p>
 * <p>
 * <strong>* 12 * * Mon</strong><br>
 * This pattern causes a task to be launched every minute during the 12th hour
 * of Monday.
 * </p>
 * <p>
 * <strong>* 12 16 * Mon</strong><br>
 * This pattern causes a task to be launched every minute during the 12th hour
 * of Monday, 16th, but only if the day is the 16th of the month.
 * </p>
 * <p>
 * Every sub-pattern can contain two or more comma separated values.
 * </p>
 * <p>
 * <strong>59 11 * * 1,2,3,4,5</strong><br>
 * This pattern causes a task to be launched at 11:59AM on Monday, Tuesday,
 * Wednesday, Thursday and Friday.
 * </p>
 * <p>
 * Values intervals are admitted and defined using the minus character.
 * </p>
 * <p>
 * <strong>59 11 * * 1-5</strong><br>
 * This pattern is equivalent to the previous one.
 * </p>
 * <p>
 * The slash character can be used to identify periodic values, in the form a/b.
 * A sub-pattern with the slash character is satisfied when the value on the
 * left divided by the one on the right gives an integer result (a % b == 0).
 * </p>
 * <p>
 * <strong>*&#47;15 9-17 * * *</strong><br>
 * This pattern causes a task to be launched every 15 minutes between the 9th
 * and 17th hour of the day (9:00, 9:15, 9:30, 9:45 and so on... note that the
 * last execution will be at 17:45).
 * </p>
 * <p>
 * All the fresh described syntax rules can be used together.
 * </p>
 * <p>
 * <strong>* 12 10-16/2 * *</strong><br>
 * This pattern causes a task to be launched every minute during the 12th hour
 * of the day, but only if the day is the 10th, the 12th, the 14th or the16th of
 * the month.
 * </p>
 * <p>
 * <strong>* 12 1-15,17,20-25 * *</strong><br>
 * This pattern causes a task to be launched every minute during the 12th hour
 * of the day, but the day of the month must be between the 1st and the 15th,
 * the 20th and the 25, or at least it must be the 17th.
 * </p>
 * <p>
 * Finally cron4j lets you combine more scheduling patterns into one, with the
 * pipe character:
 * </p>
 * <p>
 * <strong>0 5 * * *|8 10 * * *|22 17 * * *</strong><br>
 * This pattern causes a task to be launched every day at 05:00, 10:08 and
 * 17:22.
 * </p>
 * 
 * @author Carlo Pelliccia
 */
public class Scheduler {

	/**
	 * A counter for object instances.
	 */
	private static int counter = 0;

	/**
	 * The state flag. If true the scheduler is started and running, otherwise
	 * it is paused and no task is launched.
	 */
	private boolean started = false;

	/**
	 * The task list.
	 */
	private ArrayList tasks = new ArrayList();

	/**
	 * The thread checking the clock and launching the tasks.
	 */
	private Thread thread = null;

	/**
	 * The list of the threads (one for task) launched and watched by the
	 * scheduler.
	 */
	private ArrayList watchedThreads = null;

	/**
	 * A self-reference (usefull for inner-classes code).
	 */
	private Scheduler scheduler = this;

	/**
	 * This method schedules a task execution.
	 * 
	 * @param schedulingPattern
	 *            The scheduling pattern for the task.
	 * @param task
	 *            The task, as a plain Runnable object.
	 * @return The task auto-generated ID assigned by the scheduler. This ID can
	 *         be used later to reschedule and deschedule the task, and also to
	 *         retrieve informations about it.
	 * @throws InvalidPatternException
	 *             If the supplied pattern is not valid.
	 */
	public synchronized Object schedule(String schedulingPattern, Runnable task)
			throws InvalidPatternException {
		SchedulingPattern myPattern = new SchedulingPattern(schedulingPattern);
		Task myTask = new Task(myPattern, task);
		synchronized (tasks) {
			tasks.add(myTask);
		}
		return myTask.getId();
	}

	/**
	 * This method changes the scheduling pattern of a task.
	 * 
	 * @param id
	 *            The ID assigned to the previously scheduled task.
	 * @param schedulingPattern
	 *            The new scheduling pattern for the task.
	 * @throws InvalidPatternException
	 *             If the supplied pattern is not valid.
	 */
	public synchronized void reschedule(Object id, String schedulingPattern)
			throws InvalidPatternException {
		SchedulingPattern myPattern = new SchedulingPattern(schedulingPattern);
		synchronized (tasks) {
			Task myTask = getTask(id);
			if (myTask != null) {
				myTask.setSchedulingPattern(myPattern);
			}
		}
	}

	/**
	 * This methods cancels the scheduling of a task.
	 * 
	 * @param id
	 *            The ID of the task.
	 */
	public synchronized void deschedule(Object id) {
		synchronized (tasks) {
			Task myTask = getTask(id);
			if (myTask != null) {
				tasks.remove(myTask);
			}
		}
	}

	/**
	 * This method retrieves the Runnable object of a previously scheduled task.
	 * 
	 * @param id
	 *            The task ID.
	 * @return The Runnable object of the task, or null if the task was not
	 *         found.
	 */
	public synchronized Runnable getTaskRunnable(Object id) {
		synchronized (tasks) {
			Task myTask = getTask(id);
			if (myTask != null) {
				return myTask.getRunnable();
			}
		}
		return null;
	}

	/**
	 * This method retrieves the scheduling pattern of a previously scheduled
	 * task.
	 * 
	 * @param id
	 *            The task ID.
	 * @return The scheduling pattern of the task, or null if the task was not
	 *         found.
	 */
	public synchronized String getTaskSchedulingPattern(Object id) {
		synchronized (tasks) {
			Task myTask = getTask(id);
			if (myTask != null) {
				return myTask.getSchedulingPattern().getOriginalString();
			}
		}
		return null;
	}

	/**
	 * This method starts the scheduler. When the scheduled is started the
	 * supplied tasks are executed at the given moment.
	 * 
	 * @throws IllegalStateException
	 *             Thrown if this scheduler is already started.
	 */
	public synchronized void start() throws IllegalStateException {
		if (!started) {
			// Init the watched threads list.
			watchedThreads = new ArrayList();
			// Start the main scheduler thread.
			thread = new Thread(new Runner(), "cron4j-scheduler-" + (++counter));
			thread.start();
			// Change the state of the object.
			started = true;
		} else {
			throw new IllegalStateException("Scheduler already started");
		}
	}

	/**
	 * This method stops the scheduler execution. Before returning it waits the
	 * end of all the running tasks previously launched. Once the scheduler has
	 * been stopped it can be started again with a start() call.
	 * 
	 * @throws IllegalStateException
	 *             Thrown if this scheduler is not started.
	 */
	public synchronized void stop() throws IllegalStateException {
		if (started) {
			// Change the state of the object.
			started = false;
			// Interrupt the scheduler thread and force a wait for its death.
			thread.interrupt();
			boolean dead = false;
			do {
				try {
					thread.join();
					dead = true;
				} catch (InterruptedException e) {
					;
				}
			} while (!dead);
			// Discharge useless references.
			thread = null;
			watchedThreads = null;
		} else {
			throw new IllegalStateException("Scheduler not started");
		}
	}

	/**
	 * This method is called by a watched thread to notify its own dead to the
	 * scheduler.
	 */
	void notifyWatchedThreadExiting(WatchedThread watchedThread) {
		synchronized (watchedThreads) {
			watchedThreads.remove(watchedThread);
		}
	}

	/**
	 * This method scans the list of the scheduled task, retrieving a task
	 * starting from its ID.
	 * 
	 * @param id
	 *            The ID of the requested task.
	 * @return The requested task, or null f it was not found.
	 */
	private Task getTask(Object id) {
		synchronized (tasks) {
			int size = tasks.size();
			for (int i = 0; i < size; i++) {
				Task myTask = (Task) tasks.get(i);
				if (myTask.getId().equals(id)) {
					return myTask;
				}
			}
		}
		return null;
	}

	/**
	 * The Runnable class with the main scheduler thread routine.
	 */
	private class Runner implements Runnable {

		/**
		 * The scheduler routine.
		 */
		public void run() {
			// Wait 'till next minute comes.
			try {
				long millis = System.currentTimeMillis();
				long nextMinute = ((millis / 60000) + 1) * 60000;
				long sleepTime = (nextMinute - millis);
				sleep(sleepTime);
			} catch (InterruptedException e1) {
				return;
			}
			// Work untill the scheduler is started.
			while (true) {
				// What time is it?
				long millis = System.currentTimeMillis();
				// Exit?
				synchronized (scheduler) {
					if (!started) {
						break;
					}
				}
				// Check every scheduled task.
				synchronized (tasks) {
					int size = tasks.size();
					for (int i = 0; i < size; i++) {
						Task task = (Task) tasks.get(i);
						SchedulingPattern pattern = task.getSchedulingPattern();
						if (pattern.match(millis)) {
							// This task must be started now!
							WatchedThread t = new WatchedThread(scheduler, task);
							synchronized (watchedThreads) {
								// Watch this one!
								watchedThreads.add(t);
							}
							t.start();
						}
					}
				}
				// Coffee break 'till next minute comes!
				long nextMinute = ((millis / 60000) + 1) * 60000;
				long sleepTime = (nextMinute - System.currentTimeMillis());
				if (sleepTime > 0) {
					try {
						sleep(sleepTime);
					} catch (InterruptedException e) {
						// Must exit!
						break;
					}
				}
			}
			// Before exiting wait for all the active tasks end (sending them an
			// interrupt ticket).
			do {
				WatchedThread t;
				synchronized (watchedThreads) {
					if (watchedThreads.size() == 0) {
						break;
					}
					t = (WatchedThread) watchedThreads.remove(0);
				}
				t.interrupt();
				boolean dead = false;
				do {
					try {
						t.join();
						dead = true;
					} catch (InterruptedException e) {
						;
					}
				} while (!dead);
			} while (true);
		}

		/**
		 * It has been reported that the {@link Thread#sleep(long)} method
		 * sometimes exits before the requested time has passed. This one offers
		 * an alternative that sometimes could sleep a few millis more than
		 * requested, but never less.
		 * 
		 * @param millis
		 *            the length of time to sleep in milliseconds.
		 * @throws InterruptedException
		 *             if another thread has interrupted the current thread. The
		 *             <i>interrupted status</i> of the current thread is
		 *             cleared when this exception is thrown.
		 * @see {@link Thread#sleep(long)}
		 */
		private void sleep(long millis) throws InterruptedException {
			long done = 0;
			do {
				long before = System.currentTimeMillis();
				Thread.sleep(millis - done);
				long after = System.currentTimeMillis();
				done += (after - before);
			} while (done < millis);
		}

	}

}
