/*
 * Decompiled with CFR 0.152.
 */
package fi.csc.chipster.toolbox;

import fi.csc.chipster.rest.Config;
import fi.csc.chipster.rest.RestUtils;
import fi.csc.chipster.toolbox.Toolbox;
import fi.csc.chipster.toolbox.resource.ModuleResource;
import fi.csc.chipster.toolbox.resource.ToolResource;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.glassfish.grizzly.http.server.HttpServer;
import org.glassfish.jersey.grizzly2.httpserver.GrizzlyHttpServerFactory;
import org.glassfish.jersey.server.ResourceConfig;

public class ToolboxService {
    private static final String TOOLS_DIR_NAME = "tools";
    public static final String TOOLS_ZIP_NAME = "tools.zip";
    private static final String[] TOOLS_SEARCH_LOCATIONS = new String[]{".", "../chipster-tools", "../chipster-tools/build/distributions"};
    private static final String RELOAD_DIR = ".reload";
    private static final String RELOAD_FILE = "touch-me-to-reload-tools";
    private Logger logger = LogManager.getLogger();
    private Config config;
    private Toolbox toolbox;
    private String url;
    private HttpServer httpServer;
    private WatchService reloadWatcher;
    private ToolResource toolResource;
    private ModuleResource moduleResource;
    private File toolsBin;

    public ToolboxService(Config config) throws IOException, URISyntaxException {
        this.config = config;
        this.url = config.getString("toolbox-bind-url");
        this.toolsBin = new File(config.getString("tools-bin-path"));
        this.initialise();
    }

    public ToolboxService(String url, String toolsBinPath) throws IOException, URISyntaxException {
        this.url = url;
        this.toolsBin = new File(toolsBinPath);
        this.initialise();
    }

    private void initialise() throws IOException, URISyntaxException {
        Toolbox newToolbox;
        if (!this.toolsBin.exists()) {
            this.logger.warn("unable to fill tool parameters from files because tools-bin path " + this.toolsBin.getPath() + " doesn't exist");
        }
        if ((newToolbox = this.loadToolbox()) == null) {
            throw new RuntimeException("failed to load toolbox");
        }
        this.toolbox = newToolbox;
        this.startReloadWatch();
    }

    private Toolbox loadToolbox() throws IOException, URISyntaxException {
        Toolbox box;
        Path foundPath = this.findToolsDir();
        if (Files.isDirectory(foundPath, new LinkOption[0])) {
            box = new Toolbox(foundPath, this.toolsBin);
            Path tempDir = Files.createTempDirectory(TOOLS_DIR_NAME, new FileAttribute[0]);
            Path tempZipFile = tempDir.resolve(TOOLS_ZIP_NAME);
            this.dirToZip(foundPath, tempZipFile);
            byte[] zipContents = Files.readAllBytes(tempZipFile);
            box.setZipContents(zipContents);
            Files.delete(tempZipFile);
            Files.delete(tempDir);
        } else {
            FileSystem fs = FileSystems.newFileSystem(foundPath, null);
            Path toolsPath = fs.getPath(TOOLS_DIR_NAME, new String[0]);
            box = new Toolbox(toolsPath, this.toolsBin);
            byte[] zipContents = Files.readAllBytes(foundPath);
            box.setZipContents(zipContents);
        }
        return box;
    }

    private void reloadToolbox() {
        this.logger.info("reloading tools");
        try {
            Toolbox newToolbox = this.loadToolbox();
            if (newToolbox == null) {
                this.logger.warn("failed to reload tools");
                return;
            }
            this.toolbox = newToolbox;
            if (this.toolResource != null) {
                this.toolResource.setToolbox(newToolbox);
            }
            if (this.moduleResource != null) {
                this.moduleResource.setToolbox(newToolbox);
            }
        }
        catch (Exception e) {
            this.logger.warn("failed to reload tools");
            return;
        }
        this.logger.info("tools reload done");
    }

    public void startServer() throws IOException, URISyntaxException {
        this.toolResource = new ToolResource(this.toolbox);
        this.moduleResource = new ModuleResource(this.toolbox);
        ResourceConfig rc = RestUtils.getDefaultResourceConfig().register((Object)this.toolResource).register((Object)this.moduleResource);
        URI baseUri = URI.create(this.url);
        this.httpServer = GrizzlyHttpServerFactory.createHttpServer((URI)baseUri, (ResourceConfig)rc);
        this.logger.info("toolbox service running at " + baseUri);
    }

    public static void main(String[] args) throws IOException, URISyntaxException {
        ToolboxService server = new ToolboxService(new Config());
        server.startServer();
        RestUtils.waitForShutdown("toolbox", server.getHttpServer());
    }

    private HttpServer getHttpServer() {
        return this.httpServer;
    }

    public void close() {
        this.closeReloadWatcher();
        RestUtils.shutdown("toolbox", this.httpServer);
    }

    private void closeReloadWatcher() {
        if (this.reloadWatcher != null) {
            try {
                this.reloadWatcher.close();
            }
            catch (IOException e) {
                this.logger.warn("failed to close reload watcher");
            }
        }
    }

    private Path findToolsDir() throws FileNotFoundException {
        for (String location : TOOLS_SEARCH_LOCATIONS) {
            Path path = Paths.get(location, TOOLS_DIR_NAME);
            this.logger.info("looking for " + path);
            if (Files.isDirectory(path, new LinkOption[0])) {
                this.logger.info("tools directory " + path + " found");
                return path;
            }
            path = Paths.get(location, TOOLS_ZIP_NAME);
            this.logger.info("looking for " + path);
            if (!Files.exists(path, new LinkOption[0])) continue;
            this.logger.info("tools zip " + path + " found");
            return path;
        }
        this.logger.warn("tools not found");
        throw new FileNotFoundException("tools not found");
    }

    public Toolbox getToolbox() {
        return this.toolbox;
    }

    public void dirToZip(final Path srcDir, Path destZip) throws IOException, URISyntaxException {
        this.logger.debug("packaging " + srcDir + " -> " + destZip);
        if (Files.exists(destZip, new LinkOption[0])) {
            this.logger.info("deleting existing " + destZip);
            Files.delete(destZip);
        }
        URI zipLocation = new URI("jar:file:" + destZip);
        HashMap<String, String> env = new HashMap<String, String>();
        env.put("create", "true");
        final FileSystem zipFs = FileSystems.newFileSystem(zipLocation, env);
        Files.walkFileTree(srcDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                return this.copy(file);
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                return this.copy(dir);
            }

            private FileVisitResult copy(Path src) throws IOException {
                Path dest = src.subpath(srcDir.getNameCount() - 1, src.getNameCount());
                Path destInZip = zipFs.getPath(dest.toString(), new String[0]);
                Files.copy(src, destInZip, StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
                return FileVisitResult.CONTINUE;
            }
        });
        zipFs.close();
    }

    private void startReloadWatch() {
        new Thread(new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                Path reloadDir = Paths.get(ToolboxService.RELOAD_DIR, new String[0]);
                Path reloadFile = reloadDir.resolve(ToolboxService.RELOAD_FILE);
                try {
                    WatchKey key;
                    boolean valid;
                    Files.createDirectories(reloadDir, new FileAttribute[0]);
                    try {
                        Files.createFile(reloadFile, new FileAttribute[0]);
                    }
                    catch (FileAlreadyExistsException fileAlreadyExistsException) {
                        // empty catch block
                    }
                    ToolboxService.this.reloadWatcher = FileSystems.getDefault().newWatchService();
                    reloadDir.register(ToolboxService.this.reloadWatcher, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY);
                    ToolboxService.this.logger.info("watching " + reloadFile + " for triggering tools reload");
                    do {
                        try {
                            key = ToolboxService.this.reloadWatcher.take();
                        }
                        catch (InterruptedException | ClosedWatchServiceException e) {
                            break;
                        }
                        for (WatchEvent<?> event : key.pollEvents()) {
                            WatchEvent.Kind<?> kind = event.kind();
                            WatchEvent<?> ev = event;
                            Path fileName = (Path)ev.context();
                            if (kind != StandardWatchEventKinds.ENTRY_MODIFY && kind != StandardWatchEventKinds.ENTRY_CREATE || !fileName.toString().equals(ToolboxService.RELOAD_FILE)) continue;
                            ToolboxService.this.logger.info("tool reload requested");
                            ToolboxService.this.reloadToolbox();
                        }
                    } while (valid = key.reset());
                }
                catch (Exception e) {
                    ToolboxService.this.logger.warn("got exception while watching reload dir " + reloadDir, (Throwable)e);
                }
                finally {
                    ToolboxService.this.closeReloadWatcher();
                }
                ToolboxService.this.logger.info("stopped watching " + reloadDir);
            }
        }, "toolbox-reload-watch").start();
    }
}

