/* -*-C++-*-
 * ###################################################################
 *	Cpptcl - Integrating C++ with Tcl
 * 
 *	FILE: "tcl_args.cc"
 *									  created: 17/6/96 {6:20:04 pm}	
 *								  last update: 05/21/98 {14:05:25 PM}	
 *	Author:	Vince Darley
 *	E-mail:	<darley@fas.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www:	<http://www.fas.harvard.edu/~darley/>
 *	
 *	See	header file	for	further	information
 * ###################################################################
 */


#include "tcl_args.h"
#include "tcl_object.h"
#include "cpptclIncludes.h"
#include "cpptcl_metaobject.h"
#include <string.h>

tcl_args& skip(tcl_args& t) {
	t.skip();
	return t;
}

tcl_args& backup(tcl_args& t) {
	t.backup();
	return t;
}

tcl_args& optional(tcl_args& t) {
	t._optional_arg = true;
	return t;
}

tcl_args& done(tcl_args& t) {
	// are there any more arguments left?
	if(t.finding_completions()) {
		t.haveErr = false;
		t.completion_type = tcl_args::Already_complete;
		t.signal_error(tcl_args::Finding_completions);
		throw(TCL_ERROR);
	} else if(t.args_left() != 0) {
		t.signal_error(tcl_args::Too_many_args);
		throw(TCL_ERROR);
	}
	return t;
}
// copy constructor, copies only the un-parsed arguments
tcl_args::tcl_args(const tcl_args& a, const tcl_base *o)
:parsed_so_far(),failed_args(),
 my_args_(),
 stored_type(a.stored_type),
 myName(a.myName),
 _finding_completions(0),expansion(0),
 completion_type(No_completion),
 completion_orig_len(0),completion_extra_len(0),completion(),
 tcl_(a.tcl_),cmd_syntax(a.cmd_syntax),help_text(a.help_text),
 got_cmd(false),
 abbrev(a.abbrev),
 objc_(a),
 objv_(a),
 match_configuration_options(a.match_configuration_options),
 container(a.container)
{
	// Perhaps we should let these three be inherited??
	// Can't think of a good reason one way or another right now
	// We kind of assume we copy at some default 'between argument'
	// time.
	cur_cmd = 0;
	_optional_arg = false;
	haveErr = false;
	parsed_so_far << list_mode_on;
	if(o) {
		parsed_so_far << o;
	}

	args_left_ = objc_;
}


tcl_args::tcl_args(tcl_obj& i, int objc, Tcl_Obj* objv[], const tcl_base* o)
:parsed_so_far(),
 failed_args(),
 my_args_(),
 myName(0),
 _finding_completions(0),expansion(0),
 completion_type(No_completion),
 completion_orig_len(0),completion_extra_len(0),completion(),
 tcl_(i), 
 cmd_syntax(),
 help_text(),
 got_cmd(false),
 abbrev(0),
 objc_(objc),
 objv_(objv),match_configuration_options(false),container(0)
{
 	stored_type.str = 0; 
 	cur_cmd = 0;
	_optional_arg = false;

	haveErr = false;
	parsed_so_far << list_mode_on;
	if(o) {
		parsed_so_far << o;
	}
	
	args_left_ = objc_;
}

tcl_args::~tcl_args(void){
	// I think we want these
	reset_stream(failed_args);
	reset_stream(parsed_so_far);
	reset_stream(completion);
	if(expansion) {
		Tcl_DecrRefCount(expansion);
		expansion = 0;
	}
}

void tcl_args::setName(Tcl_Obj *n) {
	myName = n;
}

Tcl_Obj* tcl_args::check_name(void) const {
	if(myName) {		
		return myName;
	} else {
		return orig_objv(0);
	}
}

Tcl_Obj* tcl_args::name(void) {
	// Only skip if we haven't been touched yet
	// We know we've been touched if our name as been set.
	if(!myName)
		skip();
	return check_name();
}

// static
void tcl_args::reset_stream(tcl_obj& o) {
	o.clear_value();
}

	
void tcl_args::clear_args(void) {
	cmd_syntax.clear();
	reset_stream(parsed_so_far);
	reset_stream(failed_args);
}

void tcl_args::skip(void) {
	remove_from_buffer();
	args_left_--;
}

void tcl_args::backup(void) {
	remove_from_buffer();
	args_left_++;
}

void tcl_args::remove_from_buffer(void) {
	// removal from the buffer is now a no-op
	// (we could perhaps set the arg to zero)
	my_args_.set_arg(0);
}

void tcl_args::pretend_read_from_me(void) {
	_optional_arg = false;
}

tcl_args_reader& tcl_args::check_read(void) {
	if(!args_left()) {
		if(!finding_completions()) {
			signal_error(Syntax);
		}
		my_args_.set_arg(0);
	} else {
		my_args_.set_arg((*this)[0]);
	}	
	return my_args_;
}

tcl_args& tcl_args::operator -= (const char*s) {
	if(operator==(s)) {
		// ok
	} else {
		signal_error(Syntax);
	}
	return *this;
}

/// Match next command item with a type
bool tcl_args::is_a(Tcl_ObjType *t){
	if(_ensure_have_command() && 
	   Tcl_ConvertToType(NULL,(Tcl_Obj*)cur_cmd,t) == TCL_OK) {
		return matched_cmd(cur_cmd);
	} else {
		return false;
	}
}

/// Match next command item with a type
bool tcl_args::is_a(const char* const t){
	if(!_ensure_have_command()) return false;
	struct Tcl_ObjType* ty = Tcl_GetObjType((char*)t);
	if(t) {
		if (Tcl_ConvertToType(NULL,(Tcl_Obj*)cur_cmd,ty) == TCL_OK) {
			return matched_cmd(cur_cmd);
		} else {
			return false;
		}
	}
	if(tcl_.FindObject(cur_cmd,t)) {
		return matched_cmd(cur_cmd);
	}
	return false;
}
/// Match next command item with a type
bool tcl_args::is_a(class cpx_type& /*t*/){
	if(!_ensure_have_command()) return false;
	return false;
}

bool tcl_args::matched_cmd(const Tcl_Obj* o) {
	if(o == NULL) {o = peek_arg();}
	// this will decrement args_left_
	read_done();
	// add the current command to our command-so-far
	parsed_so_far << o;
	reset_stream(failed_args);
	// we need to read another next time through.
	got_cmd = false;
	if(finding_help() || help_flag()) {
		// we were asked for help
		if(help_text) {
			parsed_so_far << list_mode_off << ": " << help_text;
		} else {
			parsed_so_far << list_mode_off << ": no help available.";
		}
		signal_error(Help);
	}
	cur_cmd = 0;
	return true;
}

bool tcl_args::_ensure_have_command(void) {
	if(!got_cmd) {
		if(args_left()) {
			check_read();
			cur_cmd = (*this)[0];
			remove_from_buffer();
		} else {
			cur_cmd = 0;
			return false;
		}
		got_cmd = true;
	}
	return true;
}

bool tcl_args::_matching_equal(const Tcl_Obj* s, const Tcl_Obj* str) {
	const char* ss = object_to_string(s);
	const char* sstr = object_to_string(str);
	// We check the first character first for speed (standard practice)
	return ((ss[0] == sstr[0] 
			&& (!strcmp(sstr,ss)
	   		    || (abbrev != 0 && !strncmp(sstr,abbrev,strlen(abbrev))
		   	        && !strncmp(sstr,ss,strlen(sstr)))
			    || (match_configuration_options && loose_match(ss,sstr))
			)) || (match_configuration_options && sstr[0] == '-' && loose_match(ss,sstr+1)));
}

bool tcl_args::loose_match(const char* s1, const char* s2) {
	// second arg may start with '-' and may have underscores instead of spaces
	if(s2[0] == '-') { s2++;}
	int i = -1;
	while (1) {
		i++;
		if(s1[i] == s2[i]) {
			if (s1[i] == '\0') return true;
			continue;
		}
		if(s1[i] == ' ' && s2[i] == '_') continue;
		return false;
	}
}

bool tcl_args::_internal_match(const char* &s, const char* const arg_name) {
	_ensure_have_command();
	const char* str = (cur_cmd ? object_to_string(cur_cmd) : 0);
	if(!s) {
		// we always match, and set to the value!
		s = str;
	}
	// We check the first character first for speed (standard practice)
	if(str 
	   && ((s[0] == str[0] 
	        && (!strcmp(str,s)
		        || (abbrev != 0 && !strncmp(str,abbrev,strlen(abbrev))
			        && !strncmp(str,s,strlen(str))))
		        || (match_configuration_options && loose_match(s,str)))
		   || (match_configuration_options && str[0] == '-' && loose_match(s,str+1)))
	   && (!desiring_next_arg_type() || next_arg_matches())
	) {
		return matched_cmd(cur_cmd);
	} else {
		return _failed_internal_match(s ? s : arg_name);
	}	
}

bool tcl_args::_failed_internal_match(const char* s) {
	// If the var was optional or it doesn't match.
	if (finding_completions()) {
		const char* str = (cur_cmd ? object_to_string(cur_cmd) : 0);
		if(!str ||
		   (s[0] == str[0] && !strncmp(str,s,strlen(str)))) {
				remember_completion(s);
		} else if((s[0] == '\0') && !str) {
			remember_completion(s);
		}
		
	}
	failed_args << s << lappend;
	help_text.clear();
	return false;	
}

bool tcl_args::_internal_match(const Tcl_Obj* &s, const char* const arg_name) {
	_ensure_have_command();
	const Tcl_Obj* str = cur_cmd;
	if(!s) {
		// we always match, and set to the value!
		s = str;
	}
	if(str && ((s == str) || _matching_equal(s,str))
	   && (!desiring_next_arg_type() || next_arg_matches())) {
		return matched_cmd(cur_cmd);
	} else {
	    // If the var was optional or it doesn't match.
		return _failed_internal_match(s ? object_to_string(s) : arg_name);
	}	
}

bool tcl_args::help_flag(void) const {
	const char* hc = peek();
	return (hc && (!strcmp(hc,"-h") || !strcmp(hc,"-help")));
}
	
bool tcl_args::next_arg_matches(void) const {	
    if(args_left() <=1) return true;
    switch(type_of_stored_type) {
      case tcl_type: 
	return (Tcl_ConvertToType(NULL,(Tcl_Obj*)objv_[objc_-args_left()+1],
				  stored_type.tcl_type) == TCL_OK);
	break;
      case cpptcl_type: 
	{
	    struct Tcl_ObjType* t = Tcl_GetObjType((char*)stored_type.str);
	    if(t) {
		return (Tcl_ConvertToType(NULL,(Tcl_Obj*)objv_[objc_-args_left()+1],t) == TCL_OK);
	    } else {
		return (tcl_.FindObject(object_to_string(objv_[objc_-args_left()+1]),stored_type.str) ? true : false);
	    }
	}
	break;
      default:
	// we shouldn't get here.
	assert(0);
	break;
    }	
    return false;
}

bool tcl_args::try_abbreviation(void) {
	// if we're completing, trying to abbreviate, or there wasn't a command
	if(finding_completions() || !cur_cmd || expansion) {
		if(expansion) {
			Tcl_DecrRefCount(expansion);
			expansion = 0;
		}
		return false;
	} else {
		// search through 'failed_args' for one that matches 'cur_cmd' best
		tcl_ << list_mode_on << "cpp::abbreviation" << failed_args
			 << cur_cmd << list_mode_off << eval ;
		if (!tcl_.ok()) return false;
		int len;
		Tcl_GetStringFromObj(tcl_.result(),&len);
		if (len) {
			// we found one
			cur_cmd = tcl_.result();
			// so cur_cmd doesn't get screwed behind our back,
			// we must share it with Tcl.
			expansion = (*this)[0];
			Tcl_IncrRefCount(expansion);
			reset_stream(failed_args);
			haveErr = 0;
			return true;
		} else {
			return false;
		}
	}
}

int tcl_args::no_match_yet(void) {
	haveErr = CPPTCL_NOT_HANDLED;
	return haveErr;
}

int tcl_args::no_match(void) {
	//read_done();
	if(finding_completions()) {
		haveErr = false;
		signal_error(Finding_completions);
	} else {
		haveErr = 0;
		signal_error(No_match);
	}
	
	return TCL_ERROR;
}

void tcl_args::internal_signal_error(int err) {
	switch(err) {
	  case Conversion:
		args_conversion_err();
		break;
	  case No_match:
		args_no_match_err();
		break;
	  case Syntax:
		args_syntax_err();
		break;
	  case Too_many_args:
		args_too_many_err();
		break;
	  case Cpp_constructor:
		args_cpp_constructor_err();
		break;
	  case Help:
		args_help_signal();
		break;
	  case Finding_completions:
		args_completion_signal();
		break;
	  default:
		// no known error!
		break;
	}
}
	
void tcl_args::signal_error(int err){
	if(!haveErr) {
		parsed_so_far << ends;
		failed_args << ends;
		if(err) 
			internal_signal_error(err);
	}
	haveErr = true;
}

const char* tcl_args::peek(void) const {
	if (args_left() > 0)  {
		return peek_string_arg();
	} else {
		return 0;
	}
}

/// Is the next argument a real
bool tcl_args::peek_is_float(void) const {
	if(args_left() <=0) return false;
	double d;
	return (Tcl_GetDoubleFromObj(NULL,(Tcl_Obj*)peek_arg(),&d) == TCL_OK);
}

/// Is the next argument a long integer
bool tcl_args::peek_is_int(void) const {
	if(args_left() <=0) return false;
	int l;
	return (Tcl_GetIntFromObj(NULL,(Tcl_Obj*)peek_arg(),&l) == TCL_OK);
}

/// Is the next argument an integer
bool tcl_args::peek_is_long(void) const {
	if(args_left() <=0) return false;
	long l;
	return (Tcl_GetLongFromObj(NULL,(Tcl_Obj*)peek_arg(),&l) == TCL_OK);
}

/// Is the next argument a string (i.e. not a number)
bool tcl_args::peek_is_string(void) const {
	return (peek_is_float() ? false : true);
}

/// Is the next argument the given string
bool tcl_args::peek_is(const char* const s) const {
	if(args_left() <=0) return false;
	return (!strcmp(peek_string_arg(),s) ? true : false);
}

tcl_args& tcl_args::check_after(void) {
	if(my_args_.bad()) {
		tcl_ << "Internal argument handling error at:";
		signal_error(Syntax);
	} else if (my_args_.eof()) {
		signal_error(Syntax);
	} else if (my_args_.fail()) {
		signal_error(Conversion);
	}
	return *this;    
}

void tcl_args::read_done(void){
	_optional_arg = false;
	if(args_left_>0){
		// only decrement if we actually read something
		args_left_--;
	}
}


void tcl_args::args_too_many_err(void) {
	tcl_ << "got cmd '" << parsed_so_far;
	tcl_ << "' with extra arguments: ";
	reset_stream(parsed_so_far);
	for(int i=total_args() - args_left();i<total_args();i++) {
		tcl_ << objv_[i] << ' ';
	}
	tcl_ << tcl_error;
}

void tcl_args::args_cpp_constructor_err(void) {
	// append in case the constructor gave some info already.
	tcl_ << cmd_syntax << append;
}

void tcl_args::args_help_signal(void) {
	tcl_ << parsed_so_far << tcl_error;
	reset_stream(parsed_so_far);
}

void tcl_args::start_completion_check(bool with_arguments) {
	tcl_ << "Completions:";
	_finding_completions |= (with_arguments ? 2 : 1);
	haveErr = true;
}

void tcl_args::end_completion_check(void) {
	_finding_completions &= 4;
	completion_orig_len = 0;
	completion_extra_len = 0;
	reset_stream(completion);
	reset_stream(parsed_so_far);
}

void tcl_args::start_help_check(void) {
	_finding_completions |= 4;
	haveErr = false;
}

void tcl_args::end_help_check(void) {
	_finding_completions &= 3;
	reset_stream(parsed_so_far);
}

void tcl_args::args_completion_signal(void) {
	if(completion_type == No_completion) {
		tcl_ << discard << "No known completions." << result;
	} else if(completion_type == Already_complete) {
		tcl_ << discard << "Already complete." << result;
	} else {
		tcl_ << result;
	}
}

void tcl_args::remember_completion(const char* s) {
	// All completions are from the point we've reached
	// so far, so we must end the string now
	int cur_cmd_len;
	if(cur_cmd) {
		Tcl_GetStringFromObj(cur_cmd,&cur_cmd_len);
	} else {
		cur_cmd_len = 0;
	}
	if(!completion_orig_len) {
		parsed_so_far << ends;
	 	completion_orig_len = strlen(parsed_so_far.str()) + cur_cmd_len;
	 	completion_extra_len = strlen(s) - cur_cmd_len;
	 	completion << s;
	 	for(unsigned int i=cur_cmd_len;i<strlen(s);i++)
	 		completion << s[i];
	 	completion << ends;
	} else {
		const char* oldcomp = completion.str();
		const char* newcomp = s;
		int i;
		for(i=0;i<completion_extra_len;i++) {
			if(oldcomp[i+cur_cmd_len] != newcomp[i+cur_cmd_len])
				break;
		}
		completion_extra_len = i;
	}
	// However we don't reset it till we're totally done.
	tcl_ << endl << parsed_so_far << ' ' << s;
	if(finding_completions() & 2) {
		if(s[0] != '\0')
			tcl_ << ' ';
		tcl_ << cmd_syntax;
	}
	switch(completion_type) {
	  case No_completion:
	  	completion_type = Unique_completion;
	  	break;
	  case Have_completion:
	  	break;
	  case Unique_completion:
	  	completion_type = Have_completion;
	  	break;
	  case Already_complete:
	    assert(0);
    }
}


void tcl_args::args_conversion_err(void) {
	tcl_ << "read partial cmd '" << parsed_so_far;
	tcl_ << "' but failed to convert next argument '"
		 << objv_[total_args() - args_left()] 
		 << '\'';

	if(stored_type.str != NULL) {
		if(type_of_stored_type == type_name) {
			tcl_ << " to type '" << stored_type.str << '\'';
		} else {
			tcl_ << stored_type.obj;
			Tcl_DecrRefCount(stored_type.obj);
		}
		stored_type.str = 0;		
	}
	tcl_ << "; syntax should be '" << parsed_so_far;
	tcl_ << ' ' << cmd_syntax << '\'' << tcl_error;
	reset_stream(parsed_so_far);
}

void tcl_args::args_no_match_err(void) {
	// Sort the failed arguments
	tcl_ << list_mode_on;
	tcl_ << "lsort" << failed_args << eval;
	reset_stream(failed_args);
	failed_args << tcl_.result() << ends;
	tcl_.ResetResult();
	tcl_ << "join" << failed_args << ", " << eval;
	reset_stream(failed_args);
	failed_args << tcl_.result() << ends;
	tcl_.ResetResult();	
	tcl_ << list_mode_off;
	if(cur_cmd){
		tcl_ << "wrong option '" << cur_cmd << '\'';
	} else {
		tcl_ << "No argument given";
	}
	tcl_ << " to '" << parsed_so_far;
	tcl_ << "' should be one of: " << failed_args << tcl_error;
	reset_stream(parsed_so_far);
	reset_stream(failed_args);
}

void tcl_args::args_syntax_err(void) {
	if(cmd_syntax) {
		tcl_ << "wrong # args: should be '" << parsed_so_far;
		tcl_ << ' ' << cmd_syntax << '\'' << tcl_error;
	} else {
		tcl_ << "wrong # args: correct syntax for '" << parsed_so_far;
		tcl_ << "' sadly unknown." << tcl_error;
	}
	reset_stream(parsed_so_far);
}

void tcl_args::const_string_read(const char*& v){
	if(!args_left()) {
		v = 0;
		if(is_optional_arg())
			pretend_read_from_me();
		else
			check_read();
	} else {
		check_read();
		v = object_to_string((*this)[0]);
		remove_from_buffer();
	}
}

void tcl_args_reader::copy_arg(char* &dest) const {
	if(arg) {
		int length;
		dest = mystrdup(Tcl_GetStringFromObj((Tcl_Obj*)arg,&length));
	} else {
		dest = new char[0];
		*dest = '\0';
	}
}

void tcl_args_reader::copy_arg_into(char* const &dest) const {
	if(arg) {
		int length;
		char* string = Tcl_GetStringFromObj((Tcl_Obj*)arg,&length);
		strncpy(dest,string,length+1);
	} else {
		*dest = '\0';
	}
}

void tcl_args_reader::examine_arg(const char* &dest) const {
	if(arg) {
		int length;
		dest = Tcl_GetStringFromObj((Tcl_Obj*)arg,&length);
	} else {
		dest = (char*)arg;
	}
}

void tcl_args_reader::examine_arg(const Tcl_Obj* &dest) const {
	dest = arg;
}



