#!/bin/bash

# Complete tests of PLplot for the three generic build types which
# consist of shared+dynamic, shared+nondynamic, and static+nondynamic.
# These complete tests that are run for each build type are (I)
# test_noninteractive, test_interactive, and ctest in the core build
# tree; (II) CMake-based test_noninteractive, test_interactive and
# ctest in the build tree for the installed examples; and (III)
# traditional [Makefile+pkg-config] test_noninteractive and
# test_interactive of the installed examples.

usage () {
    local prog=`basename $0`
    echo "Usage: $prog [OPTIONS]
OPTIONS:
  The next option specifies the directory prefix (absolute or relative to
  the original directory from which this script was invoked) which controls
  where all files produced by this script are located.
  [--prefix (defaults to the 'comprehensive_test_disposeable'
                  subdirectory of the directory just above the
                  top-level source-tree directory)]

  The next option controls whether the shared, nondynamic, and static
  subdirectories of the prefix tree are initially removed so the
  tarball of all results is guaranteed not to contain stale results.
  Only use no for this option if you want to preserve results from a
  previous run of this script that will not be tested for this run,
  (e.g., if you previously used the option --do_shared yes and are now
  using the option --do_shared no).
  [--do_clean_first (yes/no, defaults to yes)]

  The next option controls whether the script runs clean to get rid of
  file results and save disk space after the tests are completed.
  This option is highly recommended to greatly reduce the
  the disk usage (which can be as large as 40GB [!] without this
  option).
  [--do_clean_as_you_go (yes/no, defaults to yes)]

  The next six options control the tools used for the configurations, builds, and tests.
  [--cmake_command (defaults to cmake)]
  [--generator_string (defaults to 'Unix Makefiles')]
  [--cmake_added_options (defaults to none, but can be used to specify any
                          combination of white-space-separated cmake options
                          to, e.g., refine what parts of the PLplot software are
                          built and tested)]
  [--build_command (defaults to 'make -j4')]
  [--ctest_command (defaults to 'ctest -j4')]
  [--do_submit_dashboard (yes/no defaults to no,  but if set to yes, the -D Experimental option
                      will be added to the ctest command, i.e., a dashboard of the ctest results
                      will be sent to <http://my.cdash.org/index.php?project=PLplot_git> where
                      they will be publicly displayed.]

  The next three options control which of the three principal PLplot configurations are tested.
  [--do_shared (yes/no, defaults to yes)]
  [--do_nondynamic (yes/no, defaults to yes)]
  [--do_static (yes/no, defaults to yes)]

  The next six control which of eight kinds of tests
  (test_interactive, test_noninteractive, and ctest for both the build
  and install tree and test_interactive and test_noninteractive for
  the traditional install tree) are done for each kind of build.
  [--do_test_interactive (yes/no, defaults to yes)]
  [--do_test_noninteractive (yes/no, defaults to yes)]
  [--do_ctest (yes/no, defaults to yes)]
  [--do_test_build_tree (yes/no, defaults to yes)]
  [--do_test_install_tree (yes/no, defaults to yes)]
  [--do_test_traditional_install_tree (yes/no, defaults to yes)]

  [--help] Show this message.
"
    if [ $1 -ne 0 ]; then
	exit $1
    fi
}

collect_exit() {
    # Collect all information in a tarball and exit with return
    # code = $1

    # This function should only be used after prefix,
    # RELATIVE_COMPREHENSIVE_TEST_LOG and RELATIVE_ENVIRONMENT_LOG
    # have been defined.

    return_code=$1
    cd "$prefix"

    # Clean up stale results before appending new information to the tarball.
    TARBALL="$prefix/comprehensive_test.tar"

    rm -f "$TARBALL" "$TARBALL.gz"

    # Collect relevant subset of $prefix information in the tarball
    if [ -f $RELATIVE_COMPREHENSIVE_TEST_LOG ] ; then
	tar rf "$TARBALL" $RELATIVE_COMPREHENSIVE_TEST_LOG
    fi

    if [ -f $RELATIVE_ENVIRONMENT_LOG ] ; then
	tar rf "$TARBALL" $RELATIVE_ENVIRONMENT_LOG
    fi

    for directory in shared/interactive shared/noninteractive nondynamic/interactive nondynamic/noninteractive static/interactive static/noninteractive ; do
	if [ -d $directory/output_tree ] ; then
	    tar rf "$TARBALL" $directory/output_tree
	fi
	if [ -f $directory/build_tree/CMakeCache.txt ] ; then
	    tar rf "$TARBALL" $directory/build_tree/CMakeCache.txt
	fi
	if [ -f $directory/install_build_tree/CMakeCache.txt ] ; then
	    tar rf "$TARBALL" $directory/install_build_tree/CMakeCache.txt
	fi
    done

    # Collect listing of every file generated by the script
    find . -type f |xargs ls -l >| comprehensive_test_listing.out
    tar rf "$TARBALL" comprehensive_test_listing.out

    gzip "$TARBALL"

    exit $return_code
}

echo_tee() {
    # N.B. only useful after this script defines $COMPREHENSIVE_TEST_LOG
    echo "$@" |tee -a "$COMPREHENSIVE_TEST_LOG"
}

comprehensive_test () {
    CMAKE_BUILD_TYPE_OPTION=$1
    TEST_TYPE=$2
    echo_tee "
Running comprehensive_test function with the following variables set:

The variables below are key CMake options which determine the entire
kind of build that will be tested.
CMAKE_BUILD_TYPE_OPTION = $CMAKE_BUILD_TYPE_OPTION
TEST_TYPE = ${TEST_TYPE}

The location below is where all the important *.out files will be found.
OUTPUT_TREE = \"$OUTPUT_TREE\"

The location below is the top-level build-tree directory.
BUILD_TREE = \"$BUILD_TREE\"

The location below is the top-level install-tree directory.
INSTALL_TREE = \"$INSTALL_TREE\"

The location below is the top-level directory of the build tree used
for the CMake-based build and test of the installed examples.
INSTALL_BUILD_TREE = \"$INSTALL_BUILD_TREE\""

    echo_tee "
This variable specifies whether any windows platform has been detected
ANY_WINDOWS_PLATFORM=$ANY_WINDOWS_PLATFORM"

    echo_tee "
Each of the steps in this comprehensive test may take a while...."

    PATH_SAVE="$PATH"
    mkdir -p "$OUTPUT_TREE"
    # Clean start with nonexistent install tree and empty build tree(s).
    rm -rf "$INSTALL_TREE"
    rm -rf "$BUILD_TREE"
    mkdir -p "$BUILD_TREE"
    if [ "$do_test_install_tree" = "yes" ] ; then
	rm -rf "$INSTALL_BUILD_TREE"
	mkdir -p "$INSTALL_BUILD_TREE"
    fi
    cd "$BUILD_TREE"
    if [ "$do_test_build_tree" = "yes" ] ; then
	BUILD_TEST_OPTION="-DBUILD_TEST=ON"
    else
	BUILD_TEST_OPTION=""
    fi
    output="$OUTPUT_TREE"/cmake.out
    rm -f "$output"

    if [ "$CMAKE_BUILD_TYPE_OPTION" != "-DBUILD_SHARED_LIBS=OFF" -a "$ANY_WINDOWS_PLATFORM" = "true" ] ; then
	echo_tee "Prepend \"$BUILD_TREE/dll\" to the original PATH"
	PATH="$BUILD_TREE/dll:$PATH_SAVE"
    fi

    # Process $cmake_added_options into $* to be used on the ${cmake_command} command
    # line below.
    set -- $cmake_added_options
    echo_tee "${cmake_command} in the build tree"
    ${cmake_command} "-DCMAKE_INSTALL_PREFIX=$INSTALL_TREE" $BUILD_TEST_OPTION \
		     $* $CMAKE_BUILD_TYPE_OPTION -G "$generator_string" \
		     "$DASHBOARD_LABEL_OPTION" "$SOURCE_TREE" >& "$output"
    cmake_rc=$?
    if [ "$cmake_rc" -ne 0 ] ; then
	echo_tee "ERROR: ${cmake_command} in the build tree failed"
	collect_exit 1
    fi

    # Prepare do_test_type variables.
    if [ "$TEST_TYPE" = "noninteractive" ] ; then
	do_test_type=$do_test_noninteractive
    else
	do_test_type=$do_test_interactive
    fi
    if [ "$do_test_build_tree" = "yes" ] ; then
	if [ "$do_test_type" = "yes" ] ; then
	    output="${OUTPUT_TREE}/make_${TEST_TYPE}.out"
	    rm -f "$output"
	    echo_tee "$build_command VERBOSE=1 test_${TEST_TYPE} in the build tree"
	    $build_command VERBOSE=1 test_${TEST_TYPE} >& "$output"
	    make_test_type_rc=$?
	    if [ "$make_test_type_rc" -ne 0 ] ; then
		echo_tee "ERROR: $build_command VERBOSE=1 test_${TEST_TYPE} failed in the build tree"
		collect_exit 1
	    fi
	fi

	if [ "$do_ctest" = "yes" -a "$TEST_TYPE" = "noninteractive" ] ; then
	    output="$OUTPUT_TREE"/make.out
	    rm -f "$output"
	    echo_tee "$build_command VERBOSE=1 in the build tree"
	    $build_command VERBOSE=1 >& "$output"
	    make_rc=$?
	    if [ "$make_rc" -eq 0 ] ; then
		output="$OUTPUT_TREE"/ctest.out
		rm -f "$output"
		echo_tee "$ctest_command --extra-verbose ${dashboard_option}in the build tree"
		$ctest_command --extra-verbose ${dashboard_option}>& "$output"
		ctest_rc=$?
		if [ "$ctest_rc" -eq 0 ] ; then
		    if [ "$do_clean_as_you_go" = "yes" ] ; then
			output="$OUTPUT_TREE"/clean_ctest_plot_files.out
			rm -f "$output"
			echo_tee "$build_command VERBOSE=1 clean_ctest_plot_files in the build tree (since we are done with ctest)"
			$build_command VERBOSE=1 clean_ctest_plot_files >& "$output"
			make_rc=$?
			if [ "$make_rc" -ne 0 ] ; then
			    echo_tee "ERROR: $build_command VERBOSE=1 clean_ctest_plot_files failed in the build tree"
			    collect_exit 1
			fi
		    fi
		else
		    echo_tee "ERROR: $ctest_command --extra-verbose ${dashboard_option}failed in the build tree"
		    collect_exit 1
		fi
	    else
		echo_tee "ERROR: $build_command VERBOSE=1 failed in the build tree"
		collect_exit 1
	    fi
	fi
    fi

    if [ "$do_test_install_tree" = "yes" -o "$do_test_traditional_install_tree" = "yes" ] ; then
	rm -rf "$INSTALL_TREE"
	output="$OUTPUT_TREE"/make_install.out
	rm -f "$output"
	echo_tee "$build_command VERBOSE=1 install in the build tree"
	$build_command VERBOSE=1 install >& "$output"
	make_install_rc=$?
	if [ "$make_install_rc" -ne 0 ] ; then
	    echo_tee "ERROR: $build_command VERBOSE=1 install failed in the build tree"
	    collect_exit 1
	fi

	if [[ "$OSTYPE" =~ ^linux && "$CMAKE_BUILD_TYPE_OPTION" != "-DBUILD_SHARED_LIBS=OFF" ]]; then
	    # The above install and the above "if" assure there are
	    # *.so files in both the build and install trees for the
	    # linux case that can be analyzed with ldd -r.  N.B. we
	    # deliberately drop dllplplot_stubs.so and
	    # dllinstalled_plplotstubs.so from the reports because
	    # there is no known way to set the rpath for that
	    # ocamlmklib-generated shared object at the moment.  See
	    # plplot-devel mailing discusion on 2016-12-09 with the
	    # subject line "Setting rpath for dllplplot_stubs.so
	    # appears not to be possible".
	    output="$OUTPUT_TREE"/build_tree_ldd.out
	    rm -f "$output"
	    echo_tee "find \"$BUILD_TREE\" -name \"*.so\" -a ! -name \"dll*plplot_stubs.so\" -a -print0 |xargs -0 ldd -r >& \"$output\" in the build tree just after the install for TEST_TYPE = ${TEST_TYPE})"
	    find "$BUILD_TREE" -name "*.so" -a ! -name "dll*plplot_stubs.so" -a -print0 | xargs -0 ldd -r >& "$output"
	    ldd_rc=$?
	    if [ "$ldd_rc" -ne 0 ] ; then
		echo_tee "ERROR: find \"$BUILD_TREE\" -name \"*.so\" -a ! -name \"dll*plplot_stubs.so\" -a -print0 |xargs -0 ldd -r >& \"$output\" failed in the build tree"
		collect_exit 1
	    fi

	    output="$OUTPUT_TREE"/install_tree_ldd.out
	    rm -f "$output"
	    echo_tee "find \"$INSTALL_TREE\" -name \"*.so\" -a ! -name \"dll*plplot_stubs.so\" -a -print0 |xargs -0 ldd -r >& \"$output\" in the install tree just after the install for TEST_TYPE = ${TEST_TYPE})"
	    find "$INSTALL_TREE" -name "*.so" -a ! -name "dll*plplot_stubs.so" -a -print0 | xargs -0 ldd -r >& "$output"
	    ldd_rc=$?
	    if [ "$ldd_rc" -ne 0 ] ; then
		echo_tee "ERROR: find \"$INSTALL_TREE\" -name \"*.so\" -a ! -name \"dll*plplot_stubs.so\" -a -print0 |xargs -0 ldd -r >& \"$output\" failed in the install tree"
		collect_exit 1
	    fi
	fi
    fi

    if [ "$do_clean_as_you_go" = "yes" ] ; then
	output="$OUTPUT_TREE"/clean.out
	rm -f "$output"
	echo_tee "$build_command VERBOSE=1 clean in the build tree (since we are done with it for TEST_TYPE = ${TEST_TYPE})"
	$build_command VERBOSE=1 clean >& "$output"
	make_rc=$?
	if [ "$make_rc" -ne 0 ] ; then
	    echo_tee "ERROR: $build_command VERBOSE=1 clean failed in the build tree"
	    collect_exit 1
	fi
    fi

    if [ "$do_test_install_tree" = "yes" -o "$do_test_traditional_install_tree" = "yes" ] ; then
	echo_tee "Prepend \"$INSTALL_TREE/bin\" to the original PATH"
	PATH="$INSTALL_TREE/bin:$PATH_SAVE"

	if [ "$CMAKE_BUILD_TYPE_OPTION" = "-DBUILD_SHARED_LIBS=ON" -a "$ANY_WINDOWS_PLATFORM" = "true" ] ; then
	    # Use this logic to be as version-independent as possible.
	    current_dir="$(pwd)"
	    # Wild cards must not be inside quotes.
	    cd "$INSTALL_TREE"/lib/plplot[0-9].[0-9]*.[0-9]*/drivers*
	    echo_tee "Prepend \"$(pwd)\" to the current PATH"
	    PATH="$(pwd):$PATH"
	    cd $current_dir
	fi

	if [ "$do_test_install_tree" = "yes" ] ; then
	    cd "$INSTALL_BUILD_TREE"
	    output="$OUTPUT_TREE"/installed_cmake.out
	    rm -f "$output"
	    echo_tee "${cmake_command} in the installed examples build tree"
	    ${cmake_command} -G "$generator_string" "$INSTALLED_DASHBOARD_LABEL_OPTION" "$INSTALL_TREE"/share/plplot[0-9].[0-9]*.[0-9]*/examples >& "$output"
	    cmake_rc=$?
	    if [ "$cmake_rc" -ne 0 ] ; then
		echo_tee "ERROR: ${cmake_command} in the installed examples build tree failed"
		collect_exit 1
	    fi
	    if [ "$do_test_type" = "yes" ] ; then
		output="${OUTPUT_TREE}/installed_make_${TEST_TYPE}.out"
		rm -f "$output"
		echo_tee "$build_command VERBOSE=1 test_${TEST_TYPE} in the installed examples build tree"
		$build_command VERBOSE=1 test_${TEST_TYPE} >& "$output"
		make_rc=$?
		if [ "$make_rc" -ne 0 ] ; then
		    echo_tee "ERROR: $build_command VERBOSE=1 test_${TEST_TYPE} failed in the installed examples build tree"
		    collect_exit 1
		fi

		if [ "$do_clean_as_you_go" = "yes" ] ; then
		    output="$OUTPUT_TREE"/installed_clean.out
		    rm -f "$output"
		    echo_tee "$build_command VERBOSE=1 clean in the installed examples build tree (since we are done with it for TEST_TYPE = ${TEST_TYPE})"
		    $build_command VERBOSE=1 clean >& "$output"
		    make_rc=$?
		    if [ "$make_rc" -ne 0 ] ; then
			echo_tee "ERROR: $build_command VERBOSE=1 clean failed in the installed examples build tree"
			collect_exit 1
		    fi
		fi
	    fi
	    if [ "$do_ctest" = "yes" -a "$TEST_TYPE" = "noninteractive" ] ; then
		output="$OUTPUT_TREE"/installed_make.out
		rm -f "$output"
		echo_tee "$build_command VERBOSE=1 in the installed examples build tree"
		$build_command VERBOSE=1 >& "$output"
		make_rc=$?
		if [ "$make_rc" -eq 0 ] ; then
		    output="$OUTPUT_TREE"/installed_ctest.out
		    rm -f "$output"
		    echo_tee "$ctest_command --extra-verbose ${dashboard_option}in the installed examples build tree"
		    $ctest_command --extra-verbose ${dashboard_option}>& "$output"
		    ctest_rc=$?
		    if [ "$ctest_rc" -eq 0 ] ; then
			if [ "$do_clean_as_you_go" = "yes" ] ; then
			    output="$OUTPUT_TREE"/installed_clean_ctest_plot_files.out
			    rm -f "$output"
			    echo_tee "$build_command VERBOSE=1 clean_ctest_plot_files in the installed examples build tree"
			    $build_command VERBOSE=1 clean_ctest_plot_files >& "$output"
			    make_rc=$?
			    if [ "$make_rc" -ne 0 ] ; then
				echo_tee "ERROR: $build_command VERBOSE=1 clean_ctest_plot_files failed in the installed examples build tree"
				collect_exit 1
			    fi
			fi
		    else
			echo_tee "ERROR: $ctest_command --extra-verbose ${dashboard_option}failed in the installed examples build tree"
			collect_exit 1
		    fi
		else
		    echo_tee "ERROR: $build_command VERBOSE=1 failed in the installed examples build tree"
		    collect_exit 1
		fi
	    fi
	fi

	if [ "$do_test_traditional_install_tree" = "yes" -a "$do_test_type" = "yes" ] ; then
	    cd "$INSTALL_TREE"/share/plplot[0-9].[0-9]*.[0-9]*/examples
	    output="${OUTPUT_TREE}/traditional_make_${TEST_TYPE}.out"
	    rm -f "$output"
	    echo_tee "Traditional $traditional_build_command test_${TEST_TYPE} in the installed examples tree"
	    $traditional_build_command test_${TEST_TYPE} >& "$output"
	    make_rc=$?
	    if [ "$make_rc" -ne 0 ] ; then
		echo_tee "ERROR: Traditional $traditional_build_command test_${TEST_TYPE} failed in the installed examples tree"
		collect_exit 1
	    fi
	    if [ "$do_clean_as_you_go" = "yes" ] ; then
		output="$OUTPUT_TREE"/traditional_clean.out
		rm -f "$output"
		echo_tee "Traditional $traditional_build_command clean in the installed examples tree (since we are done with it for TEST_TYPE = ${TEST_TYPE})"
		$traditional_build_command clean >& "$output"
		make_rc=$?
		if [ "$make_rc" -ne 0 ] ; then
		    echo_tee "ERROR: Traditional $traditional_build_command clean failed in the installed examples tree"
		    collect_exit 1
		fi
	    fi
	fi
    fi
    echo_tee "Restore PATH to its original form"
    PATH="$PATH_SAVE"
}

# Start of actual script after functions are defined.

# Find absolute PATH of script without using readlink (since readlink is
# not available on all platforms).  Followed advice at
# http://fritzthomas.com/open-source/linux/551-how-to-get-absolute-path-within-shell-script-part2/
ORIGINAL_PATH="$(pwd)"
cd "$(dirname "$0")"
# Absolute Path of the script
SCRIPT_PATH="$(pwd)"
cd "${ORIGINAL_PATH}"

# Assumption: top-level source tree is parent directory of where script
# is located.
SOURCE_TREE=$(dirname "${SCRIPT_PATH}")
# This is the parent tree for the BUILD_TREE, INSTALL_TREE,
# INSTALL_BUILD_TREE, and OUTPUT_TREE.  It is disposable.

# Default values for options
prefix="${SOURCE_TREE}/../comprehensive_test_disposeable"

do_clean_first=yes

do_clean_as_you_go=yes

cmake_command="cmake"
generator_string="Unix Makefiles"
cmake_added_options=
build_command="make -j4"
ctest_command="ctest -j4"
do_submit_dashboard=no

do_shared=yes
do_nondynamic=yes
do_static=yes

do_test_interactive=yes
do_test_noninteractive=yes
do_ctest=yes
do_test_build_tree=yes
do_test_install_tree=yes
do_test_traditional_install_tree=yes

usage_reported=0

while test $# -gt 0; do

    case $1 in
        --prefix)
	    prefix="$2"
	    shift
	    ;;
	--do_clean_first)
	    case $2 in
		yes|no)
		    do_clean_first=$2
		    shift
		    ;;

		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
	--do_clean_as_you_go)
	    case $2 in
		yes|no)
		    do_clean_as_you_go=$2
		    shift
		    ;;

		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
        --cmake_command)
	    cmake_command=$2
	    shift
	    ;;
        --generator_string)
	    generator_string=$2
	    shift
	    ;;
        --cmake_added_options)
	    cmake_added_options=$2
            shift
            ;;
        --build_command)
	    build_command=$2
	    shift
	    ;;
        --ctest_command)
	    ctest_command=$2
	    shift
	    ;;
	--do_submit_dashboard)
	    case $2 in
		yes|no)
		    do_submit_dashboard=$2
		    shift
		    ;;

		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
	--do_shared)
	    case $2 in
		yes|no)
		    do_shared=$2
		    shift
		    ;;

		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
	--do_nondynamic)
	    case $2 in
		yes|no)
		    do_nondynamic=$2
		    shift
		    ;;
		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
	--do_static)
	    case $2 in
		yes|no)
		    do_static=$2
		    shift
		    ;;
		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
	--do_test_interactive)
	    case $2 in
		yes|no)
		    do_test_interactive=$2
		    shift
		    ;;
		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
	--do_test_noninteractive)
	    case $2 in
		yes|no)
		    do_test_noninteractive=$2
		    shift
		    ;;
		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
	--do_ctest)
	    case $2 in
		yes|no)
		    do_ctest=$2
		    shift
		    ;;
		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
	--do_test_build_tree)
	    case $2 in
		yes|no)
		    do_test_build_tree=$2
		    shift
		    ;;
		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
	--do_test_install_tree)
	    case $2 in
		yes|no)
		    do_test_install_tree=$2
		    shift
		    ;;
		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
	--do_test_traditional_install_tree)
	    case $2 in
		yes|no)
		    do_test_traditional_install_tree=$2
		    shift
		    ;;
		*)
		    usage 1 1>&2
		    ;;
	    esac
	    ;;
        --help)
            usage 0 1>&2
	    exit 0;
            ;;
        *)
            if [ $usage_reported -eq 0 ]; then
                usage_reported=1
                usage 0 1>&2
                echo " "
            fi
            echo "Unknown option: $1"
            ;;
    esac
    shift
done

if [ $usage_reported -eq 1 ]; then
    exit 1
fi

# Create $prefix directory if it does not exist already
mkdir -p "$prefix"
# Convert $prefix to absolute path (if it is not already that).
cd "$prefix"
prefix="$(pwd)"
cd "${ORIGINAL_PATH}"

# Establish names of output files.  We do this here (as soon as
# possible after $prefix is determined) because
# $COMPREHENSIVE_TEST_LOG affects echo_tee results.
# The relative versions of these are needed for the tar command.
RELATIVE_COMPREHENSIVE_TEST_LOG=comprehensive_test.sh.out
RELATIVE_ENVIRONMENT_LOG=comprehensive_test_env.out
COMPREHENSIVE_TEST_LOG="$prefix/$RELATIVE_COMPREHENSIVE_TEST_LOG"
ENVIRONMENT_LOG="$prefix/$RELATIVE_ENVIRONMENT_LOG"

# Clean up stale results before appending new information to this file
# with the echo_tee command.
rm -f "$COMPREHENSIVE_TEST_LOG"

# Set up trap of user interrupts as soon as possible after the above variables
# have been determined and the stale $COMPREHENSIVE_TEST_LOG has been removed.
trap '{ echo_tee "Exiting because of user interrupt" ; collect_exit 1; }' INT

hash git
hash_rc=$?

if [ "$hash_rc" -ne 0 ] ; then
    echo_tee "WARNING: git not on PATH so cannot determine if SOURCE_TREE =
$SOURCE_TREE is a git repository or not"
else
    cd "$SOURCE_TREE"
    git_commit_id=$(git rev-parse --short HEAD)
    git_rc=$?
    if [ "$git_rc" -ne 0 ] ; then
	echo_tee "WARNING: SOURCE_TREE = $SOURCE_TREE is not a git repository
 so cannot determine git commit id of the version of PLplot being tested"
    else
	echo_tee "git commit id for the PLplot version being tested = $git_commit_id"
    fi
fi

echo_tee "OSTYPE = ${OSTYPE}"

hash pkg-config
hash_rc=$?
if [ "$hash_rc" -ne 0 ] ; then
    echo_tee "WARNING: pkg-config not on PATH so setting do_test_traditional_install_tree=no"
    do_test_traditional_install_tree=no
fi

# The question of what to use for the traditional build command is a
# tricky one that depends on platform.  Therefore, we hard code the
# value of this variable rather than allowing the user to change it.
if [ "$generator_string" = "MinGW Makefiles" -o "$generator_string" = "MSYS Makefiles" ] ; then
    # For both these cases the MSYS make command should be used rather than
    # the MinGW mingw32-make command.  But a
    # parallel version of the MSYS make command is problematic.
    # Therefore, specify no -j option.
    traditional_build_command="make"
else
    # For all other cases, the traditional build command should be the
    # same as the build command.
    traditional_build_command="$build_command"
fi

echo_tee "Summary of options used for these tests

--prefix \"$prefix\"

--do_clean_first $do_clean_first

--do_clean_as_you_go $do_clean_as_you_go

--cmake_command \"$cmake_command\"
--generator_string \"$generator_string\"
--cmake_added_options \"$cmake_added_options\"
--build_command \"$build_command\"
  (derived) traditional_build_command \"$traditional_build_command\"
--ctest_command \"$ctest_command\"
--do_submit_dashboard $do_submit_dashboard

--do_shared $do_shared
--do_nondynamic $do_nondynamic
--do_static $do_static

--do_test_interactive $do_test_interactive
--do_test_noninteractive $do_test_noninteractive
--do_ctest $do_ctest
--do_test_build_tree $do_test_build_tree
--do_test_install_tree $do_test_install_tree
--do_test_traditional_install_tree $do_test_traditional_install_tree

N.B. do_clean_as_you_go above should be yes unless you don't mind an
accumulation of ~40GB of plot files!  Even with this option set to yes
the high-water mark of disk usage can still be as high as 4GB so be
sure you have enough free disk space to run this test!
"

if [ "$do_submit_dashboard" = "yes" ] ; then
    echo_tee "WARNING: Because you have specified \"do_submit_dashboard $do_submit_dashboard\" above, all (anonymized) details concerning each
of your ctest results will be published at <http://my.cdash.org/index.php?project=PLplot_git>
"
    dashboard_option="-D Experimental "
else
    dashboard_option=""
fi


if [ "$do_clean_first" = "yes" ] ; then
    echo_tee "WARNING: The shared, nondynamic, and static subdirectory trees of
\"$prefix\"
are about to be removed!
"
fi
ANSWER=
while [ "$ANSWER" != "yes" -a "$ANSWER" != "no" ] ; do
    echo_tee -n "Continue (yes/no)? "
    read ANSWER
    if [ -z "$ANSWER" ] ; then
	# default of no if no ANSWER
	ANSWER=no
    fi
done
echo_tee ""

if [ "$ANSWER" = "no" ] ; then
    echo_tee "Immediate exit specified!"
    exit
fi

if [ "$do_clean_first" = "yes" ] ; then
    rm -rf "$prefix/shared" "$prefix/nondynamic" "$prefix/static"
fi

# Discover if it is a Windows platform or not.
if [[ "$OSTYPE" =~ ^cygwin ]]; then
    ANY_WINDOWS_PLATFORM="true"
elif [[ "$OSTYPE" =~ ^msys ]]; then
    ANY_WINDOWS_PLATFORM="true"
elif [[ "$OSTYPE" =~ ^win ]]; then
    ANY_WINDOWS_PLATFORM="true"
else
    ANY_WINDOWS_PLATFORM="false"
fi

# Discover if it is a Mac OS X platform or not.
if [[ "$OSTYPE" =~ ^darwin ]]; then
    ANY_MAC_OSX_PLATFORM="true"
else
    ANY_MAC_OSX_PLATFORM="false"
fi

hash printenv
hash_rc=$?
if [ "$hash_rc" -ne 0 ] ; then
    HAS_PRINTENV=false
else
    HAS_PRINTENV=true
fi

# Collect selected important environment variable results prior to testing.
rm -f "$ENVIRONMENT_LOG"
if [ "$HAS_PRINTENV" = "false" ] ; then
    echo_tee "WARNING: printenv not on PATH so cannot collect certain important environment variables in $ENVIRONMENT_LOG"
else
    echo "# Look for the following subset of environment variables:
    # Java-related,
    # non-Java compilers,
    # compiler flags,
    # PATH-related variables,
    # current directory, and
    # PLplot-related." > "$ENVIRONMENT_LOG"
    for ENVVAR in "$(printenv |grep -E '^JAVA.*=|^ADA=|^CC=|^CXX=|^DC=|^FC=|^.*FLAGS=|^PATH=|^CMAKE_.*PATH=|^PKG_CONFIG_PATH=|^LD_LIBRARY_PATH=|^PWD=|PLPLOT' | sort -u)"; do
	echo "${ENVVAR}" | sed -e 's?=?="?' -e 's?\(.*\)$?\1"?' >> "$ENVIRONMENT_LOG"
    done
fi

test_types=
if [ "$do_test_interactive" = "yes" ] ; then
    test_types="${test_types} interactive"
fi

if [ "$do_test_noninteractive" = "yes" -o "$do_ctest" = "yes" ] ; then
    test_types="${test_types} noninteractive"
fi

for test_type in ${test_types} ; do
    # Shared + dynamic
    if [ "$do_shared" = "yes" ] ; then
	OUTPUT_TREE="$prefix/shared/$test_type/output_tree"
	BUILD_TREE="$prefix/shared/$test_type/build_tree"
	INSTALL_TREE="$prefix/shared/$test_type/install_tree"
	INSTALL_BUILD_TREE="$prefix/shared/$test_type/install_build_tree"
	if [ "$do_submit_dashboard" = "yes" ] ; then
	    DASHBOARD_LABEL_OPTION="-DPLPLOT_BUILDNAME_SUFFIX:STRING=-(shared library + dynamic devices)"
	    INSTALLED_DASHBOARD_LABEL_OPTION="-DPLPLOT_BUILDNAME_SUFFIX:STRING=-(installed shared library + dynamic devices)"
	else
	    DASHBOARD_LABEL_OPTION=
	    INSTALLED_DASHBOARD_LABEL_OPTION=
	fi
	comprehensive_test "-DBUILD_SHARED_LIBS=ON" $test_type
    fi

    # Shared + nondynamic
    if [ "$do_nondynamic" = "yes" ] ; then
	OUTPUT_TREE="$prefix/nondynamic/$test_type/output_tree"
	BUILD_TREE="$prefix/nondynamic/$test_type/build_tree"
	INSTALL_TREE="$prefix/nondynamic/$test_type/install_tree"
	INSTALL_BUILD_TREE="$prefix/nondynamic/$test_type/install_build_tree"
	if [ "$do_submit_dashboard" = "yes" ] ; then
	    DASHBOARD_LABEL_OPTION="-DPLPLOT_BUILDNAME_SUFFIX:STRING=-(shared library + nondynamic devices)"
	    INSTALLED_DASHBOARD_LABEL_OPTION="-DPLPLOT_BUILDNAME_SUFFIX:STRING=-(installed shared library + nondynamic devices)"
	else
	    DASHBOARD_LABEL_OPTION=
	    INSTALLED_DASHBOARD_LABEL_OPTION=
	fi
	comprehensive_test "-DBUILD_SHARED_LIBS=ON -DENABLE_DYNDRIVERS=OFF" $test_type
    fi

    # Static + nondynamic
    if [ "$do_static" = "yes" ] ; then
	OUTPUT_TREE="$prefix/static/$test_type/output_tree"
	BUILD_TREE="$prefix/static/$test_type/build_tree"
	INSTALL_TREE="$prefix/static/$test_type/install_tree"
	INSTALL_BUILD_TREE="$prefix/static/$test_type/install_build_tree"
	if [ "$do_submit_dashboard" = "yes" ] ; then
	    DASHBOARD_LABEL_OPTION="-DPLPLOT_BUILDNAME_SUFFIX:STRING=-(static library + nondynamic devices)"
	    INSTALLED_DASHBOARD_LABEL_OPTION="-DPLPLOT_BUILDNAME_SUFFIX:STRING=-(installed static library + nondynamic devices)"
	else
	    DASHBOARD_LABEL_OPTION=
	    INSTALLED_DASHBOARD_LABEL_OPTION=
	fi
	comprehensive_test "-DBUILD_SHARED_LIBS=OFF" $test_type
    fi
done

collect_exit 0
