# Copyright 1999 by Jeffrey Chang.  All rights reserved.
# This code is part of the Biopython distribution and governed by its
# license.  Please see the LICENSE file that should have been included
# as part of this package.

import os
import unittest

import warnings
from Bio import BiopythonDeprecationWarning
with warnings.catch_warnings():
    warnings.simplefilter('ignore', BiopythonDeprecationWarning)
    from Bio.Blast import NCBIStandalone


class TestBlastRecord(unittest.TestCase):

    # - incredibly incomplete. Just testing what I'm adding -- Brad.

    def setUp(self):
        self.parser = NCBIStandalone.BlastParser()

    def test_conversion(self):
        "Test converting a Blast record multiple alignment"

        from Bio.Alphabet import IUPAC

        path = os.path.join("Blast", 'text_2010L_blastp_006.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        generic_align = record.multiple_alignment.to_generic(IUPAC.protein)
        test_seq = generic_align[0].seq
        self.assertEqual(test_seq.alphabet, IUPAC.protein)
        self.assertEqual(str(test_seq[:60]), record.multiple_alignment.alignment[0][2])


class TestNCBITextParser(unittest.TestCase):

    reference = 'Altschul, Stephen F., Thomas L. Madden, Alejandro A. Schaffer, \nJinghui Zhang, Zheng Zhang, Webb Miller, and David J. Lipman (1997), \n"Gapped BLAST and PSI-BLAST: a new generation of protein database search\nprograms",  Nucleic Acids Res. 25:3389-3402.'

    def setUp(self):
        self.parser = NCBIStandalone.BlastParser()
        self.pb_parser = NCBIStandalone.PSIBlastParser()

    def test_text_2010L_blastp_001(self):
        "Test parsing BLASTP 2.0.10 output (text_2010L_blastp_001)"

        path = os.path.join('Blast', 'text_2010L_blastp_001.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|120291|sp|P21297|FLBT_CAUCR FLBT PROTEIN")
        self.assertEqual(record.query_letters, 141)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 3)
        self.assertEqual(record.descriptions[0].title, "gi|120291|sp|P21297|FLBT_CAUCR FLBT PROTEIN")
        self.assertEqual(record.descriptions[0].score, 284)
        self.assertAlmostEqual(record.descriptions[0].e, 7.e-77)
        self.assertEqual(record.descriptions[1].title, "gi|3024946|sp|Q58368|Y958_METJA HYPOTHETICAL PROTEIN MJ0958 PRE...")
        self.assertEqual(record.descriptions[1].score, 29)
        self.assertAlmostEqual(record.descriptions[1].e, 3.4)
        self.assertEqual(record.descriptions[2].title, "gi|3024745|sp|O26320|THSA_METTH PROBABLE THERMOSOME SUBUNIT A")
        self.assertEqual(record.descriptions[2].score, 29)
        self.assertAlmostEqual(record.descriptions[2].e, 4.5)
        self.assertEqual(len(record.alignments), 3)
        self.assertEqual(record.alignments[0].title, ">gi|120291|sp|P21297|FLBT_CAUCR FLBT PROTEIN")
        self.assertEqual(record.alignments[0].length, 141)
        self.assertEqual(record.alignments[1].title, ">gi|3024946|sp|Q58368|Y958_METJA HYPOTHETICAL PROTEIN MJ0958 PRECURSOR")
        self.assertEqual(record.alignments[1].length, 426)
        self.assertEqual(record.alignments[2].title, ">gi|3024745|sp|O26320|THSA_METTH PROBABLE THERMOSOME SUBUNIT A")
        self.assertEqual(record.alignments[2].length, 542)
        self.assertEqual(record.alignments[0].hsps[0].score, 718)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 284)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 7e-77)
        self.assertEqual(record.alignments[0].hsps[0].identities, (141, 141))
        self.assertEqual(record.alignments[0].hsps[0].positives, (141, 141))
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 64)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 3.4)
        self.assertEqual(record.alignments[1].hsps[0].identities, (15, 47))
        self.assertEqual(record.alignments[1].hsps[0].positives, (23, 47))
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 63)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 29.0)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 4.5)
        self.assertEqual(record.alignments[2].hsps[0].identities, (31, 107))
        self.assertEqual(record.alignments[2].hsps[0].positives, (46, 107))
        self.assertEqual(record.alignments[2].hsps[0].gaps, (9, 107))
        self.assertEqual(len(record.alignments[2].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].query, "MPLKLSLKPGEKFVLNGAVVQNGDRRGVLVLQNKASVLREKDIMQPDQVTTPARHIYFPVMMMYLDEVGAEKFYEEFATRLNEFMGVVRNPVVLQDCIAISKHVMAREYYKALMLSRKLIEYEDERLGHVSSGVSAGGDAG")
        self.assertEqual(record.alignments[0].hsps[0].match, "MPLKLSLKPGEKFVLNGAVVQNGDRRGVLVLQNKASVLREKDIMQPDQVTTPARHIYFPVMMMYLDEVGAEKFYEEFATRLNEFMGVVRNPVVLQDCIAISKHVMAREYYKALMLSRKLIEYEDERLGHVSSGVSAGGDAG")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "MPLKLSLKPGEKFVLNGAVVQNGDRRGVLVLQNKASVLREKDIMQPDQVTTPARHIYFPVMMMYLDEVGAEKFYEEFATRLNEFMGVVRNPVVLQDCIAISKHVMAREYYKALMLSRKLIEYEDERLGHVSSGVSAGGDAG")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 141)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 141)
        self.assertEqual(record.alignments[1].hsps[0].query, "VLVLQNKASVLREKDIMQPDQVTTPARHIYFPVMMMYLDEVGAEKFY")
        self.assertEqual(record.alignments[1].hsps[0].match, "+LVL N  ++   K     D  TT   +IY P+ +    +  A+KFY")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "ILVLINNTNITELKKFEDDDYYTTFQHYIYQPIFIFTTYDSKAKKFY")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 28)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 74)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 169)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 215)
        self.assertEqual(record.alignments[2].hsps[0].query, "KASVLREKDIMQPDQVTTPARHIYFPVMMMYLDEVGAEKFYEEFATRLNEFMGVVRNPVVLQDCIAISKHVMAREYYKALMLSRKLIEYEDERLGHVSSGVSAGGDA")
        self.assertEqual(record.alignments[2].hsps[0].match, "+A V+ EK I   D+V T       P  +  +     +   EE    L + +GVV +   L+D     + V+A      + ++RKL EY D   G     VSA  DA")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "EAGVIYEKKIF--DEVLTFIEECRDPKAISIILRGSTKHVAEEMERALEDAIGVVAS--TLED-----REVVAGGGAPEVEIARKLREYADTISGREQLAVSAFADA")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 34)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 140)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 339)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 436)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Nov 15, 1999  2:55 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.321)
        self.assertAlmostEqual(record.ka_params[1], 0.138)
        self.assertAlmostEqual(record.ka_params[2], 0.390)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 7927717)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 284596)
        self.assertEqual(record.num_good_extends, 567)
        self.assertEqual(record.num_seqs_better_e, 3)
        self.assertEqual(record.hsps_no_gap, 2)
        self.assertEqual(record.hsps_prelim_gapped, 1)
        self.assertEqual(record.hsps_gapped, 3)
        self.assertEqual(record.query_length, 141)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 50)
        self.assertEqual(record.effective_query_length, 91)
        self.assertEqual(record.effective_database_length, 25539661)
        self.assertEqual(record.effective_search_space, 2324109151)
        self.assertEqual(record.effective_search_space_used, 2324109151)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.4)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.8)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 61)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.2)

    def test_text_2010L_blastp_002(self):
        "Test parsing BLASTP 2.0.10 output without hits (text_2010L_blastp_002)"

        path = os.path.join('Blast', 'text_2010L_blastp_002.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|400206|sp|Q02112|LYTA_BACSU MEMBRANE-BOUND PROTEIN LYTA\nPRECURSOR")
        self.assertEqual(record.query_letters, 102)
        self.assertEqual(record.database, "data/pdbaa")
        self.assertEqual(record.database_sequences, 6999)
        self.assertEqual(record.database_letters, 1461753)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/pdbaa'])
        self.assertEqual(record.num_letters_in_database, [1461753])
        self.assertEqual(record.num_sequences_in_database, [6999])
        self.assertEqual(record.posted_date, [('Oct 11, 1999 11:30 AM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.310)
        self.assertAlmostEqual(record.ka_params[1], 0.132)
        self.assertAlmostEqual(record.ka_params[2], 0.353)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 289135)
        self.assertEqual(record.num_sequences, 6999)
        self.assertEqual(record.num_extends, 10742)
        self.assertEqual(record.num_good_extends, 10)
        self.assertEqual(record.num_seqs_better_e, 0)
        self.assertEqual(record.hsps_no_gap, 0)
        self.assertEqual(record.hsps_prelim_gapped, 0)
        self.assertEqual(record.hsps_gapped, 0)
        self.assertEqual(record.query_length, 102)
        self.assertEqual(record.database_length, 1461753)
        self.assertEqual(record.effective_hsp_length, 45)
        self.assertEqual(record.effective_query_length, 57)
        self.assertEqual(record.effective_database_length, 1146798)
        self.assertEqual(record.effective_search_space, 65367486)
        self.assertEqual(record.effective_search_space_used, 65367486)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.1)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 42)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 47)
        self.assertAlmostEqual(record.blast_cutoff[1], 22.7)

    def test_text_2010L_blastp_004(self):
        "Test parsing BLASTP 2.0.10 output without descriptions (text_2010L_blastp_004)"

        path = os.path.join('Blast', 'text_2010L_blastp_004.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1718062|sp|P16153|UTXA_CLODI UTXA PROTEIN")
        self.assertEqual(record.query_letters, 166)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 6)
        self.assertEqual(record.alignments[0].title, ">gi|1718062|sp|P16153|UTXA_CLODI UTXA PROTEIN")
        self.assertEqual(record.alignments[0].length, 166)
        self.assertEqual(record.alignments[1].title, ">gi|140528|sp|P24811|YQXH_BACSU HYPOTHETICAL 15.7 KD PROTEIN IN SPOIIIC-CWLA INTERGENIC REGION (ORF2)")
        self.assertEqual(record.alignments[1].length, 140)
        self.assertEqual(record.alignments[2].title, ">gi|141088|sp|P26835|YNGD_CLOPE HYPOTHETICAL 14.9 KD PROTEIN IN NAGH 3'REGION (ORFD)")
        self.assertEqual(record.alignments[2].length, 132)
        self.assertEqual(record.alignments[3].title, ">gi|6014830|sp|O78935|CYB_MARAM CYTOCHROME B")
        self.assertEqual(record.alignments[3].length, 379)
        self.assertEqual(record.alignments[4].title, ">gi|1351589|sp|P47694|Y456_MYCGE HYPOTHETICAL PROTEIN MG456")
        self.assertEqual(record.alignments[4].length, 334)
        self.assertEqual(record.alignments[5].title, ">gi|2496246|sp|Q57881|Y439_METJA HYPOTHETICAL ATP-BINDING PROTEIN MJ0439")
        self.assertEqual(record.alignments[5].length, 361)
        self.assertEqual(record.alignments[0].hsps[0].score, 843)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 332)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 2e-91)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 90)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 39.5)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 0.004)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 88)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 38.7)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 0.007)
        self.assertEqual(len(record.alignments[2].hsps), 1)
        self.assertEqual(record.alignments[3].hsps[0].score, 64)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 4.6)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[4].hsps[0].score, 63)
        self.assertAlmostEqual(record.alignments[4].hsps[0].bits, 29.0)
        self.assertAlmostEqual(record.alignments[4].hsps[0].expect, 6.0)
        self.assertEqual(len(record.alignments[4].hsps), 1)
        self.assertEqual(record.alignments[5].hsps[0].score, 62)
        self.assertAlmostEqual(record.alignments[5].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.alignments[5].hsps[0].expect, 7.8)
        self.assertEqual(len(record.alignments[5].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (166, 166))
        self.assertEqual(record.alignments[0].hsps[0].positives, (166, 166))
        self.assertEqual(record.alignments[1].hsps[0].identities, (27, 130))
        self.assertEqual(record.alignments[1].hsps[0].positives, (55, 130))
        self.assertEqual(record.alignments[1].hsps[0].gaps, (19, 130))
        self.assertEqual(record.alignments[2].hsps[0].identities, (24, 110))
        self.assertEqual(record.alignments[2].hsps[0].positives, (52, 110))
        self.assertEqual(record.alignments[2].hsps[0].gaps, (18, 110))
        self.assertEqual(record.alignments[3].hsps[0].identities, (19, 57))
        self.assertEqual(record.alignments[3].hsps[0].positives, (33, 57))
        self.assertEqual(record.alignments[3].hsps[0].gaps, (2, 57))
        self.assertEqual(record.alignments[4].hsps[0].identities, (16, 44))
        self.assertEqual(record.alignments[4].hsps[0].positives, (24, 44))
        self.assertEqual(record.alignments[4].hsps[0].gaps, (2, 44))
        self.assertEqual(record.alignments[5].hsps[0].identities, (19, 56))
        self.assertEqual(record.alignments[5].hsps[0].positives, (30, 56))
        self.assertEqual(record.alignments[5].hsps[0].gaps, (12, 56))
        self.assertEqual(record.alignments[0].hsps[0].query, "MHSSSPFYISNGNKIFFYINLGGVMNMTISFLSEHIFIKLVILTISFDTLLGCLSAIKSRKFNSSFGIDGGIRKVAMIACIFFLSVVDILTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKNMCLCGLPVPKRLKEKIAILLDAMTDEMNAKDEK")
        self.assertEqual(record.alignments[0].hsps[0].match, "MHSSSPFYISNGNKIFFYINLGGVMNMTISFLSEHIFIKLVILTISFDTLLGCLSAIKSRKFNSSFGIDGGIRKVAMIACIFFLSVVDILTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKNMCLCGLPVPKRLKEKIAILLDAMTDEMNAKDEK")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "MHSSSPFYISNGNKIFFYINLGGVMNMTISFLSEHIFIKLVILTISFDTLLGCLSAIKSRKFNSSFGIDGGIRKVAMIACIFFLSVVDILTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKNMCLCGLPVPKRLKEKIAILLDAMTDEMNAKDEK")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 166)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 166)
        self.assertEqual(record.alignments[1].hsps[0].query, "FIKLVILTISFDTLLGCLSAIKSRKFNSSFGIDGGIRKVAMIACIFFLSVVDILTKFNFLFMLPQDCINFLRLKHLGISEFFSILF-ILYESVSILKNMCLCGLPVPKRLKEKIAILLDAMTDEMNAKDE")
        self.assertEqual(record.alignments[1].hsps[0].match, "++ L+++    D L G + A K +K  S     G +RK+     +   +V+D +   N                  G+  F ++LF I  E +SI +N+   G+ +P  + +++  + +      N  D+")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "YLDLLLVLSIIDVLTGVIKAWKFKKLRSRSAWFGYVRKLLNFFAVILANVIDTVLNLN------------------GVLTFGTVLFYIANEGLSITENLAQIGVKIPSSITDRLQTIENEKEQSKNNADK")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 37)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 165)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 26)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 137)
        self.assertEqual(record.alignments[2].hsps[0].query, "VILTISFDTLLGCLSAIKSRKFNSSFGIDGGIRKVAMIACIFFLSVVD-ILTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKNMCLCGLPVPKRLKEKI")
        self.assertEqual(record.alignments[2].hsps[0].match, "+++ I  D L G +   KS++  S+ G+ G  +K  ++  +    ++D +L    ++F                     +  +I+ E +SIL+N    G+P+P++LK+ +")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "LLVFIFLDYLTGVIKGCKSKELCSNIGLRGITKKGLILVVLLVAVMLDRLLDNGTWMFRT-----------------LIAYFYIMNEGISILENCAALGVPIPEKLKQAL")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 41)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 149)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 33)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 125)
        self.assertEqual(record.alignments[3].hsps[0].query, "CIFFLSVVDILTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKNMCL")
        self.assertEqual(record.alignments[3].hsps[0].match, "C+F+L V D+LT   ++   P +   F+ +  L    +F+IL IL  ++SI++N  L")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "CLFWLLVADLLT-LTWIGGQPVEH-PFITIGQLASILYFAILLILMPAISIIENNLL")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 80)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 136)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 323)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 377)
        self.assertEqual(record.alignments[4].hsps[0].query, "LTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKN")
        self.assertEqual(record.alignments[4].hsps[0].match, "LTKFN  F+ P     FLR+  +G+   FS++ I +   S  +N")
        self.assertEqual(record.alignments[4].hsps[0].sbjct, "LTKFNKFFLTPNKLNAFLRV--IGLCGLFSVIAISFGIYSYTRN")
        self.assertEqual(record.alignments[4].hsps[0].query_start, 90)
        self.assertEqual(record.alignments[4].hsps[0].query_end, 133)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_start, 4)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_end, 45)
        self.assertEqual(record.alignments[5].hsps[0].query, "FLRLKHLGIS---EFFSILFILYES----VSILKNMC-----LCGLPVPKRLKEKI")
        self.assertEqual(record.alignments[5].hsps[0].match, "++ L+ + IS   +F  +LF  YE     V I+K++      LCG+P PK   E+I")
        self.assertEqual(record.alignments[5].hsps[0].sbjct, "YINLRGIFISKYKDFIEVLFEEYEEDRKPVEIIKSLIKDVPSLCGIPTPKNTLEEI")
        self.assertEqual(record.alignments[5].hsps[0].query_start, 106)
        self.assertEqual(record.alignments[5].hsps[0].query_end, 149)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_start, 68)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_end, 123)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Nov 15, 1999  2:55 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.331)
        self.assertAlmostEqual(record.ka_params[1], 0.146)
        self.assertAlmostEqual(record.ka_params[2], 0.428)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 8801581)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 320828)
        self.assertEqual(record.num_good_extends, 892)
        self.assertEqual(record.num_seqs_better_e, 6)
        self.assertEqual(record.hsps_no_gap, 3)
        self.assertEqual(record.hsps_prelim_gapped, 3)
        self.assertEqual(record.hsps_gapped, 6)
        self.assertEqual(record.query_length, 166)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 46)
        self.assertEqual(record.effective_query_length, 120)
        self.assertEqual(record.effective_database_length, 25868693)
        self.assertEqual(record.effective_search_space, 3104243160)
        self.assertEqual(record.effective_search_space_used, 3104243160)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 15)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.2)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 40)
        self.assertAlmostEqual(record.gap_trigger[1], 21.9)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 62)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.6)

    def test_text_2010L_blastp_005(self):
        "Test parsing BLASTP 2.0.10 output without alignments (text_2010L_blastp_005)"

        path = os.path.join('Blast', 'text_2010L_blastp_005.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1718062|sp|P16153|UTXA_CLODI UTXA PROTEIN")
        self.assertEqual(record.query_letters, 166)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 6)
        self.assertEqual(record.descriptions[0].title, "gi|1718062|sp|P16153|UTXA_CLODI UTXA PROTEIN")
        self.assertEqual(record.descriptions[0].score, 332)
        self.assertAlmostEqual(record.descriptions[0].e, 2.e-91)
        self.assertEqual(record.descriptions[1].title, "gi|140528|sp|P24811|YQXH_BACSU HYPOTHETICAL 15.7 KD PROTEIN IN ...")
        self.assertEqual(record.descriptions[1].score, 39)
        self.assertAlmostEqual(record.descriptions[1].e, 0.004)
        self.assertEqual(record.descriptions[2].title, "gi|141088|sp|P26835|YNGD_CLOPE HYPOTHETICAL 14.9 KD PROTEIN IN ...")
        self.assertEqual(record.descriptions[2].score, 39)
        self.assertAlmostEqual(record.descriptions[2].e, 0.007)
        self.assertEqual(record.descriptions[3].title, "gi|6014830|sp|O78935|CYB_MARAM CYTOCHROME B")
        self.assertEqual(record.descriptions[3].score, 29)
        self.assertAlmostEqual(record.descriptions[3].e, 4.6)
        self.assertEqual(record.descriptions[4].title, "gi|1351589|sp|P47694|Y456_MYCGE HYPOTHETICAL PROTEIN MG456")
        self.assertEqual(record.descriptions[4].score, 29)
        self.assertAlmostEqual(record.descriptions[4].e, 6.0)
        self.assertEqual(record.descriptions[5].title, "gi|2496246|sp|Q57881|Y439_METJA HYPOTHETICAL ATP-BINDING PROTEI...")
        self.assertEqual(record.descriptions[5].score, 29)
        self.assertAlmostEqual(record.descriptions[5].e, 7.8)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Nov 15, 1999  2:55 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.331)
        self.assertAlmostEqual(record.ka_params[1], 0.146)
        self.assertAlmostEqual(record.ka_params[2], 0.428)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 8801581)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 320828)
        self.assertEqual(record.num_good_extends, 892)
        self.assertEqual(record.num_seqs_better_e, 6)
        self.assertEqual(record.hsps_no_gap, 3)
        self.assertEqual(record.hsps_prelim_gapped, 3)
        self.assertEqual(record.hsps_gapped, 6)
        self.assertEqual(record.query_length, 166)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 46)
        self.assertEqual(record.effective_query_length, 120)
        self.assertEqual(record.effective_database_length, 25868693)
        self.assertEqual(record.effective_search_space, 3104243160)
        self.assertEqual(record.effective_search_space_used, 3104243160)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 15)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.2)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 40)
        self.assertAlmostEqual(record.gap_trigger[1], 21.9)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 62)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.6)

    def test_text_2010L_blastp_006(self):
        "Test parsing BLASTP 2.0.10 output (text_2010L_blastp_006)"

        path = os.path.join('Blast', 'text_2010L_blastp_006.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|132349|sp|P15394|REPA_AGRTU REPLICATING PROTEIN")
        self.assertEqual(record.query_letters, 250)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 15)
        self.assertEqual(record.descriptions[0].title, "gi|132349|sp|P15394|REPA_AGRTU REPLICATING PROTEIN")
        self.assertEqual(record.descriptions[0].score, 514)
        self.assertAlmostEqual(record.descriptions[0].e, 1.e-146)
        self.assertEqual(record.descriptions[1].title, "gi|123932|sp|P19922|HYIN_BRAJA INDOLEACETAMIDE HYDROLASE (IAH) ...")
        self.assertEqual(record.descriptions[1].score, 34)
        self.assertAlmostEqual(record.descriptions[1].e, 0.29)
        self.assertEqual(record.descriptions[2].title, "gi|137670|sp|P06422|VE2_HPV08 REGULATORY PROTEIN E2")
        self.assertEqual(record.descriptions[2].score, 32)
        self.assertAlmostEqual(record.descriptions[2].e, 0.86)
        self.assertEqual(record.descriptions[3].title, "gi|5921693|sp|Q05152|CCAB_RABIT VOLTAGE-DEPENDENT N-TYPE CALCIU...")
        self.assertEqual(record.descriptions[3].score, 32)
        self.assertAlmostEqual(record.descriptions[3].e, 1.5)
        self.assertEqual(record.descriptions[4].title, "gi|121952|sp|P02256|H1_PARAN HISTONE H1, GONADAL")
        self.assertEqual(record.descriptions[4].score, 31)
        self.assertAlmostEqual(record.descriptions[4].e, 2.5)
        self.assertEqual(record.descriptions[5].title, "gi|124141|sp|P08392|ICP4_HSV11 TRANS-ACTING TRANSCRIPTIONAL PRO...")
        self.assertEqual(record.descriptions[5].score, 31)
        self.assertAlmostEqual(record.descriptions[5].e, 3.3)
        self.assertEqual(record.descriptions[6].title, "gi|3915729|sp|P51592|HYDP_DROME HYPERPLASTIC DISCS PROTEIN (HYD...")
        self.assertEqual(record.descriptions[6].score, 31)
        self.assertAlmostEqual(record.descriptions[6].e, 3.3)
        self.assertEqual(record.descriptions[7].title, "gi|462182|sp|P33438|GLT_DROME GLUTACTIN PRECURSOR")
        self.assertEqual(record.descriptions[7].score, 31)
        self.assertAlmostEqual(record.descriptions[7].e, 3.3)
        self.assertEqual(record.descriptions[8].title, "gi|731294|sp|P39713|YAG1_YEAST HYPOTHETICAL ZINC-TYPE ALCOHOL D...")
        self.assertEqual(record.descriptions[8].score, 30)
        self.assertAlmostEqual(record.descriptions[8].e, 4.3)
        self.assertEqual(record.descriptions[9].title, "gi|1708851|sp|P55268|LMB2_HUMAN LAMININ BETA-2 CHAIN PRECURSOR ...")
        self.assertEqual(record.descriptions[9].score, 30)
        self.assertAlmostEqual(record.descriptions[9].e, 4.3)
        self.assertEqual(record.descriptions[10].title, "gi|2495137|sp|Q24704|H1_DROVI HISTONE H1")
        self.assertEqual(record.descriptions[10].score, 29)
        self.assertAlmostEqual(record.descriptions[10].e, 7.5)
        self.assertEqual(record.descriptions[11].title, "gi|1172635|sp|P46466|PRS4_ORYSA 26S PROTEASE REGULATORY SUBUNIT...")
        self.assertEqual(record.descriptions[11].score, 29)
        self.assertAlmostEqual(record.descriptions[11].e, 9.8)
        self.assertEqual(record.descriptions[12].title, "gi|6093970|sp|Q61085|RHOP_MOUSE GTP-RHO BINDING PROTEIN 1 (RHOP...")
        self.assertEqual(record.descriptions[12].score, 29)
        self.assertAlmostEqual(record.descriptions[12].e, 9.8)
        self.assertEqual(record.descriptions[13].title, "gi|547963|sp|Q01989|MYS9_DROME MYOSIN HEAVY CHAIN 95F (95F MHC)")
        self.assertEqual(record.descriptions[13].score, 29)
        self.assertAlmostEqual(record.descriptions[13].e, 9.8)
        self.assertEqual(record.descriptions[14].title, "gi|6226905|sp|Q59567|TOP1_MYCTU DNA TOPOISOMERASE I (OMEGA-PROT...")
        self.assertEqual(record.descriptions[14].score, 29)
        self.assertAlmostEqual(record.descriptions[14].e, 9.8)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Nov 15, 1999  2:55 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.317)
        self.assertAlmostEqual(record.ka_params[1], 0.133)
        self.assertAlmostEqual(record.ka_params[2], 0.395)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 14679054)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 599117)
        self.assertEqual(record.num_good_extends, 1508)
        self.assertEqual(record.num_seqs_better_e, 15)
        self.assertEqual(record.hsps_no_gap, 4)
        self.assertEqual(record.hsps_prelim_gapped, 11)
        self.assertEqual(record.hsps_gapped, 15)
        self.assertEqual(record.query_length, 250)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 51)
        self.assertEqual(record.effective_query_length, 199)
        self.assertEqual(record.effective_database_length, 25457403)
        self.assertEqual(record.effective_search_space, 5066023197)
        self.assertEqual(record.effective_search_space_used, 5066023197)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 63)
        self.assertAlmostEqual(record.blast_cutoff[1], 29.0)

    def test_text_2010L_phiblast_001(self):
        "Test parsing PHI-BLAST, BLASTP 2.0.10 output, one round (text_2010L_phiblast_001)"

        path = os.path.join('Blast', 'text_2010L_phiblast_001.txt')
        handle = open(path)
        record = self.pb_parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1174726|sp|P12921|TMRB_BACSU TUNICAMYCIN RESISTANCE PROTEIN")
        self.assertEqual(record.query_letters, 197)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.rounds), 1)
        self.assertEqual(len(record.rounds[0].new_seqs), 4)
        self.assertEqual(record.rounds[0].new_seqs[0].title, "gi|1174726|sp|P12921|TMRB_BACSU TUNICAMYCIN RESISTANCE PROTEIN")
        self.assertEqual(record.rounds[0].new_seqs[0].score, 402)
        self.assertAlmostEqual(record.rounds[0].new_seqs[0].e, 1.e-112)
        self.assertEqual(record.rounds[0].new_seqs[1].title, "gi|1352934|sp|P47169|YJ9F_YEAST HYPOTHETICAL 161.2 KD PROTEIN I...")
        self.assertEqual(record.rounds[0].new_seqs[1].score, 30)
        self.assertAlmostEqual(record.rounds[0].new_seqs[1].e, 3.3)
        self.assertEqual(record.rounds[0].new_seqs[2].title, "gi|3915741|sp|P04407|KITH_HSV23 THYMIDINE KINASE")
        self.assertEqual(record.rounds[0].new_seqs[2].score, 29)
        self.assertAlmostEqual(record.rounds[0].new_seqs[2].e, 7.4)
        self.assertEqual(record.rounds[0].new_seqs[3].title, "gi|3334489|sp|P15398|RPA1_SCHPO DNA-DIRECTED RNA POLYMERASE I 1...")
        self.assertEqual(record.rounds[0].new_seqs[3].score, 29)
        self.assertAlmostEqual(record.rounds[0].new_seqs[3].e, 7.4)
        self.assertEqual(len(record.rounds[0].alignments), 4)
        self.assertEqual(record.rounds[0].alignments[0].title, ">gi|1174726|sp|P12921|TMRB_BACSU TUNICAMYCIN RESISTANCE PROTEIN")
        self.assertEqual(record.rounds[0].alignments[0].length, 197)
        self.assertEqual(record.rounds[0].alignments[1].title, ">gi|1352934|sp|P47169|YJ9F_YEAST HYPOTHETICAL 161.2 KD PROTEIN IN NMD5-HOM6 INTERGENIC REGION")
        self.assertEqual(record.rounds[0].alignments[1].length, 1442)
        self.assertEqual(record.rounds[0].alignments[2].title, ">gi|3915741|sp|P04407|KITH_HSV23 THYMIDINE KINASE")
        self.assertEqual(record.rounds[0].alignments[2].length, 376)
        self.assertEqual(record.rounds[0].alignments[3].title, ">gi|3334489|sp|P15398|RPA1_SCHPO DNA-DIRECTED RNA POLYMERASE I 190 KD POLYPEPTIDE")
        self.assertEqual(record.rounds[0].alignments[3].length, 1689)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].score, 1021)
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].bits, 402)
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].expect, 1e-112)
        self.assertEqual(len(record.rounds[0].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].score, 66)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].bits, 30.1)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].expect, 3.3)
        self.assertEqual(len(record.rounds[0].alignments[1].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].score, 63)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].bits, 29.0)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].expect, 7.4)
        self.assertEqual(len(record.rounds[0].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].score, 63)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].bits, 29.0)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].expect, 7.4)
        self.assertEqual(len(record.rounds[0].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].identities, (197, 197))
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].positives, (197, 197))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].identities, (23, 70))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].positives, (35, 70))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].gaps, (10, 70))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].identities, (15, 37))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].positives, (22, 37))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].gaps, (2, 37))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].identities, (12, 38))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].positives, (20, 38))
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query, "MIIWINGAFGSGKTQTAFELHRRLNPSYVYDPEKMGFALRSMVPQEIAKDDFQSYPLWRAFNYSLLASLTDTYRGILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASKETLLKRLRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARAELPLDPDTRGSLRRFADRLMVKLNHIRIK")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].match, "MIIWINGAFGSGKTQTAFELHRRLNPSYVYDPEKMGFALRSMVPQEIAKDDFQSYPLWRAFNYSLLASLTDTYRGILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASKETLLKRLRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARAELPLDPDTRGSLRRFADRLMVKLNHIRIK")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct, "MIIWINGAFGSGKTQTAFELHRRLNPSYVYDPEKMGFALRSMVPQEIAKDDFQSYPLWRAFNYSLLASLTDTYRGILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASKETLLKRLRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARAELPLDPDTRGSLRRFADRLMVKLNHIRIK")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query_end, 197)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct_end, 197)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query, "TLMASKETLLKR--------LRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARA")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].match, "TL+  K+  L R          TR + K S AA   D+ +EGLS P    +  +D  +  ++A+ +AARA")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct, "TLLTRKDPSLSRNLKQSAGDALTRKQEKRSKAA--FDQLLEGLSGPPLHVYYASDGGNAANLAKRLAARA")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query_start, 107)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query_end, 168)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct_start, 637)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct_end, 704)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query, "IWINGAFGSGKTQTAFELHRRLNP--SYVYDPEKMGF")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].match, "++I+G  G GKT T+ +L   L P  + VY PE M +")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct, "VYIDGPHGVGKTTTSAQLMEALGPRDNIVYVPEPMTY")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query_end, 37)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct_start, 52)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct_end, 88)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query, "GILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASK")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].match, "G +++P+   HP +F+++   LR      HHF L   K")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct, "GHIVLPIPAYHPLFFSQMYNLLRSTCLYCHHFKLSKVK")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query_start, 75)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query_end, 112)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct_start, 78)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct_end, 115)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Nov 15, 1999  2:55 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.323)
        self.assertAlmostEqual(record.ka_params[1], 0.138)
        self.assertAlmostEqual(record.ka_params[2], 0.411)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 11332752)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 435116)
        self.assertEqual(record.num_good_extends, 987)
        self.assertEqual(record.num_seqs_better_e, 4)
        self.assertEqual(record.hsps_no_gap, 2)
        self.assertEqual(record.hsps_prelim_gapped, 2)
        self.assertEqual(record.hsps_gapped, 4)
        self.assertEqual(record.query_length, 197)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 48)
        self.assertEqual(record.effective_query_length, 149)
        self.assertEqual(record.effective_database_length, 25704177)
        self.assertEqual(record.effective_search_space, 3829922373)
        self.assertEqual(record.effective_search_space_used, 3829922373)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.5)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 22.0)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 62)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.6)

    def test_text_2010L_phiblast_002(self):
        "Test parsing PHI-BLAST, BLASTP 2.0.10 output, three rounds (text_2010L_phiblast_002)"

        path = os.path.join('Blast', 'text_2010L_phiblast_002.txt')
        handle = open(path)
        record = self.pb_parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|126343|sp|P17216|LIVK_SALTY LEUCINE-SPECIFIC BINDING PROTEIN\nPRECURSOR (LS-BP) (L-BP)")
        self.assertEqual(record.query_letters, 369)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.rounds), 3)
        self.assertEqual(len(record.rounds[0].new_seqs), 14)
        # Rest of test broken up to avoid Jython JVM limitations
        self._check_text_2010L_phiblast_002_round0(record)
        self._check_text_2010L_phiblast_002_round1(record)
        self._check_text_2010L_phiblast_002_round2(record)
        self._check_text_2010L_phiblast_002_hsps(record)
        self._check_text_2010L_phiblast_002_hsps_details(record)
        self._check_text_2010L_phiblast_002_footer(record)

    def _check_text_2010L_phiblast_002_round0(self, record):
        self.assertEqual(record.rounds[0].new_seqs[0].title, "gi|126343|sp|P17216|LIVK_SALTY LEUCINE-SPECIFIC BINDING PROTEIN...")
        self.assertEqual(record.rounds[0].new_seqs[0].score, 743)
        self.assertAlmostEqual(record.rounds[0].new_seqs[0].e, 0.0)
        self.assertEqual(record.rounds[0].new_seqs[1].title, "gi|126349|sp|P04816|LIVK_ECOLI LEUCINE-SPECIFIC BINDING PROTEIN...")
        self.assertEqual(record.rounds[0].new_seqs[1].score, 679)
        self.assertAlmostEqual(record.rounds[0].new_seqs[1].e, 0.0)
        self.assertEqual(record.rounds[0].new_seqs[2].title, "gi|126348|sp|P02917|LIVJ_ECOLI LEU/ILE/VAL-BINDING PROTEIN PREC...")
        self.assertEqual(record.rounds[0].new_seqs[2].score, 581)
        self.assertAlmostEqual(record.rounds[0].new_seqs[2].e, 1.e-166)
        self.assertEqual(record.rounds[0].new_seqs[3].title, "gi|126342|sp|P17215|LIVJ_SALTY LEU/ILE/VAL/THR-BINDING PROTEIN ...")
        self.assertEqual(record.rounds[0].new_seqs[3].score, 577)
        self.assertAlmostEqual(record.rounds[0].new_seqs[3].e, 1.e-164)
        self.assertEqual(record.rounds[0].new_seqs[4].title, "gi|126347|sp|P25399|LIVJ_CITFR LEU/ILE/VAL-BINDING PROTEIN PREC...")
        self.assertEqual(record.rounds[0].new_seqs[4].score, 570)
        self.assertAlmostEqual(record.rounds[0].new_seqs[4].e, 1.e-162)
        self.assertEqual(record.rounds[0].new_seqs[5].title, "gi|115120|sp|P21175|BRAC_PSEAE LEUCINE-, ISOLEUCINE-, VALINE-, ...")
        self.assertEqual(record.rounds[0].new_seqs[5].score, 413)
        self.assertAlmostEqual(record.rounds[0].new_seqs[5].e, 1.e-115)
        self.assertEqual(record.rounds[0].new_seqs[6].title, "gi|113709|sp|P27017|AMIC_PSEAE ALIPHATIC AMIDASE EXPRESSION-REG...")
        self.assertEqual(record.rounds[0].new_seqs[6].score, 40)
        self.assertAlmostEqual(record.rounds[0].new_seqs[6].e, 0.006)
        self.assertEqual(record.rounds[0].new_seqs[7].title, "gi|127751|sp|P02567|MYSD_CAEEL MYOSIN HEAVY CHAIN D (MHC D)")
        self.assertEqual(record.rounds[0].new_seqs[7].score, 32)
        self.assertAlmostEqual(record.rounds[0].new_seqs[7].e, 1.4)
        self.assertEqual(record.rounds[0].new_seqs[8].title, "gi|131068|sp|P23596|PRTD_ERWCH PROTEASES SECRETION ATP-BINDING ...")
        self.assertEqual(record.rounds[0].new_seqs[8].score, 32)
        self.assertAlmostEqual(record.rounds[0].new_seqs[8].e, 1.8)
        self.assertEqual(record.rounds[0].new_seqs[9].title, "gi|2495081|sp|Q09630|MGR1_CAEEL PROBABLE METABOTROPIC GLUTAMATE...")
        self.assertEqual(record.rounds[0].new_seqs[9].score, 32)
        self.assertAlmostEqual(record.rounds[0].new_seqs[9].e, 1.8)
        self.assertEqual(record.rounds[0].new_seqs[10].title, "gi|2506848|sp|P80040|MDH_CHLAU MALATE DEHYDROGENASE")
        self.assertEqual(record.rounds[0].new_seqs[10].score, 32)
        self.assertAlmostEqual(record.rounds[0].new_seqs[10].e, 1.8)
        self.assertEqual(record.rounds[0].new_seqs[11].title, "gi|3915245|sp|Q38394|VG37_BPK3 TAIL FIBER PROTEIN GP37 (RECEPTO...")
        self.assertEqual(record.rounds[0].new_seqs[11].score, 31)
        self.assertAlmostEqual(record.rounds[0].new_seqs[11].e, 4.0)
        self.assertEqual(record.rounds[0].new_seqs[12].title, "gi|1351210|sp|P47209|TCPE_CAEEL T-COMPLEX PROTEIN 1, EPSILON SU...")
        self.assertEqual(record.rounds[0].new_seqs[12].score, 31)
        self.assertAlmostEqual(record.rounds[0].new_seqs[12].e, 4.0)
        self.assertEqual(record.rounds[0].new_seqs[13].title, "gi|1351310|sp|P43496|TRXB_PENCH THIOREDOXIN REDUCTASE")
        self.assertEqual(record.rounds[0].new_seqs[13].score, 31)
        self.assertAlmostEqual(record.rounds[0].new_seqs[13].e, 5.2)
        self.assertEqual(len(record.rounds[0].alignments), 14)
        self.assertEqual(record.rounds[0].alignments[0].title, ">gi|126343|sp|P17216|LIVK_SALTY LEUCINE-SPECIFIC BINDING PROTEIN PRECURSOR (LS-BP) (L-BP)")
        self.assertEqual(record.rounds[0].alignments[0].length, 369)
        self.assertEqual(record.rounds[0].alignments[1].title, ">gi|126349|sp|P04816|LIVK_ECOLI LEUCINE-SPECIFIC BINDING PROTEIN PRECURSOR (LS-BP) (L-BP)")
        self.assertEqual(record.rounds[0].alignments[1].length, 369)
        self.assertEqual(record.rounds[0].alignments[2].title, ">gi|126348|sp|P02917|LIVJ_ECOLI LEU/ILE/VAL-BINDING PROTEIN PRECURSOR (LIV-BP)")
        self.assertEqual(record.rounds[0].alignments[2].length, 367)
        self.assertEqual(record.rounds[0].alignments[3].title, ">gi|126342|sp|P17215|LIVJ_SALTY LEU/ILE/VAL/THR-BINDING PROTEIN PRECURSOR (LIVT-BP)")
        self.assertEqual(record.rounds[0].alignments[3].length, 365)
        self.assertEqual(record.rounds[0].alignments[4].title, ">gi|126347|sp|P25399|LIVJ_CITFR LEU/ILE/VAL-BINDING PROTEIN PRECURSOR (LIV-BP)")
        self.assertEqual(record.rounds[0].alignments[4].length, 367)
        self.assertEqual(record.rounds[0].alignments[5].title, ">gi|115120|sp|P21175|BRAC_PSEAE LEUCINE-, ISOLEUCINE-, VALINE-, THREONINE-, AND ALANINE-BINDING PROTEIN PRECURSOR (LIVAT-BP) (LEU/ILE/VAL/THR/ALA-BINDING PROTEIN)")
        self.assertEqual(record.rounds[0].alignments[5].length, 373)
        self.assertEqual(record.rounds[0].alignments[6].title, ">gi|113709|sp|P27017|AMIC_PSEAE ALIPHATIC AMIDASE EXPRESSION-REGULATING PROTEIN")
        self.assertEqual(record.rounds[0].alignments[6].length, 385)
        self.assertEqual(record.rounds[0].alignments[7].title, ">gi|127751|sp|P02567|MYSD_CAEEL MYOSIN HEAVY CHAIN D (MHC D)")
        self.assertEqual(record.rounds[0].alignments[7].length, 1938)
        self.assertEqual(record.rounds[0].alignments[8].title, ">gi|131068|sp|P23596|PRTD_ERWCH PROTEASES SECRETION ATP-BINDING PROTEIN PRTD")
        self.assertEqual(record.rounds[0].alignments[8].length, 575)
        self.assertEqual(record.rounds[0].alignments[9].title, ">gi|2495081|sp|Q09630|MGR1_CAEEL PROBABLE METABOTROPIC GLUTAMATE RECEPTOR MGL-1")
        self.assertEqual(record.rounds[0].alignments[9].length, 999)
        self.assertEqual(record.rounds[0].alignments[10].title, ">gi|2506848|sp|P80040|MDH_CHLAU MALATE DEHYDROGENASE")
        self.assertEqual(record.rounds[0].alignments[10].length, 309)
        self.assertEqual(record.rounds[0].alignments[11].title, ">gi|3915245|sp|Q38394|VG37_BPK3 TAIL FIBER PROTEIN GP37 (RECEPTOR RECOGNIZING PROTEIN)")
        self.assertEqual(record.rounds[0].alignments[11].length, 1243)
        self.assertEqual(record.rounds[0].alignments[12].title, ">gi|1351210|sp|P47209|TCPE_CAEEL T-COMPLEX PROTEIN 1, EPSILON SUBUNIT (TCP-1-EPSILON) (CCT-EPSILON)")
        self.assertEqual(record.rounds[0].alignments[12].length, 542)
        self.assertEqual(record.rounds[0].alignments[13].title, ">gi|1351310|sp|P43496|TRXB_PENCH THIOREDOXIN REDUCTASE")
        self.assertEqual(record.rounds[0].alignments[13].length, 334)

    def _check_text_2010L_phiblast_002_round1(self, record):
        self.assertEqual(len(record.rounds[1].new_seqs), 18)
        self.assertEqual(record.rounds[1].new_seqs[0].title, "gi|113709|sp|P27017|AMIC_PSEAE ALIPHATIC AMIDASE EXPRESSION-REG...")
        self.assertEqual(record.rounds[1].new_seqs[0].score, 49)
        self.assertAlmostEqual(record.rounds[1].new_seqs[0].e, 2e-5)
        self.assertEqual(record.rounds[1].new_seqs[1].title, "gi|3024131|sp|P91685|MGR_DROME METABOTROPIC GLUTAMATE RECEPTOR ...")
        self.assertEqual(record.rounds[1].new_seqs[1].score, 41)
        self.assertAlmostEqual(record.rounds[1].new_seqs[1].e, 0.003)
        self.assertEqual(record.rounds[1].new_seqs[2].title, "gi|121300|sp|P23385|MGR1_RAT METABOTROPIC GLUTAMATE RECEPTOR 1 ...")
        self.assertEqual(record.rounds[1].new_seqs[2].score, 39)
        self.assertAlmostEqual(record.rounds[1].new_seqs[2].e, 0.012)
        self.assertEqual(record.rounds[1].new_seqs[3].title, "gi|2495074|sp|Q13255|MGR1_HUMAN METABOTROPIC GLUTAMATE RECEPTOR...")
        self.assertEqual(record.rounds[1].new_seqs[3].score, 39)
        self.assertAlmostEqual(record.rounds[1].new_seqs[3].e, 0.012)
        self.assertEqual(record.rounds[1].new_seqs[4].title, "gi|1170947|sp|P31424|MGR5_RAT METABOTROPIC GLUTAMATE RECEPTOR 5...")
        self.assertEqual(record.rounds[1].new_seqs[4].score, 36)
        self.assertAlmostEqual(record.rounds[1].new_seqs[4].e, 0.11)
        self.assertEqual(record.rounds[1].new_seqs[5].title, "gi|1709020|sp|P41594|MGR5_HUMAN METABOTROPIC GLUTAMATE RECEPTOR...")
        self.assertEqual(record.rounds[1].new_seqs[5].score, 36)
        self.assertAlmostEqual(record.rounds[1].new_seqs[5].e, 0.14)
        self.assertEqual(record.rounds[1].new_seqs[6].title, "gi|2495081|sp|Q09630|MGR1_CAEEL PROBABLE METABOTROPIC GLUTAMATE...")
        self.assertEqual(record.rounds[1].new_seqs[6].score, 35)
        self.assertAlmostEqual(record.rounds[1].new_seqs[6].e, 0.18)
        self.assertEqual(record.rounds[1].new_seqs[7].title, "gi|6014748|sp|Q10900|CTPI_MYCTU PROBABLE CATION-TRANSPORTING AT...")
        self.assertEqual(record.rounds[1].new_seqs[7].score, 33)
        self.assertAlmostEqual(record.rounds[1].new_seqs[7].e, 0.71)
        self.assertEqual(record.rounds[1].new_seqs[8].title, "gi|2833574|sp|Q58178|Y768_METJA HYPOTHETICAL PROTEIN MJ0768")
        self.assertEqual(record.rounds[1].new_seqs[8].score, 32)
        self.assertAlmostEqual(record.rounds[1].new_seqs[8].e, 1.6)
        self.assertEqual(record.rounds[1].new_seqs[9].title, "gi|1169299|sp|P45513|DHAT_CITFR 1,3-PROPANEDIOL DEHYDROGENASE (...")
        self.assertEqual(record.rounds[1].new_seqs[9].score, 31)
        self.assertAlmostEqual(record.rounds[1].new_seqs[9].e, 3.6)
        self.assertEqual(record.rounds[1].new_seqs[10].title, "gi|2495080|sp|P70579|MGR8_RAT METABOTROPIC GLUTAMATE RECEPTOR 8...")
        self.assertEqual(record.rounds[1].new_seqs[10].score, 31)
        self.assertAlmostEqual(record.rounds[1].new_seqs[10].e, 3.6)
        self.assertEqual(record.rounds[1].new_seqs[11].title, "gi|2495079|sp|O00222|MGR8_HUMAN METABOTROPIC GLUTAMATE RECEPTOR...")
        self.assertEqual(record.rounds[1].new_seqs[11].score, 31)
        self.assertAlmostEqual(record.rounds[1].new_seqs[11].e, 3.6)
        self.assertEqual(record.rounds[1].new_seqs[12].title, "gi|1346533|sp|P47743|MGR8_MOUSE METABOTROPIC GLUTAMATE RECEPTOR...")
        self.assertEqual(record.rounds[1].new_seqs[12].score, 30)
        self.assertAlmostEqual(record.rounds[1].new_seqs[12].e, 6.2)
        self.assertEqual(record.rounds[1].new_seqs[13].title, "gi|113381|sp|P06758|ADH2_ZYMMO ALCOHOL DEHYDROGENASE II (ADH II)")
        self.assertEqual(record.rounds[1].new_seqs[13].score, 30)
        self.assertAlmostEqual(record.rounds[1].new_seqs[13].e, 8.1)
        self.assertEqual(record.rounds[1].new_seqs[14].title, "gi|127751|sp|P02567|MYSD_CAEEL MYOSIN HEAVY CHAIN D (MHC D)")
        self.assertEqual(record.rounds[1].new_seqs[14].score, 30)
        self.assertAlmostEqual(record.rounds[1].new_seqs[14].e, 8.1)
        self.assertEqual(record.rounds[1].new_seqs[15].title, "gi|1346321|sp|P23897|HSER_RAT HEAT-STABLE ENTEROTOXIN RECEPTOR ...")
        self.assertEqual(record.rounds[1].new_seqs[15].score, 30)
        self.assertAlmostEqual(record.rounds[1].new_seqs[15].e, 8.1)
        self.assertEqual(record.rounds[1].new_seqs[16].title, "gi|1175674|sp|P45116|YCIH_HAEIN HYPOTHETICAL PROTEIN HI1225")
        self.assertEqual(record.rounds[1].new_seqs[16].score, 30)
        self.assertAlmostEqual(record.rounds[1].new_seqs[16].e, 8.1)
        self.assertEqual(record.rounds[1].new_seqs[17].title, "gi|3025270|sp|P77269|YPHF_ECOLI ABC TRANSPORTER PERIPLASMIC BIN...")
        self.assertEqual(record.rounds[1].new_seqs[17].score, 30)
        self.assertAlmostEqual(record.rounds[1].new_seqs[17].e, 8.1)
        self.assertEqual(len(record.rounds[1].alignments), 24)
        self.assertEqual(record.rounds[1].alignments[0].title, ">gi|126349|sp|P04816|LIVK_ECOLI LEUCINE-SPECIFIC BINDING PROTEIN PRECURSOR (LS-BP) (L-BP)")
        self.assertEqual(record.rounds[1].alignments[0].length, 369)
        self.assertEqual(record.rounds[1].alignments[1].title, ">gi|126342|sp|P17215|LIVJ_SALTY LEU/ILE/VAL/THR-BINDING PROTEIN PRECURSOR (LIVT-BP)")
        self.assertEqual(record.rounds[1].alignments[1].length, 365)
        self.assertEqual(record.rounds[1].alignments[2].title, ">gi|126347|sp|P25399|LIVJ_CITFR LEU/ILE/VAL-BINDING PROTEIN PRECURSOR (LIV-BP)")
        self.assertEqual(record.rounds[1].alignments[2].length, 367)
        self.assertEqual(record.rounds[1].alignments[3].title, ">gi|126343|sp|P17216|LIVK_SALTY LEUCINE-SPECIFIC BINDING PROTEIN PRECURSOR (LS-BP) (L-BP)")
        self.assertEqual(record.rounds[1].alignments[3].length, 369)
        self.assertEqual(record.rounds[1].alignments[4].title, ">gi|126348|sp|P02917|LIVJ_ECOLI LEU/ILE/VAL-BINDING PROTEIN PRECURSOR (LIV-BP)")
        self.assertEqual(record.rounds[1].alignments[4].length, 367)
        self.assertEqual(record.rounds[1].alignments[5].title, ">gi|115120|sp|P21175|BRAC_PSEAE LEUCINE-, ISOLEUCINE-, VALINE-, THREONINE-, AND ALANINE-BINDING PROTEIN PRECURSOR (LIVAT-BP) (LEU/ILE/VAL/THR/ALA-BINDING PROTEIN)")
        self.assertEqual(record.rounds[1].alignments[5].length, 373)
        self.assertEqual(record.rounds[1].alignments[6].title, ">gi|113709|sp|P27017|AMIC_PSEAE ALIPHATIC AMIDASE EXPRESSION-REGULATING PROTEIN")
        self.assertEqual(record.rounds[1].alignments[6].length, 385)
        self.assertEqual(record.rounds[1].alignments[7].title, ">gi|3024131|sp|P91685|MGR_DROME METABOTROPIC GLUTAMATE RECEPTOR PRECURSOR")
        self.assertEqual(record.rounds[1].alignments[7].length, 976)
        self.assertEqual(record.rounds[1].alignments[8].title, ">gi|121300|sp|P23385|MGR1_RAT METABOTROPIC GLUTAMATE RECEPTOR 1 PRECURSOR")
        self.assertEqual(record.rounds[1].alignments[8].length, 1199)
        self.assertEqual(record.rounds[1].alignments[9].title, ">gi|2495074|sp|Q13255|MGR1_HUMAN METABOTROPIC GLUTAMATE RECEPTOR 1 PRECURSOR")
        self.assertEqual(record.rounds[1].alignments[9].length, 1194)
        self.assertEqual(record.rounds[1].alignments[10].title, ">gi|1170947|sp|P31424|MGR5_RAT METABOTROPIC GLUTAMATE RECEPTOR 5 PRECURSOR")
        self.assertEqual(record.rounds[1].alignments[10].length, 1203)
        self.assertEqual(record.rounds[1].alignments[11].title, ">gi|1709020|sp|P41594|MGR5_HUMAN METABOTROPIC GLUTAMATE RECEPTOR 5 PRECURSOR")
        self.assertEqual(record.rounds[1].alignments[11].length, 1212)
        self.assertEqual(record.rounds[1].alignments[12].title, ">gi|2495081|sp|Q09630|MGR1_CAEEL PROBABLE METABOTROPIC GLUTAMATE RECEPTOR MGL-1")
        self.assertEqual(record.rounds[1].alignments[12].length, 999)
        self.assertEqual(record.rounds[1].alignments[13].title, ">gi|6014748|sp|Q10900|CTPI_MYCTU PROBABLE CATION-TRANSPORTING ATPASE I")
        self.assertEqual(record.rounds[1].alignments[13].length, 1632)
        self.assertEqual(record.rounds[1].alignments[14].title, ">gi|2833574|sp|Q58178|Y768_METJA HYPOTHETICAL PROTEIN MJ0768")
        self.assertEqual(record.rounds[1].alignments[14].length, 249)
        self.assertEqual(record.rounds[1].alignments[15].title, ">gi|1169299|sp|P45513|DHAT_CITFR 1,3-PROPANEDIOL DEHYDROGENASE (3-HYDROXYPROPIONALDEHYDE REDUCTASE) (1,3-PROPANEDIOL OXIDOREDUCTASE)")
        self.assertEqual(record.rounds[1].alignments[15].length, 387)
        self.assertEqual(record.rounds[1].alignments[16].title, ">gi|2495080|sp|P70579|MGR8_RAT METABOTROPIC GLUTAMATE RECEPTOR 8 PRECURSOR")
        self.assertEqual(record.rounds[1].alignments[16].length, 908)
        self.assertEqual(record.rounds[1].alignments[17].title, ">gi|2495079|sp|O00222|MGR8_HUMAN METABOTROPIC GLUTAMATE RECEPTOR 8 PRECURSOR")
        self.assertEqual(record.rounds[1].alignments[17].length, 908)
        self.assertEqual(record.rounds[1].alignments[18].title, ">gi|1346533|sp|P47743|MGR8_MOUSE METABOTROPIC GLUTAMATE RECEPTOR 8 PRECURSOR")
        self.assertEqual(record.rounds[1].alignments[18].length, 908)
        self.assertEqual(record.rounds[1].alignments[19].title, ">gi|113381|sp|P06758|ADH2_ZYMMO ALCOHOL DEHYDROGENASE II (ADH II)")
        self.assertEqual(record.rounds[1].alignments[19].length, 383)
        self.assertEqual(record.rounds[1].alignments[20].title, ">gi|127751|sp|P02567|MYSD_CAEEL MYOSIN HEAVY CHAIN D (MHC D)")
        self.assertEqual(record.rounds[1].alignments[20].length, 1938)
        self.assertEqual(record.rounds[1].alignments[21].title, ">gi|1346321|sp|P23897|HSER_RAT HEAT-STABLE ENTEROTOXIN RECEPTOR PRECURSOR (GC-C) (INTESTINAL GUANYLATE CYCLASE) (STA RECEPTOR)")
        self.assertEqual(record.rounds[1].alignments[21].length, 1072)
        self.assertEqual(record.rounds[1].alignments[22].title, ">gi|1175674|sp|P45116|YCIH_HAEIN HYPOTHETICAL PROTEIN HI1225")
        self.assertEqual(record.rounds[1].alignments[22].length, 106)
        self.assertEqual(record.rounds[1].alignments[23].title, ">gi|3025270|sp|P77269|YPHF_ECOLI ABC TRANSPORTER PERIPLASMIC BINDING PROTEIN YPHF PRECURSOR")

    def _check_text_2010L_phiblast_002_round2(self, record):
        self.assertEqual(record.rounds[1].alignments[23].length, 327)
        self.assertEqual(len(record.rounds[2].new_seqs), 16)
        self.assertEqual(record.rounds[2].new_seqs[0].title, "gi|3024134|sp|O15303|MGR6_HUMAN METABOTROPIC GLUTAMATE RECEPTOR...")
        self.assertEqual(record.rounds[2].new_seqs[0].score, 42)
        self.assertAlmostEqual(record.rounds[2].new_seqs[0].e, 0.002)
        self.assertEqual(record.rounds[2].new_seqs[1].title, "gi|2495077|sp|Q14833|MGR4_HUMAN METABOTROPIC GLUTAMATE RECEPTOR...")
        self.assertEqual(record.rounds[2].new_seqs[1].score, 39)
        self.assertAlmostEqual(record.rounds[2].new_seqs[1].e, 0.012)
        self.assertEqual(record.rounds[2].new_seqs[2].title, "gi|3024131|sp|P91685|MGR_DROME METABOTROPIC GLUTAMATE RECEPTOR ...")
        self.assertEqual(record.rounds[2].new_seqs[2].score, 38)
        self.assertAlmostEqual(record.rounds[2].new_seqs[2].e, 0.021)
        self.assertEqual(record.rounds[2].new_seqs[3].title, "gi|400255|sp|P31423|MGR4_RAT METABOTROPIC GLUTAMATE RECEPTOR 4 ...")
        self.assertEqual(record.rounds[2].new_seqs[3].score, 38)
        self.assertAlmostEqual(record.rounds[2].new_seqs[3].e, 0.036)
        self.assertEqual(record.rounds[2].new_seqs[4].title, "gi|547903|sp|P35349|MGR6_RAT METABOTROPIC GLUTAMATE RECEPTOR 6 ...")
        self.assertEqual(record.rounds[2].new_seqs[4].score, 37)
        self.assertAlmostEqual(record.rounds[2].new_seqs[4].e, 0.047)
        self.assertEqual(record.rounds[2].new_seqs[5].title, "gi|2495080|sp|P70579|MGR8_RAT METABOTROPIC GLUTAMATE RECEPTOR 8...")
        self.assertEqual(record.rounds[2].new_seqs[5].score, 36)
        self.assertAlmostEqual(record.rounds[2].new_seqs[5].e, 0.10)
        self.assertEqual(record.rounds[2].new_seqs[6].title, "gi|2495079|sp|O00222|MGR8_HUMAN METABOTROPIC GLUTAMATE RECEPTOR...")
        self.assertEqual(record.rounds[2].new_seqs[6].score, 36)
        self.assertAlmostEqual(record.rounds[2].new_seqs[6].e, 0.10)
        self.assertEqual(record.rounds[2].new_seqs[7].title, "gi|1346533|sp|P47743|MGR8_MOUSE METABOTROPIC GLUTAMATE RECEPTOR...")
        self.assertEqual(record.rounds[2].new_seqs[7].score, 35)
        self.assertAlmostEqual(record.rounds[2].new_seqs[7].e, 0.18)
        self.assertEqual(record.rounds[2].new_seqs[8].title, "gi|400402|sp|P13595|NCA1_MOUSE NEURAL CELL ADHESION MOLECULE, L...")
        self.assertEqual(record.rounds[2].new_seqs[8].score, 33)
        self.assertAlmostEqual(record.rounds[2].new_seqs[8].e, 0.69)
        self.assertEqual(record.rounds[2].new_seqs[9].title, "gi|1170947|sp|P31424|MGR5_RAT METABOTROPIC GLUTAMATE RECEPTOR 5...")
        self.assertEqual(record.rounds[2].new_seqs[9].score, 33)
        self.assertAlmostEqual(record.rounds[2].new_seqs[9].e, 0.69)
        self.assertEqual(record.rounds[2].new_seqs[10].title, "gi|1706242|sp|P51840|CYGE_RAT GUANYLYL CYCLASE GC-E PRECURSOR (...")
        self.assertEqual(record.rounds[2].new_seqs[10].score, 33)
        self.assertAlmostEqual(record.rounds[2].new_seqs[10].e, 0.91)
        self.assertEqual(record.rounds[2].new_seqs[11].title, "gi|1709020|sp|P41594|MGR5_HUMAN METABOTROPIC GLUTAMATE RECEPTOR...")
        self.assertEqual(record.rounds[2].new_seqs[11].score, 33)
        self.assertAlmostEqual(record.rounds[2].new_seqs[11].e, 0.91)
        self.assertEqual(record.rounds[2].new_seqs[12].title, "gi|549451|sp|Q06278|ADO_HUMAN ALDEHYDE OXIDASE")
        self.assertEqual(record.rounds[2].new_seqs[12].score, 32)
        self.assertAlmostEqual(record.rounds[2].new_seqs[12].e, 2.0)
        self.assertEqual(record.rounds[2].new_seqs[13].title, "gi|6014748|sp|Q10900|CTPI_MYCTU PROBABLE CATION-TRANSPORTING AT...")
        self.assertEqual(record.rounds[2].new_seqs[13].score, 32)
        self.assertAlmostEqual(record.rounds[2].new_seqs[13].e, 2.0)
        self.assertEqual(record.rounds[2].new_seqs[14].title, "gi|127861|sp|P13594|NCA2_MOUSE NEURAL CELL ADHESION MOLECULE, P...")
        self.assertEqual(record.rounds[2].new_seqs[14].score, 31)
        self.assertAlmostEqual(record.rounds[2].new_seqs[14].e, 2.7)
        self.assertEqual(record.rounds[2].new_seqs[15].title, "gi|127971|sp||NCPR_SALTR_1 [Segment 1 of 3] NADPH-CYTOCHROME P4...")
        self.assertEqual(record.rounds[2].new_seqs[15].score, 31)
        self.assertAlmostEqual(record.rounds[2].new_seqs[15].e, 3.5)
        self.assertEqual(len(record.rounds[2].alignments), 23)
        self.assertEqual(record.rounds[2].alignments[0].title, ">gi|126342|sp|P17215|LIVJ_SALTY LEU/ILE/VAL/THR-BINDING PROTEIN PRECURSOR (LIVT-BP)")
        self.assertEqual(record.rounds[2].alignments[0].length, 365)
        self.assertEqual(record.rounds[2].alignments[1].title, ">gi|126349|sp|P04816|LIVK_ECOLI LEUCINE-SPECIFIC BINDING PROTEIN PRECURSOR (LS-BP) (L-BP)")
        self.assertEqual(record.rounds[2].alignments[1].length, 369)
        self.assertEqual(record.rounds[2].alignments[2].title, ">gi|126348|sp|P02917|LIVJ_ECOLI LEU/ILE/VAL-BINDING PROTEIN PRECURSOR (LIV-BP)")
        self.assertEqual(record.rounds[2].alignments[2].length, 367)
        self.assertEqual(record.rounds[2].alignments[3].title, ">gi|126347|sp|P25399|LIVJ_CITFR LEU/ILE/VAL-BINDING PROTEIN PRECURSOR (LIV-BP)")
        self.assertEqual(record.rounds[2].alignments[3].length, 367)
        self.assertEqual(record.rounds[2].alignments[4].title, ">gi|126343|sp|P17216|LIVK_SALTY LEUCINE-SPECIFIC BINDING PROTEIN PRECURSOR (LS-BP) (L-BP)")
        self.assertEqual(record.rounds[2].alignments[4].length, 369)
        self.assertEqual(record.rounds[2].alignments[5].title, ">gi|115120|sp|P21175|BRAC_PSEAE LEUCINE-, ISOLEUCINE-, VALINE-, THREONINE-, AND ALANINE-BINDING PROTEIN PRECURSOR (LIVAT-BP) (LEU/ILE/VAL/THR/ALA-BINDING PROTEIN)")
        self.assertEqual(record.rounds[2].alignments[5].length, 373)
        self.assertEqual(record.rounds[2].alignments[6].title, ">gi|113709|sp|P27017|AMIC_PSEAE ALIPHATIC AMIDASE EXPRESSION-REGULATING PROTEIN")
        self.assertEqual(record.rounds[2].alignments[6].length, 385)
        self.assertEqual(record.rounds[2].alignments[7].title, ">gi|3024134|sp|O15303|MGR6_HUMAN METABOTROPIC GLUTAMATE RECEPTOR 6 PRECURSOR")
        self.assertEqual(record.rounds[2].alignments[7].length, 877)
        self.assertEqual(record.rounds[2].alignments[8].title, ">gi|2495077|sp|Q14833|MGR4_HUMAN METABOTROPIC GLUTAMATE RECEPTOR 4 PRECURSOR")
        self.assertEqual(record.rounds[2].alignments[8].length, 912)
        self.assertEqual(record.rounds[2].alignments[9].title, ">gi|3024131|sp|P91685|MGR_DROME METABOTROPIC GLUTAMATE RECEPTOR PRECURSOR")
        self.assertEqual(record.rounds[2].alignments[9].length, 976)
        self.assertEqual(record.rounds[2].alignments[10].title, ">gi|400255|sp|P31423|MGR4_RAT METABOTROPIC GLUTAMATE RECEPTOR 4 PRECURSOR")
        self.assertEqual(record.rounds[2].alignments[10].length, 912)
        self.assertEqual(record.rounds[2].alignments[11].title, ">gi|547903|sp|P35349|MGR6_RAT METABOTROPIC GLUTAMATE RECEPTOR 6 PRECURSOR")
        self.assertEqual(record.rounds[2].alignments[11].length, 871)
        self.assertEqual(record.rounds[2].alignments[12].title, ">gi|2495080|sp|P70579|MGR8_RAT METABOTROPIC GLUTAMATE RECEPTOR 8 PRECURSOR")
        self.assertEqual(record.rounds[2].alignments[12].length, 908)
        self.assertEqual(record.rounds[2].alignments[13].title, ">gi|2495079|sp|O00222|MGR8_HUMAN METABOTROPIC GLUTAMATE RECEPTOR 8 PRECURSOR")
        self.assertEqual(record.rounds[2].alignments[13].length, 908)
        self.assertEqual(record.rounds[2].alignments[14].title, ">gi|1346533|sp|P47743|MGR8_MOUSE METABOTROPIC GLUTAMATE RECEPTOR 8 PRECURSOR")
        self.assertEqual(record.rounds[2].alignments[14].length, 908)
        self.assertEqual(record.rounds[2].alignments[15].title, ">gi|400402|sp|P13595|NCA1_MOUSE NEURAL CELL ADHESION MOLECULE, LARGE ISOFORM PRECURSOR (N-CAM 180) (NCAM-180) [CONTAINS: N-CAM 140 (NCAM-140)]")
        self.assertEqual(record.rounds[2].alignments[15].length, 1115)
        self.assertEqual(record.rounds[2].alignments[16].title, ">gi|1170947|sp|P31424|MGR5_RAT METABOTROPIC GLUTAMATE RECEPTOR 5 PRECURSOR")
        self.assertEqual(record.rounds[2].alignments[16].length, 1203)
        self.assertEqual(record.rounds[2].alignments[17].title, ">gi|1706242|sp|P51840|CYGE_RAT GUANYLYL CYCLASE GC-E PRECURSOR (GUANYLATE CYCLASE 2E)")
        self.assertEqual(record.rounds[2].alignments[17].length, 1108)
        self.assertEqual(record.rounds[2].alignments[18].title, ">gi|1709020|sp|P41594|MGR5_HUMAN METABOTROPIC GLUTAMATE RECEPTOR 5 PRECURSOR")
        self.assertEqual(record.rounds[2].alignments[18].length, 1212)
        self.assertEqual(record.rounds[2].alignments[19].title, ">gi|549451|sp|Q06278|ADO_HUMAN ALDEHYDE OXIDASE")
        self.assertEqual(record.rounds[2].alignments[19].length, 1338)
        self.assertEqual(record.rounds[2].alignments[20].title, ">gi|6014748|sp|Q10900|CTPI_MYCTU PROBABLE CATION-TRANSPORTING ATPASE I")
        self.assertEqual(record.rounds[2].alignments[20].length, 1632)
        self.assertEqual(record.rounds[2].alignments[21].title, ">gi|127861|sp|P13594|NCA2_MOUSE NEURAL CELL ADHESION MOLECULE, PHOSPHATIDYLINOSITOL-LINKED ISOFORM PRECURSOR (N-CAM 120) (NCAM-120)")
        self.assertEqual(record.rounds[2].alignments[21].length, 725)
        self.assertEqual(record.rounds[2].alignments[22].title, ">gi|127971|sp||NCPR_SALTR_1 [Segment 1 of 3] NADPH-CYTOCHROME P450 REDUCTASE (CPR)")
        self.assertEqual(record.rounds[2].alignments[22].length, 426)

    def _check_text_2010L_phiblast_002_hsps(self, record):
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].score, 1897)
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].bits, 743)
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[0].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].score, 1733)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].bits, 679)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[0].alignments[1].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].score, 1482)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].bits, 581)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].expect, 1e-166)
        self.assertEqual(len(record.rounds[0].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].score, 1470)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].bits, 577)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].expect, 1e-164)
        self.assertEqual(len(record.rounds[0].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].score, 1453)
        self.assertAlmostEqual(record.rounds[0].alignments[4].hsps[0].bits, 570)
        self.assertAlmostEqual(record.rounds[0].alignments[4].hsps[0].expect, 1e-162)
        self.assertEqual(len(record.rounds[0].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].score, 1050)
        self.assertAlmostEqual(record.rounds[0].alignments[5].hsps[0].bits, 413)
        self.assertAlmostEqual(record.rounds[0].alignments[5].hsps[0].expect, 1e-115)
        self.assertEqual(len(record.rounds[0].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].score, 92)
        self.assertAlmostEqual(record.rounds[0].alignments[6].hsps[0].bits, 40.2)
        self.assertAlmostEqual(record.rounds[0].alignments[6].hsps[0].expect, 0.006)
        self.assertEqual(len(record.rounds[0].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].score, 72)
        self.assertAlmostEqual(record.rounds[0].alignments[7].hsps[0].bits, 32.5)
        self.assertAlmostEqual(record.rounds[0].alignments[7].hsps[0].expect, 1.4)
        self.assertEqual(len(record.rounds[0].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].score, 71)
        self.assertAlmostEqual(record.rounds[0].alignments[8].hsps[0].bits, 32.1)
        self.assertAlmostEqual(record.rounds[0].alignments[8].hsps[0].expect, 1.8)
        self.assertEqual(len(record.rounds[0].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].score, 71)
        self.assertAlmostEqual(record.rounds[0].alignments[9].hsps[0].bits, 32.1)
        self.assertAlmostEqual(record.rounds[0].alignments[9].hsps[0].expect, 1.8)
        self.assertEqual(len(record.rounds[0].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].score, 71)
        self.assertAlmostEqual(record.rounds[0].alignments[10].hsps[0].bits, 32.1)
        self.assertAlmostEqual(record.rounds[0].alignments[10].hsps[0].expect, 1.8)
        self.assertEqual(len(record.rounds[0].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].score, 68)
        self.assertAlmostEqual(record.rounds[0].alignments[11].hsps[0].bits, 30.9)
        self.assertAlmostEqual(record.rounds[0].alignments[11].hsps[0].expect, 4.0)
        self.assertEqual(len(record.rounds[0].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].score, 68)
        self.assertAlmostEqual(record.rounds[0].alignments[12].hsps[0].bits, 30.9)
        self.assertAlmostEqual(record.rounds[0].alignments[12].hsps[0].expect, 4.0)
        self.assertEqual(len(record.rounds[0].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].score, 67)
        self.assertAlmostEqual(record.rounds[0].alignments[13].hsps[0].bits, 30.5)
        self.assertAlmostEqual(record.rounds[0].alignments[13].hsps[0].expect, 5.2)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].score, 1683)
        self.assertAlmostEqual(record.rounds[1].alignments[0].hsps[0].bits, 660)
        self.assertAlmostEqual(record.rounds[1].alignments[0].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[1].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].score, 1673)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[0].bits, 656)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[1].alignments[1].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].score, 1672)
        self.assertAlmostEqual(record.rounds[1].alignments[2].hsps[0].bits, 655)
        self.assertAlmostEqual(record.rounds[1].alignments[2].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[1].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].score, 1671)
        self.assertAlmostEqual(record.rounds[1].alignments[3].hsps[0].bits, 655)
        self.assertAlmostEqual(record.rounds[1].alignments[3].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[1].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].score, 1670)
        self.assertAlmostEqual(record.rounds[1].alignments[4].hsps[0].bits, 655)
        self.assertAlmostEqual(record.rounds[1].alignments[4].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[1].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].score, 1569)
        self.assertAlmostEqual(record.rounds[1].alignments[5].hsps[0].bits, 615)
        self.assertAlmostEqual(record.rounds[1].alignments[5].hsps[0].expect, 1e-176)
        self.assertEqual(len(record.rounds[1].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].score, 113)
        self.assertAlmostEqual(record.rounds[1].alignments[6].hsps[0].bits, 48.6)
        self.assertAlmostEqual(record.rounds[1].alignments[6].hsps[0].expect, 2e-5)
        self.assertEqual(len(record.rounds[1].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].score, 94)
        self.assertAlmostEqual(record.rounds[1].alignments[7].hsps[0].bits, 41.2)
        self.assertAlmostEqual(record.rounds[1].alignments[7].hsps[0].expect, 0.003)
        self.assertEqual(len(record.rounds[1].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].score, 89)
        self.assertAlmostEqual(record.rounds[1].alignments[8].hsps[0].bits, 39.2)
        self.assertAlmostEqual(record.rounds[1].alignments[8].hsps[0].expect, 0.012)
        self.assertEqual(len(record.rounds[1].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].score, 89)
        self.assertAlmostEqual(record.rounds[1].alignments[9].hsps[0].bits, 39.2)
        self.assertAlmostEqual(record.rounds[1].alignments[9].hsps[0].expect, 0.012)
        self.assertEqual(len(record.rounds[1].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].score, 81)
        self.assertAlmostEqual(record.rounds[1].alignments[10].hsps[0].bits, 36.1)
        self.assertAlmostEqual(record.rounds[1].alignments[10].hsps[0].expect, 0.11)
        self.assertEqual(len(record.rounds[1].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].score, 80)
        self.assertAlmostEqual(record.rounds[1].alignments[11].hsps[0].bits, 35.7)
        self.assertAlmostEqual(record.rounds[1].alignments[11].hsps[0].expect, 0.14)
        self.assertEqual(len(record.rounds[1].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].score, 79)
        self.assertAlmostEqual(record.rounds[1].alignments[12].hsps[0].bits, 35.3)
        self.assertAlmostEqual(record.rounds[1].alignments[12].hsps[0].expect, 0.18)
        self.assertEqual(len(record.rounds[1].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].score, 74)
        self.assertAlmostEqual(record.rounds[1].alignments[13].hsps[0].bits, 33.4)
        self.assertAlmostEqual(record.rounds[1].alignments[13].hsps[0].expect, 0.71)
        self.assertEqual(len(record.rounds[1].alignments[13].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].score, 71)
        self.assertAlmostEqual(record.rounds[1].alignments[14].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.rounds[1].alignments[14].hsps[0].expect, 1.6)
        self.assertEqual(len(record.rounds[1].alignments[14].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].score, 68)
        self.assertAlmostEqual(record.rounds[1].alignments[15].hsps[0].bits, 31.1)
        self.assertAlmostEqual(record.rounds[1].alignments[15].hsps[0].expect, 3.6)
        self.assertEqual(len(record.rounds[1].alignments[15].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].score, 68)
        self.assertAlmostEqual(record.rounds[1].alignments[16].hsps[0].bits, 31.1)
        self.assertAlmostEqual(record.rounds[1].alignments[16].hsps[0].expect, 3.6)
        self.assertEqual(len(record.rounds[1].alignments[16].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].score, 68)
        self.assertAlmostEqual(record.rounds[1].alignments[17].hsps[0].bits, 31.1)
        self.assertAlmostEqual(record.rounds[1].alignments[17].hsps[0].expect, 3.6)
        self.assertEqual(len(record.rounds[1].alignments[17].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].score, 66)
        self.assertAlmostEqual(record.rounds[1].alignments[18].hsps[0].bits, 30.3)
        self.assertAlmostEqual(record.rounds[1].alignments[18].hsps[0].expect, 6.2)
        self.assertEqual(len(record.rounds[1].alignments[18].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].score, 65)
        self.assertAlmostEqual(record.rounds[1].alignments[19].hsps[0].bits, 29.9)
        self.assertAlmostEqual(record.rounds[1].alignments[19].hsps[0].expect, 8.1)
        self.assertEqual(len(record.rounds[1].alignments[19].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].score, 65)
        self.assertAlmostEqual(record.rounds[1].alignments[20].hsps[0].bits, 29.9)
        self.assertAlmostEqual(record.rounds[1].alignments[20].hsps[0].expect, 8.1)
        self.assertEqual(len(record.rounds[1].alignments[20].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].score, 65)
        self.assertAlmostEqual(record.rounds[1].alignments[21].hsps[0].bits, 29.9)
        self.assertAlmostEqual(record.rounds[1].alignments[21].hsps[0].expect, 8.1)
        self.assertEqual(len(record.rounds[1].alignments[21].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].score, 65)
        self.assertAlmostEqual(record.rounds[1].alignments[22].hsps[0].bits, 29.9)
        self.assertAlmostEqual(record.rounds[1].alignments[22].hsps[0].expect, 8.1)
        self.assertEqual(len(record.rounds[1].alignments[22].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].score, 65)
        self.assertAlmostEqual(record.rounds[1].alignments[23].hsps[0].bits, 29.9)
        self.assertAlmostEqual(record.rounds[1].alignments[23].hsps[0].expect, 8.1)
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].score, 1478)
        self.assertAlmostEqual(record.rounds[2].alignments[0].hsps[0].bits, 580)
        self.assertAlmostEqual(record.rounds[2].alignments[0].hsps[0].expect, 1e-165)
        self.assertEqual(len(record.rounds[2].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].score, 1477)
        self.assertAlmostEqual(record.rounds[2].alignments[1].hsps[0].bits, 579)
        self.assertAlmostEqual(record.rounds[2].alignments[1].hsps[0].expect, 1e-165)
        self.assertEqual(len(record.rounds[2].alignments[1].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].score, 1475)
        self.assertAlmostEqual(record.rounds[2].alignments[2].hsps[0].bits, 579)
        self.assertAlmostEqual(record.rounds[2].alignments[2].hsps[0].expect, 1e-165)
        self.assertEqual(len(record.rounds[2].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].score, 1474)
        self.assertAlmostEqual(record.rounds[2].alignments[3].hsps[0].bits, 578)
        self.assertAlmostEqual(record.rounds[2].alignments[3].hsps[0].expect, 1e-165)
        self.assertEqual(len(record.rounds[2].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].score, 1472)
        self.assertAlmostEqual(record.rounds[2].alignments[4].hsps[0].bits, 577)
        self.assertAlmostEqual(record.rounds[2].alignments[4].hsps[0].expect, 1e-165)
        self.assertEqual(len(record.rounds[2].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].score, 1351)
        self.assertAlmostEqual(record.rounds[2].alignments[5].hsps[0].bits, 530)
        self.assertAlmostEqual(record.rounds[2].alignments[5].hsps[0].expect, 1e-150)
        self.assertEqual(len(record.rounds[2].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].score, 976)
        self.assertAlmostEqual(record.rounds[2].alignments[6].hsps[0].bits, 384)
        self.assertAlmostEqual(record.rounds[2].alignments[6].hsps[0].expect, 1e-106)
        self.assertEqual(len(record.rounds[2].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].score, 95)
        self.assertAlmostEqual(record.rounds[2].alignments[7].hsps[0].bits, 41.6)
        self.assertAlmostEqual(record.rounds[2].alignments[7].hsps[0].expect, 0.002)
        self.assertEqual(len(record.rounds[2].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].score, 89)
        self.assertAlmostEqual(record.rounds[2].alignments[8].hsps[0].bits, 39.3)
        self.assertAlmostEqual(record.rounds[2].alignments[8].hsps[0].expect, 0.012)
        self.assertEqual(len(record.rounds[2].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].score, 87)
        self.assertAlmostEqual(record.rounds[2].alignments[9].hsps[0].bits, 38.5)
        self.assertAlmostEqual(record.rounds[2].alignments[9].hsps[0].expect, 0.021)
        self.assertEqual(len(record.rounds[2].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].score, 85)
        self.assertAlmostEqual(record.rounds[2].alignments[10].hsps[0].bits, 37.7)
        self.assertAlmostEqual(record.rounds[2].alignments[10].hsps[0].expect, 0.036)
        self.assertEqual(len(record.rounds[2].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].score, 84)
        self.assertAlmostEqual(record.rounds[2].alignments[11].hsps[0].bits, 37.3)
        self.assertAlmostEqual(record.rounds[2].alignments[11].hsps[0].expect, 0.047)
        self.assertEqual(len(record.rounds[2].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].score, 81)
        self.assertAlmostEqual(record.rounds[2].alignments[12].hsps[0].bits, 36.1)
        self.assertAlmostEqual(record.rounds[2].alignments[12].hsps[0].expect, 0.10)
        self.assertEqual(len(record.rounds[2].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].score, 81)
        self.assertAlmostEqual(record.rounds[2].alignments[13].hsps[0].bits, 36.1)
        self.assertAlmostEqual(record.rounds[2].alignments[13].hsps[0].expect, 0.10)
        self.assertEqual(len(record.rounds[2].alignments[13].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].score, 79)
        self.assertAlmostEqual(record.rounds[2].alignments[14].hsps[0].bits, 35.4)
        self.assertAlmostEqual(record.rounds[2].alignments[14].hsps[0].expect, 0.18)
        self.assertEqual(len(record.rounds[2].alignments[14].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].score, 74)
        self.assertAlmostEqual(record.rounds[2].alignments[15].hsps[0].bits, 33.4)
        self.assertAlmostEqual(record.rounds[2].alignments[15].hsps[0].expect, 0.69)
        self.assertEqual(len(record.rounds[2].alignments[15].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].score, 74)
        self.assertAlmostEqual(record.rounds[2].alignments[16].hsps[0].bits, 33.4)
        self.assertAlmostEqual(record.rounds[2].alignments[16].hsps[0].expect, 0.69)
        self.assertEqual(len(record.rounds[2].alignments[16].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].score, 73)
        self.assertAlmostEqual(record.rounds[2].alignments[17].hsps[0].bits, 33.0)
        self.assertAlmostEqual(record.rounds[2].alignments[17].hsps[0].expect, 0.91)
        self.assertEqual(len(record.rounds[2].alignments[17].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].score, 73)
        self.assertAlmostEqual(record.rounds[2].alignments[18].hsps[0].bits, 33.0)
        self.assertAlmostEqual(record.rounds[2].alignments[18].hsps[0].expect, 0.91)
        self.assertEqual(len(record.rounds[2].alignments[18].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].score, 70)
        self.assertAlmostEqual(record.rounds[2].alignments[19].hsps[0].bits, 31.9)
        self.assertAlmostEqual(record.rounds[2].alignments[19].hsps[0].expect, 2.0)
        self.assertEqual(len(record.rounds[2].alignments[19].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].score, 70)
        self.assertAlmostEqual(record.rounds[2].alignments[20].hsps[0].bits, 31.9)
        self.assertAlmostEqual(record.rounds[2].alignments[20].hsps[0].expect, 2.0)
        self.assertEqual(len(record.rounds[2].alignments[20].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].score, 69)
        self.assertAlmostEqual(record.rounds[2].alignments[21].hsps[0].bits, 31.5)
        self.assertAlmostEqual(record.rounds[2].alignments[21].hsps[0].expect, 2.7)
        self.assertEqual(len(record.rounds[2].alignments[21].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].score, 68)
        self.assertAlmostEqual(record.rounds[2].alignments[22].hsps[0].bits, 31.1)
        self.assertAlmostEqual(record.rounds[2].alignments[22].hsps[0].expect, 3.5)
        self.assertEqual(len(record.rounds[2].alignments[22].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].identities, (369, 369))
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].positives, (369, 369))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].identities, (333, 369))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].positives, (351, 369))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].identities, (282, 369))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].positives, (320, 369))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].gaps, (2, 369))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].identities, (280, 366))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].positives, (316, 366))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].gaps, (2, 366))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].identities, (276, 369))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].positives, (316, 369))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].gaps, (2, 369))
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].identities, (193, 368))
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].positives, (268, 368))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].identities, (60, 334))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].positives, (130, 334))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].gaps, (23, 334))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].identities, (18, 67))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].positives, (32, 67))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].gaps, (2, 67))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].identities, (28, 89))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].positives, (42, 89))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].gaps, (11, 89))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].identities, (29, 125))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].positives, (57, 125))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].gaps, (6, 125))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].identities, (24, 93))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].positives, (44, 93))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].gaps, (2, 93))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].identities, (24, 65))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].positives, (31, 65))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].gaps, (9, 65))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].identities, (18, 55))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].positives, (31, 55))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].gaps, (1, 55))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].identities, (24, 90))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].positives, (41, 90))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].gaps, (10, 90))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].identities, (333, 369))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].positives, (351, 369))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].identities, (280, 367))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].positives, (316, 367))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].gaps, (2, 367))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].identities, (276, 369))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].positives, (316, 369))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].gaps, (2, 369))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].identities, (369, 369))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].positives, (369, 369))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].identities, (282, 369))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].positives, (320, 369))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].gaps, (2, 369))
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].identities, (193, 368))
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].positives, (268, 368))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].identities, (60, 314))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].positives, (114, 314))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].gaps, (12, 314))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].identities, (36, 154))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].positives, (66, 154))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].gaps, (5, 154))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].identities, (32, 121))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].positives, (55, 121))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].gaps, (4, 121))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].identities, (32, 121))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].positives, (55, 121))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].gaps, (4, 121))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].identities, (30, 121))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].positives, (54, 121))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].gaps, (4, 121))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].identities, (30, 120))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].positives, (53, 120))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].gaps, (4, 120))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].identities, (29, 125))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].positives, (57, 125))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].gaps, (6, 125))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].identities, (20, 68))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].positives, (26, 68))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].gaps, (7, 68))
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].identities, (33, 171))
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].positives, (67, 171))
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].gaps, (10, 171))
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].identities, (22, 88))
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].positives, (38, 88))
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].gaps, (8, 88))
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].identities, (50, 196))
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].positives, (77, 196))
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].gaps, (17, 196))
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].identities, (50, 196))
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].positives, (77, 196))
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].gaps, (17, 196))
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].identities, (50, 196))
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].positives, (76, 196))
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].gaps, (17, 196))
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].identities, (19, 69))
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].positives, (28, 69))
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].gaps, (3, 69))
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].identities, (18, 72))
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].positives, (31, 72))
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].gaps, (2, 72))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].identities, (16, 85))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].positives, (35, 85))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].gaps, (2, 85))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].identities, (11, 57))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].positives, (27, 57))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].gaps, (3, 57))
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].identities, (19, 79))
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].positives, (31, 79))
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].gaps, (2, 79))
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].identities, (280, 367))
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].positives, (316, 367))
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].gaps, (2, 367))
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].identities, (333, 369))
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].positives, (351, 369))
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].identities, (282, 369))
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].positives, (320, 369))
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].gaps, (2, 369))
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].identities, (276, 369))
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].positives, (316, 369))
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].gaps, (2, 369))
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].identities, (369, 369))
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].positives, (369, 369))
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].identities, (193, 368))
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].positives, (268, 368))
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].identities, (60, 314))
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].positives, (114, 314))
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].gaps, (12, 314))
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].identities, (47, 266))
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].positives, (90, 266))
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].gaps, (20, 266))
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].identities, (43, 216))
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].positives, (78, 216))
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].gaps, (13, 216))
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].identities, (42, 182))
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].positives, (74, 182))
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].gaps, (8, 182))
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].identities, (42, 215))
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].positives, (76, 215))
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].gaps, (11, 215))
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].identities, (46, 266))
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].positives, (88, 266))
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].gaps, (20, 266))
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].identities, (46, 222))
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].positives, (82, 222))
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].gaps, (15, 222))
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].identities, (46, 222))
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].positives, (82, 222))
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].gaps, (15, 222))
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].identities, (46, 222))
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].positives, (81, 222))
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].gaps, (15, 222))
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].identities, (46, 221))
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].positives, (73, 221))
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].gaps, (24, 221))
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].identities, (31, 155))
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].positives, (58, 155))
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].gaps, (4, 155))
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].identities, (29, 193))
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].positives, (67, 193))
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].gaps, (12, 193))
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].identities, (31, 155))
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].positives, (57, 155))
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].gaps, (4, 155))
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].identities, (26, 105))
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].positives, (37, 105))
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].gaps, (3, 105))
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].identities, (16, 55))
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].positives, (22, 55))
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].gaps, (3, 55))
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].identities, (39, 188))
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].positives, (62, 188))
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].gaps, (23, 188))
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].identities, (28, 146))
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].positives, (53, 146))
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].gaps, (8, 146))

    def _check_text_2010L_phiblast_002_hsps_details(self, record):
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].match, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct_end, 369)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].match, "MKR AKTIIAG++ALA+S  AMADDIKVA+VGAMSGP+AQWG MEFNGA QAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGI+YVIGHLCSSSTQPASDIYEDEGILMISPGAT PELTQRGYQ+IMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLK  NAN+VFFDGITAGEKDFSALIARL+KENIDFVYYGGYYPEMGQ++RQAR+ GLKTQFMGPEGVGNASLSNIAG AAEGMLVTMPKRYDQDPAN+ IV+ALKADKKDPSGPYVWITYAAVQSLATA+ R+ S  PL LVKDLKANGA+TVIGPL WDEKGDLKGF+FGVFQWHADGSST AK")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct, "MKRNAKTIIAGMIALAISHTAMADDIKVAVVGAMSGPIAQWGIMEFNGAEQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIKYVIGHLCSSSTQPASDIYEDEGILMISPGATAPELTQRGYQHIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKAANANVVFFDGITAGEKDFSALIARLKKENIDFVYYGGYYPEMGQMLRQARSVGLKTQFMGPEGVGNASLSNIAGDAAEGMLVTMPKRYDQDPANQGIVDALKADKKDPSGPYVWITYAAVQSLATALERTGSDEPLALVKDLKANGANTVIGPLNWDEKGDLKGFDFGVFQWHADGSSTAAK")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct_end, 369)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].match, "M  K K ++AG++ALA S  A+A+DIKVA+VGAMSGPVAQ+GD EF GA QA+ DINAKGGIKG+KL   +YDDACDPKQAVAVANK+VNDGI+YVIGHLCSSSTQPASDIYEDEGILMI+P AT PELT RGYQ I+RT GLDS QGPTAAKYILE VKPQRIAI+HDKQQYGEGLAR+VQDGLK+GNAN+VFFDGITAGEKDFS L+ARL+KENIDFVYYGGY+PEMGQI+RQARA GLKTQFMGPEGV N SLSNIAG +AEG+LVT PK YDQ PANK IV+A+KA K+DPSG +VW TYAA+QSL   + +S    P ++ K LKAN  DTV+GPL WDEKGDLKGFEFGVF WHA+G++T AK")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct, "MNTKGKALLAGLIALAFSNMALAEDIKVAVVGAMSGPVAQYGDQEFTGAEQAVADINAKGGIKGNKLQIAKYDDACDPKQAVAVANKVVNDGIKYVIGHLCSSSTQPASDIYEDEGILMITPAATAPELTARGYQLILRTTGLDSDQGPTAAKYILEKVKPQRIAIVHDKQQYGEGLARAVQDGLKKGNANVVFFDGITAGEKDFSTLVARLKKENIDFVYYGGYHPEMGQILRQARAAGLKTQFMGPEGVANVSLSNIAGESAEGLLVTKPKNYDQVPANKPIVDAIKAKKQDPSGAFVWTTYAALQSLQAGLNQSDD--PAEIAKYLKANSVDTVMGPLTWDEKGDLKGFEFGVFDWHANGTATDAK")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct_end, 367)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query, "KAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].match, "K KT++AG +AL++S  A ADDIKVA+VGAMSGPVAQ+GD EF GA QAI DINAKGGIKGDKLV V+YDDACDPKQAVAVANK+VNDGI+YVIGHLCSSSTQPASDIYEDEGILMI+P AT PELT RGY+ ++RT GLDS QGPTAAKYILE VKPQRIAIIHDKQQYGEGLAR+VQDGLK+G  N+VFFDGITAGEKDFS L+ARL+KENIDFVYYGGY+PEMGQI+RQ+RA GLKTQFMGPEGV N SLSNIAG +AEG+LVT PK YDQ PANK IV+A+KA K+DPSG +VW TYAA+QSL   +  S    P ++ K LK    DTV+GPL WDEKGDLKGFEFGVF WHA+G++T AK")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct, "KGKTLLAGCIALSLSHMAFADDIKVAVVGAMSGPVAQYGDQEFTGAEQAIADINAKGGIKGDKLVAVKYDDACDPKQAVAVANKVVNDGIKYVIGHLCSSSTQPASDIYEDEGILMITPAATAPELTARGYKLVLRTTGLDSDQGPTAAKYILEKVKPQRIAIIHDKQQYGEGLARAVQDGLKKGGVNVVFFDGITAGEKDFSTLVARLKKENIDFVYYGGYHPEMGQILRQSRAAGLKTQFMGPEGVANVSLSNIAGESAEGLLVTKPKNYDQVPANKPIVDAIKAKKQDPSGAFVWTTYAALQSLQAGLNHSDD--PAEIAKYLKGATVDTVMGPLSWDEKGDLKGFEFGVFDWHANGTATDAK")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query_start, 4)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct_start, 2)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct_end, 365)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].match, "M  K K ++AG +AL++S  A A DIKVA+VGAMSGPVAQ+GD EF GA QAI DINAKGG+KGDKLV V+YDDACDPKQAVAVANK+VNDGI+YVIGHLCSSSTQPASDIYEDEGILMI+P AT PELT RGY  ++RT GLDS QGPTAAKYI+E VKP+RIAI+HDKQQYGEGLARSVQD LK+ NA++VFFDGITAGEKDFS L+ARL+KENIDFVYYGGY+PEMGQI+RQARA GLKTQFMGPEGV N SLSNIAG +AEG+LVT PK YDQ PANK IV+A+KA K+DPSG +VW TYAA+QSL   + +S    P ++ K LKAN  +TV+GPL WD KGDLKGFEFGVF WHA+G++T AK")
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].sbjct, "MNMKGKALLAGCIALSLSNMAFAKDIKVAVVGAMSGPVAQYGDQEFTGAEQAIADINAKGGVKGDKLVMVKYDDACDPKQAVAVANKVVNDGIKYVIGHLCSSSTQPASDIYEDEGILMITPAATAPELTARGYNLVLRTTGLDSDQGPTAAKYIVEKVKPKRIAIVHDKQQYGEGLARSVQDNLKKANADVVFFDGITAGEKDFSTLVARLKKENIDFVYYGGYHPEMGQILRQARAAGLKTQFMGPEGVANVSLSNIAGESAEGLLVTKPKNYDQVPANKPIVDAIKAKKQDPSGAFVWTTYAALQSLQAGLNQSDD--PAEIAKYLKANSVETVMGPLSWDAKGDLKGFEFGVFDWHANGTATDAK")
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].sbjct_end, 367)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].query, "KRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].match, "+R ++   A  +A   S    AD IK+A+ G ++GPVAQ+GDM+  GA  AI+ IN  GG+ G +L GV YDDACDPKQAVAVANK+VNDG+++V+GH+CSSSTQPA+DIYEDEG+LMI+P AT PE+T RGY+ I RT GLD+ QGP A K+I E  K + IA++HDKQQYGEG+A  V+  ++     +  F+G+ AG+KDF+ALI++L+K  + FVY+GGY+PEMG ++RQA+  GL  +FMGPEGVGN+ ++ IAG A+EGML T+P+ ++QDP NKA+++A KA  +DPSG +V   Y+AV  +A  + ++    P  + + L+AN  +T  G L +DEKGDLK F+F V++WH D + T  K")
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].sbjct, "QRLSRLFAAMAIAGFASYSMAADTIKIALAGPVTGPVAQYGDMQRAGALMAIEQINKAGGVNGAQLEGVIYDDACDPKQAVAVANKVVNDGVKFVVGHVCSSSTQPATDIYEDEGVLMITPSATAPEITSRGYKLIFRTIGLDNMQGPVAGKFIAERYKVKTIAVLHDKQQYGEGIATEVKKTVEDAGIKVAVFEGLNAGDKDFNALISKLKKAGVQFVYFGGYHPEMGLLLRQAKQAGLDARFMGPEGVGNSEITAIAGDASEGMLATLPRAFEQDPKNKALIDAFKAKNQDPSGIFVLPAYSAVTVIAKGIEKAGEADPEKVAEALRANTFETPTGNLGFDEKGDLKNFDFTVYEWHKDATRTEVK")
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].query_start, 2)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].sbjct_start, 6)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].sbjct_end, 373)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].query, "IVGAMSGPVAQWGDMEFN---GARQAIKDINAKGGIKGDKLVGVEYDDACDP-KQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQY---IMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIV--FFDGITAGEKDFSALIARLQKENIDFVY---YGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQD-PANKAIVEALKADKKDPSGPYVWITYAAVQS--LATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGD")
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].match, "++G +        D+E +   GA  A++ +N +GG+ G  +  +  D   DP +  +   + I N G+++++G   S + +    + E    L+  P          G++Y   I+      +      A Y++     +R+  I     Y       ++   +Q    ++   +  +   + D    + R+ +   D V+    G    E+ + + +   +G +   +       A ++ +    AEG +V  P     D PA++A V+A      + +    W   A  Q+  L  A   + + R  D+ + L     D   GP++ + + +")
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].sbjct, "LIGLLFSETGVTADIERSHAYGALLAVEQLNREGGVGGRPIETLSQDPGGDPDRYRLCAEDFIRNRGVRFLVGCYMSHTRKAVMPVVERADALLCYP------TPYEGFEYSPNIVYGGPAPNQNSAPLAAYLIRHY-GERVVFIGSDYIYPRESNHVMRHLYRQHGGTVLEEIYIPLYPSDDDLQRAVERIYQARADVVFSTVVGTGTAELYRAIARRYGDGRRPP-IASLTTSEAEVAKMESDVAEGQVVVAPYFSSIDTPASRAFVQACHGFFPENATITAWAEAAYWQTLLLGRAAQAAGNWRVEDVQRHLYDIDIDAPQGPVRVERQNN")
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].query_start, 30)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].query_end, 348)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].sbjct_start, 9)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].sbjct_end, 334)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].query, "GPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATA")
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].match, "G +G  + ++ + AG     +L  + K  ++DP N  +V  +KA KK+     +W  Y   +  A A")
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].sbjct, "GKQGEAHLAMRHYAGTVRYNVLNWLEK--NKDPLNDTVVSVMKASKKNDLLVEIWQDYTTQEEAAAA")
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].query_start, 247)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].query_end, 313)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].sbjct_start, 570)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].sbjct_end, 634)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].query, "ASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKP------QRIAIIHDKQQYGEGLARSVQDGLKQGNA")
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].match, "A  +Y D  +L++     N  L   G Q +M+       +G T    +L T +P      Q+I I+H+ QQ   GLAR V   L+Q +A")
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].sbjct, "ARAMYGDPCLLILDE--PNASLDSEGDQALMQAIVALQKRGATV---VLITHRPALTTLAQKILILHEGQQQRMGLARDVLTELQQRSA")
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].query_start, 108)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].query_end, 190)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].sbjct_start, 478)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].sbjct_end, 561)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRG-YQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSA----LIARLQKE")
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].match, "V+G   SS +   +++     I  +SP +TN +L+ +  ++Y  RT   D  Q   A   I    K   +++++   +YGE  A + +   ++    I   + I   ++ F+     L+ +LQ E")
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].sbjct, "VVGGSYSSVSVQLANLLRLFRIAQVSPASTNADLSDKNRFEYFARTVPSDDYQA-MAMVEIAVKFKWSYVSLVYSADEYGELGADAFKKEARKKGICIALEERIQNKKESFTESINNLVQKLQPE")
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].query_end, 215)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].sbjct_start, 196)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].sbjct_end, 319)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].query, "KDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLK-TQFMGPEGVGN-ASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKK")
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].match, "+ F A+ A +  E++  +  GG+  EM  + R +  +G+  ++F+ P+ +      +   GG    +L T    Y    A   +VEA+  DKK")
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].sbjct, "RTFIAMEAGVSVEDVQAMLMGGHGDEMVPLPRFSTISGIPVSEFIAPDRLAQIVERTRKGGGEIVNLLKTGSAYYAPAAATAQMVEAVLKDKK")
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].query_start, 203)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].query_end, 293)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].sbjct_start, 153)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].sbjct_end, 245)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].query, "QQYGEGLARS-----VQDGLKQGNANIVFFDGITAGEKDFSALIARL-QKENIDFVYYG---GYY")
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].match, "Q YG GL  S     +   LK GNA +     IT G  +F+ L   L +K N  F+ YG   G+Y")
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].sbjct, "QSYGHGLVISDNFVSISKPLKVGNAQLGTDGNITGGSGNFANLNTTLNRKVNSGFITYGATSGWY")
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].query_start, 171)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].query_end, 226)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].sbjct_start, 716)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].sbjct_end, 780)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].query, "AIVEALKADKKDPSGPYVWITYA-AVQSLATAMTRSASHRPLDLVKDLKANGADT")
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].match, "AI  A +AD+ D    Y +  +A A++S+  A+  ++   P+D + DLKA   +T")
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].sbjct, "AIQVAKEADRIDGIEQYAFRAFADALESIPMALAENSGLAPIDALSDLKAKQIET")
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].query_start, 283)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].query_end, 336)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].sbjct_start, 429)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].sbjct_end, 483)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].query, "KQAVAVANKIVN-DGIQYVIGHLCSSSTQPASDIYEDEGILMISPGA--TNPE-------LTQRGYQYIMRTAGLDSSQGPTAAKYILET")
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].match, "K  ++ A ++V  +G+ Y +GH  +S         +DEG ++  PG   TN E       +  + Y+  + +AG        A K+I ET")
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].sbjct, "KDVLSNAEEVVEANGLFYAVGHDPASGLVKGQVELDDEGYIITKPGTSFTNVEGVFACGDVQDKRYRQAITSAGSGCVAALEAEKFIAET")
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].query_start, 79)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].query_end, 158)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].sbjct_start, 235)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].sbjct_end, 324)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].match, "MKR AKTIIAG++ALA+S  AMADDIKVA+VGAMSGP+AQWG MEFNGA QAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGI+YVIGHLCSSSTQPASDIYEDEGILMISPGAT PELTQRGYQ+IMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLK  NAN+VFFDGITAGEKDFSALIARL+KENIDFVYYGGYYPEMGQ++RQAR+ GLKTQFMGPEGVGNASLSNIAG AAEGMLVTMPKRYDQDPAN+ IV+ALKADKKDPSGPYVWITYAAVQSLATA+ R+ S  PL LVKDLKANGA+TVIGPL WDEKGDLKGF+FGVFQWHADGSST AK")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct, "MKRNAKTIIAGMIALAISHTAMADDIKVAVVGAMSGPIAQWGIMEFNGAEQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIKYVIGHLCSSSTQPASDIYEDEGILMISPGATAPELTQRGYQHIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKAANANVVFFDGITAGEKDFSALIARLKKENIDFVYYGGYYPEMGQMLRQARSVGLKTQFMGPEGVGNASLSNIAGDAAEGMLVTMPKRYDQDPANQGIVDALKADKKDPSGPYVWITYAAVQSLATALERTGSDEPLALVKDLKANGANTVIGPLNWDEKGDLKGFDFGVFQWHADGSSTAAK")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct_end, 369)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query, "RKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].match, " K KT++AG +AL++S  A ADDIKVA+VGAMSGPVAQ+GD EF GA QAI DINAKGGIKGDKLV V+YDDACDPKQAVAVANK+VNDGI+YVIGHLCSSSTQPASDIYEDEGILMI+P AT PELT RGY+ ++RT GLDS QGPTAAKYILE VKPQRIAIIHDKQQYGEGLAR+VQDGLK+G  N+VFFDGITAGEKDFS L+ARL+KENIDFVYYGGY+PEMGQI+RQ+RA GLKTQFMGPEGV N SLSNIAG +AEG+LVT PK YDQ PANK IV+A+KA K+DPSG +VW TYAA+QSL   +  S    P ++ K LK    DTV+GPL WDEKGDLKGFEFGVF WHA+G++T AK")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct, "MKGKTLLAGCIALSLSHMAFADDIKVAVVGAMSGPVAQYGDQEFTGAEQAIADINAKGGIKGDKLVAVKYDDACDPKQAVAVANKVVNDGIKYVIGHLCSSSTQPASDIYEDEGILMITPAATAPELTARGYKLVLRTTGLDSDQGPTAAKYILEKVKPQRIAIIHDKQQYGEGLARAVQDGLKKGGVNVVFFDGITAGEKDFSTLVARLKKENIDFVYYGGYHPEMGQILRQSRAAGLKTQFMGPEGVANVSLSNIAGESAEGLLVTKPKNYDQVPANKPIVDAIKAKKQDPSGAFVWTTYAALQSLQAGLNHSD--DPAEIAKYLKGATVDTVMGPLSWDEKGDLKGFEFGVFDWHANGTATDAK")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct_end, 365)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].match, "M  K K ++AG +AL++S  A A DIKVA+VGAMSGPVAQ+GD EF GA QAI DINAKGG+KGDKLV V+YDDACDPKQAVAVANK+VNDGI+YVIGHLCSSSTQPASDIYEDEGILMI+P AT PELT RGY  ++RT GLDS QGPTAAKYI+E VKP+RIAI+HDKQQYGEGLARSVQD LK+ NA++VFFDGITAGEKDFS L+ARL+KENIDFVYYGGY+PEMGQI+RQARA GLKTQFMGPEGV N SLSNIAG +AEG+LVT PK YDQ PANK IV+A+KA K+DPSG +VW TYAA+QSL   + +S    P ++ K LKAN  +TV+GPL WD KGDLKGFEFGVF WHA+G++T AK")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct, "MNMKGKALLAGCIALSLSNMAFAKDIKVAVVGAMSGPVAQYGDQEFTGAEQAIADINAKGGVKGDKLVMVKYDDACDPKQAVAVANKVVNDGIKYVIGHLCSSSTQPASDIYEDEGILMITPAATAPELTARGYNLVLRTTGLDSDQGPTAAKYIVEKVKPKRIAIVHDKQQYGEGLARSVQDNLKKANADVVFFDGITAGEKDFSTLVARLKKENIDFVYYGGYHPEMGQILRQARAAGLKTQFMGPEGVANVSLSNIAGESAEGLLVTKPKNYDQVPANKPIVDAIKAKKQDPSGAFVWTTYAALQSLQAGLNQSD--DPAEIAKYLKANSVETVMGPLSWDAKGDLKGFEFGVFDWHANGTATDAK")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct_end, 367)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].match, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct_end, 369)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].match, "M  K K ++AG++ALA S  A+A+DIKVA+VGAMSGPVAQ+GD EF GA QA+ DINAKGGIKG+KL   +YDDACDPKQAVAVANK+VNDGI+YVIGHLCSSSTQPASDIYEDEGILMI+P AT PELT RGYQ I+RT GLDS QGPTAAKYILE VKPQRIAI+HDKQQYGEGLAR+VQDGLK+GNAN+VFFDGITAGEKDFS L+ARL+KENIDFVYYGGY+PEMGQI+RQARA GLKTQFMGPEGV N SLSNIAG +AEG+LVT PK YDQ PANK IV+A+KA K+DPSG +VW TYAA+QSL   + +S    P ++ K LKAN  DTV+GPL WDEKGDLKGFEFGVF WHA+G++T AK")
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].sbjct, "MNTKGKALLAGLIALAFSNMALAEDIKVAVVGAMSGPVAQYGDQEFTGAEQAVADINAKGGIKGNKLQIAKYDDACDPKQAVAVANKVVNDGIKYVIGHLCSSSTQPASDIYEDEGILMITPAATAPELTARGYQLILRTTGLDSDQGPTAAKYILEKVKPQRIAIVHDKQQYGEGLARAVQDGLKKGNANVVFFDGITAGEKDFSTLVARLKKENIDFVYYGGYHPEMGQILRQARAAGLKTQFMGPEGVANVSLSNIAGESAEGLLVTKPKNYDQVPANKPIVDAIKAKKQDPSGAFVWTTYAALQSLQAGLNQSD--DPAEIAKYLKANSVDTVMGPLTWDEKGDLKGFEFGVFDWHANGTATDAK")
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].sbjct_end, 367)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].query, "KRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].match, "+R ++   A  +A   S    AD IK+A+ G ++GPVAQ+GDM+  GA  AI+ IN  GG+ G +L GV YDDACDPKQAVAVANK+VNDG+++V+GH+CSSSTQPA+DIYEDEG+LMI+P AT PE+T RGY+ I RT GLD+ QGP A K+I E  K + IA++HDKQQYGEG+A  V+  ++     +  F+G+ AG+KDF+ALI++L+K  + FVY+GGY+PEMG ++RQA+  GL  +FMGPEGVGN+ ++ IAG A+EGML T+P+ ++QDP NKA+++A KA  +DPSG +V   Y+AV  +A  + ++    P  + + L+AN  +T  G L +DEKGDLK F+F V++WH D + T  K")
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].sbjct, "QRLSRLFAAMAIAGFASYSMAADTIKIALAGPVTGPVAQYGDMQRAGALMAIEQINKAGGVNGAQLEGVIYDDACDPKQAVAVANKVVNDGVKFVVGHVCSSSTQPATDIYEDEGVLMITPSATAPEITSRGYKLIFRTIGLDNMQGPVAGKFIAERYKVKTIAVLHDKQQYGEGIATEVKKTVEDAGIKVAVFEGLNAGDKDFNALISKLKKAGVQFVYFGGYHPEMGLLLRQAKQAGLDARFMGPEGVGNSEITAIAGDASEGMLATLPRAFEQDPKNKALIDAFKAKNQDPSGIFVLPAYSAVTVIAKGIEKAGEADPEKVAEALRANTFETPTGNLGFDEKGDLKNFDFTVYEWHKDATRTEVK")
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].query_start, 2)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].sbjct_start, 6)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].sbjct_end, 373)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].query, "SGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIV-NDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIV--FFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQ-ARANGLKTQ-FMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQD-PANKAIVEALKADKKDPSGPYVW--ITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGP")
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].match, "+G  A        GA  A++ +N +GG+ G  +  +  D   DP +    A   + N G+++++G   S + +    + E    L+  P  T  E  +     +      + +  P AA  I      +R+  I     Y       ++   +Q    ++   +  +   + D    + R+ +   D V+         ++ R  AR  G   +  +       A ++ +    AEG +V  P     D PA++A V+A      + +    W    Y     L  A   + + R  D+ + L     D   GP")
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].sbjct, "TGVTADIERSHAYGALLAVEQLNREGGVGGRPIETLSQDPGGDPDRYRLCAEDFIRNRGVRFLVGCYMSHTRKAVMPVVERADALLCYP--TPYEGFEYSPNIVYGGPAPNQNSAPLAAYLI--RHYGERVVFIGSDYIYPRESNHVMRHLYRQHGGTVLEEIYIPLYPSDDDLQRAVERIYQARADVVFSTVVGTGTAELYRAIARRYGDGRRPPIASLTTSEAEVAKMESDVAEGQVVVAPYFSSIDTPASRAFVQACHGFFPENATITAWAEAAYWQTLLLGRAAQAAGNWRVEDVQRHLYDIDIDAPQGP")
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].query_start, 35)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].query_end, 340)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].sbjct_start, 17)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].sbjct_end, 326)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRG-YQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGI--TAGEKDFSALIARL-QKENIDFVYYGGYYPEMGQIVRQARANGLKTQF")
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].match, "VIG   SS +   +++     I  +SP +T   L+ +  +    RT   D+ Q   A   IL+      ++ IH +  YGE    ++     + N  I   + +   A +K F ++I++L +K N   V       +  +I++ A+   L   F")
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].sbjct, "VIGGSYSSVSLQVANLLRLFHIPQVSPASTAKTLSDKTRFDLFARTVPPDTFQS-VALVDILKNFNWSYVSTIHSEGSYGEYGIEALHKEATERNVCIAVAEKVPSAADDKVFDSIISKLQKKPNARGVVLFTRAEDARRILQAAKRANLSQPF")
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].query_end, 245)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].sbjct_start, 152)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].sbjct_end, 304)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRG-YQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGI--TAGEKDFSALIARLQ")
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].match, "VIG   SS      ++ +   I  I+  AT+ +L+ +  Y+Y +R    D+ Q   A   I++      ++ +H +  YGE    + ++   Q    I   D I   AGEK F  L+ +L+")
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].sbjct, "VIGPGSSSVAIQVQNLLQLFDIPQIAYSATSIDLSDKTLYKYFLRVVPSDTLQA-RAMLDIVKRYNWTYVSAVHTEGNYGESGMDAFKELAAQEGLCIAHSDKIYSNAGEKSFDRLLRKLR")
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].query_end, 213)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].sbjct_start, 159)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].sbjct_end, 278)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRG-YQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGI--TAGEKDFSALIARLQ")
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].match, "VIG   SS      ++ +   I  I+  AT+ +L+ +  Y+Y +R    D+ Q   A   I++      ++ +H +  YGE    + ++   Q    I   D I   AGEK F  L+ +L+")
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].sbjct, "VIGPGSSSVAIQVQNLLQLFDIPQIAYSATSIDLSDKTLYKYFLRVVPSDTLQA-RAMLDIVKRYNWTYVSAVHTEGNYGESGMDAFKELAAQEGLCIAHSDKIYSNAGEKSFDRLLRKLR")
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].query_end, 213)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].sbjct_start, 159)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].sbjct_end, 278)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRG-YQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGI--TAGEKDFSALIARLQ")
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].match, "VIG   SS      ++ +   I  I+  AT+ +L+ +  ++Y MR    D+ Q   A   I++      ++ +H +  YGE    + +D   +    I     I   AGE+ F  L+ +L+")
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].sbjct, "VIGPGSSSVAIQVQNLLQLFNIPQIAYSATSMDLSDKTLFKYFMRVVPSDAQQA-RAMVDIVKRYNWTYVSAVHTEGNYGESGMEAFKDMSAKEGICIAHSYKIYSNAGEQSFDKLLKKLR")
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].query_end, 213)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].sbjct_start, 145)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].sbjct_end, 264)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRG-YQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGI--TAGEKDFSALIARL")
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].match, "VIG   SS      ++ +   I  I+  AT+ +L+ +  ++Y MR    D+ Q   A   I++      ++ +H +  YGE    + +D   +    I     I   AGE+ F  L+ +L")
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].sbjct, "VIGPGSSSVAIQVQNLLQLFNIPQIAYSATSMDLSDKTLFKYFMRVVPSDAQQA-RAMVDIVKRYNWTYVSAVHTEGNYGESGMEAFKDMSAKEGICIAHSYKIYSNAGEQSFDKLLKKL")
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].query_end, 212)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].sbjct_start, 146)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].sbjct_end, 264)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRG-YQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDF----SALIARLQKE")
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].match, "V+G   SS +   +++     I  +SP +TN +L+ +  ++Y  RT   D  Q   A   I    K   +++++   +YGE  A + +   ++    I   + I   ++ F    + L+ +LQ E")
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].sbjct, "VVGGSYSSVSVQLANLLRLFRIAQVSPASTNADLSDKNRFEYFARTVPSDDYQA-MAMVEIAVKFKWSYVSLVYSADEYGELGADAFKKEARKKGICIALEERIQNKKESFTESINNLVQKLQPE")
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].query_end, 215)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].sbjct_start, 196)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].sbjct_end, 319)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].query, "AGIVALAVSQGAMADDIKVAI---VGAMSGPVAQWGDMEFNGARQAIKDINAKGG----IKGDKLVGV")
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].match, "A   A   S   +   I  AI    G+ +GPV Q+ +   NG+  A       GG      G  L GV")
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].sbjct, "AAAAAGEASHVVVGGSIDAAIDTAKGSRAGPVEQYVNQAANGSLIAAASALVAGGGTEDAAGAILAGV")
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].query_start, 10)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].query_end, 70)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].sbjct_start, 315)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].sbjct_end, 382)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].query, "AIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGL---DSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDF")
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].match, "AI+     G  K  K+V V  D+A   K+ V V    +    ++  G +C++S    S+IY+   IL  +P  +  ++ +   +   +  G+   DS   P     +   +    I  + D++   +   R ++         +     +  GE D    +  ++  N+ F")
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].sbjct, "AIELAKKTG--KDPKVVQVILDEA---KEIVKVGKNFIITETKH--GFVCANSGVDESNIYKGIKILPKNPDESAEKIRKEIEKLTGKRVGVIISDSVGRPFRKGAVGIAIGVSGILALWDRKGEKDLFGRELKTTEVAIADELASMANVVMGEADEGIPVVIIRGANVPF")
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].query_start, 52)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].query_end, 219)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].sbjct_start, 67)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].sbjct_end, 230)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].query, "GPTAAKYILETVKP---QRIAIIHDK--QQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSAL--IARLQKENID-FVYYGGYYP")
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].match, "GP A   + E  K    ++  ++ DK  +   +G        L++   ++V FDG+    KD +    +   +KE+ D  V  GG  P")
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].sbjct, "GPNAISVVGERCKLLGGKKALLVTDKGLRAIKDGAVDKTLTHLREAGIDVVVFDGVEPNPKDTNVRDGLEVFRKEHCDIIVTVGGGSP")
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].query_start, 148)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].query_end, 227)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].sbjct_start, 17)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].sbjct_end, 104)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].query, "DGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQR-GYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGE-GLARSVQDGLKQGNANIVFFDGITAGEK--DFSALIAR-LQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFM--GPEGVGNASLSNIA-----GGAAEGMLVTMPKR")
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].match, "D I  VIG   SS +   ++I     I  IS  +T PEL+    Y +  R    DS Q   A   I+  +    ++ +  +  YGE G+    Q   + G   I     I    +  +F  +I R L+  N   V       ++ +I+  A+       F+  G +  G    S IA        AEG +  +PKR")
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].sbjct, "DKISGVIGAAASSVSIMVANILRLFKIPQISYASTAPELSDNTRYDFFSRVVPPDSYQA-QAMVDIVTALGWNYVSTLASEGNYGESGVEAFTQISREIGGVCIAQSQKIPREPRPGEFEKIIKRLLETPNARAVIMFANEDDIRRILEAAKKLNQSGHFLWIGSDSWG----SKIAPVYQQEEIAEGAVTILPKR")
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].query_start, 91)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].query_end, 274)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].sbjct_start, 145)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].sbjct_end, 335)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].query, "DGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQR-GYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGE-GLARSVQDGLKQGNANIVFFDGITAGEK--DFSALIAR-LQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFM--GPEGVGNASLSNIA-----GGAAEGMLVTMPKR")
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].match, "D I  VIG   SS +   ++I     I  IS  +T PEL+    Y +  R    DS Q   A   I+  +    ++ +  +  YGE G+    Q   + G   I     I    +  +F  +I R L+  N   V       ++ +I+  A+       F+  G +  G    S IA        AEG +  +PKR")
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].sbjct, "DKISGVIGAAASSVSIMVANILRLFKIPQISYASTAPELSDNTRYDFFSRVVPPDSYQA-QAMVDIVTALGWNYVSTLASEGNYGESGVEAFTQISREIGGVCIAQSQKIPREPRPGEFEKIIKRLLETPNARAVIMFANEDDIRRILEAAKKLNQSGHFLWIGSDSWG----SKIAPVYQQEEIAEGAVTILPKR")
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].query_start, 91)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].query_end, 274)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].sbjct_start, 145)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].sbjct_end, 335)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].query, "DGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQR-GYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGE-GLARSVQDGLKQGNANIVFFDGITAGEK--DFSALIAR-LQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFM--GPEGVGNASLSNIA-----GGAAEGMLVTMPKR")
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].match, "D I  VIG   SS +   ++I     I  IS  +T PEL+    Y +  R    DS Q   A   I+  +    ++ +  +  YGE G+    Q   + G   I     I    +  +F  +I R L+  N   V       ++  I+  A+       F+  G +  G    S IA        AEG +  +PKR")
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].sbjct, "DKISGVIGAAASSVSIMVANILRLFKIPQISYASTAPELSDNTRYDFFSRVVPPDSYQA-QAMVDIVTALGWNYVSTLASEGNYGESGVEAFTQISREIGGVCIAQSQKIPREPRPGEFEKIIKRLLETPNARAVIMFANEDDIRGILEAAKKLNQSGHFLWIGSDSWG----SKIAPVYQQEEIAEGAVTILPKR")
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].query_start, 91)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].query_end, 274)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].sbjct_start, 145)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].sbjct_end, 335)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].query, "QRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSAL--IARLQKENIDFVY-YGGYYP")
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].match, "+   I+ D      G+ + V D LK    N   +DG+       + L  +  L+  N DFV   GG  P")
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].sbjct, "KNALIVSDAFMNKSGVVKQVADLLKAQGINSAVYDGVMPNPTVTAVLEGLKILKDNNSDFVISLGGGSP")
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].query_start, 162)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].query_end, 227)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].sbjct_start, 32)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].sbjct_end, 100)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].query, "GPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSA")
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].match, "G +G  + ++ + AG     +L  + K   +DP N  +V  +KA KK+     +W  Y   +  A A     ")
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].sbjct, "GKQGEAHLAMRHYAGTVRYNVLNWLEKN--KDPLNDTVVSVMKASKKNDLLVEIWQDYTTQEEAAAAAKAGG")
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].query_start, 247)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].query_end, 318)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].sbjct_start, 570)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].sbjct_end, 639)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].query, "MLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKG")
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].match, "+++T+P   ++  AN ++     +++ D S  Y+  T      L T +    S       +  +      + GP+  D+ GD+  ")
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].sbjct, "LVLTLPP--EKFIANASVSGRFPSERSDFSLAYLEGTLLFGHMLQTFLENGESVTTPKFARAFRNLTFQGLEGPVTLDDSGDIDN")
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].query_start, 267)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].query_end, 351)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].sbjct_start, 294)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].sbjct_end, 376)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].query, "TVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQK")
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].match, " +K ++ +++  K   G+G+ R  +    +  A +    G+   +++   L A L+K")
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].sbjct, "RIKEEKASVVRPK---GDGVVRIQKQTSGRKGAGVSVITGLDLSDEELKKLAAELKK")
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].query_start, 158)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].query_end, 214)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].sbjct_start, 14)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].sbjct_end, 67)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].query, "AIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGE-KDFSALIARLQKENIDFVYYGGYYPE-MGQIVRQARANGL")
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].match, "AI  D Q Y  G+ + VQD  K  +  +   +    G+    S  +  L   N+D +           + VR+A   G+")
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].sbjct, "AIYLDTQGYYAGVRQGVQDAAKDSSVQVQLIETNAQGDISKESTFVDTLVARNVDAIILSAVSENGSSRTVRRASEAGI")
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].query_start, 165)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].query_end, 241)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].sbjct_start, 35)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].sbjct_end, 113)
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].query, "RKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].match, " K KT++AG +AL++S  A ADDIKVA+VGAMSGPVAQ+GD EF GA QAI DINAKGGIKGDKLV V+YDDACDPKQAVAVANK+VNDGI+YVIGHLCSSSTQPASDIYEDEGILMI+P AT PELT RGY+ ++RT GLDS QGPTAAKYILE VKPQRIAIIHDKQQYGEGLAR+VQDGLK+G  N+VFFDGITAGEKDFS L+ARL+KENIDFVYYGGY+PEMGQI+RQ+RA GLKTQFMGPEGV N SLSNIAG +AEG+LVT PK YDQ PANK IV+A+KA K+DPSG +VW TYAA+QSL   +  S    P ++ K LK    DTV+GPL WDEKGDLKGFEFGVF WHA+G++T AK")
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].sbjct, "MKGKTLLAGCIALSLSHMAFADDIKVAVVGAMSGPVAQYGDQEFTGAEQAIADINAKGGIKGDKLVAVKYDDACDPKQAVAVANKVVNDGIKYVIGHLCSSSTQPASDIYEDEGILMITPAATAPELTARGYKLVLRTTGLDSDQGPTAAKYILEKVKPQRIAIIHDKQQYGEGLARAVQDGLKKGGVNVVFFDGITAGEKDFSTLVARLKKENIDFVYYGGYHPEMGQILRQSRAAGLKTQFMGPEGVANVSLSNIAGESAEGLLVTKPKNYDQVPANKPIVDAIKAKKQDPSGAFVWTTYAALQSLQAGLNHSD--DPAEIAKYLKGATVDTVMGPLSWDEKGDLKGFEFGVFDWHANGTATDAK")
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].sbjct_end, 365)
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].match, "MKR AKTIIAG++ALA+S  AMADDIKVA+VGAMSGP+AQWG MEFNGA QAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGI+YVIGHLCSSSTQPASDIYEDEGILMISPGAT PELTQRGYQ+IMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLK  NAN+VFFDGITAGEKDFSALIARL+KENIDFVYYGGYYPEMGQ++RQAR+ GLKTQFMGPEGVGNASLSNIAG AAEGMLVTMPKRYDQDPAN+ IV+ALKADKKDPSGPYVWITYAAVQSLATA+ R+ S  PL LVKDLKANGA+TVIGPL WDEKGDLKGF+FGVFQWHADGSST AK")
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].sbjct, "MKRNAKTIIAGMIALAISHTAMADDIKVAVVGAMSGPIAQWGIMEFNGAEQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIKYVIGHLCSSSTQPASDIYEDEGILMISPGATAPELTQRGYQHIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKAANANVVFFDGITAGEKDFSALIARLKKENIDFVYYGGYYPEMGQMLRQARSVGLKTQFMGPEGVGNASLSNIAGDAAEGMLVTMPKRYDQDPANQGIVDALKADKKDPSGPYVWITYAAVQSLATALERTGSDEPLALVKDLKANGANTVIGPLNWDEKGDLKGFDFGVFQWHADGSSTAAK")
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].sbjct_end, 369)
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].match, "M  K K ++AG++ALA S  A+A+DIKVA+VGAMSGPVAQ+GD EF GA QA+ DINAKGGIKG+KL   +YDDACDPKQAVAVANK+VNDGI+YVIGHLCSSSTQPASDIYEDEGILMI+P AT PELT RGYQ I+RT GLDS QGPTAAKYILE VKPQRIAI+HDKQQYGEGLAR+VQDGLK+GNAN+VFFDGITAGEKDFS L+ARL+KENIDFVYYGGY+PEMGQI+RQARA GLKTQFMGPEGV N SLSNIAG +AEG+LVT PK YDQ PANK IV+A+KA K+DPSG +VW TYAA+QSL   + +S    P ++ K LKAN  DTV+GPL WDEKGDLKGFEFGVF WHA+G++T AK")
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].sbjct, "MNTKGKALLAGLIALAFSNMALAEDIKVAVVGAMSGPVAQYGDQEFTGAEQAVADINAKGGIKGNKLQIAKYDDACDPKQAVAVANKVVNDGIKYVIGHLCSSSTQPASDIYEDEGILMITPAATAPELTARGYQLILRTTGLDSDQGPTAAKYILEKVKPQRIAIVHDKQQYGEGLARAVQDGLKKGNANVVFFDGITAGEKDFSTLVARLKKENIDFVYYGGYHPEMGQILRQARAAGLKTQFMGPEGVANVSLSNIAGESAEGLLVTKPKNYDQVPANKPIVDAIKAKKQDPSGAFVWTTYAALQSLQAGLNQSD--DPAEIAKYLKANSVDTVMGPLTWDEKGDLKGFEFGVFDWHANGTATDAK")
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].sbjct_end, 367)
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].match, "M  K K ++AG +AL++S  A A DIKVA+VGAMSGPVAQ+GD EF GA QAI DINAKGG+KGDKLV V+YDDACDPKQAVAVANK+VNDGI+YVIGHLCSSSTQPASDIYEDEGILMI+P AT PELT RGY  ++RT GLDS QGPTAAKYI+E VKP+RIAI+HDKQQYGEGLARSVQD LK+ NA++VFFDGITAGEKDFS L+ARL+KENIDFVYYGGY+PEMGQI+RQARA GLKTQFMGPEGV N SLSNIAG +AEG+LVT PK YDQ PANK IV+A+KA K+DPSG +VW TYAA+QSL   + +S    P ++ K LKAN  +TV+GPL WD KGDLKGFEFGVF WHA+G++T AK")
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].sbjct, "MNMKGKALLAGCIALSLSNMAFAKDIKVAVVGAMSGPVAQYGDQEFTGAEQAIADINAKGGVKGDKLVMVKYDDACDPKQAVAVANKVVNDGIKYVIGHLCSSSTQPASDIYEDEGILMITPAATAPELTARGYNLVLRTTGLDSDQGPTAAKYIVEKVKPKRIAIVHDKQQYGEGLARSVQDNLKKANADVVFFDGITAGEKDFSTLVARLKKENIDFVYYGGYHPEMGQILRQARAAGLKTQFMGPEGVANVSLSNIAGESAEGLLVTKPKNYDQVPANKPIVDAIKAKKQDPSGAFVWTTYAALQSLQAGLNQSD--DPAEIAKYLKANSVETVMGPLSWDAKGDLKGFEFGVFDWHANGTATDAK")
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].sbjct_end, 367)
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].query, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].match, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].sbjct, "MKRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].sbjct_end, 369)
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].query, "KRKAKTIIAGIVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDLKGFEFGVFQWHADGSSTVAK")
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].match, "+R ++   A  +A   S    AD IK+A+ G ++GPVAQ+GDM+  GA  AI+ IN  GG+ G +L GV YDDACDPKQAVAVANK+VNDG+++V+GH+CSSSTQPA+DIYEDEG+LMI+P AT PE+T RGY+ I RT GLD+ QGP A K+I E  K + IA++HDKQQYGEG+A  V+  ++     +  F+G+ AG+KDF+ALI++L+K  + FVY+GGY+PEMG ++RQA+  GL  +FMGPEGVGN+ ++ IAG A+EGML T+P+ ++QDP NKA+++A KA  +DPSG +V   Y+AV  +A  + ++    P  + + L+AN  +T  G L +DEKGDLK F+F V++WH D + T  K")
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].sbjct, "QRLSRLFAAMAIAGFASYSMAADTIKIALAGPVTGPVAQYGDMQRAGALMAIEQINKAGGVNGAQLEGVIYDDACDPKQAVAVANKVVNDGVKFVVGHVCSSSTQPATDIYEDEGVLMITPSATAPEITSRGYKLIFRTIGLDNMQGPVAGKFIAERYKVKTIAVLHDKQQYGEGIATEVKKTVEDAGIKVAVFEGLNAGDKDFNALISKLKKAGVQFVYFGGYHPEMGLLLRQAKQAGLDARFMGPEGVGNSEITAIAGDASEGMLATLPRAFEQDPKNKALIDAFKAKNQDPSGIFVLPAYSAVTVIAKGIEKAGEADPEKVAEALRANTFETPTGNLGFDEKGDLKNFDFTVYEWHKDATRTEVK")
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].query_start, 2)
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].query_end, 369)
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].sbjct_start, 6)
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].sbjct_end, 373)
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].query, "SGPVAQWGDMEFNGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIV-NDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIV--FFDGITAGEKDFSALIARLQKENIDFVYYGGYYPEMGQIVRQ-ARANGLKTQ-FMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQD-PANKAIVEALKADKKDPSGPYVW--ITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGP")
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].match, "+G  A        GA  A++ +N +GG+ G  +  +  D   DP +    A   + N G+++++G   S + +    + E    L+  P  T  E  +     +      + +  P AA  I      +R+  I     Y       ++   +Q    ++   +  +   + D    + R+ +   D V+         ++ R  AR  G   +  +       A ++ +    AEG +V  P     D PA++A V+A      + +    W    Y     L  A   + + R  D+ + L     D   GP")
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].sbjct, "TGVTADIERSHAYGALLAVEQLNREGGVGGRPIETLSQDPGGDPDRYRLCAEDFIRNRGVRFLVGCYMSHTRKAVMPVVERADALLCYP--TPYEGFEYSPNIVYGGPAPNQNSAPLAAYLI--RHYGERVVFIGSDYIYPRESNHVMRHLYRQHGGTVLEEIYIPLYPSDDDLQRAVERIYQARADVVFSTVVGTGTAELYRAIARRYGDGRRPPIASLTTSEAEVAKMESDVAEGQVVVAPYFSSIDTPASRAFVQACHGFFPENATITAWAEAAYWQTLLLGRAAQAAGNWRVEDVQRHLYDIDIDAPQGP")
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].query_start, 35)
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].query_end, 340)
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].sbjct_start, 17)
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].sbjct_end, 326)
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].query, "NGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELT-QRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIV---FFDGITAGEKDFSALIARLQKE-NIDFVYYGGYYPEMGQIVRQARANGLKTQFM----GPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWIT")
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].match, "    QA+  + A    +GD            P    A   ++V      V+G   SS +   +++     I  IS  +T PEL+    Y +  R    DS Q   A   I+  +    ++ +  +  YGE    +     ++     +             +FS +I RL +  N   +       ++ +++  AR   L   F+       G   + + ++    A G +  +PKR   D       +       + +   +W  ")
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].sbjct, "YALEQALSFVQALIRGRGDGDEVGVRCPGGVPPLRPAPPERVVA-----VVGASASSVSIMVANVLRLFAIPQISYASTAPELSDSTRYDFFSRVVPPDSYQA-QAMVDIVRALGWNYVSTLASEGNYGESGVEAFVQISREAGGVCIAQSIKIPREPKPGEFSKVIRRLMETPNARGIIIFANEDDIRRVLEAARQANLTGHFLWVGSDSWGAKTSPILSLE-DVAVGAITILPKRASID----GFDQYFMTRSLENNRRNIWFA")
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].query_start, 47)
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].query_end, 303)
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].sbjct_start, 104)
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].sbjct_end, 358)
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGE-GLARSVQDGLKQGNANIV--FFDGITAGEKDFSALIAR-LQKENIDFVYYGGYYPEMGQIVRQARANGLKTQF--MGPEGVGN--ASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWIT")
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].match, "VIG   SS +   ++I     I  IS  +T P+L+               +    A   I+  +K   ++ +  +  YGE G+   +Q   + G   I             +F  +I R L+  N   V       ++ +++  AR       F  MG +  G+  A + ++    AEG +  +PKR       +       +   D +   +W  ")
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].sbjct, "VIGASGSSVSIMVANILRLFKIPQISYASTAPDLSDNSRYDFFSRVVPSDTYQAQAMVDIVRALKWNYVSTVASEGSYGESGVEAFIQKSREDGGVCIAQSVKIPREPKAGEFDKIIRRLLETSNARAVIIFANEDDIRRVLEAARRANQTGHFFWMGSDSWGSKIAPVLHLE-EVAEGAVTILPKRMSV----RGFDRYFSSRTLDNNRRNIWFA")
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].query_end, 303)
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].sbjct_start, 153)
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].sbjct_end, 363)
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRG-YQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFD--GITAGEKDFSALIARL-QKENIDFVYYGGYYPEMGQIVRQARANGLKTQF--MGPEGVG-NASLSNIAGGAAEGMLVT")
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].match, "VIG   SS +   +++     I  +SP +T   L+ +  +    RT   D+ Q   A   IL+      ++ IH +  YGE    ++     + N  I   +     A +K F ++I++L +K N   V       +  +I++ A+   L   F  +  +G G    L       AEG +  ")
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].sbjct, "VIGGSYSSVSLQVANLLRLFHIPQVSPASTAKTLSDKTRFDLFARTVPPDTFQS-VALVDILKNFNWSYVSTIHSEGSYGEYGIEALHKEATERNVCIAVAEKVPSAADDKVFDSIISKLQKKPNARGVVLFTRAEDARRILQAAKRANLSQPFHWIASDGWGKQQKLLEGLEDIAEGAITV")
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].query_end, 270)
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].sbjct_start, 152)
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].sbjct_end, 332)
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGE-GLARSVQDGLKQGNANIV--FFDGITAGEKDFSALIAR-LQKENIDFVYYGGYYPEMGQIVRQARANGLKTQF--MGPEGVGNASLSNIA-GGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWIT")
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].match, "VIG   SS +   ++I     I  IS  +T P+L+               +    A   I+  +K   ++ +  +  YGE G+   +Q   + G   I             +F  +I R L+  N   +       ++ +++  AR       F  MG +  G+ S   +     AEG +  +PKR       +       +   D +   +W  ")
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].sbjct, "VIGASGSSVSIMVANILRLFKIPQISYASTAPDLSDNSRYDFFSRVVPSDTYQAQAMVDIVRALKWNYVSTLASEGSYGESGVEAFIQKSRENGGVCIAQSVKIPREPKTGEFDKIIKRLLETSNARGIIIFANEDDIRRVLEAARRANQTGHFFWMGSDSWGSKSAPVLRLEEVAEGAVTILPKRMSV----RGFDRYFSSRTLDNNRRNIWFA")
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].query_end, 303)
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].sbjct_start, 153)
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].sbjct_end, 363)
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].query, "NGARQAIKDINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELT-QRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIV---FFDGITAGEKDFSALIARLQKE-NIDFVYYGGYYPEMGQIVRQARANGLKTQFM----GPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWIT")
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].match, "    QA+  + A    +GD            P    A   ++V      V+G   SS +   +++     I  IS  +T PEL+    Y +  R    DS Q   A   I+  +    ++ +  +  YGE    +     ++     +             +F  +I RL +  N   +       ++ +++   R   L   F+       G   + + N+    A G +  +PKR   D       +       + +   +W  ")
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].sbjct, "YALEQALSFVQALIRGRGDGDEASVRCPGGVPPLRSAPPERVVA-----VVGASASSVSIMVANVLRLFAIPQISYASTAPELSDSTRYDFFSRVVPPDSYQA-QAMVDIVRALGWNYVSTLASEGNYGESGVEAFVQISREAGGVCIAQSIKIPREPKPGEFHKVIRRLMETPNARGIIIFANEDDIRRVLEATRQANLTGHFLWVGSDSWGSKISPILNLE-EEAVGAITILPKRASID----GFDQYFMTRSLENNRRNIWFA")
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].query_start, 47)
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].query_end, 303)
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].sbjct_start, 98)
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].sbjct_end, 352)
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].query, "DGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRG-YQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFF---DGITAGEKDFSALIAR-LQKENIDFVYYGGYYPEMGQIVRQARANGLKTQF--MGPEGVGNASLSNI--AGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWIT")
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].match, "D I  VIG   SS +   ++I     I  IS  +T PEL+    Y +  R    DS Q   A   I+  +    ++ +  +  YGE    +     ++     +             +F  +I R L+  N   V       ++ +I+  A+       F  +G +  G + ++ +      AEG +  +PKR   D  ++       A+ +      VW  ")
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].sbjct, "DKISGVIGAAASSVSIMVANILRLFKIPQISYASTAPELSDNTRYDFFSRVVPPDSYQA-QAMVDIVTALGWNYVSTLASEGNYGESGVEAFTQISREIGGVCIAQSQKIPREPRPGEFEKIIKRLLETPNARAVIMFANEDDIRRILEAAKKLNQSGHFLWIGSDSWG-SKIAPVYQQEEIAEGAVTILPKRASIDGFDRYFRSRTLANNRRN----VWFA")
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].query_start, 91)
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].query_end, 303)
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].sbjct_start, 145)
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].sbjct_end, 360)
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].query, "DGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRG-YQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFF---DGITAGEKDFSALIAR-LQKENIDFVYYGGYYPEMGQIVRQARANGLKTQF--MGPEGVGNASLSNI--AGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWIT")
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].match, "D I  VIG   SS +   ++I     I  IS  +T PEL+    Y +  R    DS Q   A   I+  +    ++ +  +  YGE    +     ++     +             +F  +I R L+  N   V       ++ +I+  A+       F  +G +  G + ++ +      AEG +  +PKR   D  ++       A+ +      VW  ")
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].sbjct, "DKISGVIGAAASSVSIMVANILRLFKIPQISYASTAPELSDNTRYDFFSRVVPPDSYQA-QAMVDIVTALGWNYVSTLASEGNYGESGVEAFTQISREIGGVCIAQSQKIPREPRPGEFEKIIKRLLETPNARAVIMFANEDDIRRILEAAKKLNQSGHFLWIGSDSWG-SKIAPVYQQEEIAEGAVTILPKRASIDGFDRYFRSRTLANNRRN----VWFA")
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].query_start, 91)
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].query_end, 303)
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].sbjct_start, 145)
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].sbjct_end, 360)
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].query, "DGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRG-YQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFF---DGITAGEKDFSALIAR-LQKENIDFVYYGGYYPEMGQIVRQARANGLKTQF--MGPEGVGNASLSNI--AGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWIT")
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].match, "D I  VIG   SS +   ++I     I  IS  +T PEL+    Y +  R    DS Q   A   I+  +    ++ +  +  YGE    +     ++     +             +F  +I R L+  N   V       ++  I+  A+       F  +G +  G + ++ +      AEG +  +PKR   D  ++       A+ +      VW  ")
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].sbjct, "DKISGVIGAAASSVSIMVANILRLFKIPQISYASTAPELSDNTRYDFFSRVVPPDSYQA-QAMVDIVTALGWNYVSTLASEGNYGESGVEAFTQISREIGGVCIAQSQKIPREPRPGEFEKIIKRLLETPNARAVIMFANEDDIRGILEAAKKLNQSGHFLWIGSDSWG-SKIAPVYQQEEIAEGAVTILPKRASIDGFDRYFRSRTLANNRRN----VWFA")
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].query_start, 91)
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].query_end, 303)
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].sbjct_start, 145)
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].sbjct_end, 360)
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].query, "DGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDG----------LKQGNANIVFFDGITA-GEKDFS----ALIARLQKENIDFVYYGGYYPEMGQIVR-QARANGLKTQFMGPEGVGNASLSNIAGG-AAEGMLVTMPKRYDQDPANKAIVEALKADKKD")
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].match, "DG   V  H   SS    S  Y D G  M +   T   + Q      +        QGP A      T +  ++ I  +   Y        +DG          +K  N     +  +T   E DF       + R+ +E+++F+      P    I R +  ++  + QF  PE  G   +        + G      K YD   AN   +  +   K +")
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].sbjct, "DGHMVVRSHARVSSLTLKSIQYRDAGEYMCTASNT---IGQDSQSIDLEFQYAPKLQGPVAVY----TWEGNQVNITCEVFAYPSATISWFRDGQLLPSSNYSNIKIYNTPSASYLEVTPDSENDFGNYNCTAVNRIGQESLEFILVQADTPSSPSIDRVEPYSSTAQVQFDEPEATGGVPILKYKAEWKSLGEESWHFKWYDAKEANMEGIVTIMGLKPE")
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].query_start, 91)
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].query_end, 294)
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].sbjct_start, 357)
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].sbjct_end, 570)
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGIT--AGEKDFSALIARLQK--ENIDFVYYGGYYPEMGQIVRQARANGLKTQFM")
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].match, "VIG   SS      ++ +   I  I+  AT+ +L+ +             +Q   A   I++      ++ +H +  YGE    + +D   +    I     I   AGE+ F  L+ +L+        V        +  ++   R  GL  +F+")
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].sbjct, "VIGPGSSSVAIQVQNLLQLFNIPQIAYSATSMDLSDKTLFKYFMRVVPSDAQQARAMVDIVKRYNWTYVSAVHTEGNYGESGMEAFKDMSAKEGICIAHSYKIYSNAGEQSFDKLLKKLRSHLPKARVVACFCEGMTVRGLLMAMRRLGLAGEFL")
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].query_end, 246)
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].sbjct_start, 145)
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].sbjct_end, 299)
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].query, "IVALAVSQGAMADDIKVAIVGAMSGPVAQWGDMEFNGARQAIKDINAKGGIKGDKL--VGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGITAGE")
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].match, "++ L  S  A +   KV ++G  +             AR A   +N    + G     V +  +    P    AV++ +    +  ++G +  ++ +PA  + ++ G+ ++  G                T     +    A   +L+  +  R+A+I   Q       R++   L+     +     +   +")
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].sbjct, "LLLLLPSPSAFSAVFKVGVLGPWACDPIFARARPDLAARLATDRLNRDLALDGGPWFEVTLLPEPCLTPGSLGAVSSALTR--VSGLVGPVNPAACRPAELLAQEAGVALVPWGCPGTRAA--------GTTAPAVTPAADALYVLLKAFRWARVALITAPQDLWVEAGRALSTALRARGLPVALVTSMVPSD")
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].query_start, 12)
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].query_end, 202)
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].sbjct_start, 43)
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].sbjct_end, 225)
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].query, "VIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDGLKQGNANIVFFDGIT--AGEKDFSALIARLQK--ENIDFVYYGGYYPEMGQIVRQARANGLKTQFM")
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].match, "VIG   SS      ++ +   I  I+  AT+ +L+ +             +Q   A   I++      ++ +H +  YGE    + +D   +    I     I   AGE+ F  L+ +L         V        +  ++   R  GL  +F+")
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].sbjct, "VIGPGSSSVAIQVQNLLQLFNIPQIAYSATSMDLSDKTLFKYFMRVVPSDAQQARAMVDIVKRYNWTYVSAVHTEGNYGESGMEAFKDMSAKEGICIAHSYKIYSNAGEQSFDKLLKKLTSHLPKARVVACFCEGMTVRGLLMAMRRLGLAGEFL")
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].query_start, 96)
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].query_end, 246)
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].sbjct_start, 146)
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].sbjct_end, 300)
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].query, "DINAKGGIKGDKLVGVEYDDACDPKQAVAVANKIVNDGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETV")
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].match, "D+N + G +G       Y  AC   +   +     N     V+   CS +  PA DI + EG  +   G    E      Q I+ T G D  + P       E  ")
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].sbjct, "DMNWEKG-EGQPFEYFVYGAACSEVEIDCLTGDHKNIRTDIVMDVGCSIN--PAIDIGQIEGAFIQGMGLYTIEELNYSPQGILHTRGPDQYKIPAICDMPTELH")
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].query_start, 55)
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].query_end, 159)
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].sbjct_start, 1148)
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].sbjct_end, 1249)
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].query, "AGIVALAVSQGAMADDIKVAI---VGAMSGPVAQWGDMEFNGARQAIKDINAKGG")
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].match, "A   A   S   +   I  AI    G+ +GPV Q+ +   NG+  A       GG")
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].sbjct, "AAAAAGEASHVVVGGSIDAAIDTAKGSRAGPVEQYVNQAANGSLIAAASALVAGG")
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].query_start, 10)
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].query_end, 61)
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].sbjct_start, 315)
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].sbjct_end, 369)
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].query, "DGIQYVIGHLCSSSTQPASDIYEDEGILMISPGATNPELTQRGYQYIMRTAGLDSSQGPTAAKYILETVKPQRIAIIHDKQQYGEGLARSVQDG----------LKQGNANIVFFDGITA-GEKDFS----ALIARLQKENIDFVYYGGYYPEMGQIVR-QARANGLKTQFMGPEGVGNASLSNIAGG")
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].match, "DG   V  H   SS    S  Y D G  M +   T   + Q      +        QGP A      T +  ++ I  +   Y        +DG          +K  N     +  +T   E DF       + R+ +E+++F+      P    I R +  ++  + QF  PE  G   +      ")
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].sbjct, "DGHMVVRSHARVSSLTLKSIQYRDAGEYMCTASNT---IGQDSQSIDLEFQYAPKLQGPVAVY----TWEGNQVNITCEVFAYPSATISWFRDGQLLPSSNYSNIKIYNTPSASYLEVTPDSENDFGNYNCTAVNRIGQESLEFILVQADTPSSPSIDRVEPYSSTAQVQFDEPEATGGVPILKYKAE")
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].query_start, 91)
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].query_end, 262)
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].sbjct_start, 357)
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].sbjct_end, 537)
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].query, "IARLQKENIDFVYYGGYYPEMGQ-----IVRQARANGLKTQFMGPEGVGNASLSNIAGGAAEGMLVTMPKRYDQDPANKAIVEALKADKKDPSGPYVWITYAAVQSLATAMTRSASHRPLDLVKDLKANGADTVIGPLKWDEKGDL")
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].match, "I +++K   + V + G     G+     + + A   G+ +    PE    + LS +A       +  M    + DP + A  +      ++  G    + Y  V +L        +     + K L+  GA  V      D+ G+L")
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].sbjct, "IEKMKKTGRNIVVFYGSQTGTGEEFANRLSKDAHRYGMGSMAADPEEYDMSELSRLAEIGNSLAIFCMATYGEGDPTDNA--QDFYDWLQETDGQLSGVNY-PVFALGDKTYEHYNAMGAYVDKRLEELGAKRVFDLGMGDDDGNL")
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].query_start, 209)
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].query_end, 349)
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].sbjct_start, 15)
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].sbjct_end, 157)

    def _check_text_2010L_phiblast_002_footer(self, record):
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Nov 15, 1999  2:55 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.319)
        self.assertAlmostEqual(record.ka_params[1], 0.118)
        self.assertAlmostEqual(record.ka_params[2], 0.300)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0413)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 51436041)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 1847150)
        self.assertEqual(record.num_good_extends, 5653)
        self.assertEqual(record.num_seqs_better_e, 61)
        self.assertEqual(record.hsps_no_gap, 21)
        self.assertEqual(record.hsps_prelim_gapped, 40)
        self.assertEqual(record.hsps_gapped, 65)
        self.assertEqual(record.query_length, 369)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 53)
        self.assertEqual(record.effective_query_length, 316)
        self.assertEqual(record.effective_database_length, 25292887)
        self.assertEqual(record.effective_search_space, 7992552292)
        self.assertEqual(record.effective_search_space_used, 7992552292)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.4)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.9)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 65)
        self.assertAlmostEqual(record.blast_cutoff[1], 29.9)

    def test_text_2010L_phiblast_003(self):
        "Test parsing PHI-BLAST, BLASTP 2.0.10 output, two rounds (text_2010L_phiblast_003)"

        path = os.path.join('Blast', 'text_2010L_phiblast_003.txt')
        handle = open(path)
        record = self.pb_parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|399896|sp|Q02134|HIS7_LACLA IMIDAZOLEGLYCEROL-PHOSPHATE\nDEHYDRATASE (IGPD)")
        self.assertEqual(record.query_letters, 200)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.rounds), 2)
        self.assertEqual(len(record.rounds[0].new_seqs), 30)
        # Rest of test broken up to avoid Jython JVM limitations
        self._check_text_2010L_phiblast_003_round0(record)
        self._check_text_2010L_phiblast_003_round1(record)
        self._check_text_2010L_phiblast_003_hsps(record)
        self._check_text_2010L_phiblast_003_hsps_details(record)
        self._check_text_2010L_phiblast_003_footer(record)

    def _check_text_2010L_phiblast_003_round0(self, record):
        self.assertEqual(record.rounds[0].new_seqs[0].title, "gi|399896|sp|Q02134|HIS7_LACLA IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[0].score, 409)
        self.assertAlmostEqual(record.rounds[0].new_seqs[0].e, 1.e-114)
        self.assertEqual(record.rounds[0].new_seqs[1].title, "gi|462273|sp|P34047|HIS7_ARATH IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[1].score, 198)
        self.assertAlmostEqual(record.rounds[0].new_seqs[1].e, 6e-51)
        self.assertEqual(record.rounds[0].new_seqs[2].title, "gi|2495230|sp|Q43072|HIS7_PEA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYD...")
        self.assertEqual(record.rounds[0].new_seqs[2].score, 196)
        self.assertAlmostEqual(record.rounds[0].new_seqs[2].e, 4e-50)
        self.assertEqual(record.rounds[0].new_seqs[3].title, "gi|123157|sp|P18787|HIS7_AZOBR IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[3].score, 185)
        self.assertAlmostEqual(record.rounds[0].new_seqs[3].e, 5.e-47)
        self.assertEqual(record.rounds[0].new_seqs[4].title, "gi|462275|sp|P34048|HIS7_WHEAT IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[4].score, 181)
        self.assertAlmostEqual(record.rounds[0].new_seqs[4].e, 8e-46)
        self.assertEqual(record.rounds[0].new_seqs[5].title, "gi|123161|sp|P16247|HIS7_STRCO IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[5].score, 178)
        self.assertAlmostEqual(record.rounds[0].new_seqs[5].e, 7e-45)
        self.assertEqual(record.rounds[0].new_seqs[6].title, "gi|462272|sp|Q05068|HIS7_ANASP IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[6].score, 178)
        self.assertAlmostEqual(record.rounds[0].new_seqs[6].e, 7e-45)
        self.assertEqual(record.rounds[0].new_seqs[7].title, "gi|123158|sp|P06987|HIS7_ECOLI HISTIDINE BIOSYNTHESIS BIFUNCTIO...")
        self.assertEqual(record.rounds[0].new_seqs[7].score, 175)
        self.assertAlmostEqual(record.rounds[0].new_seqs[7].e, 8e-44)
        self.assertEqual(record.rounds[0].new_seqs[8].title, "gi|1346293|sp|P48054|HIS7_SYNY3 IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[8].score, 174)
        self.assertAlmostEqual(record.rounds[0].new_seqs[8].e, 1e-43)
        self.assertEqual(record.rounds[0].new_seqs[9].title, "gi|1170286|sp|P44327|HIS7_HAEIN HISTIDINE BIOSYNTHESIS BIFUNCTI...")
        self.assertEqual(record.rounds[0].new_seqs[9].score, 168)
        self.assertAlmostEqual(record.rounds[0].new_seqs[9].e, 8e-42)
        self.assertEqual(record.rounds[0].new_seqs[10].title, "gi|2495224|sp|O06590|HIS7_MYCTU IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[10].score, 167)
        self.assertAlmostEqual(record.rounds[0].new_seqs[10].e, 2e-41)
        self.assertEqual(record.rounds[0].new_seqs[11].title, "gi|123160|sp|P10368|HIS7_SALTY HISTIDINE BIOSYNTHESIS BIFUNCTIO...")
        self.assertEqual(record.rounds[0].new_seqs[11].score, 166)
        self.assertAlmostEqual(record.rounds[0].new_seqs[11].e, 2e-41)
        self.assertEqual(record.rounds[0].new_seqs[12].title, "gi|2495226|sp|Q50504|HIS7_METTH PROBABLE IMIDAZOLEGLYCEROL-PHOS...")
        self.assertEqual(record.rounds[0].new_seqs[12].score, 153)
        self.assertAlmostEqual(record.rounds[0].new_seqs[12].e, 3e-37)
        self.assertEqual(record.rounds[0].new_seqs[13].title, "gi|729718|sp|P40919|HIS7_CRYNE IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[13].score, 152)
        self.assertAlmostEqual(record.rounds[0].new_seqs[13].e, 7e-37)
        self.assertEqual(record.rounds[0].new_seqs[14].title, "gi|3334215|sp|O33773|HIS7_SULSO PROBABLE IMIDAZOLEGLYCEROL-PHOS...")
        self.assertEqual(record.rounds[0].new_seqs[14].score, 151)
        self.assertAlmostEqual(record.rounds[0].new_seqs[14].e, 9e-37)
        self.assertEqual(record.rounds[0].new_seqs[15].title, "gi|123159|sp|P28624|HIS7_PHYPR IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[15].score, 149)
        self.assertAlmostEqual(record.rounds[0].new_seqs[15].e, 3e-36)
        self.assertEqual(record.rounds[0].new_seqs[16].title, "gi|729719|sp|P40374|HIS7_SCHPO IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[16].score, 136)
        self.assertAlmostEqual(record.rounds[0].new_seqs[16].e, 3e-32)
        self.assertEqual(record.rounds[0].new_seqs[17].title, "gi|2495227|sp|P56090|HIS7_CANAL IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[17].score, 128)
        self.assertAlmostEqual(record.rounds[0].new_seqs[17].e, 9e-30)
        self.assertEqual(record.rounds[0].new_seqs[18].title, "gi|2495225|sp|Q58109|HIS7_METJA PROBABLE IMIDAZOLEGLYCEROL-PHOS...")
        self.assertEqual(record.rounds[0].new_seqs[18].score, 126)
        self.assertAlmostEqual(record.rounds[0].new_seqs[18].e, 4e-29)
        self.assertEqual(record.rounds[0].new_seqs[19].title, "gi|399897|sp|Q02986|HIS7_SACKL IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[19].score, 125)
        self.assertAlmostEqual(record.rounds[0].new_seqs[19].e, 6e-29)
        self.assertEqual(record.rounds[0].new_seqs[20].title, "gi|2495229|sp|Q92447|HIS7_PICPA IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[20].score, 125)
        self.assertAlmostEqual(record.rounds[0].new_seqs[20].e, 6e-29)
        self.assertEqual(record.rounds[0].new_seqs[21].title, "gi|2495228|sp|Q12578|HIS7_CANGA IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[21].score, 123)
        self.assertAlmostEqual(record.rounds[0].new_seqs[21].e, 2e-28)
        self.assertEqual(record.rounds[0].new_seqs[22].title, "gi|2506514|sp|P06633|HIS7_YEAST IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[22].score, 122)
        self.assertAlmostEqual(record.rounds[0].new_seqs[22].e, 4e-28)
        self.assertEqual(record.rounds[0].new_seqs[23].title, "gi|462274|sp|P34041|HIS7_TRIHA IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[23].score, 106)
        self.assertAlmostEqual(record.rounds[0].new_seqs[23].e, 3e-23)
        self.assertEqual(record.rounds[0].new_seqs[24].title, "gi|1345641|sp|P49264|C7B1_THLAR CYTOCHROME P450 71B1 (CYPLXXIB1)")
        self.assertEqual(record.rounds[0].new_seqs[24].score, 35)
        self.assertAlmostEqual(record.rounds[0].new_seqs[24].e, 0.13)
        self.assertEqual(record.rounds[0].new_seqs[25].title, "gi|1731346|sp|Q10698|YY29_MYCTU PROBABLE DIPEPTIDASE CY49.29C")
        self.assertEqual(record.rounds[0].new_seqs[25].score, 32)
        self.assertAlmostEqual(record.rounds[0].new_seqs[25].e, 1.1)
        self.assertEqual(record.rounds[0].new_seqs[26].title, "gi|3287839|sp|Q01812|GLK4_RAT GLUTAMATE RECEPTOR, IONOTROPIC KA...")
        self.assertEqual(record.rounds[0].new_seqs[26].score, 30)
        self.assertAlmostEqual(record.rounds[0].new_seqs[26].e, 3.3)
        self.assertEqual(record.rounds[0].new_seqs[27].title, "gi|3123025|sp|Q94637|VIT6_OSCBR VITELLOGENIN 6 PRECURSOR")
        self.assertEqual(record.rounds[0].new_seqs[27].score, 29)
        self.assertAlmostEqual(record.rounds[0].new_seqs[27].e, 5.6)
        self.assertEqual(record.rounds[0].new_seqs[28].title, "gi|3287848|sp|Q16099|GLK4_HUMAN GLUTAMATE RECEPTOR, IONOTROPIC ...")
        self.assertEqual(record.rounds[0].new_seqs[28].score, 29)
        self.assertAlmostEqual(record.rounds[0].new_seqs[28].e, 9.7)
        self.assertEqual(record.rounds[0].new_seqs[29].title, "gi|1174406|sp|P36126|SP14_YEAST PHOSPHOLIPASE D1 (PLD 1) (CHOLI...")
        self.assertEqual(record.rounds[0].new_seqs[29].score, 29)
        self.assertAlmostEqual(record.rounds[0].new_seqs[29].e, 9.7)
        self.assertEqual(len(record.rounds[0].alignments), 30)
        self.assertEqual(record.rounds[0].alignments[0].title, ">gi|399896|sp|Q02134|HIS7_LACLA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[0].length, 200)
        self.assertEqual(record.rounds[0].alignments[1].title, ">gi|462273|sp|P34047|HIS7_ARATH IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[1].length, 270)
        self.assertEqual(record.rounds[0].alignments[2].title, ">gi|2495230|sp|Q43072|HIS7_PEA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[2].length, 281)
        self.assertEqual(record.rounds[0].alignments[3].title, ">gi|123157|sp|P18787|HIS7_AZOBR IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[3].length, 207)
        self.assertEqual(record.rounds[0].alignments[4].title, ">gi|462275|sp|P34048|HIS7_WHEAT IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[4].length, 195)
        self.assertEqual(record.rounds[0].alignments[5].title, ">gi|123161|sp|P16247|HIS7_STRCO IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[5].length, 197)
        self.assertEqual(record.rounds[0].alignments[6].title, ">gi|462272|sp|Q05068|HIS7_ANASP IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[6].length, 209)
        self.assertEqual(record.rounds[0].alignments[7].title, ">gi|123158|sp|P06987|HIS7_ECOLI HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[0].alignments[7].length, 355)
        self.assertEqual(record.rounds[0].alignments[8].title, ">gi|1346293|sp|P48054|HIS7_SYNY3 IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[8].length, 210)
        self.assertEqual(record.rounds[0].alignments[9].title, ">gi|1170286|sp|P44327|HIS7_HAEIN HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[0].alignments[9].length, 362)
        self.assertEqual(record.rounds[0].alignments[10].title, ">gi|2495224|sp|O06590|HIS7_MYCTU IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[10].length, 210)
        self.assertEqual(record.rounds[0].alignments[11].title, ">gi|123160|sp|P10368|HIS7_SALTY HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[0].alignments[11].length, 354)
        self.assertEqual(record.rounds[0].alignments[12].title, ">gi|2495226|sp|Q50504|HIS7_METTH PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[12].length, 194)
        self.assertEqual(record.rounds[0].alignments[13].title, ">gi|729718|sp|P40919|HIS7_CRYNE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[13].length, 202)
        self.assertEqual(record.rounds[0].alignments[14].title, ">gi|3334215|sp|O33773|HIS7_SULSO PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[14].length, 193)
        self.assertEqual(record.rounds[0].alignments[15].title, ">gi|123159|sp|P28624|HIS7_PHYPR IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[15].length, 452)
        self.assertEqual(record.rounds[0].alignments[16].title, ">gi|729719|sp|P40374|HIS7_SCHPO IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[16].length, 216)
        self.assertEqual(record.rounds[0].alignments[17].title, ">gi|2495227|sp|P56090|HIS7_CANAL IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[17].length, 223)
        self.assertEqual(record.rounds[0].alignments[18].title, ">gi|2495225|sp|Q58109|HIS7_METJA PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[18].length, 197)
        self.assertEqual(record.rounds[0].alignments[19].title, ">gi|399897|sp|Q02986|HIS7_SACKL IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[19].length, 232)
        self.assertEqual(record.rounds[0].alignments[20].title, ">gi|2495229|sp|Q92447|HIS7_PICPA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[20].length, 224)
        self.assertEqual(record.rounds[0].alignments[21].title, ">gi|2495228|sp|Q12578|HIS7_CANGA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[21].length, 210)
        self.assertEqual(record.rounds[0].alignments[22].title, ">gi|2506514|sp|P06633|HIS7_YEAST IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[22].length, 220)
        self.assertEqual(record.rounds[0].alignments[23].title, ">gi|462274|sp|P34041|HIS7_TRIHA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[23].length, 208)
        self.assertEqual(record.rounds[0].alignments[24].title, ">gi|1345641|sp|P49264|C7B1_THLAR CYTOCHROME P450 71B1 (CYPLXXIB1)")
        self.assertEqual(record.rounds[0].alignments[24].length, 496)
        self.assertEqual(record.rounds[0].alignments[25].title, ">gi|1731346|sp|Q10698|YY29_MYCTU PROBABLE DIPEPTIDASE CY49.29C")
        self.assertEqual(record.rounds[0].alignments[25].length, 375)
        self.assertEqual(record.rounds[0].alignments[26].title, ">gi|3287839|sp|Q01812|GLK4_RAT GLUTAMATE RECEPTOR, IONOTROPIC KAINATE 4 PRECURSOR (GLUTAMATE RECEPTOR KA-1) (KA1)")
        self.assertEqual(record.rounds[0].alignments[26].length, 956)
        self.assertEqual(record.rounds[0].alignments[27].title, ">gi|3123025|sp|Q94637|VIT6_OSCBR VITELLOGENIN 6 PRECURSOR")
        self.assertEqual(record.rounds[0].alignments[27].length, 1660)
        self.assertEqual(record.rounds[0].alignments[28].title, ">gi|3287848|sp|Q16099|GLK4_HUMAN GLUTAMATE RECEPTOR, IONOTROPIC KAINATE 4 PRECURSOR (GLUTAMATE RECEPTOR KA-1) (KA1) (EXCITATORY AMINO ACID RECEPTOR 1) (EAA1)")
        self.assertEqual(record.rounds[0].alignments[28].length, 956)
        self.assertEqual(record.rounds[0].alignments[29].title, ">gi|1174406|sp|P36126|SP14_YEAST PHOSPHOLIPASE D1 (PLD 1) (CHOLINE PHOSPHATASE 1) (PHOSPHATIDYLCHOLINE-HYDROLYZING PHOSPHOLIPASE D1) (MEIOSIS-SPECIFIC SPORULATION PROTEIN SPO14)")
        self.assertEqual(record.rounds[0].alignments[29].length, 1380)

    def _check_text_2010L_phiblast_003_round1(self, record):
        self.assertEqual(len(record.rounds[1].new_seqs), 0)
        self.assertEqual(len(record.rounds[1].alignments), 24)
        self.assertEqual(record.rounds[1].alignments[0].title, ">gi|2495230|sp|Q43072|HIS7_PEA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[0].length, 281)
        self.assertEqual(record.rounds[1].alignments[1].title, ">gi|462273|sp|P34047|HIS7_ARATH IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[1].length, 270)
        self.assertEqual(record.rounds[1].alignments[2].title, ">gi|399896|sp|Q02134|HIS7_LACLA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[2].length, 200)
        self.assertEqual(record.rounds[1].alignments[3].title, ">gi|1346293|sp|P48054|HIS7_SYNY3 IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[3].length, 210)
        self.assertEqual(record.rounds[1].alignments[4].title, ">gi|462272|sp|Q05068|HIS7_ANASP IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[4].length, 209)
        self.assertEqual(record.rounds[1].alignments[5].title, ">gi|462275|sp|P34048|HIS7_WHEAT IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[5].length, 195)
        self.assertEqual(record.rounds[1].alignments[6].title, ">gi|123161|sp|P16247|HIS7_STRCO IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[6].length, 197)
        self.assertEqual(record.rounds[1].alignments[7].title, ">gi|2506514|sp|P06633|HIS7_YEAST IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[7].length, 220)
        self.assertEqual(record.rounds[1].alignments[8].title, ">gi|2495227|sp|P56090|HIS7_CANAL IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[8].length, 223)
        self.assertEqual(record.rounds[1].alignments[9].title, ">gi|399897|sp|Q02986|HIS7_SACKL IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[9].length, 232)
        self.assertEqual(record.rounds[1].alignments[10].title, ">gi|2495228|sp|Q12578|HIS7_CANGA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[10].length, 210)
        self.assertEqual(record.rounds[1].alignments[11].title, ">gi|123158|sp|P06987|HIS7_ECOLI HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[1].alignments[11].length, 355)
        self.assertEqual(record.rounds[1].alignments[12].title, ">gi|123157|sp|P18787|HIS7_AZOBR IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[12].length, 207)
        self.assertEqual(record.rounds[1].alignments[13].title, ">gi|729718|sp|P40919|HIS7_CRYNE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[13].length, 202)
        self.assertEqual(record.rounds[1].alignments[14].title, ">gi|2495229|sp|Q92447|HIS7_PICPA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[14].length, 224)
        self.assertEqual(record.rounds[1].alignments[15].title, ">gi|1170286|sp|P44327|HIS7_HAEIN HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[1].alignments[15].length, 362)
        self.assertEqual(record.rounds[1].alignments[16].title, ">gi|729719|sp|P40374|HIS7_SCHPO IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[16].length, 216)
        self.assertEqual(record.rounds[1].alignments[17].title, ">gi|123159|sp|P28624|HIS7_PHYPR IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[17].length, 452)
        self.assertEqual(record.rounds[1].alignments[18].title, ">gi|123160|sp|P10368|HIS7_SALTY HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[1].alignments[18].length, 354)
        self.assertEqual(record.rounds[1].alignments[19].title, ">gi|2495224|sp|O06590|HIS7_MYCTU IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[19].length, 210)
        self.assertEqual(record.rounds[1].alignments[20].title, ">gi|2495226|sp|Q50504|HIS7_METTH PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[20].length, 194)
        self.assertEqual(record.rounds[1].alignments[21].title, ">gi|2495225|sp|Q58109|HIS7_METJA PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[21].length, 197)
        self.assertEqual(record.rounds[1].alignments[22].title, ">gi|462274|sp|P34041|HIS7_TRIHA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[22].length, 208)
        self.assertEqual(record.rounds[1].alignments[23].title, ">gi|3334215|sp|O33773|HIS7_SULSO PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[23].length, 193)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].score, 1040)

    def _check_text_2010L_phiblast_003_hsps(self, record):
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].bits, 409)
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].expect, 1e-114)
        self.assertEqual(len(record.rounds[0].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].score, 499)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].bits, 198)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].expect, 6e-51)
        self.assertEqual(len(record.rounds[0].alignments[1].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].score, 492)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].bits, 196)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].expect, 4e-50)
        self.assertEqual(len(record.rounds[0].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].score, 465)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].bits, 185)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].expect, 5e-47)
        self.assertEqual(len(record.rounds[0].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].score, 455)
        self.assertAlmostEqual(record.rounds[0].alignments[4].hsps[0].bits, 181)
        self.assertAlmostEqual(record.rounds[0].alignments[4].hsps[0].expect, 8e-46)
        self.assertEqual(len(record.rounds[0].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].score, 447)
        self.assertAlmostEqual(record.rounds[0].alignments[5].hsps[0].bits, 178)
        self.assertAlmostEqual(record.rounds[0].alignments[5].hsps[0].expect, 7e-45)
        self.assertEqual(len(record.rounds[0].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].score, 447)
        self.assertAlmostEqual(record.rounds[0].alignments[6].hsps[0].bits, 178)
        self.assertAlmostEqual(record.rounds[0].alignments[6].hsps[0].expect, 7e-45)
        self.assertEqual(len(record.rounds[0].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].score, 438)
        self.assertAlmostEqual(record.rounds[0].alignments[7].hsps[0].bits, 175)
        self.assertAlmostEqual(record.rounds[0].alignments[7].hsps[0].expect, 8e-44)
        self.assertEqual(len(record.rounds[0].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].score, 437)
        self.assertAlmostEqual(record.rounds[0].alignments[8].hsps[0].bits, 174)
        self.assertAlmostEqual(record.rounds[0].alignments[8].hsps[0].expect, 1e-43)
        self.assertEqual(len(record.rounds[0].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].score, 421)
        self.assertAlmostEqual(record.rounds[0].alignments[9].hsps[0].bits, 168)
        self.assertAlmostEqual(record.rounds[0].alignments[9].hsps[0].expect, 8e-42)
        self.assertEqual(len(record.rounds[0].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].score, 418)
        self.assertAlmostEqual(record.rounds[0].alignments[10].hsps[0].bits, 167)
        self.assertAlmostEqual(record.rounds[0].alignments[10].hsps[0].expect, 2e-41)
        self.assertEqual(len(record.rounds[0].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].score, 417)
        self.assertAlmostEqual(record.rounds[0].alignments[11].hsps[0].bits, 166)
        self.assertAlmostEqual(record.rounds[0].alignments[11].hsps[0].expect, 2e-41)
        self.assertEqual(len(record.rounds[0].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].score, 382)
        self.assertAlmostEqual(record.rounds[0].alignments[12].hsps[0].bits, 153)
        self.assertAlmostEqual(record.rounds[0].alignments[12].hsps[0].expect, 3e-37)
        self.assertEqual(len(record.rounds[0].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].score, 379)
        self.assertAlmostEqual(record.rounds[0].alignments[13].hsps[0].bits, 152)
        self.assertAlmostEqual(record.rounds[0].alignments[13].hsps[0].expect, 7e-37)
        self.assertEqual(len(record.rounds[0].alignments[13].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].score, 378)
        self.assertAlmostEqual(record.rounds[0].alignments[14].hsps[0].bits, 151)
        self.assertAlmostEqual(record.rounds[0].alignments[14].hsps[0].expect, 9e-37)
        self.assertEqual(len(record.rounds[0].alignments[14].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].score, 373)
        self.assertAlmostEqual(record.rounds[0].alignments[15].hsps[0].bits, 149)
        self.assertAlmostEqual(record.rounds[0].alignments[15].hsps[0].expect, 3e-36)
        self.assertEqual(len(record.rounds[0].alignments[15].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].score, 339)
        self.assertAlmostEqual(record.rounds[0].alignments[16].hsps[0].bits, 136)
        self.assertAlmostEqual(record.rounds[0].alignments[16].hsps[0].expect, 3e-32)
        self.assertEqual(len(record.rounds[0].alignments[16].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].score, 318)
        self.assertAlmostEqual(record.rounds[0].alignments[17].hsps[0].bits, 128)
        self.assertAlmostEqual(record.rounds[0].alignments[17].hsps[0].expect, 9e-30)
        self.assertEqual(len(record.rounds[0].alignments[17].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].score, 313)
        self.assertAlmostEqual(record.rounds[0].alignments[18].hsps[0].bits, 126)
        self.assertAlmostEqual(record.rounds[0].alignments[18].hsps[0].expect, 4e-29)
        self.assertEqual(len(record.rounds[0].alignments[18].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].score, 311)
        self.assertAlmostEqual(record.rounds[0].alignments[19].hsps[0].bits, 125)
        self.assertAlmostEqual(record.rounds[0].alignments[19].hsps[0].expect, 6e-29)
        self.assertEqual(len(record.rounds[0].alignments[19].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].score, 311)
        self.assertAlmostEqual(record.rounds[0].alignments[20].hsps[0].bits, 125)
        self.assertAlmostEqual(record.rounds[0].alignments[20].hsps[0].expect, 6e-29)
        self.assertEqual(len(record.rounds[0].alignments[20].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].score, 306)
        self.assertAlmostEqual(record.rounds[0].alignments[21].hsps[0].bits, 123)
        self.assertAlmostEqual(record.rounds[0].alignments[21].hsps[0].expect, 2e-28)
        self.assertEqual(len(record.rounds[0].alignments[21].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].score, 304)
        self.assertAlmostEqual(record.rounds[0].alignments[22].hsps[0].bits, 122)
        self.assertAlmostEqual(record.rounds[0].alignments[22].hsps[0].expect, 4e-28)
        self.assertEqual(len(record.rounds[0].alignments[22].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].score, 263)
        self.assertAlmostEqual(record.rounds[0].alignments[23].hsps[0].bits, 106)
        self.assertAlmostEqual(record.rounds[0].alignments[23].hsps[0].expect, 3e-23)
        self.assertEqual(len(record.rounds[0].alignments[23].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].score, 78)
        self.assertAlmostEqual(record.rounds[0].alignments[24].hsps[0].bits, 34.8)
        self.assertAlmostEqual(record.rounds[0].alignments[24].hsps[0].expect, 0.13)
        self.assertEqual(len(record.rounds[0].alignments[24].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].score, 70)
        self.assertAlmostEqual(record.rounds[0].alignments[25].hsps[0].bits, 31.7)
        self.assertAlmostEqual(record.rounds[0].alignments[25].hsps[0].expect, 1.1)
        self.assertEqual(len(record.rounds[0].alignments[25].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].score, 66)
        self.assertAlmostEqual(record.rounds[0].alignments[26].hsps[0].bits, 30.1)
        self.assertAlmostEqual(record.rounds[0].alignments[26].hsps[0].expect, 3.3)
        self.assertEqual(len(record.rounds[0].alignments[26].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].score, 64)
        self.assertAlmostEqual(record.rounds[0].alignments[27].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.rounds[0].alignments[27].hsps[0].expect, 5.6)
        self.assertEqual(len(record.rounds[0].alignments[27].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].score, 62)
        self.assertAlmostEqual(record.rounds[0].alignments[28].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.rounds[0].alignments[28].hsps[0].expect, 9.7)
        self.assertEqual(len(record.rounds[0].alignments[28].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].score, 62)
        self.assertAlmostEqual(record.rounds[0].alignments[29].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.rounds[0].alignments[29].hsps[0].expect, 9.7)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].score, 820)
        self.assertAlmostEqual(record.rounds[1].alignments[0].hsps[0].bits, 323)
        self.assertAlmostEqual(record.rounds[1].alignments[0].hsps[0].expect, 1e-88)
        self.assertEqual(len(record.rounds[1].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].score, 817)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[0].bits, 322)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[0].expect, 3e-88)
        self.assertEqual(len(record.rounds[1].alignments[1].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].score, 808)
        self.assertAlmostEqual(record.rounds[1].alignments[2].hsps[0].bits, 318)
        self.assertAlmostEqual(record.rounds[1].alignments[2].hsps[0].expect, 4e-87)
        self.assertEqual(len(record.rounds[1].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].score, 798)
        self.assertAlmostEqual(record.rounds[1].alignments[3].hsps[0].bits, 315)
        self.assertAlmostEqual(record.rounds[1].alignments[3].hsps[0].expect, 5e-86)
        self.assertEqual(len(record.rounds[1].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].score, 795)
        self.assertAlmostEqual(record.rounds[1].alignments[4].hsps[0].bits, 313)
        self.assertAlmostEqual(record.rounds[1].alignments[4].hsps[0].expect, 1e-85)
        self.assertEqual(len(record.rounds[1].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].score, 793)
        self.assertAlmostEqual(record.rounds[1].alignments[5].hsps[0].bits, 313)
        self.assertAlmostEqual(record.rounds[1].alignments[5].hsps[0].expect, 2e-85)
        self.assertEqual(len(record.rounds[1].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].score, 776)
        self.assertAlmostEqual(record.rounds[1].alignments[6].hsps[0].bits, 306)
        self.assertAlmostEqual(record.rounds[1].alignments[6].hsps[0].expect, 2e-83)
        self.assertEqual(len(record.rounds[1].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].score, 772)
        self.assertAlmostEqual(record.rounds[1].alignments[7].hsps[0].bits, 304)
        self.assertAlmostEqual(record.rounds[1].alignments[7].hsps[0].expect, 6e-83)
        self.assertEqual(len(record.rounds[1].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].score, 771)
        self.assertAlmostEqual(record.rounds[1].alignments[8].hsps[0].bits, 304)
        self.assertAlmostEqual(record.rounds[1].alignments[8].hsps[0].expect, 8e-83)
        self.assertEqual(len(record.rounds[1].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].score, 770)
        self.assertAlmostEqual(record.rounds[1].alignments[9].hsps[0].bits, 304)
        self.assertAlmostEqual(record.rounds[1].alignments[9].hsps[0].expect, 1e-82)
        self.assertEqual(len(record.rounds[1].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].score, 767)
        self.assertAlmostEqual(record.rounds[1].alignments[10].hsps[0].bits, 303)
        self.assertAlmostEqual(record.rounds[1].alignments[10].hsps[0].expect, 2e-82)
        self.assertEqual(len(record.rounds[1].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].score, 765)
        self.assertAlmostEqual(record.rounds[1].alignments[11].hsps[0].bits, 302)
        self.assertAlmostEqual(record.rounds[1].alignments[11].hsps[0].expect, 4e-82)
        self.assertEqual(len(record.rounds[1].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].score, 762)
        self.assertAlmostEqual(record.rounds[1].alignments[12].hsps[0].bits, 301)
        self.assertAlmostEqual(record.rounds[1].alignments[12].hsps[0].expect, 9e-82)
        self.assertEqual(len(record.rounds[1].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].score, 759)
        self.assertAlmostEqual(record.rounds[1].alignments[13].hsps[0].bits, 299)
        self.assertAlmostEqual(record.rounds[1].alignments[13].hsps[0].expect, 2e-81)
        self.assertEqual(len(record.rounds[1].alignments[13].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].score, 756)
        self.assertAlmostEqual(record.rounds[1].alignments[14].hsps[0].bits, 298)
        self.assertAlmostEqual(record.rounds[1].alignments[14].hsps[0].expect, 5e-81)
        self.assertEqual(len(record.rounds[1].alignments[14].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].score, 741)
        self.assertAlmostEqual(record.rounds[1].alignments[15].hsps[0].bits, 292)
        self.assertAlmostEqual(record.rounds[1].alignments[15].hsps[0].expect, 3e-79)
        self.assertEqual(len(record.rounds[1].alignments[15].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].score, 734)
        self.assertAlmostEqual(record.rounds[1].alignments[16].hsps[0].bits, 290)
        self.assertAlmostEqual(record.rounds[1].alignments[16].hsps[0].expect, 2e-78)
        self.assertEqual(len(record.rounds[1].alignments[16].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].score, 734)
        self.assertAlmostEqual(record.rounds[1].alignments[17].hsps[0].bits, 290)
        self.assertAlmostEqual(record.rounds[1].alignments[17].hsps[0].expect, 2e-78)
        self.assertEqual(len(record.rounds[1].alignments[17].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].score, 726)
        self.assertAlmostEqual(record.rounds[1].alignments[18].hsps[0].bits, 287)
        self.assertAlmostEqual(record.rounds[1].alignments[18].hsps[0].expect, 1e-77)
        self.assertEqual(len(record.rounds[1].alignments[18].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].score, 716)
        self.assertAlmostEqual(record.rounds[1].alignments[19].hsps[0].bits, 283)
        self.assertAlmostEqual(record.rounds[1].alignments[19].hsps[0].expect, 2e-76)
        self.assertEqual(len(record.rounds[1].alignments[19].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].score, 695)
        self.assertAlmostEqual(record.rounds[1].alignments[20].hsps[0].bits, 274)
        self.assertAlmostEqual(record.rounds[1].alignments[20].hsps[0].expect, 6e-74)
        self.assertEqual(len(record.rounds[1].alignments[20].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].score, 685)
        self.assertAlmostEqual(record.rounds[1].alignments[21].hsps[0].bits, 271)
        self.assertAlmostEqual(record.rounds[1].alignments[21].hsps[0].expect, 1e-72)
        self.assertEqual(len(record.rounds[1].alignments[21].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].score, 680)
        self.assertAlmostEqual(record.rounds[1].alignments[22].hsps[0].bits, 269)
        self.assertAlmostEqual(record.rounds[1].alignments[22].hsps[0].expect, 4e-72)
        self.assertEqual(len(record.rounds[1].alignments[22].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].score, 662)
        self.assertAlmostEqual(record.rounds[1].alignments[23].hsps[0].bits, 262)
        self.assertAlmostEqual(record.rounds[1].alignments[23].hsps[0].expect, 5e-70)
        self.assertEqual(len(record.rounds[1].alignments[23].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].identities, (200, 200))
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].positives, (200, 200))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].identities, (99, 198))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].positives, (135, 198))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].identities, (96, 199))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].positives, (136, 199))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].gaps, (4, 199))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].identities, (91, 194))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].positives, (126, 194))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].gaps, (4, 194))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].identities, (93, 194))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].positives, (128, 194))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].gaps, (4, 194))
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].identities, (89, 200))
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].positives, (124, 200))
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].gaps, (3, 200))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].identities, (91, 198))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].positives, (131, 198))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].identities, (91, 198))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].positives, (130, 198))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].gaps, (9, 198))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].identities, (88, 198))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].positives, (129, 198))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].identities, (89, 198))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].positives, (127, 198))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].gaps, (9, 198))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].identities, (92, 207))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].positives, (125, 207))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].gaps, (14, 207))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].identities, (89, 198))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].positives, (129, 198))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].gaps, (10, 198))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].identities, (81, 198))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].positives, (122, 198))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].gaps, (8, 198))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].identities, (83, 203))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].positives, (120, 203))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].gaps, (11, 203))
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].identities, (88, 201))
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].positives, (128, 201))
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].gaps, (9, 201))
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].identities, (86, 198))
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].positives, (120, 198))
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].gaps, (6, 198))
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].identities, (84, 221))
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].positives, (114, 221))
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].gaps, (29, 221))
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].identities, (81, 227))
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].positives, (119, 227))
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].gaps, (33, 227))
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].identities, (80, 196))
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].positives, (107, 196))
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].gaps, (9, 196))
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].identities, (81, 222))
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].positives, (119, 222))
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].gaps, (30, 222))
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].identities, (84, 223))
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].positives, (116, 223))
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].gaps, (31, 223))
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].identities, (78, 215))
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].positives, (116, 215))
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].gaps, (24, 215))
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].identities, (79, 218))
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].positives, (114, 218))
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].gaps, (30, 218))
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].identities, (68, 202))
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].positives, (102, 202))
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].gaps, (28, 202))
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].identities, (34, 134))
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].positives, (60, 134))
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].gaps, (11, 134))
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].identities, (16, 45))
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].positives, (21, 45))
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].gaps, (3, 45))
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].identities, (17, 48))
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].positives, (24, 48))
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].gaps, (3, 48))
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].identities, (25, 70))
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].positives, (32, 70))
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].gaps, (5, 70))
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].identities, (16, 48))
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].positives, (24, 48))
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].gaps, (3, 48))
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].identities, (20, 65))
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].positives, (31, 65))
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].gaps, (7, 65))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].identities, (96, 199))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].positives, (136, 199))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].gaps, (4, 199))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].identities, (99, 198))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].positives, (135, 198))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].identities, (200, 200))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].positives, (200, 200))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].identities, (88, 198))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].positives, (129, 198))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].identities, (91, 198))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].positives, (131, 198))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].identities, (93, 196))
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].positives, (128, 196))
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].gaps, (4, 196))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].identities, (89, 200))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].positives, (124, 200))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].gaps, (3, 200))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].identities, (78, 220))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].positives, (115, 220))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].gaps, (30, 220))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].identities, (81, 227))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].positives, (119, 227))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].gaps, (33, 227))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].identities, (79, 222))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].positives, (118, 222))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].gaps, (30, 222))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].identities, (76, 214))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].positives, (113, 214))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].gaps, (24, 214))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].identities, (91, 198))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].positives, (130, 198))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].gaps, (9, 198))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].identities, (91, 196))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].positives, (127, 196))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].gaps, (4, 196))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].identities, (83, 203))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].positives, (120, 203))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].gaps, (11, 203))
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].identities, (82, 223))
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].positives, (115, 223))
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].gaps, (31, 223))
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].identities, (89, 198))
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].positives, (127, 198))
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].gaps, (9, 198))
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].identities, (83, 221))
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].positives, (114, 221))
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].gaps, (29, 221))
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].identities, (86, 199))
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].positives, (121, 199))
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].gaps, (8, 199))
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].identities, (89, 198))
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].positives, (129, 198))
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].gaps, (10, 198))
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].identities, (92, 207))
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].positives, (124, 207))
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].gaps, (14, 207))
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].identities, (81, 198))
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].positives, (122, 198))
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].gaps, (8, 198))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].identities, (79, 196))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].positives, (106, 196))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].gaps, (9, 196))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].identities, (68, 202))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].positives, (102, 202))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].gaps, (28, 202))
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].identities, (83, 200))
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].positives, (124, 200))
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].gaps, (7, 200))

    def _check_text_2010L_phiblast_003_hsps_details(self, record):
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].match, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct_end, 200)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].match, "RI  + R TKET + + INLDGTG AD S+GI FLDHML  L  H  FD+ +   GD   V +D HH  ED+A+A+G  + + LG + GI R+G FT P+DEAL+   LD+SGRPYL ++ ++   Q++G YDT++ E FF++L   +G+TLH+ +  G+N+HHIIE  FK+ ARAL+QA   D  + G IPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct, "RIGEVKRVTKETNVSVKINLDGTGVADSSSGIPFLDHMLDQLASHGLFDVHVRATGD---VHIDDHHTNEDIALAIGTALLKALGERKGINRFGDFTAPLDEALIHVSLDLSGRPYLGYNLEIP-TQRVGTYDTQLVEHFFQSLVNTSGMTLHIRQLAGENSHHIIEATFKAFARALRQATETDPRRGGTIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct_start, 74)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct_end, 267)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query, "TRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].match, "TR+  + R TKET + + INLDG+G AD STGI FLDHML  L  H  FD+ +   GD   V +D HH  EDVA+A+G  + + LG++ GI R+G F+ P+DEAL+   LD+SGRP+L ++ D+   Q++G YDT++ E F +++   +G+TLH+ +  G+N+HHIIE  FK+ ARAL+QA   D  + G +PSSKGVL")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct, "TRVGEVKRVTKETNVSVKINLDGSGVADSSTGIPFLDHMLDQLASHGLFDVHVKATGD---VHIDDHHTNEDVALAIGTALLQALGDRKGINRFGDFSAPLDEALIHVSLDLSGRPHLSYNLDIP-TQRVGTYDTQVVEHFLQSIVNTSGMTLHIRQLAGRNSHHIIEATFKAFARALRQATEYDPRRRGSVPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query_start, 2)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct_start, 84)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct_end, 278)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query, "ITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].match, "I RNT ET+I +++NLDGTG  D+ TG+GFLDHML  L+ HS  DL +   GD   V +D HH  E   IA+G+ +++ +G++ GI+RYG   +PMDE L    LD S RPYL++    S   K+G  DTE+  E+F+A A  AG+TLH+   YG+N HHI+E  +K+ ARAL+  + ID  K   +PS+KG L")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct, "IERNTTETRIRVAVNLDGTGVYDVKTGVGFLDHMLEQLSRHSLMDLSVAAEGD---VHIDAHHTTEHSGIAIGQAVAKAVGDRKGIQRYGHAYVPMDETLTRVALDFSNRPYLIWKVSFS-RDKIGDMDTELFREWFQAFAMAAGVTLHVECLYGENNHHIVESCYKALARALRAGIEIDPRKRDAVPSTKGTL")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query_start, 7)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct_start, 14)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct_end, 203)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].query, "ITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].match, "+ R TKET + + INLDGTG A+ STGI FLDHML  L  H  FD+ +   GD     +D HH  ED+A+A+G  + + LG++ GI R+G FT P+DEA V   LD+SGRP+L     +   +++G YDT++ E FF++L   +G+TLH+ +  G N+HHIIE  FK+ ARAL+QA   D  + G +PSSKGVL")
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].sbjct, "VKRVTKETNVHVKINLDGTGVANSSTGIPFLDHMLDQLASHGLFDVYVKATGDTH---IDDHHSNEDIALAIGTALLQALGDRKGINRFGHFTAPLDEAAVEVILDLSGRPHLSCGLSIP-TERVGTYDTQLVEHFFQSLVNTSGMTLHIRQLAGNNSHHIIEATFKAFARALRQATEYDLRRQGTMPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].query_start, 7)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].sbjct_start, 3)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].sbjct_end, 192)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].match, "M+R+  + R TKET + + I+LDGTG+ DI+TG+GF DHML  L  H  FDL +   GD   + +D HH IED A+ALG    + LG+K+GI R+G+ T+P+DE+L    +D+SGRPYLV     +    +G YD  MT     +    A + LH++  YG+N HHI+E  FK+ ARAL+ A   D    G +PS+KG L")
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].sbjct, "MSRVGRVERTTKETSVLVEIDLDGTGKTDIATGVGFYDHMLDQLGRHGLFDLTVKTDGD---LHIDSHHTIEDTALALGAAFRQALGDKVGIYRFGNCTVPLDESLAQVTVDLSGRPYLVHTEPENMAPMIGEYDVTMTRHILESFVAQAQVALHVHVPYGRNAHHIVECQFKALARALRYASERDPRAAGILPSTKGAL")
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].sbjct_end, 197)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].match, "RI+ + R T ET +++++NLDGTG    +TGI FLDHML  ++ H   DL +   GD E   +D HH  EDV I LG+ +++ LG++ GI R+G+F  P+DEALV   LD SGRP+L +   +   +++G YDT++  EFF AL  ++ +TLH+ +  G N+HHIIE  FK+ ARA + A+ +D  + G IPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].sbjct, "RIASVHRITGETNVQVTVNLDGTGICKAATGIPFLDHMLHQISSHGLIDLDVQAKGDWE---IDDHHTNEDVGITLGQALAKALGDRKGIVRFGNFLAPLDEALVQVALDFSGRPHLSYGLQIP-TERVGTYDTQLVREFFVALVNHSQMTLHIRQLDGINSHHIIEATFKAFARAARMAIEVDPRRAGTIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].sbjct_start, 16)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].sbjct_end, 209)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].match, "R +H+ RNTKETQI++ + LD  G + I+TG+GF DHML  +  H  F ++I   GD   + +D HH +ED  +ALG+ +   LG+K GI R+G F +PMDE L  C LDISGRP+L + A+ +  Q++G   TEM E FFR+L++  G+TLHL    G+N HH +E +FK+  R L+QA+ ++      +PSSKGVL")
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].sbjct, "RYAHVVRNTKETQIDVQVWLDREGGSKINTGVGFFDHMLDQIATHGGFRMEINVKGD---LYIDDHHTVEDTGLALGEALKIALGDKRGICRFG-FVLPMDECLARCALDISGRPHLEYKAEFT-YQRVGDLSTEMIEHFFRSLSYTMGVTLHLKTK-GKNDHHRVESLFKAFGRTLRQAIRVEGD---TLPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].sbjct_start, 167)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].sbjct_end, 355)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].match, "R + + R TKET + +S+NL G+G   ++TG+ FLDHML  +  H   DL++   GD E   +D HH  EDV I LG+ ++E LG++ GI R+G F  P+DEALV   LD SGRP+L +   +   +++G YDT++  EFF A+  ++ +TLH+ +  G N+HHIIE  FK+ ARA++ A+ +D  +   IPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].sbjct, "RAAAVHRVTKETDVRVSLNLMGSGLCHVATGVPFLDHMLHQIASHGLIDLEVNATGDIE---IDDHHTNEDVGITLGQALAEALGDRRGINRFGHFIAPLDEALVQVTLDFSGRPHLSYGLQIP-TERVGTYDTQLVREFFVAVVNHSQMTLHIRQLDGINSHHIIEATFKAFARAMRMAIEVDPRRADTIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].sbjct_start, 17)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].sbjct_end, 210)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].match, "R + + R TKET I++ + LD  G  +I TG+GF DHML  +  H  F + +   GD   + +D HH +ED A+ALG+ + + +G+K GI R+G F +PMDE    C LD+SGRP++ F+A      K+G + TE+TE FF++LAF+   TLHLN   G N HH IE +FK+  R L+QA+ I+ +   E+PSSKGVL")
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].sbjct, "RFAEVIRQTKETDIKVQVWLDEAGVNEIKTGVGFFDHMLDQIATHGGFRMNVQCKGD---LWIDEHHTVEDTALALGQALKQAVGDKRGIARFG-FVLPMDECKAECALDLSGRPWIKFNACFK-RDKVGDFSTELTEHFFQSLAFSMLATLHLNV-TGNNDHHKIESLFKAFGRTLRQAIRIEGN---EMPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].sbjct_start, 174)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].sbjct_end, 362)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].query, "TRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVF--------HADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].match, "+R + I R T+E+ I + ++LDGTGQ  + TG+ F DHMLT L  H+ FDL +   GD E   ++ HH IED AIALG  + + LG+K GIRR+G   IPMDE L    +D+SGRPY V         H  ++G+     Y T +    F +LA NA I LH+   YG++ HHI E  +K+ ARAL+QAV  D  +V  +PS+KG L")
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].sbjct, "SRRARIERRTRESDIVIELDLDGTGQVAVDTGVPFYDHMLTALGSHASFDLTVRATGDVE---IEAHHTIEDTAIALGTALGQALGDKRGIRRFGDAFIPMDETLAHAAVDLSGRPYCVHTGEPDHLQHTTIAGSSV--PYHTVINRHVFESLAANARIALHVRVLYGRDPHHITEAQYKAVARALRQAVEPD-PRVSGVPSTKGAL")
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].query_start, 2)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].sbjct_start, 10)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].sbjct_end, 210)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].match, "R +H+ RNTKETQI++S+ LD  G + I+TG+GF DHML  +  H  F ++I   GD   + +D HH +ED  +AL + +   L +K GI R+G F +PMDE L  C LDISGRP+L + A+ +  Q++G   TEM E FFR+L++  G+TLHL    G+N HH +E +FK+  R ++QA+ ++      +PSSKGVL")
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].sbjct, "RYAHVVRNTKETQIDVSVWLDREGNSKINTGVGFFDHMLDQIATHGGFRMEITVKGD---LYIDDHHTVEDTGLALREALKLALRDKRGICRFG-FVLPMDECL-ACALDISGRPHLEYKAEFT-YQRVGNLSTEMIEHFFRSLSYTMGVTLHLKTK-GKNDHHRVESLFKAFGRTVRQAIRVEGD---TLPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].sbjct_start, 167)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].sbjct_end, 354)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].match, "R S  TR T ET +++ + +DG+G++ ++TG+GFLDHML  +  H   DL++   GD E   +D HH +EDVA+ LG+ + E LG+K GIRR     +PMD+AL T  LD+SGRPY V   +   +  +G   ++    F  +LA +A + +H +   G+N HH  E +FK+ A A++ AV ++    GEIPS+KG L")
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].sbjct, "RRSMKTRETLETHVKVDLEIDGSGKSSVNTGLGFLDHMLESVARHGLLDLEVEARGDLE---VDDHHTVEDVALTLGEALREALGDKSGIRRMAHAMVPMDDALATVALDLSGRPYTVLELEFD-DAVIGDVKSQNIGHFIESLAVSAAMNIHASVR-GRNDHHKAEALFKALALAIRDAVRVEH---GEIPSTKGKL")
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].sbjct_start, 5)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].sbjct_end, 194)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].query, "RISHITRNTKETQIELSINLDGTG-----QADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].match, "RI+ + R T ET I  +I+LD        + ++STGIGFLDHM T L  H    L++   GD   + +D HH  ED A+ALG+   + LG + GI+RYG    P+DE+L    +DIS RPY + H   +  +K+G   TEM     ++ AF AG+TLH++   G+N HHI E  FK+ A A++ A+S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].sbjct, "RIASVERTTSETHISCTIDLDHIPGVTEQKINVSTGIGFLDHMFTALAKHGGMSLQLQCKGD---LHIDDHHTAEDCALALGEAFKKALGERKGIKRYGYAYAPLDESLSRAVIDISSRPYFMCHLPFT-REKVGDLSTEMVSHLLQSFAFAAGVTLHIDSIRGENNHHIAESAFKALALAIRMAIS--RTGGDDVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].sbjct_start, 4)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].sbjct_end, 200)
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHML-TLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].match, "M+R ++ITR TKET+IE+ +++D  G+  +ST I F +HML TLLT+ +     I+   D   +  D HH++EDVAI LG  I   LG+K GI+R+    IPMD+ALV   LDIS R     + +L  ++ +GG  TE    FF++ A+N+GITLH+++  G NTHHIIE  FK+   AL +A  I ++   EI S+KG++")
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].sbjct, "MSRSANITRETKETKIEVLLDIDRKGEVKVSTPIPFFNHMLITLLTYMNS--TAIVSATDK--LPYDDHHIVEDVAITLGLAIKTALGDKRGIKRFSHQIIPMDDALVLVSLDISNRGMAFVNLNLKRSE-IGGLATENVPHFFQSFAYNSGITLHISQLSGYNTHHIIEASFKALGLALYEATRIVDN---EIRSTKGII")
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].sbjct_end, 193)
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].match, "R ++I+R TKET I + ++LDGTG++ +S+GIGFLDHMLT L  HS FDL++   GD     +D HH  ED A+ LG+     LG++ GI R+GS  +P+DEAL    +DIS R +   +  L     +G   +EM   FF + A  A  TLH++   G+N HH  E  FK+ A AL+ AV  D +    +PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].sbjct, "REANISRVTKETSISVKLSLDGTGKSKVSSGIGFLDHMLTALAKHSRFDLELDCKGD---TWIDDHHTTEDCALTLGEAFDVALGDRAGIARFGSACVPLDEALSRAIVDISSRAHSEINLQLV-RPSVGELSSEMITHFFESFASAALXTLHVDVLRGRNDHHRAEASFKALAVALRTAVKHDAT--AGVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].sbjct_start, 260)
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].sbjct_end, 451)
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].query, "RISHITRNTKETQIELSINLD-----------------------GTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].match, "R + + RNT ET+I ++I LD                       G     + TGIGFLDHM   L  H+ + L++   GD   + +D HH  ED AIALG    + +GN  G++R+G    P+DEAL    +D+SGRPY V    L   +K+G    EM      + +  AGITLH+   YG N HH  E  FKS A A++ A S+  S   E+PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].sbjct, "RRAFVERNTNETKISVAIALDKAPLPEESNFIDELITSKHANQKGEQVIQVDTGIGFLDHMYHALAKHAGWSLRLYSRGD---LIIDDHHTAEDTAIALGIAFKQAMGNFAGVKRFGHAYCPLDEALSRSVVDLSGRPYAVIDLGLK-REKVGELSCEMIPHLLYSFSVAAGITLHVTCLYGSNDHHRAESAFKSLAVAMRAATSLTGS--SEVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].sbjct_start, 2)
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].sbjct_end, 216)
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].query, "MTRISHITRNTKETQIELSINLDG---------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].match, "M+R + I R T ET+I++++NLDG                           +   ++ TGIGFLDHM+  L  HS + L +   GD   + +D HH  EDV I+LG    + LG   G++R+G    P+DEAL    +D+S RP+ V    L   +K+G   TEM      + A   GIT+H++   G N HH  E  FK+ A A+K+A+S  ++   +IPS+KGVL")
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].sbjct, "MSREALINRITNETKIQIALNLDGGKLELKESIFPNQSIIIDEHHAKQVSGSQYINVQTGIGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTAEDVGISLGMAFKQALGQIKGVKRFGHGFAPLDEALSRAVVDLSNRPFAVIELGLK-REKIGDLSTEMIPHVLESFAGAVGITIHVDCLRGFNDHHRAESAFKALAIAIKEAIS--KTGKNDIPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].sbjct_end, 221)
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKG")
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].match, "RI  + R TKET I L IN+DGTG+  I TGI F DH+L     H  FDL +   GD E   +D HH +EDV I LG  +++    K  I R+G   IPMD+A  T  +D+SGR Y V + +    + +G   TE    FF ++A    + +H  E  G+N HH  E +FK+   AL  A  IDE K   + S+KG")
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].sbjct, "RIFEVMRETKETNIYLKINIDGTGKYKIDTGIPFFDHLLASFAKHGCFDLIVKARGDLE---IDDHHTVEDVGICLGLALNQI--EKRNIFRFGWAIIPMDDARATVAIDLSGRSYCVGNYE-PKREFVGDLATENINHFFESVASYGMLNIHY-EVIGKNEHHKAEALFKAFGVALDLATKIDERK--GVISTKG")
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].query_end, 198)
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].sbjct_start, 7)
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].sbjct_end, 193)
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].query, "RISHITRNTKETQIELSINLDG------------------TGQA------DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].match, "R + I+R T ET+I+++I+L+G                    QA      DI TG+GFLDHM+  L  HS + L +   GD   + +D HH  ED  IALG+   E +G   G++R+G+   P+DEAL    +D+S RP+ V    L   + +G   TEM   F  + A  A ITLH++   G N HH  E  FK+ A A+++A+S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].sbjct, "RKAFISRITNETKIQIAISLNGGYIQIKDSILPAKKDDDVASQATQSQVIDIHTGVGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTTEDCGIALGQAFKEAMGAVRGVKRFGTGFAPLDEALSRAVVDLSNRPFAVIDLGLK-REMIGDLSTEMIPHFLESFAEAARITLHVDCLRGFNDHHRSESAFKALAVAIREAIS--SNGTNDVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].sbjct_start, 16)
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].sbjct_end, 231)
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].query, "RISHITRNTKETQIELSINLDG-------------------TGQA------DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].match, "R S I R T ET+I+++++LDG                     QA       ++TGIGFLDHML  L  H  + + I   GD   + +D HH  ED  IALG    E LG+  GI+R+GS   P+DEAL    +D+S RPY V    L   +K+G    EM      + A  A +T+H++   G N HH  E  FK+ A A+K+A+S   +   +IPS+KGVL")
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].sbjct, "RSSLIKRITNETKIQIALSLDGGPVSLAQSLFKDKDYSAEHAAQATSSQFISVNTGIGFLDHMLHALAKHGGWSVIIECVGD---LHIDDHHSAEDTGIALGMAFKEALGHVRGIKRFGSGFAPLDEALSRAVIDMSNRPYAVVDLGLK-REKIGDLSCEMIPHVLESFAQGAHVTMHVDCLRGFNDHHRAESAFKALAIAIKEAIS--SNGTDDIPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].sbjct_start, 7)
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].sbjct_end, 223)
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].query, "ISHITRNTKETQIELSINLDG-----------------TGQA-DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].match, "++ + R T+ET I+L+++LDG                  GQ   + TG+GFLDHMLT L  H  + L +   GD   + +D HH +ED  IALG+   E LG+  GI+R+G    P+DEAL    +D S RP+ V    L   +++G   TEM   F  + A    IT+H++   G N HH  E  FK+ A A+++A +   +   ++PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].sbjct, "MAFVKRVTQETNIQLALDLDGGSVSVRESILGKEYASGDGQTIHVHTGVGFLDHMLTALAKHGGWSLILECIGD---LHIDDHHTVEDCGIALGQAFKEALGSVRGIKRFGHGFAPLDEALSRAVVDFSNRPFAVVELGLK-RERIGQLSTEMIPHFLESFATEGRITMHVDCLRGTNDHHRSESGFKALAIAIREART--PTGRDDVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].query_start, 4)
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].sbjct_end, 209)
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].query, "ITRNTKETQIELSINLDGTGQA------------------------DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].match, "+ R T ET+I+++I+L G   A                        ++ TGIGFLDHM+  L  HS + L +   GD   + +D HH  ED  IALG+   E LG   G++R+GS   P+DEAL    +D+S RPY V    L   +K+G    EM   F  + A  + ITLH++   G+N HH  E  FK+ A A+++A S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].sbjct, "VKRITNETKIQIAISLKGGPLAIEHSIFPEKEAEAVAEQATQSQVINVHTGIGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTTEDCGIALGQAFKEALGAVRGVKRFGSGFAPLDEALSRAVVDLSNRPYAVVELGLQ-REKVGDLSCEMIPHFLESFAEASRITLHVDCLRGKNDHHRSESAFKALAVAIREATS--PNGTNDVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].query_start, 7)
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].sbjct_start, 8)
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].sbjct_end, 219)
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].query, "RISHITRNTKETQIELSINLDG------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALK")
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].match, "R + ++R+T ET I++++++DG                        +    I+TGIGFLDHML  L  H+ + + +   GD   + +D HH  ED  IA+G   ++ LG   G+ R+G    P+DEAL    +D+S RPY V    L   +KLG    EM     ++ A  A ITLH++   G N HH  E  FK+ A A++")
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].sbjct, "RAAALSRDTNETSIQIALSIDGGELPQDADPRLLEASSAHASQTSKSQVISINTGIGFLDHMLHALAKHAGWSMALNCKGD---LHIDDHHTAEDCCIAVGTTFAKALGALTGVARFGYAYAPLDEALSRAVVDLSNRPYTVVDLGLK-REKLGELSCEMIPHCLQSFAQAARITLHVDCLRGDNDHHRAESAFKALAVAVR")
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].query_end, 180)
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].sbjct_start, 8)
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].sbjct_end, 205)
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].query, "GDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQK---LGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDES")
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].match, "G+   +G  PH  + +++   G  +S  LG+   +      T+   + L T DL+   RPY+ + A ++ N K      YD     +++R +     + L+  +   Q+  HI E    S  R  KQA S +E+")
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].sbjct, "GNLHQLGEKPHRAMVELSKTYGPLMSLKLGSVTTVVATSVETVR--DVLKTYDLECCSRPYMTYPARITYNLKDLVFSPYD-----KYWRQVRKLTVVELYTAKRV-QSFRHIREEEVASFVRFNKQAASSEET")
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].query_start, 58)
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].query_end, 188)
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].sbjct_start, 40)
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].sbjct_end, 165)
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].query, "HSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLG")
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].match, "HS+    I+G G H   G DPHH   D  +  G  +  D+G   G")
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].sbjct, "HSEVAFVIVGSGPH---GADPHHGYSDRELREGDIVVVDIGGTYG")
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].query_start, 47)
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].query_end, 91)
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].sbjct_start, 195)
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].sbjct_end, 236)
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].query, "PYLVF---HADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYG")
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].match, "PYL+    H D+ GN +  G+  +M +E    L FN  I L  +  YG")
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].sbjct, "PYLMLKGNHQDMEGNDRYEGFCVDMLKELAEILRFNYKIRLVGDGVYG")
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].query_start, 117)
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].query_end, 161)
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].sbjct_start, 427)
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].sbjct_end, 474)
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].query, "GKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLG---GYDTEMTEEFFRA")
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].match, "GK ISED     GI + G     +D   VT + D  G    +  + L  N++ G    YD E T EF RA")
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].sbjct, "GKEISEDKWEDFGISQRGEEKFFIDAEKVTVEFD--GFQAKIQMSSLYKNKQCGLCGHYDGEKTNEFRRA")
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].query_start, 79)
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].query_end, 145)
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].sbjct_start, 1436)
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].sbjct_end, 1503)
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].query, "PYLVF---HADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYG")
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].match, "PYL+    H ++ GN +  G+  +M +E    L FN  I L  +  YG")
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].sbjct, "PYLMLKGNHQEMEGNDRYEGFCVDMLKELAEILRFNYKIRLVGDGVYG")
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].query_start, 117)
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].query_end, 161)
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].sbjct_start, 427)
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].sbjct_end, 474)
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].query, "RYGSFTIPMDEAL-----VTCDLDISGRPYLVFHADLSGNQKL--GGYDTEMTEEFFRALAFNAG")
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].match, "R GS ++P++E          D DI G  Y   H D+  NQ+L  G  D  +++   +   FN+G")
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].sbjct, "RNGSNSLPLNEKSNEGESTNVDQDIEGDEYHRLHEDILKNQELDDGSLDDLLSQIIPKITNFNSG")
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].query_start, 94)
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].query_end, 151)
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].sbjct_start, 1141)
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].sbjct_end, 1205)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query, "TRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].match, "TR+  + R TKET + + INLDG+G AD STGI FLDHML  L  H  FD+ +   GD   V +D HH  EDVA+A+G  + + LG++ GI R+G F+ P+DEAL+   LD+SGRP+L ++ D+   Q++G YDT++ E F +++   +G+TLH+ +  G+N+HHIIE  FK+ ARAL+QA   D  + G +PSSKGVL")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct, "TRVGEVKRVTKETNVSVKINLDGSGVADSSTGIPFLDHMLDQLASHGLFDVHVKATGD---VHIDDHHTNEDVALAIGTALLQALGDRKGINRFGDFSAPLDEALIHVSLDLSGRPHLSYNLDIP-TQRVGTYDTQVVEHFLQSIVNTSGMTLHIRQLAGRNSHHIIEATFKAFARALRQATEYDPRRRGSVPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query_start, 2)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct_start, 84)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct_end, 278)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].match, "RI  + R TKET + + INLDGTG AD S+GI FLDHML  L  H  FD+ +   GD   V +D HH  ED+A+A+G  + + LG + GI R+G FT P+DEAL+   LD+SGRPYL ++ ++   Q++G YDT++ E FF++L   +G+TLH+ +  G+N+HHIIE  FK+ ARAL+QA   D  + G IPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct, "RIGEVKRVTKETNVSVKINLDGTGVADSSSGIPFLDHMLDQLASHGLFDVHVRATGD---VHIDDHHTNEDIALAIGTALLKALGERKGINRFGDFTAPLDEALIHVSLDLSGRPYLGYNLEIP-TQRVGTYDTQLVEHFFQSLVNTSGMTLHIRQLAGENSHHIIEATFKAFARALRQATETDPRRGGTIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct_start, 74)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct_end, 267)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].match, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct_end, 200)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].match, "R + + R TKET + +S+NL G+G   ++TG+ FLDHML  +  H   DL++   GD E   +D HH  EDV I LG+ ++E LG++ GI R+G F  P+DEALV   LD SGRP+L +   +   +++G YDT++  EFF A+  ++ +TLH+ +  G N+HHIIE  FK+ ARA++ A+ +D  +   IPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct, "RAAAVHRVTKETDVRVSLNLMGSGLCHVATGVPFLDHMLHQIASHGLIDLEVNATGDIE---IDDHHTNEDVGITLGQALAEALGDRRGINRFGHFIAPLDEALVQVTLDFSGRPHLSYGLQIP-TERVGTYDTQLVREFFVAVVNHSQMTLHIRQLDGINSHHIIEATFKAFARAMRMAIEVDPRRADTIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct_start, 17)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct_end, 210)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].match, "RI+ + R T ET +++++NLDGTG    +TGI FLDHML  ++ H   DL +   GD E   +D HH  EDV I LG+ +++ LG++ GI R+G+F  P+DEALV   LD SGRP+L +   +   +++G YDT++  EFF AL  ++ +TLH+ +  G N+HHIIE  FK+ ARA + A+ +D  + G IPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].sbjct, "RIASVHRITGETNVQVTVNLDGTGICKAATGIPFLDHMLHQISSHGLIDLDVQAKGDWE---IDDHHTNEDVGITLGQALAKALGDRKGIVRFGNFLAPLDEALVQVALDFSGRPHLSYGLQIP-TERVGTYDTQLVREFFVALVNHSQMTLHIRQLDGINSHHIIEATFKAFARAARMAIEVDPRRAGTIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].sbjct_start, 16)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].sbjct_end, 209)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].query, "SHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].match, "  + R TKET + + INLDGTG A+ STGI FLDHML  L  H  FD+ +   GD     +D HH  ED+A+A+G  + + LG++ GI R+G FT P+DEA V   LD+SGRP+L     +   +++G YDT++ E FF++L   +G+TLH+ +  G N+HHIIE  FK+ ARAL+QA   D  + G +PSSKGVL")
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].sbjct, "GEVKRVTKETNVHVKINLDGTGVANSSTGIPFLDHMLDQLASHGLFDVYVKATGDT---HIDDHHSNEDIALAIGTALLQALGDRKGINRFGHFTAPLDEAAVEVILDLSGRPHLSCGLSIP-TERVGTYDTQLVEHFFQSLVNTSGMTLHIRQLAGNNSHHIIEATFKAFARALRQATEYDLRRQGTMPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].query_start, 5)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].sbjct_end, 192)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].match, "M+R+  + R TKET + + I+LDGTG+ DI+TG+GF DHML  L  H  FDL +   GD   + +D HH IED A+ALG    + LG+K+GI R+G+ T+P+DE+L    +D+SGRPYLV     +    +G YD  MT     +    A + LH++  YG+N HHI+E  FK+ ARAL+ A   D    G +PS+KG L")
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].sbjct, "MSRVGRVERTTKETSVLVEIDLDGTGKTDIATGVGFYDHMLDQLGRHGLFDLTVKTDGD---LHIDSHHTIEDTALALGAAFRQALGDKVGIYRFGNCTVPLDESLAQVTVDLSGRPYLVHTEPENMAPMIGEYDVTMTRHILESFVAQAQVALHVHVPYGRNAHHIVECQFKALARALRYASERDPRAAGILPSTKGAL")
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].sbjct_end, 197)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].query, "SHITRNTKETQIELSINLDG------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].match, "+ + R T ET+I+++I+L G                        +   ++ TGIGFLDHM+  L  HS + L +   GD   + +D HH  ED  IALG+   E LG   G++R+GS   P+DEAL    +D+S RPY V    L   +K+G    EM   F  + A  + ITLH++   G+N HH  E  FK+ A A+++A S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].sbjct, "ALVKRITNETKIQIAISLKGGPLAIEHSIFPEKEAEAVAEQATQSQVINVHTGIGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTTEDCGIALGQAFKEALGAVRGVKRFGSGFAPLDEALSRAVVDLSNRPYAVVELGL-QREKVGDLSCEMIPHFLESFAEASRITLHVDCLRGKNDHHRSESAFKALAVAIREATS--PNGTNDVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].query_start, 5)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].sbjct_start, 6)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].sbjct_end, 219)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].query, "MTRISHITRNTKETQIELSINLDG---------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].match, "M+R + I R T ET+I++++NLDG                           +   ++ TGIGFLDHM+  L  HS + L +   GD   + +D HH  EDV I+LG    + LG   G++R+G    P+DEAL    +D+S RP+ V    L   +K+G   TEM      + A   GIT+H++   G N HH  E  FK+ A A+K+A+S  ++   +IPS+KGVL")
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].sbjct, "MSREALINRITNETKIQIALNLDGGKLELKESIFPNQSIIIDEHHAKQVSGSQYINVQTGIGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTAEDVGISLGMAFKQALGQIKGVKRFGHGFAPLDEALSRAVVDLSNRPFAVIELGLK-REKIGDLSTEMIPHVLESFAGAVGITIHVDCLRGFNDHHRAESAFKALAIAIKEAIS--KTGKNDIPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].sbjct_end, 221)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].query, "RISHITRNTKETQIELSINLDG------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].match, "R + I+R T ET+I+++I+L+G                        +   DI TG+GFLDHM+  L  HS + L +   GD   + +D HH  ED  IALG+   E +G   G++R+G+   P+DEAL    +D+S RP+ V    L   + +G   TEM   F  + A  A ITLH++   G N HH  E  FK+ A A+++A+S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].sbjct, "RKAFISRITNETKIQIAISLNGGYIQIKDSILPAKKDDDVASQATQSQVIDIHTGVGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTTEDCGIALGQAFKEAMGAVRGVKRFGTGFAPLDEALSRAVVDLSNRPFAVIDLGLK-REMIGDLSTEMIPHFLESFAEAARITLHVDCLRGFNDHHRSESAFKALAVAIREAIS--SNGTNDVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].sbjct_start, 16)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].sbjct_end, 231)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].query, "SHITRNTKETQIELSINLDGTGQA------------------DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].match, "+ + R T+ET I+L+++LDG   +                   + TG+GFLDHMLT L  H  + L +   GD   + +D HH +ED  IALG+   E LG+  GI+R+G    P+DEAL    +D S RP+ V    L   +++G   TEM   F  + A    IT+H++   G N HH  E  FK+ A A+++A     +   ++PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].sbjct, "AFVKRVTQETNIQLALDLDGGSVSVRESILGKEYASGDGQTIHVHTGVGFLDHMLTALAKHGGWSLILECIGD---LHIDDHHTVEDCGIALGQAFKEALGSVRGIKRFGHGFAPLDEALSRAVVDFSNRPFAVVELGLK-RERIGQLSTEMIPHFLESFATEGRITMHVDCLRGTNDHHRSESGFKALAIAIREA--RTPTGRDDVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].query_start, 5)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].sbjct_start, 2)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].sbjct_end, 209)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].match, "R +H+ RNTKETQI++ + LD  G + I+TG+GF DHML  +  H  F ++I   GD   + +D HH +ED  +ALG+ +   LG+K GI R+G F +PMDE L  C LDISGRP+L + A+ +  Q++G   TEM E FFR+L++  G+TLHL    G+N HH +E +FK+  R L+QA+ ++      +PSSKGVL")
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].sbjct, "RYAHVVRNTKETQIDVQVWLDREGGSKINTGVGFFDHMLDQIATHGGFRMEINVKGD---LYIDDHHTVEDTGLALGEALKIALGDKRGICRFG-FVLPMDECLARCALDISGRPHLEYKAEFT-YQRVGDLSTEMIEHFFRSLSYTMGVTLHLKT-KGKNDHHRVESLFKAFGRTLRQAIRVEG---DTLPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].sbjct_start, 167)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].sbjct_end, 355)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].query, "SHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].match, "+ I RNT ET+I +++NLDGTG  D+ TG+GFLDHML  L+ HS  DL +   GD   V +D HH  E   IA+G+ +++ +G++ GI+RYG   +PMDE L    LD S RPYL++    S   K+G  DTE+  E+F+A A  AG+TLH+   YG+N HHI+E  +K+ ARAL+  + ID  K   +PS+KG L")
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].sbjct, "ASIERNTTETRIRVAVNLDGTGVYDVKTGVGFLDHMLEQLSRHSLMDLSVAAEGD---VHIDAHHTTEHSGIAIGQAVAKAVGDRKGIQRYGHAYVPMDETLTRVALDFSNRPYLIWKVSFS-RDKIGDMDTELFREWFQAFAMAAGVTLHVECLYGENNHHIVESCYKALARALRAGIEIDPRKRDAVPSTKGTL")
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].query_start, 5)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].sbjct_start, 12)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].sbjct_end, 203)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].query, "RISHITRNTKETQIELSINLDG-----TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].match, "RI+ + R T ET I  +I+LD        + ++STGIGFLDHM T L  H    L++   GD   + +D HH  ED A+ALG+   + LG + GI+RYG    P+DE+L    +DIS RPY + H   +  +K+G   TEM     ++ AF AG+TLH++   G+N HHI E  FK+ A A++ A+S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].sbjct, "RIASVERTTSETHISCTIDLDHIPGVTEQKINVSTGIGFLDHMFTALAKHGGMSLQLQCKGD---LHIDDHHTAEDCALALGEAFKKALGERKGIKRYGYAYAPLDESLSRAVIDISSRPYFMCHLPFT-REKVGDLSTEMVSHLLQSFAFAAGVTLHIDSIRGENNHHIAESAFKALALAIRMAIS--RTGGDDVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].sbjct_start, 4)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].sbjct_end, 200)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQA-------------------------DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].match, "R S I R T ET+I+++++LDG   +                          ++TGIGFLDHML  L  H  + + I   GD   + +D HH  ED  IALG    E LG+  GI+R+GS   P+DEAL    +D+S RPY V    L   +K+G    EM      + A  A +T+H++   G N HH  E  FK+ A A+K+A+S   +   +IPS+KGVL")
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].sbjct, "RSSLIKRITNETKIQIALSLDGGPVSLAQSLFKDKDYSAEHAAQATSSQFISVNTGIGFLDHMLHALAKHGGWSVIIECVGD---LHIDDHHSAEDTGIALGMAFKEALGHVRGIKRFGSGFAPLDEALSRAVIDMSNRPYAVVDLGLK-REKIGDLSCEMIPHVLESFAQGAHVTMHVDCLRGFNDHHRAESAFKALAIAIKEAIS--SNGTDDIPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].sbjct_start, 7)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].sbjct_end, 223)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].match, "R + + R TKET I++ + LD  G  +I TG+GF DHML  +  H  F + +   GD   + +D HH +ED A+ALG+ + + +G+K GI R+G F +PMDE    C LD+SGRP++ F+A      K+G + TE+TE FF++LAF+   TLHLN   G N HH IE +FK+  R L+QA+ I+ +   E+PSSKGVL")
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].sbjct, "RFAEVIRQTKETDIKVQVWLDEAGVNEIKTGVGFFDHMLDQIATHGGFRMNVQCKGD---LWIDEHHTVEDTALALGQALKQAVGDKRGIARFG-FVLPMDECKAECALDLSGRPWIKFNACFK-RDKVGDFSTELTEHFFQSLAFSMLATLHLNV-TGNNDHHKIESLFKAFGRTLRQAIRIEGN---EMPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].sbjct_start, 174)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].sbjct_end, 362)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].query, "RISHITRNTKETQIELSINLD-----------------------GTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].match, "R + + RNT ET+I ++I LD                       G     + TGIGFLDHM   L  H+ + L++   GD   + +D HH  ED AIALG    + +GN  G++R+G    P+DEAL    +D+SGRPY V    L   +K+G    EM      + +  AGITLH+   YG N HH  E  FKS A A++ A S+  +   E+PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].sbjct, "RRAFVERNTNETKISVAIALDKAPLPEESNFIDELITSKHANQKGEQVIQVDTGIGFLDHMYHALAKHAGWSLRLYSRGD---LIIDDHHTAEDTAIALGIAFKQAMGNFAGVKRFGHAYCPLDEALSRSVVDLSGRPYAVIDLGLK-REKVGELSCEMIPHLLYSFSVAAGITLHVTCLYGSNDHHRAESAFKSLAVAMRAATSL--TGSSEVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].sbjct_start, 2)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].sbjct_end, 216)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGI-TLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].match, "R ++I+R TKET I + ++LDGTG++ +S+GIGFLDHMLT L  HS FDL++   GD     +D HH  ED A+ LG+     LG++ GI R+GS  +P+DEAL    +DIS R +   +  L     +G   +EM   FF + A  A + TLH++   G+N HH  E  FK+ A AL+ AV  D +    +PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].sbjct, "REANISRVTKETSISVKLSLDGTGKSKVSSGIGFLDHMLTALAKHSRFDLELDCKGDT---WIDDHHTTEDCALTLGEAFDVALGDRAGIARFGSACVPLDEALSRAIVDISSRAHSEINLQLV-RPSVGELSSEMITHFFESFASAA-LXTLHVDVLRGRNDHHRAEASFKALAVALRTAVKHDAT--AGVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].sbjct_start, 260)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].sbjct_end, 451)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].match, "R +H+ RNTKETQI++S+ LD  G + I+TG+GF DHML  +  H  F ++I   GD   + +D HH +ED  +AL + +   L +K GI R+G F +PMDE L  C LDISGRP+L + A+ +  Q++G   TEM E FFR+L++  G+TLHL    G+N HH +E +FK+  R ++QA+ ++      +PSSKGVL")
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].sbjct, "RYAHVVRNTKETQIDVSVWLDREGNSKINTGVGFFDHMLDQIATHGGFRMEITVKGD---LYIDDHHTVEDTGLALREALKLALRDKRGICRFG-FVLPMDECLA-CALDISGRPHLEYKAEFT-YQRVGNLSTEMIEHFFRSLSYTMGVTLHLKT-KGKNDHHRVESLFKAFGRTVRQAIRVEG---DTLPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].sbjct_start, 167)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].sbjct_end, 354)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].query, "TRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVF--------HADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].match, "+R + I R T+E+ I + ++LDGTGQ  + TG+ F DHMLT L  H+ FDL +   GD E   ++ HH IED AIALG  + + LG+K GIRR+G   IPMDE L    +D+SGRPY V         H  ++G+     Y T +    F +LA NA I LH+   YG++ HHI E  +K+ ARAL+QAV  D   V  +PS+KG L")
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].sbjct, "SRRARIERRTRESDIVIELDLDGTGQVAVDTGVPFYDHMLTALGSHASFDLTVRATGDVE---IEAHHTIEDTAIALGTALGQALGDKRGIRRFGDAFIPMDETLAHAAVDLSGRPYCVHTGEPDHLQHTTIAGSSV--PYHTVINRHVFESLAANARIALHVRVLYGRDPHHITEAQYKAVARALRQAVEPDPR-VSGVPSTKGAL")
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].query_start, 2)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].sbjct_start, 10)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].sbjct_end, 210)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].match, "R S  TR T ET +++ + +DG+G++ ++TG+GFLDHML  +  H   DL++   GD E   +D HH +EDVA+ LG+ + E LG+K GIRR     +PMD+AL T  LD+SGRPY V   +   +  +G   ++    F  +LA +A + +H +   G+N HH  E +FK+ A A++ AV ++    GEIPS+KG L")
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].sbjct, "RRSMKTRETLETHVKVDLEIDGSGKSSVNTGLGFLDHMLESVARHGLLDLEVEARGDLE---VDDHHTVEDVALTLGEALREALGDKSGIRRMAHAMVPMDDALATVALDLSGRPYTVLELEFD-DAVIGDVKSQNIGHFIESLAVSAAMNIHASV-RGRNDHHKAEALFKALALAIRDAVRVEH---GEIPSTKGKL")
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].sbjct_start, 5)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].sbjct_end, 194)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKG")
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].match, "RI  + R TKET I L IN+DGTG+  I TGI F DH+L     H  FDL +   GD E   +D HH +EDV I LG  +++    K  I R+G   IPMD+A  T  +D+SGR Y V + +    + +G   TE    FF ++A    + +H     G+N HH  E +FK+   AL  A  IDE K   + S+KG")
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].sbjct, "RIFEVMRETKETNIYLKINIDGTGKYKIDTGIPFFDHLLASFAKHGCFDLIVKARGDLE---IDDHHTVEDVGICLGLALNQI--EKRNIFRFGWAIIPMDDARATVAIDLSGRSYCVGNYE-PKREFVGDLATENINHFFESVASYGMLNIHYEVI-GKNEHHKAEALFKAFGVALDLATKIDERK--GVISTKG")
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].query_end, 198)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].sbjct_start, 7)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].sbjct_end, 193)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].query, "RISHITRNTKETQIELSINLDG------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALK")
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].match, "R + ++R+T ET I++++++DG                        +    I+TGIGFLDHML  L  H+ + + +   GD   + +D HH  ED  IA+G   ++ LG   G+ R+G    P+DEAL    +D+S RPY V    L   +KLG    EM     ++ A  A ITLH++   G N HH  E  FK+ A A++")
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].sbjct, "RAAALSRDTNETSIQIALSIDGGELPQDADPRLLEASSAHASQTSKSQVISINTGIGFLDHMLHALAKHAGWSMALNCKGD---LHIDDHHTAEDCCIAVGTTFAKALGALTGVARFGYAYAPLDEALSRAVVDLSNRPYTVVDLGLK-REKLGELSCEMIPHCLQSFAQAARITLHVDCLRGDNDHHRAESAFKALAVAVR")
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].query_end, 180)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].sbjct_start, 8)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].sbjct_end, 205)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].match, "M+R ++ITR TKET+IE+ +++D  G+  +ST I F +HML  L  + +    +      + +  D HH++EDVAI LG  I   LG+K GI+R+    IPMD+ALV   LDIS R     + +L  ++ +GG  TE    FF++ A+N+GITLH+++  G NTHHIIE  FK+   AL +A  I ++   EI S+KG++")
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].sbjct, "MSRSANITRETKETKIEVLLDIDRKGEVKVSTPIPFFNHMLITLLTYMNSTAIVSAT---DKLPYDDHHIVEDVAITLGLAIKTALGDKRGIKRFSHQIIPMDDALVLVSLDISNRGMAFVNLNLKRSE-IGGLATENVPHFFQSFAYNSGITLHISQLSGYNTHHIIEASFKALGLALYEATRIVDN---EIRSTKGII")
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].sbjct_end, 193)

    def _check_text_2010L_phiblast_003_footer(self, record):
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Nov 15, 1999  2:55 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.317)
        self.assertAlmostEqual(record.ka_params[1], 0.149)
        self.assertAlmostEqual(record.ka_params[2], 0.479)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0524)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 23641501)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 1047546)
        self.assertEqual(record.num_good_extends, 2481)
        self.assertEqual(record.num_seqs_better_e, 54)
        self.assertEqual(record.hsps_no_gap, 48)
        self.assertEqual(record.hsps_prelim_gapped, 6)
        self.assertEqual(record.hsps_gapped, 56)
        self.assertEqual(record.query_length, 200)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 50)
        self.assertEqual(record.effective_query_length, 150)
        self.assertEqual(record.effective_database_length, 25539661)
        self.assertEqual(record.effective_search_space, 3830949150)
        self.assertEqual(record.effective_search_space_used, 3830949150)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.5)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 63)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.8)

    def test_text_2010L_blastn_001(self):
        "Test parsing BLASTN 2.0.10 output (text_2010L_blastn_001)"

        path = os.path.join('Blast', 'text_2010L_blastn_001.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTN")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1348916|gb|G26684|G26684 human STS\nSTS_D11570.\x01gi|1375195|gb|G26945|G26945 human STS SHGC-32699.")
        self.assertEqual(record.query_letters, 285)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 4)
        self.assertEqual(record.descriptions[0].title, "gi|1348916|gb|G26684|G26684 human STS STS_D11570. >gi|1375195|g...")
        self.assertEqual(record.descriptions[0].score, 517)
        self.assertAlmostEqual(record.descriptions[0].e, 1e-146)
        self.assertEqual(record.descriptions[1].title, "gi|720683|gb|G03725|G03725 human STS WI-344.")
        self.assertEqual(record.descriptions[1].score, 32)
        self.assertAlmostEqual(record.descriptions[1].e, 1.6)
        self.assertEqual(record.descriptions[2].title, "gi|4516686|dbj|AU026763.1|AU026763 Rattus norvegicus, OTSUKA cl...")
        self.assertEqual(record.descriptions[2].score, 32)
        self.assertAlmostEqual(record.descriptions[2].e, 1.6)
        self.assertEqual(record.descriptions[3].title, "gi|6120827|gb|G55508.1|G55508 SHGC-100856 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[3].score, 32)
        self.assertAlmostEqual(record.descriptions[3].e, 1.6)
        self.assertEqual(len(record.alignments), 4)
        self.assertEqual(record.alignments[0].title, ">gi|1348916|gb|G26684|G26684 human STS STS_D11570. >gi|1375195|gb|G26945|G26945 human STS SHGC-32699.")
        self.assertEqual(record.alignments[0].length, 285)
        self.assertEqual(record.alignments[1].title, ">gi|720683|gb|G03725|G03725 human STS WI-344.")
        self.assertEqual(record.alignments[1].length, 246)
        self.assertEqual(record.alignments[2].title, ">gi|4516686|dbj|AU026763.1|AU026763 Rattus norvegicus, OTSUKA clone, OT33.16/752f07, microsatellite sequence, sequence tagged site")
        self.assertEqual(record.alignments[2].length, 307)
        self.assertEqual(record.alignments[3].title, ">gi|6120827|gb|G55508.1|G55508 SHGC-100856 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[3].length, 711)
        self.assertEqual(record.alignments[0].hsps[0].score, 261)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 517)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 1e-146)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 1.6)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 1.6)
        self.assertEqual(len(record.alignments[2].hsps), 1)
        self.assertEqual(record.alignments[3].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 1.6)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (285, 285))
        self.assertEqual(record.alignments[1].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[2].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[3].hsps[0].identities, (18, 19))
        self.assertEqual(record.alignments[0].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[1].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[2].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[3].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[0].hsps[0].query, "gatccctacccttnccgttggtctctntcgctgactcgaggcacctaacatccattcacacccaacacaggccagcgacttctggggctcagccacagacatggtttgtnactnttgagcttctgttcctagagaatcctagaggcttgattggcccaggctgctgtntgtnctggaggcaaagaatccctacctcctaggggtgaaaggaaatnaaaatggaaagttcttgtagcgcaaggcctgacatgggtagctgctcaataaatgctagtntgttatttc")
        self.assertEqual(record.alignments[0].hsps[0].match, "|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "gatccctacccttnccgttggtctctntcgctgactcgaggcacctaacatccattcacacccaacacaggccagcgacttctggggctcagccacagacatggtttgtnactnttgagcttctgttcctagagaatcctagaggcttgattggcccaggctgctgtntgtnctggaggcaaagaatccctacctcctaggggtgaaaggaaatnaaaatggaaagttcttgtagcgcaaggcctgacatgggtagctgctcaataaatgctagtntgttatttc")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 285)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 285)
        self.assertEqual(record.alignments[1].hsps[0].query, "ctcaataaatgctagt")
        self.assertEqual(record.alignments[1].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "ctcaataaatgctagt")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 260)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 275)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 178)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 163)
        self.assertEqual(record.alignments[2].hsps[0].query, "ggaaagttcttgtagc")
        self.assertEqual(record.alignments[2].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "ggaaagttcttgtagc")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 221)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 236)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 32)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 47)
        self.assertEqual(record.alignments[3].hsps[0].query, "gaaatnaaaatggaaagtt")
        self.assertEqual(record.alignments[3].hsps[0].match, "||||| |||||||||||||")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "gaaataaaaatggaaagtt")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 210)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 228)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 588)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 606)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(record.posted_date, [('Nov 26, 1999  5:52 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 1.37)
        self.assertAlmostEqual(record.ka_params[1], 0.711)
        self.assertAlmostEqual(record.ka_params[2], 1.31)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 1.37)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.711)
        self.assertAlmostEqual(record.ka_params_gap[2], 1.31)
        self.assertEqual(record.matrix, 'blastn matrix:1 -3')
        self.assertEqual(record.gap_penalties, [5, 2])
        self.assertEqual(record.num_hits, 3835)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 3835)
        self.assertEqual(record.num_good_extends, 1155)
        self.assertEqual(record.num_seqs_better_e, 4)
        self.assertEqual(record.query_length, 285)
        self.assertEqual(record.database_length, 31998854)
        self.assertEqual(record.effective_hsp_length, 17)
        self.assertEqual(record.effective_query_length, 268)
        self.assertEqual(record.effective_database_length, 30506390)
        self.assertEqual(record.effective_search_space, 8175712520)
        self.assertEqual(record.effective_search_space_used, 8175712520)
        self.assertEqual(record.threshold, 0)
        self.assertEqual(record.window_size, 0)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 6)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 11.9)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 25)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 49.6)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 12)
        self.assertAlmostEqual(record.gap_trigger[1], 24.3)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 15)
        self.assertAlmostEqual(record.blast_cutoff[1], 30.2)

    def test_text_2010L_blastn_002(self):
        "Test parsing BLASTN 2.0.10 output without alignments (text_2010L_blastn_002)"

        path = os.path.join('Blast', 'text_2010L_blastn_002.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTN")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1348400|gb|G26168|G26168 human STS\nEST47274.\x01gi|1375380|gb|G27130|G27130 human STS SHGC-32751.")
        self.assertEqual(record.query_letters, 434)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 19)
        self.assertEqual(record.descriptions[0].title, "gi|1348400|gb|G26168|G26168 human STS EST47274. >gi|1375380|gb|...")
        self.assertEqual(record.descriptions[0].score, 718)
        self.assertAlmostEqual(record.descriptions[0].e, 0.0)
        self.assertEqual(record.descriptions[1].title, "gi|6121436|gb|G56117.1|G56117 SHGC-101583 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[1].score, 34)
        self.assertAlmostEqual(record.descriptions[1].e, 0.65)
        self.assertEqual(record.descriptions[2].title, "gi|4632200|dbj|AU047565.1|AU047565 Rattus norvegicus, OTSUKA cl...")
        self.assertEqual(record.descriptions[2].score, 34)
        self.assertAlmostEqual(record.descriptions[2].e, 0.65)
        self.assertEqual(record.descriptions[3].title, "gi|3249175|gb|G38401|G38401 SHGC-57345 Human Homo sapiens STS g...")
        self.assertEqual(record.descriptions[3].score, 34)
        self.assertAlmostEqual(record.descriptions[3].e, 0.65)
        self.assertEqual(record.descriptions[4].title, "gi|605557|gb|L31312|HUMUT937B Human STS UT937, 3' primer bind.")
        self.assertEqual(record.descriptions[4].score, 34)
        self.assertAlmostEqual(record.descriptions[4].e, 0.65)
        self.assertEqual(record.descriptions[5].title, "gi|720383|gb|G03425|G03425 human STS WI-5868.")
        self.assertEqual(record.descriptions[5].score, 34)
        self.assertAlmostEqual(record.descriptions[5].e, 0.65)
        self.assertEqual(record.descriptions[6].title, "gi|859592|gb|G06347|G06347 human STS WI-7005.")
        self.assertEqual(record.descriptions[6].score, 32)
        self.assertAlmostEqual(record.descriptions[6].e, 2.6)
        self.assertEqual(record.descriptions[7].title, "gi|6121347|gb|G56178.1|G56178 SHGC-101470 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[7].score, 32)
        self.assertAlmostEqual(record.descriptions[7].e, 2.6)
        self.assertEqual(record.descriptions[8].title, "gi|1233216|emb|Z51916|HSA082WB5 H.sapiens (D1S2890) DNA segment...")
        self.assertEqual(record.descriptions[8].score, 32)
        self.assertAlmostEqual(record.descriptions[8].e, 2.6)
        self.assertEqual(record.descriptions[9].title, "gi|1232198|emb|Z50898|HS038XD8 H.sapiens (D18S1106) DNA segment...")
        self.assertEqual(record.descriptions[9].score, 32)
        self.assertAlmostEqual(record.descriptions[9].e, 2.6)
        self.assertEqual(record.descriptions[10].title, "gi|1348143|gb|G25911|G25911 human STS EST349382.")
        self.assertEqual(record.descriptions[10].score, 32)
        self.assertAlmostEqual(record.descriptions[10].e, 2.6)
        self.assertEqual(record.descriptions[11].title, "gi|6122805|gb|G57486.1|G57486 SHGC-103345 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[11].score, 32)
        self.assertAlmostEqual(record.descriptions[11].e, 2.6)
        self.assertEqual(record.descriptions[12].title, "gi|1396897|gb|G28178|G28178 human STS SHGC-34170.")
        self.assertEqual(record.descriptions[12].score, 32)
        self.assertAlmostEqual(record.descriptions[12].e, 2.6)
        self.assertEqual(record.descriptions[13].title, "gi|3893806|emb|AL034156|HSPE59A01 H.sapiens flow-sorted chromos...")
        self.assertEqual(record.descriptions[13].score, 32)
        self.assertAlmostEqual(record.descriptions[13].e, 2.6)
        self.assertEqual(record.descriptions[14].title, "gi|1161890|gb|G16001|G16001 human STS CHLC.GCT8D08.P11278 clone...")
        self.assertEqual(record.descriptions[14].score, 32)
        self.assertAlmostEqual(record.descriptions[14].e, 2.6)
        self.assertEqual(record.descriptions[15].title, "gi|1017612|gb|G11520|G11520 human STS SHGC-14676.")
        self.assertEqual(record.descriptions[15].score, 32)
        self.assertAlmostEqual(record.descriptions[15].e, 2.6)
        self.assertEqual(record.descriptions[16].title, "gi|1130137|gb|G14398|G14398 human STS SHGC-9310 clone pG-5191.")
        self.assertEqual(record.descriptions[16].score, 32)
        self.assertAlmostEqual(record.descriptions[16].e, 2.6)
        self.assertEqual(record.descriptions[17].title, "gi|5224295|gb|G52968.1|G52968 SHGC-86325 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[17].score, 32)
        self.assertAlmostEqual(record.descriptions[17].e, 2.6)
        self.assertEqual(record.descriptions[18].title, "gi|6123581|gb|G58262.1|G58262 SHGC-104352 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[18].score, 32)
        self.assertAlmostEqual(record.descriptions[18].e, 2.6)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(record.posted_date, [('Nov 26, 1999  5:52 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 1.37)
        self.assertAlmostEqual(record.ka_params[1], 0.711)
        self.assertAlmostEqual(record.ka_params[2], 1.31)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 1.37)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.711)
        self.assertAlmostEqual(record.ka_params_gap[2], 1.31)
        self.assertEqual(record.matrix, 'blastn matrix:1 -3')
        self.assertEqual(record.gap_penalties, [5, 2])
        self.assertEqual(record.num_hits, 8762)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 8762)
        self.assertEqual(record.num_good_extends, 2655)
        self.assertEqual(record.num_seqs_better_e, 27)
        self.assertEqual(record.query_length, 434)
        self.assertEqual(record.database_length, 31998854)
        self.assertEqual(record.effective_hsp_length, 17)
        self.assertEqual(record.effective_query_length, 417)
        self.assertEqual(record.effective_database_length, 30506390)
        self.assertEqual(record.effective_search_space, 12721164630)
        self.assertEqual(record.effective_search_space_used, 12721164630)
        self.assertEqual(record.threshold, 0)
        self.assertEqual(record.window_size, 0)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 6)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 11.9)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 25)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 49.6)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 12)
        self.assertAlmostEqual(record.gap_trigger[1], 24.3)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 16)
        self.assertAlmostEqual(record.blast_cutoff[1], 32.2)

    def test_text_2010L_blastn_003(self):
        "Test parsing BLASTN 2.0.10 output without descriptions (text_2010L_blastn_003)"

        path = os.path.join('Blast', 'text_2010L_blastn_003.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTN")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1347201|gb|G24969|G24969 human STS\nEST21946.\x01gi|1375315|gb|G27065|G27065 human STS SHGC-31731.")
        self.assertEqual(record.query_letters, 331)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 9)
        self.assertEqual(record.alignments[0].title, ">gi|1347201|gb|G24969|G24969 human STS EST21946. >gi|1375315|gb|G27065|G27065 human STS SHGC-31731.")
        self.assertEqual(record.alignments[0].length, 331)
        self.assertEqual(record.alignments[1].title, ">gi|2665277|emb|AL010115|HSPE77H4 H.sapiens flow-sorted chromosome 1 HindIII fragment, SC1pE77H4, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[1].length, 554)
        self.assertEqual(record.alignments[2].title, ">gi|6120731|gb|G55412.1|G55412 SHGC-100745 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[2].length, 652)
        self.assertEqual(record.alignments[3].title, ">gi|4493602|gb|G47248.1|G47248 Z17392_1 Zebrafish AB Danio rerio STS genomic clone Z17392 5', sequence tagged site")
        self.assertEqual(record.alignments[3].length, 454)
        self.assertEqual(record.alignments[4].title, ">gi|1342139|gb|G21813|G21813 human STS WI-12408.")
        self.assertEqual(record.alignments[4].length, 418)
        self.assertEqual(record.alignments[5].title, ">gi|1235411|emb|Z53965|HSC009WH1 H.sapiens (D2S2321) DNA segment containing (CA) repeat; clone AFMc009wh1; single read, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[5].length, 382)
        self.assertEqual(record.alignments[6].title, ">gi|4757440|gb|G49267.1|G49267 stbK343C1_96809 chromosome 22 genomic clone Homo sapiens STS genomic clone 343C1, sequence tagged site")
        self.assertEqual(record.alignments[6].length, 360)
        self.assertEqual(record.alignments[7].title, ">gi|719782|gb|G02824|G02824 human STS WI-1312.")
        self.assertEqual(record.alignments[7].length, 349)
        self.assertEqual(record.alignments[8].title, ">gi|939357|gb|G08807|G08807 human STS CHLC.GATA70E11.P18111 clone GATA70E11")
        self.assertEqual(record.alignments[8].length, 643)
        self.assertEqual(record.alignments[0].hsps[0].score, 331)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 656)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 0.0)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 17)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 34.2)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 0.49)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[2].hsps), 1)
        self.assertEqual(record.alignments[3].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[4].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[4].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[4].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[4].hsps), 1)
        self.assertEqual(record.alignments[5].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[5].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[5].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[5].hsps), 1)
        self.assertEqual(record.alignments[6].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[6].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[6].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[6].hsps), 1)
        self.assertEqual(record.alignments[7].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[7].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[7].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[7].hsps), 1)
        self.assertEqual(record.alignments[8].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[8].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[8].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[8].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (331, 331))
        self.assertEqual(record.alignments[1].hsps[0].identities, (17, 17))
        self.assertEqual(record.alignments[2].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[3].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[4].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[5].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[6].hsps[0].identities, (22, 24))
        self.assertEqual(record.alignments[7].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[8].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[0].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[1].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[2].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[3].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[4].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[5].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[6].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[7].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[8].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[0].hsps[0].query, "cctccaccctctcatgagcaacaggatatgtgaaagtacttgcagccagaagcaaaaccacaatcctcgggtgctagatggagctccccaaggagcagagaggaaaaggcaggaggagagggccaggcagcagggatggagactaagtttggcccaaggctgcccgcaagcactgatgccatcatgccctctggtaggtgtctatttctgtctgaaccagaaatacaccaagctccacacatgggggctttgctggcttcgacatcactggttcaactatgtcactgctttgttatatttagtgctccagaacctcaggttccttcagatt")
        self.assertEqual(record.alignments[0].hsps[0].match, "|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "cctccaccctctcatgagcaacaggatatgtgaaagtacttgcagccagaagcaaaaccacaatcctcgggtgctagatggagctccccaaggagcagagaggaaaaggcaggaggagagggccaggcagcagggatggagactaagtttggcccaaggctgcccgcaagcactgatgccatcatgccctctggtaggtgtctatttctgtctgaaccagaaatacaccaagctccacacatgggggctttgctggcttcgacatcactggttcaactatgtcactgctttgttatatttagtgctccagaacctcaggttccttcagatt")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 331)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 331)
        self.assertEqual(record.alignments[1].hsps[0].query, "ccaggcagcagggatgg")
        self.assertEqual(record.alignments[1].hsps[0].match, "|||||||||||||||||")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "ccaggcagcagggatgg")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 123)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 139)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 434)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 418)
        self.assertEqual(record.alignments[2].hsps[0].query, "agaggaaaaggcagga")
        self.assertEqual(record.alignments[2].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "agaggaaaaggcagga")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 99)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 114)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 431)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 416)
        self.assertEqual(record.alignments[3].hsps[0].query, "agaagcaaaaccacaa")
        self.assertEqual(record.alignments[3].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "agaagcaaaaccacaa")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 48)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 63)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 434)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 449)
        self.assertEqual(record.alignments[4].hsps[0].query, "cagaagcaaaaccaca")
        self.assertEqual(record.alignments[4].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[4].hsps[0].sbjct, "cagaagcaaaaccaca")
        self.assertEqual(record.alignments[4].hsps[0].query_start, 47)
        self.assertEqual(record.alignments[4].hsps[0].query_end, 62)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_start, 193)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_end, 178)
        self.assertEqual(record.alignments[5].hsps[0].query, "agagaggaaaaggcag")
        self.assertEqual(record.alignments[5].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[5].hsps[0].sbjct, "agagaggaaaaggcag")
        self.assertEqual(record.alignments[5].hsps[0].query_start, 97)
        self.assertEqual(record.alignments[5].hsps[0].query_end, 112)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_start, 107)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_end, 122)
        self.assertEqual(record.alignments[6].hsps[0].query, "ggaggagagggccaggcagcaggg")
        self.assertEqual(record.alignments[6].hsps[0].match, "||||||||||| | ||||||||||")
        self.assertEqual(record.alignments[6].hsps[0].sbjct, "ggaggagaggggctggcagcaggg")
        self.assertEqual(record.alignments[6].hsps[0].query_start, 112)
        self.assertEqual(record.alignments[6].hsps[0].query_end, 135)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_start, 287)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_end, 264)
        self.assertEqual(record.alignments[7].hsps[0].query, "tgctttgttatattta")
        self.assertEqual(record.alignments[7].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[7].hsps[0].sbjct, "tgctttgttatattta")
        self.assertEqual(record.alignments[7].hsps[0].query_start, 286)
        self.assertEqual(record.alignments[7].hsps[0].query_end, 301)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_start, 111)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_end, 126)
        self.assertEqual(record.alignments[8].hsps[0].query, "gaggaaaaggcaggag")
        self.assertEqual(record.alignments[8].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[8].hsps[0].sbjct, "gaggaaaaggcaggag")
        self.assertEqual(record.alignments[8].hsps[0].query_start, 100)
        self.assertEqual(record.alignments[8].hsps[0].query_end, 115)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_start, 482)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_end, 467)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(record.posted_date, [('Nov 26, 1999  5:52 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 1.37)
        self.assertAlmostEqual(record.ka_params[1], 0.711)
        self.assertAlmostEqual(record.ka_params[2], 1.31)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 1.37)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.711)
        self.assertAlmostEqual(record.ka_params_gap[2], 1.31)
        self.assertEqual(record.matrix, 'blastn matrix:1 -3')
        self.assertEqual(record.gap_penalties, [5, 2])
        self.assertEqual(record.num_hits, 6844)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 6844)
        self.assertEqual(record.num_good_extends, 1887)
        self.assertEqual(record.num_seqs_better_e, 14)
        self.assertEqual(record.query_length, 331)
        self.assertEqual(record.database_length, 31998854)
        self.assertEqual(record.effective_hsp_length, 17)
        self.assertEqual(record.effective_query_length, 314)
        self.assertEqual(record.effective_database_length, 30506390)
        self.assertEqual(record.effective_search_space, 9579006460)
        self.assertEqual(record.effective_search_space_used, 9579006460)
        self.assertEqual(record.threshold, 0)
        self.assertEqual(record.window_size, 0)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 6)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 11.9)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 25)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 49.6)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 12)
        self.assertAlmostEqual(record.gap_trigger[1], 24.3)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 15)
        self.assertAlmostEqual(record.blast_cutoff[1], 30.2)

    def test_text_2010L_blastn_004(self):
        "Test parsing BLASTN 2.0.10 output (text_2010L_blastn_004)"

        path = os.path.join('Blast', 'text_2010L_blastn_004.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTN")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|859351|gb|G06106|G06106 human STS WI-6344.")
        self.assertEqual(record.query_letters, 183)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 6)
        self.assertEqual(record.descriptions[0].title, "gi|859351|gb|G06106|G06106 human STS WI-6344.")
        self.assertEqual(record.descriptions[0].score, 327)
        self.assertAlmostEqual(record.descriptions[0].e, 1e-89)
        self.assertEqual(record.descriptions[1].title, "gi|1341350|gb|G21024|G21024 human STS WI-30979.")
        self.assertEqual(record.descriptions[1].score, 32)
        self.assertAlmostEqual(record.descriptions[1].e, 1.0)
        self.assertEqual(record.descriptions[2].title, "gi|6126285|gb|G60966.1|G60966 SHGC-84377 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[2].score, 30)
        self.assertAlmostEqual(record.descriptions[2].e, 4.1)
        self.assertEqual(record.descriptions[3].title, "gi|1340656|gb|G20319|G20319 human STS A005L39.")
        self.assertEqual(record.descriptions[3].score, 30)
        self.assertAlmostEqual(record.descriptions[3].e, 4.1)
        self.assertEqual(record.descriptions[4].title, "gi|5222421|gb|G51244.1|G51244 SHGC-80725 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[4].score, 30)
        self.assertAlmostEqual(record.descriptions[4].e, 4.1)
        self.assertEqual(record.descriptions[5].title, "gi|860526|gb|G07281|G07281 human STS WI-9430.")
        self.assertEqual(record.descriptions[5].score, 30)
        self.assertAlmostEqual(record.descriptions[5].e, 4.1)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(record.posted_date, [('Nov 26, 1999  5:52 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 1.37)
        self.assertAlmostEqual(record.ka_params[1], 0.711)
        self.assertAlmostEqual(record.ka_params[2], 1.31)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 1.37)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.711)
        self.assertAlmostEqual(record.ka_params_gap[2], 1.31)
        self.assertEqual(record.matrix, 'blastn matrix:1 -3')
        self.assertEqual(record.gap_penalties, [5, 2])
        self.assertEqual(record.num_hits, 1902)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 1902)
        self.assertEqual(record.num_good_extends, 481)
        self.assertEqual(record.num_seqs_better_e, 8)
        self.assertEqual(record.query_length, 183)
        self.assertEqual(record.database_length, 31998854)
        self.assertEqual(record.effective_hsp_length, 16)
        self.assertEqual(record.effective_query_length, 167)
        self.assertEqual(record.effective_database_length, 30594182)
        self.assertEqual(record.effective_search_space, 5109228394)
        self.assertEqual(record.effective_search_space_used, 5109228394)
        self.assertEqual(record.threshold, 0)
        self.assertEqual(record.window_size, 0)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 6)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 11.9)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 25)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 49.6)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 12)
        self.assertAlmostEqual(record.gap_trigger[1], 24.3)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 15)
        self.assertAlmostEqual(record.blast_cutoff[1], 30.2)

    def test_text_2010L_blastx_001(self):
        "Test parsing BLASTX 2.0.10 output (text_2010L_blastx_001)"

        path = os.path.join('Blast', 'text_2010L_blastx_001.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTX")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1347369|gb|G25137|G25137 human STS EST48004.")
        self.assertEqual(record.query_letters, 556)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 4)
        self.assertEqual(record.descriptions[0].title, "gi|1731448|sp|P54103|ZRF1_MOUSE ZUOTIN RELATED FACTOR-1")
        self.assertEqual(record.descriptions[0].score, 87)
        self.assertAlmostEqual(record.descriptions[0].e, 3e-17)
        self.assertEqual(record.descriptions[1].title, "gi|465911|sp|P34454|YMA9_CAEEL HYPOTHETICAL 31.6 KD PROTEIN F54...")
        self.assertEqual(record.descriptions[1].score, 42)
        self.assertAlmostEqual(record.descriptions[1].e, 0.001)
        self.assertEqual(record.descriptions[2].title, "gi|2494160|sp|Q61712|MTJ1_MOUSE DNAJ PROTEIN HOMOLOG MTJ1")
        self.assertEqual(record.descriptions[2].score, 37)
        self.assertAlmostEqual(record.descriptions[2].e, 0.033)
        self.assertEqual(record.descriptions[3].title, "gi|1730688|sp|P53745|YN8X_YEAST HYPOTHETICAL 68.1 KD PROTEIN IN...")
        self.assertEqual(record.descriptions[3].score, 29)
        self.assertAlmostEqual(record.descriptions[3].e, 7.4)
        self.assertEqual(len(record.alignments), 4)
        self.assertEqual(record.alignments[0].title, ">gi|1731448|sp|P54103|ZRF1_MOUSE ZUOTIN RELATED FACTOR-1")
        self.assertEqual(record.alignments[0].length, 514)
        self.assertEqual(record.alignments[1].title, ">gi|465911|sp|P34454|YMA9_CAEEL HYPOTHETICAL 31.6 KD PROTEIN F54F2.9 IN CHROMOSOME III")
        self.assertEqual(record.alignments[1].length, 275)
        self.assertEqual(record.alignments[2].title, ">gi|2494160|sp|Q61712|MTJ1_MOUSE DNAJ PROTEIN HOMOLOG MTJ1")
        self.assertEqual(record.alignments[2].length, 552)
        self.assertEqual(record.alignments[3].title, ">gi|1730688|sp|P53745|YN8X_YEAST HYPOTHETICAL 68.1 KD PROTEIN IN BIO3-FRE4 INTERGENIC REGION")
        self.assertEqual(record.alignments[3].length, 580)
        self.assertEqual(record.alignments[0].hsps[0].score, 211)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 86.6)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 3e-17)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 96)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 41.8)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 0.001)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 83)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 36.7)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 0.033)
        self.assertEqual(record.alignments[2].hsps[1].score, 69)
        self.assertAlmostEqual(record.alignments[2].hsps[1].bits, 31.3)
        self.assertAlmostEqual(record.alignments[2].hsps[1].expect, 1.5)
        self.assertEqual(len(record.alignments[2].hsps), 2)
        self.assertEqual(record.alignments[3].hsps[0].score, 63)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 29.0)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 7.4)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (41, 47))
        self.assertEqual(record.alignments[0].hsps[0].positives, (44, 47))
        self.assertEqual(record.alignments[1].hsps[0].identities, (30, 122))
        self.assertEqual(record.alignments[1].hsps[0].positives, (54, 122))
        self.assertEqual(record.alignments[2].hsps[0].identities, (17, 36))
        self.assertEqual(record.alignments[2].hsps[0].positives, (19, 36))
        self.assertEqual(record.alignments[2].hsps[1].identities, (18, 50))
        self.assertEqual(record.alignments[2].hsps[1].positives, (26, 50))
        self.assertEqual(record.alignments[3].hsps[0].identities, (27, 99))
        self.assertEqual(record.alignments[3].hsps[0].positives, (41, 99))
        self.assertEqual(record.alignments[0].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[1].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[2].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[2].hsps[1].frame, ("+1", ))
        self.assertEqual(record.alignments[3].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[0].hsps[0].query, "DLQLLIKAVNLFPAGTNSRWEVIANYMNIHSSSGVKRTAKDVIGKAK")
        self.assertEqual(record.alignments[0].hsps[0].match, "DLQLLIKAVNLFPAG NSRW+VIANYMNIHSSSGVKRTAKDVI + +")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "DLQLLIKAVNLFPAGRNSRWDVIANYMNIHSSSGVKRTAKDVISEVR")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 141)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 458)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 504)
        self.assertEqual(record.alignments[1].hsps[0].query, "FPAGTNSRWEVIANYMNIHSSSGVKRTAKDVIGKAKSLQKLDPHQKDDINKKAFDKFKKEHGVVPQADNATPSERFXGPYTDFTPXTTEXQKLXEQALNTYPVNTXERWXXIAVAVPGRXKE")
        self.assertEqual(record.alignments[1].hsps[0].match, "+PAGT +RWE +   +N        R+A+DVI  A  ++++   +++D  K      ++   V  ++++                 +   QK  E AL  YP  T ERW  I+  +  + K+")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "YPAGTPNRWEQMGRVLN--------RSAEDVIAMAGKMKQM---KQEDYTKLLMTTIQQSVPVEEKSED---------------DWSQAEQKAFETALQKYPKGTDERWERISEEIGSKTKK")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 34)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 399)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 159)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 254)
        self.assertEqual(record.alignments[2].hsps[0].query, "TTEXQKLXEQALNTYPVNTXERWXXIAVAVPGRXKE")
        self.assertEqual(record.alignments[2].hsps[0].match, "T   QKL E AL  YP    +RW  IA  VP + KE")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "TQSQQKLLELALQQYPKGASDRWDKIAKCVPSKSKE")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 292)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 399)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 496)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 531)
        self.assertEqual(record.alignments[2].hsps[1].query, "DLQLLIKAVNLFPAGTNSRWEVIANYMNIHSSSGVKRTAKDVIGKAKSLQ")
        self.assertEqual(record.alignments[2].hsps[1].match, "DL  L +++  FP GT  RW+ IA+ +         R+  DV  KAK L+")
        self.assertEqual(record.alignments[2].hsps[1].sbjct, "DLSQLTRSMVKFPGGTPGRWDKIAHELG--------RSVTDVTTKAKELK")
        self.assertEqual(record.alignments[2].hsps[1].query_start, 1)
        self.assertEqual(record.alignments[2].hsps[1].query_end, 150)
        self.assertEqual(record.alignments[2].hsps[1].sbjct_start, 332)
        self.assertEqual(record.alignments[2].hsps[1].sbjct_end, 373)
        self.assertEqual(record.alignments[3].hsps[0].query, "SRWEVIANYMNIHSSSGVKRTAKDVIGKAKSLQKLDPHQKDDINKKAFDKFKKEHGVVPQADNATPSERFXGPYTDFTPXTTEXQKLXEQALNTYPVNT")
        self.assertEqual(record.alignments[3].hsps[0].match, "+RW+   +Y        V R+ KDV   ++SL  LD +QK     +A       H +          E    PY +FT   +      EQ+ N +PV+T")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "NRWKSFISY--------VTRSRKDVKTVSRSLSNLDLYQKCSKEIRADQDISLLHSI----------ETKLFPYINFTALNS------EQSHNFWPVHT")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 52)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 348)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 75)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 149)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Nov 15, 1999  2:55 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.318)
        self.assertAlmostEqual(record.ka_params[1], 0.135)
        self.assertAlmostEqual(record.ka_params[2], 0.401)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 23174157)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 387821)
        self.assertEqual(record.num_good_extends, 980)
        self.assertEqual(record.num_seqs_better_e, 8)
        self.assertEqual(record.hsps_no_gap, 3)
        self.assertEqual(record.hsps_prelim_gapped, 1)
        self.assertEqual(record.hsps_gapped, 7)
        self.assertEqual(record.query_length, 185)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 49)
        self.assertEqual(record.effective_query_length, 135)
        self.assertEqual(record.effective_database_length, 25621919)
        self.assertEqual(record.effective_search_space, 3458959065)
        self.assertEqual(record.effective_search_space_used, 3458959065)
        self.assertEqual(record.frameshift, ('50,', '0.1'))
        self.assertEqual(record.threshold, 12)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 62)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.6)

    def test_text_2010L_blastx_002(self):
        "Test parsing BLASTX 2.0.10 output without hits (text_2010L_blastx_002)"

        path = os.path.join('Blast', 'text_2010L_blastx_002.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTX")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1347782|gb|G25550|G25550 human STS\nEST47652.\x01gi|1592937|gb|G29386|G29386 human STS SHGC-32770")
        self.assertEqual(record.query_letters, 379)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Nov 15, 1999  2:55 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.318)
        self.assertAlmostEqual(record.ka_params[1], 0.135)
        self.assertAlmostEqual(record.ka_params[2], 0.401)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 14686001)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 235383)
        self.assertEqual(record.num_good_extends, 396)
        self.assertEqual(record.num_seqs_better_e, 0)
        self.assertEqual(record.hsps_no_gap, 0)
        self.assertEqual(record.hsps_prelim_gapped, 0)
        self.assertEqual(record.hsps_gapped, 0)
        self.assertEqual(record.query_length, 126)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 48)
        self.assertEqual(record.effective_query_length, 77)
        self.assertEqual(record.effective_database_length, 25704177)
        self.assertEqual(record.effective_search_space, 1979221629)
        self.assertEqual(record.effective_search_space_used, 1979221629)
        self.assertEqual(record.frameshift, ('50,', '0.1'))
        self.assertEqual(record.threshold, 12)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 60)
        self.assertAlmostEqual(record.blast_cutoff[1], 27.8)

    def test_text_2010L_tblastn_001(self):
        "Test parsing TBLASTN 2.0.10 output (text_2010L_tblastn_001)"

        path = os.path.join('Blast', 'text_2010L_tblastn_001.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "TBLASTN")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|729325|sp|P39483|DHG2_BACME GLUCOSE 1-DEHYDROGENASE II\n(GLCDH-II)")
        self.assertEqual(record.query_letters, 261)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 7)
        self.assertEqual(record.descriptions[0].title, "gi|3820341|emb|AJ229891|KLAJ9891 Kluyveromyces lactis DNA fragm...")
        self.assertEqual(record.descriptions[0].score, 47)
        self.assertAlmostEqual(record.descriptions[0].e, 0.00001)
        self.assertEqual(record.descriptions[1].title, "gi|1375419|gb|G27169|G27169 human STS SHGC-31983.")
        self.assertEqual(record.descriptions[1].score, 43)
        self.assertAlmostEqual(record.descriptions[1].e, 0.0001)
        self.assertEqual(record.descriptions[2].title, "gi|3819804|emb|AJ230012|KLAJ0012 Kluyveromyces lactis DNA fragm...")
        self.assertEqual(record.descriptions[2].score, 39)
        self.assertAlmostEqual(record.descriptions[2].e, 0.002)
        self.assertEqual(record.descriptions[3].title, "gi|1375215|gb|G26965|G26965 human STS SHGC-31083.")
        self.assertEqual(record.descriptions[3].score, 31)
        self.assertAlmostEqual(record.descriptions[3].e, 0.73)
        self.assertEqual(record.descriptions[4].title, "gi|177714|gb|L09988|HUM4STS889 Human Chromosome 4 (clone p4-109...")
        self.assertEqual(record.descriptions[4].score, 29)
        self.assertAlmostEqual(record.descriptions[4].e, 2.2)
        self.assertEqual(record.descriptions[5].title, "gi|5714409|gb|AF106665.1|AF106665 Mus musculus chromosome 6 clo...")
        self.assertEqual(record.descriptions[5].score, 29)
        self.assertAlmostEqual(record.descriptions[5].e, 2.2)
        self.assertEqual(record.descriptions[6].title, "gi|1341648|gb|G21322|G21322 human STS WI-12250.")
        self.assertEqual(record.descriptions[6].score, 29)
        self.assertAlmostEqual(record.descriptions[6].e, 3.7)
        self.assertEqual(len(record.alignments), 7)
        self.assertEqual(record.alignments[0].title, ">gi|3820341|emb|AJ229891|KLAJ9891 Kluyveromyces lactis DNA fragment for sequence tagged site, clone okam5d07r [Kluyveromyces lactis]")
        self.assertEqual(record.alignments[0].length, 230)
        self.assertEqual(record.alignments[1].title, ">gi|1375419|gb|G27169|G27169 human STS SHGC-31983.")
        self.assertEqual(record.alignments[1].length, 594)
        self.assertEqual(record.alignments[2].title, ">gi|3819804|emb|AJ230012|KLAJ0012 Kluyveromyces lactis DNA fragment for sequence tagged site, clone okam6d01d [Kluyveromyces lactis]")
        self.assertEqual(record.alignments[2].length, 199)
        self.assertEqual(record.alignments[3].title, ">gi|1375215|gb|G26965|G26965 human STS SHGC-31083.")
        self.assertEqual(record.alignments[3].length, 268)
        self.assertEqual(record.alignments[4].title, ">gi|177714|gb|L09988|HUM4STS889 Human Chromosome 4 (clone p4-1095) STS4-889.")
        self.assertEqual(record.alignments[4].length, 412)
        self.assertEqual(record.alignments[5].title, ">gi|5714409|gb|AF106665.1|AF106665 Mus musculus chromosome 6 clone D6wum9 map between Nkrp1 and Prp strain C57BL/6J, sequence tagged site")
        self.assertEqual(record.alignments[5].length, 299)
        self.assertEqual(record.alignments[6].title, ">gi|1341648|gb|G21322|G21322 human STS WI-12250.")
        self.assertEqual(record.alignments[6].length, 586)
        self.assertEqual(record.alignments[0].hsps[0].score, 109)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 46.9)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 1e-5)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 100)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 43.4)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 1e-4)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 90)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 39.5)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 0.002)
        self.assertEqual(len(record.alignments[2].hsps), 1)
        self.assertEqual(record.alignments[3].hsps[0].score, 68)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 30.9)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 0.73)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[4].hsps[0].score, 64)
        self.assertAlmostEqual(record.alignments[4].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.alignments[4].hsps[0].expect, 2.2)
        self.assertEqual(len(record.alignments[4].hsps), 1)
        self.assertEqual(record.alignments[5].hsps[0].score, 64)
        self.assertAlmostEqual(record.alignments[5].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.alignments[5].hsps[0].expect, 2.2)
        self.assertEqual(len(record.alignments[5].hsps), 1)
        self.assertEqual(record.alignments[6].hsps[0].score, 62)
        self.assertAlmostEqual(record.alignments[6].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.alignments[6].hsps[0].expect, 3.7)
        self.assertEqual(len(record.alignments[6].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (25, 72))
        self.assertEqual(record.alignments[0].hsps[0].positives, (44, 72))
        self.assertEqual(record.alignments[0].hsps[0].gaps, (3, 72))
        self.assertEqual(record.alignments[1].hsps[0].identities, (21, 73))
        self.assertEqual(record.alignments[1].hsps[0].positives, (34, 73))
        self.assertEqual(record.alignments[2].hsps[0].identities, (18, 49))
        self.assertEqual(record.alignments[2].hsps[0].positives, (26, 49))
        self.assertEqual(record.alignments[3].hsps[0].identities, (12, 37))
        self.assertEqual(record.alignments[3].hsps[0].positives, (19, 37))
        self.assertEqual(record.alignments[4].hsps[0].identities, (14, 34))
        self.assertEqual(record.alignments[4].hsps[0].positives, (22, 34))
        self.assertEqual(record.alignments[5].hsps[0].identities, (17, 55))
        self.assertEqual(record.alignments[5].hsps[0].positives, (32, 55))
        self.assertEqual(record.alignments[5].hsps[0].gaps, (2, 55))
        self.assertEqual(record.alignments[6].hsps[0].identities, (16, 39))
        self.assertEqual(record.alignments[6].hsps[0].positives, (20, 39))
        self.assertEqual(record.alignments[6].hsps[0].gaps, (1, 39))
        self.assertEqual(record.alignments[0].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[1].hsps[0].frame, ("-1", ))
        self.assertEqual(record.alignments[2].hsps[0].frame, ("-1", ))
        self.assertEqual(record.alignments[3].hsps[0].frame, ("-1", ))
        self.assertEqual(record.alignments[4].hsps[0].frame, ("-1", ))
        self.assertEqual(record.alignments[5].hsps[0].frame, ("-2", ))
        self.assertEqual(record.alignments[6].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[0].hsps[0].query, "NWNQVIDTNLTGAFLGSREAIKYFVEN---DIKGNVINMSSVHEMIPWPLFVHYAASKGGMKLMTETLALEYAPK")
        self.assertEqual(record.alignments[0].hsps[0].match, "+W QVIDTN+ G F   + A+     +   D +  V+N+S+V+ ++  P    Y A+K  +  +T+++ALEYA +")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "SWRQVIDTNINGTFYTLKYALPLMESSSSPDSEAAVVNLSAVNGLVGIPGISPYTATKHAVIGITQSVALEYAER")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 108)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 179)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 225)
        self.assertEqual(record.alignments[1].hsps[0].query, "APKGIRVNNIGPGAIDTPINAEKFADPEQRADVESMIPMGYIGKPEEIASVAAFLASSQASYVTGITLFADGG")
        self.assertEqual(record.alignments[1].hsps[0].match, "AP   RVN + P  + T +    + DP +   + +  P+G   + E +     FL S ++   TG TL  +GG")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "APXRXRVNAVXPXVVMTSMGQATWXDPXKAXTMLNRXPLGXFAEVEHVVKAILFLLSDRSGMTTGSTLPVEGG")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 177)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 249)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 312)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 94)
        self.assertEqual(record.alignments[2].hsps[0].query, "FADPEQRADVESMIPMGYIGKPEEIASVAAFLASSQASYVTGITLFADG")
        self.assertEqual(record.alignments[2].hsps[0].match, "F D + +    S+IPMG  G P+E+     + AS  ++Y TG  L  DG")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "FVDEDLKNKWHSLIPMGREGLPQELVGAYLYFASDASTYTTGSDLLVDG")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 200)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 248)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 157)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 11)
        self.assertEqual(record.alignments[3].hsps[0].query, "PMGYIGKPEEIASVAAFLASSQASYVTGITLFADGGM")
        self.assertEqual(record.alignments[3].hsps[0].match, "PMG  G PE++  V A      +  +TG ++   GG+")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "PMGXXGDPEDVXDVXAXXXXEXSGXITGTSVEVTGGL")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 214)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 250)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 268)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 158)
        self.assertEqual(record.alignments[4].hsps[0].query, "DKVVVVTGGSKGLGRAMAVRFGQEQSKVVVNYRS")
        self.assertEqual(record.alignments[4].hsps[0].match, "DKV  V GGS+G+GRA+A    ++  ++ V  R+")
        self.assertEqual(record.alignments[4].hsps[0].sbjct, "DKVCAVFGGSRGIGRAVAQLMARKGYRLAVIARN")
        self.assertEqual(record.alignments[4].hsps[0].query_start, 7)
        self.assertEqual(record.alignments[4].hsps[0].query_end, 40)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_start, 316)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_end, 215)
        self.assertEqual(record.alignments[5].hsps[0].query, "NMSSVHEMIPWPLFVHYAASKGGMKLMTETL--ALEYAPKGIRVNNIGPGAIDTPIN")
        self.assertEqual(record.alignments[5].hsps[0].match, "++S    +I +P F+    S  G  L+  +L  A+ + P GI V+++GP ++ T +N")
        self.assertEqual(record.alignments[5].hsps[0].sbjct, "SLSPTQYLIMFPSFLPCPLSHPGPFLLPSSLVIAVFFLPNGIEVSSLGPFSLRTLLN")
        self.assertEqual(record.alignments[5].hsps[0].query_start, 142)
        self.assertEqual(record.alignments[5].hsps[0].query_end, 196)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_start, 172)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_end, 2)
        self.assertEqual(record.alignments[6].hsps[0].query, "PVPSHELSLENWNQ-VIDTNLTGAFLGSREAIKYFVENDI")
        self.assertEqual(record.alignments[6].hsps[0].match, "PVP  ELS  +W+Q  + T+ T  F  S     YF  N I")
        self.assertEqual(record.alignments[6].hsps[0].sbjct, "PVPMQELSKVHWSQFFLTTSPTMTFFFSHYLANYFFRNSI")
        self.assertEqual(record.alignments[6].hsps[0].query_start, 98)
        self.assertEqual(record.alignments[6].hsps[0].query_end, 136)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_start, 220)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_end, 339)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(record.posted_date, [('Nov 26, 1999  5:52 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.315)
        self.assertAlmostEqual(record.ka_params[1], 0.134)
        self.assertAlmostEqual(record.ka_params[2], 0.378)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 6165875)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 55665)
        self.assertEqual(record.num_good_extends, 148)
        self.assertEqual(record.num_seqs_better_e, 14)
        self.assertEqual(record.hsps_no_gap, 5)
        self.assertEqual(record.hsps_prelim_gapped, 2)
        self.assertEqual(record.hsps_gapped, 7)
        self.assertEqual(record.query_length, 261)
        self.assertEqual(record.database_length, 10666284)
        self.assertEqual(record.effective_hsp_length, 50)
        self.assertEqual(record.effective_query_length, 211)
        self.assertEqual(record.effective_database_length, 6276684)
        self.assertEqual(record.effective_search_space, 1324380324)
        self.assertEqual(record.effective_search_space_used, 1324380324)
        self.assertEqual(record.frameshift, ('50,', '0.1'))
        self.assertEqual(record.threshold, 13)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 42)
        self.assertAlmostEqual(record.gap_trigger[1], 22.0)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 58)
        self.assertAlmostEqual(record.blast_cutoff[1], 27.0)

    def test_text_2010L_tblastn_002(self):
        "Test parsing TBLASTN 2.0.10 output without hits (text_2010L_tblastn_002)"

        path = os.path.join('Blast', 'text_2010L_tblastn_002.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "TBLASTN")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|127420|sp|P19888|MTBA_BACAR MODIFICATION METHYLASE BANI\n(CYTOSINE-SPECIFIC METHYLTRANSFERASE BANI) (M.BANI)")
        self.assertEqual(record.query_letters, 428)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(record.posted_date, [('Nov 26, 1999  5:52 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.320)
        self.assertAlmostEqual(record.ka_params[1], 0.140)
        self.assertAlmostEqual(record.ka_params[2], 0.403)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 13588598)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 162273)
        self.assertEqual(record.num_good_extends, 546)
        self.assertEqual(record.num_seqs_better_e, 0)
        self.assertEqual(record.hsps_no_gap, 0)
        self.assertEqual(record.hsps_prelim_gapped, 0)
        self.assertEqual(record.hsps_gapped, 0)
        self.assertEqual(record.query_length, 428)
        self.assertEqual(record.database_length, 10666284)
        self.assertEqual(record.effective_hsp_length, 48)
        self.assertEqual(record.effective_query_length, 380)
        self.assertEqual(record.effective_database_length, 6452268)
        self.assertEqual(record.effective_search_space, 2451861840)
        self.assertEqual(record.effective_search_space_used, 2451861840)
        self.assertEqual(record.frameshift, ('50,', '0.1'))
        self.assertEqual(record.threshold, 13)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.4)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.8)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 61)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.2)

    def test_text_2010L_tblastx_001(self):
        "Test parsing TBLASTX 2.0.10 output (text_2010L_tblastx_001)"

        path = os.path.join('Blast', 'text_2010L_tblastx_001.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "TBLASTX")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1348853|gb|G26621|G26621 human STS\nSTS_D12006.\x01gi|1396339|gb|G27620|G27620 human STS SHGC-32705.")
        self.assertEqual(record.query_letters, 615)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 19)
        self.assertEqual(record.descriptions[0].title, "gi|1348853|gb|G26621|G26621 human STS STS_D12006. >gi|1396339|g...")
        self.assertEqual(record.descriptions[0].score, 398)
        self.assertAlmostEqual(record.descriptions[0].e, 1e-111)
        self.assertEqual(record.descriptions[1].title, "gi|1348016|gb|G25784|G25784 human STS EST47998.")
        self.assertEqual(record.descriptions[1].score, 302)
        self.assertAlmostEqual(record.descriptions[1].e, 1e-83)
        self.assertEqual(record.descriptions[2].title, "gi|3403105|gb|G41148|G41148 Z7324 Zebrafish AB Danio rerio STS ...")
        self.assertEqual(record.descriptions[2].score, 31)
        self.assertAlmostEqual(record.descriptions[2].e, 0.6)
        self.assertEqual(record.descriptions[3].title, "gi|1234967|emb|Z53521|HSB316XA9 H.sapiens (D22S1166) DNA segmen...")
        self.assertEqual(record.descriptions[3].score, 30)
        self.assertAlmostEqual(record.descriptions[3].e, 0.83)
        self.assertEqual(record.descriptions[4].title, "gi|4185670|gb|G42865|G42865 Xq3845 KWOK Homo sapiens STS genomi...")
        self.assertEqual(record.descriptions[4].score, 30)
        self.assertAlmostEqual(record.descriptions[4].e, 1.1)
        self.assertEqual(record.descriptions[5].title, "gi|4757419|gb|G49246.1|G49246 stbK116F5_30376 chromosome 22 gen...")
        self.assertEqual(record.descriptions[5].score, 29)
        self.assertAlmostEqual(record.descriptions[5].e, 1.6)
        self.assertEqual(record.descriptions[6].title, "gi|6120694|gb|G55375.1|G55375 SHGC-100697 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[6].score, 29)
        self.assertAlmostEqual(record.descriptions[6].e, 3.0)
        self.assertEqual(record.descriptions[7].title, "gi|5225124|gb|G53947.1|G53947 SHGC-85304 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[7].score, 29)
        self.assertAlmostEqual(record.descriptions[7].e, 3.0)
        self.assertEqual(record.descriptions[8].title, "gi|1311530|gb|L77996|HUMSWX945 Human chromosome X STS sWXD945, ...")
        self.assertEqual(record.descriptions[8].score, 28)
        self.assertAlmostEqual(record.descriptions[8].e, 4.1)
        self.assertEqual(record.descriptions[9].title, "gi|2734624|gb|G36957|G36957 SHGC-56642 Human Homo sapiens STS cDNA")
        self.assertEqual(record.descriptions[9].score, 28)
        self.assertAlmostEqual(record.descriptions[9].e, 4.1)
        self.assertEqual(record.descriptions[10].title, "gi|4631600|dbj|AU046965.1|AU046965 Rattus norvegicus, OTSUKA cl...")
        self.assertEqual(record.descriptions[10].score, 28)
        self.assertAlmostEqual(record.descriptions[10].e, 4.1)
        self.assertEqual(record.descriptions[11].title, "gi|4631518|dbj|AU046883.1|AU046883 Rattus norvegicus, OTSUKA cl...")
        self.assertEqual(record.descriptions[11].score, 28)
        self.assertAlmostEqual(record.descriptions[11].e, 4.1)
        self.assertEqual(record.descriptions[12].title, "gi|2226478|gb|G33174|G33174 human STS SHGC-6097 clone pG-2470")
        self.assertEqual(record.descriptions[12].score, 28)
        self.assertAlmostEqual(record.descriptions[12].e, 4.1)
        self.assertEqual(record.descriptions[13].title, "gi|938611|gb|G08061|G08061 human STS CHLC.GGAA7E02.P7438 clone ...")
        self.assertEqual(record.descriptions[13].score, 27)
        self.assertAlmostEqual(record.descriptions[13].e, 7.7)
        self.assertEqual(record.descriptions[14].title, "gi|307789|gb|L18105|HUMUT1736 Human STS UT1736.")
        self.assertEqual(record.descriptions[14].score, 27)
        self.assertAlmostEqual(record.descriptions[14].e, 7.7)
        self.assertEqual(record.descriptions[15].title, "gi|859804|gb|G06559|G06559 human STS WI-7401.")
        self.assertEqual(record.descriptions[15].score, 27)
        self.assertAlmostEqual(record.descriptions[15].e, 7.7)
        self.assertEqual(record.descriptions[16].title, "gi|4493143|gb|G46852.1|G46852 Z14841_1 Zebrafish AB Danio rerio...")
        self.assertEqual(record.descriptions[16].score, 27)
        self.assertAlmostEqual(record.descriptions[16].e, 7.7)
        self.assertEqual(record.descriptions[17].title, "gi|4492122|gb|G45831.1|G45831 Z4588_1 Zebrafish AB Danio rerio ...")
        self.assertEqual(record.descriptions[17].score, 27)
        self.assertAlmostEqual(record.descriptions[17].e, 7.7)
        self.assertEqual(record.descriptions[18].title, "gi|6121804|gb|G56635.1|G56635 SHGC-102032 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[18].score, 27)
        self.assertAlmostEqual(record.descriptions[18].e, 7.7)
        self.assertEqual(len(record.alignments), 19)
        self.assertEqual(record.alignments[0].title, ">gi|1348853|gb|G26621|G26621 human STS STS_D12006. >gi|1396339|gb|G27620|G27620 human STS SHGC-32705.")
        self.assertEqual(record.alignments[0].length, 615)
        self.assertEqual(record.alignments[1].title, ">gi|1348016|gb|G25784|G25784 human STS EST47998.")
        self.assertEqual(record.alignments[1].length, 617)
        self.assertEqual(record.alignments[2].title, ">gi|3403105|gb|G41148|G41148 Z7324 Zebrafish AB Danio rerio STS genomic")
        self.assertEqual(record.alignments[2].length, 351)
        self.assertEqual(record.alignments[3].title, ">gi|1234967|emb|Z53521|HSB316XA9 H.sapiens (D22S1166) DNA segment containing (CA) repeat; clone AFMb316xa9; single read, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[3].length, 345)
        self.assertEqual(record.alignments[4].title, ">gi|4185670|gb|G42865|G42865 Xq3845 KWOK Homo sapiens STS genomic, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[4].length, 1200)
        self.assertEqual(record.alignments[5].title, ">gi|4757419|gb|G49246.1|G49246 stbK116F5_30376 chromosome 22 genomic clone Homo sapiens STS genomic clone 116F5, sequence tagged site")
        self.assertEqual(record.alignments[5].length, 375)
        self.assertEqual(record.alignments[6].title, ">gi|6120694|gb|G55375.1|G55375 SHGC-100697 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[6].length, 460)
        self.assertEqual(record.alignments[7].title, ">gi|5225124|gb|G53947.1|G53947 SHGC-85304 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[7].length, 444)
        self.assertEqual(record.alignments[8].title, ">gi|1311530|gb|L77996|HUMSWX945 Human chromosome X STS sWXD945, single read.")
        self.assertEqual(record.alignments[8].length, 196)
        self.assertEqual(record.alignments[9].title, ">gi|2734624|gb|G36957|G36957 SHGC-56642 Human Homo sapiens STS cDNA")
        self.assertEqual(record.alignments[9].length, 466)
        self.assertEqual(record.alignments[10].title, ">gi|4631600|dbj|AU046965.1|AU046965 Rattus norvegicus, OTSUKA clone, 108a02, microsatellite sequence, sequence tagged site")
        self.assertEqual(record.alignments[10].length, 330)
        self.assertEqual(record.alignments[11].title, ">gi|4631518|dbj|AU046883.1|AU046883 Rattus norvegicus, OTSUKA clone, 085f03, microsatellite sequence, sequence tagged site")
        self.assertEqual(record.alignments[11].length, 351)
        self.assertEqual(record.alignments[12].title, ">gi|2226478|gb|G33174|G33174 human STS SHGC-6097 clone pG-2470")
        self.assertEqual(record.alignments[12].length, 299)
        self.assertEqual(record.alignments[13].title, ">gi|938611|gb|G08061|G08061 human STS CHLC.GGAA7E02.P7438 clone GGAA7E02")
        self.assertEqual(record.alignments[13].length, 338)
        self.assertEqual(record.alignments[14].title, ">gi|307789|gb|L18105|HUMUT1736 Human STS UT1736.")
        self.assertEqual(record.alignments[14].length, 355)
        self.assertEqual(record.alignments[15].title, ">gi|859804|gb|G06559|G06559 human STS WI-7401.")
        self.assertEqual(record.alignments[15].length, 3280)
        self.assertEqual(record.alignments[16].title, ">gi|4493143|gb|G46852.1|G46852 Z14841_1 Zebrafish AB Danio rerio STS genomic clone Z14841 5', sequence tagged site")
        self.assertEqual(record.alignments[16].length, 291)
        self.assertEqual(record.alignments[17].title, ">gi|4492122|gb|G45831.1|G45831 Z4588_1 Zebrafish AB Danio rerio STS genomic clone Z4588 5', sequence tagged site")
        self.assertEqual(record.alignments[17].length, 398)
        self.assertEqual(record.alignments[18].title, ">gi|6121804|gb|G56635.1|G56635 SHGC-102032 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[18].length, 541)
        self.assertEqual(record.alignments[0].hsps[0].score, 796)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 367)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 1e-102)
        self.assertEqual(record.alignments[0].hsps[1].score, 759)
        self.assertAlmostEqual(record.alignments[0].hsps[1].bits, 350)
        self.assertAlmostEqual(record.alignments[0].hsps[1].expect, 3e-97)
        self.assertEqual(record.alignments[0].hsps[2].score, 387)
        self.assertAlmostEqual(record.alignments[0].hsps[2].bits, 180)
        self.assertAlmostEqual(record.alignments[0].hsps[2].expect, 9e-91)
        self.assertEqual(record.alignments[0].hsps[3].score, 368)
        self.assertAlmostEqual(record.alignments[0].hsps[3].bits, 171)
        self.assertAlmostEqual(record.alignments[0].hsps[3].expect, 9e-91)
        self.assertEqual(record.alignments[0].hsps[4].score, 864)
        self.assertAlmostEqual(record.alignments[0].hsps[4].bits, 398)
        self.assertAlmostEqual(record.alignments[0].hsps[4].expect, 1e-111)
        self.assertEqual(record.alignments[0].hsps[5].score, 846)
        self.assertAlmostEqual(record.alignments[0].hsps[5].bits, 390)
        self.assertAlmostEqual(record.alignments[0].hsps[5].expect, 1e-109)
        self.assertEqual(record.alignments[0].hsps[6].score, 684)
        self.assertAlmostEqual(record.alignments[0].hsps[6].bits, 316)
        self.assertAlmostEqual(record.alignments[0].hsps[6].expect, 7e-87)
        self.assertEqual(len(record.alignments[0].hsps), 7)
        self.assertEqual(record.alignments[1].hsps[0].score, 366)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 170)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 3e-63)
        self.assertEqual(record.alignments[1].hsps[1].score, 188)
        self.assertAlmostEqual(record.alignments[1].hsps[1].bits, 89.0)
        self.assertAlmostEqual(record.alignments[1].hsps[1].expect, 3e-63)
        self.assertEqual(record.alignments[1].hsps[2].score, 590)
        self.assertAlmostEqual(record.alignments[1].hsps[2].bits, 273)
        self.assertAlmostEqual(record.alignments[1].hsps[2].expect, 7e-74)
        self.assertEqual(record.alignments[1].hsps[3].score, 593)
        self.assertAlmostEqual(record.alignments[1].hsps[3].bits, 274)
        self.assertAlmostEqual(record.alignments[1].hsps[3].expect, 8e-76)
        self.assertEqual(record.alignments[1].hsps[4].score, 53)
        self.assertAlmostEqual(record.alignments[1].hsps[4].bits, 27.2)
        self.assertAlmostEqual(record.alignments[1].hsps[4].expect, 8e-76)
        self.assertEqual(record.alignments[1].hsps[5].score, 653)
        self.assertAlmostEqual(record.alignments[1].hsps[5].bits, 302)
        self.assertAlmostEqual(record.alignments[1].hsps[5].expect, 1e-82)
        self.assertEqual(record.alignments[1].hsps[6].score, 598)
        self.assertAlmostEqual(record.alignments[1].hsps[6].bits, 276)
        self.assertAlmostEqual(record.alignments[1].hsps[6].expect, 5e-75)
        self.assertEqual(record.alignments[1].hsps[7].score, 628)
        self.assertAlmostEqual(record.alignments[1].hsps[7].bits, 290)
        self.assertAlmostEqual(record.alignments[1].hsps[7].expect, 4e-79)
        self.assertEqual(len(record.alignments[1].hsps), 8)
        self.assertEqual(record.alignments[2].hsps[0].score, 61)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 30.8)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 0.60)
        self.assertEqual(len(record.alignments[2].hsps), 1)
        self.assertEqual(record.alignments[3].hsps[0].score, 60)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 30.4)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 0.83)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[4].hsps[0].score, 59)
        self.assertAlmostEqual(record.alignments[4].hsps[0].bits, 29.9)
        self.assertAlmostEqual(record.alignments[4].hsps[0].expect, 1.1)
        self.assertEqual(len(record.alignments[4].hsps), 1)
        self.assertEqual(record.alignments[5].hsps[0].score, 58)
        self.assertAlmostEqual(record.alignments[5].hsps[0].bits, 29.5)
        self.assertAlmostEqual(record.alignments[5].hsps[0].expect, 1.6)
        self.assertEqual(len(record.alignments[5].hsps), 1)
        self.assertEqual(record.alignments[6].hsps[0].score, 56)
        self.assertAlmostEqual(record.alignments[6].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.alignments[6].hsps[0].expect, 3.0)
        self.assertEqual(len(record.alignments[6].hsps), 1)
        self.assertEqual(record.alignments[7].hsps[0].score, 56)
        self.assertAlmostEqual(record.alignments[7].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.alignments[7].hsps[0].expect, 3.0)
        self.assertEqual(len(record.alignments[7].hsps), 1)
        self.assertEqual(record.alignments[8].hsps[0].score, 55)
        self.assertAlmostEqual(record.alignments[8].hsps[0].bits, 28.1)
        self.assertAlmostEqual(record.alignments[8].hsps[0].expect, 4.1)
        self.assertEqual(len(record.alignments[8].hsps), 1)
        self.assertEqual(record.alignments[9].hsps[0].score, 55)
        self.assertAlmostEqual(record.alignments[9].hsps[0].bits, 28.1)
        self.assertAlmostEqual(record.alignments[9].hsps[0].expect, 4.1)
        self.assertEqual(len(record.alignments[9].hsps), 1)
        self.assertEqual(record.alignments[10].hsps[0].score, 55)
        self.assertAlmostEqual(record.alignments[10].hsps[0].bits, 28.1)
        self.assertAlmostEqual(record.alignments[10].hsps[0].expect, 4.1)
        self.assertEqual(len(record.alignments[10].hsps), 1)
        self.assertEqual(record.alignments[11].hsps[0].score, 55)
        self.assertAlmostEqual(record.alignments[11].hsps[0].bits, 28.1)
        self.assertAlmostEqual(record.alignments[11].hsps[0].expect, 4.1)
        self.assertEqual(len(record.alignments[11].hsps), 1)
        self.assertEqual(record.alignments[12].hsps[0].score, 55)
        self.assertAlmostEqual(record.alignments[12].hsps[0].bits, 28.1)
        self.assertAlmostEqual(record.alignments[12].hsps[0].expect, 4.1)
        self.assertEqual(len(record.alignments[12].hsps), 1)
        self.assertEqual(record.alignments[13].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[13].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[13].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[13].hsps), 1)
        self.assertEqual(record.alignments[14].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[14].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[14].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[14].hsps), 1)
        self.assertEqual(record.alignments[15].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[15].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[15].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[15].hsps), 1)
        self.assertEqual(record.alignments[16].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[16].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[16].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[16].hsps), 1)
        self.assertEqual(record.alignments[17].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[17].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[17].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[17].hsps), 1)
        self.assertEqual(record.alignments[18].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[18].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[18].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[18].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (192, 200))
        self.assertEqual(record.alignments[0].hsps[0].positives, (192, 200))
        self.assertEqual(record.alignments[0].hsps[1].identities, (195, 205))
        self.assertEqual(record.alignments[0].hsps[1].positives, (195, 205))
        self.assertEqual(record.alignments[0].hsps[2].identities, (74, 74))
        self.assertEqual(record.alignments[0].hsps[2].positives, (74, 74))
        self.assertEqual(record.alignments[0].hsps[3].identities, (114, 114))
        self.assertEqual(record.alignments[0].hsps[3].positives, (114, 114))
        self.assertEqual(record.alignments[0].hsps[4].identities, (205, 205))
        self.assertEqual(record.alignments[0].hsps[4].positives, (205, 205))
        self.assertEqual(record.alignments[0].hsps[5].identities, (196, 196))
        self.assertEqual(record.alignments[0].hsps[5].positives, (196, 196))
        self.assertEqual(record.alignments[0].hsps[6].identities, (146, 146))
        self.assertEqual(record.alignments[0].hsps[6].positives, (146, 146))
        self.assertEqual(record.alignments[1].hsps[0].identities, (71, 74))
        self.assertEqual(record.alignments[1].hsps[0].positives, (71, 74))
        self.assertEqual(record.alignments[1].hsps[1].identities, (42, 67))
        self.assertEqual(record.alignments[1].hsps[1].positives, (43, 67))
        self.assertEqual(record.alignments[1].hsps[2].identities, (121, 133))
        self.assertEqual(record.alignments[1].hsps[2].positives, (121, 133))
        self.assertEqual(record.alignments[1].hsps[3].identities, (112, 131))
        self.assertEqual(record.alignments[1].hsps[3].positives, (112, 131))
        self.assertEqual(record.alignments[1].hsps[4].identities, (9, 13))
        self.assertEqual(record.alignments[1].hsps[4].positives, (10, 13))
        self.assertEqual(record.alignments[1].hsps[5].identities, (128, 157))
        self.assertEqual(record.alignments[1].hsps[5].positives, (132, 157))
        self.assertEqual(record.alignments[1].hsps[6].identities, (122, 130))
        self.assertEqual(record.alignments[1].hsps[6].positives, (122, 130))
        self.assertEqual(record.alignments[1].hsps[7].identities, (119, 131))
        self.assertEqual(record.alignments[1].hsps[7].positives, (120, 131))
        self.assertEqual(record.alignments[2].hsps[0].identities, (11, 27))
        self.assertEqual(record.alignments[2].hsps[0].positives, (18, 27))
        self.assertEqual(record.alignments[3].hsps[0].identities, (10, 19))
        self.assertEqual(record.alignments[3].hsps[0].positives, (13, 19))
        self.assertEqual(record.alignments[4].hsps[0].identities, (10, 24))
        self.assertEqual(record.alignments[4].hsps[0].positives, (14, 24))
        self.assertEqual(record.alignments[5].hsps[0].identities, (15, 34))
        self.assertEqual(record.alignments[5].hsps[0].positives, (17, 34))
        self.assertEqual(record.alignments[6].hsps[0].identities, (9, 28))
        self.assertEqual(record.alignments[6].hsps[0].positives, (16, 28))
        self.assertEqual(record.alignments[7].hsps[0].identities, (10, 24))
        self.assertEqual(record.alignments[7].hsps[0].positives, (15, 24))
        self.assertEqual(record.alignments[8].hsps[0].identities, (9, 33))
        self.assertEqual(record.alignments[8].hsps[0].positives, (19, 33))
        self.assertEqual(record.alignments[9].hsps[0].identities, (10, 15))
        self.assertEqual(record.alignments[9].hsps[0].positives, (11, 15))
        self.assertEqual(record.alignments[10].hsps[0].identities, (12, 39))
        self.assertEqual(record.alignments[10].hsps[0].positives, (19, 39))
        self.assertEqual(record.alignments[11].hsps[0].identities, (8, 26))
        self.assertEqual(record.alignments[11].hsps[0].positives, (15, 26))
        self.assertEqual(record.alignments[12].hsps[0].identities, (13, 38))
        self.assertEqual(record.alignments[12].hsps[0].positives, (19, 38))
        self.assertEqual(record.alignments[13].hsps[0].identities, (12, 24))
        self.assertEqual(record.alignments[13].hsps[0].positives, (13, 24))
        self.assertEqual(record.alignments[14].hsps[0].identities, (12, 24))
        self.assertEqual(record.alignments[14].hsps[0].positives, (13, 24))
        self.assertEqual(record.alignments[15].hsps[0].identities, (9, 21))
        self.assertEqual(record.alignments[15].hsps[0].positives, (13, 21))
        self.assertEqual(record.alignments[16].hsps[0].identities, (8, 20))
        self.assertEqual(record.alignments[16].hsps[0].positives, (13, 20))
        self.assertEqual(record.alignments[17].hsps[0].identities, (7, 22))
        self.assertEqual(record.alignments[17].hsps[0].positives, (13, 22))
        self.assertEqual(record.alignments[18].hsps[0].identities, (9, 20))
        self.assertEqual(record.alignments[18].hsps[0].positives, (13, 20))
        self.assertEqual(record.alignments[0].hsps[0].frame, ("+2", "+2"))
        self.assertEqual(record.alignments[0].hsps[1].frame, ("+1", "+1"))
        self.assertEqual(record.alignments[0].hsps[2].frame, ("+3", "+3"))
        self.assertEqual(record.alignments[0].hsps[3].frame, ("+3", "+3"))
        self.assertEqual(record.alignments[0].hsps[4].frame, ("-1", "-1"))
        self.assertEqual(record.alignments[0].hsps[5].frame, ("-3", "-3"))
        self.assertEqual(record.alignments[0].hsps[6].frame, ("-2", "-2"))
        self.assertEqual(record.alignments[1].hsps[0].frame, ("+3", "+3"))
        self.assertEqual(record.alignments[1].hsps[1].frame, ("+3", "+3"))
        self.assertEqual(record.alignments[1].hsps[2].frame, ("+1", "+1"))
        self.assertEqual(record.alignments[1].hsps[3].frame, ("+2", "+2"))
        self.assertEqual(record.alignments[1].hsps[4].frame, ("+2", "+2"))
        self.assertEqual(record.alignments[1].hsps[5].frame, ("-3", "-2"))
        self.assertEqual(record.alignments[1].hsps[6].frame, ("-2", "-1"))
        self.assertEqual(record.alignments[1].hsps[7].frame, ("-1", "-3"))
        self.assertEqual(record.alignments[2].hsps[0].frame, ("+1", "+2"))
        self.assertEqual(record.alignments[3].hsps[0].frame, ("-3", "+1"))
        self.assertEqual(record.alignments[4].hsps[0].frame, ("-3", "-2"))
        self.assertEqual(record.alignments[5].hsps[0].frame, ("-3", "-2"))
        self.assertEqual(record.alignments[6].hsps[0].frame, ("+3", "+1"))
        self.assertEqual(record.alignments[7].hsps[0].frame, ("+1", "-2"))
        self.assertEqual(record.alignments[8].hsps[0].frame, ("+3", "-2"))
        self.assertEqual(record.alignments[9].hsps[0].frame, ("-3", "-3"))
        self.assertEqual(record.alignments[10].hsps[0].frame, ("+2", "-3"))
        self.assertEqual(record.alignments[11].hsps[0].frame, ("+1", "+3"))
        self.assertEqual(record.alignments[12].hsps[0].frame, ("+1", "+2"))
        self.assertEqual(record.alignments[13].hsps[0].frame, ("-3", "+3"))
        self.assertEqual(record.alignments[14].hsps[0].frame, ("-3", "-1"))
        self.assertEqual(record.alignments[15].hsps[0].frame, ("-1", "-1"))
        self.assertEqual(record.alignments[16].hsps[0].frame, ("-3", "+1"))
        self.assertEqual(record.alignments[17].hsps[0].frame, ("+1", "+1"))
        self.assertEqual(record.alignments[18].hsps[0].frame, ("-1", "+2"))
        self.assertEqual(record.alignments[0].hsps[0].query, "IRMPLHS*DSSFCPL*QEKWECMXXXXXXXXRPKRCLQPHPLNWP*LGLNALMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFSPLQPLFWWTNNLGTPNPGRKNIQHYEXALX*S*NGFPKLVTHGPGXVXLXLXGLSSFQEFXSTVANPWGX*XXXXXXXXXFXXGXRXXXLXXXXGGCXXVXXVXXXWXXXF")
        self.assertEqual(record.alignments[0].hsps[0].match, "IRMPLHS*DSSFCPL*QEKWECM        RPKRCLQPHPLNWP*LGLNALMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFSPLQPLFWWTNNLGTPNPGRKNIQHYE AL *S*NGFPKLVTHGPG V L L GLSSFQEF STVANPWG *         F  G R   L    GGC  V  V   W   F")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "IRMPLHS*DSSFCPL*QEKWECMQSSQKKQKRPKRCLQPHPLNWP*LGLNALMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFSPLQPLFWWTNNLGTPNPGRKNIQHYEXALX*S*NGFPKLVTHGPGXVXLXLXGLSSFQEFXSTVANPWGX*XXXXXXXXXFXXGXRXXXLXXXXGGCXXVXXVXXXWXXXF")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 2)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 601)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 2)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 601)
        self.assertEqual(record.alignments[0].hsps[1].query, "DQNAPPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTKCTNAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL*XSTXVILKWISKAGYTWPWXXXIXFXXAFFXXXXXXXXXXXXXXLXGXVX*XXXXSXGPXXXXVXXPXGWVXXGXFXFXXXXXXFXXXLG")
        self.assertEqual(record.alignments[0].hsps[1].match, "DQNAPPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTKCTNAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL* ST VILKWISKAGYTWPW   I F  AFF              L G V *    S GP    V  P GWV  G F F      F   LG")
        self.assertEqual(record.alignments[0].hsps[1].sbjct, "DQNAPPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTKCTNAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL*XSTXVILKWISKAGYTWPWXXXIXFXXAFFXSGVXVNGGKSXGXLXGXVX*XXXXSXGPXXXXVXXPXGWVXXGXFXFXXXXXXFXXXLG")
        self.assertEqual(record.alignments[0].hsps[1].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[1].query_end, 615)
        self.assertEqual(record.alignments[0].hsps[1].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[1].sbjct_end, 615)
        self.assertEqual(record.alignments[0].hsps[2].query, "SECPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*MH*CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[0].hsps[2].match, "SECPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*MH*CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[0].hsps[2].sbjct, "SECPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*MH*CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[0].hsps[2].query_start, 3)
        self.assertEqual(record.alignments[0].hsps[2].query_end, 224)
        self.assertEqual(record.alignments[0].hsps[2].sbjct_start, 3)
        self.assertEqual(record.alignments[0].hsps[2].sbjct_end, 224)
        self.assertEqual(record.alignments[0].hsps[3].query, "YSHCSGGQIT*VLQILAGKIYSIMXQHSXNPKMDFQSWLHMALXXSY*XXXGFLXFRSXGQRWQIXGVXXWXGXLXXXXFXGAXGGXGXXSXXVGAXGXXXFXXXGXXXXXXXW")
        self.assertEqual(record.alignments[0].hsps[3].match, "YSHCSGGQIT*VLQILAGKIYSIM QHS NPKMDFQSWLHMAL  SY*   GFL FRS GQRWQI GV  W G L    F GA GG G  S  VGA G   F   G       W")
        self.assertEqual(record.alignments[0].hsps[3].sbjct, "YSHCSGGQIT*VLQILAGKIYSIMXQHSXNPKMDFQSWLHMALXXSY*XXXGFLXFRSXGQRWQIXGVXXWXGXLXXXXFXGAXGGXGXXSXXVGAXGXXXFXXXGXXXXXXXW")
        self.assertEqual(record.alignments[0].hsps[3].query_start, 273)
        self.assertEqual(record.alignments[0].hsps[3].query_end, 614)
        self.assertEqual(record.alignments[0].hsps[3].sbjct_start, 273)
        self.assertEqual(record.alignments[0].hsps[3].sbjct_end, 614)
        self.assertEqual(record.alignments[0].hsps[4].query, "PKXXXKXXXXXXKTKXTXXHPPXWXXNXXXLWPXXXXXXLXNXXX*XPXGFATVDXNS*XEESPXKXNXTXPGPCVTSFGNPF*DYXSAXS*CCIFFLPGFGVPRLFVHQNNGCKGEN*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCISAFSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRGILI")
        self.assertEqual(record.alignments[0].hsps[4].match, "PK   K      KTK T  HPP W  N   LWP      L N   * P GFATVD NS* EESP K N T PGPCVTSFGNPF*DY SA S*CCIFFLPGFGVPRLFVHQNNGCKGEN*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCISAFSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRGILI")
        self.assertEqual(record.alignments[0].hsps[4].sbjct, "PKXXXKXXXXXXKTKXTXXHPPXWXXNXXXLWPXXXXXXLXNXXX*XPXGFATVDXNS*XEESPXKXNXTXPGPCVTSFGNPF*DYXSAXS*CCIFFLPGFGVPRLFVHQNNGCKGEN*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCISAFSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRGILI")
        self.assertEqual(record.alignments[0].hsps[4].query_start, 615)
        self.assertEqual(record.alignments[0].hsps[4].query_end, 1)
        self.assertEqual(record.alignments[0].hsps[4].sbjct_start, 615)
        self.assertEqual(record.alignments[0].hsps[4].sbjct_end, 1)
        self.assertEqual(record.alignments[0].hsps[5].query, "PXXXNXXNPXAPTXXXX*PXPPXAPXKXXXXXXPXXLXTPXICHR*PKLLKXRKPX*XQXDXXRAMCNQLWKSILGLXECXFIMLYIFPARIWST*VICPPEQWL*RRKLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*CI*SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGHSD")
        self.assertEqual(record.alignments[0].hsps[5].match, "P   N  NP APT    *P PP AP K      P  L TP ICHR*PKLLK RKP * Q D  RAMCNQLWKSILGL EC FIMLYIFPARIWST*VICPPEQWL*RRKLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*CI*SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGHSD")
        self.assertEqual(record.alignments[0].hsps[5].sbjct, "PXXXNXXNPXAPTXXXX*PXPPXAPXKXXXXXXPXXLXTPXICHR*PKLLKXRKPX*XQXDXXRAMCNQLWKSILGLXECXFIMLYIFPARIWST*VICPPEQWL*RRKLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*CI*SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGHSD")
        self.assertEqual(record.alignments[0].hsps[5].query_start, 589)
        self.assertEqual(record.alignments[0].hsps[5].query_end, 2)
        self.assertEqual(record.alignments[0].hsps[5].sbjct_start, 589)
        self.assertEqual(record.alignments[0].hsps[5].sbjct_end, 2)
        self.assertEqual(record.alignments[0].hsps[6].query, "EXKKAXXXSIXXXQGHV*PALEIHFRIT*VLXHNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFALVHLVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGAF*")
        self.assertEqual(record.alignments[0].hsps[6].match, "E KKA   SI   QGHV*PALEIHFRIT*VL HNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFALVHLVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGAF*")
        self.assertEqual(record.alignments[0].hsps[6].sbjct, "EXKKAXXXSIXXXQGHV*PALEIHFRIT*VLXHNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFALVHLVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGAF*")
        self.assertEqual(record.alignments[0].hsps[6].query_start, 440)
        self.assertEqual(record.alignments[0].hsps[6].query_end, 3)
        self.assertEqual(record.alignments[0].hsps[6].sbjct_start, 440)
        self.assertEqual(record.alignments[0].hsps[6].sbjct_end, 3)
        self.assertEqual(record.alignments[1].hsps[0].query, "SECPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*MH*CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[1].hsps[0].match, "S CPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*  *CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "SXCPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*XX*CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 3)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 224)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 3)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 224)
        self.assertEqual(record.alignments[1].hsps[1].query, "YSHCSGGQIT*VLQILAGKIYSIMXQHSXNPKMDFQSWLHMALXXSY*XXXGFLXFRSXGQRWQIXG")
        self.assertEqual(record.alignments[1].hsps[1].match, "YSHCSGGQIT*VLQILAGKIYSIM QHS   K  FQSWLHM     +     F   R  GQR Q  G")
        self.assertEqual(record.alignments[1].hsps[1].sbjct, "YSHCSGGQIT*VLQILAGKIYSIMKQHSVILKWIFQSWLHMXCKVLFKFKRPFSFTRGLGQRXQTPG")
        self.assertEqual(record.alignments[1].hsps[1].query_start, 273)
        self.assertEqual(record.alignments[1].hsps[1].query_end, 473)
        self.assertEqual(record.alignments[1].hsps[1].sbjct_start, 273)
        self.assertEqual(record.alignments[1].hsps[1].sbjct_end, 473)
        self.assertEqual(record.alignments[1].hsps[2].query, "DQNAPPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTKCTNAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL*XSTXVILKWISKAGYTWP")
        self.assertEqual(record.alignments[1].hsps[2].match, "DQ APPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTK  NAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL* ST       SKAGYT P")
        self.assertEqual(record.alignments[1].hsps[2].sbjct, "DQXAPPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTKXXNAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL*NSTQ*S*NGFSKAGYTXP")
        self.assertEqual(record.alignments[1].hsps[2].query_start, 1)
        self.assertEqual(record.alignments[1].hsps[2].query_end, 399)
        self.assertEqual(record.alignments[1].hsps[2].sbjct_start, 1)
        self.assertEqual(record.alignments[1].hsps[2].sbjct_end, 399)
        self.assertEqual(record.alignments[1].hsps[3].query, "IRMPLHS*DSSFCPL*QEKWECMXXXXXXXXRPKRCLQPHPLNWP*LGLNALMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFSPLQPLFWWTNNLGTPNPGRKNIQHYEXALX*S*NGFPKLVTH")
        self.assertEqual(record.alignments[1].hsps[3].match, "IR PLHS*DSSFCPL*QEKWECM        RPKRCLQPHPLNWP*LGL  LMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQF PLQPLFWWTNNLGTPNPGRKNIQHYE AL      FPKLVTH")
        self.assertEqual(record.alignments[1].hsps[3].sbjct, "IRXPLHS*DSSFCPL*QEKWECMQSSQKKQKRPKRCLQPHPLNWP*LGLXPLMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFFPLQPLFWWTNNLGTPNPGRKNIQHYETALSNPKMDFPKLVTH")
        self.assertEqual(record.alignments[1].hsps[3].query_start, 2)
        self.assertEqual(record.alignments[1].hsps[3].query_end, 394)
        self.assertEqual(record.alignments[1].hsps[3].sbjct_start, 2)
        self.assertEqual(record.alignments[1].hsps[3].sbjct_end, 394)
        self.assertEqual(record.alignments[1].hsps[4].query, "FQEFXSTVANPWG")
        self.assertEqual(record.alignments[1].hsps[4].match, "+Q F ST ANPWG")
        self.assertEqual(record.alignments[1].hsps[4].sbjct, "YQGFRSTXANPWG")
        self.assertEqual(record.alignments[1].hsps[4].query_start, 437)
        self.assertEqual(record.alignments[1].hsps[4].query_end, 475)
        self.assertEqual(record.alignments[1].hsps[4].sbjct_start, 437)
        self.assertEqual(record.alignments[1].hsps[4].sbjct_end, 475)
        self.assertEqual(record.alignments[1].hsps[5].query, "PXICHR*PKLLKXRKPX*XQXDXXRAMCNQLWKSILGLXECXFIMLYIFPARIWST*VICPPEQWL*RRKLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*CI*SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGHSD")
        self.assertEqual(record.alignments[1].hsps[5].match, "P +C R*PK L   K         + MCNQLWK    + EC FIMLYIFPARIWST*VICPPEQWL*R+KLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*  *SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEG SD")
        self.assertEqual(record.alignments[1].hsps[5].sbjct, "PGVCXR*PKPLVNEKGLLNLNKTLQXMCNQLWKIHFRITECCFIMLYIFPARIWST*VICPPEQWL*RKKLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*XX*SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGXSD")
        self.assertEqual(record.alignments[1].hsps[5].query_start, 472)
        self.assertEqual(record.alignments[1].hsps[5].query_end, 2)
        self.assertEqual(record.alignments[1].hsps[5].sbjct_start, 472)
        self.assertEqual(record.alignments[1].hsps[5].sbjct_end, 2)
        self.assertEqual(record.alignments[1].hsps[6].query, "GHV*PALEIHFRIT*VLXHNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFALVHLVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGA")
        self.assertEqual(record.alignments[1].hsps[6].match, "GHV*PALE  F   *VL HNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFAL  LVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGA")
        self.assertEqual(record.alignments[1].hsps[6].sbjct, "GHV*PALENPF*DY*VLFHNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFALXXLVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGA")
        self.assertEqual(record.alignments[1].hsps[6].query_start, 398)
        self.assertEqual(record.alignments[1].hsps[6].query_end, 9)
        self.assertEqual(record.alignments[1].hsps[6].sbjct_start, 398)
        self.assertEqual(record.alignments[1].hsps[6].sbjct_end, 9)
        self.assertEqual(record.alignments[1].hsps[7].query, "CVTSFGNPF*DYXSAXS*CCIFFLPGFGVPRLFVHQNNGCKGEN*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCISAFSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRGILI")
        self.assertEqual(record.alignments[1].hsps[7].match, "CVTSFG       SA S*CCIFFLPGFGVPRLFVHQNNGCKG+N*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCI  FSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRG LI")
        self.assertEqual(record.alignments[1].hsps[7].sbjct, "CVTSFGKSILGLLSAVS*CCIFFLPGFGVPRLFVHQNNGCKGKN*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCIXGFSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRGXLI")
        self.assertEqual(record.alignments[1].hsps[7].query_start, 393)
        self.assertEqual(record.alignments[1].hsps[7].query_end, 1)
        self.assertEqual(record.alignments[1].hsps[7].sbjct_start, 393)
        self.assertEqual(record.alignments[1].hsps[7].sbjct_end, 1)
        self.assertEqual(record.alignments[2].hsps[0].query, "LCNLFYRIFSAASDHCSIFSFTAIVLV")
        self.assertEqual(record.alignments[2].hsps[0].match, "L NL +R+F++  DHCS+  F   V++")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "LTNLIFRLFTSK*DHCSLKQFNNKVVI")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 211)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 291)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 2)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 82)
        self.assertEqual(record.alignments[3].hsps[0].query, "IWST*VICPPEQWL*RRKL")
        self.assertEqual(record.alignments[3].hsps[0].match, "+W T  +CPP  W *RR+L")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "LWGTAPLCPPVPWA*RRQL")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 316)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 260)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 226)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 282)
        self.assertEqual(record.alignments[4].hsps[0].query, "PFWPFLFLLRALHTFPLFLSKWTK")
        self.assertEqual(record.alignments[4].hsps[0].match, "P WP + LL  +H+FP  +  W K")
        self.assertEqual(record.alignments[4].hsps[0].sbjct, "PEWPSINLLPGMHSFPRVIPCWEK")
        self.assertEqual(record.alignments[4].hsps[0].query_start, 106)
        self.assertEqual(record.alignments[4].hsps[0].query_end, 35)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_start, 716)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_end, 645)
        self.assertEqual(record.alignments[5].hsps[0].query, "PFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGHS")
        self.assertEqual(record.alignments[5].hsps[0].match, "P W   FLLR LHT P    K+    V+  EG S")
        self.assertEqual(record.alignments[5].hsps[0].sbjct, "PGWTLRFLLRGLHTPPKAAEKFPSQGVAEPEGFS")
        self.assertEqual(record.alignments[5].hsps[0].query_start, 106)
        self.assertEqual(record.alignments[5].hsps[0].query_end, 5)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_start, 176)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_end, 75)
        self.assertEqual(record.alignments[6].hsps[0].query, "STHETLHFVHFDRKSGNVCRALKRNKKG")
        self.assertEqual(record.alignments[6].hsps[0].match, "+TH    F+H  +K  N+C  +  N++G")
        self.assertEqual(record.alignments[6].hsps[0].sbjct, "TTHSMTVFLHVKKKLSNICIYMPENREG")
        self.assertEqual(record.alignments[6].hsps[0].query_start, 15)
        self.assertEqual(record.alignments[6].hsps[0].query_end, 98)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_start, 82)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_end, 165)
        self.assertEqual(record.alignments[7].hsps[0].query, "HCFLCNLFYRIFSAASDHCSIFSF")
        self.assertEqual(record.alignments[7].hsps[0].match, "HC L ++F RIF    +  S+F+F")
        self.assertEqual(record.alignments[7].hsps[0].sbjct, "HCHLSSVFCRIFFTLEESLSLFAF")
        self.assertEqual(record.alignments[7].hsps[0].query_start, 202)
        self.assertEqual(record.alignments[7].hsps[0].query_end, 273)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_start, 443)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_end, 372)
        self.assertEqual(record.alignments[8].hsps[0].query, "ETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHI")
        self.assertEqual(record.alignments[8].hsps[0].match, "E +H+V F +  G +  + ++ ++ QN   SH+")
        self.assertEqual(record.alignments[8].hsps[0].sbjct, "EKIHYVLF*KNCGKIMTS*RQRQENQNNLLSHV")
        self.assertEqual(record.alignments[8].hsps[0].query_start, 24)
        self.assertEqual(record.alignments[8].hsps[0].query_end, 122)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_start, 141)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_end, 43)
        self.assertEqual(record.alignments[9].hsps[0].query, "LKAPFWPFLFLLRAL")
        self.assertEqual(record.alignments[9].hsps[0].match, "L  PFW FLF L+AL")
        self.assertEqual(record.alignments[9].hsps[0].sbjct, "LSVPFWKFLFYLQAL")
        self.assertEqual(record.alignments[9].hsps[0].query_start, 115)
        self.assertEqual(record.alignments[9].hsps[0].query_end, 71)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_start, 242)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_end, 198)
        self.assertEqual(record.alignments[10].hsps[0].query, "PLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFSPLQPL")
        self.assertEqual(record.alignments[10].hsps[0].match, "PL    RLF + +  ++ C T     L  +  F P++PL")
        self.assertEqual(record.alignments[10].hsps[0].sbjct, "PLKTQGRLFYSKVSLFLKCRTTLCLFLNVSEGFXPIEPL")
        self.assertEqual(record.alignments[10].hsps[0].query_start, 167)
        self.assertEqual(record.alignments[10].hsps[0].query_end, 283)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_start, 211)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_end, 95)
        self.assertEqual(record.alignments[11].hsps[0].query, "CFLCNLFYRIFSAASDHCSIFSFTAI")
        self.assertEqual(record.alignments[11].hsps[0].match, "CF+C L  +I+  A +   +  FT++")
        self.assertEqual(record.alignments[11].hsps[0].sbjct, "CFICKLVLKIYLRAEERTQLIEFTSL")
        self.assertEqual(record.alignments[11].hsps[0].query_start, 205)
        self.assertEqual(record.alignments[11].hsps[0].query_end, 282)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_start, 249)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_end, 326)
        self.assertEqual(record.alignments[12].hsps[0].query, "KSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVL")
        self.assertEqual(record.alignments[12].hsps[0].match, "+SA +I   HC    +   +FS ASD  S    T ++L")
        self.assertEqual(record.alignments[12].hsps[0].sbjct, "QSAWIIGMSHCAWAVIHCLLFSTASDEKSAVDLTGVLL")
        self.assertEqual(record.alignments[12].hsps[0].query_start, 175)
        self.assertEqual(record.alignments[12].hsps[0].query_end, 288)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_start, 119)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_end, 232)
        self.assertEqual(record.alignments[13].hsps[0].query, "WLKAPFWPFLFLLRALHTFPLFLS")
        self.assertEqual(record.alignments[13].hsps[0].match, "WL  PF PFL  L +L   P  LS")
        self.assertEqual(record.alignments[13].hsps[0].sbjct, "WLLFPFLPFLPFLPSLPFLPFLLS")
        self.assertEqual(record.alignments[13].hsps[0].query_start, 118)
        self.assertEqual(record.alignments[13].hsps[0].query_end, 47)
        self.assertEqual(record.alignments[13].hsps[0].sbjct_start, 153)
        self.assertEqual(record.alignments[13].hsps[0].sbjct_end, 224)
        self.assertEqual(record.alignments[14].hsps[0].query, "WLKAPFWPFLFLLRALHTFPLFLS")
        self.assertEqual(record.alignments[14].hsps[0].match, "WL  PF PFL  L +L   P  LS")
        self.assertEqual(record.alignments[14].hsps[0].sbjct, "WLLFPFLPFLPFLPSLPFLPFLLS")
        self.assertEqual(record.alignments[14].hsps[0].query_start, 118)
        self.assertEqual(record.alignments[14].hsps[0].query_end, 47)
        self.assertEqual(record.alignments[14].hsps[0].sbjct_start, 235)
        self.assertEqual(record.alignments[14].hsps[0].sbjct_end, 164)
        self.assertEqual(record.alignments[15].hsps[0].query, "CCIFFLPGFGVPRLFVHQNNG")
        self.assertEqual(record.alignments[15].hsps[0].match, "C IF++P F    LF+H+  G")
        self.assertEqual(record.alignments[15].hsps[0].sbjct, "CFIFYVPDFPWSNLFLHRGRG")
        self.assertEqual(record.alignments[15].hsps[0].query_start, 339)
        self.assertEqual(record.alignments[15].hsps[0].query_end, 277)
        self.assertEqual(record.alignments[15].hsps[0].sbjct_start, 1396)
        self.assertEqual(record.alignments[15].hsps[0].sbjct_end, 1334)
        self.assertEqual(record.alignments[16].hsps[0].query, "NQLWKSILGLXECXFIMLYI")
        self.assertEqual(record.alignments[16].hsps[0].match, "NQ+WK  L +  C F+ +Y+")
        self.assertEqual(record.alignments[16].hsps[0].sbjct, "NQMWKHSLEVCMCVFVYIYV")
        self.assertEqual(record.alignments[16].hsps[0].query_start, 388)
        self.assertEqual(record.alignments[16].hsps[0].query_end, 329)
        self.assertEqual(record.alignments[16].hsps[0].sbjct_start, 37)
        self.assertEqual(record.alignments[16].hsps[0].sbjct_end, 96)
        self.assertEqual(record.alignments[17].hsps[0].query, "CFLCNLFYRIFSAASDHCSIFS")
        self.assertEqual(record.alignments[17].hsps[0].match, "CF   + +R+F+    HC+ F+")
        self.assertEqual(record.alignments[17].hsps[0].sbjct, "CFALTVIWRVFAGCRPHCATFT")
        self.assertEqual(record.alignments[17].hsps[0].query_start, 205)
        self.assertEqual(record.alignments[17].hsps[0].query_end, 270)
        self.assertEqual(record.alignments[17].hsps[0].sbjct_start, 13)
        self.assertEqual(record.alignments[17].hsps[0].sbjct_end, 78)
        self.assertEqual(record.alignments[18].hsps[0].query, "VKGFCISAFSPSQGQFRGCG")
        self.assertEqual(record.alignments[18].hsps[0].match, "V GFC+++FSP +    G G")
        self.assertEqual(record.alignments[18].hsps[0].sbjct, "VDGFCVTSFSPKKDTHPGSG")
        self.assertEqual(record.alignments[18].hsps[0].query_start, 174)
        self.assertEqual(record.alignments[18].hsps[0].query_end, 115)
        self.assertEqual(record.alignments[18].hsps[0].sbjct_start, 125)
        self.assertEqual(record.alignments[18].hsps[0].sbjct_end, 184)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(record.posted_date, [('Nov 26, 1999  5:52 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.318)
        self.assertAlmostEqual(record.ka_params[1], 0.135)
        self.assertAlmostEqual(record.ka_params[2], 0.401)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.num_hits, 40473548)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 487631)
        self.assertEqual(record.num_good_extends, 13175)
        self.assertEqual(record.num_seqs_better_e, 38)
        self.assertEqual(record.query_length, 205)
        self.assertEqual(record.database_length, 10666284)
        self.assertEqual(record.effective_hsp_length, 46)
        self.assertEqual(record.effective_query_length, 158)
        self.assertEqual(record.effective_database_length, 6627852)
        self.assertEqual(record.effective_search_space, 1047200616)
        self.assertEqual(record.effective_search_space_used, 1047200616)
        self.assertEqual(record.frameshift, ('50,', '0.1'))
        self.assertEqual(record.threshold, 13)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 0)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 0.0)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 52)
        self.assertAlmostEqual(record.blast_cutoff[1], 26.7)

    def test_text_2010L_blastp_003(self):
        "Test parsing BLASTP 2.0.10 output (text_2010L_blastp_003)"

        path = os.path.join('Blast', 'text_2010L_blastp_003.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|585505|sp|Q08386|MOPB_RHOCA MOLYBDENUM-PTERIN BINDING\nPROTEIN MOPB")
        self.assertEqual(record.query_letters, 270)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 13)
        self.assertEqual(record.descriptions[0].title, "gi|585505|sp|Q08386|MOPB_RHOCA MOLYBDENUM-PTERIN BINDING PROTEI...")
        self.assertEqual(record.descriptions[0].score, 467)
        self.assertAlmostEqual(record.descriptions[0].e, 1e-131)
        self.assertEqual(record.descriptions[1].title, "gi|585504|sp|Q08385|MOPA_RHOCA MOLYBDENUM-PTERIN BINDING PROTEI...")
        self.assertEqual(record.descriptions[1].score, 207)
        self.assertAlmostEqual(record.descriptions[1].e, 2e-53)
        self.assertEqual(record.descriptions[2].title, "gi|585492|sp|P37733|MODA_AZOVI MOLYBDENUM TRANSPORT PROTEIN MODA")
        self.assertEqual(record.descriptions[2].score, 145)
        self.assertAlmostEqual(record.descriptions[2].e, 9e-35)
        self.assertEqual(record.descriptions[3].title, "gi|1709070|sp|P46930|MODE_ECOLI MOLYBDENUM TRANSPORT PROTEIN MODE")
        self.assertEqual(record.descriptions[3].score, 87)
        self.assertAlmostEqual(record.descriptions[3].e, 5e-17)
        self.assertEqual(record.descriptions[4].title, "gi|1709071|sp|P45324|MODE_HAEIN MOLYBDENUM TRANSPORT PROTEIN MO...")
        self.assertEqual(record.descriptions[4].score, 54)
        self.assertAlmostEqual(record.descriptions[4].e, 2e-7)
        self.assertEqual(record.descriptions[5].title, "gi|585502|sp|P04952|MOP1_CLOPA MOLYBDENUM-PTERIN BINDING PROTEIN I")
        self.assertEqual(record.descriptions[5].score, 53)
        self.assertAlmostEqual(record.descriptions[5].e, 6e-7)
        self.assertEqual(record.descriptions[6].title, "gi|127241|sp|P08854|MOP2_CLOPA MOLYBDENUM-PTERIN BINDING PROTEI...")
        self.assertEqual(record.descriptions[6].score, 52)
        self.assertAlmostEqual(record.descriptions[6].e, 0.000001)
        self.assertEqual(record.descriptions[7].title, "gi|585503|sp|P38366|MOP3_CLOPA MOLYBDENUM-PTERIN BINDING PROTEI...")
        self.assertEqual(record.descriptions[7].score, 51)
        self.assertAlmostEqual(record.descriptions[7].e, 0.000003)
        self.assertEqual(record.descriptions[8].title, "gi|1170996|sp|P45183|MOP_HAEIN PROBABLE MOLYBDENUM-PTERIN BINDI...")
        self.assertEqual(record.descriptions[8].score, 46)
        self.assertAlmostEqual(record.descriptions[8].e, 0.00005)
        self.assertEqual(record.descriptions[9].title, "gi|1709069|sp|P09833|MODC_ECOLI MOLYBDENUM TRANSPORT ATP-BINDIN...")
        self.assertEqual(record.descriptions[9].score, 38)
        self.assertAlmostEqual(record.descriptions[9].e, 0.021)
        self.assertEqual(record.descriptions[10].title, "gi|585500|sp|P37732|MODD_AZOVI MOLYBDENUM TRANSPORT ATP-BINDING...")
        self.assertEqual(record.descriptions[10].score, 33)
        self.assertAlmostEqual(record.descriptions[10].e, 0.53)
        self.assertEqual(record.descriptions[11].title, "gi|2507168|sp|P08838|PT1_BACSU PHOSPHOENOLPYRUVATE-PROTEIN PHOS...")
        self.assertEqual(record.descriptions[11].score, 30)
        self.assertAlmostEqual(record.descriptions[11].e, 4.6)
        self.assertEqual(record.descriptions[12].title, "gi|729786|sp|Q05355|HYDL_STRHA PUTATIVE POLYKETIDE HYDROXYLASE")
        self.assertEqual(record.descriptions[12].score, 29)
        self.assertAlmostEqual(record.descriptions[12].e, 7.9)
        self.assertEqual(len(record.alignments), 13)
        self.assertEqual(record.alignments[0].title, ">gi|585505|sp|Q08386|MOPB_RHOCA MOLYBDENUM-PTERIN BINDING PROTEIN MOPB")
        self.assertEqual(record.alignments[0].length, 270)
        self.assertEqual(record.alignments[1].title, ">gi|585504|sp|Q08385|MOPA_RHOCA MOLYBDENUM-PTERIN BINDING PROTEIN MOPA")
        self.assertEqual(record.alignments[1].length, 265)
        self.assertEqual(record.alignments[2].title, ">gi|585492|sp|P37733|MODA_AZOVI MOLYBDENUM TRANSPORT PROTEIN MODA")
        self.assertEqual(record.alignments[2].length, 270)
        self.assertEqual(record.alignments[3].title, ">gi|1709070|sp|P46930|MODE_ECOLI MOLYBDENUM TRANSPORT PROTEIN MODE")
        self.assertEqual(record.alignments[3].length, 262)
        self.assertEqual(record.alignments[4].title, ">gi|1709071|sp|P45324|MODE_HAEIN MOLYBDENUM TRANSPORT PROTEIN MODE HOMOLOG")
        self.assertEqual(record.alignments[4].length, 255)
        self.assertEqual(record.alignments[5].title, ">gi|585502|sp|P04952|MOP1_CLOPA MOLYBDENUM-PTERIN BINDING PROTEIN I")
        self.assertEqual(record.alignments[5].length, 68)
        self.assertEqual(record.alignments[6].title, ">gi|127241|sp|P08854|MOP2_CLOPA MOLYBDENUM-PTERIN BINDING PROTEIN II")
        self.assertEqual(record.alignments[6].length, 68)
        self.assertEqual(record.alignments[7].title, ">gi|585503|sp|P38366|MOP3_CLOPA MOLYBDENUM-PTERIN BINDING PROTEIN III")
        self.assertEqual(record.alignments[7].length, 68)
        self.assertEqual(record.alignments[8].title, ">gi|1170996|sp|P45183|MOP_HAEIN PROBABLE MOLYBDENUM-PTERIN BINDING PROTEIN")
        self.assertEqual(record.alignments[8].length, 69)
        self.assertEqual(record.alignments[9].title, ">gi|1709069|sp|P09833|MODC_ECOLI MOLYBDENUM TRANSPORT ATP-BINDING PROTEIN MODC")
        self.assertEqual(record.alignments[9].length, 352)
        self.assertEqual(record.alignments[10].title, ">gi|585500|sp|P37732|MODD_AZOVI MOLYBDENUM TRANSPORT ATP-BINDING PROTEIN MODD")
        self.assertEqual(record.alignments[10].length, 380)
        self.assertEqual(record.alignments[11].title, ">gi|2507168|sp|P08838|PT1_BACSU PHOSPHOENOLPYRUVATE-PROTEIN PHOSPHOTRANSFERASE (PHOSPHOTRANSFERASE SYSTEM, ENZYME I)")
        self.assertEqual(record.alignments[11].length, 570)
        self.assertEqual(record.alignments[12].title, ">gi|729786|sp|Q05355|HYDL_STRHA PUTATIVE POLYKETIDE HYDROXYLASE")
        self.assertEqual(record.alignments[12].length, 555)
        self.assertEqual(record.alignments[0].hsps[0].score, 1189)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 467)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 1e-131)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 521)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 207)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 2e-53)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 362)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 145)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 9e-35)
        self.assertEqual(record.alignments[2].hsps[1].score, 98)
        self.assertAlmostEqual(record.alignments[2].hsps[1].bits, 42.6)
        self.assertAlmostEqual(record.alignments[2].hsps[1].expect, 8e-4)
        self.assertEqual(len(record.alignments[2].hsps), 2)
        self.assertEqual(record.alignments[3].hsps[0].score, 211)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 86.6)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 5e-17)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[4].hsps[0].score, 128)
        self.assertAlmostEqual(record.alignments[4].hsps[0].bits, 54.3)
        self.assertAlmostEqual(record.alignments[4].hsps[0].expect, 2e-7)
        self.assertEqual(len(record.alignments[4].hsps), 1)
        self.assertEqual(record.alignments[5].hsps[0].score, 125)
        self.assertAlmostEqual(record.alignments[5].hsps[0].bits, 53.1)
        self.assertAlmostEqual(record.alignments[5].hsps[0].expect, 6e-7)
        self.assertEqual(record.alignments[5].hsps[1].score, 84)
        self.assertAlmostEqual(record.alignments[5].hsps[1].bits, 37.1)
        self.assertAlmostEqual(record.alignments[5].hsps[1].expect, 0.036)
        self.assertEqual(len(record.alignments[5].hsps), 2)
        self.assertEqual(record.alignments[6].hsps[0].score, 123)
        self.assertAlmostEqual(record.alignments[6].hsps[0].bits, 52.3)
        self.assertAlmostEqual(record.alignments[6].hsps[0].expect, 1e-6)
        self.assertEqual(record.alignments[6].hsps[1].score, 86)
        self.assertAlmostEqual(record.alignments[6].hsps[1].bits, 37.9)
        self.assertAlmostEqual(record.alignments[6].hsps[1].expect, 0.021)
        self.assertEqual(len(record.alignments[6].hsps), 2)
        self.assertEqual(record.alignments[7].hsps[0].score, 119)
        self.assertAlmostEqual(record.alignments[7].hsps[0].bits, 50.8)
        self.assertAlmostEqual(record.alignments[7].hsps[0].expect, 3e-6)
        self.assertEqual(record.alignments[7].hsps[1].score, 83)
        self.assertAlmostEqual(record.alignments[7].hsps[1].bits, 36.7)
        self.assertAlmostEqual(record.alignments[7].hsps[1].expect, 0.047)
        self.assertEqual(len(record.alignments[7].hsps), 2)
        self.assertEqual(record.alignments[8].hsps[0].score, 108)
        self.assertAlmostEqual(record.alignments[8].hsps[0].bits, 46.5)
        self.assertAlmostEqual(record.alignments[8].hsps[0].expect, 5e-5)
        self.assertEqual(len(record.alignments[8].hsps), 1)
        self.assertEqual(record.alignments[9].hsps[0].score, 86)
        self.assertAlmostEqual(record.alignments[9].hsps[0].bits, 37.9)
        self.assertAlmostEqual(record.alignments[9].hsps[0].expect, 0.021)
        self.assertEqual(len(record.alignments[9].hsps), 1)
        self.assertEqual(record.alignments[10].hsps[0].score, 74)
        self.assertAlmostEqual(record.alignments[10].hsps[0].bits, 33.2)
        self.assertAlmostEqual(record.alignments[10].hsps[0].expect, 0.53)
        self.assertEqual(len(record.alignments[10].hsps), 1)
        self.assertEqual(record.alignments[11].hsps[0].score, 66)
        self.assertAlmostEqual(record.alignments[11].hsps[0].bits, 30.1)
        self.assertAlmostEqual(record.alignments[11].hsps[0].expect, 4.6)
        self.assertEqual(len(record.alignments[11].hsps), 1)
        self.assertEqual(record.alignments[12].hsps[0].score, 64)
        self.assertAlmostEqual(record.alignments[12].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.alignments[12].hsps[0].expect, 7.9)
        self.assertEqual(len(record.alignments[12].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (247, 270))
        self.assertEqual(record.alignments[0].hsps[0].positives, (247, 270))
        self.assertEqual(record.alignments[1].hsps[0].identities, (123, 259))
        self.assertEqual(record.alignments[1].hsps[0].positives, (155, 259))
        self.assertEqual(record.alignments[1].hsps[0].gaps, (13, 259))
        self.assertEqual(record.alignments[2].hsps[0].identities, (93, 253))
        self.assertEqual(record.alignments[2].hsps[0].positives, (132, 253))
        self.assertEqual(record.alignments[2].hsps[0].gaps, (8, 253))
        self.assertEqual(record.alignments[2].hsps[1].identities, (33, 99))
        self.assertEqual(record.alignments[2].hsps[1].positives, (47, 99))
        self.assertEqual(record.alignments[2].hsps[1].gaps, (7, 99))
        self.assertEqual(record.alignments[3].hsps[0].identities, (76, 247))
        self.assertEqual(record.alignments[3].hsps[0].positives, (114, 247))
        self.assertEqual(record.alignments[3].hsps[0].gaps, (17, 247))
        self.assertEqual(record.alignments[4].hsps[0].identities, (46, 170))
        self.assertEqual(record.alignments[4].hsps[0].positives, (76, 170))
        self.assertEqual(record.alignments[4].hsps[0].gaps, (3, 170))
        self.assertEqual(record.alignments[5].hsps[0].identities, (25, 64))
        self.assertEqual(record.alignments[5].hsps[0].positives, (43, 64))
        self.assertEqual(record.alignments[5].hsps[1].identities, (19, 63))
        self.assertEqual(record.alignments[5].hsps[1].positives, (36, 63))
        self.assertEqual(record.alignments[6].hsps[0].identities, (24, 64))
        self.assertEqual(record.alignments[6].hsps[0].positives, (43, 64))
        self.assertEqual(record.alignments[6].hsps[1].identities, (21, 63))
        self.assertEqual(record.alignments[6].hsps[1].positives, (36, 63))
        self.assertEqual(record.alignments[7].hsps[0].identities, (24, 64))
        self.assertEqual(record.alignments[7].hsps[0].positives, (43, 64))
        self.assertEqual(record.alignments[7].hsps[1].identities, (20, 63))
        self.assertEqual(record.alignments[7].hsps[1].positives, (37, 63))
        self.assertEqual(record.alignments[8].hsps[0].identities, (19, 67))
        self.assertEqual(record.alignments[8].hsps[0].positives, (46, 67))
        self.assertEqual(record.alignments[9].hsps[0].identities, (23, 62))
        self.assertEqual(record.alignments[9].hsps[0].positives, (37, 62))
        self.assertEqual(record.alignments[9].hsps[0].gaps, (1, 62))
        self.assertEqual(record.alignments[10].hsps[0].identities, (41, 143))
        self.assertEqual(record.alignments[10].hsps[0].positives, (62, 143))
        self.assertEqual(record.alignments[10].hsps[0].gaps, (12, 143))
        self.assertEqual(record.alignments[11].hsps[0].identities, (32, 141))
        self.assertEqual(record.alignments[11].hsps[0].positives, (61, 141))
        self.assertEqual(record.alignments[11].hsps[0].gaps, (6, 141))
        self.assertEqual(record.alignments[12].hsps[0].identities, (21, 62))
        self.assertEqual(record.alignments[12].hsps[0].positives, (29, 62))
        self.assertEqual(record.alignments[0].hsps[0].query, "MAATKQGGGDDGRCARGVVLERTGARMGAERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNXXXXXXXXXXXXXXXXXXXXXXXAGEKLIAAYGAIEAGVAKLLSSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILALP")
        self.assertEqual(record.alignments[0].hsps[0].match, "MAATKQGGGDDGRCARGVVLERTGARMGAERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNN                       AGEKLIAAYGAIEAGVAKLLSSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILALP")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "MAATKQGGGDDGRCARGVVLERTGARMGAERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNLLAAPVVTAAPGGKAGGGAVLTPAGEKLIAAYGAIEAGVAKLLSSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILALP")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 270)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 270)
        self.assertEqual(record.alignments[1].hsps[0].query, "LERTGA-RMGAERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNXXXXXXXXXXXXXXXXXXXXXXXAGEKLIAAYGAIEAGVAKLL-------SSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILALP")
        self.assertEqual(record.alignments[1].hsps[0].match, "L+R GA R+G +R+ LL AI R G+I+ AAREVGLSYK AWD V  +NN                       AG+ LIA +G +E  + K L       S+ EK+LN      L  L++RTS RN   C V  V    V A+V + L +G  LTAVIT RSA EM LAPG EV AL+K++FV+LA  G P R+S  NR+ G V  R D P+++E+ LDLG  K+ITA IT  SA+ L L PGV  TAL K+SHVILA+P")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "LQRAGAPRVGGDRIRLLEAIARHGTIAGAAREVGLSYKTAWDAVGTLNNLFEQPLVEAAPGGRTGGNARVTEAGQALIAGFGLLEGALTKALGVLEGGVSAPEKALN-----TLWSLTMRTSNRNTLRCTVTRVTLGAVNAEVELALTDGHSLTAVITERSATEMGLAPGVEVFALIKASFVMLAAGGDPGRISACNRLTGIVAARTDGPVNTEIILDLGNCKSITAVITHTSADALGLAPGVPATALFKASHVILAMP")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 20)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 270)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 12)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 265)
        self.assertEqual(record.alignments[2].hsps[0].query, "GARMGAERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNXXXXXXXXXXXXXXXXXXXXXXXAGEKLIAAYGAIEAGVAKLLSSFEKSLNLDPA-------EVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILAL")
        self.assertEqual(record.alignments[2].hsps[0].match, "G  +   R+ LL AI R GSI+ AA+ V LSYKAAWD +  MNN                        G +++A Y A+E      L    + LN            ++  +S++TSARN +A  V  +    V  +VR+RL    ++ AVIT  SA  + LA G EV ALVKS+ V+L       +L+ RN++ G VI+  + P+++EVTL L  G+++T  +T DS + L L PGV   A  KSS VILA+")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "GTALSDTRIRLLEAIEREGSINRAAKVVPLSYKAAWDAIDTMNNLAPEPLVVRVAGGRQGGGTQLTDYGRRIVAMYRALEIEYQSALDRLSERLNEVTGGDIQAFQRLMHSMSMKTSARNQFAGIVTGLRVGGVDYEVRIRLDAENEIAAVITKASAENLELAIGKEVFALVKSSSVMLT-TEPSLKLTARNQLWGEVIDIHEGPVNNEVTLALPSGRSVTCVVTADSCKALGLAPGVAACAFFKSSSVILAV")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 24)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 269)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 17)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 268)
        self.assertEqual(record.alignments[2].hsps[1].query, "AIEAGVAKLLSSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAG")
        self.assertEqual(record.alignments[2].hsps[1].match, "AI   V  L+ S    L  +P       SL+ +ARN    +V  +    V  +V + L  G+ +T V+TA S   + LAPG    A  KS+ V+LA  G")
        self.assertEqual(record.alignments[2].hsps[1].sbjct, "AIGKEVFALVKSSSVMLTTEP-------SLKLTARNQLWGEVIDIHEGPVNNEVTLALPSGRSVTCVVTADSCKALGLAPGVAACAFFKSSSVILAVYG")
        self.assertEqual(record.alignments[2].hsps[1].query_start, 101)
        self.assertEqual(record.alignments[2].hsps[1].query_end, 199)
        self.assertEqual(record.alignments[2].hsps[1].sbjct_start, 179)
        self.assertEqual(record.alignments[2].hsps[1].sbjct_end, 270)
        self.assertEqual(record.alignments[3].hsps[0].query, "RVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNXXXXXXXXXXXXXXXXXXXXXXXAGEKLIAAY---GAIEAGVAKLLSSFEK-SLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQD-LTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVR----NRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILA")
        self.assertEqual(record.alignments[3].hsps[0].match, "R++LL  I  +GSIS  A++ G+SYK+AWD +  MN                         G++LI  Y     I+     +LS  +   LN   A + R  SL+TSARN W   + +   DDV   V + L +G+  L   ITA+S A + L  G EVL L+K+ +V     G+ +  +V     N++ G +          EV + L  G+T+ AT+  +  E   L  G   TA   +  VI+A")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "RISLLKHIALSGSISQGAKDAGISYKSAWDAINEMNQLSEHILVERATGGKGGGGAVLTRYGQRLIQLYDLLAQIQQKAFDVLSDDDALPLNSLLAAISR-FSLQTSARNQWFGTITARDHDDVQQHVDVLLADGKTRLKVAITAQSGARLGLDEGKEVLILLKAPWV-----GITQDEAVAQNADNQLPGIISHIERGAEQCEVLMALPDGQTLCATVPVN--EATSLQQGQNVTAYFNADSVIIA")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 31)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 268)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 21)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 259)
        self.assertEqual(record.alignments[4].hsps[0].query, "ERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNXXXXXXXXXXXXXXXXXXXXXXXAGEKLIAAYGAIEAGVAKLLSSF-EKSLNLDP-AEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRL-GEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLA")
        self.assertEqual(record.alignments[4].hsps[0].match, "+RV LL  I + GSI+ AA+   +SYK+AWD ++AMN                          E+L+  Y  +E           ++S+ LD         SL++SARN +  +V      D    V + + G    L   IT +S+A ++L    EV+ + K+ +V ++")
        self.assertEqual(record.alignments[4].hsps[0].sbjct, "KRVRLLKEIQQCGSINQAAKNAKVSYKSAWDHLEAMNKISPRPLLERNTGGKNGGGTALTTYAERLLQLYDLLERTQEHAFHILQDESVPLDSLLTATARFSLQSSARNQFFGRVAQQRIIDSRCVVDVNVQGLPTPLQVSITTKSSARLKLITEKEVMLMFKAPWVKIS")
        self.assertEqual(record.alignments[4].hsps[0].query_start, 30)
        self.assertEqual(record.alignments[4].hsps[0].query_end, 196)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_start, 21)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_end, 190)
        self.assertEqual(record.alignments[5].hsps[0].query, "LSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVIL")
        self.assertEqual(record.alignments[5].hsps[0].match, "+S RN+++G+V+      +++EV L++ GG  IT+ I+ DS E L +  G E TA+IKS+ V++")
        self.assertEqual(record.alignments[5].hsps[0].sbjct, "ISARNQLKGKVVGLKKGVITAEVVLEIAGGNKITSIISLDSVEELGVKEGAELTAVIKSTDVMI")
        self.assertEqual(record.alignments[5].hsps[0].query_start, 204)
        self.assertEqual(record.alignments[5].hsps[0].query_end, 267)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_start, 3)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_end, 66)
        self.assertEqual(record.alignments[5].hsps[1].query, "SARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLL")
        self.assertEqual(record.alignments[5].hsps[1].match, "SARN    KV  +    + A+V + +  G  +T++I+  S  E+ +  G+E+ A++KS  V++")
        self.assertEqual(record.alignments[5].hsps[1].sbjct, "SARNQLKGKVVGLKKGVITAEVVLEIAGGNKITSIISLDSVEELGVKEGAELTAVIKSTDVMI")
        self.assertEqual(record.alignments[5].hsps[1].query_start, 133)
        self.assertEqual(record.alignments[5].hsps[1].query_end, 195)
        self.assertEqual(record.alignments[5].hsps[1].sbjct_start, 4)
        self.assertEqual(record.alignments[5].hsps[1].sbjct_end, 66)
        self.assertEqual(record.alignments[6].hsps[0].query, "LSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVIL")
        self.assertEqual(record.alignments[6].hsps[0].match, "+S RN+++G+V+      +++EV L++ GG  IT+ I+ DS E L +  G E TA++KS+ V++")
        self.assertEqual(record.alignments[6].hsps[0].sbjct, "ISARNQLKGKVVGLKKGVVTAEVVLEIAGGNKITSIISLDSVEELGVKEGAELTAVVKSTDVMI")
        self.assertEqual(record.alignments[6].hsps[0].query_start, 204)
        self.assertEqual(record.alignments[6].hsps[0].query_end, 267)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_start, 3)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_end, 66)
        self.assertEqual(record.alignments[6].hsps[1].query, "SARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLL")
        self.assertEqual(record.alignments[6].hsps[1].match, "SARN    KV  +    V A+V + +  G  +T++I+  S  E+ +  G+E+ A+VKS  V++")
        self.assertEqual(record.alignments[6].hsps[1].sbjct, "SARNQLKGKVVGLKKGVVTAEVVLEIAGGNKITSIISLDSVEELGVKEGAELTAVVKSTDVMI")
        self.assertEqual(record.alignments[6].hsps[1].query_start, 133)
        self.assertEqual(record.alignments[6].hsps[1].query_end, 195)
        self.assertEqual(record.alignments[6].hsps[1].sbjct_start, 4)
        self.assertEqual(record.alignments[6].hsps[1].sbjct_end, 66)
        self.assertEqual(record.alignments[7].hsps[0].query, "LSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVIL")
        self.assertEqual(record.alignments[7].hsps[0].match, "+S RN+++G+V+      +++EV L++ GG  +T+ I+ DS E L +  G E TA+IKS+ V++")
        self.assertEqual(record.alignments[7].hsps[0].sbjct, "ISARNQLKGKVVAVKKGLVTAEVVLEIAGGDKVTSIISLDSIEDLGVKEGTELTAVIKSTDVMI")
        self.assertEqual(record.alignments[7].hsps[0].query_start, 204)
        self.assertEqual(record.alignments[7].hsps[0].query_end, 267)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_start, 3)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_end, 66)
        self.assertEqual(record.alignments[7].hsps[1].query, "SARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLL")
        self.assertEqual(record.alignments[7].hsps[1].match, "SARN    KV +V    V A+V + +  G  +T++I+  S  ++ +  G+E+ A++KS  V++")
        self.assertEqual(record.alignments[7].hsps[1].sbjct, "SARNQLKGKVVAVKKGLVTAEVVLEIAGGDKVTSIISLDSIEDLGVKEGTELTAVIKSTDVMI")
        self.assertEqual(record.alignments[7].hsps[1].query_start, 133)
        self.assertEqual(record.alignments[7].hsps[1].query_end, 195)
        self.assertEqual(record.alignments[7].hsps[1].sbjct_start, 4)
        self.assertEqual(record.alignments[7].hsps[1].sbjct_end, 66)
        self.assertEqual(record.alignments[8].hsps[0].query, "RLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILAL")
        self.assertEqual(record.alignments[8].hsps[0].match, "++S RN+++G+V+   +  +++ V +D+GGG  +++T++  + + L+L  G E  A+IK++ V++ +")
        self.assertEqual(record.alignments[8].hsps[0].sbjct, "KISARNQLKGKVVSIENGSVNAIVHIDIGGGNVLSSTVSLAAVKELNLEVGKEAYAIIKATSVMVGV")
        self.assertEqual(record.alignments[8].hsps[0].query_start, 203)
        self.assertEqual(record.alignments[8].hsps[0].query_end, 269)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_start, 2)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_end, 68)
        self.assertEqual(record.alignments[9].hsps[0].query, "PERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKS")
        self.assertEqual(record.alignments[9].hsps[0].match, "P++ S+RN +R +V+   D     EV L++ GGKT+ A I+  + + L + PG+   A IKS")
        self.assertEqual(record.alignments[9].hsps[0].sbjct, "PQQTSIRNVLRAKVVNSYDDNGQVEVELEV-GGKTLWARISPWARDELAIKPGLWLYAQIKS")
        self.assertEqual(record.alignments[9].hsps[0].query_start, 201)
        self.assertEqual(record.alignments[9].hsps[0].query_end, 262)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_start, 287)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_end, 347)
        self.assertEqual(record.alignments[10].hsps[0].query, "EVLRGLSLRTSARNAWACKVWSVAA--DDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGG-GKTITATITRDSAEMLDLHPGVETTALIKS")
        self.assertEqual(record.alignments[10].hsps[0].match, "+++  L L T+        + SV A  DD     R+    G    AV+ AR       APG  +   V +  V LA + + E  S+ N +   V E ++A   + V + L   G  + A ITR S + L + PG    A IK+")
        self.assertEqual(record.alignments[10].hsps[0].sbjct, "DIMARLDLPTAFHEDAGVVIESVVAEHDDHYHLTRLAFPGG----AVLVARRPE----APGQRLRLRVHARDVSLANSRI-EDSSITNVLPATVREVVEADTPAHVLVRLEAEGTPLIARITRRSCDQLGIAPGRRMWAQIKA")
        self.assertEqual(record.alignments[10].hsps[0].query_start, 123)
        self.assertEqual(record.alignments[10].hsps[0].query_end, 262)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_start, 242)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_end, 375)
        self.assertEqual(record.alignments[11].hsps[0].query, "AAYGAIEAGVAKLLSSFEKSLNLDP-AEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTI")
        self.assertEqual(record.alignments[11].hsps[0].match, "AA G I+ GV  ++      + +DP AE ++    + +A  A   + W+   ++       + G   +L A I      +  L  G E + L ++ F+ +    +P      +  +  V+ER++       TLD+GG K +")
        self.assertEqual(record.alignments[11].hsps[0].sbjct, "AATGTIQNGVTVIVDGINGDVIIDPSAETVKEYEEKHNAYLAQKAE-WAKLVNEPTVS---KDGHHVELAANIGTPDDVKGVLENGGEAVGLYRTEFLYMGRDQLPTEDEQFDAYK-TVLERMEGKSVVVRTLDIGGDKEL")
        self.assertEqual(record.alignments[11].hsps[0].query_start, 97)
        self.assertEqual(record.alignments[11].hsps[0].query_end, 236)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_start, 207)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_end, 342)
        self.assertEqual(record.alignments[12].hsps[0].query, "AIEAGVAKLLSSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQ")
        self.assertEqual(record.alignments[12].hsps[0].match, "A+E G     S+  +S   DPA V   +  R S  +      + VAAD   + VR +LG GQ")
        self.assertEqual(record.alignments[12].hsps[0].sbjct, "AVELGGEIRFSTELQSFEQDPAGVTAVIKSRRSGEHTTVRADYLVAADGPRSPVREQLGIGQ")
        self.assertEqual(record.alignments[12].hsps[0].query_start, 101)
        self.assertEqual(record.alignments[12].hsps[0].query_end, 162)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_start, 136)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_end, 197)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Nov 29, 1999  4:39 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.316)
        self.assertAlmostEqual(record.ka_params[1], 0.131)
        self.assertAlmostEqual(record.ka_params[2], 0.361)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 12068104)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 396723)
        self.assertEqual(record.num_good_extends, 1066)
        self.assertEqual(record.num_seqs_better_e, 13)
        self.assertEqual(record.hsps_no_gap, 10)
        self.assertEqual(record.hsps_prelim_gapped, 3)
        self.assertEqual(record.hsps_gapped, 18)
        self.assertEqual(record.query_length, 270)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 56)
        self.assertEqual(record.effective_query_length, 214)
        self.assertEqual(record.effective_database_length, 25046113)
        self.assertEqual(record.effective_search_space, 5359868182)
        self.assertEqual(record.effective_search_space_used, 5359868182)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.6)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 64)
        self.assertAlmostEqual(record.blast_cutoff[1], 29.3)

    def test_text_2010L_blastp_007(self):
        "Test parsing BLASTP 2.0.10 output (text_2010L_blastp_007)"

        path = os.path.join('Blast', 'text_2010L_blastp_007.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.10')
        self.assertEqual(record.date, "Aug-26-1999")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|730725|sp|Q05362|SCHB_STRHA SCHB PROTEIN")
        self.assertEqual(record.query_letters, 138)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Nov 29, 1999  4:39 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.319)
        self.assertAlmostEqual(record.ka_params[1], 0.139)
        self.assertAlmostEqual(record.ka_params[2], 0.415)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 8952568)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 387403)
        self.assertEqual(record.num_good_extends, 727)
        self.assertEqual(record.num_seqs_better_e, 23)
        self.assertEqual(record.hsps_no_gap, 13)
        self.assertEqual(record.hsps_prelim_gapped, 10)
        self.assertEqual(record.hsps_gapped, 23)
        self.assertEqual(record.query_length, 138)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 47)
        self.assertEqual(record.effective_query_length, 91)
        self.assertEqual(record.effective_database_length, 25786435)
        self.assertEqual(record.effective_search_space, 2346565585)
        self.assertEqual(record.effective_search_space_used, 2346565585)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.4)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 61)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.2)

    def test_text_2011L_blastp_001(self):
        "Test parsing BLASTP 2.0.11 output (text_2011L_blastp_001)"

        path = os.path.join('Blast', 'text_2011L_blastp_001.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|120291|sp|P21297|FLBT_CAUCR FLBT PROTEIN")
        self.assertEqual(record.query_letters, 141)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 2)
        self.assertEqual(record.descriptions[0].title, "gi|120291|sp|P21297|FLBT_CAUCR FLBT PROTEIN")
        self.assertEqual(record.descriptions[0].score, 256)
        self.assertAlmostEqual(record.descriptions[0].e, 2e-68)
        self.assertEqual(record.descriptions[1].title, "gi|3024946|sp|Q58368|Y958_METJA HYPOTHETICAL PROTEIN MJ0958 PRE...")
        self.assertEqual(record.descriptions[1].score, 29)
        self.assertAlmostEqual(record.descriptions[1].e, 3.4)
        self.assertEqual(len(record.alignments), 2)
        self.assertEqual(record.alignments[0].title, ">gi|120291|sp|P21297|FLBT_CAUCR FLBT PROTEIN")
        self.assertEqual(record.alignments[0].length, 141)
        self.assertEqual(record.alignments[1].title, ">gi|3024946|sp|Q58368|Y958_METJA HYPOTHETICAL PROTEIN MJ0958 PRECURSOR")
        self.assertEqual(record.alignments[1].length, 426)
        self.assertEqual(record.alignments[0].hsps[0].score, 646)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 256)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 2e-68)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 64)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 3.4)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (127, 127))
        self.assertEqual(record.alignments[0].hsps[0].positives, (127, 127))
        self.assertEqual(record.alignments[1].hsps[0].identities, (15, 47))
        self.assertEqual(record.alignments[1].hsps[0].positives, (23, 47))
        self.assertEqual(record.alignments[0].hsps[0].query, "MPLKLSLKPGEKFVLNGAVVQNGDRRGVLVLQNKASVLREKDIMQPDQVTTPARHIYFPVMMMYLDEVGAEKFYEEFATRLNEFMGVVRNPVVLQDCIAISKHVMAREYYKALMLSRKLIEYEDERL")
        self.assertEqual(record.alignments[0].hsps[0].match, "MPLKLSLKPGEKFVLNGAVVQNGDRRGVLVLQNKASVLREKDIMQPDQVTTPARHIYFPVMMMYLDEVGAEKFYEEFATRLNEFMGVVRNPVVLQDCIAISKHVMAREYYKALMLSRKLIEYEDERL")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "MPLKLSLKPGEKFVLNGAVVQNGDRRGVLVLQNKASVLREKDIMQPDQVTTPARHIYFPVMMMYLDEVGAEKFYEEFATRLNEFMGVVRNPVVLQDCIAISKHVMAREYYKALMLSRKLIEYEDERL")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 127)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 127)
        self.assertEqual(record.alignments[1].hsps[0].query, "VLVLQNKASVLREKDIMQPDQVTTPARHIYFPVMMMYLDEVGAEKFY")
        self.assertEqual(record.alignments[1].hsps[0].match, "+LVL N  ++   K     D  TT   +IY P+ +    +  A+KFY")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "ILVLINNTNITELKKFEDDDYYTTFQHYIYQPIFIFTTYDSKAKKFY")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 28)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 74)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 169)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 215)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Feb 2, 2000  9:39 AM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.323)
        self.assertAlmostEqual(record.ka_params[1], 0.139)
        self.assertAlmostEqual(record.ka_params[2], 0.392)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 6932673)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 246623)
        self.assertEqual(record.num_good_extends, 486)
        self.assertEqual(record.num_seqs_better_e, 2)
        self.assertEqual(record.hsps_no_gap, 2)
        self.assertEqual(record.hsps_prelim_gapped, 0)
        self.assertEqual(record.hsps_gapped, 2)
        self.assertEqual(record.query_length, 141)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 50)
        self.assertEqual(record.effective_query_length, 91)
        self.assertEqual(record.effective_database_length, 25539661)
        self.assertEqual(record.effective_search_space, 2324109151)
        self.assertEqual(record.effective_search_space_used, 2324109151)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.5)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.9)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 61)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.2)

    def test_text_2011L_blastp_002(self):
        "Test parsing BLASTP 2.0.11 output without hits (text_2011L_blastp_002)"

        path = os.path.join('Blast', 'text_2011L_blastp_002.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|400206|sp|Q02112|LYTA_BACSU MEMBRANE-BOUND PROTEIN LYTA\nPRECURSOR")
        self.assertEqual(record.query_letters, 102)
        self.assertEqual(record.database, "data/pdbaa")
        self.assertEqual(record.database_sequences, 6999)
        self.assertEqual(record.database_letters, 1461753)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/pdbaa'])
        self.assertEqual(record.num_letters_in_database, [1461753])
        self.assertEqual(record.num_sequences_in_database, [6999])
        self.assertEqual(record.posted_date, [('Feb 11, 2000  2:32 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.302)
        self.assertAlmostEqual(record.ka_params[1], 0.126)
        self.assertAlmostEqual(record.ka_params[2], 0.332)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 270485)
        self.assertEqual(record.num_sequences, 6999)
        self.assertEqual(record.num_extends, 9861)
        self.assertEqual(record.num_good_extends, 7)
        self.assertEqual(record.num_seqs_better_e, 0)
        self.assertEqual(record.hsps_no_gap, 0)
        self.assertEqual(record.hsps_prelim_gapped, 0)
        self.assertEqual(record.hsps_gapped, 0)
        self.assertEqual(record.query_length, 102)
        self.assertEqual(record.database_length, 1461753)
        self.assertEqual(record.effective_hsp_length, 47)
        self.assertEqual(record.effective_query_length, 55)
        self.assertEqual(record.effective_database_length, 1132800)
        self.assertEqual(record.effective_search_space, 62304000)
        self.assertEqual(record.effective_search_space_used, 62304000)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 17)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.4)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 43)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 47)
        self.assertAlmostEqual(record.blast_cutoff[1], 22.7)

    def test_text_2011L_blastp_004(self):
        "Test parsing BLASTP 2.0.11 output without descriptions (text_2011L_blastp_004)"

        path = os.path.join('Blast', 'text_2011L_blastp_004.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1718062|sp|P16153|UTXA_CLODI UTXA PROTEIN")
        self.assertEqual(record.query_letters, 166)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 6)
        self.assertEqual(record.alignments[0].title, ">gi|1718062|sp|P16153|UTXA_CLODI UTXA PROTEIN")
        self.assertEqual(record.alignments[0].length, 166)
        self.assertEqual(record.alignments[1].title, ">gi|140528|sp|P24811|YQXH_BACSU HYPOTHETICAL 15.7 KD PROTEIN IN SPOIIIC-CWLA INTERGENIC REGION (ORF2)")
        self.assertEqual(record.alignments[1].length, 140)
        self.assertEqual(record.alignments[2].title, ">gi|141088|sp|P26835|YNGD_CLOPE HYPOTHETICAL 14.9 KD PROTEIN IN NAGH 3'REGION (ORFD)")
        self.assertEqual(record.alignments[2].length, 132)
        self.assertEqual(record.alignments[3].title, ">gi|6014830|sp|O78935|CYB_MARAM CYTOCHROME B")
        self.assertEqual(record.alignments[3].length, 379)
        self.assertEqual(record.alignments[4].title, ">gi|1351589|sp|P47694|Y456_MYCGE HYPOTHETICAL PROTEIN MG456")
        self.assertEqual(record.alignments[4].length, 334)
        self.assertEqual(record.alignments[5].title, ">gi|2496246|sp|Q57881|Y439_METJA HYPOTHETICAL ATP-BINDING PROTEIN MJ0439")
        self.assertEqual(record.alignments[5].length, 361)
        self.assertEqual(record.alignments[0].hsps[0].score, 843)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 332)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 2e-91)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 90)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 39.5)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 0.004)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 88)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 38.7)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 0.007)
        self.assertEqual(len(record.alignments[2].hsps), 1)
        self.assertEqual(record.alignments[3].hsps[0].score, 64)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 4.6)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[4].hsps[0].score, 63)
        self.assertAlmostEqual(record.alignments[4].hsps[0].bits, 29.0)
        self.assertAlmostEqual(record.alignments[4].hsps[0].expect, 6.0)
        self.assertEqual(len(record.alignments[4].hsps), 1)
        self.assertEqual(record.alignments[5].hsps[0].score, 62)
        self.assertAlmostEqual(record.alignments[5].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.alignments[5].hsps[0].expect, 7.8)
        self.assertEqual(len(record.alignments[5].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (166, 166))
        self.assertEqual(record.alignments[0].hsps[0].positives, (166, 166))
        self.assertEqual(record.alignments[1].hsps[0].identities, (27, 130))
        self.assertEqual(record.alignments[1].hsps[0].positives, (55, 130))
        self.assertEqual(record.alignments[1].hsps[0].gaps, (19, 130))
        self.assertEqual(record.alignments[2].hsps[0].identities, (24, 110))
        self.assertEqual(record.alignments[2].hsps[0].positives, (52, 110))
        self.assertEqual(record.alignments[2].hsps[0].gaps, (18, 110))
        self.assertEqual(record.alignments[3].hsps[0].identities, (19, 57))
        self.assertEqual(record.alignments[3].hsps[0].positives, (33, 57))
        self.assertEqual(record.alignments[3].hsps[0].gaps, (2, 57))
        self.assertEqual(record.alignments[4].hsps[0].identities, (16, 44))
        self.assertEqual(record.alignments[4].hsps[0].positives, (24, 44))
        self.assertEqual(record.alignments[4].hsps[0].gaps, (2, 44))
        self.assertEqual(record.alignments[5].hsps[0].identities, (19, 56))
        self.assertEqual(record.alignments[5].hsps[0].positives, (30, 56))
        self.assertEqual(record.alignments[5].hsps[0].gaps, (12, 56))
        self.assertEqual(record.alignments[0].hsps[0].query, "MHSSSPFYISNGNKIFFYINLGGVMNMTISFLSEHIFIKLVILTISFDTLLGCLSAIKSRKFNSSFGIDGGIRKVAMIACIFFLSVVDILTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKNMCLCGLPVPKRLKEKIAILLDAMTDEMNAKDEK")
        self.assertEqual(record.alignments[0].hsps[0].match, "MHSSSPFYISNGNKIFFYINLGGVMNMTISFLSEHIFIKLVILTISFDTLLGCLSAIKSRKFNSSFGIDGGIRKVAMIACIFFLSVVDILTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKNMCLCGLPVPKRLKEKIAILLDAMTDEMNAKDEK")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "MHSSSPFYISNGNKIFFYINLGGVMNMTISFLSEHIFIKLVILTISFDTLLGCLSAIKSRKFNSSFGIDGGIRKVAMIACIFFLSVVDILTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKNMCLCGLPVPKRLKEKIAILLDAMTDEMNAKDEK")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 166)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 166)
        self.assertEqual(record.alignments[1].hsps[0].query, "FIKLVILTISFDTLLGCLSAIKSRKFNSSFGIDGGIRKVAMIACIFFLSVVDILTKFNFLFMLPQDCINFLRLKHLGISEFFSILF-ILYESVSILKNMCLCGLPVPKRLKEKIAILLDAMTDEMNAKDE")
        self.assertEqual(record.alignments[1].hsps[0].match, "++ L+++    D L G + A K +K  S     G +RK+     +   +V+D +   N                  G+  F ++LF I  E +SI +N+   G+ +P  + +++  + +      N  D+")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "YLDLLLVLSIIDVLTGVIKAWKFKKLRSRSAWFGYVRKLLNFFAVILANVIDTVLNLN------------------GVLTFGTVLFYIANEGLSITENLAQIGVKIPSSITDRLQTIENEKEQSKNNADK")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 37)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 165)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 26)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 137)
        self.assertEqual(record.alignments[2].hsps[0].query, "VILTISFDTLLGCLSAIKSRKFNSSFGIDGGIRKVAMIACIFFLSVVD-ILTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKNMCLCGLPVPKRLKEKI")
        self.assertEqual(record.alignments[2].hsps[0].match, "+++ I  D L G +   KS++  S+ G+ G  +K  ++  +    ++D +L    ++F                     +  +I+ E +SIL+N    G+P+P++LK+ +")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "LLVFIFLDYLTGVIKGCKSKELCSNIGLRGITKKGLILVVLLVAVMLDRLLDNGTWMFRT-----------------LIAYFYIMNEGISILENCAALGVPIPEKLKQAL")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 41)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 149)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 33)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 125)
        self.assertEqual(record.alignments[3].hsps[0].query, "CIFFLSVVDILTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKNMCL")
        self.assertEqual(record.alignments[3].hsps[0].match, "C+F+L V D+LT   ++   P +   F+ +  L    +F+IL IL  ++SI++N  L")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "CLFWLLVADLLT-LTWIGGQPVEH-PFITIGQLASILYFAILLILMPAISIIENNLL")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 80)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 136)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 323)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 377)
        self.assertEqual(record.alignments[4].hsps[0].query, "LTKFNFLFMLPQDCINFLRLKHLGISEFFSILFILYESVSILKN")
        self.assertEqual(record.alignments[4].hsps[0].match, "LTKFN  F+ P     FLR+  +G+   FS++ I +   S  +N")
        self.assertEqual(record.alignments[4].hsps[0].sbjct, "LTKFNKFFLTPNKLNAFLRV--IGLCGLFSVIAISFGIYSYTRN")
        self.assertEqual(record.alignments[4].hsps[0].query_start, 90)
        self.assertEqual(record.alignments[4].hsps[0].query_end, 133)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_start, 4)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_end, 45)
        self.assertEqual(record.alignments[5].hsps[0].query, "FLRLKHLGIS---EFFSILFILYES----VSILKNMC-----LCGLPVPKRLKEKI")
        self.assertEqual(record.alignments[5].hsps[0].match, "++ L+ + IS   +F  +LF  YE     V I+K++      LCG+P PK   E+I")
        self.assertEqual(record.alignments[5].hsps[0].sbjct, "YINLRGIFISKYKDFIEVLFEEYEEDRKPVEIIKSLIKDVPSLCGIPTPKNTLEEI")
        self.assertEqual(record.alignments[5].hsps[0].query_start, 106)
        self.assertEqual(record.alignments[5].hsps[0].query_end, 149)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_start, 68)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_end, 123)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Feb 2, 2000  9:39 AM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.331)
        self.assertAlmostEqual(record.ka_params[1], 0.146)
        self.assertAlmostEqual(record.ka_params[2], 0.428)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 8801581)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 320828)
        self.assertEqual(record.num_good_extends, 892)
        self.assertEqual(record.num_seqs_better_e, 6)
        self.assertEqual(record.hsps_no_gap, 3)
        self.assertEqual(record.hsps_prelim_gapped, 3)
        self.assertEqual(record.hsps_gapped, 6)
        self.assertEqual(record.query_length, 166)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 46)
        self.assertEqual(record.effective_query_length, 120)
        self.assertEqual(record.effective_database_length, 25868693)
        self.assertEqual(record.effective_search_space, 3104243160)
        self.assertEqual(record.effective_search_space_used, 3104243160)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 15)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.2)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 40)
        self.assertAlmostEqual(record.gap_trigger[1], 21.9)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 62)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.6)

    def test_text_2011L_blastp_005(self):
        "Test parsing BLASTP 2.0.11 output without alignments (text_2011L_blastp_005)"

        path = os.path.join('Blast', 'text_2011L_blastp_005.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1718062|sp|P16153|UTXA_CLODI UTXA PROTEIN")
        self.assertEqual(record.query_letters, 166)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 6)
        self.assertEqual(record.descriptions[0].title, "gi|1718062|sp|P16153|UTXA_CLODI UTXA PROTEIN")
        self.assertEqual(record.descriptions[0].score, 332)
        self.assertAlmostEqual(record.descriptions[0].e, 2e-91)
        self.assertEqual(record.descriptions[1].title, "gi|140528|sp|P24811|YQXH_BACSU HYPOTHETICAL 15.7 KD PROTEIN IN ...")
        self.assertEqual(record.descriptions[1].score, 39)
        self.assertAlmostEqual(record.descriptions[1].e, 0.004)
        self.assertEqual(record.descriptions[2].title, "gi|141088|sp|P26835|YNGD_CLOPE HYPOTHETICAL 14.9 KD PROTEIN IN ...")
        self.assertEqual(record.descriptions[2].score, 39)
        self.assertAlmostEqual(record.descriptions[2].e, 0.007)
        self.assertEqual(record.descriptions[3].title, "gi|6014830|sp|O78935|CYB_MARAM CYTOCHROME B")
        self.assertEqual(record.descriptions[3].score, 29)
        self.assertAlmostEqual(record.descriptions[3].e, 4.6)
        self.assertEqual(record.descriptions[4].title, "gi|1351589|sp|P47694|Y456_MYCGE HYPOTHETICAL PROTEIN MG456")
        self.assertEqual(record.descriptions[4].score, 29)
        self.assertAlmostEqual(record.descriptions[4].e, 6.0)
        self.assertEqual(record.descriptions[5].title, "gi|2496246|sp|Q57881|Y439_METJA HYPOTHETICAL ATP-BINDING PROTEI...")
        self.assertEqual(record.descriptions[5].score, 29)
        self.assertAlmostEqual(record.descriptions[5].e, 7.8)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Feb 2, 2000  9:39 AM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.331)
        self.assertAlmostEqual(record.ka_params[1], 0.146)
        self.assertAlmostEqual(record.ka_params[2], 0.428)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 8801581)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 320828)
        self.assertEqual(record.num_good_extends, 892)
        self.assertEqual(record.num_seqs_better_e, 6)
        self.assertEqual(record.hsps_no_gap, 3)
        self.assertEqual(record.hsps_prelim_gapped, 3)
        self.assertEqual(record.hsps_gapped, 6)
        self.assertEqual(record.query_length, 166)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 46)
        self.assertEqual(record.effective_query_length, 120)
        self.assertEqual(record.effective_database_length, 25868693)
        self.assertEqual(record.effective_search_space, 3104243160)
        self.assertEqual(record.effective_search_space_used, 3104243160)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 15)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.2)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 40)
        self.assertAlmostEqual(record.gap_trigger[1], 21.9)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 62)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.6)

    def test_text_2011L_blastp_006(self):
        "Test parsing BLASTP 2.0.11 output (text_2011L_blastp_006)"

        path = os.path.join('Blast', 'text_2011L_blastp_006.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|132349|sp|P15394|REPA_AGRTU REPLICATING PROTEIN")
        self.assertEqual(record.query_letters, 250)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 15)
        self.assertEqual(record.descriptions[0].title, "gi|132349|sp|P15394|REPA_AGRTU REPLICATING PROTEIN")
        self.assertEqual(record.descriptions[0].score, 514)
        self.assertAlmostEqual(record.descriptions[0].e, 1e-146)
        self.assertEqual(record.descriptions[1].title, "gi|123932|sp|P19922|HYIN_BRAJA INDOLEACETAMIDE HYDROLASE (IAH) ...")
        self.assertEqual(record.descriptions[1].score, 34)
        self.assertAlmostEqual(record.descriptions[1].e, 0.29)
        self.assertEqual(record.descriptions[2].title, "gi|137670|sp|P06422|VE2_HPV08 REGULATORY PROTEIN E2")
        self.assertEqual(record.descriptions[2].score, 32)
        self.assertAlmostEqual(record.descriptions[2].e, 0.86)
        self.assertEqual(record.descriptions[3].title, "gi|5921693|sp|Q05152|CCAB_RABIT VOLTAGE-DEPENDENT N-TYPE CALCIU...")
        self.assertEqual(record.descriptions[3].score, 32)
        self.assertAlmostEqual(record.descriptions[3].e, 1.5)
        self.assertEqual(record.descriptions[4].title, "gi|121952|sp|P02256|H1_PARAN HISTONE H1, GONADAL")
        self.assertEqual(record.descriptions[4].score, 31)
        self.assertAlmostEqual(record.descriptions[4].e, 2.5)
        self.assertEqual(record.descriptions[5].title, "gi|3915729|sp|P51592|HYDP_DROME HYPERPLASTIC DISCS PROTEIN (HYD...")
        self.assertEqual(record.descriptions[5].score, 31)
        self.assertAlmostEqual(record.descriptions[5].e, 3.3)
        self.assertEqual(record.descriptions[6].title, "gi|124141|sp|P08392|ICP4_HSV11 TRANS-ACTING TRANSCRIPTIONAL PRO...")
        self.assertEqual(record.descriptions[6].score, 31)
        self.assertAlmostEqual(record.descriptions[6].e, 3.3)
        self.assertEqual(record.descriptions[7].title, "gi|462182|sp|P33438|GLT_DROME GLUTACTIN PRECURSOR")
        self.assertEqual(record.descriptions[7].score, 31)
        self.assertAlmostEqual(record.descriptions[7].e, 3.3)
        self.assertEqual(record.descriptions[8].title, "gi|1708851|sp|P55268|LMB2_HUMAN LAMININ BETA-2 CHAIN PRECURSOR ...")
        self.assertEqual(record.descriptions[8].score, 30)
        self.assertAlmostEqual(record.descriptions[8].e, 4.3)
        self.assertEqual(record.descriptions[9].title, "gi|731294|sp|P39713|YAG1_YEAST HYPOTHETICAL ZINC-TYPE ALCOHOL D...")
        self.assertEqual(record.descriptions[9].score, 30)
        self.assertAlmostEqual(record.descriptions[9].e, 4.3)
        self.assertEqual(record.descriptions[10].title, "gi|2495137|sp|Q24704|H1_DROVI HISTONE H1")
        self.assertEqual(record.descriptions[10].score, 29)
        self.assertAlmostEqual(record.descriptions[10].e, 7.5)
        self.assertEqual(record.descriptions[11].title, "gi|6226905|sp|Q59567|TOP1_MYCTU DNA TOPOISOMERASE I (OMEGA-PROT...")
        self.assertEqual(record.descriptions[11].score, 29)
        self.assertAlmostEqual(record.descriptions[11].e, 9.8)
        self.assertEqual(record.descriptions[12].title, "gi|6093970|sp|Q61085|RHOP_MOUSE GTP-RHO BINDING PROTEIN 1 (RHOP...")
        self.assertEqual(record.descriptions[12].score, 29)
        self.assertAlmostEqual(record.descriptions[12].e, 9.8)
        self.assertEqual(record.descriptions[13].title, "gi|1172635|sp|P46466|PRS4_ORYSA 26S PROTEASE REGULATORY SUBUNIT...")
        self.assertEqual(record.descriptions[13].score, 29)
        self.assertAlmostEqual(record.descriptions[13].e, 9.8)
        self.assertEqual(record.descriptions[14].title, "gi|547963|sp|Q01989|MYS9_DROME MYOSIN HEAVY CHAIN 95F (95F MHC)")
        self.assertEqual(record.descriptions[14].score, 29)
        self.assertAlmostEqual(record.descriptions[14].e, 9.8)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.posted_date, [('Feb 2, 2000  9:39 AM',)])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.317)
        self.assertAlmostEqual(record.ka_params[1], 0.133)
        self.assertAlmostEqual(record.ka_params[2], 0.395)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 14679054)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 599117)
        self.assertEqual(record.num_good_extends, 1508)
        self.assertEqual(record.num_seqs_better_e, 15)
        self.assertEqual(record.hsps_no_gap, 4)
        self.assertEqual(record.hsps_prelim_gapped, 11)
        self.assertEqual(record.hsps_gapped, 15)
        self.assertEqual(record.query_length, 250)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 51)
        self.assertEqual(record.effective_query_length, 199)
        self.assertEqual(record.effective_database_length, 25457403)
        self.assertEqual(record.effective_search_space, 5066023197)
        self.assertEqual(record.effective_search_space_used, 5066023197)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 63)
        self.assertAlmostEqual(record.blast_cutoff[1], 29.0)

    def test_text_2011L_psiblast_001(self):
        "Test parsing PHI-BLAST, BLASTP 2.0.11 output, two rounds (text_2011L_psiblast_001)"

        path = os.path.join('Blast', 'text_2011L_phiblast_001.txt')
        handle = open(path)
        record = self.pb_parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1174726|sp|P12921|TMRB_BACSU TUNICAMYCIN RESISTANCE PROTEIN")
        self.assertEqual(record.query_letters, 197)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.rounds), 2)
        self.assertEqual(len(record.rounds[0].new_seqs), 4)
        self.assertEqual(record.rounds[0].new_seqs[0].title, "gi|1174726|sp|P12921|TMRB_BACSU TUNICAMYCIN RESISTANCE PROTEIN")
        self.assertEqual(record.rounds[0].new_seqs[0].score, 402)
        self.assertAlmostEqual(record.rounds[0].new_seqs[0].e, 1e-112)
        self.assertEqual(record.rounds[0].new_seqs[1].title, "gi|1352934|sp|P47169|YJ9F_YEAST HYPOTHETICAL 161.2 KD PROTEIN I...")
        self.assertEqual(record.rounds[0].new_seqs[1].score, 30)
        self.assertAlmostEqual(record.rounds[0].new_seqs[1].e, 3.3)
        self.assertEqual(record.rounds[0].new_seqs[2].title, "gi|3915741|sp|P04407|KITH_HSV23 THYMIDINE KINASE")
        self.assertEqual(record.rounds[0].new_seqs[2].score, 29)
        self.assertAlmostEqual(record.rounds[0].new_seqs[2].e, 7.4)
        self.assertEqual(record.rounds[0].new_seqs[3].title, "gi|3334489|sp|P15398|RPA1_SCHPO DNA-DIRECTED RNA POLYMERASE I 1...")
        self.assertEqual(record.rounds[0].new_seqs[3].score, 29)
        self.assertAlmostEqual(record.rounds[0].new_seqs[3].e, 7.4)
        self.assertEqual(len(record.rounds[0].alignments), 4)
        self.assertEqual(record.rounds[0].alignments[0].title, ">gi|1174726|sp|P12921|TMRB_BACSU TUNICAMYCIN RESISTANCE PROTEIN")
        self.assertEqual(record.rounds[0].alignments[0].length, 197)
        self.assertEqual(record.rounds[0].alignments[1].title, ">gi|1352934|sp|P47169|YJ9F_YEAST HYPOTHETICAL 161.2 KD PROTEIN IN NMD5-HOM6 INTERGENIC REGION")
        self.assertEqual(record.rounds[0].alignments[1].length, 1442)
        self.assertEqual(record.rounds[0].alignments[2].title, ">gi|3915741|sp|P04407|KITH_HSV23 THYMIDINE KINASE")
        self.assertEqual(record.rounds[0].alignments[2].length, 376)
        self.assertEqual(record.rounds[0].alignments[3].title, ">gi|3334489|sp|P15398|RPA1_SCHPO DNA-DIRECTED RNA POLYMERASE I 190 KD POLYPEPTIDE")
        self.assertEqual(record.rounds[0].alignments[3].length, 1689)
        self.assertEqual(len(record.rounds[1].new_seqs), 3)
        self.assertEqual(record.rounds[1].new_seqs[0].title, "gi|1352934|sp|P47169|YJ9F_YEAST HYPOTHETICAL 161.2 KD PROTEIN I...")
        self.assertEqual(record.rounds[1].new_seqs[0].score, 30)
        self.assertAlmostEqual(record.rounds[1].new_seqs[0].e, 3.3)
        self.assertEqual(record.rounds[1].new_seqs[1].title, "gi|3334489|sp|P15398|RPA1_SCHPO DNA-DIRECTED RNA POLYMERASE I 1...")
        self.assertEqual(record.rounds[1].new_seqs[1].score, 30)
        self.assertAlmostEqual(record.rounds[1].new_seqs[1].e, 3.3)
        self.assertEqual(record.rounds[1].new_seqs[2].title, "gi|3915741|sp|P04407|KITH_HSV23 THYMIDINE KINASE")
        self.assertEqual(record.rounds[1].new_seqs[2].score, 29)
        self.assertAlmostEqual(record.rounds[1].new_seqs[2].e, 7.4)
        self.assertEqual(len(record.rounds[1].alignments), 4)
        self.assertEqual(record.rounds[1].alignments[0].title, ">gi|1174726|sp|P12921|TMRB_BACSU TUNICAMYCIN RESISTANCE PROTEIN")
        self.assertEqual(record.rounds[1].alignments[0].length, 197)
        self.assertEqual(record.rounds[1].alignments[1].title, ">gi|1352934|sp|P47169|YJ9F_YEAST HYPOTHETICAL 161.2 KD PROTEIN IN NMD5-HOM6 INTERGENIC REGION")
        self.assertEqual(record.rounds[1].alignments[1].length, 1442)
        self.assertEqual(record.rounds[1].alignments[2].title, ">gi|3334489|sp|P15398|RPA1_SCHPO DNA-DIRECTED RNA POLYMERASE I 190 KD POLYPEPTIDE")
        self.assertEqual(record.rounds[1].alignments[2].length, 1689)
        self.assertEqual(record.rounds[1].alignments[3].title, ">gi|3915741|sp|P04407|KITH_HSV23 THYMIDINE KINASE")
        self.assertEqual(record.rounds[1].alignments[3].length, 376)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].score, 1021)
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].bits, 402)
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].expect, 1e-112)
        self.assertEqual(len(record.rounds[0].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].score, 66)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].bits, 30.1)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].expect, 3.3)
        self.assertEqual(len(record.rounds[0].alignments[1].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].score, 63)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].bits, 29.0)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].expect, 7.4)
        self.assertEqual(len(record.rounds[0].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].score, 63)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].bits, 29.0)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].expect, 7.4)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].score, 1031)
        self.assertAlmostEqual(record.rounds[1].alignments[0].hsps[0].bits, 406)
        self.assertAlmostEqual(record.rounds[1].alignments[0].hsps[0].expect, 1e-113)
        self.assertEqual(len(record.rounds[1].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].score, 66)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[0].bits, 30.1)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[0].expect, 3.3)
        self.assertEqual(len(record.rounds[1].alignments[1].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].score, 66)
        self.assertAlmostEqual(record.rounds[1].alignments[2].hsps[0].bits, 30.1)
        self.assertAlmostEqual(record.rounds[1].alignments[2].hsps[0].expect, 3.3)
        self.assertEqual(len(record.rounds[1].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].score, 63)
        self.assertAlmostEqual(record.rounds[1].alignments[3].hsps[0].bits, 28.9)
        self.assertAlmostEqual(record.rounds[1].alignments[3].hsps[0].expect, 7.4)
        self.assertEqual(len(record.rounds[1].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].identities, (197, 197))
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].positives, (197, 197))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].identities, (23, 70))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].positives, (35, 70))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].gaps, (10, 70))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].identities, (15, 37))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].positives, (22, 37))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].gaps, (2, 37))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].identities, (12, 38))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].positives, (20, 38))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].identities, (197, 197))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].positives, (197, 197))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].identities, (23, 70))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].positives, (35, 70))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].gaps, (10, 70))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].identities, (12, 38))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].positives, (20, 38))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].identities, (15, 37))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].positives, (22, 37))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].gaps, (2, 37))
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query, "MIIWINGAFGSGKTQTAFELHRRLNPSYVYDPEKMGFALRSMVPQEIAKDDFQSYPLWRAFNYSLLASLTDTYRGILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASKETLLKRLRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARAELPLDPDTRGSLRRFADRLMVKLNHIRIK")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].match, "MIIWINGAFGSGKTQTAFELHRRLNPSYVYDPEKMGFALRSMVPQEIAKDDFQSYPLWRAFNYSLLASLTDTYRGILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASKETLLKRLRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARAELPLDPDTRGSLRRFADRLMVKLNHIRIK")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct, "MIIWINGAFGSGKTQTAFELHRRLNPSYVYDPEKMGFALRSMVPQEIAKDDFQSYPLWRAFNYSLLASLTDTYRGILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASKETLLKRLRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARAELPLDPDTRGSLRRFADRLMVKLNHIRIK")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query_end, 197)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct_end, 197)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query, "TLMASKETLLKR--------LRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARA")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].match, "TL+  K+  L R          TR + K S AA   D+ +EGLS P    +  +D  +  ++A+ +AARA")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct, "TLLTRKDPSLSRNLKQSAGDALTRKQEKRSKAA--FDQLLEGLSGPPLHVYYASDGGNAANLAKRLAARA")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query_start, 107)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query_end, 168)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct_start, 637)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct_end, 704)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query, "IWINGAFGSGKTQTAFELHRRLNP--SYVYDPEKMGF")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].match, "++I+G  G GKT T+ +L   L P  + VY PE M +")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct, "VYIDGPHGVGKTTTSAQLMEALGPRDNIVYVPEPMTY")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query_end, 37)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct_start, 52)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct_end, 88)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query, "GILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASK")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].match, "G +++P+   HP +F+++   LR      HHF L   K")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct, "GHIVLPIPAYHPLFFSQMYNLLRSTCLYCHHFKLSKVK")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query_start, 75)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query_end, 112)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct_start, 78)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct_end, 115)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query, "MIIWINGAFGSGKTQTAFELHRRLNPSYVYDPEKMGFALRSMVPQEIAKDDFQSYPLWRAFNYSLLASLTDTYRGILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASKETLLKRLRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARAELPLDPDTRGSLRRFADRLMVKLNHIRIK")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].match, "MIIWINGAFGSGKTQTAFELHRRLNPSYVYDPEKMGFALRSMVPQEIAKDDFQSYPLWRAFNYSLLASLTDTYRGILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASKETLLKRLRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARAELPLDPDTRGSLRRFADRLMVKLNHIRIK")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct, "MIIWINGAFGSGKTQTAFELHRRLNPSYVYDPEKMGFALRSMVPQEIAKDDFQSYPLWRAFNYSLLASLTDTYRGILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASKETLLKRLRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARAELPLDPDTRGSLRRFADRLMVKLNHIRIK")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query_end, 197)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct_end, 197)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query, "TLMASKETLLKR--------LRTRAEGKNSWAAKQIDRCVEGLSSPIFEDHIQTDNLSIQDVAENIAARA")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].match, "TL+  K+  L R          TR + K S AA   D+ +EGLS P    +  +D  +  ++A+ +AARA")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct, "TLLTRKDPSLSRNLKQSAGDALTRKQEKRSKAA--FDQLLEGLSGPPLHVYYASDGGNAANLAKRLAARA")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query_start, 107)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query_end, 168)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct_start, 637)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct_end, 704)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query, "GILIVPMTIVHPEYFNEIIGRLRQEGRIVHHFTLMASK")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].match, "G +++P+   HP +F+++   LR      HHF L   K")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct, "GHIVLPIPAYHPLFFSQMYNLLRSTCLYCHHFKLSKVK")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query_start, 75)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query_end, 112)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct_start, 78)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct_end, 115)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query, "IWINGAFGSGKTQTAFELHRRLNP--SYVYDPEKMGF")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].match, "++I+G  G GKT T+ +L   L P  + VY PE M +")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct, "VYIDGPHGVGKTTTSAQLMEALGPRDNIVYVPEPMTY")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query_end, 37)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct_start, 52)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct_end, 88)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Feb 2, 2000  9:39 AM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.318)
        self.assertAlmostEqual(record.ka_params[1], 0.134)
        self.assertAlmostEqual(record.ka_params[2], 0.412)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0471)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 23669955)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 918938)
        self.assertEqual(record.num_good_extends, 2130)
        self.assertEqual(record.num_seqs_better_e, 8)
        self.assertEqual(record.hsps_no_gap, 4)
        self.assertEqual(record.hsps_prelim_gapped, 4)
        self.assertEqual(record.hsps_gapped, 8)
        self.assertEqual(record.query_length, 197)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 48)
        self.assertEqual(record.effective_query_length, 149)
        self.assertEqual(record.effective_database_length, 25704177)
        self.assertEqual(record.effective_search_space, 3829922373)
        self.assertEqual(record.effective_search_space_used, 3829922373)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 62)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.6)

    def test_text_2011L_psiblast_002(self):
        "Test parsing PHI-BLAST, BLASTP 2.0.11 output, two rounds (text_2011L_psiblast_002)"

        path = os.path.join('Blast', 'text_2011L_phiblast_002.txt')
        handle = open(path)
        record = self.pb_parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|399896|sp|Q02134|HIS7_LACLA IMIDAZOLEGLYCEROL-PHOSPHATE\nDEHYDRATASE (IGPD)")
        self.assertEqual(record.query_letters, 200)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.rounds), 2)
        self.assertEqual(len(record.rounds[0].new_seqs), 30)
        # Rest of test broken up to avoid Jython JVM limitations
        self._check_text_2011L_psiblast_002_round0(record)
        self._check_text_2011L_psiblast_002_round1(record)
        self._check_text_2011L_psiblast_002_hsps(record)
        self._check_text_2011L_psiblast_002_hsps_details(record)
        self._check_text_2011L_psiblast_002_footer(record)

    def _check_text_2011L_psiblast_002_round0(self, record):
        self.assertEqual(record.rounds[0].new_seqs[0].title, "gi|399896|sp|Q02134|HIS7_LACLA IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[0].score, 409)
        self.assertAlmostEqual(record.rounds[0].new_seqs[0].e, 1e-114)
        self.assertEqual(record.rounds[0].new_seqs[1].title, "gi|462273|sp|P34047|HIS7_ARATH IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[1].score, 198)
        self.assertAlmostEqual(record.rounds[0].new_seqs[1].e, 6e-51)
        self.assertEqual(record.rounds[0].new_seqs[2].title, "gi|2495230|sp|Q43072|HIS7_PEA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYD...")
        self.assertEqual(record.rounds[0].new_seqs[2].score, 196)
        self.assertAlmostEqual(record.rounds[0].new_seqs[2].e, 4e-50)
        self.assertEqual(record.rounds[0].new_seqs[3].title, "gi|123157|sp|P18787|HIS7_AZOBR IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[3].score, 185)
        self.assertAlmostEqual(record.rounds[0].new_seqs[3].e, 5e-47)
        self.assertEqual(record.rounds[0].new_seqs[4].title, "gi|462275|sp|P34048|HIS7_WHEAT IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[4].score, 181)
        self.assertAlmostEqual(record.rounds[0].new_seqs[4].e, 8e-46)
        self.assertEqual(record.rounds[0].new_seqs[5].title, "gi|123161|sp|P16247|HIS7_STRCO IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[5].score, 178)
        self.assertAlmostEqual(record.rounds[0].new_seqs[5].e, 7e-45)
        self.assertEqual(record.rounds[0].new_seqs[6].title, "gi|462272|sp|Q05068|HIS7_ANASP IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[6].score, 178)
        self.assertAlmostEqual(record.rounds[0].new_seqs[6].e, 7e-45)
        self.assertEqual(record.rounds[0].new_seqs[7].title, "gi|123158|sp|P06987|HIS7_ECOLI HISTIDINE BIOSYNTHESIS BIFUNCTIO...")
        self.assertEqual(record.rounds[0].new_seqs[7].score, 175)
        self.assertAlmostEqual(record.rounds[0].new_seqs[7].e, 8e-44)
        self.assertEqual(record.rounds[0].new_seqs[8].title, "gi|1346293|sp|P48054|HIS7_SYNY3 IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[8].score, 174)
        self.assertAlmostEqual(record.rounds[0].new_seqs[8].e, 1e-43)
        self.assertEqual(record.rounds[0].new_seqs[9].title, "gi|1170286|sp|P44327|HIS7_HAEIN HISTIDINE BIOSYNTHESIS BIFUNCTI...")
        self.assertEqual(record.rounds[0].new_seqs[9].score, 168)
        self.assertAlmostEqual(record.rounds[0].new_seqs[9].e, 8e-42)
        self.assertEqual(record.rounds[0].new_seqs[10].title, "gi|2495224|sp|O06590|HIS7_MYCTU IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[10].score, 167)
        self.assertAlmostEqual(record.rounds[0].new_seqs[10].e, 2e-41)
        self.assertEqual(record.rounds[0].new_seqs[11].title, "gi|123160|sp|P10368|HIS7_SALTY HISTIDINE BIOSYNTHESIS BIFUNCTIO...")
        self.assertEqual(record.rounds[0].new_seqs[11].score, 166)
        self.assertAlmostEqual(record.rounds[0].new_seqs[11].e, 2e-41)
        self.assertEqual(record.rounds[0].new_seqs[12].title, "gi|2495226|sp|Q50504|HIS7_METTH PROBABLE IMIDAZOLEGLYCEROL-PHOS...")
        self.assertEqual(record.rounds[0].new_seqs[12].score, 153)
        self.assertAlmostEqual(record.rounds[0].new_seqs[12].e, 3e-37)
        self.assertEqual(record.rounds[0].new_seqs[13].title, "gi|729718|sp|P40919|HIS7_CRYNE IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[13].score, 152)
        self.assertAlmostEqual(record.rounds[0].new_seqs[13].e, 7e-37)
        self.assertEqual(record.rounds[0].new_seqs[14].title, "gi|3334215|sp|O33773|HIS7_SULSO PROBABLE IMIDAZOLEGLYCEROL-PHOS...")
        self.assertEqual(record.rounds[0].new_seqs[14].score, 151)
        self.assertAlmostEqual(record.rounds[0].new_seqs[14].e, 9e-37)
        self.assertEqual(record.rounds[0].new_seqs[15].title, "gi|123159|sp|P28624|HIS7_PHYPR IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[15].score, 149)
        self.assertAlmostEqual(record.rounds[0].new_seqs[15].e, 3e-36)
        self.assertEqual(record.rounds[0].new_seqs[16].title, "gi|729719|sp|P40374|HIS7_SCHPO IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[16].score, 136)
        self.assertAlmostEqual(record.rounds[0].new_seqs[16].e, 3e-32)
        self.assertEqual(record.rounds[0].new_seqs[17].title, "gi|2495227|sp|P56090|HIS7_CANAL IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[17].score, 128)
        self.assertAlmostEqual(record.rounds[0].new_seqs[17].e, 9e-30)
        self.assertEqual(record.rounds[0].new_seqs[18].title, "gi|2495225|sp|Q58109|HIS7_METJA PROBABLE IMIDAZOLEGLYCEROL-PHOS...")
        self.assertEqual(record.rounds[0].new_seqs[18].score, 126)
        self.assertAlmostEqual(record.rounds[0].new_seqs[18].e, 4e-29)
        self.assertEqual(record.rounds[0].new_seqs[19].title, "gi|2495229|sp|Q92447|HIS7_PICPA IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[19].score, 125)
        self.assertAlmostEqual(record.rounds[0].new_seqs[19].e, 6e-29)
        self.assertEqual(record.rounds[0].new_seqs[20].title, "gi|399897|sp|Q02986|HIS7_SACKL IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[20].score, 125)
        self.assertAlmostEqual(record.rounds[0].new_seqs[20].e, 6e-29)
        self.assertEqual(record.rounds[0].new_seqs[21].title, "gi|2495228|sp|Q12578|HIS7_CANGA IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[21].score, 123)
        self.assertAlmostEqual(record.rounds[0].new_seqs[21].e, 2e-28)
        self.assertEqual(record.rounds[0].new_seqs[22].title, "gi|2506514|sp|P06633|HIS7_YEAST IMIDAZOLEGLYCEROL-PHOSPHATE DEH...")
        self.assertEqual(record.rounds[0].new_seqs[22].score, 122)
        self.assertAlmostEqual(record.rounds[0].new_seqs[22].e, 4e-28)
        self.assertEqual(record.rounds[0].new_seqs[23].title, "gi|462274|sp|P34041|HIS7_TRIHA IMIDAZOLEGLYCEROL-PHOSPHATE DEHY...")
        self.assertEqual(record.rounds[0].new_seqs[23].score, 106)
        self.assertAlmostEqual(record.rounds[0].new_seqs[23].e, 3e-23)
        self.assertEqual(record.rounds[0].new_seqs[24].title, "gi|1345641|sp|P49264|C7B1_THLAR CYTOCHROME P450 71B1 (CYPLXXIB1)")
        self.assertEqual(record.rounds[0].new_seqs[24].score, 35)
        self.assertAlmostEqual(record.rounds[0].new_seqs[24].e, 0.13)
        self.assertEqual(record.rounds[0].new_seqs[25].title, "gi|1731346|sp|Q10698|YY29_MYCTU PROBABLE DIPEPTIDASE CY49.29C")
        self.assertEqual(record.rounds[0].new_seqs[25].score, 32)
        self.assertAlmostEqual(record.rounds[0].new_seqs[25].e, 1.1)
        self.assertEqual(record.rounds[0].new_seqs[26].title, "gi|3287839|sp|Q01812|GLK4_RAT GLUTAMATE RECEPTOR, IONOTROPIC KA...")
        self.assertEqual(record.rounds[0].new_seqs[26].score, 30)
        self.assertAlmostEqual(record.rounds[0].new_seqs[26].e, 3.3)
        self.assertEqual(record.rounds[0].new_seqs[27].title, "gi|3123025|sp|Q94637|VIT6_OSCBR VITELLOGENIN 6 PRECURSOR")
        self.assertEqual(record.rounds[0].new_seqs[27].score, 29)
        self.assertAlmostEqual(record.rounds[0].new_seqs[27].e, 5.6)
        self.assertEqual(record.rounds[0].new_seqs[28].title, "gi|1174406|sp|P36126|SP14_YEAST PHOSPHOLIPASE D1 (PLD 1) (CHOLI...")
        self.assertEqual(record.rounds[0].new_seqs[28].score, 29)
        self.assertAlmostEqual(record.rounds[0].new_seqs[28].e, 9.7)
        self.assertEqual(record.rounds[0].new_seqs[29].title, "gi|3287848|sp|Q16099|GLK4_HUMAN GLUTAMATE RECEPTOR, IONOTROPIC ...")
        self.assertEqual(record.rounds[0].new_seqs[29].score, 29)
        self.assertAlmostEqual(record.rounds[0].new_seqs[29].e, 9.7)
        self.assertEqual(len(record.rounds[0].alignments), 30)
        self.assertEqual(record.rounds[0].alignments[0].title, ">gi|399896|sp|Q02134|HIS7_LACLA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[0].length, 200)
        self.assertEqual(record.rounds[0].alignments[1].title, ">gi|462273|sp|P34047|HIS7_ARATH IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[1].length, 270)
        self.assertEqual(record.rounds[0].alignments[2].title, ">gi|2495230|sp|Q43072|HIS7_PEA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[2].length, 281)
        self.assertEqual(record.rounds[0].alignments[3].title, ">gi|123157|sp|P18787|HIS7_AZOBR IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[3].length, 207)
        self.assertEqual(record.rounds[0].alignments[4].title, ">gi|462275|sp|P34048|HIS7_WHEAT IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[4].length, 195)
        self.assertEqual(record.rounds[0].alignments[5].title, ">gi|123161|sp|P16247|HIS7_STRCO IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[5].length, 197)
        self.assertEqual(record.rounds[0].alignments[6].title, ">gi|462272|sp|Q05068|HIS7_ANASP IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[6].length, 209)
        self.assertEqual(record.rounds[0].alignments[7].title, ">gi|123158|sp|P06987|HIS7_ECOLI HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[0].alignments[7].length, 355)
        self.assertEqual(record.rounds[0].alignments[8].title, ">gi|1346293|sp|P48054|HIS7_SYNY3 IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[8].length, 210)
        self.assertEqual(record.rounds[0].alignments[9].title, ">gi|1170286|sp|P44327|HIS7_HAEIN HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[0].alignments[9].length, 362)
        self.assertEqual(record.rounds[0].alignments[10].title, ">gi|2495224|sp|O06590|HIS7_MYCTU IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[10].length, 210)
        self.assertEqual(record.rounds[0].alignments[11].title, ">gi|123160|sp|P10368|HIS7_SALTY HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[0].alignments[11].length, 354)
        self.assertEqual(record.rounds[0].alignments[12].title, ">gi|2495226|sp|Q50504|HIS7_METTH PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[12].length, 194)
        self.assertEqual(record.rounds[0].alignments[13].title, ">gi|729718|sp|P40919|HIS7_CRYNE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[13].length, 202)
        self.assertEqual(record.rounds[0].alignments[14].title, ">gi|3334215|sp|O33773|HIS7_SULSO PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[14].length, 193)
        self.assertEqual(record.rounds[0].alignments[15].title, ">gi|123159|sp|P28624|HIS7_PHYPR IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[15].length, 452)
        self.assertEqual(record.rounds[0].alignments[16].title, ">gi|729719|sp|P40374|HIS7_SCHPO IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[16].length, 216)
        self.assertEqual(record.rounds[0].alignments[17].title, ">gi|2495227|sp|P56090|HIS7_CANAL IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[17].length, 223)
        self.assertEqual(record.rounds[0].alignments[18].title, ">gi|2495225|sp|Q58109|HIS7_METJA PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[18].length, 197)
        self.assertEqual(record.rounds[0].alignments[19].title, ">gi|2495229|sp|Q92447|HIS7_PICPA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[19].length, 224)
        self.assertEqual(record.rounds[0].alignments[20].title, ">gi|399897|sp|Q02986|HIS7_SACKL IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[20].length, 232)
        self.assertEqual(record.rounds[0].alignments[21].title, ">gi|2495228|sp|Q12578|HIS7_CANGA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[21].length, 210)
        self.assertEqual(record.rounds[0].alignments[22].title, ">gi|2506514|sp|P06633|HIS7_YEAST IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[22].length, 220)
        self.assertEqual(record.rounds[0].alignments[23].title, ">gi|462274|sp|P34041|HIS7_TRIHA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[0].alignments[23].length, 208)
        self.assertEqual(record.rounds[0].alignments[24].title, ">gi|1345641|sp|P49264|C7B1_THLAR CYTOCHROME P450 71B1 (CYPLXXIB1)")
        self.assertEqual(record.rounds[0].alignments[24].length, 496)
        self.assertEqual(record.rounds[0].alignments[25].title, ">gi|1731346|sp|Q10698|YY29_MYCTU PROBABLE DIPEPTIDASE CY49.29C")
        self.assertEqual(record.rounds[0].alignments[25].length, 375)
        self.assertEqual(record.rounds[0].alignments[26].title, ">gi|3287839|sp|Q01812|GLK4_RAT GLUTAMATE RECEPTOR, IONOTROPIC KAINATE 4 PRECURSOR (GLUTAMATE RECEPTOR KA-1) (KA1)")
        self.assertEqual(record.rounds[0].alignments[26].length, 956)
        self.assertEqual(record.rounds[0].alignments[27].title, ">gi|3123025|sp|Q94637|VIT6_OSCBR VITELLOGENIN 6 PRECURSOR")
        self.assertEqual(record.rounds[0].alignments[27].length, 1660)
        self.assertEqual(record.rounds[0].alignments[28].title, ">gi|1174406|sp|P36126|SP14_YEAST PHOSPHOLIPASE D1 (PLD 1) (CHOLINE PHOSPHATASE 1) (PHOSPHATIDYLCHOLINE-HYDROLYZING PHOSPHOLIPASE D1) (MEIOSIS-SPECIFIC SPORULATION PROTEIN SPO14)")
        self.assertEqual(record.rounds[0].alignments[28].length, 1380)
        self.assertEqual(record.rounds[0].alignments[29].title, ">gi|3287848|sp|Q16099|GLK4_HUMAN GLUTAMATE RECEPTOR, IONOTROPIC KAINATE 4 PRECURSOR (GLUTAMATE RECEPTOR KA-1) (KA1) (EXCITATORY AMINO ACID RECEPTOR 1) (EAA1)")
        self.assertEqual(record.rounds[0].alignments[29].length, 956)

    def _check_text_2011L_psiblast_002_round1(self, record):
        self.assertEqual(len(record.rounds[1].new_seqs), 2)
        self.assertEqual(record.rounds[1].new_seqs[0].title, "gi|2833252|sp|Q14571|IP3S_HUMAN INOSITOL 1,4,5-TRISPHOSPHATE-BI...")
        self.assertEqual(record.rounds[1].new_seqs[0].score, 30)
        self.assertAlmostEqual(record.rounds[1].new_seqs[0].e, 3.7)
        self.assertEqual(record.rounds[1].new_seqs[1].title, "gi|266389|sp|P29995|IP3S_RAT INOSITOL 1,4,5-TRISPHOSPHATE-BINDI...")
        self.assertEqual(record.rounds[1].new_seqs[1].score, 29)
        self.assertAlmostEqual(record.rounds[1].new_seqs[1].e, 8.2)
        self.assertEqual(len(record.rounds[1].alignments), 26)
        self.assertEqual(record.rounds[1].alignments[0].title, ">gi|2495230|sp|Q43072|HIS7_PEA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[0].length, 281)
        self.assertEqual(record.rounds[1].alignments[1].title, ">gi|462273|sp|P34047|HIS7_ARATH IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[1].length, 270)
        self.assertEqual(record.rounds[1].alignments[2].title, ">gi|399896|sp|Q02134|HIS7_LACLA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[2].length, 200)
        self.assertEqual(record.rounds[1].alignments[3].title, ">gi|1346293|sp|P48054|HIS7_SYNY3 IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[3].length, 210)
        self.assertEqual(record.rounds[1].alignments[4].title, ">gi|462272|sp|Q05068|HIS7_ANASP IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[4].length, 209)
        self.assertEqual(record.rounds[1].alignments[5].title, ">gi|462275|sp|P34048|HIS7_WHEAT IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[5].length, 195)
        self.assertEqual(record.rounds[1].alignments[6].title, ">gi|123161|sp|P16247|HIS7_STRCO IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[6].length, 197)
        self.assertEqual(record.rounds[1].alignments[7].title, ">gi|2506514|sp|P06633|HIS7_YEAST IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[7].length, 220)
        self.assertEqual(record.rounds[1].alignments[8].title, ">gi|2495227|sp|P56090|HIS7_CANAL IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[8].length, 223)
        self.assertEqual(record.rounds[1].alignments[9].title, ">gi|399897|sp|Q02986|HIS7_SACKL IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[9].length, 232)
        self.assertEqual(record.rounds[1].alignments[10].title, ">gi|2495228|sp|Q12578|HIS7_CANGA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[10].length, 210)
        self.assertEqual(record.rounds[1].alignments[11].title, ">gi|123158|sp|P06987|HIS7_ECOLI HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[1].alignments[11].length, 355)
        self.assertEqual(record.rounds[1].alignments[12].title, ">gi|123157|sp|P18787|HIS7_AZOBR IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[12].length, 207)
        self.assertEqual(record.rounds[1].alignments[13].title, ">gi|729718|sp|P40919|HIS7_CRYNE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[13].length, 202)
        self.assertEqual(record.rounds[1].alignments[14].title, ">gi|2495229|sp|Q92447|HIS7_PICPA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[14].length, 224)
        self.assertEqual(record.rounds[1].alignments[15].title, ">gi|1170286|sp|P44327|HIS7_HAEIN HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[1].alignments[15].length, 362)
        self.assertEqual(record.rounds[1].alignments[16].title, ">gi|123159|sp|P28624|HIS7_PHYPR IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[16].length, 452)
        self.assertEqual(record.rounds[1].alignments[17].title, ">gi|729719|sp|P40374|HIS7_SCHPO IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[17].length, 216)
        self.assertEqual(record.rounds[1].alignments[18].title, ">gi|123160|sp|P10368|HIS7_SALTY HISTIDINE BIOSYNTHESIS BIFUNCTIONAL PROTEIN HISB [INCLUDES: IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD); HISTIDINOL-PHOSPHATASE ]")
        self.assertEqual(record.rounds[1].alignments[18].length, 354)
        self.assertEqual(record.rounds[1].alignments[19].title, ">gi|2495224|sp|O06590|HIS7_MYCTU IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[19].length, 210)
        self.assertEqual(record.rounds[1].alignments[20].title, ">gi|2495226|sp|Q50504|HIS7_METTH PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[20].length, 194)
        self.assertEqual(record.rounds[1].alignments[21].title, ">gi|2495225|sp|Q58109|HIS7_METJA PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[21].length, 197)
        self.assertEqual(record.rounds[1].alignments[22].title, ">gi|462274|sp|P34041|HIS7_TRIHA IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[22].length, 208)
        self.assertEqual(record.rounds[1].alignments[23].title, ">gi|3334215|sp|O33773|HIS7_SULSO PROBABLE IMIDAZOLEGLYCEROL-PHOSPHATE DEHYDRATASE (IGPD)")
        self.assertEqual(record.rounds[1].alignments[23].length, 193)
        self.assertEqual(record.rounds[1].alignments[24].title, ">gi|2833252|sp|Q14571|IP3S_HUMAN INOSITOL 1,4,5-TRISPHOSPHATE-BINDING PROTEIN TYPE 2 RECEPTOR (TYPE 2 INSP3 RECEPTOR) (TYPE 2 INOSITOL 1,4,5-TRISPHOSPHATE RECEPTOR)")
        self.assertEqual(record.rounds[1].alignments[24].length, 2701)
        self.assertEqual(record.rounds[1].alignments[25].title, ">gi|266389|sp|P29995|IP3S_RAT INOSITOL 1,4,5-TRISPHOSPHATE-BINDING PROTEIN TYPE 2 RECEPTOR (TYPE 2 INSP3 RECEPTOR) (TYPE 2 INOSITOL 1,4,5-TRISPHOSPHATE RECEPTOR)")
        self.assertEqual(record.rounds[1].alignments[25].length, 2701)

    def _check_text_2011L_psiblast_002_hsps(self, record):
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].score, 1040)
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].bits, 409)
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].expect, 1e-114)
        self.assertEqual(len(record.rounds[0].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].score, 499)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].bits, 198)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].expect, 6e-51)
        self.assertEqual(len(record.rounds[0].alignments[1].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].score, 492)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].bits, 196)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].expect, 4e-50)
        self.assertEqual(len(record.rounds[0].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].score, 465)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].bits, 185)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].expect, 5e-47)
        self.assertEqual(len(record.rounds[0].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].score, 455)
        self.assertAlmostEqual(record.rounds[0].alignments[4].hsps[0].bits, 181)
        self.assertAlmostEqual(record.rounds[0].alignments[4].hsps[0].expect, 8e-46)
        self.assertEqual(len(record.rounds[0].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].score, 447)
        self.assertAlmostEqual(record.rounds[0].alignments[5].hsps[0].bits, 178)
        self.assertAlmostEqual(record.rounds[0].alignments[5].hsps[0].expect, 7e-45)
        self.assertEqual(len(record.rounds[0].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].score, 447)
        self.assertAlmostEqual(record.rounds[0].alignments[6].hsps[0].bits, 178)
        self.assertAlmostEqual(record.rounds[0].alignments[6].hsps[0].expect, 7e-45)
        self.assertEqual(len(record.rounds[0].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].score, 438)
        self.assertAlmostEqual(record.rounds[0].alignments[7].hsps[0].bits, 175)
        self.assertAlmostEqual(record.rounds[0].alignments[7].hsps[0].expect, 8e-44)
        self.assertEqual(len(record.rounds[0].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].score, 437)
        self.assertAlmostEqual(record.rounds[0].alignments[8].hsps[0].bits, 174)
        self.assertAlmostEqual(record.rounds[0].alignments[8].hsps[0].expect, 1e-43)
        self.assertEqual(len(record.rounds[0].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].score, 421)
        self.assertAlmostEqual(record.rounds[0].alignments[9].hsps[0].bits, 168)
        self.assertAlmostEqual(record.rounds[0].alignments[9].hsps[0].expect, 8e-42)
        self.assertEqual(len(record.rounds[0].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].score, 418)
        self.assertAlmostEqual(record.rounds[0].alignments[10].hsps[0].bits, 167)
        self.assertAlmostEqual(record.rounds[0].alignments[10].hsps[0].expect, 2e-41)
        self.assertEqual(len(record.rounds[0].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].score, 417)
        self.assertAlmostEqual(record.rounds[0].alignments[11].hsps[0].bits, 166)
        self.assertAlmostEqual(record.rounds[0].alignments[11].hsps[0].expect, 2e-41)
        self.assertEqual(len(record.rounds[0].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].score, 382)
        self.assertAlmostEqual(record.rounds[0].alignments[12].hsps[0].bits, 153)
        self.assertAlmostEqual(record.rounds[0].alignments[12].hsps[0].expect, 3e-37)
        self.assertEqual(len(record.rounds[0].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].score, 379)
        self.assertAlmostEqual(record.rounds[0].alignments[13].hsps[0].bits, 152)
        self.assertAlmostEqual(record.rounds[0].alignments[13].hsps[0].expect, 7e-37)
        self.assertEqual(len(record.rounds[0].alignments[13].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].score, 378)
        self.assertAlmostEqual(record.rounds[0].alignments[14].hsps[0].bits, 151)
        self.assertAlmostEqual(record.rounds[0].alignments[14].hsps[0].expect, 9e-37)
        self.assertEqual(len(record.rounds[0].alignments[14].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].score, 373)
        self.assertAlmostEqual(record.rounds[0].alignments[15].hsps[0].bits, 149)
        self.assertAlmostEqual(record.rounds[0].alignments[15].hsps[0].expect, 3e-36)
        self.assertEqual(len(record.rounds[0].alignments[15].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].score, 339)
        self.assertAlmostEqual(record.rounds[0].alignments[16].hsps[0].bits, 136)
        self.assertAlmostEqual(record.rounds[0].alignments[16].hsps[0].expect, 3e-32)
        self.assertEqual(len(record.rounds[0].alignments[16].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].score, 318)
        self.assertAlmostEqual(record.rounds[0].alignments[17].hsps[0].bits, 128)
        self.assertAlmostEqual(record.rounds[0].alignments[17].hsps[0].expect, 9e-30)
        self.assertEqual(len(record.rounds[0].alignments[17].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].score, 313)
        self.assertAlmostEqual(record.rounds[0].alignments[18].hsps[0].bits, 126)
        self.assertAlmostEqual(record.rounds[0].alignments[18].hsps[0].expect, 4e-29)
        self.assertEqual(len(record.rounds[0].alignments[18].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].score, 311)
        self.assertAlmostEqual(record.rounds[0].alignments[19].hsps[0].bits, 125)
        self.assertAlmostEqual(record.rounds[0].alignments[19].hsps[0].expect, 6e-29)
        self.assertEqual(len(record.rounds[0].alignments[19].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].score, 311)
        self.assertAlmostEqual(record.rounds[0].alignments[20].hsps[0].bits, 125)
        self.assertAlmostEqual(record.rounds[0].alignments[20].hsps[0].expect, 6e-29)
        self.assertEqual(len(record.rounds[0].alignments[20].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].score, 306)
        self.assertAlmostEqual(record.rounds[0].alignments[21].hsps[0].bits, 123)
        self.assertAlmostEqual(record.rounds[0].alignments[21].hsps[0].expect, 2e-28)
        self.assertEqual(len(record.rounds[0].alignments[21].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].score, 304)
        self.assertAlmostEqual(record.rounds[0].alignments[22].hsps[0].bits, 122)
        self.assertAlmostEqual(record.rounds[0].alignments[22].hsps[0].expect, 4e-28)
        self.assertEqual(len(record.rounds[0].alignments[22].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].score, 263)
        self.assertAlmostEqual(record.rounds[0].alignments[23].hsps[0].bits, 106)
        self.assertAlmostEqual(record.rounds[0].alignments[23].hsps[0].expect, 3e-23)
        self.assertEqual(len(record.rounds[0].alignments[23].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].score, 78)
        self.assertAlmostEqual(record.rounds[0].alignments[24].hsps[0].bits, 34.8)
        self.assertAlmostEqual(record.rounds[0].alignments[24].hsps[0].expect, 0.13)
        self.assertEqual(len(record.rounds[0].alignments[24].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].score, 70)
        self.assertAlmostEqual(record.rounds[0].alignments[25].hsps[0].bits, 31.7)
        self.assertAlmostEqual(record.rounds[0].alignments[25].hsps[0].expect, 1.1)
        self.assertEqual(len(record.rounds[0].alignments[25].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].score, 66)
        self.assertAlmostEqual(record.rounds[0].alignments[26].hsps[0].bits, 30.1)
        self.assertAlmostEqual(record.rounds[0].alignments[26].hsps[0].expect, 3.3)
        self.assertEqual(len(record.rounds[0].alignments[26].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].score, 64)
        self.assertAlmostEqual(record.rounds[0].alignments[27].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.rounds[0].alignments[27].hsps[0].expect, 5.6)
        self.assertEqual(len(record.rounds[0].alignments[27].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].score, 62)
        self.assertAlmostEqual(record.rounds[0].alignments[28].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.rounds[0].alignments[28].hsps[0].expect, 9.7)
        self.assertEqual(len(record.rounds[0].alignments[28].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].score, 62)
        self.assertAlmostEqual(record.rounds[0].alignments[29].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.rounds[0].alignments[29].hsps[0].expect, 9.7)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].score, 820)
        self.assertAlmostEqual(record.rounds[1].alignments[0].hsps[0].bits, 323)
        self.assertAlmostEqual(record.rounds[1].alignments[0].hsps[0].expect, 1e-88)
        self.assertEqual(len(record.rounds[1].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].score, 817)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[0].bits, 322)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[0].expect, 3e-88)
        self.assertEqual(len(record.rounds[1].alignments[1].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].score, 808)
        self.assertAlmostEqual(record.rounds[1].alignments[2].hsps[0].bits, 318)
        self.assertAlmostEqual(record.rounds[1].alignments[2].hsps[0].expect, 4e-87)
        self.assertEqual(len(record.rounds[1].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].score, 798)
        self.assertAlmostEqual(record.rounds[1].alignments[3].hsps[0].bits, 315)
        self.assertAlmostEqual(record.rounds[1].alignments[3].hsps[0].expect, 5e-86)
        self.assertEqual(len(record.rounds[1].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].score, 795)
        self.assertAlmostEqual(record.rounds[1].alignments[4].hsps[0].bits, 313)
        self.assertAlmostEqual(record.rounds[1].alignments[4].hsps[0].expect, 1e-85)
        self.assertEqual(len(record.rounds[1].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].score, 793)
        self.assertAlmostEqual(record.rounds[1].alignments[5].hsps[0].bits, 313)
        self.assertAlmostEqual(record.rounds[1].alignments[5].hsps[0].expect, 2e-85)
        self.assertEqual(len(record.rounds[1].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].score, 776)
        self.assertAlmostEqual(record.rounds[1].alignments[6].hsps[0].bits, 306)
        self.assertAlmostEqual(record.rounds[1].alignments[6].hsps[0].expect, 2e-83)
        self.assertEqual(len(record.rounds[1].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].score, 772)
        self.assertAlmostEqual(record.rounds[1].alignments[7].hsps[0].bits, 304)
        self.assertAlmostEqual(record.rounds[1].alignments[7].hsps[0].expect, 6e-83)
        self.assertEqual(len(record.rounds[1].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].score, 771)
        self.assertAlmostEqual(record.rounds[1].alignments[8].hsps[0].bits, 304)
        self.assertAlmostEqual(record.rounds[1].alignments[8].hsps[0].expect, 8e-83)
        self.assertEqual(len(record.rounds[1].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].score, 770)
        self.assertAlmostEqual(record.rounds[1].alignments[9].hsps[0].bits, 304)
        self.assertAlmostEqual(record.rounds[1].alignments[9].hsps[0].expect, 1e-82)
        self.assertEqual(len(record.rounds[1].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].score, 767)
        self.assertAlmostEqual(record.rounds[1].alignments[10].hsps[0].bits, 303)
        self.assertAlmostEqual(record.rounds[1].alignments[10].hsps[0].expect, 2e-82)
        self.assertEqual(len(record.rounds[1].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].score, 765)
        self.assertAlmostEqual(record.rounds[1].alignments[11].hsps[0].bits, 302)
        self.assertAlmostEqual(record.rounds[1].alignments[11].hsps[0].expect, 4e-82)
        self.assertEqual(len(record.rounds[1].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].score, 762)
        self.assertAlmostEqual(record.rounds[1].alignments[12].hsps[0].bits, 301)
        self.assertAlmostEqual(record.rounds[1].alignments[12].hsps[0].expect, 9e-82)
        self.assertEqual(len(record.rounds[1].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].score, 759)
        self.assertAlmostEqual(record.rounds[1].alignments[13].hsps[0].bits, 299)
        self.assertAlmostEqual(record.rounds[1].alignments[13].hsps[0].expect, 2e-81)
        self.assertEqual(len(record.rounds[1].alignments[13].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].score, 756)
        self.assertAlmostEqual(record.rounds[1].alignments[14].hsps[0].bits, 298)
        self.assertAlmostEqual(record.rounds[1].alignments[14].hsps[0].expect, 5e-81)
        self.assertEqual(len(record.rounds[1].alignments[14].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].score, 741)
        self.assertAlmostEqual(record.rounds[1].alignments[15].hsps[0].bits, 292)
        self.assertAlmostEqual(record.rounds[1].alignments[15].hsps[0].expect, 3e-79)
        self.assertEqual(len(record.rounds[1].alignments[15].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].score, 734)
        self.assertAlmostEqual(record.rounds[1].alignments[16].hsps[0].bits, 290)
        self.assertAlmostEqual(record.rounds[1].alignments[16].hsps[0].expect, 2e-78)
        self.assertEqual(len(record.rounds[1].alignments[16].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].score, 734)
        self.assertAlmostEqual(record.rounds[1].alignments[17].hsps[0].bits, 290)
        self.assertAlmostEqual(record.rounds[1].alignments[17].hsps[0].expect, 2e-78)
        self.assertEqual(len(record.rounds[1].alignments[17].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].score, 726)
        self.assertAlmostEqual(record.rounds[1].alignments[18].hsps[0].bits, 287)
        self.assertAlmostEqual(record.rounds[1].alignments[18].hsps[0].expect, 1e-77)
        self.assertEqual(len(record.rounds[1].alignments[18].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].score, 716)
        self.assertAlmostEqual(record.rounds[1].alignments[19].hsps[0].bits, 283)
        self.assertAlmostEqual(record.rounds[1].alignments[19].hsps[0].expect, 2e-76)
        self.assertEqual(len(record.rounds[1].alignments[19].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].score, 695)
        self.assertAlmostEqual(record.rounds[1].alignments[20].hsps[0].bits, 274)
        self.assertAlmostEqual(record.rounds[1].alignments[20].hsps[0].expect, 6e-74)
        self.assertEqual(len(record.rounds[1].alignments[20].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].score, 685)
        self.assertAlmostEqual(record.rounds[1].alignments[21].hsps[0].bits, 271)
        self.assertAlmostEqual(record.rounds[1].alignments[21].hsps[0].expect, 1e-72)
        self.assertEqual(len(record.rounds[1].alignments[21].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].score, 680)
        self.assertAlmostEqual(record.rounds[1].alignments[22].hsps[0].bits, 269)
        self.assertAlmostEqual(record.rounds[1].alignments[22].hsps[0].expect, 4e-72)
        self.assertEqual(len(record.rounds[1].alignments[22].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].score, 662)
        self.assertAlmostEqual(record.rounds[1].alignments[23].hsps[0].bits, 262)
        self.assertAlmostEqual(record.rounds[1].alignments[23].hsps[0].expect, 5e-70)
        self.assertEqual(len(record.rounds[1].alignments[23].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].score, 66)
        self.assertAlmostEqual(record.rounds[1].alignments[24].hsps[0].bits, 30.0)
        self.assertAlmostEqual(record.rounds[1].alignments[24].hsps[0].expect, 3.7)
        self.assertEqual(len(record.rounds[1].alignments[24].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].score, 63)
        self.assertAlmostEqual(record.rounds[1].alignments[25].hsps[0].bits, 28.8)
        self.assertAlmostEqual(record.rounds[1].alignments[25].hsps[0].expect, 8.2)
        self.assertEqual(len(record.rounds[1].alignments[25].hsps), 1)

    def _check_text_2011L_psiblast_002_hsps_details(self, record):
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].identities, (200, 200))
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].positives, (200, 200))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].identities, (99, 198))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].positives, (135, 198))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].identities, (96, 199))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].positives, (136, 199))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].gaps, (4, 199))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].identities, (91, 194))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].positives, (126, 194))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].gaps, (4, 194))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].identities, (93, 194))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].positives, (128, 194))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].gaps, (4, 194))
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].identities, (89, 200))
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].positives, (124, 200))
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].gaps, (3, 200))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].identities, (91, 198))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].positives, (131, 198))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].identities, (91, 198))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].positives, (130, 198))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].gaps, (9, 198))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].identities, (88, 198))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].positives, (129, 198))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].identities, (89, 198))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].positives, (127, 198))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].gaps, (9, 198))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].identities, (92, 207))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].positives, (125, 207))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].gaps, (14, 207))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].identities, (89, 198))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].positives, (129, 198))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].gaps, (10, 198))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].identities, (81, 198))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].positives, (122, 198))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].gaps, (8, 198))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].identities, (83, 203))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].positives, (120, 203))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].gaps, (11, 203))
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].identities, (88, 201))
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].positives, (128, 201))
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].gaps, (9, 201))
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].identities, (86, 198))
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].positives, (120, 198))
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].gaps, (6, 198))
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].identities, (84, 221))
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].positives, (114, 221))
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].gaps, (29, 221))
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].identities, (81, 227))
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].positives, (119, 227))
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].gaps, (33, 227))
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].identities, (80, 196))
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].positives, (107, 196))
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].gaps, (9, 196))
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].identities, (84, 223))
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].positives, (116, 223))
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].gaps, (31, 223))
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].identities, (81, 222))
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].positives, (119, 222))
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].gaps, (30, 222))
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].identities, (78, 215))
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].positives, (116, 215))
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].gaps, (24, 215))
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].identities, (79, 218))
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].positives, (114, 218))
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].gaps, (30, 218))
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].identities, (68, 202))
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].positives, (102, 202))
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].gaps, (28, 202))
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].identities, (34, 134))
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].positives, (60, 134))
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].gaps, (11, 134))
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].identities, (16, 45))
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].positives, (21, 45))
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].gaps, (3, 45))
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].identities, (17, 48))
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].positives, (24, 48))
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].gaps, (3, 48))
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].identities, (25, 70))
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].positives, (32, 70))
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].gaps, (5, 70))
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].identities, (20, 65))
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].positives, (31, 65))
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].gaps, (7, 65))
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].identities, (16, 48))
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].positives, (24, 48))
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].gaps, (3, 48))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].identities, (96, 199))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].positives, (136, 199))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].gaps, (4, 199))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].identities, (99, 198))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].positives, (135, 198))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].identities, (200, 200))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].positives, (200, 200))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].identities, (88, 198))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].positives, (129, 198))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].identities, (91, 198))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].positives, (131, 198))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].gaps, (4, 198))
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].identities, (93, 196))
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].positives, (128, 196))
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].gaps, (4, 196))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].identities, (89, 200))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].positives, (124, 200))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].gaps, (3, 200))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].identities, (78, 220))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].positives, (115, 220))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].gaps, (30, 220))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].identities, (81, 227))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].positives, (119, 227))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].gaps, (33, 227))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].identities, (79, 222))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].positives, (118, 222))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].gaps, (30, 222))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].identities, (76, 214))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].positives, (113, 214))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].gaps, (24, 214))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].identities, (91, 198))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].positives, (130, 198))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].gaps, (9, 198))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].identities, (91, 196))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].positives, (127, 196))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].gaps, (4, 196))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].identities, (83, 203))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].positives, (120, 203))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].gaps, (11, 203))
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].identities, (82, 223))
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].positives, (115, 223))
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].gaps, (31, 223))
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].identities, (89, 198))
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].positives, (127, 198))
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].gaps, (9, 198))
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].identities, (86, 199))
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].positives, (121, 199))
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].gaps, (8, 199))
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].identities, (83, 221))
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].positives, (114, 221))
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].gaps, (29, 221))
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].identities, (89, 198))
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].positives, (129, 198))
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].gaps, (10, 198))
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].identities, (92, 207))
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].positives, (124, 207))
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].gaps, (14, 207))
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].identities, (81, 198))
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].positives, (122, 198))
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].gaps, (8, 198))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].identities, (79, 196))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].positives, (106, 196))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].gaps, (9, 196))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].identities, (68, 202))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].positives, (102, 202))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].gaps, (28, 202))
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].identities, (83, 200))
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].positives, (124, 200))
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].gaps, (7, 200))
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].identities, (18, 120))
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].positives, (39, 120))
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].gaps, (19, 120))
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].identities, (18, 120))
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].positives, (37, 120))
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].gaps, (19, 120))
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].match, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].sbjct_end, 200)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].match, "RI  + R TKET + + INLDGTG AD S+GI FLDHML  L  H  FD+ +   GD   V +D HH  ED+A+A+G  + + LG + GI R+G FT P+DEAL+   LD+SGRPYL ++ ++   Q++G YDT++ E FF++L   +G+TLH+ +  G+N+HHIIE  FK+ ARAL+QA   D  + G IPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct, "RIGEVKRVTKETNVSVKINLDGTGVADSSSGIPFLDHMLDQLASHGLFDVHVRATGD---VHIDDHHTNEDIALAIGTALLKALGERKGINRFGDFTAPLDEALIHVSLDLSGRPYLGYNLEIP-TQRVGTYDTQLVEHFFQSLVNTSGMTLHIRQLAGENSHHIIEATFKAFARALRQATETDPRRGGTIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct_start, 74)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].sbjct_end, 267)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query, "TRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].match, "TR+  + R TKET + + INLDG+G AD STGI FLDHML  L  H  FD+ +   GD   V +D HH  EDVA+A+G  + + LG++ GI R+G F+ P+DEAL+   LD+SGRP+L ++ D+   Q++G YDT++ E F +++   +G+TLH+ +  G+N+HHIIE  FK+ ARAL+QA   D  + G +PSSKGVL")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct, "TRVGEVKRVTKETNVSVKINLDGSGVADSSTGIPFLDHMLDQLASHGLFDVHVKATGD---VHIDDHHTNEDVALAIGTALLQALGDRKGINRFGDFSAPLDEALIHVSLDLSGRPHLSYNLDIP-TQRVGTYDTQVVEHFLQSIVNTSGMTLHIRQLAGRNSHHIIEATFKAFARALRQATEYDPRRRGSVPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query_start, 2)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct_start, 84)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].sbjct_end, 278)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query, "ITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].match, "I RNT ET+I +++NLDGTG  D+ TG+GFLDHML  L+ HS  DL +   GD   V +D HH  E   IA+G+ +++ +G++ GI+RYG   +PMDE L    LD S RPYL++    S   K+G  DTE+  E+F+A A  AG+TLH+   YG+N HHI+E  +K+ ARAL+  + ID  K   +PS+KG L")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct, "IERNTTETRIRVAVNLDGTGVYDVKTGVGFLDHMLEQLSRHSLMDLSVAAEGD---VHIDAHHTTEHSGIAIGQAVAKAVGDRKGIQRYGHAYVPMDETLTRVALDFSNRPYLIWKVSFS-RDKIGDMDTELFREWFQAFAMAAGVTLHVECLYGENNHHIVESCYKALARALRAGIEIDPRKRDAVPSTKGTL")
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query_start, 7)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct_start, 14)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].sbjct_end, 203)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].query, "ITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].match, "+ R TKET + + INLDGTG A+ STGI FLDHML  L  H  FD+ +   GD     +D HH  ED+A+A+G  + + LG++ GI R+G FT P+DEA V   LD+SGRP+L     +   +++G YDT++ E FF++L   +G+TLH+ +  G N+HHIIE  FK+ ARAL+QA   D  + G +PSSKGVL")
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].sbjct, "VKRVTKETNVHVKINLDGTGVANSSTGIPFLDHMLDQLASHGLFDVYVKATGDTH---IDDHHSNEDIALAIGTALLQALGDRKGINRFGHFTAPLDEAAVEVILDLSGRPHLSCGLSIP-TERVGTYDTQLVEHFFQSLVNTSGMTLHIRQLAGNNSHHIIEATFKAFARALRQATEYDLRRQGTMPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].query_start, 7)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].sbjct_start, 3)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].sbjct_end, 192)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].match, "M+R+  + R TKET + + I+LDGTG+ DI+TG+GF DHML  L  H  FDL +   GD   + +D HH IED A+ALG    + LG+K+GI R+G+ T+P+DE+L    +D+SGRPYLV     +    +G YD  MT     +    A + LH++  YG+N HHI+E  FK+ ARAL+ A   D    G +PS+KG L")
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].sbjct, "MSRVGRVERTTKETSVLVEIDLDGTGKTDIATGVGFYDHMLDQLGRHGLFDLTVKTDGD---LHIDSHHTIEDTALALGAAFRQALGDKVGIYRFGNCTVPLDESLAQVTVDLSGRPYLVHTEPENMAPMIGEYDVTMTRHILESFVAQAQVALHVHVPYGRNAHHIVECQFKALARALRYASERDPRAAGILPSTKGAL")
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].sbjct_end, 197)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].match, "RI+ + R T ET +++++NLDGTG    +TGI FLDHML  ++ H   DL +   GD E   +D HH  EDV I LG+ +++ LG++ GI R+G+F  P+DEALV   LD SGRP+L +   +   +++G YDT++  EFF AL  ++ +TLH+ +  G N+HHIIE  FK+ ARA + A+ +D  + G IPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].sbjct, "RIASVHRITGETNVQVTVNLDGTGICKAATGIPFLDHMLHQISSHGLIDLDVQAKGDWE---IDDHHTNEDVGITLGQALAKALGDRKGIVRFGNFLAPLDEALVQVALDFSGRPHLSYGLQIP-TERVGTYDTQLVREFFVALVNHSQMTLHIRQLDGINSHHIIEATFKAFARAARMAIEVDPRRAGTIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].sbjct_start, 16)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].sbjct_end, 209)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].match, "R +H+ RNTKETQI++ + LD  G + I+TG+GF DHML  +  H  F ++I   GD   + +D HH +ED  +ALG+ +   LG+K GI R+G F +PMDE L  C LDISGRP+L + A+ +  Q++G   TEM E FFR+L++  G+TLHL    G+N HH +E +FK+  R L+QA+ ++      +PSSKGVL")
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].sbjct, "RYAHVVRNTKETQIDVQVWLDREGGSKINTGVGFFDHMLDQIATHGGFRMEINVKGD---LYIDDHHTVEDTGLALGEALKIALGDKRGICRFG-FVLPMDECLARCALDISGRPHLEYKAEFT-YQRVGDLSTEMIEHFFRSLSYTMGVTLHLKTK-GKNDHHRVESLFKAFGRTLRQAIRVEGD---TLPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].sbjct_start, 167)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].sbjct_end, 355)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].match, "R + + R TKET + +S+NL G+G   ++TG+ FLDHML  +  H   DL++   GD E   +D HH  EDV I LG+ ++E LG++ GI R+G F  P+DEALV   LD SGRP+L +   +   +++G YDT++  EFF A+  ++ +TLH+ +  G N+HHIIE  FK+ ARA++ A+ +D  +   IPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].sbjct, "RAAAVHRVTKETDVRVSLNLMGSGLCHVATGVPFLDHMLHQIASHGLIDLEVNATGDIE---IDDHHTNEDVGITLGQALAEALGDRRGINRFGHFIAPLDEALVQVTLDFSGRPHLSYGLQIP-TERVGTYDTQLVREFFVAVVNHSQMTLHIRQLDGINSHHIIEATFKAFARAMRMAIEVDPRRADTIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].sbjct_start, 17)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].sbjct_end, 210)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].match, "R + + R TKET I++ + LD  G  +I TG+GF DHML  +  H  F + +   GD   + +D HH +ED A+ALG+ + + +G+K GI R+G F +PMDE    C LD+SGRP++ F+A      K+G + TE+TE FF++LAF+   TLHLN   G N HH IE +FK+  R L+QA+ I+ +   E+PSSKGVL")
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].sbjct, "RFAEVIRQTKETDIKVQVWLDEAGVNEIKTGVGFFDHMLDQIATHGGFRMNVQCKGD---LWIDEHHTVEDTALALGQALKQAVGDKRGIARFG-FVLPMDECKAECALDLSGRPWIKFNACFK-RDKVGDFSTELTEHFFQSLAFSMLATLHLNV-TGNNDHHKIESLFKAFGRTLRQAIRIEGN---EMPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].sbjct_start, 174)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].sbjct_end, 362)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].query, "TRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVF--------HADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].match, "+R + I R T+E+ I + ++LDGTGQ  + TG+ F DHMLT L  H+ FDL +   GD E   ++ HH IED AIALG  + + LG+K GIRR+G   IPMDE L    +D+SGRPY V         H  ++G+     Y T +    F +LA NA I LH+   YG++ HHI E  +K+ ARAL+QAV  D  +V  +PS+KG L")
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].sbjct, "SRRARIERRTRESDIVIELDLDGTGQVAVDTGVPFYDHMLTALGSHASFDLTVRATGDVE---IEAHHTIEDTAIALGTALGQALGDKRGIRRFGDAFIPMDETLAHAAVDLSGRPYCVHTGEPDHLQHTTIAGSSV--PYHTVINRHVFESLAANARIALHVRVLYGRDPHHITEAQYKAVARALRQAVEPD-PRVSGVPSTKGAL")
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].query_start, 2)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].sbjct_start, 10)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].sbjct_end, 210)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].match, "R +H+ RNTKETQI++S+ LD  G + I+TG+GF DHML  +  H  F ++I   GD   + +D HH +ED  +AL + +   L +K GI R+G F +PMDE L  C LDISGRP+L + A+ +  Q++G   TEM E FFR+L++  G+TLHL    G+N HH +E +FK+  R ++QA+ ++      +PSSKGVL")
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].sbjct, "RYAHVVRNTKETQIDVSVWLDREGNSKINTGVGFFDHMLDQIATHGGFRMEITVKGD---LYIDDHHTVEDTGLALREALKLALRDKRGICRFG-FVLPMDECL-ACALDISGRPHLEYKAEFT-YQRVGNLSTEMIEHFFRSLSYTMGVTLHLKTK-GKNDHHRVESLFKAFGRTVRQAIRVEGD---TLPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].sbjct_start, 167)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].sbjct_end, 354)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].match, "R S  TR T ET +++ + +DG+G++ ++TG+GFLDHML  +  H   DL++   GD E   +D HH +EDVA+ LG+ + E LG+K GIRR     +PMD+AL T  LD+SGRPY V   +   +  +G   ++    F  +LA +A + +H +   G+N HH  E +FK+ A A++ AV ++    GEIPS+KG L")
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].sbjct, "RRSMKTRETLETHVKVDLEIDGSGKSSVNTGLGFLDHMLESVARHGLLDLEVEARGDLE---VDDHHTVEDVALTLGEALREALGDKSGIRRMAHAMVPMDDALATVALDLSGRPYTVLELEFD-DAVIGDVKSQNIGHFIESLAVSAAMNIHASVR-GRNDHHKAEALFKALALAIRDAVRVEH---GEIPSTKGKL")
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].sbjct_start, 5)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].sbjct_end, 194)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].query, "RISHITRNTKETQIELSINLDGTG-----QADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].match, "RI+ + R T ET I  +I+LD        + ++STGIGFLDHM T L  H    L++   GD   + +D HH  ED A+ALG+   + LG + GI+RYG    P+DE+L    +DIS RPY + H   +  +K+G   TEM     ++ AF AG+TLH++   G+N HHI E  FK+ A A++ A+S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].sbjct, "RIASVERTTSETHISCTIDLDHIPGVTEQKINVSTGIGFLDHMFTALAKHGGMSLQLQCKGD---LHIDDHHTAEDCALALGEAFKKALGERKGIKRYGYAYAPLDESLSRAVIDISSRPYFMCHLPFT-REKVGDLSTEMVSHLLQSFAFAAGVTLHIDSIRGENNHHIAESAFKALALAIRMAIS--RTGGDDVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].sbjct_start, 4)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].sbjct_end, 200)
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHML-TLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].match, "M+R ++ITR TKET+IE+ +++D  G+  +ST I F +HML TLLT+ +     I+   D   +  D HH++EDVAI LG  I   LG+K GI+R+    IPMD+ALV   LDIS R     + +L  ++ +GG  TE    FF++ A+N+GITLH+++  G NTHHIIE  FK+   AL +A  I ++   EI S+KG++")
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].sbjct, "MSRSANITRETKETKIEVLLDIDRKGEVKVSTPIPFFNHMLITLLTYMNS--TAIVSATDK--LPYDDHHIVEDVAITLGLAIKTALGDKRGIKRFSHQIIPMDDALVLVSLDISNRGMAFVNLNLKRSE-IGGLATENVPHFFQSFAYNSGITLHISQLSGYNTHHIIEASFKALGLALYEATRIVDN---EIRSTKGII")
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].sbjct_end, 193)
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].match, "R ++I+R TKET I + ++LDGTG++ +S+GIGFLDHMLT L  HS FDL++   GD     +D HH  ED A+ LG+     LG++ GI R+GS  +P+DEAL    +DIS R +   +  L     +G   +EM   FF + A  A  TLH++   G+N HH  E  FK+ A AL+ AV  D +    +PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].sbjct, "REANISRVTKETSISVKLSLDGTGKSKVSSGIGFLDHMLTALAKHSRFDLELDCKGD---TWIDDHHTTEDCALTLGEAFDVALGDRAGIARFGSACVPLDEALSRAIVDISSRAHSEINLQLV-RPSVGELSSEMITHFFESFASAALXTLHVDVLRGRNDHHRAEASFKALAVALRTAVKHDAT--AGVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].sbjct_start, 260)
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].sbjct_end, 451)
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].query, "RISHITRNTKETQIELSINLD-----------------------GTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].match, "R + + RNT ET+I ++I LD                       G     + TGIGFLDHM   L  H+ + L++   GD   + +D HH  ED AIALG    + +GN  G++R+G    P+DEAL    +D+SGRPY V    L   +K+G    EM      + +  AGITLH+   YG N HH  E  FKS A A++ A S+  S   E+PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].sbjct, "RRAFVERNTNETKISVAIALDKAPLPEESNFIDELITSKHANQKGEQVIQVDTGIGFLDHMYHALAKHAGWSLRLYSRGD---LIIDDHHTAEDTAIALGIAFKQAMGNFAGVKRFGHAYCPLDEALSRSVVDLSGRPYAVIDLGLK-REKVGELSCEMIPHLLYSFSVAAGITLHVTCLYGSNDHHRAESAFKSLAVAMRAATSLTGS--SEVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].sbjct_start, 2)
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].sbjct_end, 216)
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].query, "MTRISHITRNTKETQIELSINLDG---------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].match, "M+R + I R T ET+I++++NLDG                           +   ++ TGIGFLDHM+  L  HS + L +   GD   + +D HH  EDV I+LG    + LG   G++R+G    P+DEAL    +D+S RP+ V    L   +K+G   TEM      + A   GIT+H++   G N HH  E  FK+ A A+K+A+S  ++   +IPS+KGVL")
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].sbjct, "MSREALINRITNETKIQIALNLDGGKLELKESIFPNQSIIIDEHHAKQVSGSQYINVQTGIGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTAEDVGISLGMAFKQALGQIKGVKRFGHGFAPLDEALSRAVVDLSNRPFAVIELGLK-REKIGDLSTEMIPHVLESFAGAVGITIHVDCLRGFNDHHRAESAFKALAIAIKEAIS--KTGKNDIPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].sbjct_end, 221)
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKG")
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].match, "RI  + R TKET I L IN+DGTG+  I TGI F DH+L     H  FDL +   GD E   +D HH +EDV I LG  +++    K  I R+G   IPMD+A  T  +D+SGR Y V + +    + +G   TE    FF ++A    + +H  E  G+N HH  E +FK+   AL  A  IDE K   + S+KG")
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].sbjct, "RIFEVMRETKETNIYLKINIDGTGKYKIDTGIPFFDHLLASFAKHGCFDLIVKARGDLE---IDDHHTVEDVGICLGLALNQI--EKRNIFRFGWAIIPMDDARATVAIDLSGRSYCVGNYE-PKREFVGDLATENINHFFESVASYGMLNIHY-EVIGKNEHHKAEALFKAFGVALDLATKIDERK--GVISTKG")
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].query_end, 198)
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].sbjct_start, 7)
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].sbjct_end, 193)
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].query, "RISHITRNTKETQIELSINLDG-------------------TGQA------DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].match, "R S I R T ET+I+++++LDG                     QA       ++TGIGFLDHML  L  H  + + I   GD   + +D HH  ED  IALG    E LG+  GI+R+GS   P+DEAL    +D+S RPY V    L   +K+G    EM      + A  A +T+H++   G N HH  E  FK+ A A+K+A+S   +   +IPS+KGVL")
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].sbjct, "RSSLIKRITNETKIQIALSLDGGPVSLAQSLFKDKDYSAEHAAQATSSQFISVNTGIGFLDHMLHALAKHGGWSVIIECVGD---LHIDDHHSAEDTGIALGMAFKEALGHVRGIKRFGSGFAPLDEALSRAVIDMSNRPYAVVDLGLK-REKIGDLSCEMIPHVLESFAQGAHVTMHVDCLRGFNDHHRAESAFKALAIAIKEAIS--SNGTDDIPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].sbjct_start, 7)
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].sbjct_end, 223)
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].query, "RISHITRNTKETQIELSINLDG------------------TGQA------DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].match, "R + I+R T ET+I+++I+L+G                    QA      DI TG+GFLDHM+  L  HS + L +   GD   + +D HH  ED  IALG+   E +G   G++R+G+   P+DEAL    +D+S RP+ V    L   + +G   TEM   F  + A  A ITLH++   G N HH  E  FK+ A A+++A+S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].sbjct, "RKAFISRITNETKIQIAISLNGGYIQIKDSILPAKKDDDVASQATQSQVIDIHTGVGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTTEDCGIALGQAFKEAMGAVRGVKRFGTGFAPLDEALSRAVVDLSNRPFAVIDLGLK-REMIGDLSTEMIPHFLESFAEAARITLHVDCLRGFNDHHRSESAFKALAVAIREAIS--SNGTNDVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].sbjct_start, 16)
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].sbjct_end, 231)
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].query, "ISHITRNTKETQIELSINLDG-----------------TGQA-DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].match, "++ + R T+ET I+L+++LDG                  GQ   + TG+GFLDHMLT L  H  + L +   GD   + +D HH +ED  IALG+   E LG+  GI+R+G    P+DEAL    +D S RP+ V    L   +++G   TEM   F  + A    IT+H++   G N HH  E  FK+ A A+++A +   +   ++PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].sbjct, "MAFVKRVTQETNIQLALDLDGGSVSVRESILGKEYASGDGQTIHVHTGVGFLDHMLTALAKHGGWSLILECIGD---LHIDDHHTVEDCGIALGQAFKEALGSVRGIKRFGHGFAPLDEALSRAVVDFSNRPFAVVELGLK-RERIGQLSTEMIPHFLESFATEGRITMHVDCLRGTNDHHRSESGFKALAIAIREART--PTGRDDVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].query_start, 4)
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].sbjct_end, 209)
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].query, "ITRNTKETQIELSINLDGTGQA------------------------DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].match, "+ R T ET+I+++I+L G   A                        ++ TGIGFLDHM+  L  HS + L +   GD   + +D HH  ED  IALG+   E LG   G++R+GS   P+DEAL    +D+S RPY V    L   +K+G    EM   F  + A  + ITLH++   G+N HH  E  FK+ A A+++A S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].sbjct, "VKRITNETKIQIAISLKGGPLAIEHSIFPEKEAEAVAEQATQSQVINVHTGIGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTTEDCGIALGQAFKEALGAVRGVKRFGSGFAPLDEALSRAVVDLSNRPYAVVELGLQ-REKVGDLSCEMIPHFLESFAEASRITLHVDCLRGKNDHHRSESAFKALAVAIREATS--PNGTNDVPSTKGVL")
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].query_start, 7)
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].sbjct_start, 8)
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].sbjct_end, 219)
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].query, "RISHITRNTKETQIELSINLDG------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALK")
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].match, "R + ++R+T ET I++++++DG                        +    I+TGIGFLDHML  L  H+ + + +   GD   + +D HH  ED  IA+G   ++ LG   G+ R+G    P+DEAL    +D+S RPY V    L   +KLG    EM     ++ A  A ITLH++   G N HH  E  FK+ A A++")
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].sbjct, "RAAALSRDTNETSIQIALSIDGGELPQDADPRLLEASSAHASQTSKSQVISINTGIGFLDHMLHALAKHAGWSMALNCKGD---LHIDDHHTAEDCCIAVGTTFAKALGALTGVARFGYAYAPLDEALSRAVVDLSNRPYTVVDLGLK-REKLGELSCEMIPHCLQSFAQAARITLHVDCLRGDNDHHRAESAFKALAVAVR")
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].query_end, 180)
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].sbjct_start, 8)
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].sbjct_end, 205)
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].query, "GDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQK---LGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDES")
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].match, "G+   +G  PH  + +++   G  +S  LG+   +      T+   + L T DL+   RPY+ + A ++ N K      YD     +++R +     + L+  +   Q+  HI E    S  R  KQA S +E+")
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].sbjct, "GNLHQLGEKPHRAMVELSKTYGPLMSLKLGSVTTVVATSVETVR--DVLKTYDLECCSRPYMTYPARITYNLKDLVFSPYD-----KYWRQVRKLTVVELYTAKRV-QSFRHIREEEVASFVRFNKQAASSEET")
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].query_start, 58)
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].query_end, 188)
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].sbjct_start, 40)
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].sbjct_end, 165)
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].query, "HSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLG")
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].match, "HS+    I+G G H   G DPHH   D  +  G  +  D+G   G")
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].sbjct, "HSEVAFVIVGSGPH---GADPHHGYSDRELREGDIVVVDIGGTYG")
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].query_start, 47)
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].query_end, 91)
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].sbjct_start, 195)
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].sbjct_end, 236)
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].query, "PYLVF---HADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYG")
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].match, "PYL+    H D+ GN +  G+  +M +E    L FN  I L  +  YG")
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].sbjct, "PYLMLKGNHQDMEGNDRYEGFCVDMLKELAEILRFNYKIRLVGDGVYG")
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].query_start, 117)
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].query_end, 161)
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].sbjct_start, 427)
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].sbjct_end, 474)
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].query, "GKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLG---GYDTEMTEEFFRA")
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].match, "GK ISED     GI + G     +D   VT + D  G    +  + L  N++ G    YD E T EF RA")
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].sbjct, "GKEISEDKWEDFGISQRGEEKFFIDAEKVTVEFD--GFQAKIQMSSLYKNKQCGLCGHYDGEKTNEFRRA")
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].query_start, 79)
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].query_end, 145)
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].sbjct_start, 1436)
        self.assertEqual(record.rounds[0].alignments[27].hsps[0].sbjct_end, 1503)
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].query, "RYGSFTIPMDEAL-----VTCDLDISGRPYLVFHADLSGNQKL--GGYDTEMTEEFFRALAFNAG")
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].match, "R GS ++P++E          D DI G  Y   H D+  NQ+L  G  D  +++   +   FN+G")
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].sbjct, "RNGSNSLPLNEKSNEGESTNVDQDIEGDEYHRLHEDILKNQELDDGSLDDLLSQIIPKITNFNSG")
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].query_start, 94)
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].query_end, 151)
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].sbjct_start, 1141)
        self.assertEqual(record.rounds[0].alignments[28].hsps[0].sbjct_end, 1205)
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].query, "PYLVF---HADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYG")
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].match, "PYL+    H ++ GN +  G+  +M +E    L FN  I L  +  YG")
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].sbjct, "PYLMLKGNHQEMEGNDRYEGFCVDMLKELAEILRFNYKIRLVGDGVYG")
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].query_start, 117)
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].query_end, 161)
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].sbjct_start, 427)
        self.assertEqual(record.rounds[0].alignments[29].hsps[0].sbjct_end, 474)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query, "TRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].match, "TR+  + R TKET + + INLDG+G AD STGI FLDHML  L  H  FD+ +   GD   V +D HH  EDVA+A+G  + + LG++ GI R+G F+ P+DEAL+   LD+SGRP+L ++ D+   Q++G YDT++ E F +++   +G+TLH+ +  G+N+HHIIE  FK+ ARAL+QA   D  + G +PSSKGVL")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct, "TRVGEVKRVTKETNVSVKINLDGSGVADSSTGIPFLDHMLDQLASHGLFDVHVKATGD---VHIDDHHTNEDVALAIGTALLQALGDRKGINRFGDFSAPLDEALIHVSLDLSGRPHLSYNLDIP-TQRVGTYDTQVVEHFLQSIVNTSGMTLHIRQLAGRNSHHIIEATFKAFARALRQATEYDPRRRGSVPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query_start, 2)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct_start, 84)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].sbjct_end, 278)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].match, "RI  + R TKET + + INLDGTG AD S+GI FLDHML  L  H  FD+ +   GD   V +D HH  ED+A+A+G  + + LG + GI R+G FT P+DEAL+   LD+SGRPYL ++ ++   Q++G YDT++ E FF++L   +G+TLH+ +  G+N+HHIIE  FK+ ARAL+QA   D  + G IPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct, "RIGEVKRVTKETNVSVKINLDGTGVADSSSGIPFLDHMLDQLASHGLFDVHVRATGD---VHIDDHHTNEDIALAIGTALLKALGERKGINRFGDFTAPLDEALIHVSLDLSGRPYLGYNLEIP-TQRVGTYDTQLVEHFFQSLVNTSGMTLHIRQLAGENSHHIIEATFKAFARALRQATETDPRRGGTIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct_start, 74)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].sbjct_end, 267)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].match, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].sbjct_end, 200)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].match, "R + + R TKET + +S+NL G+G   ++TG+ FLDHML  +  H   DL++   GD E   +D HH  EDV I LG+ ++E LG++ GI R+G F  P+DEALV   LD SGRP+L +   +   +++G YDT++  EFF A+  ++ +TLH+ +  G N+HHIIE  FK+ ARA++ A+ +D  +   IPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct, "RAAAVHRVTKETDVRVSLNLMGSGLCHVATGVPFLDHMLHQIASHGLIDLEVNATGDIE---IDDHHTNEDVGITLGQALAEALGDRRGINRFGHFIAPLDEALVQVTLDFSGRPHLSYGLQIP-TERVGTYDTQLVREFFVAVVNHSQMTLHIRQLDGINSHHIIEATFKAFARAMRMAIEVDPRRADTIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct_start, 17)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].sbjct_end, 210)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].match, "RI+ + R T ET +++++NLDGTG    +TGI FLDHML  ++ H   DL +   GD E   +D HH  EDV I LG+ +++ LG++ GI R+G+F  P+DEALV   LD SGRP+L +   +   +++G YDT++  EFF AL  ++ +TLH+ +  G N+HHIIE  FK+ ARA + A+ +D  + G IPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].sbjct, "RIASVHRITGETNVQVTVNLDGTGICKAATGIPFLDHMLHQISSHGLIDLDVQAKGDWE---IDDHHTNEDVGITLGQALAKALGDRKGIVRFGNFLAPLDEALVQVALDFSGRPHLSYGLQIP-TERVGTYDTQLVREFFVALVNHSQMTLHIRQLDGINSHHIIEATFKAFARAARMAIEVDPRRAGTIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].sbjct_start, 16)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].sbjct_end, 209)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].query, "SHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].match, "  + R TKET + + INLDGTG A+ STGI FLDHML  L  H  FD+ +   GD     +D HH  ED+A+A+G  + + LG++ GI R+G FT P+DEA V   LD+SGRP+L     +   +++G YDT++ E FF++L   +G+TLH+ +  G N+HHIIE  FK+ ARAL+QA   D  + G +PSSKGVL")
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].sbjct, "GEVKRVTKETNVHVKINLDGTGVANSSTGIPFLDHMLDQLASHGLFDVYVKATGDT---HIDDHHSNEDIALAIGTALLQALGDRKGINRFGHFTAPLDEAAVEVILDLSGRPHLSCGLSIP-TERVGTYDTQLVEHFFQSLVNTSGMTLHIRQLAGNNSHHIIEATFKAFARALRQATEYDLRRQGTMPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].query_start, 5)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].sbjct_end, 192)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].match, "M+R+  + R TKET + + I+LDGTG+ DI+TG+GF DHML  L  H  FDL +   GD   + +D HH IED A+ALG    + LG+K+GI R+G+ T+P+DE+L    +D+SGRPYLV     +    +G YD  MT     +    A + LH++  YG+N HHI+E  FK+ ARAL+ A   D    G +PS+KG L")
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].sbjct, "MSRVGRVERTTKETSVLVEIDLDGTGKTDIATGVGFYDHMLDQLGRHGLFDLTVKTDGD---LHIDSHHTIEDTALALGAAFRQALGDKVGIYRFGNCTVPLDESLAQVTVDLSGRPYLVHTEPENMAPMIGEYDVTMTRHILESFVAQAQVALHVHVPYGRNAHHIVECQFKALARALRYASERDPRAAGILPSTKGAL")
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].sbjct_end, 197)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].query, "SHITRNTKETQIELSINLDG------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].match, "+ + R T ET+I+++I+L G                        +   ++ TGIGFLDHM+  L  HS + L +   GD   + +D HH  ED  IALG+   E LG   G++R+GS   P+DEAL    +D+S RPY V    L   +K+G    EM   F  + A  + ITLH++   G+N HH  E  FK+ A A+++A S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].sbjct, "ALVKRITNETKIQIAISLKGGPLAIEHSIFPEKEAEAVAEQATQSQVINVHTGIGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTTEDCGIALGQAFKEALGAVRGVKRFGSGFAPLDEALSRAVVDLSNRPYAVVELGL-QREKVGDLSCEMIPHFLESFAEASRITLHVDCLRGKNDHHRSESAFKALAVAIREATS--PNGTNDVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].query_start, 5)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].sbjct_start, 6)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].sbjct_end, 219)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].query, "MTRISHITRNTKETQIELSINLDG---------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].match, "M+R + I R T ET+I++++NLDG                           +   ++ TGIGFLDHM+  L  HS + L +   GD   + +D HH  EDV I+LG    + LG   G++R+G    P+DEAL    +D+S RP+ V    L   +K+G   TEM      + A   GIT+H++   G N HH  E  FK+ A A+K+A+S  ++   +IPS+KGVL")
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].sbjct, "MSREALINRITNETKIQIALNLDGGKLELKESIFPNQSIIIDEHHAKQVSGSQYINVQTGIGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTAEDVGISLGMAFKQALGQIKGVKRFGHGFAPLDEALSRAVVDLSNRPFAVIELGLK-REKIGDLSTEMIPHVLESFAGAVGITIHVDCLRGFNDHHRAESAFKALAIAIKEAIS--KTGKNDIPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].sbjct_end, 221)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].query, "RISHITRNTKETQIELSINLDG------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].match, "R + I+R T ET+I+++I+L+G                        +   DI TG+GFLDHM+  L  HS + L +   GD   + +D HH  ED  IALG+   E +G   G++R+G+   P+DEAL    +D+S RP+ V    L   + +G   TEM   F  + A  A ITLH++   G N HH  E  FK+ A A+++A+S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].sbjct, "RKAFISRITNETKIQIAISLNGGYIQIKDSILPAKKDDDVASQATQSQVIDIHTGVGFLDHMIHALAKHSGWSLIVECIGD---LHIDDHHTTEDCGIALGQAFKEAMGAVRGVKRFGTGFAPLDEALSRAVVDLSNRPFAVIDLGLK-REMIGDLSTEMIPHFLESFAEAARITLHVDCLRGFNDHHRSESAFKALAVAIREAIS--SNGTNDVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].sbjct_start, 16)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].sbjct_end, 231)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].query, "SHITRNTKETQIELSINLDGTGQA------------------DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].match, "+ + R T+ET I+L+++LDG   +                   + TG+GFLDHMLT L  H  + L +   GD   + +D HH +ED  IALG+   E LG+  GI+R+G    P+DEAL    +D S RP+ V    L   +++G   TEM   F  + A    IT+H++   G N HH  E  FK+ A A+++A     +   ++PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].sbjct, "AFVKRVTQETNIQLALDLDGGSVSVRESILGKEYASGDGQTIHVHTGVGFLDHMLTALAKHGGWSLILECIGD---LHIDDHHTVEDCGIALGQAFKEALGSVRGIKRFGHGFAPLDEALSRAVVDFSNRPFAVVELGLK-RERIGQLSTEMIPHFLESFATEGRITMHVDCLRGTNDHHRSESGFKALAIAIREA--RTPTGRDDVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].query_start, 5)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].sbjct_start, 2)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].sbjct_end, 209)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].match, "R +H+ RNTKETQI++ + LD  G + I+TG+GF DHML  +  H  F ++I   GD   + +D HH +ED  +ALG+ +   LG+K GI R+G F +PMDE L  C LDISGRP+L + A+ +  Q++G   TEM E FFR+L++  G+TLHL    G+N HH +E +FK+  R L+QA+ ++      +PSSKGVL")
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].sbjct, "RYAHVVRNTKETQIDVQVWLDREGGSKINTGVGFFDHMLDQIATHGGFRMEINVKGD---LYIDDHHTVEDTGLALGEALKIALGDKRGICRFG-FVLPMDECLARCALDISGRPHLEYKAEFT-YQRVGDLSTEMIEHFFRSLSYTMGVTLHLKT-KGKNDHHRVESLFKAFGRTLRQAIRVEG---DTLPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].sbjct_start, 167)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].sbjct_end, 355)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].query, "SHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].match, "+ I RNT ET+I +++NLDGTG  D+ TG+GFLDHML  L+ HS  DL +   GD   V +D HH  E   IA+G+ +++ +G++ GI+RYG   +PMDE L    LD S RPYL++    S   K+G  DTE+  E+F+A A  AG+TLH+   YG+N HHI+E  +K+ ARAL+  + ID  K   +PS+KG L")
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].sbjct, "ASIERNTTETRIRVAVNLDGTGVYDVKTGVGFLDHMLEQLSRHSLMDLSVAAEGD---VHIDAHHTTEHSGIAIGQAVAKAVGDRKGIQRYGHAYVPMDETLTRVALDFSNRPYLIWKVSFS-RDKIGDMDTELFREWFQAFAMAAGVTLHVECLYGENNHHIVESCYKALARALRAGIEIDPRKRDAVPSTKGTL")
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].query_start, 5)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].sbjct_start, 12)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].sbjct_end, 203)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].query, "RISHITRNTKETQIELSINLDG-----TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].match, "RI+ + R T ET I  +I+LD        + ++STGIGFLDHM T L  H    L++   GD   + +D HH  ED A+ALG+   + LG + GI+RYG    P+DE+L    +DIS RPY + H   +  +K+G   TEM     ++ AF AG+TLH++   G+N HHI E  FK+ A A++ A+S   +   ++PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].sbjct, "RIASVERTTSETHISCTIDLDHIPGVTEQKINVSTGIGFLDHMFTALAKHGGMSLQLQCKGD---LHIDDHHTAEDCALALGEAFKKALGERKGIKRYGYAYAPLDESLSRAVIDISSRPYFMCHLPFT-REKVGDLSTEMVSHLLQSFAFAAGVTLHIDSIRGENNHHIAESAFKALALAIRMAIS--RTGGDDVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].sbjct_start, 4)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].sbjct_end, 200)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQA-------------------------DISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].match, "R S I R T ET+I+++++LDG   +                          ++TGIGFLDHML  L  H  + + I   GD   + +D HH  ED  IALG    E LG+  GI+R+GS   P+DEAL    +D+S RPY V    L   +K+G    EM      + A  A +T+H++   G N HH  E  FK+ A A+K+A+S   +   +IPS+KGVL")
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].sbjct, "RSSLIKRITNETKIQIALSLDGGPVSLAQSLFKDKDYSAEHAAQATSSQFISVNTGIGFLDHMLHALAKHGGWSVIIECVGD---LHIDDHHSAEDTGIALGMAFKEALGHVRGIKRFGSGFAPLDEALSRAVIDMSNRPYAVVDLGLK-REKIGDLSCEMIPHVLESFAQGAHVTMHVDCLRGFNDHHRAESAFKALAIAIKEAIS--SNGTDDIPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].sbjct_start, 7)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].sbjct_end, 223)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].match, "R + + R TKET I++ + LD  G  +I TG+GF DHML  +  H  F + +   GD   + +D HH +ED A+ALG+ + + +G+K GI R+G F +PMDE    C LD+SGRP++ F+A      K+G + TE+TE FF++LAF+   TLHLN   G N HH IE +FK+  R L+QA+ I+ +   E+PSSKGVL")
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].sbjct, "RFAEVIRQTKETDIKVQVWLDEAGVNEIKTGVGFFDHMLDQIATHGGFRMNVQCKGD---LWIDEHHTVEDTALALGQALKQAVGDKRGIARFG-FVLPMDECKAECALDLSGRPWIKFNACFK-RDKVGDFSTELTEHFFQSLAFSMLATLHLNV-TGNNDHHKIESLFKAFGRTLRQAIRIEGN---EMPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].sbjct_start, 174)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].sbjct_end, 362)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGI-TLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].match, "R ++I+R TKET I + ++LDGTG++ +S+GIGFLDHMLT L  HS FDL++   GD     +D HH  ED A+ LG+     LG++ GI R+GS  +P+DEAL    +DIS R +   +  L     +G   +EM   FF + A  A + TLH++   G+N HH  E  FK+ A AL+ AV  D +    +PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].sbjct, "REANISRVTKETSISVKLSLDGTGKSKVSSGIGFLDHMLTALAKHSRFDLELDCKGDT---WIDDHHTTEDCALTLGEAFDVALGDRAGIARFGSACVPLDEALSRAIVDISSRAHSEINLQLV-RPSVGELSSEMITHFFESFASAA-LXTLHVDVLRGRNDHHRAEASFKALAVALRTAVKHDAT--AGVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].sbjct_start, 260)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].sbjct_end, 451)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].query, "RISHITRNTKETQIELSINLD-----------------------GTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].match, "R + + RNT ET+I ++I LD                       G     + TGIGFLDHM   L  H+ + L++   GD   + +D HH  ED AIALG    + +GN  G++R+G    P+DEAL    +D+SGRPY V    L   +K+G    EM      + +  AGITLH+   YG N HH  E  FKS A A++ A S+  +   E+PS+KGVL")
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].sbjct, "RRAFVERNTNETKISVAIALDKAPLPEESNFIDELITSKHANQKGEQVIQVDTGIGFLDHMYHALAKHAGWSLRLYSRGD---LIIDDHHTAEDTAIALGIAFKQAMGNFAGVKRFGHAYCPLDEALSRSVVDLSGRPYAVIDLGLK-REKVGELSCEMIPHLLYSFSVAAGITLHVTCLYGSNDHHRAESAFKSLAVAMRAATSL--TGSSEVPSTKGVL")
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].sbjct_start, 2)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].sbjct_end, 216)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].match, "R +H+ RNTKETQI++S+ LD  G + I+TG+GF DHML  +  H  F ++I   GD   + +D HH +ED  +AL + +   L +K GI R+G F +PMDE L  C LDISGRP+L + A+ +  Q++G   TEM E FFR+L++  G+TLHL    G+N HH +E +FK+  R ++QA+ ++      +PSSKGVL")
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].sbjct, "RYAHVVRNTKETQIDVSVWLDREGNSKINTGVGFFDHMLDQIATHGGFRMEITVKGD---LYIDDHHTVEDTGLALREALKLALRDKRGICRFG-FVLPMDECLA-CALDISGRPHLEYKAEFT-YQRVGNLSTEMIEHFFRSLSYTMGVTLHLKT-KGKNDHHRVESLFKAFGRTVRQAIRVEG---DTLPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].sbjct_start, 167)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].sbjct_end, 354)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].query, "TRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVF--------HADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].match, "+R + I R T+E+ I + ++LDGTGQ  + TG+ F DHMLT L  H+ FDL +   GD E   ++ HH IED AIALG  + + LG+K GIRR+G   IPMDE L    +D+SGRPY V         H  ++G+     Y T +    F +LA NA I LH+   YG++ HHI E  +K+ ARAL+QAV  D   V  +PS+KG L")
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].sbjct, "SRRARIERRTRESDIVIELDLDGTGQVAVDTGVPFYDHMLTALGSHASFDLTVRATGDVE---IEAHHTIEDTAIALGTALGQALGDKRGIRRFGDAFIPMDETLAHAAVDLSGRPYCVHTGEPDHLQHTTIAGSSV--PYHTVINRHVFESLAANARIALHVRVLYGRDPHHITEAQYKAVARALRQAVEPDPR-VSGVPSTKGAL")
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].query_start, 2)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].sbjct_start, 10)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].sbjct_end, 210)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].match, "R S  TR T ET +++ + +DG+G++ ++TG+GFLDHML  +  H   DL++   GD E   +D HH +EDVA+ LG+ + E LG+K GIRR     +PMD+AL T  LD+SGRPY V   +   +  +G   ++    F  +LA +A + +H +   G+N HH  E +FK+ A A++ AV ++    GEIPS+KG L")
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].sbjct, "RRSMKTRETLETHVKVDLEIDGSGKSSVNTGLGFLDHMLESVARHGLLDLEVEARGDLE---VDDHHTVEDVALTLGEALREALGDKSGIRRMAHAMVPMDDALATVALDLSGRPYTVLELEFD-DAVIGDVKSQNIGHFIESLAVSAAMNIHASV-RGRNDHHKAEALFKALALAIRDAVRVEH---GEIPSTKGKL")
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].sbjct_start, 5)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].sbjct_end, 194)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].query, "RISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKG")
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].match, "RI  + R TKET I L IN+DGTG+  I TGI F DH+L     H  FDL +   GD E   +D HH +EDV I LG  +++    K  I R+G   IPMD+A  T  +D+SGR Y V + +    + +G   TE    FF ++A    + +H     G+N HH  E +FK+   AL  A  IDE K   + S+KG")
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].sbjct, "RIFEVMRETKETNIYLKINIDGTGKYKIDTGIPFFDHLLASFAKHGCFDLIVKARGDLE---IDDHHTVEDVGICLGLALNQI--EKRNIFRFGWAIIPMDDARATVAIDLSGRSYCVGNYE-PKREFVGDLATENINHFFESVASYGMLNIHYEVI-GKNEHHKAEALFKAFGVALDLATKIDERK--GVISTKG")
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].query_end, 198)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].sbjct_start, 7)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].sbjct_end, 193)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].query, "RISHITRNTKETQIELSINLDG------------------------TGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALK")
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].match, "R + ++R+T ET I++++++DG                        +    I+TGIGFLDHML  L  H+ + + +   GD   + +D HH  ED  IA+G   ++ LG   G+ R+G    P+DEAL    +D+S RPY V    L   +KLG    EM     ++ A  A ITLH++   G N HH  E  FK+ A A++")
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].sbjct, "RAAALSRDTNETSIQIALSIDGGELPQDADPRLLEASSAHASQTSKSQVISINTGIGFLDHMLHALAKHAGWSMALNCKGD---LHIDDHHTAEDCCIAVGTTFAKALGALTGVARFGYAYAPLDEALSRAVVDLSNRPYTVVDLGLK-REKLGELSCEMIPHCLQSFAQAARITLHVDCLRGDNDHHRAESAFKALAVAVR")
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].query_start, 3)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].query_end, 180)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].sbjct_start, 8)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].sbjct_end, 205)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].query, "MTRISHITRNTKETQIELSINLDGTGQADISTGIGFLDHMLTLLTFHSDFDLKIIGHGDHETVGMDPHHLIEDVAIALGKCISEDLGNKLGIRRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEEFFRALAFNAGITLHLNEHYGQNTHHIIEGMFKSTARALKQAVSIDESKVGEIPSSKGVL")
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].match, "M+R ++ITR TKET+IE+ +++D  G+  +ST I F +HML  L  + +    +      + +  D HH++EDVAI LG  I   LG+K GI+R+    IPMD+ALV   LDIS R     + +L  ++ +GG  TE    FF++ A+N+GITLH+++  G NTHHIIE  FK+   AL +A  I ++   EI S+KG++")
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].sbjct, "MSRSANITRETKETKIEVLLDIDRKGEVKVSTPIPFFNHMLITLLTYMNSTAIVSAT---DKLPYDDHHIVEDVAITLGLAIKTALGDKRGIKRFSHQIIPMDDALVLVSLDISNRGMAFVNLNLKRSE-IGGLATENVPHFFQSFAYNSGITLHISQLSGYNTHHIIEASFKALGLALYEATRIVDN---EIRSTKGII")
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].query_start, 1)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].query_end, 200)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].sbjct_start, 1)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].sbjct_end, 193)
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].query, "LTLLTFHSDFDLKIIGHGDHETVGMDPH-HLIEDVAIAL------GKCISEDLG----NKLGI--------RRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEE")
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].match, "L  +  +    + +       T+ M  H ++++  A+        G    E LG    +   I                PM +A  +  +D+   P L+F        + G + +++   ")
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].sbjct, "LAEVAKNRGIAIPVDLDSQVNTLFMKSHSNMVQRAAMGWRLSARSGPRFKEALGGPAWDYRNIIEKLQDVVASLEHQFSPMMQAEFSVLVDVLYSPELLFPEGSDARIRCGAFMSKLINH")
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].query_start, 41)
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].query_end, 141)
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].sbjct_start, 1540)
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].sbjct_end, 1659)
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].query, "LTLLTFHSDFDLKIIGHGDHETVGMDPHH-LIEDVAIAL------GKCISEDLG----NKLGI--------RRYGSFTIPMDEALVTCDLDISGRPYLVFHADLSGNQKLGGYDTEMTEE")
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].match, "L  +  +    + +       T+ M  H   ++  A+        G    E LG    +   I                PM +A  +  +D+   P L+F        + G + +++   ")
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].sbjct, "LAEVAKNRGIAIPVDLDSQVNTLFMKNHSSTVQRAAMGWRLSARSGPRFKEALGGPAWDYRNIIEKLQDVVASLEQQFSPMMQAEFSVLVDVLYSPELLFPEGSDARIRCGAFMSKLINH")
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].query_start, 41)
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].query_end, 141)
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].sbjct_start, 1540)
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].sbjct_end, 1659)

    def _check_text_2011L_psiblast_002_footer(self, record):
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Feb 2, 2000  9:39 AM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.317)
        self.assertAlmostEqual(record.ka_params[1], 0.149)
        self.assertAlmostEqual(record.ka_params[2], 0.479)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0524)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 26734132)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 1229035)
        self.assertEqual(record.num_good_extends, 2616)
        self.assertEqual(record.num_seqs_better_e, 56)
        self.assertEqual(record.hsps_no_gap, 48)
        self.assertEqual(record.hsps_prelim_gapped, 8)
        self.assertEqual(record.hsps_gapped, 58)
        self.assertEqual(record.query_length, 200)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 50)
        self.assertEqual(record.effective_query_length, 150)
        self.assertEqual(record.effective_database_length, 25539661)
        self.assertEqual(record.effective_search_space, 3830949150)
        self.assertEqual(record.effective_search_space_used, 3830949150)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.5)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 63)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.8)

    def test_text_2011L_blastn_001(self):
        "Test parsing BLASTN 2.0.11 output (text_2011L_blastn_001)"

        path = os.path.join('Blast', 'text_2011L_blastn_001.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTN")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1348916|gb|G26684|G26684 human STS\nSTS_D11570.\x01gi|1375195|gb|G26945|G26945 human STS SHGC-32699.")
        self.assertEqual(record.query_letters, 285)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 23)
        self.assertEqual(record.descriptions[0].title, "gi|1348916|gb|G26684|G26684 human STS STS_D11570. >gi|1375195|g...")
        self.assertEqual(record.descriptions[0].score, 517)
        self.assertAlmostEqual(record.descriptions[0].e, 1e-146)
        self.assertEqual(record.descriptions[1].title, "gi|4516686|dbj|AU026763.1|AU026763 Rattus norvegicus, OTSUKA cl...")
        self.assertEqual(record.descriptions[1].score, 32)
        self.assertAlmostEqual(record.descriptions[1].e, 1.6)
        self.assertEqual(record.descriptions[2].title, "gi|6120827|gb|G55508.1|G55508 SHGC-100856 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[2].score, 32)
        self.assertAlmostEqual(record.descriptions[2].e, 1.6)
        self.assertEqual(record.descriptions[3].title, "gi|720683|gb|G03725|G03725 human STS WI-344.")
        self.assertEqual(record.descriptions[3].score, 32)
        self.assertAlmostEqual(record.descriptions[3].e, 1.6)
        self.assertEqual(record.descriptions[4].title, "gi|5690111|gb|G54226.1|G54226 B124N23/SP6 Human Chromosome 12 H...")
        self.assertEqual(record.descriptions[4].score, 30)
        self.assertAlmostEqual(record.descriptions[4].e, 6.5)
        self.assertEqual(record.descriptions[5].title, "gi|4493307|gb|G47007.1|G47007 Z15259_1 Zebrafish AB Danio rerio...")
        self.assertEqual(record.descriptions[5].score, 30)
        self.assertAlmostEqual(record.descriptions[5].e, 6.5)
        self.assertEqual(record.descriptions[6].title, "gi|4491799|gb|G45508.1|G45508 Z24506_1 Zebrafish AB Danio rerio...")
        self.assertEqual(record.descriptions[6].score, 30)
        self.assertAlmostEqual(record.descriptions[6].e, 6.5)
        self.assertEqual(record.descriptions[7].title, "gi|6121596|gb|G56277.1|G56277 SHGC-101791 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[7].score, 30)
        self.assertAlmostEqual(record.descriptions[7].e, 6.5)
        self.assertEqual(record.descriptions[8].title, "gi|5222417|gb|G51240.1|G51240 SHGC-80720 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[8].score, 30)
        self.assertAlmostEqual(record.descriptions[8].e, 6.5)
        self.assertEqual(record.descriptions[9].title, "gi|5221977|gb|G50800.1|G50800 SHGC-83850 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[9].score, 30)
        self.assertAlmostEqual(record.descriptions[9].e, 6.5)
        self.assertEqual(record.descriptions[10].title, "gi|5224501|gb|G53324.1|G53324 SHGC-82315 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[10].score, 30)
        self.assertAlmostEqual(record.descriptions[10].e, 6.5)
        self.assertEqual(record.descriptions[11].title, "gi|4529247|gb|G48587.1|G48587 SHGC-82546 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[11].score, 30)
        self.assertAlmostEqual(record.descriptions[11].e, 6.5)
        self.assertEqual(record.descriptions[12].title, "gi|3359917|gb|G40708|G40708 Z8947 Zebrafish AB Danio rerio STS ...")
        self.assertEqual(record.descriptions[12].score, 30)
        self.assertAlmostEqual(record.descriptions[12].e, 6.5)
        self.assertEqual(record.descriptions[13].title, "gi|3359244|gb|G40035|G40035 Z13538 Zebrafish AB Danio rerio STS...")
        self.assertEqual(record.descriptions[13].score, 30)
        self.assertAlmostEqual(record.descriptions[13].e, 6.5)
        self.assertEqual(record.descriptions[14].title, "gi|1347715|gb|G25483|G25483 human STS EST334642.")
        self.assertEqual(record.descriptions[14].score, 30)
        self.assertAlmostEqual(record.descriptions[14].e, 6.5)
        self.assertEqual(record.descriptions[15].title, "gi|1244262|gb|G19475|G19475 human STS SHGC-18755.")
        self.assertEqual(record.descriptions[15].score, 30)
        self.assertAlmostEqual(record.descriptions[15].e, 6.5)
        self.assertEqual(record.descriptions[16].title, "gi|1232611|emb|Z51311|HS302WC9 H.sapiens (D5S2069) DNA segment ...")
        self.assertEqual(record.descriptions[16].score, 30)
        self.assertAlmostEqual(record.descriptions[16].e, 6.5)
        self.assertEqual(record.descriptions[17].title, "gi|1223022|gb|G18565|G18565 BMS485 cow Bos taurus STS genomic, ...")
        self.assertEqual(record.descriptions[17].score, 30)
        self.assertAlmostEqual(record.descriptions[17].e, 6.5)
        self.assertEqual(record.descriptions[18].title, "gi|1161779|gb|G15890|G15890 human STS CHLC.UTR_01448_M84721.P56...")
        self.assertEqual(record.descriptions[18].score, 30)
        self.assertAlmostEqual(record.descriptions[18].e, 6.5)
        self.assertEqual(record.descriptions[19].title, "gi|858803|gb|G05558|G05558 human STS WI-7105.")
        self.assertEqual(record.descriptions[19].score, 30)
        self.assertAlmostEqual(record.descriptions[19].e, 6.5)
        self.assertEqual(record.descriptions[20].title, "gi|1342455|gb|G22129|G22129 human STS WI-14200.")
        self.assertEqual(record.descriptions[20].score, 30)
        self.assertAlmostEqual(record.descriptions[20].e, 6.5)
        self.assertEqual(record.descriptions[21].title, "gi|1347001|gb|G24769|G24769 human STS EST129834.")
        self.assertEqual(record.descriptions[21].score, 30)
        self.assertAlmostEqual(record.descriptions[21].e, 6.5)
        self.assertEqual(record.descriptions[22].title, "gi|605469|gb|L31223|HUMUT821B Human STS UT821, 3' primer bind.")
        self.assertEqual(record.descriptions[22].score, 30)
        self.assertAlmostEqual(record.descriptions[22].e, 6.5)
        self.assertEqual(len(record.alignments), 23)
        self.assertEqual(record.alignments[0].title, ">gi|1348916|gb|G26684|G26684 human STS STS_D11570. >gi|1375195|gb|G26945|G26945 human STS SHGC-32699.")
        self.assertEqual(record.alignments[0].length, 285)
        self.assertEqual(record.alignments[1].title, ">gi|4516686|dbj|AU026763.1|AU026763 Rattus norvegicus, OTSUKA clone, OT33.16/752f07, microsatellite sequence, sequence tagged site")
        self.assertEqual(record.alignments[1].length, 307)
        self.assertEqual(record.alignments[2].title, ">gi|6120827|gb|G55508.1|G55508 SHGC-100856 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[2].length, 711)
        self.assertEqual(record.alignments[3].title, ">gi|720683|gb|G03725|G03725 human STS WI-344.")
        self.assertEqual(record.alignments[3].length, 246)
        self.assertEqual(record.alignments[4].title, ">gi|5690111|gb|G54226.1|G54226 B124N23/SP6 Human Chromosome 12 Homo sapiens STS genomic clone RPCI-11-B124N23 SP6, sequence tagged site")
        self.assertEqual(record.alignments[4].length, 550)
        self.assertEqual(record.alignments[5].title, ">gi|4493307|gb|G47007.1|G47007 Z15259_1 Zebrafish AB Danio rerio STS genomic clone Z15259 5', sequence tagged site")
        self.assertEqual(record.alignments[5].length, 442)
        self.assertEqual(record.alignments[6].title, ">gi|4491799|gb|G45508.1|G45508 Z24506_1 Zebrafish AB Danio rerio STS genomic clone Z24506 5', sequence tagged site")
        self.assertEqual(record.alignments[6].length, 272)
        self.assertEqual(record.alignments[7].title, ">gi|6121596|gb|G56277.1|G56277 SHGC-101791 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[7].length, 641)
        self.assertEqual(record.alignments[8].title, ">gi|5222417|gb|G51240.1|G51240 SHGC-80720 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[8].length, 712)
        self.assertEqual(record.alignments[9].title, ">gi|5221977|gb|G50800.1|G50800 SHGC-83850 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[9].length, 422)
        self.assertEqual(record.alignments[10].title, ">gi|5224501|gb|G53324.1|G53324 SHGC-82315 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[10].length, 428)
        self.assertEqual(record.alignments[11].title, ">gi|4529247|gb|G48587.1|G48587 SHGC-82546 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[11].length, 694)
        self.assertEqual(record.alignments[12].title, ">gi|3359917|gb|G40708|G40708 Z8947 Zebrafish AB Danio rerio STS genomic")
        self.assertEqual(record.alignments[12].length, 549)
        self.assertEqual(record.alignments[13].title, ">gi|3359244|gb|G40035|G40035 Z13538 Zebrafish AB Danio rerio STS genomic")
        self.assertEqual(record.alignments[13].length, 536)
        self.assertEqual(record.alignments[14].title, ">gi|1347715|gb|G25483|G25483 human STS EST334642.")
        self.assertEqual(record.alignments[14].length, 407)
        self.assertEqual(record.alignments[15].title, ">gi|1244262|gb|G19475|G19475 human STS SHGC-18755.")
        self.assertEqual(record.alignments[15].length, 400)
        self.assertEqual(record.alignments[16].title, ">gi|1232611|emb|Z51311|HS302WC9 H.sapiens (D5S2069) DNA segment containing (CA) repeat; clone AFM302wc9; single read, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[16].length, 374)
        self.assertEqual(record.alignments[17].title, ">gi|1223022|gb|G18565|G18565 BMS485 cow Bos taurus STS genomic, sequence tagged site [Bos taurus]")
        self.assertEqual(record.alignments[17].length, 181)
        self.assertEqual(record.alignments[18].title, ">gi|1161779|gb|G15890|G15890 human STS CHLC.UTR_01448_M84721.P56085 clone UTR_01448_M84721.")
        self.assertEqual(record.alignments[18].length, 729)
        self.assertEqual(record.alignments[19].title, ">gi|858803|gb|G05558|G05558 human STS WI-7105.")
        self.assertEqual(record.alignments[19].length, 735)
        self.assertEqual(record.alignments[20].title, ">gi|1342455|gb|G22129|G22129 human STS WI-14200.")
        self.assertEqual(record.alignments[20].length, 373)
        self.assertEqual(record.alignments[21].title, ">gi|1347001|gb|G24769|G24769 human STS EST129834.")
        self.assertEqual(record.alignments[21].length, 306)
        self.assertEqual(record.alignments[22].title, ">gi|605469|gb|L31223|HUMUT821B Human STS UT821, 3' primer bind.")
        self.assertEqual(record.alignments[22].length, 127)
        self.assertEqual(record.alignments[0].hsps[0].score, 261)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 517)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 1e-146)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 1.6)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 1.6)
        self.assertEqual(len(record.alignments[2].hsps), 1)
        self.assertEqual(record.alignments[3].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 1.6)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[4].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[4].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[4].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[4].hsps), 1)
        self.assertEqual(record.alignments[5].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[5].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[5].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[5].hsps), 1)
        self.assertEqual(record.alignments[6].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[6].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[6].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[6].hsps), 1)
        self.assertEqual(record.alignments[7].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[7].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[7].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[7].hsps), 1)
        self.assertEqual(record.alignments[8].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[8].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[8].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[8].hsps), 1)
        self.assertEqual(record.alignments[9].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[9].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[9].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[9].hsps), 1)
        self.assertEqual(record.alignments[10].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[10].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[10].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[10].hsps), 1)
        self.assertEqual(record.alignments[11].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[11].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[11].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[11].hsps), 1)
        self.assertEqual(record.alignments[12].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[12].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[12].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[12].hsps), 1)
        self.assertEqual(record.alignments[13].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[13].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[13].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[13].hsps), 1)
        self.assertEqual(record.alignments[14].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[14].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[14].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[14].hsps), 1)
        self.assertEqual(record.alignments[15].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[15].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[15].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[15].hsps), 1)
        self.assertEqual(record.alignments[16].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[16].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[16].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[16].hsps), 1)
        self.assertEqual(record.alignments[17].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[17].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[17].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[17].hsps), 1)
        self.assertEqual(record.alignments[18].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[18].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[18].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[18].hsps), 1)
        self.assertEqual(record.alignments[19].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[19].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[19].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[19].hsps), 1)
        self.assertEqual(record.alignments[20].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[20].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[20].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[20].hsps), 1)
        self.assertEqual(record.alignments[21].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[21].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[21].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[21].hsps), 1)
        self.assertEqual(record.alignments[22].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[22].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[22].hsps[0].expect, 6.5)
        self.assertEqual(len(record.alignments[22].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (285, 285))
        self.assertEqual(record.alignments[1].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[2].hsps[0].identities, (18, 19))
        self.assertEqual(record.alignments[3].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[4].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[5].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[6].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[7].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[8].hsps[0].identities, (17, 18))
        self.assertEqual(record.alignments[9].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[10].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[11].hsps[0].identities, (17, 18))
        self.assertEqual(record.alignments[12].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[13].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[14].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[15].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[16].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[17].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[18].hsps[0].identities, (18, 19))
        self.assertEqual(record.alignments[19].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[20].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[21].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[22].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[0].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[1].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[2].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[3].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[4].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[5].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[6].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[7].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[8].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[9].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[10].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[11].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[12].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[13].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[14].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[15].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[16].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[17].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[18].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[19].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[20].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[21].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[22].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[0].hsps[0].query, "gatccctacccttnccgttggtctctntcgctgactcgaggcacctaacatccattcacacccaacacaggccagcgacttctggggctcagccacagacatggtttgtnactnttgagcttctgttcctagagaatcctagaggcttgattggcccaggctgctgtntgtnctggaggcaaagaatccctacctcctaggggtgaaaggaaatnaaaatggaaagttcttgtagcgcaaggcctgacatgggtagctgctcaataaatgctagtntgttatttc")
        self.assertEqual(record.alignments[0].hsps[0].match, "|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "gatccctacccttnccgttggtctctntcgctgactcgaggcacctaacatccattcacacccaacacaggccagcgacttctggggctcagccacagacatggtttgtnactnttgagcttctgttcctagagaatcctagaggcttgattggcccaggctgctgtntgtnctggaggcaaagaatccctacctcctaggggtgaaaggaaatnaaaatggaaagttcttgtagcgcaaggcctgacatgggtagctgctcaataaatgctagtntgttatttc")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 285)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 285)
        self.assertEqual(record.alignments[1].hsps[0].query, "ggaaagttcttgtagc")
        self.assertEqual(record.alignments[1].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "ggaaagttcttgtagc")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 221)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 236)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 32)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 47)
        self.assertEqual(record.alignments[2].hsps[0].query, "gaaatnaaaatggaaagtt")
        self.assertEqual(record.alignments[2].hsps[0].match, "||||| |||||||||||||")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "gaaataaaaatggaaagtt")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 210)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 228)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 588)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 606)
        self.assertEqual(record.alignments[3].hsps[0].query, "ctcaataaatgctagt")
        self.assertEqual(record.alignments[3].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "ctcaataaatgctagt")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 260)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 275)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 178)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 163)
        self.assertEqual(record.alignments[4].hsps[0].query, "aaaatggaaagttct")
        self.assertEqual(record.alignments[4].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[4].hsps[0].sbjct, "aaaatggaaagttct")
        self.assertEqual(record.alignments[4].hsps[0].query_start, 216)
        self.assertEqual(record.alignments[4].hsps[0].query_end, 230)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_start, 330)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_end, 344)
        self.assertEqual(record.alignments[5].hsps[0].query, "ttctgttcctagaga")
        self.assertEqual(record.alignments[5].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[5].hsps[0].sbjct, "ttctgttcctagaga")
        self.assertEqual(record.alignments[5].hsps[0].query_start, 121)
        self.assertEqual(record.alignments[5].hsps[0].query_end, 135)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_start, 384)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_end, 370)
        self.assertEqual(record.alignments[6].hsps[0].query, "ggaaagttcttgtag")
        self.assertEqual(record.alignments[6].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[6].hsps[0].sbjct, "ggaaagttcttgtag")
        self.assertEqual(record.alignments[6].hsps[0].query_start, 221)
        self.assertEqual(record.alignments[6].hsps[0].query_end, 235)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_start, 138)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_end, 124)
        self.assertEqual(record.alignments[7].hsps[0].query, "tgctcaataaatgct")
        self.assertEqual(record.alignments[7].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[7].hsps[0].sbjct, "tgctcaataaatgct")
        self.assertEqual(record.alignments[7].hsps[0].query_start, 258)
        self.assertEqual(record.alignments[7].hsps[0].query_end, 272)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_start, 216)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_end, 202)
        self.assertEqual(record.alignments[8].hsps[0].query, "taaatgctagtntgttat")
        self.assertEqual(record.alignments[8].hsps[0].match, "||||||||||| ||||||")
        self.assertEqual(record.alignments[8].hsps[0].sbjct, "taaatgctagtttgttat")
        self.assertEqual(record.alignments[8].hsps[0].query_start, 265)
        self.assertEqual(record.alignments[8].hsps[0].query_end, 282)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_start, 293)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_end, 310)
        self.assertEqual(record.alignments[9].hsps[0].query, "tgctcaataaatgct")
        self.assertEqual(record.alignments[9].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[9].hsps[0].sbjct, "tgctcaataaatgct")
        self.assertEqual(record.alignments[9].hsps[0].query_start, 258)
        self.assertEqual(record.alignments[9].hsps[0].query_end, 272)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_start, 32)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_end, 18)
        self.assertEqual(record.alignments[10].hsps[0].query, "tgctcaataaatgct")
        self.assertEqual(record.alignments[10].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[10].hsps[0].sbjct, "tgctcaataaatgct")
        self.assertEqual(record.alignments[10].hsps[0].query_start, 258)
        self.assertEqual(record.alignments[10].hsps[0].query_end, 272)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_start, 47)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_end, 33)
        self.assertEqual(record.alignments[11].hsps[0].query, "taaatgctagtntgttat")
        self.assertEqual(record.alignments[11].hsps[0].match, "||||||||||| ||||||")
        self.assertEqual(record.alignments[11].hsps[0].sbjct, "taaatgctagtttgttat")
        self.assertEqual(record.alignments[11].hsps[0].query_start, 265)
        self.assertEqual(record.alignments[11].hsps[0].query_end, 282)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_start, 292)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_end, 309)
        self.assertEqual(record.alignments[12].hsps[0].query, "aacatccattcacac")
        self.assertEqual(record.alignments[12].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[12].hsps[0].sbjct, "aacatccattcacac")
        self.assertEqual(record.alignments[12].hsps[0].query_start, 47)
        self.assertEqual(record.alignments[12].hsps[0].query_end, 61)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_start, 479)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_end, 465)
        self.assertEqual(record.alignments[13].hsps[0].query, "ttctgttcctagaga")
        self.assertEqual(record.alignments[13].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[13].hsps[0].sbjct, "ttctgttcctagaga")
        self.assertEqual(record.alignments[13].hsps[0].query_start, 121)
        self.assertEqual(record.alignments[13].hsps[0].query_end, 135)
        self.assertEqual(record.alignments[13].hsps[0].sbjct_start, 433)
        self.assertEqual(record.alignments[13].hsps[0].sbjct_end, 419)
        self.assertEqual(record.alignments[14].hsps[0].query, "ctaacatccattcac")
        self.assertEqual(record.alignments[14].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[14].hsps[0].sbjct, "ctaacatccattcac")
        self.assertEqual(record.alignments[14].hsps[0].query_start, 45)
        self.assertEqual(record.alignments[14].hsps[0].query_end, 59)
        self.assertEqual(record.alignments[14].hsps[0].sbjct_start, 389)
        self.assertEqual(record.alignments[14].hsps[0].sbjct_end, 403)
        self.assertEqual(record.alignments[15].hsps[0].query, "tgctcaataaatgct")
        self.assertEqual(record.alignments[15].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[15].hsps[0].sbjct, "tgctcaataaatgct")
        self.assertEqual(record.alignments[15].hsps[0].query_start, 258)
        self.assertEqual(record.alignments[15].hsps[0].query_end, 272)
        self.assertEqual(record.alignments[15].hsps[0].sbjct_start, 324)
        self.assertEqual(record.alignments[15].hsps[0].sbjct_end, 310)
        self.assertEqual(record.alignments[16].hsps[0].query, "acagacatggtttgt")
        self.assertEqual(record.alignments[16].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[16].hsps[0].sbjct, "acagacatggtttgt")
        self.assertEqual(record.alignments[16].hsps[0].query_start, 95)
        self.assertEqual(record.alignments[16].hsps[0].query_end, 109)
        self.assertEqual(record.alignments[16].hsps[0].sbjct_start, 246)
        self.assertEqual(record.alignments[16].hsps[0].sbjct_end, 260)
        self.assertEqual(record.alignments[17].hsps[0].query, "ctcaataaatgctag")
        self.assertEqual(record.alignments[17].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[17].hsps[0].sbjct, "ctcaataaatgctag")
        self.assertEqual(record.alignments[17].hsps[0].query_start, 260)
        self.assertEqual(record.alignments[17].hsps[0].query_end, 274)
        self.assertEqual(record.alignments[17].hsps[0].sbjct_start, 145)
        self.assertEqual(record.alignments[17].hsps[0].sbjct_end, 159)
        self.assertEqual(record.alignments[18].hsps[0].query, "gtagctgctcaataaatgc")
        self.assertEqual(record.alignments[18].hsps[0].match, "|||| ||||||||||||||")
        self.assertEqual(record.alignments[18].hsps[0].sbjct, "gtaggtgctcaataaatgc")
        self.assertEqual(record.alignments[18].hsps[0].query_start, 253)
        self.assertEqual(record.alignments[18].hsps[0].query_end, 271)
        self.assertEqual(record.alignments[18].hsps[0].sbjct_start, 698)
        self.assertEqual(record.alignments[18].hsps[0].sbjct_end, 716)
        self.assertEqual(record.alignments[19].hsps[0].query, "gaaagttcttgtagc")
        self.assertEqual(record.alignments[19].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[19].hsps[0].sbjct, "gaaagttcttgtagc")
        self.assertEqual(record.alignments[19].hsps[0].query_start, 222)
        self.assertEqual(record.alignments[19].hsps[0].query_end, 236)
        self.assertEqual(record.alignments[19].hsps[0].sbjct_start, 543)
        self.assertEqual(record.alignments[19].hsps[0].sbjct_end, 529)
        self.assertEqual(record.alignments[20].hsps[0].query, "tgctcaataaatgct")
        self.assertEqual(record.alignments[20].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[20].hsps[0].sbjct, "tgctcaataaatgct")
        self.assertEqual(record.alignments[20].hsps[0].query_start, 258)
        self.assertEqual(record.alignments[20].hsps[0].query_end, 272)
        self.assertEqual(record.alignments[20].hsps[0].sbjct_start, 33)
        self.assertEqual(record.alignments[20].hsps[0].sbjct_end, 19)
        self.assertEqual(record.alignments[21].hsps[0].query, "tggaaagttcttgta")
        self.assertEqual(record.alignments[21].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[21].hsps[0].sbjct, "tggaaagttcttgta")
        self.assertEqual(record.alignments[21].hsps[0].query_start, 220)
        self.assertEqual(record.alignments[21].hsps[0].query_end, 234)
        self.assertEqual(record.alignments[21].hsps[0].sbjct_start, 144)
        self.assertEqual(record.alignments[21].hsps[0].sbjct_end, 158)
        self.assertEqual(record.alignments[22].hsps[0].query, "acagacatggtttgt")
        self.assertEqual(record.alignments[22].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[22].hsps[0].sbjct, "acagacatggtttgt")
        self.assertEqual(record.alignments[22].hsps[0].query_start, 95)
        self.assertEqual(record.alignments[22].hsps[0].query_end, 109)
        self.assertEqual(record.alignments[22].hsps[0].sbjct_start, 106)
        self.assertEqual(record.alignments[22].hsps[0].sbjct_end, 92)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.posted_date, [('Feb 11, 2000  2:37 PM',)])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 1.37)
        self.assertAlmostEqual(record.ka_params[1], 0.711)
        self.assertAlmostEqual(record.ka_params[2], 1.31)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 1.37)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.711)
        self.assertAlmostEqual(record.ka_params_gap[2], 1.31)
        self.assertEqual(record.matrix, 'blastn matrix:1 -3')
        self.assertEqual(record.gap_penalties, [5, 2])
        self.assertEqual(record.num_hits, 3835)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 3835)
        self.assertEqual(record.num_good_extends, 1155)
        self.assertEqual(record.num_seqs_better_e, 24)
        self.assertEqual(record.query_length, 285)
        self.assertEqual(record.database_length, 31998854)
        self.assertEqual(record.effective_hsp_length, 17)
        self.assertEqual(record.effective_query_length, 268)
        self.assertEqual(record.effective_database_length, 30506390)
        self.assertEqual(record.effective_search_space, 8175712520)
        self.assertEqual(record.effective_search_space_used, 8175712520)
        self.assertEqual(record.threshold, 0)
        self.assertEqual(record.window_size, 0)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 6)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 11.9)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 10)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 19.8)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 12)
        self.assertAlmostEqual(record.gap_trigger[1], 24.3)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 15)
        self.assertAlmostEqual(record.blast_cutoff[1], 30.2)

    def test_text_2011L_blastn_002(self):
        "Test parsing BLASTN 2.0.11 output without alignments (text_2011L_blastn_002)"

        path = os.path.join('Blast', 'text_2011L_blastn_002.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTN")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1348400|gb|G26168|G26168 human STS\nEST47274.\x01gi|1375380|gb|G27130|G27130 human STS SHGC-32751.")
        self.assertEqual(record.query_letters, 434)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 19)
        self.assertEqual(record.descriptions[0].title, "gi|1348400|gb|G26168|G26168 human STS EST47274. >gi|1375380|gb|...")
        self.assertEqual(record.descriptions[0].score, 718)
        self.assertAlmostEqual(record.descriptions[0].e, 0.0)
        self.assertEqual(record.descriptions[1].title, "gi|4632200|dbj|AU047565.1|AU047565 Rattus norvegicus, OTSUKA cl...")
        self.assertEqual(record.descriptions[1].score, 34)
        self.assertAlmostEqual(record.descriptions[1].e, 0.65)
        self.assertEqual(record.descriptions[2].title, "gi|6121436|gb|G56117.1|G56117 SHGC-101583 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[2].score, 34)
        self.assertAlmostEqual(record.descriptions[2].e, 0.65)
        self.assertEqual(record.descriptions[3].title, "gi|3249175|gb|G38401|G38401 SHGC-57345 Human Homo sapiens STS g...")
        self.assertEqual(record.descriptions[3].score, 34)
        self.assertAlmostEqual(record.descriptions[3].e, 0.65)
        self.assertEqual(record.descriptions[4].title, "gi|720383|gb|G03425|G03425 human STS WI-5868.")
        self.assertEqual(record.descriptions[4].score, 34)
        self.assertAlmostEqual(record.descriptions[4].e, 0.65)
        self.assertEqual(record.descriptions[5].title, "gi|605557|gb|L31312|HUMUT937B Human STS UT937, 3' primer bind.")
        self.assertEqual(record.descriptions[5].score, 34)
        self.assertAlmostEqual(record.descriptions[5].e, 0.65)
        self.assertEqual(record.descriptions[6].title, "gi|6123581|gb|G58262.1|G58262 SHGC-104352 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[6].score, 32)
        self.assertAlmostEqual(record.descriptions[6].e, 2.6)
        self.assertEqual(record.descriptions[7].title, "gi|6122805|gb|G57486.1|G57486 SHGC-103345 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[7].score, 32)
        self.assertAlmostEqual(record.descriptions[7].e, 2.6)
        self.assertEqual(record.descriptions[8].title, "gi|6121347|gb|G56178.1|G56178 SHGC-101470 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[8].score, 32)
        self.assertAlmostEqual(record.descriptions[8].e, 2.6)
        self.assertEqual(record.descriptions[9].title, "gi|3893806|emb|AL034156|HSPE59A01 H.sapiens flow-sorted chromos...")
        self.assertEqual(record.descriptions[9].score, 32)
        self.assertAlmostEqual(record.descriptions[9].e, 2.6)
        self.assertEqual(record.descriptions[10].title, "gi|5224295|gb|G52968.1|G52968 SHGC-86325 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[10].score, 32)
        self.assertAlmostEqual(record.descriptions[10].e, 2.6)
        self.assertEqual(record.descriptions[11].title, "gi|1348143|gb|G25911|G25911 human STS EST349382.")
        self.assertEqual(record.descriptions[11].score, 32)
        self.assertAlmostEqual(record.descriptions[11].e, 2.6)
        self.assertEqual(record.descriptions[12].title, "gi|1233216|emb|Z51916|HSA082WB5 H.sapiens (D1S2890) DNA segment...")
        self.assertEqual(record.descriptions[12].score, 32)
        self.assertAlmostEqual(record.descriptions[12].e, 2.6)
        self.assertEqual(record.descriptions[13].title, "gi|1232198|emb|Z50898|HS038XD8 H.sapiens (D18S1106) DNA segment...")
        self.assertEqual(record.descriptions[13].score, 32)
        self.assertAlmostEqual(record.descriptions[13].e, 2.6)
        self.assertEqual(record.descriptions[14].title, "gi|1161890|gb|G16001|G16001 human STS CHLC.GCT8D08.P11278 clone...")
        self.assertEqual(record.descriptions[14].score, 32)
        self.assertAlmostEqual(record.descriptions[14].e, 2.6)
        self.assertEqual(record.descriptions[15].title, "gi|1130137|gb|G14398|G14398 human STS SHGC-9310 clone pG-5191.")
        self.assertEqual(record.descriptions[15].score, 32)
        self.assertAlmostEqual(record.descriptions[15].e, 2.6)
        self.assertEqual(record.descriptions[16].title, "gi|1017612|gb|G11520|G11520 human STS SHGC-14676.")
        self.assertEqual(record.descriptions[16].score, 32)
        self.assertAlmostEqual(record.descriptions[16].e, 2.6)
        self.assertEqual(record.descriptions[17].title, "gi|1396897|gb|G28178|G28178 human STS SHGC-34170.")
        self.assertEqual(record.descriptions[17].score, 32)
        self.assertAlmostEqual(record.descriptions[17].e, 2.6)
        self.assertEqual(record.descriptions[18].title, "gi|859592|gb|G06347|G06347 human STS WI-7005.")
        self.assertEqual(record.descriptions[18].score, 32)
        self.assertAlmostEqual(record.descriptions[18].e, 2.6)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(record.posted_date, [('Feb 11, 2000  2:37 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 1.37)
        self.assertAlmostEqual(record.ka_params[1], 0.711)
        self.assertAlmostEqual(record.ka_params[2], 1.31)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 1.37)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.711)
        self.assertAlmostEqual(record.ka_params_gap[2], 1.31)
        self.assertEqual(record.matrix, 'blastn matrix:1 -3')
        self.assertEqual(record.gap_penalties, [5, 2])
        self.assertEqual(record.num_hits, 8762)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 8762)
        self.assertEqual(record.num_good_extends, 2655)
        self.assertEqual(record.num_seqs_better_e, 27)
        self.assertEqual(record.query_length, 434)
        self.assertEqual(record.database_length, 31998854)
        self.assertEqual(record.effective_hsp_length, 17)
        self.assertEqual(record.effective_query_length, 417)
        self.assertEqual(record.effective_database_length, 30506390)
        self.assertEqual(record.effective_search_space, 12721164630)
        self.assertEqual(record.effective_search_space_used, 12721164630)
        self.assertEqual(record.threshold, 0)
        self.assertEqual(record.window_size, 0)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 6)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 11.9)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 10)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 19.8)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 12)
        self.assertAlmostEqual(record.gap_trigger[1], 24.3)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 16)
        self.assertAlmostEqual(record.blast_cutoff[1], 32.2)

    def test_text_2011L_blastn_003(self):
        "Test parsing BLASTN 2.0.11 output without descriptions (text_2011L_blastn_003)"

        path = os.path.join('Blast', 'text_2011L_blastn_003.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTN")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1347201|gb|G24969|G24969 human STS\nEST21946.\x01gi|1375315|gb|G27065|G27065 human STS SHGC-31731.")
        self.assertEqual(record.query_letters, 331)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 45)
        self.assertEqual(record.alignments[0].title, ">gi|1347201|gb|G24969|G24969 human STS EST21946. >gi|1375315|gb|G27065|G27065 human STS SHGC-31731.")
        self.assertEqual(record.alignments[0].length, 331)
        self.assertEqual(record.alignments[1].title, ">gi|2665277|emb|AL010115|HSPE77H4 H.sapiens flow-sorted chromosome 1 HindIII fragment, SC1pE77H4, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[1].length, 554)
        self.assertEqual(record.alignments[2].title, ">gi|4757440|gb|G49267.1|G49267 stbK343C1_96809 chromosome 22 genomic clone Homo sapiens STS genomic clone 343C1, sequence tagged site")
        self.assertEqual(record.alignments[2].length, 360)
        self.assertEqual(record.alignments[3].title, ">gi|4493602|gb|G47248.1|G47248 Z17392_1 Zebrafish AB Danio rerio STS genomic clone Z17392 5', sequence tagged site")
        self.assertEqual(record.alignments[3].length, 454)
        self.assertEqual(record.alignments[4].title, ">gi|6120731|gb|G55412.1|G55412 SHGC-100745 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[4].length, 652)
        self.assertEqual(record.alignments[5].title, ">gi|1235411|emb|Z53965|HSC009WH1 H.sapiens (D2S2321) DNA segment containing (CA) repeat; clone AFMc009wh1; single read, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[5].length, 382)
        self.assertEqual(record.alignments[6].title, ">gi|939357|gb|G08807|G08807 human STS CHLC.GATA70E11.P18111 clone GATA70E11")
        self.assertEqual(record.alignments[6].length, 643)
        self.assertEqual(record.alignments[7].title, ">gi|1342139|gb|G21813|G21813 human STS WI-12408.")
        self.assertEqual(record.alignments[7].length, 418)
        self.assertEqual(record.alignments[8].title, ">gi|719782|gb|G02824|G02824 human STS WI-1312.")
        self.assertEqual(record.alignments[8].length, 349)
        self.assertEqual(record.alignments[9].title, ">gi|5902652|gb|G54536.1|G54536 Xq4072 KWOK Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[9].length, 997)
        self.assertEqual(record.alignments[10].title, ">gi|5566455|gb|AF167528.1|AF167528 Mus musculus chromosome 10 STS D10Jhu41, sequence tagged site")
        self.assertEqual(record.alignments[10].length, 749)
        self.assertEqual(record.alignments[11].title, ">gi|5566418|gb|AF167504.1|AF167504 Mus musculus chromosome 10 STS D10Jhu59, sequence tagged site")
        self.assertEqual(record.alignments[11].length, 550)
        self.assertEqual(record.alignments[12].title, ">gi|5565757|gb|AF096569.1|AF096569 Rattus norvegicus clone D5Uwm59, sequence tagged site")
        self.assertEqual(record.alignments[12].length, 580)
        self.assertEqual(record.alignments[13].title, ">gi|4757436|gb|G49263.1|G49263 stbK343C1_70109 chromosome 22 genomic clone Homo sapiens STS genomic clone 343C1, sequence tagged site")
        self.assertEqual(record.alignments[13].length, 359)
        self.assertEqual(record.alignments[14].title, ">gi|4518571|dbj|AU028648.1|AU028648 Rattus norvegicus, OTSUKA clone, OT10.41/02694, microsatellite sequence, sequence tagged site")
        self.assertEqual(record.alignments[14].length, 465)
        self.assertEqual(record.alignments[15].title, ">gi|6123014|gb|G57845.1|G57845 SHGC-103599 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[15].length, 680)
        self.assertEqual(record.alignments[16].title, ">gi|6124406|gb|G59237.1|G59237 SHGC-110189 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[16].length, 505)
        self.assertEqual(record.alignments[17].title, ">gi|6121931|gb|G56612.1|G56612 SHGC-102181 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[17].length, 489)
        self.assertEqual(record.alignments[18].title, ">gi|3599905|gb|G41846|G41846 Z1061 Zebrafish AB Danio rerio STS genomic, sequence tagged site [Danio rerio]")
        self.assertEqual(record.alignments[18].length, 410)
        self.assertEqual(record.alignments[19].title, ">gi|5224504|gb|G53327.1|G53327 SHGC-82321 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[19].length, 421)
        self.assertEqual(record.alignments[20].title, ">gi|5221389|gb|G50212.1|G50212 SHGC-82917 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[20].length, 297)
        self.assertEqual(record.alignments[21].title, ">gi|3168757|gb|G38183|G38183 RPCI-6-164E4Sp6 Human Homo sapiens STS genomic")
        self.assertEqual(record.alignments[21].length, 395)
        self.assertEqual(record.alignments[22].title, ">gi|3150191|emb|AL023622|HS863J11T H.sapiens STS from genomic clone 863J11, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[22].length, 558)
        self.assertEqual(record.alignments[23].title, ">gi|3123402|emb|AL023351|DM171A7S Drosophila melanogaster STS determined from European Mapping Project cosmid, sequence tagged site [Drosophila melanogaster]")
        self.assertEqual(record.alignments[23].length, 600)
        self.assertEqual(record.alignments[24].title, ">gi|2665031|emb|AL009868|HSPE36B05 H.sapiens flow-sorted chromosome 1 HindIII fragment, SC1pE36B05, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[24].length, 397)
        self.assertEqual(record.alignments[25].title, ">gi|2641969|dbj|AB004264|AB004264 Mus spretus genomic DNA, RLGS spot, D5Rik122, sequence tagged site [Mus spretus]")
        self.assertEqual(record.alignments[25].length, 237)
        self.assertEqual(record.alignments[26].title, ">gi|4529138|gb|G48478.1|G48478 SHGC-68947 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[26].length, 415)
        self.assertEqual(record.alignments[27].title, ">gi|2484048|gb|G36284|G36284   STS h14a2375 5")
        self.assertEqual(record.alignments[27].length, 305)
        self.assertEqual(record.alignments[28].title, ">gi|1871265|gb|G31295|G31295 sy625c11-R Human (A.Gnirke) Homo sapiens STS genomic, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[28].length, 279)
        self.assertEqual(record.alignments[29].title, ">gi|1263801|emb|Z70926|DM63D12T D. melanogaster STS determined from European Mapping Project cosmid, sequence tagged site [Drosophila melanogaster]")
        self.assertEqual(record.alignments[29].length, 350)
        self.assertEqual(record.alignments[30].title, ">gi|1244248|gb|G19461|G19461 human STS SHGC-11979.")
        self.assertEqual(record.alignments[30].length, 400)
        self.assertEqual(record.alignments[31].title, ">gi|1233253|emb|Z51953|HSA084XF1 H.sapiens (D19S932) DNA segment containing (CA) repeat; clone AFMa084xf1; single read, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[31].length, 276)
        self.assertEqual(record.alignments[32].title, ">gi|1233152|emb|Z51852|HSA070WH1 H.sapiens (D9S1879) DNA segment containing (CA) repeat; clone AFMa070wh1; single read, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[32].length, 363)
        self.assertEqual(record.alignments[33].title, ">gi|1215106|gb|G17680|G17680 human STS SHGC-3112 clone pG-956.")
        self.assertEqual(record.alignments[33].length, 313)
        self.assertEqual(record.alignments[34].title, ">gi|1594113|gb|G30562|G30562 human STS SHGC-37420")
        self.assertEqual(record.alignments[34].length, 447)
        self.assertEqual(record.alignments[35].title, ">gi|1593022|gb|G29471|G29471 human STS SHGC-33749")
        self.assertEqual(record.alignments[35].length, 437)
        self.assertEqual(record.alignments[36].title, ">gi|1052040|emb|Z67257|HSA090WF9 H.sapiens DNA segment containing (CA) repeat; clone AFMa090wf9; single read, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[36].length, 312)
        self.assertEqual(record.alignments[37].title, ">gi|881858|gb|G07651|G07651 human STS SHGC-5830 clone pG-2175.")
        self.assertEqual(record.alignments[37].length, 336)
        self.assertEqual(record.alignments[38].title, ">gi|1344417|gb|G24091|G24091 human STS WI-12544.")
        self.assertEqual(record.alignments[38].length, 452)
        self.assertEqual(record.alignments[39].title, ">gi|860391|gb|G07146|G07146 human STS WI-9143.")
        self.assertEqual(record.alignments[39].length, 1946)
        self.assertEqual(record.alignments[40].title, ">gi|858959|gb|G05714|G05714 human STS WI-8961.")
        self.assertEqual(record.alignments[40].length, 1337)
        self.assertEqual(record.alignments[41].title, ">gi|1341506|gb|G21180|G21180 human STS WI-14638.")
        self.assertEqual(record.alignments[41].length, 375)
        self.assertEqual(record.alignments[42].title, ">gi|1341551|gb|G21225|G21225 human STS WI-11777.")
        self.assertEqual(record.alignments[42].length, 348)
        self.assertEqual(record.alignments[43].title, ">gi|465219|gb|L29850|HUMUT50 Human STS UT50.")
        self.assertEqual(record.alignments[43].length, 382)
        self.assertEqual(record.alignments[44].title, ">gi|860599|gb|G07354|G07354 human STS WI-9600.")
        self.assertEqual(record.alignments[44].length, 226)
        self.assertEqual(record.alignments[0].hsps[0].score, 331)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 656)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 0.0)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 17)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 34.2)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 0.49)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[2].hsps), 1)
        self.assertEqual(record.alignments[3].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[4].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[4].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[4].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[4].hsps), 1)
        self.assertEqual(record.alignments[5].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[5].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[5].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[5].hsps), 1)
        self.assertEqual(record.alignments[6].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[6].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[6].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[6].hsps), 1)
        self.assertEqual(record.alignments[7].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[7].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[7].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[7].hsps), 1)
        self.assertEqual(record.alignments[8].hsps[0].score, 16)
        self.assertAlmostEqual(record.alignments[8].hsps[0].bits, 32.2)
        self.assertAlmostEqual(record.alignments[8].hsps[0].expect, 1.9)
        self.assertEqual(len(record.alignments[8].hsps), 1)
        self.assertEqual(record.alignments[9].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[9].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[9].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[9].hsps), 1)
        self.assertEqual(record.alignments[10].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[10].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[10].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[10].hsps), 1)
        self.assertEqual(record.alignments[11].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[11].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[11].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[11].hsps), 1)
        self.assertEqual(record.alignments[12].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[12].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[12].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[12].hsps), 1)
        self.assertEqual(record.alignments[13].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[13].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[13].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[13].hsps), 1)
        self.assertEqual(record.alignments[14].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[14].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[14].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[14].hsps), 1)
        self.assertEqual(record.alignments[15].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[15].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[15].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[15].hsps), 1)
        self.assertEqual(record.alignments[16].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[16].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[16].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[16].hsps), 1)
        self.assertEqual(record.alignments[17].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[17].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[17].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[17].hsps), 1)
        self.assertEqual(record.alignments[18].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[18].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[18].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[18].hsps), 1)
        self.assertEqual(record.alignments[19].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[19].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[19].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[19].hsps), 1)
        self.assertEqual(record.alignments[20].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[20].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[20].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[20].hsps), 1)
        self.assertEqual(record.alignments[21].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[21].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[21].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[21].hsps), 1)
        self.assertEqual(record.alignments[22].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[22].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[22].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[22].hsps), 1)
        self.assertEqual(record.alignments[23].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[23].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[23].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[23].hsps), 1)
        self.assertEqual(record.alignments[24].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[24].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[24].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[24].hsps), 1)
        self.assertEqual(record.alignments[25].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[25].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[25].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[25].hsps), 1)
        self.assertEqual(record.alignments[26].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[26].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[26].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[26].hsps), 1)
        self.assertEqual(record.alignments[27].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[27].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[27].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[27].hsps), 1)
        self.assertEqual(record.alignments[28].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[28].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[28].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[28].hsps), 1)
        self.assertEqual(record.alignments[29].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[29].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[29].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[29].hsps), 1)
        self.assertEqual(record.alignments[30].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[30].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[30].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[30].hsps), 1)
        self.assertEqual(record.alignments[31].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[31].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[31].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[31].hsps), 1)
        self.assertEqual(record.alignments[32].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[32].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[32].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[32].hsps), 1)
        self.assertEqual(record.alignments[33].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[33].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[33].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[33].hsps), 1)
        self.assertEqual(record.alignments[34].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[34].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[34].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[34].hsps), 1)
        self.assertEqual(record.alignments[35].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[35].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[35].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[35].hsps), 1)
        self.assertEqual(record.alignments[36].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[36].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[36].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[36].hsps), 1)
        self.assertEqual(record.alignments[37].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[37].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[37].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[37].hsps), 1)
        self.assertEqual(record.alignments[38].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[38].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[38].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[38].hsps), 1)
        self.assertEqual(record.alignments[39].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[39].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[39].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[39].hsps), 1)
        self.assertEqual(record.alignments[40].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[40].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[40].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[40].hsps), 1)
        self.assertEqual(record.alignments[41].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[41].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[41].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[41].hsps), 1)
        self.assertEqual(record.alignments[42].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[42].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[42].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[42].hsps), 1)
        self.assertEqual(record.alignments[43].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[43].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[43].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[43].hsps), 1)
        self.assertEqual(record.alignments[44].hsps[0].score, 15)
        self.assertAlmostEqual(record.alignments[44].hsps[0].bits, 30.2)
        self.assertAlmostEqual(record.alignments[44].hsps[0].expect, 7.6)
        self.assertEqual(len(record.alignments[44].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (331, 331))
        self.assertEqual(record.alignments[1].hsps[0].identities, (17, 17))
        self.assertEqual(record.alignments[2].hsps[0].identities, (22, 24))
        self.assertEqual(record.alignments[3].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[4].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[5].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[6].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[7].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[8].hsps[0].identities, (16, 16))
        self.assertEqual(record.alignments[9].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[10].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[11].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[12].hsps[0].identities, (21, 23))
        self.assertEqual(record.alignments[13].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[14].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[15].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[16].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[17].hsps[0].identities, (18, 19))
        self.assertEqual(record.alignments[18].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[19].hsps[0].identities, (18, 19))
        self.assertEqual(record.alignments[20].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[21].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[22].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[23].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[24].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[25].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[26].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[27].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[28].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[29].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[30].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[31].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[32].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[33].hsps[0].identities, (18, 19))
        self.assertEqual(record.alignments[34].hsps[0].identities, (18, 19))
        self.assertEqual(record.alignments[35].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[36].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[37].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[38].hsps[0].identities, (18, 19))
        self.assertEqual(record.alignments[39].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[40].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[41].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[42].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[43].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[44].hsps[0].identities, (15, 15))
        self.assertEqual(record.alignments[0].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[1].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[2].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[3].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[4].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[5].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[6].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[7].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[8].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[9].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[10].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[11].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[12].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[13].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[14].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[15].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[16].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[17].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[18].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[19].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[20].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[21].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[22].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[23].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[24].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[25].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[26].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[27].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[28].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[29].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[30].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[31].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[32].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[33].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[34].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[35].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[36].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[37].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[38].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[39].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[40].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[41].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[42].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[43].hsps[0].strand, ("Plus", "Minus"))
        self.assertEqual(record.alignments[44].hsps[0].strand, ("Plus", "Plus"))
        self.assertEqual(record.alignments[0].hsps[0].query, "cctccaccctctcatgagcaacaggatatgtgaaagtacttgcagccagaagcaaaaccacaatcctcgggtgctagatggagctccccaaggagcagagaggaaaaggcaggaggagagggccaggcagcagggatggagactaagtttggcccaaggctgcccgcaagcactgatgccatcatgccctctggtaggtgtctatttctgtctgaaccagaaatacaccaagctccacacatgggggctttgctggcttcgacatcactggttcaactatgtcactgctttgttatatttagtgctccagaacctcaggttccttcagatt")
        self.assertEqual(record.alignments[0].hsps[0].match, "|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "cctccaccctctcatgagcaacaggatatgtgaaagtacttgcagccagaagcaaaaccacaatcctcgggtgctagatggagctccccaaggagcagagaggaaaaggcaggaggagagggccaggcagcagggatggagactaagtttggcccaaggctgcccgcaagcactgatgccatcatgccctctggtaggtgtctatttctgtctgaaccagaaatacaccaagctccacacatgggggctttgctggcttcgacatcactggttcaactatgtcactgctttgttatatttagtgctccagaacctcaggttccttcagatt")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 331)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 331)
        self.assertEqual(record.alignments[1].hsps[0].query, "ccaggcagcagggatgg")
        self.assertEqual(record.alignments[1].hsps[0].match, "|||||||||||||||||")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "ccaggcagcagggatgg")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 123)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 139)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 434)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 418)
        self.assertEqual(record.alignments[2].hsps[0].query, "ggaggagagggccaggcagcaggg")
        self.assertEqual(record.alignments[2].hsps[0].match, "||||||||||| | ||||||||||")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "ggaggagaggggctggcagcaggg")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 112)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 135)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 287)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 264)
        self.assertEqual(record.alignments[3].hsps[0].query, "agaagcaaaaccacaa")
        self.assertEqual(record.alignments[3].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "agaagcaaaaccacaa")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 48)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 63)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 434)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 449)
        self.assertEqual(record.alignments[4].hsps[0].query, "agaggaaaaggcagga")
        self.assertEqual(record.alignments[4].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[4].hsps[0].sbjct, "agaggaaaaggcagga")
        self.assertEqual(record.alignments[4].hsps[0].query_start, 99)
        self.assertEqual(record.alignments[4].hsps[0].query_end, 114)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_start, 431)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_end, 416)
        self.assertEqual(record.alignments[5].hsps[0].query, "agagaggaaaaggcag")
        self.assertEqual(record.alignments[5].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[5].hsps[0].sbjct, "agagaggaaaaggcag")
        self.assertEqual(record.alignments[5].hsps[0].query_start, 97)
        self.assertEqual(record.alignments[5].hsps[0].query_end, 112)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_start, 107)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_end, 122)
        self.assertEqual(record.alignments[6].hsps[0].query, "gaggaaaaggcaggag")
        self.assertEqual(record.alignments[6].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[6].hsps[0].sbjct, "gaggaaaaggcaggag")
        self.assertEqual(record.alignments[6].hsps[0].query_start, 100)
        self.assertEqual(record.alignments[6].hsps[0].query_end, 115)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_start, 482)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_end, 467)
        self.assertEqual(record.alignments[7].hsps[0].query, "cagaagcaaaaccaca")
        self.assertEqual(record.alignments[7].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[7].hsps[0].sbjct, "cagaagcaaaaccaca")
        self.assertEqual(record.alignments[7].hsps[0].query_start, 47)
        self.assertEqual(record.alignments[7].hsps[0].query_end, 62)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_start, 193)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_end, 178)
        self.assertEqual(record.alignments[8].hsps[0].query, "tgctttgttatattta")
        self.assertEqual(record.alignments[8].hsps[0].match, "||||||||||||||||")
        self.assertEqual(record.alignments[8].hsps[0].sbjct, "tgctttgttatattta")
        self.assertEqual(record.alignments[8].hsps[0].query_start, 286)
        self.assertEqual(record.alignments[8].hsps[0].query_end, 301)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_start, 111)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_end, 126)
        self.assertEqual(record.alignments[9].hsps[0].query, "cctccaccctctcat")
        self.assertEqual(record.alignments[9].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[9].hsps[0].sbjct, "cctccaccctctcat")
        self.assertEqual(record.alignments[9].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[9].hsps[0].query_end, 15)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_start, 618)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_end, 604)
        self.assertEqual(record.alignments[10].hsps[0].query, "agagggccaggcagc")
        self.assertEqual(record.alignments[10].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[10].hsps[0].sbjct, "agagggccaggcagc")
        self.assertEqual(record.alignments[10].hsps[0].query_start, 117)
        self.assertEqual(record.alignments[10].hsps[0].query_end, 131)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_start, 487)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_end, 501)
        self.assertEqual(record.alignments[11].hsps[0].query, "gagctccccaaggag")
        self.assertEqual(record.alignments[11].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[11].hsps[0].sbjct, "gagctccccaaggag")
        self.assertEqual(record.alignments[11].hsps[0].query_start, 81)
        self.assertEqual(record.alignments[11].hsps[0].query_end, 95)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_start, 278)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_end, 264)
        self.assertEqual(record.alignments[12].hsps[0].query, "gcagagaggaaaaggcaggagga")
        self.assertEqual(record.alignments[12].hsps[0].match, "||||| |||| ||||||||||||")
        self.assertEqual(record.alignments[12].hsps[0].sbjct, "gcagacaggagaaggcaggagga")
        self.assertEqual(record.alignments[12].hsps[0].query_start, 95)
        self.assertEqual(record.alignments[12].hsps[0].query_end, 117)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_start, 84)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_end, 106)
        self.assertEqual(record.alignments[13].hsps[0].query, "aggcagcagggatgg")
        self.assertEqual(record.alignments[13].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[13].hsps[0].sbjct, "aggcagcagggatgg")
        self.assertEqual(record.alignments[13].hsps[0].query_start, 125)
        self.assertEqual(record.alignments[13].hsps[0].query_end, 139)
        self.assertEqual(record.alignments[13].hsps[0].sbjct_start, 58)
        self.assertEqual(record.alignments[13].hsps[0].sbjct_end, 44)
        self.assertEqual(record.alignments[14].hsps[0].query, "atgccctctggtagg")
        self.assertEqual(record.alignments[14].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[14].hsps[0].sbjct, "atgccctctggtagg")
        self.assertEqual(record.alignments[14].hsps[0].query_start, 184)
        self.assertEqual(record.alignments[14].hsps[0].query_end, 198)
        self.assertEqual(record.alignments[14].hsps[0].sbjct_start, 114)
        self.assertEqual(record.alignments[14].hsps[0].sbjct_end, 100)
        self.assertEqual(record.alignments[15].hsps[0].query, "ggatggagactaagt")
        self.assertEqual(record.alignments[15].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[15].hsps[0].sbjct, "ggatggagactaagt")
        self.assertEqual(record.alignments[15].hsps[0].query_start, 134)
        self.assertEqual(record.alignments[15].hsps[0].query_end, 148)
        self.assertEqual(record.alignments[15].hsps[0].sbjct_start, 451)
        self.assertEqual(record.alignments[15].hsps[0].sbjct_end, 465)
        self.assertEqual(record.alignments[16].hsps[0].query, "cagcagggatggaga")
        self.assertEqual(record.alignments[16].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[16].hsps[0].sbjct, "cagcagggatggaga")
        self.assertEqual(record.alignments[16].hsps[0].query_start, 128)
        self.assertEqual(record.alignments[16].hsps[0].query_end, 142)
        self.assertEqual(record.alignments[16].hsps[0].sbjct_start, 163)
        self.assertEqual(record.alignments[16].hsps[0].sbjct_end, 149)
        self.assertEqual(record.alignments[17].hsps[0].query, "tggagactaagtttggccc")
        self.assertEqual(record.alignments[17].hsps[0].match, "||||||||||||| |||||")
        self.assertEqual(record.alignments[17].hsps[0].sbjct, "tggagactaagttgggccc")
        self.assertEqual(record.alignments[17].hsps[0].query_start, 137)
        self.assertEqual(record.alignments[17].hsps[0].query_end, 155)
        self.assertEqual(record.alignments[17].hsps[0].sbjct_start, 302)
        self.assertEqual(record.alignments[17].hsps[0].sbjct_end, 320)
        self.assertEqual(record.alignments[18].hsps[0].query, "tcactggttcaacta")
        self.assertEqual(record.alignments[18].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[18].hsps[0].sbjct, "tcactggttcaacta")
        self.assertEqual(record.alignments[18].hsps[0].query_start, 265)
        self.assertEqual(record.alignments[18].hsps[0].query_end, 279)
        self.assertEqual(record.alignments[18].hsps[0].sbjct_start, 185)
        self.assertEqual(record.alignments[18].hsps[0].sbjct_end, 199)
        self.assertEqual(record.alignments[19].hsps[0].query, "aaggagcagagaggaaaag")
        self.assertEqual(record.alignments[19].hsps[0].match, "|||||| ||||||||||||")
        self.assertEqual(record.alignments[19].hsps[0].sbjct, "aaggagaagagaggaaaag")
        self.assertEqual(record.alignments[19].hsps[0].query_start, 90)
        self.assertEqual(record.alignments[19].hsps[0].query_end, 108)
        self.assertEqual(record.alignments[19].hsps[0].sbjct_start, 209)
        self.assertEqual(record.alignments[19].hsps[0].sbjct_end, 227)
        self.assertEqual(record.alignments[20].hsps[0].query, "gagagggccaggcag")
        self.assertEqual(record.alignments[20].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[20].hsps[0].sbjct, "gagagggccaggcag")
        self.assertEqual(record.alignments[20].hsps[0].query_start, 116)
        self.assertEqual(record.alignments[20].hsps[0].query_end, 130)
        self.assertEqual(record.alignments[20].hsps[0].sbjct_start, 155)
        self.assertEqual(record.alignments[20].hsps[0].sbjct_end, 169)
        self.assertEqual(record.alignments[21].hsps[0].query, "gcagccagaagcaaa")
        self.assertEqual(record.alignments[21].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[21].hsps[0].sbjct, "gcagccagaagcaaa")
        self.assertEqual(record.alignments[21].hsps[0].query_start, 42)
        self.assertEqual(record.alignments[21].hsps[0].query_end, 56)
        self.assertEqual(record.alignments[21].hsps[0].sbjct_start, 259)
        self.assertEqual(record.alignments[21].hsps[0].sbjct_end, 273)
        self.assertEqual(record.alignments[22].hsps[0].query, "tgtctatttctgtct")
        self.assertEqual(record.alignments[22].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[22].hsps[0].sbjct, "tgtctatttctgtct")
        self.assertEqual(record.alignments[22].hsps[0].query_start, 199)
        self.assertEqual(record.alignments[22].hsps[0].query_end, 213)
        self.assertEqual(record.alignments[22].hsps[0].sbjct_start, 155)
        self.assertEqual(record.alignments[22].hsps[0].sbjct_end, 169)
        self.assertEqual(record.alignments[23].hsps[0].query, "acaccaagctccaca")
        self.assertEqual(record.alignments[23].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[23].hsps[0].sbjct, "acaccaagctccaca")
        self.assertEqual(record.alignments[23].hsps[0].query_start, 225)
        self.assertEqual(record.alignments[23].hsps[0].query_end, 239)
        self.assertEqual(record.alignments[23].hsps[0].sbjct_start, 163)
        self.assertEqual(record.alignments[23].hsps[0].sbjct_end, 177)
        self.assertEqual(record.alignments[24].hsps[0].query, "tgtctatttctgtct")
        self.assertEqual(record.alignments[24].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[24].hsps[0].sbjct, "tgtctatttctgtct")
        self.assertEqual(record.alignments[24].hsps[0].query_start, 199)
        self.assertEqual(record.alignments[24].hsps[0].query_end, 213)
        self.assertEqual(record.alignments[24].hsps[0].sbjct_start, 244)
        self.assertEqual(record.alignments[24].hsps[0].sbjct_end, 258)
        self.assertEqual(record.alignments[25].hsps[0].query, "ggagagggccaggca")
        self.assertEqual(record.alignments[25].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[25].hsps[0].sbjct, "ggagagggccaggca")
        self.assertEqual(record.alignments[25].hsps[0].query_start, 115)
        self.assertEqual(record.alignments[25].hsps[0].query_end, 129)
        self.assertEqual(record.alignments[25].hsps[0].sbjct_start, 53)
        self.assertEqual(record.alignments[25].hsps[0].sbjct_end, 39)
        self.assertEqual(record.alignments[26].hsps[0].query, "acatcactggttcaa")
        self.assertEqual(record.alignments[26].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[26].hsps[0].sbjct, "acatcactggttcaa")
        self.assertEqual(record.alignments[26].hsps[0].query_start, 262)
        self.assertEqual(record.alignments[26].hsps[0].query_end, 276)
        self.assertEqual(record.alignments[26].hsps[0].sbjct_start, 331)
        self.assertEqual(record.alignments[26].hsps[0].sbjct_end, 345)
        self.assertEqual(record.alignments[27].hsps[0].query, "ggccaggcagcaggg")
        self.assertEqual(record.alignments[27].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[27].hsps[0].sbjct, "ggccaggcagcaggg")
        self.assertEqual(record.alignments[27].hsps[0].query_start, 121)
        self.assertEqual(record.alignments[27].hsps[0].query_end, 135)
        self.assertEqual(record.alignments[27].hsps[0].sbjct_start, 108)
        self.assertEqual(record.alignments[27].hsps[0].sbjct_end, 122)
        self.assertEqual(record.alignments[28].hsps[0].query, "tttagtgctccagaa")
        self.assertEqual(record.alignments[28].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[28].hsps[0].sbjct, "tttagtgctccagaa")
        self.assertEqual(record.alignments[28].hsps[0].query_start, 298)
        self.assertEqual(record.alignments[28].hsps[0].query_end, 312)
        self.assertEqual(record.alignments[28].hsps[0].sbjct_start, 261)
        self.assertEqual(record.alignments[28].hsps[0].sbjct_end, 275)
        self.assertEqual(record.alignments[29].hsps[0].query, "caggttccttcagat")
        self.assertEqual(record.alignments[29].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[29].hsps[0].sbjct, "caggttccttcagat")
        self.assertEqual(record.alignments[29].hsps[0].query_start, 316)
        self.assertEqual(record.alignments[29].hsps[0].query_end, 330)
        self.assertEqual(record.alignments[29].hsps[0].sbjct_start, 323)
        self.assertEqual(record.alignments[29].hsps[0].sbjct_end, 309)
        self.assertEqual(record.alignments[30].hsps[0].query, "cagccagaagcaaaa")
        self.assertEqual(record.alignments[30].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[30].hsps[0].sbjct, "cagccagaagcaaaa")
        self.assertEqual(record.alignments[30].hsps[0].query_start, 43)
        self.assertEqual(record.alignments[30].hsps[0].query_end, 57)
        self.assertEqual(record.alignments[30].hsps[0].sbjct_start, 356)
        self.assertEqual(record.alignments[30].hsps[0].sbjct_end, 342)
        self.assertEqual(record.alignments[31].hsps[0].query, "aacctcaggttcctt")
        self.assertEqual(record.alignments[31].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[31].hsps[0].sbjct, "aacctcaggttcctt")
        self.assertEqual(record.alignments[31].hsps[0].query_start, 311)
        self.assertEqual(record.alignments[31].hsps[0].query_end, 325)
        self.assertEqual(record.alignments[31].hsps[0].sbjct_start, 54)
        self.assertEqual(record.alignments[31].hsps[0].sbjct_end, 40)
        self.assertEqual(record.alignments[32].hsps[0].query, "cctccaccctctcat")
        self.assertEqual(record.alignments[32].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[32].hsps[0].sbjct, "cctccaccctctcat")
        self.assertEqual(record.alignments[32].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[32].hsps[0].query_end, 15)
        self.assertEqual(record.alignments[32].hsps[0].sbjct_start, 59)
        self.assertEqual(record.alignments[32].hsps[0].sbjct_end, 73)
        self.assertEqual(record.alignments[33].hsps[0].query, "ccccaaggagcagagagga")
        self.assertEqual(record.alignments[33].hsps[0].match, "|||| ||||||||||||||")
        self.assertEqual(record.alignments[33].hsps[0].sbjct, "ccccgaggagcagagagga")
        self.assertEqual(record.alignments[33].hsps[0].query_start, 86)
        self.assertEqual(record.alignments[33].hsps[0].query_end, 104)
        self.assertEqual(record.alignments[33].hsps[0].sbjct_start, 213)
        self.assertEqual(record.alignments[33].hsps[0].sbjct_end, 195)
        self.assertEqual(record.alignments[34].hsps[0].query, "ggaggagagggccaggcag")
        self.assertEqual(record.alignments[34].hsps[0].match, "||||||| |||||||||||")
        self.assertEqual(record.alignments[34].hsps[0].sbjct, "ggaggagggggccaggcag")
        self.assertEqual(record.alignments[34].hsps[0].query_start, 112)
        self.assertEqual(record.alignments[34].hsps[0].query_end, 130)
        self.assertEqual(record.alignments[34].hsps[0].sbjct_start, 234)
        self.assertEqual(record.alignments[34].hsps[0].sbjct_end, 252)
        self.assertEqual(record.alignments[35].hsps[0].query, "ccaggcagcagggat")
        self.assertEqual(record.alignments[35].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[35].hsps[0].sbjct, "ccaggcagcagggat")
        self.assertEqual(record.alignments[35].hsps[0].query_start, 123)
        self.assertEqual(record.alignments[35].hsps[0].query_end, 137)
        self.assertEqual(record.alignments[35].hsps[0].sbjct_start, 320)
        self.assertEqual(record.alignments[35].hsps[0].sbjct_end, 306)
        self.assertEqual(record.alignments[36].hsps[0].query, "gtaggtgtctatttc")
        self.assertEqual(record.alignments[36].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[36].hsps[0].sbjct, "gtaggtgtctatttc")
        self.assertEqual(record.alignments[36].hsps[0].query_start, 194)
        self.assertEqual(record.alignments[36].hsps[0].query_end, 208)
        self.assertEqual(record.alignments[36].hsps[0].sbjct_start, 201)
        self.assertEqual(record.alignments[36].hsps[0].sbjct_end, 215)
        self.assertEqual(record.alignments[37].hsps[0].query, "ggccaggcagcaggg")
        self.assertEqual(record.alignments[37].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[37].hsps[0].sbjct, "ggccaggcagcaggg")
        self.assertEqual(record.alignments[37].hsps[0].query_start, 121)
        self.assertEqual(record.alignments[37].hsps[0].query_end, 135)
        self.assertEqual(record.alignments[37].hsps[0].sbjct_start, 86)
        self.assertEqual(record.alignments[37].hsps[0].sbjct_end, 100)
        self.assertEqual(record.alignments[38].hsps[0].query, "aaaaggcaggaggagaggg")
        self.assertEqual(record.alignments[38].hsps[0].match, "|||||||||||| ||||||")
        self.assertEqual(record.alignments[38].hsps[0].sbjct, "aaaaggcaggagaagaggg")
        self.assertEqual(record.alignments[38].hsps[0].query_start, 104)
        self.assertEqual(record.alignments[38].hsps[0].query_end, 122)
        self.assertEqual(record.alignments[38].hsps[0].sbjct_start, 234)
        self.assertEqual(record.alignments[38].hsps[0].sbjct_end, 216)
        self.assertEqual(record.alignments[39].hsps[0].query, "tgtcactgctttgtt")
        self.assertEqual(record.alignments[39].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[39].hsps[0].sbjct, "tgtcactgctttgtt")
        self.assertEqual(record.alignments[39].hsps[0].query_start, 280)
        self.assertEqual(record.alignments[39].hsps[0].query_end, 294)
        self.assertEqual(record.alignments[39].hsps[0].sbjct_start, 492)
        self.assertEqual(record.alignments[39].hsps[0].sbjct_end, 478)
        self.assertEqual(record.alignments[40].hsps[0].query, "acacatgggggcttt")
        self.assertEqual(record.alignments[40].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[40].hsps[0].sbjct, "acacatgggggcttt")
        self.assertEqual(record.alignments[40].hsps[0].query_start, 237)
        self.assertEqual(record.alignments[40].hsps[0].query_end, 251)
        self.assertEqual(record.alignments[40].hsps[0].sbjct_start, 117)
        self.assertEqual(record.alignments[40].hsps[0].sbjct_end, 131)
        self.assertEqual(record.alignments[41].hsps[0].query, "ctttgttatatttag")
        self.assertEqual(record.alignments[41].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[41].hsps[0].sbjct, "ctttgttatatttag")
        self.assertEqual(record.alignments[41].hsps[0].query_start, 288)
        self.assertEqual(record.alignments[41].hsps[0].query_end, 302)
        self.assertEqual(record.alignments[41].hsps[0].sbjct_start, 74)
        self.assertEqual(record.alignments[41].hsps[0].sbjct_end, 60)
        self.assertEqual(record.alignments[42].hsps[0].query, "aggagcagagaggaa")
        self.assertEqual(record.alignments[42].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[42].hsps[0].sbjct, "aggagcagagaggaa")
        self.assertEqual(record.alignments[42].hsps[0].query_start, 91)
        self.assertEqual(record.alignments[42].hsps[0].query_end, 105)
        self.assertEqual(record.alignments[42].hsps[0].sbjct_start, 216)
        self.assertEqual(record.alignments[42].hsps[0].sbjct_end, 230)
        self.assertEqual(record.alignments[43].hsps[0].query, "aggcagcagggatgg")
        self.assertEqual(record.alignments[43].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[43].hsps[0].sbjct, "aggcagcagggatgg")
        self.assertEqual(record.alignments[43].hsps[0].query_start, 125)
        self.assertEqual(record.alignments[43].hsps[0].query_end, 139)
        self.assertEqual(record.alignments[43].hsps[0].sbjct_start, 273)
        self.assertEqual(record.alignments[43].hsps[0].sbjct_end, 259)
        self.assertEqual(record.alignments[44].hsps[0].query, "aggagagggccaggc")
        self.assertEqual(record.alignments[44].hsps[0].match, "|||||||||||||||")
        self.assertEqual(record.alignments[44].hsps[0].sbjct, "aggagagggccaggc")
        self.assertEqual(record.alignments[44].hsps[0].query_start, 114)
        self.assertEqual(record.alignments[44].hsps[0].query_end, 128)
        self.assertEqual(record.alignments[44].hsps[0].sbjct_start, 124)
        self.assertEqual(record.alignments[44].hsps[0].sbjct_end, 138)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(record.posted_date, [('Feb 11, 2000  2:37 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 1.37)
        self.assertAlmostEqual(record.ka_params[1], 0.711)
        self.assertAlmostEqual(record.ka_params[2], 1.31)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 1.37)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.711)
        self.assertAlmostEqual(record.ka_params_gap[2], 1.31)
        self.assertEqual(record.matrix, 'blastn matrix:1 -3')
        self.assertEqual(record.gap_penalties, [5, 2])
        self.assertEqual(record.num_hits, 6844)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 6844)
        self.assertEqual(record.num_good_extends, 1887)
        self.assertEqual(record.num_seqs_better_e, 51)
        self.assertEqual(record.query_length, 331)
        self.assertEqual(record.database_length, 31998854)
        self.assertEqual(record.effective_hsp_length, 17)
        self.assertEqual(record.effective_query_length, 314)
        self.assertEqual(record.effective_database_length, 30506390)
        self.assertEqual(record.effective_search_space, 9579006460)
        self.assertEqual(record.effective_search_space_used, 9579006460)
        self.assertEqual(record.threshold, 0)
        self.assertEqual(record.window_size, 0)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 6)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 11.9)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 10)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 19.8)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 12)
        self.assertAlmostEqual(record.gap_trigger[1], 24.3)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 15)
        self.assertAlmostEqual(record.blast_cutoff[1], 30.2)

    def test_text_2011L_blastn_004(self):
        "Test parsing BLASTN 2.0.11 output (text_2011L_blastn_004)"

        path = os.path.join('Blast', 'text_2011L_blastn_004.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTN")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|859351|gb|G06106|G06106 human STS WI-6344.")
        self.assertEqual(record.query_letters, 183)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 6)
        self.assertEqual(record.descriptions[0].title, "gi|859351|gb|G06106|G06106 human STS WI-6344.")
        self.assertEqual(record.descriptions[0].score, 327)
        self.assertAlmostEqual(record.descriptions[0].e, 1e-89)
        self.assertEqual(record.descriptions[1].title, "gi|1341350|gb|G21024|G21024 human STS WI-30979.")
        self.assertEqual(record.descriptions[1].score, 32)
        self.assertAlmostEqual(record.descriptions[1].e, 1.0)
        self.assertEqual(record.descriptions[2].title, "gi|6126285|gb|G60966.1|G60966 SHGC-84377 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[2].score, 30)
        self.assertAlmostEqual(record.descriptions[2].e, 4.1)
        self.assertEqual(record.descriptions[3].title, "gi|5222421|gb|G51244.1|G51244 SHGC-80725 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[3].score, 30)
        self.assertAlmostEqual(record.descriptions[3].e, 4.1)
        self.assertEqual(record.descriptions[4].title, "gi|1340656|gb|G20319|G20319 human STS A005L39.")
        self.assertEqual(record.descriptions[4].score, 30)
        self.assertAlmostEqual(record.descriptions[4].e, 4.1)
        self.assertEqual(record.descriptions[5].title, "gi|860526|gb|G07281|G07281 human STS WI-9430.")
        self.assertEqual(record.descriptions[5].score, 30)
        self.assertAlmostEqual(record.descriptions[5].e, 4.1)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(record.posted_date, [('Feb 11, 2000  2:37 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 1.37)
        self.assertAlmostEqual(record.ka_params[1], 0.711)
        self.assertAlmostEqual(record.ka_params[2], 1.31)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 1.37)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.711)
        self.assertAlmostEqual(record.ka_params_gap[2], 1.31)
        self.assertEqual(record.matrix, 'blastn matrix:1 -3')
        self.assertEqual(record.gap_penalties, [5, 2])
        self.assertEqual(record.num_hits, 1902)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 1902)
        self.assertEqual(record.num_good_extends, 481)
        self.assertEqual(record.num_seqs_better_e, 8)
        self.assertEqual(record.query_length, 183)
        self.assertEqual(record.database_length, 31998854)
        self.assertEqual(record.effective_hsp_length, 16)
        self.assertEqual(record.effective_query_length, 167)
        self.assertEqual(record.effective_database_length, 30594182)
        self.assertEqual(record.effective_search_space, 5109228394)
        self.assertEqual(record.effective_search_space_used, 5109228394)
        self.assertEqual(record.threshold, 0)
        self.assertEqual(record.window_size, 0)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 6)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 11.9)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 10)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 19.8)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 12)
        self.assertAlmostEqual(record.gap_trigger[1], 24.3)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 15)
        self.assertAlmostEqual(record.blast_cutoff[1], 30.2)

    def test_text_2011L_blastx_001(self):
        "Test parsing BLASTX 2.0.11 output (text_2011L_blastx_001)"

        path = os.path.join('Blast', 'text_2011L_blastx_001.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTX")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1347369|gb|G25137|G25137 human STS EST48004.")
        self.assertEqual(record.query_letters, 556)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 4)
        self.assertEqual(record.descriptions[0].title, "gi|1731448|sp|P54103|ZRF1_MOUSE ZUOTIN RELATED FACTOR-1")
        self.assertEqual(record.descriptions[0].score, 87)
        self.assertAlmostEqual(record.descriptions[0].e, 3e-17)
        self.assertEqual(record.descriptions[1].title, "gi|465911|sp|P34454|YMA9_CAEEL HYPOTHETICAL 31.6 KD PROTEIN F54...")
        self.assertEqual(record.descriptions[1].score, 42)
        self.assertAlmostEqual(record.descriptions[1].e, 0.001)
        self.assertEqual(record.descriptions[2].title, "gi|2494160|sp|Q61712|MTJ1_MOUSE DNAJ PROTEIN HOMOLOG MTJ1")
        self.assertEqual(record.descriptions[2].score, 37)
        self.assertAlmostEqual(record.descriptions[2].e, 0.033)
        self.assertEqual(record.descriptions[3].title, "gi|1730688|sp|P53745|YN8X_YEAST HYPOTHETICAL 68.1 KD PROTEIN IN...")
        self.assertEqual(record.descriptions[3].score, 29)
        self.assertAlmostEqual(record.descriptions[3].e, 7.4)
        self.assertEqual(len(record.alignments), 4)
        self.assertEqual(record.alignments[0].title, ">gi|1731448|sp|P54103|ZRF1_MOUSE ZUOTIN RELATED FACTOR-1")
        self.assertEqual(record.alignments[0].length, 514)
        self.assertEqual(record.alignments[1].title, ">gi|465911|sp|P34454|YMA9_CAEEL HYPOTHETICAL 31.6 KD PROTEIN F54F2.9 IN CHROMOSOME III")
        self.assertEqual(record.alignments[1].length, 275)
        self.assertEqual(record.alignments[2].title, ">gi|2494160|sp|Q61712|MTJ1_MOUSE DNAJ PROTEIN HOMOLOG MTJ1")
        self.assertEqual(record.alignments[2].length, 552)
        self.assertEqual(record.alignments[3].title, ">gi|1730688|sp|P53745|YN8X_YEAST HYPOTHETICAL 68.1 KD PROTEIN IN BIO3-FRE4 INTERGENIC REGION")
        self.assertEqual(record.alignments[3].length, 580)
        self.assertEqual(record.alignments[0].hsps[0].score, 211)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 86.6)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 3e-17)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 96)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 41.8)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 0.001)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 83)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 36.7)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 0.033)
        self.assertEqual(record.alignments[2].hsps[1].score, 69)
        self.assertAlmostEqual(record.alignments[2].hsps[1].bits, 31.3)
        self.assertAlmostEqual(record.alignments[2].hsps[1].expect, 1.5)
        self.assertEqual(len(record.alignments[2].hsps), 2)
        self.assertEqual(record.alignments[3].hsps[0].score, 63)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 29.0)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 7.4)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (41, 47))
        self.assertEqual(record.alignments[0].hsps[0].positives, (44, 47))
        self.assertEqual(record.alignments[1].hsps[0].identities, (30, 122))
        self.assertEqual(record.alignments[1].hsps[0].positives, (54, 122))
        self.assertEqual(record.alignments[2].hsps[0].identities, (17, 36))
        self.assertEqual(record.alignments[2].hsps[0].positives, (19, 36))
        self.assertEqual(record.alignments[2].hsps[1].identities, (18, 50))
        self.assertEqual(record.alignments[2].hsps[1].positives, (26, 50))
        self.assertEqual(record.alignments[3].hsps[0].identities, (27, 99))
        self.assertEqual(record.alignments[3].hsps[0].positives, (41, 99))
        self.assertEqual(record.alignments[0].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[1].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[2].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[2].hsps[1].frame, ("+1", ))
        self.assertEqual(record.alignments[3].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[0].hsps[0].query, "DLQLLIKAVNLFPAGTNSRWEVIANYMNIHSSSGVKRTAKDVIGKAK")
        self.assertEqual(record.alignments[0].hsps[0].match, "DLQLLIKAVNLFPAG NSRW+VIANYMNIHSSSGVKRTAKDVI + +")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "DLQLLIKAVNLFPAGRNSRWDVIANYMNIHSSSGVKRTAKDVISEVR")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 141)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 458)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 504)
        self.assertEqual(record.alignments[1].hsps[0].query, "FPAGTNSRWEVIANYMNIHSSSGVKRTAKDVIGKAKSLQKLDPHQKDDINKKAFDKFKKEHGVVPQADNATPSERFXGPYTDFTPXTTEXQKLXEQALNTYPVNTXERWXXIAVAVPGRXKE")
        self.assertEqual(record.alignments[1].hsps[0].match, "+PAGT +RWE +   +N        R+A+DVI  A  ++++   +++D  K      ++   V  ++++                 +   QK  E AL  YP  T ERW  I+  +  + K+")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "YPAGTPNRWEQMGRVLN--------RSAEDVIAMAGKMKQM---KQEDYTKLLMTTIQQSVPVEEKSED---------------DWSQAEQKAFETALQKYPKGTDERWERISEEIGSKTKK")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 34)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 399)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 159)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 254)
        self.assertEqual(record.alignments[2].hsps[0].query, "TTEXQKLXEQALNTYPVNTXERWXXIAVAVPGRXKE")
        self.assertEqual(record.alignments[2].hsps[0].match, "T   QKL E AL  YP    +RW  IA  VP + KE")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "TQSQQKLLELALQQYPKGASDRWDKIAKCVPSKSKE")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 292)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 399)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 496)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 531)
        self.assertEqual(record.alignments[2].hsps[1].query, "DLQLLIKAVNLFPAGTNSRWEVIANYMNIHSSSGVKRTAKDVIGKAKSLQ")
        self.assertEqual(record.alignments[2].hsps[1].match, "DL  L +++  FP GT  RW+ IA+ +         R+  DV  KAK L+")
        self.assertEqual(record.alignments[2].hsps[1].sbjct, "DLSQLTRSMVKFPGGTPGRWDKIAHELG--------RSVTDVTTKAKELK")
        self.assertEqual(record.alignments[2].hsps[1].query_start, 1)
        self.assertEqual(record.alignments[2].hsps[1].query_end, 150)
        self.assertEqual(record.alignments[2].hsps[1].sbjct_start, 332)
        self.assertEqual(record.alignments[2].hsps[1].sbjct_end, 373)
        self.assertEqual(record.alignments[3].hsps[0].query, "SRWEVIANYMNIHSSSGVKRTAKDVIGKAKSLQKLDPHQKDDINKKAFDKFKKEHGVVPQADNATPSERFXGPYTDFTPXTTEXQKLXEQALNTYPVNT")
        self.assertEqual(record.alignments[3].hsps[0].match, "+RW+   +Y        V R+ KDV   ++SL  LD +QK     +A       H +          E    PY +FT   +      EQ+ N +PV+T")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "NRWKSFISY--------VTRSRKDVKTVSRSLSNLDLYQKCSKEIRADQDISLLHSI----------ETKLFPYINFTALNS------EQSHNFWPVHT")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 52)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 348)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 75)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 149)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.posted_date, [('Feb 2, 2000  9:39 AM',)])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.318)
        self.assertAlmostEqual(record.ka_params[1], 0.135)
        self.assertAlmostEqual(record.ka_params[2], 0.401)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 23174157)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 387821)
        self.assertEqual(record.num_good_extends, 980)
        self.assertEqual(record.num_seqs_better_e, 8)
        self.assertEqual(record.hsps_no_gap, 3)
        self.assertEqual(record.hsps_prelim_gapped, 1)
        self.assertEqual(record.hsps_gapped, 7)
        self.assertEqual(record.query_length, 185)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 49)
        self.assertEqual(record.effective_query_length, 135)
        self.assertEqual(record.effective_database_length, 25621919)
        self.assertEqual(record.effective_search_space, 3458959065)
        self.assertEqual(record.effective_search_space_used, 3458959065)
        self.assertEqual(record.frameshift, ('50,', '0.1'))
        self.assertEqual(record.threshold, 12)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 62)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.6)

    def test_text_2011L_blastx_002(self):
        "Test parsing BLASTX 2.0.11 output without hits (text_2011L_blastx_002)"

        path = os.path.join('Blast', 'text_2011L_blastx_002.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTX")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1347782|gb|G25550|G25550 human STS\nEST47652.\x01gi|1592937|gb|G29386|G29386 human STS SHGC-32770")
        self.assertEqual(record.query_letters, 379)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Feb 2, 2000  9:39 AM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.318)
        self.assertAlmostEqual(record.ka_params[1], 0.135)
        self.assertAlmostEqual(record.ka_params[2], 0.401)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 14686001)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 235383)
        self.assertEqual(record.num_good_extends, 396)
        self.assertEqual(record.num_seqs_better_e, 0)
        self.assertEqual(record.hsps_no_gap, 0)
        self.assertEqual(record.hsps_prelim_gapped, 0)
        self.assertEqual(record.hsps_gapped, 0)
        self.assertEqual(record.query_length, 126)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 48)
        self.assertEqual(record.effective_query_length, 77)
        self.assertEqual(record.effective_database_length, 25704177)
        self.assertEqual(record.effective_search_space, 1979221629)
        self.assertEqual(record.effective_search_space_used, 1979221629)
        self.assertEqual(record.frameshift, ('50,', '0.1'))
        self.assertEqual(record.threshold, 12)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 60)
        self.assertAlmostEqual(record.blast_cutoff[1], 27.8)

    def test_text_2011L_tblastn_001(self):
        "Test parsing TBLASTN 2.0.11 output (text_2011L_tblastn_001)"

        path = os.path.join('Blast', 'text_2011L_tblastn_001.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "TBLASTN")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|729325|sp|P39483|DHG2_BACME GLUCOSE 1-DEHYDROGENASE II\n(GLCDH-II)")
        self.assertEqual(record.query_letters, 261)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 7)
        self.assertEqual(record.descriptions[0].title, "gi|3820341|emb|AJ229891|KLAJ9891 Kluyveromyces lactis DNA fragm...")
        self.assertEqual(record.descriptions[0].score, 47)
        self.assertAlmostEqual(record.descriptions[0].e, 0.00001)
        self.assertEqual(record.descriptions[1].title, "gi|1375419|gb|G27169|G27169 human STS SHGC-31983.")
        self.assertEqual(record.descriptions[1].score, 43)
        self.assertAlmostEqual(record.descriptions[1].e, 0.0001)
        self.assertEqual(record.descriptions[2].title, "gi|3819804|emb|AJ230012|KLAJ0012 Kluyveromyces lactis DNA fragm...")
        self.assertEqual(record.descriptions[2].score, 39)
        self.assertAlmostEqual(record.descriptions[2].e, 0.002)
        self.assertEqual(record.descriptions[3].title, "gi|1375215|gb|G26965|G26965 human STS SHGC-31083.")
        self.assertEqual(record.descriptions[3].score, 31)
        self.assertAlmostEqual(record.descriptions[3].e, 0.73)
        self.assertEqual(record.descriptions[4].title, "gi|5714409|gb|AF106665.1|AF106665 Mus musculus chromosome 6 clo...")
        self.assertEqual(record.descriptions[4].score, 29)
        self.assertAlmostEqual(record.descriptions[4].e, 2.2)
        self.assertEqual(record.descriptions[5].title, "gi|177714|gb|L09988|HUM4STS889 Human Chromosome 4 (clone p4-109...")
        self.assertEqual(record.descriptions[5].score, 29)
        self.assertAlmostEqual(record.descriptions[5].e, 2.2)
        self.assertEqual(record.descriptions[6].title, "gi|1341648|gb|G21322|G21322 human STS WI-12250.")
        self.assertEqual(record.descriptions[6].score, 29)
        self.assertAlmostEqual(record.descriptions[6].e, 3.7)
        self.assertEqual(len(record.alignments), 7)
        self.assertEqual(record.alignments[0].title, ">gi|3820341|emb|AJ229891|KLAJ9891 Kluyveromyces lactis DNA fragment for sequence tagged site, clone okam5d07r [Kluyveromyces lactis]")
        self.assertEqual(record.alignments[0].length, 230)
        self.assertEqual(record.alignments[1].title, ">gi|1375419|gb|G27169|G27169 human STS SHGC-31983.")
        self.assertEqual(record.alignments[1].length, 594)
        self.assertEqual(record.alignments[2].title, ">gi|3819804|emb|AJ230012|KLAJ0012 Kluyveromyces lactis DNA fragment for sequence tagged site, clone okam6d01d [Kluyveromyces lactis]")
        self.assertEqual(record.alignments[2].length, 199)
        self.assertEqual(record.alignments[3].title, ">gi|1375215|gb|G26965|G26965 human STS SHGC-31083.")
        self.assertEqual(record.alignments[3].length, 268)
        self.assertEqual(record.alignments[4].title, ">gi|5714409|gb|AF106665.1|AF106665 Mus musculus chromosome 6 clone D6wum9 map between Nkrp1 and Prp strain C57BL/6J, sequence tagged site")
        self.assertEqual(record.alignments[4].length, 299)
        self.assertEqual(record.alignments[5].title, ">gi|177714|gb|L09988|HUM4STS889 Human Chromosome 4 (clone p4-1095) STS4-889.")
        self.assertEqual(record.alignments[5].length, 412)
        self.assertEqual(record.alignments[6].title, ">gi|1341648|gb|G21322|G21322 human STS WI-12250.")
        self.assertEqual(record.alignments[6].length, 586)
        self.assertEqual(record.alignments[0].hsps[0].score, 109)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 46.9)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 1e-5)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 100)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 43.4)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 1e-4)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 90)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 39.5)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 0.002)
        self.assertEqual(len(record.alignments[2].hsps), 1)
        self.assertEqual(record.alignments[3].hsps[0].score, 68)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 30.9)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 0.73)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[4].hsps[0].score, 64)
        self.assertAlmostEqual(record.alignments[4].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.alignments[4].hsps[0].expect, 2.2)
        self.assertEqual(len(record.alignments[4].hsps), 1)
        self.assertEqual(record.alignments[5].hsps[0].score, 64)
        self.assertAlmostEqual(record.alignments[5].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.alignments[5].hsps[0].expect, 2.2)
        self.assertEqual(len(record.alignments[5].hsps), 1)
        self.assertEqual(record.alignments[6].hsps[0].score, 62)
        self.assertAlmostEqual(record.alignments[6].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.alignments[6].hsps[0].expect, 3.7)
        self.assertEqual(len(record.alignments[6].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (25, 72))
        self.assertEqual(record.alignments[0].hsps[0].positives, (44, 72))
        self.assertEqual(record.alignments[0].hsps[0].gaps, (3, 72))
        self.assertEqual(record.alignments[1].hsps[0].identities, (21, 73))
        self.assertEqual(record.alignments[1].hsps[0].positives, (34, 73))
        self.assertEqual(record.alignments[2].hsps[0].identities, (18, 49))
        self.assertEqual(record.alignments[2].hsps[0].positives, (26, 49))
        self.assertEqual(record.alignments[3].hsps[0].identities, (12, 37))
        self.assertEqual(record.alignments[3].hsps[0].positives, (19, 37))
        self.assertEqual(record.alignments[4].hsps[0].identities, (17, 55))
        self.assertEqual(record.alignments[4].hsps[0].positives, (32, 55))
        self.assertEqual(record.alignments[4].hsps[0].gaps, (2, 55))
        self.assertEqual(record.alignments[5].hsps[0].identities, (14, 34))
        self.assertEqual(record.alignments[5].hsps[0].positives, (22, 34))
        self.assertEqual(record.alignments[6].hsps[0].identities, (16, 39))
        self.assertEqual(record.alignments[6].hsps[0].positives, (20, 39))
        self.assertEqual(record.alignments[6].hsps[0].gaps, (1, 39))
        self.assertEqual(record.alignments[0].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[1].hsps[0].frame, ("-1", ))
        self.assertEqual(record.alignments[2].hsps[0].frame, ("-1", ))
        self.assertEqual(record.alignments[3].hsps[0].frame, ("-1", ))
        self.assertEqual(record.alignments[4].hsps[0].frame, ("-2", ))
        self.assertEqual(record.alignments[5].hsps[0].frame, ("-1", ))
        self.assertEqual(record.alignments[6].hsps[0].frame, ("+1", ))
        self.assertEqual(record.alignments[0].hsps[0].query, "NWNQVIDTNLTGAFLGSREAIKYFVEN---DIKGNVINMSSVHEMIPWPLFVHYAASKGGMKLMTETLALEYAPK")
        self.assertEqual(record.alignments[0].hsps[0].match, "+W QVIDTN+ G F   + A+     +   D +  V+N+S+V+ ++  P    Y A+K  +  +T+++ALEYA +")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "SWRQVIDTNINGTFYTLKYALPLMESSSSPDSEAAVVNLSAVNGLVGIPGISPYTATKHAVIGITQSVALEYAER")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 108)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 179)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 225)
        self.assertEqual(record.alignments[1].hsps[0].query, "APKGIRVNNIGPGAIDTPINAEKFADPEQRADVESMIPMGYIGKPEEIASVAAFLASSQASYVTGITLFADGG")
        self.assertEqual(record.alignments[1].hsps[0].match, "AP   RVN + P  + T +    + DP +   + +  P+G   + E +     FL S ++   TG TL  +GG")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "APXRXRVNAVXPXVVMTSMGQATWXDPXKAXTMLNRXPLGXFAEVEHVVKAILFLLSDRSGMTTGSTLPVEGG")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 177)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 249)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 312)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 94)
        self.assertEqual(record.alignments[2].hsps[0].query, "FADPEQRADVESMIPMGYIGKPEEIASVAAFLASSQASYVTGITLFADG")
        self.assertEqual(record.alignments[2].hsps[0].match, "F D + +    S+IPMG  G P+E+     + AS  ++Y TG  L  DG")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "FVDEDLKNKWHSLIPMGREGLPQELVGAYLYFASDASTYTTGSDLLVDG")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 200)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 248)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 157)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 11)
        self.assertEqual(record.alignments[3].hsps[0].query, "PMGYIGKPEEIASVAAFLASSQASYVTGITLFADGGM")
        self.assertEqual(record.alignments[3].hsps[0].match, "PMG  G PE++  V A      +  +TG ++   GG+")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "PMGXXGDPEDVXDVXAXXXXEXSGXITGTSVEVTGGL")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 214)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 250)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 268)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 158)
        self.assertEqual(record.alignments[4].hsps[0].query, "NMSSVHEMIPWPLFVHYAASKGGMKLMTETL--ALEYAPKGIRVNNIGPGAIDTPIN")
        self.assertEqual(record.alignments[4].hsps[0].match, "++S    +I +P F+    S  G  L+  +L  A+ + P GI V+++GP ++ T +N")
        self.assertEqual(record.alignments[4].hsps[0].sbjct, "SLSPTQYLIMFPSFLPCPLSHPGPFLLPSSLVIAVFFLPNGIEVSSLGPFSLRTLLN")
        self.assertEqual(record.alignments[4].hsps[0].query_start, 142)
        self.assertEqual(record.alignments[4].hsps[0].query_end, 196)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_start, 172)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_end, 2)
        self.assertEqual(record.alignments[5].hsps[0].query, "DKVVVVTGGSKGLGRAMAVRFGQEQSKVVVNYRS")
        self.assertEqual(record.alignments[5].hsps[0].match, "DKV  V GGS+G+GRA+A    ++  ++ V  R+")
        self.assertEqual(record.alignments[5].hsps[0].sbjct, "DKVCAVFGGSRGIGRAVAQLMARKGYRLAVIARN")
        self.assertEqual(record.alignments[5].hsps[0].query_start, 7)
        self.assertEqual(record.alignments[5].hsps[0].query_end, 40)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_start, 316)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_end, 215)
        self.assertEqual(record.alignments[6].hsps[0].query, "PVPSHELSLENWNQ-VIDTNLTGAFLGSREAIKYFVENDI")
        self.assertEqual(record.alignments[6].hsps[0].match, "PVP  ELS  +W+Q  + T+ T  F  S     YF  N I")
        self.assertEqual(record.alignments[6].hsps[0].sbjct, "PVPMQELSKVHWSQFFLTTSPTMTFFFSHYLANYFFRNSI")
        self.assertEqual(record.alignments[6].hsps[0].query_start, 98)
        self.assertEqual(record.alignments[6].hsps[0].query_end, 136)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_start, 220)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_end, 339)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(record.posted_date, [('Feb 11, 2000  2:37 PM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.315)
        self.assertAlmostEqual(record.ka_params[1], 0.134)
        self.assertAlmostEqual(record.ka_params[2], 0.378)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 6165875)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 55665)
        self.assertEqual(record.num_good_extends, 148)
        self.assertEqual(record.num_seqs_better_e, 14)
        self.assertEqual(record.hsps_no_gap, 5)
        self.assertEqual(record.hsps_prelim_gapped, 2)
        self.assertEqual(record.hsps_gapped, 7)
        self.assertEqual(record.query_length, 261)
        self.assertEqual(record.database_length, 10666284)
        self.assertEqual(record.effective_hsp_length, 50)
        self.assertEqual(record.effective_query_length, 211)
        self.assertEqual(record.effective_database_length, 6276684)
        self.assertEqual(record.effective_search_space, 1324380324)
        self.assertEqual(record.effective_search_space_used, 1324380324)
        self.assertEqual(record.frameshift, ('50,', '0.1'))
        self.assertEqual(record.threshold, 13)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 42)
        self.assertAlmostEqual(record.gap_trigger[1], 22.0)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 58)
        self.assertAlmostEqual(record.blast_cutoff[1], 27.0)

    def test_text_2011L_tblastn_002(self):
        "Test parsing TBLASTN 2.0.11 output without hits (text_2011L_tblastn_002)"

        path = os.path.join('Blast', 'text_2011L_tblastn_002.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "TBLASTN")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|127420|sp|P19888|MTBA_BACAR MODIFICATION METHYLASE BANI\n(CYTOSINE-SPECIFIC METHYLTRANSFERASE BANI) (M.BANI)")
        self.assertEqual(record.query_letters, 428)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.posted_date, [('Feb 11, 2000  2:37 PM',)])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.320)
        self.assertAlmostEqual(record.ka_params[1], 0.140)
        self.assertAlmostEqual(record.ka_params[2], 0.403)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 13588598)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 162273)
        self.assertEqual(record.num_good_extends, 546)
        self.assertEqual(record.num_seqs_better_e, 0)
        self.assertEqual(record.hsps_no_gap, 0)
        self.assertEqual(record.hsps_prelim_gapped, 0)
        self.assertEqual(record.hsps_gapped, 0)
        self.assertEqual(record.query_length, 428)
        self.assertEqual(record.database_length, 10666284)
        self.assertEqual(record.effective_hsp_length, 48)
        self.assertEqual(record.effective_query_length, 380)
        self.assertEqual(record.effective_database_length, 6452268)
        self.assertEqual(record.effective_search_space, 2451861840)
        self.assertEqual(record.effective_search_space_used, 2451861840)
        self.assertEqual(record.frameshift, ('50,', '0.1'))
        self.assertEqual(record.threshold, 13)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.4)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.8)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 61)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.2)

    def test_text_2011L_tblastx_001(self):
        "Test parsing TBLASTX 2.0.11 output (text_2011L_tblastx_001)"

        path = os.path.join('Blast', 'text_2011L_tblastx_001.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "TBLASTX")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|1348853|gb|G26621|G26621 human STS\nSTS_D12006.\x01gi|1396339|gb|G27620|G27620 human STS SHGC-32705.")
        self.assertEqual(record.query_letters, 615)
        self.assertEqual(record.database, "data/sts")
        self.assertEqual(record.database_sequences, 87792)
        self.assertEqual(record.database_letters, 31998854)
        self.assertEqual(len(record.descriptions), 19)
        self.assertEqual(record.descriptions[0].title, "gi|1348853|gb|G26621|G26621 human STS STS_D12006. >gi|1396339|g...")
        self.assertEqual(record.descriptions[0].score, 398)
        self.assertAlmostEqual(record.descriptions[0].e, 1e-111)
        self.assertEqual(record.descriptions[1].title, "gi|1348016|gb|G25784|G25784 human STS EST47998.")
        self.assertEqual(record.descriptions[1].score, 302)
        self.assertAlmostEqual(record.descriptions[1].e, 1e-82)
        self.assertEqual(record.descriptions[2].title, "gi|3403105|gb|G41148|G41148 Z7324 Zebrafish AB Danio rerio STS ...")
        self.assertEqual(record.descriptions[2].score, 31)
        self.assertAlmostEqual(record.descriptions[2].e, 0.6)
        self.assertEqual(record.descriptions[3].title, "gi|1234967|emb|Z53521|HSB316XA9 H.sapiens (D22S1166) DNA segmen...")
        self.assertEqual(record.descriptions[3].score, 30)
        self.assertAlmostEqual(record.descriptions[3].e, 0.83)
        self.assertEqual(record.descriptions[4].title, "gi|4185670|gb|G42865|G42865 Xq3845 KWOK Homo sapiens STS genomi...")
        self.assertEqual(record.descriptions[4].score, 30)
        self.assertAlmostEqual(record.descriptions[4].e, 1.1)
        self.assertEqual(record.descriptions[5].title, "gi|4757419|gb|G49246.1|G49246 stbK116F5_30376 chromosome 22 gen...")
        self.assertEqual(record.descriptions[5].score, 29)
        self.assertAlmostEqual(record.descriptions[5].e, 1.6)
        self.assertEqual(record.descriptions[6].title, "gi|6120694|gb|G55375.1|G55375 SHGC-100697 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[6].score, 29)
        self.assertAlmostEqual(record.descriptions[6].e, 3.0)
        self.assertEqual(record.descriptions[7].title, "gi|5225124|gb|G53947.1|G53947 SHGC-85304 Human Homo sapiens STS...")
        self.assertEqual(record.descriptions[7].score, 29)
        self.assertAlmostEqual(record.descriptions[7].e, 3.0)
        self.assertEqual(record.descriptions[8].title, "gi|1311530|gb|L77996|HUMSWX945 Human chromosome X STS sWXD945, ...")
        self.assertEqual(record.descriptions[8].score, 28)
        self.assertAlmostEqual(record.descriptions[8].e, 4.1)
        self.assertEqual(record.descriptions[9].title, "gi|4631600|dbj|AU046965.1|AU046965 Rattus norvegicus, OTSUKA cl...")
        self.assertEqual(record.descriptions[9].score, 28)
        self.assertAlmostEqual(record.descriptions[9].e, 4.1)
        self.assertEqual(record.descriptions[10].title, "gi|4631518|dbj|AU046883.1|AU046883 Rattus norvegicus, OTSUKA cl...")
        self.assertEqual(record.descriptions[10].score, 28)
        self.assertAlmostEqual(record.descriptions[10].e, 4.1)
        self.assertEqual(record.descriptions[11].title, "gi|2226478|gb|G33174|G33174 human STS SHGC-6097 clone pG-2470")
        self.assertEqual(record.descriptions[11].score, 28)
        self.assertAlmostEqual(record.descriptions[11].e, 4.1)
        self.assertEqual(record.descriptions[12].title, "gi|2734624|gb|G36957|G36957 SHGC-56642 Human Homo sapiens STS cDNA")
        self.assertEqual(record.descriptions[12].score, 28)
        self.assertAlmostEqual(record.descriptions[12].e, 4.1)
        self.assertEqual(record.descriptions[13].title, "gi|859804|gb|G06559|G06559 human STS WI-7401.")
        self.assertEqual(record.descriptions[13].score, 27)
        self.assertAlmostEqual(record.descriptions[13].e, 7.7)
        self.assertEqual(record.descriptions[14].title, "gi|938611|gb|G08061|G08061 human STS CHLC.GGAA7E02.P7438 clone ...")
        self.assertEqual(record.descriptions[14].score, 27)
        self.assertAlmostEqual(record.descriptions[14].e, 7.7)
        self.assertEqual(record.descriptions[15].title, "gi|307789|gb|L18105|HUMUT1736 Human STS UT1736.")
        self.assertEqual(record.descriptions[15].score, 27)
        self.assertAlmostEqual(record.descriptions[15].e, 7.7)
        self.assertEqual(record.descriptions[16].title, "gi|4492122|gb|G45831.1|G45831 Z4588_1 Zebrafish AB Danio rerio ...")
        self.assertEqual(record.descriptions[16].score, 27)
        self.assertAlmostEqual(record.descriptions[16].e, 7.7)
        self.assertEqual(record.descriptions[17].title, "gi|6121804|gb|G56635.1|G56635 SHGC-102032 Human Homo sapiens ST...")
        self.assertEqual(record.descriptions[17].score, 27)
        self.assertAlmostEqual(record.descriptions[17].e, 7.7)
        self.assertEqual(record.descriptions[18].title, "gi|4493143|gb|G46852.1|G46852 Z14841_1 Zebrafish AB Danio rerio...")
        self.assertEqual(record.descriptions[18].score, 27)
        self.assertAlmostEqual(record.descriptions[18].e, 7.7)
        self.assertEqual(len(record.alignments), 19)
        self.assertEqual(record.alignments[0].title, ">gi|1348853|gb|G26621|G26621 human STS STS_D12006. >gi|1396339|gb|G27620|G27620 human STS SHGC-32705.")
        self.assertEqual(record.alignments[0].length, 615)
        self.assertEqual(record.alignments[1].title, ">gi|1348016|gb|G25784|G25784 human STS EST47998.")
        self.assertEqual(record.alignments[1].length, 617)
        self.assertEqual(record.alignments[2].title, ">gi|3403105|gb|G41148|G41148 Z7324 Zebrafish AB Danio rerio STS genomic")
        self.assertEqual(record.alignments[2].length, 351)
        self.assertEqual(record.alignments[3].title, ">gi|1234967|emb|Z53521|HSB316XA9 H.sapiens (D22S1166) DNA segment containing (CA) repeat; clone AFMb316xa9; single read, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[3].length, 345)
        self.assertEqual(record.alignments[4].title, ">gi|4185670|gb|G42865|G42865 Xq3845 KWOK Homo sapiens STS genomic, sequence tagged site [Homo sapiens]")
        self.assertEqual(record.alignments[4].length, 1200)
        self.assertEqual(record.alignments[5].title, ">gi|4757419|gb|G49246.1|G49246 stbK116F5_30376 chromosome 22 genomic clone Homo sapiens STS genomic clone 116F5, sequence tagged site")
        self.assertEqual(record.alignments[5].length, 375)
        self.assertEqual(record.alignments[6].title, ">gi|6120694|gb|G55375.1|G55375 SHGC-100697 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[6].length, 460)
        self.assertEqual(record.alignments[7].title, ">gi|5225124|gb|G53947.1|G53947 SHGC-85304 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[7].length, 444)
        self.assertEqual(record.alignments[8].title, ">gi|1311530|gb|L77996|HUMSWX945 Human chromosome X STS sWXD945, single read.")
        self.assertEqual(record.alignments[8].length, 196)
        self.assertEqual(record.alignments[9].title, ">gi|4631600|dbj|AU046965.1|AU046965 Rattus norvegicus, OTSUKA clone, 108a02, microsatellite sequence, sequence tagged site")
        self.assertEqual(record.alignments[9].length, 330)
        self.assertEqual(record.alignments[10].title, ">gi|4631518|dbj|AU046883.1|AU046883 Rattus norvegicus, OTSUKA clone, 085f03, microsatellite sequence, sequence tagged site")
        self.assertEqual(record.alignments[10].length, 351)
        self.assertEqual(record.alignments[11].title, ">gi|2226478|gb|G33174|G33174 human STS SHGC-6097 clone pG-2470")
        self.assertEqual(record.alignments[11].length, 299)
        self.assertEqual(record.alignments[12].title, ">gi|2734624|gb|G36957|G36957 SHGC-56642 Human Homo sapiens STS cDNA")
        self.assertEqual(record.alignments[12].length, 466)
        self.assertEqual(record.alignments[13].title, ">gi|859804|gb|G06559|G06559 human STS WI-7401.")
        self.assertEqual(record.alignments[13].length, 3280)
        self.assertEqual(record.alignments[14].title, ">gi|938611|gb|G08061|G08061 human STS CHLC.GGAA7E02.P7438 clone GGAA7E02")
        self.assertEqual(record.alignments[14].length, 338)
        self.assertEqual(record.alignments[15].title, ">gi|307789|gb|L18105|HUMUT1736 Human STS UT1736.")
        self.assertEqual(record.alignments[15].length, 355)
        self.assertEqual(record.alignments[16].title, ">gi|4492122|gb|G45831.1|G45831 Z4588_1 Zebrafish AB Danio rerio STS genomic clone Z4588 5', sequence tagged site")
        self.assertEqual(record.alignments[16].length, 398)
        self.assertEqual(record.alignments[17].title, ">gi|6121804|gb|G56635.1|G56635 SHGC-102032 Human Homo sapiens STS genomic, sequence tagged site")
        self.assertEqual(record.alignments[17].length, 541)
        self.assertEqual(record.alignments[18].title, ">gi|4493143|gb|G46852.1|G46852 Z14841_1 Zebrafish AB Danio rerio STS genomic clone Z14841 5', sequence tagged site")
        self.assertEqual(record.alignments[18].length, 291)
        self.assertEqual(record.alignments[0].hsps[0].score, 796)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 367)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 1e-102)
        self.assertEqual(record.alignments[0].hsps[1].score, 759)
        self.assertAlmostEqual(record.alignments[0].hsps[1].bits, 350)
        self.assertAlmostEqual(record.alignments[0].hsps[1].expect, 3e-97)
        self.assertEqual(record.alignments[0].hsps[2].score, 387)
        self.assertAlmostEqual(record.alignments[0].hsps[2].bits, 180)
        self.assertAlmostEqual(record.alignments[0].hsps[2].expect, 9e-91)
        self.assertEqual(record.alignments[0].hsps[3].score, 368)
        self.assertAlmostEqual(record.alignments[0].hsps[3].bits, 171)
        self.assertAlmostEqual(record.alignments[0].hsps[3].expect, 9e-91)
        self.assertEqual(record.alignments[0].hsps[4].score, 864)
        self.assertAlmostEqual(record.alignments[0].hsps[4].bits, 398)
        self.assertAlmostEqual(record.alignments[0].hsps[4].expect, 1e-111)
        self.assertEqual(record.alignments[0].hsps[5].score, 846)
        self.assertAlmostEqual(record.alignments[0].hsps[5].bits, 390)
        self.assertAlmostEqual(record.alignments[0].hsps[5].expect, 1e-109)
        self.assertEqual(record.alignments[0].hsps[6].score, 684)
        self.assertAlmostEqual(record.alignments[0].hsps[6].bits, 316)
        self.assertAlmostEqual(record.alignments[0].hsps[6].expect, 7e-87)
        self.assertEqual(len(record.alignments[0].hsps), 7)
        self.assertEqual(record.alignments[1].hsps[0].score, 366)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 170)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 3e-63)
        self.assertEqual(record.alignments[1].hsps[1].score, 188)
        self.assertAlmostEqual(record.alignments[1].hsps[1].bits, 89.0)
        self.assertAlmostEqual(record.alignments[1].hsps[1].expect, 3e-63)
        self.assertEqual(record.alignments[1].hsps[2].score, 590)
        self.assertAlmostEqual(record.alignments[1].hsps[2].bits, 273)
        self.assertAlmostEqual(record.alignments[1].hsps[2].expect, 7e-74)
        self.assertEqual(record.alignments[1].hsps[3].score, 593)
        self.assertAlmostEqual(record.alignments[1].hsps[3].bits, 274)
        self.assertAlmostEqual(record.alignments[1].hsps[3].expect, 8e-76)
        self.assertEqual(record.alignments[1].hsps[4].score, 53)
        self.assertAlmostEqual(record.alignments[1].hsps[4].bits, 27.2)
        self.assertAlmostEqual(record.alignments[1].hsps[4].expect, 8e-76)
        self.assertEqual(record.alignments[1].hsps[5].score, 653)
        self.assertAlmostEqual(record.alignments[1].hsps[5].bits, 302)
        self.assertAlmostEqual(record.alignments[1].hsps[5].expect, 1e-82)
        self.assertEqual(record.alignments[1].hsps[6].score, 598)
        self.assertAlmostEqual(record.alignments[1].hsps[6].bits, 276)
        self.assertAlmostEqual(record.alignments[1].hsps[6].expect, 5e-75)
        self.assertEqual(record.alignments[1].hsps[7].score, 628)
        self.assertAlmostEqual(record.alignments[1].hsps[7].bits, 290)
        self.assertAlmostEqual(record.alignments[1].hsps[7].expect, 4e-79)
        self.assertEqual(len(record.alignments[1].hsps), 8)
        self.assertEqual(record.alignments[2].hsps[0].score, 61)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 30.8)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 0.60)
        self.assertEqual(len(record.alignments[2].hsps), 1)
        self.assertEqual(record.alignments[3].hsps[0].score, 60)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 30.4)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 0.83)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[4].hsps[0].score, 59)
        self.assertAlmostEqual(record.alignments[4].hsps[0].bits, 29.9)
        self.assertAlmostEqual(record.alignments[4].hsps[0].expect, 1.1)
        self.assertEqual(len(record.alignments[4].hsps), 1)
        self.assertEqual(record.alignments[5].hsps[0].score, 58)
        self.assertAlmostEqual(record.alignments[5].hsps[0].bits, 29.5)
        self.assertAlmostEqual(record.alignments[5].hsps[0].expect, 1.6)
        self.assertEqual(len(record.alignments[5].hsps), 1)
        self.assertEqual(record.alignments[6].hsps[0].score, 56)
        self.assertAlmostEqual(record.alignments[6].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.alignments[6].hsps[0].expect, 3.0)
        self.assertEqual(len(record.alignments[6].hsps), 1)
        self.assertEqual(record.alignments[7].hsps[0].score, 56)
        self.assertAlmostEqual(record.alignments[7].hsps[0].bits, 28.6)
        self.assertAlmostEqual(record.alignments[7].hsps[0].expect, 3.0)
        self.assertEqual(len(record.alignments[7].hsps), 1)
        self.assertEqual(record.alignments[8].hsps[0].score, 55)
        self.assertAlmostEqual(record.alignments[8].hsps[0].bits, 28.1)
        self.assertAlmostEqual(record.alignments[8].hsps[0].expect, 4.1)
        self.assertEqual(len(record.alignments[8].hsps), 1)
        self.assertEqual(record.alignments[9].hsps[0].score, 55)
        self.assertAlmostEqual(record.alignments[9].hsps[0].bits, 28.1)
        self.assertAlmostEqual(record.alignments[9].hsps[0].expect, 4.1)
        self.assertEqual(len(record.alignments[9].hsps), 1)
        self.assertEqual(record.alignments[10].hsps[0].score, 55)
        self.assertAlmostEqual(record.alignments[10].hsps[0].bits, 28.1)
        self.assertAlmostEqual(record.alignments[10].hsps[0].expect, 4.1)
        self.assertEqual(len(record.alignments[10].hsps), 1)
        self.assertEqual(record.alignments[11].hsps[0].score, 55)
        self.assertAlmostEqual(record.alignments[11].hsps[0].bits, 28.1)
        self.assertAlmostEqual(record.alignments[11].hsps[0].expect, 4.1)
        self.assertEqual(len(record.alignments[11].hsps), 1)
        self.assertEqual(record.alignments[12].hsps[0].score, 55)
        self.assertAlmostEqual(record.alignments[12].hsps[0].bits, 28.1)
        self.assertAlmostEqual(record.alignments[12].hsps[0].expect, 4.1)
        self.assertEqual(len(record.alignments[12].hsps), 1)
        self.assertEqual(record.alignments[13].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[13].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[13].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[13].hsps), 1)
        self.assertEqual(record.alignments[14].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[14].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[14].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[14].hsps), 1)
        self.assertEqual(record.alignments[15].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[15].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[15].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[15].hsps), 1)
        self.assertEqual(record.alignments[16].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[16].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[16].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[16].hsps), 1)
        self.assertEqual(record.alignments[17].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[17].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[17].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[17].hsps), 1)
        self.assertEqual(record.alignments[18].hsps[0].score, 53)
        self.assertAlmostEqual(record.alignments[18].hsps[0].bits, 27.2)
        self.assertAlmostEqual(record.alignments[18].hsps[0].expect, 7.7)
        self.assertEqual(len(record.alignments[18].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (192, 200))
        self.assertEqual(record.alignments[0].hsps[0].positives, (192, 200))
        self.assertEqual(record.alignments[0].hsps[1].identities, (195, 205))
        self.assertEqual(record.alignments[0].hsps[1].positives, (195, 205))
        self.assertEqual(record.alignments[0].hsps[2].identities, (74, 74))
        self.assertEqual(record.alignments[0].hsps[2].positives, (74, 74))
        self.assertEqual(record.alignments[0].hsps[3].identities, (114, 114))
        self.assertEqual(record.alignments[0].hsps[3].positives, (114, 114))
        self.assertEqual(record.alignments[0].hsps[4].identities, (205, 205))
        self.assertEqual(record.alignments[0].hsps[4].positives, (205, 205))
        self.assertEqual(record.alignments[0].hsps[5].identities, (196, 196))
        self.assertEqual(record.alignments[0].hsps[5].positives, (196, 196))
        self.assertEqual(record.alignments[0].hsps[6].identities, (146, 146))
        self.assertEqual(record.alignments[0].hsps[6].positives, (146, 146))
        self.assertEqual(record.alignments[1].hsps[0].identities, (71, 74))
        self.assertEqual(record.alignments[1].hsps[0].positives, (71, 74))
        self.assertEqual(record.alignments[1].hsps[1].identities, (42, 67))
        self.assertEqual(record.alignments[1].hsps[1].positives, (43, 67))
        self.assertEqual(record.alignments[1].hsps[2].identities, (121, 133))
        self.assertEqual(record.alignments[1].hsps[2].positives, (121, 133))
        self.assertEqual(record.alignments[1].hsps[3].identities, (112, 131))
        self.assertEqual(record.alignments[1].hsps[3].positives, (112, 131))
        self.assertEqual(record.alignments[1].hsps[4].identities, (9, 13))
        self.assertEqual(record.alignments[1].hsps[4].positives, (10, 13))
        self.assertEqual(record.alignments[1].hsps[5].identities, (128, 157))
        self.assertEqual(record.alignments[1].hsps[5].positives, (132, 157))
        self.assertEqual(record.alignments[1].hsps[6].identities, (122, 130))
        self.assertEqual(record.alignments[1].hsps[6].positives, (122, 130))
        self.assertEqual(record.alignments[1].hsps[7].identities, (119, 131))
        self.assertEqual(record.alignments[1].hsps[7].positives, (120, 131))
        self.assertEqual(record.alignments[2].hsps[0].identities, (11, 27))
        self.assertEqual(record.alignments[2].hsps[0].positives, (18, 27))
        self.assertEqual(record.alignments[3].hsps[0].identities, (10, 19))
        self.assertEqual(record.alignments[3].hsps[0].positives, (13, 19))
        self.assertEqual(record.alignments[4].hsps[0].identities, (10, 24))
        self.assertEqual(record.alignments[4].hsps[0].positives, (14, 24))
        self.assertEqual(record.alignments[5].hsps[0].identities, (15, 34))
        self.assertEqual(record.alignments[5].hsps[0].positives, (17, 34))
        self.assertEqual(record.alignments[6].hsps[0].identities, (9, 28))
        self.assertEqual(record.alignments[6].hsps[0].positives, (16, 28))
        self.assertEqual(record.alignments[7].hsps[0].identities, (10, 24))
        self.assertEqual(record.alignments[7].hsps[0].positives, (15, 24))
        self.assertEqual(record.alignments[8].hsps[0].identities, (9, 33))
        self.assertEqual(record.alignments[8].hsps[0].positives, (19, 33))
        self.assertEqual(record.alignments[9].hsps[0].identities, (12, 39))
        self.assertEqual(record.alignments[9].hsps[0].positives, (19, 39))
        self.assertEqual(record.alignments[10].hsps[0].identities, (8, 26))
        self.assertEqual(record.alignments[10].hsps[0].positives, (15, 26))
        self.assertEqual(record.alignments[11].hsps[0].identities, (13, 38))
        self.assertEqual(record.alignments[11].hsps[0].positives, (19, 38))
        self.assertEqual(record.alignments[12].hsps[0].identities, (10, 15))
        self.assertEqual(record.alignments[12].hsps[0].positives, (11, 15))
        self.assertEqual(record.alignments[13].hsps[0].identities, (9, 21))
        self.assertEqual(record.alignments[13].hsps[0].positives, (13, 21))
        self.assertEqual(record.alignments[14].hsps[0].identities, (12, 24))
        self.assertEqual(record.alignments[14].hsps[0].positives, (13, 24))
        self.assertEqual(record.alignments[15].hsps[0].identities, (12, 24))
        self.assertEqual(record.alignments[15].hsps[0].positives, (13, 24))
        self.assertEqual(record.alignments[16].hsps[0].identities, (7, 22))
        self.assertEqual(record.alignments[16].hsps[0].positives, (13, 22))
        self.assertEqual(record.alignments[17].hsps[0].identities, (9, 20))
        self.assertEqual(record.alignments[17].hsps[0].positives, (13, 20))
        self.assertEqual(record.alignments[18].hsps[0].identities, (8, 20))
        self.assertEqual(record.alignments[18].hsps[0].positives, (13, 20))
        self.assertEqual(record.alignments[0].hsps[0].frame, ("+2", "+2"))
        self.assertEqual(record.alignments[0].hsps[1].frame, ("+1", "+1"))
        self.assertEqual(record.alignments[0].hsps[2].frame, ("+3", "+3"))
        self.assertEqual(record.alignments[0].hsps[3].frame, ("+3", "+3"))
        self.assertEqual(record.alignments[0].hsps[4].frame, ("-1", "-1"))
        self.assertEqual(record.alignments[0].hsps[5].frame, ("-3", "-3"))
        self.assertEqual(record.alignments[0].hsps[6].frame, ("-2", "-2"))
        self.assertEqual(record.alignments[1].hsps[0].frame, ("+3", "+3"))
        self.assertEqual(record.alignments[1].hsps[1].frame, ("+3", "+3"))
        self.assertEqual(record.alignments[1].hsps[2].frame, ("+1", "+1"))
        self.assertEqual(record.alignments[1].hsps[3].frame, ("+2", "+2"))
        self.assertEqual(record.alignments[1].hsps[4].frame, ("+2", "+2"))
        self.assertEqual(record.alignments[1].hsps[5].frame, ("-3", "-2"))
        self.assertEqual(record.alignments[1].hsps[6].frame, ("-2", "-1"))
        self.assertEqual(record.alignments[1].hsps[7].frame, ("-1", "-3"))
        self.assertEqual(record.alignments[2].hsps[0].frame, ("+1", "+2"))
        self.assertEqual(record.alignments[3].hsps[0].frame, ("-3", "+1"))
        self.assertEqual(record.alignments[4].hsps[0].frame, ("-3", "-2"))
        self.assertEqual(record.alignments[5].hsps[0].frame, ("-3", "-2"))
        self.assertEqual(record.alignments[6].hsps[0].frame, ("+3", "+1"))
        self.assertEqual(record.alignments[7].hsps[0].frame, ("+1", "-2"))
        self.assertEqual(record.alignments[8].hsps[0].frame, ("+3", "-2"))
        self.assertEqual(record.alignments[9].hsps[0].frame, ("+2", "-3"))
        self.assertEqual(record.alignments[10].hsps[0].frame, ("+1", "+3"))
        self.assertEqual(record.alignments[11].hsps[0].frame, ("+1", "+2"))
        self.assertEqual(record.alignments[12].hsps[0].frame, ("-3", "-3"))
        self.assertEqual(record.alignments[13].hsps[0].frame, ("-1", "-1"))
        self.assertEqual(record.alignments[14].hsps[0].frame, ("-3", "+3"))
        self.assertEqual(record.alignments[15].hsps[0].frame, ("-3", "-1"))
        self.assertEqual(record.alignments[16].hsps[0].frame, ("+1", "+1"))
        self.assertEqual(record.alignments[17].hsps[0].frame, ("-1", "+2"))
        self.assertEqual(record.alignments[18].hsps[0].frame, ("-3", "+1"))
        self.assertEqual(record.alignments[0].hsps[0].query, "IRMPLHS*DSSFCPL*QEKWECMXXXXXXXXRPKRCLQPHPLNWP*LGLNALMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFSPLQPLFWWTNNLGTPNPGRKNIQHYEXALX*S*NGFPKLVTHGPGXVXLXLXGLSSFQEFXSTVANPWGX*XXXXXXXXXFXXGXRXXXLXXXXGGCXXVXXVXXXWXXXF")
        self.assertEqual(record.alignments[0].hsps[0].match, "IRMPLHS*DSSFCPL*QEKWECM        RPKRCLQPHPLNWP*LGLNALMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFSPLQPLFWWTNNLGTPNPGRKNIQHYE AL *S*NGFPKLVTHGPG V L L GLSSFQEF STVANPWG *         F  G R   L    GGC  V  V   W   F")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "IRMPLHS*DSSFCPL*QEKWECMQSSQKKQKRPKRCLQPHPLNWP*LGLNALMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFSPLQPLFWWTNNLGTPNPGRKNIQHYEXALX*S*NGFPKLVTHGPGXVXLXLXGLSSFQEFXSTVANPWGX*XXXXXXXXXFXXGXRXXXLXXXXGGCXXVXXVXXXWXXXF")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 2)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 601)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 2)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 601)
        self.assertEqual(record.alignments[0].hsps[1].query, "DQNAPPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTKCTNAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL*XSTXVILKWISKAGYTWPWXXXIXFXXAFFXXXXXXXXXXXXXXLXGXVX*XXXXSXGPXXXXVXXPXGWVXXGXFXFXXXXXXFXXXLG")
        self.assertEqual(record.alignments[0].hsps[1].match, "DQNAPPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTKCTNAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL* ST VILKWISKAGYTWPW   I F  AFF              L G V *    S GP    V  P GWV  G F F      F   LG")
        self.assertEqual(record.alignments[0].hsps[1].sbjct, "DQNAPPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTKCTNAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL*XSTXVILKWISKAGYTWPWXXXIXFXXAFFXSGVXVNGGKSXGXLXGXVX*XXXXSXGPXXXXVXXPXGWVXXGXFXFXXXXXXFXXXLG")
        self.assertEqual(record.alignments[0].hsps[1].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[1].query_end, 615)
        self.assertEqual(record.alignments[0].hsps[1].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[1].sbjct_end, 615)
        self.assertEqual(record.alignments[0].hsps[2].query, "SECPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*MH*CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[0].hsps[2].match, "SECPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*MH*CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[0].hsps[2].sbjct, "SECPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*MH*CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[0].hsps[2].query_start, 3)
        self.assertEqual(record.alignments[0].hsps[2].query_end, 224)
        self.assertEqual(record.alignments[0].hsps[2].sbjct_start, 3)
        self.assertEqual(record.alignments[0].hsps[2].sbjct_end, 224)
        self.assertEqual(record.alignments[0].hsps[3].query, "YSHCSGGQIT*VLQILAGKIYSIMXQHSXNPKMDFQSWLHMALXXSY*XXXGFLXFRSXGQRWQIXGVXXWXGXLXXXXFXGAXGGXGXXSXXVGAXGXXXFXXXGXXXXXXXW")
        self.assertEqual(record.alignments[0].hsps[3].match, "YSHCSGGQIT*VLQILAGKIYSIM QHS NPKMDFQSWLHMAL  SY*   GFL FRS GQRWQI GV  W G L    F GA GG G  S  VGA G   F   G       W")
        self.assertEqual(record.alignments[0].hsps[3].sbjct, "YSHCSGGQIT*VLQILAGKIYSIMXQHSXNPKMDFQSWLHMALXXSY*XXXGFLXFRSXGQRWQIXGVXXWXGXLXXXXFXGAXGGXGXXSXXVGAXGXXXFXXXGXXXXXXXW")
        self.assertEqual(record.alignments[0].hsps[3].query_start, 273)
        self.assertEqual(record.alignments[0].hsps[3].query_end, 614)
        self.assertEqual(record.alignments[0].hsps[3].sbjct_start, 273)
        self.assertEqual(record.alignments[0].hsps[3].sbjct_end, 614)
        self.assertEqual(record.alignments[0].hsps[4].query, "PKXXXKXXXXXXKTKXTXXHPPXWXXNXXXLWPXXXXXXLXNXXX*XPXGFATVDXNS*XEESPXKXNXTXPGPCVTSFGNPF*DYXSAXS*CCIFFLPGFGVPRLFVHQNNGCKGEN*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCISAFSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRGILI")
        self.assertEqual(record.alignments[0].hsps[4].match, "PK   K      KTK T  HPP W  N   LWP      L N   * P GFATVD NS* EESP K N T PGPCVTSFGNPF*DY SA S*CCIFFLPGFGVPRLFVHQNNGCKGEN*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCISAFSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRGILI")
        self.assertEqual(record.alignments[0].hsps[4].sbjct, "PKXXXKXXXXXXKTKXTXXHPPXWXXNXXXLWPXXXXXXLXNXXX*XPXGFATVDXNS*XEESPXKXNXTXPGPCVTSFGNPF*DYXSAXS*CCIFFLPGFGVPRLFVHQNNGCKGEN*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCISAFSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRGILI")
        self.assertEqual(record.alignments[0].hsps[4].query_start, 615)
        self.assertEqual(record.alignments[0].hsps[4].query_end, 1)
        self.assertEqual(record.alignments[0].hsps[4].sbjct_start, 615)
        self.assertEqual(record.alignments[0].hsps[4].sbjct_end, 1)
        self.assertEqual(record.alignments[0].hsps[5].query, "PXXXNXXNPXAPTXXXX*PXPPXAPXKXXXXXXPXXLXTPXICHR*PKLLKXRKPX*XQXDXXRAMCNQLWKSILGLXECXFIMLYIFPARIWST*VICPPEQWL*RRKLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*CI*SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGHSD")
        self.assertEqual(record.alignments[0].hsps[5].match, "P   N  NP APT    *P PP AP K      P  L TP ICHR*PKLLK RKP * Q D  RAMCNQLWKSILGL EC FIMLYIFPARIWST*VICPPEQWL*RRKLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*CI*SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGHSD")
        self.assertEqual(record.alignments[0].hsps[5].sbjct, "PXXXNXXNPXAPTXXXX*PXPPXAPXKXXXXXXPXXLXTPXICHR*PKLLKXRKPX*XQXDXXRAMCNQLWKSILGLXECXFIMLYIFPARIWST*VICPPEQWL*RRKLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*CI*SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGHSD")
        self.assertEqual(record.alignments[0].hsps[5].query_start, 589)
        self.assertEqual(record.alignments[0].hsps[5].query_end, 2)
        self.assertEqual(record.alignments[0].hsps[5].sbjct_start, 589)
        self.assertEqual(record.alignments[0].hsps[5].sbjct_end, 2)
        self.assertEqual(record.alignments[0].hsps[6].query, "EXKKAXXXSIXXXQGHV*PALEIHFRIT*VLXHNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFALVHLVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGAF*")
        self.assertEqual(record.alignments[0].hsps[6].match, "E KKA   SI   QGHV*PALEIHFRIT*VL HNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFALVHLVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGAF*")
        self.assertEqual(record.alignments[0].hsps[6].sbjct, "EXKKAXXXSIXXXQGHV*PALEIHFRIT*VLXHNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFALVHLVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGAF*")
        self.assertEqual(record.alignments[0].hsps[6].query_start, 440)
        self.assertEqual(record.alignments[0].hsps[6].query_end, 3)
        self.assertEqual(record.alignments[0].hsps[6].sbjct_start, 440)
        self.assertEqual(record.alignments[0].hsps[6].sbjct_end, 3)
        self.assertEqual(record.alignments[1].hsps[0].query, "SECPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*MH*CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[1].hsps[0].match, "S CPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*  *CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "SXCPSTHETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHIL*IGPDLD*XX*CKIP*QKRIGYFKPALFFM*PV")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 3)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 224)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 3)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 224)
        self.assertEqual(record.alignments[1].hsps[1].query, "YSHCSGGQIT*VLQILAGKIYSIMXQHSXNPKMDFQSWLHMALXXSY*XXXGFLXFRSXGQRWQIXG")
        self.assertEqual(record.alignments[1].hsps[1].match, "YSHCSGGQIT*VLQILAGKIYSIM QHS   K  FQSWLHM     +     F   R  GQR Q  G")
        self.assertEqual(record.alignments[1].hsps[1].sbjct, "YSHCSGGQIT*VLQILAGKIYSIMKQHSVILKWIFQSWLHMXCKVLFKFKRPFSFTRGLGQRXQTPG")
        self.assertEqual(record.alignments[1].hsps[1].query_start, 273)
        self.assertEqual(record.alignments[1].hsps[1].query_end, 473)
        self.assertEqual(record.alignments[1].hsps[1].sbjct_start, 273)
        self.assertEqual(record.alignments[1].hsps[1].sbjct_end, 473)
        self.assertEqual(record.alignments[1].hsps[2].query, "DQNAPPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTKCTNAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL*XSTXVILKWISKAGYTWP")
        self.assertEqual(record.alignments[1].hsps[2].match, "DQ APPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTK  NAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL* ST       SKAGYT P")
        self.assertEqual(record.alignments[1].hsps[2].sbjct, "DQXAPPLMRLFILSTLTGKVGMYAELSKETKKAKTVPSATSSELALTWTKXXNAKSLDKSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVLVDK*PRYSKSWQEKYTAL*NSTQ*S*NGFSKAGYTXP")
        self.assertEqual(record.alignments[1].hsps[2].query_start, 1)
        self.assertEqual(record.alignments[1].hsps[2].query_end, 399)
        self.assertEqual(record.alignments[1].hsps[2].sbjct_start, 1)
        self.assertEqual(record.alignments[1].hsps[2].sbjct_end, 399)
        self.assertEqual(record.alignments[1].hsps[3].query, "IRMPLHS*DSSFCPL*QEKWECMXXXXXXXXRPKRCLQPHPLNWP*LGLNALMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFSPLQPLFWWTNNLGTPNPGRKNIQHYEXALX*S*NGFPKLVTH")
        self.assertEqual(record.alignments[1].hsps[3].match, "IR PLHS*DSSFCPL*QEKWECM        RPKRCLQPHPLNWP*LGL  LMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQF PLQPLFWWTNNLGTPNPGRKNIQHYE AL      FPKLVTH")
        self.assertEqual(record.alignments[1].hsps[3].sbjct, "IRXPLHS*DSSFCPL*QEKWECMQSSQKKQKRPKRCLQPHPLNWP*LGLXPLMQNPLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFFPLQPLFWWTNNLGTPNPGRKNIQHYETALSNPKMDFPKLVTH")
        self.assertEqual(record.alignments[1].hsps[3].query_start, 2)
        self.assertEqual(record.alignments[1].hsps[3].query_end, 394)
        self.assertEqual(record.alignments[1].hsps[3].sbjct_start, 2)
        self.assertEqual(record.alignments[1].hsps[3].sbjct_end, 394)
        self.assertEqual(record.alignments[1].hsps[4].query, "FQEFXSTVANPWG")
        self.assertEqual(record.alignments[1].hsps[4].match, "+Q F ST ANPWG")
        self.assertEqual(record.alignments[1].hsps[4].sbjct, "YQGFRSTXANPWG")
        self.assertEqual(record.alignments[1].hsps[4].query_start, 437)
        self.assertEqual(record.alignments[1].hsps[4].query_end, 475)
        self.assertEqual(record.alignments[1].hsps[4].sbjct_start, 437)
        self.assertEqual(record.alignments[1].hsps[4].sbjct_end, 475)
        self.assertEqual(record.alignments[1].hsps[5].query, "PXICHR*PKLLKXRKPX*XQXDXXRAMCNQLWKSILGLXECXFIMLYIFPARIWST*VICPPEQWL*RRKLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*CI*SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGHSD")
        self.assertEqual(record.alignments[1].hsps[5].match, "P +C R*PK L   K         + MCNQLWK    + EC FIMLYIFPARIWST*VICPPEQWL*R+KLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*  *SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEG SD")
        self.assertEqual(record.alignments[1].hsps[5].sbjct, "PGVCXR*PKPLVNEKGLLNLNKTLQXMCNQLWKIHFRITECCFIMLYIFPARIWST*VICPPEQWL*RKKLSSGQKLLRRCGKTGYIKNNAGLK*PMRFCQGILH*XX*SKSGPIQRMWLKAPFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGXSD")
        self.assertEqual(record.alignments[1].hsps[5].query_start, 472)
        self.assertEqual(record.alignments[1].hsps[5].query_end, 2)
        self.assertEqual(record.alignments[1].hsps[5].sbjct_start, 472)
        self.assertEqual(record.alignments[1].hsps[5].sbjct_end, 2)
        self.assertEqual(record.alignments[1].hsps[6].query, "GHV*PALEIHFRIT*VLXHNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFALVHLVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGA")
        self.assertEqual(record.alignments[1].hsps[6].match, "GHV*PALE  F   *VL HNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFAL  LVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGA")
        self.assertEqual(record.alignments[1].hsps[6].sbjct, "GHV*PALENPF*DY*VLFHNAVYFSCQDLEYLGYLSTRTMAVKEKIEQWSEAAEKMR*NRLHKKQCWFEITYALLSRDFALXXLVQVRANSEDVAEGTVLAFFVSFESSAYIPTFPVKVDKMKSLMSGGA")
        self.assertEqual(record.alignments[1].hsps[6].query_start, 398)
        self.assertEqual(record.alignments[1].hsps[6].query_end, 9)
        self.assertEqual(record.alignments[1].hsps[6].sbjct_start, 398)
        self.assertEqual(record.alignments[1].hsps[6].sbjct_end, 9)
        self.assertEqual(record.alignments[1].hsps[7].query, "CVTSFGNPF*DYXSAXS*CCIFFLPGFGVPRLFVHQNNGCKGEN*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCISAFSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRGILI")
        self.assertEqual(record.alignments[1].hsps[7].match, "CVTSFG       SA S*CCIFFLPGFGVPRLFVHQNNGCKG+N*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCI  FSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRG LI")
        self.assertEqual(record.alignments[1].hsps[7].sbjct, "CVTSFGKSILGLLSAVS*CCIFFLPGFGVPRLFVHQNNGCKGKN*AVVRSC*EDAVKQVT*KTMLV*NNLCAFVKGFCIXGFSPSQGQFRGCG*RHRFGLFCFF*ELCIHSHFSCQSGQNEESHEWRGXLI")
        self.assertEqual(record.alignments[1].hsps[7].query_start, 393)
        self.assertEqual(record.alignments[1].hsps[7].query_end, 1)
        self.assertEqual(record.alignments[1].hsps[7].sbjct_start, 393)
        self.assertEqual(record.alignments[1].hsps[7].sbjct_end, 1)
        self.assertEqual(record.alignments[2].hsps[0].query, "LCNLFYRIFSAASDHCSIFSFTAIVLV")
        self.assertEqual(record.alignments[2].hsps[0].match, "L NL +R+F++  DHCS+  F   V++")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "LTNLIFRLFTSK*DHCSLKQFNNKVVI")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 211)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 291)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 2)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 82)
        self.assertEqual(record.alignments[3].hsps[0].query, "IWST*VICPPEQWL*RRKL")
        self.assertEqual(record.alignments[3].hsps[0].match, "+W T  +CPP  W *RR+L")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "LWGTAPLCPPVPWA*RRQL")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 316)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 260)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 226)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 282)
        self.assertEqual(record.alignments[4].hsps[0].query, "PFWPFLFLLRALHTFPLFLSKWTK")
        self.assertEqual(record.alignments[4].hsps[0].match, "P WP + LL  +H+FP  +  W K")
        self.assertEqual(record.alignments[4].hsps[0].sbjct, "PEWPSINLLPGMHSFPRVIPCWEK")
        self.assertEqual(record.alignments[4].hsps[0].query_start, 106)
        self.assertEqual(record.alignments[4].hsps[0].query_end, 35)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_start, 716)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_end, 645)
        self.assertEqual(record.alignments[5].hsps[0].query, "PFWPFLFLLRALHTFPLFLSKWTK*RVS*VEGHS")
        self.assertEqual(record.alignments[5].hsps[0].match, "P W   FLLR LHT P    K+    V+  EG S")
        self.assertEqual(record.alignments[5].hsps[0].sbjct, "PGWTLRFLLRGLHTPPKAAEKFPSQGVAEPEGFS")
        self.assertEqual(record.alignments[5].hsps[0].query_start, 106)
        self.assertEqual(record.alignments[5].hsps[0].query_end, 5)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_start, 176)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_end, 75)
        self.assertEqual(record.alignments[6].hsps[0].query, "STHETLHFVHFDRKSGNVCRALKRNKKG")
        self.assertEqual(record.alignments[6].hsps[0].match, "+TH    F+H  +K  N+C  +  N++G")
        self.assertEqual(record.alignments[6].hsps[0].sbjct, "TTHSMTVFLHVKKKLSNICIYMPENREG")
        self.assertEqual(record.alignments[6].hsps[0].query_start, 15)
        self.assertEqual(record.alignments[6].hsps[0].query_end, 98)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_start, 82)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_end, 165)
        self.assertEqual(record.alignments[7].hsps[0].query, "HCFLCNLFYRIFSAASDHCSIFSF")
        self.assertEqual(record.alignments[7].hsps[0].match, "HC L ++F RIF    +  S+F+F")
        self.assertEqual(record.alignments[7].hsps[0].sbjct, "HCHLSSVFCRIFFTLEESLSLFAF")
        self.assertEqual(record.alignments[7].hsps[0].query_start, 202)
        self.assertEqual(record.alignments[7].hsps[0].query_end, 273)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_start, 443)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_end, 372)
        self.assertEqual(record.alignments[8].hsps[0].query, "ETLHFVHFDRKSGNVCRALKRNKKGQNGAFSHI")
        self.assertEqual(record.alignments[8].hsps[0].match, "E +H+V F +  G +  + ++ ++ QN   SH+")
        self.assertEqual(record.alignments[8].hsps[0].sbjct, "EKIHYVLF*KNCGKIMTS*RQRQENQNNLLSHV")
        self.assertEqual(record.alignments[8].hsps[0].query_start, 24)
        self.assertEqual(record.alignments[8].hsps[0].query_end, 122)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_start, 141)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_end, 43)
        self.assertEqual(record.alignments[9].hsps[0].query, "PLTKAHRLFQTSIVFYVTCFTASSQQLLTTAQFSPLQPL")
        self.assertEqual(record.alignments[9].hsps[0].match, "PL    RLF + +  ++ C T     L  +  F P++PL")
        self.assertEqual(record.alignments[9].hsps[0].sbjct, "PLKTQGRLFYSKVSLFLKCRTTLCLFLNVSEGFXPIEPL")
        self.assertEqual(record.alignments[9].hsps[0].query_start, 167)
        self.assertEqual(record.alignments[9].hsps[0].query_end, 283)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_start, 211)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_end, 95)
        self.assertEqual(record.alignments[10].hsps[0].query, "CFLCNLFYRIFSAASDHCSIFSFTAI")
        self.assertEqual(record.alignments[10].hsps[0].match, "CF+C L  +I+  A +   +  FT++")
        self.assertEqual(record.alignments[10].hsps[0].sbjct, "CFICKLVLKIYLRAEERTQLIEFTSL")
        self.assertEqual(record.alignments[10].hsps[0].query_start, 205)
        self.assertEqual(record.alignments[10].hsps[0].query_end, 282)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_start, 249)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_end, 326)
        self.assertEqual(record.alignments[11].hsps[0].query, "KSA*VISNQHCFLCNLFYRIFSAASDHCSIFSFTAIVL")
        self.assertEqual(record.alignments[11].hsps[0].match, "+SA +I   HC    +   +FS ASD  S    T ++L")
        self.assertEqual(record.alignments[11].hsps[0].sbjct, "QSAWIIGMSHCAWAVIHCLLFSTASDEKSAVDLTGVLL")
        self.assertEqual(record.alignments[11].hsps[0].query_start, 175)
        self.assertEqual(record.alignments[11].hsps[0].query_end, 288)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_start, 119)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_end, 232)
        self.assertEqual(record.alignments[12].hsps[0].query, "LKAPFWPFLFLLRAL")
        self.assertEqual(record.alignments[12].hsps[0].match, "L  PFW FLF L+AL")
        self.assertEqual(record.alignments[12].hsps[0].sbjct, "LSVPFWKFLFYLQAL")
        self.assertEqual(record.alignments[12].hsps[0].query_start, 115)
        self.assertEqual(record.alignments[12].hsps[0].query_end, 71)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_start, 242)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_end, 198)
        self.assertEqual(record.alignments[13].hsps[0].query, "CCIFFLPGFGVPRLFVHQNNG")
        self.assertEqual(record.alignments[13].hsps[0].match, "C IF++P F    LF+H+  G")
        self.assertEqual(record.alignments[13].hsps[0].sbjct, "CFIFYVPDFPWSNLFLHRGRG")
        self.assertEqual(record.alignments[13].hsps[0].query_start, 339)
        self.assertEqual(record.alignments[13].hsps[0].query_end, 277)
        self.assertEqual(record.alignments[13].hsps[0].sbjct_start, 1396)
        self.assertEqual(record.alignments[13].hsps[0].sbjct_end, 1334)
        self.assertEqual(record.alignments[14].hsps[0].query, "WLKAPFWPFLFLLRALHTFPLFLS")
        self.assertEqual(record.alignments[14].hsps[0].match, "WL  PF PFL  L +L   P  LS")
        self.assertEqual(record.alignments[14].hsps[0].sbjct, "WLLFPFLPFLPFLPSLPFLPFLLS")
        self.assertEqual(record.alignments[14].hsps[0].query_start, 118)
        self.assertEqual(record.alignments[14].hsps[0].query_end, 47)
        self.assertEqual(record.alignments[14].hsps[0].sbjct_start, 153)
        self.assertEqual(record.alignments[14].hsps[0].sbjct_end, 224)
        self.assertEqual(record.alignments[15].hsps[0].query, "WLKAPFWPFLFLLRALHTFPLFLS")
        self.assertEqual(record.alignments[15].hsps[0].match, "WL  PF PFL  L +L   P  LS")
        self.assertEqual(record.alignments[15].hsps[0].sbjct, "WLLFPFLPFLPFLPSLPFLPFLLS")
        self.assertEqual(record.alignments[15].hsps[0].query_start, 118)
        self.assertEqual(record.alignments[15].hsps[0].query_end, 47)
        self.assertEqual(record.alignments[15].hsps[0].sbjct_start, 235)
        self.assertEqual(record.alignments[15].hsps[0].sbjct_end, 164)
        self.assertEqual(record.alignments[16].hsps[0].query, "CFLCNLFYRIFSAASDHCSIFS")
        self.assertEqual(record.alignments[16].hsps[0].match, "CF   + +R+F+    HC+ F+")
        self.assertEqual(record.alignments[16].hsps[0].sbjct, "CFALTVIWRVFAGCRPHCATFT")
        self.assertEqual(record.alignments[16].hsps[0].query_start, 205)
        self.assertEqual(record.alignments[16].hsps[0].query_end, 270)
        self.assertEqual(record.alignments[16].hsps[0].sbjct_start, 13)
        self.assertEqual(record.alignments[16].hsps[0].sbjct_end, 78)
        self.assertEqual(record.alignments[17].hsps[0].query, "VKGFCISAFSPSQGQFRGCG")
        self.assertEqual(record.alignments[17].hsps[0].match, "V GFC+++FSP +    G G")
        self.assertEqual(record.alignments[17].hsps[0].sbjct, "VDGFCVTSFSPKKDTHPGSG")
        self.assertEqual(record.alignments[17].hsps[0].query_start, 174)
        self.assertEqual(record.alignments[17].hsps[0].query_end, 115)
        self.assertEqual(record.alignments[17].hsps[0].sbjct_start, 125)
        self.assertEqual(record.alignments[17].hsps[0].sbjct_end, 184)
        self.assertEqual(record.alignments[18].hsps[0].query, "NQLWKSILGLXECXFIMLYI")
        self.assertEqual(record.alignments[18].hsps[0].match, "NQ+WK  L +  C F+ +Y+")
        self.assertEqual(record.alignments[18].hsps[0].sbjct, "NQMWKHSLEVCMCVFVYIYV")
        self.assertEqual(record.alignments[18].hsps[0].query_start, 388)
        self.assertEqual(record.alignments[18].hsps[0].query_end, 329)
        self.assertEqual(record.alignments[18].hsps[0].sbjct_start, 37)
        self.assertEqual(record.alignments[18].hsps[0].sbjct_end, 96)
        self.assertEqual(record.database_name, ['data/sts'])
        self.assertEqual(record.posted_date, [('Feb 11, 2000  2:37 PM',)])
        self.assertEqual(record.num_letters_in_database, [31998854])
        self.assertEqual(record.num_sequences_in_database, [87792])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.318)
        self.assertAlmostEqual(record.ka_params[1], 0.135)
        self.assertAlmostEqual(record.ka_params[2], 0.401)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.num_hits, 40473548)
        self.assertEqual(record.num_sequences, 87792)
        self.assertEqual(record.num_extends, 487631)
        self.assertEqual(record.num_good_extends, 13175)
        self.assertEqual(record.num_seqs_better_e, 38)
        self.assertEqual(record.query_length, 205)
        self.assertEqual(record.database_length, 10666284)
        self.assertEqual(record.effective_hsp_length, 46)
        self.assertEqual(record.effective_query_length, 158)
        self.assertEqual(record.effective_database_length, 6627852)
        self.assertEqual(record.effective_search_space, 1047200616)
        self.assertEqual(record.effective_search_space_used, 1047200616)
        self.assertEqual(record.frameshift, ('50,', '0.1'))
        self.assertEqual(record.threshold, 13)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 0)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 0.0)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 52)
        self.assertAlmostEqual(record.blast_cutoff[1], 26.7)

    def test_text_2011L_blastp_003(self):
        "Test parsing BLASTP 2.0.11 output (text_2011L_blastp_003)"

        path = os.path.join('Blast', 'text_2011L_blastp_003.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|585505|sp|Q08386|MOPB_RHOCA MOLYBDENUM-PTERIN BINDING\nPROTEIN MOPB")
        self.assertEqual(record.query_letters, 270)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 13)
        self.assertEqual(record.descriptions[0].title, "gi|585505|sp|Q08386|MOPB_RHOCA MOLYBDENUM-PTERIN BINDING PROTEI...")
        self.assertEqual(record.descriptions[0].score, 467)
        self.assertAlmostEqual(record.descriptions[0].e, 1e-131)
        self.assertEqual(record.descriptions[1].title, "gi|585504|sp|Q08385|MOPA_RHOCA MOLYBDENUM-PTERIN BINDING PROTEI...")
        self.assertEqual(record.descriptions[1].score, 207)
        self.assertAlmostEqual(record.descriptions[1].e, 2e-53)
        self.assertEqual(record.descriptions[2].title, "gi|585492|sp|P37733|MODA_AZOVI MOLYBDENUM TRANSPORT PROTEIN MODA")
        self.assertEqual(record.descriptions[2].score, 145)
        self.assertAlmostEqual(record.descriptions[2].e, 9e-35)
        self.assertEqual(record.descriptions[3].title, "gi|1709070|sp|P46930|MODE_ECOLI MOLYBDENUM TRANSPORT PROTEIN MODE")
        self.assertEqual(record.descriptions[3].score, 87)
        self.assertAlmostEqual(record.descriptions[3].e, 5e-17)
        self.assertEqual(record.descriptions[4].title, "gi|1709071|sp|P45324|MODE_HAEIN MOLYBDENUM TRANSPORT PROTEIN MO...")
        self.assertEqual(record.descriptions[4].score, 54)
        self.assertAlmostEqual(record.descriptions[4].e, 2e-7)
        self.assertEqual(record.descriptions[5].title, "gi|585502|sp|P04952|MOP1_CLOPA MOLYBDENUM-PTERIN BINDING PROTEIN I")
        self.assertEqual(record.descriptions[5].score, 53)
        self.assertAlmostEqual(record.descriptions[5].e, 6e-7)
        self.assertEqual(record.descriptions[6].title, "gi|127241|sp|P08854|MOP2_CLOPA MOLYBDENUM-PTERIN BINDING PROTEI...")
        self.assertEqual(record.descriptions[6].score, 52)
        self.assertAlmostEqual(record.descriptions[6].e, 0.000001)
        self.assertEqual(record.descriptions[7].title, "gi|585503|sp|P38366|MOP3_CLOPA MOLYBDENUM-PTERIN BINDING PROTEI...")
        self.assertEqual(record.descriptions[7].score, 51)
        self.assertAlmostEqual(record.descriptions[7].e, 0.000003)
        self.assertEqual(record.descriptions[8].title, "gi|1170996|sp|P45183|MOP_HAEIN PROBABLE MOLYBDENUM-PTERIN BINDI...")
        self.assertEqual(record.descriptions[8].score, 46)
        self.assertAlmostEqual(record.descriptions[8].e, 0.00005)
        self.assertEqual(record.descriptions[9].title, "gi|1709069|sp|P09833|MODC_ECOLI MOLYBDENUM TRANSPORT ATP-BINDIN...")
        self.assertEqual(record.descriptions[9].score, 38)
        self.assertAlmostEqual(record.descriptions[9].e, 0.021)
        self.assertEqual(record.descriptions[10].title, "gi|585500|sp|P37732|MODD_AZOVI MOLYBDENUM TRANSPORT ATP-BINDING...")
        self.assertEqual(record.descriptions[10].score, 33)
        self.assertAlmostEqual(record.descriptions[10].e, 0.53)
        self.assertEqual(record.descriptions[11].title, "gi|2507168|sp|P08838|PT1_BACSU PHOSPHOENOLPYRUVATE-PROTEIN PHOS...")
        self.assertEqual(record.descriptions[11].score, 30)
        self.assertAlmostEqual(record.descriptions[11].e, 4.6)
        self.assertEqual(record.descriptions[12].title, "gi|729786|sp|Q05355|HYDL_STRHA PUTATIVE POLYKETIDE HYDROXYLASE")
        self.assertEqual(record.descriptions[12].score, 29)
        self.assertAlmostEqual(record.descriptions[12].e, 7.9)
        self.assertEqual(len(record.alignments), 13)
        self.assertEqual(record.alignments[0].title, ">gi|585505|sp|Q08386|MOPB_RHOCA MOLYBDENUM-PTERIN BINDING PROTEIN MOPB")
        self.assertEqual(record.alignments[0].length, 270)
        self.assertEqual(record.alignments[1].title, ">gi|585504|sp|Q08385|MOPA_RHOCA MOLYBDENUM-PTERIN BINDING PROTEIN MOPA")
        self.assertEqual(record.alignments[1].length, 265)
        self.assertEqual(record.alignments[2].title, ">gi|585492|sp|P37733|MODA_AZOVI MOLYBDENUM TRANSPORT PROTEIN MODA")
        self.assertEqual(record.alignments[2].length, 270)
        self.assertEqual(record.alignments[3].title, ">gi|1709070|sp|P46930|MODE_ECOLI MOLYBDENUM TRANSPORT PROTEIN MODE")
        self.assertEqual(record.alignments[3].length, 262)
        self.assertEqual(record.alignments[4].title, ">gi|1709071|sp|P45324|MODE_HAEIN MOLYBDENUM TRANSPORT PROTEIN MODE HOMOLOG")
        self.assertEqual(record.alignments[4].length, 255)
        self.assertEqual(record.alignments[5].title, ">gi|585502|sp|P04952|MOP1_CLOPA MOLYBDENUM-PTERIN BINDING PROTEIN I")
        self.assertEqual(record.alignments[5].length, 68)
        self.assertEqual(record.alignments[6].title, ">gi|127241|sp|P08854|MOP2_CLOPA MOLYBDENUM-PTERIN BINDING PROTEIN II")
        self.assertEqual(record.alignments[6].length, 68)
        self.assertEqual(record.alignments[7].title, ">gi|585503|sp|P38366|MOP3_CLOPA MOLYBDENUM-PTERIN BINDING PROTEIN III")
        self.assertEqual(record.alignments[7].length, 68)
        self.assertEqual(record.alignments[8].title, ">gi|1170996|sp|P45183|MOP_HAEIN PROBABLE MOLYBDENUM-PTERIN BINDING PROTEIN")
        self.assertEqual(record.alignments[8].length, 69)
        self.assertEqual(record.alignments[9].title, ">gi|1709069|sp|P09833|MODC_ECOLI MOLYBDENUM TRANSPORT ATP-BINDING PROTEIN MODC")
        self.assertEqual(record.alignments[9].length, 352)
        self.assertEqual(record.alignments[10].title, ">gi|585500|sp|P37732|MODD_AZOVI MOLYBDENUM TRANSPORT ATP-BINDING PROTEIN MODD")
        self.assertEqual(record.alignments[10].length, 380)
        self.assertEqual(record.alignments[11].title, ">gi|2507168|sp|P08838|PT1_BACSU PHOSPHOENOLPYRUVATE-PROTEIN PHOSPHOTRANSFERASE (PHOSPHOTRANSFERASE SYSTEM, ENZYME I)")
        self.assertEqual(record.alignments[11].length, 570)
        self.assertEqual(record.alignments[12].title, ">gi|729786|sp|Q05355|HYDL_STRHA PUTATIVE POLYKETIDE HYDROXYLASE")
        self.assertEqual(record.alignments[12].length, 555)
        self.assertEqual(record.alignments[0].hsps[0].score, 1189)
        self.assertAlmostEqual(record.alignments[0].hsps[0].bits, 467)
        self.assertAlmostEqual(record.alignments[0].hsps[0].expect, 1e-131)
        self.assertEqual(len(record.alignments[0].hsps), 1)
        self.assertEqual(record.alignments[1].hsps[0].score, 521)
        self.assertAlmostEqual(record.alignments[1].hsps[0].bits, 207)
        self.assertAlmostEqual(record.alignments[1].hsps[0].expect, 2e-53)
        self.assertEqual(len(record.alignments[1].hsps), 1)
        self.assertEqual(record.alignments[2].hsps[0].score, 362)
        self.assertAlmostEqual(record.alignments[2].hsps[0].bits, 145)
        self.assertAlmostEqual(record.alignments[2].hsps[0].expect, 9e-35)
        self.assertEqual(record.alignments[2].hsps[1].score, 98)
        self.assertAlmostEqual(record.alignments[2].hsps[1].bits, 42.6)
        self.assertAlmostEqual(record.alignments[2].hsps[1].expect, 8e-4)
        self.assertEqual(len(record.alignments[2].hsps), 2)
        self.assertEqual(record.alignments[3].hsps[0].score, 211)
        self.assertAlmostEqual(record.alignments[3].hsps[0].bits, 86.6)
        self.assertAlmostEqual(record.alignments[3].hsps[0].expect, 5e-17)
        self.assertEqual(len(record.alignments[3].hsps), 1)
        self.assertEqual(record.alignments[4].hsps[0].score, 128)
        self.assertAlmostEqual(record.alignments[4].hsps[0].bits, 54.3)
        self.assertAlmostEqual(record.alignments[4].hsps[0].expect, 2e-7)
        self.assertEqual(len(record.alignments[4].hsps), 1)
        self.assertEqual(record.alignments[5].hsps[0].score, 125)
        self.assertAlmostEqual(record.alignments[5].hsps[0].bits, 53.1)
        self.assertAlmostEqual(record.alignments[5].hsps[0].expect, 6e-7)
        self.assertEqual(record.alignments[5].hsps[1].score, 84)
        self.assertAlmostEqual(record.alignments[5].hsps[1].bits, 37.1)
        self.assertAlmostEqual(record.alignments[5].hsps[1].expect, 0.036)
        self.assertEqual(len(record.alignments[5].hsps), 2)
        self.assertEqual(record.alignments[6].hsps[0].score, 123)
        self.assertAlmostEqual(record.alignments[6].hsps[0].bits, 52.3)
        self.assertAlmostEqual(record.alignments[6].hsps[0].expect, 1e-6)
        self.assertEqual(record.alignments[6].hsps[1].score, 86)
        self.assertAlmostEqual(record.alignments[6].hsps[1].bits, 37.9)
        self.assertAlmostEqual(record.alignments[6].hsps[1].expect, 0.021)
        self.assertEqual(len(record.alignments[6].hsps), 2)
        self.assertEqual(record.alignments[7].hsps[0].score, 119)
        self.assertAlmostEqual(record.alignments[7].hsps[0].bits, 50.8)
        self.assertAlmostEqual(record.alignments[7].hsps[0].expect, 3e-6)
        self.assertEqual(record.alignments[7].hsps[1].score, 83)
        self.assertAlmostEqual(record.alignments[7].hsps[1].bits, 36.7)
        self.assertAlmostEqual(record.alignments[7].hsps[1].expect, 0.047)
        self.assertEqual(len(record.alignments[7].hsps), 2)
        self.assertEqual(record.alignments[8].hsps[0].score, 108)
        self.assertAlmostEqual(record.alignments[8].hsps[0].bits, 46.5)
        self.assertAlmostEqual(record.alignments[8].hsps[0].expect, 5e-5)
        self.assertEqual(len(record.alignments[8].hsps), 1)
        self.assertEqual(record.alignments[9].hsps[0].score, 86)
        self.assertAlmostEqual(record.alignments[9].hsps[0].bits, 37.9)
        self.assertAlmostEqual(record.alignments[9].hsps[0].expect, 0.021)
        self.assertEqual(len(record.alignments[9].hsps), 1)
        self.assertEqual(record.alignments[10].hsps[0].score, 74)
        self.assertAlmostEqual(record.alignments[10].hsps[0].bits, 33.2)
        self.assertAlmostEqual(record.alignments[10].hsps[0].expect, 0.53)
        self.assertEqual(len(record.alignments[10].hsps), 1)
        self.assertEqual(record.alignments[11].hsps[0].score, 66)
        self.assertAlmostEqual(record.alignments[11].hsps[0].bits, 30.1)
        self.assertAlmostEqual(record.alignments[11].hsps[0].expect, 4.6)
        self.assertEqual(len(record.alignments[11].hsps), 1)
        self.assertEqual(record.alignments[12].hsps[0].score, 64)
        self.assertAlmostEqual(record.alignments[12].hsps[0].bits, 29.3)
        self.assertAlmostEqual(record.alignments[12].hsps[0].expect, 7.9)
        self.assertEqual(len(record.alignments[12].hsps), 1)
        self.assertEqual(record.alignments[0].hsps[0].identities, (247, 270))
        self.assertEqual(record.alignments[0].hsps[0].positives, (247, 270))
        self.assertEqual(record.alignments[1].hsps[0].identities, (123, 259))
        self.assertEqual(record.alignments[1].hsps[0].positives, (155, 259))
        self.assertEqual(record.alignments[1].hsps[0].gaps, (13, 259))
        self.assertEqual(record.alignments[2].hsps[0].identities, (93, 253))
        self.assertEqual(record.alignments[2].hsps[0].positives, (132, 253))
        self.assertEqual(record.alignments[2].hsps[0].gaps, (8, 253))
        self.assertEqual(record.alignments[2].hsps[1].identities, (33, 99))
        self.assertEqual(record.alignments[2].hsps[1].positives, (47, 99))
        self.assertEqual(record.alignments[2].hsps[1].gaps, (7, 99))
        self.assertEqual(record.alignments[3].hsps[0].identities, (76, 247))
        self.assertEqual(record.alignments[3].hsps[0].positives, (114, 247))
        self.assertEqual(record.alignments[3].hsps[0].gaps, (17, 247))
        self.assertEqual(record.alignments[4].hsps[0].identities, (46, 170))
        self.assertEqual(record.alignments[4].hsps[0].positives, (76, 170))
        self.assertEqual(record.alignments[4].hsps[0].gaps, (3, 170))
        self.assertEqual(record.alignments[5].hsps[0].identities, (25, 64))
        self.assertEqual(record.alignments[5].hsps[0].positives, (43, 64))
        self.assertEqual(record.alignments[5].hsps[1].identities, (19, 63))
        self.assertEqual(record.alignments[5].hsps[1].positives, (36, 63))
        self.assertEqual(record.alignments[6].hsps[0].identities, (24, 64))
        self.assertEqual(record.alignments[6].hsps[0].positives, (43, 64))
        self.assertEqual(record.alignments[6].hsps[1].identities, (21, 63))
        self.assertEqual(record.alignments[6].hsps[1].positives, (36, 63))
        self.assertEqual(record.alignments[7].hsps[0].identities, (24, 64))
        self.assertEqual(record.alignments[7].hsps[0].positives, (43, 64))
        self.assertEqual(record.alignments[7].hsps[1].identities, (20, 63))
        self.assertEqual(record.alignments[7].hsps[1].positives, (37, 63))
        self.assertEqual(record.alignments[8].hsps[0].identities, (19, 67))
        self.assertEqual(record.alignments[8].hsps[0].positives, (46, 67))
        self.assertEqual(record.alignments[9].hsps[0].identities, (23, 62))
        self.assertEqual(record.alignments[9].hsps[0].positives, (37, 62))
        self.assertEqual(record.alignments[9].hsps[0].gaps, (1, 62))
        self.assertEqual(record.alignments[10].hsps[0].identities, (41, 143))
        self.assertEqual(record.alignments[10].hsps[0].positives, (62, 143))
        self.assertEqual(record.alignments[10].hsps[0].gaps, (12, 143))
        self.assertEqual(record.alignments[11].hsps[0].identities, (32, 141))
        self.assertEqual(record.alignments[11].hsps[0].positives, (61, 141))
        self.assertEqual(record.alignments[11].hsps[0].gaps, (6, 141))
        self.assertEqual(record.alignments[12].hsps[0].identities, (21, 62))
        self.assertEqual(record.alignments[12].hsps[0].positives, (29, 62))
        self.assertEqual(record.alignments[0].hsps[0].query, "MAATKQGGGDDGRCARGVVLERTGARMGAERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNXXXXXXXXXXXXXXXXXXXXXXXAGEKLIAAYGAIEAGVAKLLSSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILALP")
        self.assertEqual(record.alignments[0].hsps[0].match, "MAATKQGGGDDGRCARGVVLERTGARMGAERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNN                       AGEKLIAAYGAIEAGVAKLLSSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILALP")
        self.assertEqual(record.alignments[0].hsps[0].sbjct, "MAATKQGGGDDGRCARGVVLERTGARMGAERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNLLAAPVVTAAPGGKAGGGAVLTPAGEKLIAAYGAIEAGVAKLLSSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILALP")
        self.assertEqual(record.alignments[0].hsps[0].query_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].query_end, 270)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_start, 1)
        self.assertEqual(record.alignments[0].hsps[0].sbjct_end, 270)
        self.assertEqual(record.alignments[1].hsps[0].query, "LERTGA-RMGAERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNXXXXXXXXXXXXXXXXXXXXXXXAGEKLIAAYGAIEAGVAKLL-------SSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILALP")
        self.assertEqual(record.alignments[1].hsps[0].match, "L+R GA R+G +R+ LL AI R G+I+ AAREVGLSYK AWD V  +NN                       AG+ LIA +G +E  + K L       S+ EK+LN      L  L++RTS RN   C V  V    V A+V + L +G  LTAVIT RSA EM LAPG EV AL+K++FV+LA  G P R+S  NR+ G V  R D P+++E+ LDLG  K+ITA IT  SA+ L L PGV  TAL K+SHVILA+P")
        self.assertEqual(record.alignments[1].hsps[0].sbjct, "LQRAGAPRVGGDRIRLLEAIARHGTIAGAAREVGLSYKTAWDAVGTLNNLFEQPLVEAAPGGRTGGNARVTEAGQALIAGFGLLEGALTKALGVLEGGVSAPEKALN-----TLWSLTMRTSNRNTLRCTVTRVTLGAVNAEVELALTDGHSLTAVITERSATEMGLAPGVEVFALIKASFVMLAAGGDPGRISACNRLTGIVAARTDGPVNTEIILDLGNCKSITAVITHTSADALGLAPGVPATALFKASHVILAMP")
        self.assertEqual(record.alignments[1].hsps[0].query_start, 20)
        self.assertEqual(record.alignments[1].hsps[0].query_end, 270)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_start, 12)
        self.assertEqual(record.alignments[1].hsps[0].sbjct_end, 265)
        self.assertEqual(record.alignments[2].hsps[0].query, "GARMGAERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNXXXXXXXXXXXXXXXXXXXXXXXAGEKLIAAYGAIEAGVAKLLSSFEKSLNLDPA-------EVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILAL")
        self.assertEqual(record.alignments[2].hsps[0].match, "G  +   R+ LL AI R GSI+ AA+ V LSYKAAWD +  MNN                        G +++A Y A+E      L    + LN            ++  +S++TSARN +A  V  +    V  +VR+RL    ++ AVIT  SA  + LA G EV ALVKS+ V+L       +L+ RN++ G VI+  + P+++EVTL L  G+++T  +T DS + L L PGV   A  KSS VILA+")
        self.assertEqual(record.alignments[2].hsps[0].sbjct, "GTALSDTRIRLLEAIEREGSINRAAKVVPLSYKAAWDAIDTMNNLAPEPLVVRVAGGRQGGGTQLTDYGRRIVAMYRALEIEYQSALDRLSERLNEVTGGDIQAFQRLMHSMSMKTSARNQFAGIVTGLRVGGVDYEVRIRLDAENEIAAVITKASAENLELAIGKEVFALVKSSSVMLT-TEPSLKLTARNQLWGEVIDIHEGPVNNEVTLALPSGRSVTCVVTADSCKALGLAPGVAACAFFKSSSVILAV")
        self.assertEqual(record.alignments[2].hsps[0].query_start, 24)
        self.assertEqual(record.alignments[2].hsps[0].query_end, 269)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_start, 17)
        self.assertEqual(record.alignments[2].hsps[0].sbjct_end, 268)
        self.assertEqual(record.alignments[2].hsps[1].query, "AIEAGVAKLLSSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAG")
        self.assertEqual(record.alignments[2].hsps[1].match, "AI   V  L+ S    L  +P       SL+ +ARN    +V  +    V  +V + L  G+ +T V+TA S   + LAPG    A  KS+ V+LA  G")
        self.assertEqual(record.alignments[2].hsps[1].sbjct, "AIGKEVFALVKSSSVMLTTEP-------SLKLTARNQLWGEVIDIHEGPVNNEVTLALPSGRSVTCVVTADSCKALGLAPGVAACAFFKSSSVILAVYG")
        self.assertEqual(record.alignments[2].hsps[1].query_start, 101)
        self.assertEqual(record.alignments[2].hsps[1].query_end, 199)
        self.assertEqual(record.alignments[2].hsps[1].sbjct_start, 179)
        self.assertEqual(record.alignments[2].hsps[1].sbjct_end, 270)
        self.assertEqual(record.alignments[3].hsps[0].query, "RVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNXXXXXXXXXXXXXXXXXXXXXXXAGEKLIAAY---GAIEAGVAKLLSSFEK-SLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQD-LTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVR----NRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILA")
        self.assertEqual(record.alignments[3].hsps[0].match, "R++LL  I  +GSIS  A++ G+SYK+AWD +  MN                         G++LI  Y     I+     +LS  +   LN   A + R  SL+TSARN W   + +   DDV   V + L +G+  L   ITA+S A + L  G EVL L+K+ +V     G+ +  +V     N++ G +          EV + L  G+T+ AT+  +  E   L  G   TA   +  VI+A")
        self.assertEqual(record.alignments[3].hsps[0].sbjct, "RISLLKHIALSGSISQGAKDAGISYKSAWDAINEMNQLSEHILVERATGGKGGGGAVLTRYGQRLIQLYDLLAQIQQKAFDVLSDDDALPLNSLLAAISR-FSLQTSARNQWFGTITARDHDDVQQHVDVLLADGKTRLKVAITAQSGARLGLDEGKEVLILLKAPWV-----GITQDEAVAQNADNQLPGIISHIERGAEQCEVLMALPDGQTLCATVPVN--EATSLQQGQNVTAYFNADSVIIA")
        self.assertEqual(record.alignments[3].hsps[0].query_start, 31)
        self.assertEqual(record.alignments[3].hsps[0].query_end, 268)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_start, 21)
        self.assertEqual(record.alignments[3].hsps[0].sbjct_end, 259)
        self.assertEqual(record.alignments[4].hsps[0].query, "ERVALLAAIGRTGSISAAAREVGLSYKAAWDGVQAMNNXXXXXXXXXXXXXXXXXXXXXXXAGEKLIAAYGAIEAGVAKLLSSF-EKSLNLDP-AEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRL-GEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLA")
        self.assertEqual(record.alignments[4].hsps[0].match, "+RV LL  I + GSI+ AA+   +SYK+AWD ++AMN                          E+L+  Y  +E           ++S+ LD         SL++SARN +  +V      D    V + + G    L   IT +S+A ++L    EV+ + K+ +V ++")
        self.assertEqual(record.alignments[4].hsps[0].sbjct, "KRVRLLKEIQQCGSINQAAKNAKVSYKSAWDHLEAMNKISPRPLLERNTGGKNGGGTALTTYAERLLQLYDLLERTQEHAFHILQDESVPLDSLLTATARFSLQSSARNQFFGRVAQQRIIDSRCVVDVNVQGLPTPLQVSITTKSSARLKLITEKEVMLMFKAPWVKIS")
        self.assertEqual(record.alignments[4].hsps[0].query_start, 30)
        self.assertEqual(record.alignments[4].hsps[0].query_end, 196)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_start, 21)
        self.assertEqual(record.alignments[4].hsps[0].sbjct_end, 190)
        self.assertEqual(record.alignments[5].hsps[0].query, "LSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVIL")
        self.assertEqual(record.alignments[5].hsps[0].match, "+S RN+++G+V+      +++EV L++ GG  IT+ I+ DS E L +  G E TA+IKS+ V++")
        self.assertEqual(record.alignments[5].hsps[0].sbjct, "ISARNQLKGKVVGLKKGVITAEVVLEIAGGNKITSIISLDSVEELGVKEGAELTAVIKSTDVMI")
        self.assertEqual(record.alignments[5].hsps[0].query_start, 204)
        self.assertEqual(record.alignments[5].hsps[0].query_end, 267)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_start, 3)
        self.assertEqual(record.alignments[5].hsps[0].sbjct_end, 66)
        self.assertEqual(record.alignments[5].hsps[1].query, "SARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLL")
        self.assertEqual(record.alignments[5].hsps[1].match, "SARN    KV  +    + A+V + +  G  +T++I+  S  E+ +  G+E+ A++KS  V++")
        self.assertEqual(record.alignments[5].hsps[1].sbjct, "SARNQLKGKVVGLKKGVITAEVVLEIAGGNKITSIISLDSVEELGVKEGAELTAVIKSTDVMI")
        self.assertEqual(record.alignments[5].hsps[1].query_start, 133)
        self.assertEqual(record.alignments[5].hsps[1].query_end, 195)
        self.assertEqual(record.alignments[5].hsps[1].sbjct_start, 4)
        self.assertEqual(record.alignments[5].hsps[1].sbjct_end, 66)
        self.assertEqual(record.alignments[6].hsps[0].query, "LSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVIL")
        self.assertEqual(record.alignments[6].hsps[0].match, "+S RN+++G+V+      +++EV L++ GG  IT+ I+ DS E L +  G E TA++KS+ V++")
        self.assertEqual(record.alignments[6].hsps[0].sbjct, "ISARNQLKGKVVGLKKGVVTAEVVLEIAGGNKITSIISLDSVEELGVKEGAELTAVVKSTDVMI")
        self.assertEqual(record.alignments[6].hsps[0].query_start, 204)
        self.assertEqual(record.alignments[6].hsps[0].query_end, 267)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_start, 3)
        self.assertEqual(record.alignments[6].hsps[0].sbjct_end, 66)
        self.assertEqual(record.alignments[6].hsps[1].query, "SARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLL")
        self.assertEqual(record.alignments[6].hsps[1].match, "SARN    KV  +    V A+V + +  G  +T++I+  S  E+ +  G+E+ A+VKS  V++")
        self.assertEqual(record.alignments[6].hsps[1].sbjct, "SARNQLKGKVVGLKKGVVTAEVVLEIAGGNKITSIISLDSVEELGVKEGAELTAVVKSTDVMI")
        self.assertEqual(record.alignments[6].hsps[1].query_start, 133)
        self.assertEqual(record.alignments[6].hsps[1].query_end, 195)
        self.assertEqual(record.alignments[6].hsps[1].sbjct_start, 4)
        self.assertEqual(record.alignments[6].hsps[1].sbjct_end, 66)
        self.assertEqual(record.alignments[7].hsps[0].query, "LSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVIL")
        self.assertEqual(record.alignments[7].hsps[0].match, "+S RN+++G+V+      +++EV L++ GG  +T+ I+ DS E L +  G E TA+IKS+ V++")
        self.assertEqual(record.alignments[7].hsps[0].sbjct, "ISARNQLKGKVVAVKKGLVTAEVVLEIAGGDKVTSIISLDSIEDLGVKEGTELTAVIKSTDVMI")
        self.assertEqual(record.alignments[7].hsps[0].query_start, 204)
        self.assertEqual(record.alignments[7].hsps[0].query_end, 267)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_start, 3)
        self.assertEqual(record.alignments[7].hsps[0].sbjct_end, 66)
        self.assertEqual(record.alignments[7].hsps[1].query, "SARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLL")
        self.assertEqual(record.alignments[7].hsps[1].match, "SARN    KV +V    V A+V + +  G  +T++I+  S  ++ +  G+E+ A++KS  V++")
        self.assertEqual(record.alignments[7].hsps[1].sbjct, "SARNQLKGKVVAVKKGLVTAEVVLEIAGGDKVTSIISLDSIEDLGVKEGTELTAVIKSTDVMI")
        self.assertEqual(record.alignments[7].hsps[1].query_start, 133)
        self.assertEqual(record.alignments[7].hsps[1].query_end, 195)
        self.assertEqual(record.alignments[7].hsps[1].sbjct_start, 4)
        self.assertEqual(record.alignments[7].hsps[1].sbjct_end, 66)
        self.assertEqual(record.alignments[8].hsps[0].query, "RLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKSSHVILAL")
        self.assertEqual(record.alignments[8].hsps[0].match, "++S RN+++G+V+   +  +++ V +D+GGG  +++T++  + + L+L  G E  A+IK++ V++ +")
        self.assertEqual(record.alignments[8].hsps[0].sbjct, "KISARNQLKGKVVSIENGSVNAIVHIDIGGGNVLSSTVSLAAVKELNLEVGKEAYAIIKATSVMVGV")
        self.assertEqual(record.alignments[8].hsps[0].query_start, 203)
        self.assertEqual(record.alignments[8].hsps[0].query_end, 269)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_start, 2)
        self.assertEqual(record.alignments[8].hsps[0].sbjct_end, 68)
        self.assertEqual(record.alignments[9].hsps[0].query, "PERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTITATITRDSAEMLDLHPGVETTALIKS")
        self.assertEqual(record.alignments[9].hsps[0].match, "P++ S+RN +R +V+   D     EV L++ GGKT+ A I+  + + L + PG+   A IKS")
        self.assertEqual(record.alignments[9].hsps[0].sbjct, "PQQTSIRNVLRAKVVNSYDDNGQVEVELEV-GGKTLWARISPWARDELAIKPGLWLYAQIKS")
        self.assertEqual(record.alignments[9].hsps[0].query_start, 201)
        self.assertEqual(record.alignments[9].hsps[0].query_end, 262)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_start, 287)
        self.assertEqual(record.alignments[9].hsps[0].sbjct_end, 347)
        self.assertEqual(record.alignments[10].hsps[0].query, "EVLRGLSLRTSARNAWACKVWSVAA--DDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGG-GKTITATITRDSAEMLDLHPGVETTALIKS")
        self.assertEqual(record.alignments[10].hsps[0].match, "+++  L L T+        + SV A  DD     R+    G    AV+ AR       APG  +   V +  V LA + + E  S+ N +   V E ++A   + V + L   G  + A ITR S + L + PG    A IK+")
        self.assertEqual(record.alignments[10].hsps[0].sbjct, "DIMARLDLPTAFHEDAGVVIESVVAEHDDHYHLTRLAFPGG----AVLVARRPE----APGQRLRLRVHARDVSLANSRI-EDSSITNVLPATVREVVEADTPAHVLVRLEAEGTPLIARITRRSCDQLGIAPGRRMWAQIKA")
        self.assertEqual(record.alignments[10].hsps[0].query_start, 123)
        self.assertEqual(record.alignments[10].hsps[0].query_end, 262)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_start, 242)
        self.assertEqual(record.alignments[10].hsps[0].sbjct_end, 375)
        self.assertEqual(record.alignments[11].hsps[0].query, "AAYGAIEAGVAKLLSSFEKSLNLDP-AEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQDLTAVITARSAAEMRLAPGSEVLALVKSNFVLLAGAGVPERLSVRNRVRGRVIERIDAPLSSEVTLDLGGGKTI")
        self.assertEqual(record.alignments[11].hsps[0].match, "AA G I+ GV  ++      + +DP AE ++    + +A  A   + W+   ++       + G   +L A I      +  L  G E + L ++ F+ +    +P      +  +  V+ER++       TLD+GG K +")
        self.assertEqual(record.alignments[11].hsps[0].sbjct, "AATGTIQNGVTVIVDGINGDVIIDPSAETVKEYEEKHNAYLAQKAE-WAKLVNEPTVS---KDGHHVELAANIGTPDDVKGVLENGGEAVGLYRTEFLYMGRDQLPTEDEQFDAYK-TVLERMEGKSVVVRTLDIGGDKEL")
        self.assertEqual(record.alignments[11].hsps[0].query_start, 97)
        self.assertEqual(record.alignments[11].hsps[0].query_end, 236)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_start, 207)
        self.assertEqual(record.alignments[11].hsps[0].sbjct_end, 342)
        self.assertEqual(record.alignments[12].hsps[0].query, "AIEAGVAKLLSSFEKSLNLDPAEVLRGLSLRTSARNAWACKVWSVAADDVAAQVRMRLGEGQ")
        self.assertEqual(record.alignments[12].hsps[0].match, "A+E G     S+  +S   DPA V   +  R S  +      + VAAD   + VR +LG GQ")
        self.assertEqual(record.alignments[12].hsps[0].sbjct, "AVELGGEIRFSTELQSFEQDPAGVTAVIKSRRSGEHTTVRADYLVAADGPRSPVREQLGIGQ")
        self.assertEqual(record.alignments[12].hsps[0].query_start, 101)
        self.assertEqual(record.alignments[12].hsps[0].query_end, 162)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_start, 136)
        self.assertEqual(record.alignments[12].hsps[0].sbjct_end, 197)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Feb 2, 2000  9:39 AM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.316)
        self.assertAlmostEqual(record.ka_params[1], 0.131)
        self.assertAlmostEqual(record.ka_params[2], 0.361)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 12068104)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 396723)
        self.assertEqual(record.num_good_extends, 1066)
        self.assertEqual(record.num_seqs_better_e, 13)
        self.assertEqual(record.hsps_no_gap, 10)
        self.assertEqual(record.hsps_prelim_gapped, 3)
        self.assertEqual(record.hsps_gapped, 18)
        self.assertEqual(record.query_length, 270)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 56)
        self.assertEqual(record.effective_query_length, 214)
        self.assertEqual(record.effective_database_length, 25046113)
        self.assertEqual(record.effective_search_space, 5359868182)
        self.assertEqual(record.effective_search_space_used, 5359868182)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.6)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 64)
        self.assertAlmostEqual(record.blast_cutoff[1], 29.3)

    def test_text_2011L_blastp_007(self):
        "Test parsing BLASTP 2.0.11 output (text_2011L_blastp_007)"

        path = os.path.join('Blast', 'text_2011L_blastp_007.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|730725|sp|Q05362|SCHB_STRHA SCHB PROTEIN")
        self.assertEqual(record.query_letters, 138)
        self.assertEqual(record.database, "data/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 0)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['data/swissprot'])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(record.posted_date, [('Feb 2, 2000  9:39 AM',)])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.319)
        self.assertAlmostEqual(record.ka_params[1], 0.139)
        self.assertAlmostEqual(record.ka_params[2], 0.415)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 8952568)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 387403)
        self.assertEqual(record.num_good_extends, 727)
        self.assertEqual(record.num_seqs_better_e, 23)
        self.assertEqual(record.hsps_no_gap, 13)
        self.assertEqual(record.hsps_prelim_gapped, 10)
        self.assertEqual(record.hsps_gapped, 23)
        self.assertEqual(record.query_length, 138)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 47)
        self.assertEqual(record.effective_query_length, 91)
        self.assertEqual(record.effective_database_length, 25786435)
        self.assertEqual(record.effective_search_space, 2346565585)
        self.assertEqual(record.effective_search_space_used, 2346565585)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.4)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.7)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 61)
        self.assertAlmostEqual(record.blast_cutoff[1], 28.2)

    def test_text_2011L_psiblast_003(self):
        "Test parsing BLASTP 2.0.11 output (text_2011L_psiblast_003)"

        path = os.path.join('Blast', 'text_2011L_phiblast_003.txt')
        handle = open(path)
        record = self.parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.11')
        self.assertEqual(record.date, "Jan-20-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "gi|129628|sp|P07175|PARA_AGRTU PARA PROTEIN")
        self.assertEqual(record.query_letters, 222)
        self.assertEqual(record.database, "/taiyang/jchang/parsers/swissprot")
        self.assertEqual(record.database_sequences, 82258)
        self.assertEqual(record.database_letters, 29652561)
        self.assertEqual(len(record.descriptions), 107)
        self.assertEqual(record.descriptions[0].title, "gi|129628|sp|P07175|PARA_AGRTU PARA PROTEIN")
        self.assertEqual(record.descriptions[0].score, 427)
        self.assertAlmostEqual(record.descriptions[0].e, 1e-120)
        self.assertEqual(record.descriptions[1].title, "gi|138456|sp|P06665|VIC1_AGRT6 VIRC1 PROTEIN")
        self.assertEqual(record.descriptions[1].score, 53)
        self.assertAlmostEqual(record.descriptions[1].e, 3e-7)
        self.assertEqual(record.descriptions[2].title, "gi|138454|sp|P13459|VIC1_AGRRA VIRC1 PROTEIN")
        self.assertEqual(record.descriptions[2].score, 52)
        self.assertAlmostEqual(record.descriptions[2].e, 7e-7)
        self.assertEqual(record.descriptions[3].title, "gi|138455|sp|P07165|VIC1_AGRT5 VIRC1 PROTEIN")
        self.assertEqual(record.descriptions[3].score, 49)
        self.assertAlmostEqual(record.descriptions[3].e, 0.000006)
        self.assertEqual(record.descriptions[4].title, "gi|2497977|sp|P72190|YCAB_PSEFR HYPOTHETICAL 30.2 KD PROTEIN IN...")
        self.assertEqual(record.descriptions[4].score, 42)
        self.assertAlmostEqual(record.descriptions[4].e, 0.001)
        self.assertEqual(record.descriptions[5].title, "gi|586852|sp|P37522|SOJ_BACSU SOJ PROTEIN")
        self.assertEqual(record.descriptions[5].score, 42)
        self.assertAlmostEqual(record.descriptions[5].e, 0.001)
        self.assertEqual(record.descriptions[6].title, "gi|132348|sp|P05682|REPA_AGRRA POSSIBLE REPLICATION PROTEIN A")
        self.assertEqual(record.descriptions[6].score, 41)
        self.assertAlmostEqual(record.descriptions[6].e, 0.001)
        self.assertEqual(record.descriptions[7].title, "gi|120545|sp|P28373|CHLL_SYNY3 PROTOCHLOROPHYLLIDE REDUCTASE IR...")
        self.assertEqual(record.descriptions[7].score, 41)
        self.assertAlmostEqual(record.descriptions[7].e, 0.001)
        self.assertEqual(record.descriptions[8].title, "gi|2496226|sp|Q60283|YZ24_METJA HYPOTHETICAL PROTEIN MJECL24")
        self.assertEqual(record.descriptions[8].score, 40)
        self.assertAlmostEqual(record.descriptions[8].e, 0.003)
        self.assertEqual(record.descriptions[9].title, "gi|3024144|sp|Q55900|MIND_SYNY3 SEPTUM SITE-DETERMINING PROTEIN...")
        self.assertEqual(record.descriptions[9].score, 40)
        self.assertAlmostEqual(record.descriptions[9].e, 0.003)
        self.assertEqual(record.descriptions[10].title, "gi|1709100|sp|P53382|MRP_MYCLE MRP PROTEIN HOMOLOG")
        self.assertEqual(record.descriptions[10].score, 40)
        self.assertAlmostEqual(record.descriptions[10].e, 0.003)
        self.assertEqual(record.descriptions[11].title, "gi|462615|sp|P21590|MRP_ECOLI MRP PROTEIN")
        self.assertEqual(record.descriptions[11].score, 40)
        self.assertAlmostEqual(record.descriptions[11].e, 0.004)
        self.assertEqual(record.descriptions[12].title, "gi|3024135|sp|P56346|MIND_CHLVU PUTATIVE SEPTUM SITE-DETERMININ...")
        self.assertEqual(record.descriptions[12].score, 39)
        self.assertAlmostEqual(record.descriptions[12].e, 0.005)
        self.assertEqual(record.descriptions[13].title, "gi|1709101|sp|P53383|MRP_SYNY3 MRP PROTEIN HOMOLOG")
        self.assertEqual(record.descriptions[13].score, 39)
        self.assertAlmostEqual(record.descriptions[13].e, 0.009)
        self.assertEqual(record.descriptions[14].title, "gi|127097|sp|P18197|MIND_ECOLI SEPTUM SITE-DETERMINING PROTEIN ...")
        self.assertEqual(record.descriptions[14].score, 39)
        self.assertAlmostEqual(record.descriptions[14].e, 0.009)
        self.assertEqual(record.descriptions[15].title, "gi|3183081|sp|O24999|MRP_HELPY MRP PROTEIN HOMOLOG")
        self.assertEqual(record.descriptions[15].score, 38)
        self.assertAlmostEqual(record.descriptions[15].e, 0.012)
        self.assertEqual(record.descriptions[16].title, "gi|1345782|sp|P48110|CHLL_CYAPA PROTOCHLOROPHYLLIDE REDUCTASE I...")
        self.assertEqual(record.descriptions[16].score, 38)
        self.assertAlmostEqual(record.descriptions[16].e, 0.012)
        self.assertEqual(record.descriptions[17].title, "gi|400260|sp|Q01464|MIND_BACSU SEPTUM SITE-DETERMINING PROTEIN ...")
        self.assertEqual(record.descriptions[17].score, 38)
        self.assertAlmostEqual(record.descriptions[17].e, 0.012)
        self.assertEqual(record.descriptions[18].title, "gi|3913244|sp|O47041|CHLL_PICAB PROTOCHLOROPHYLLIDE REDUCTASE I...")
        self.assertEqual(record.descriptions[18].score, 38)
        self.assertAlmostEqual(record.descriptions[18].e, 0.016)
        self.assertEqual(record.descriptions[19].title, "gi|1168937|sp|P41645|CHLL_PINTH PROTOCHLOROPHYLLIDE REDUCTASE I...")
        self.assertEqual(record.descriptions[19].score, 38)
        self.assertAlmostEqual(record.descriptions[19].e, 0.016)
        self.assertEqual(record.descriptions[20].title, "gi|120543|sp|P26181|CHLL_PINCO PROTOCHLOROPHYLLIDE REDUCTASE IR...")
        self.assertEqual(record.descriptions[20].score, 38)
        self.assertAlmostEqual(record.descriptions[20].e, 0.016)
        self.assertEqual(record.descriptions[21].title, "gi|401555|sp|P31856|YGI1_PSEPU HYPOTHETICAL 28.9 KD PROTEIN IN ...")
        self.assertEqual(record.descriptions[21].score, 38)
        self.assertAlmostEqual(record.descriptions[21].e, 0.016)
        self.assertEqual(record.descriptions[22].title, "gi|120544|sp|Q00237|CHLL_PLEBO PROTOCHLOROPHYLLIDE REDUCTASE IR...")
        self.assertEqual(record.descriptions[22].score, 38)
        self.assertAlmostEqual(record.descriptions[22].e, 0.016)
        self.assertEqual(record.descriptions[23].title, "gi|6225723|sp|O33225|MRP_MYCTU MRP PROTEIN HOMOLOG")
        self.assertEqual(record.descriptions[23].score, 38)
        self.assertAlmostEqual(record.descriptions[23].e, 0.021)
        self.assertEqual(record.descriptions[24].title, "gi|2496603|sp|P55393|Y4CK_RHISN PUTATIVE REPLICATION PROTEIN A")
        self.assertEqual(record.descriptions[24].score, 38)
        self.assertAlmostEqual(record.descriptions[24].e, 0.021)
        self.assertEqual(record.descriptions[25].title, "gi|120542|sp|P06267|CHLL_MARPO PROTOCHLOROPHYLLIDE REDUCTASE IR...")
        self.assertEqual(record.descriptions[25].score, 38)
        self.assertAlmostEqual(record.descriptions[25].e, 0.021)
        self.assertEqual(record.descriptions[26].title, "gi|1705820|sp|P54207|CHLL_SYNP7 PROTOCHLOROPHYLLIDE REDUCTASE I...")
        self.assertEqual(record.descriptions[26].score, 38)
        self.assertAlmostEqual(record.descriptions[26].e, 0.021)
        self.assertEqual(record.descriptions[27].title, "gi|124472|sp|P07673|INC2_ECOLI INCC PROTEIN")
        self.assertEqual(record.descriptions[27].score, 38)
        self.assertAlmostEqual(record.descriptions[27].e, 0.021)
        self.assertEqual(record.descriptions[28].title, "gi|3023485|sp|P56291|CHLL_CHLVU PROTOCHLOROPHYLLIDE REDUCTASE I...")
        self.assertEqual(record.descriptions[28].score, 37)
        self.assertAlmostEqual(record.descriptions[28].e, 0.027)
        self.assertEqual(record.descriptions[29].title, "gi|6016572|sp|O78436|MIND_GUITH PUTATIVE SEPTUM SITE-DETERMININ...")
        self.assertEqual(record.descriptions[29].score, 36)
        self.assertAlmostEqual(record.descriptions[29].e, 0.047)
        self.assertEqual(record.descriptions[30].title, "gi|6225722|sp|O66946|MRP_AQUAE MRP PROTEIN HOMOLOG")
        self.assertEqual(record.descriptions[30].score, 36)
        self.assertAlmostEqual(record.descriptions[30].e, 0.047)
        self.assertEqual(record.descriptions[31].title, "gi|1723296|sp|P50863|MRP_BACSU MRP PROTEIN HOMOLOG")
        self.assertEqual(record.descriptions[31].score, 36)
        self.assertAlmostEqual(record.descriptions[31].e, 0.047)
        self.assertEqual(record.descriptions[32].title, "gi|128211|sp|P08625|NIF2_METTL NITROGENASE IRON PROTEIN 2 (NITR...")
        self.assertEqual(record.descriptions[32].score, 36)
        self.assertAlmostEqual(record.descriptions[32].e, 0.047)
        self.assertEqual(record.descriptions[33].title, "gi|120541|sp|Q00469|CHLL_CHLRE PROTOCHLOROPHYLLIDE REDUCTASE IR...")
        self.assertEqual(record.descriptions[33].score, 36)
        self.assertAlmostEqual(record.descriptions[33].e, 0.047)
        self.assertEqual(record.descriptions[34].title, "gi|585623|sp|Q07733|OPPD_LACLA OLIGOPEPTIDE TRANSPORT ATP-BINDI...")
        self.assertEqual(record.descriptions[34].score, 36)
        self.assertAlmostEqual(record.descriptions[34].e, 0.047)
        self.assertEqual(record.descriptions[35].title, "gi|1705819|sp|P51187|CHLL_PORPU PROTOCHLOROPHYLLIDE REDUCTASE I...")
        self.assertEqual(record.descriptions[35].score, 36)
        self.assertAlmostEqual(record.descriptions[35].e, 0.061)
        self.assertEqual(record.descriptions[36].title, "gi|6226409|sp|O58667|Y949_PYRHO HYPOTHETICAL PROTEIN PH0949")
        self.assertEqual(record.descriptions[36].score, 36)
        self.assertAlmostEqual(record.descriptions[36].e, 0.081)
        self.assertEqual(record.descriptions[37].title, "gi|6226573|sp|P31897|COOC_RHORU CARBON MONOXIDE DEHYDROGENASE A...")
        self.assertEqual(record.descriptions[37].score, 36)
        self.assertAlmostEqual(record.descriptions[37].e, 0.081)
        self.assertEqual(record.descriptions[38].title, "gi|544021|sp|P36439|CHLL_POLAC PROTOCHLOROPHYLLIDE REDUCTASE IR...")
        self.assertEqual(record.descriptions[38].score, 36)
        self.assertAlmostEqual(record.descriptions[38].e, 0.081)
        self.assertEqual(record.descriptions[39].title, "gi|2499207|sp|Q58289|NIFH_METJA NITROGENASE IRON PROTEIN (NITRO...")
        self.assertEqual(record.descriptions[39].score, 35)
        self.assertAlmostEqual(record.descriptions[39].e, 0.11)
        self.assertEqual(record.descriptions[40].title, "gi|2495840|sp|Q57633|Y169_METJA HYPOTHETICAL ATP-BINDING PROTEI...")
        self.assertEqual(record.descriptions[40].score, 35)
        self.assertAlmostEqual(record.descriptions[40].e, 0.11)
        self.assertEqual(record.descriptions[41].title, "gi|2496129|sp|Q58233|Y823_METJA HYPOTHETICAL ATP-BINDING PROTEI...")
        self.assertEqual(record.descriptions[41].score, 35)
        self.assertAlmostEqual(record.descriptions[41].e, 0.14)
        self.assertEqual(record.descriptions[42].title, "gi|1171022|sp|P45135|MRP_HAEIN MRP PROTEIN HOMOLOG")
        self.assertEqual(record.descriptions[42].score, 34)
        self.assertAlmostEqual(record.descriptions[42].e, 0.18)
        self.assertEqual(record.descriptions[43].title, "gi|132518|sp|P22670|RFX1_HUMAN MHC CLASS II REGULATORY FACTOR R...")
        self.assertEqual(record.descriptions[43].score, 34)
        self.assertAlmostEqual(record.descriptions[43].e, 0.18)
        self.assertEqual(record.descriptions[44].title, "gi|113780|sp|P08144|AMYA_DROME ALPHA-AMYLASE A PRECURSOR (1,4-A...")
        self.assertEqual(record.descriptions[44].score, 34)
        self.assertAlmostEqual(record.descriptions[44].e, 0.18)
        self.assertEqual(record.descriptions[45].title, "gi|1709471|sp|P50980|OPPD_LACLC OLIGOPEPTIDE TRANSPORT ATP-BIND...")
        self.assertEqual(record.descriptions[45].score, 34)
        self.assertAlmostEqual(record.descriptions[45].e, 0.18)
        self.assertEqual(record.descriptions[46].title, "gi|6225724|sp|Q9ZE27|MRP_RICPR MRP PROTEIN HOMOLOG")
        self.assertEqual(record.descriptions[46].score, 34)
        self.assertAlmostEqual(record.descriptions[46].e, 0.24)
        self.assertEqual(record.descriptions[47].title, "gi|2496034|sp|Q57967|Y547_METJA HYPOTHETICAL ATP-BINDING PROTEI...")
        self.assertEqual(record.descriptions[47].score, 34)
        self.assertAlmostEqual(record.descriptions[47].e, 0.24)
        self.assertEqual(record.descriptions[48].title, "gi|128215|sp|P22548|NIF4_CLOPA NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[48].score, 34)
        self.assertAlmostEqual(record.descriptions[48].e, 0.24)
        self.assertEqual(record.descriptions[49].title, "gi|1706104|sp|Q04663|CPSC_STRAG CPSC PROTEIN")
        self.assertEqual(record.descriptions[49].score, 34)
        self.assertAlmostEqual(record.descriptions[49].e, 0.31)
        self.assertEqual(record.descriptions[50].title, "gi|548362|sp|P26248|NIF1_AZOCH NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[50].score, 34)
        self.assertAlmostEqual(record.descriptions[50].e, 0.31)
        self.assertEqual(record.descriptions[51].title, "gi|128203|sp|P00459|NIF1_AZOVI NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[51].score, 34)
        self.assertAlmostEqual(record.descriptions[51].e, 0.31)
        self.assertEqual(record.descriptions[52].title, "gi|2497979|sp|Q57731|Y283_METJA HYPOTHETICAL PROTEIN MJ0283")
        self.assertEqual(record.descriptions[52].score, 33)
        self.assertAlmostEqual(record.descriptions[52].e, 0.41)
        self.assertEqual(record.descriptions[53].title, "gi|2499205|sp|Q44044|NIFH_ALCFA NITROGENASE IRON PROTEIN (NITRO...")
        self.assertEqual(record.descriptions[53].score, 33)
        self.assertAlmostEqual(record.descriptions[53].e, 0.41)
        self.assertEqual(record.descriptions[54].title, "gi|1703363|sp|P52145|ARA2_ECOLI ARSENICAL PUMP-DRIVING ATPASE")
        self.assertEqual(record.descriptions[54].score, 33)
        self.assertAlmostEqual(record.descriptions[54].e, 0.41)
        self.assertEqual(record.descriptions[55].title, "gi|132821|sp|P27683|RK24_SPIOL 50S RIBOSOMAL PROTEIN L24, CHLOR...")
        self.assertEqual(record.descriptions[55].score, 33)
        self.assertAlmostEqual(record.descriptions[55].e, 0.41)
        self.assertEqual(record.descriptions[56].title, "gi|399100|sp|Q02431|BCHX_RHOSH CHLOROPHYLLIDE REDUCTASE 35.5 KD...")
        self.assertEqual(record.descriptions[56].score, 33)
        self.assertAlmostEqual(record.descriptions[56].e, 0.41)
        self.assertEqual(record.descriptions[57].title, "gi|128207|sp|P06118|NIF2_AZOCH NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[57].score, 33)
        self.assertAlmostEqual(record.descriptions[57].e, 0.41)
        self.assertEqual(record.descriptions[58].title, "gi|128267|sp|P00458|NIFH_KLEPN NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[58].score, 33)
        self.assertAlmostEqual(record.descriptions[58].e, 0.41)
        self.assertEqual(record.descriptions[59].title, "gi|128208|sp|P15335|NIF2_AZOVI NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[59].score, 33)
        self.assertAlmostEqual(record.descriptions[59].e, 0.41)
        self.assertEqual(record.descriptions[60].title, "gi|1703299|sp|P54215|AMYA_DROMA ALPHA-AMYLASE A PRECURSOR (1,4-...")
        self.assertEqual(record.descriptions[60].score, 33)
        self.assertAlmostEqual(record.descriptions[60].e, 0.53)
        self.assertEqual(record.descriptions[61].title, "gi|114867|sp|P26177|BCHX_RHOCA CHLOROPHYLLIDE REDUCTASE 35.5 KD...")
        self.assertEqual(record.descriptions[61].score, 33)
        self.assertAlmostEqual(record.descriptions[61].e, 0.53)
        self.assertEqual(record.descriptions[62].title, "gi|128217|sp|P09555|NIF6_CLOPA NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[62].score, 33)
        self.assertAlmostEqual(record.descriptions[62].e, 0.53)
        self.assertEqual(record.descriptions[63].title, "gi|128277|sp|P06661|NIFH_THIFE NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[63].score, 33)
        self.assertAlmostEqual(record.descriptions[63].e, 0.53)
        self.assertEqual(record.descriptions[64].title, "gi|1709270|sp|P54800|NIF2_METBA NITROGENASE IRON PROTEIN 2 (NIT...")
        self.assertEqual(record.descriptions[64].score, 32)
        self.assertAlmostEqual(record.descriptions[64].e, 0.70)
        self.assertEqual(record.descriptions[65].title, "gi|6014740|sp|O33946|CTB1_ACILW MUCONATE CYCLOISOMERASE I 1 (CI...")
        self.assertEqual(record.descriptions[65].score, 32)
        self.assertAlmostEqual(record.descriptions[65].e, 0.92)
        self.assertEqual(record.descriptions[66].title, "gi|134731|sp|P08866|SOPA_ECOLI SOPA PROTEIN (PROTEIN A)")
        self.assertEqual(record.descriptions[66].score, 32)
        self.assertAlmostEqual(record.descriptions[66].e, 0.92)
        self.assertEqual(record.descriptions[67].title, "gi|114220|sp|P08690|ARA1_ECOLI ARSENICAL PUMP-DRIVING ATPASE")
        self.assertEqual(record.descriptions[67].score, 32)
        self.assertAlmostEqual(record.descriptions[67].e, 0.92)
        self.assertEqual(record.descriptions[68].title, "gi|3334342|sp|O29633|SR54_ARCFU PROBABLE SIGNAL RECOGNITION 54 ...")
        self.assertEqual(record.descriptions[68].score, 32)
        self.assertAlmostEqual(record.descriptions[68].e, 1.2)
        self.assertEqual(record.descriptions[69].title, "gi|2499206|sp|Q59270|NIFH_CLOCB NITROGENASE IRON PROTEIN (NITRO...")
        self.assertEqual(record.descriptions[69].score, 32)
        self.assertAlmostEqual(record.descriptions[69].e, 1.2)
        self.assertEqual(record.descriptions[70].title, "gi|2842582|sp|Q58334|Y924_METJA HYPOTHETICAL ATP-BINDING PROTEI...")
        self.assertEqual(record.descriptions[70].score, 31)
        self.assertAlmostEqual(record.descriptions[70].e, 1.6)
        self.assertEqual(record.descriptions[71].title, "gi|128264|sp|P00463|NIFH_BRASP NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[71].score, 31)
        self.assertAlmostEqual(record.descriptions[71].e, 1.6)
        self.assertEqual(record.descriptions[72].title, "gi|128268|sp|P06119|NIFH_METVO NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[72].score, 31)
        self.assertAlmostEqual(record.descriptions[72].e, 1.6)
        self.assertEqual(record.descriptions[73].title, "gi|128263|sp|P06117|NIFH_BRAJA NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[73].score, 31)
        self.assertAlmostEqual(record.descriptions[73].e, 1.6)
        self.assertEqual(record.descriptions[74].title, "gi|2833531|sp|Q58098|Y685_METJA HYPOTHETICAL ATP-BINDING PROTEI...")
        self.assertEqual(record.descriptions[74].score, 31)
        self.assertAlmostEqual(record.descriptions[74].e, 2.1)
        self.assertEqual(record.descriptions[75].title, "gi|1709997|sp|P53692|RA18_SCHPO DNA REPAIR PROTEIN RAD18")
        self.assertEqual(record.descriptions[75].score, 31)
        self.assertAlmostEqual(record.descriptions[75].e, 2.1)
        self.assertEqual(record.descriptions[76].title, "gi|127221|sp|P22900|MOBD_THIFE MOBD PROTEIN")
        self.assertEqual(record.descriptions[76].score, 31)
        self.assertAlmostEqual(record.descriptions[76].e, 2.1)
        self.assertEqual(record.descriptions[77].title, "gi|114863|sp|P26237|BCHL_RHOCA PROTOCHLOROPHILLIDE REDUCTASE 33...")
        self.assertEqual(record.descriptions[77].score, 31)
        self.assertAlmostEqual(record.descriptions[77].e, 2.7)
        self.assertEqual(record.descriptions[78].title, "gi|128216|sp|P09554|NIF5_CLOPA NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[78].score, 31)
        self.assertAlmostEqual(record.descriptions[78].e, 2.7)
        self.assertEqual(record.descriptions[79].title, "gi|128204|sp|P00456|NIF1_CLOPA NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[79].score, 31)
        self.assertAlmostEqual(record.descriptions[79].e, 2.7)
        self.assertEqual(record.descriptions[80].title, "gi|128209|sp|P09552|NIF2_CLOPA NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[80].score, 31)
        self.assertAlmostEqual(record.descriptions[80].e, 2.7)
        self.assertEqual(record.descriptions[81].title, "gi|128261|sp|P00457|NIFH_ANASP NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[81].score, 31)
        self.assertAlmostEqual(record.descriptions[81].e, 2.7)
        self.assertEqual(record.descriptions[82].title, "gi|266624|sp|Q00240|NIFH_PLEBO NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[82].score, 31)
        self.assertAlmostEqual(record.descriptions[82].e, 2.7)
        self.assertEqual(record.descriptions[83].title, "gi|6225393|sp|O67066|FTSY_AQUAE CELL DIVISION PROTEIN FTSY HOMOLOG")
        self.assertEqual(record.descriptions[83].score, 30)
        self.assertAlmostEqual(record.descriptions[83].e, 3.5)
        self.assertEqual(record.descriptions[84].title, "gi|6166125|sp|O14936|CSKP_HUMAN PERIPHERAL PLASMA MEMBRANE PROT...")
        self.assertEqual(record.descriptions[84].score, 30)
        self.assertAlmostEqual(record.descriptions[84].e, 3.5)
        self.assertEqual(record.descriptions[85].title, "gi|1709099|sp|P53381|MRP_CLOPE MRP PROTEIN HOMOLOG")
        self.assertEqual(record.descriptions[85].score, 30)
        self.assertAlmostEqual(record.descriptions[85].e, 3.5)
        self.assertEqual(record.descriptions[86].title, "gi|1171710|sp|P46034|NIFH_FRASP NITROGENASE IRON PROTEIN (NITRO...")
        self.assertEqual(record.descriptions[86].score, 30)
        self.assertAlmostEqual(record.descriptions[86].e, 3.5)
        self.assertEqual(record.descriptions[87].title, "gi|128212|sp|P26252|NIF2_RHISO NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[87].score, 30)
        self.assertAlmostEqual(record.descriptions[87].e, 3.5)
        self.assertEqual(record.descriptions[88].title, "gi|128206|sp|P26251|NIF1_RHISO NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[88].score, 30)
        self.assertAlmostEqual(record.descriptions[88].e, 3.5)
        self.assertEqual(record.descriptions[89].title, "gi|2495787|sp|Q60392|Y084_METJA HYPOTHETICAL ATP-BINDING PROTEI...")
        self.assertEqual(record.descriptions[89].score, 30)
        self.assertAlmostEqual(record.descriptions[89].e, 4.6)
        self.assertEqual(record.descriptions[90].title, "gi|1709283|sp|P52336|NIFH_NOSSN NITROGENASE IRON PROTEIN (NITRO...")
        self.assertEqual(record.descriptions[90].score, 30)
        self.assertAlmostEqual(record.descriptions[90].e, 4.6)
        self.assertEqual(record.descriptions[91].title, "gi|731773|sp|P40558|YIA3_YEAST HYPOTHETICAL 31.9 KD PROTEIN IN ...")
        self.assertEqual(record.descriptions[91].score, 30)
        self.assertAlmostEqual(record.descriptions[91].e, 4.6)
        self.assertEqual(record.descriptions[92].title, "gi|128266|sp|P08925|NIFH_FRAAL NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[92].score, 30)
        self.assertAlmostEqual(record.descriptions[92].e, 4.6)
        self.assertEqual(record.descriptions[93].title, "gi|732146|sp|P40742|YLXH_BACSU HYPOTHETICAL 33.2 KD PROTEIN IN ...")
        self.assertEqual(record.descriptions[93].score, 30)
        self.assertAlmostEqual(record.descriptions[93].e, 4.6)
        self.assertEqual(record.descriptions[94].title, "gi|417362|sp|P33178|NIFH_ANASL NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[94].score, 30)
        self.assertAlmostEqual(record.descriptions[94].e, 4.6)
        self.assertEqual(record.descriptions[95].title, "gi|1709281|sp|P26250|NIFH_NOSCO NITROGENASE IRON PROTEIN (NITRO...")
        self.assertEqual(record.descriptions[95].score, 30)
        self.assertAlmostEqual(record.descriptions[95].e, 4.6)
        self.assertEqual(record.descriptions[96].title, "gi|6225290|sp|Q9Z900|EFP_CHLPN ELONGATION FACTOR P (EF-P)")
        self.assertEqual(record.descriptions[96].score, 29)
        self.assertAlmostEqual(record.descriptions[96].e, 6.1)
        self.assertEqual(record.descriptions[97].title, "gi|3183458|sp|P75796|YLIA_ECOLI HYPOTHETICAL ABC TRANSPORTER AT...")
        self.assertEqual(record.descriptions[97].score, 29)
        self.assertAlmostEqual(record.descriptions[97].e, 6.1)
        self.assertEqual(record.descriptions[98].title, "gi|2499682|sp|Q52312|INC1_ECOLI INCC PROTEIN")
        self.assertEqual(record.descriptions[98].score, 29)
        self.assertAlmostEqual(record.descriptions[98].e, 6.1)
        self.assertEqual(record.descriptions[99].title, "gi|1345941|sp|P15555|DAC_STRSQ D-ALANYL-D-ALANINE CARBOXYPEPTID...")
        self.assertEqual(record.descriptions[99].score, 29)
        self.assertAlmostEqual(record.descriptions[99].e, 6.1)
        self.assertEqual(record.descriptions[100].title, "gi|128275|sp|P08718|NIF1_RHOCA NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[100].score, 29)
        self.assertAlmostEqual(record.descriptions[100].e, 6.1)
        self.assertEqual(record.descriptions[101].title, "gi|267440|sp|P29811|YAC3_PSEAE HYPOTHETICAL 23.9 KD PROTEIN IN ...")
        self.assertEqual(record.descriptions[101].score, 29)
        self.assertAlmostEqual(record.descriptions[101].e, 6.1)
        self.assertEqual(record.descriptions[102].title, "gi|732050|sp|P39342|YJGR_ECOLI HYPOTHETICAL 54.3 KD PROTEIN IN ...")
        self.assertEqual(record.descriptions[102].score, 29)
        self.assertAlmostEqual(record.descriptions[102].e, 6.1)
        self.assertEqual(record.descriptions[103].title, "gi|2833520|sp|Q58000|Y580_METJA HYPOTHETICAL PROTEIN MJ0580")
        self.assertEqual(record.descriptions[103].score, 29)
        self.assertAlmostEqual(record.descriptions[103].e, 7.9)
        self.assertEqual(record.descriptions[104].title, "gi|128272|sp|P00460|NIFH_RHIME NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[104].score, 29)
        self.assertAlmostEqual(record.descriptions[104].e, 7.9)
        self.assertEqual(record.descriptions[105].title, "gi|128271|sp|P00461|NIFH_RHILT NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[105].score, 29)
        self.assertAlmostEqual(record.descriptions[105].e, 7.9)
        self.assertEqual(record.descriptions[106].title, "gi|128276|sp|P22921|NIFH_RHORU NITROGENASE IRON PROTEIN (NITROG...")
        self.assertEqual(record.descriptions[106].score, 29)
        self.assertAlmostEqual(record.descriptions[106].e, 7.9)
        self.assertEqual(len(record.alignments), 0)
        self.assertEqual(record.database_name, ['/taiyang/jchang/parsers/swissprot'])
        self.assertEqual(record.posted_date, [('May 12, 2000 11:25 AM',)])
        self.assertEqual(record.num_letters_in_database, [29652561])
        self.assertEqual(record.num_sequences_in_database, [82258])
        self.assertEqual(len(record.ka_params), 3)
        self.assertAlmostEqual(record.ka_params[0], 0.315)
        self.assertAlmostEqual(record.ka_params[1], 0.130)
        self.assertAlmostEqual(record.ka_params[2], 0.349)
        self.assertEqual(len(record.ka_params_gap), 3)
        self.assertAlmostEqual(record.ka_params_gap[0], 0.270)
        self.assertAlmostEqual(record.ka_params_gap[1], 0.0470)
        self.assertAlmostEqual(record.ka_params_gap[2], 0.230)
        self.assertEqual(record.matrix, 'BLOSUM62')
        self.assertEqual(record.gap_penalties, [11, 1])
        self.assertEqual(record.num_hits, 11011949)
        self.assertEqual(record.num_sequences, 82258)
        self.assertEqual(record.num_extends, 406641)
        self.assertEqual(record.num_good_extends, 1349)
        self.assertEqual(record.num_seqs_better_e, 107)
        self.assertEqual(record.hsps_no_gap, 89)
        self.assertEqual(record.hsps_prelim_gapped, 18)
        self.assertEqual(record.hsps_gapped, 131)
        self.assertEqual(record.query_length, 222)
        self.assertEqual(record.database_length, 29652561)
        self.assertEqual(record.effective_hsp_length, 57)
        self.assertEqual(record.effective_query_length, 165)
        self.assertEqual(record.effective_database_length, 24963855)
        self.assertEqual(record.effective_search_space, 4119036075)
        self.assertEqual(record.effective_search_space_used, 4119036075)
        self.assertEqual(record.threshold, 11)
        self.assertEqual(record.window_size, 40)
        self.assertEqual(len(record.dropoff_1st_pass), 2)
        self.assertEqual(record.dropoff_1st_pass[0], 16)
        self.assertAlmostEqual(record.dropoff_1st_pass[1], 7.3)
        self.assertEqual(len(record.gap_x_dropoff), 2)
        self.assertEqual(record.gap_x_dropoff[0], 38)
        self.assertAlmostEqual(record.gap_x_dropoff[1], 14.8)
        self.assertEqual(len(record.gap_x_dropoff_final), 2)
        self.assertEqual(record.gap_x_dropoff_final[0], 64)
        self.assertAlmostEqual(record.gap_x_dropoff_final[1], 24.9)
        self.assertEqual(len(record.gap_trigger), 2)
        self.assertEqual(record.gap_trigger[0], 41)
        self.assertAlmostEqual(record.gap_trigger[1], 21.6)
        self.assertEqual(len(record.blast_cutoff), 2)
        self.assertEqual(record.blast_cutoff[0], 63)
        self.assertAlmostEqual(record.blast_cutoff[1], 29.0)

    def test_text_2012L_psiblast_001(self):
        "Test parsing BLASTP 2.0.12 output (text_2012L_psiblast_001)"

        path = os.path.join('Blast', 'text_2012L_psiblast_001.txt')
        handle = open(path)
        record = self.pb_parser.parse(handle)
        handle.close()
        self.assertEqual(record.application, "BLASTP")
        self.assertEqual(record.version, '2.0.12')
        self.assertEqual(record.date, "Apr-21-2000")
        self.assertEqual(record.reference, TestNCBITextParser.reference)
        self.assertEqual(record.query, "aaaa")
        self.assertEqual(record.query_letters, 889)
        self.assertEqual(record.database, "/dbase/swissprot/main/release/sp; /dbase/swissprot/main/update/spu")
        self.assertEqual(record.database_sequences, 88201)
        self.assertEqual(record.database_letters, 31957340)
        self.assertEqual(len(record.rounds), 5)
        # Rest of test broken up to avoid Jython JVM limitations
        self._check_text_2012L_psiblast_001_round0(record)
        self._check_text_2012L_psiblast_001_round1(record)
        self._check_text_2012L_psiblast_001_round2(record)
        self._check_text_2012L_psiblast_001_round4(record)
        self._check_text_2012L_psiblast_001_hsps(record)
        self._check_text_2012L_psiblast_001_hsps_counts(record)
        self._check_text_2012L_psiblast_001_hsps_details(record)
        self._check_text_2012L_psiblast_001_footer(record)

    def _check_text_2012L_psiblast_001_round0(self, record):
        self.assertEqual(len(record.rounds[0].new_seqs), 27)
        self.assertEqual(record.rounds[0].new_seqs[0].title, "100K_RAT Q62671 rattus norvegicus (rat). 100 kda protein (ec...")
        self.assertEqual(record.rounds[0].new_seqs[0].score, 1516)
        self.assertAlmostEqual(record.rounds[0].new_seqs[0].e, 0.0)
        self.assertEqual(record.rounds[0].new_seqs[1].title, "HYDP_DROME P51592 drosophila melanogaster (fruit fly). hyper...")
        self.assertEqual(record.rounds[0].new_seqs[1].score, 513)
        self.assertAlmostEqual(record.rounds[0].new_seqs[1].e, 1e-145)
        self.assertEqual(record.rounds[0].new_seqs[2].title, "PUB1_SCHPO Q92462 schizosaccharomyces pombe (fission yeast)....")
        self.assertEqual(record.rounds[0].new_seqs[2].score, 95)
        self.assertAlmostEqual(record.rounds[0].new_seqs[2].e, 7e-19)
        self.assertEqual(record.rounds[0].new_seqs[3].title, "RSP5_YEAST P39940 saccharomyces cerevisiae (baker's yeast). ...")
        self.assertEqual(record.rounds[0].new_seqs[3].score, 93)
        self.assertAlmostEqual(record.rounds[0].new_seqs[3].e, 3e-18)
        self.assertEqual(record.rounds[0].new_seqs[4].title, "NED4_HUMAN P46934 homo sapiens (human). nedd-4 protein (ec 6...")
        self.assertEqual(record.rounds[0].new_seqs[4].score, 88)
        self.assertAlmostEqual(record.rounds[0].new_seqs[4].e, 9e-17)
        self.assertEqual(record.rounds[0].new_seqs[5].title, "NED4_MOUSE P46935 mus musculus (mouse). nedd-4 protein (ec 6...")
        self.assertEqual(record.rounds[0].new_seqs[5].score, 87)
        self.assertAlmostEqual(record.rounds[0].new_seqs[5].e, 9e-17)
        self.assertEqual(record.rounds[0].new_seqs[6].title, "URB1_RAT P51593 rattus norvegicus (rat). dna binding protein...")
        self.assertEqual(record.rounds[0].new_seqs[6].score, 85)
        self.assertAlmostEqual(record.rounds[0].new_seqs[6].e, 4e-16)
        self.assertEqual(record.rounds[0].new_seqs[7].title, "UE3A_HUMAN Q05086 homo sapiens (human). ubiquitin-protein li...")
        self.assertEqual(record.rounds[0].new_seqs[7].score, 84)
        self.assertAlmostEqual(record.rounds[0].new_seqs[7].e, 1e-15)
        self.assertEqual(record.rounds[0].new_seqs[8].title, "HUL5_YEAST P53119 saccharomyces cerevisiae (baker's yeast). ...")
        self.assertEqual(record.rounds[0].new_seqs[8].score, 83)
        self.assertAlmostEqual(record.rounds[0].new_seqs[8].e, 2e-15)
        self.assertEqual(record.rounds[0].new_seqs[9].title, "UE3A_MOUSE O08759 mus musculus (mouse). ubiquitin-protein li...")
        self.assertEqual(record.rounds[0].new_seqs[9].score, 82)
        self.assertAlmostEqual(record.rounds[0].new_seqs[9].e, 6e-15)
        self.assertEqual(record.rounds[0].new_seqs[10].title, "HUL4_YEAST P40985 saccharomyces cerevisiae (baker's yeast). ...")
        self.assertEqual(record.rounds[0].new_seqs[10].score, 69)
        self.assertAlmostEqual(record.rounds[0].new_seqs[10].e, 3e-11)
        self.assertEqual(record.rounds[0].new_seqs[11].title, "UFD4_YEAST P33202 saccharomyces cerevisiae (baker's yeast). ...")
        self.assertEqual(record.rounds[0].new_seqs[11].score, 58)
        self.assertAlmostEqual(record.rounds[0].new_seqs[11].e, 7e-8)
        self.assertEqual(record.rounds[0].new_seqs[12].title, "Y032_HUMAN Q15034 homo sapiens (human). hypothetical protein...")
        self.assertEqual(record.rounds[0].new_seqs[12].score, 56)
        self.assertAlmostEqual(record.rounds[0].new_seqs[12].e, 3e-7)
        self.assertEqual(record.rounds[0].new_seqs[13].title, "TR12_HUMAN Q14669 homo sapiens (human). thyroid receptor int...")
        self.assertEqual(record.rounds[0].new_seqs[13].score, 50)
        self.assertAlmostEqual(record.rounds[0].new_seqs[13].e, 0.00002)
        self.assertEqual(record.rounds[0].new_seqs[14].title, "PAB1_MOUSE P29341 mus musculus (mouse). polyadenylate-bindin...")
        self.assertEqual(record.rounds[0].new_seqs[14].score, 49)
        self.assertAlmostEqual(record.rounds[0].new_seqs[14].e, 0.00003)
        self.assertEqual(record.rounds[0].new_seqs[15].title, "PAB1_HUMAN P11940 homo sapiens (human). polyadenylate-bindin...")
        self.assertEqual(record.rounds[0].new_seqs[15].score, 49)
        self.assertAlmostEqual(record.rounds[0].new_seqs[15].e, 0.00005)
        self.assertEqual(record.rounds[0].new_seqs[16].title, "PABP_XENLA P20965 xenopus laevis (african clawed frog). poly...")
        self.assertEqual(record.rounds[0].new_seqs[16].score, 48)
        self.assertAlmostEqual(record.rounds[0].new_seqs[16].e, 0.00008)
        self.assertEqual(record.rounds[0].new_seqs[17].title, "PABP_DROME P21187 drosophila melanogaster (fruit fly). polya...")
        self.assertEqual(record.rounds[0].new_seqs[17].score, 41)
        self.assertAlmostEqual(record.rounds[0].new_seqs[17].e, 0.008)
        self.assertEqual(record.rounds[0].new_seqs[18].title, "PAB5_ARATH Q05196 arabidopsis thaliana (mouse-ear cress). po...")
        self.assertEqual(record.rounds[0].new_seqs[18].score, 37)
        self.assertAlmostEqual(record.rounds[0].new_seqs[18].e, 0.15)
        self.assertEqual(record.rounds[0].new_seqs[19].title, "PAB2_ARATH P42731 arabidopsis thaliana (mouse-ear cress). po...")
        self.assertEqual(record.rounds[0].new_seqs[19].score, 34)
        self.assertAlmostEqual(record.rounds[0].new_seqs[19].e, 0.99)
        self.assertEqual(record.rounds[0].new_seqs[20].title, "MCM3_MOUSE P25206 mus musculus (mouse). dna replication lice...")
        self.assertEqual(record.rounds[0].new_seqs[20].score, 34)
        self.assertAlmostEqual(record.rounds[0].new_seqs[20].e, 1.7)
        self.assertEqual(record.rounds[0].new_seqs[21].title, "PABP_SCHPO P31209 schizosaccharomyces pombe (fission yeast)....")
        self.assertEqual(record.rounds[0].new_seqs[21].score, 32)
        self.assertAlmostEqual(record.rounds[0].new_seqs[21].e, 3.8)
        self.assertEqual(record.rounds[0].new_seqs[22].title, "PROB_SERMA P17856 serratia marcescens. glutamate 5-kinase (e...")
        self.assertEqual(record.rounds[0].new_seqs[22].score, 32)
        self.assertAlmostEqual(record.rounds[0].new_seqs[22].e, 6.6)
        self.assertEqual(record.rounds[0].new_seqs[23].title, "ST20_CANAL Q92212 candida albicans (yeast). serine/threonine...")
        self.assertEqual(record.rounds[0].new_seqs[23].score, 31)
        self.assertAlmostEqual(record.rounds[0].new_seqs[23].e, 8.6)
        self.assertEqual(record.rounds[0].new_seqs[24].title, "KEND_HUMAN O95613 homo sapiens (human). kendrin (kiaa0402). ...")
        self.assertEqual(record.rounds[0].new_seqs[24].score, 31)
        self.assertAlmostEqual(record.rounds[0].new_seqs[24].e, 8.6)
        self.assertEqual(record.rounds[0].new_seqs[25].title, "FIXK_RHIME P13295 rhizobium meliloti (sinorhizobium meliloti...")
        self.assertEqual(record.rounds[0].new_seqs[25].score, 31)
        self.assertAlmostEqual(record.rounds[0].new_seqs[25].e, 8.6)
        self.assertEqual(record.rounds[0].new_seqs[26].title, "CC24_YEAST P11433 saccharomyces cerevisiae (baker's yeast). ...")
        self.assertEqual(record.rounds[0].new_seqs[26].score, 31)
        self.assertAlmostEqual(record.rounds[0].new_seqs[26].e, 8.6)
        self.assertEqual(len(record.rounds[0].alignments), 27)
        self.assertEqual(record.rounds[0].alignments[0].title, ">100K_RAT Q62671 rattus norvegicus (rat). 100 kda protein (ec 6.3.2.-). 7/1999")
        self.assertEqual(record.rounds[0].alignments[0].length, 889)
        self.assertEqual(record.rounds[0].alignments[1].title, ">HYDP_DROME P51592 drosophila melanogaster (fruit fly). hyperplastic discs protein (hyd protein) (ec 6.3.2.-). 12/1998")
        self.assertEqual(record.rounds[0].alignments[1].length, 2895)
        self.assertEqual(record.rounds[0].alignments[2].title, ">PUB1_SCHPO Q92462 schizosaccharomyces pombe (fission yeast). ubiquitin--protein ligase pub1 (ec 6.3.2.-). 12/1998")
        self.assertEqual(record.rounds[0].alignments[2].length, 767)
        self.assertEqual(record.rounds[0].alignments[3].title, ">RSP5_YEAST P39940 saccharomyces cerevisiae (baker's yeast). ubiquitin--protein ligase rsp5 (ec 6.3.2.-). 7/1999")
        self.assertEqual(record.rounds[0].alignments[3].length, 809)
        self.assertEqual(record.rounds[0].alignments[4].title, ">NED4_HUMAN P46934 homo sapiens (human). nedd-4 protein (ec 6.3.2.-) (kiaa0093) (fragment). 7/1999")
        self.assertEqual(record.rounds[0].alignments[4].length, 927)
        self.assertEqual(record.rounds[0].alignments[5].title, ">NED4_MOUSE P46935 mus musculus (mouse). nedd-4 protein (ec 6.3.2.-) (fragment). 11/1997")
        self.assertEqual(record.rounds[0].alignments[5].length, 957)
        self.assertEqual(record.rounds[0].alignments[6].title, ">URB1_RAT P51593 rattus norvegicus (rat). dna binding protein ure-b1 (ec 6.3.2.-). 10/1996")
        self.assertEqual(record.rounds[0].alignments[6].length, 310)
        self.assertEqual(record.rounds[0].alignments[7].title, ">UE3A_HUMAN Q05086 homo sapiens (human). ubiquitin-protein ligase e3a (ec 6.3.2.-) (oncogenic protein- associated protein e6-ap) (human papillomavirus e6-associated protein). 5/2000")
        self.assertEqual(record.rounds[0].alignments[7].length, 875)
        self.assertEqual(record.rounds[0].alignments[8].title, ">HUL5_YEAST P53119 saccharomyces cerevisiae (baker's yeast). probable ubiquitin--protein ligase hul5 (ec 6.3.2.-). 5/2000")
        self.assertEqual(record.rounds[0].alignments[8].length, 910)
        self.assertEqual(record.rounds[0].alignments[9].title, ">UE3A_MOUSE O08759 mus musculus (mouse). ubiquitin-protein ligase e3a (ec 6.3.2.-) (oncogenic protein- associated protein e6-ap). 5/2000")
        self.assertEqual(record.rounds[0].alignments[9].length, 885)
        self.assertEqual(record.rounds[0].alignments[10].title, ">HUL4_YEAST P40985 saccharomyces cerevisiae (baker's yeast). probable ubiquitin--protein ligase hul4 (ec 6.3.2.-). 5/2000")
        self.assertEqual(record.rounds[0].alignments[10].length, 892)
        self.assertEqual(record.rounds[0].alignments[11].title, ">UFD4_YEAST P33202 saccharomyces cerevisiae (baker's yeast). ubiquitin fusion degradation protein 4 (ub fusion protein 4). 11/1997")
        self.assertEqual(record.rounds[0].alignments[11].length, 1483)
        self.assertEqual(record.rounds[0].alignments[12].title, ">Y032_HUMAN Q15034 homo sapiens (human). hypothetical protein kiaa0032. 5/2000")
        self.assertEqual(record.rounds[0].alignments[12].length, 1050)
        self.assertEqual(record.rounds[0].alignments[13].title, ">TR12_HUMAN Q14669 homo sapiens (human). thyroid receptor interacting protein 12 (trip12) (kiaa0045). 12/1998")
        self.assertEqual(record.rounds[0].alignments[13].length, 1992)
        self.assertEqual(record.rounds[0].alignments[14].title, ">PAB1_MOUSE P29341 mus musculus (mouse). polyadenylate-binding protein 1 (poly(a) binding protein 1) (pabp 1). 7/1998")
        self.assertEqual(record.rounds[0].alignments[14].length, 636)
        self.assertEqual(record.rounds[0].alignments[15].title, ">PAB1_HUMAN P11940 homo sapiens (human). polyadenylate-binding protein 1 (poly(a) binding protein 1) (pabp 1). 7/1998")
        self.assertEqual(record.rounds[0].alignments[15].length, 636)
        self.assertEqual(record.rounds[0].alignments[16].title, ">PABP_XENLA P20965 xenopus laevis (african clawed frog). polyadenylate-binding protein (poly(a) binding protein) (pabp). 7/1998")
        self.assertEqual(record.rounds[0].alignments[16].length, 633)
        self.assertEqual(record.rounds[0].alignments[17].title, ">PABP_DROME P21187 drosophila melanogaster (fruit fly). polyadenylate-binding protein (poly(a) binding protein) (pabp). 2/1995")
        self.assertEqual(record.rounds[0].alignments[17].length, 632)
        self.assertEqual(record.rounds[0].alignments[18].title, ">PAB5_ARATH Q05196 arabidopsis thaliana (mouse-ear cress). polyadenylate-binding protein 5 (poly(a) binding protein 5) (pabp 5). 11/1995")
        self.assertEqual(record.rounds[0].alignments[18].length, 668)
        self.assertEqual(record.rounds[0].alignments[19].title, ">PAB2_ARATH P42731 arabidopsis thaliana (mouse-ear cress). polyadenylate-binding protein 2 (poly(a) binding protein 2) (pabp 2). 12/1998")
        self.assertEqual(record.rounds[0].alignments[19].length, 629)
        self.assertEqual(record.rounds[0].alignments[20].title, ">MCM3_MOUSE P25206 mus musculus (mouse). dna replication licensing factor mcm3 (dna polymerase alpha holoenzyme-associated protein p1) (p1-mcm3). 5/2000")
        self.assertEqual(record.rounds[0].alignments[20].length, 812)
        self.assertEqual(record.rounds[0].alignments[21].title, ">PABP_SCHPO P31209 schizosaccharomyces pombe (fission yeast). polyadenylate-binding protein (poly(a) binding protein) (pabp). 7/1998")
        self.assertEqual(record.rounds[0].alignments[21].length, 653)
        self.assertEqual(record.rounds[0].alignments[22].title, ">PROB_SERMA P17856 serratia marcescens. glutamate 5-kinase (ec 2.7.2.11) (gamma-glutamyl kinase) (gk). 10/1996")
        self.assertEqual(record.rounds[0].alignments[22].length, 367)
        self.assertEqual(record.rounds[0].alignments[23].title, ">ST20_CANAL Q92212 candida albicans (yeast). serine/threonine-protein kinase ste20 homolog (ec 2.7.1.-). 5/2000")
        self.assertEqual(record.rounds[0].alignments[23].length, 1230)
        self.assertEqual(record.rounds[0].alignments[24].title, ">KEND_HUMAN O95613 homo sapiens (human). kendrin (kiaa0402). 5/2000")
        self.assertEqual(record.rounds[0].alignments[24].length, 3321)
        self.assertEqual(record.rounds[0].alignments[25].title, ">FIXK_RHIME P13295 rhizobium meliloti (sinorhizobium meliloti). nitrogen fixation regulation protein fixk. 5/2000")
        self.assertEqual(record.rounds[0].alignments[25].length, 211)
        self.assertEqual(record.rounds[0].alignments[26].title, ">CC24_YEAST P11433 saccharomyces cerevisiae (baker's yeast). cell division control protein 24 (calcium regulatory protein). 7/1999")
        self.assertEqual(record.rounds[0].alignments[26].length, 854)

    def _check_text_2012L_psiblast_001_round1(self, record):
        self.assertEqual(len(record.rounds[1].new_seqs), 9)
        self.assertEqual(record.rounds[1].new_seqs[0].title, "PABP_DROME P21187 drosophila melanogaster (fruit fly). polya...")
        self.assertEqual(record.rounds[1].new_seqs[0].score, 67)
        self.assertAlmostEqual(record.rounds[1].new_seqs[0].e, 1e-10)
        self.assertEqual(record.rounds[1].new_seqs[1].title, "PABP_SCHPO P31209 schizosaccharomyces pombe (fission yeast)....")
        self.assertEqual(record.rounds[1].new_seqs[1].score, 44)
        self.assertAlmostEqual(record.rounds[1].new_seqs[1].e, 0.001)
        self.assertEqual(record.rounds[1].new_seqs[2].title, "PAB2_ARATH P42731 arabidopsis thaliana (mouse-ear cress). po...")
        self.assertEqual(record.rounds[1].new_seqs[2].score, 43)
        self.assertAlmostEqual(record.rounds[1].new_seqs[2].e, 0.003)
        self.assertEqual(record.rounds[1].new_seqs[3].title, "PAB5_ARATH Q05196 arabidopsis thaliana (mouse-ear cress). po...")
        self.assertEqual(record.rounds[1].new_seqs[3].score, 43)
        self.assertAlmostEqual(record.rounds[1].new_seqs[3].e, 0.003)
        self.assertEqual(record.rounds[1].new_seqs[4].title, "PABP_YEAST P04147 saccharomyces cerevisiae polyadenylate-bin...")
        self.assertEqual(record.rounds[1].new_seqs[4].score, 41)
        self.assertAlmostEqual(record.rounds[1].new_seqs[4].e, 0.01)
        self.assertEqual(record.rounds[1].new_seqs[5].title, "RNR_AQUAE O67834 aquifex aeolicus. ribonuclease r (ec 3.1.-....")
        self.assertEqual(record.rounds[1].new_seqs[5].score, 33)
        self.assertAlmostEqual(record.rounds[1].new_seqs[5].e, 2.2)
        self.assertEqual(record.rounds[1].new_seqs[6].title, "NIRA_EMENI P28348 emericella nidulans (aspergillus nidulans)...")
        self.assertEqual(record.rounds[1].new_seqs[6].score, 33)
        self.assertAlmostEqual(record.rounds[1].new_seqs[6].e, 2.9)
        self.assertEqual(record.rounds[1].new_seqs[7].title, "YK44_YEAST P36023 saccharomyces cerevisiae (baker's yeast). ...")
        self.assertEqual(record.rounds[1].new_seqs[7].score, 31)
        self.assertAlmostEqual(record.rounds[1].new_seqs[7].e, 8.4)
        self.assertEqual(record.rounds[1].new_seqs[8].title, "SYQ_YEAST P13188 saccharomyces cerevisiae (baker's yeast). g...")
        self.assertEqual(record.rounds[1].new_seqs[8].score, 31)
        self.assertAlmostEqual(record.rounds[1].new_seqs[8].e, 8.4)
        self.assertEqual(len(record.rounds[1].alignments), 26)
        self.assertEqual(record.rounds[1].alignments[0].title, ">100K_RAT Q62671 rattus norvegicus (rat). 100 kda protein (ec 6.3.2.-). 7/1999")
        self.assertEqual(record.rounds[1].alignments[0].length, 889)
        self.assertEqual(record.rounds[1].alignments[1].title, ">HYDP_DROME P51592 drosophila melanogaster (fruit fly). hyperplastic discs protein (hyd protein) (ec 6.3.2.-). 12/1998")
        self.assertEqual(record.rounds[1].alignments[1].length, 2895)
        self.assertEqual(record.rounds[1].alignments[2].title, ">NED4_HUMAN P46934 homo sapiens (human). nedd-4 protein (ec 6.3.2.-) (kiaa0093) (fragment). 7/1999")
        self.assertEqual(record.rounds[1].alignments[2].length, 927)
        self.assertEqual(record.rounds[1].alignments[3].title, ">PUB1_SCHPO Q92462 schizosaccharomyces pombe (fission yeast). ubiquitin--protein ligase pub1 (ec 6.3.2.-). 12/1998")
        self.assertEqual(record.rounds[1].alignments[3].length, 767)
        self.assertEqual(record.rounds[1].alignments[4].title, ">NED4_MOUSE P46935 mus musculus (mouse). nedd-4 protein (ec 6.3.2.-) (fragment). 11/1997")
        self.assertEqual(record.rounds[1].alignments[4].length, 957)
        self.assertEqual(record.rounds[1].alignments[5].title, ">RSP5_YEAST P39940 saccharomyces cerevisiae (baker's yeast). ubiquitin--protein ligase rsp5 (ec 6.3.2.-). 7/1999")
        self.assertEqual(record.rounds[1].alignments[5].length, 809)
        self.assertEqual(record.rounds[1].alignments[6].title, ">UE3A_HUMAN Q05086 homo sapiens (human). ubiquitin-protein ligase e3a (ec 6.3.2.-) (oncogenic protein- associated protein e6-ap) (human papillomavirus e6-associated protein). 5/2000")
        self.assertEqual(record.rounds[1].alignments[6].length, 875)
        self.assertEqual(record.rounds[1].alignments[7].title, ">UE3A_MOUSE O08759 mus musculus (mouse). ubiquitin-protein ligase e3a (ec 6.3.2.-) (oncogenic protein- associated protein e6-ap). 5/2000")
        self.assertEqual(record.rounds[1].alignments[7].length, 885)
        self.assertEqual(record.rounds[1].alignments[8].title, ">HUL4_YEAST P40985 saccharomyces cerevisiae (baker's yeast). probable ubiquitin--protein ligase hul4 (ec 6.3.2.-). 5/2000")
        self.assertEqual(record.rounds[1].alignments[8].length, 892)
        self.assertEqual(record.rounds[1].alignments[9].title, ">URB1_RAT P51593 rattus norvegicus (rat). dna binding protein ure-b1 (ec 6.3.2.-). 10/1996")
        self.assertEqual(record.rounds[1].alignments[9].length, 310)
        self.assertEqual(record.rounds[1].alignments[10].title, ">TR12_HUMAN Q14669 homo sapiens (human). thyroid receptor interacting protein 12 (trip12) (kiaa0045). 12/1998")
        self.assertEqual(record.rounds[1].alignments[10].length, 1992)
        self.assertEqual(record.rounds[1].alignments[11].title, ">Y032_HUMAN Q15034 homo sapiens (human). hypothetical protein kiaa0032. 5/2000")
        self.assertEqual(record.rounds[1].alignments[11].length, 1050)
        self.assertEqual(record.rounds[1].alignments[12].title, ">HUL5_YEAST P53119 saccharomyces cerevisiae (baker's yeast). probable ubiquitin--protein ligase hul5 (ec 6.3.2.-). 5/2000")
        self.assertEqual(record.rounds[1].alignments[12].length, 910)
        self.assertEqual(record.rounds[1].alignments[13].title, ">UFD4_YEAST P33202 saccharomyces cerevisiae (baker's yeast). ubiquitin fusion degradation protein 4 (ub fusion protein 4). 11/1997")
        self.assertEqual(record.rounds[1].alignments[13].length, 1483)
        self.assertEqual(record.rounds[1].alignments[14].title, ">PAB1_HUMAN P11940 homo sapiens (human). polyadenylate-binding protein 1 (poly(a) binding protein 1) (pabp 1). 7/1998")
        self.assertEqual(record.rounds[1].alignments[14].length, 636)
        self.assertEqual(record.rounds[1].alignments[15].title, ">PABP_XENLA P20965 xenopus laevis (african clawed frog). polyadenylate-binding protein (poly(a) binding protein) (pabp). 7/1998")
        self.assertEqual(record.rounds[1].alignments[15].length, 633)
        self.assertEqual(record.rounds[1].alignments[16].title, ">PAB1_MOUSE P29341 mus musculus (mouse). polyadenylate-binding protein 1 (poly(a) binding protein 1) (pabp 1). 7/1998")
        self.assertEqual(record.rounds[1].alignments[16].length, 636)
        self.assertEqual(record.rounds[1].alignments[17].title, ">PABP_DROME P21187 drosophila melanogaster (fruit fly). polyadenylate-binding protein (poly(a) binding protein) (pabp). 2/1995")
        self.assertEqual(record.rounds[1].alignments[17].length, 632)
        self.assertEqual(record.rounds[1].alignments[18].title, ">PABP_SCHPO P31209 schizosaccharomyces pombe (fission yeast). polyadenylate-binding protein (poly(a) binding protein) (pabp). 7/1998")
        self.assertEqual(record.rounds[1].alignments[18].length, 653)
        self.assertEqual(record.rounds[1].alignments[19].title, ">PAB2_ARATH P42731 arabidopsis thaliana (mouse-ear cress). polyadenylate-binding protein 2 (poly(a) binding protein 2) (pabp 2). 12/1998")
        self.assertEqual(record.rounds[1].alignments[19].length, 629)
        self.assertEqual(record.rounds[1].alignments[20].title, ">PAB5_ARATH Q05196 arabidopsis thaliana (mouse-ear cress). polyadenylate-binding protein 5 (poly(a) binding protein 5) (pabp 5). 11/1995")
        self.assertEqual(record.rounds[1].alignments[20].length, 668)
        self.assertEqual(record.rounds[1].alignments[21].title, ">PABP_YEAST P04147 saccharomyces cerevisiae polyadenylate-binding protein, cytoplasmic and nuclear (pabp) (ars consensus binding protein acbp-67) (polyadenylate tail-binding protein). 2/1996")
        self.assertEqual(record.rounds[1].alignments[21].length, 576)
        self.assertEqual(record.rounds[1].alignments[22].title, ">RNR_AQUAE O67834 aquifex aeolicus. ribonuclease r (ec 3.1.-.-) (rnase r) (vacb protein homolog). 5/2000")
        self.assertEqual(record.rounds[1].alignments[22].length, 705)
        self.assertEqual(record.rounds[1].alignments[23].title, ">NIRA_EMENI P28348 emericella nidulans (aspergillus nidulans). nitrogen assimilation transcription factor nira. 4/1993")
        self.assertEqual(record.rounds[1].alignments[23].length, 892)
        self.assertEqual(record.rounds[1].alignments[24].title, ">YK44_YEAST P36023 saccharomyces cerevisiae (baker's yeast). putative 101.8 kda transcriptional regulatory protein in las1-ccp1 intergenic region. 2/1995")
        self.assertEqual(record.rounds[1].alignments[24].length, 863)
        self.assertEqual(record.rounds[1].alignments[25].title, ">SYQ_YEAST P13188 saccharomyces cerevisiae (baker's yeast). glutaminyl-trna synthetase (ec 6.1.1.18) (glutamine--trna ligase) (glnrs). 11/1997")
        self.assertEqual(record.rounds[1].alignments[25].length, 809)

    def _check_text_2012L_psiblast_001_round2(self, record):
        self.assertEqual(len(record.rounds[2].new_seqs), 6)
        self.assertEqual(record.rounds[2].new_seqs[0].title, "PAB2_ARATH P42731 arabidopsis thaliana (mouse-ear cress). po...")
        self.assertEqual(record.rounds[2].new_seqs[0].score, 48)
        self.assertAlmostEqual(record.rounds[2].new_seqs[0].e, 1e-4)
        self.assertEqual(record.rounds[2].new_seqs[1].title, "PABP_SCHPO P31209 schizosaccharomyces pombe (fission yeast)....")
        self.assertEqual(record.rounds[2].new_seqs[1].score, 45)
        self.assertAlmostEqual(record.rounds[2].new_seqs[1].e, 7e-4)
        self.assertEqual(record.rounds[2].new_seqs[2].title, "PAB5_ARATH Q05196 arabidopsis thaliana (mouse-ear cress). po...")
        self.assertEqual(record.rounds[2].new_seqs[2].score, 44)
        self.assertAlmostEqual(record.rounds[2].new_seqs[2].e, 0.001)
        self.assertEqual(record.rounds[2].new_seqs[3].title, "PABP_YEAST P04147 saccharomyces cerevisiae polyadenylate-bin...")
        self.assertEqual(record.rounds[2].new_seqs[3].score, 42)
        self.assertAlmostEqual(record.rounds[2].new_seqs[3].e, 0.006)
        self.assertEqual(record.rounds[2].new_seqs[4].title, "NIRA_EMENI P28348 emericella nidulans (aspergillus nidulans)...")
        self.assertEqual(record.rounds[2].new_seqs[4].score, 33)
        self.assertAlmostEqual(record.rounds[2].new_seqs[4].e, 2.9)
        self.assertEqual(record.rounds[2].new_seqs[5].title, "RNR_AQUAE O67834 aquifex aeolicus. ribonuclease r (ec 3.1.-....")
        self.assertEqual(record.rounds[2].new_seqs[5].score, 32)
        self.assertAlmostEqual(record.rounds[2].new_seqs[5].e, 3.7)
        self.assertEqual(len(record.rounds[2].alignments), 24)
        self.assertEqual(record.rounds[2].alignments[0].title, ">100K_RAT Q62671 rattus norvegicus (rat). 100 kda protein (ec 6.3.2.-). 7/1999")
        self.assertEqual(record.rounds[2].alignments[0].length, 889)
        self.assertEqual(record.rounds[2].alignments[1].title, ">HYDP_DROME P51592 drosophila melanogaster (fruit fly). hyperplastic discs protein (hyd protein) (ec 6.3.2.-). 12/1998")
        self.assertEqual(record.rounds[2].alignments[1].length, 2895)
        self.assertEqual(record.rounds[2].alignments[2].title, ">PUB1_SCHPO Q92462 schizosaccharomyces pombe (fission yeast). ubiquitin--protein ligase pub1 (ec 6.3.2.-). 12/1998")
        self.assertEqual(record.rounds[2].alignments[2].length, 767)
        self.assertEqual(record.rounds[2].alignments[3].title, ">NED4_HUMAN P46934 homo sapiens (human). nedd-4 protein (ec 6.3.2.-) (kiaa0093) (fragment). 7/1999")
        self.assertEqual(record.rounds[2].alignments[3].length, 927)
        self.assertEqual(record.rounds[2].alignments[4].title, ">NED4_MOUSE P46935 mus musculus (mouse). nedd-4 protein (ec 6.3.2.-) (fragment). 11/1997")
        self.assertEqual(record.rounds[2].alignments[4].length, 957)
        self.assertEqual(record.rounds[2].alignments[5].title, ">RSP5_YEAST P39940 saccharomyces cerevisiae (baker's yeast). ubiquitin--protein ligase rsp5 (ec 6.3.2.-). 7/1999")
        self.assertEqual(record.rounds[2].alignments[5].length, 809)
        self.assertEqual(record.rounds[2].alignments[6].title, ">UE3A_HUMAN Q05086 homo sapiens (human). ubiquitin-protein ligase e3a (ec 6.3.2.-) (oncogenic protein- associated protein e6-ap) (human papillomavirus e6-associated protein). 5/2000")
        self.assertEqual(record.rounds[2].alignments[6].length, 875)
        self.assertEqual(record.rounds[2].alignments[7].title, ">UE3A_MOUSE O08759 mus musculus (mouse). ubiquitin-protein ligase e3a (ec 6.3.2.-) (oncogenic protein- associated protein e6-ap). 5/2000")
        self.assertEqual(record.rounds[2].alignments[7].length, 885)
        self.assertEqual(record.rounds[2].alignments[8].title, ">HUL4_YEAST P40985 saccharomyces cerevisiae (baker's yeast). probable ubiquitin--protein ligase hul4 (ec 6.3.2.-). 5/2000")
        self.assertEqual(record.rounds[2].alignments[8].length, 892)
        self.assertEqual(record.rounds[2].alignments[9].title, ">TR12_HUMAN Q14669 homo sapiens (human). thyroid receptor interacting protein 12 (trip12) (kiaa0045). 12/1998")
        self.assertEqual(record.rounds[2].alignments[9].length, 1992)
        self.assertEqual(record.rounds[2].alignments[10].title, ">URB1_RAT P51593 rattus norvegicus (rat). dna binding protein ure-b1 (ec 6.3.2.-). 10/1996")
        self.assertEqual(record.rounds[2].alignments[10].length, 310)
        self.assertEqual(record.rounds[2].alignments[11].title, ">Y032_HUMAN Q15034 homo sapiens (human). hypothetical protein kiaa0032. 5/2000")
        self.assertEqual(record.rounds[2].alignments[11].length, 1050)
        self.assertEqual(record.rounds[2].alignments[12].title, ">UFD4_YEAST P33202 saccharomyces cerevisiae (baker's yeast). ubiquitin fusion degradation protein 4 (ub fusion protein 4). 11/1997")
        self.assertEqual(record.rounds[2].alignments[12].length, 1483)
        self.assertEqual(record.rounds[2].alignments[13].title, ">HUL5_YEAST P53119 saccharomyces cerevisiae (baker's yeast). probable ubiquitin--protein ligase hul5 (ec 6.3.2.-). 5/2000")
        self.assertEqual(record.rounds[2].alignments[13].length, 910)
        self.assertEqual(record.rounds[2].alignments[14].title, ">PABP_DROME P21187 drosophila melanogaster (fruit fly). polyadenylate-binding protein (poly(a) binding protein) (pabp). 2/1995")
        self.assertEqual(record.rounds[2].alignments[14].length, 632)
        self.assertEqual(record.rounds[2].alignments[15].title, ">PAB1_HUMAN P11940 homo sapiens (human). polyadenylate-binding protein 1 (poly(a) binding protein 1) (pabp 1). 7/1998")
        self.assertEqual(record.rounds[2].alignments[15].length, 636)
        self.assertEqual(record.rounds[2].alignments[16].title, ">PABP_XENLA P20965 xenopus laevis (african clawed frog). polyadenylate-binding protein (poly(a) binding protein) (pabp). 7/1998")
        self.assertEqual(record.rounds[2].alignments[16].length, 633)
        self.assertEqual(record.rounds[2].alignments[17].title, ">PAB1_MOUSE P29341 mus musculus (mouse). polyadenylate-binding protein 1 (poly(a) binding protein 1) (pabp 1). 7/1998")
        self.assertEqual(record.rounds[2].alignments[17].length, 636)
        self.assertEqual(record.rounds[2].alignments[18].title, ">PAB2_ARATH P42731 arabidopsis thaliana (mouse-ear cress). polyadenylate-binding protein 2 (poly(a) binding protein 2) (pabp 2). 12/1998")
        self.assertEqual(record.rounds[2].alignments[18].length, 629)
        self.assertEqual(record.rounds[2].alignments[19].title, ">PABP_SCHPO P31209 schizosaccharomyces pombe (fission yeast). polyadenylate-binding protein (poly(a) binding protein) (pabp). 7/1998")
        self.assertEqual(record.rounds[2].alignments[19].length, 653)
        self.assertEqual(record.rounds[2].alignments[20].title, ">PAB5_ARATH Q05196 arabidopsis thaliana (mouse-ear cress). polyadenylate-binding protein 5 (poly(a) binding protein 5) (pabp 5). 11/1995")
        self.assertEqual(record.rounds[2].alignments[20].length, 668)
        self.assertEqual(record.rounds[2].alignments[21].title, ">PABP_YEAST P04147 saccharomyces cerevisiae polyadenylate-binding protein, cytoplasmic and nuclear (pabp) (ars consensus binding protein acbp-67) (polyadenylate tail-binding protein). 2/1996")
        self.assertEqual(record.rounds[2].alignments[21].length, 576)
        self.assertEqual(record.rounds[2].alignments[22].title, ">NIRA_EMENI P28348 emericella nidulans (aspergillus nidulans). nitrogen assimilation transcription factor nira. 4/1993")
        self.assertEqual(record.rounds[2].alignments[22].length, 892)
        self.assertEqual(record.rounds[2].alignments[23].title, ">RNR_AQUAE O67834 aquifex aeolicus. ribonuclease r (ec 3.1.-.-) (rnase r) (vacb protein homolog). 5/2000")
        self.assertEqual(record.rounds[2].alignments[23].length, 705)

    def _check_text_2012L_psiblast_001_round3(self, record):
        self.assertEqual(len(record.rounds[3].new_seqs), 4)
        self.assertEqual(record.rounds[3].new_seqs[0].title, "PAB5_ARATH Q05196 arabidopsis thaliana (mouse-ear cress). po...")
        self.assertEqual(record.rounds[3].new_seqs[0].score, 51)
        self.assertAlmostEqual(record.rounds[3].new_seqs[0].e, 9e-6)
        self.assertEqual(record.rounds[3].new_seqs[1].title, "PABP_YEAST P04147 saccharomyces cerevisiae polyadenylate-bin...")
        self.assertEqual(record.rounds[3].new_seqs[1].score, 45)
        self.assertAlmostEqual(record.rounds[3].new_seqs[1].e, 5e-4)
        self.assertEqual(record.rounds[3].new_seqs[2].title, "RNR_AQUAE O67834 aquifex aeolicus. ribonuclease r (ec 3.1.-....")
        self.assertEqual(record.rounds[3].new_seqs[2].score, 33)
        self.assertAlmostEqual(record.rounds[3].new_seqs[2].e, 2.9)
        self.assertEqual(record.rounds[3].new_seqs[3].title, "NIRA_EMENI P28348 emericella nidulans (aspergillus nidulans)...")
        self.assertEqual(record.rounds[3].new_seqs[3].score, 33)
        self.assertAlmostEqual(record.rounds[3].new_seqs[3].e, 2.9)
        self.assertEqual(len(record.rounds[3].alignments), 24)
        self.assertEqual(record.rounds[3].alignments[0].title, ">100K_RAT Q62671 rattus norvegicus (rat). 100 kda protein (ec 6.3.2.-). 7/1999")
        self.assertEqual(record.rounds[3].alignments[0].length, 889)
        self.assertEqual(record.rounds[3].alignments[1].title, ">HYDP_DROME P51592 drosophila melanogaster (fruit fly). hyperplastic discs protein (hyd protein) (ec 6.3.2.-). 12/1998")
        self.assertEqual(record.rounds[3].alignments[1].length, 2895)
        self.assertEqual(record.rounds[3].alignments[2].title, ">PUB1_SCHPO Q92462 schizosaccharomyces pombe (fission yeast). ubiquitin--protein ligase pub1 (ec 6.3.2.-). 12/1998")
        self.assertEqual(record.rounds[3].alignments[2].length, 767)
        self.assertEqual(record.rounds[3].alignments[3].title, ">NED4_HUMAN P46934 homo sapiens (human). nedd-4 protein (ec 6.3.2.-) (kiaa0093) (fragment). 7/1999")
        self.assertEqual(record.rounds[3].alignments[3].length, 927)
        self.assertEqual(record.rounds[3].alignments[4].title, ">NED4_MOUSE P46935 mus musculus (mouse). nedd-4 protein (ec 6.3.2.-) (fragment). 11/1997")
        self.assertEqual(record.rounds[3].alignments[4].length, 957)
        self.assertEqual(record.rounds[3].alignments[5].title, ">RSP5_YEAST P39940 saccharomyces cerevisiae (baker's yeast). ubiquitin--protein ligase rsp5 (ec 6.3.2.-). 7/1999")
        self.assertEqual(record.rounds[3].alignments[5].length, 809)
        self.assertEqual(record.rounds[3].alignments[6].title, ">UE3A_HUMAN Q05086 homo sapiens (human). ubiquitin-protein ligase e3a (ec 6.3.2.-) (oncogenic protein- associated protein e6-ap) (human papillomavirus e6-associated protein). 5/2000")
        self.assertEqual(record.rounds[3].alignments[6].length, 875)
        self.assertEqual(record.rounds[3].alignments[7].title, ">UE3A_MOUSE O08759 mus musculus (mouse). ubiquitin-protein ligase e3a (ec 6.3.2.-) (oncogenic protein- associated protein e6-ap). 5/2000")
        self.assertEqual(record.rounds[3].alignments[7].length, 885)
        self.assertEqual(record.rounds[3].alignments[8].title, ">HUL4_YEAST P40985 saccharomyces cerevisiae (baker's yeast). probable ubiquitin--protein ligase hul4 (ec 6.3.2.-). 5/2000")
        self.assertEqual(record.rounds[3].alignments[8].length, 892)
        self.assertEqual(record.rounds[3].alignments[9].title, ">TR12_HUMAN Q14669 homo sapiens (human). thyroid receptor interacting protein 12 (trip12) (kiaa0045). 12/1998")
        self.assertEqual(record.rounds[3].alignments[9].length, 1992)
        self.assertEqual(record.rounds[3].alignments[10].title, ">URB1_RAT P51593 rattus norvegicus (rat). dna binding protein ure-b1 (ec 6.3.2.-). 10/1996")
        self.assertEqual(record.rounds[3].alignments[10].length, 310)
        self.assertEqual(record.rounds[3].alignments[11].title, ">Y032_HUMAN Q15034 homo sapiens (human). hypothetical protein kiaa0032. 5/2000")
        self.assertEqual(record.rounds[3].alignments[11].length, 1050)
        self.assertEqual(record.rounds[3].alignments[12].title, ">UFD4_YEAST P33202 saccharomyces cerevisiae (baker's yeast). ubiquitin fusion degradation protein 4 (ub fusion protein 4). 11/1997")
        self.assertEqual(record.rounds[3].alignments[12].length, 1483)
        self.assertEqual(record.rounds[3].alignments[13].title, ">HUL5_YEAST P53119 saccharomyces cerevisiae (baker's yeast). probable ubiquitin--protein ligase hul5 (ec 6.3.2.-). 5/2000")
        self.assertEqual(record.rounds[3].alignments[13].length, 910)
        self.assertEqual(record.rounds[3].alignments[14].title, ">PAB1_HUMAN P11940 homo sapiens (human). polyadenylate-binding protein 1 (poly(a) binding protein 1) (pabp 1). 7/1998")
        self.assertEqual(record.rounds[3].alignments[14].length, 636)
        self.assertEqual(record.rounds[3].alignments[15].title, ">PABP_DROME P21187 drosophila melanogaster (fruit fly). polyadenylate-binding protein (poly(a) binding protein) (pabp). 2/1995")
        self.assertEqual(record.rounds[3].alignments[15].length, 632)
        self.assertEqual(record.rounds[3].alignments[16].title, ">PABP_XENLA P20965 xenopus laevis (african clawed frog). polyadenylate-binding protein (poly(a) binding protein) (pabp). 7/1998")
        self.assertEqual(record.rounds[3].alignments[16].length, 633)
        self.assertEqual(record.rounds[3].alignments[17].title, ">PAB1_MOUSE P29341 mus musculus (mouse). polyadenylate-binding protein 1 (poly(a) binding protein 1) (pabp 1). 7/1998")
        self.assertEqual(record.rounds[3].alignments[17].length, 636)
        self.assertEqual(record.rounds[3].alignments[18].title, ">PABP_SCHPO P31209 schizosaccharomyces pombe (fission yeast). polyadenylate-binding protein (poly(a) binding protein) (pabp). 7/1998")
        self.assertEqual(record.rounds[3].alignments[18].length, 653)
        self.assertEqual(record.rounds[3].alignments[19].title, ">PAB2_ARATH P42731 arabidopsis thaliana (mouse-ear cress). polyadenylate-binding protein 2 (poly(a) binding protein 2) (pabp 2). 12/1998")
        self.assertEqual(record.rounds[3].alignments[19].length, 629)
        self.assertEqual(record.rounds[3].alignments[20].title, ">PAB5_ARATH Q05196 arabidopsis thaliana (mouse-ear cress). polyadenylate-binding protein 5 (poly(a) binding protein 5) (pabp 5). 11/1995")
        self.assertEqual(record.rounds[3].alignments[20].length, 668)
        self.assertEqual(record.rounds[3].alignments[21].title, ">PABP_YEAST P04147 saccharomyces cerevisiae polyadenylate-binding protein, cytoplasmic and nuclear (pabp) (ars consensus binding protein acbp-67) (polyadenylate tail-binding protein). 2/1996")
        self.assertEqual(record.rounds[3].alignments[21].length, 576)
        self.assertEqual(record.rounds[3].alignments[22].title, ">RNR_AQUAE O67834 aquifex aeolicus. ribonuclease r (ec 3.1.-.-) (rnase r) (vacb protein homolog). 5/2000")
        self.assertEqual(record.rounds[3].alignments[22].length, 705)
        self.assertEqual(record.rounds[3].alignments[23].title, ">NIRA_EMENI P28348 emericella nidulans (aspergillus nidulans). nitrogen assimilation transcription factor nira. 4/1993")
        self.assertEqual(record.rounds[3].alignments[23].length, 892)

    def _check_text_2012L_psiblast_001_round4(self, record):
        self.assertEqual(len(record.rounds[4].new_seqs), 2)
        self.assertEqual(record.rounds[4].new_seqs[0].title, "RNR_AQUAE O67834 aquifex aeolicus. ribonuclease r (ec 3.1.-....")
        self.assertEqual(record.rounds[4].new_seqs[0].score, 33)
        self.assertAlmostEqual(record.rounds[4].new_seqs[0].e, 2.2)
        self.assertEqual(record.rounds[4].new_seqs[1].title, "NIRA_EMENI P28348 emericella nidulans (aspergillus nidulans)...")
        self.assertEqual(record.rounds[4].new_seqs[1].score, 33)
        self.assertAlmostEqual(record.rounds[4].new_seqs[1].e, 2.9)
        self.assertEqual(len(record.rounds[4].alignments), 24)
        self.assertEqual(record.rounds[4].alignments[0].title, ">100K_RAT Q62671 rattus norvegicus (rat). 100 kda protein (ec 6.3.2.-). 7/1999")
        self.assertEqual(record.rounds[4].alignments[0].length, 889)
        self.assertEqual(record.rounds[4].alignments[1].title, ">HYDP_DROME P51592 drosophila melanogaster (fruit fly). hyperplastic discs protein (hyd protein) (ec 6.3.2.-). 12/1998")
        self.assertEqual(record.rounds[4].alignments[1].length, 2895)
        self.assertEqual(record.rounds[4].alignments[2].title, ">PUB1_SCHPO Q92462 schizosaccharomyces pombe (fission yeast). ubiquitin--protein ligase pub1 (ec 6.3.2.-). 12/1998")
        self.assertEqual(record.rounds[4].alignments[2].length, 767)
        self.assertEqual(record.rounds[4].alignments[3].title, ">NED4_HUMAN P46934 homo sapiens (human). nedd-4 protein (ec 6.3.2.-) (kiaa0093) (fragment). 7/1999")
        self.assertEqual(record.rounds[4].alignments[3].length, 927)
        self.assertEqual(record.rounds[4].alignments[4].title, ">NED4_MOUSE P46935 mus musculus (mouse). nedd-4 protein (ec 6.3.2.-) (fragment). 11/1997")
        self.assertEqual(record.rounds[4].alignments[4].length, 957)
        self.assertEqual(record.rounds[4].alignments[5].title, ">RSP5_YEAST P39940 saccharomyces cerevisiae (baker's yeast). ubiquitin--protein ligase rsp5 (ec 6.3.2.-). 7/1999")
        self.assertEqual(record.rounds[4].alignments[5].length, 809)
        self.assertEqual(record.rounds[4].alignments[6].title, ">UE3A_HUMAN Q05086 homo sapiens (human). ubiquitin-protein ligase e3a (ec 6.3.2.-) (oncogenic protein- associated protein e6-ap) (human papillomavirus e6-associated protein). 5/2000")
        self.assertEqual(record.rounds[4].alignments[6].length, 875)
        self.assertEqual(record.rounds[4].alignments[7].title, ">UE3A_MOUSE O08759 mus musculus (mouse). ubiquitin-protein ligase e3a (ec 6.3.2.-) (oncogenic protein- associated protein e6-ap). 5/2000")
        self.assertEqual(record.rounds[4].alignments[7].length, 885)
        self.assertEqual(record.rounds[4].alignments[8].title, ">HUL4_YEAST P40985 saccharomyces cerevisiae (baker's yeast). probable ubiquitin--protein ligase hul4 (ec 6.3.2.-). 5/2000")
        self.assertEqual(record.rounds[4].alignments[8].length, 892)
        self.assertEqual(record.rounds[4].alignments[9].title, ">TR12_HUMAN Q14669 homo sapiens (human). thyroid receptor interacting protein 12 (trip12) (kiaa0045). 12/1998")
        self.assertEqual(record.rounds[4].alignments[9].length, 1992)
        self.assertEqual(record.rounds[4].alignments[10].title, ">URB1_RAT P51593 rattus norvegicus (rat). dna binding protein ure-b1 (ec 6.3.2.-). 10/1996")
        self.assertEqual(record.rounds[4].alignments[10].length, 310)
        self.assertEqual(record.rounds[4].alignments[11].title, ">Y032_HUMAN Q15034 homo sapiens (human). hypothetical protein kiaa0032. 5/2000")
        self.assertEqual(record.rounds[4].alignments[11].length, 1050)
        self.assertEqual(record.rounds[4].alignments[12].title, ">UFD4_YEAST P33202 saccharomyces cerevisiae (baker's yeast). ubiquitin fusion degradation protein 4 (ub fusion protein 4). 11/1997")
        self.assertEqual(record.rounds[4].alignments[12].length, 1483)
        self.assertEqual(record.rounds[4].alignments[13].title, ">HUL5_YEAST P53119 saccharomyces cerevisiae (baker's yeast). probable ubiquitin--protein ligase hul5 (ec 6.3.2.-). 5/2000")
        self.assertEqual(record.rounds[4].alignments[13].length, 910)
        self.assertEqual(record.rounds[4].alignments[14].title, ">PABP_DROME P21187 drosophila melanogaster (fruit fly). polyadenylate-binding protein (poly(a) binding protein) (pabp). 2/1995")
        self.assertEqual(record.rounds[4].alignments[14].length, 632)
        self.assertEqual(record.rounds[4].alignments[15].title, ">PAB1_HUMAN P11940 homo sapiens (human). polyadenylate-binding protein 1 (poly(a) binding protein 1) (pabp 1). 7/1998")
        self.assertEqual(record.rounds[4].alignments[15].length, 636)
        self.assertEqual(record.rounds[4].alignments[16].title, ">PAB1_MOUSE P29341 mus musculus (mouse). polyadenylate-binding protein 1 (poly(a) binding protein 1) (pabp 1). 7/1998")
        self.assertEqual(record.rounds[4].alignments[16].length, 636)
        self.assertEqual(record.rounds[4].alignments[17].title, ">PABP_XENLA P20965 xenopus laevis (african clawed frog). polyadenylate-binding protein (poly(a) binding protein) (pabp). 7/1998")
        self.assertEqual(record.rounds[4].alignments[17].length, 633)
        self.assertEqual(record.rounds[4].alignments[18].title, ">PABP_SCHPO P31209 schizosaccharomyces pombe (fission yeast). polyadenylate-binding protein (poly(a) binding protein) (pabp). 7/1998")
        self.assertEqual(record.rounds[4].alignments[18].length, 653)
        self.assertEqual(record.rounds[4].alignments[19].title, ">PAB2_ARATH P42731 arabidopsis thaliana (mouse-ear cress). polyadenylate-binding protein 2 (poly(a) binding protein 2) (pabp 2). 12/1998")
        self.assertEqual(record.rounds[4].alignments[19].length, 629)
        self.assertEqual(record.rounds[4].alignments[20].title, ">PAB5_ARATH Q05196 arabidopsis thaliana (mouse-ear cress). polyadenylate-binding protein 5 (poly(a) binding protein 5) (pabp 5). 11/1995")
        self.assertEqual(record.rounds[4].alignments[20].length, 668)
        self.assertEqual(record.rounds[4].alignments[21].title, ">PABP_YEAST P04147 saccharomyces cerevisiae polyadenylate-binding protein, cytoplasmic and nuclear (pabp) (ars consensus binding protein acbp-67) (polyadenylate tail-binding protein). 2/1996")
        self.assertEqual(record.rounds[4].alignments[21].length, 576)
        self.assertEqual(record.rounds[4].alignments[22].title, ">RNR_AQUAE O67834 aquifex aeolicus. ribonuclease r (ec 3.1.-.-) (rnase r) (vacb protein homolog). 5/2000")
        self.assertEqual(record.rounds[4].alignments[22].length, 705)
        self.assertEqual(record.rounds[4].alignments[23].title, ">NIRA_EMENI P28348 emericella nidulans (aspergillus nidulans). nitrogen assimilation transcription factor nira. 4/1993")
        self.assertEqual(record.rounds[4].alignments[23].length, 892)

    def _check_text_2012L_psiblast_001_hsps(self, record):
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].score, 3882)
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].bits, 1516)
        self.assertAlmostEqual(record.rounds[0].alignments[0].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[0].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].score, 1308)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].bits, 513)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[0].expect, 1e-145)
        self.assertEqual(record.rounds[0].alignments[1].hsps[1].score, 272)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[1].bits, 110)
        self.assertAlmostEqual(record.rounds[0].alignments[1].hsps[1].expect, 1e-23)
        self.assertEqual(len(record.rounds[0].alignments[1].hsps), 2)
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].score, 232)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].bits, 94.8)
        self.assertAlmostEqual(record.rounds[0].alignments[2].hsps[0].expect, 7e-19)
        self.assertEqual(len(record.rounds[0].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].score, 227)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].bits, 92.8)
        self.assertAlmostEqual(record.rounds[0].alignments[3].hsps[0].expect, 3e-18)
        self.assertEqual(len(record.rounds[0].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].score, 214)
        self.assertAlmostEqual(record.rounds[0].alignments[4].hsps[0].bits, 87.8)
        self.assertAlmostEqual(record.rounds[0].alignments[4].hsps[0].expect, 9e-17)
        self.assertEqual(len(record.rounds[0].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].score, 212)
        self.assertAlmostEqual(record.rounds[0].alignments[5].hsps[0].bits, 87.0)
        self.assertAlmostEqual(record.rounds[0].alignments[5].hsps[0].expect, 1e-16)
        self.assertEqual(len(record.rounds[0].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].score, 208)
        self.assertAlmostEqual(record.rounds[0].alignments[6].hsps[0].bits, 85.4)
        self.assertAlmostEqual(record.rounds[0].alignments[6].hsps[0].expect, 4e-16)
        self.assertEqual(len(record.rounds[0].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].score, 204)
        self.assertAlmostEqual(record.rounds[0].alignments[7].hsps[0].bits, 83.9)
        self.assertAlmostEqual(record.rounds[0].alignments[7].hsps[0].expect, 1e-15)
        self.assertEqual(len(record.rounds[0].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].score, 202)
        self.assertAlmostEqual(record.rounds[0].alignments[8].hsps[0].bits, 83.1)
        self.assertAlmostEqual(record.rounds[0].alignments[8].hsps[0].expect, 2e-15)
        self.assertEqual(len(record.rounds[0].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].score, 198)
        self.assertAlmostEqual(record.rounds[0].alignments[9].hsps[0].bits, 81.5)
        self.assertAlmostEqual(record.rounds[0].alignments[9].hsps[0].expect, 6e-15)
        self.assertEqual(len(record.rounds[0].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].score, 167)
        self.assertAlmostEqual(record.rounds[0].alignments[10].hsps[0].bits, 69.5)
        self.assertAlmostEqual(record.rounds[0].alignments[10].hsps[0].expect, 3e-11)
        self.assertEqual(len(record.rounds[0].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].score, 138)
        self.assertAlmostEqual(record.rounds[0].alignments[11].hsps[0].bits, 58.2)
        self.assertAlmostEqual(record.rounds[0].alignments[11].hsps[0].expect, 7e-8)
        self.assertEqual(len(record.rounds[0].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].score, 133)
        self.assertAlmostEqual(record.rounds[0].alignments[12].hsps[0].bits, 56.2)
        self.assertAlmostEqual(record.rounds[0].alignments[12].hsps[0].expect, 3e-7)
        self.assertEqual(len(record.rounds[0].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].score, 118)
        self.assertAlmostEqual(record.rounds[0].alignments[13].hsps[0].bits, 50.4)
        self.assertAlmostEqual(record.rounds[0].alignments[13].hsps[0].expect, 2e-5)
        self.assertEqual(len(record.rounds[0].alignments[13].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].score, 115)
        self.assertAlmostEqual(record.rounds[0].alignments[14].hsps[0].bits, 49.2)
        self.assertAlmostEqual(record.rounds[0].alignments[14].hsps[0].expect, 3e-5)
        self.assertEqual(len(record.rounds[0].alignments[14].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].score, 114)
        self.assertAlmostEqual(record.rounds[0].alignments[15].hsps[0].bits, 48.8)
        self.assertAlmostEqual(record.rounds[0].alignments[15].hsps[0].expect, 5e-5)
        self.assertEqual(len(record.rounds[0].alignments[15].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].score, 112)
        self.assertAlmostEqual(record.rounds[0].alignments[16].hsps[0].bits, 48.0)
        self.assertAlmostEqual(record.rounds[0].alignments[16].hsps[0].expect, 8e-5)
        self.assertEqual(len(record.rounds[0].alignments[16].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].score, 95)
        self.assertAlmostEqual(record.rounds[0].alignments[17].hsps[0].bits, 41.4)
        self.assertAlmostEqual(record.rounds[0].alignments[17].hsps[0].expect, 0.008)
        self.assertEqual(len(record.rounds[0].alignments[17].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].score, 84)
        self.assertAlmostEqual(record.rounds[0].alignments[18].hsps[0].bits, 37.1)
        self.assertAlmostEqual(record.rounds[0].alignments[18].hsps[0].expect, 0.15)
        self.assertEqual(len(record.rounds[0].alignments[18].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].score, 77)
        self.assertAlmostEqual(record.rounds[0].alignments[19].hsps[0].bits, 34.4)
        self.assertAlmostEqual(record.rounds[0].alignments[19].hsps[0].expect, 0.99)
        self.assertEqual(len(record.rounds[0].alignments[19].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].score, 75)
        self.assertAlmostEqual(record.rounds[0].alignments[20].hsps[0].bits, 33.6)
        self.assertAlmostEqual(record.rounds[0].alignments[20].hsps[0].expect, 1.7)
        self.assertEqual(len(record.rounds[0].alignments[20].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].score, 72)
        self.assertAlmostEqual(record.rounds[0].alignments[21].hsps[0].bits, 32.5)
        self.assertAlmostEqual(record.rounds[0].alignments[21].hsps[0].expect, 3.8)
        self.assertEqual(len(record.rounds[0].alignments[21].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].score, 70)
        self.assertAlmostEqual(record.rounds[0].alignments[22].hsps[0].bits, 31.7)
        self.assertAlmostEqual(record.rounds[0].alignments[22].hsps[0].expect, 6.6)
        self.assertEqual(len(record.rounds[0].alignments[22].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].score, 69)
        self.assertAlmostEqual(record.rounds[0].alignments[23].hsps[0].bits, 31.3)
        self.assertAlmostEqual(record.rounds[0].alignments[23].hsps[0].expect, 8.6)
        self.assertEqual(len(record.rounds[0].alignments[23].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].score, 69)
        self.assertAlmostEqual(record.rounds[0].alignments[24].hsps[0].bits, 31.3)
        self.assertAlmostEqual(record.rounds[0].alignments[24].hsps[0].expect, 8.6)
        self.assertEqual(len(record.rounds[0].alignments[24].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].score, 69)
        self.assertAlmostEqual(record.rounds[0].alignments[25].hsps[0].bits, 31.3)
        self.assertAlmostEqual(record.rounds[0].alignments[25].hsps[0].expect, 8.6)
        self.assertEqual(len(record.rounds[0].alignments[25].hsps), 1)
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].score, 69)
        self.assertAlmostEqual(record.rounds[0].alignments[26].hsps[0].bits, 31.3)
        self.assertAlmostEqual(record.rounds[0].alignments[26].hsps[0].expect, 8.6)
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].score, 3163)
        self.assertAlmostEqual(record.rounds[1].alignments[0].hsps[0].bits, 1236)
        self.assertAlmostEqual(record.rounds[1].alignments[0].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[1].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].score, 1810)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[0].bits, 709)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[0].expect, 0.0)
        self.assertEqual(record.rounds[1].alignments[1].hsps[1].score, 649)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[1].bits, 257)
        self.assertAlmostEqual(record.rounds[1].alignments[1].hsps[1].expect, 8e-68)
        self.assertEqual(len(record.rounds[1].alignments[1].hsps), 2)
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].score, 873)
        self.assertAlmostEqual(record.rounds[1].alignments[2].hsps[0].bits, 344)
        self.assertAlmostEqual(record.rounds[1].alignments[2].hsps[0].expect, 4e-94)
        self.assertEqual(len(record.rounds[1].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].score, 870)
        self.assertAlmostEqual(record.rounds[1].alignments[3].hsps[0].bits, 343)
        self.assertAlmostEqual(record.rounds[1].alignments[3].hsps[0].expect, 1e-93)
        self.assertEqual(len(record.rounds[1].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].score, 870)
        self.assertAlmostEqual(record.rounds[1].alignments[4].hsps[0].bits, 343)
        self.assertAlmostEqual(record.rounds[1].alignments[4].hsps[0].expect, 1e-93)
        self.assertEqual(len(record.rounds[1].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].score, 825)
        self.assertAlmostEqual(record.rounds[1].alignments[5].hsps[0].bits, 325)
        self.assertAlmostEqual(record.rounds[1].alignments[5].hsps[0].expect, 2e-88)
        self.assertEqual(len(record.rounds[1].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].score, 795)
        self.assertAlmostEqual(record.rounds[1].alignments[6].hsps[0].bits, 314)
        self.assertAlmostEqual(record.rounds[1].alignments[6].hsps[0].expect, 6e-85)
        self.assertEqual(len(record.rounds[1].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].score, 765)
        self.assertAlmostEqual(record.rounds[1].alignments[7].hsps[0].bits, 302)
        self.assertAlmostEqual(record.rounds[1].alignments[7].hsps[0].expect, 2e-81)
        self.assertEqual(len(record.rounds[1].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].score, 711)
        self.assertAlmostEqual(record.rounds[1].alignments[8].hsps[0].bits, 281)
        self.assertAlmostEqual(record.rounds[1].alignments[8].hsps[0].expect, 4e-75)
        self.assertEqual(len(record.rounds[1].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].score, 709)
        self.assertAlmostEqual(record.rounds[1].alignments[9].hsps[0].bits, 280)
        self.assertAlmostEqual(record.rounds[1].alignments[9].hsps[0].expect, 8e-75)
        self.assertEqual(len(record.rounds[1].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].score, 687)
        self.assertAlmostEqual(record.rounds[1].alignments[10].hsps[0].bits, 272)
        self.assertAlmostEqual(record.rounds[1].alignments[10].hsps[0].expect, 3e-72)
        self.assertEqual(len(record.rounds[1].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].score, 684)
        self.assertAlmostEqual(record.rounds[1].alignments[11].hsps[0].bits, 270)
        self.assertAlmostEqual(record.rounds[1].alignments[11].hsps[0].expect, 6e-72)
        self.assertEqual(len(record.rounds[1].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].score, 633)
        self.assertAlmostEqual(record.rounds[1].alignments[12].hsps[0].bits, 251)
        self.assertAlmostEqual(record.rounds[1].alignments[12].hsps[0].expect, 6e-66)
        self.assertEqual(len(record.rounds[1].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].score, 566)
        self.assertAlmostEqual(record.rounds[1].alignments[13].hsps[0].bits, 224)
        self.assertAlmostEqual(record.rounds[1].alignments[13].hsps[0].expect, 4e-58)
        self.assertEqual(len(record.rounds[1].alignments[13].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].score, 216)
        self.assertAlmostEqual(record.rounds[1].alignments[14].hsps[0].bits, 88.6)
        self.assertAlmostEqual(record.rounds[1].alignments[14].hsps[0].expect, 5e-17)
        self.assertEqual(len(record.rounds[1].alignments[14].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].score, 215)
        self.assertAlmostEqual(record.rounds[1].alignments[15].hsps[0].bits, 88.2)
        self.assertAlmostEqual(record.rounds[1].alignments[15].hsps[0].expect, 6e-17)
        self.assertEqual(len(record.rounds[1].alignments[15].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].score, 214)
        self.assertAlmostEqual(record.rounds[1].alignments[16].hsps[0].bits, 87.8)
        self.assertAlmostEqual(record.rounds[1].alignments[16].hsps[0].expect, 8e-17)
        self.assertEqual(len(record.rounds[1].alignments[16].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].score, 161)
        self.assertAlmostEqual(record.rounds[1].alignments[17].hsps[0].bits, 67.2)
        self.assertAlmostEqual(record.rounds[1].alignments[17].hsps[0].expect, 1e-10)
        self.assertEqual(len(record.rounds[1].alignments[17].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].score, 102)
        self.assertAlmostEqual(record.rounds[1].alignments[18].hsps[0].bits, 44.2)
        self.assertAlmostEqual(record.rounds[1].alignments[18].hsps[0].expect, 0.001)
        self.assertEqual(len(record.rounds[1].alignments[18].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].score, 99)
        self.assertAlmostEqual(record.rounds[1].alignments[19].hsps[0].bits, 43.0)
        self.assertAlmostEqual(record.rounds[1].alignments[19].hsps[0].expect, 0.003)
        self.assertEqual(len(record.rounds[1].alignments[19].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].score, 98)
        self.assertAlmostEqual(record.rounds[1].alignments[20].hsps[0].bits, 42.6)
        self.assertAlmostEqual(record.rounds[1].alignments[20].hsps[0].expect, 0.003)
        self.assertEqual(len(record.rounds[1].alignments[20].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].score, 94)
        self.assertAlmostEqual(record.rounds[1].alignments[21].hsps[0].bits, 41.1)
        self.assertAlmostEqual(record.rounds[1].alignments[21].hsps[0].expect, 0.010)
        self.assertEqual(len(record.rounds[1].alignments[21].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].score, 74)
        self.assertAlmostEqual(record.rounds[1].alignments[22].hsps[0].bits, 33.3)
        self.assertAlmostEqual(record.rounds[1].alignments[22].hsps[0].expect, 2.2)
        self.assertEqual(len(record.rounds[1].alignments[22].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].score, 73)
        self.assertAlmostEqual(record.rounds[1].alignments[23].hsps[0].bits, 32.9)
        self.assertAlmostEqual(record.rounds[1].alignments[23].hsps[0].expect, 2.9)
        self.assertEqual(len(record.rounds[1].alignments[23].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].score, 69)
        self.assertAlmostEqual(record.rounds[1].alignments[24].hsps[0].bits, 31.3)
        self.assertAlmostEqual(record.rounds[1].alignments[24].hsps[0].expect, 8.4)
        self.assertEqual(len(record.rounds[1].alignments[24].hsps), 1)
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].score, 69)
        self.assertAlmostEqual(record.rounds[1].alignments[25].hsps[0].bits, 31.3)
        self.assertAlmostEqual(record.rounds[1].alignments[25].hsps[0].expect, 8.4)
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].score, 3143)
        self.assertAlmostEqual(record.rounds[2].alignments[0].hsps[0].bits, 1228)
        self.assertAlmostEqual(record.rounds[2].alignments[0].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[2].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].score, 1792)
        self.assertAlmostEqual(record.rounds[2].alignments[1].hsps[0].bits, 702)
        self.assertAlmostEqual(record.rounds[2].alignments[1].hsps[0].expect, 0.0)
        self.assertEqual(record.rounds[2].alignments[1].hsps[1].score, 643)
        self.assertAlmostEqual(record.rounds[2].alignments[1].hsps[1].bits, 254)
        self.assertAlmostEqual(record.rounds[2].alignments[1].hsps[1].expect, 4e-67)
        self.assertEqual(len(record.rounds[2].alignments[1].hsps), 2)
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].score, 864)
        self.assertAlmostEqual(record.rounds[2].alignments[2].hsps[0].bits, 340)
        self.assertAlmostEqual(record.rounds[2].alignments[2].hsps[0].expect, 5e-93)
        self.assertEqual(len(record.rounds[2].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].score, 860)
        self.assertAlmostEqual(record.rounds[2].alignments[3].hsps[0].bits, 339)
        self.assertAlmostEqual(record.rounds[2].alignments[3].hsps[0].expect, 1e-92)
        self.assertEqual(len(record.rounds[2].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].score, 859)
        self.assertAlmostEqual(record.rounds[2].alignments[4].hsps[0].bits, 339)
        self.assertAlmostEqual(record.rounds[2].alignments[4].hsps[0].expect, 2e-92)
        self.assertEqual(len(record.rounds[2].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].score, 836)
        self.assertAlmostEqual(record.rounds[2].alignments[5].hsps[0].bits, 330)
        self.assertAlmostEqual(record.rounds[2].alignments[5].hsps[0].expect, 1e-89)
        self.assertEqual(len(record.rounds[2].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].score, 801)
        self.assertAlmostEqual(record.rounds[2].alignments[6].hsps[0].bits, 316)
        self.assertAlmostEqual(record.rounds[2].alignments[6].hsps[0].expect, 1e-85)
        self.assertEqual(len(record.rounds[2].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].score, 775)
        self.assertAlmostEqual(record.rounds[2].alignments[7].hsps[0].bits, 306)
        self.assertAlmostEqual(record.rounds[2].alignments[7].hsps[0].expect, 1e-82)
        self.assertEqual(len(record.rounds[2].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].score, 733)
        self.assertAlmostEqual(record.rounds[2].alignments[8].hsps[0].bits, 289)
        self.assertAlmostEqual(record.rounds[2].alignments[8].hsps[0].expect, 1e-77)
        self.assertEqual(len(record.rounds[2].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].score, 719)
        self.assertAlmostEqual(record.rounds[2].alignments[9].hsps[0].bits, 284)
        self.assertAlmostEqual(record.rounds[2].alignments[9].hsps[0].expect, 5e-76)
        self.assertEqual(len(record.rounds[2].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].score, 707)
        self.assertAlmostEqual(record.rounds[2].alignments[10].hsps[0].bits, 279)
        self.assertAlmostEqual(record.rounds[2].alignments[10].hsps[0].expect, 1e-74)
        self.assertEqual(len(record.rounds[2].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].score, 694)
        self.assertAlmostEqual(record.rounds[2].alignments[11].hsps[0].bits, 274)
        self.assertAlmostEqual(record.rounds[2].alignments[11].hsps[0].expect, 4e-73)
        self.assertEqual(len(record.rounds[2].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].score, 655)
        self.assertAlmostEqual(record.rounds[2].alignments[12].hsps[0].bits, 259)
        self.assertAlmostEqual(record.rounds[2].alignments[12].hsps[0].expect, 2e-68)
        self.assertEqual(len(record.rounds[2].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].score, 649)
        self.assertAlmostEqual(record.rounds[2].alignments[13].hsps[0].bits, 257)
        self.assertAlmostEqual(record.rounds[2].alignments[13].hsps[0].expect, 8e-68)
        self.assertEqual(len(record.rounds[2].alignments[13].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].score, 215)
        self.assertAlmostEqual(record.rounds[2].alignments[14].hsps[0].bits, 88.2)
        self.assertAlmostEqual(record.rounds[2].alignments[14].hsps[0].expect, 6e-17)
        self.assertEqual(len(record.rounds[2].alignments[14].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].score, 213)
        self.assertAlmostEqual(record.rounds[2].alignments[15].hsps[0].bits, 87.4)
        self.assertAlmostEqual(record.rounds[2].alignments[15].hsps[0].expect, 1e-16)
        self.assertEqual(len(record.rounds[2].alignments[15].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].score, 210)
        self.assertAlmostEqual(record.rounds[2].alignments[16].hsps[0].bits, 86.2)
        self.assertAlmostEqual(record.rounds[2].alignments[16].hsps[0].expect, 2e-16)
        self.assertEqual(len(record.rounds[2].alignments[16].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].score, 209)
        self.assertAlmostEqual(record.rounds[2].alignments[17].hsps[0].bits, 85.9)
        self.assertAlmostEqual(record.rounds[2].alignments[17].hsps[0].expect, 3e-16)
        self.assertEqual(len(record.rounds[2].alignments[17].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].score, 111)
        self.assertAlmostEqual(record.rounds[2].alignments[18].hsps[0].bits, 47.7)
        self.assertAlmostEqual(record.rounds[2].alignments[18].hsps[0].expect, 1e-4)
        self.assertEqual(len(record.rounds[2].alignments[18].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].score, 104)
        self.assertAlmostEqual(record.rounds[2].alignments[19].hsps[0].bits, 45.0)
        self.assertAlmostEqual(record.rounds[2].alignments[19].hsps[0].expect, 7e-4)
        self.assertEqual(len(record.rounds[2].alignments[19].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].score, 101)
        self.assertAlmostEqual(record.rounds[2].alignments[20].hsps[0].bits, 43.8)
        self.assertAlmostEqual(record.rounds[2].alignments[20].hsps[0].expect, 0.001)
        self.assertEqual(len(record.rounds[2].alignments[20].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].score, 96)
        self.assertAlmostEqual(record.rounds[2].alignments[21].hsps[0].bits, 41.8)
        self.assertAlmostEqual(record.rounds[2].alignments[21].hsps[0].expect, 0.006)
        self.assertEqual(len(record.rounds[2].alignments[21].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].score, 73)
        self.assertAlmostEqual(record.rounds[2].alignments[22].hsps[0].bits, 32.9)
        self.assertAlmostEqual(record.rounds[2].alignments[22].hsps[0].expect, 2.9)
        self.assertEqual(len(record.rounds[2].alignments[22].hsps), 1)
        self.assertEqual(record.rounds[2].alignments[23].hsps[0].score, 72)
        self.assertAlmostEqual(record.rounds[2].alignments[23].hsps[0].bits, 32.5)
        self.assertAlmostEqual(record.rounds[2].alignments[23].hsps[0].expect, 3.7)
        self.assertEqual(record.rounds[3].alignments[0].hsps[0].score, 3136)
        self.assertAlmostEqual(record.rounds[3].alignments[0].hsps[0].bits, 1225)
        self.assertAlmostEqual(record.rounds[3].alignments[0].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[3].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[1].hsps[0].score, 1776)
        self.assertAlmostEqual(record.rounds[3].alignments[1].hsps[0].bits, 696)
        self.assertAlmostEqual(record.rounds[3].alignments[1].hsps[0].expect, 0.0)
        self.assertEqual(record.rounds[3].alignments[1].hsps[1].score, 643)
        self.assertAlmostEqual(record.rounds[3].alignments[1].hsps[1].bits, 254)
        self.assertAlmostEqual(record.rounds[3].alignments[1].hsps[1].expect, 4e-67)
        self.assertEqual(len(record.rounds[3].alignments[1].hsps), 2)
        self.assertEqual(record.rounds[3].alignments[2].hsps[0].score, 866)
        self.assertAlmostEqual(record.rounds[3].alignments[2].hsps[0].bits, 341)
        self.assertAlmostEqual(record.rounds[3].alignments[2].hsps[0].expect, 3e-93)
        self.assertEqual(len(record.rounds[3].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[3].hsps[0].score, 862)
        self.assertAlmostEqual(record.rounds[3].alignments[3].hsps[0].bits, 340)
        self.assertAlmostEqual(record.rounds[3].alignments[3].hsps[0].expect, 9e-93)
        self.assertEqual(len(record.rounds[3].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[4].hsps[0].score, 861)
        self.assertAlmostEqual(record.rounds[3].alignments[4].hsps[0].bits, 339)
        self.assertAlmostEqual(record.rounds[3].alignments[4].hsps[0].expect, 1e-92)
        self.assertEqual(len(record.rounds[3].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[5].hsps[0].score, 839)
        self.assertAlmostEqual(record.rounds[3].alignments[5].hsps[0].bits, 331)
        self.assertAlmostEqual(record.rounds[3].alignments[5].hsps[0].expect, 4e-90)
        self.assertEqual(len(record.rounds[3].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[6].hsps[0].score, 802)
        self.assertAlmostEqual(record.rounds[3].alignments[6].hsps[0].bits, 316)
        self.assertAlmostEqual(record.rounds[3].alignments[6].hsps[0].expect, 9e-86)
        self.assertEqual(len(record.rounds[3].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[7].hsps[0].score, 777)
        self.assertAlmostEqual(record.rounds[3].alignments[7].hsps[0].bits, 307)
        self.assertAlmostEqual(record.rounds[3].alignments[7].hsps[0].expect, 8e-83)
        self.assertEqual(len(record.rounds[3].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[8].hsps[0].score, 735)
        self.assertAlmostEqual(record.rounds[3].alignments[8].hsps[0].bits, 290)
        self.assertAlmostEqual(record.rounds[3].alignments[8].hsps[0].expect, 7e-78)
        self.assertEqual(len(record.rounds[3].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[9].hsps[0].score, 720)
        self.assertAlmostEqual(record.rounds[3].alignments[9].hsps[0].bits, 284)
        self.assertAlmostEqual(record.rounds[3].alignments[9].hsps[0].expect, 4e-76)
        self.assertEqual(len(record.rounds[3].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[10].hsps[0].score, 708)
        self.assertAlmostEqual(record.rounds[3].alignments[10].hsps[0].bits, 280)
        self.assertAlmostEqual(record.rounds[3].alignments[10].hsps[0].expect, 1e-74)
        self.assertEqual(len(record.rounds[3].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[11].hsps[0].score, 696)
        self.assertAlmostEqual(record.rounds[3].alignments[11].hsps[0].bits, 275)
        self.assertAlmostEqual(record.rounds[3].alignments[11].hsps[0].expect, 3e-73)
        self.assertEqual(len(record.rounds[3].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[12].hsps[0].score, 656)
        self.assertAlmostEqual(record.rounds[3].alignments[12].hsps[0].bits, 259)
        self.assertAlmostEqual(record.rounds[3].alignments[12].hsps[0].expect, 1e-68)
        self.assertEqual(len(record.rounds[3].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[13].hsps[0].score, 652)
        self.assertAlmostEqual(record.rounds[3].alignments[13].hsps[0].bits, 258)
        self.assertAlmostEqual(record.rounds[3].alignments[13].hsps[0].expect, 4e-68)
        self.assertEqual(len(record.rounds[3].alignments[13].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[14].hsps[0].score, 211)
        self.assertAlmostEqual(record.rounds[3].alignments[14].hsps[0].bits, 86.6)
        self.assertAlmostEqual(record.rounds[3].alignments[14].hsps[0].expect, 2e-16)
        self.assertEqual(len(record.rounds[3].alignments[14].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[15].hsps[0].score, 209)
        self.assertAlmostEqual(record.rounds[3].alignments[15].hsps[0].bits, 85.9)
        self.assertAlmostEqual(record.rounds[3].alignments[15].hsps[0].expect, 3e-16)
        self.assertEqual(len(record.rounds[3].alignments[15].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[16].hsps[0].score, 208)
        self.assertAlmostEqual(record.rounds[3].alignments[16].hsps[0].bits, 85.5)
        self.assertAlmostEqual(record.rounds[3].alignments[16].hsps[0].expect, 4e-16)
        self.assertEqual(len(record.rounds[3].alignments[16].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[17].hsps[0].score, 208)
        self.assertAlmostEqual(record.rounds[3].alignments[17].hsps[0].bits, 85.5)
        self.assertAlmostEqual(record.rounds[3].alignments[17].hsps[0].expect, 4e-16)
        self.assertEqual(len(record.rounds[3].alignments[17].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[18].hsps[0].score, 189)
        self.assertAlmostEqual(record.rounds[3].alignments[18].hsps[0].bits, 78.1)
        self.assertAlmostEqual(record.rounds[3].alignments[18].hsps[0].expect, 7e-14)
        self.assertEqual(len(record.rounds[3].alignments[18].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[19].hsps[0].score, 158)
        self.assertAlmostEqual(record.rounds[3].alignments[19].hsps[0].bits, 66.0)
        self.assertAlmostEqual(record.rounds[3].alignments[19].hsps[0].expect, 3e-10)
        self.assertEqual(len(record.rounds[3].alignments[19].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[20].hsps[0].score, 120)
        self.assertAlmostEqual(record.rounds[3].alignments[20].hsps[0].bits, 51.2)
        self.assertAlmostEqual(record.rounds[3].alignments[20].hsps[0].expect, 9e-6)
        self.assertEqual(len(record.rounds[3].alignments[20].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[21].hsps[0].score, 105)
        self.assertAlmostEqual(record.rounds[3].alignments[21].hsps[0].bits, 45.3)
        self.assertAlmostEqual(record.rounds[3].alignments[21].hsps[0].expect, 5e-4)
        self.assertEqual(len(record.rounds[3].alignments[21].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[22].hsps[0].score, 73)
        self.assertAlmostEqual(record.rounds[3].alignments[22].hsps[0].bits, 32.9)
        self.assertAlmostEqual(record.rounds[3].alignments[22].hsps[0].expect, 2.9)
        self.assertEqual(len(record.rounds[3].alignments[22].hsps), 1)
        self.assertEqual(record.rounds[3].alignments[23].hsps[0].score, 73)
        self.assertAlmostEqual(record.rounds[3].alignments[23].hsps[0].bits, 32.9)
        self.assertAlmostEqual(record.rounds[3].alignments[23].hsps[0].expect, 2.9)
        self.assertEqual(record.rounds[4].alignments[0].hsps[0].score, 3106)
        self.assertAlmostEqual(record.rounds[4].alignments[0].hsps[0].bits, 1214)
        self.assertAlmostEqual(record.rounds[4].alignments[0].hsps[0].expect, 0.0)
        self.assertEqual(len(record.rounds[4].alignments[0].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[1].hsps[0].score, 1758)
        self.assertAlmostEqual(record.rounds[4].alignments[1].hsps[0].bits, 689)
        self.assertAlmostEqual(record.rounds[4].alignments[1].hsps[0].expect, 0.0)
        self.assertEqual(record.rounds[4].alignments[1].hsps[1].score, 644)
        self.assertAlmostEqual(record.rounds[4].alignments[1].hsps[1].bits, 255)
        self.assertAlmostEqual(record.rounds[4].alignments[1].hsps[1].expect, 3e-67)
        self.assertEqual(len(record.rounds[4].alignments[1].hsps), 2)
        self.assertEqual(record.rounds[4].alignments[2].hsps[0].score, 867)
        self.assertAlmostEqual(record.rounds[4].alignments[2].hsps[0].bits, 342)
        self.assertAlmostEqual(record.rounds[4].alignments[2].hsps[0].expect, 2e-93)
        self.assertEqual(len(record.rounds[4].alignments[2].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[3].hsps[0].score, 865)
        self.assertAlmostEqual(record.rounds[4].alignments[3].hsps[0].bits, 341)
        self.assertAlmostEqual(record.rounds[4].alignments[3].hsps[0].expect, 4e-93)
        self.assertEqual(len(record.rounds[4].alignments[3].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[4].hsps[0].score, 864)
        self.assertAlmostEqual(record.rounds[4].alignments[4].hsps[0].bits, 340)
        self.assertAlmostEqual(record.rounds[4].alignments[4].hsps[0].expect, 5e-93)
        self.assertEqual(len(record.rounds[4].alignments[4].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[5].hsps[0].score, 840)
        self.assertAlmostEqual(record.rounds[4].alignments[5].hsps[0].bits, 331)
        self.assertAlmostEqual(record.rounds[4].alignments[5].hsps[0].expect, 3e-90)
        self.assertEqual(len(record.rounds[4].alignments[5].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[6].hsps[0].score, 802)
        self.assertAlmostEqual(record.rounds[4].alignments[6].hsps[0].bits, 316)
        self.assertAlmostEqual(record.rounds[4].alignments[6].hsps[0].expect, 9e-86)
        self.assertEqual(len(record.rounds[4].alignments[6].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[7].hsps[0].score, 777)
        self.assertAlmostEqual(record.rounds[4].alignments[7].hsps[0].bits, 307)
        self.assertAlmostEqual(record.rounds[4].alignments[7].hsps[0].expect, 8e-83)
        self.assertEqual(len(record.rounds[4].alignments[7].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[8].hsps[0].score, 735)
        self.assertAlmostEqual(record.rounds[4].alignments[8].hsps[0].bits, 290)
        self.assertAlmostEqual(record.rounds[4].alignments[8].hsps[0].expect, 7e-78)
        self.assertEqual(len(record.rounds[4].alignments[8].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[9].hsps[0].score, 720)
        self.assertAlmostEqual(record.rounds[4].alignments[9].hsps[0].bits, 284)
        self.assertAlmostEqual(record.rounds[4].alignments[9].hsps[0].expect, 4e-76)
        self.assertEqual(len(record.rounds[4].alignments[9].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[10].hsps[0].score, 709)
        self.assertAlmostEqual(record.rounds[4].alignments[10].hsps[0].bits, 280)
        self.assertAlmostEqual(record.rounds[4].alignments[10].hsps[0].expect, 8e-75)
        self.assertEqual(len(record.rounds[4].alignments[10].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[11].hsps[0].score, 697)
        self.assertAlmostEqual(record.rounds[4].alignments[11].hsps[0].bits, 275)
        self.assertAlmostEqual(record.rounds[4].alignments[11].hsps[0].expect, 2e-73)
        self.assertEqual(len(record.rounds[4].alignments[11].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[12].hsps[0].score, 657)
        self.assertAlmostEqual(record.rounds[4].alignments[12].hsps[0].bits, 260)
        self.assertAlmostEqual(record.rounds[4].alignments[12].hsps[0].expect, 9e-69)
        self.assertEqual(len(record.rounds[4].alignments[12].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[13].hsps[0].score, 651)
        self.assertAlmostEqual(record.rounds[4].alignments[13].hsps[0].bits, 258)
        self.assertAlmostEqual(record.rounds[4].alignments[13].hsps[0].expect, 5e-68)
        self.assertEqual(len(record.rounds[4].alignments[13].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[14].hsps[0].score, 196)
        self.assertAlmostEqual(record.rounds[4].alignments[14].hsps[0].bits, 80.8)
        self.assertAlmostEqual(record.rounds[4].alignments[14].hsps[0].expect, 1e-14)
        self.assertEqual(len(record.rounds[4].alignments[14].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[15].hsps[0].score, 194)
        self.assertAlmostEqual(record.rounds[4].alignments[15].hsps[0].bits, 80.0)
        self.assertAlmostEqual(record.rounds[4].alignments[15].hsps[0].expect, 2e-14)
        self.assertEqual(len(record.rounds[4].alignments[15].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[16].hsps[0].score, 192)
        self.assertAlmostEqual(record.rounds[4].alignments[16].hsps[0].bits, 79.2)
        self.assertAlmostEqual(record.rounds[4].alignments[16].hsps[0].expect, 3e-14)
        self.assertEqual(len(record.rounds[4].alignments[16].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[17].hsps[0].score, 190)
        self.assertAlmostEqual(record.rounds[4].alignments[17].hsps[0].bits, 78.5)
        self.assertAlmostEqual(record.rounds[4].alignments[17].hsps[0].expect, 5e-14)
        self.assertEqual(len(record.rounds[4].alignments[17].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[18].hsps[0].score, 168)
        self.assertAlmostEqual(record.rounds[4].alignments[18].hsps[0].bits, 69.9)
        self.assertAlmostEqual(record.rounds[4].alignments[18].hsps[0].expect, 2e-11)
        self.assertEqual(len(record.rounds[4].alignments[18].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[19].hsps[0].score, 155)
        self.assertAlmostEqual(record.rounds[4].alignments[19].hsps[0].bits, 64.8)
        self.assertAlmostEqual(record.rounds[4].alignments[19].hsps[0].expect, 7e-10)
        self.assertEqual(len(record.rounds[4].alignments[19].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[20].hsps[0].score, 154)
        self.assertAlmostEqual(record.rounds[4].alignments[20].hsps[0].bits, 64.4)
        self.assertAlmostEqual(record.rounds[4].alignments[20].hsps[0].expect, 9e-10)
        self.assertEqual(len(record.rounds[4].alignments[20].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[21].hsps[0].score, 137)
        self.assertAlmostEqual(record.rounds[4].alignments[21].hsps[0].bits, 57.8)
        self.assertAlmostEqual(record.rounds[4].alignments[21].hsps[0].expect, 9e-8)
        self.assertEqual(len(record.rounds[4].alignments[21].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[22].hsps[0].score, 74)
        self.assertAlmostEqual(record.rounds[4].alignments[22].hsps[0].bits, 33.3)
        self.assertAlmostEqual(record.rounds[4].alignments[22].hsps[0].expect, 2.2)
        self.assertEqual(len(record.rounds[4].alignments[22].hsps), 1)
        self.assertEqual(record.rounds[4].alignments[23].hsps[0].score, 73)
        self.assertAlmostEqual(record.rounds[4].alignments[23].hsps[0].bits, 32.9)
        self.assertAlmostEqual(record.rounds[4].alignments[23].hsps[0].expect, 2.9)
        self.assertEqual(len(record.rounds[4].alignments[23].hsps), 1)

    def _check_text_2012L_psiblast_001_hsps_counts(self, record):
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].identities, (765, 889))
        self.assertEqual(record.rounds[0].alignments[0].hsps[0].positives, (765, 889))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].identities, (281, 634))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].positives, (375, 634))
        self.assertEqual(record.rounds[0].alignments[1].hsps[0].gaps, (32, 634))
        self.assertEqual(record.rounds[0].alignments[1].hsps[1].identities, (69, 209))
        self.assertEqual(record.rounds[0].alignments[1].hsps[1].positives, (107, 209))
        self.assertEqual(record.rounds[0].alignments[1].hsps[1].gaps, (26, 209))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].identities, (69, 267))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].positives, (116, 267))
        self.assertEqual(record.rounds[0].alignments[2].hsps[0].gaps, (22, 267))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].identities, (71, 267))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].positives, (122, 267))
        self.assertEqual(record.rounds[0].alignments[3].hsps[0].gaps, (22, 267))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].identities, (72, 268))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].positives, (122, 268))
        self.assertEqual(record.rounds[0].alignments[4].hsps[0].gaps, (23, 268))
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].identities, (73, 270))
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].positives, (120, 270))
        self.assertEqual(record.rounds[0].alignments[5].hsps[0].gaps, (27, 270))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].identities, (65, 250))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].positives, (110, 250))
        self.assertEqual(record.rounds[0].alignments[6].hsps[0].gaps, (19, 250))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].identities, (68, 249))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].positives, (117, 249))
        self.assertEqual(record.rounds[0].alignments[7].hsps[0].gaps, (12, 249))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].identities, (67, 252))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].positives, (125, 252))
        self.assertEqual(record.rounds[0].alignments[8].hsps[0].gaps, (21, 252))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].identities, (67, 243))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].positives, (114, 243))
        self.assertEqual(record.rounds[0].alignments[9].hsps[0].gaps, (12, 243))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].identities, (58, 264))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].positives, (114, 264))
        self.assertEqual(record.rounds[0].alignments[10].hsps[0].gaps, (16, 264))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].identities, (51, 210))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].positives, (92, 210))
        self.assertEqual(record.rounds[0].alignments[11].hsps[0].gaps, (17, 210))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].identities, (60, 255))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].positives, (97, 255))
        self.assertEqual(record.rounds[0].alignments[12].hsps[0].gaps, (21, 255))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].identities, (63, 289))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].positives, (116, 289))
        self.assertEqual(record.rounds[0].alignments[13].hsps[0].gaps, (34, 289))
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].identities, (25, 70))
        self.assertEqual(record.rounds[0].alignments[14].hsps[0].positives, (36, 70))
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].identities, (25, 70))
        self.assertEqual(record.rounds[0].alignments[15].hsps[0].positives, (35, 70))
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].identities, (25, 70))
        self.assertEqual(record.rounds[0].alignments[16].hsps[0].positives, (34, 70))
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].identities, (21, 59))
        self.assertEqual(record.rounds[0].alignments[17].hsps[0].positives, (29, 59))
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].identities, (17, 57))
        self.assertEqual(record.rounds[0].alignments[18].hsps[0].positives, (29, 57))
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].identities, (18, 56))
        self.assertEqual(record.rounds[0].alignments[19].hsps[0].positives, (28, 56))
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].identities, (23, 87))
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].positives, (38, 87))
        self.assertEqual(record.rounds[0].alignments[20].hsps[0].gaps, (1, 87))
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].identities, (18, 61))
        self.assertEqual(record.rounds[0].alignments[21].hsps[0].positives, (26, 61))
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].identities, (29, 99))
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].positives, (40, 99))
        self.assertEqual(record.rounds[0].alignments[22].hsps[0].gaps, (14, 99))
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].identities, (26, 100))
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].positives, (46, 100))
        self.assertEqual(record.rounds[0].alignments[23].hsps[0].gaps, (3, 100))
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].identities, (23, 90))
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].positives, (44, 90))
        self.assertEqual(record.rounds[0].alignments[24].hsps[0].gaps, (10, 90))
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].identities, (22, 74))
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].positives, (39, 74))
        self.assertEqual(record.rounds[0].alignments[25].hsps[0].gaps, (10, 74))
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].identities, (24, 78))
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].positives, (35, 78))
        self.assertEqual(record.rounds[0].alignments[26].hsps[0].gaps, (8, 78))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].identities, (765, 889))
        self.assertEqual(record.rounds[1].alignments[0].hsps[0].positives, (765, 889))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].identities, (281, 635))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].positives, (374, 635))
        self.assertEqual(record.rounds[1].alignments[1].hsps[0].gaps, (34, 635))
        self.assertEqual(record.rounds[1].alignments[1].hsps[1].identities, (69, 209))
        self.assertEqual(record.rounds[1].alignments[1].hsps[1].positives, (107, 209))
        self.assertEqual(record.rounds[1].alignments[1].hsps[1].gaps, (26, 209))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].identities, (72, 268))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].positives, (119, 268))
        self.assertEqual(record.rounds[1].alignments[2].hsps[0].gaps, (23, 268))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].identities, (69, 265))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].positives, (115, 265))
        self.assertEqual(record.rounds[1].alignments[3].hsps[0].gaps, (18, 265))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].identities, (72, 268))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].positives, (117, 268))
        self.assertEqual(record.rounds[1].alignments[4].hsps[0].gaps, (23, 268))
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].identities, (69, 265))
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].positives, (116, 265))
        self.assertEqual(record.rounds[1].alignments[5].hsps[0].gaps, (18, 265))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].identities, (67, 249))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].positives, (116, 249))
        self.assertEqual(record.rounds[1].alignments[6].hsps[0].gaps, (12, 249))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].identities, (67, 249))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].positives, (115, 249))
        self.assertEqual(record.rounds[1].alignments[7].hsps[0].gaps, (13, 249))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].identities, (58, 271))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].positives, (116, 271))
        self.assertEqual(record.rounds[1].alignments[8].hsps[0].gaps, (16, 271))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].identities, (65, 250))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].positives, (109, 250))
        self.assertEqual(record.rounds[1].alignments[9].hsps[0].gaps, (19, 250))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].identities, (58, 286))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].positives, (109, 286))
        self.assertEqual(record.rounds[1].alignments[10].hsps[0].gaps, (28, 286))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].identities, (58, 256))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].positives, (96, 256))
        self.assertEqual(record.rounds[1].alignments[11].hsps[0].gaps, (15, 256))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].identities, (68, 259))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].positives, (126, 259))
        self.assertEqual(record.rounds[1].alignments[12].hsps[0].gaps, (21, 259))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].identities, (60, 277))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].positives, (109, 277))
        self.assertEqual(record.rounds[1].alignments[13].hsps[0].gaps, (29, 277))
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].identities, (25, 70))
        self.assertEqual(record.rounds[1].alignments[14].hsps[0].positives, (35, 70))
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].identities, (25, 70))
        self.assertEqual(record.rounds[1].alignments[15].hsps[0].positives, (34, 70))
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].identities, (25, 70))
        self.assertEqual(record.rounds[1].alignments[16].hsps[0].positives, (36, 70))
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].identities, (22, 72))
        self.assertEqual(record.rounds[1].alignments[17].hsps[0].positives, (30, 72))
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].identities, (18, 62))
        self.assertEqual(record.rounds[1].alignments[18].hsps[0].positives, (26, 62))
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].identities, (18, 63))
        self.assertEqual(record.rounds[1].alignments[19].hsps[0].positives, (28, 63))
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].identities, (17, 60))
        self.assertEqual(record.rounds[1].alignments[20].hsps[0].positives, (29, 60))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].identities, (20, 49))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].positives, (23, 49))
        self.assertEqual(record.rounds[1].alignments[21].hsps[0].gaps, (6, 49))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].identities, (17, 92))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].positives, (25, 92))
        self.assertEqual(record.rounds[1].alignments[22].hsps[0].gaps, (13, 92))
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].identities, (22, 95))
        self.assertEqual(record.rounds[1].alignments[23].hsps[0].positives, (35, 95))
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].identities, (26, 238))
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].positives, (59, 238))
        self.assertEqual(record.rounds[1].alignments[24].hsps[0].gaps, (45, 238))
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].identities, (20, 117))
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].positives, (44, 117))
        self.assertEqual(record.rounds[1].alignments[25].hsps[0].gaps, (18, 117))
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].identities, (765, 889))
        self.assertEqual(record.rounds[2].alignments[0].hsps[0].positives, (765, 889))
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].identities, (281, 635))
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].positives, (374, 635))
        self.assertEqual(record.rounds[2].alignments[1].hsps[0].gaps, (34, 635))
        self.assertEqual(record.rounds[2].alignments[1].hsps[1].identities, (69, 209))
        self.assertEqual(record.rounds[2].alignments[1].hsps[1].positives, (107, 209))
        self.assertEqual(record.rounds[2].alignments[1].hsps[1].gaps, (26, 209))
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].identities, (69, 265))
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].positives, (115, 265))
        self.assertEqual(record.rounds[2].alignments[2].hsps[0].gaps, (18, 265))
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].identities, (72, 268))
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].positives, (119, 268))
        self.assertEqual(record.rounds[2].alignments[3].hsps[0].gaps, (23, 268))
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].identities, (72, 268))
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].positives, (117, 268))
        self.assertEqual(record.rounds[2].alignments[4].hsps[0].gaps, (23, 268))
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].identities, (69, 265))
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].positives, (116, 265))
        self.assertEqual(record.rounds[2].alignments[5].hsps[0].gaps, (18, 265))
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].identities, (67, 249))
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].positives, (116, 249))
        self.assertEqual(record.rounds[2].alignments[6].hsps[0].gaps, (12, 249))
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].identities, (67, 249))
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].positives, (114, 249))
        self.assertEqual(record.rounds[2].alignments[7].hsps[0].gaps, (13, 249))
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].identities, (58, 271))
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].positives, (116, 271))
        self.assertEqual(record.rounds[2].alignments[8].hsps[0].gaps, (16, 271))
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].identities, (58, 286))
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].positives, (109, 286))
        self.assertEqual(record.rounds[2].alignments[9].hsps[0].gaps, (28, 286))
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].identities, (65, 250))
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].positives, (109, 250))
        self.assertEqual(record.rounds[2].alignments[10].hsps[0].gaps, (19, 250))
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].identities, (58, 256))
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].positives, (96, 256))
        self.assertEqual(record.rounds[2].alignments[11].hsps[0].gaps, (15, 256))
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].identities, (60, 277))
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].positives, (109, 277))
        self.assertEqual(record.rounds[2].alignments[12].hsps[0].gaps, (29, 277))
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].identities, (68, 259))
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].positives, (125, 259))
        self.assertEqual(record.rounds[2].alignments[13].hsps[0].gaps, (21, 259))
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].identities, (22, 72))
        self.assertEqual(record.rounds[2].alignments[14].hsps[0].positives, (30, 72))
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].identities, (25, 71))
        self.assertEqual(record.rounds[2].alignments[15].hsps[0].positives, (35, 71))
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].identities, (25, 71))
        self.assertEqual(record.rounds[2].alignments[16].hsps[0].positives, (34, 71))
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].identities, (25, 71))
        self.assertEqual(record.rounds[2].alignments[17].hsps[0].positives, (36, 71))
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].identities, (18, 63))
        self.assertEqual(record.rounds[2].alignments[18].hsps[0].positives, (28, 63))
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].identities, (22, 80))
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].positives, (30, 80))
        self.assertEqual(record.rounds[2].alignments[19].hsps[0].gaps, (6, 80))
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].identities, (17, 62))
        self.assertEqual(record.rounds[2].alignments[20].hsps[0].positives, (29, 62))
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].identities, (20, 49))
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].positives, (23, 49))
        self.assertEqual(record.rounds[2].alignments[21].hsps[0].gaps, (6, 49))
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].identities, (22, 95))
        self.assertEqual(record.rounds[2].alignments[22].hsps[0].positives, (35, 95))
        self.assertEqual(record.rounds[2].alignments[23].hsps[0].identities, (17, 92))
        self.assertEqual(record.rounds[2].alignments[23].hsps[0].positives, (25, 92))
        self.assertEqual(record.rounds[2].alignments[23].hsps[0].gaps, (13, 92))
        self.assertEqual(record.rounds[3].alignments[0].hsps[0].identities, (765, 889))
        self.assertEqual(record.rounds[3].alignments[0].hsps[0].positives, (765, 889))
        self.assertEqual(record.rounds[3].alignments[1].hsps[0].identities, (281, 635))
        self.assertEqual(record.rounds[3].alignments[1].hsps[0].positives, (374, 635))
        self.assertEqual(record.rounds[3].alignments[1].hsps[0].gaps, (34, 635))
        self.assertEqual(record.rounds[3].alignments[1].hsps[1].identities, (69, 209))
        self.assertEqual(record.rounds[3].alignments[1].hsps[1].positives, (107, 209))
        self.assertEqual(record.rounds[3].alignments[1].hsps[1].gaps, (26, 209))
        self.assertEqual(record.rounds[3].alignments[2].hsps[0].identities, (69, 265))
        self.assertEqual(record.rounds[3].alignments[2].hsps[0].positives, (115, 265))
        self.assertEqual(record.rounds[3].alignments[2].hsps[0].gaps, (18, 265))
        self.assertEqual(record.rounds[3].alignments[3].hsps[0].identities, (71, 268))
        self.assertEqual(record.rounds[3].alignments[3].hsps[0].positives, (119, 268))
        self.assertEqual(record.rounds[3].alignments[3].hsps[0].gaps, (23, 268))
        self.assertEqual(record.rounds[3].alignments[4].hsps[0].identities, (71, 268))
        self.assertEqual(record.rounds[3].alignments[4].hsps[0].positives, (117, 268))
        self.assertEqual(record.rounds[3].alignments[4].hsps[0].gaps, (23, 268))
        self.assertEqual(record.rounds[3].alignments[5].hsps[0].identities, (69, 265))
        self.assertEqual(record.rounds[3].alignments[5].hsps[0].positives, (116, 265))
        self.assertEqual(record.rounds[3].alignments[5].hsps[0].gaps, (18, 265))
        self.assertEqual(record.rounds[3].alignments[6].hsps[0].identities, (67, 249))
        self.assertEqual(record.rounds[3].alignments[6].hsps[0].positives, (116, 249))
        self.assertEqual(record.rounds[3].alignments[6].hsps[0].gaps, (12, 249))
        self.assertEqual(record.rounds[3].alignments[7].hsps[0].identities, (67, 249))
        self.assertEqual(record.rounds[3].alignments[7].hsps[0].positives, (114, 249))
        self.assertEqual(record.rounds[3].alignments[7].hsps[0].gaps, (13, 249))
        self.assertEqual(record.rounds[3].alignments[8].hsps[0].identities, (58, 271))
        self.assertEqual(record.rounds[3].alignments[8].hsps[0].positives, (116, 271))
        self.assertEqual(record.rounds[3].alignments[8].hsps[0].gaps, (16, 271))
        self.assertEqual(record.rounds[3].alignments[9].hsps[0].identities, (58, 286))
        self.assertEqual(record.rounds[3].alignments[9].hsps[0].positives, (109, 286))
        self.assertEqual(record.rounds[3].alignments[9].hsps[0].gaps, (28, 286))
        self.assertEqual(record.rounds[3].alignments[10].hsps[0].identities, (65, 250))
        self.assertEqual(record.rounds[3].alignments[10].hsps[0].positives, (109, 250))
        self.assertEqual(record.rounds[3].alignments[10].hsps[0].gaps, (19, 250))
        self.assertEqual(record.rounds[3].alignments[11].hsps[0].identities, (58, 256))
        self.assertEqual(record.rounds[3].alignments[11].hsps[0].positives, (96, 256))
        self.assertEqual(record.rounds[3].alignments[11].hsps[0].gaps, (15, 256))
        self.assertEqual(record.rounds[3].alignments[12].hsps[0].identities, (60, 277))
        self.assertEqual(record.rounds[3].alignments[12].hsps[0].positives, (109, 277))
        self.assertEqual(record.rounds[3].alignments[12].hsps[0].gaps, (29, 277))
        self.assertEqual(record.rounds[3].alignments[13].hsps[0].identities, (68, 259))
        self.assertEqual(record.rounds[3].alignments[13].hsps[0].positives, (125, 259))
        self.assertEqual(record.rounds[3].alignments[13].hsps[0].gaps, (21, 259))
        self.assertEqual(record.rounds[3].alignments[14].hsps[0].identities, (25, 71))
        self.assertEqual(record.rounds[3].alignments[14].hsps[0].positives, (35, 71))
        self.assertEqual(record.rounds[3].alignments[15].hsps[0].identities, (22, 72))
        self.assertEqual(record.rounds[3].alignments[15].hsps[0].positives, (30, 72))
        self.assertEqual(record.rounds[3].alignments[16].hsps[0].identities, (25, 71))
        self.assertEqual(record.rounds[3].alignments[16].hsps[0].positives, (34, 71))
        self.assertEqual(record.rounds[3].alignments[17].hsps[0].identities, (25, 71))
        self.assertEqual(record.rounds[3].alignments[17].hsps[0].positives, (36, 71))
        self.assertEqual(record.rounds[3].alignments[18].hsps[0].identities, (22, 80))
        self.assertEqual(record.rounds[3].alignments[18].hsps[0].positives, (30, 80))
        self.assertEqual(record.rounds[3].alignments[18].hsps[0].gaps, (6, 80))
        self.assertEqual(record.rounds[3].alignments[19].hsps[0].identities, (18, 63))
        self.assertEqual(record.rounds[3].alignments[19].hsps[0].positives, (28, 63))
        self.assertEqual(record.rounds[3].alignments[20].hsps[0].identities, (17, 62))
        self.assertEqual(record.rounds[3].alignments[20].hsps[0].positives, (29, 62))
        self.assertEqual(record.rounds[3].alignments[21].hsps[0].identities, (20, 49))
        self.assertEqual(record.rounds[3].alignments[21].hsps[0].positives, (23, 49))
        self.assertEqual(record.rounds[3].alignments[21].hsps[0].gaps, (6, 49))
        self.assertEqual(record.rounds[3].alignments[22].hsps[0].identities, (17, 92))
        self.assertEqual(record.rounds[3].alignments[22].hsps[0].positives, (25, 92))
        self.assertEqual(record.rounds[3].alignments[22].hsps[0].gaps, (13, 92))
        self.assertEqual(record.rounds[3].alignments[23].hsps[0].identities, (22, 95))
        self.assertEqual(record.rounds[3].alignments[23].hsps[0].positives, (35, 95))
        self.assertEqual(record.rounds[4].alignments[0].hsps[0].identities, (765, 889))
        self.assertEqual(record.rounds[4].alignments[0].hsps[0].positives, (765, 889))
        self.assertEqual(record.rounds[4].alignments[1].hsps[0].identities, (281, 635))
        self.assertEqual(record.rounds[4].alignments[1].hsps[0].positives, (374, 635))
        self.assertEqual(record.rounds[4].alignments[1].hsps[0].gaps, (34, 635))
        self.assertEqual(record.rounds[4].alignments[1].hsps[1].identities, (69, 209))
        self.assertEqual(record.rounds[4].alignments[1].hsps[1].positives, (107, 209))
        self.assertEqual(record.rounds[4].alignments[1].hsps[1].gaps, (26, 209))
        self.assertEqual(record.rounds[4].alignments[2].hsps[0].identities, (69, 265))
        self.assertEqual(record.rounds[4].alignments[2].hsps[0].positives, (115, 265))
        self.assertEqual(record.rounds[4].alignments[2].hsps[0].gaps, (18, 265))
        self.assertEqual(record.rounds[4].alignments[3].hsps[0].identities, (71, 268))
        self.assertEqual(record.rounds[4].alignments[3].hsps[0].positives, (119, 268))
        self.assertEqual(record.rounds[4].alignments[3].hsps[0].gaps, (23, 268))
        self.assertEqual(record.rounds[4].alignments[4].hsps[0].identities, (71, 268))
        self.assertEqual(record.rounds[4].alignments[4].hsps[0].positives, (117, 268))
        self.assertEqual(record.rounds[4].alignments[4].hsps[0].gaps, (23, 268))
        self.assertEqual(record.rounds[4].alignments[5].hsps[0].identities, (69, 265))
        self.assertEqual(record.rounds[4].alignments[5].hsps[0].positives, (116, 265))
        self.assertEqual(record.rounds[4].alignments[5].hsps[0].gaps, (18, 265))
        self.assertEqual(record.rounds[4].alignments[6].hsps[0].identities, (67, 249))
        self.assertEqual(record.rounds[4].alignments[6].hsps[0].positives, (116, 249))
        self.assertEqual(record.rounds[4].alignments[6].hsps[0].gaps, (12, 249))
        self.assertEqual(record.rounds[4].alignments[7].hsps[0].identities, (67, 249))
        self.assertEqual(record.rounds[4].alignments[7].hsps[0].positives, (114, 249))
        self.assertEqual(record.rounds[4].alignments[7].hsps[0].gaps, (13, 249))
        self.assertEqual(record.rounds[4].alignments[8].hsps[0].identities, (58, 271))
        self.assertEqual(record.rounds[4].alignments[8].hsps[0].positives, (116, 271))
        self.assertEqual(record.rounds[4].alignments[8].hsps[0].gaps, (16, 271))
        self.assertEqual(record.rounds[4].alignments[9].hsps[0].identities, (58, 286))
        self.assertEqual(record.rounds[4].alignments[9].hsps[0].positives, (109, 286))
        self.assertEqual(record.rounds[4].alignments[9].hsps[