#ifdef PDB
/****************************************************************************
 * WWW Fish-Search for Mosaic                                               *
 * Information Systems Group                                                *
 * Computing Science Department                                             *
 * Eindhoven University of Technology (TUE)                                 *
 * PO Box 513, NL 5600 MB Eindhoven, The Netherlands                        *
 *                                                                          *
 * Copyright (C) 1993, 1994, Eindhoven University of Technology             *
 *                                                                          *
 * The WWW Fish-Search for Mosaic is subject to the GNU Copyleft license.   *
 * This license only applies to the current version of the search.          *
 * The Eindhoven University of Technology reserves the right to change      *
 * its licensing scheme at any time.                                        *
 *                                                                          *
 * Comments and questions are welcome and can be sent to www@win.tue.nl     *
 *                                                                          *
 * This file contains parts of other files from NCSA Mosaic for X.          *
 * Therefore the NCSA Mosaic copyright notice is included below.            *
 ****************************************************************************/

/****************************************************************************
 * NCSA Mosaic for the X Window System                                      *
 * Software Development Group                                               *
 * National Center for Supercomputing Applications                          *
 * University of Illinois at Urbana-Champaign                               *
 * 605 E. Springfield, Champaign IL 61820                                   *
 * mosaic@ncsa.uiuc.edu                                                     *
 *                                                                          *
 * Copyright (C) 1993, Board of Trustees of the University of Illinois      *
 *                                                                          *
 * NCSA Mosaic software, both binary and source (hereafter, Software) is    *
 * copyrighted by The Board of Trustees of the University of Illinois       *
 * (UI), and ownership remains with the UI.                                 *
 *                                                                          *
 * The UI grants you (hereafter, Licensee) a license to use the Software    *
 * for academic, research and internal business purposes only, without a    *
 * fee.  Licensee may distribute the binary and source code (if released)   *
 * to third parties provided that the copyright notice and this statement   *
 * appears on all copies and that no charge is associated with such         *
 * copies.                                                                  *
 *                                                                          *
 * Licensee may make derivative works.  However, if Licensee distributes    *
 * any derivative work based on or derived from the Software, then          *
 * Licensee will (1) notify NCSA regarding its distribution of the          *
 * derivative work, and (2) clearly notify users that such derivative       *
 * work is a modified version and not the original NCSA Mosaic              *
 * distributed by the UI.                                                   *
 *                                                                          *
 * Any Licensee wishing to make commercial use of the Software should       *
 * contact the UI, c/o NCSA, to negotiate an appropriate license for such   *
 * commercial use.  Commercial use includes (1) integration of all or       *
 * part of the source code into a product for sale or license by or on      *
 * behalf of Licensee to third parties, or (2) distribution of the binary   *
 * code or source code to third parties that need it to utilize a           *
 * commercial product sold or licensed by or on behalf of Licensee.         *
 *                                                                          *
 * UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR   *
 * ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED          *
 * WARRANTY.  THE UI SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY THE    *
 * USERS OF THIS SOFTWARE.                                                  *
 *                                                                          *
 * By using or copying this Software, Licensee agrees to abide by the       *
 * copyright law and all other applicable laws of the U.S. including, but   *
 * not limited to, export control laws, and the terms of this license.      *
 * UI shall have the right to terminate this license immediately by         *
 * written notice upon Licensee's breach of, or non-compliance with, any    *
 * of its terms.  Licensee may be held legally responsible for any          *
 * copyright infringement that is caused or encouraged by Licensee's        *
 * failure to abide by the terms of this license.                           *
 *                                                                          *
 * Comments and questions are welcome and can be sent to                    *
 * mosaic-x@ncsa.uiuc.edu.                                                  *
 ****************************************************************************/

#include "mosaic.h"
#include <time.h>
#include <Xm/List.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/time.h>
#include "libwww2/HTParse.h"
#include "libhtmlw/HTML.h"
#include "libwww2/HText.h"

#include <math.h>
#include "../libregex/regex.h"
#include <signal.h>

extern char *cached_url;

char *normalAppVersion = NULL;
char *fishAppVersion = NULL;
extern char *HTAppVersion;

#include "bitmaps/search.xbm"

/* This file provides support for keyword search in html documents.
   the search is combined with keyword search in index nodes.
   
   Initially there will be a single answerlist, 'Default'.

   The bulk of this code is borrowed from 'hotlist.c'.
*/

static mo_answerlist *default_answerlist = NULL;

/*
 * Given an answerlist and a answernode, append the node
 * to the list.
 * Change fields nodelist and nodelist_last in the answerlist,
 * and fields next and previous in the answernode.
 * Also fill in field position in the answernode.
 * Return nothing.
 */
static void mo_append_answernode_to_answerlist (mo_answerlist *list, 
                                          mo_answernode *node)
{
  if (list->nodelist == NULL)
    {
      /* Nothing yet. */
      list->nodelist = node;
      list->nodelist_last = node;
      node->next = NULL;
      node->previous = NULL;
      node->position = 1;
    }
  else
    {
      /* The new node becomes nodelist_last. */
      /* But first, set up node. */
      node->previous = list->nodelist_last;
      node->next = NULL;
      node->position = node->previous->position + 1;
      
      /* Now point forward from previous nodelist_last. */
      list->nodelist_last->next = node;
      
      /* Now set up new nodelist_last. */
      list->nodelist_last = node;
    }
  
  return;
}

/* --------------- mo_delete_position_from_default_answerlist ---------------- */

/* Given an answerlist and an answernode, rip the node out of the list.
   No check is made as to whether the node is actually in the list;
   it better be. */
static void mo_remove_answernode_from_answerlist (mo_answerlist *list,
                                            mo_answernode *node)
{
  if (node->previous == NULL)
    {
      /* Node was the first member of the list. */
      if (node->next != NULL)
        {
          /* Node was the first member of the list and had
             a next node. */
          /* The next node is now the first node in the list. */
          node->next->previous = NULL;
          list->nodelist = node->next;
        }
      else
        {
          /* Node was the first member of the list and
             didn't have a next node. */
          /* The list is now empty. */
          list->nodelist = NULL;
          list->nodelist_last = NULL;
        }
    }
  else
    {
      /* Node had a previous. */
      if (node->next != NULL)
        {
          /* Node had a previous and a next. */
          node->previous->next = node->next;
          node->next->previous = node->previous;
        }
      else
        {
          /* Node had a previous but no next. */
          node->previous->next = NULL;
          list->nodelist_last = node->previous;
        }
    }
  
  return;
}

/* Go through an answerlist (sequence of answernodes) and assign position
   numbers for all of 'em. */
static void mo_recalculate_answerlist_positions (mo_answerlist *list)
{
  mo_answernode *node;
  int count = 1;
  
  for (node = list->nodelist; node != NULL;
       node = node->next)
    node->position = count++;
  
  return;
}

/*
 * Delete an element of the default answerlist.
 * The element is referenced by its position.
 * Algorithm for removal:
 *   Find answernode with the position.
 *   Remove the answernode from the answerlist data structure.
 *   Recalculate positions of the answerlist.
 *   Remove the element in the position in the list widgets.
 * Return status.
 */
mo_status mo_delete_position_from_default_answerlist (int position)
{
  mo_answerlist *list = default_answerlist;
  mo_answernode *node;
  mo_window *win = NULL;
  
  for (node = list->nodelist; node != NULL;
       node = node->next)
    {
      if (node->position == position)
        goto foundit;
    }
  
  return mo_fail;
  
  /* OK, now we have node loaded. */
 foundit:
  /* Pull the answernode out of the answerlist. */
  mo_remove_answernode_from_answerlist (list, node);
  free (node);
  /* Recalculate positions in this answerlist. */
  mo_recalculate_answerlist_positions (list);
  
  /* Do the GUI stuff. */
  while (win = mo_next_window (win))
    {
      if (win->answerlist_list)
        XmListDeletePos (win->answerlist_list, position);
    }
  
  return mo_succeed;
}

/* --------------- mo_clear_answerlist ---------------- */

static void mo_clear_answerlist (mo_answerlist *list)
{
  /* Do the GUI stuff. */
  mo_window *win = NULL;
  while (win = mo_next_window (win))
    {
      if (win->answerlist_list) {
        mo_answernode *node = list->nodelist;
	while (node != NULL) {
          XmListDeletePos (win->answerlist_list, 1);
	  node = node->next;
	}
      }
    }
  list->nodelist = NULL;
  list->nodelist_last = NULL;
}

void mo_clear_default_answerlist ()
{
  mo_clear_answerlist(default_answerlist);
}

/*
 * Create a new mo_answerlist.
 */
mo_answerlist *mo_new_answerlist (char *title)
{
  mo_answerlist *list;
  
  list = (mo_answerlist *)malloc (sizeof (mo_answerlist));
  list->nodelist = list->nodelist_last = 0;
  return list;
}

/* ------------------------------------------------------------------------ */
/* ----------------------------- HITLIST GUI ------------------------------ */
/* ------------------------------------------------------------------------ */

/* Initial GUI support for answerlist will work like this:

   There will be a single answerlist, called 'Default'.
   It will be persistent across all windows.

*/

/*
 * Called on initialization.  
 * Tries to load the default answerlist.
 */
mo_status mo_setup_default_answerlist (void)
{
  default_answerlist = mo_new_answerlist ("Default");
  
  return mo_succeed;
}

/* ------------------------------------------------------------------------ */
/* ------------------------- gui support routines ------------------------- */
/* ------------------------------------------------------------------------ */

/* We've just init'd a new answerlist list widget; look at the default
   answerlist and load 'er up. */
static void mo_load_answerlist_list (mo_window *win, Widget list)
{
  mo_answernode *node;
  char *s, buf[2048]; /* should be big enough for title */
  
  for (node = default_answerlist->nodelist; node != NULL; node = node->next) {
    sprintf(buf,"%4d : %s",node->relevance,
            (Rdata.display_urls_not_titles || !strncmp (node->url, "gopher:", 7)) ?
                               node->url : node->title);
    s = strchr(buf,'\n');
    if (s)
      *s = '\0'; /* only take first line of title */
    XmListAddItemUnselected 
      (list, 
       XmxMakeXmstrFromString (buf), 
       0);
  }
  return;
}

static void mo_visit_answerlist_position (mo_window *win, int position)
{
  mo_answernode *node;

  /* We should grab the string corresponding to the entry in the
     list and use that as the ref, then call load_window_text
     directly instead of calling mo_access_document. */
  for (node = default_answerlist->nodelist; node != NULL;
       node = node->next)
    {
      if (node->position == position)
        mo_access_document (win, node->url);
    }
  
  return;
}

/* ----------------------------- mail answerlist ----------------------------- */

static XmxCallback (mailanswer_win_cb)
{
  mo_window *win = mo_fetch_window_by_id 
    (XmxExtractUniqid ((int)client_data));
  char *to, *subj, *fnam, *cmd;
  FILE *fp;

  switch (XmxExtractToken ((int)client_data))
    {
    case 0:
      if (!win->setupsearch_win) {
	XmxMakeErrorDialog
	  (win->base,
	   "You must perform a search before\nyou can mail the answer.",
	   "Mail Answerlist Error");
	XtManageChild (Xmx_w);
        return;
      }
      XtUnmanageChild (win->mailanswer_win);

      mo_busy ();
      to = XmxTextGetString (win->mailanswer_to_text);
      if (!to)
        return;
      if (to[0] == '\0')
        return;

      subj = XmxTextGetString (win->mailanswer_subj_text);

      fnam = mo_tmpnam();

      /* Open a file descriptor to sendmail. */
      fp = mo_start_sending_mail_message (to, subj, "text/x-html", NULL);
      if (!fp)
        goto oops;

      {
        mo_answernode *node;
        struct passwd *pw = getpwuid (getuid ());
        char *author;
  
        if (Rdata.default_author_name)
          author = Rdata.default_author_name;
        else
          author = pw->pw_gecos;
        
	fprintf (fp, "<HTML>\n");
        fprintf (fp, "<H1>result of fish search From %s</H1>\n", author);
	fprintf (fp, "keywords: %s\n<p>\n", XmxTextGetString(win->setupsearch_win_text));
        fprintf (fp, "<DL>\n");
        for (node = default_answerlist->nodelist; node != NULL; node = node->next)
          {
            fprintf (fp, "<DT>%d:\t%s\n<DD><A HREF=\"%s\">%s</A>\n", 
                     node->relevance, node->title, node->url, node->url);
          }
        fprintf (fp, "</DL>\n");
	fprintf (fp, "</HTML>\n");
      }

      mo_finish_sending_mail_message ();

    oops:
      free (to);
      free (subj);

      mo_not_busy ();
            
      break;
    case 1:
      XtUnmanageChild (win->mailanswer_win);
      /* Do nothing. */
      break;
    case 2:
      mo_open_another_window
        (win, 
         mo_assemble_fish_help_url ("help-on-answerlist-view.html"), 
         NULL, NULL);
      break;
    }

  return;
}

static mo_status mo_post_mailanswer_win (mo_window *win)
{
  /* This shouldn't happen. */
  if (!win->answer_win)
    return mo_fail;

  if (!win->mailanswer_win)
    {
      Widget dialog_frame;
      Widget dialog_sep, buttons_form;
      Widget mailanswer_form, to_label, subj_label;
      
      /* Create it for the first time. */
      XmxSetUniqid (win->id);
      win->mailanswer_win = XmxMakeFormDialog 
        (win->answer_win, "NCSA Mosaic: Mail answer to keyword search");
      dialog_frame = XmxMakeFrame (win->mailanswer_win, XmxShadowOut);

      /* Constraints for base. */
      XmxSetConstraints 
        (dialog_frame, XmATTACH_FORM, XmATTACH_FORM, 
         XmATTACH_FORM, XmATTACH_FORM, NULL, NULL, NULL, NULL);
      
      /* Main form. */
      mailanswer_form = XmxMakeForm (dialog_frame);
      
      to_label = XmxMakeLabel (mailanswer_form, "Mail To: ");
      XmxSetArg (XmNwidth, 335);
      win->mailanswer_to_text = XmxMakeTextField (mailanswer_form);
      
      subj_label = XmxMakeLabel (mailanswer_form, "Subject: ");
      win->mailanswer_subj_text = XmxMakeTextField (mailanswer_form);

      dialog_sep = XmxMakeHorizontalSeparator (mailanswer_form);
      
      buttons_form = XmxMakeFormAndThreeButtons
        (mailanswer_form, mailanswer_win_cb, "Mail", "Dismiss", "Help...", 0, 1, 2);

      /* Constraints for mailanswer_form. */
      XmxSetOffsets (to_label, 14, 0, 10, 0);
      XmxSetConstraints
        (to_label, XmATTACH_FORM, XmATTACH_NONE, XmATTACH_FORM, XmATTACH_NONE,
         NULL, NULL, NULL, NULL);
      XmxSetOffsets (win->mailanswer_to_text, 10, 0, 5, 10);
      XmxSetConstraints
        (win->mailanswer_to_text, XmATTACH_FORM, XmATTACH_NONE, XmATTACH_WIDGET,
         XmATTACH_FORM, NULL, NULL, to_label, NULL);

      XmxSetOffsets (subj_label, 14, 0, 10, 0);
      XmxSetConstraints
        (subj_label, XmATTACH_WIDGET, XmATTACH_NONE, XmATTACH_FORM, 
         XmATTACH_NONE,
         win->mailanswer_to_text, NULL, NULL, NULL);
      XmxSetOffsets (win->mailanswer_subj_text, 10, 0, 5, 10);
      XmxSetConstraints
        (win->mailanswer_subj_text, XmATTACH_WIDGET, 
         XmATTACH_NONE, XmATTACH_WIDGET,
         XmATTACH_FORM, win->mailanswer_to_text, NULL, subj_label, NULL);

      XmxSetArg (XmNtopOffset, 10);
      XmxSetConstraints 
        (dialog_sep, XmATTACH_WIDGET, XmATTACH_WIDGET, XmATTACH_FORM, 
         XmATTACH_FORM,
         win->mailanswer_subj_text, buttons_form, NULL, NULL);
      XmxSetConstraints 
        (buttons_form, XmATTACH_NONE, XmATTACH_FORM, XmATTACH_FORM, 
         XmATTACH_FORM,
         NULL, NULL, NULL, NULL);
    }
  
  XtManageChild (win->mailanswer_win);
  
  return mo_succeed;
}

/* ---------------------------- answerlist_win_cb ---------------------------- */

static XmxCallback (answerlist_win_cb)
{
  mo_window *win = mo_fetch_window_by_id (XmxExtractUniqid ((int)client_data));

  switch (XmxExtractToken ((int)client_data))
    {
    case 0:
      XtUnmanageChild (win->answer_win);
      /* Dismissed -- do nothing. */
      break;
    case 1:
      mo_post_mailanswer_win (win);
      break;
    case 2:
      mo_open_another_window
        (win, 
         mo_assemble_fish_help_url ("help-on-answerlist-view.html"),
         NULL, NULL);
      break;
    case 3:
      /* configure search parameters. */
      mo_post_fishsearch_win (win);
      break;
    case 4:
      /* Goto selected. */
      {
        Boolean rv;
        int *pos_list;
        int pos_cnt;
        rv = XmListGetSelectedPos (win->answerlist_list, &pos_list, &pos_cnt);
        if (rv && pos_cnt)
          {
            mo_visit_answerlist_position (win, pos_list[0]);
          }
        else
          {
            XmxMakeErrorDialog
              (win->answer_win, "No entry in the answerlist is currently selected.\n\nTo go to an entry in the answerlist,\nselect it with a single mouse click\nand press the Go To button again.", "Error: Nothing Selected");
            XtManageChild (Xmx_w);
          }
      }
      break;
    case 5:
      /* Remove selected. */
      {
        Boolean rv;
        int *pos_list;
        int pos_cnt;
        rv = XmListGetSelectedPos (win->answerlist_list, &pos_list, &pos_cnt);
        if (rv && pos_cnt)
          {
            mo_delete_position_from_default_answerlist (pos_list[0]);
          }
        else
          {
            XmxMakeErrorDialog
              (win->answer_win, "No entry in the answerlist is currently selected.\n\nTo remove an entry in the answerlist,\nselect it with a single mouse click\nand press the Remove button again.", "Error: Nothing Selected");
            XtManageChild (Xmx_w);
          }
      }
      break;
    case 6:
      /* clear list */
      mo_clear_default_answerlist ();
      break;
    case 7:
      /* make answerlist be an annotation to current node */
      /* turn answerlist into personal annotation. user can change
	 that into group or public annotation later */
      /* code borrowed from src/annotate.c */
      if (!win->setupsearch_win && !win->setupsearch_win) {
	XmxMakeErrorDialog
	  (win->base,
	   "You must perform a search before\nyou can turn the answer\ninto an annotation.",
	   "Annotate Answerlist Error");
	XtManageChild (Xmx_w);
        return;
      }
      {
	char *txt; /* to be filled with list */
        struct passwd *pw = getpwuid (getuid ());
        char namestr[500], *author;
        extern char *machine;
        mo_answernode *node;

	if (!win->annotate_win)
	  make_annotate_win (win);

        if (Rdata.default_author_name)
          author = Rdata.default_author_name;
        else
          author = pw->pw_gecos;
      
        sprintf (namestr, "%s (%s@%s)", author, pw->pw_name, machine);
        XmxTextSetString (win->annotate_author, namestr);
      
        sprintf (namestr, "Annotation by %s", author);
        XmxTextSetString (win->annotate_title, namestr);
      
#ifdef GRPAN_PASSWD
        XmxTextSetString (win->annotate_passwd, "\0");
#endif
        XmxTextSetString (win->annotate_text, "\0");

	txt = malloc(500);
	sprintf (txt, "<H2>result of fish search</H2>\n<B>keywords</B>: %s<P>\n<DL>\n",
		XmxTextGetString(win->setupsearch_win_text));
	for (node = default_answerlist->nodelist; node != NULL; node = node->next)
          {
            sprintf (namestr, "<DT>%d:\t%s\n<DD><A HREF=\"%s\">%s</A>\n", 
                     node->relevance, node->title, node->url, node->url);
	    txt = smart_append (txt, namestr);
          }
        sprintf (namestr, "</DL>\n");
	txt = smart_append (txt, namestr);

	mo_new_pan
            (win->current_node->url,
             XmxTextGetString (win->annotate_title),
             XmxTextGetString (win->annotate_author),
             txt);
        /* Inefficient, but safe. */
        mo_write_pan_list ();
        mo_set_win_current_node (win, win->current_node);
	free(txt);
        break;
      }
    }

  return;
}

static XmxCallback (answerlist_list_cb)
{
  mo_window *win = mo_fetch_window_by_id (XmxExtractUniqid ((int)client_data));
  XmListCallbackStruct *cs = (XmListCallbackStruct *)call_data;
  
  mo_visit_answerlist_position (win, cs->item_position);
  
  /* Don't unmanage the list. */
  
  return;
}

/* ------------------------- mo_post_answerist_win -------------------------- */

/*
 * Pop up an answerlist window for an mo_window.
 */
mo_status mo_post_answer_win (mo_window *win)
{
  if (!win->answer_win)
    {
      Dimension oldx, oldy;
      Dimension width, height;
      Widget dialog_frame;
      Widget dialog_sep, buttons_form, buttons1_form;
      Widget answerlist_form, buttons1_frame;
      Widget label, logo;
      XtTranslations listTable;
      static char listTranslations[] =
	"~Shift ~Ctrl ~Meta ~Alt <Btn2Down>: ListBeginSelect() \n\
	 ~Shift ~Ctrl ~Meta ~Alt <Btn2Up>:   ListEndSelect() ListKbdActivate()";

      listTable = XtParseTranslationTable(listTranslations);

      XtVaGetValues (win->base, XmNx, &oldx, XmNy, &oldy, 
                 XmNwidth, &width, XmNheight, &height, NULL);
      XmxSetArg (XmNdefaultPosition, False);
      if (Rdata.auto_place_windows) {
        char geom[20];
        sprintf (geom, "+%d+%d\0", oldx+60, oldy+4);
        XmxSetArg (XmNgeometry, (long)geom);
      }
      /* Create it for the first time. */
      XmxSetUniqid (win->id);
      XmxSetArg (XmNwidth, 475);
      XmxSetArg (XmNheight, 352);

      win->answer_win = XmxMakeFormDialog 
        (win->base, "NCSA Mosaic: Query Result View");
      dialog_frame = XmxMakeFrame (win->answer_win, XmxShadowOut);
      
      /* Constraints for base. */
      XmxSetConstraints 
        (dialog_frame, XmATTACH_FORM, XmATTACH_FORM, 
         XmATTACH_FORM, XmATTACH_FORM, NULL, NULL, NULL, NULL);
      
      /* Main form. */
      answerlist_form = XmxMakeForm (dialog_frame);

      label = XmxMakeLabel (answerlist_form, "Default Query Result:");

      buttons1_form = XmxMakeFormAndFourButtons
        (answerlist_form, answerlist_win_cb, "Setup", "Go To",
         "Remove", "Clear List", 3, 4, 5, 6);

      logo = XmxMakeNamedLabel (answerlist_form, NULL, "logo");
      XmxApplyBitmapToLabelWidget
        (logo, search_bits, search_width, search_height);
      
      /* answerlist list itself. */
      XmxSetArg (XmNresizable, False);
      XmxSetArg (XmNscrollBarDisplayPolicy, XmSTATIC);
      XmxSetArg (XmNlistSizePolicy, XmCONSTANT);
      win->answerlist_list = 
        XmxMakeScrolledList (answerlist_form, answerlist_list_cb, 0);
      XtAugmentTranslations (win->answerlist_list, listTable);

      dialog_sep = XmxMakeHorizontalSeparator (answerlist_form);
      
      buttons_form = XmxMakeFormAndFourButtons
        (answerlist_form, answerlist_win_cb, "Dismiss", "Mail To...", "Annotation", "Help...", 
         0, 1, 7, 2);
      
      /* Constraints for answerlist_form. */
      /* Label: top to nothing, bottom to buttons1_form,
         left to form, right to nothing. */
      XmxSetOffsets (label, 0, 5, 7, 10);
      XmxSetConstraints
        (label, XmATTACH_NONE, XmATTACH_WIDGET, XmATTACH_FORM,
         XmATTACH_NONE, NULL, buttons1_form, NULL, NULL);
      /* buttons1_form: top to nothing, bottom to answerlist_list,
         left to form, right to logo. */
      XmxSetOffsets (buttons1_form, 0, 2, 2, 0);
      XmxSetConstraints
        (buttons1_form, XmATTACH_NONE, XmATTACH_WIDGET, XmATTACH_FORM,
         XmATTACH_WIDGET, 
         NULL, XtParent (win->answerlist_list), NULL, logo);
      /* Logo: top to form, bottom to nothing,
         left to nothing, right to form. */
      XmxSetOffsets (logo, 2, 0, 0, 4);
      XmxSetConstraints
        (logo, XmATTACH_FORM, XmATTACH_NONE, XmATTACH_NONE, XmATTACH_FORM,
         NULL, NULL, NULL, NULL);
      /* list: top to logo, bottom to dialog_sep,
         etc... */
      XmxSetOffsets (XtParent (win->answerlist_list), 10, 10, 10, 10);
      XmxSetConstraints
        (XtParent (win->answerlist_list), 
         XmATTACH_WIDGET, XmATTACH_WIDGET, XmATTACH_FORM, XmATTACH_FORM, 
         logo, dialog_sep, NULL, NULL);
      XmxSetConstraints 
        (dialog_sep, 
         XmATTACH_NONE, XmATTACH_WIDGET, XmATTACH_FORM, XmATTACH_FORM,
         NULL, buttons_form, NULL, NULL);
      XmxSetConstraints 
        (buttons_form, XmATTACH_NONE, XmATTACH_FORM, XmATTACH_FORM, 
         XmATTACH_FORM,
         NULL, NULL, NULL, NULL);
      
      /* Go get the answerlist up to this point set up... */
      mo_load_answerlist_list (win, win->answerlist_list);
    }
  
  XmxManageRemanage (win->answer_win);
  
  return mo_succeed;
}

/* -------------------- mo_post_search_win ------------------------------ */

typedef enum
{
  mo_30seconds = 30,
  mo_1minute = 60,
  mo_2minutes = 120,
  mo_5minutes = 300,
  mo_10minutes = 600,
  mo_20minutes = 1200,
  mo_60minutes = 3600,
  mo_infinite = 1000000
} mo_timeout_token;

static XmxOptionMenuStruct regextype_opts[] = 
{
  { "default",	0,	XmxNotSet },
  { "emacs",	1,	XmxNotSet },
  { "awk",	2,	XmxNotSet },
  { "Posix awk",	3,	XmxNotSet },
  { "grep",	4,	XmxNotSet },
  { "egrep",	5,	XmxNotSet },
  { "Posix egrep",	6,	XmxNotSet },
  { "ed",	7,	XmxNotSet },
  { "sed",	8,	XmxNotSet },
  { "Posix basic",	9,	XmxNotSet },
  { "Posix ext.",	10,	XmxNotSet },
  { NULL },
};

static XmxOptionMenuStruct domain_opts[] = 
{
  { "World",	0,	XmxNotSet },
  { "*.x",	1,	XmxNotSet },
  { "*.y.x",	2,	XmxNotSet },
  { "*.z.y.x",	3,	XmxNotSet },
  { "*.t.z.y.x",	4,	XmxNotSet },
  { "One Host",	5,	XmxNotSet },
  { NULL },
};

static XmxOptionMenuStruct timeout_opts[] =
{
  { "30 sec",	30,	XmxNotSet },
  { "1 min",	60,	XmxNotSet },
  { "2 min",	120,	XmxNotSet },
  { "5 min",	300,	XmxNotSet },
#ifndef STUDENT
  { "10 min",	600,	XmxNotSet },
  { "20 min",	1200,	XmxNotSet },
  { "1 hour",	3600,	XmxNotSet },
  { "unlimited",10000,	XmxNotSet },
#endif
  { NULL },
};

static XmxOptionMenuStruct answers_opts[] =
{
  { "10",	10,	XmxNotSet },
  { "20",	20,	XmxNotSet },
  { "50",	50,	XmxNotSet },
  { "100",	100,	XmxNotSet },
  { "unlimited",10000,	XmxNotSet },
  { NULL },
};

static XmxOptionMenuStruct depth_opts[] =
{
  { "1",	1,	XmxNotSet },
  { "2",	2,	XmxNotSet },
  { "3",	3,	XmxNotSet },
  { "5",	5,	XmxNotSet },
  { "8",	8,	XmxNotSet },
  { "10",	10,	XmxNotSet },
  { "15",	15,	XmxNotSet },
  { "unlimited",100,	XmxNotSet },
  { NULL },
};

static XmxOptionMenuStruct width_opts[] =
{
  { "3",	3,	XmxNotSet },
  { "5",	5,	XmxNotSet },
  { "8",	8,	XmxNotSet },
  { "10",	10,	XmxNotSet },
  { "15",	15,	XmxNotSet },
  { "unlimited",10000,	XmxNotSet },
  { NULL },
};

static XmxCallback (search_win_cb)
{
  mo_window *win = mo_fetch_window_by_id (XmxExtractUniqid ((int)client_data));
  switch (XmxExtractToken ((int)client_data))
    {
    case 0: /* search */
      {
         char *keyword = XmxTextGetString (win->setupsearch_win_text);

	 if (keyword && *keyword)
	   mo_do_query (win, keyword);
	 else {
	    XmxMakeErrorDialog
	    (win->setupsearch_win,
	       "You must enter a keyword before\nyou can perform a search.",
	       "Search Setup Error");
	    XtManageChild (Xmx_w);
	    return;
	  }
         break;
      }
    case 1: /* reset */
      XmxTextSetString (win->setupsearch_win_text, "");
      break;
    case 2: /* dismiss */
      XtUnmanageChild (win->setupsearch_win);
      break;
    case 3: /* help */
      mo_open_another_window
        (win,
         mo_assemble_fish_help_url ("help-on-search-setup.html"),
         NULL, NULL);
      break;
    }
  return;
}

static unsigned syntaxes[] =
{ RE_SYNTAX_EMACS,
  RE_SYNTAX_AWK,
  RE_SYNTAX_POSIX_AWK,
  RE_SYNTAX_GREP,
  RE_SYNTAX_EGREP,
  RE_SYNTAX_POSIX_EGREP,
  RE_SYNTAX_ED,
  RE_SYNTAX_SED,
  RE_SYNTAX_POSIX_BASIC,
  RE_SYNTAX_POSIX_EXTENDED
};

static XmxCallback (regextype_menu_cb)
{
  int i;
  mo_window *win = mo_fetch_window_by_id (XmxExtractUniqid ((int)client_data));

/* kludge because the parameter can only be a small int... */
  win->regex_type = syntaxes[XmxExtractToken ((int)client_data)];

  return;
}

static XmxCallback (domain_menu_cb)
{
  mo_window *win = mo_fetch_window_by_id (XmxExtractUniqid ((int)client_data));
  
  win->domain = XmxExtractToken ((int)client_data);

  return;
}

static XmxCallback (timeout_menu_cb)
{
  mo_window *win = mo_fetch_window_by_id (XmxExtractUniqid ((int)client_data));
  
  win->timeout_val = XmxExtractToken ((int)client_data);

  return;
}

static XmxCallback (answers_menu_cb)
{
  mo_window *win = mo_fetch_window_by_id (XmxExtractUniqid ((int)client_data));
  
  win->max_answers = XmxExtractToken ((int)client_data);

  return;
}

static XmxCallback (depth_menu_cb)
{
  mo_window *win = mo_fetch_window_by_id (XmxExtractUniqid ((int)client_data));
  
  win->max_depth = XmxExtractToken ((int)client_data);

  return;
}

static XmxCallback (width_menu_cb)
{
  mo_window *win = mo_fetch_window_by_id (XmxExtractUniqid ((int)client_data));
  
  win->max_width = XmxExtractToken ((int)client_data);

  return;
}

#define RE_SYNTAX_DEFAULT  ( \
	RE_BACKSLASH_ESCAPE_IN_LISTS | \
	RE_CHAR_CLASSES | \
	RE_DOT_NOT_NULL | \
	RE_HAT_LISTS_NOT_NEWLINE | \
	RE_INTERVALS | \
	RE_NEWLINE_ALT | \
	RE_NO_BK_BRACES | \
	RE_NO_BK_PARENS | \
	RE_NO_BK_VBAR |	\
	RE_NO_EMPTY_RANGES \
	)

mo_status mo_post_fishsearch_win (mo_window *win)
{
  char *keyword;
  if (!win->setupsearch_win)
    {
      Dimension oldx, oldy;
      Dimension width, height;
      Widget dialog_frame;
      Widget dialog_sep, buttons_form;
      Widget search_form, buttons_frame;
      Widget label;
      Widget timeout_label, answers_label, depth_label, width_label;
      Widget regextype_label, domain_label;

      normalAppVersion = HTAppVersion;
      fishAppVersion = malloc(strlen(HTAppVersion)+10);
      sprintf(fishAppVersion,"%s%s",HTAppVersion," (spider)");
      XtVaGetValues (win->base, XmNx, &oldx, XmNy, &oldy, 
                 XmNwidth, &width, XmNheight, &height, NULL);
      XmxSetArg (XmNdefaultPosition, False);
      if (Rdata.auto_place_windows) {
        char geom[20];
        sprintf (geom, "+%d+%d\0", oldx+60, oldy+366);
        XmxSetArg (XmNgeometry, (long)geom);
      }
       /* Create it for the first time. */
      XmxSetUniqid (win->id);
      XmxSetArg (XmNwidth, 500);
      XmxSetArg (XmNheight, 254);
	win->setupsearch_win = XmxMakeFormDialog 
        (win->base, "NCSA Mosaic: Query Setup");
      dialog_frame = XmxMakeFrame (win->setupsearch_win, XmxShadowOut);
      
      /* Constraints for base. */
      XmxSetConstraints 
        (dialog_frame, XmATTACH_FORM, XmATTACH_FORM, 
         XmATTACH_FORM, XmATTACH_FORM, NULL, NULL, NULL, NULL);
      
      /* Main form. */
      search_form = XmxMakeForm (dialog_frame);

      label = XmxMakeLabel (search_form, "Search For:");
      XmxSetArg (XmNcolumns, 25);
      win->setupsearch_win_text = XmxMakeText (search_form);
      XmxAddCallbackToText (win->setupsearch_win_text, search_win_cb, 0);

      win->query_caseless_toggle = XmxMakeToggleButton 
        (search_form, "Caseless Search", NULL, 0);
      if (Rdata.search_caseless)
        XmxSetToggleButton (win->query_caseless_toggle, XmxSet);
      win->query_html_toggle = XmxMakeToggleButton 
        (search_form, "HTML Only", NULL, 0);
      if (Rdata.search_htmlonly)
        XmxSetToggleButton (win->query_html_toggle, XmxSet);
      win->query_andor_toggle = XmxMakeToggleButton 
        (search_form, "Find All Keywords", NULL, 0);
      if (Rdata.search_andor)
        XmxSetToggleButton (win->query_andor_toggle, XmxSet);
      win->query_server_toggle = XmxMakeToggleButton 
        (search_form, "Query Servers", NULL, 0);
      if (Rdata.search_useservers)
        XmxSetToggleButton (win->query_server_toggle, XmxSet);
      win->search_from_hotlist = XmxMakeToggleButton
	(search_form, "Search Hotlist", NULL, 0);
      win->search_from_current = XmxMakeToggleButton
	(search_form, "Search From Current", NULL, 0);
      XmxSetToggleButton (win->search_from_current, XmxSet);

      regextype_label = XmxMakeLabel (search_form, "Regex type:");
      if (caseless_equal(Rdata.search_regextype,"default")) {
	regextype_opts[0].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_DEFAULT;
      } else if (caseless_equal(Rdata.search_regextype, "emacs")) {
	regextype_opts[1].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_EMACS;
      } else if (caseless_equal(Rdata.search_regextype, "awk")) {
	regextype_opts[2].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_AWK;
      } else if (caseless_equal(Rdata.search_regextype, "posix awk")) {
	regextype_opts[3].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_POSIX_AWK;
      } else if (caseless_equal(Rdata.search_regextype, "grep")) {
	regextype_opts[4].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_GREP;
      } else if (caseless_equal(Rdata.search_regextype, "egrep")) {
	regextype_opts[5].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_EGREP;
      } else if (caseless_equal(Rdata.search_regextype, "posix egrep")) {
	regextype_opts[6].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_POSIX_EGREP;
      } else if (caseless_equal(Rdata.search_regextype, "ed")) {
	regextype_opts[7].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_ED;
      } else if (caseless_equal(Rdata.search_regextype, "sed")) {
	regextype_opts[8].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_SED;
      } else if (caseless_equal(Rdata.search_regextype, "posix basic")) {
	regextype_opts[9].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_POSIX_BASIC;
      } else if (caseless_equal(Rdata.search_regextype, "posix ext.")) {
	regextype_opts[10].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_POSIX_EXTENDED;
      } else {
	regextype_opts[2].set_state = XmxSet;
	win->regex_type = RE_SYNTAX_DEFAULT;
      }
      win->regextype_menu = XmxRMakeOptionMenu (search_form, "",
						  regextype_menu_cb,
						  regextype_opts);

      domain_label = XmxMakeLabel (search_form, "Domain:");
      if (Rdata.search_domain==0)
	domain_opts[0].set_state = XmxSet;
      else if (Rdata.search_domain==1)
	domain_opts[1].set_state = XmxSet;
      else if (Rdata.search_domain==2)
	domain_opts[2].set_state = XmxSet;
      else if (Rdata.search_domain==3)
	domain_opts[3].set_state = XmxSet;
      else if (Rdata.search_domain==4)
	domain_opts[4].set_state = XmxSet;
      else
	domain_opts[5].set_state = XmxSet;
      win->domain = Rdata.search_domain;
      win->domain_menu = XmxRMakeOptionMenu (search_form, "",
						domain_menu_cb,
						domain_opts);
      timeout_label = XmxMakeLabel (search_form, "Timeout after:");
      if (Rdata.search_maxtime <= 1)
	timeout_opts[1].set_state = XmxSet;
      else if (Rdata.search_maxtime <= 2)
	timeout_opts[2].set_state = XmxSet;
      else if (Rdata.search_maxtime <= 5)
	timeout_opts[3].set_state = XmxSet;
      else if (Rdata.search_maxtime <= 10)
	timeout_opts[4].set_state = XmxSet;
      else if (Rdata.search_maxtime <= 20)
	timeout_opts[5].set_state = XmxSet;
      else if (Rdata.search_maxtime <= 60)
	timeout_opts[6].set_state = XmxSet;
      else timeout_opts[7].set_state = XmxSet;
      win->timeout_val = Rdata.search_maxtime * 60;
      win->timeout_menu = XmxRMakeOptionMenu (search_form, "",
						timeout_menu_cb,
						timeout_opts);

      answers_label = XmxMakeLabel (search_form, "Max # Answers:");
      if (Rdata.search_maxanswers <= 10)
	answers_opts[0].set_state = XmxSet;
      else if (Rdata.search_maxanswers <= 20)
	answers_opts[1].set_state = XmxSet;
      else if (Rdata.search_maxanswers <= 50)
	answers_opts[2].set_state = XmxSet;
      else if (Rdata.search_maxanswers <= 100)
	answers_opts[3].set_state = XmxSet;
      else answers_opts[4].set_state = XmxSet;
      win->max_answers = Rdata.search_maxanswers;
      win->answers_menu = XmxRMakeOptionMenu (search_form, "",
						answers_menu_cb,
						answers_opts);

      depth_label = XmxMakeLabel (search_form, "Depth of Search:");
      if (Rdata.search_maxdepth <= 1)
	depth_opts[0].set_state = XmxSet;
      else if (Rdata.search_maxdepth <= 2)
	depth_opts[1].set_state = XmxSet;
      else if (Rdata.search_maxdepth <= 3)
	depth_opts[2].set_state = XmxSet;
      else if (Rdata.search_maxdepth <= 5)
	depth_opts[3].set_state = XmxSet;
      else if (Rdata.search_maxdepth <= 8)
	depth_opts[4].set_state = XmxSet;
      else if (Rdata.search_maxdepth <= 10)
	depth_opts[5].set_state = XmxSet;
      else if (Rdata.search_maxdepth <= 15)
	depth_opts[6].set_state = XmxSet;
      else depth_opts[7].set_state = XmxSet;
      win->max_depth = Rdata.search_maxdepth;
      win->depth_menu = XmxRMakeOptionMenu (search_form, "",
						depth_menu_cb,
						depth_opts);

      width_label = XmxMakeLabel (search_form, "Width of Search:");
      if (Rdata.search_maxwidth <= 3)
	width_opts[0].set_state = XmxSet;
      else if (Rdata.search_maxwidth <= 5)
	width_opts[1].set_state = XmxSet;
      else if (Rdata.search_maxwidth <= 8)
	width_opts[2].set_state = XmxSet;
      else if (Rdata.search_maxwidth <= 10)
	width_opts[3].set_state = XmxSet;
      else if (Rdata.search_maxwidth <= 15)
	width_opts[4].set_state = XmxSet;
      else width_opts[5].set_state = XmxSet;
      win->max_width = Rdata.search_maxwidth;
      win->width_menu = XmxRMakeOptionMenu (search_form, "",
						width_menu_cb,
						width_opts);

      dialog_sep = XmxMakeHorizontalSeparator (search_form);
      
      buttons_form = XmxMakeFormAndFourButtons
        (search_form, search_win_cb, "Search", "Reset", "Dismiss", "Help...", 
         0, 1, 2, 3);
      /* Constraints for search_form. */
      XmxSetOffsets (label, 8, 0, 10, 0);
      /* label attaches top to form, bottom to nothing,
         left to form, right to nothing. */
      XmxSetConstraints
        (label, XmATTACH_FORM, XmATTACH_NONE, XmATTACH_FORM, XmATTACH_NONE,
         NULL, NULL, NULL, NULL);
      XmxSetOffsets (win->setupsearch_win_text, 5, 0, 5, 8);
      /* setupsearch_win_text attaches top to form, bottom to nothing,
         left to label, right to form. */
      XmxSetConstraints
        (win->setupsearch_win_text, XmATTACH_FORM, XmATTACH_NONE, XmATTACH_WIDGET,
         XmATTACH_FORM, NULL, NULL, label, NULL);

      /* query_caseless_toggle attaches top to setupsearch_win_text, bottom to nothing,
         left to position, right to position. */
      XmxSetConstraints
        (win->query_caseless_toggle, XmATTACH_WIDGET, XmATTACH_NONE,
         XmATTACH_FORM, XmATTACH_NONE, 
         label, NULL, NULL, NULL);
      XmxSetOffsets (win->query_caseless_toggle, 10, 0, 10, 0);
      /* query_html_toggle attaches top to setupsearch_win_text, bottom to nothing,
         left to position, right to position. */
      XmxSetConstraints
        (win->query_html_toggle, XmATTACH_WIDGET, XmATTACH_NONE,
         XmATTACH_FORM, XmATTACH_NONE, 
         win->query_caseless_toggle, NULL, NULL, NULL);
      XmxSetOffsets (win->query_html_toggle, 7, 0, 10, 0);
      XmxSetConstraints
        (win->query_andor_toggle, XmATTACH_WIDGET, XmATTACH_NONE,
         XmATTACH_WIDGET, XmATTACH_NONE, 
         label, NULL, label, NULL);
      XmxSetOffsets (win->query_andor_toggle, 10, 0, 75, 0);
      XmxSetConstraints
        (win->query_server_toggle, XmATTACH_WIDGET, XmATTACH_NONE,
         XmATTACH_WIDGET, XmATTACH_NONE, 
         win->query_andor_toggle, NULL, label, NULL);
      XmxSetOffsets (win->query_server_toggle, 7, 0, 75, 0);
      XmxSetConstraints
	(win->search_from_current, XmATTACH_WIDGET, XmATTACH_NONE,
         XmATTACH_WIDGET, XmATTACH_NONE, 
         label, NULL, label, NULL);
      XmxSetOffsets (win->search_from_current, 10, 0, 215, 0);
      XmxSetConstraints
	(win->search_from_hotlist, XmATTACH_WIDGET, XmATTACH_NONE,
         XmATTACH_WIDGET, XmATTACH_NONE, 
         win->search_from_current, NULL, label, NULL);
      XmxSetOffsets (win->search_from_hotlist, 7, 0, 215, 0);
      XmxSetConstraints
          (regextype_label, XmATTACH_WIDGET, XmATTACH_NONE, XmATTACH_FORM,
           XmATTACH_NONE, win->query_html_toggle, NULL, NULL, NULL);
      XmxSetOffsets (regextype_label, 14, 0, 10, 0);
      XmxSetConstraints
          (win->regextype_menu->base, XmATTACH_WIDGET, XmATTACH_NONE, 
           XmATTACH_WIDGET,
           XmATTACH_NONE, win->query_html_toggle, NULL, label, NULL);
      XmxSetOffsets (win->regextype_menu->base, 7, 0, 9, 0);
      XmxSetConstraints
          (timeout_label, XmATTACH_WIDGET, XmATTACH_NONE, XmATTACH_FORM,
           XmATTACH_NONE, regextype_label, NULL, NULL, NULL);
      XmxSetOffsets (timeout_label, 17, 0, 10, 0);
      XmxSetConstraints
          (win->timeout_menu->base, XmATTACH_WIDGET, XmATTACH_NONE, 
           XmATTACH_WIDGET,
           XmATTACH_NONE, regextype_label, NULL, label, NULL);
      XmxSetOffsets (win->timeout_menu->base, 10, 0, 9, 0);
      XmxSetConstraints
          (answers_label, XmATTACH_WIDGET, XmATTACH_NONE, XmATTACH_FORM,
           XmATTACH_NONE, timeout_label, NULL, NULL, NULL);
      XmxSetOffsets (answers_label, 17, 0, 10, 0);
      XmxSetConstraints
          (win->answers_menu->base, XmATTACH_WIDGET, XmATTACH_NONE, 
           XmATTACH_WIDGET,
           XmATTACH_NONE, timeout_label, NULL, label, NULL);
      XmxSetOffsets (win->answers_menu->base, 10, 0, 9, 0);
      XmxSetConstraints
          (domain_label, XmATTACH_WIDGET, XmATTACH_NONE, XmATTACH_WIDGET,
           XmATTACH_NONE, win->query_server_toggle, NULL, label, NULL);
      XmxSetOffsets (domain_label, 14, 0, 154, 0);
      XmxSetConstraints
          (win->domain_menu->base, XmATTACH_WIDGET, XmATTACH_NONE, 
           XmATTACH_WIDGET,
           XmATTACH_NONE, win->query_server_toggle, NULL, label, NULL);
      XmxSetOffsets (win->domain_menu->base, 7, 0, 256, 0);
      XmxSetConstraints
          (depth_label, XmATTACH_WIDGET, XmATTACH_NONE, XmATTACH_WIDGET,
           XmATTACH_NONE, domain_label, NULL, label, NULL);
      XmxSetOffsets (depth_label, 17, 0, 154, 0);
      XmxSetConstraints
          (win->depth_menu->base, XmATTACH_WIDGET, XmATTACH_NONE, 
           XmATTACH_WIDGET,
           XmATTACH_NONE, domain_label, NULL, label, NULL);
      XmxSetOffsets (win->depth_menu->base, 10, 0, 256, 0);
      XmxSetConstraints
          (width_label, XmATTACH_WIDGET, XmATTACH_NONE, XmATTACH_WIDGET,
           XmATTACH_NONE, depth_label, NULL, label, NULL);
      XmxSetOffsets (width_label, 17, 0, 154, 0);
      XmxSetConstraints
          (win->width_menu->base, XmATTACH_WIDGET, XmATTACH_NONE, 
           XmATTACH_WIDGET,
           XmATTACH_NONE, depth_label, NULL, label, NULL);
      XmxSetOffsets (win->width_menu->base, 10, 0, 256, 0);
      XmxSetConstraints 
        (dialog_sep, XmATTACH_WIDGET, XmATTACH_WIDGET, XmATTACH_FORM, 
         XmATTACH_FORM,
         width_label, buttons_form, NULL, NULL);
      XmxSetOffsets (dialog_sep, 4, 0, 0, 0);
      /* dialog_sep attaches top to search_titles_toggle,
         bottom to buttons_form, left to form, right to form */
      XmxSetConstraints 
        (buttons_form, XmATTACH_NONE, XmATTACH_FORM, XmATTACH_FORM, 
         XmATTACH_FORM,
         NULL, NULL, NULL, NULL);
    }
  XmxManageRemanage (win->setupsearch_win);

  return mo_succeed;
}

/* -------------------- mo_add_node_to_default_answerlist -------------------- */

mo_status mo_add_node_to_answerlist (mo_answerlist *list, mo_node *node, int relevance)
{
  mo_answernode *answernode = (mo_answernode *)malloc (sizeof (mo_answernode));
  mo_window *win = NULL;
  char *s, buf[2048];
  
  answernode->relevance = relevance;
  s = strchr(node->title,'\n');
  if (s)
    *s = '\0'; /* only take first line of title */
  if (node->title)
    answernode->title = strdup (node->title);
  else
    answernode->title = strdup ("Unnamed");
  mo_convert_newlines_to_spaces (node->title);

  answernode->url = strdup (node->url);
  mo_convert_newlines_to_spaces (answernode->url);
  mo_append_answernode_to_answerlist (list, answernode);
  
  /* Now we've got to update all active answerlist_list's. */
  while (win = mo_next_window (win))
    {
      if (win->answerlist_list)
        {
	  sprintf(buf,"%4d : %s",answernode->relevance,
                  (Rdata.display_urls_not_titles || !strncmp (answernode->url, "gopher:", 7)) ? 
                                     answernode->url : answernode->title);
          XmListAddItemUnselected 
            (win->answerlist_list, 
             XmxMakeXmstrFromString (buf), 
             node->position);
          XmListSetBottomPos (win->answerlist_list, 0);
        }
    }
  
  return mo_succeed;
}

mo_status mo_add_node_to_default_answerlist (mo_node *node, int relevance)
{
  return mo_add_node_to_answerlist (default_answerlist, node, relevance);
}

mo_status mo_already_in_list(char *url)
{
  mo_answernode *node = default_answerlist->nodelist;
  while (node != NULL) {
    if (!strcmp(node->url, url))
      return mo_succeed;
    node = node->next;
  }
  return mo_fail;
}

/* -------------------- hashing url's ----------------------------- */
/* the hash code here is somewhat borrowed from globalhist.c */

typedef struct entry
{
  /* Canonical URL for this document. */
  char *url;
  /* pointer to next entry */
  struct entry *next;
} entry;

#define HASH_TABLE_SIZE 200

static entry *hash_table[HASH_TABLE_SIZE];
/* how nice C guarantees this is filled with 0's */

/* Given a URL, hash it and return the hash value, mod'd by the size
   of the hash table. */
static int hash_url (char *url)
{
  int len, i, val;

  if (!url)
    return 0;
  len = strlen (url);
  val = 0;
  for (i = 0; i < 10; i++)
    val += url[(i * val + 7) % len];

  return val % HASH_TABLE_SIZE;
}

static void clear_hash_table(void)
{
  int buck;
  entry *ptr, *newptr;
  for (buck = 0; buck < HASH_TABLE_SIZE; ++buck) {
    for (ptr = hash_table[buck]; ptr != NULL; ptr = newptr) {
      free(ptr->url);
      newptr = ptr->next;
      free(ptr);
    }
    hash_table[buck] = NULL;
  }
}

static void add_url_to_hashlist (char *url)
{
  int hash = hash_url (url);
  entry *l = (entry *)malloc (sizeof(entry));
  l->url = url; /* assume we get copies of url's, only needed until
                   we clear the hash table */
  l->next = hash_table[hash];
  hash_table[hash] = l;
}

/* check whether we searched this url before */
static Boolean searched_here_before (char *url)
{
  entry *l;

  for (l = hash_table[hash_url(url)]; l!= NULL; l = l->next)
    if (!strcmp (l->url, url))
      return True;
  return False;
}


/*
 * To use faster TOLOWER as set up in HTMLparse.c
 */
#ifndef TOLOWER
#ifdef NOT_ASCII
#define TOLOWER(x)      (tolower(x))
#else
extern char map_table[];
#define TOLOWER(x)      (map_table[x])
#endif /* NOT_ASCII */
#endif /* TOLOWER */

/* ---------------------- determine relevance -------------------- */

/* three types of search exist for now (more are planned) */

#define MULTIKEY_SEARCH 0
#define EXTERNAL_SEARCH 1
#define REGEXP_SEARCH 2

/* for external and regexp search, whitespace is changed into a single space, */
/* and tags are replaced with a special character */
/* that is assumed not to match */

#define UNMATCHED_CHAR '\n'

/* now the external and regular expression searching routines */

/*
 * If the search string starts with !, the rest is interpreted as an
 * external command; this command must be a filter, taking the text
 * on stdin, and returning on stdout a single number. The command is called in
 * a shell; if this procedure should fail, or the command fails to
 * return a single number, currently we will simply return 0 relevance.
 * The boolean caseless is simply ignored for external searches.
 * Neither can we be sophisticated about simultaneous matching of
 * multiple keywords (maybe in future versions where more than one number
 * can be returned). Assume that the number returned is the number of
 * hits, at least when the search is a single keyword search.
 * (We ought to prepare an HTML page explaining the basic assumptions behind
 * the relevance algorithm so people can provide well-motivated external
 * search return values for complicated searches.)
 */

static int on_broken_pipe()
{
  /* this will happen if the external command is malformed and	*/
  /* terminates prematurely					*/
  return 2;
}

int external_search (char *text, char *cmd)
/* negative if search is very unlikely to find anything on any docum.	*/
/* (ideally, iff, but we can't predict the behavior of all commands)	*/
/* (i.e. if command not found, syntax error, or something similar)	*/
/* this is to be used to abort the search after the first document	*/
{
  int rc;
  FILE *stream;
  int hits;  /* holds external search result */
  void (*prev_sigpipehandler)();
  /* append a redirection to a tmp file */
  char *tmpnm = tmpnam(NULL);
  char *pipecmd = (char *)malloc(strlen(cmd)+strlen(tmpnm)+4);
  strcpy(pipecmd,cmd);
  strcpy(&pipecmd[strlen(cmd)]," > ");
  strcpy(&pipecmd[strlen(cmd)+3],tmpnm);
  /* open a pipe to the command specified as argument */
  /* do not crash if command should fail */
  prev_sigpipehandler = signal(SIGPIPE,on_broken_pipe);
  if ((stream = popen(pipecmd,"w")) == (FILE *)NULL)
  {
    fprintf(stderr,"Mosaic error: cannot open pipe to command '%s'\n",cmd);
    free(pipecmd);
    signal(SIGPIPE,prev_sigpipehandler);
    return(-1);
  }
  /* write all text to the stream */
  fwrite(text, sizeof(char), strlen(text), stream);
  rc = pclose(stream);
  if (rc == -1)
  {
    fprintf(stderr,"Mosaic error: cannot close pipe to command '%s', rc=%d\n",cmd,rc);
    hits = -1;
  }
  else if (rc % 256)
  {
    fprintf(stderr,"Mosaic error: cannot close pipe to command '%s', rc=%d\n",cmd,rc%256);
    hits = -1;
  }
  else
  {
    if ((rc = rc / 256))  /* command return code */
    {
      /* fprintf(stderr,"command '%s' returned error code %d\n",cmd,rc); */
      /* grep returns 1 if no matches were found, 2 if expression syntax
         was incorrect; we do not (yet?) require all programs to follow grep */
      /* simply ignore this return code */
    }
    /* retrieve the result from the tmp file */
    if (!(stream = fopen(tmpnm,"r")))
    {
      fprintf(stderr,"Mosaic error: cannot open command output\n");
      hits = -1;
    }
    /* read a number */
    else if (!fscanf(stream," %d ",&hits))
    /* no number was returned */
    {
      fprintf(stderr,"Mosaic error: no number could be extracted from the result of '%s'\n",
              cmd);
      hits = -1;
    }
    else if (!feof(stream))
    {
      fprintf(stderr,"Mosaic error: garbage in the result of '%s'\n",
              cmd);
      hits = -1;
    }
    else if (ferror(stream))
    {
      fprintf(stderr,"Mosaic error: problem occurred reading the result of '%s'\n", cmd);
      hits = -1;
    }
    else if (hits < -1)
    {
      fprintf(stderr,"Mosaic error: '%s' returns a negative value, %d\n", cmd, hits);
      hits = -1;
    }
    fclose(stream);  /* so what if the fopen failed */
  }
  unlink(tmpnm);
  free(pipecmd);
  signal(SIGPIPE,prev_sigpipehandler);
  return(hits);
}

#ifdef NOT_ASCII
/* no idea how to do case conversion if not ascii */
unsigned char case_fold[] = {
 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10,
 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
 21, 22, 23, 24, 25, 26, 27, 28, 29, 30,
 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
 41, 42, 43, 44, 45, 46, 47, 48, 49, 50,
 51, 52, 53, 54, 55, 56, 57, 58, 59, 60,
 61, 62, 63, 64, 65, 66, 67, 68, 69, 70,
 71, 72, 73, 74, 75, 76, 77, 78, 79, 80,
 81, 82, 83, 84, 85, 86, 87, 88, 89, 90,
 91, 92, 93, 94, 95, 96, 97, 98, 99, 100,
 101, 102, 103, 104, 105, 106, 107, 108, 109, 110,
 111, 112, 113, 114, 115, 116, 117, 118, 119, 120,
 121, 122, 123, 124, 125, 126, 127, 128, 129, 130,
 131, 132, 133, 134, 135, 136, 137, 138, 139, 140,
 141, 142, 143, 144, 145, 146, 147, 148, 149, 150,
 151, 152, 153, 154, 155, 156, 157, 158, 159, 160,
 161, 162, 163, 164, 165, 166, 167, 168, 169, 170,
 171, 172, 173, 174, 175, 176, 177, 178, 179, 180,
 181, 182, 183, 184, 185, 186, 187, 188, 189, 190,
 191, 192, 193, 194, 195, 196, 197, 198, 199, 200,
 201, 202, 203, 204, 205, 206, 207, 208, 209, 210,
 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 
 221, 222, 223, 224, 225, 226, 227, 228, 229, 230,
 231, 232, 233, 234, 235, 236, 237, 238, 239, 240,
 241, 242, 243, 244, 245, 246, 247, 248, 249, 250,
 251, 252, 253, 254, 255};

#else

unsigned char case_fold[] = {
 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10,
 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
 21, 22, 23, 24, 25, 26, 27, 28, 29, 30,
 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
 41, 42, 43, 44, 45, 46, 47, 48, 49, 50,
 51, 52, 53, 54, 55, 56, 57, 58, 59, 60,
 61, 62, 63, 64, 65, 66, 67, 68, 69, 70,
 71, 72, 73, 74, 75, 76, 77, 78, 79, 80,
 81, 82, 83, 84, 85, 86, 87, 88, 89, 90,
 91, 92, 93, 94, 95, 96, 65, 66, 67, 68,
 69, 70, 71, 72, 73, 74, 75, 76, 77, 78,
 79, 80, 81, 82, 83, 84, 85, 86, 87, 88,
 89,  90,  123, 124, 125, 126, 127, 128, 129, 130,
 131, 132, 133, 134, 135, 136, 137, 138, 139, 140,
 141, 142, 143, 144, 145, 146, 147, 148, 149, 150,
 151, 152, 153, 154, 155, 156, 157, 158, 159, 160,
 161, 162, 163, 164, 165, 166, 167, 168, 169, 170,
 171, 172, 173, 174, 175, 176, 177, 178, 179, 180,
 181, 182, 183, 184, 185, 186, 187, 188, 189, 190,
 191, 192, 193, 194, 195, 196, 197, 198, 199, 200,
 201, 202, 203, 204, 205, 206, 207, 208, 209, 210,
 211, 212, 213, 214, 215, 216, 217, 218, 219, 220,
 221, 222, 223, 224, 225, 226, 227, 228, 229, 230,
 231, 232, 233, 234, 235, 236, 237, 238, 239, 240,
 241, 242, 243, 244, 245, 246, 247, 248, 249, 250,
 251, 252, 253, 254, 255};

#endif

#define FASTMAPSIZE 256
/* I expect this to be enough even for non-ASCII ... */
char fastmapbuf[FASTMAPSIZE];

int regex_search (char *text, const char *expr, Boolean caseless)
{
  int textlen = strlen(text);
  int exprlen = strlen(expr);
  register int textoffset = 0;
  const char *comperr;
  int rc;
  int hits = 0;
  int matchlen;
  struct re_pattern_buffer *pat =
    (struct re_pattern_buffer *)malloc(sizeof(struct re_pattern_buffer));
  struct re_registers *regs =
    (struct re_registers *)malloc(sizeof(struct re_registers));
  bzero(pat,sizeof(struct re_pattern_buffer));
  bzero(regs,sizeof(struct re_registers));

/* following determines the syntax for the regular expression search    */
/* at the momen this is not configurable while Mosaic is running.       */

/* see regex.h for more details						*/  
/*	RE_BACKSLASH_ESCAPE_IN_LISTS |	/* [\a] is [a]			*/
/*	! RE_BK_PLUS_QM |		/* * is closure, \* literal	*/
/*	RE_CHAR_CLASSES |		/* [:alpha:] etc.		*/
/*	! RE_CONTEXT_INDEP_ANCHORS |	/* $a^ matches '$a^'		*/
/*	! RE_CONTEXT_INDEP_OPS |	/* e.g. initial * is literal	*/
/*	! RE_CONTEXT_INVALID_OPS |	/* initial * is valid		*/
/*	! RE_DOT_NEWLINE |		/* . doesn't match newline	*/
/*	RE_DOT_NOT_NULL |		/* . doesn't match \0		*/
/*	RE_HAT_LISTS_NOT_NEWLINE |	/* [^.] doesn't match newline	*/
/*	RE_INTERVALS |			/* intervals ({,}) exist	*/
/*	! RE_LIMITED_OPS |		/* +, |, ? exist		*/
/*	RE_NEWLINE_ALT |		/* newline is not a literal	*/
/*	RE_NO_BK_BRACES |		/* {} is interval, \{ \} literal*/
/*	RE_NO_BK_PARENS |		/* () is group, \( \) literal	*/
/*	! RE_NO_BK_REFS |		/* \digit is a back reference	*/
/*	RE_NO_BK_VBAR |			/* | is alternation, \| literal */
/*	RE_NO_EMPTY_RANGES |		/* [z-a] is invalid		*/
/*	! RE_UNMATCHED_RIGHT_PAREN_ORD	/* unmatched ) is invalid	*/

  if (caseless)
  {
    pat->translate = strdup(case_fold);
  }
  else
  {
    pat->translate = (char *)NULL;
  }
  pat->fastmap = fastmapbuf;
  pat->buffer = NULL;  /* leave allocation to routine */
  pat->allocated = REGS_UNALLOCATED;

  comperr = re_compile_pattern(expr,exprlen,pat);
  if (comperr)
  {
    fprintf(stderr,"Mosaic error: compilation of pattern '%s' failed: %s\n",expr,comperr);
    return -1;
  }
  /* register allocation is dealt with automatically, says the manual */
  while ((rc = re_search(pat,text,textlen,textoffset,textlen-textoffset,
	regs))>=0)
  {
    matchlen = regs->end[0] - regs->start[0];
    if (!matchlen) {
      matchlen = 1;  /* if pattern was matched, but match has zero length,
			do move forward ! */
    }
    if (matchlen > 100 || matchlen < 0)
    {
       fprintf(stderr,"Mosaic warning: found a match of %d characters.\n",matchlen);
    }
    ++hits;
    /* textoffset += (rc?rc:1);  /* this is too expensive for closures! */
    textoffset = rc + matchlen;
  }
  if (rc < -1) { /* not -1, that is */
    fprintf(stderr,"Mosaic error: internal error in regexp search, rc = %d\n",rc);
    hits = -1;
  }
  regfree(pat);
  return hits;
}

extern char *ParseMarkTag(char *, char *, char *);
extern struct mark_up *get_mark(char *, char **);
#define MAX_REFS 1000
int mo_evaluate (char *text, char *keywords, Boolean caseless, int andor_type, char ***phrefs, int *pnum_hrefs, char **title, Boolean *isindex)
{
  int relevance, textlen;
  struct keyword {
    char word[256];
    int len;
    int matches;
  } keywordlist[12];
  /* max 12 keywords should be enough */
  struct mark_up *mark = (struct mark_up *) 0;
  char *hrefs[MAX_REFS], **list;
  int num_hrefs = 0;
  int nb = 0, i, j = 0;
  register char c;
  char *s = keywords, *l = keywordlist[0].word;

  int search_type = MULTIKEY_SEARCH;
  char *untagged_text = (char *)malloc(sizeof(char)*(strlen(text)+1));
  /* wouldn't it be easy if we could just get he length of the text... */
  char *copy = untagged_text;
  /* will become copy with altered white space */

  *isindex = False;
  *title = NULL;
  *pnum_hrefs = 0;
  /* if no keywords given, fail the search */
  if (s == NULL || *s == '\0')
    return 0;
  if (!strncmp(text,"<H1>ERROR</H1>",14))
    return 0; /* nothing wrong with the search per se, but the error
		 node is not considered relevant */
  /* skip leading whitespace in keywords */
  while ((c = *s) == ' ' || c == '\t')
    ++s;
  /* if keywords start with !, then interpret the rest as an external
     search command; if with /, use the regex library to match a reg. expr. */
  if (c == '!')
  {
    search_type = EXTERNAL_SEARCH;
    strncpy(l,++s,255);
    l[255] = '\0';  /* in case it has length 255 */
    keywordlist[0].len = strlen(l); /* but not used */
    if (caseless)
    {
      for (s = l; (c = *s) != '\0'; ++s) *s = TOLOWER(c);
    }
  }
  else if (c == '/')
  {
    search_type = REGEXP_SEARCH;
    strncpy(l,++s,255);
    l[255] = '\0';  /* in case it has length 255 */
    /* strip off possible trailing '/' */
    if (l[strlen(l)-1] == '/') l[strlen(l)-1] = '\0';
    keywordlist[0].len = strlen(l); /* but not used */
    if (caseless)
    {
      for (s = l; (c = *s) != '\0'; ++s) *s = TOLOWER(c);
    }
    /* we don't need to use the inherent capability of regexp to */
    /* do case insensitive matching */
  }
  /* otherwise, proceed as before: parse the keywords */
  else
  {
  while ((c = *s++) != '\0') {
    if (c == ' ' || c == '\t') {
      *l = '\0';
      keywordlist[nb].len = j;
      keywordlist[nb].matches = 0;
      j = 0;
      ++nb;
      if (nb >= 12)
        break; /* ignore rest */
      l = keywordlist[nb].word;
      while ((c = *s) == ' ' || c == '\t' || c == '\n')
	++s;
    } else {
      if (caseless)
	*l++ = TOLOWER(c);
      else
	*l++ = c;
      ++j;
      if (j >= 255)
        break; /* ignore rest */
    }
  }
  *l = '\0';
  keywordlist[nb].len = j;
  keywordlist[nb].matches = 0;
  }  /* end of keyword parse */
  /* now search for the keywords */
  /* changed this back to one loop, sacrificing performance for */
  /* maintainability.  eliminated a bug in the process.  RP */
  for (s = text; *s != '\0'; ++s) {
    /* if we've stumbled upon a tag, treat it right */
    if (*s == '<') {
	mark = get_mark(s, &s);
	if (mark) {
	if (mark->type == M_ANCHOR && mark->start) {
	  char *tptr;
	  tptr = ParseMarkTag(mark->start, MT_ANCHOR, AT_HREF);
	  if (tptr)
	    if (*tptr && num_hrefs < MAX_REFS-1) {
	      /* add href to list */
		hrefs[num_hrefs++] = HTParse(tptr, cached_url, PARSE_HOST |
					PARSE_ACCESS | PARSE_HOST | PARSE_PATH |
					PARSE_PUNCTUATION);
		free(tptr);
	    } else
		free(tptr);
	} else
	if (mark->type == M_TITLE && !mark->is_end) {
	  int i;
	  char tmp[257], *c, p;
	  for (i=0; i<256 && ((p= *++s)==' '||p=='\t'||p=='\n'); ++i);
	  for (c = &tmp[0]; i<256 && *s!='\0' && *s != '<'; ++i,++s)
		*c++ = *s;
	  *c = '\0';
	  --s;
	  if (isdigit(tmp[0]) && isdigit(tmp[1]) && isdigit(tmp[2]) && isspace(tmp[3]))
	    return 0; /* title is known error title, not real node */
	  *title = strdup(tmp);
	} else
	if (mark->type == M_INDEX)
	  *isindex = True;
	}
	if (*s == '\0')
	  break;
    }

    /* always skip whitespace! */
    if ((c = *s) == ' ' || c == '\n' || c == '\t')
    {
      *copy++ = c;  /* copy to single white-space of same kind */
      while ((c = *++s) == ' ' || c == '\n' || c == '\t');  /* skip rest wsp */
      --s;  /* compensate for last increment */
    }
    else /* we weren't on whitespace */
    /* now, we are not before or inside a tag */
    /* and not on whitespace, either */
    if (search_type != MULTIKEY_SEARCH)
    {
      if (mark)
      /* we've just parsed a mark and we need to remember */
      {
        if (mark->text)
	  free(mark->text);
	free(mark);
	mark = (struct mark_up *) 0;
        /* and we replace the tag with nothing in the copy */
      }
      else
      /* it's an ordinary character; drop it into the copy */
      {
	*copy++ = caseless ? TOLOWER(*s) : *s;
      }
    }
    /* now look for the keyword only if we have MULTIKEY_SEARCH */
    else
    {
    for (i = 0; i <= nb; ++i) {
      l = &keywordlist[i].word[0];
      c = caseless ? TOLOWER(*s) : *s;
      if (c == *l) { /* first char is same */
        char *m = s;
        while (*l != '\0' && c == *l) {
          ++l;
	  ++m;
	  c = caseless ? TOLOWER(*m) : *m;
	  if (*l =='\0') {
	    ++keywordlist[i].matches;
	  }
	}
      }
    }
    }  /* end of treatment of this character */
  }  /* end of text scan */
  if (search_type != MULTIKEY_SEARCH)
  {
    *copy = '\0';  /* terminate copy */
    /* we only copied, changing whitespace to a single space, and */
    /* removing tags; now SEARCH */
    if (search_type == EXTERNAL_SEARCH)
    {
      keywordlist[0].matches =
		external_search(untagged_text,keywordlist[0].word);
    }
    else /* (search_type == REGEXP_SEARCH) */
    {
      keywordlist[0].matches = 
		regex_search(untagged_text,keywordlist[0].word,0);
    /* for caseless, we already transformed both args to lowercase */
    }
  }
  free(untagged_text);

  /* end of keyword searching */
  textlen = s - text;

  /* now some arbitrary way to determine relevance, depending on how
     often some of the keywords occur */
  relevance = 0;
  j = 0;
  for (i = 0; i <= nb; ++i) {
    j += (keywordlist[i].matches != 0);
    relevance += keywordlist[i].matches;
  }
  /* relevance is now the total number of matches, or -1 if an error occurred */

  if (relevance > 0)  /* leave the values 0 and -1 untouched */
  {
    /* temper the influence of many matches */
    relevance = (int)sqrt(100 * (double)relevance);
    /* have a REAL BIG influence of many keywords matching */
    relevance = relevance * j * j;
    /* have a moderate influence of document size */
    relevance = relevance*50 / ((textlen>625)?(int)sqrt((double)textlen):25);
    /* never return 0 if there were any matches */
    relevance = relevance + 1;
  }
  if (andor_type) {
    /* all keywords must occur or else relevance is 0 */
    if (j <= nb)
      relevance = 0;
  }
  /* now -1 means error, 0 means no matches, >0 means at least one match */

  /* collect and return results */

  list = malloc (num_hrefs * sizeof(char *));
  for (i=0; i<num_hrefs; ++i)
    list[i] = hrefs[i];
  *pnum_hrefs = num_hrefs;
  *phrefs = list;
  return relevance;
}

/*
 * The algorithm to determine relevance: the original code employed
 * the following: relevance is the total number of matches times
 * the fraction of matching keywords (times 10); divided by document
 * size (500 bytes minimum).
 *
 * This means that before using the text length, relevance is 0 only if
 * no matches were found at all, 10 for a single match, and is linear
 * both in the number of matches and the number of matching keywords.
 * This appears to exaggerate the value of the former; for the latter,
 * the measure appears reasonable.
 * As to the usage of text length, relevance is 20 for a single match
 * on a small (<= 500 byte) document, and shrinks inversely proportionally
 * with size, becoming 2 for 5K bytes, 1 for 10K bytes, 0 for 20K.  With
 * three keywords, all of which match three times, the numbers are 180 for
 * small documents, 2 for 100K documents, 1 for 200K documents, 0 for 400K.
 *
 * In my opinion, accidental hits should always be distinguishable from
 * complete failures.  Besides, matching many out of many keywords should
 * be rewarded even more.  Finally, the 'normalizing' action of dividing
 * by the total number of keywords seems inappropriate.  It seems to me
 * that not finding a given keyword should always return the same value
 * as not specifying the keyword at all.  This will happen when the
 * normalizing division is removed.
 *
 * The usage of file size appears in practice to overrate small documents.
 * We might simply give an advantage to documents with a comfortable size,
 * i.e., roughly one page.  Up to this sentence, this comment is 1580 bytes;
 * for XMosaic, 500 bytes seem an ideal reading length.  Smaller documents
 * are rightly considered equally big for rating purposes; but larger
 * documents may be too heavily penalized for being large.  If only a
 * portion of the document is relevant, then the only reason for having a
 * low rate on the document is transmission cost, and the cost of looking up
 * the relevant passage; scaling relevance down proportionally to size may
 * be too harsh.
 *
 * These considerations lead to the new algorithm given above.
 *
 * Still, an algorithm based on known theoretical assumptions (cosine measure
 * like things) would be preferable.
 */

/*      Gopher entity types:
*/
/* taken from libwww/HTGopher.c */

#define GOPHER_TEXT             '0'
#define GOPHER_MENU             '1'
#define GOPHER_CSO              '2'
#define GOPHER_ERROR            '3'
#define GOPHER_MACBINHEX        '4'
#define GOPHER_PCBINHEX         '5'
#define GOPHER_UUENCODED        '6'
#define GOPHER_INDEX            '7'
#define GOPHER_TELNET           '8'
#define GOPHER_BINARY           '9'
#define GOPHER_DUPLICATE        '+'

#define GOPHER_GIF              'g'
#define GOPHER_IMAGE            'I'
#define GOPHER_TN3270           'T'

#define GOPHER_HTML             'h'             /* HTML */
#define GOPHER_WWW              'w'             /* W3 address */

#define GOPHER_SOUND            's'

static char mo_get_gophertype(char *url)
{
  /* url must start with gopher:, then //hostname... then / and port,
     and then / followed by the gother type */
  char *s;

  s = strchr(url, '/'); /* now at //hostname... */
  if (s && (s[1] == '/')) {
    s += 2;
    s = strchr(s, '/'); /* now at /type */
    if (s)
      return *(s+1);
  }
  return '9'; /* pretend to have found binary file */
}

void xsleep (usec)
int usec;
{
  struct timeval timeout;

  timeout.tv_sec = usec / 1000000;
  timeout.tv_usec = usec % 1000000;
  (void) select(0, NULL, NULL, NULL, &timeout);
}

void mo_do_query(mo_window *win, char *keyword)
{
    unsigned long gtime;
    int i, nb_of_documents = 0, nb_on_same_host = 0;
    Boolean isindex=False; /* there's no way to know ... */
    Boolean may_still_have_to_query_this_node;
    char **hrefs;
    int num_hrefs;
    char *href, *firsthost, *lasthost, *host, *title;
    char *text, *texthead = NULL;
    int relevance;
    int delay_image_loads = win->delay_image_loads;
    int caseless = win->setupsearch_win ? XmToggleButtonGetState (win->query_caseless_toggle) : True;
    int htmlonly = win->setupsearch_win ? XmToggleButtonGetState (win->query_html_toggle) : False;
    int andor = win->setupsearch_win ? XmToggleButtonGetState (win->query_andor_toggle) : False;
    int servers = win->setupsearch_win ? XmToggleButtonGetState (win->query_server_toggle) : False;
    int dohotlist = win->setupsearch_win ? XmToggleButtonGetState (win->search_from_hotlist) : False;
    int docurrent = win->setupsearch_win ? XmToggleButtonGetState (win->search_from_current) : False;
#ifndef STUDENT
    int maxtime = win->timeout_val;
#else
    int maxtime = (win->timeout_val < 300) ? win->timeout_val : 300;
#endif
    int maxanswers = win->max_answers;
    int maxdepth = win->max_depth;
    int maxwidth = win->max_width;
    int width;
    int answers = 0;
    struct list {
      char *url;
      struct list *next;
      struct list *prev;
      int life;
    } *queue = NULL, *ptr, *point1 = NULL, *point2 = NULL, *last;
    if (keyword == NULL || *keyword == '\0')
      return; /* ignore search when no keywords given */
    if (!dohotlist && !docurrent) {
      XmxMakeErrorDialog
	    (win->setupsearch_win,
	       "You must select Search From Current\nor Search From Hotlist or both.",
	       "Search Setup Error");
      XtManageChild (Xmx_w);
      return;
    }
    HTAppVersion = fishAppVersion;
    re_set_syntax (win->regex_type);
    win->delay_image_loads = 1;
    /* would like to use StartClock and StopClock from mo-www.c, but alas
       these don't work. they use the 'times' call which erroneously
       always returns 0 on the Sun */
    gtime = time(NULL);
    mo_clear_default_answerlist();
    mo_post_answer_win (win);
    mo_busy ();
    XmUpdateDisplay(win->answer_win);

    clear_hash_table(); /* clear hash table for queue */
			/* this also frees the memory allocated for urls
			   during the previous search */

    firsthost = HTParse(cached_url, cached_url, PARSE_HOST);
    if (!firsthost || *firsthost=='\0')
      firsthost = strdup("localhost");
    host = strdup("no_host_will_have_this_name");
    if (docurrent) {
      queue = last = malloc(sizeof(struct list));
      queue->url = HTParse(win->current_node->url, cached_url,
                   PARSE_ACCESS | PARSE_HOST | PARSE_PATH |
                   PARSE_PUNCTUATION);
;
      queue->next = NULL;
      queue->prev = NULL;
      queue->life = maxdepth;
    }
    if (dohotlist) {
      mo_hotnode *node;
      for (node = default_hotlist->nodelist; node != NULL; node = node->next) {
	struct list *ptr;
	if (docurrent && !strcmp(queue->url, node->url))
	  /* don't enter this url a second time, with short life */
	  continue;
        ptr = malloc(sizeof(struct list));
	ptr->url = HTParse(node->url, "",
                   PARSE_ACCESS | PARSE_HOST | PARSE_PATH |
                   PARSE_PUNCTUATION);
	ptr->life = maxdepth;
	ptr->next = queue;
	ptr->prev = NULL;
	if (ptr->next != NULL) {
	  /* queue not empty */
	  ptr->next->prev = ptr;
	  queue = ptr;
	} else {
	  /* queue was empty */
	  queue = last = ptr;
	}
      }
    }
    /* we dup so that cached_url can be freed */
    cached_url = strdup(queue->url);
    while (queue != NULL) {
       /* queue is always the start of the queue */
      int i, life, max;
      if (searched_here_before(queue->url)) {
	/* this URL has been scanned, so skip it */
	if (queue == last)
	  /* queue becomes empty */
	  last = NULL;
	if (queue == point1)
	  /* point1 moves back */
	  point1 = point1->next;
	if (queue == point2)
	  /* point2 moves back */
	  point2 = point2->next;
	ptr = queue;
	queue = queue->next;
	free(ptr->url);
	free(ptr);
	if (queue!=NULL)
	  queue->prev = NULL;
	continue;
      }
      lasthost = host;
      /* we don't just take the first element in the queue but look for an
	 URL from a different host */
      ptr = queue;
      max = 3*maxwidth;
      if (win->domain <=  4) {
	/* documents from different hosts allowed */
        if (max>30) max=30;
        for (i=0;i<max && ptr != NULL; ) {
	  if (searched_here_before(ptr->url)) {
	    struct list *tmp;
	    /* this ptr will be removed, so check other pointers */
	    if (ptr == last)
	      last = ptr->prev;
	    if (ptr == point1)
	      /* point1 moves back */
	      point1 = point1->next;
	    if (ptr == point2)
	      /* point2 moves back */
	      point2 = point2->next;
	    if (ptr->prev)
	      ptr->prev->next = ptr->next;
	    else
	      /* if no prev then ptr is queue */
	      queue = ptr->next;
	    tmp = ptr;
	    ptr = tmp->next;
	    if (ptr != NULL)
	      ptr->prev = tmp->prev;
	    free(tmp->url);
	    free(tmp);
	  } else {
	    host = HTParse(ptr->url, cached_url, PARSE_HOST);
	    if (strcmp(host, lasthost))
	      break;
            ptr = ptr->next;
	    free(host);
	    ++i;
	  }
        }
        if (i<max && ptr!=NULL && ptr!=queue) {
	  /* URL from different host found */
	  /* we move this URL forward. */
	  struct list *newptr = malloc(sizeof(struct list));
	  /* we move the 'queue' position back by one */
	  /* never mind the old position, that will be removed when we
	     get there, because it will have been searched before */
	  /* have to strdup the url because it will be freed twice */
	  newptr->url = strdup(ptr->url);
	  newptr->next = queue;
	  newptr->prev = NULL;
	  newptr->life = ptr->life;
	  queue->prev = newptr;
	  queue = newptr;
        } else
	  /* no URL from different host found */
	  host = HTParse(queue->url, cached_url, PARSE_HOST);
      } else {
	/* only one host allowed */
	host = HTParse(queue->url, cached_url, PARSE_HOST);
      }
      free(cached_url);
      href = queue->url;
      cached_url = strdup(href);
      life = (queue->life < 100) ? (queue->life - 1) : 100;
      if (Rdata.track_pointer_motion)
      {
        XmString xmstr = XmStringCreateLtoR (href, XmSTRING_DEFAULT_CHARSET);
          XtVaSetValues
        (win->tracker_label,
         XmNlabelString, (XtArgVal)xmstr,
         NULL);
        XmStringFree (xmstr);
      }
      XmUpdateDisplay(win->tracker_label);
      /* we have to wait in between network accesses to avoid blocking and
	 crashing servers, and to avoid clogging the network.
	 the more we retrieve from the same host the longer we wait */
      if (host == lasthost) {
        xsleep(100000 * (++nb_on_same_host));
      } else
	nb_on_same_host = 0;
      text = mo_pull_er_over(href, &texthead);
      add_url_to_hashlist (href);
      ptr = queue;
      if (queue == last)
	last=NULL;
      if (queue == point1)
	point1 = point1->next;
      if (queue == point2)
	point2 = point2->next;
      if (point1 == point2 && point2 != NULL)
	point2 = point2->next;
      queue = queue->next;
      free(ptr);
      /* note: cannot free ptr->url since that's used in the hashlist */
      if (queue!=NULL)
	queue->prev = NULL;
      if (text == NULL) /* interrupted */
        goto end_search;
      may_still_have_to_query_this_node = True;
evaluate_this_node:
      relevance = mo_evaluate(texthead, keyword, caseless, andor, &hrefs, &num_hrefs, &title, &isindex);
      /* a negative relevance means an error occurred - stop searching */
      if (relevance < 0)
      {
	relevance = 0;
	life = 0;
	goto end_search;
      }
      if (relevance > 0) {
        char type;
        if (strncmp (href,"gopher:", 7)
	   || ((type = mo_get_gophertype(href)) != GOPHER_MENU
	   && type != GOPHER_CSO && type != GOPHER_INDEX)) {
	  mo_node *node = (mo_node *)malloc (sizeof (mo_node));
	  node->url = href;
	  ++answers;
	  node->text = text;
	  node->texthead = texthead;
	  node->ref = NULL;
	  node->title = title ? title : href;
	  node->position = answers+1;
	  node->docid = 1;
          mo_add_node_to_default_answerlist(node, relevance);
	  if (life < 100)
	    life = (life > 2*maxdepth/3) ? maxdepth : 3*life/2;
	}
      }
      if (life > 0) {
	int phase = 0;
	/* during phase 0 we pick url's for the front of the queue;
	   during phase 1 we put the rest at point2 */
	width = (relevance > 0) ? maxwidth * 3 / 2 : maxwidth;
	width = (width > num_hrefs) ? num_hrefs : width;
	if (isindex) /* all children are relevant according to server */
	  width = num_hrefs;
        for (i=0; i < num_hrefs && phase < 2 ;++i) {
	  char *extension;
	  char *node_type;
	  int j;
	  if (phase == 0 && i == width) {
	    phase = 1;
	    i = 0;
	  }
	  j = (phase == 1 || num_hrefs == width) ? i : (rand() % num_hrefs);
	  if (hrefs[j] == NULL)
	    /* we make them NULL after use, to avoid second use in phase 1 */
	    continue;
          href = HTParse(hrefs[j], cached_url,
                   PARSE_ACCESS | PARSE_HOST | PARSE_PATH |
                   PARSE_PUNCTUATION);
	  for (extension=href+strlen(href)-5;
	        (extension>href) && (*extension != '.') && (*extension!='/');
		  --extension) /* search */ ;
          if (htmlonly && strncmp("http:", href, 5) && strcmp(".html", extension)) {
	    free(href);
	    free(hrefs[j]);
	    hrefs[j] = NULL;
	    continue;
	  }
	  node_type = HTFileMimeType(href, "text");
	  if (!strncmp(node_type, "audio", 5) ||
	      !strncmp(node_type, "image", 5) ||
	      !strncmp(node_type, "video", 5) ||
	      !strncmp(node_type, "application", 11) ||
	      !strncmp(node_type, "message", 7) ||
	      !strncmp(node_type, "mosaic", 6)) {
	    free(href);
	    free(hrefs[j]);
	    hrefs[j] = NULL;
	    continue;
	  }

          if (strncmp (href, "http:", 5) && strncmp (href, "file:", 5) && strncmp (href, "news:", 5)) {
	    if (strncmp (href, "gopher:", 7)) {
	      free(href);
	      free(hrefs[j]);
	      hrefs[j] = NULL;
	      continue;
	    } else {
	      char type = mo_get_gophertype(href);
	      if (type != GOPHER_TEXT && type != GOPHER_MENU &&
                  type != GOPHER_CSO && type != GOPHER_INDEX) {
		free(href);
		free(hrefs[j]);
		hrefs[j] = NULL;
		continue;
	      }
	    }
	  }

	  /* domain  0 means any domain is fine */
          if (win->domain) {
	    char *host;
	    host = HTParse(href, cached_url, PARSE_HOST);
	    if (host) {
	      int level = 0;
	      char *firstptr, *hostptr, f, h;

	      /* domain > 4 means hosts must be identical */
	      if (win->domain > 4 && strcmp(firsthost?firsthost:"localhost", host)) {
	        free(host);
	        free(href);
		free(hrefs[j]);
		hrefs[j] = NULL;
	        continue;
              }

	      for (firstptr = firsthost + strlen(firsthost) - 1,
		   hostptr = host + strlen(host) - 1;
		   firstptr >= firsthost && hostptr >= host;
		   --firstptr, --hostptr) {
		h = *hostptr; f = *firstptr;
		if (h == f) {
		  if (h == '.') {
		    ++level;
		    if (level >= win->domain) {
		      break; /* enough domains match */
		    }
		  } else
		    continue;
		} else {
		  /* hosts differ, but do they have enough in common? */
		  if (level < win->domain) {
		    free(host);
		    free(href);
		    free(hrefs[j]);
		    hrefs[j] = NULL;
		    goto docontinue;
		  }
		}
	      }
	      free(host);
            } else {
	      free(href);
	      free(hrefs[j]);
	      hrefs[j] = NULL;
	      continue;
	    }
	  }
	  if (searched_here_before (href)) {
	    free(href);
	    free(hrefs[j]);
	    hrefs[j] = NULL;
	    continue;
	  } else {
	    ptr = malloc(sizeof(struct list));
	    ptr->url = href;
	    ptr->life = life;
	    if (phase == 0) {
	      if (relevance > 0) {
	        /* insert at beginning of queue */
	        ptr->next = queue;
	        ptr->prev = NULL;
	        if (queue)
	          queue->prev = ptr;
	        else
		  last = ptr;
	        queue = ptr;
	      } else {
	        /* insert at point1 */
	        if (point1 != NULL) {
		  /* second part of queue exists */
		  ptr->next = point1;
		  ptr->prev = point1->prev;
		  if (point1->prev != NULL)
		    point1->prev->next = ptr;
		  point1->prev = ptr;
		  if (queue==point1)
		    queue = ptr;
		  point1 = ptr;
	        } else {
		  /* second part of queue doesn't exist */
		  if (point2 != NULL) {
		    ptr->next = point2;
		    ptr->prev = point2->prev;
		    if (point2->prev != NULL)
		      point2->prev->next = ptr;
		    point2->prev = ptr;
		    if (queue == point2)
		      queue = ptr;
		    point1 = ptr;
		  } else {
		    if (last != NULL) {
		      /* but first part exists */
		      last->next = ptr;
		      ptr->next = NULL;
		      ptr->prev = last;
		      point1 = ptr;
		      last = ptr;
		    } else {
		      /* queue empty */
		      queue = last = point1 = ptr;
		      ptr->next = NULL;
		      ptr->prev = NULL;
		    }
		  }
	        }
	      }
	    } else {
	      /* insert at point2 */
	      if (point2 != NULL) {
		/* third part of queue exists */
		if (point1 == point2)
		  point1 = ptr; /* move point1 forward */
		ptr->next = point2;
		ptr->prev = point2->prev;
		if (point2->prev != NULL)
		  point2->prev->next = ptr;
		point2->prev = ptr;
		if (queue==point2)
		  queue = ptr;
		point2 = ptr;
		if (last==NULL)
		  last=ptr;
	      } else {
		/* third part of queue doesn't exist */
		if (last != NULL) {
		  /* but first part exists */
		  last->next = ptr;
		  ptr->next = NULL;
		  ptr->prev = last;
		  point2 = ptr;
		  last = ptr;
		} else {
		  /* queue empty */
		  queue = last = point1 = point2 = ptr;
		  ptr->next = NULL;
		  ptr->prev = NULL;
		}
	      }
	    }
	  }
docontinue:;
        }
      }
      /*
       * now here's an extremely stupid hack:
       * if this was an index node, and we want to query servers
       * we propagate the query to the server, replace the pending
       * document with the resulting document, and jump back up
       */
      if (isindex==1 && servers && may_still_have_to_query_this_node) {
	/* call remote server to search index nodes */
	char *url = (char *)malloc ((strlen (href) + strlen (keyword) + 10) * sizeof (char));
	char *convertedkeyword = strdup(keyword);
	char *i = keyword, *j = convertedkeyword;
	/* Convert spaces to plus's: see
	   http://info.cern.ch/hypertext/WWW/Addressing/Search.html */
	/* PDB: more efficient than in gui.c but does the same */
	do {
	  *j = (*i == ' ') ? '+' : *i;
	  ++i;
	  ++j;
	} while (*i != '\0');

	/* Clip out previous query, if any, by calling strtok. */
	strtok (href, "?");

	if (href[strlen (href)-1] == '?')
	  /* Question mark is already there... */
	  sprintf (url, "%s%s", href, convertedkeyword);
	else
	  sprintf (url, "%s?%s", href, convertedkeyword);
	free (convertedkeyword);
	ptr = malloc(sizeof(struct list));
	ptr->url = url;
	ptr->next = queue;
	ptr->life = life;
	ptr->prev = NULL;
	if (queue != NULL)
	  queue->prev = ptr;
	else {
	  last = queue;
	}
	queue = ptr;
	free (href);
	href = url;
	free(texthead);
	texthead = NULL;
	text = mo_pull_er_over(url, &texthead);
	add_url_to_hashlist (href);
        if (text == NULL) /* interrupted */
          goto end_search;
	may_still_have_to_query_this_node = False;
	goto evaluate_this_node;
      }
      if (maxtime < 10000 && time(NULL) - gtime > maxtime) {
	break;
      }
      if (maxanswers < 10000 && answers >= maxanswers) {
	break;
      }
    }
end_search:
    if (texthead != NULL) {
      free(texthead);
      texthead = NULL;
    }
    win->delay_image_loads = delay_image_loads;
    if (firsthost)
      free(firsthost);
    /* now free queue */
    if (Rdata.track_pointer_motion)
    {
      XmString xmstr = XmStringCreateLtoR (" ", XmSTRING_DEFAULT_CHARSET);
        XtVaSetValues
      (win->tracker_label,
       XmNlabelString, (XtArgVal)xmstr,
       NULL);
      XmStringFree (xmstr);
    }
    XmUpdateDisplay(win->tracker_label);
    mo_not_busy ();
    HTAppVersion = normalAppVersion;
    return;
}
#endif /* PDB */
