/* Blur 3.0 
 * Copyright (C) 1998-1999 Shawn T. Amundson
 * 
 * Algorithm and some code based on Blur 2.0.  Those portions:
 *
 * Copyright (C) 1997-98 Miles O'Neal <meo@rru.com> http://www.rru.com/~meo/
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * GUI for Blur 2.0 based on GTK code from:
 *    alienmap (Copyright (C) 1996, 1997 Daniel Cotting)
 *    plasma   (Copyright (C) 1996 Stephen Norris),
 *    oilify   (Copyright (C) 1996 Torsten Martinsen),
 *    ripple   (Copyright (C) 1997 Brian Degenhardt) and
 *    whirl    (Copyright (C) 1997 Federico Mena Quintero).
 *
 * Almost all, if not all, GUI code has been rewritten for Blur 3.0.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <libgimp/gimp.h>
#include "gimppluginwindow.h"
#include "gimppreview.h"

#define PLUG_IN_NAME "plug_in_blur3"
#define BLUR_VERSION "Blur 3.0"

#define ENTRY_WIDTH 75
#define SCALE_WIDTH 100

enum {
  SEED_TIME,
  SEED_USER
};

typedef struct {
  gdouble blur_pct;     /* likelihood of randomization (as %age) */
  gdouble blur_rcount;  /* repeat count */
  gint seed_type;       /* seed init. type - current time or user value */
  gint blur_seed;       /* seed value for rand() function */
} BlurVals;

typedef struct {
  gint run;
} BlurInterface;

typedef struct {
  GtkWidget *window;
  GtkWidget *entry;

  GtkWidget *preview;
  GDrawable *drawable;

  BlurVals *vals;
} BlurData;  

typedef struct {
  GimpPreviewEvent *event;
  GimpPreview *preview;
  guint event_id;
  GDrawable *drawable;
  
  gdouble scale;
  guint x;
  guint y;
  guint width;
  guint height;

  GPixelRgn *region;

  guchar *scaled_data;

  BlurVals *blur_vals;
} BlurIdleInfo; 

static void query (void);
static void run   (gchar *name, 
		   gint nparams, 
		   GParam *param, 
		   gint *nreturn_vals,
		   GParam **return_vals);

static gint blur_dialog();

static void ok_callback      (GtkWidget *widget, gpointer data);
static void preview_callback (GtkWidget *widget, GimpPreviewEvent *event, gpointer data);
static void entry_changed_callback (GtkWidget *widget, gpointer data);
static void radio_time_callback (GtkWidget *widget, gpointer data);
static void radio_other_callback (GtkWidget *widget, gpointer data);
static void scale_random_callback (GtkObject *object, gpointer data);
static void scale_repeat_callback (GtkObject *object, gpointer data);
static void close_callback (GtkWidget *widget, gpointer data);
static gint blur_idle (gpointer data);


GPlugInInfo PLUG_IN_INFO = {
  NULL,    /* init_proc */
  NULL,    /* quit_proc */
  query,   /* query_proc */
  run,     /* run_proc */
};

static BlurVals blur_vals = {
  50.0,
  1.0,
  SEED_TIME,
  0,
};

static BlurInterface blur_int = {
  FALSE     /*  have we run? */
};

static gint32 drawable_id = 0;


MAIN ();

static void
query ()
{
  static GParamDef args_ni[] = {
    { PARAM_INT32, "run_mode", "non-interactive" },
    { PARAM_IMAGE, "image", "Input image (unused)" },
    { PARAM_DRAWABLE, "drawable", "Input drawable" },
  };
  static int nargs_ni = sizeof(args_ni) / sizeof (args_ni[0]);
  
  static GParamDef args[] = {
    { PARAM_INT32, "run_mode", "Interactive, non-interactive" },
    { PARAM_IMAGE, "image", "Input image (unused)" },
    { PARAM_DRAWABLE, "drawable", "Input drawable" },
    { PARAM_FLOAT, "blur_pct", "Randomization percentage (1 - 100)" },
    { PARAM_FLOAT, "blur_rcount", "Repeat count(1 - 100)" },
    { PARAM_INT32, "seed_type", "Seed type (10 = current time, 11 = seed value)" },
    { PARAM_INT32, "blur_seed", "Seed value (used only if seed type is 11)" },
  };
  static int nargs = sizeof(args) / sizeof (args[0]);
  
  static GParamDef *return_vals = NULL;
  static int nreturn_vals = 0;
  
  const char *blurb = "Apply a 3x3 blurring convolution kernel to the specified drawable.";
  const char *help = "This plug-in randomly blurs the specified drawable, using a 3x3 blur.  You control the percentage of the pixels that are blurred and the number of times blurring is applied.  Indexed images are not supported.";
  const char *author = "Shawn T. Amundson <amundson@gimp.org>";
  const char *copyrights = "Shawn T. Amundson, Miles O'Neal, Spencer Kimball, Peter Mattis, Torsten Martinsen, Brian Degenhardt, Federico Mena Quintero, Stephen Norris, Daniel Cotting";
  const char *copyright_date = "1995-1998";
  
  gimp_install_procedure("plug_in_blur_randomize3",
			 (char *) blurb,
			 (char *) help,
			 (char *) author,
			 (char *) copyrights,
			 (char *) copyright_date,
			 "<Image>/Filters/Blur3",
			 "RGB*, GRAY*",
			 PROC_PLUG_IN,
			 nargs, nreturn_vals,
			 args, return_vals);

  gimp_install_procedure(PLUG_IN_NAME,
			 (char *) blurb,
			 (char *) help,
			 (char *) author,
			 (char *) copyrights,
			 (char *) copyright_date,
			 NULL,
			 "RGB*, GRAY*",
			 PROC_PLUG_IN,
			 nargs_ni, nreturn_vals,
			 args_ni, return_vals);

}


static void 
run (gchar *name, 
     gint nparams, 
     GParam *param, 
     gint *nreturn_vals,
     GParam **return_vals)
{
  GDrawable *drawable;
  GRunModeType run_mode;
  GStatusType status = STATUS_SUCCESS;   /* assume the best! */
  static GParam values[1];

  /* Get the specified drawable, do standard initialization. */
  run_mode = param[0].data.d_int32;
  drawable_id = param[2].data.d_drawable;
  drawable = gimp_drawable_get(param[2].data.d_drawable);

  /* Initial return values */
  values[0].type = PARAM_STATUS;
  values[0].data.d_status = STATUS_SUCCESS;
  *nreturn_vals = 1;
  *return_vals = values;

  /* Make sure the drawable type is appropriate. */
  if (gimp_drawable_color (drawable->id) ||
      gimp_drawable_gray (drawable->id))
    {
      switch (run_mode) 
	{
	case RUN_INTERACTIVE:
	  gimp_get_data (PLUG_IN_NAME, &blur_vals);
	  if (!blur_dialog ()) /* return on cancel */
	    return;
	  break;

	case RUN_NONINTERACTIVE:
	  /* If we're not interactive (probably scripting), we
	   *  get the parameters from the param[] array, since
	   *  we don't use the dialog box.  Make sure all
	   *  parameters have legitimate values.
	   */
	  /*TODO*/
	  
	case RUN_WITH_LAST_VALS:
	  gimp_get_data(PLUG_IN_NAME, &blur_vals);
	  break;

	default:
	  break;
	}

      /*TODO*/
    }

  values[0].data.d_status = status;
  gimp_drawable_detach(drawable);

}

static gint
blur_dialog()
{
  gchar **argv;
  gint argc;
  BlurData *pdata;
  GtkWidget *vbox;
  GtkWidget *frame_seed;
  GtkWidget *table_seed;
  GSList *radio_group;
  GtkWidget *toggle_time;
  GtkWidget *toggle_other;
  GtkWidget *table;
  GtkWidget *label_random;
  GtkWidget *label_repeat;
  GtkWidget *scale_random;
  GtkWidget *scale_repeat;
  GtkObject *scale_random_data;
  GtkObject *scale_repeat_data;
  GDrawable *drawable;
  gchar buffer[255];
  GimpPreview *preview;

  argc = 1;
  argv = g_new(gchar *, 1);
  argv[0] = g_strdup("Blur");
  
  gtk_init(&argc, &argv);
  gtk_rc_parse(gimp_gtkrc());
    
  pdata = g_malloc (sizeof(BlurData));
  pdata->window = NULL;
  pdata->vals = &blur_vals;

  /* Create window */
  drawable = gimp_drawable_get(drawable_id);
  pdata->drawable = drawable;
  pdata->window = gimp_plugin_window_new (GIMP_PLUGIN_NORMAL, BLUR_VERSION,
					  drawable);
  
  gtk_signal_connect (GTK_OBJECT (pdata->window), "clicked_ok",
		      GTK_SIGNAL_FUNC (ok_callback), 
		      (gpointer) (pdata->window)); 
  gtk_signal_connect (GTK_OBJECT (pdata->window), "destroy",
		      GTK_SIGNAL_FUNC (close_callback), 
		      NULL); 

  preview = GIMP_PREVIEW (((GimpPluginWindow*)(pdata->window))->preview);
  gtk_signal_connect (GTK_OBJECT (preview), "update_preview",
		      GTK_SIGNAL_FUNC (preview_callback), 
		      (gpointer) pdata); 
  pdata->preview = GTK_WIDGET (preview);

  vbox = gtk_vbox_new(0, 8);
  gtk_container_add (GTK_CONTAINER (pdata->window), vbox);
  
  /* Randomization Seed Configuration Options */
  frame_seed = gtk_frame_new ("Randomization Seed");
  gtk_box_pack_start (GTK_BOX (vbox), frame_seed, FALSE, FALSE, 0);
  
  table_seed = gtk_table_new (2, 2, FALSE);
  gtk_container_border_width (GTK_CONTAINER (table_seed), 8);
  gtk_container_add (GTK_CONTAINER (frame_seed), table_seed);
  
  radio_group = NULL;
  toggle_time = gtk_radio_button_new_with_label (radio_group, "Current Time");
  radio_group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle_time));
  toggle_other = gtk_radio_button_new_with_label (radio_group, "Other Value");
  pdata->entry = gtk_entry_new();

  sprintf(buffer, "%d", blur_vals.blur_seed);
  gtk_entry_set_text(GTK_ENTRY(pdata->entry), buffer);
  gtk_widget_set_sensitive(pdata->entry, FALSE);
  
  gtk_signal_connect(GTK_OBJECT(toggle_time), "toggled",
		     (GtkSignalFunc) radio_time_callback, 
		     pdata);
  gtk_signal_connect(GTK_OBJECT(toggle_other), "toggled",
		     (GtkSignalFunc) radio_other_callback, 
		     pdata);
  gtk_signal_connect(GTK_OBJECT (pdata->entry), "changed",
		     (GtkSignalFunc) entry_changed_callback, 
		     pdata);
  
  gtk_table_attach(GTK_TABLE(table_seed), toggle_time, 0, 1, 0, 1,
		   GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(table_seed), toggle_other, 0, 1, 1, 2,
		   GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(table_seed), pdata->entry, 1, 2, 1, 2,
		   GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL, 0, 0);
  
  /* This table holds both Randomization % and Repeat Option */
  table = gtk_table_new(2, 2, 0);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  
  
  /* Randomization Percent */
  label_random = gtk_label_new("Randomization %:");
  gtk_label_set_justify (GTK_LABEL (label_random), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label_random), 0.0, 0.5);
  scale_random_data = gtk_adjustment_new(1.0,1.0,100.0,1.0,1.0,0.0);
  scale_random = gtk_hscale_new(GTK_ADJUSTMENT(scale_random_data));
  gtk_scale_set_digits(GTK_SCALE(scale_random), 0);
  gtk_range_set_update_policy(GTK_RANGE(scale_random), GTK_UPDATE_DELAYED);
  gtk_signal_connect (GTK_OBJECT(scale_random_data), "value_changed",
		      GTK_SIGNAL_FUNC (scale_random_callback),
		      pdata);
  
  /* Repeat Option */
  label_repeat = gtk_label_new("Repeat:");
  gtk_label_set_justify (GTK_LABEL (label_repeat), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label_repeat), 0.0, 0.5);
  scale_repeat_data = gtk_adjustment_new(1.0,1.0,100.0,1.0,1.0,0);
  scale_repeat = gtk_hscale_new(GTK_ADJUSTMENT(scale_repeat_data));
  gtk_scale_set_digits(GTK_SCALE(scale_repeat), 0);
  gtk_range_set_update_policy(GTK_RANGE(scale_repeat), GTK_UPDATE_DELAYED);
  gtk_signal_connect (GTK_OBJECT(scale_repeat_data), "value_changed",
		      GTK_SIGNAL_FUNC (scale_repeat_callback),
		      pdata);
  
  /* Pack all Randomization % and Repeat Options */
  gtk_table_set_col_spacing (GTK_TABLE (table), 0, 8);
  gtk_table_set_row_spacing (GTK_TABLE (table), 0, 4);
  gtk_table_attach(GTK_TABLE(table), label_random, 0, 1, 0, 1,
		   GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(table), label_repeat, 0, 1, 1, 2,
		   GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(table), scale_random, 1, 2, 0, 1,
		   GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(table), scale_repeat, 1, 2, 1, 2,
		   GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  
  
  gtk_widget_show (label_random);
  gtk_widget_show (label_repeat);
  gtk_widget_show (scale_random);
  gtk_widget_show (scale_repeat);
  gtk_widget_show (table);
  gtk_widget_show (toggle_time);
  gtk_widget_show (toggle_other);
  gtk_widget_show (pdata->entry);
  gtk_widget_show (table_seed);
  gtk_widget_show (frame_seed);
  gtk_widget_show (vbox);
  gtk_widget_show (pdata->window);

  gimp_tile_cache_ntiles (9);  /*  FIXME */

  gimp_preview_update (GIMP_PREVIEW (preview));

  gtk_main();
  gdk_flush();
  
  return blur_int.run;
}

static void
ok_callback(GtkWidget *widget, gpointer data) 
{
  blur_int.run = TRUE;
  gtk_widget_destroy(GTK_WIDGET(data));
}

static void
preview_callback (GtkWidget *widget, GimpPreviewEvent *event, gpointer data) 
{
  static guint event_id = 0;
  BlurIdleInfo *info;
  BlurData *blur_data;

  event_id++;
  
  blur_data = (BlurData*) data;

  info = g_malloc (sizeof (BlurIdleInfo));
  info->preview = GIMP_PREVIEW (blur_data->preview);
  info->drawable = blur_data->drawable;
  info->event_id = event_id;
  info->scale = event->scale;
  info->x = (guint) event->x;
  info->y = (guint) event->y;
  info->width = event->width;
  info->height = event->height;
  info->blur_vals = blur_data->vals;
  info->scaled_data = event->scaled_data;

  gtk_idle_add ((GtkFunction) blur_idle, (gpointer) info);
}

static gint
blur_idle (gpointer data)
{
  static guint in_progress = 0;
  static guint count = 0;
  static guchar *src = NULL;
  static guchar *dest = NULL;
  BlurIdleInfo *info;
  BlurVals *blur_vals;
  guint row;
  int y;
  int x;
  int prev_x, next_x;
  int prev_y, next_y;
  guchar *prev_row, *dest_row, *next_row;
  guchar *curr_row;
  gdouble scale;
  gint scaleint;

  info = (BlurIdleInfo*) data;
  blur_vals = info->blur_vals;

  if (in_progress > info->event_id)
    return FALSE;

  if (in_progress < info->event_id)
    {
      in_progress = info->event_id;
      count = 0;
      
      if (src)
	g_free (src);

      if (dest)
	g_free (dest);
      
      src = g_malloc (sizeof (guchar) 
		      * info->preview->width
		      * info->preview->height
		      * 4);
      dest = g_malloc (sizeof (guchar) 
		      * info->preview->width
		      * info->preview->height
		      * 4);

      memcpy (src, info->scaled_data, 
	      sizeof (guchar) * info->preview->width * info->preview->height * 4);

      /* this next one should be nuked */
      memcpy (dest, info->scaled_data, 
	      sizeof (guchar) * info->preview->width * info->preview->height * 4);
  
/*
      printf("\nrandom percent %f\n", blur_vals->blur_pct);
      printf("repeat count %f\n", blur_vals->blur_rcount);
      printf("seed type %d\n", blur_vals->seed_type);
      printf("blur seed %d\n", blur_vals->blur_seed);
*/
    }


/*
  printf("[%d]\n", count);
*/

  scale = info->scale;
  scaleint = (int) scale;

  for (y = 0; y < info->preview->height; y++)
    {
      prev_y = MAX ((y - scaleint), 0);
      next_y = MIN ((y + scaleint), info->preview->height - 1);

      prev_row = &(src[prev_y * info->preview->width * 4]);
      curr_row = &(src[y * info->preview->width * 4]);
      next_row = &(src[next_y * info->preview->width * 4]);
      dest_row = &(dest[y * info->preview->width * 4]);

      for (x = 0; x < info->preview->width * 4; x++)
        {
          prev_x = x - scaleint * 4;
          next_x = x + scaleint * 4;
	  
	  if (prev_x < 0)
	    prev_x += scaleint * 4;
	  
	  if (next_x >= info->preview->width * 4)
	    next_x -= scaleint * 4;
	  
	  dest_row[x] = (prev_row[prev_x] + prev_row[x] + prev_row[next_x]
			 + curr_row[prev_x] + curr_row[x] + curr_row[next_x]
			 + next_row[prev_x] + next_row[x] + next_row[next_x]) / 9;
	  
        }
    }
  
  memcpy (src, dest,
          sizeof (guchar) * info->preview->width * info->preview->height * 4);

  gtk_progress_bar_update (GTK_PROGRESS_BAR (info->preview->progress_bar), 
	                   (count + 1) / blur_vals->blur_rcount);

  count++;

  if (count < blur_vals->blur_rcount)
    return TRUE;
  else
    {
      gimp_preview_force_redraw (info->preview);
      gtk_progress_bar_update (GTK_PROGRESS_BAR (info->preview->progress_bar), 
	                       1.0);

      for (row = 0; row < info->preview->height; row++)
        {
          gimp_preview_draw_row (info->preview, RGBA_IMAGE, row,
			         &(dest[row * info->preview->width * 4]));
        }
      gimp_preview_force_redraw (info->preview);

      /* Should we force a redraw of progress bar before we reset it? */
      gtk_progress_bar_update (GTK_PROGRESS_BAR (info->preview->progress_bar),
			       0.0);
      gimp_preview_force_redraw (info->preview);
      return FALSE;
    }
}

static void
entry_changed_callback (GtkWidget *widget, gpointer data) {
  BlurData *plugin_data;
  gint new_value;
  
  new_value = atoi(gtk_entry_get_text(GTK_ENTRY(widget)));
  
  plugin_data = (BlurData*)data;
  if (new_value != plugin_data->vals->blur_seed) {
    plugin_data->vals->blur_seed = new_value;
  }

  gimp_preview_update (GIMP_PREVIEW (plugin_data->preview));
}

static void 
radio_time_callback (GtkWidget *widget, gpointer data)
{
  BlurData *plugin_data;

  plugin_data = (BlurData*)data;

  if (GTK_TOGGLE_BUTTON(widget)->active) {
    plugin_data->vals->seed_type = SEED_TIME;
  }

  gimp_preview_update (GIMP_PREVIEW (plugin_data->preview));
}

static void 
radio_other_callback (GtkWidget *widget, gpointer data)
{
  BlurData *plugin_data;

  plugin_data = (BlurData*)data;

  if (GTK_TOGGLE_BUTTON(widget)->active) {
    plugin_data->vals->seed_type = SEED_USER;
    gtk_widget_set_sensitive(plugin_data->entry, TRUE);
  } else {
    gtk_widget_set_sensitive(plugin_data->entry, FALSE);
  }
  
  gimp_preview_update (GIMP_PREVIEW (plugin_data->preview));
}

static void 
scale_random_callback (GtkObject *object, gpointer data)
{
  GtkAdjustment *adjustment;
  BlurData *plugin_data;
  gdouble new_value;

  plugin_data = (BlurData*)data;

  adjustment = GTK_ADJUSTMENT (object);
  new_value = adjustment->value;

  if (plugin_data->vals->blur_pct != new_value) {
    plugin_data->vals->blur_pct = new_value;

    gimp_preview_update (GIMP_PREVIEW (plugin_data->preview));
  }
}

static void 
scale_repeat_callback (GtkObject *object, gpointer data)
{
  GtkAdjustment *adjustment;
  BlurData *plugin_data;
  gdouble new_value;

  plugin_data = (BlurData*)data;

  adjustment = GTK_ADJUSTMENT (object);
  new_value = adjustment->value;

  if (plugin_data->vals->blur_rcount != new_value) {
    plugin_data->vals->blur_rcount = new_value;

    gimp_preview_update (GIMP_PREVIEW (plugin_data->preview));
  }
}

static void
close_callback (GtkWidget *widget, gpointer data)
{
  gtk_main_quit();
}
