#include <gtk/gtk.h>
#include "gtkthreads.h"
#include <pthread.h>

#include <fcntl.h>

typedef enum {
  GTK_UNLOCKED,			/* Nobody has GTK locked */
  GTK_LOCKED,			/* Somebody has GTK locked */
  GTK_MAINLOOP			/* The mainloop has GTK locked, and needs
				 * to be signalled */
} GtkLockState;

GtkLockState gtk_lock_state = GTK_UNLOCKED;

/* The number of threads other than the mainloop waiting
 * for the lock. If this is > 0 when we reenter the mainloop,
 * then we need to signal to release the lock on the next iteration.
 */
gint gtk_lock_nwaiting = 0;

/* This mutex must be held when reading or modifying gtk_lock_state */
pthread_mutex_t gtk_lock_mutex = PTHREAD_MUTEX_INITIALIZER;

/* Used to signal setting gtk_lock_state to GTK_UNLOCKED */
pthread_cond_t gtk_lock_cond = PTHREAD_COND_INITIALIZER;

/* Used to signal mainloop to release lock */
gint gtk_lock_pipe[2];

static void gtk_threads_callback (gpointer          data, 
				 gint              source,
				 GdkInputCondition condition);

void
gtk_threads_init (void)
{
  pipe (gtk_lock_pipe);
  fcntl (gtk_lock_pipe[0], F_SETFL, O_NONBLOCK);
  fcntl (gtk_lock_pipe[1], F_SETFL, O_NONBLOCK);

  gdk_input_add (gtk_lock_pipe[0], GDK_INPUT_READ, gtk_threads_callback, NULL);
}

void
gtk_threads_enter (void)
{
  pthread_mutex_lock (&gtk_lock_mutex);

  while (gtk_lock_state != GTK_UNLOCKED)
    {
      gtk_lock_nwaiting++;
      if (gtk_lock_state == GTK_MAINLOOP)
	{
	  gtk_lock_state = GTK_LOCKED;
	  write(gtk_lock_pipe[1], "A", 1);
	}
      pthread_cond_wait (&gtk_lock_cond, &gtk_lock_mutex);
      gtk_lock_nwaiting--;
    }
  gtk_lock_state = GTK_LOCKED;
  pthread_mutex_unlock (&gtk_lock_mutex);
}

void
gtk_threads_leave (void)
{
  pthread_mutex_lock (&gtk_lock_mutex);
  gtk_lock_state = GTK_UNLOCKED;
  pthread_cond_signal (&gtk_lock_cond);

  pthread_mutex_unlock (&gtk_lock_mutex);
}

void
gtk_threads_enter_main (void)
{
  pthread_mutex_lock (&gtk_lock_mutex);
  while (gtk_lock_state != GTK_UNLOCKED)
    pthread_cond_wait (&gtk_lock_cond, &gtk_lock_mutex);

  if (gtk_lock_nwaiting > 0)
    write(gtk_lock_pipe[1], "A", 1);

  gtk_lock_state = GTK_MAINLOOP;
  pthread_mutex_unlock (&gtk_lock_mutex);
}

void
gtk_threads_leave_main (void)
{
  char c;

  pthread_mutex_lock (&gtk_lock_mutex);
  read (gtk_lock_pipe[0], &c, 1);
  gtk_lock_state = GTK_UNLOCKED;
  pthread_cond_signal (&gtk_lock_cond);
  pthread_mutex_unlock (&gtk_lock_mutex);
}

void
gtk_threads_main (void)
{
  gtk_threads_enter_main();
  gtk_main();
  gtk_threads_leave_main ();
}

/* This routine is just like enter_main / leave_main, except
 * we omit an unecessary lock/unlock, and we always do at
 * least one cond_wait() to make sure someone else gets
 * control.
 */
static void 
gtk_threads_callback (gpointer          data, 
		     gint              source,
		     GdkInputCondition condition)
{
  char c;
  pthread_mutex_lock (&gtk_lock_mutex);
  read (gtk_lock_pipe[0], &c, 1);
  gtk_lock_state = GTK_UNLOCKED;

  pthread_cond_signal (&gtk_lock_cond);

  /* Now wait for control to return */
  do
    pthread_cond_wait (&gtk_lock_cond, &gtk_lock_mutex);
  while (gtk_lock_state != GTK_UNLOCKED);

  if (gtk_lock_nwaiting > 0)
    write(gtk_lock_pipe[1], "A", 1);

  gtk_lock_state = GTK_MAINLOOP;
  pthread_mutex_unlock (&gtk_lock_mutex);
}

