#include "glib.h"
#include "gthread.h"
#include <errno.h>

#ifdef G_THREAD_POSIX
#  include <pthread.h>
#endif

const char      *g_log_domain_gthread = "GThread";

#ifdef G_THREAD_POSIX

#define PRINT_ERROR( name, num )                                          \
  g_error( "file %s: line %d (%s): error %s during %s",                   \
           __FILE__, __LINE__, G_GNUC_PRETTY_FUNCTION,                    \
           g_strerror((num)), #name )
#define CHECK_ERROR( what ) G_STMT_START{                                 \
  int error = (what);                                                     \
  if( error ) { PRINT_ERROR( what, error ); }                             \
  }G_STMT_END


typedef struct _GRealThread       GRealThread;

struct _GRealThread
{
  pthread_t id;
};

static void
thread_has_ended(gpointer thread)
{
  g_free( thread );
}

static GThreadPrivate* thread_private = NULL;

static void
init_threads()
{
  GRealThread* main_thread;
  thread_private = g_thread_private_new(thread_has_ended);
  g_return_if_fail( thread_private );
  main_thread  = g_new (GRealThread, 1);
  main_thread->id = pthread_self();
  g_thread_private_set( thread_private, main_thread );
}

GThread* 
g_thread_new(GThreadFunc thread_func, 
	     gpointer arg, 
	     guint stack_size)
{     
  GRealThread* thread;
  pthread_attr_t attr;

  g_return_val_if_fail( thread_func, NULL );

  if( !thread_private )
    init_threads();

  thread  = g_new (GRealThread, 1);
  
  CHECK_ERROR( pthread_attr_init(&attr) );
  
  if( stack_size )
    {
      CHECK_ERROR( pthread_attr_setstacksize( &attr, stack_size ) );
    }
  
  CHECK_ERROR( pthread_create (&thread->id, NULL, 
			       (void*(*)(void*))thread_func, arg) );

  CHECK_ERROR( pthread_attr_destroy (&attr) );

  g_thread_private_set (thread_private, thread);
  return (GThread*) thread;
}

gboolean 
g_thread_join(GThread* thread)
{     
  void *ignore;
  GRealThread *real = (GRealThread*) thread;

  g_return_val_if_fail( real, FALSE );
  CHECK_ERROR( pthread_join (real->id, &ignore ) );

  g_free( real );
  return TRUE;
}

void 
g_thread_exit()
{
  pthread_exit(NULL);
}

GThread* 
g_thread_get_current()
{
  return g_thread_private_get( thread_private );
}

void
g_thread_sleep(gulong microseconds)
{
  static GMonitor* wait_monitor = NULL;

  if( !wait_monitor )
    wait_monitor = g_monitor_new();

  if( microseconds == 0 ) /* This means Yield */
    {
      microseconds = 1000; 
    }
  g_monitor_wait( wait_monitor, microseconds );
}

typedef struct _GRealThreadPrivate      GRealThreadPrivate;

struct _GRealThreadPrivate
{
  pthread_key_t key;
};

GThreadPrivate* 
g_thread_private_new(GDestroyNotify destroy_func)
{
  GRealThreadPrivate* result = g_new(GRealThreadPrivate,1);
  CHECK_ERROR( pthread_key_create (&result->key, destroy_func) );
  return (GThreadPrivate*) result;
}

void
g_thread_private_set(GThreadPrivate* id, gpointer data)
{
  GRealThreadPrivate* real = (GRealThreadPrivate*) id;
  g_return_if_fail( real );
  CHECK_ERROR( pthread_setspecific( real->key, data ) );
}

gpointer 
g_thread_private_get(GThreadPrivate* id)
{
  GRealThreadPrivate* real = (GRealThreadPrivate*) id;
  g_return_val_if_fail( real, NULL );
  return pthread_getspecific( real->key );
}

void 
g_thread_private_free(GThreadPrivate* id)
{
  GRealThreadPrivate* real = (GRealThreadPrivate*) id;
  g_return_if_fail( real );
  CHECK_ERROR( pthread_key_delete( real->key ) );
  /* FIXME: call all private destructors */
}

typedef struct _GRealMonitor      GRealMonitor;

static pthread_mutex_t* mutex_mutex = NULL;
static GHashTable* mutex_hash_table = NULL;

static void
init_mutex_mutex()
{
  mutex_mutex = g_new(pthread_mutex_t,1);
  CHECK_ERROR( pthread_mutex_init (mutex_mutex, NULL) );
}

static void
init_mutex_hash_table()
{
  mutex_hash_table = g_hash_table_new( NULL, NULL );
}

#define G_MUTEX_MUTEX_LOCK   if (!mutex_mutex)                             \
                               init_mutex_mutex();                         \
			       CHECK_ERROR( pthread_mutex_lock(mutex_mutex) )
#define G_MUTEX_MUTEX_UNLOCK CHECK_ERROR( pthread_mutex_unlock(mutex_mutex) )

struct _GRealMonitor
{
  pthread_mutex_t mutex;
  pthread_cond_t* cond;
  gpointer resource;
  guint entry_counter;
  guint ref_counter;
  pthread_t holder;
};

GMonitor* 
g_monitor_new()
{
  GRealMonitor* result = g_new(GRealMonitor,1);
  CHECK_ERROR( pthread_mutex_init (&result->mutex, NULL));
  result->entry_counter = 0;
  result->ref_counter = 0;
  result->resource = NULL;
  result->cond = NULL;
  return (GMonitor*) result;
}

GMonitor* 
g_monitor_new_for_resource(gpointer resource)
{
  GRealMonitor* result;

  g_return_val_if_fail( resource, NULL );

  G_MUTEX_MUTEX_LOCK;
  if( !mutex_hash_table )
    init_mutex_hash_table();

  result = g_hash_table_lookup( mutex_hash_table, resource );
  if( !result )
    {
      result = (GRealMonitor*) g_monitor_new();
      result->resource = resource;
      g_hash_table_insert( mutex_hash_table, result->resource, result );
    }

  result->ref_counter++;
  G_MUTEX_MUTEX_UNLOCK;
  
  return (GMonitor*) result;
}

void 
g_monitor_free(GMonitor* monitor)
{
  GRealMonitor* real = (GRealMonitor*) monitor;
  g_return_if_fail( real );
  G_MUTEX_MUTEX_LOCK;

  real->ref_counter--;

  if( !real->resource || 
      ( real->ref_counter == 0 && real->entry_counter == 0 ) )
    {
      CHECK_ERROR( pthread_mutex_destroy( &real->mutex ) );
      if( real->cond )
	{
	  CHECK_ERROR( pthread_cond_destroy( real->cond ) );
	}
      if( real->resource )
	{
	  g_hash_table_remove( mutex_hash_table, real->resource );
	}
      g_free( real );
    }
  G_MUTEX_MUTEX_UNLOCK;
}

void 
g_monitor_enter(GMonitor* monitor)
{
  GRealMonitor* real = (GRealMonitor*) monitor;
  g_return_if_fail( real );
  if( pthread_equal( real->holder, pthread_self() ) )
    {
      real->entry_counter++;
      return;
    }
  CHECK_ERROR( pthread_mutex_lock(&real->mutex));
  real->holder = pthread_self();
  g_assert( real->entry_counter == 0);
  real->entry_counter = 1;
}

gboolean 
g_monitor_exit(GMonitor* monitor)
{
  GRealMonitor* real = (GRealMonitor*) monitor;
  g_return_val_if_fail( real, FALSE );
  if( pthread_equal( real->holder, pthread_self() ) )
    {
      real->entry_counter--;
      if( real->entry_counter == 0 )
	{
	  real->holder = 0; 
	  CHECK_ERROR( pthread_mutex_unlock(&real->mutex) );
	  return TRUE;
	}
      return TRUE;
    }
  return FALSE;
}

#define G_MICROSEC 1000000
#define G_NANOSEC 1000000000

gboolean 
g_monitor_wait(GMonitor* monitor, gulong microseconds)
{
  GRealMonitor* real = (GRealMonitor*) monitor;
  int old_counter = real->entry_counter;
  pthread_t old_holder = real->holder;
  gboolean entered = TRUE;
  gboolean timed_out = FALSE;
  g_return_val_if_fail( real, FALSE );
  if( !pthread_equal( old_holder, pthread_self() ) )
    {
      entered = FALSE;
      CHECK_ERROR( pthread_mutex_lock(&real->mutex) );
      real->holder = pthread_self();
      real->entry_counter = 1;
    }
  if( !real->cond )
    {
      pthread_cond_t* new_cond = g_new(pthread_cond_t,1);
      CHECK_ERROR( pthread_cond_init(new_cond, NULL ) );
      real->cond = new_cond; /* this should be atomic */
    } 
  real->entry_counter = 0;
  if( microseconds == 0 ) /* infinite wait */
    {      
      CHECK_ERROR( pthread_cond_wait( real->cond, &real->mutex ) );
    }
  else /* finite wait */
    {
      int result;
      timestruc_t end_time;
      struct timeval current_time;
      gettimeofday( &current_time, NULL);
      end_time.tv_sec = current_time.tv_sec + microseconds / G_MICROSEC;
      end_time.tv_nsec = ( current_time.tv_usec + microseconds % G_MICROSEC ) 
	* ( G_NANOSEC / G_MICROSEC );
      while( end_time.tv_nsec >= G_NANOSEC )
	{
	  end_time.tv_sec++;
	  end_time.tv_nsec-= G_NANOSEC;
	}
      result = pthread_cond_timedwait( real->cond, &real->mutex, &end_time );
      timed_out = ( result == ETIMEDOUT );
      if( !timed_out ) CHECK_ERROR( result );
   }
  if( entered )
    {
      real->entry_counter = old_counter;
      real->holder = old_holder;
    }
  else
    {
      real->holder = 0;
      real->entry_counter = 0;
      CHECK_ERROR( pthread_mutex_unlock(&real->mutex) );
    }
  return !timed_out;
}

void 
g_monitor_notify(GMonitor* monitor)
{
  GRealMonitor* real = (GRealMonitor*) monitor;
  g_return_if_fail( real );
  if( real->cond )
    {
      CHECK_ERROR( pthread_cond_signal(real->cond) );
    }
}

void 
g_monitor_notify_all(GMonitor* monitor)
{
  GRealMonitor* real = (GRealMonitor*) monitor;
  g_return_if_fail( real );
  if( real->cond )
    {
      CHECK_ERROR( pthread_cond_broadcast(real->cond) );
    }
}

#endif /* G_THREAD_POSIX */
