/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include "buffer.h"


static GimpBufferClass my_class =
{
  STORAGE_NONE,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL
};


void
gimp_buffer_init (GimpBuffer * buffer,
                  Tag tag,
                  gint width,
                  gint height,
                  gint x_period,
                  gint x_phase,
                  gint y_period,
                  gint y_phase)
{
  g_return_if_fail (buffer != NULL);
  g_return_if_fail (width > 0);
  g_return_if_fail (height > 0);
  g_return_if_fail (tag_valid (tag) == TRUE);

  buffer->width = width;
  buffer->height = height;
  buffer->depth = tag_bytes (tag);
  buffer->tag = tag;
  buffer->x.period = x_period;
  buffer->x.phase = x_phase;
  buffer->y.period = y_period;
  buffer->y.phase = y_phase;
  buffer->autoalloc = TRUE;
  buffer->autovalidate = TRUE;
  buffer->klass = (void*) &my_class;
}


void
gimp_buffer_uninit (GimpBuffer * buffer)
{
}


gint
gimp_buffer_width (GimpBuffer *buffer)
{
  g_return_val_if_fail (buffer != NULL, 0);

  return buffer->width;
}


gint
gimp_buffer_height (GimpBuffer *buffer)
{
  g_return_val_if_fail (buffer != NULL, 0);

  return buffer->height;
}


gint
gimp_buffer_depth (GimpBuffer *buffer)
{
  g_return_val_if_fail (buffer != NULL, 0);

  return buffer->depth;
}


/* the guts of this routine, as well as the tiling structure data, was
   moved to the base class for speed */
gboolean
gimp_buffer_focus (GimpBuffer   *buffer,
                   GimpPortion  *portion,
                   GimpArea     *roi)
{
  GimpArea *bounds, *focus;

  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (portion != NULL, FALSE);
  g_return_val_if_fail (roi != NULL, FALSE);

  bounds = &portion->bounds;
  focus = &portion->focus;

  /* consider buffer height and width */
  if (roi->a.x < 0 ||
      roi->a.y < 0 ||
      roi->a.x >= buffer->width ||
      roi->a.y >= buffer->height ||
      roi->b.x <= roi->a.x ||
      roi->b.y <= roi->a.y)
    return FALSE;
  
  /* find the contiguous area */
  bounds->a.x = roi->a.x - (roi->a.x % buffer->x.period);
  bounds->a.y = roi->a.y - (roi->a.y % buffer->y.period);
  bounds->b.x = bounds->a.x + buffer->x.period;
  bounds->b.y = bounds->a.y + buffer->y.period;

  /* find the focus, trimming roi if required */
  focus->a.x = roi->a.x;
  focus->a.y = roi->a.y;
  focus->b.x = MIN (roi->b.x, bounds->b.x);
  focus->b.y = MIN (roi->b.y, bounds->b.y);

  return TRUE;
}




void
gimp_buffer_delete (GimpBuffer *buffer)
{
  g_return_if_fail (buffer != NULL);
  g_return_if_fail (buffer->klass != NULL);
  g_return_if_fail (GIMP_BUFFER_CLASS(buffer->klass)->delete != NULL);

  GIMP_BUFFER_CLASS(buffer->klass)->delete (buffer);
}


gboolean
gimp_buffer_alloc (GimpBuffer        *buffer,
                   GimpPortion       *portion,
                   Alloc              how)
{
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);
  g_return_val_if_fail (GIMP_BUFFER_CLASS(buffer->klass)->alloc != NULL, FALSE);
  g_return_val_if_fail (portion != NULL, FALSE);

  return GIMP_BUFFER_CLASS(buffer->klass)->alloc (buffer, portion, how);
}


gboolean
gimp_buffer_validate (GimpBuffer        *buffer,
                      GimpPortion       *portion,
                      Validate           how)
{
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);
  g_return_val_if_fail (GIMP_BUFFER_CLASS(buffer->klass)->validate != NULL, FALSE);
  g_return_val_if_fail (portion != NULL, FALSE);

  return GIMP_BUFFER_CLASS(buffer->klass)->validate (buffer, portion, how);
}


gboolean
gimp_buffer_use (GimpBuffer        *buffer,
                 GimpPortion       *portion,
                 Use                how)
{
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);
  g_return_val_if_fail (GIMP_BUFFER_CLASS(buffer->klass)->use != NULL, FALSE);
  g_return_val_if_fail (portion != NULL, FALSE);
  
  return GIMP_BUFFER_CLASS(buffer->klass)->use (buffer, portion, how);
}


gboolean
gimp_buffer_query (GimpBuffer    *buffer,
                   GimpPortion   *portion,
                   GimpMemStatus *status)
{
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);
  g_return_val_if_fail (GIMP_BUFFER_CLASS(buffer->klass)->query != NULL, FALSE);
  g_return_val_if_fail (portion != NULL, FALSE);
  g_return_val_if_fail (status != NULL, FALSE);
  
  return GIMP_BUFFER_CLASS(buffer->klass)->query (buffer, portion, status);
}


gboolean
gimp_buffer_data (GimpBuffer     *buffer,
                  GimpPortion    *portion,
                  GimpPixelArray *array)
{
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (buffer->klass != NULL, FALSE);
  g_return_val_if_fail (GIMP_BUFFER_CLASS(buffer->klass)->data != NULL, FALSE);
  g_return_val_if_fail (portion != NULL, FALSE);
  g_return_val_if_fail (array != NULL, FALSE);
  
  return GIMP_BUFFER_CLASS(buffer->klass)->data (buffer, portion, array);
}
