/* GLE - The GTK+ Layout Editor
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"glemisc.h"
#include	"glepopups.h"
#include	<gdk/gdk.h>
#include	<string.h>
#include	<ctype.h>


/* --- functions --- */
void
gle_widget_make_sensitive (GtkWidget	*widget)
{
  if (!GTK_WIDGET_SENSITIVE (widget))
    gtk_widget_set_sensitive (widget, TRUE);
}

void
gle_widget_make_insensitive (GtkWidget	*widget)
{
  if (GTK_WIDGET_SENSITIVE (widget))
    gtk_widget_set_sensitive (widget, FALSE);
}

GtkWidget*
gle_class_show_signals (GtkObjectClass *klass,
			const gchar	*selected)
{
  GtkWidget	*window;
  GList		*item_list;
  GtkWidget	*item;
  guint		 i;
  gint		 pos;
  gchar		*message;

  g_return_val_if_fail (klass != NULL, NULL);
  g_return_val_if_fail (GTK_OBJECT_CLASS (klass) != NULL, NULL);
  if (!selected)
    selected = "";

  /* create list items
   */
  pos = -1;
  item_list = NULL;
  for (i = 0; i < klass->nsignals; i++)
    {
      gchar  *tmp_signal_name;
      
      tmp_signal_name = gtk_signal_name (klass->signals[i]);
      if (!tmp_signal_name)
	{
	  if (i == 0)
	    g_warning ("signal slot 0 empty");
	  else
	    g_warning ("signal slot %d (after \"%s\") empty",
		       i,
		       gtk_signal_name (klass->signals[i - 1]));
	  continue;
	}
      
      item = gtk_list_item_new_with_label (tmp_signal_name);
      gtk_label_get (GTK_LABEL (GTK_BIN (item)->child), &tmp_signal_name);
      gtk_object_set_user_data (GTK_OBJECT(item), tmp_signal_name);
      gtk_widget_show (GTK_WIDGET (item));
      if (g_str_equal ((gchar*) selected, tmp_signal_name) == 0)
	pos = i + 1;
      
      item_list = g_list_prepend (item_list, item);
    }
  item_list = g_list_reverse (item_list);
  if (pos != -1)
    pos = i - pos;

  /* pop up the signal_name list
   */
  message = g_strconcat (gtk_type_name (klass->type), " signals:", NULL);
  window = gle_popup_simple_list (NULL,
				  "GLE Signal List",
				  NULL,
				  NULL,
				  NULL,
				  FALSE,
				  item_list,
				  pos,
				  NULL,
				  TRUE,
				  message);
  g_free (message);

  return window;
}

static struct
{
  gchar *name;
  guint32 value;
} gle_widget_flags[] =
{
  { "need_destroy",	GTK_NEED_DESTROY },
  { "being_destroyed",	GTK_BEING_DESTROYED },
  { "in_call",		GTK_IN_CALL },
  { "----------------",	0 },
  { "visible",		GTK_VISIBLE },
  { "mapped",		GTK_MAPPED },
  { "unmapped",		GTK_UNMAPPED },
  { "realized",		GTK_REALIZED },
  { "sensitive",	GTK_SENSITIVE },
  { "parent_sensitive",	GTK_PARENT_SENSITIVE },
  { "no_window",	GTK_NO_WINDOW },
  { "has_focus",	GTK_HAS_FOCUS },
  { "can_focus",	GTK_CAN_FOCUS },
  { "has_default",	GTK_HAS_DEFAULT },
  { "can_default",	GTK_CAN_DEFAULT },
  { "propagate_state",	GTK_PROPAGATE_STATE },
  { "anchored",		GTK_ANCHORED },
  { "basic",		GTK_BASIC },
  { "user_style",	GTK_USER_STYLE },
  { "has_grab",		GTK_HAS_GRAB },
  { "redraw_pending",	GTK_REDRAW_PENDING },
  { "resize_pending",	GTK_RESIZE_PENDING },
  { "resize_needed",	GTK_RESIZE_NEEDED },
  { "has_shape_mask",	GTK_HAS_SHAPE_MASK },
};
static guint	gle_n_widget_flag_names = sizeof (gle_widget_flags) / sizeof (gle_widget_flags[0]);


GtkWidget*
gle_widget_show_flags (GtkWidget        *widget)
{
  GtkWidget	*window;
  GList		*item_list;
  GtkWidget	*item;
  guint		 i;
  gint		 pos;
  gchar		*message;

  g_return_val_if_fail (widget != NULL, NULL);
  g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);

  /* create list items
   */
  pos = -1;
  item_list = NULL;
  for (i = 0; i < gle_n_widget_flag_names; i++)
    {
      gchar  *string;

      string = g_strconcat (" ", gle_widget_flags[i].name, NULL);
      if (! (GTK_OBJECT (widget)->flags & gle_widget_flags[i].value))
	string[0] = '-';
      else
	string[0] = '+';
      item = gtk_list_item_new_with_label (string);
      g_free (string);
      gtk_label_get (GTK_LABEL (GTK_BIN (item)->child), &string);
      gtk_object_set_user_data (GTK_OBJECT(item), string);
      gtk_widget_show (GTK_WIDGET (item));
      
      item_list = g_list_prepend (item_list, item);
    }
  item_list = g_list_reverse (item_list);
  if (pos != -1)
    pos = i - pos;

  /* pop up the flag list
   */
  message = g_strconcat (gtk_type_name (GTK_OBJECT (widget)->klass->type),
			 "\"",
			 widget->name ? widget->name : "",
			 "\"",
			 " flags:",
			 NULL);
  window = gle_popup_simple_list (NULL,
				  "GLE Flag List",
				  NULL,
				  NULL,
				  NULL,
				  FALSE,
				  item_list,
				  pos,
				  NULL,
				  TRUE,
				  message);
  g_free (message);

  return window;
}

static GtkWidget *gle_flash_widget = NULL;

static gint
gle_flash_timer (gpointer data)
{
  static gboolean toggle = FALSE;
  GtkWidget *widget;
  GdkGC *gc;

  toggle = !toggle;

  widget = gle_flash_widget;
  if (!widget || !widget->window)
    return TRUE;

  if (toggle)
    gc = gtk_widget_get_default_style ()->white_gc;
  else
    gc = gtk_widget_get_default_style ()->black_gc;

  gdk_draw_rectangle	  (widget->window,
			   gc,
			   FALSE,
			   widget->allocation.x,
			   widget->allocation.y,
			   widget->allocation.width-1,
			   widget->allocation.height-1);
  gdk_draw_rectangle	  (widget->window,
			   gc,
			   FALSE,
			   widget->allocation.x,
			   widget->allocation.y,
			   widget->allocation.width,
			   widget->allocation.height);
  gdk_draw_line	  (widget->window,
		   gc,
		   widget->allocation.x,
		   widget->allocation.y,
		   widget->allocation.x +
		   widget->allocation.width,
		   widget->allocation.y +
		   widget->allocation.height);
  gdk_draw_line	  (widget->window,
		   gc,
		   widget->allocation.x +
		   widget->allocation.width,
		   widget->allocation.y,
		   widget->allocation.x,
		   widget->allocation.y +
		   widget->allocation.height);
  

  return TRUE;
}

static void
gle_expose_widget (GtkWidget *widget)
{
  if (widget->window)
    gdk_window_clear_area_e (widget->window, 0, 0, 0, 0);
}

void
gle_set_flash_widget (GtkWidget *widget)
{
  static gint flash_timer = 0;
  static gint flash_widget_handler1;
  static gint flash_widget_handler2;

  if (widget)
    g_return_if_fail (GTK_IS_WIDGET (widget));

  if (!flash_timer && widget)
    flash_timer = gtk_timeout_add (GLE_FLASH_MSEC, gle_flash_timer, NULL);

  if (flash_timer && !widget)
    {
      gtk_timeout_remove (flash_timer);
      flash_timer = 0;
    }

  if (gle_flash_widget)
    {
      gtk_signal_disconnect (GTK_OBJECT (gle_flash_widget), flash_widget_handler1);
      gtk_signal_disconnect (GTK_OBJECT (gle_flash_widget), flash_widget_handler2);
      gle_expose_widget (gle_flash_widget);
    }

  gle_flash_widget = widget;
  if (gle_flash_widget)
    {
      flash_widget_handler1 = gtk_signal_connect_object (GTK_OBJECT (gle_flash_widget),
							 "destroy",
							 GTK_SIGNAL_FUNC (gle_set_flash_widget),
							 NULL);
      flash_widget_handler2 = gtk_signal_connect_object (GTK_OBJECT (gle_flash_widget),
							 "size_allocate",
							 GTK_SIGNAL_FUNC (gle_expose_widget),
							 GTK_OBJECT (gle_flash_widget));
    }
}

GtkWidget*
gle_get_flash_widget (void)
{
  return gle_flash_widget;
}

gint
gtk_window_activate_focus (GtkWindow	  *window)
{
  g_return_val_if_fail (window != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_WINDOW (window), FALSE);

  if (window->focus_widget)
    {
      gtk_widget_activate (window->focus_widget);
      return TRUE;
    }
  return FALSE;
}

gint
gtk_window_activate_default (GtkWindow	    *window)
{
  g_return_val_if_fail (window != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_WINDOW (window), FALSE);

  if (window->default_widget)
    {
      gtk_widget_activate (window->default_widget);
      return TRUE;
    }
  return FALSE;
}

static	gint	destroy_handler = 0;
static GList	*destroy_queue = NULL;

static gint
gtk_delayed_destroy_handler (gpointer data)
{
  GList *list;

  list = destroy_queue;
  while (list)
    {
      GtkWidget *widget;

      widget = list->data;
      list = list->next;

      if (!GTK_OBJECT_NEED_DESTROY (widget))
	gtk_widget_destroy (widget);
      destroy_queue = g_list_remove (destroy_queue, widget);
    }

  destroy_handler = 0;
  return FALSE;
}

void
gtk_widget_queue_destroy (GtkWidget      *widget)
{
  if (!GTK_OBJECT_NEED_DESTROY (widget) && !g_list_find (destroy_queue, widget))
    {
      if (!destroy_handler)
	destroy_handler = gtk_idle_add (gtk_delayed_destroy_handler, NULL);
      destroy_queue = g_list_prepend (destroy_queue, widget);
    }
}

void
gtk_widget_destroyed (GtkWidget	     *widget,
		      GtkWidget	     **widget_pointer)
{
  if (widget_pointer)
    *widget_pointer = NULL;
}

GList*
gtk_widget_query_arg_list (GtkWidget *widget,
			   GtkType    class_type)
{
  GtkArg* args;
  gint nargs;
  GList *list;
  guint i;

  nargs = 0;
  args = gtk_object_query_args	 (class_type, &nargs);
  gtk_widget_set (widget, "GtkObject::signal::destroy", g_free, args, NULL);

  list = NULL;
  for (i = nargs; i > 0; i--)
    {
      list = g_list_prepend (list, &args[i - 1]);
    }

  return list;
}

typedef struct _GtkDisconnectInfo	GtkDisconnectInfo;
struct _GtkDisconnectInfo
{
  GtkObject	*object1;
  gint		disconnect_handler1;
  gint		signal_handler;
  GtkObject	*object2;
  gint		disconnect_handler2;
};

static gint
gtk_alive_disconnecter (GtkDisconnectInfo *info)
{
  g_return_val_if_fail (info != NULL, 0);

  gtk_signal_disconnect (info->object1, info->disconnect_handler1);
  gtk_signal_disconnect (info->object1, info->signal_handler);
  gtk_signal_disconnect (info->object2, info->disconnect_handler2);
  g_free (info);

  return 0;
}

void
gtk_signal_connect_object_while_alive (GtkObject	*object,
				       const gchar	*signal,
				       GtkSignalFunc	 func,
				       GtkObject	*alive_object)
{
  GtkDisconnectInfo *info;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_OBJECT (object));
  g_return_if_fail (signal != NULL);
  g_return_if_fail (func != NULL);
  g_return_if_fail (alive_object != NULL);
  g_return_if_fail (GTK_IS_OBJECT (alive_object));

  info = g_new (GtkDisconnectInfo, 1);
  info->object1 = object;
  info->object2 = alive_object;

  info->signal_handler = gtk_signal_connect_object (object, signal, func, alive_object);
  info->disconnect_handler1 = gtk_signal_connect_object (info->object1,
							 "destroy",
							 GTK_SIGNAL_FUNC (gtk_alive_disconnecter),
							 (GtkObject*) info);
  info->disconnect_handler2 = gtk_signal_connect_object (info->object2,
							 "destroy",
							 GTK_SIGNAL_FUNC (gtk_alive_disconnecter),
							 (GtkObject*) info);
}

void
gtk_adjustment_set_value (GtkAdjustment	       *adjustment,
			  gfloat		value)
{
  g_return_if_fail (adjustment != NULL);
  g_return_if_fail (GTK_IS_ADJUSTMENT (adjustment));

  adjustment->value = CLAMP (value, adjustment->lower, adjustment->upper);

  gtk_signal_emit_by_name (GTK_OBJECT (adjustment), "value_changed");
}

gchar*
g_strdown (gchar  *string)
{
  register gchar *s;

  g_return_val_if_fail (string, NULL);

  s = string;

  while (*s)
    {
      *s = tolower (*s);
      s++;
    }

  return string;
}

gchar*
g_strup (gchar	*string)
{
  register gchar *s;

  g_return_val_if_fail (string, NULL);

  s = string;

  while (*s)
    {
      *s = toupper (*s);
      s++;
    }

  return string;
}

gchar*
g_basename (const gchar	   *file_name)
{
  register gchar *base;

  g_return_val_if_fail (file_name != NULL, NULL);

  base = strrchr (file_name, '/');
  if (base)
    base++;
  else
    base = (gchar*) file_name;

  return base;
}

gchar*
g_strconvert (gchar	     *mod_string,
	      gchar	     new_delim,
	      gboolean	     downcase,
	      gboolean	     upcase)
{
  register gchar *delimiters = "_-|> <.";
  register gchar *string;

  g_return_val_if_fail (mod_string != NULL, NULL);

  string = mod_string;
  while (*string)
    {
      if (new_delim && strchr (delimiters, *string))
	*string = new_delim;
      else if (upcase)
	*string = toupper (*string);
      else if (downcase)
	*string = tolower (*string);

      string++;
    }

  return mod_string;
}
