/* Copyright (C) 1996, 1997 Aladdin Enterprises.  All rights reserved.
  
  This file is part of GNU Ghostscript.
  
  GNU Ghostscript is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
  to anyone for the consequences of using it or for whether it serves any
  particular purpose or works at all, unless he says so in writing.  Refer
  to the GNU General Public License for full details.
  
  Everyone is granted permission to copy, modify and redistribute GNU
  Ghostscript, but only under the conditions described in the GNU General
  Public License.  A copy of this license is supposed to have been given
  to you along with GNU Ghostscript so you can know your rights and
  responsibilities.  It should be in a file named COPYING.  Among other
  things, the copyright notice and this notice must be preserved on all
  copies.
  
  Aladdin Enterprises supports the work of the GNU Project, but is not
  affiliated with the Free Software Foundation or the GNU Project.  GNU
  Ghostscript, as distributed by Aladdin Enterprises, does not require any
  GNU software to build or run it.
*/

/*$Id: gdevpdfp.c $ */
/* Get/put parameters for PDF-writing driver */
#include "gx.h"
#include "gserrors.h"
#include "gdevpdfx.h"

/*
 * The pdfwrite device supports the following "real" parameters:
 *      OutputFile <string>
 *      all the Distiller parameters -- see gdevpsdp.c
 * Only some of the Distiller parameters actually have any effect.
 *
 * The device also supports the following write-only pseudo-parameters that
 * serve only to communicate other information from the PostScript file.
 * Their "value" is an array of strings, some of which may be the result
 * of converting arbitrary PostScript objects to string form.
 *      pdfmark - see gdevpdfm.c
 *      show - see gdevpdft.c
 */

private const int CoreDistVersion = 3000;	/* Distiller 3.0 */

/* ---------------- Get parameters ---------------- */

/* Get parameters. */
int
gdev_pdf_get_params(gx_device * dev, gs_param_list * plist)
{
    gx_device_pdf *pdev = (gx_device_pdf *) dev;
    int code = gdev_psdf_get_params(dev, plist);

    if (code < 0 ||
	(code = param_write_float(plist, "CompatibilityLevel",
				  &pdev->CompatibilityLevel)) < 0 ||
	(code = param_write_int(plist, "CoreDistVersion",
				(int *)&CoreDistVersion)) < 0 ||
	(code = param_write_bool(plist, "ReAssignCharacters",
				 &pdev->ReAssignCharacters)) < 0 ||
	(code = param_write_bool(plist, "ReEncodeCharacters",
				 &pdev->ReEncodeCharacters)) < 0 ||
	(code = param_write_long(plist, "FirstObjectNumber",
				 &pdev->FirstObjectNumber)) < 0
	);
    return code;
}

/* ---------------- Put parameters ---------------- */

/* Put parameters. */
int
gdev_pdf_put_params(gx_device * dev, gs_param_list * plist)
{
    gx_device_pdf *pdev = (gx_device_pdf *) dev;
    int ecode = 0;
    int code;
    float cl = pdev->CompatibilityLevel;
    bool rac = pdev->ReAssignCharacters;
    bool rec = pdev->ReEncodeCharacters;
    long fon = pdev->FirstObjectNumber;
    gs_param_name param_name;
    psdf_version save_version = pdev->version;

    /*
     * If this is one of the pseudo-parameters (show or pdfmark),
     * don't bother checking for any real ones.
     */

    {
	gs_param_string pps;

	code = param_read_string(plist, (param_name = "show"), &pps);
	switch (code) {
	    case 0:
		pdf_open_document(pdev);
		code = pdfshow_process(pdev, plist, &pps);
		if (code >= 0)
		    return code;
		/* falls through for errors */
	    default:
		param_signal_error(plist, param_name, code);
		return code;
	    case 1:
		break;
	}
    }

    {
	gs_param_string_array ppa;

	code = param_read_string_array(plist, (param_name = "pdfmark"), &ppa);
	switch (code) {
	    case 0:
		pdf_open_document(pdev);
		code = pdfmark_process(pdev, &ppa);
		if (code >= 0)
		    return code;
		/* falls through for errors */
	    default:
		param_signal_error(plist, param_name, code);
		return code;
	    case 1:
		break;
	}
    }

    /* General parameters. */

    switch (code = param_read_float(plist, (param_name = "CompatibilityLevel"), &cl)) {
	default:
	    ecode = code;
	    param_signal_error(plist, param_name, ecode);
	case 0:
	case 1:
	    break;
    }

    {
	int cdv = CoreDistVersion;

	ecode = psdf_put_int_param(plist, (param_name = "CoreDistVersion"), &cdv, ecode);
	if (cdv != CoreDistVersion)
	    param_signal_error(plist, param_name, ecode = gs_error_rangecheck);
    }

    ecode = psdf_put_bool_param(plist, "ReAssignCharacters", &rac, ecode);
    ecode = psdf_put_bool_param(plist, "ReEncodeCharacters", &rec, ecode);
    switch (code = param_read_long(plist, (param_name = "FirstObjectNumber"), &fon)) {
	case 0:
	    /*
	     * Setting this parameter is only legal if the file
	     * has just been opened and nothing has been written,
	     * or if we are setting it to the same value.
	     */
	    if (fon == pdev->FirstObjectNumber)
		break;
	    if (fon <= 0 || fon > 0x7fff0000 ||
		(pdev->next_id != 0 &&
		 pdev->next_id !=
		 pdev->FirstObjectNumber + pdf_num_initial_ids)
		)
		ecode = gs_error_rangecheck;
	    else
		break;
	default:
	    ecode = code;
	    param_signal_error(plist, param_name, ecode);
	case 1:
	    break;
    }

    if (ecode < 0)
	return ecode;
    /*
     * We have to set version to the new value, because the set of
     * legal parameter values for psdf_put_params varies according to
     * the version.
     */
    pdev->version =
	(cl < 1.2 ? psdf_version_level2 : psdf_version_ll3);
    code = gdev_psdf_put_params(dev, plist);
    if (code < 0) {
	pdev->version = save_version;
	return code;
    }
    pdev->CompatibilityLevel = cl;
    pdev->ReAssignCharacters = rac;
    pdev->ReEncodeCharacters = rec;
    if (fon != pdev->FirstObjectNumber) {
	pdev->FirstObjectNumber = fon;
	if (pdev->tfile != 0) {
	    fseek(pdev->tfile, 0L, SEEK_SET);
	    pdf_initialize_ids(pdev);
	}
    }
    return 0;
}
