#
# Copyright 2005 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

import math
from gnuradio import gr, optfir
from gnuradio.blksimpl.fm_emph import fm_deemph

class nbfm_rx(gr.hier_block):
    def __init__(self, fg, audio_rate, quad_rate, tau=75e-6, max_dev=5e3):
        """
        Narrow Band FM Receiver.

        Takes a single complex baseband input stream and produces a single
        float output stream of audio sample in the range [-1, +1].

        @param fg: flow graph
        @param audio_rate: sample rate of audio stream, >= 16k
        @type audio_rate: integer
        @param quad_rate: sample rate of output stream
        @type quad_rate: integer
        @param tau: preemphasis time constant (default 75e-6)
        @type tau: float
        @param max_dev: maximum deviation in Hz (default 5e3)
        @type max_dev: float

        quad_rate must be an integer multiple of audio_rate.

        Exported sub-blocks (attributes):
          squelch
          quad_demod
          deemph
          audio_filter
        """

        # FIXME audio_rate and quad_rate ought to be exact rationals
        audio_rate = int(audio_rate)
        quad_rate = int(quad_rate)

        if quad_rate % audio_rate != 0:
            raise ValueError, "quad_rate is not an integer multiple of audio_rate"

        squelch_threshold = 20		# dB
        self.squelch = gr.simple_squelch_cc(squelch_threshold, 0.001)

        # FM Demodulator  input: complex; output: float
        k = quad_rate/(2*math.pi*max_dev)
        self.quad_demod = gr.quadrature_demod_cf(k)

        # FM Deemphasis IIR filter
        self.deemph = fm_deemph (fg, quad_rate, tau=tau)

        # compute FIR taps for audio filter
        audio_decim = quad_rate // audio_rate
        audio_taps = gr.firdes.low_pass (1.0,            # gain
                                         quad_rate,      # sampling rate
                                         4.5e3,          # Audio LPF cutoff
                                         2.5e3,          # Transition band
                                         gr.firdes.WIN_HAMMING)  # filter type

        print "len(audio_taps) =", len(audio_taps)

        # Decimating audio filter
        # input: float; output: float; taps: float
        self.audio_filter = gr.fir_filter_fff(audio_decim, audio_taps)

        # wire the block together
        fg.connect(self.squelch, self.quad_demod, self.deemph, self.audio_filter)
        gr.hier_block.__init__(self, fg, self.squelch, self.audio_filter)
