;;; xemacsbug.el --- command to report XEmacs bugs to appropriate mailing list.

;; Copyright (C) 2001 Free Software Foundation, Inc.
;; Copyright (C) 2001 Steve Youngs <youngs@xemacs.org>

;; Author: Steve Youngs <youngs@xemacs.org>
;; Based on 'emacsbug.el' by: K. Shane Hartman

;; Maintainer: XEmacs Development Team
;; Keywords: maint mail bugs

;; Not fully installed because it can work only on Internet hosts.
;; This file is part of XEmacs.

;; XEmacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XEmacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Synched up with: Not synched.

;;; Commentary:

;; `M-x report-xemacs-bug ' starts an email note to the XEmacs maintainers
;; describing a problem.  Here's how it's done...

;;; Code:

(require 'sendmail)

(defgroup xemacsbug nil
  "Sending XEmacs bug reports."
  :group 'maint
  :group 'mail)

;; >> These should be addresses which are accessible to your machine,
;; >> otherwise you can't use this file.  It will only work on the
;; >> internet with this address.

(defcustom report-xemacs-bug-address "xemacs-beta@xemacs.org"
  "*Address of mailing list for XEmacs bugs."
  :group 'xemacsbug
  :type 'string)

(defcustom report-xemacs-bug-beta-address "xemacs-beta@xemacs.org"
  "*Address of mailing list for XEmacs beta bugs."
  :group 'xemacsbug
  :type 'string)

(defvar report-xemacs-bug-orig-text nil
  "The automatically-created initial text of bug report.")

(defcustom report-xemacs-bug-no-confirmation nil
  "*If non-nil, suppress the confirmations asked for the sake of novice users."
  :group 'xemacsbug
  :type 'boolean)

(defcustom report-xemacs-bug-no-explanations nil
  "*If non-nil, suppress the explanations given for the sake of novice users."
  :group 'xemacsbug
  :type 'boolean)

(defcustom report-xemacs-bug-send-init nil
  "*If non-nil, include the user's init.el file in the bug report."
  :group 'xemacsbug
  :type 'boolean)

;;;###autoload
(defun report-xemacs-bug (topic &optional recent-keys)
  "Report a bug in XEmacs.
Prompts for bug subject.  Leaves you in a mail buffer."
  ;; This strange form ensures that (recent-keys) is the value before
  ;; the bug subject string is read.
  (interactive (reverse (list (recent-keys) (read-string "Bug Subject: "))))
  (let (user-point)
    (compose-mail (if (string-match "(beta[0-9]+)" emacs-version)
		      report-xemacs-bug-beta-address
		    report-xemacs-bug-address)
		  topic)
    ;; The rest of this does not execute
    ;; if the user was asked to confirm and said no.
    (goto-char (point-min))
    (re-search-forward (concat "^" (regexp-quote mail-header-separator) "$"))
    (forward-line 1)
    (unless report-xemacs-bug-no-explanations
      ;; Insert warnings for novice users.
      (insert "This bug report will be sent to the XEmacs Development Team,\n")
      (insert-face " not to your local site managers!!" 'italic)
      (insert "\nPlease write in ")
      (insert-face "English" 'italic)
      (insert ", because the XEmacs maintainers do not have
translators to read other languages for them.\n\n"))
    (insert "\n")
    (insert "Please describe as succinctly as possible:
\t- What happened.
\t- What you thought should have happened.
\t- Precisely what you were doing at the time.\n
Please also include any C or lisp back-traces that you may have.
================================================================
Dear Bug Team!\n\n")
    (setq user-point (point))
    (insert "\n\n================================================================\n
System Info to help track down your bug:
---------------------------------------\n\n")
    ;; Insert the output of 'describe-installation'.
    (insert (symbol-value 'Installation-string))
    ;; Insert a list of installed packages.
    (insert "\n\nInstalled XEmacs Packages:\n"
	    "-------------------------\n")
    (cl-prettyprint
     (symbol-value 'packages-package-list))
    (insert "\n")
    ;; Insert a list of loaded features
    (let ((before-features (point)))
      (insert
       (format "\n\nFeatures:\n--------\n\n%s" (symbol-value 'features)))
      (save-restriction
	(narrow-to-region before-features (point))
	(fill-paragraph t)
	(insert "\n")))
    ;; Insert recent keystrokes.
    (insert "\n\n"
	    "Recent keystrokes:\n-----------------\n\n")
    (let ((before-keys (point)))
      (insert (key-description recent-keys))
      (save-restriction
	(narrow-to-region before-keys (point))
	(goto-char before-keys)
	(while (progn (move-to-column 50) (not (eobp)))
	  (search-forward " " nil t)
	  (insert "\n"))))
    ;; Insert recent minibuffer messages.
    (insert "\n\n\nRecent messages (most recent first):\n"
	    "-----------------------------------\n")
    (let ((standard-output (current-buffer)))
      (print-recent-messages 20)
      (insert "\n"))
    ;; Insert the contents of the user's init file if it exists.
    (if report-xemacs-bug-send-init
      (if (file-readable-p user-init-file)
	  (save-excursion
	    (goto-char (point-max))
	    (beginning-of-line)
	    (insert "\n\nUser Init File:\n--------------\n\n")
	    (insert-file-contents user-init-file))))
    ;; This is so the user has to type something
    ;; in order to send easily.
    (use-local-map (let ((map (make-sparse-keymap)))
		     (set-keymap-parents map (current-local-map))
		     map))
    (define-key (current-local-map) "\C-c\C-i" 'report-xemacs-bug-info)
    (unless report-xemacs-bug-no-explanations
      (with-output-to-temp-buffer "*Bug Help*"
	(if (eq mail-user-agent 'sendmail-user-agent)
	    (princ (substitute-command-keys
		    "Type \\[mail-send-and-exit] to send the bug report.\n"))
	  (princ (substitute-command-keys
		  "Type \\[message-send-and-exit] to send the bug report.\n")))
	(princ (substitute-command-keys
		"Type \\[kill-buffer] RET to cancel (don't send it).\n"))
	(terpri)
	(princ (substitute-command-keys
		"\
Type \\[report-xemacs-bug-info] to visit in Info the XEmacs Manual section
about when and how to write a bug report,
and what information to supply so that the bug can be fixed.
Type SPC to scroll through this section and its subsections."))
	(princ "\n\n
You are very welcome to scan through the bug report and remove any
potentially sensitive data.")))
    ;; Make it less likely people will send empty messages.
    (make-local-variable 'mail-send-hook)
    (add-hook 'mail-send-hook 'report-emacs-bug-hook)
    (save-excursion
      (goto-char (point-max))
      (skip-chars-backward " \t\n")
      (make-local-variable 'report-xemacs-bug-orig-text)
      (setq report-xemacs-bug-orig-text (buffer-substring (point-min) (point))))
    (goto-char user-point)))

;; For backwards compatibility
(defalias 'report-emacs-bug 'report-xemacs-bug)

(defun report-xemacs-bug-info ()
  "Go to the Info node on reporting Emacs bugs."
  (interactive)
  (Info-goto-node "(xemacs)Bugs"))

(defun report-xemacs-bug-hook ()
  (save-excursion
    (goto-char (point-max))
    (skip-chars-backward " \t\n")
    (if (and (= (- (point) (point-min))
		(length report-xemacs-bug-orig-text))
	     (equal (buffer-substring (point-min) (point))
		    report-xemacs-bug-orig-text))
	(error "No text entered in bug report"))

    ;; Check the buffer contents and reject non-English letters.
    (save-excursion
      (goto-char (point-min))
      (skip-chars-forward "\0-\177")
      (if (not (eobp))
	  (if (or report-xemacs-bug-no-confirmation
		  (y-or-n-p "Convert non-ASCII letters to hexadecimal? "))
	      (while (progn (skip-chars-forward "\0-\177")
			    (not (eobp)))
		(let ((ch (following-char)))
		  (delete-char 1)
		  (insert (format "=%02x" ch)))))))

    ;; The last warning for novice users.
    (if (or report-xemacs-bug-no-confirmation
	    (yes-or-no-p
	     "Send this bug report to the XEmacs maintainers? "))
	;; Just send the current mail.
	nil
      (goto-char (point-min))
      (if (search-forward "To: ")
	  (let ((pos (point)))
	    (end-of-line)
	    (delete-region pos (point))))
      (kill-local-variable 'mail-send-hook)
      (with-output-to-temp-buffer "*Bug Help*"
	(princ (substitute-command-keys "\
You invoked the command M-x report-xemacs-bug,
but you decided not to mail the bug report to the XEmacs maintainers.

If you want to mail it to someone else instead,
please insert the proper e-mail address after \"To: \",
and send the mail again using \\[mail-send-and-exit].")))
      (error "M-x report-xemacs-bug was cancelled, please read *Bug Help* buffer"))))

(provide 'xemacsbug)

;;; xemacsbug.el ends here
