;;; patcher.el --- Utility for mailing patch information

;; Copyright (C) 1999-2001 Didier Verna.

;; Author:        Didier Verna <didier@xemacs.org>
;; Maintainer:    Didier Verna <didier@xemacs.org>
;; Created:       Tue Sep 28 18:12:43 1999
;; Last Revision: Wed Apr 25 14:57:40 2001
;; Keywords:      maint

;; This file is part of Patcher.

;; Patcher is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; Patcher is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


;;; Commentary:

;; Contents management by FCM version 0.1.

;; Many thanks to Ben Wing <ben@xemacs.org> for good ideas and code.

;; This package automates the process of building and submitting patches for
;; archive-based projects you're working on.  In one or two keystrokes, it
;; prepares a mail with a patch corresponding to the differences between your
;; working version and the archive state, and prepares a skeleton for the
;; ChangeLog entries, that you can fill in and insert into the message before
;; sending.  You also have the possibility of committing your changes directly
;; from your XEmacs session.

;; Patcher currently supports the `compose-mail' mail sending interface, as
;; well as the `sendmail' and `message' libraries individually, which is
;; probably redundant. There is also a special support for sending mail from a
;; running Gnus session.

;; Here's a typical usage:

;; 0/ After loading the package, customize the `patcher' group to suit your
;;    needs.
;; 1/ Once you have made modifications to one of your projects and you'd like
;;    to submit them, type `M-x patcher-mail' (see the docstring).  The
;;    message will be prepared and the ChangeLog skeletons built.
;;    Alternately, use `M-x patcher-mail-subproject' to mail (and possibly
;;    commit) changes to particular directories and/or files in the project.
;; 2/ Edit the different ChangeLog buffers that have opened, and fill in the
;;    skeletons.  You can save the ChangeLogs buffers, BUT DON't KILL THEM !!
;; 3/ If you want to commit your changes immediately, type `C-c C-p c' in
;;    the message buffer.  You'll have the opportunity to edit the message log
;;    and/or the commit command, and finally commit your changes.
;; 4/ Otherwise, or just after that, type `C-c C-p i' in the message buffer.
;;    The new ChangeLog entries will be inserted just above the patch.
;; 5/ At any time after step 1, you can change your mind (like further modify
;;    source files) and regenerate the diff again. To do that, just type
;;    `C-c C-p d' in the message buffer.  If ChangeLog skeletons had already
;;    been created, and possibly edited, you'll be asked whether you want to
;;    regenerate them also. The diff regeneration feature is also usefull when
;;    the initial diff command failed for some reason.
;; 6/ That's all folks.  You can send the message.

;; Requirements:

;; This package requires a recent `add-log' library (from the xemacs-base
;; package version 1.48 or later), and the `sendmail' library.  The different
;; mail methods will also have specific requirements (especially the Gnus
;; one).  Please, refer to them for further information.

;; Suggestions for further improvements:

;; #### Consider doing a better job of handling overlapping patcher
;; instances.  e.g. use a different extent property for each instance,
;; and keep a usage count for the ChangeLog files so that they're only
;; killed when no instance uses them any more.
;;
;; #### Also add an option to kill, not just bury, the mail message when
;; it's sent.
;;
;; #### When the message is sent, the cvs commit results window should be
;; removed and the corresponding buffer buried.
;;
;; #### Have a control window to clarify the progress of everything.
;; Make it perhaps a small buffer, above the mail message/ChangeLog
;; buffer.  It shows what steps have been completed, what haven't, and
;; what to do.  It should have buttons in it for the various actions.
;; One button is for include-changelogs, one for commit, one for send
;; the mail, and one for execute the commit.  These should be enabled
;; or grayed out appropriately.  It should also have buttons that show
;; the associated changelogs; clicking on a button puts that changelog
;; buffer in the main window below the control window.  By each
;; changelog button should be an indication of whether the changelog
;; has been modified so far.  The control window should stay around as
;; much as it can during the whole process (e.g. use
;; set-buffer-dedicated-p or something), so that it's always there to
;; drive the whole process.  One corollary is that you don't actually
;; have to switch to the mail buffer to (e.g.) execute
;; include-changelogs -- you just click on the control window, and it
;; does it automatically.  also, when you execute include-changelogs,
;; it can issue a warning if not all changelogs have been visited, and
;; prompt you to make sure you want to continue.  Similarly, you can
;; run include-changelogs more than once even if it succeeded the
;; first time (e.g. for some reason you didn't like the result and
;; deleted it), but it will prompt "are you sure?".  there could also
;; be an "undo include-changelogs", if you make a mistake after doing
;; include-changelogs and realize you want to go back and fix the
;; problem and do include-changelogs again.


;;; Code:

(require 'add-log)
;; Require 'sendmail for getting `mail-header-separator'.
;; #### Now that a fake mail sending function exists, sendmail shoudn't be
;; systematically required like this. However, since most users will really
;; want do send real messasges, it probably doesn't hurt to keep things like
;; this.
(require 'sendmail)

(defgroup patcher nil
  "Patch mailing utilities.")

(defmacro patcher-globally-declare-fboundp (symbol)
  ;; copied from bytecomp-runtime.el
  (when (cl-compiling-file)
    (setq symbol (eval symbol))
    (if (not (consp symbol))
	(setq symbol (list symbol)))
    ;; Another hack.  This works because the autoload environment is
    ;; currently used ONLY to suppress warnings, and the actual
    ;; autoload definition is not used. (NOTE: With this definition,
    ;; we will get spurious "multiple autoloads for %s" warnings if we
    ;; have an autoload later in the file for any functions in SYMBOL.
    ;; This is not something that code should ever do, though.)
    (setq byte-compile-autoload-environment
	  (append (mapcar #'(lambda (sym) (cons sym nil)) symbol)
		  byte-compile-autoload-environment)))
  nil)

(patcher-globally-declare-fboundp
 '(gnus-alive-p gnus-post-news message-goto-subject message-goto-body))


;; Projects description =====================================================

(defgroup patcher-default nil
  "Default settings for Patcher projects."
  :group 'patcher)

(defcustom patcher-default-mail-method 'compose-mail
  "*Default method used by Patcher to prepare a mail \(a symbol).
Currently, there are four built-in methods: 'compose-mail \(the default),
'sendmail, 'message, 'gnus and 'fake.  Please refer to the corresponding
`patcher-mail-*' function for a description of each method.  You can also
define your own method, say `foo'.  In that case, you *must* provide a
function named `patcher-mail-foo' which takes two arguments: a project
descriptor and a string containing the subject of the message.  This
function must prepare a mail buffer.  If you want to do this, please see
how it's done for the built-in methods."
  :group 'patcher-default
  :type '(radio (const :value compose-mail)
		(const :value sendmail)
		(const :value message)
		(const :value gnus)
		(const :value fake)
		(symbol :tag "other")))

(defcustom patcher-default-subject ""
  "*Default subject to use when sending a Patcher mail \(a string).
A %n occurring in this string will be replaced by the project's name.
All subjects are unconditionally prefixed with the string \"[PATCH] \"."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-mail-prologue ""
  "*Default string to insert at the beginning of every Patcher mail."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-diff-command "cvs -q diff -u %f"
  "*Default method used by Patcher to generate a patch \(a string).
A %f occurring in this string will be replaced by the files affected by
the patch.  These files can be specified by using `patcher-mail-subproject'
instead of `patcher-mail' to prepare the patch.  Otherwise, the %f will
simply be removed."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-diff-prologue "%n Patch (%c):"
  "*Default string to insert just before the patch in every Patcher mails.
A %n occurring in this string will be replaced with the project's name.
A %c occurring in this string will be replaced with the diff command."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-commit-command "cvs commit -F %s %f"
  "*Default method used by Patcher to commit a patch \(a string).
This command must contain a %s which will be replaced with the name of a
file containing the commit message log.
A %f occurring in the command will be replaced with the files affected by
the patch.  These files can be specified by using `patcher-mail-subproject'
instead of `patcher-mail' to prepare the patch.  Otherwise, the %f will
simply be removed."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-confirm-commits t
  "*Whether Patcher asks for a confirmation before doing a commit."
  :group 'patcher-default
  :type 'boolean)

(defcustom patcher-default-committed-notice
  "NOTE: This patch has been committed."
  "*Notice added to a mail when the patch is committed before sending."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-failed-command-regexp "^cvs \\[[^]]* aborted\\]"
  "*Default regular expression for matching the result of a failed command.
Commands in question are the diff and the commit one."
  :group 'patcher-default
  :type 'regexp)

(defcustom patcher-default-init-log-message 'subject
  "*How to initialize the commit log message \(a symbol).
The values currently supported are:
- 'subject \(the default): use the subject of the mail.
- 'change-logs: use the ChangeLog entries.
- 'compressed-change-logs: use the ChangeLog entries, but compress them
   into something more suitable as a log message.
- 'compressed-change-logs-with-original: insert the compressed ChangeLog
   entries, as above, but also append the original entries at the end for
   easy reference.  You should delete the originals before committing.
- nil: don't initialize the log message.

If at commit time, the log message is empty, \"\(none)\" will be used."
  :group 'patcher-default
  :type '(radio (const :value subject)
		(const :value change-logs)
		(const :value compressed-change-logs)
		(const :value compressed-change-logs-with-original)
		(const :value nil :tag "nothing")))

(defcustom patcher-default-edit-log-message t
  "*Whether Patcher lets you edit the commit log message.
If nil, Patcher will directly use the initialization value \(see
`patcher-default-init-log-message')."
  :group 'patcher-default
  :type 'boolean)

(defcustom patcher-default-kill-source-files-after-sending t
  "*Whether to kill source files after sending the mail.
This is effective only when sources files have not been killed already
\(see the variable `patcher-default-kill-source-files-after-diffing').

That feature is not implemented yet."
  :group 'patcher-default
  :type 'boolean)

(defcustom patcher-default-kill-change-logs-after-sending t
  "*Whether to kill the ChangeLog files after sending the mail."
  :group 'patcher-default
  :type 'boolean)

(defcustom patcher-default-kill-source-files-after-diffing t
  "*Whether to kill source files after building the ChangeLog skeletons.
These files are loaded temporarily by `patch-to-change-log'.  If this
variable is non nil, `patch-to-change-log' will be instructed to remove
them when they are not needed anymore.
See also the variable `patcher-default-kill-source-files-after-sending'."
  :group 'patcher-default
  :type 'boolean)

(defcustom patcher-default-to-address "xemacs-patches@xemacs.org"
  "*Default email address to use when sending a Patcher mail (a string).
This variable is used by all mail methods except the 'gnus one \(see
`patcher-default-mail-method').  If not defined, it is prompted for."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-gnus-group ""
  "*Default Gnus group to use when sending a Patcher mail (a string).
This variable is used only in the 'gnus mail method \(see
`patcher-default-mail-method').  The mail sending process will behave as if
you had type `C-u a' in the group buffer on that Gnus group.  If not
defined, it is prompted for."
  :group 'patcher-default
  :type 'string)

;; Defining these const avoids coding special cases for the :inheritance,
;; :subdirectory and :files (sub)project option in the accessor functions.
(defconst patcher-default-inheritance nil)
(defconst patcher-default-subdirectory nil)
(defconst patcher-default-files nil)

(defconst patcher-project-options-custom-type
  '((list :inline t :tag "Mail method"
	  :format "%{%t%}: %v"
	  (const :tag "" :value :mail-method)
	  (choice (const :value compose-mail)
		  (const :value sendmail)
		  (const :value message)
		  (const :value gnus)
		  (const :value fake)
		  (symbol :tag "other")))
    (list :inline t :tag "Subject"
	  :format "%{%t%}: %v"
	  (const  :tag "" :value :subject)
	  (string :tag "Value"))
    (list :inline t :tag "Mail Prologue"
	  :format "%{%t%}: %v"
	  (const :tag "" :value :mail-prologue)
	  (string :tag "Value"))
    (list :inline t :tag "Diff command"
	  :format "%{%t%}: %v"
	  (const :tag "" :value :diff-command)
	  (string :tag "Value"))
    (list :inline t :tag "Diff prologue"
	  :format "%{%t%}: %v"
	  (const :tag "" :value :diff-prologue)
	  (string :tag "Value"))
    (list :inline t :tag "Commit command"
	  :format "%{%t%}: %v"
	  (const :tag "" :value :commit-command)
	  (string :tag "Value"))
    (list :inline t :tag "Confirm commits"
	  :format "%{%t%}: %v"
	  (const :tag "" :value :confirm-commits)
	  (boolean :tag "Value"))
    (list :inline t :tag "Committed-notice"
	  :format "%{%t%}: %v"
	  (const :tag "" :value :committed-notice)
	  (string :tag "Value"))
    (list :inline t :tag "Failed command regexp"
	  :format "%{%t%}: %v"
	  (const :tag "" :value :failed-command-regexp)
	  (regexp :tag "Value"))
    (list :inline t :tag "Init log message"
	  :format "%{%t%}: %v"
	  (const :tag "" :value :init-log-message)
	  (choice (const :value subject)
		  (const :value change-logs)
		  (const :value compressed-change-logs)
		  (const :value compressed-change-logs-with-original)))
    (list :inline t :tag "Edit log message"
	  :format "%{%t%}: %v"
	  (const :tag "" :value :edit-log-message)
	  (boolean :tag "Value"))
    (list :inline t
	  :tag "Kill source files after sending"
	  :format "%{%t%}: %v"
	  (const :tag ""
		 :value
		 :kill-source-files-after-sending)
	  (boolean :tag "Value"))
    (list :inline t
	  :tag "Kill ChangeLogs after sending"
	  :format "%{%t%}: %v"
	  (const :tag ""
		 :value
		 :kill-change-logs-after-sending)
	  (boolean :tag "Value"))
    (list :inline t
	  :tag "Kill source files after diffing"
	  :format "%{%t%}: %v"
	  (const :tag ""
		 :value
		 :kill-source-files-after-diffing)
	  (boolean :tag "Value"))
    (list :inline t :tag "To Address"
	  :format "%{%t%}: %v"
	  (const  :tag "" :value :to-address)
	  (string :tag "Value"))
    (list :inline t :tag "Gnus Group"
	  :format "%{%t%}: %v"
	  (const  :tag "" :value :gnus-group)
	  (string :tag "Value"))
    (list :inline t :tag "Other"
	  symbol
	  sexp))
  ;; Custom type elements for Patcher project options common to
  ;; `patcher-projects' and `patcher-subprojects'.
  )

(defcustom patcher-projects '()
  "*List of project descriptors used by `patcher-mail'.
Each project descriptor looks like \(NAME DIR OPTIONS...):
- NAME is the project's name \(it serves to identify the project),
- DIR  is the top level directory where the project's sources live,

The remainder of the project descriptor is composed of \"project options\"
\(keywords and associated values).  Keywords correspond to the variables
`patcher-default-*'.  When Patcher needs a keyword value, it tries to find
at different places:
- first, it looks for the keyword in the project options.
- if that fails, it tries to find it in the project options of the projects
  appearing in the `:inheritance' option of the current project.
- if that fails, it falls back to the corresponding `patcher-default-*'
variable.

As an exception, the `:inheritance' keyword does not have a corresponding
`patcher-default-inheritance' variable."
  :group 'patcher
  :type `(repeat
	  (group (string :tag "Project")
		 (directory :tag "Project directory")
		 (repeat :inline t :tag "Options"
			 (choice :inline t :value (:mail-method compose-mail)
				 ,@patcher-project-options-custom-type
				 (list :inline t :tag "Inheritance"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :inheritance)
				       (repeat :tag "From"
					       (string :tag "Project")))))
		 ))
  )

(defcustom patcher-subprojects '()
  "*List of subproject descriptors used by `patcher-mail'.
Subproject descriptors are similar to project descriptors \(see the
variable `patcher-projects') with a few exceptions:

- instead of the project directory field DIR, you specify the name of the
  project this subproject is based on.
- two project options are available in addition to the standard ones:
  - :subdirectory lets you specify a subdirectory \(of the original
     project's directory) in which the whole subproject resides.  There is
     no corresponding `patcher-default-subdirectory' variable.
  - :files lets you specify a list of files or directories composing the
     subproject. Each file specification can contain wildcards.  If a
     :subdirectory option is given, these files or directories should be
     relative to this subdirectory. Otherwise, they should be relative to
     the base project's directory.  There is no corresponding
     `patcher-default-files' variable.
  Note that a subproject with neither a :subdirectory nor a :files option
  behaves exactly like the corresponding base project.
- subprojects don't have an :inheritance mechanism. Instead, they
  implicitly inherit from their base project \(which in turn can inherit
  from other projects).

Note: the normal way to use predefined Patcher subprojects is to call
`patcher-mail', *not* `patcher-mail-subproject'.  Using the former will
directly use the set of files and/or directory you have specified.  Using
the latter will also let you modify this set."
  :group 'patcher
  :type `(repeat
	  (group (string :tag "Subproject")
		 (string :tag "Of project")
		 (repeat :inline t :tag "Options"
			 (choice :inline t :value (:subdirectory "")
				 ;; #### Look inside the widget library to see
				 ;; how we can modify the completion behavior
				 (list :inline t :tag "Subdirectory"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :subdirectory)
				       (directory :tag "Value"))
				 (list :inline t :tag "Files"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :files)
				       (repeat :format "\n%v%i\n"
					       (file :tag "File")))
				 ,@patcher-project-options-custom-type))
		 ))
  )


;; Project descriptors Accessors:
(defsubst patcher-project-name (project) (nth 0 project))

(defun patcher-project-directory (project)
  ;; Returns the project directory of PROJECT, possibly expanded as a project
  ;; subdir if PROJECT is a subproject.
  (if (member project patcher-subprojects)
      (let ((prj (assoc (nth 1 project) patcher-projects)))
	(unless prj
	  (error "Can't find base project for subproject `%s'"
		 (patcher-project-name project)))
	(let ((subdir (patcher-project-option project :subdirectory)))
	  (if subdir
	      (expand-file-name subdir (patcher-project-directory prj))
	    (patcher-project-directory prj))))
    ;; else: (member project patcher-projects)
    (nth 1 project)))

(defun patcher-project-option-1 (project option)
  ;; Try to find an option in the project descriptor, otherwise, try in each
  ;; project from the project's inheritance list.
  ;; The whole option form is returned: '(:stuff value)
  (let* ((is-subproject (member project patcher-subprojects))
	 (options (cddr project))
	 (value (member option options)))
    (unless value
      (let ((projs (cadr (member :inheritance options)))
	    proj)
	;; If PROJECT is a subproject, we use only the base project as an
	;; inheritance list.
	(when is-subproject
	    (if projs
		(warn "Option :inheritance in subproject `%s' will be unused"
		      (patcher-project-name project)))
	    (setq projs (list (nth 1 project))))
	(when projs
	  (while (and (not value) (setq proj (pop projs)))
	    (setq value (patcher-project-option-1 (assoc proj patcher-projects)
						  option))))
	))
    ;; Now some checkings.
    (when (and (eq option :files) value)
      (if is-subproject
	  ;; Return the files as a string, not as the original list.
	  (setq value (list :files (mapconcat #'identity (cadr value) " ")))
	;; Projects shouldn't have a :file option.
	(warn "Option :file in project `%s' will be unused"
	      (patcher-project-name project))
	(setq value nil)))
    value))

(defun patcher-project-option (project option)
  ;; Returns either a project's option, or the patcher-default-* value.
  (let ((opt (patcher-project-option-1 project option)))
    (if opt
	(cadr opt)
      (symbol-value
       (intern-soft
	(concat "patcher-default-" (substring (symbol-name option) 1))))
      )))


;; Version management =======================================================

;; $Format: "(defconst patcher-prcs-major-version \"$ProjectMajorVersion$\")"$
(defconst patcher-prcs-major-version "version-2-5")
;; $Format: "(defconst patcher-prcs-minor-version \"$ProjectMinorVersion$\")"$
(defconst patcher-prcs-minor-version "1")
(defconst patcher-version
  (let ((level patcher-prcs-minor-version)
	major minor status)
    (string-match "\\(branch\\|version\\)-\\([0-9]+\\)-\\([0-9]+\\)"
		  patcher-prcs-major-version)
    (setq major (match-string 2 patcher-prcs-major-version)
	  minor (match-string 3 patcher-prcs-major-version)
	  status (match-string 1 patcher-prcs-major-version))
    (cond ((string= status "version")
	   (setq level (int-to-string (1- (string-to-int level))))
	   (if (string-equal level "0")
	       (concat major "." minor)
	     (concat major "." minor "." level)))
	  ((string= status "branch")
	   (concat major "." minor "-b" level)))
    ))

;;;###autoload
(defun patcher-version ()
  "Show the current version of Patcher."
  (interactive)
  (message "Patcher version %s" patcher-version))


;; Internal variables =======================================================

;; These variables get proper values in each mail buffer.

(make-variable-buffer-local
 (defvar patcher-project nil
   ;; Patcher project related to the current patch.
   ))

(make-variable-buffer-local
 (defvar patcher-files nil
   ;; Files and/or directories concerned by the current patch.
   ;; This is set by `patcher-mail-subproject'.
   ))

(make-variable-buffer-local
 (defvar patcher-diff-command nil
   ;; Complete diff command to use for making the current patch.
   ))

(make-variable-buffer-local
 (defvar patcher-change-logs nil
   ;; List of ChangeLog file buffers concerned by the current patch.
   ))

(make-variable-buffer-local
 (defvar patcher-change-logs-marker nil
   ;; Marker indicating the beginning of the ChangeLog entries in the mail
   ;; buffer.
   ))

(make-variable-buffer-local
 (defvar patcher-patch-marker nil
   ;; Marker indicating the beginning of the patch in the mail buffer.
   ))

(make-variable-buffer-local
 (defvar patcher-commit-output-buffer nil
   ;; Buffer containing the output of the commit command.
   ))

(make-variable-buffer-local
 (defvar patcher-commit-logmsg-buffer nil
   ;; Buffer containing the log message of the commit command.
   ))

(make-variable-buffer-local
 (defvar patcher-pre-commit-window-config nil
   ;; Window configuration when we're in the mail buffer, just prior to
   ;; beginning a commit operation, so we can get back to it at the
   ;; appropriate times after prompting for log messages, displaying
   ;; subprocess output, etc.
   ))

(make-variable-buffer-local
 (defvar patcher-logmsg-file-name
   ;; Name of the temporary file where the log message is stored.
   ))


;; Internal functions =======================================================

(defun patcher-file-relative-name (file dir)
  ;; Construct a filename relative to DIR.
  (file-relative-name (expand-file-name file (expand-file-name dir))
		      (expand-file-name dir)))

(defun patcher-offer-save-buffers (buffers)
  ;; Offer to save some buffers.
  ;; #### this should be a standard function somewhere.
  (map-y-or-n-p
   (lambda (buffer)
     (and (buffer-modified-p buffer)
	  (not (buffer-base-buffer buffer))
	  (buffer-file-name buffer)
	  (format "Save file %s? "
		  (buffer-file-name buffer))))
   (lambda (buffer)
     (save-excursion
       (set-buffer buffer)
       (condition-case ()
	   (save-buffer)
	 (error nil))))
   buffers
   '("buffer" "buffers" "save")))

(defun patcher-insert-change-logs-1 (mail-buffer)
  ;; Insert ChangeLog entries in the current buffer at the current position.
  ;; ChangeLog entries are those corresponding to the patch being edited in
  ;; MAIL_BUFFER.
  (let* ((project (symbol-value-in-buffer 'patcher-project mail-buffer))
	 (directory (patcher-project-directory project))
	 (change-logs (symbol-value-in-buffer 'patcher-change-logs
					      mail-buffer)))
    (dolist (change-log change-logs)
      (map-extents
       #'(lambda (extent unused)
	   ;; Force forward slashes (for native Windows). -- APA
	   (let ((directory-sep-char ?/))
	     (message "%s %s" (extent-start-position extent)
		      (extent-end-position extent))
	     (sit-for 1)
	     (insert (format "%s addition:\n\n"
			     (file-relative-name (buffer-file-name change-log)
						 directory)))
	     (insert (extent-string extent))
	     ;; Be sure to map all extents.
	     nil))
       change-log nil nil nil nil 'patcher mail-buffer))
    ))

(defun patcher-construct-command (command files)
  ;; Replace the %f with the specified files (if any), or append.
  (or files (setq files ""))
  (setq files (replace-in-string files "\\\\" "/"))
  (cond ((string-match "%f" command)
	 (replace-in-string (replace-in-string command "%f" files)
			    "[ \t]+$" ""))
	((> (length files) 0)
	 (concat command " " files))
	(t command)))

(defun patcher-after-send (&optional unused)
  ;; Do some cleanup after sending the mail.
  (when (patcher-project-option patcher-project
				:kill-change-logs-after-sending)
    (patcher-offer-save-buffers patcher-change-logs)
    (mapcar #'kill-buffer patcher-change-logs))
  ;; #### Implement kill-source-files-after-sending here.
  (when patcher-pre-commit-window-config
    (set-window-configuration patcher-pre-commit-window-config))
  (when patcher-commit-logmsg-buffer
    (kill-buffer patcher-commit-logmsg-buffer))
  (when patcher-commit-output-buffer
    (bury-buffer patcher-commit-output-buffer)))


;; Patcher LogMsg mode ======================================================

(defcustom patcher-logmsg-mode-hook nil
  "*Hook to run after setting up Patcher-Logmsg mode."
  :group 'patcher
  :type 'hook)

(make-variable-buffer-local
 (defvar patcher-logmsg-mail-buffer
   ;; Mail buffer where the relevant Patcher message is being edited.
   ;; This variable gets a proper value in the logmsg buffers.
   ))

(defun patcher-logmsg-do-it (arg)
  "Function used to commit a change reported by `patcher-mail'.
When called interactively, use a prefix to override the default commit
command for this project."
  (interactive "P")
  (let* ((project (symbol-value-in-buffer 'patcher-project
					  patcher-logmsg-mail-buffer))
	 (command (patcher-project-option project :commit-command))
	 (failed-command-regexp (patcher-project-option
				 project :failed-command-regexp))
	 (confirm-commits (patcher-project-option project :confirm-commits))
	 (change-logs (symbol-value-in-buffer 'patcher-change-logs
					      patcher-logmsg-mail-buffer))
	 (pre-commit-window-config
	  (symbol-value-in-buffer 'patcher-pre-commit-window-config
				  patcher-logmsg-mail-buffer))
	 (logmsg-file-name
	  (symbol-value-in-buffer 'patcher-logmsg-file-name
				  patcher-logmsg-mail-buffer))
	 (output-buffer (get-buffer-create "*Patcher-Commit-Output*")))
    ;; First, make sure the ChangeLogs are saved.
    (patcher-offer-save-buffers change-logs)
    ;; Now, construct the commit command by starting with what was specified
    ;; in this project's options (or read from the user, if the prefix arg was
    ;; given) and (if subproject files were given) combining the specified
    ;; files with the relevant ChangeLogs.  If the whole project is being
    ;; committed, the ChangeLogs will automatically be committed, otherwise we
    ;; have to specify them explicitly.
    (let* ((directory (patcher-project-directory project))
	   (files
	    (let ((f (symbol-value-in-buffer
		      'patcher-files
		      patcher-logmsg-mail-buffer)))
	      (if (not f)
		  ""
		(mapconcat
		 #'identity
		 (cons f (mapcar
			  #'(lambda (buf)
			      (patcher-file-relative-name
			       (buffer-file-name buf)
			       directory))
			  change-logs)) " ")))))
      (when arg
	(setq command (read-shell-command "Commit command: " command)))
      (setq command (patcher-construct-command
		     (replace-in-string command "%s"
					logmsg-file-name t)
		     files)))
    ;; Maybe display the commit command, and make sure the user agrees.
    (when (or (not confirm-commits)
	      (save-window-excursion
		(let ((runbuf (get-buffer-create "*Patcher-Commit-Command*")))
		  (erase-buffer runbuf)
		  (insert-string (format "Command to run:\n\n%s" command)
				 runbuf)
		  (display-buffer runbuf)
		  (y-or-n-p "Run commit command? "))))
      ;; Write out the log message and ...
      (write-region (point-min) (point-max) logmsg-file-name nil 'silent)
      (erase-buffer output-buffer)
      ;; ... commit!
      (shell-command command output-buffer)
      ;; Record the buffers that we will get rid of later.  Specifically, we
      ;; will bury the output buffer and kill the log message buffer when we
      ;; send the patch mail -- i.e. when we are done with this project.  We
      ;; don't kill the log message buffer now in case the user needs it later
      ;; -- e.g. if the commit failed and needs to be redone (we try to
      ;; detect this, but we might not succeed in all cases.).  The reason we
      ;; kill, not bury, the log message buffer is that it is generated anew
      ;; for each commit action, and we don't want to accumulate these buffers
      ;; endlessly. the output buffer is reused each time we execute the
      ;; commit, so no such problem exists here.
      (let ((curbuf (current-buffer)))
	(with-current-buffer patcher-logmsg-mail-buffer
	  ;; Note that this sets a buffer-local variable in the mail buffer,
	  ;; not our own buffer.
	  (setq patcher-commit-logmsg-buffer curbuf)
	  (if (buffer-live-p output-buffer)
	      (setq patcher-commit-output-buffer output-buffer))))
      ;; Try to see if the commit failed.
      (if (and (buffer-live-p output-buffer)
	       (with-current-buffer output-buffer
		 (save-excursion
		   (goto-char (point-min))
		   (re-search-forward failed-command-regexp nil t))))
	  ;; It failed.
	  (progn
	    (display-buffer output-buffer)
	    (message "Error during commit.  Please correct and try again."))
	;; Otherwise, record the successful commit in the mail message.
	(with-current-buffer patcher-logmsg-mail-buffer
	  (save-excursion
	    (goto-char (point-min))
	    (when (search-forward mail-header-separator nil t)
	      (forward-line 1)
	      (insert (concat (patcher-project-option patcher-project
						      :committed-notice)
			      "\n")))))
	;; Bury the log message (see above).  Remove the log message window
	;; and display the output buffer.
	(bury-buffer (current-buffer))
	(set-window-configuration pre-commit-window-config)
	(if (buffer-live-p output-buffer)
	    (display-buffer output-buffer)))
      )))

(defvar patcher-logmsg-mode-map
  (let ((map (make-sparse-keymap)))
    (define-key map [(control c) (control c)] 'patcher-logmsg-do-it)
    map))

(defun patcher-logmsg-mode ()
  "Sets up Patcher-LogMsg major mode.
Used for editing the log message for `patcher-commit-change'.  To commit
the change, use \\<patcher-logmsg-mode-map>\\[patcher-logmsg-do-it].
You're not supposed to use this, unless you know what you're doing."
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'patcher-logmsg)
  (setq mode-name "Patcher-LogMsg")
  (use-local-map patcher-logmsg-mode-map)
  (run-hooks 'patcher-logmsg-mode-hook))


;; Patcher FakeMail mode ====================================================

(defun patcher-fakemail-send ()
  "Pretend to send a fake Patcher mail.
Only perform the usual cleanup after real Patcher mails are sent."
  (interactive)
  (patcher-after-send nil)
  (kill-buffer (current-buffer)))

(defvar patcher-fakemail-mode-map
  (let ((map (make-sparse-keymap)))
    (define-key map [(control c) (control c)] 'patcher-fakemail-send)
    map))

(defun patcher-fakemail-mode ()
  "Sets up Patcher-FakeMail major mode.
Used for editing a fake Patcher mail.  "
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'patcher-fakemail)
  (setq mode-name "Patcher-FakeMail")
  (use-local-map patcher-fakemail-mode-map)
  (run-hooks 'patcher-logmsg-mode-hook))


;; Patcher minor-mode =======================================================

(defcustom patcher-minor-mode-string " Patch"
  "*String to use in the modeline when Patcher minor mode is active."
  :group 'patcher
  :type 'string)

(defcustom patcher-minor-mode-hook nil
  "*Hooks to run after setting up Patcher minor mode."
  :group 'patcher
  :type 'hook)

(defun patcher-generate-diff ()
  "(Re)generate the diff in the current Patcher mail buffer.
When appropriate, this function first removes any formerly generated diff.
The diff output will be inserted at the current position, or at the place
where a former diff output was found, if any.  Normally, you don't have to
call this function by hand, as `patcher-mail' and `patcher-mail-subproject'
will do it for you.  However, you may have to if the initial diff command
failed for some reason."
  (interactive)
  (let ((diff-extent
	 (car (mapcar-extents 'identity nil nil nil nil nil 'patcher-diff)))
	(change-logs-extent
	 (car (mapcar-extents 'identity nil nil nil nil nil 'patcher-change-logs)))
	(regenerate t))
    (save-excursion
      ;; First, clean up the place.
      (when patcher-change-logs
	(save-window-excursion
	  (save-excursion
	    (with-output-to-temp-buffer " *Patcher Information"
	      (set-buffer " *Patcher Information")
	      (if change-logs-extent
		  (insert "\
It appears that ChangeLog entries for this patch have already been generated,
edited and inserted into the mail buffer.\n\n")
		(insert "\
It appears that ChangeLog skeletons for this patch have already been generated,
and maybe edited.\n\n"))
	      (insert "\
If you answer `yes' to the question below, both the diff and the ChangeLog
entries will be regenerated. This means that current ChangeLog entries will be
lost.  If otherwise your answer is `no', only the diff will be regenerated."))
	    (setq regenerate (yes-or-no-p "\
Regenerate both patch and ChangeLog skeleton ? "))))
	(when regenerate
	  (when change-logs-extent
	    (delete-region (extent-start-position change-logs-extent)
			   (extent-end-position change-logs-extent))
	    (delete-extent change-logs-extent))
	  (dolist (change-log patcher-change-logs)
	    (map-extents
	     #'(lambda (extent unused)
		 (delete-region (extent-start-position extent)
				(extent-end-position extent)
				(extent-object extent))
		 (delete-extent extent)
		 ;; Be sure to map all extents.
		 nil)
	     change-log nil nil nil nil 'patcher (current-buffer)))))
      (when diff-extent
	(delete-region (extent-start-position diff-extent)
		       (extent-end-position diff-extent))
	(delete-extent diff-extent))
      (goto-char patcher-patch-marker)
      (message "Generating the diff...")
      (sit-for 0) ;; Let XEmacs redisplay the message buffer
      (apply 'call-process shell-file-name nil t nil
	     shell-command-switch `(,patcher-diff-command))
      (message "Generating the diff... done")
      (sit-for 0) ;; Let XEmacs redisplay the message buffer
      ;; Remember the command output region.
      (setq diff-extent (make-extent patcher-patch-marker (point)))
      (set-extent-properties diff-extent '(start-open t patcher-diff t))
      ;; Try to see if the diff failed.
      (if (save-excursion
	    (goto-char (extent-start-position diff-extent))
	    (re-search-forward
	     (patcher-project-option patcher-project
				     :failed-command-regexp)
	     nil t))
	  ;; It failed.
	  (message (substitute-command-keys "Error during diff.  \
Please fix the problem and type \\[patcher-generate-diff] to try again."))
	;; It succeeded.
	(when regenerate
	  (message "Generating the ChangeLog skeletons...")
	  (sit-for 0) ;; Let XEmacs redisplay the message buffer
	  (narrow-to-region (extent-start-position diff-extent)
			    (extent-end-position diff-extent))
	  (setq patcher-change-logs
		(patch-to-change-log (patcher-project-directory
				      patcher-project)
				     :keep-source-files
				     (not (patcher-project-option
					   patcher-project
					   :kill-source-files-after-diffing))
				     :extent-property 'patcher
				     :extent-property-value (current-buffer)))
	  (widen)
	  (message (substitute-command-keys
		    "Please annotate the ChangeLogs, \
and run \\[patcher-insert-change-logs] from this buffer when done."))))
      )))

(defun patcher-insert-change-logs ()
  "Insert annotated ChangeLog entries, after `patcher-mail' has been run.
Run this command after you've run `patcher-mail' and then annotated the
ChangeLog skeletons that were created."
  (interactive)
  (save-excursion
    (let ((extent (car (mapcar-extents 'identity nil nil nil nil nil 'patcher-change-logs))))
      (when (and extent
		 (y-or-n-p "ChangeLog entries already inserted. Replace ? "))
	(delete-region (extent-start-position extent)
		       (extent-end-position extent))
	(delete-extent extent))
      (goto-char patcher-change-logs-marker)
      (patcher-insert-change-logs-1 (current-buffer))
      (setq extent (make-extent patcher-change-logs-marker (point)))
      (set-extent-properties extent '(start-open t patcher-change-logs t)))
    ))

(defun patcher-compress-commit-log ()
  ;; After the ChangeLogs have been inserted into the commit log message
  ;; buffer, this function compresses the output into something that
  ;; conveys the essence of what has been changed, but much more compactly.

  ;;(interactive) for testing
  (goto-char (point-min))
  (delete-matching-lines "^\\(\\S-+\\)?ChangeLog addition:$")
  (delete-matching-lines "^[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9] ")
  ;; Now compress the change log specs into just files, so that mostly just
  ;; the annotations are left.
  (let ((change-log-change-line
	 "^\\([ \t]+\\)\\* \\(\\S-+\\)\\( (.*)\\)?:\\( New\\.\\)?"))
    (while (re-search-forward change-log-change-line nil t)
      (let ((beg (match-beginning 1)) ;; Change to match-end if you want the
				      ;; indentation.
	    (end (match-end 0))
	    files)
	(push (match-string 2) files)
	(forward-line 1)
	(while (looking-at change-log-change-line)
	  (setq end (match-end 0))
	  (unless (member (match-string 2) files)
	    (push (match-string 2) files))
	  (forward-line 1))
	(goto-char beg)
	(delete-region beg end)
	(insert (mapconcat 'identity (nreverse files) ", ") ":")
	(when (looking-at "\\s-+")
	  (let ((p (point))
		(end (match-end 0)))
	    ;; If there's no annotation at all for this change, make sure we
	    ;; don't treat the next change as an annotation for this one!
	    (if (save-excursion
		  (goto-char end)
		  (beginning-of-line)
		  (looking-at change-log-change-line))
		(progn
		  (if (looking-at "[ \t]+")
		      (delete-region p (match-end 0))))
	      (delete-region p end)
	      (insert " "))))
	)))
  ;; Shrink extra blank lines.
  (let ((blank-line "^\\s-*$"))
    (goto-char (point-min))
    (while (and (not (eobp))
		(progn (forward-line 1) (re-search-forward blank-line nil t)))
      (delete-blank-lines))
    (goto-char (point-min))
    (if (looking-at blank-line)
	(delete-blank-lines))))

(defun patcher-commit-change (arg)
  "Prepare to (and possibly) commit a change to a project's repository.
The change is the one that is announced in the mail buffer.

When called interactively, use a prefix (ARG) to override the commit
command to use."
  (interactive "P")
  (let* ((init-log-message
	  (patcher-project-option patcher-project :init-log-message))
	 (edit-log-message
	  (patcher-project-option patcher-project :edit-log-message))
	 (logmsg-buffer (generate-new-buffer "*Patcher-Log-Message*"))
	 (mail-buffer (current-buffer)))
    (setq patcher-pre-commit-window-config (current-window-configuration))
    (pop-to-buffer logmsg-buffer)
    (patcher-logmsg-mode)
    (setq patcher-logmsg-mail-buffer mail-buffer)
    (set-symbol-value-in-buffer
     'patcher-logmsg-file-name
     (replace-in-string
      (make-temp-name (expand-file-name "patch" (temp-directory)))
      "\\\\" "/")
     mail-buffer)
    (cond ((eq init-log-message 'subject)
	   (insert (with-current-buffer mail-buffer
		     (save-excursion
		       (goto-char (point-min))
		       (and (re-search-forward
			     "\
^Subject:[ \t\f\r]+\\(\\[PATCH\\][ \t\f\r]*\\)?\\(.*\\)$"
			     nil t)
			    (concat (match-string 2) "\n"))))))
	  ((eq init-log-message 'change-logs)
	   (patcher-insert-change-logs-1 mail-buffer))
	  ((eq init-log-message 'compressed-change-logs)
	   (patcher-insert-change-logs-1 mail-buffer)
	   (patcher-compress-commit-log))
	  ((eq init-log-message 'compressed-change-logs-with-original)
	   (patcher-insert-change-logs-1 mail-buffer)
	   (let ((orig (buffer-string)))
	     (patcher-compress-commit-log)
	     (goto-char (point-max))
	     (insert "\
\n\n---------------- Original ChangeLog entries follow: ----------------\n\n")
	     (insert orig)))
	  (t
	   (insert "(none)")))
    (goto-char (point-min))
    (if edit-log-message
	(message (substitute-command-keys
		  "Edit the log message, \
and press \\[patcher-logmsg-do-it] when done."))
      (patcher-logmsg-do-it arg))
    ))

(defvar patcher-minor-mode-map
  (let ((map (make-sparse-keymap 'patcher-minor-mode-map)))
    (define-key map [(control c) (control p) d] 'patcher-generate-diff)
    (define-key map [(control c) (control p) i] 'patcher-insert-change-logs)
    (define-key map [(control c) (control p) c] 'patcher-commit-change)
    (define-key map [(control c) (control p) v] 'patcher-version)
    map)
  ;; Patcher minor mode keymap.
  )

(make-variable-buffer-local
 (defvar patcher-minor-mode nil))

(defun patcher-minor-mode (arg)
  "Toggles Patcher minor mode.
Used for mails prepared with `patcher-mail'.  You're not supposed to use
this, unless you know what you're doing."
  (interactive "*P")
  (setq patcher-minor-mode
	(if (null arg) (not patcher-minor-mode)
	  (> (prefix-numeric-value arg) 0)))
  (run-hooks 'patcher-minor-mode-hook))

(add-minor-mode
 'patcher-minor-mode patcher-minor-mode-string patcher-minor-mode-map)


;; Mail preparation routines =================================================

(defun patcher-mail-compose-mail (project subject)
  "Prepare a patch-related mail with the `compose-mail' function.
See also the `mail-user-agent' variable.

This function uses the `:to-address' project option to determine the email
address for sending the message.  Otherwise, the address is prompted for."
  (compose-mail (or (patcher-project-option project :to-address)
		    (read-string "To address: "))
		subject nil nil nil nil '((patcher-after-send))))

(defun patcher-mail-sendmail (project subject)
  "Prepare a patch-related mail with the `mail' function.
This method requires the `sendmail' library.

This function uses the `:to-address' project option to determine the email
address for sending the message.  Otherwise, the address is prompted for."
  (require 'sendmail)
  (mail nil (or (patcher-project-option project :to-address)
		(read-string "To address: "))
		subject nil nil nil '((patcher-after-send))))

(defvar message-exit-actions)
(defun patcher-mail-message (project subject)
  "Prepare a patch-related mail with the `message-mail' function.
This method requires the `message' library.

This function uses the `:to-address' project option to determine the email
address for sending the message.  Otherwise, the address is prompted for."
  (require 'message)
  (message-mail (or (patcher-project-option project :to-address)
		    (read-string "To address: "))
		    subject nil nil nil nil '(patcher-after-send))
  ;; #### There appears to be a bug in message.el: message-mail completely
  ;; ignores the `send-actions' argument!  So we need to set the value
  ;; ourselves.  `message-exit-actions' seems more appropriate than
  ;; `message-send-actions'.
  (push '(patcher-after-send) message-exit-actions))

(defun patcher-mail-gnus (project subject)
  "Prepare a patch-related mail with the `gnus-post-news' function.
Don't worry, this function can also send mails ;-).  This method
requires that you have Gnus *running* in your XEmacs session.

This function uses the `:gnus-group' project option to determine the Gnus
group to use \(as if you had typed `C-u a' on that group in the Group
buffer).  Otherwise, the group is prompted for."
  (require 'gnus-util)
  (unless (gnus-alive-p)
    (error "You should have Gnus running in this XEmacs session"))
  ;; This binding is necessary to let message-mode hooks perform correctly.
  (let ((gnus-newsgroup-name (or (patcher-project-option project :gnus-group)
				 (read-string "Gnus group name: "))))
    (gnus-post-news 'post gnus-newsgroup-name)
    (message-goto-subject)
    (insert subject)
    (message-goto-body)
    (push '(patcher-after-send) message-exit-actions)))

(defun patcher-mail-fake (project subject)
  "Prepare a patch-related fake mail.
Use this function if you want to do all that Patcher can do, apart from
sending a real mail. This function generates a fake mail buffer which acts
as a standard Patcher mail buffer, apart from the fact that when you type
\\<patcher-fakemail-mode-map>\\[patcher-fakemail-send] in it, it doesn't
really send a mail, but just clean things up."
  (let ((buffer (generate-new-buffer "*Patcher Fake Mail*")))
    (switch-to-buffer buffer)
    (patcher-fakemail-mode)
    ))

(defun patcher-mail-1 (project subject files &optional override)
  ;; Perform the real job of preparing the mail buffer.
  (let ((command   (patcher-project-option project :diff-command))
	(directory (patcher-project-directory project)))
    (funcall (intern (concat "patcher-mail-"
			     (symbol-name
			      (patcher-project-option project :mail-method))))
	     project (concat "[PATCH] " subject))
    (patcher-minor-mode t)
    (cd directory)
    (let ((mail-prologue (patcher-project-option project :mail-prologue)))
      (and mail-prologue (insert "\n" mail-prologue)))
    (save-excursion
      (insert "\n\n")
      (setq patcher-change-logs-marker (point-marker))
      (insert "\n")
      (and override
	   (setq command (read-shell-command "Diff command: " command)))
      (setq command (patcher-construct-command command files))
      (let ((diff-prologue (patcher-project-option project :diff-prologue)))
	(and diff-prologue
	     (insert (replace-in-string
		      (replace-in-string diff-prologue
					 "%n" (patcher-project-name project))
		      "%c" command)
		     "\n\n")))
      (setq patcher-project project
	    patcher-files files
	    patcher-diff-command command
	    patcher-patch-marker (point-marker))
      (patcher-generate-diff))
    ))

(defvar patcher-projects-history nil
  ;; History used for prompting patcher projects.
  )

;;;###autoload
(defun patcher-mail-subproject (project subject files &optional arg)
  "Prepare a mail about a patch to apply on part of a project.
PROJECT is the name of the project (see the variables `patcher-projects'
and `patcher-subprojects').
SUBJECT is the subject of the mail.
FILES is a string listing one or more files, possibly with wild cards --
  essentially, part of a command line to be interpolated into the `diff'
  and maybe the `commit' commands issued by Patcher.

When called interactively, use a prefix (ARG) to override the value of
the diff command to use for this project.

This function is intended for one-time only subprojects.  Alternately, you
can define subprojects in the variable `patcher-subprojects' and continue
using `patcher-mail'.  If you call this function on a predefine subproject,
you will have the opportunity to modify the predefined list of files or
directories the subproject is composed of.

When you use this command instead of `patcher-mail', any commits issued
from the mail buffer (using \\<patcher-minor-mode-map>\\[patcher-commit-change]) will automatically include the associated ChangeLogs,
as well as the file(s) specified as part of this command.

Please note that you can have multiple occurrences of a Patcher mail at
the same time, but not more than one at a time on the same project unless
you use `patcher-mail-subproject' and the sections of the project don't
overlap."
  (interactive
   (let* ((prj (assoc (completing-read "Project: " (append patcher-subprojects
							   patcher-projects)
				       nil t nil 'patcher-projects-history)
		      (append patcher-subprojects patcher-projects)))
	  (sbj (read-string
		"Subject: "
		(let ((s (patcher-project-option prj :subject)))
		  (and s
		       (replace-in-string s "%n" (patcher-project-name prj)))
		  )))
	  (dir (patcher-project-directory prj))
	  (fls (let ((default-directory (file-name-as-directory dir)))
		 (or (let ((f (patcher-project-option prj :files)))
		       (and f (read-shell-command "Files: "
						  (concat f " ") nil f)))
		     ;; - Ben sez: Supply the current buffer's file as the
		     ;; default, since often we just finished working on a
		     ;; file and want to submit it as a patch.
		     ;; - Didier answers: ... without even testing what you've
		     ;; done ? You're a BBDB (Bad Boy, Dear Ben ;-) !!
		     (let* ((default-file (and (buffer-file-name)
					       (patcher-file-relative-name
						(buffer-file-name)
						dir)))
			    (default-file
			      ;; If the file is not actually underneath the
			      ;; project, then don't suggest it as a
			      ;; possibility.
			      (and default-file
				   (if (string-match "^\\.\\.$\\|^\\.\\.[/\\]"
						     default-file)
				       nil default-file))))
		       (read-shell-command
			(if default-file (format "Files (%s): " default-file)
			  "Files: ")
			nil nil default-file))))))
     (list prj sbj fls current-prefix-arg)))
  (patcher-mail-1 project subject files (and arg (interactive-p))))

;;;###autoload
(defun patcher-mail (project subject &optional arg)
  "Prepare a mail about a patch to apply on a project.
PROJECT is the name of the project (see the variables `patcher-projects'
and `patcher-subprojects').
SUBJECT is the subject of the mail.

When called interactively, use a prefix (ARG) to override the value of
the diff command to use for this project.  If you want to work on a
subset of the project (e.g. some files, subdirectories etc), you have two
alternatives:

- for temporary subprojects, you can use the function
  `patcher-mail-subproject', which lets you specify the list of modified
  files / directories.
- otherwise, you can also define the subprojects in the variable
  `patcher-subprojects' and continue using this function.

Please note that you can have multiple occurrences of a Patcher mail at
the same time, but not more than one at a time on the same project unless
you use `patcher-mail-subproject' and the sections of the project don't
overlap."
  (interactive
   (let* ((prj (assoc (completing-read "Project: " (append patcher-subprojects
							   patcher-projects)
				       nil t nil 'patcher-projects-history)
		      (append patcher-subprojects patcher-projects)))
	  (sbj (read-string
		"Subject: "
		(let ((s (patcher-project-option prj :subject)))
		  (and s
		       (replace-in-string s "%n" (patcher-project-name prj)))
		  ))))
     (list prj sbj current-prefix-arg)))
  (patcher-mail-1 project subject (patcher-project-option project :files)
		  (and arg (interactive-p))))

(provide 'patcher)

;;; patcher.el ends here
