;;; xwem-interactive.el --- XWEM interactive interface.

;; Copyright (C) 2003 by Free Software Foundation, Inc.

;; Author: Zajcev Evgeny <zevlg@yandex.ru>
;; Created: Thu Dec 18 05:49:52 MSK 2003
;; Keywords: xwem, xlib
;; X-CVS: $Id: xwem-interactive.el,v 1.4 2004/05/11 09:00:15 youngs Exp $

;; This file is part of XWEM.

;; XWEM is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XWEM is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
;; License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
;; 02111-1307, USA.

;;; Synched up with: Not in FSF

;;; Commentary:

;; 

;;; Code:

(eval-when-compile
  ;; Shutup compiler
  (defvar iswitchb-buflist nil)
  (autoload 'iswitchb-read-buffer "iswitchb"))

;;;###autoload
(defvar xwem-interactively nil
  "Non-nil when xwem in interactive mode.
Internal variabel, do not modify.")

;;;###autoload
(fset 'read-from-minibuffer-for-xwem (symbol-function 'read-from-minibuffer))


;;;###autoload
(defun xwem-interactive-p ()
  "Return non-nil when xwem in interactive mode."
  xwem-interactively)

(defun xwem-interactive-itimer ()
  "Itimer used by `xwem-read-from-minibuffer'."
  (when xwem-executing-kbd-macro
    (xwem-key-send-ekeys
     (vector (aref xwem-executing-kbd-macro (incf xwem-executing-kbd-macro-index))))
    ))

;;; Keymacros safe variant of `read-from-minibuffer', i.e. does not
;;  block if we are executing keyboard macro.
;;;###autoload
(defun xwem-read-from-minibuffer (prompt &optional initial-contents keymap readp history abbrev-table &rest notused)
  "Read data from xwem minibuffer.
Arguments PROMPT, INITIAL-CONTENTS, KEYMAP, READP, HISTORY and
ABBREV-TABLE are same as for `read-from-minibuffer'."
  (let (ret)

    (xwem-under-minib-focus
     (xwem-kbd-stop-grabbing)

     (when xwem-executing-kbd-macro
       (start-itimer "xwem-macro-itimer" 'xwem-interactive-itimer 0.01 0.01 t))
     
     (unwind-protect
         (setq ret (read-from-minibuffer-for-xwem prompt initial-contents keymap readp history abbrev-table))

       (when xwem-executing-kbd-macro
         (delete-itimer "xwem-macro-itimer")))

     ret)))

;;;###autoload
(defun xwem-completing-read (prompt table &optional predicate require-match initial-contents history)
  "XWEM awared varian of `completing-read'."
  (flet ((read-from-minibuffer (&rest args) nil))
    (fset 'read-from-minibuffer (symbol-function 'xwem-read-from-minibuffer))
    (completing-read prompt table predicate require-match initial-contents history)))
  
;;;###autoload
(defun xwem-read-command (prompt)
  "Just like `read-command', but for XWEM.
Argument PROMPT is same as for `read-command'."

  ;; Temporary redefine `read-from-minibuffer'
  (flet ((read-from-minibuffer (&rest args) nil))
    (fset 'read-from-minibuffer (symbol-function 'xwem-read-from-minibuffer))
    (read-command prompt)))

;;;###autoload
(defun xwem-read-filename (prompt &optional dir default must-match initial-contents history)
  "Just like `read-file-name', but for XWEM.
PROMPT, DIR, DEFAULT, MUST-MATCH, INITIAL-CONTENTS and HISTORY are
same as for `read-file-name'."

  ;; Temporary redefine `read-from-minibuffer'
  (flet ((read-from-minibuffer (&rest args) nil))
    (fset 'read-from-minibuffer (symbol-function 'xwem-read-from-minibuffer))
    (read-file-name prompt dir default must-match initial-contents history)))

;;;###autoload
(defun xwem-read-shell-command (prompt &optional initial-input history)
  "Just like `read-shell-command', but for XWEM.
PROMPT, INITIAL-INPUT and HISTORY are same as for `read-shell-command'."
  
  ;; Temporary redefine `read-from-minibuffer'
  (flet ((read-from-minibuffer (&rest args) nil))
    (fset 'read-from-minibuffer (symbol-function 'xwem-read-from-minibuffer))
    (read-shell-command prompt initial-input history)))

;;;###autoload
(defun xwem-read-external-command (prompt)
  "Read for external command using PROMPT."
  (xwem-launcher-query prompt))

;;;###autoload
(defun xwem-read-client (prompt &optional predic)
  "Read for client name prompting PROMPT and return xwem client.
Additional argument PREDIC is passed right to `xwem-client-name'."
  (flet ((read-from-minibuffer (&rest args) nil))
    (fset 'read-from-minibuffer (symbol-function 'xwem-read-from-minibuffer))

   (let* ((rcl nil)
	  (clsnl (xwem-cl-normal-list xwem-clients predic))
	  (clns (mapcar
		 (lambda (cl)
		   (cons (xwem-client-name cl clsnl) cl))
		 clsnl))
	  (name (flet ((iswitchb-make-buflist
			(default)
			(setq iswitchb-buflist (mapcar (lambda (cl) (car cl)) clns))))
		  (iswitchb-read-buffer prompt))))

     (while clns
       (when (string= (caar clns) name)
	 (setq rcl (cdar clns))
	 (setq clns nil))
       (setq clns (cdr clns)))
     rcl)))

;;;###autoload
(defun xwem-read-frame (prompt)
  "Read for frame number prompting PROMPT and return xwem frame."
  )

;; For `xwem-interactive' take a look at xwem-macros.el
;;;###autoload
(defun xwem-interactive-ilist (spec)
  "Return list valid for `interactive'.
SPEC is specification of list items."
  (let ((xwem-interactively t))
    (declare (special xwem-interactively))

    ;; XXX if ?* mean wait keyrelease
    (when (and spec (eq (aref (car spec) 0) ?*))
      (when (= (X-Event-type xwem-last-xevent) X-KeyPress)
        (xwem-kbd-wait-key-release (X-Event-xkey-keycode xwem-last-xevent)))

      ;; Remove ?* from first element in SPEC
      (if (= (length (car spec)) 1)
	  (setq spec (cdr spec))
	(setq spec (cons (substring (car spec) 1) (cdr spec)))))

    ;; XXX if ?_ is first than command need to run with ungrabbed
    ;; keyboard.
    (when (and spec (eq (aref (car spec) 0) ?_))
      (xwem-kbd-stop-grabbing)

      ;; Remove ?_ from first element in SPEC
      (if (= (length (car spec)) 1)
	  (setq spec (cdr spec))
	(setq spec (cons (substring (car spec) 1) (cdr spec)))))

    (mapcar (lambda (el)
	      (let ((code (aref el 0))
		    (prompt (substring el 1)))
		(cond ((eq code ?P) xwem-prefix-arg)
		      ((eq code ?p) (prefix-numeric-value xwem-prefix-arg))
		      
		      ((eq code ?k) (xwem-read-key prompt))
		      ((eq code ?K) (xwem-read-key-sequence prompt))
		      ((eq code ?c) (xwem-read-client prompt))
		      ((eq code ?f) (xwem-read-filename prompt nil nil t))
		      ((eq code ?F) (xwem-read-filename prompt))
		      ((eq code ?s) (xwem-read-from-minibuffer prompt))
		      ((eq code ?C) (xwem-read-command prompt))
		      ((eq code ?c) (xwem-read-client prompt))
		      ((eq code ?e) (xwem-read-external-command prompt))
		      )))
	    spec)))

(provide 'xwem-interactive)

;;; xwem-interactive.el ends here
