/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:	 Mark Allender (allender@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */
/*
 *  This file contains all of the routines that are needed to destroy
 *  a graph (or many graphs for that matter).  This includes some callback
 *  routines that get called when a graph is killed (either by WM, or from
 *  the control panel.
*/

/*
 *  This little routine merely destroys all of the GC's associated with
 *  a graph in a nice neat fashion.
*/

#include <stdio.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include "config.h"
#include "graph.h"
#include "extern.h"
#include "edge.h"

void
FreeGCs(graph)
     Graph *graph;
{
  XFreeGC(XtDisplay(graph->parent), graph->xor_gc);
  XFreeGC(XtDisplay(graph->parent), graph->graph_gc);
  XFreeGC(XtDisplay(graph->parent), graph->text_gc);
  XFreeGC(XtDisplay(graph->parent), graph->virt_xor_gc);
}  

/*
 *  This little routine just frees up the space that is located in the 
 *  common areas for both nodes and edges.  It uses 'free' to return
 *  the space back to the heap.
*/
void
FreeCommons(commons)
     CommonVals *commons;
{
  int i;
  KeyList *keylist;

  free(commons->i_name);
  free(commons->type);
  for (i = 0; i < MAX_LABELS; i++)
  {
    if (commons->labels[i])
	free (commons->labels[i]);
  }
/*
 *  get rid of the key/value pair
*/
  while (keylist = commons->key_list) {
    commons->key_list = keylist->next;
    free(keylist->key);
    free(keylist->value);
  }
  free(commons);
}

/*
 *  This routine removes a link from the link structure linking nodes and
 *  edges.  We have to see what the link structure is pointing to, and
 *  fix up the pointers accordingly.
*/
void
RemoveLink(link)
     Link *link;
{

  if (link->ptr_type & EDGEMASK) {
    ((Edge *)(link->p_edge_adj_node))->links = link->n_edge_adj_node;
    if (link->n_edge_adj_node != NULL) {
      link->n_edge_adj_node->p_edge_adj_node = (Link *)link->p_edge_adj_node;
      link->n_edge_adj_node->ptr_type |= EDGEMASK;
    }
  }
  else {
    link->p_edge_adj_node->n_edge_adj_node = link->n_edge_adj_node;
    if (link->n_edge_adj_node != NULL)
      link->n_edge_adj_node->p_edge_adj_node = link->p_edge_adj_node;
  }
  if (link->ptr_type & NODEMASK) {
    ((Vertex *)(link->p_adj_node))->links = link->n_adj_node;
    if (link->n_adj_node != NULL) {
      link->n_adj_node->p_adj_node = (Link *)link->p_adj_node;
      link->n_adj_node->ptr_type |= NODEMASK;
    }
  }
  else {
    link->p_adj_node->n_adj_node = link->n_adj_node;
    if (link->n_adj_node != NULL)
      link->n_adj_node->p_adj_node = link->p_adj_node;
  }
  /* Now we want to check if removing the link caused the edge to
   * no longer point to the nodes it used to.  If so, we need to
   * remove the nodes from the edge->edge_nodes (and modify the count).
   */
  if (!EdgeConnects(link->edge, link->fnode, TO_NODE | FROM_NODE))
  {
    RemoveNodefromList (&link->edge->edge_nodes, link->fnode);
    link->edge->node_count--;
  }
  if (!EdgeConnects(link->edge, link->tnode, TO_NODE | FROM_NODE))
  {
    RemoveNodefromList (&link->edge->edge_nodes, link->tnode);
    link->edge->node_count--;
  }
  free(link);
}

/*
 * FreeEdge actually frees the space that the edge held.
 */
void
FreeEdge (edge)
  Edge *edge;
{
  /* The following while loop should remove the edge links AND free
   * up all the nodes that this edge points to.
   */
  while (edge->links)
    RemoveLink(edge->links);
  FreeCommons(edge->values);
  free(edge);
}

/*
 *  DeleteEdge removes the edge that is passed in as a paramter from the
 *  graph.  As with the function DeleteNode, we must (possibly) fix up
 *  any of the internal link structures that might get hosed as a result.
*/
void
DeleteEdge(edge)
     Edge *edge;
{
  Edge *ptr;
  Link *link, *tmplink;

/*
 * First, remove the edge from the linked list of edges
*/
  if (graph->edges == edge)
    graph->edges = edge->next;
  else
    for (ptr = graph->edges; ptr != NULL; ptr = ptr->next)
      if (ptr->next == edge) {
	ptr->next = edge->next;
	break;
      }
  FreeEdge(edge);
}
    
/*
 *  DeleteNode deletes the node passed in as a parameter from the graph.
 *  We must take care that all of the link structure for the graph are
 *  fixed up when this is done.
*/
void
DeleteNode(node)
     Vertex *node;
{
  Vertex *ptr;
  Link *link, *tmplink;
  Edge *edge;
  EdgeIdList *list, *tmpedge;

/*
 *  First, remove the node from the node list
*/
  if (graph->v == node)
    graph->v = node->next;
  else
    for (ptr = graph->v; ptr != NULL; ptr = ptr->next)
      if (ptr->next == node) {
	ptr->next = node->next;
	break;
      }
/*
 *  Now, skip down the links of the node.  We remove all the links on
 *  the node.  We can catch the edges that may need to be deleted by
 *  following the From_Node_List on the node being deleted.
*/
  while (node->links)
    RemoveLink(node->links);
/*
 *  Now, we must also look at all of the edges that pointed to the
 *  node we are deleting.  This is kept in a linked list off of the node.
 *  We now want to see if those edges are now only pointing to one link
 *  box (because link boxes may have gotten deleted above), and if so, then
 *  delete the edge.  Otherwise, remove any box that points to the node
 *  just deleted from the links.
*/
  list = node->edge_id_list;
  while (list) {
    tmpedge = list->next;
    if ((edge = EdgeLookup(list->edge_id, graph->edges))) {
      link = edge->links;
      while (link) {
	tmplink = link->n_edge_adj_node;
	if (link->tnode == node)
	  RemoveLink(link);
	link = tmplink;
      }
      if (edge->links == NULL)
	DeleteEdge(edge);
    }
    free(list);
    list = tmpedge;
  }
  FreeCommons(node->values);
  free(node);
}

/*
 *  The routine ClearGraph deletes all of the node and edges from a graph
*/
void
ClearGraph(graph)
  Graph *graph;
{
  Vertex *node, *tmpn;
  Edge *edge, *tmpe;

  node = graph->v;
  while (node) {
    tmpn = node->next;
    DeleteNode(node);
    node = tmpn;
  }
  edge = graph->edges;
  while (edge) {
    tmpe = edge->next;
    DeleteEdge(edge);
    edge = tmpe;
  }
}

/*
 *  DestroyGraph is the callback that gets called when a graph
 *  is destroyed either from the control panel, or from the WM.
*/
void
DestroyGraph(widget, graph, call_data)
     Widget widget;
     Graph *graph;
     caddr_t call_data;
{
  struct TableStruct *table_ptr, *prev;
  struct FunctionBindings *fbinding;
  struct KeyBindings *key_binding;
  struct FunctionList *flist;
  struct SuperNodeList *super_node;
  struct DisplayTable *ctable_entry;

  for (table_ptr = GraphList, prev = NULL; table_ptr;
       prev = table_ptr, table_ptr = table_ptr->next)
    if (table_ptr->graph == graph) break;
  if (!table_ptr) {
    printf ("Error: deleted graph not in Graph List.  Consult guru.\n");
    exit(1);
  }
  if (prev == NULL)
    GraphList = table_ptr->next;
  else
    prev->next = table_ptr->next;
  free(table_ptr->id);
  free(table_ptr);		  /* get rid of the space */
/*
 *  We need to get rid of functions and key bindings.
*/
  while (fbinding = graph->callback_functions) {
    graph->callback_functions = fbinding->next;
    free(fbinding->name);
    free(fbinding->function);
    free(fbinding->data);
    free(fbinding->tag);
    free(fbinding->domain);
    free(fbinding);
  }
  while (key_binding = graph->callback_keys) {
    graph->callback_keys = key_binding->next;
    while (flist = key_binding->func_list) {
      key_binding->func_list = flist->next;
      free(flist->function);
      free(flist);
    }
    free(key_binding);
  }
/*
 *  Also, free the color table stuff
*/
  while (ctable_entry = graph->display_table->common) {
    graph->display_table->common = ctable_entry->next;
    free(ctable_entry->type);
    free(ctable_entry);
  }
  while (ctable_entry = graph->display_table->nodes) {
    graph->display_table->nodes = ctable_entry->next;
    free(ctable_entry->type);
    free(ctable_entry);
  }
  while (ctable_entry = graph->display_table->edges) {
    graph->display_table->edges = ctable_entry->next;
    free(ctable_entry->type);
    free(ctable_entry);
  }
  free(graph->display_table);
  ClearGraph(graph);
  if (graph->title) free(graph->title);
  FreeGCs(graph);			/* check xdraw.c */
  free(graph);
}
