
/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:  Mark Allender (allender@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

#ifndef ELISION_H
#define ELISION_H

#include "graph.h"

/*
 *  This header file contains information related to the processes
 *  of eliding nodes from the graph.  A variety of generic structures
 *  are defined here.
*/

/*
 *  The following definitions relate to the type of elision policy
 *  defined.  The definitions are as follows:

REMOVE_ALL -- This is the default action.  In either case of a node elision
              or a region elision, then all edges the point into the elided
	      region, or out from the elided region (this only makes sense
	      in the context of a region elision) will completely be
	      hidden in the graph.  The only exception to this rule
	      will be the edges that point to the node (in the case of
	      node elision) because we need a way to attach the elided
	      region to the rest of the graph.

INBOUND_EDGE -- This is used to signify that any edges that point from
                a node outside the elided region to a node inside the elided
		region, then this edge will appear, and will now point from
		the node outside the elided region to the supernode that
		is being created.

INBOUND_EDGE_AND_NODE -- This is used to mean that if an edge points from
                         outside the region being elided to a point inside
			 the elided region, then the edge *AND* the node to
			 which it points will both remain on the graph.
			 This is true for both node elision and region elision.

OUTBOUND_EDGE -- Similar in function to the INBOUND_EDGE, this specifies that
                 any edges outbound from a region being elided (this only
		 applies to this case) that point to a node outside the
		 elided region will now point from the supernode being
		 created to the node outside the region.

OUTBOUND_EDGE_AND_NODE -- As with INBOUND_EDGE_AND_NODE, if node inside
                          an elided region that points outside the region,
			  then both the node *AND* the edge will remain
			  in the graph.

UNDIRECTED_EDGE -- This definition is used to signify that all undirected
                   edges into the region will now be connected to the supernode
		   that is created.

UNDIRECTED_EDGE_AND_NODE -- The undirected edge and the node inside the
                            region to which it is connected are both kept
			    in the graph definition.

SUPER_NODE -- This is merely a combination of INBOUND_EDGE and OUTBOUND_EDGE
              in the fact that if this is chosen as the method of elision in
	      either a node elision (in which only INBOUND_EDGES to the
	      elided region will be affected) or a region elision, then
	      all edges pointing into and out from the region will now all
	      point to the supernode, and out from the supernode.

 *  We have to have a seperate policy for each kind of edges coming
 *  into the region.  i.e. there will be a policy for inbound edges,
 *  a policy for outbound edges, and a policy for undirected edges.
 *  Please see the programmers manual for more detail.
*/

/*
 *  These are the mechanisms for elision.  Some of the numbers are duplicated
 *  because of the nature of one of the antielision policies.
*/

#define HIDE_EDGE 0
#define CONNECT_TO_SUPERNODE 1
#define HIDE_NODE 2
#define HIDE_SUB 3
#define UNHIDE_NODE 4
#define UNHIDE_SUB 5

/*
 *  Some antielision mechanisms for adding an edge between two nodes
 *  in different elision sets.
*/
#define HIDE_IN_FROM_NODE 2
#define HIDE_IN_TO_NODE 4

/*
 *  some definitions for elision mechanisms to help with coding
*/
#define INBOUND 0x07
#define OUTBOUND 0x07 << 3
#define UNDIRECTED 0x07 << 6
#define INFINAL 0x01 << 9
#define OUTFINAL 0x01 << 10
#define UNFINAL 0x01 << 11
#define BETWEENSETS 0x07 << 9

/*
 *  These definition correspond to how the super node of a elided region
 *  is going to get named.
*/

#define ROOT_IS_LABEL 1			/* this is default */
#define QUERY_FOR_LABEL 2

/*
 *  We will use these two definitions to help us keep track of certain`
 *  operations in elision.
*/
#define NODE_ELISION 1
#define SWEEP_ELISION 2

/*
 *  This structure is a linked list that will hold all of the super nodes
 *  and what parts of the graph they elide.  It makes more sense to do things
 *  this way, since we will probably save space here rather than having
 *  every node have a (possible) pointer to such a location.  This would
 *  probably be a waste of pointer space if done this way.
*/

struct SuperNodeList
{
  Vertex *super_node;
  NodeList *nodes;
  EdgeList *edges;
  struct SuperNodeList *next;
};

/*
 *  ElisionPolicy is the variable that holds the browser wide type of elision
 *  that is going to be done.  SuperCounter is the variable that will help
 *  to determine the internal-label of the supernode and superedges that
 *  might possibly be created with an elision
*/
short ElisionPolicy;
int SuperNodeLabel;
int SuperCounter;

#endif
