/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees, University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	clerk.c: Main file for clerk
 *	Author:  Alan M. Carroll (carroll@cs.uiuc.edu)
 *      Modified:  Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Enquiries should be sent to
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 */

/*
 * 02/14/92 - Doug Bogia - Changed the way the domains worked and changed
 *   the clerk flags to support these domains.
 * 03/09/92 - Doug Bogia - Caused the clerk to re-read the directories file
 *   anytime it changes (well, the next time it gets a message bus message).
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
/* ------------------------------------------------------------------------ */
#include "mbus/api.h"
#include "mbus/mbus.h"
#include "mbus/keyword.h"
#include "getopt.h"
#include "config.h"
#include "cb-defs.h"

int MBLogLevel = 1;

#define BUF_SIZE 2048
/* Flags that tell if the clerk is allowed to read/write/remove files
 * from a particular dir.
 */
#define READ_PERMISSION 1
#define WRITE_PERMISSION 2
#define REMOVE_PERMISSION 4

#define FALSE 0
#define TRUE 1

/* SEPARATORS are the characters that can come between items in the file */
#define SEPARATORS " \t\n"
/* BAK_EXT is the extension to add to a file to create a backup file. */
#define BAK_EXT "~"

#define FREE(x) if (NULL != (x)) free(x)
#define ARRAY_SIZE(a) (sizeof(a) / (sizeof((a)[0])))

/* ------------------------------------------------------------------------ */
struct directory_entry
{
  char *path;
  int length;			/* length of path */
  int ops;			/* What operations are allowed in dir. */
  char prefix;			/* subdirectories ok too? */
  struct directory_entry *next;
} ;

struct temp_file_record
{
  char *path;			/* path to tmp file */
  struct temp_file_record *next;
} ;

char *host = NULL;
int port = DEF_MBUS_PORT;
char *cb_base_domain = NULL;
char *cb_server_domain = NULL;
char *cb_ui_domain = NULL;
char *cb_ws_domain = NULL;
char *cb_user = NULL;
char *cb_clerk_domain = NULL;
char *dir_path = ".cb-directories";

t_mb_connection MBus;

struct temp_file_record *temp_files = NULL;
/* ------------------------------------------------------------------------ */
#if !(defined NeXT) && !(defined HPUX)
char *strerror(err) int err;
{
  extern char *sys_errlist[];
  extern int sys_nerr;

  return err < 0 || err >= sys_nerr ? "Unknown error" : sys_errlist[err];
}
#endif
/* ------------------------------------------------------------------------ */
void
StartLog(Message)
  char *Message;
{
#ifdef USE_WIDGET_SERVER
  extern char *cb_ws_domain, *cb_user, *cb_ui_domain, *cb_base_domain;
  extern t_mb_connection MBus;
  char buf[BUF_SIZE];
  if (!strcmp(cb_user, ""))
      sprintf(buf, "(\"info\" \"%s.%s.%s\" (info clerk.info \"%s (Server)\"",
	      cb_ws_domain, cb_ui_domain, cb_base_domain, Message);
  else
      sprintf(buf, "(\"info\" \"%s.%s.%s.%s\" (info clerk.info \"%s\"",
	      cb_ws_domain, cb_user, cb_ui_domain, cb_base_domain, Message);
  MBtransmit_Cstring(MBus, buf);
#else
  fprintf(stderr, "%s\n", Message);
#endif
}
/* ------------------------------------------------------------------------ */
void
EndLog()
{
#ifdef USE_WIDGET_SERVER
  extern t_mb_connection MBus;
  MBtransmit_Cstring(MBus, "))");
#endif
}  
/* ------------------------------------------------------------------------ */
void
InfoLine(level, StartMessage, Info, Queue)
  int level;			/* Level of this logging line */
  char *StartMessage;		/* Starting message */
  char *Info;			/* Current line of information */
  int Queue;			/* Should we queue this info up? */
{
  extern int MBLogLevel;
  static int HaveQueued = 0;
  if (Info && level <= MBLogLevel)
  {
    /* If we haven't queued data, send a start message. */
    if (!HaveQueued) StartLog(StartMessage);
#ifdef USE_WIDGET_SERVER
    MBtransmit_Cstring(MBus, " \"");
    MBtransmit_Cstring(MBus, Info);
    MBtransmit_Cstring(MBus, "\"");
#else
    fprintf(stderr, "%s\n", Info);
#endif    
    HaveQueued = 1;
  }    
  if (!Queue && HaveQueued)
  {
    EndLog();
    HaveQueued = 0;
  }
}  
/* ------------------------------------------------------------------------ */
int
CheckPermissions(match, mask)
  struct directory_entry *match;
  int mask;
{
  return (match && ((match->ops & mask) == mask));
}

struct directory_entry *
FindBestMatch(path, valid_dirs)
  char *path;
  struct directory_entry *valid_dirs;
{
  int length;
  char *ptr;
  struct directory_entry *best_match = NULL;
  int best_length = 0;
  ptr = strrchr (path, '/');
  if (!ptr)
  {
    path = ".";
    length = 1;
  }
  else
  {
    length = ptr - path;
  }
  while (valid_dirs)
  {
    if (length == valid_dirs->length)
    {
      if (!strncmp(valid_dirs->path, path, length))
      {
	best_match = valid_dirs;
	break;			/* We found the best possible */
      }
    }
    else if (valid_dirs->prefix)
    {
      if (valid_dirs->length < length && /* It will match only part */
	  valid_dirs->length > best_length && /* It will be best match */
	  !strncmp(valid_dirs->path, path, valid_dirs->length))
      {
	best_match = valid_dirs;
	best_length = valid_dirs->length;
      }
    }
    valid_dirs = valid_dirs->next;
  }
  return best_match;
}
      
/* ------------------------------------------------------------------------ */
struct directory_entry *
readDirectories(fpath) char *fpath;
{
  FILE *f;
  struct directory_entry *base = NULL, *new;
  int n, prefix, lineno = 0;
  struct stat info;
  char buff[BUF_SIZE], err_buf[BUF_SIZE+200], StartMessage[BUF_SIZE];
  char *ptr, *nextptr;
  int end, start, length, opscheck;

  sprintf(StartMessage, "Clerk Message about %s", fpath);
  f = fopen(fpath, "r");
  if (NULL == f)
  {
    sprintf(buff, "Cannot read `%s' file - no valid directories", fpath);
    InfoLine(1, StartMessage, buff, FALSE);
    return NULL;
  }

  while (NULL != fgets(buff, BUF_SIZE, f))
  {
    lineno += 1;
    ptr = strchr(buff, '#');
    if (ptr) *ptr = 0;		/* Throw away comments */
      
    ptr = strtok(buff, SEPARATORS);
    if (ptr)
    {
      n = strlen(ptr);
      if ('/' == buff[n-1])
      {
	prefix = 1;
	buff[--n] = 0;
      }
      else prefix = 0;

      /* check for access to directory, etc. */
      if (stat(ptr, &info) < 0)
      {
	sprintf(err_buf, "Line %d - `%s': %s", lineno, ptr,
		strerror(errno));
	InfoLine(1, StartMessage, err_buf, TRUE);
      }
      else if (!(info.st_mode & S_IFDIR))
      {
	sprintf(err_buf, "Line %d - `%s': Not a directory", lineno, ptr);
	InfoLine(1, StartMessage, err_buf, TRUE);
      }
      else
      {
	new = (struct directory_entry *)
	    malloc(sizeof(struct directory_entry));
	new->path = (char *) malloc(n + 1);
	new->length = n;
	strcpy(new->path, ptr);
	new->prefix = prefix;
	new->next = base;
	new->ops = 0;		/* No permissions, yet... */
	base = new;
	while ((ptr = strtok(NULL, SEPARATORS)))
	{
	  if (!strcmp(ptr, "read"))
	  {
	    new->ops |= READ_PERMISSION;
	  }
	  else if (!strcmp(ptr, "write"))
	  {
	    new->ops |= WRITE_PERMISSION;
	  }
	  else if (!strcmp(ptr, "remove"))
	  {
	    new->ops |= REMOVE_PERMISSION;
	  }
	  else
	  {
	    sprintf(err_buf, "Unknown option:  Line = %d Option = %s",
		    lineno, ptr);
	    InfoLine(1, StartMessage, err_buf, TRUE);
	  }
	}
      }
    }
  }

  InfoLine(1, NULL, NULL, FALSE); /* Flush the log */
  
  fclose(f);

  return base;
}
/* ------------------------------------------------------------------------ */
void
freeDirectories (valid_dirs)
  struct directory_entry **valid_dirs;
{
  struct directory_entry *tmp;
  for ( ; *valid_dirs; *valid_dirs = (*valid_dirs)->next)
  {
    tmp = *valid_dirs;
    FREE(tmp->path);
    FREE(tmp);
  }
}
/* ------------------------------------------------------------------------ */
void
checkDirectories (dir_path, mod_time, valid_dirs)
  char *dir_path;
  time_t *mod_time;
  struct directory_entry **valid_dirs;
{
  struct stat file_status;
  char buf[BUF_SIZE];
  if (stat(dir_path, &file_status) == 0)
  {
    if (*mod_time != file_status.st_mtime)
    {
      *mod_time = file_status.st_mtime;
      freeDirectories (valid_dirs);
      *valid_dirs = readDirectories(dir_path);
    }
  }
  else
  {
    sprintf(buf,
	    "Error reading %s\nThe valid directories remain unchanged\n",
	    dir_path, strerror(errno));
    InfoLine(1, "Clerk Message", buf, FALSE);
  }
}
/* ------------------------------------------------------------------------ */
#ifdef DEBUG
void
reportDirectories(valid_dirs)
  struct directory_entry *valid_dirs;
{
  char *StartMessage = "List of valid directories";
  char buf[BUF_SIZE];
  for ( ; valid_dirs; valid_dirs = valid_dirs->next)
  {
    sprintf (buf, "%s", valid_dirs->path);
    if (valid_dirs->prefix)
    {
      strcat (buf, " and sub directories");
    }
    strcat (buf, " Permissions:");
    if (valid_dirs->ops & READ_PERMISSION)
    {
      strcat (buf, " read");
    }
    if (valid_dirs->ops & WRITE_PERMISSION)
    {
      strcat (buf, " write");
    }
    if (valid_dirs->ops & REMOVE_PERMISSION)
    {
      strcat (buf, " remove");
    }
    if (!valid_dirs->ops)
    {
      strcat (buf, " none");
    }
    InfoLine(0, StartMessage, buf, TRUE);
  }
  InfoLine(0, NULL, NULL, FALSE); /* Flush the queue. */
}
#endif  
/* ------------------------------------------------------------------------ */
int DealWithCommandLine(argc,argv)
     int argc;
     char **argv;
{
  int opt;
  char *tmp;
  char Usage = 0;
  extern char *getenv(), *GetUserName();

  /* environment variable checks */
  host = getenv(ENV_MBUS_HOST);
  if (NULL != (tmp = getenv(ENV_MBUS_PORT))) port = strtol(tmp, NULL, 0);
  cb_base_domain = getenv(ENV_BASE_DOMAIN);
  cb_server_domain = getenv(ENV_SERVER_DOMAIN);
  cb_ui_domain = getenv(ENV_UI_DOMAIN);
  cb_clerk_domain = getenv(ENV_CLERK_DOMAIN);
  cb_ws_domain = getenv(ENV_WS_DOMAIN);

  while ( (opt = getopt(argc,argv,"b:s:i:u:d:h:p:L;")) != EOF)
    switch (opt)
      {
      case '?' :
	fprintf (stderr, "Unknown option -%c.\n", opt_err_char);
	Usage = 1;
	break;
      case 'b':
	cb_base_domain = optarg; break;
      case 's':
	cb_server_domain = optarg; break;
      case 'i':
	cb_ui_domain = optarg; break;
      case 'u':
	cb_user = optarg; break;
      case 'd' : dir_path = optarg; break;
      case 'h' : host = optarg; break;
      case 'p' : port = strtol(optarg, NULL, 0); break;
      case 'L':
        if (NULL == optarg) MBLogLevel = 1;
        else MBLogLevel = strtol(optarg, NULL, 0);
        break;
      }

  if (Usage)
  {
    printf ("Usage:  %s -b b_domain -s s_domain -i i_domain\n",
	    argv[0]);
    printf ("           -u u_domain -h host -p port -d dir_path\n");
    printf ("where:  b_domain is the base domain (default \"%s\")\n",
	    DEF_BASE_DOMAIN);
    printf ("        i_domain is the user interface domain (default \"%s\")\n",
	    DEF_UI_DOMAIN);
    printf ("        u_domain is the user domain (default $USER)\n");
    printf ("        host is the host of the message bus\n");
    printf ("        port is the port of the message bus\n");
    printf ("        dir_path is where to find the file specifying legal directories\n");
    printf ("\nEnvironment variables used:\n");
    printf ("CB_BASE_DOMAIN     the base domain\n");
    printf ("CB_SERVER_DOMAIN   the server domain\n");
    printf ("CB_UI_DOMAIN       the user interface domain\n");
    printf ("CB_USER            the user domain\n");
    printf ("CB_CLERK_DOMAIN    the clerk domain\n");
    printf ("CB_WS_DOMAIN       the widget server domain\n");
    printf ("MBUS_HOST          the message bus host\n");
    printf ("MBUS_PORT          the message bus port\n");
    exit(1);
  }

  if (NULL == cb_base_domain) cb_base_domain = DEF_BASE_DOMAIN;
  if (NULL == cb_server_domain) cb_server_domain = DEF_SERVER_DOMAIN;
  if (NULL == cb_ui_domain) cb_ui_domain = DEF_UI_DOMAIN;
  if (NULL == cb_user) cb_user = GetUserName();
  if (NULL == cb_clerk_domain) cb_clerk_domain = DEF_CLERK_DOMAIN;
  if (NULL == cb_ws_domain) cb_ws_domain = DEF_WS_DOMAIN;

  return optind;
}
/* ------------------------------------------------------------------------- */
#define SEND_SIZE 1024

/* (send TAG DOMAIN FILE :command :id :key :args) */
static struct keyword_entry_struct skw[] =
{
  { ":command", (void *)"receive", KEYWORD_RAW },
  { ":id", NULL, KEYWORD_RAW },
  { ":key", NULL, KEYWORD_RAW },
  { ":args", NULL, KEYWORD_SEXP },
  { ":remove", (void *)KEYWORD_NONE, KEYWORD_FLAG },
};

void
SendFile(comm, valid_dirs)
  t_sexp comm;
  struct directory_entry *valid_dirs;
{
  t_sexp tmp;			/* temporary holder for S-exps */
  char *path;			/* path of the file to send */
  struct stat info;		/* file information structure */
  int fd;			/* fd of the opened file */
  int i;			/* scratch counter */
  long remaining;		/* bytes left in the file to send */
  long n;			/* number of bytes from last read() */
  char *domain;			/* domain of the file message */
  char *tag;			/* tag for the message */
  char *command;		/* command for the message */
  char *id;			/* id for the message */
  char *key;			/* keyword for file, if any */
  t_sexp s_args;		/* additional args for the command */
  t_keyword_flag f_remove;	/* remove file flag */
  char *args;			/* printed version of s_args */
  char buff[SEND_SIZE];		/* buffer for sending the file */
  struct directory_entry *best_match; /* Best match of the path */
  
  tmp = MBnth(comm, 2);		/* alledged file name */
  if (!MBstringp(tmp)) return;		/* no file name */
  path = MBCstring(tmp);
  best_match = FindBestMatch(path, valid_dirs);
  if (CheckPermissions (best_match, READ_PERMISSION))
  {
    if (stat(path, &info) == 0)
    {
      if (S_IFREG & info.st_mode)
      {
	if ((fd = open(path, O_RDONLY)) >= 0)
	{
	  remaining = info.st_size; /* length of file */

	  tag = MBprint_Cstring(MBcar(comm));
	  domain = MBprint_Cstring(MBnth(comm, 1));

	  MBparse_keywords(MBnthcdr(comm, 3), skw, ARRAY_SIZE(skw));
	  i = 0;
	  command = (char *) skw[i++].result;
	  id = (char *) skw[i++].result;
	  key = (char *) skw[i++].result;
	  s_args = (t_sexp) skw[i++].result;
	  f_remove = (t_keyword_flag) skw[i++].result;
	  
	  MBtransmit_Cstring(MBus, "(");
	  MBtransmit_Cstring(MBus, tag);
	  MBtransmit_Cstring(MBus, " ");
	  MBtransmit_Cstring(MBus, domain);
	  MBtransmit_Cstring(MBus, "(");
	  MBtransmit_Cstring(MBus, command);
	  if (NULL != id)
	  {
	    MBtransmit_Cstring(MBus, " ");
	    MBtransmit_Cstring(MBus, id);
	  }
	  if (NULL != key)
	  {
	    MBtransmit_Cstring(MBus, " ");
	    MBtransmit_Cstring(MBus, key);
	  }

	  sprintf(buff, " #%ld!", remaining);
	  MBtransmit_Cstring(MBus, buff);
	  while (remaining > 0)
	  {
	    n = read(fd, buff, SEND_SIZE);
	    remaining = remaining - n;
	    MBtransmit_buffer(MBus, buff, n);
	  }

	  /* now do the extended args */
	  if (NULL != s_args)
	  {
	    args = MBprint_Cstring(s_args);
	    if (MBconsp(s_args))
		MBtransmit_buffer(MBus, args + 1, strlen(args) - 2);
	    else
		MBtransmit_Cstring(MBus, args);
	  }

	  MBtransmit_Cstring(MBus, "))");
	  close(fd);
	  FREE(domain); FREE(tag); FREE(command);
	  FREE(id); FREE(key); FREE(args);

	  if (f_remove == KEYWORD_TRUE)
	  {
	    if (CheckPermissions (best_match, REMOVE_PERMISSION))
	    {
	      unlink(path);
	    }
	    else
	    {
	      sprintf (buff, "No clerk remove permission on %s", path);
	      InfoLine (1, "Clerk Message", buff, FALSE);
	    }
	  }
	}
	else
	{
	  sprintf (buff, "Unable to open the file %s", path);
	  InfoLine (1, "Clerk Message", buff, FALSE);
	}	  
      }
      else
      {
	sprintf (buff, "File %s is not a regular file", path);
	InfoLine (1, "Clerk Message", buff, FALSE);
      }
    }
    else
    {
      sprintf (buff, "Could not stat %s.  Error %s", path, strerror(errno));
      InfoLine (1, "Clerk Message", buff, FALSE);
    }
  }
  else
  {
    sprintf (buff, "No clerk read permission on %s", path);
    InfoLine (1, "Clerk Message", buff, FALSE);
  }      
  FREE(path);
}
/* ------------------------------------------------------------------------- */
/* We know that we will be called with a count of zero immediately before the
 * first chunk of valid data. The last call will be with remaining == 0,
 * at which time we'll have to return a t_sexp.
 */
t_sexp
FileBlockHandler(buff, count, remaining)
     char *buff;		/* data buffer */
     int count;			/* # of bytes in buffer */
     int remaining;		/* # of bytes left in raw data */
{
  static char *tmp_path = NULL;	/* get temporary file name */
  static int fd = -1;
  t_sexp zret = NULL;
  struct temp_file_record *tf;

  if (NULL == buff)		/* first time, do set up */
    {
      tmp_path = tmpnam(NULL); /* get temporary file name */
      fd = open(tmp_path, O_CREAT | O_RDWR, 0600);
    }
  else if (fd >= 0)
    {
      write(fd, buff, count);
      if (0 == remaining)
	{
	  /* last time for this one, close the file and return a t_sexp */
	  close(fd);
	  fd = -1;
	  zret = MBput_Cstring(NULL, tmp_path);
	  tf = (struct temp_file_record *)
	    malloc(sizeof (struct temp_file_record));
	  tf->path = MBCstring(zret);
	  tf->next = temp_files;
	  temp_files = tf;
	}
    }
  return zret;
}
/* ------------------------------------------------------------------------- */
void
RemoveTempFile(path) char *path;
{
  struct temp_file_record *tf = temp_files;
  struct temp_file_record *prev = NULL;

  while (NULL != tf)
    {
      if (!strcmp(path, tf->path))
	{
	  /* clip out of the list first */
	  if (NULL == prev) temp_files = tf->next;
	  else prev->next = tf->next;
	  unlink(tf->path);	/* remove file */
	  FREE(tf->path);	/* Added 03/10/92 by DPB */
	  FREE(tf);		/* release memory */
	  break;		/* leave loop */
	}
      prev = tf;
      tf = tf->next;
    }
}

/* ------------------------------------------------------------------------- */
void
RemoveAllTempFiles()
{
  struct temp_file_record *tf = temp_files, *next;

  while (NULL != tf)
    {
      next = tf->next;
      unlink(tf->path);
      FREE(tf->path);
      FREE(tf);
      tf = next;
    }
}
/* ------------------------------------------------------------------------- */
int
VerifyTempFile(path) char *path;
{
  struct temp_file_record *tf;

  for (tf = temp_files ; NULL != tf ; tf = tf->next )
    {
      if (!strcmp(path, tf->path)) return 1;
    }
  return 0;
}
/* ------------------------------------------------------------------------- */
struct keyword_entry_struct nkw[] =
{
  { ":command", NULL, KEYWORD_RAW },
  { ":id", NULL, KEYWORD_RAW },
  { ":key", NULL, KEYWORD_RAW },
  { ":args", NULL, KEYWORD_SEXP },
  { ":name", NULL, KEYWORD_RAW },
} ;

void
SendCallback(c)
     t_sexp c;                 /* Sexp for callback */
{
  if (MBstringp(c))
      MBtransmit_Cstring(MBus, MBCstring(c));
}

void
SendNotify(e, path)
     t_sexp e;			/* Sexp for notify request */
     char *path;		/* path of the file */
{
  char *domain;			/* notify domain */
  char *tag;			/* notify tag */
  char *command;		/* notify command */
  char *id;			/* notify id */
  char *key;			/* notify key (for file name) */
  char *name;			/* name for id, file name if not set */
  t_sexp s_args;		/* notify extended args */
  t_sexp s_path;		/* Sexp for quoting path */
  int i;			/* scratch index variable */

  if (MBconsp(e) && MBlength(e) >= 2)
    {
      /* output is (tag domain (command id key name args))
       * where name defaults to the file name if not set
       */
      domain = MBprint_Cstring(MBnth(e, 1));
      tag = MBprint_Cstring(MBnth(e, 0));

      i = 0;
      MBparse_keywords(MBnthcdr(e, 2), nkw, ARRAY_SIZE(nkw));
      command = (char *) nkw[i++].result;
      id = (char *) nkw[i++].result;
      key = (char *) nkw[i++].result;
      s_args = (t_sexp) nkw[i++].result;
      name = (char *) nkw[i++].result;

      MBtransmit_Cstring(MBus, "(");
      MBtransmit_Cstring(MBus, tag);
      MBtransmit_Cstring(MBus, " ");
      MBtransmit_Cstring(MBus, domain);
      MBtransmit_Cstring(MBus, " (");
      if (NULL != command) MBtransmit_Cstring(MBus, command);
      if (NULL != id)
	{
	  MBtransmit_Cstring(MBus, " ");
	  MBtransmit_Cstring(MBus, id);
	}
      if (NULL != key)
	{
	  MBtransmit_Cstring(MBus, " ");
	  MBtransmit_Cstring(MBus, key);
	}

      MBtransmit_Cstring(MBus, " ");
      if (NULL == name)
	{
	  /* generate quoted version of file name */
	  s_path = MBput_Cstring(NULL, path);
	  path = MBprint_Cstring(s_path);
	  MBtransmit_Cstring(MBus, path);
	  FREE(path);
	  MBfree(s_path);
	}
      else
	MBtransmit_Cstring(MBus, name);

      if (NULL != s_args)
	{
	  char *args = MBprint_Cstring(s_args);
	  MBtransmit_Cstring(MBus, " ");
	  if (MBconsp(s_args))
	    MBtransmit_buffer(MBus, args + 1, strlen(args) - 2);
	  else
	    MBtransmit_Cstring(MBus, args);
	  FREE(args);
	}
      MBtransmit_Cstring(MBus, "))");

      FREE(tag); FREE(domain);
      FREE(command); FREE(id); FREE(key);
      FREE(name);
    }
}
/* ------------------------------------------------------------------------- */
int
CopyFile(to, from) char *to, *from;
{
  int f_fd, t_fd;
  int n;
  char buff[BUF_SIZE], bak_file[BUF_SIZE];

  sprintf(bak_file, "%s%s", to, BAK_EXT);
  unlink(bak_file);			/* Get rid of backup if it exists */
  if (0 != link(to, bak_file) && errno != ENOENT)
  {
    sprintf (buff, "Unable to make backup of %s\nError %s", to,
	     strerror(errno));
    InfoLine(1, "Clerk Message", buff, FALSE);
    return FALSE;
  }
  unlink(to);
  if (0 != link(from, to))
  {
    if (errno == EXDEV) /* different file systems */
    {
      if ((f_fd = open(from, O_RDONLY)) >= 0)
      {
	/* The 666 mode will be tailored by the user's umask */
	if ((t_fd = open(to, O_CREAT | O_RDWR, 0666)) >= 0)
	{
	  /* Seems like we could have a problem here. That is, it is
	   * possible to get only part of the file copied.  Maybe we
	   * should test for that also?
	   */
	  while ((n = read(f_fd, buff, BUF_SIZE)) > 0
		 && write(t_fd, buff, n) == n)
	      ;
	  close(f_fd);
	  close(t_fd);
	  unlink(bak_file);	/* Get rid of backup if it exists */
	  return TRUE;
	}
	else 
	{
	  link(bak_file, to);	/* Restore the backup */
	  sprintf (buff, "Unable to open %s", to);
	  InfoLine(1, "Clerk Message", buff, FALSE);
	  close(f_fd);
	  return FALSE;
	}
      }
      else
      {
	link(bak_file, to);	/* Restore the backup */
	sprintf (buff, "Unable to open %s for read", from);
	InfoLine(1, "Clerk Message", buff, FALSE);
	return FALSE;
      }
    }
    else
    {
      link(bak_file, to);	/* Restore the backup */
      sprintf (buff, "Unable to more the file %s to %s.\nError %s",
	       from, to, strerror(errno));
      InfoLine(1, "Clerk Message", buff, FALSE);
      return FALSE;
    }
  }
  unlink(bak_file);		/* Get rid of backup if it exists */
  return TRUE;			/* File moved */
}
/* ------------------------------------------------------------------------- */
/* (receive <file> :file "newfile"
		   :notify (<tag> <domain> :command :id :key :args)
   )
 */

struct keyword_entry_struct rkw[] =
{
  { ":file", NULL, KEYWORD_COOKED },
  { ":notify", NULL, KEYWORD_SEXP },
  { ":callback", NULL, KEYWORD_SEXP },
} ;

void
ReceiveFile(e, valid_dirs)
     t_sexp e;
     struct directory_entry *valid_dirs;
{
  t_sexp s_file;		/* the file object - name of the temp file */
  char *path;			/* path to temp file */
  char *file;			/* path to move temp file to */
  t_sexp s_notify;		/* notify request */
  t_sexp s_callback;            /* callback request */
  int i;			/* scratch index variable */
  struct directory_entry *best_match;
  char buf[BUF_SIZE];

  /* the temp file name */
  s_file = MBnth(e, 0);

  i = 0;
  MBparse_keywords(MBnthcdr(e, 1), rkw, ARRAY_SIZE(rkw));
  file = (char *) rkw[i++].result;
  s_notify = (t_sexp) rkw[i++].result;
  s_callback = (t_sexp) rkw[i++].result;

  if (MBstringp(s_file))
    {
      path = MBCstring(s_file);
      if (VerifyTempFile(path))
	{
	  if (file)
	  {
	    best_match = FindBestMatch(file, valid_dirs);
	    if (CheckPermissions (best_match, WRITE_PERMISSION))
	    {
	      if (CopyFile(file, path))
	      {
		RemoveTempFile(path); /* clips list and removes file */
		SendNotify(s_notify, file);
		SendCallback(s_callback);
	      }
	      /* Question:  do we send the notify and the callback here
	       * if the copy failed?  For now, don't.
	       */
	    }
	    else
	    {
	      sprintf(buf, "No clerk write permission on %s", file);
	      InfoLine(1, "Clerk message", buf, FALSE);
	      /* Question:  do we send the notify and the callback here
	       * if the copy failed?  For now, don't.
	       */
	    }
	  }
	  else {
	    SendNotify(s_notify, path);
	    SendCallback(s_callback);
	  }
	}
      FREE(path);
    }
  
  FREE(file);
  return;

}
/* ------------------------------------------------------------------------- */
void
RemoveFiles(e) t_sexp e;
{
  char *path;
  t_sexp f;

  for ( ; NULL != e ; e = MBcdr(e))
    {
      f = MBcar(e);
      if (MBstringp(f))
	{
	  path = MBCstring(f);
	  RemoveTempFile(path);
	  FREE(path);
	}
    }
}
/* ------------------------------------------------------------------------- */
main(argc, argv)
     int argc;
     char *argv[];
{
  int loop;
  struct directory_entry *valid_dirs = NULL;
  time_t mod_time = 0;
  t_sexp m, comm;

  DealWithCommandLine(argc, argv);
  MBus = MBConnect(host, port);
  if (NULL == MBus)
    {
      fprintf(stderr, "Could not connect to MBus\n");
      exit(1);
    }

  MBSetConnectionBlocking(MBus, 1);
  {
    char buff[2048];

    /* If we have no user name, then we assume this is a server. */
    if (!strcmp(cb_user, ""))
    {
      sprintf(buff, "(id \"File Clerk for server\") (accept \".*\" (\"%s.%s.%s\"))",
	      cb_clerk_domain, cb_server_domain, cb_base_domain);
    } else {      
      sprintf(buff, "(id \"File Clerk for %s\") (accept \".*\" (\"%s.%s.%s.%s\" \"%s.%s.%s\" \"%s.%s\" \"%s.%s.%s\"))",
	      cb_user,
	      cb_clerk_domain, cb_user, cb_ui_domain, cb_base_domain,
	      cb_user, cb_ui_domain, cb_base_domain,
	      cb_ui_domain, cb_base_domain,
	      cb_clerk_domain, cb_ui_domain, cb_base_domain);
    }
    MBtransmit_buffer(MBus, buff, strlen(buff));
  }
  
  /* Intercept raw block requests and store them in a temp file */
  MBraw_filter = FileBlockHandler;

  /* We could do without this call here and things would work.  However,
   * warnings and such wouldn't appear until the first time a message bus
   * message came in.
   */
  checkDirectories(dir_path, &mod_time, &valid_dirs);

  loop = 1;
  while (loop)
    {
      while (!(m = MBNextMessage(MBus)))
	;
      checkDirectories(dir_path, &mod_time, &valid_dirs);
      for ( comm = MBnthcdr(m, 2) ; NULL != comm ; comm = MBcdr(comm))
	{
	  t_sexp item = MBcar(comm);
	  t_sexp key = MBcar(item);

	  if (!MBcompare_Cstring(key, "send"))
	      SendFile(MBcdr(item), valid_dirs);
	  if (!MBcompare_Cstring(key, "receive"))
	      ReceiveFile(MBcdr(item), valid_dirs);
	  if (!MBcompare_Cstring(key, "remove")) RemoveFiles(MBcdr(item));
	  if (!MBcompare_Cstring(key, "clean-up")) RemoveAllTempFiles();
#ifdef DEBUG
	  if (!MBcompare_Cstring(key, "report")) reportDirectories(valid_dirs);
#endif
	  if (!MBcompare_Cstring(key, "shutdown"))
	  {
	    loop=0;
	    RemoveAllTempFiles();
	    break;
	  }
	}
      MBfree(m);
    }
  MBtransmit_Cstring(MBus, "(close)");
  exit (0);
}
