/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/*	block.c: Block manipulation functions */
/* Block allocation */
/* $Source: /import/kaplan/kaplan/carroll/cb/mbus/lib/RCS/block.c,v $ */

static char rcsid[] = "block.c $Revision: 2.1.1.3 $ $Date: 91/11/15 13:35:21 $ $State: Exp $ $Author: carroll $";
/* ------------------------------------------------------------------------ */

#include <stdio.h>
#include "api.h"
#include "mbus.h"

/* ------------------------------------------------------------------------ */
/* All the data block stuff for managing them. */

/* Don't worry about pointer complaints here */
#ifdef __STDC__
t_malloc (*MBmalloc_func)(unsigned int) = malloc; /* function to malloc with */
int (*MBfree_func)(void *) = free;	/* function to free memory */
#else
t_malloc (*MBmalloc_func)() = malloc;	/* function to malloc with */
int (*MBfree_func)() = free;		/* function to free memory */
#endif

#define N_BLOCK_GROUP	8		/* blocks per block group */

/* Actual size of a block structure with n data bytes */
#define BLOCK_SIZE(n)   (sizeof(struct mb_block_stub_struct) + (n))

typedef struct mb_block_group_struct
{
  struct mb_block_group_struct *next;		/* next block group */
  void *blocks;				/* pointer to blocks in the group */
} * t_mb_block_group ;

struct mb_block_free_list
{
  t_block_int size;			/* size of block data area */
  int total;				/* total number malloc'd */
  int in_use;				/* number of blocks in use */
  struct mb_block_struct *free_list;		/* free blocks */
  struct mb_block_group_struct *groups;	/* groups of blocks */
};

static struct mb_block_free_list BlockFreeLists[] =
{
  { 32, 0, 0, NULL, NULL },
  { 256, 0, 0, NULL, NULL },
  { 4096, 0, 0, NULL, NULL },
};

#define N_BLOCK_SIZES	(sizeof(BlockFreeLists)/sizeof(struct mb_block_free_list))

/* ------------------------------------------------------------------------ */
/* The object types */

#define N_OBJECT_GROUP 16		/* objects per group */

/* This is different from blocks because the objects are all the same
 * size.
 */
typedef struct mb_object_group_struct
{
  struct mb_object_group_struct *next;
  struct mb_object object[N_OBJECT_GROUP];
} * t_mb_object_group ;

static struct mb_object_free_list
{
  int total;
  int in_use;
  struct mb_object_group_struct *groups;
  struct mb_object nil;
} ObjectFreeList = { 0, 0, NULL, { MB_NULL, MB_CHECK_WORD_VALUE, NULL } };

struct mb_object *MBnil = &(ObjectFreeList.nil);

/* Don't use the external version, in case someone messes up MBnil. This
 * version will always be correct
 */
#undef FREEP
#define FREEP(o) ((o) != NULL && (o)->next != &ObjectFreeList.nil)
/* ------------------------------------------------------------------------ */
/* This is really just a debugging function */
void
MBPrintMemoryUsage()
{
  int i;

  printf("Objects: %d of %d in use\n",
         ObjectFreeList.in_use,ObjectFreeList.total);
  printf("Blocks:  Size  In Use  Total\n");
  /* The spaces and size values for the printf in the for loop are magic,
   * and derived from the printf statement directly above.
   */
  for ( i = 0 ; i < N_BLOCK_SIZES ; ++i )
    printf(" %12d %7d %6d\n",
           BlockFreeLists[i].size,
           BlockFreeLists[i].in_use,
           BlockFreeLists[i].total);
}
/* ------------------------------------------------------------------------ */
/* This is really just a debugging function */
char *
MBDumpMemoryUsage()
{
  int i;
  char *s;
  static char buff[128];

  s = buff;
  sprintf(buff,"(objects %d %d) ",
	 ObjectFreeList.in_use,ObjectFreeList.total);
  s = buff + strlen(buff);
  sprintf(s,"(blocks");
  s = s + strlen(s);
  for ( i = 0 ; i < N_BLOCK_SIZES ; ++i )
    {
      sprintf(s," (%d %d %d)",
	      BlockFreeLists[i].size,
	      BlockFreeLists[i].in_use,
	      BlockFreeLists[i].total);
      s = s + strlen(s);
    }
  sprintf(s,")");
  return buff;
}
   
/* ------------------------------------------------------------------------ */
/* Return an element of the the block free list array */
static struct mb_block_free_list *
GetBlockFreeList(size) t_block_int size;
{
  int i;

  for ( i = 0 ; i < N_BLOCK_SIZES ; ++i )
    if (BlockFreeLists[i].size >= size)
      return &(BlockFreeLists[i]);
  return NULL;
}

/* ------------------------------------------------------------------------ */
/* called when a new block is needed but the free list is empty. */
static void
MBGetBlockGroup(flist) struct mb_block_free_list *flist;
{
  t_mb_block_group group;
  t_mb_block block;
  char *base;
  int size;
  int i;

  /* allocate the new group */
  group = (t_mb_block_group)
    MBmalloc_func(sizeof(struct mb_block_group_struct));

  if (NULL == group)			/* allocation failure */
    {
      if (MBLogLevel) fprintf(stderr, "Cannot allocate block group header\n");
      return;
    }

  size = BLOCK_SIZE(flist->size);

  group->blocks = MBmalloc_func(N_BLOCK_GROUP * size);
  if (NULL == group->blocks)		/* allocation failure */
    {
      if (MBLogLevel) fprintf(stderr, "Cannot allocate block group\n");
      MBfree_func(group);
      return;
    }

  /* put this group in the group list */
  group->next = flist->groups;
  flist->groups = group;

  for ( i = 0, base = group->blocks ; i < N_BLOCK_GROUP ; ++i, base += size )
    {
      block = (t_mb_block) base;
      block->size = flist->size;	/* set the size */
      block->flags = 0;			/* clear all flags */
      block->next = flist->free_list;	/* link into free list */
      flist->free_list = block;		/* update free list */
      flist->total += 1;		/* one more block */
    }
}
/* ------------------------------------------------------------------------ */
/* Allocate a new block (or get one from the free list) */
t_mb_block
MBGetBlock(size) t_block_int size;
{
  t_mb_block block = NULL;
  struct mb_block_free_list *list = GetBlockFreeList(size);

  if (NULL == list)			/* couldn't find a free list */
    {
      if (MBLogLevel)
	fprintf(stderr,"Bad size request for data block (%u)\n",size);
      return NULL;
    }

  size = list->size;
  if (NULL == list->free_list) MBGetBlockGroup(list);

  if (NULL != list->free_list)
    {
      block = list->free_list;
      list->free_list = block->next;
      block->count = 0;
      block->next = NULL;
      block->flags |= MB_IN_USE;	/* mark as in use */
      list->in_use += 1;
    }
  return block;				/* NULL if no block available */
}

/* ------------------------------------------------------------------------ */
/* release a block (put it on a free list) */
void
MBFreeBlock(block) t_mb_block block;
{
  struct mb_block_free_list *list = GetBlockFreeList(block->size);

  if (NULL == list)
    {
      if (MBLogLevel)
	fprintf(stderr,"Bad size on block return (%ud)\n",block->size);
    }
  else
    {
      block->next = list->free_list;
      list->free_list = block;
      block->flags &= ~MB_IN_USE;
      list->in_use -= 1;
    }
}

/* ------------------------------------------------------------------------ */
/* Are there blocks of a larger size? */
int
MBisLargestBlock(b) t_mb_block b;
{
  struct mb_block_free_list *list = GetBlockFreeList(b->size);

  return list - BlockFreeLists >= N_BLOCK_SIZES-1;
}
/* ------------------------------------------------------------------------ */
/* Get a new block of larger size, and copy the given block to it,
 * freeing the given block is a new one is allocated
 */
t_mb_block
MBUpgradeBlock(block)
     t_mb_block block;
{
  struct mb_block_free_list *list = GetBlockFreeList(block->size);
  t_mb_block new;
  int index;

  if (NULL == list) return NULL;
  index = list - BlockFreeLists;

  if (index >= N_BLOCK_SIZES-1) return NULL;

  new = MBGetBlock(BlockFreeLists[index+1].size);
  if (new)
    {
      memcpy((void *)new->data, (void *)block->data, block->size);
      new->count = block->count;
      MBFreeBlock(block);
    }
  return new;
}
/* ------------------------------------------------------------------------ */
static void
MBGetObjectGroup()
{
  struct mb_object *flist = &(ObjectFreeList.nil);
  t_mb_object_group group;
  int i;

  group = (t_mb_object_group) MBmalloc_func(sizeof(struct mb_object_group_struct));

  if (NULL == group) return;

  group->next = ObjectFreeList.groups;
  ObjectFreeList.groups = group;

  for ( i = 0 ; i < N_OBJECT_GROUP ; ++i )
    {
      group->object[i].check_word = MB_CHECK_WORD_VALUE; /* mark it */
      group->object[i].next = flist->next;
      flist->next = &(group->object[i]);
      group->object[i].flags = 0;	/* clear all flags */
      ObjectFreeList.total += 1;
    }
}
/* ------------------------------------------------------------------------ */
struct mb_object *
MBGetObject()
{
  struct mb_object *list = &(ObjectFreeList.nil);
  struct mb_object *new;

  if (NULL == list->next) MBGetObjectGroup();

  if (NULL == list->next)
    {
      if (MBLogLevel) fprintf(stderr, "MBObject: out of memory\n");
      return NULL;
    }

  new = list->next;			/* get the first one */
  list->next = new->next;		/* set free to second object */

  /* every allocated object should have its next field set to the address of
   * the nil object (ObjectFreeList.nil), to mark it as allocated.
   */
  new->next = list;
  new->type = MB_INVALID;		/* set invalid type here */
  new->flags |= MB_IN_USE;

  ObjectFreeList.in_use += 1;
  return new;
}
/* ------------------------------------------------------------------------ */
void
MBFreeObject(o) struct mb_object *o;
{
  if (NULL == o) return;
  else if (o->check_word != MB_CHECK_WORD_VALUE)
    {
      fprintf(stderr,
	      "MBus says ``Hey! This isn't my object, I can't free it''\n");
    }
  else if (o == &ObjectFreeList.nil)
    {
      if (MBLogLevel > 0)
	fprintf(stderr, "MBus: Attempt to free nil object\n");
    }
  else if (FREEP(o))
    {
      if (MBLogLevel > 1) fprintf(stderr, "Double free of object\n");
    }
  else
    {
      o->next = ObjectFreeList.nil.next;
      o->flags &= ~MB_IN_USE;
      ObjectFreeList.nil.next = o;
      ObjectFreeList.in_use -= 1;
    }
}
/* ------------------------------------------------------------------------ */
struct mb_object *
MBGetCons()
{
  struct mb_object *obj = MBGetObject();

  obj->type = MB_CONS;
  obj->object.cons.car = obj->object.cons.cdr = NULL;
  return obj;
}
/* ------------------------------------------------------------------------ */
struct mb_object *
MBGetName()
{
  struct mb_object *obj = MBGetObject();

  obj->type = MB_NAME;
  MBChunkReset(&(obj->object.chunk));
  return obj;
}
/* ------------------------------------------------------------------------ */
struct mb_object *
MBGetRaw()
{
  struct mb_object *obj = MBGetObject();

  obj->type = MB_RAW;
  MBChunkReset(&(obj->object.chunk));
  return obj;
}
/* ------------------------------------------------------------------------ */
struct mb_object *
MBGetString()
{
  struct mb_object *obj = MBGetObject();

  obj->type = MB_STRING;
  MBChunkReset(&(obj->object.chunk));
  return obj;
}
/* ------------------------------------------------------------------------ */
struct mb_object *
MBGetChunkPointer()
{
  struct mb_object *obj = MBGetObject();

  obj->type = MB_CHUNK_POINTER;
  MBCPSet(&(obj->object.cp),(struct mb_chunk *)NULL);	/* reset */
  return obj;
}
/* ------------------------------------------------------------------------ */
static void
MBClearBlockMarks()
{
  t_mb_block_group group;
  int i, j;
  int size;
  char *base;

  for ( i = 0 ; i < N_BLOCK_SIZES ; ++i )
    {
      for ( group = BlockFreeLists[i].groups
	   ; NULL != group
	   ; group = group->next ) 
	{
	  size = BLOCK_SIZE(BlockFreeLists[i].size);
	  for ( j = 0, base = group->blocks
	       ; j < N_BLOCK_GROUP
	       ; ++j, base += size )
	    ((t_mb_block)base)->flags &= ~MB_MARKED;
	}
    }

}
/* ------------------------------------------------------------------------ */
static void
MBClearObjectMarks()
{
  struct mb_object_group_struct *group;
  int i;

  for ( group = ObjectFreeList.groups ; NULL != group ; group = group->next )
    {
      for ( i = 0 ; i < N_OBJECT_GROUP ; ++i )
	group->object[i].flags &= ~MB_MARKED;
    }
}
/* ------------------------------------------------------------------------ */
void
MBClearMarks()
{
  MBClearBlockMarks();
  MBClearObjectMarks();
}
/* ------------------------------------------------------------------------ */
static void
MBGCBlocks()
{
  t_mb_block_group group;
  int i, j;
  int size;
  char *base;
  t_mb_block block;

  for ( i = 0 ; i < N_BLOCK_SIZES ; ++i )
    {
      for ( group = BlockFreeLists[i].groups
	   ; NULL != group
	   ; group = group->next ) 
	{
	  size = BLOCK_SIZE(BlockFreeLists[i].size);
	  BlockFreeLists[i].free_list = NULL; /* clear free list */
	  BlockFreeLists[i].in_use = BlockFreeLists[i].total;
	  for ( j = 0, base = group->blocks
	       ; j < N_BLOCK_GROUP
	       ; ++j, base += size )
	    {
	      block = (t_mb_block) base;
	      if (! (block->flags & MB_MARKED))
		{
		  block->next = BlockFreeLists[i].free_list;
		  BlockFreeLists[i].free_list = block;
		  BlockFreeLists[i].in_use -= 1;
		  block->flags &= ~MB_IN_USE;
		}
	    }
	}
    }
}
/* ------------------------------------------------------------------------ */
static void
MBGCObjects()
{
  struct mb_object_group_struct *group;
  t_sexp obj;
  int i;

  ObjectFreeList.nil.next = NULL;		/* clear free list */
  ObjectFreeList.in_use = ObjectFreeList.total;

  for ( group = ObjectFreeList.groups ; NULL != group ; group = group->next )
    {
      for ( i = 0 ; i < N_OBJECT_GROUP ; ++i )
	{
	  obj = group->object + i;
	  if (! (obj->flags & MB_MARKED))
	    MBFreeObject(obj);
	}
    }
}
/* ------------------------------------------------------------------------ */
void
MBGC()
{
  MBGCBlocks();
  MBGCObjects();
}
/* ------------------------------------------------------------------------ */
