/* Minibuffer input and completion.
   Copyright (C) 1985, 1986, 1992, 1993, 1994 Free Software Foundation, Inc.
   Copyright (C) 1994, 1995 Amdahl Corporation.

This file is part of XEmacs.

XEmacs is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

XEmacs is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with XEmacs; see the file COPYING.  If not, write to the Free
Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Synched up with: Mule 2.0, FSF 19.28.  Mule-ized except as noted.
   Substantially different from FSF. */

#include <config.h>
#include "lisp.h"

#include "buffer.h"
#include "commands.h"
#include "device-stream.h"
#include "frame.h"
#include "insdel.h"
#include "redisplay.h"
#include "window.h"

/* Depth in minibuffer invocations.  */
int minibuf_level;

Lisp_Object Qcompletion_ignore_case;

/* Nonzero means completion ignores case.  */
int completion_ignore_case;

/* List of regexps that should restrict possible completions.  */
Lisp_Object Vcompletion_regexp_list;

/* The echo area buffer. */
Lisp_Object Vecho_area_buffer;

/* location echo area was activated from */
static enum echo_area_type internal_echo_area_status;

/* Prompt to display in front of the minibuffer contents */
Lisp_Object Vminibuf_prompt;

/* Hook to run just after entry to minibufer. */
Lisp_Object Qminibuffer_setup_hook, Vminibuffer_setup_hook;


DEFUN ("minibuffer-depth", Fminibuffer_depth, Sminibuffer_depth, 0, 0, 0,
  "Return current depth of activations of minibuffer, a nonnegative integer.")
  ()
{
  return make_number (minibuf_level);
}

/* The default buffer to use as the window-buffer of minibuffer windows */
/*  Note there is special code in kill-buffer to make this unkillable */
Lisp_Object Vminibuffer_zero;


/* Actual minibuffer invocation. */

static Lisp_Object
read_minibuffer_internal_unwind (Lisp_Object unwind_data)
{
  Lisp_Object frame;
  XWINDOW (minibuf_window)->last_modified[CURRENT_DISP] = Qzero;
  XWINDOW (minibuf_window)->last_modified[DESIRED_DISP] = Qzero;
  XWINDOW (minibuf_window)->last_modified[CMOTION_DISP] = Qzero;
  XWINDOW (minibuf_window)->last_facechange[CURRENT_DISP] = Qzero;
  XWINDOW (minibuf_window)->last_facechange[DESIRED_DISP] = Qzero;
  XWINDOW (minibuf_window)->last_facechange[CMOTION_DISP] = Qzero;
  Vminibuf_prompt = Felt (unwind_data, Qzero);
  minibuf_level = XINT (Felt (unwind_data, make_number (1)));
  while (CONSP (unwind_data))
  {
    Lisp_Object victim = unwind_data;
    unwind_data = XCDR (unwind_data);
    free_cons (XCONS (victim));
  }

  /* If cursor is on the minibuffer line,
     show the user we have exited by putting it in column 0.  */
  frame = Fselected_frame (Qnil);
  if (!noninteractive
      && !NILP (frame)
      && !NILP (XFRAME (frame)->minibuffer_window))
    {
      struct window *w = XWINDOW (XFRAME (frame)->minibuffer_window);
      redisplay_move_cursor (w, 0, 0);
    }

  return Qnil;
}

DEFUN ("read-minibuffer-internal", 
       Fread_minibuffer_internal, Sread_minibuffer_internal, 
       1, 1, 0,
       "Lowest-level interface to minibuffers.  Don't call this.")
  (prompt)
     Lisp_Object prompt;
{
  /* This function can GC */
  int speccount = specpdl_depth ();
  Lisp_Object val;

  CHECK_STRING (prompt, 0);

  record_unwind_protect (read_minibuffer_internal_unwind,
                         list2 (Vminibuf_prompt,
                                make_number (minibuf_level)));
  Vminibuf_prompt = LISP_GETTEXT (prompt);

  if (!NILP (Vrun_hooks) && !NILP (Vminibuffer_setup_hook))
    call1 (Vrun_hooks, Qminibuffer_setup_hook);
    
  minibuf_level++;
  clear_echo_area (selected_frame ());

  val = call_command_loop (Qt);

  return (unbind_to (speccount, val));
}



/* Completion hair */

/* Compare exactly LEN chars of strings at S1 and S2,
   ignoring case if appropriate.
   Return -1 if strings match,
   else number of chars that match at the beginning.  */

/* Note that this function works in Charcounts, unlike most functions.
   This is necessary for many reasons, one of which is that two
   strings may match even if they have different numbers of bytes,
   if IGNORE_CASE is true. */

Charcount
scmp_1 (CONST Bufbyte *s1, CONST Bufbyte *s2, Charcount len,
	int ignore_case)
{
  Charcount l = len;

  if (ignore_case)
    {
      while (l)
        {
          Bufbyte c1 = DOWNCASE (current_buffer, charptr_to_emchar (s1));
          Bufbyte c2 = DOWNCASE (current_buffer, charptr_to_emchar (s2));

          if (c1 == c2)
            {
              l--;
              INC_CHARPTR (s1);
              INC_CHARPTR (s2);
            }
          else
            break;
        }
    }
  else
    {
      while (l && charptr_to_emchar (s1) == charptr_to_emchar (s2))
	{
	  l--;
	  INC_CHARPTR (s1);
	  INC_CHARPTR (s2);
	}
    }

  if (l == 0)
    return -1;
  else return len - l;
}


int
regexp_ignore_completion_p (CONST Bufbyte *completion, Bytecount len)
{
  /* Ignore this element if it fails to match all the regexps.  */
  if (!NILP (Vcompletion_regexp_list))
  {
    Lisp_Object regexps;
    for (regexps = Vcompletion_regexp_list;
         CONSP (regexps);
         regexps = XCDR (regexps))
    {
      Lisp_Object re = XCAR (regexps);
      if (STRINGP (re)
          && (fast_string_match (re, completion, len, 0) >= 0))
        return (1);
    }
  }
  return (0);
}


/* Callers should GCPRO, since this may call eval */
static int
ignore_completion_p (Lisp_Object completion_string,
                     Lisp_Object pred, Lisp_Object completion)
{
  if (regexp_ignore_completion_p (string_data (XSTRING (completion_string)),
                                  string_length (XSTRING (completion_string))))
    return (1);
  
  /* Ignore this element if there is a predicate
     and the predicate doesn't like it. */
  if (!NILP (pred))
  {
    Lisp_Object tem;
    if (EQ (pred, Qcommandp))
      tem = Fcommandp (completion);
    else
      tem = call1 (pred, completion);
    if (NILP (tem))
      return (1);
  }
  return (0);
}




DEFUN ("try-completion", Ftry_completion, Stry_completion, 2, 3, 0,
  "Return common substring of all completions of STRING in ALIST.\n\
Each car of each element of ALIST is tested to see if it begins with STRING.\n\
All that match are compared together; the longest initial sequence\n\
common to all matches is returned as a string.\n\
If there is no match at all, nil is returned.\n\
For an exact match, t is returned.\n\
\n\
ALIST can be an obarray instead of an alist.\n\
Then the print names of all symbols in the obarray are the possible matches.\n\
\n\
ALIST can also be a function to do the completion itself.\n\
It receives three arguments: the values STRING, PREDICATE and nil.\n\
Whatever it returns becomes the value of `try-completion'.\n\
\n\
If optional third argument PREDICATE is non-nil,\n\
it is used to test each possible match.\n\
The match is a candidate only if PREDICATE returns non-nil.\n\
The argument given to PREDICATE is the alist element or the symbol from the obarray.")
  (string, alist, pred)
     Lisp_Object string, alist, pred;
{
  /* This function can GC */
  Lisp_Object bestmatch, tail;
  Charcount bestmatchsize = 0;
  int list;
  int indice = 0;
  int obsize = 0;
  int matchcount = 0;
  Lisp_Object bucket;
  Charcount slength, blength;

  CHECK_STRING (string, 0);

  if (CONSP (alist))
  {
    Lisp_Object tem = XCAR (alist);
    if (SYMBOLP (tem))          /* lambda, autoload, etc.  Emacs-lisp sucks */
      return call3 (alist, string, pred, Qnil);
    else
      list = 1;
  }
  else if (VECTORP (alist))
    list = 0;
  else if (NILP (alist))
    list = 1;
  else
    return call3 (alist, string, pred, Qnil);

  bestmatch = Qnil;
  blength = 0;
  slength = string_char_length (XSTRING (string));

  /* If ALIST is not a list, set TAIL just for gc pro.  */
  tail = alist;
  if (!list)
    {
      obsize = vector_length (XVECTOR (alist));
      bucket = vector_data (XVECTOR (alist))[indice];
    }

  while (1)
    {
      /* Get the next element of the alist or obarray. */
      /* Exit the loop if the elements are all used up. */
      /* elt gets the alist element or symbol.
	 eltstring gets the name to check as a completion. */
      Lisp_Object elt;
      Lisp_Object eltstring;

      if (list)
	{
	  if (NILP (tail))
	    break;
	  elt = Fcar (tail);
	  eltstring = Fcar (elt);
	  tail = Fcdr (tail);
	}
      else
	{
	  if (!EQ (bucket, Qzero))
	    {
              struct Lisp_Symbol *next = symbol_next (XSYMBOL (bucket));
	      elt = bucket;
	      eltstring = Fsymbol_name (elt);
              if (next)
		XSETSYMBOL (bucket, next);
	      else
		bucket = Qzero;
	    }
	  else if (++indice >= obsize)
	    break;
	  else
	    {
	      bucket = vector_data (XVECTOR (alist))[indice];
	      continue;
	    }
	}

      /* Is this element a possible completion? */

      if (STRINGP (eltstring))
	{
	  Charcount eltlength = string_char_length (XSTRING (eltstring));
	  if (slength <= eltlength
	      && (0 > scmp (string_data (XSTRING (eltstring)),
                            string_data (XSTRING (string)),
                            slength)))
	    {
              {
                struct gcpro gcpro1, gcpro2, gcpro3, gcpro4;
                int loser;
                GCPRO4 (tail, string, eltstring, bestmatch);
                loser = ignore_completion_p (eltstring, pred, elt);
                UNGCPRO;
                if (loser)      /* reject this one */
                  continue;
              }

	      /* Update computation of how much all possible
		 completions match */

	      matchcount++;
	      if (NILP (bestmatch))
		{
		  bestmatch = eltstring;
                  blength = eltlength;
		  bestmatchsize = eltlength;
		}
	      else
		{
		  Charcount compare = min (bestmatchsize, eltlength);
		  Charcount matchsize =
		    scmp (string_data (XSTRING (bestmatch)),
			  string_data (XSTRING (eltstring)),
			  compare);
		  if (matchsize < 0)
		    matchsize = compare;
		  if (completion_ignore_case)
		    {
		      /* If this is an exact match except for case,
			 use it as the best match rather than one that is not
			 an exact match.  This way, we get the case pattern
			 of the actual match.  */
		      if ((matchsize == eltlength
			   && matchsize < blength)
			  ||
			  /* If there is more than one exact match ignoring
			     case, and one of them is exact including case,
			     prefer that one.  */
			  /* If there is no exact match ignoring case,
			     prefer a match that does not change the case
			     of the input.  */
			  ((matchsize == eltlength)
			   ==
			   (matchsize == blength)
			   && 0 > scmp_1 (string_data (XSTRING (eltstring)),
					  string_data (XSTRING (string)),
					  slength, 0)
			   && 0 <= scmp_1 (string_data (XSTRING (bestmatch)),
					   string_data (XSTRING (string)), 
					   slength, 0)))
                      {
			bestmatch = eltstring;
                        blength = eltlength;
                      }
		    }
		  bestmatchsize = matchsize;
		}
	    }
	}
    }

  if (NILP (bestmatch))
    return Qnil;		/* No completions found */
  /* If we are ignoring case, and there is no exact match,
     and no additional text was supplied,
     don't change the case of what the user typed.  */
  if (completion_ignore_case
      && bestmatchsize == slength
      && blength > bestmatchsize)
    return string;

  /* Return t if the supplied string is an exact match (counting case);
     it does not require any change to be made.  */
  if (matchcount == 1
      && bestmatchsize == slength
      && 0 > scmp_1 (string_data (XSTRING (bestmatch)),
		     string_data (XSTRING (string)),
		     bestmatchsize, 0))
    return Qt;

  /* Else extract the part in which all completions agree */
  return Fsubstring (bestmatch, Qzero, make_number (bestmatchsize));
}


DEFUN ("all-completions", Fall_completions, Sall_completions, 2, 3, 0,
  "Search for partial matches to STRING in ALIST.\n\
Each car of each element of ALIST is tested to see if it begins with STRING.\n\
The value is a list of all the strings from ALIST that match.\n\
ALIST can be an obarray instead of an alist.\n\
Then the print names of all symbols in the obarray are the possible matches.\n\
\n\
ALIST can also be a function to do the completion itself.\n\
It receives three arguments: the values STRING, PREDICATE and t.\n\
Whatever it returns becomes the value of `all-completions'.\n\
\n\
If optional third argument PREDICATE is non-nil,\n\
it is used to test each possible match.\n\
The match is a candidate only if PREDICATE returns non-nil.\n\
The argument given to PREDICATE is the alist element or\n\
the symbol from the obarray.")
  (string, alist, pred)
     Lisp_Object string, alist, pred;
{
  /* This function can GC */
  Lisp_Object tail;
  Lisp_Object allmatches;
  int list;
  int indice = 0;
  int obsize = 0;
  Lisp_Object bucket;
  Charcount slength;

  CHECK_STRING (string, 0);

  if (CONSP (alist))
  {
    Lisp_Object tem = XCAR (alist);
    if (SYMBOLP (tem))          /* lambda, autoload, etc.  Emacs-lisp sucks */
      return call3 (alist, string, pred, Qt);
    else
      list = 1;
  }
  else if (VECTORP (alist))
    list = 0;
  else if (NILP (alist))
    list = 1;
  else
    return call3 (alist, string, pred, Qt);

  allmatches = Qnil;
  slength = string_char_length (XSTRING (string));

  /* If ALIST is not a list, set TAIL just for gc pro.  */
  tail = alist;
  if (!list)
    {
      obsize = vector_length (XVECTOR (alist));
      bucket = vector_data (XVECTOR (alist))[indice];
    }

  while (1)
    {
      /* Get the next element of the alist or obarray. */
      /* Exit the loop if the elements are all used up. */
      /* elt gets the alist element or symbol.
	 eltstring gets the name to check as a completion. */
      Lisp_Object elt;
      Lisp_Object eltstring;

      if (list)
	{
	  if (NILP (tail))
	    break;
	  elt = Fcar (tail);
	  eltstring = Fcar (elt);
	  tail = Fcdr (tail);
	}
      else
	{
	  if (!EQ (bucket, Qzero))
	    {
              struct Lisp_Symbol *next = symbol_next (XSYMBOL (bucket));
	      elt = bucket;
	      eltstring = Fsymbol_name (elt);
              if (next)
		XSETSYMBOL (bucket, next);
	      else
		bucket = Qzero;
            }
	  else if (++indice >= obsize)
	    break;
	  else
	    {
	      bucket = vector_data (XVECTOR (alist))[indice];
	      continue;
	    }
	}

      /* Is this element a possible completion? */

      if (STRINGP (eltstring) 
          && (slength <= string_char_length (XSTRING (eltstring)))
	  /* Reject alternatives that start with space
	     unless the input starts with space.  */
	  && ((string_char_length (XSTRING (string)) > 0 &&
	       string_char (XSTRING (string), 0) == ' ')
	      || string_char (XSTRING (eltstring), 0) != ' ')
          && (0 > scmp (string_data (XSTRING (eltstring)),
                        string_data (XSTRING (string)),
                        slength)))
	{
	  /* Yes.  Now check whether predicate likes it. */
          struct gcpro gcpro1, gcpro2, gcpro3, gcpro4;
          int loser;
          GCPRO4 (tail, eltstring, allmatches, string);
          loser = ignore_completion_p (eltstring, pred, elt);
          UNGCPRO;
          if (!loser)
            /* Ok => put it on the list. */
            allmatches = Fcons (eltstring, allmatches);
        }
    }

  return Fnreverse (allmatches);
}

/* Useless FSFmacs functions */
/* More than useless.  I've nuked minibuf_prompt_width so they won't
   function at all in XEmacs at the moment.  They are used to
   implement some braindamage in FSF which we aren't including. --cet */

#if 0
xxDEFUN ("minibuffer-prompt", Fminibuffer_prompt, Sminibuffer_prompt, 0, 0, 0,
  "Return the prompt string of the currently-active minibuffer.\n\
If no minibuffer is active, return nil.")
  ()
{
  return (Fcopy_sequence (Vminibuf_prompt));
}

xxDEFUN ("minibuffer-prompt-width", Fminibuffer_prompt_width,
  Sminibuffer_prompt_width, 0, 0, 0,
  "Return the display width of the minibuffer prompt.")
  ()
{
  return (make_number (minibuf_prompt_width));
}
#endif

/*
 * #### The new echo area really is a buffer.  This is not
 * #### really a final implementation.  It basically just duplicates
 * #### echo_area_glyphs with a buffer.  It is a start, though,
 * #### towards providing history in the echo area.
 */

DEFUN ("message-displayed-p", Fmessage_displayed_p, Smessage_displayed_p,
       0, 1, 0,
  "Return a non-nil value if a message is presently displayed in the\n\
minibuffer's echo area.  If optional argument RETURN-STRING is non-nil,\n\
return a string containing the message, otherwise just return t.")
  (return_string)
   Lisp_Object return_string;
{
  if (echo_area_active (selected_frame ()))
  {
    if (!NILP (return_string))
      return echo_area_contents (selected_frame ());
    else
      return (Qt);
  }
  else
    return (Qnil);
}

static void
clear_echo_area_internal (struct frame *f, int from_print)
{
  int stream_clear = internal_echo_area_status != ECHO_AREA_INACTIVE;

  internal_echo_area_status = ECHO_AREA_INACTIVE;
  erase_buffer (XBUFFER (Vecho_area_buffer), 1);
  if (stream_clear && DEVICE_IS_STREAM (XDEVICE (FRAME_DEVICE (f))))
    stream_echo_area_clear (f, !from_print);
}

void
clear_echo_area (struct frame *f)
{
  clear_echo_area_internal (f, 0);
}

void
clear_echo_area_from_print (struct frame *f)
{
  clear_echo_area_internal (f, 1);
}

void
echo_area_append (struct frame *f, CONST Bufbyte *nonreloc, Lisp_Object reloc,
		  Bytecount offset, Bytecount length,
		  enum echo_area_type type)
{
  /* some callers pass in a null string as a way of clearing the echo area.
     check for length == 0 now; if this case, neither nonreloc nor reloc
     may be valid.  */
  if (length == 0)
    return;
  
  fixup_internal_substring (nonreloc, reloc, offset, &length);
  
  /* also check it here, in case the string was really blank. */
  if (length == 0)
    return;

  internal_echo_area_status = type;
  buffer_insert_string (XBUFFER (Vecho_area_buffer), nonreloc, reloc,
			offset, length);
  /* Conditionalizing on the device type in this way is not that clean,
     but neither is having a device method, as I originally implemented
     it: all non-stream devices behave in the same way.  Perhaps
     the cleanest way is to make the concept of a "redisplayable"
     device, which stream devices are not.  Look into this more if
     we ever create another non-redisplayable device type (e.g.
     processes?  printers?). */
  if (DEVICE_IS_STREAM (XDEVICE (FRAME_DEVICE (f))))
    {
      if (type == ECHO_AREA_PRINT)
	stream_echo_area_display (f, 0);
      else
	stream_echo_area_display (f, 1);
    }
  else
    redisplay_echo_area ();
}

void
echo_area_message (struct frame *f, CONST Bufbyte *nonreloc,
		   Lisp_Object reloc, Bytecount offset, Bytecount length,
		   enum echo_area_type type)
{
  clear_echo_area (f);
  echo_area_append (f, nonreloc, reloc, offset, length, type);
}

int
echo_area_active (struct frame *f)
{
  return internal_echo_area_status != ECHO_AREA_INACTIVE;
}

enum echo_area_type
echo_area_status (struct frame *f)
{
  /* In the future there might be more than one echo area */
  return internal_echo_area_status;
}

Lisp_Object
echo_area_contents (struct frame *f)
{
  struct buffer *b = XBUFFER (Vecho_area_buffer);
  return make_string_from_buffer (b, BUF_BEGV (b), BUF_ZV (b) - BUF_BEGV (b));
}

/* Dump an informative message to the echo area.  This function takes a
   string in internal format. */
void
message_internal (CONST Bufbyte *nonreloc, Lisp_Object reloc,
		  Bytecount offset, Bytecount length)
{
  if (NILP (Vexecuting_macro))
    echo_area_message (selected_frame (), nonreloc, reloc, offset, length,
		       ECHO_AREA_MESSAGE);
}

/* The next three functions are interfaces to message_internal() that
   take strings in external format.  message() does I18N3 translating
   on the format string; message_no_translate() does not. */

static void
message_1 (CONST char *fmt, va_list args)
{
  if (fmt)
    message_internal (0, emacs_doprnt_string_va ((CONST Bufbyte *) fmt, Qnil,
						 -1, args),
		      0, -1);
  else
    message_internal (0, Qnil, 0, 0);
}

void
message (CONST char *fmt, ...)
{
  va_list args;

  va_start (args, fmt);
  if (fmt)
    fmt = GETTEXT (fmt);
  message_1 (fmt, args);
  va_end (args);
}

void
message_no_translate (CONST char *fmt, ...)
{
  va_list args;

  va_start (args, fmt);
  message_1 (fmt, args);
  va_end (args);
}


void
init_minibuf_once (void)
{
  /* This function can GC */
#ifdef I18N3
  /* #### This needs to be fixed up so that the gettext() gets called
     at runtime instead of at load time. */
#endif
  Vminibuffer_zero
    = Fget_buffer_create
      (Fpurecopy (build_string (DEFER_GETTEXT (" *Minibuf-0*"))));
  Vecho_area_buffer
    = Fget_buffer_create
      (Fpurecopy (build_string (DEFER_GETTEXT (" *Echo Area*"))));
}

void
syms_of_minibuf (void)
{
  minibuf_level = 0;
  internal_echo_area_status = ECHO_AREA_INACTIVE;
  defsubr (&Smessage_displayed_p);

  staticpro (&Vminibuf_prompt);
  Vminibuf_prompt = Qnil;

  defsymbol (&Qminibuffer_setup_hook, "minibuffer-setup-hook");
  DEFVAR_LISP ("minibuffer-setup-hook", &Vminibuffer_setup_hook,
    "Normal hook run just after entry to minibuffer.");
  Vminibuffer_setup_hook = Qnil;

  defsymbol (&Qcompletion_ignore_case, "completion-ignore-case");
  DEFVAR_BOOL ("completion-ignore-case", &completion_ignore_case,
    "Non-nil means don't consider case significant in completion.");
  completion_ignore_case = 0;

  /* Worthless doc string */
  DEFVAR_LISP ("completion-regexp-list", &Vcompletion_regexp_list,
    "List of regexps that should restrict possible completions.");
  Vcompletion_regexp_list = Qnil;

  defsubr (&Sminibuffer_depth);
#if 0
  defsubr (&Sminibuffer_prompt);
  defsubr (&Sminibuffer_prompt_width);
#endif

  defsubr (&Sread_minibuffer_internal);

  defsubr (&Stry_completion);
  defsubr (&Sall_completions);
}
