/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:  Thomas Fruchterman
 *		 John Jozwiak
 *		 Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */
/*
     Nature -- a force-directed graph-drawing program
     author: Thomas Fruchterman
     copyright 1990 Thomas Fruchterman

     Rewritten in C, Jan 1991   Mark Allender
*/

/*
     This file contains only two functions
     why does it exist at all?
     well, I hoped that I could isolate the code for
     computing the force functions so they could be 
     hard-wired instead of interpreted.
     I experimented with this hard-wiring and I cant
     understand why it make no appreciable difference
     maybe integer ops take way longer than function
     calls

     Another reason for putting these functions in a 
     separate file is to abstract "folding"
*/
/*
 * There were major problems with the force functions.  Tom assumed
 * that the force functions would never return a force greater than
 * the distance between the two nodes.  The later was a very bad
 * assumption.  If I have an attractive force that is 3 times the
 * distance between the two nodes, then when the new position was
 * computed it would move the node 3 times the distance between
 * the nodes (thus it would move the node past the attacting node
 * 2 times the original distance).  In this way, attractive forces
 * were affectively becoming repulsive forces.  This didn't appear
 * for Tom to a great extent since he had interger graphs and thus
 * integer distances.  However, we noticed it since we changed to
 * a zero to one graph and our forces were often much greater than
 * our distances.
 *   The other problem was that the forces computed only acted
 * on one of the nodes, but in fact when there is a single edge
 * between nodes, the two nodes should be attracted mutually towards
 * one another.
 * These were fixed on 11/04/91 by Doug Bogia
 */
#include <stdio.h>
#include <sys/types.h>
#include <math.h>
#include "extern.h"
#include "graph.h"

/*
 * Ok, the way to fix the problem with the forces is to transform the
 * calculated force into a ratio that runs from zero to one.  I didn't
 * know the best way to do this, so I put in a function that will take
 * a force (which runs from 0 to infinity) and will convert it into
 * a zero to one scale.  We wanted this conversion to be fast so
 * using a tangent wasn't much of an option.  The other problem is that
 * Tom's thesis says that the more attactive the force, the more we want
 * to move the nodes together; however, in reality not knowing the upper
 * bound of the force makes this difficult.  The end solution was to
 * look at the force.  If it is greater than (or equal to 1) then
 * we want to compute the ratio as an inverse and subtrace it 
 * from one (i.e. 1 - 1/force).  If the force is < 1, then we will simply 
 * use the force; however, these two methods overlap so we want
 * to tell the point where one stops and the other starts.
 * Thus, we introduce the #define SWITCHPOINT that 
 * tells the point on the 0 to 1 segment that the functions meet.
 * So if we want them to meet at .5 we would compute 1 - 1/2*1/force
 * and 1/2*force.  BUT, we want each of these to contribute 1/2 to
 * the movement of the two nodes, so the functions we really want
 * are 1/2 - 1/2*1/2*1/force and 1/2*1/2*force.  So to move the
 * SWITCHPOINT around decide where you want the switch to happen and
 * divide that number by 2.  Thus for a switch point of .5 you have
 * SWITCHPOINT 0.25 and for a switch point of .25 you would have
 * SWITCHPOINT 0.125, etc.
 */
#define SWITCHPOINT 0.25

extern coord_t OptDist, OptDistSqr;

void
appAFunc(node, node1)
Vertex *node, *node1;
{
  coord_t x, y, xoffset, yoffset;
  coord_t Dist, DistSqr;
  double ratio;
  
  x = node1->pos[0] - node->pos[0];
  y = node1->pos[1] - node->pos[1];
  Dist = (coord_t)sqrt((double)((x * x) + (y * y)));
  if (Dist == 0.0)
  {
    return;			/* The nodes can't get any closer */
  }
  
  DistSqr = Dist * Dist;
  
  /*
   * What we want is to see if force > 1.  We can save ourselves some
   * computation if we make the comparison here and then do the
   * 1/force by computing the force inverted.  The force is
   * supposed to be Dist * Dist / OptDist according to the thesis.
   */
  if (DistSqr > OptDist)
  {
    ratio = 0.5 - SWITCHPOINT * OptDist / DistSqr;
  }
  else
  {
    ratio = SWITCHPOINT * DistSqr / OptDist;
  }

  xoffset = x * ratio;
  yoffset = y * ratio;

  node->newpos[0] += xoffset;
  node->newpos[1] += yoffset;
  node1->newpos[0] -= xoffset;
  node1->newpos[1] -= yoffset;

#ifdef PRINTFORCE
  printf ("Attractive ratio = %f from %20s to %20s\n",
	  ratio, node->values->labels[0], node1->values->labels[0]);
#endif
}

void
appRFunc(node, node1)
Vertex *node, *node1;
{
  coord_t x, y, xoffset, yoffset;
  coord_t Dist;
  double ratio;
  
  x = node1->pos[0] - node->pos[0];
  y = node1->pos[1] - node->pos[1];
  Dist = (coord_t)sqrt((double)((x * x) + (y * y)));

  if (Dist == 0.0)
  {
    Dist = 0.000001;		/* Some very small number */
  }
  
  /*
   * What we want is to see if force > 1.  We can save ourselves some
   * computation if we make the comparison here and then do the
   * 1/force by computing the force inverted.  The force is
   * supposed to be OptDist * OptDist / Dist according to the thesis.
   */
  if (OptDistSqr > Dist)
  {
    ratio = 0.5 - SWITCHPOINT * Dist / OptDistSqr;
  }
  else
  {
    ratio = SWITCHPOINT * OptDistSqr / Dist;
  }

  xoffset = x * ratio;
  yoffset = y * ratio;

  node->newpos[0] -= xoffset;
  node->newpos[1] -= yoffset;
  node1->newpos[0] += xoffset;
  node1->newpos[1] += yoffset;

#ifdef PRINTFORCE
  printf ("Repulsive ratio = %f from %20s to %20s\n",
	  ratio, node->values->labels[0], node1->values->labels[0]);
#endif
}
