/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:	 Mark Allender (allender@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
*/

/*
 *  This file contains all the support routines for the key-function
 *  association lists that are part of the callback structure.
*/

#include <stdio.h>
#include <mbus/api.h>
#include <mbus/keyword.h>
#include "callback.h"

/*
 *  The function GetAssociations takes a sexpression in message bus
 *  format, and creates a linked list of functions out of it.  This
 *  head of this list is returned.
*/
struct FunctionList *
GetAssociations(assoc_list)
     struct mb_object *assoc_list;
{
  char *function_name;
  struct FunctionList *head, *new_entry, *end;

  end = head = NULL;
  for (; MBconsp(assoc_list); assoc_list = MBcdr(assoc_list)) {
    function_name = MBCstring(MBcar(assoc_list));
    new_entry = (struct FunctionList *)malloc(sizeof(struct FunctionList));
    new_entry->function = function_name;
    new_entry->next = NULL;
    if (end)
    {
      end->next = new_entry;
      end = new_entry;
    }
    else
    {
      head = new_entry;
      end = new_entry;
    }
  }
  return head;
}

/*
 *  FreeAssociations takes a FunctionList and frees all of it`s members.
*/
void
FreeAssociations(function_list)
     struct FunctionList **function_list;
{
  struct FunctionList *func;

  while (func = *function_list) {
    *function_list = func->next;
    free(func->function);
    free(func);
  }
}

void
AddAssociations (old_list, new_list)
     struct FunctionList **old_list, *new_list;
{
  struct FunctionList *old_entry, *new_entry, *tmp_old, *prev;

/*
 *  look down the list of new functions to be added.  We don't want
 *  to add duplicates to this list (or do we?).  If we find a match, then
 *  we just free the space of the new FunctionList entry, and proceed to
 *  the next one.  Otherwise, we add the new function to the end of
 *  the old_list.  This code is very similar to the code ReassignKeys in
 *  the mbus-parse code.
*/
  if (*old_list == NULL) {
    /* If nothing on the old list, just return the new list */
    *old_list = new_list;
    return;
  }
  /* Go through all of the elements on the new list */
  for (; new_entry = new_list; new_list = new_list->next)
  {
    /* Now step down the old list and see if this new entry matches
     * one of the old entries.
     */
    for (tmp_old = *old_list, prev = NULL;
	 /* Go while we have a list and it doesn't match the new entry */
	 tmp_old && strcmp(tmp_old->function, new_entry->function);
	 prev = tmp_old, tmp_old = tmp_old->next)
	{};
    if (tmp_old == NULL) {
      /* If tmp_old is NULL then we went through the entire list without
       * a match.  In this case, we want to add this new element to the
       * end.  In the loop above we kept a prev pointer so we can do this.
       * Notice too that we don't have to check if prev == NULL since
       * we *know* that *old_list is not NULL (see the check at the beginning
       * of this routine.
       */
      prev->next = new_entry;
      new_entry->next = NULL;
    }
    else {
      /* Otherwise, we have a match.  Since they match exactly we
       * can just toss the information.
       */
      free (new_entry->function);
      free(new_entry);
    }
  }
}

/*
 *  This routine removes any functions from the given function list.
 *  We just have a simple list traversal and deletion here.
*/
void
RemoveAssociations(old_list, remove_list)
     struct FunctionList **old_list, *remove_list;
{
  struct FunctionList *prev_entry, *current_entry, *current_rm, *old_rm;

  for (; current_rm = remove_list; remove_list = remove_list->next)
  {
    for (prev_entry = NULL, current_entry = *old_list;
	 /* Go while we have an entry and it doesn't match */
	 current_entry &&
	 strcmp(current_entry->function, current_rm->function);
	 prev_entry = current_entry, current_entry = current_entry->next)
	{};

    if (current_entry) {
      /* If we found an entry that matched then we want to remove it */
      if (!prev_entry)
	  *old_list = current_entry->next;
      else
	  prev_entry->next = current_entry->next;
      free(current_entry->function);
      free(current_entry);
    }
    free(current_rm->function);
    free(current_rm);
  }
}
