/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:	 Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

#include "extern.h"
#include "graph.h"

/*
 * NodeFixed will return TRUE or FALSE depending on whether the node
 * is fixed or not.
 */
int
NodeFixed(node)
  Vertex *node;
{
  return ((node->values->status & FIXED) ? TRUE : FALSE);
}
  
/*
 * FixNode will fix the node.
 */
void
FixNode(node)
  Vertex *node;
{
  if (node)
      node->values->status |= FIXED;
}

/*
 * UnfixNode will mark a node as not fixed.
 */
void UnfixNode(node)
  Vertex *node;
{
  if (node)
      node->values->status &= ~FIXED;
}

/*
 * NodeSelected will return TRUE or FALSE depending on whether the node
 * is selected or not.
 */
int
NodeSelected(node)
  Vertex *node;
{
  return ((node->values->status & SELECTED) ? TRUE : FALSE);
}
  
/*
 * SelectNode will select the node.
 */
void
SelectNode(node)
  Vertex *node;
{
  if (node)
      node->values->status |= SELECTED;
}

/*
 * UnselectNode will mark a node as not selected.
 */
void UnselectNode(node)
  Vertex *node;
{
  if (node)
      node->values->status &= ~SELECTED;
}

/*
 * NodeMarked will return TRUE or FALSE depending on whether the node
 * is markedor not.
 */
int
NodeMarked(node, mark)
  Vertex *node;
  int mark;
{
  return ((node->values->status & mark) ? TRUE : FALSE);
}
  
/*
 * MarkNode will mark the node.
 */
void
MarkNode(node, mark)
  Vertex *node;
  int mark;
{
  if (node)
      node->values->status |= mark;
}

/*
 * UnmarkNode will mark a node as not marked.
 */
void UnmarkNode(node, mark)
  Vertex *node;
  int mark;
{
  if (node)
      node->values->status &= ~mark;
}

/*
 * NodeHidden will return TRUE or FALSE depending on whether the node
 * is hidden or not.
 */
int
NodeHidden(node)
  Vertex *node;
{
  return ((node->values->status & HIDDEN) ? TRUE : FALSE);
}
  
/*
 * HideNode will Hide the node.  We will also remove any marking status...
 */
void
HideNode(node)
  Vertex *node;
{
  if (node) {
    node->values->status |= HIDDEN;
    UnmarkNode(node, MARKED | VISITED);
  }
}

/*
 * UnhideNode will mark a node as not hidden.
 */
void UnhideNode(node)
  Vertex *node;
{
  if (node) 
      node->values->status &= ~HIDDEN;
}

/*
 * NodeSuper will return TRUE or FALSE depending on whether the node
 * is a super_node or not.
 */
int
NodeSuper(node)
  Vertex *node;
{
  return ((node->values->status & SUPER) ? TRUE : FALSE);
}
  
/*
 * SuperNode will mark the node as a supernode
 */
void
SuperNode(node)
  Vertex *node;
{
  if (node)
      node->values->status |= SUPER;
}

/*
 * UnsuperNode will mark a node as not being a supernode.
 */
void UnsuperNode(node)
  Vertex *node;
{
  if (node) 
      node->values->status &= ~SUPER;
}

/*
 * RandomizeNode gives the particular node a random placement in the
 * graph (providing it is not fixed).
 */
void
RandomizeNode(node)
  Vertex *node;
{
  if (!NodeFixed(node))
  {
    node->pos[0] = 0.25 + (drand48() / 2.0);
    node->pos[1] = 0.25 + (drand48() / 2.0);
  }
}

/*
 * RandomizeNodes gives all the nodes in the graph that aren't fixed a
 * random location.
 */
void
RandomizeNodes()
{
  Vertex *node;

  for (node = graph->v; node != NULL; node = node->next)
      RandomizeNode(node);
}

void
AssignNodeLabel(node)
  Vertex *node;
{
  XCharStruct      size;        /* Holds the size of our character struct. */
  XFontStruct      *font;               /* font for the node */
  char *           str;         /* Holds the string we are working on */
  int              ascent;      /* The ascent of the string */
  int              descent;     /* The descent of the string */
  int              tmp;         /* Just a temporary. */
  short            label_index; /* This holds the current label */
  unsigned int     color;       /* Dummy variable */

  label_index = graph->name_type;
  /* Look to see if we have a null string */
  if (*node->values->labels[label_index])
      node->label_index = label_index; /* Nope, use the label */
  else
      node->label_index = 0;	/* Yep, use the default */

  /* Get the string */
  str = node->values->labels[node->label_index];
  RetTypeInfo(node->values->type, NODEMASK, &color, &font);
  /* Find out it's sizes */
  XTextExtents(font,str,strlen(str),&tmp,&ascent,&descent,&size);
  /* We want to convert the label offsets into our internal coords.
   * To do this we need to divide by 2 since we will be starting
   * at the midpoint of the node and we need to divide by the
   * graph width or height to convert to our internal coords.
   */
  node->label_x_offset = (coord_t)(size.width + 2) /
      (coord_t)(2.0 * graph->viewing_area.width);
  node->label_y_offset = (coord_t)(ascent + descent + 2) /
      (coord_t)(2.0 * graph->viewing_area.height);
}

/*
   AssignNodeLabels was created by Doug Bogia on Aug. 26, 1991
   Description:
     This is meant to make the assignment to each node the actual lable
     that it is to plot and also it will compute the offsets of the
     label size.
   Entry:
     None.
   Exit:
     None.
   Returns:
     None.
*/
void
AssignNodeLabels ()
{
  /* Start of variables section */
  Vertex *         node;        /* This is used to spin down the nodes */
  /* End of variables section */

  for (node = graph->v; node != NULL; node = node->next)
      AssignNodeLabel(node);
} /* End of AssignNodeLabels */

/*
 * SetNodePos will take the x and y location of the node to be the correct
 * position and it will recompute the Pos[0,1] of the node.  If these
 * are such that they are not from 0 to 1, then the x/y position is reset
 * according to Pos[0,1].
 */
void
SetNodePos(node)
  Vertex *node;
{
  coord_t virtx_scale, virty_scale;
  virtx_scale = (graph->virtx_max - graph->virtx_min);
  virty_scale = (graph->virty_max - graph->virty_min);
  node->pos[0] = node->x * virtx_scale + graph->virtx_min;
  node->pos[1] = node->y * virty_scale + graph->virty_min;
  if (node->pos[0] < LEFTSIDE) node->pos[0] = LEFTSIDE;
  if (node->pos[0] > RIGHTSIDE) node->pos[0] = RIGHTSIDE;
  if (node->pos[1] < TOPSIDE) node->pos[1] = TOPSIDE;
  if (node->pos[1] > BOTTOMSIDE) node->pos[1] = BOTTOMSIDE;
  node->x = (node->pos[0] - graph->virtx_min) / virtx_scale;
  node->y = (node->pos[1] - graph->virty_min) / virty_scale;
}
