/*
The contents of this file contain text and code describing and 
implementing the 'DES' encryption algorithm. Despite the fact 
that this information is freely available overseas, it remains 
a violation of ITAR and/or EAR to export this information 
from inside the US or Canada to outside the US or Canada, or 
to pass it to a non-US or non-Canadian citizen within the US 
or Canada. The US Government evidently defines 'Export' to 
include placing this information on a non-restricted FTP server 
or Web site. Please do not do so, and be sure that any person you
pass this on to is made aware of this restriction.
									Peter Trei
									ptrei@acm.org

 * THIS SOFTWARE IS PROVIDED BY PETER TREI ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.

This software is copyright (c) 1997 Peter Trei (ptrei@acm.org), except for
those portions written by Phil Karn, which retain their
original ownership.

This software may be redistributed freely for use in the RSA DES Challenge,
but please obey the restrictions imposed by the US Government, and make
sure that anyone you pass it to is also aware of them.

This software may not be used for commercial purposes without the written
permission of Peter Trei and the other owners.

Please redistribute only as a complete, unmodified package, including 
source code, and ptrei@acm.org's PGP signature file and key.

 */


/* Portable C code to create DES key schedules from user-provided keys
 * This doesn't have to be fast unless you're cracking keys or UNIX
 * passwords
 */

#include <string.h>
#include <stdio.h>
#include "deskr.h"
#include "dessp.h"

unsigned int tmp3[256];
unsigned int tmp4[256];

/* map of bits, so we don't have to do those slow rotates */
static long bit32[] =
{ 
0x00000001, 0x00000002, 0x00000004, 0x00000008,
0x00000010, 0x00000020, 0x00000040, 0x00000080,
0x00000100, 0x00000200, 0x00000400, 0x00000800, 
0x00001000, 0x00002000, 0x00004000, 0x00008000,
0x00010000, 0x00020000, 0x00040000, 0x00080000, 
0x00100000, 0x00200000, 0x00400000, 0x00800000,
0x01000000, 0x02000000, 0x04000000, 0x08000000, 
0x10000000, 0x20000000, 0x40000000, 0x80000000
}; 

/*
This table describes how to re-arrange the key bits so that we
can 1: get rid of the parity bits, and 2: put the bits that don't
contribute to the first round subkey in the low order end of the
key schedule, thus allowing us to (usually) eliminate the first
round.

This table assumes that the bits in the initial key are numbered
0-63, and the position in the array gives the position in the 
crunched, 56 bit key. The number in the array entry specifies which
bit in the 64 bit key the bit in that position is derived from. */ 

static unsigned char dkr[] = {
	 4,  6, 10, 13, 44, 45, 49, 50,
	 1,  2,  3,  5,  7,  9, 11, 12,
	14, 15, 17, 18, 19, 20, 21, 22,
	23, 25, 26, 27, 28, 29, 30, 31,
	33, 34, 35, 36, 37, 38, 39, 41,
	42, 43, 46, 47, 51, 52, 53, 54,
	55, 57, 58, 59, 60, 61, 62, 63
};

/*
6666 5555 5555 5544 4444 4444 3333 3333 3322 2222 2222 1111 1111 1100 0000 0000
3210 9876 5432 1098 7654 3210 9876 5432 1098 7654 3210 9876 5432 1098 7654 3210
                !!    !!                                      !   !    ! !
*/

/* temp hack, without the round 0 rearrangement. */
static unsigned char tmpdkr[] = {
	 1,  2,  3,  4,  5,  6,  7,
	 9, 10, 11, 12, 13, 14, 15,
	17, 18, 19, 20, 21, 22, 23,
	25, 26, 27, 28, 29, 30, 31,
	33, 34, 35, 36, 37, 38, 39,
	41, 42, 43, 44, 45, 46, 47,
	49, 50, 51, 52, 53, 54, 55,
	57, 58, 59, 60, 61, 62, 63
};

/* Key schedule-related tables from FIPS-46 */

/* permuted choice table (key) */
static unsigned char pc1[] = {
	57, 49, 41, 33, 25, 17,  9,
	 1, 58, 50, 42, 34, 26, 18,
	10,  2, 59, 51, 43, 35, 27,
	19, 11,  3, 60, 52, 44, 36,

	63, 55, 47, 39, 31, 23, 15,
	 7, 62, 54, 46, 38, 30, 22,
	14,  6, 61, 53, 45, 37, 29,
	21, 13,  5, 28, 20, 12,  4
};

/* number left rotations of pc1 */
static unsigned char totrot[] = {
	1,2,4,6,8,10,12,14,15,17,19,21,23,25,27,28
};

/* permuted choice key (table) */
static unsigned char pc2[] = {
	14, 17, 11, 24,  1,  5,
	 3, 28, 15,  6, 21, 10,
	23, 19, 12,  4, 26,  8,
	16,  7, 27, 20, 13,  2,
	41, 52, 31, 37, 47, 55,
	30, 40, 51, 45, 33, 48,
	44, 49, 39, 56, 34, 53,
	46, 42, 50, 36, 29, 32
};



/* End of DES-defined tables */


/* bit 0 is left-most in byte */
static int bytebit[] = {
	0200,0100,040,020,010,04,02,01
};

unsigned int bit8[]={0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80};

/************************/
/*
 * This sets up a 256 entry table called gray8. Each entry describes
 * which bit will change as we step through 256 values in Gray code
 * order. It is used in incrementing the key schedule, to decide 
 * which set of masks to xor into the current key schedule.
 */
void set_gray8()
{
unsigned int x,y,g,gd,go;
#if 0
int tmp1[256]; int tmp,tmp2;
#endif
go = 0;
for (x =0; x<256; x++)
	{
	g = x ^ (x >>1);
	gd = g ^ go;
	go = g;
	for (y = 0; y<8; y++) 
		if (bit8[y] == gd) gray8[x] = y;
	}
gray8[0]=7;
}

/******************/
/* Generate key schedule for encryption or decryption
 * depending on the value of "decrypt", using Phil Karn's
 * original code.
 *
 * This is the original very slow routine for generating a key
 * key schedule. It is used to generate the mask arrays 
 * needed for faster mechanisms in the other deskey routines.
 */
void
deskey(k,key,decrypt)
unsigned long k[16][2];		/* Key schedule array */
unsigned char *key;		/* 64 bits (will use only 56) */
int decrypt;			/* 0 = encrypt, 1 = decrypt */
{
	unsigned char pc1m[56];		/* place to modify pc1 into */
	unsigned char pcr[56];		/* place to rotate pc1 into */
	register int i,j,l;
	int m;
	unsigned char ks[8];

	for (j=0; j<56; j++) {		/* convert pc1 to bits of key */
		l=pc1[j]-1;		/* integer bit location	 */
		m = l & 07;		/* find bit		 */
		pc1m[j]=(key[l>>3] &	/* find which key byte l is in */
			bytebit[m])	/* and which bit of that byte */
			? 1 : 0;	/* and store 1-bit result */
	}
	for (i=0; i<16; i++) {		/* key chunk for each iteration */
		memset(ks,0,sizeof(ks));	/* Clear key schedule */
		for (j=0; j<56; j++)	/* rotate pc1 the right amount */
			pcr[j] = pc1m[(l=j+totrot[decrypt? 15-i : i])<(j<28? 28 : 56) ? l: l-28];
			/* rotate left and right halves independently */
		for (j=0; j<48; j++){	/* select bits individually */
			/* check bit that goes to ks[j] */
			if (pcr[pc2[j]-1]){
				/* mask it in if it's there */
				l= j % 6;
				ks[j/6] |= bytebit[l] >> 2;
			}
		}
		/* Now convert to packed odd/even interleaved form */
		
		k[i][0] = ((long)ks[0] << 24)
		 | ((long)ks[2] << 16)
		 | ((long)ks[4] << 8)
		 | ((long)ks[6]);
		k[i][1] = ((long)ks[1] << 24)
		 | ((long)ks[3] << 16)
		 | ((long)ks[5] << 8)
		 | ((long)ks[7]);
		 
		if(Asmversion){
			/* The assembler versions pre-shift each subkey 2 bits
			 * so the Spbox indexes are already computed
			 */
			k[i][0] <<= 2;
			k[i][1] <<= 2;
		}
	}
}

/************************/
/* print out a key schedule */
void print_ks(unsigned long ks[16][2])
{
	int y;
	for (y= 0; y < 8; y++) printf("%08lx ",ks[y][0]);printf("\n");
	for (y= 0; y < 8; y++) printf("%08lx ",ks[y][1]);printf("\n");
	for (y= 8; y < 16;y++) printf("%08lx ",ks[y][0]);printf("\n");
	for (y= 8; y < 16;y++) printf("%08lx ",ks[y][1]);printf("\n\n");
}
/************************/
/* print out the decryption key difference table. */
void print_dkeydiffs()
{
	int x;

for (x = 0; x < 64; x++)
	{
	printf("%2d:\n",x);
	print_ks((unsigned long(*)[2])dkeydiffs[x]);
	}
}
/************************/
void ks_to_key(unsigned long ks[16][2],
			   unsigned int *kh,
			   unsigned int *kl)
/* turn a key schedule into a 'crunched' key. This is used to
	derive a key when a match is found, or the program must
	checkpoint it's status to a file */
{
int x,y,z;

*kh = 0; *kl = 0;
for (x = 0; x < 56; x++) /* increment bit we're trying to find */
	{
	for (y = 0; y < 15; y++) /* increment round we're looking in */
							 /* we don't use 0 or 15 since they */
							 /* may not get incremented as often */
		{
		z = 0;
		if (drkeydiffs[x][y][z] > 0) break;
		z = 1;
		if (drkeydiffs[x][y][z] > 0) break;
		}
	if (drkeydiffs[x][y][z] & ks[y][z])
		if (x < 32)
			*kl |= bit32[x];
		else
			*kh |= bit32[x-32];
	}
}
/*********************/
void derive_b_values(unsigned long kl)
{
/* figure out the b? values which corrospond to a given kl
	value 
first guess - the b? value is the nth gray code value reached,
the gray code value which is equal to that byte in kl.

This is untested and undebugged. I don't use it at the moment.
*/
int x;
unsigned long c3,c2,c1,c0,bi;

bi = 0;
c3 = kl & 0xff;
c2 = (kl >>  8) & 0xff;
c1 = (kl >> 16) & 0xff;
c0 = (kl >> 24) & 0xff;
bi = 0;
for (x = 0; x< 256; x++)
	{
	if (c0 == bi) b0 = x;
	if (c1 == bi) b1 = x;
	if (c2 == bi) b2 = x;
	if (c3 == bi) b3 = x;
    bi ^= gray8[x];
	}
}
/*********************/
void restore_key_schedule(
			   unsigned long kh,
			   unsigned int b3,
			   unsigned int b2,
			   unsigned int b1,
			   unsigned int b0)
/* use kh (the chunk number), and b? to restore the
   global key schedule in knd */
/* I'm sure a macro could do this more compactly, but I tend to
eschew anything which makes the code less clear. */
{
unsigned int x,y,bi;

for (x = 0; x < 16; x++) /* zero out the key schedule */
	{
		knd[x][0] = 0;
		knd[x][1] = 0;
	}
deskey3(knd,kh,0,1); /* restore the bits defined by kh */
/* restore the b0 bits */
if (b0 > 0)
	{
	for (x=0; x< b0; x++)
		{
		bi = gray8[x];
		for (y = 0; y < 16; y++)
			{
			knd[y][0] ^= drkeydiffs[bi][y][0];
			knd[y][1] ^= drkeydiffs[bi][y][1];
			}
		}
	}
/* restore the b1 bits */
if (b1 > 0)
	{
	for (x=0; x< b1; x++)
		{
		bi = gray8[x]+8;
		for (y = 0; y < 16; y++)
			{
			knd[y][0] ^= drkeydiffs[bi][y][0];
			knd[y][1] ^= drkeydiffs[bi][y][1];
			}
		}
	}
/* restore the b2 bits */
if (b2 > 0)
	{
	for (x=0; x< b2; x++)
		{
		bi = gray8[x]+16;
		for (y = 0; y < 16; y++)
			{
			knd[y][0] ^= drkeydiffs[bi][y][0];
			knd[y][1] ^= drkeydiffs[bi][y][1];
			}
		}
	}
/* restore the b3 bits */
if (b3 > 0)
	{
	for (x=0; x< b3; x++)
		{
		bi = gray8[x]+24;
		for (y = 0; y < 16; y++)
			{
			knd[y][0] ^= drkeydiffs[bi][y][0];
			knd[y][1] ^= drkeydiffs[bi][y][1];
			}
		}
	}
}
/************************/
/* set up keydiffs - this specifies which bits change in
   the key schedule for each bit in the key */
/*
 * There are actually four tables generated: 
 * [e|d]{r}keydiffs: e|d specifies whether this is an Encrypt or
 * Decrypt table, and the presence of r indicates that this table
 * is rearranged to allow the elimination of the first round.
 */

/* It operates by creating a key schedule for 56 different keys, 
 * each of which has only one bit set 
 */

void set_keydiffs()
{
	int x,y,z;
	unsigned char key[8];

	z = 0;
	for (y=0;y<8;y++) key[y] = 0;
	for (x=7;x>=0;x--)
		{
		for (y=0;y<8;y++)
			{
			key[x]= (1<<y);
			deskey((unsigned long(*)[2])ekeydiffs[z],key,0);
			deskey((unsigned long(*)[2])dkeydiffs[z],key,1);
			z++;
			}
		key[x] = 0;
		}
/* now, rearrange for use with crunched keys, putting the 
	rearranged table into er- and dr-keydiffs */
for (x = 0; x<56; x++)
	{
	z = dkr[x];
	for (y = 0; y < 16; y++)
		{
		erkeydiffs[x][y][0] = ekeydiffs[z][y][0];
		erkeydiffs[x][y][1] = ekeydiffs[z][y][1];
		drkeydiffs[x][y][0] = dkeydiffs[z][y][0];
		drkeydiffs[x][y][1] = dkeydiffs[z][y][1];
		}
	}
}
/************************/
/*
 Generate a key schedule de-novo, using the keydiff tables, from the
 8 byte key array. This is much faster than the original in deskey(),
 but deskey was needed to generate the keydiff tables in the first 
 place.
 */
void deskey2(k,key,decrypt)
unsigned long k[16][2];		/* Key schedule array */
unsigned char *key;		/* 64 bits (will use only 56) */
unsigned int decrypt;

{
int w,x,y,z;
for (y = 0; y < 16; y++)
	{
	k[y][0] = 0;
	k[y][1] = 0;
	}
z = 0;
for (x=7;x>=0;x--)
	{
	for (y=0;y<8;y++)
		{
		if (key[x] & (1<<y))
			{
			for (w = 0; w<16; w++)
				{
				if (decrypt == 0)
					{
					k[w][0] |= ekeydiffs[z][w][0];
					k[w][1] |= ekeydiffs[z][w][1];
					}
				else
					{
					k[w][0] |= dkeydiffs[z][w][0];
					k[w][1] |= dkeydiffs[z][w][1];
					}
				}
			}
		z++;
		}
	}
}
/************************/
/*
 Generate a key schedule de-novo, using the keydiff tables, from a
 a crunched key in keyh and keyl. This is much faster than the 
 original in deskey(), but deskey was needed to generate the keydiff 
 tables in the first place.
 */

void deskey3(k,keyh,keyl,decrypt)
unsigned long k[16][2];		/* Key schedule array */
unsigned long keyh,keyl;
unsigned int decrypt;
{
int w,x;
for (x = 0; x < 16; x++)
	{
	k[x][0] = 0;
	k[x][1] = 0;
	}
for (x = 0; x < 32; x++)
	{
	if (keyl & bit32[x])
		{
		for (w = 0; w<16; w++)
			{
			if (decrypt == 0)
				{
				k[w][0] |= erkeydiffs[x][w][0];
				k[w][1] |= erkeydiffs[x][w][1];
				}
			else
				{
				k[w][0] |= drkeydiffs[x][w][0];
				k[w][1] |= drkeydiffs[x][w][1];
				}
			}
		}
	}
for (x = 0; x < 24; x++)
	{
	if (keyh & bit32[x])
		{
		for (w = 0; w<16; w++)
			{
			if (decrypt == 0)
				{
				k[w][0] |= erkeydiffs[x+32][w][0];
				k[w][1] |= erkeydiffs[x+32][w][1];
				}
			else
				{
				k[w][0] |= drkeydiffs[x+32][w][0];
				k[w][1] |= drkeydiffs[x+32][w][1];
				}
			}	
		}
	}
}

/************************/

void deskey4(k,nkeyh,nkeyl,okeyh,okeyl)
unsigned long k[16][2];		/* Key schedule array */
unsigned long nkeyh,nkeyl,okeyl,okeyh;
{
	/* this version assumes that the key schedule in k is generated
	from okeyh and okeyl. We need to update it to the key in nkeyl,
	nkeyh. If the old and new keys are close in value, this is a lot 
	faster than deskey().  
	*/
unsigned long dkeyh,dkeyl;
int x,y,z;
/* lower 32 bits */
x = 0;
dkeyl = okeyl ^ nkeyl;
while (dkeyl)
	{
	if (dkeyl & bit32[x])
		{
		for (y = 0; y <16; y++)
			{
			k[y][0] ^= drkeydiffs[x][y][0];
			k[y][1] ^= drkeydiffs[x][y][1];
			}
		dkeyl ^= bit32[x];
		}
	x++;
	}
/* upper 24 bits */
x = 0;
dkeyh = okeyh ^ nkeyh;
while (dkeyh)
	{
	if (dkeyh & bit32[x])
		{
		z = x+24;
		for (y = 0; y <16; y++)
			{
			k[y][0] ^= drkeydiffs[z][y][0];
			k[y][1] ^= drkeydiffs[z][y][1];
			}
		dkeyh ^= bit32[x];
		}
	x++;
	}
}
/************************/

void crunchkey(unsigned char *key, long *pkhi, long *pklo)
{
/* key is an 8 byte array, containing a key. We need to
	turn it into 2 longs, one containing the low order
	32 bits, and the other the high order 24 bits.
	the key goes from 64 to 56 bits because we strip
	out the parity bits. We also permute it according to
	dkr as we do so. Keys in this format (2 longs) are 
	easier to deal with than the easy-to-print array form. */
long hi, lo;
int x,z;
unsigned long klleft, klright;

hi = 0; lo = 0;

/* first, rearrange the bits into the correct endian state. */
klleft = ((unsigned long)key[0] << 24)
 | ((unsigned long)key[1] << 16)
 | ((unsigned long)key[2] << 8)
 | (unsigned long)key[3];
klright = ((unsigned long)key[4] << 24)
 | ((unsigned long)key[5] << 16)
 | ((unsigned long)key[6] << 8)
 | (unsigned long)key[7];

/* now permute klleft and klright into hi and lo, using dkr */

for (x =0; x<56; x++)
	{
	z = /*tmp*/dkr[x];
	/* z points to a bit, in range 0-63, 
	   which will go into lo or hi*/
	if (z < 32)
		{
		/* the bit we're looking for is low order, and will
			be found in klright*/
		if (klright & bit32[z])
			{
			/* the bit is set, so we have to set the destination*/
			if (x < 32)
				lo |= bit32[x];
			else
				hi |= bit32[x-32];
			}
		}
	else
		{
		/* the bit we're looking for is hi order, and will
			be found in klleft*/
		if (klleft & bit32[z-32])
			{
			/* the bit is set, so we have to set the destination*/
			if (x < 32)
				lo |= bit32[x];
			else
				hi |= bit32[x-32];
			}
		}
	}
*pkhi = hi;
*pklo = lo;
}
/************************/

void uncrunchkey(unsigned char *key, long hi, long lo)
/*
Turn a 56 bit key stored in 2 longs (low order 32 bits in lo,
and high order 24 bits in hi) into a 64 bit array stored in a 
string. Undo the permutation caused by dkr as we do so. 
*/
{
unsigned long klleft, klright,a;
int x,z;

klleft = 0; klright = 0;

for (x = 0; x < 24; x++)
	{
	/* hi */
	if (hi & bit32[x])
		{
		z = /*tmp*/dkr[x+32];
		if (z < 32)
			klleft |= bit32[z];
		else
			klright |= bit32[z-32];
		}
	}
for (x=0; x < 32; x++)
	{
	/* lo */
	if (lo & bit32[x])
		{
		z = /*tmp*/dkr[x];
		if (z < 32)
			klleft |= bit32[z];
		else
			klright |= bit32[z-32];
		}
	}

/* rearrange for endian issues */
key[0] = (unsigned char) (klright >> 24);
key[1] = (unsigned char) (klright >> 16);
key[2] = (unsigned char) (klright >> 8);
key[3] = (unsigned char) (klright);
key[4] = (unsigned char) (klleft >> 24);
key[5] = (unsigned char) (klleft >> 16);
key[6] = (unsigned char) (klleft >> 8);
key[7] = (unsigned char) (klleft);
/* add back the parity bits */

for (x=0; x<8; x++)
	{
	a = 0;
	for (z=0; z<8; z++)
		{
		if (key[x] & bit32[z]) a++;
		}
	if (!(a & 1))
		{
		key[x] ^= 1;
		}
	}

}

