/*
 *  $Id: print.c,v 1.3 1999/01/18 22:14:07 ezk Exp $
 */

/* Print debugging functions */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <fist.h>
// #include <stdarg.h>
#include FIST_HEADER


#ifdef FIST_DEBUG

static int fist_debug_var = 18;

/* get value of debugging variable */
int
fist_get_debug_value(void)
{
  return fist_debug_var;
}

/* set debug level variable and return the previous value */
int
fist_set_debug_value(int val)
{
  int prev = fist_debug_var;

  fist_debug_var = val;
  return prev;
}

/*
 * Utilities used by both client and server
 * Standard levels:
 * 0) no debugging
 * 1) hard failures
 * 2) soft failures
 * 3) current test software
 * 4) main procedure entry points
 * 5) main procedure exit points
 * 6) utility procedure entry points
 * 7) utility procedure exit points
 * 8) obscure procedure entry points
 * 9) obscure procedure exit points
 * 10) random stuff
 * 11) all <= 1
 * 12) all <= 2
 * 13) all <= 3
 * ...
 */

static char buf[256];
void
fist_dprint_internal(int level, char *str,...)
{
  va_list ap;
  int var = fist_get_debug_value();

  if (var == level || (var > 10 && (var - 10) >= level)) {
    va_start(ap, str);
    vsprintf(buf, str, ap);
    printk("%s", buf);
    va_end(ap);
  }
  return;
}

#if 0
void
fist_print_page(char *str, page_t *page)
{
    fist_dprint(8, "        %s: %s=%x\n", str, "next", (int) page->next);
    fist_dprint(8, "        %s: %s=%x\n", str, "prev", (int) page->prev);
    fist_dprint(8, "        %s: %s=%x\n", str, "inode", (int) page->inode);
    fist_dprint(8, "        %s: %s=%x\n", str, "offset", (int) page->offset);
    fist_dprint(8, "        %s: %s=%x\n", str, "next_hash", (int) page->next_hash);
    fist_dprint(8, "        %s: %s=%x\n", str, "count", (int) page->count);
    fist_dprint(8, "        %s: %s=%x\n", str, "flags", (int) page->flags);
    fist_dprint(8, "        %s: %s=%x\n", str, "dirty", (int) page->dirty);
    fist_dprint(8, "        %s: %s=%x\n", str, "age", (int) page->age);
    fist_dprint(8, "        %s: %s=%x\n", str, "wait", (int) page->wait);
    fist_dprint(8, "        %s: %s=%x\n", str, "prev_hash", (int) page->prev_hash);
    fist_dprint(8, "        %s: %s=%x\n", str, "buffers", (int) page->buffers);
    fist_dprint(8, "        %s: %s=%x\n", str, "swap_unlock_entry", (int) page->swap_unlock_entry);
    fist_dprint(8, "        %s: %s=%x\n", str, "map_nr", (int) page->map_nr);
}
#endif


void
fist_print_page_bytes(char *str, page_t *page)
{
    int i;
    unsigned long page_data = page_address(page);
    char *cp = (char *) page_data;

    fist_dprint(8, "PPB %s (0x%x):", str, (int) page);

    for (i=0;i<10;i++)
	fist_dprint(8, "%c,",cp[i]);
    fist_dprint(8, "\n");
}


void
fist_print_buffer_flags(char *str, struct buffer_head *buffer)
{
    if (!buffer) {
	printk("PBF %s 0x%x\n", str, (int) buffer);
	return;
    }

    fist_dprint(8, "PBF %s 0x%x: Uptodate:%d Dirty:%d Locked:%d Req:%d Protected:%d\n",
		str,
		(int) buffer,
		buffer_uptodate(buffer),
		buffer_dirty(buffer),
		buffer_locked(buffer),
		buffer_req(buffer),
		buffer_protected(buffer)
		);
}


void
fist_print_page_flags(char *str, page_t *page)
{
    fist_dprint(8, "PPF %s 0x%x: Locked:%d Error:%d Referenced:%d Uptodate:%d FreeAfter:%d DecrAfter:%d SwapUnlockAfter:%d DMA:%d Slab:%d SwapCache:%d Reserved:%d\n",
		str,
		(int) page,
		(PageLocked(page) ? 1 : 0),
		(PageError(page) ? 1 : 0),
		(PageReferenced(page) ? 1 : 0),
		(PageUptodate(page) ? 1 : 0),
		(PageFreeAfter(page) ? 1 : 0),
		(PageDecrAfter(page) ? 1 : 0),
		(PageSwapUnlockAfter(page) ? 1 : 0),
		(PageDMA(page) ? 1 : 0),
		(PageSlab(page) ? 1 : 0),
		(PageSwapCache(page) ? 1 : 0),
		(PageReserved(page) ? 1 : 0)
		);
}


void
fist_print_inode(char *str, const inode_t *inode)
{
    if (!inode) {
	printk("PI:%s: NULL INODE PASSED!\n", str);
	return;
    }
    fist_dprint(8, "PI:%s: %s=%lu\n", str, "i_ino", inode->i_ino);
    fist_dprint(8, "PI:%s: %s=%u\n", str, "i_count", inode->i_count);
    fist_dprint(8, "PI:%s: %s=%x\n", str, "i_op", (int) inode->i_op);
    fist_dprint(8, "PI:%s: %s=%x (%s)\n", str, "i_sb",
		(int) inode->i_sb, sbt(inode->i_sb));
    fist_dprint(8, "\n");
    //    fist_dprint(8, "PI:%s: %s=%x\n", str, "itopd(inode)", (int) itopd(inode));
}


void
fist_print_pte_flags(char *str, const page_t *page)
{
    unsigned long address;
    pmd_t *pmd;
    pgd_t *pgd;
    pte_t *ppte;

    ASSERT(page != NULL);
    ASSERT(page->inode != NULL);
    ASSERT(page->inode->i_mmap != NULL);
    ASSERT(page->inode->i_mmap->vm_mm != NULL);
    ASSERT(page->inode->i_mmap->vm_mm->pgd != NULL);
    address = page->offset;
    pgd = page->inode->i_mmap->vm_mm->pgd;
    pmd = pmd_offset(pgd, address);
    ppte = pte_offset(pmd, address);

    fist_dprint(8, "PTE-FL:%s address=0x%x dirty=%d read=%d write=%d young=%d\n",
		str,
		(int) address,
		pte_dirty(*ppte), pte_read(*ppte),
		pte_write(*ppte), pte_young(*ppte));
}


void
fist_print_vma(char *str, const vm_area_t *vma)
{
    return;
}


void
fist_print_file(char *str, const file_t *file)
{
    if (!file) {
	fist_dprint(8, "PF:%s: NULL FILE PASSED!\n", str);
	return;
    }
    fist_dprint(8, "PF:%s: %s=0x%x\n", str, "f_dentry", (int) file->f_dentry);
    fist_dprint(8, "PF:%s: %s=0x%x\n", str, "f_op", (int) file->f_op);
    fist_dprint(8, "PF:%s: %s=0x%x\n", str, "f_mode", file->f_mode);
    fist_dprint(8, "PF:%s: %s=0x%x\n", str, "f_pos", file->f_pos);
    fist_dprint(8, "PF:%s: %s=%u\n", str, "f_count", file->f_count);
    fist_dprint(8, "PF:%s: %s=0x%x\n", str, "f_flags", file->f_flags);
    fist_dprint(8, "PF:%s: %s=0x%x\n", str, "f_reada", file->f_reada);
    fist_dprint(8, "PF:%s: %s=%d\n", str, "f_ramax", file->f_ramax);
    fist_dprint(8, "PF:%s: %s=%d\n", str, "f_raend", file->f_raend);
    fist_dprint(8, "PF:%s: %s=%d\n", str, "f_ralen", file->f_ralen);
    fist_dprint(8, "PF:%s: %s=%d\n", str, "f_rawin", file->f_rawin);
    fist_dprint(8, "PF:%s: %s=%lu\n", str, "f_version", file->f_version);
    fist_dprint(8, "\n");
    //    fist_dprint(8, "PF:%s: %s=%x\n", str, "itopd(file)", (int) itopd(file));
}


void
fist_print_dentry(char *str, const dentry_t *dentry)
{
    if (!dentry) {
	fist_dprint(8, "PD:%s: NULL DENTRY PASSED!\n", str);
	return;
    }
    if (IS_ERR(dentry)) {
	fist_dprint(8, "PD:%s: ERROR DENTRY (%ld)!\n", str, PTR_ERR(dentry));
	return;
    }
    fist_dprint(8, "PD:%s: %s=%d\n", str, "d_count", dentry->d_count);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_flags", (int) dentry->d_flags);
    fist_dprint(8, "PD:%s: %s=%x (%s)\n", str, "d_inode",
		(int) dentry->d_inode,
		(dentry->d_inode ? sbt(dentry->d_inode->i_sb) : "nil") );
    fist_dprint(8, "PD:%s: %s=%x (%s)\n", str, "d_parent",
		(int) dentry->d_parent,
		(dentry->d_parent ? sbt(dentry->d_parent->d_sb) : "nil") );
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_mounts", (int) dentry->d_mounts);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_covers", (int) dentry->d_covers);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_hash", (int) & dentry->d_hash);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_lru", (int) & dentry->d_lru);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_child", (int) & dentry->d_child);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_subdirs", (int) & dentry->d_subdirs);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_alias", (int) & dentry->d_alias);

    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_name", (int) & dentry->d_name);
    fist_dprint(8, "PD:  %s: %s=\"%s\"\n", str, "d_name.name", dentry->d_name.name);
    fist_dprint(8, "PD:  %s: %s=%d\n", str, "d_name.len", (int) dentry->d_name.len);
    //    fist_dprint(8, "PD:  %s: %s=%x\n", str, "d_name.hash", (int) dentry->d_name.hash);

    //    fist_dprint(8, "PD:%s: %s=%lu\n", str, "d_time", dentry->d_time);
    //    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_op", (int) dentry->d_op);
    fist_dprint(8, "PD:%s: %s=%x (%s)\n", str, "d_sb",
		(int) dentry->d_sb, sbt(dentry->d_sb));
    //    fist_dprint(8, "PD:%s: %s=%lu\n", str, "d_reftime", dentry->d_reftime);
    fist_dprint(8, "PD:%s: %s=%x\n", str, "d_fsdata", (int) dentry->d_fsdata);
    fist_dprint(8, "PD:%s: %s=\"%s\"\n", str, "d_iname", dentry->d_iname);
    //    fist_dprint(8, "\n");
}


void
fist_checkinode(inode_t *inode, char *msg)
{
    if (!inode) {
	printk(KERN_WARNING "fist_checkinode - inode is NULL! (%s)\n", msg);
	return;
    }
    if (!itopd(inode)) {
	fist_dprint(8, KERN_WARNING "fist_checkinode(%ld) - no private data (%s)\n", inode->i_ino, msg);
	return;
    }
    if (!itohi(inode)) {
	fist_dprint(8, KERN_WARNING "fist_checkinode(%ld) - underlying is NULL! (%s)\n", inode->i_ino, msg);
	return;
    }
    if (!inode->i_sb) {
	fist_dprint(8, KERN_WARNING "fist_checkinode(%ld) - inode->i_sb is NULL! (%s)\n", inode->i_ino, msg);
	return;
    }
    if (!inode->i_sb->s_type) {
	fist_dprint(8, KERN_WARNING "fist_checkinode(%ld) - inode->i_sb->s_type is NULL! (%s)\n", inode->i_ino, msg);
	return;
    }
#if 0
    fist_dprint(6, "%s: inode->i_count = %d, hidden_inode->i_count = %d, inode = %d, sb = %s, hidden_sb = %s\n",
		msg,
		inode->i_count, itohi(inode)->i_count, inode->i_ino,
		inode->i_sb->s_type->name, itohi(inode)->i_sb->s_type->name);
#endif
}

#endif /* FIST_DEBUG */

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
