/*
 *
			   IPSEC for Linux
		         Preliminary Release
 
	 Copyright (C) 1996, 1997, John Ioannidis <ji@hol.gr>
 
		 LIMITED PRELIMINARY RELEASE LICENCE
 	
  Permission to copy, use, and distribute unmodified copies of this
  software without fee is hereby granted, provided that this entire
  notice is included in all copies.

  No modified copies may be distributed.

  [[ This restriction will, of course, change when the code becomes
  more stable. While you may of course still distribute context-diffs
  (or anything equivalent), I strongly urge you to send any changes
  you have directly to me. This will help the community by providing a
  reference base for the code. Thanks, /ji ]]
 
  THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR
  IMPLIED WARRANTY. IN PARTICULAR, NEITHER THE AUTHOR NOR ANYONE
  DISTRIBUTING THIS SOFTWARE MAKE ANY REPRESENTATION OR WARRANTY OF
  ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR ITS
  FITNESS FOR ANY PARTICULAR PURPOSE.
 
 *
 */

/*
 * $Id: ipsec_ahhmacsha1.c,v 0.4 1997/01/15 01:28:15 ji Rel $
 *
 * $Log: ipsec_ahhmacsha1.c,v $
 * Revision 0.4  1997/01/15 01:28:15  ji
 * New transform.
 *
 *
 */

#include <linux/config.h>
#include <asm/segment.h>
#include <asm/system.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/config.h>

#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/icmp.h>
#include <linux/udp.h>
#include <net/ip.h>
#include <net/protocol.h>
#include <net/route.h>
#include <net/tcp.h>
#include <net/udp.h>
#include <net/sock.h>
#include <net/icmp.h>

#include <net/checksum.h>

#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/miscdevice.h>

#include <linux/skbuff.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>

#include <net/netlink.h>
#include <unistd.h>
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_radij.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"
#ifdef CONFIG_IPSEC_AH
#include "ipsec_ah.h"
#endif

static u_long sha1zeroes[64];

#ifdef NOLONGER_DEBUG_IPSEC_AH
static void
print_ip(struct iphdr *ip)
{
	unsigned char *ipaddr;

	printk("IP packet:\n");
	printk("--- header len = %d\n", ip->ihl*4);
	printk("--- ip version: %d\n", ip->version);
	printk("--- ip protocol: %d\n", ip->protocol);
	ipaddr=(unsigned char *)&ip->saddr;
	printk("--- source address: %u.%u.%u.%u\n", 
			*ipaddr, *(ipaddr+1), *(ipaddr+2), *(ipaddr+3));
	ipaddr=(unsigned char *)&ip->daddr;
	printk("--- destination address: %u.%u.%u.%u\n", 
			*ipaddr, *(ipaddr+1), *(ipaddr+2), *(ipaddr+3));
	printk("--- total packet len: %d\n", ntohs(ip->tot_len));
}
#endif

/*
 * Check-replay-window routine, adapted from the original 
 * by J. Hughes, from draft-ietf-ipsec-esp-des-md5-03.txt
 *
 *  This is a routine that implements a 64 packet window. This is intend-
 *  ed on being an implementation sample.
 */

static int
ah_checkreplaywindow(struct ahhmacsha1_xdata *xd, unsigned long long seq)
{
	__u64 diff;
	
	if (seq == 0) 
		return 0;		/* first == 0 or wrapped */

	if (seq > xd->amx_lastseq)	/* new larger sequence number */
	{
		diff = seq - xd->amx_lastseq;
		if (diff < xd->amx_ooowin) /* In win, set bit for this pkt */
			xd->amx_bitmap = (xd->amx_bitmap << diff) | 1;
		else
			xd->amx_bitmap = 1; /* This packet has way larger */

		xd->amx_lastseq = seq;
		return 1;		/* larger is good */
	}
	diff = xd->amx_lastseq - seq;
	if (diff >= xd->amx_ooowin)	/* too old or wrapped */
		return 0;
	
	if (xd->amx_bitmap & (1l << diff)) /* this packet already seen */
		return 0;
	xd->amx_bitmap |= (1l << diff);	/* mark as seen */
	return 1;			/* out of order but good */
}

#ifdef DEBUG_IPSEC_AH
static void dmp(char *s, caddr_t bb, int len)
{
	int i;
	unsigned char *b = bb;
  
	if (debug_ah & DB_AH_DMP)
	{
		printk("at %s, len=%d:", s, len);
		for (i=0; i < len; i++)
			printk(" %02x", *b++);
		printk("\n");
	}
}
#else
#define dmp(_x, _y, _z)
#endif

struct sk_buff *
ahhmacsha1_input(struct sk_buff *skb, struct tdb *tdbp)
{
	struct iphdr *ipp, ipo;
	struct ahhmacsha1_xdata *xd;
	struct ah aho;
	struct ah *ahp;
	int iphlen, adoff, authlen, proto, len;
	
#ifdef DEBUG_IPSEC_AH
	int i;
#endif
	unsigned char *dat;
	SHA1_CTX ictx, octx; 
	
	len = skb->len; dat = skb->data;

	ipp = (struct iphdr *)dat;
	iphlen = ipp->ihl << 2;
	ahp = (struct ah *)(dat + iphlen);
	
	authlen = (ahp->ah_hl << 2) + AH_FLENGTH;
	proto = ahp->ah_nh;
	
	xd = (struct ahhmacsha1_xdata *)tdbp->tdb_xdata;
	adoff = xd->amx_replayp ? 8 : 0; /* offset of authenticator */
	

	if ((authlen - AH_FLENGTH - adoff) != xd->amx_alen)
	{
#ifdef DEBUG_IPSEC_AH
		if (debug_ah & DB_AH_INAU)
			printk("ahhmacsha1_input: bad authenticator length %d, expected %d\n", authlen, xd->amx_alen);
#endif
	}

	ictx = xd->amx_ictx;
	octx = xd->amx_octx;;

	ipo = *ipp;
	ipo.tos = 0;
	ipo.frag_off = htons(ntohs(ipo.frag_off) & IP_DF);	/* XXX -- and the C bit? */
	ipo.ttl = 0;
	ipo.check = 0;

	dmp("ictx", (caddr_t)&ictx, 24);
	dmp("octx", (caddr_t)&octx, 24);
	
	dmp("pseudoheader", (caddr_t)&ipo, sizeof (struct iphdr));
	SHA1Update(&ictx, (unsigned char *)&ipo, sizeof (struct iphdr));
	dmp("authenticator", (caddr_t)ahp, AH_FLENGTH + adoff);
	SHA1Update(&ictx, (unsigned char *)ahp, AH_FLENGTH + adoff);
	dmp("zeroes", (caddr_t)sha1zeroes, xd->amx_alen);
	SHA1Update(&ictx, (unsigned char *)sha1zeroes, xd->amx_alen);
	dmp("packet", (caddr_t)(dat + iphlen + authlen), len - iphlen - authlen);
	SHA1Update(&ictx,  dat + iphlen + authlen, len - iphlen - authlen);
	SHA1Final((unsigned char *)(&(aho.ah_data[0])), &ictx);
	dmp("outer", (caddr_t)(&(aho.ah_data[0])), 20);
	SHA1Update(&octx, (unsigned char *)(&(aho.ah_data[0])), 20);
	SHA1Final((unsigned char *)(&(aho.ah_data[0])), &octx);

	if (memcmp(aho.ah_data, &(ahp->ah_data[adoff]), xd->amx_alen))
	{
#ifdef DEBUG_IPSEC_AH
		if (debug_ah & DB_AH_INAU)
		{
			printk("ahhmacsha1_input: bad auth\n");
			dmp("got", (caddr_t)&(aho.ah_data), xd->amx_alen);
			dmp("shg", (caddr_t)&(ahp->ah_data[adoff]), xd->amx_alen);
		}
#endif
		return NULL;
	}

	if (xd->amx_replayp)
	{
		unsigned long long seq;

		seq = ntohq(*((unsigned long long *)(ahp->ah_data)));
		if (!ah_checkreplaywindow(xd, seq))
		{
#ifdef DEBUG_IPSEC_AH
			if (debug_ah & DB_AH_REPLAY)
				printk("ahhmacsha1_input: duplicate frame\n");
#endif
			return NULL;
		}
	}

	/*
	 *	Discard the original IP header
	 */
	 
	ipp->tot_len = htons(ntohs(ipp->tot_len) - authlen);
	ipp->protocol = proto;

	memmove((void *)(skb->data + authlen), (void *)skb->data, iphlen);

	skb_pull(skb, authlen);

	/*
	 *	Adjust pointers
	 */
	 
	
	len = skb->len; dat = skb->data;
	
#ifdef DEBUG_IPSEC_AH
	if (debug_ah & DB_AH_PKTRX2)
	{
		printk("ipsec_ah: new ip packet is %d bytes:", len);
		for (i = 0; i < len; i++)
			printk(" %02x", dat[i]);
		printk("\n");
	}
#endif

	skb->h.iph=(struct iphdr *)skb->data;
	skb->ip_hdr=(struct iphdr *)skb->data;

	memset(skb->proto_priv, 0, sizeof(struct options));
	
	ipp = (struct iphdr *)dat;
	ipp->check = ip_fast_csum((unsigned char *)dat, iphlen >> 2);

	return skb;
}

int
ahhmacsha1_attach(void)
{
	printk("ahhmacsha1_attach: called.\n");
	return 0;
}

/*
 * ahhmacsha1_init() is called when an SPI is being set up. It interprets the
 * encap_msghdr present in m, and sets up the transformation data.
 */

int
ahhmacsha1_init(struct tdb *tdbp, struct xformsw *xsp, struct encap_msghdr *em)
{
	struct ahhmacsha1_edata *ed;
	struct ahhmacsha1_xdata *xd;
	unsigned char kb[AHSHA1_KMAX];
	
	int i;
	
	tdbp->tdb_xform = xsp;

	if (em->em_msglen - EMT_SETSPI_FLEN > sizeof (struct ahhmacsha1_edata))
		return EINVAL;

	tdbp->tdb_xdata = (caddr_t)kmalloc(sizeof (struct ahhmacsha1_xdata), GFP_ATOMIC); /* XXX - memory leak here if updating old TDB */

	if (tdbp->tdb_xdata == NULL)
	  return ENOBUFS;
	memset(tdbp->tdb_xdata, 0, sizeof (struct ahhmacsha1_xdata));
	
	xd = (struct ahhmacsha1_xdata *)tdbp->tdb_xdata;
	ed = (struct ahhmacsha1_edata *)em->em_dat;
	
	xd->amx_alen = ed->ame_alen;
	xd->amx_replayp = ed->ame_replayp;
	xd->amx_ooowin = ed->ame_ooowin;
	xd->amx_lastseq = xd->amx_bitmap = 0;
	
	SHA1Init(&xd->amx_octx);
	SHA1Init(&xd->amx_ictx);
	
	for (i = 0; i < ed->ame_klen; i++)
		kb[i] = ed->ame_key[i] ^ 0x36;
	for (; i < AHSHA1_KMAX; i++)
		kb[i] = 0x36;
	
	dmp("key is ", kb, AHSHA1_KMAX);

	SHA1Update(&xd->amx_ictx, kb, AHSHA1_KMAX);

	for (i = 0; i < AHSHA1_KMAX; i++)
		kb[i] ^= (0x36 ^ 0x5c);
	SHA1Update(&xd->amx_octx, kb, AHSHA1_KMAX);
	
	return 0;
}



int ahhmacsha1_zeroize(void)
{
	return 0;
}

int ahhmacsha1_print(void *xdat, char *buf)
{
	int size, ds;
#ifdef AHPRINTKEYS
	int i;
#endif
	struct ahhmacsha1_xdata *xd = (struct ahhmacsha1_xdata *)xdat;
	size = sprintf(buf, "alen = %d, win = %d", xd->amx_alen, xd->amx_ooowin);
	buf += size;

	if (xd->amx_replayp)
	{
		ds = sprintf(buf, ", seq = %16Lx", xd->amx_lastseq);
		buf += ds;
		size += ds;
		ds = sprintf(buf, ", bit = %016Lx", xd->amx_bitmap);
		buf += ds;
		size += ds;
	}
	
		
#ifdef AHPRINTKEYS
	/* XXX - print the key in /proc/net/ipsec-spi !!! */
	ds = sprintf(buf, ", key = ");
	buf += ds;
	size += ds;
	for (i = 0; i < xd->amx_klen; i++){
		ds = sprintf(buf, " %02x", xd->amx_key[i]);
		buf += ds;
		size += ds;
	}
#endif
	return size;
}

int
ahhmacsha1_room(struct tdb *tp, int iphlen, int tlen, int *hr, int *tr)
{
	struct ahhmacsha1_xdata *xd = (struct ahhmacsha1_xdata *)(tp->tdb_xdata) ;
	
	*hr += xd->amx_alen + AH_FLENGTH + (xd->amx_replayp ? 8 : 0);
	*tr += 0;
	return 0;
}

int ahhmacsha1_output(struct sk_buff *skb, struct tdb *tdbp)
{
	struct iphdr *ipp, ipo;
	struct ahhmacsha1_xdata *xd;
	struct ah aho;
	struct ah *ahp;
	int iphlen, authlen, len, adoff=0;
	unsigned char *dat;
	unsigned long long seq;
	SHA1_CTX ictx, octx; 
	
	ipp = (struct iphdr *)(skb->data);
	
	iphlen = ipp->ihl << 2;
	if (iphlen != sizeof (struct iphdr))
	{
		printk("ahhmacsha1_output: cannot process options yet\n"); /* XXX */
		return -1;
	}

	xd = (struct ahhmacsha1_xdata *)tdbp->tdb_xdata;
	adoff = xd->amx_replayp ? 8 : 0; /* offset of authenticator */
	authlen = xd->amx_alen + AH_FLENGTH  + adoff; /* same as in xxx_room() */

	dat = skb_push(skb, authlen);
	len = skb->len;
	
	memmove((void *)skb->data, (void *)(skb->data + authlen), iphlen);
	ahp = (struct ah *)(dat + iphlen);

	ipp = (struct iphdr *)dat;

	if (ntohs(ipp->frag_off) & IP_OFFSET)
	{
		printk("ahhmacsha1_output: attempted to authenticate fragment!\n"); /* XXX */
		return -1;
	}

	ahp->ah_nh = ipp->protocol;
	ahp->ah_hl = (authlen - AH_FLENGTH) >> 2;
	ahp->ah_rv = 0;
	ahp->ah_spi = tdbp->tdb_spi;

	if (xd->amx_replayp)
	{
		seq = htonq(++xd->amx_lastseq);
		*((unsigned long long *)(ahp->ah_data)) = seq;
	}

	ictx = xd->amx_ictx;
	octx = xd->amx_octx;
	
	ipp->protocol		=	51;
	ipp->tot_len		=	htons(skb->len);

	ipo = *ipp;
	ipo.tos = 0;
	ipo.frag_off = htons(ntohs(ipo.frag_off) & IP_DF);	/* XXX -- and the C bit? */
	ipo.ttl = 0;
	ipo.check = 0;

	dmp("ictx", (caddr_t)&ictx, 24);
	dmp("octx", (caddr_t)&octx, 24);

	dmp("pseudoheader", (caddr_t)&ipo, sizeof (struct iphdr));
	SHA1Update(&ictx, (unsigned char *)&ipo, sizeof (struct iphdr));
	dmp("authenticator", (caddr_t)ahp, AH_FLENGTH + adoff);
	SHA1Update(&ictx, (unsigned char *)ahp, AH_FLENGTH + adoff);
	dmp("zeroes", (caddr_t)sha1zeroes, xd->amx_alen);
	SHA1Update(&ictx, (unsigned char *)sha1zeroes, xd->amx_alen);
	dmp("packet", (caddr_t)(dat + iphlen + authlen), len - iphlen - authlen);
	SHA1Update(&ictx,  dat + iphlen + authlen, len - iphlen - authlen);
	SHA1Final((unsigned char *)(&(aho.ah_data[0])), &ictx);
	dmp("outer", (caddr_t)(&(aho.ah_data[0])), 20);
	SHA1Update(&octx, (unsigned char *)(&(aho.ah_data[0])), 20);
	SHA1Final((unsigned char *)(&(aho.ah_data[0])), &octx);
	
	memcpy(ahp->ah_data + adoff, aho.ah_data, xd->amx_alen);

	skb->ip_hdr = skb->h.iph = (struct iphdr *) skb->data;
	
	ipp->check = 0;
	ipp->check = ip_fast_csum((unsigned char *)ipp, ipp->ihl);

	return 0;
}
	






