/*
 *
			   IPSEC for Linux
		         Preliminary Release
 
	    Copyright (C) 1996, John Ioannidis <ji@hol.gr>
 
		 LIMITED PRELIMINARY RELEASE LICENCE
 	
  Permission to copy, use, and distribute unmodified copies of this
  software without fee is hereby granted, provided that this entire
  notice is included in all copies.

  No modified copies may be distributed.

  [[ This restriction will, of course, change when the code becomes
  more stable. While you may of course still distribute context-diffs
  (or anything equivalent), I strongly urge you to send any changes
  you have directly to me. This will help the community by providing a
  reference base for the code. Thanks, /ji ]]
 
  THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR
  IMPLIED WARRANTY. IN PARTICULAR, NEITHER THE AUTHOR NOR ANYONE
  DISTRIBUTING THIS SOFTWARE MAKE ANY REPRESENTATION OR WARRANTY OF
  ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR ITS
  FITNESS FOR ANY PARTICULAR PURPOSE.
 
  *
  */

/*
 * $Id: setsa.c,v 0.5 1997/06/03 04:31:55 ji Rel $
 *
 * $Log: setsa.c,v $
 * Revision 0.5  1997/06/03 04:31:55  ji
 * Added esp 3des-md5-96
 *
 * Revision 0.4  1997/01/15 01:37:54  ji
 * New program in this release, replaces set* programs.
 *
 *
 */

/*
 * All-in-one program to set Security Association parameters
 */

#include <sys/types.h>
#include <string.h>
#include <errno.h>

#include <sys/socket.h>
#include <linux/skbuff.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <unistd.h>
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"
#include "ipsec_ipe4.h"
#include "ipsec_ah.h"
#include "ipsec_esp.h"
#include <getopt.h>
#include <ctype.h>
#include <stdio.h>

u_long bigbuf[1024];
struct encap_msghdr *em = (struct encap_msghdr *)bigbuf;

#define streql(_a,_b) (!strcmp((_a),(_b)))

inline int x2i(char *s)
{
	char ss[3];
	ss[0] = s[0];
	ss[1] = s[1];
	ss[2] = 0;

	return strtol(ss, NULL, 16);
}

static char *usage_string = "\n\
Usage: setsa <dstaddr> <spi>\n\
followed by transform-specific argments:\n\
IP-in-IP encapsulation:\n\
    ip4 <encap-src> <encap-dst>\n\
\n\
Authentication Header:\n\
    ah md5 <key>\n\
    ah hmac-md5|hmac-sha1 [ -w <replay_window> ] <key>\n\
\n\
Encapsulating Security Payload:\n\
    esp <dstaddr> <spi> \n\
        des-cbc <iv> <key>\n\
        des-md5|3des-md5|3des-md5-96 i|r [ -w <replay-window> ] [ <iv> ] <key>\n\
\n\
Where:\n\
    <dstaddr> is a dotted-decimal IP address\n\
    <spi>, <iv>, and <key> are given as hex strings with no leading 0x\n\
    <replay-window> is in decimal.\n\
         hmac-md5 and hmac-sha1 have replay checking disabled by default,\n\
            and the range is 1 through 64.\n\
         des-md5 and 3des-md5 have a default replay window size of 1.\n\
    `i' and `r' stand for Initiator and Responder, respectively.\n\
";

struct cmd
{
	char *cm_name;
	int (*cm_nxtok)(int, char *([]), struct cmd *);
	int cm_alg;			/* see ipsec_xform.h */
};

int f_ip4(int ac, char *av[], struct cmd *cp);
int f_ah(int ac, char *av[], struct cmd *cp);
int f_esp(int ac, char *av[], struct cmd *cp);
int f_md5(int ac, char *av[], struct cmd *cp);
int f_hmac(int ac, char *av[], struct cmd *cp);
int f_des(int ac, char *av[], struct cmd *cp);
int f_blkrply(int ac, char *av[], struct cmd *cp);


struct cmd main_cmds[] = {
	{ "ip4", f_ip4, XF_IP4},
	{ "ah", f_ah, 0},
	{ "esp", f_esp, 0},
	{ NULL, NULL, 0},
};

struct cmd ah_cmds[] = {
	{ "md5", f_md5, XF_AHMD5},
	{ "hmac-md5", f_hmac, XF_AHHMACMD5},
	{ "hmac-sha1", f_hmac, XF_AHHMACSHA1},
	{ NULL, NULL, 0},
};

struct cmd esp_cmds[] = {
	{ "des-cbc", f_des, XF_ESPDES},
	{ "des-md5", f_blkrply, XF_ESPDESMD5},
	{ "3des-md5", f_blkrply, XF_ESP3DESMD5},
	{ "3des-md5-96", f_blkrply, XF_ESP3DESMD596},
	{ NULL, NULL, 0},
};
		
void
usage(char *s)
{
	fprintf(stderr, "usage error in %s\n%s", s, usage_string);
	exit(-1);
}

int
f_ip4(int ac, char *av[], struct cmd *cp)
{
	struct ipe4_xdata *xd = (struct ipe4_xdata *)(em->em_dat);

	if (ac != 2)
		usage("f_ip4");

	em->em_msglen = EMT_SETSPI_FLEN + EMT_IPE4_ULEN;
	em->em_alg = XF_IP4;
	
	xd->i4_src.s_addr = inet_addr(av[0]);
	xd->i4_dst.s_addr = inet_addr(av[1]);
	
	return 0;
}

int
f_ah(int ac, char *av[], struct cmd *cp)
{
	if (ac <= 0)
		usage("f_ah");
	
	return gparse(ac, av, ah_cmds);
}

int
f_esp(int ac, char *av[], struct cmd *cp)
{
	/*
	 * XXX - Why do we need this? It's identical to f_ah!
	 */

	if (ac <= 0)
		usage("f_esp");
	
	return gparse(ac, av, esp_cmds);
}
	
int
f_md5(int ac, char *av[], struct cmd *cp)
{
	struct ahmd5_xdata *xd = (struct ahmd5_xdata *)(em->em_dat);
	int i, dlen;
	
	if (ac != 1)
		usage("f_md5");
	
	dlen = strlen(av[0])/2;

	em->em_msglen = EMT_SETSPI_FLEN + 2 * sizeof (u_short) + dlen;
	em->em_alg = cp->cm_alg;
	
	xd->amx_klen = dlen;
	xd->amx_alen = 16;
	for (i = 0; i < dlen; i++ )
		xd->amx_key[i] = x2i(&(av[0][2*i]));
	return 0;
}

int
f_hmac(int ac, char *av[], struct cmd *cp)
{
	struct ahhmacmd5_edata *xd = (struct ahhmacmd5_edata *)(em->em_dat);
	int i, dlen, wl=0;
	
	if ((ac != 1) && (ac != 3))
		usage("f_hmac");
	
	if (ac == 3)
	{
		if (!streql(av[0], "-w") || ((wl = atoi(av[1])) < 1) || (wl > 64))
			usage("f_hmac:2");
		av += 2;
	}
	dlen = strlen(av[0])/2;

	em->em_msglen = EMT_SETSPI_FLEN + 3 * sizeof (u_short) + 2 * sizeof (u_char) + dlen;
	em->em_alg = cp->cm_alg;

	xd->ame_klen = dlen;
	if (cp->cm_alg == XF_AHHMACMD5)
		xd->ame_alen = 16;
	else
		xd->ame_alen = 20;
	xd->ame_x0 = 0;
	if (wl)
	{
		xd->ame_replayp = 1;
		xd->ame_ooowin = wl;
	}
	else
		xd->ame_replayp = xd->ame_ooowin = 0;
	for (i = 0; i < dlen; i++ )
		xd->ame_key[i] = x2i(&(av[0][2*i]));
	return 0;
}

int
f_des(int ac, char *av[], struct cmd *cp)
{
	struct espdes_xdata *xd;
	int i, dlen;
	
	if (ac != 2)
		usage("f_des");

	em->em_msglen = EMT_SETSPI_FLEN + EMT_ESPDESCBC_ULEN;
	em->em_alg = XF_ESPDES;
	dlen = strlen(av[0])/2;

	if ((dlen != 4) && (dlen != 8))
	{
		
		fprintf(stderr, "f_des: only IV lengths of 4 and 8 supported\n");
		exit(1);
	}
	
	xd = (struct espdes_xdata *)em->em_dat;
	xd->edx_ivlen = dlen;

	for (i = 0; i < dlen; i++ )
	  xd->edx_iv[i] = x2i(&(av[0][2*i]));

	dlen = strlen(av[1])/2;

	if (dlen != 8)
	{
		fprintf(stderr, "f_des: only key length of 8 supported\n");
		exit(1);
	}
	
	for (i = 0; i < dlen; i++ )
	  xd->edx_rk[i] = x2i(&(av[1][2*i]));

	return 0;
}

int
f_blkrply(int ac, char *av[], struct cmd *cp)
{
	int i, dlen, wl=1, dir;
	struct espblkrply_edata *ed;

	ed = (struct espblkrply_edata *)(em->em_dat);

	if (ac < 2)
		usage("f_blk");

	if (streql(av[0], "i"))
		ed->eme_flags = EME_INITIATOR;
	else if (streql(av[0], "r"))
		ed->eme_flags = 0;
	else
		usage("bad i/r specifier");
	
	if (streql(av[1], "-w"))
		if (ac < 3)
			usage("bad or missing window size");
		else 
		{
			wl = atoi(av[2]);
			av += 2;
			ac -= 2;
		}

	if (ac == 2)
		ed->eme_flags |= EME_CONSTIV;
	else if (ac != 3)
		usage("f_blk");
	else				/* we have an explicit IV */
	{
		dlen = strlen(av[1])/2;
		if (dlen != 8)
		{
			fprintf(stderr, "f_blk: only IV length of 8 supported\n");
			exit(1);
		}
		for (i = 0; i < dlen; i++ )
			ed->eme_iv[i] = x2i(&(av[1][2*i]));
		av++;
		ac--;
	}
	
	dlen = strlen(av[1])/2;
	if (dlen > EMT_ESPDES_KMAX)
	{
		fprintf(stderr, "f_blk: max key size is %d\n", EMT_ESPDES_KMAX);
		exit(1);
	}
	ed->eme_klen = dlen;
	for (i = 0; i < dlen; i++ )
		ed->eme_key[i] = x2i(&(av[1][2*i]));

	em->em_alg = cp->cm_alg;
	em->em_msglen = EMT_SETSPI_FLEN + sizeof (struct espblkrply_edata);
	ed->eme_ooowin = wl;

	return 0;
}


/*
 * This is effectively a simple recursive-descent parser.
 */
int
gparse(int ac, char *av[], struct cmd *cmp)
{
	struct cmd *cp = cmp;
	
	if (ac == 0)
		return 0;
	
	for (cp = cmp; cp->cm_name && cp->cm_nxtok; cp++)
		if (streql(av[0], cp->cm_name))
			return (*(cp->cm_nxtok))(ac-1, av+1, cp);
	usage("gparse");
}


/*
 * Main driver (what else?!)
 */

/* #define XDUMP */

main(int argc, char *argv[])
{
	int fd;

	if (argc < 4)
		usage("main");		/* also exits */
	
#ifndef XDUMP
	fd = open("/dev/ipsec", 2);
	if (fd < 0)
		perror("open"), exit(1);
#endif
	em->em_magic = EM_MAGIC;
	/* em->em_msglen is set by the appropriate routine */
	em->em_version = 0;
	em->em_type = EMT_SETSPI;
	em->em_dst.s_addr = inet_addr(argv[1]);
	em->em_spi = htonl(strtol(argv[2], NULL, 16));
	em->em_if = 0;
	/* em->em_alg is set by the appropriate routine */

	if (gparse(argc-3, argv+3, main_cmds))
		exit(1);
#ifdef XDUMP
	xdump((caddr_t)em, em->em_msglen, "SETSPI:");
#else
	if (write(fd, (caddr_t)em, em->em_msglen) == em->em_msglen)
		exit(0);
	else
		perror("write");
#endif
	exit(1);
}



#define isdot(_x)  (((_x)&0x80)|!(isprint(_x)))

/*
 * NAME
 *	xdump -- make a hex dump of a region in memory
 *
 * SYNOPSIS
 *	xdump(base, length, title)
 *	caddr_t base;
 *	int length;
 *	char *title;
 */
	
static char line[80];
static char hd[17]="0123456789ABCDEF";

#define HI(_x)  (hd[((_x)>>4)&0xF])
#define LO(_x)  (hd[(_x)&0xF])

xdump(base, length, title)
caddr_t base;
int length;
char *title;
{
	register char *bp, *hp, *cp;
	register int cnt;

	printf("%s\n", title);
	
	bp = base;

	hp = line;
	cp = line+50;
	
	for (cnt=80; cnt; line[--cnt]=' ')
	  ;

	line[49] = line[66] = '*';
	line[67] = '\0';

	while(length-- > 0)
	{
		*hp++ = HI(*bp);
		*hp++ = LO(*bp);
		hp++;
		*cp++ = isdot(*bp)?'.':(*bp);
		bp++;

		if (++cnt == 16)
		{
			cnt = 0;
			hp = line;
			cp = line+50;
			puts(line);
		}
	}

	if (cnt)
	{
		while (cnt++ < 16)
		{
			*hp++=' ';
			*hp++=' ';
			*hp++=' ';
			*cp++=' ';
		}
		puts(line);
	}

	return 0;
}

