/*
 * This program is an implementation of the ISAKMP Internet Standard.
 * Copyright (C) 1997 Angelos D. Keromytis.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * This code was written while the author was in Greece, in May/June
 * 1997.
 *
 * You may contact the author by:
 *   e-mail: angelos@dsl.cis.upenn.edu
 *  US-mail: Angelos D. Keromytis
 *           Distributed Systems Lab
 *           Computer and Information Science Department
 *           University of Pennsylvania
 *           Moore Building
 *           200 South 33rd Street
 *           Philadelphia, PA 19104	   
 */

#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "constants.h"
#include "state.h"
#include "packet.h"

/*
 * This file has all the IPsec DOI and Oakley resolution routines
 */

/*
 * XXX This should be replaced by a call to the kernel when
 * we get an API.
 * The returned SPI is in network byte order.
 */
static u_int32_t
oakley_get_spi(void)
{
    static u_int32_t spi = 0x100;
    
    spi++;
    return htonl(spi);
}

/*
 * Get an Oakley Proposal. The second argument could be used
 * to decide on a per-peer basis what the proposal(s) should be.
 * XXX For now, we offer one very basic proposal.
 */
u_char *
oakley_get_oakley_prop(int *length, struct sockaddr_in sin)
{
    u_char p[8192], *q;
    int i, proposal = 1, transform = 1;
    u_int32_t spi;
    struct isakmp_proposal *isap;
    struct isakmp_transform *isat;
    struct isakmp_attribute *isaa;
    
    bzero(p, 8192);
    
    /* Proposal header -- ISAKMP */
    isap = (struct isakmp_proposal *) p;
    i = sizeof(struct isakmp_proposal);
	
    isap->isap_np = ISAKMP_NEXT_NONE;
    isap->isap_proposal = proposal++;
    isap->isap_protoid = PROTO_ISAKMP;
    isap->isap_spisize = IPSEC_DOI_SPI_SIZE;
    isap->isap_notrans = 1;
	
    /* Get an SPI -- it is assumed it's already in network byte order */
    spi = oakley_get_spi();
    bcopy(&spi, p + i, sizeof(spi));
    
    i += IPSEC_DOI_SPI_SIZE;
    
    /* Transform header -- Oakley */
    isat = (struct isakmp_transform *) (p + i);
	
    isat->isat_np = ISAKMP_NEXT_NONE;
    isat->isat_transnum = transform++;
    isat->isat_transid = KEY_OAKLEY;
    isat->isat_length = i;		/* Use as a temporary variable */
    i += sizeof(struct isakmp_transform);
    
    /* Encryption algorithm attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_ENCRYPTION_ALGORITHM;
    isaa->isaat_lv = htons(OAKLEY_3DES_CBC);
    isaa->isaat_af = 1;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);
    
    /* Hash algorithm attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_HASH_ALGORITHM;
    isaa->isaat_lv = htons(OAKLEY_SHA);
    isaa->isaat_af = 1;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);

    /* Authentication method attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_AUTHENTICATION_METHOD;
    isaa->isaat_lv = htons(OAKLEY_PRESHARED_KEY);
    isaa->isaat_af = 1;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);

    /* Group Description attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_GROUP_DESCRIPTION;
    isaa->isaat_lv = htons(OAKLEY_DEFAULT_GROUP);
    isaa->isaat_af = 1;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);

    /* Lifetime Type attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_LIFE_TYPE;
    isaa->isaat_lv = htons(OAKLEY_LIFE_SECONDS);
    isaa->isaat_af = 1;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);

    /* Lifetime Duration attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_LIFE_DURATION;
    isaa->isaat_lv = htons(OAKLEY_ISAKMP_SA_LIFETIME);
    isaa->isaat_af = 1;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);

    isat->isat_length = htons(i - isat->isat_length);
    isap->isap_length = htons(i);
    
    q = (u_char *)calloc(i, sizeof(u_char));
    if (q == (u_char *) NULL)
    {
	log(1, "calloc() failed in oakley_get_oakleu_prop()", 0, 0, 0);
	return q;
    }
    
    bcopy(p, q, i);
    *length = i;
    return q;
}

/*
 * Construct an SA payload for Oakley, depending on the goal. Return
 * a valid SA payload, length of the buffer on the first argument.
 */
u_char *
oakley_get_sa(int *length, struct sockaddr_in sin, int goal)
{
    struct isakmp_sa *isa;
    u_char p[8192], *q;
    int i, j;

    bzero(p, 8192);
    
    isa = (struct isakmp_sa *) p;
    
    isa->isasa_np = ISAKMP_NEXT_NONE;
    isa->isasa_doi = htonl(ISAKMP_DOI_IPSEC);

    i = sizeof(struct isakmp_sa);

    /* We don't do labeled integrity/secrecy, so forget the LDI */
    p[i + IPSEC_DOI_SITUATION_LENGTH - 1] = SIT_IDENTITY_ONLY;
    i += IPSEC_DOI_SITUATION_LENGTH;
    
    if (goal & GOAL_KEYMANAGEMENT)
    {
	q = oakley_get_oakley_prop(&j, sin);
	if (q == (u_char *) NULL)
	  return q;

	bcopy(q, p + i, j);
	free(q);
	i += j;
    }
    else
    {
	if (goal & GOAL_ENCRYPT)
	{
	    /* XXX */
	}
	else
	  if (goal & GOAL_AUTHENTICATE)
	  {
	      /* XXX */
	  }
	  else
	    if (goal & (GOAL_ENCRYPT|GOAL_AUTHENTICATE))
	    {
		/* XXX */
	    }
    }
    
    isa->isasa_length = htons(i);
    q = (u_char *) calloc(i, sizeof(u_char));
    if (q == (u_char *) NULL)
    {
	log(1, "calloc() failed in oakley_get_sa()", 0, 0, 0);
	return q;
    }

    bcopy(p, q, i);
    *length = i;
    return q;
}

 
/*
 * Initiate an exchange for Oakley - we don't do the
 * disgusting aggressive mode
 */
void
oakley_initiate(int sock, struct sockaddr_in sin, int goal)
{
    u_char *buffer, *p;
    struct isakmp_hdr isa;
    struct isakmp_sa *isasa;
    struct event *ev;
    struct state *st;
    int j;

    /* XXX Check whether we should go to Quick mode directly */

    bzero(&isa, sizeof(isa));
    
    /* R-cookie, flags and MessageID is left zero */

    isa.isa_maj = ISAKMP_MAJOR_VERSION;
    isa.isa_min = ISAKMP_MINOR_VERSION;
    isa.isa_np = ISAKMP_NEXT_SA;
    isa.isa_xchg = ISAKMP_XCHG_BASE;
    isa.isa_length = sizeof(isa);
    
    get_cookie(ISAKMP_INITIATOR, &(isa.isa_icookie), COOKIE_SIZE, sin);

    p = oakley_get_sa(&j, sin, GOAL_KEYMANAGEMENT);
    if (p == (u_char *) NULL)
      return;

    isa.isa_length += j;
    isa.isa_length = htonl(isa.isa_length);
    
    buffer = (u_char *) calloc(sizeof(isa) + j, sizeof(u_char));
    if (buffer == (u_char *) NULL)
    {
	free(p);
	log(1, "calloc() failed in oakley_initiate()", 0, 0, 0);
	return;
    }
    
    bcopy(&isa, buffer, sizeof(isa));
    bcopy(p, buffer + sizeof(isa), j);
    free(p);

    st = (struct state *) get_state();

    isasa = (struct isakmp_sa *) (buffer + sizeof(isa));
    st->st_mysituation = SIT_IDENTITY_ONLY;
    st->st_goal = goal;
    st->st_state = OAKLEY_MAIN_1;
    bcopy(isa.isa_icookie, st->st_icookie, COOKIE_SIZE);
    bcopy(&sin, &(st->st_peer), sizeof(sin));
    st->st_packet = buffer;
    st->st_packet_len = sizeof(isa) + j;

    insert_state(st);
    
    if (sendto(sock, buffer, sizeof(isa) + j, 0, (struct sockaddr *)&sin,
	       sizeof(sin)) != sizeof(isa) + j)
      log(1, "sendto() failed in oakley_initiate()", 0, 0, 0);

    /* Set up a retransmission event, half a minute henceforth */
    event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st, 0);
}


/*
 * Demux for IPsec DOI initiator
 */
void
ipsecdoi_initiate(int sock, struct sockaddr_in sin, int goal, 
		  int protocol)
{
    switch (protocol)
    {
	case KEY_OAKLEY:
	    oakley_initiate(sock, sin, goal);
	    break;

	case KEY_MANUAL:
	case KEY_KDC:
	default:
	    log(0, "unsupported protocol %d specified in ipsecdoi_initiate()",
		protocol, 0, 0);
	    return;
    }
}
