/*
 * This program is an implementation of the ISAKMP Internet Standard.
 * Copyright (C) 1997 Angelos D. Keromytis.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * This code was written while the author was in Greece, in May/June
 * 1997.
 *
 * You may contact the author by:
 *   e-mail: angelos@dsl.cis.upenn.edu
 *  US-mail: Angelos D. Keromytis
 *           Distributed Systems Lab
 *           Computer and Information Science Department
 *           University of Pennsylvania
 *           Moore Building
 *           200 South 33rd Street
 *           Philadelphia, PA 19104	   
 */

#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "constants.h"
#include "state.h"
#include "packet.h"
#include "defs.h"
#include "sha1.h"
#include "md5.h"

#define PRINTADDRESS(x) (get_address(x) != (char *) NULL ? get_address(x) : "(unknown address family)")

extern char *get_address();

/*
 * This file has all the IPsec DOI and Oakley resolution routines.
 */

/*
 * XXX This should be replaced by a call to the kernel when
 * we get an API.
 * The returned SPI is in network byte order.
 */
static u_int32_t
oakley_get_spi(void)
{
    static u_int32_t spi = 0x100;
    
    spi++;
    return htonl(spi);
}

/*
 * Get an Oakley Proposal. The second argument could be used
 * to decide on a per-peer basis what the proposal(s) should be.
 * XXX For now, we offer one very basic proposal.
 */
u_char *
oakley_get_oakley_prop(int *length, struct sockaddr_in sin, struct state *st)
{
    u_char p[8192], *q;
    int i, proposal = 1, transform = 1;
    u_int32_t spi;
    struct isakmp_proposal *isap;
    struct isakmp_transform *isat;
    struct isakmp_attribute *isaa;
    
    bzero(p, 8192);
    
    /* Proposal header -- ISAKMP */
    isap = (struct isakmp_proposal *) p;
    i = sizeof(struct isakmp_proposal);
	
    isap->isap_np = ISAKMP_NEXT_NONE;
    isap->isap_proposal = proposal++;
    isap->isap_protoid = PROTO_ISAKMP;
    isap->isap_spisize = IPSEC_DOI_SPI_SIZE;
    isap->isap_notrans = 1;
	
    /* Get an SPI -- it is assumed it's already in network byte order */
    spi = oakley_get_spi();
    bcopy(&spi, p + i, sizeof(spi));
    
    i += IPSEC_DOI_SPI_SIZE;
    
    /* Transform header -- Oakley */
    isat = (struct isakmp_transform *) (p + i);
	
    isat->isat_np = ISAKMP_NEXT_NONE;
    isat->isat_transnum = transform++;
    isat->isat_transid = KEY_OAKLEY;
    isat->isat_length = i;		/* Use as a temporary variable */
    i += sizeof(struct isakmp_transform);
    
    /* Encryption algorithm attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_ENCRYPTION_ALGORITHM;
    isaa->isaat_lv = htons(OAKLEY_DES_CBC);
    isaa->isaat_af = 1;
    st->st_enc = OAKLEY_DES_CBC;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);
    
    /* Hash algorithm attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_HASH_ALGORITHM;
    isaa->isaat_lv = htons(OAKLEY_SHA);
    isaa->isaat_af = 1;
    st->st_hash = OAKLEY_SHA;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);

    /* Authentication method attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_AUTHENTICATION_METHOD;
    isaa->isaat_lv = htons(OAKLEY_PRESHARED_KEY);
    isaa->isaat_af = 1;
    st->st_auth = OAKLEY_PRESHARED_KEY;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);

    /* Group Description attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_GROUP_DESCRIPTION;
    isaa->isaat_lv = htons(OAKLEY_DEFAULT_GROUP);
    isaa->isaat_af = 1;
    st->st_group = OAKLEY_DEFAULT_GROUP;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);

    /* Lifetime Type attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_LIFE_TYPE;
    isaa->isaat_lv = htons(OAKLEY_LIFE_SECONDS);
    isaa->isaat_af = 1;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);

    /* Lifetime Duration attribute */
    isaa = (struct isakmp_attribute *) (p + i);
    isaa->isaat_type = OAKLEY_LIFE_DURATION;
    isaa->isaat_lv = htons(OAKLEY_ISAKMP_SA_LIFETIME);
    isaa->isaat_af = 1;
    st->st_expire = OAKLEY_ISAKMP_SA_LIFETIME;
    isaa->isaat_type = htons(isaa->isaat_type);
    i += sizeof(struct isakmp_attribute);

    isat->isat_length = htons(i - isat->isat_length);
    isap->isap_length = htons(i);
    
    q = (u_char *)calloc(i, sizeof(u_char));
    if (q == (u_char *) NULL)
      exit_log("calloc() failed in oakley_get_oakleu_prop()", 0, 0, 0);
    
    bcopy(p, q, i);
    *length = i;
    return q;
}

/*
 * Construct an SA payload for Oakley, depending on the goal. Return
 * a valid SA payload, length of the buffer on the first argument.
 */
u_char *
oakley_get_sa(int *length, struct sockaddr_in sin, int goal, struct state *st)
{
    struct isakmp_sa *isa;
    u_char p[8192], *q;
    int i, j;

    bzero(p, 8192);
    
    isa = (struct isakmp_sa *) p;
    
    isa->isasa_np = ISAKMP_NEXT_NONE;
    isa->isasa_doi = htonl(ISAKMP_DOI_IPSEC);

    i = sizeof(struct isakmp_sa);

    /* We don't do labeled integrity/secrecy, so forget the LDI */
    p[i + IPSEC_DOI_SITUATION_LENGTH - 1] = SIT_IDENTITY_ONLY;
    i += IPSEC_DOI_SITUATION_LENGTH;
    
    if (goal & GOAL_KEYMANAGEMENT)
    {
	q = oakley_get_oakley_prop(&j, sin, st);
	if (q == (u_char *) NULL)
	  return q;

	bcopy(q, p + i, j);
	free(q);
	i += j;
    }
    else
    {
	if (goal & GOAL_ENCRYPT)
	{
	    /* XXX */
	}
	else
	  if (goal & GOAL_AUTHENTICATE)
	  {
	      /* XXX */
	  }
	  else
	    if (goal & (GOAL_ENCRYPT|GOAL_AUTHENTICATE))
	    {
		/* XXX */
	    }
    }
    
    isa->isasa_length = htons(i);
    q = (u_char *) calloc(i, sizeof(u_char));
    if (q == (u_char *) NULL)
      exit_log("calloc() failed in oakley_get_sa()", 0, 0, 0);

    bcopy(p, q, i);
    *length = i;
    return q;
}

 
/*
 * Initiate an exchange for Oakley.
 */
void
oakley_initiate(int sock, struct sockaddr_in sin, int goal)
{
    u_char *buffer, *p;
    struct isakmp_hdr isa;
    struct isakmp_sa *isasa;
    struct event *ev;
    struct state *st;
    int j;

    /* XXX Check whether we should go to Quick mode directly */

    bzero(&isa, sizeof(isa));
    
    /* R-cookie, flags and MessageID is left zero */

    isa.isa_maj = ISAKMP_MAJOR_VERSION;
    isa.isa_min = ISAKMP_MINOR_VERSION;
    isa.isa_np = ISAKMP_NEXT_SA;
    isa.isa_xchg = ISAKMP_XCHG_IDPROT;
    isa.isa_length = sizeof(isa);
    
    get_cookie(ISAKMP_INITIATOR, &(isa.isa_icookie), COOKIE_SIZE, sin);

    st = (struct state *) get_state();

    p = oakley_get_sa(&j, sin, GOAL_KEYMANAGEMENT, st);
    if (p == (u_char *) NULL)
    {
	free_state(st);
	return;
    }
    
    isa.isa_length += j;
    isa.isa_length = htonl(isa.isa_length);
    
    buffer = (u_char *) calloc(sizeof(isa) + j, sizeof(u_char));
    if (buffer == (u_char *) NULL)
      exit_log("calloc() failed in oakley_initiate()", 0, 0, 0);
    
    bcopy(&isa, buffer, sizeof(isa));
    bcopy(p, buffer + sizeof(isa), j);

    isasa = (struct isakmp_sa *) (buffer + sizeof(isa));
    st->st_situation = SIT_IDENTITY_ONLY;
    st->st_goal = goal;
    st->st_state = OAKLEY_MAIN_I_1;
    bcopy(isa.isa_icookie, st->st_icookie, COOKIE_SIZE);
    bcopy(&sin, &(st->st_peer), sizeof(sin));
    st->st_packet = buffer;
    st->st_packet_len = sizeof(isa) + j;
    st->st_doi = ISAKMP_DOI_IPSEC;
    
    /* Copy sent SA */
    st->st_sa = (u_char *) calloc(j, sizeof(u_char));
    if (st->st_sa == (u_char *) NULL)
      exit_log("calloc() failed in oakley_initiate()", 0, 0, 0);
    bcopy(p, st->st_sa, j);
    free(p);

    insert_state(st);
    
    if (sendto(sock, buffer, sizeof(isa) + j, 0, (struct sockaddr *)&sin,
	       sizeof(sin)) != sizeof(isa) + j)
      log(1, "sendto() failed in oakley_initiate()", 0, 0, 0);

    /* Set up a retransmission event, half a minute henceforth */
    event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st, 0);
}

/*
 * Demux for IPsec DOI initiator.
 */
void
ipsecdoi_initiate(int sock, struct sockaddr_in sin, int goal, 
		  int protocol)
{
    switch (protocol)
    {
	case KEY_OAKLEY:
	    oakley_initiate(sock, sin, goal);
	    break;

	case KEY_MANUAL:
	case KEY_KDC:
	default:
	    log(0, "unsupported protocol %d specified in ipsecdoi_initiate()",
		protocol, 0, 0);
	    return;
    }
}

/*
 * Examine a list of IPsec DOI policies and copy the acceptable one. If
 * none is acceptable, return 0. Also copy the accepted proposal in the
 * state object.
 * XXX The policy is hardwired now; we only accept a policy
 * XXX with the Oakley transform.
 * XXX It also doesn't handle policies that span multiple protocols.
 */
int
ipsecdoi_oakley_sa_copy(u_char *from, u_char *to, int length,
			struct sockaddr sa, struct state *st)
{
    struct isakmp_proposal *isapf, *isapt;
    struct isakmp_attribute *isaat;
    struct isakmp_transform *isat;
    struct isakmp_sa *isasa;
    int l, k, j, i = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa) +
		     IPSEC_DOI_SITUATION_LENGTH;
    int enc, hash, group, auth;
    int acceptable, len;
        
    isapt = (struct isakmp_proposal *) (to + i);

    isasa = (struct isakmp_sa *) (from + sizeof(struct isakmp_hdr));
    len = ntohs(isasa->isasa_length) - sizeof(struct isakmp_sa) -
	  IPSEC_DOI_SITUATION_LENGTH;

    isasa = (struct isakmp_sa *) (to + sizeof(struct isakmp_hdr));
    
    while (1)
    {
	enc = 0;
	hash = 0;
	group = 0;
	auth = 0;
	isapf = (struct isakmp_proposal *) (from + i);
	acceptable = 1;
	
	/* Length mismatch */
	if (ntohs(isapf->isap_length) > len)
	  return 0;
	
	j = i + sizeof(struct isakmp_proposal) + isapf->isap_spisize;
	isat = (struct isakmp_transform *) (from + j);

	if (ntohs(isat->isat_length) > len + sizeof(struct isakmp_proposal) +
	    isapf->isap_spisize)
	  return 0;

	if ((isapf->isap_protoid != PROTO_ISAKMP) ||
	    (isapf->isap_spisize != IPSEC_DOI_SPI_SIZE) ||
	    (isapf->isap_notrans != 1) ||
	    (isat->isat_np != ISAKMP_NEXT_NONE) ||
	    (isat->isat_transid != KEY_OAKLEY))
	{
	    /* Next proposal, if there */
	    if (isapf->isap_np == ISAKMP_NEXT_P)
	    {
		i += ntohs(isapf->isap_length);

		if (length - i <= sizeof(struct isakmp_proposal))
		  return 0;

		len += ntohs(isapf->isap_length) + isapf->isap_spisize;
		isapf = (struct isakmp_proposal *) (from + i);

		if (ntohs(isapf->isap_length) + i > length)
		  return 0;

		continue;
	    }
	    else
	      return 0;
	}

	j += sizeof(struct isakmp_transform);

	k = ntohs(isat->isat_length) - sizeof(struct isakmp_transform);
	while (k != 0)
	{
	    isaat = (struct isakmp_attribute *) (from + j);
	    isaat->isaat_type = ntohs(isaat->isaat_type);

	    l = ntohs(isaat->isaat_lv);
	    
	    if (isaat->isaat_af)
	    {
		j += sizeof(struct isakmp_attribute);
		k -= sizeof(struct isakmp_attribute);
	    }
	    else
	    {
		j += l;
		k -= l;
	    }

	    /* 
	     * XXX We should also be checking whether the attribute
	     * is/should be TLV or TV 
	     */
	    switch (isaat->isaat_rtype)
	    {
		case OAKLEY_ENCRYPTION_ALGORITHM:
		    enc = 1;
#ifdef DEBUG
		    log(0, "encryption algorithm %d", l, 0, 0);
#endif
		    if (l != OAKLEY_DES_CBC)
		      acceptable = 0;
		    st->st_enc = l;
		    break;
		    
		case OAKLEY_HASH_ALGORITHM:
#ifdef DEBUG
		    log(0, "hash algorithm %d", l, 0, 0);
#endif
		    if ((l != OAKLEY_MD5) &&
			(l != OAKLEY_SHA))
		      acceptable = 0;
		    st->st_hash = l;
		    hash = 1;
		    break;
		    
		case OAKLEY_AUTHENTICATION_METHOD:
#ifdef DEBUG
		    log(0, "authentication method %d", l, 0, 0);
#endif
		    if ((l != OAKLEY_PRESHARED_KEY) &&
			(l != OAKLEY_RSA_SIG))
		      acceptable = 0;
		    st->st_auth = l;
		    auth = 1;
		    break;
		    
		case OAKLEY_GROUP_DESCRIPTION:
#ifdef DEBUG
		    log(0, "group description %d", l, 0, 0);
#endif
		    if (l != OAKLEY_DEFAULT_GROUP)
		      acceptable = 0;
		    st->st_group = l;
		    group = 1;
		    break;
		    
		case OAKLEY_LIFE_TYPE:
#ifdef DEBUG
		    log(0, "life type %d", l, 0, 0);
#endif
		    if (l != OAKLEY_LIFE_SECONDS)
		      acceptable = 0;
		    break;
		    
		case OAKLEY_LIFE_DURATION:
#ifdef DEBUG
		    log(0, "life duration %d", l, 0, 0);
#endif
		    if (l < OAKLEY_ISAKMP_SA_LIFETIME / 3)
		      acceptable = 0;
		    st->st_expire = l;
		    break;
		    
		    /* XXX Handle more */
		default:
		    acceptable = 0;
		    log(0, "unsupported attribute %d from %s, port %d",
			isaat->isaat_rtype, PRINTADDRESS(sa), get_port(sa));
		    break;
	    }
	    
	    isaat->isaat_type = htons(isaat->isaat_type);

	    /* Something unacceptable with this policy */
	    if (acceptable == 0)
	      break;
	}

	/* 
	 * One of the default attributes not negotiated or unacceptable
	 * policy 
	 */
	if ((enc == 0) || (auth == 0) || (hash == 0) || (group == 0) ||
	    (acceptable == 0))
	{
	    /* Next proposal, if there */
	    if (isapf->isap_np == ISAKMP_NEXT_P)
	    {
		i += ntohs(isapf->isap_length);
		
		if (length - i <= sizeof(struct isakmp_proposal))
		  return 0;
		
		len += ntohs(isapf->isap_length) + isapf->isap_spisize;
		isapf = (struct isakmp_proposal *) (from + i);
		
		if (ntohs(isapf->isap_length) + i > length)
		  return 0;

		continue;
	    }
	    else
	      return 0;
	}
	
	/* Copy proposal */
	bcopy(isapf, isapt, ntohs(isapf->isap_length));
	isapt->isap_np = ISAKMP_NEXT_NONE;
	
	isasa->isasa_length = htons(ntohs(isapf->isap_length) +
				    sizeof(struct isakmp_sa) +
				    IPSEC_DOI_SITUATION_LENGTH);
	
	st->st_proposal = (u_char *) calloc(ntohs(isapt->isap_length), 
					    sizeof(u_char));
	if (st->st_proposal == (u_char *) NULL)
	  exit_log("calloc() failed in ipsecdoi_oakley_sa_copy()", 0, 0, 0);
	
	bcopy(isapt, st->st_proposal, ntohs(isapt->isap_length));
	
	return 1;
    }
}

/*
 * Handle a Main Mode Oakley first packet (responder side).
 */
void
ipsecdoi_handle_rfirst(int sock, u_char *buffer, int length, 
		      struct sockaddr sa)
{
    struct isakmp_hdr *isa;
    struct isakmp_sa *isasa, *isas;
    struct state *st;
    u_char *packet;
    int i, k;
    struct sockaddr_in sin;
    
    isa = (struct isakmp_hdr *) buffer;
    isasa = (struct isakmp_sa *) (buffer + sizeof(struct isakmp_hdr));

    if (ntohs(isasa->isasa_length) + sizeof(struct isakmp_hdr) >
	length)
    {
	free(buffer);
	log(0, "malformed packet from %s, port %d",
	    PRINTADDRESS(sa), get_port(sa), 0);
	/* XXX Could send notification back */
	return;
    }
	   
    /* XXX Check various fields ? */

    /* Set up state */
    st = (struct state *) get_state();

    bcopy(isa->isa_icookie, st->st_icookie, COOKIE_SIZE);

    get_cookie(ISAKMP_RESPONDER, st->st_rcookie, COOKIE_SIZE, sa);

    st->st_peer = sa;
    st->st_doi = ISAKMP_DOI_IPSEC;
    st->st_situation = SIT_IDENTITY_ONLY; /* We only support this */
    st->st_state = OAKLEY_MAIN_R_1;
    
    /* The reply packet can't be larger than the original request */
    packet = (u_char *) calloc(length, sizeof(u_char));
    if (packet == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_rfirst()", 0, 0, 0);

    isa = (struct isakmp_hdr *) packet;
    
    /* Copy header as received, and responder cookie */
    bcopy(buffer, packet, sizeof(struct isakmp_hdr));
    bcopy(st->st_rcookie, isa->isa_rcookie, COOKIE_SIZE);
    isa->isa_length = 0;
    
    isas = (struct isakmp_sa *) (packet + sizeof(struct isakmp_hdr));

    isas->isasa_doi = ntohl(ISAKMP_DOI_IPSEC);
    i = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa) +
	IPSEC_DOI_SITUATION_LENGTH;
    packet[i - 1] = SIT_IDENTITY_ONLY;

    if (ipsecdoi_oakley_sa_copy(buffer, packet, length, sa, st) == 0)
    {
	free(buffer);
	free(packet);
	free_state(st);
	log(0, "no acceptable proposal from %s, port %d",
	    PRINTADDRESS(sa), get_port(sa), 0);
	/* XXX Could send notification back */
	return;
    }

    st->st_sa = (u_char *) calloc(ntohs(isas->isasa_length), sizeof(u_char));
    if (st->st_sa == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_rfirst()", 0, 0, 0);
    
    bcopy(isas, st->st_sa, ntohs(isas->isasa_length));
    
    /* XXX Handle Aggressive Mode here ? */

    free(buffer);

    st->st_packet = packet;

    /* Get my identity */
    k = sizeof(sin);
    if (getsockname(sock, (struct sockaddr *)&sin, &k) < 0)
      exit_log("getsockname() in ipsecdoi_handle_rfirst()", 0, 0, 0);
    
    if (k <= 0)
      exit_log("getsockname() in ipsecdoi_handle_rfirst()", 0, 0, 0);

    st->st_myidentity = (u_char *) calloc(1, sizeof(struct in_addr));
    if (st->st_myidentity == (u_char *) NULL)
      exit_log("strdup() in ipsecdoi_handle_rfirst()", 0, 0, 0);
    bcopy(&(sin.sin_addr), st->st_myidentity, sizeof(struct in_addr));
    st->st_myidentity_len = sizeof(struct in_addr);
    st->st_myidentity_type = ID_IPV4_ADDR;
    
    length = ntohs(isas->isasa_length) + sizeof(struct isakmp_hdr);
#ifdef DEBUG
    log(0, "my identity is %s", inet_ntoa(sin.sin_addr), 0, 0);
    log(0, "sending %d bytes to %s, port %d", length,
	PRINTADDRESS(sa), get_port(sa));
#endif
    isa->isa_length = ntohl(length);
    st->st_packet_len = length;
    
    if (sendto(sock, packet, length, 0, &sa, sizeof(sa)) != length)
      log(1, "sendto() failed in ipsecdoi_handle_rfirst() for %s, port %d",
	  PRINTADDRESS(sa), get_port(sa), 0);
    
    insert_state(st);
    event_schedule(EVENT_CLEANUP, EVENT_CLEANUP_DELAY, st, 0);
}

/*
 * Get preshared key for remote IP.
 */
u_char *
get_preshared_key(struct state *st, int *length)
{
    u_char *p;
    
    /* XXX Lookup in a file */
    p = (char *)strdup("Angelos");
    *length = strlen("Angelos");
    return p;
}

/*
 * SKEYID for preshared keys.
 */
void
skeyid_preshared(struct state *st, u_char *id, int length)
{
    int k;
    MD5_CTX md5ctx;
    SHA1_CTX sha1ctx;
    char buffer[HMAC_BUFSIZE], buffer2[HMAC_BUFSIZE];
    struct isakmp_generic *isag;
    
    if (st->st_prf == 0)
    {
	/* Prepare the two pads for the HMAC */
	if (length >= HMAC_BUFSIZE)
	{
	    bcopy(id, buffer, HMAC_BUFSIZE);
	    bcopy(id, buffer2, HMAC_BUFSIZE);
	}
	else
	{
	    for (k = 0; k + length < HMAC_BUFSIZE; k += length)
	    {
		bcopy(id, buffer + k, length);
		bcopy(id, buffer2 + k, length);
	    }

	    bcopy(id, buffer + k, HMAC_BUFSIZE - k);
	    bcopy(id, buffer2 + k, HMAC_BUFSIZE - k);
	}
	
	for (k = 0; k < HMAC_BUFSIZE; k++)
	{
	    buffer[k] ^= HMAC_IPAD;
	    buffer2[k] ^= HMAC_OPAD;
	}
	
	switch (st->st_hash)
	{
	    case OAKLEY_MD5:
		st->st_skeyid = (u_char *) calloc(16, sizeof(u_char));
		if (st->st_skeyid == (u_char *) NULL)
		  exit_log("calloc() in skeyid_preshared()", 0, 0, 0);
		st->st_skeyid_len = 16;
		
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, buffer, HMAC_BUFSIZE);

		k = sizeof(struct isakmp_generic);

		isag = (struct isakmp_generic *) st->st_ni;
		MD5Update(&md5ctx, st->st_ni + k, 
			  ntohs(isag->isag_length) - k);
		
		isag = (struct isakmp_generic *) st->st_nr;
		MD5Update(&md5ctx, st->st_nr + k, 
			  ntohs(isag->isag_length) - k);
		
		MD5Final(st->st_skeyid, &md5ctx);
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, buffer2, HMAC_BUFSIZE);
		MD5Update(&md5ctx, st->st_skeyid, 16);
		MD5Final(st->st_skeyid, &md5ctx);
		return;
		
	    case OAKLEY_SHA:
		st->st_skeyid = (u_char *) calloc(20, sizeof(u_char));
		if (st->st_skeyid == (u_char *) NULL)
		  exit_log("calloc() in skeyid_preshared()", 0, 0, 0);
		st->st_skeyid_len = 20;

		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, buffer, HMAC_BUFSIZE);

		k = sizeof(struct isakmp_generic);

		isag = (struct isakmp_generic *) st->st_ni;
		SHA1Update(&sha1ctx, st->st_ni + k, 
			  ntohs(isag->isag_length) - k);
		
		isag = (struct isakmp_generic *) st->st_nr;
		SHA1Update(&sha1ctx, st->st_nr + k, 
			  ntohs(isag->isag_length) - k);
		
		SHA1Final(st->st_skeyid, &sha1ctx);
		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, buffer2, HMAC_BUFSIZE);
		SHA1Update(&sha1ctx, st->st_skeyid, 16);
		SHA1Final(st->st_skeyid, &sha1ctx);
		return;
		
	    default:
		exit_log("unknown hash algorithm %d", st->st_hash, 0, 0);
	}
    }
    else
    {
	/* XXX Handle 3DES */
    }
}


/*
 * Generate the SKEYID_*
 */
void
generate_skeyids(struct state *st)
{
    u_char *id;
    int len, k;
    MD5_CTX md5ctx;
    SHA1_CTX sha1ctx;
    u_char buf1[HMAC_BUFSIZE], buf2[HMAC_BUFSIZE];
    u_char *p, c;
    MP_INT temp, temp2;
    u_char *gi, *gr;
    int gilen, grlen;

    mpz_init(&temp);
    mpz_init(&temp2);
    
    gilen = mpz_sizeinbase(&(st->st_gi), 16) / 2 + 1;
    gi = (u_char *) calloc(gilen, sizeof(u_char));
    if (gi == (u_char *) NULL)
      exit_log("calloc() failed in get_hash()", 0, 0, 0);
    
    mpz_set(&temp, &(st->st_gi));

    for (k = gilen - 1 ; k >= 0; k--)
    {
	gi[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	mpz_set(&temp, &temp2);
    }

    grlen = mpz_sizeinbase(&(st->st_gr), 16) / 2 + 1;
    gr = (u_char *) calloc(grlen, sizeof(u_char));
    if (gr == (u_char *) NULL)
      exit_log("calloc() failed in get_hash()", 0, 0, 0);
    
    mpz_set(&temp, &(st->st_gr));

    for (k = grlen - 1; k >= 0; k--)
    {
	gr[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	mpz_set(&temp, &temp2);
    }

    mpz_clear(&temp);
    mpz_clear(&temp2);

    switch (st->st_auth)
    {
	case OAKLEY_PRESHARED_KEY:
	    /* Generate the SKEYID */
	    id = get_preshared_key(st, &len);
	    skeyid_preshared(st, id, len);
	    free(id);
	    break;
	    
	case OAKLEY_DSS_SIG:
	case OAKLEY_RSA_SIG:
	    /* XXX */
	    break;
	    
	case OAKLEY_RSA_ENC:
	    /* XXX */
	    break;
	    
	default:
	    exit_log("generate_skeyids(): unknown authentication method",
		     0, 0, 0);
    }

    /* Get the g^xy for HMAC'ing */
    len = mpz_sizeinbase(&(st->st_shared), 16) / 2 + 1;
    
#ifdef DEBUG
    log(0, "size of g^xy is %d", len, 0, 0);
#endif

    p = (u_char *) calloc(len, sizeof(u_char));
    if (p == (u_char *) NULL)
      exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);

    mpz_init(&temp);
    mpz_init(&temp2);
    mpz_set(&temp, &(st->st_shared));
    
    for (k = len - 1; k >= 0; k--)
    {
	p[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	mpz_set(&temp, &temp2);
    }
   
    mpz_clear(&temp);
    mpz_clear(&temp2);
    
    if (st->st_prf == 0)
    {
	/* Prepare the two pads for the HMAC */
	if (st->st_skeyid_len >= HMAC_BUFSIZE)
	{
	    bcopy(st->st_skeyid, buf1, HMAC_BUFSIZE);
	    bcopy(st->st_skeyid, buf2, HMAC_BUFSIZE);
	}
	else
	{
	    for (k = 0; k + st->st_skeyid_len < HMAC_BUFSIZE; 
		 k += st->st_skeyid_len)
	    {
		bcopy(st->st_skeyid, buf1 + k, st->st_skeyid_len);
		bcopy(st->st_skeyid, buf2 + k, st->st_skeyid_len);
	    }

	    bcopy(st->st_skeyid, buf1 + k, HMAC_BUFSIZE - k);
	    bcopy(st->st_skeyid, buf2 + k, HMAC_BUFSIZE - k);
	}
	
	for (k = 0; k < HMAC_BUFSIZE; k++)
	{
	    buf1[k] ^= HMAC_IPAD;
	    buf2[k] ^= HMAC_OPAD;
	}

	switch (st->st_hash)
	{
	    case OAKLEY_MD5:
		st->st_skeyid_d_len = 16;
		st->st_skeyid_d = (u_char *) calloc(16, sizeof(u_char));
		if (st->st_skeyid_d == (u_char *) NULL)
		  exit_log("calloc() in generate_skeyids()", 0, 0, 0);
		
		st->st_skeyid_a_len = 16;
		st->st_skeyid_a = (u_char *) calloc(16, sizeof(u_char));
		if (st->st_skeyid_a == (u_char *) NULL)
		  exit_log("calloc() in generate_skeyids()", 0, 0, 0);

		st->st_skeyid_e_len = 16;
		st->st_skeyid_e = (u_char *) calloc(16, sizeof(u_char));
		if (st->st_skeyid_e == (u_char *) NULL)
		  exit_log("calloc() in generate_skeyids()", 0, 0, 0);

		/* SKEYID_D */
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);
		MD5Update(&md5ctx, p, len);
		MD5Update(&md5ctx, st->st_icookie, COOKIE_SIZE);
		MD5Update(&md5ctx, st->st_rcookie, COOKIE_SIZE);
		c = 0;
		MD5Update(&md5ctx, &c, 1);
		MD5Final(st->st_skeyid_d, &md5ctx);
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);
		MD5Update(&md5ctx, st->st_skeyid_d, st->st_skeyid_d_len);
		MD5Final(st->st_skeyid_d, &md5ctx);
		
		/* SKEYID_A */
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);
		MD5Update(&md5ctx, st->st_skeyid_d, st->st_skeyid_d_len);
		MD5Update(&md5ctx, p, len);
		MD5Update(&md5ctx, st->st_icookie, COOKIE_SIZE);
		MD5Update(&md5ctx, st->st_rcookie, COOKIE_SIZE);
		c = 1;
		MD5Update(&md5ctx, &c, 1);
		MD5Final(st->st_skeyid_a, &md5ctx);
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);
		MD5Update(&md5ctx, st->st_skeyid_a, st->st_skeyid_a_len);
		MD5Final(st->st_skeyid_a, &md5ctx);
		
		/* SKEYID_E */
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);
		MD5Update(&md5ctx, st->st_skeyid_a, st->st_skeyid_a_len);
		MD5Update(&md5ctx, p, len);
		MD5Update(&md5ctx, st->st_icookie, COOKIE_SIZE);
		MD5Update(&md5ctx, st->st_rcookie, COOKIE_SIZE);
		c = 2;
		MD5Update(&md5ctx, &c, 1);
		MD5Final(st->st_skeyid_e, &md5ctx);
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);
		MD5Update(&md5ctx, st->st_skeyid_e, st->st_skeyid_e_len);
		MD5Final(st->st_skeyid_e, &md5ctx);

		/* IV */
		st->st_iv_len = 16;
		st->st_iv = (u_char *) calloc(st->st_iv_len, sizeof(u_char));
		if (st->st_iv == (u_char *) NULL)
		  exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
		
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, gi, gilen);
		MD5Update(&md5ctx, gr, grlen);
		MD5Final(st->st_iv, &md5ctx);
		
		break;
		
	    case OAKLEY_SHA:
		st->st_skeyid_d_len = 20;
		st->st_skeyid_d = (u_char *) calloc(20, sizeof(u_char));
		if (st->st_skeyid_d == (u_char *) NULL)
		  exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);

		st->st_skeyid_a_len = 20;
		st->st_skeyid_a = (u_char *) calloc(20, sizeof(u_char));
		if (st->st_skeyid_a == (u_char *) NULL)
		  exit_log("calloc() in generate_skeyids()", 0, 0, 0);

		st->st_skeyid_e_len = 20;
		st->st_skeyid_e = (u_char *) calloc(20, sizeof(u_char));
		if (st->st_skeyid_e == (u_char *) NULL)
		  exit_log("calloc() in generate_skeyids()", 0, 0, 0);

		/* SKEYID_D */
		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, buf1, HMAC_BUFSIZE);
		SHA1Update(&sha1ctx, p, len);
		SHA1Update(&sha1ctx, st->st_icookie, COOKIE_SIZE);
		SHA1Update(&sha1ctx, st->st_rcookie, COOKIE_SIZE);
		c = 0;
		SHA1Update(&sha1ctx, &c, 1);
		SHA1Final(st->st_skeyid_d, &sha1ctx);
		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, buf2, HMAC_BUFSIZE);
		SHA1Update(&sha1ctx, st->st_skeyid_d, st->st_skeyid_d_len);
		SHA1Final(st->st_skeyid_d, &sha1ctx);
		
		/* SKEYID_A */
		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, buf1, HMAC_BUFSIZE);
		SHA1Update(&sha1ctx, st->st_skeyid_d, st->st_skeyid_d_len);
		SHA1Update(&sha1ctx, p, len);
		SHA1Update(&sha1ctx, st->st_icookie, COOKIE_SIZE);
		SHA1Update(&sha1ctx, st->st_rcookie, COOKIE_SIZE);
		c = 1;
		SHA1Update(&sha1ctx, &c, 1);
		SHA1Final(st->st_skeyid_a, &sha1ctx);
		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, buf2, HMAC_BUFSIZE);
		SHA1Update(&sha1ctx, st->st_skeyid_a, st->st_skeyid_a_len);
		SHA1Final(st->st_skeyid_a, &sha1ctx);
		
		/* SKEYID_E */
		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, buf1, HMAC_BUFSIZE);
		SHA1Update(&sha1ctx, st->st_skeyid_a, st->st_skeyid_a_len);
		SHA1Update(&sha1ctx, p, len);
		SHA1Update(&sha1ctx, st->st_icookie, COOKIE_SIZE);
		SHA1Update(&sha1ctx, st->st_rcookie, COOKIE_SIZE);
		c = 2;
		SHA1Update(&sha1ctx, &c, 1);
		SHA1Final(st->st_skeyid_e, &sha1ctx);
		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, buf2, HMAC_BUFSIZE);
		SHA1Update(&sha1ctx, st->st_skeyid_e, st->st_skeyid_e_len);
		SHA1Final(st->st_skeyid_e, &sha1ctx);
		
                /* IV */
		st->st_iv_len = 20;
		st->st_iv = (u_char *) calloc(st->st_iv_len, sizeof(u_char));
		if (st->st_iv == (u_char *) NULL)
		  exit_log("calloc() failed in generate_skeyids()", 0, 0, 0);
		
		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, gi, gilen);
		SHA1Update(&sha1ctx, gr, grlen);
		SHA1Final(st->st_iv, &sha1ctx);
				
		break;
		
	    default:
		exit_log("generate_skeyids(): unknown hash function",
			 0, 0, 0);
	}
    }
    else
    {
	/* XXX Handle 3DES */
    }

    free(gi);
    free(gr);
    free(p);
}

/*
 * Handle HDR;SA from responder (verify proposal), and send back
 * a HDR;KE;Ni.
 */
void
ipsecdoi_handle_i1(int sock, u_char *buffer, int length, struct sockaddr sa,
		   struct state *st)
{
    struct sockaddr_in sa2;
    struct isakmp_hdr *isa, *isa2;
    struct isakmp_sa *isasa;
    struct isakmp_proposal *isap;
    struct isakmp_transform *isat;
    struct isakmp_attribute *isaat;
    struct isakmp_generic *isag;
    u_int32_t situation;
    u_char tmp[LOCALSECRETSIZE];
    u_char *packet;
    MP_INT temp, temp2;
    int i, k;
    
    isa = (struct isakmp_hdr *) buffer;

    if (length <= sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa))
    {
	free(buffer);
	log(0, "too short (%d bytes) packet received from %s, port %d", length,
	    PRINTADDRESS(sa), get_port(sa));
	/* XXX Could send notification back */
	return;
    }
    
    isasa = (struct isakmp_sa *) (buffer + sizeof(struct isakmp_hdr));
    
    if (htons(isasa->isasa_length) + sizeof(struct isakmp_hdr) !=
	length)
    {
	free(buffer);
	log(0, "malformed HDR;SA packet from %s, port %d",
	    PRINTADDRESS(sa), get_port(sa), 0);
	/* XXX Could send notification back */
	return;
    }

    i = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_sa);

    /* Not sure if this should ever happen...i'm just paranoid */
    bcopy(buffer + i, &situation, IPSEC_DOI_SITUATION_LENGTH);
    situation = ntohl(situation);
    if (situation != SIT_IDENTITY_ONLY)
    {
	free(buffer);
	log(0, "bad situation %d from %s, port %d", situation,
	    PRINTADDRESS(sa), get_port(sa));
	/* XXX Could send notification back */
	return;
    }
    
    i += IPSEC_DOI_SITUATION_LENGTH;
    
    isap = (struct isakmp_proposal *) (buffer + i);
    if (ntohs(isap->isap_length) + i != length)
    {
	free(buffer);
	log(0, "bad proposal length from %s, port %d",
	    PRINTADDRESS(sa), get_port(sa), 0);
	/* XXX Could send notification back */
	return;
    }

    i += sizeof(struct isakmp_proposal) + isap->isap_spisize;
    
    /* XXX Verify proposal we received back */

    mpz_init(&(st->st_sec));
    mpz_init(&(st->st_gi));
    st->st_sec_in_use = 1;
    st->st_gi_in_use = 1;
    
    mpz_init(&temp);
    mpz_init(&temp2);
    
    get_rnd_bytes(tmp, LOCALSECRETSIZE);

    for (k = 0; k < LOCALSECRETSIZE; k++)
    {
	mpz_mul_ui(&(st->st_gi), &(st->st_sec), 256);
	mpz_add_ui(&(st->st_sec), &(st->st_gi), tmp[k]);
    }

    mpz_powm(&(st->st_gi), &groupgenerator, &(st->st_sec), &groupmodulo);
    
    i = sizeof(struct isakmp_hdr) + 2 * sizeof(struct isakmp_generic) +
	1 + mpz_sizeinbase(&(st->st_gi), 16) / 2 + DEFAULTNONCESIZE;

    packet = (u_char *) calloc(i, sizeof(u_char));
    if (packet == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_i1()", 0, 0, 0);

    bcopy(isa, packet, sizeof(struct isakmp_hdr));

    isa2 = (struct isakmp_hdr *) packet;
    isa2->isa_np = ISAKMP_NEXT_KE;
    isag = (struct isakmp_generic *) (packet + sizeof(struct isakmp_hdr));
    isag->isag_np = ISAKMP_NEXT_NONCE;
    isa2->isa_length = htonl(i);
    isag->isag_length = htons(mpz_sizeinbase(&(st->st_gi), 16) / 2 + 1 +
			      sizeof(struct isakmp_generic));

    i -= (sizeof(struct isakmp_generic) + DEFAULTNONCESIZE);

#ifdef DEBUG
    log(0, "Local secret: ", 0, 0, 0);
    mpz_out_str(stdout, 16, &(st->st_sec));
    printf("\n");
    log(0, "Public value sent: ", 0, 0, 0);
    mpz_out_str(stdout, 16, &(st->st_gi));
    printf("\nPublic value is %d bytes long.\n", 
	   mpz_sizeinbase(&(st->st_gi), 16) / 2 + 1);
#endif

    mpz_set(&temp, &(st->st_gi));
    
    for (k = i - 1; 
	 k >= sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic);
	 k--)
    {
	packet[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	mpz_set(&temp, &temp2);
    }

#ifdef DEBUG
    log(0, "k = %d, i = %d, isag->isag_length = %d", k, i, 
	ntohs(isag->isag_length));

    if (mpz_cmp_ui(&temp, 0) != 0)
    {
	mpz_out_str(stdout, 16, &temp);
	printf("\n");
	exit_log("ipsecdoi_handle_i1(): remainder not zero", 0, 0, 0);
    }
#endif

    mpz_clear(&temp);
    mpz_clear(&temp2);

    /* Generate Ni */
    isag = (struct isakmp_generic *) (packet + i);
    isag->isag_length = htons(sizeof(struct isakmp_generic) +
			      DEFAULTNONCESIZE);
    get_rnd_bytes(packet + i + sizeof(struct isakmp_generic),
		  DEFAULTNONCESIZE);
    
    /* Copy the Ni nonce in the state object */
    st->st_ni = (u_char *) calloc(ntohs(isag->isag_length), sizeof(u_char));
    if (st->st_ni == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_i1()", 0, 0, 0);
    
    bcopy(isag, st->st_ni, ntohs(isag->isag_length));
    
    /* Copy accepted/received proposal in state object */
    st->st_proposal = (u_char *) calloc(ntohs(isap->isap_length),
					sizeof(u_char));
    if (st->st_proposal == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_i1()", 0, 0, 0);

#ifdef DEBUG
    log(0, "copying %d bytes of proposal into state object",
	ntohs(isap->isap_length), 0, 0);
#endif
    bcopy(isap, st->st_proposal, ntohs(isap->isap_length));
    
    /* Reinsert the state, using the responder cookie we just received */
    delete_state(st);
    bcopy(isa->isa_rcookie, st->st_rcookie, COOKIE_SIZE);
    insert_state(st);
    
    /* Delete previous retransmission event */
    delete_event(st);

    /* Free previous packet and place the new one in the state object */
    free(st->st_packet);
    st->st_packet_len = ntohl(isa2->isa_length);
    st->st_packet = packet;
    st->st_state = OAKLEY_MAIN_I_2;

    /* Get my identity */
    k = sizeof(sa2);
    if (getsockname(sock, (struct sockaddr *)&sa2, &k) < 0)
      exit_log("getsockname() in ipsecdoi_handle_i1()", 0, 0, 0);
    
    if (k <= 0)
      exit_log("getsockname() in ipsecdoi_handle_i1()", 0, 0, 0);
    
    st->st_myidentity = (u_char *) calloc(1, sizeof(struct in_addr));
    if (st->st_myidentity == (u_char *) NULL)
      exit_log("strdup() in ipsecdoi_handle_i1()", 0, 0, 0);
    bcopy(&(sa2.sin_addr), st->st_myidentity, sizeof(struct in_addr));
    st->st_myidentity_len = sizeof(struct in_addr);
    st->st_myidentity_type = ID_IPV4_ADDR;
    
#ifdef DEBUG
    log(0, "my identity is %s", inet_ntoa(sa2.sin_addr), 0, 0);
#endif

    free(buffer);

    if (sendto(sock, packet, st->st_packet_len, 0, &sa, sizeof(sa)) !=
	st->st_packet_len)
      log(1, "sendto() failed in ipsecdoi_handle_i1()", 0, 0, 0);
#ifdef DEBUG
    else
      log(0, "transmitted %d bytes", st->st_packet_len, 0, 0);
#endif

    /* Schedule for retransmission */
    event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st, 0);    
}

/*
 * If the third argument is 1, generate HASH_I, if 0 generate HASH_R.
 * If fourth argument is 1, i'm the Initiator.
 */
u_char *
get_hash(struct state *st, int *length, int hashi, int initiator)
{
    u_char *p, *gi, *gr;
    MD5_CTX md5ctx;
    SHA1_CTX sha1ctx;
    u_char buf1[HMAC_BUFSIZE], buf2[HMAC_BUFSIZE];
    int k, gilen, grlen;
    MP_INT temp, temp2;
    struct identity id;
    struct isakmp_sa *isasa;
    
    mpz_init(&temp);
    mpz_init(&temp2);
    
    gilen = mpz_sizeinbase(&(st->st_gi), 16) / 2 + 1;
    gi = (u_char *) calloc(gilen, sizeof(u_char));
    if (gi == (u_char *) NULL)
      exit_log("calloc() failed in get_hash()", 0, 0, 0);
    
    mpz_set(&temp, &(st->st_gi));

    for (k = gilen - 1 ; k >= 0; k--)
    {
	gi[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	mpz_set(&temp, &temp2);
    }

    grlen = mpz_sizeinbase(&(st->st_gr), 16) / 2 + 1;
    gr = (u_char *) calloc(grlen, sizeof(u_char));
    if (gr == (u_char *) NULL)
      exit_log("calloc() failed in get_hash()", 0, 0, 0);
    
    mpz_set(&temp, &(st->st_gr));

    for (k = grlen - 1; k >= 0; k--)
    {
	gr[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	mpz_set(&temp, &temp2);
    }

    mpz_clear(&temp);
    mpz_clear(&temp2);

    isasa = (struct isakmp_sa *) st->st_sa;
    
    if (st->st_prf == 0)
    {
	/* Prepare the two pads for the HMAC */
	if (st->st_skeyid_len >= HMAC_BUFSIZE)
	{
	    bcopy(st->st_skeyid, buf1, HMAC_BUFSIZE);
	    bcopy(st->st_skeyid, buf2, HMAC_BUFSIZE);
	}
	else
	{
	    for (k = 0; k + st->st_skeyid_len < HMAC_BUFSIZE; 
		 k += st->st_skeyid_len)
	    {
		bcopy(st->st_skeyid, buf1 + k, st->st_skeyid_len);
		bcopy(st->st_skeyid, buf2 + k, st->st_skeyid_len);
	    }

	    bcopy(st->st_skeyid, buf1 + k, HMAC_BUFSIZE - k);
	    bcopy(st->st_skeyid, buf2 + k, HMAC_BUFSIZE - k);
	}
	
	for (k = 0; k < HMAC_BUFSIZE; k++)
	{
	    buf1[k] ^= HMAC_IPAD;
	    buf2[k] ^= HMAC_OPAD;
	}

	switch (st->st_hash)
	{
	    case OAKLEY_MD5:
		*length = 16;
		p = (u_char *) calloc(16, sizeof(u_char));
		if (p == (u_char *) NULL)
		  exit_log("calloc() failed in get_hash()", 0, 0, 0);
	
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, buf1, HMAC_BUFSIZE);

		if (hashi)
		{
		    MD5Update(&md5ctx, gi, gilen);
		    MD5Update(&md5ctx, gr, grlen);
		    MD5Update(&md5ctx, st->st_icookie, COOKIE_SIZE);
		    MD5Update(&md5ctx, st->st_rcookie, COOKIE_SIZE);
		}
		else
		{
		    MD5Update(&md5ctx, gr, grlen);
		    MD5Update(&md5ctx, gi, gilen);
		    MD5Update(&md5ctx, st->st_rcookie, COOKIE_SIZE);
		    MD5Update(&md5ctx, st->st_icookie, COOKIE_SIZE);
		}

		/* SAp */
		MD5Update(&md5ctx, st->st_sa + sizeof(struct isakmp_generic), 
			  ntohs(isasa->isasa_length) - 
			  sizeof(struct isakmp_generic));

		bzero(&id, sizeof(id));

		if (hashi)
		{
		    if (initiator)
		      id.id_type = st->st_myidentity_type;
		    else
		      id.id_type = st->st_peeridentity_type;
		}
		else
		{
		    if (initiator)
		      id.id_type = st->st_peeridentity_type;
		    else
		      id.id_type = st->st_myidentity_type;
		}
		
		MD5Update(&md5ctx, (u_char *) &id, sizeof(id));
		
		if (hashi)
		{
		    if (initiator)
		      MD5Update(&md5ctx, st->st_myidentity, 
				st->st_myidentity_len);
		    else
		      MD5Update(&md5ctx, st->st_peeridentity,
				st->st_peeridentity_len);
		}
		else
		{
		    if (initiator)
		      MD5Update(&md5ctx, st->st_peeridentity,
				st->st_peeridentity_len);
		    else
		      MD5Update(&md5ctx, st->st_myidentity,
				st->st_myidentity_len);
		}
		
		MD5Final(p, &md5ctx);
		MD5Init(&md5ctx);
		MD5Update(&md5ctx, buf2, HMAC_BUFSIZE);
		MD5Update(&md5ctx, p, 16);
		MD5Final(p, &md5ctx);
		
		break;
		
	    case OAKLEY_SHA:
		*length = 20;
		p = (u_char *) calloc(20, sizeof(u_char));
		if (p == (u_char *) NULL)
		  exit_log("calloc() failed in get_hash()", 0, 0, 0);
		
		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, buf1, HMAC_BUFSIZE);

		if (hashi)
		{
		    SHA1Update(&sha1ctx, gi, gilen);
		    SHA1Update(&sha1ctx, gr, grlen);
		    SHA1Update(&sha1ctx, st->st_icookie, COOKIE_SIZE);
		    SHA1Update(&sha1ctx, st->st_rcookie, COOKIE_SIZE);
		}
		else
		{
		    SHA1Update(&sha1ctx, gr, grlen);
		    SHA1Update(&sha1ctx, gi, gilen);
		    SHA1Update(&sha1ctx, st->st_rcookie, COOKIE_SIZE);
		    SHA1Update(&sha1ctx, st->st_icookie, COOKIE_SIZE);
		}
		
		/* SAp */
		SHA1Update(&sha1ctx, st->st_sa + 
			   sizeof(struct isakmp_generic), 
			   ntohs(isasa->isasa_length) - 
			   sizeof(struct isakmp_generic));

		bzero(&id, sizeof(id));

		if (hashi)
		{
		    if (initiator)
		      id.id_type = st->st_myidentity_type;
		    else
		      id.id_type = st->st_peeridentity_type;
		}
		else
		{
		    if (initiator)
		      id.id_type = st->st_peeridentity_type;
		    else
		      id.id_type = st->st_myidentity_type;
		}
		
		SHA1Update(&sha1ctx, (u_char *) &id, sizeof(id));

		if (hashi)
		{
		    if (initiator)
		      SHA1Update(&sha1ctx, st->st_myidentity, 
				 st->st_myidentity_len);
		    else
		      SHA1Update(&sha1ctx, st->st_peeridentity, 
				 st->st_peeridentity_len);
		}
		else
		{
		    if (initiator)
		      SHA1Update(&sha1ctx, st->st_peeridentity, 
				 st->st_peeridentity_len);
		    else
		      SHA1Update(&sha1ctx, st->st_myidentity, 
				 st->st_myidentity_len);
		}
		
		SHA1Final(p, &sha1ctx);
		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, buf2, HMAC_BUFSIZE);
		SHA1Update(&sha1ctx, p, 20);
		SHA1Final(p, &sha1ctx);
		break;
		
	    default:
		exit_log("get_hash(): unknown hash algorithm %d", st->st_hash,
			 0, 0);
	}
    }
    else
    {
	/* XXX Handle 3DES */
    }

    free(gi);
    free(gr);
    return p;
}

/*
 * Handle HDR;KE;Nr from responder. Send a HDR;IDii;HASH_I back.
 */
void
ipsecdoi_handle_i2(int sock, u_char *buffer, int length, struct sockaddr sa,
		   struct state *st)
{
    struct isakmp_hdr *isa;
    struct isakmp_generic *isag;
    MP_INT temp, temp2;
    int i, k, blocksize;
    u_char *hash;
    struct identity *ide;
    u_int32_t des_cbc_keys[16][2];
    
    isa = (struct isakmp_hdr *) buffer;
    if (length <= sizeof(struct isakmp_hdr) + 
	2 * sizeof(struct isakmp_generic))
    {
	free(buffer);
	log(0, "too short HDR;KE;Nr packet (%d bytes) from %s, port %d", 
	    length, PRINTADDRESS(sa), get_port(sa));
	/* XXX Could send notification back */
	return;
    }

    isag = (struct isakmp_generic *) (buffer + sizeof(struct isakmp_hdr));
    i = htons(isag->isag_length);
    if (i + sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic) > 
	length)
    {
	free(buffer);
	log(0, "malformed packet from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa), 0);
	/* XXX Could send notification back */
	return;
    }

    if (i - sizeof(struct isakmp_generic) < MINIMUM_PUBLIC_VALUE_SIZE)
    {
	free(buffer);
	log(0, "too short public size (%d) sent from %s, port %d",
	    i - sizeof(struct isakmp_generic), PRINTADDRESS(sa), 
	    get_port(sa));
	/* XXX Could send notification back */
	return;
    }

    if (isag->isag_np != ISAKMP_NEXT_NONCE)
    {
	free(buffer);
	log(0, "unexpected header %d in HDR;KE;Nr packet from %s, port %d",
	    isag->isag_np, PRINTADDRESS(sa), get_port(sa));
	/* XXX Could send notification back */
	return;
    }
    
    mpz_init(&temp);
    mpz_init(&temp2);
    
    i += sizeof(struct isakmp_hdr);
    for (k = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic);
	 k < i; k++)
    {
	mpz_mul_ui(&temp2, &temp, 256);
	mpz_add_ui(&temp, &temp2, buffer[k]);
    }

    /* Check the Nr header */
    isag = (struct isakmp_generic *) (buffer + i);
    if (ntohs(isag->isag_length) + i != length)
    {
	free(buffer);
	log(0, "malformed packet from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa), 0);
	/* XXX Could send notification back */
	return;
    }    

    mpz_init(&(st->st_shared));
    mpz_init(&(st->st_gr));
    
    st->st_shared_in_use = 1;
    st->st_gr_in_use = 1;
    
    mpz_set(&(st->st_gr), &temp);
    
    mpz_powm(&(st->st_shared), &(st->st_gr), &(st->st_sec), &groupmodulo);
    
#ifdef DEBUG
    log(0, "public value received: ", 0, 0, 0);
    mpz_out_str(stdout, 16, &(st->st_gr));
    printf("\n");
    log(0, "shared secret: ", 0, 0, 0);
    mpz_out_str(stdout, 16, &(st->st_shared));
    printf("\n");
#endif

    /* Save Nr */
    st->st_nr = (u_char *) calloc(ntohs(isag->isag_length), sizeof(u_char));
    if (st->st_nr == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_i2()", 0, 0, 0);

    bcopy(isag, st->st_nr, ntohs(isag->isag_length));
    
    /* Delete old retransmit event */
    delete_event(st);
    
    /* Free old packet */
    free(st->st_packet);

    st->st_state = OAKLEY_MAIN_I_3;
    
    generate_skeyids(st);

#ifdef DEBUG
    log(0, "Skeyid(%d): ", st->st_skeyid_len, 0, 0);
    for (k = 0; k < st->st_skeyid_len; k++)
      printf("%02x ", st->st_skeyid[k]);
    printf("\n");
    log(0, "Skeyid_d(%d): ", st->st_skeyid_d_len, 0, 0);
    for (k = 0; k < st->st_skeyid_d_len; k++)
      printf("%02x ", st->st_skeyid_d[k]);
    printf("\n");
    log(0, "Skeyid_a(%d): ", st->st_skeyid_a_len, 0, 0);
    for (k = 0; k < st->st_skeyid_a_len; k++)
      printf("%02x ", st->st_skeyid_a[k]);
    printf("\n");
    log(0, "Skeyid_e(%d): ", st->st_skeyid_e_len, 0, 0);
    for (k = 0; k < st->st_skeyid_e_len; k++)
      printf("%02x ", st->st_skeyid_e[k]);
    printf("\n");    
    log(0, "IV(%d): ", st->st_iv_len, 0, 0);
    for (k = 0; k < st->st_iv_len; k++)
      printf("%02x ", st->st_iv[k]);
    printf("\n");
#endif

    /* Generate HASH_I */
    hash = get_hash(st, &i, 1, 1);
    
    st->st_packet_len = sizeof(struct isakmp_hdr) + 
			2 * sizeof(struct isakmp_generic) + i +
			st->st_myidentity_len + sizeof(struct identity);
    /* Padding */
    switch (st->st_enc)
    {
	case OAKLEY_DES_CBC:
	case OAKLEY_3DES_CBC:
	    blocksize = DES_CBC_BLOCK_SIZE;
	    break;
	    
	default:
	    exit_log("unknown/unsupported encryption algorithm %d specified",
		     st->st_enc, 0, 0);
    }
    
    k = (blocksize - 
	 (st->st_packet_len - sizeof(struct isakmp_hdr)) % blocksize);
    st->st_packet_len += k;
    
    st->st_packet = (u_char *) calloc(st->st_packet_len, sizeof(u_char));
    if (st->st_packet == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_i2()", 0, 0, 0);

    /* Fix padding */
    st->st_packet[st->st_packet_len - 1] = k;
    
#ifdef DEBUG
    log(0, "padding is %d bytes", st->st_packet[st->st_packet_len - 1],
	0, 0);
#endif

    bcopy(isa, st->st_packet, sizeof(struct isakmp_hdr));
    isa = (struct isakmp_hdr *) st->st_packet;
    isa->isa_length = htonl(st->st_packet_len);
    isa->isa_np = ISAKMP_NEXT_ID;
    
    /* IDii */
    isag = (struct isakmp_generic *) (st->st_packet + 
				      sizeof(struct isakmp_hdr));
    isag->isag_length = htons(sizeof(struct isakmp_generic) +
			      st->st_myidentity_len + sizeof(struct identity));
    isag->isag_np = ISAKMP_NEXT_HASH;

    ide = (struct identity *) (st->st_packet + sizeof(struct isakmp_hdr) +
			       sizeof(struct isakmp_generic));

    ide->id_type = st->st_myidentity_type;

    bcopy(st->st_myidentity,
	  st->st_packet + sizeof(struct isakmp_hdr) +
	  sizeof(struct isakmp_generic) + sizeof(struct identity), 
	  st->st_myidentity_len);
    
    /* HASH_I */
    isag = (struct isakmp_generic *) (st->st_packet +
				      sizeof(struct isakmp_hdr) +
				      ntohs(isag->isag_length));
    isag->isag_length = htons(i + sizeof(struct isakmp_generic));
    
    bcopy(hash, st->st_packet + st->st_packet_len - 
	  (i + st->st_packet[st->st_packet_len - 1]) , i);

#ifdef DEBUG
    log(0, "HASH_I sent: ");
    for (k = 0; k < i; k++)
      printf("%02x ", hash[k]);
    printf("\n");
#endif

    free(hash);

    isa->isa_flags |= ISAKMP_FLAG_ENCRYPTION;
    
    switch (st->st_enc)
    {
	case OAKLEY_DES_CBC:
	    /* XXX Do weak key detection */
	    des_set_key(st->st_skeyid_e, des_cbc_keys);
	    des_cbc_encrypt(st->st_packet + sizeof(struct isakmp_hdr),
			    st->st_packet + sizeof(struct isakmp_hdr),
			    st->st_packet_len - sizeof(struct isakmp_hdr),
			    des_cbc_keys, st->st_iv, 1);
#ifdef DEBUG
	    log(0, "encrypting using DES CBC", 0, 0, 0);
#endif
	    
	    /* Update IV */
	    if ((st->st_iv != (u_char *) NULL) &&
		(st->st_iv_len < DES_CBC_BLOCK_SIZE))
	    {
		free(st->st_iv);
		st->st_iv = (u_char *) NULL;
	    }

	    if (st->st_iv == (u_char *) NULL)
	    {
		st->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					      sizeof(u_char));
		if (st->st_iv == (u_char *) NULL)
		  exit_log("calloc() failed in ipsecdoi_handle_i2()",
			   0, 0, 0);
	    }

	    st->st_iv_len = DES_CBC_BLOCK_SIZE;
	    bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE,
		  st->st_iv, DES_CBC_BLOCK_SIZE);

	    break;
	
	    /* XXX Support more */
	default:
	    exit_log("unsupport encryption algorithm %d", st->st_enc,
		     0, 0);
    }

#ifdef DEBUG
    log(0, "new IV: ", 0, 0, 0);
    for (k = 0; k < st->st_iv_len; k++)
      printf("%02x ", st->st_iv[k]);
    printf("\n");
#endif

    if (sendto(sock, st->st_packet, st->st_packet_len, 0, &sa, sizeof(sa)) !=
	st->st_packet_len)
      log(1, "sendto() failed in ipsecdoi_handle_i2()", 0, 0, 0);
#ifdef DEBUG
    else
      log(0, "transmitted %d bytes", st->st_packet_len, 0, 0);
#endif

    event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st, 0);
}

/*
 * Handle HDR;IDir;HASH_R from responder.
 */
void
ipsecdoi_handle_i3(int sock, u_char *buffer, int length, struct sockaddr sa,
		   struct state *st)
{
    struct isakmp_hdr *isa;
    struct isakmp_generic *isag;
    int i, k;
    struct identity *id;
    u_char *hash;
    struct sockaddr_in sin;
    
    isa = (struct isakmp_hdr *) buffer;
    if (length <= sizeof(struct isakmp_hdr) + 
	2 * sizeof(struct isakmp_generic) + sizeof(struct identity))
    {
	log(0, "too short packet from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa), 0);
	free(buffer);
	/* XXX Could send notification back */
	return;
    }

    if (isa->isa_np != ISAKMP_NEXT_ID)
    {
	log(0, "bad packet received from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa));
	free(buffer);
	/* XXX Could send notification back */
	return;
    }

    i = sizeof(struct isakmp_hdr);

    /* IDir */
    isag = (struct isakmp_generic *) (buffer + i);
    if (ntohs(isag->isag_length) + i >= length)
    {
	log(0, "too short packet from %s, port %d", PRINTADDRESS(sa), 
	    get_port(sa), 0);
	free(buffer);
	/* XXX Could send notification back */
	return;
    }

    i += sizeof(struct isakmp_generic);
    if (isag->isag_np != ISAKMP_NEXT_HASH)
    {
	log(0, "bad packet received from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa));
	free(buffer);
	/* XXX Could send notification back */
	return;
    }
    
    id = (struct identity *) (buffer + i);

    /* XXX Check for valid ID types ? */

    if ((id->id_port != 0) || (id->id_protoid))
    {
	log(0, "port or protocol id in ID not zero (%d/%d)", id->id_port,
	    id->id_protoid, 0);
	free(buffer);
	/* XXX Could send notification back */
	return;
    }
    
    i += sizeof(struct identity);
    st->st_peeridentity_type = id->id_type;
    st->st_peeridentity_len = ntohs(isag->isag_length) -
			      (sizeof(struct isakmp_generic) + 
			       sizeof(struct identity));
    st->st_peeridentity = (u_char *) calloc(st->st_peeridentity_len,
					    sizeof(u_char));
    if (st->st_peeridentity == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_r2()", 0, 0, 0);

    bcopy(buffer + i, st->st_peeridentity, st->st_peeridentity_len);
    i += st->st_peeridentity_len;

#ifdef DEBUG
    log(0, "IDir type is %d, length %d", id->id_type, 
	st->st_peeridentity_len, 0);
    if (id->id_type == ID_IPV4_ADDR)
    {
	bcopy(st->st_peeridentity, &(sin.sin_addr), sizeof(sin));
	log(0, "IDir is %s", inet_ntoa(sin.sin_addr), 0, 0);
    }
#endif

    /* HASH_R */
    isag = (struct isakmp_generic *) (buffer + i);
    
    if ((isag->isag_np != ISAKMP_NEXT_NONE) ||
	(ntohs(isag->isag_length) + i != length))
    {
	log(0, "bad packet from %s, port %d", PRINTADDRESS(sa), 
	    get_port(sa), 0);
	free(buffer);
	free(st->st_peeridentity);	/* Rollback */
	/* XXX Could send notification back */
	return;
    }
    
    i += sizeof(struct isakmp_generic);
    
    /* Compute HASH_R */
    hash = get_hash(st, &k, 0, 1);
    
    if (length - i != k)
    {
	log(0, "computed and received HASH_I lengths don't match (%d/%d)",
	    length - i, k, 0);
	free(buffer);
	free(hash);
	free(st->st_peeridentity);
	/* XXX Could send notification back */
	return;
    }
    
#ifdef DEBUG
    log(0, "computed HASH_R: ", 0, 0, 0);
    for (k = 0; k < length - i; k++)
      printf("%02x ", hash[k]);
    printf("\n");
    log(0, "received HASH_R: ", 0, 0, 0);
    for (k = 0; k < length - i; k++)
      printf("%02x ", buffer[i + k]);
    printf("\n");
#endif

    if (bcmp(hash, buffer + i, length - i))
    {
	log(0, "received and computed HASH_I values don't match", 0, 0, 0);
	free(buffer);
	free(hash);
	free(st->st_peeridentity);
	/* XXX Could send notification back */
	return;
    }
#ifdef DEBUG
    else
      log (0, "HASH_R verified", 0, 0, 0);
#endif

    /* Free old packet */
    free(st->st_packet);

    /* Free computed HASH_I */
    free(hash);
    
    /* Free received packet */
    free(buffer);
    
    /* Delete pending event */
    delete_event(st);
    
    /* Advance state */
    st->st_state = OAKLEY_MAIN_I_4;

    if (st->st_goal != 0)
    {
#ifdef DEBUG
    log(0, "Doing Quick Mode with %s, port %d, goal %d",
	PRINTADDRESS(sa), get_port(sa), st->st_goal);
#endif
	/* XXX Continue with Quick Mode */
    }
    else
    {
	/* Schedule for SA expiration */
	if (st->st_expire != 0)
	  event_schedule(EVENT_SA_EXPIRE, st->st_expire, st, 0);
	else
	  event_schedule(EVENT_SA_EXPIRE, DEFAULT_SA_EXPIRE, st, 0);
    }
}

/*
 * Handle HDR;KE;Ni from Initiator. Send a HDR;KE;Nr back.
 */
void
ipsecdoi_handle_r1(int sock, u_char *buffer, int length, struct sockaddr sa,
		   struct state *st)
{
    struct isakmp_hdr *isa;
    struct isakmp_generic *isag;
    u_char *packet;
    int i, k;
    u_char tmp[LOCALSECRETSIZE];
    MP_INT temp, temp2;
    
    isa = (struct isakmp_hdr *) buffer;

    if (length <= sizeof(struct isakmp_hdr) + 
	2 * sizeof(struct isakmp_generic))
    {
	free(buffer);
	log(0, "too short HDR;KE;Ni packet (%d bytes) from %s, port %d", 
	    length, PRINTADDRESS(sa), get_port(sa));
	/* XXX Could send notification back */
	return;
    }
    
    isag = (struct isakmp_generic *) (buffer + sizeof(struct isakmp_hdr));
    i = htons(isag->isag_length);
    if (i + sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic) > 
	length)
    {
	free(buffer);
	log(0, "malformed packet from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa), 0);
	/* XXX Could send notification back */
	return;
    }

    if (i - sizeof(struct isakmp_generic) < MINIMUM_PUBLIC_VALUE_SIZE)
    {
	free(buffer);
	log(0, "too short public size (%d) sent from %s, port %d",
	    i - sizeof(struct isakmp_generic), PRINTADDRESS(sa), 
	    get_port(sa));
	/* XXX Could send notification back */
	return;
    }

    if (isag->isag_np != ISAKMP_NEXT_NONCE)
    {
	free(buffer);
	log(0, "unexpected header %d in HDR;KE;Ni packet from %s, port %d",
	    isag->isag_np, PRINTADDRESS(sa), get_port(sa));
	/* XXX Could send notification back */
	return;
    }
    
    mpz_init(&temp);
    mpz_init(&temp2);
    
    i += sizeof(struct isakmp_hdr);
    for (k = sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic);
	 k < i; k++)
    {
	mpz_mul_ui(&temp2, &temp, 256);
	mpz_add_ui(&temp, &temp2, buffer[k]);
    }

    /* Check the Ni header */
    isag = (struct isakmp_generic *) (buffer + i);
    if (ntohs(isag->isag_length) + i != length)
    {
	free(buffer);
	log(0, "malformed packet from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa), 0);
	/* XXX Could send notification back */
	return;
    }
    
    mpz_init(&(st->st_gi));
    mpz_init(&(st->st_gr));
    mpz_init(&(st->st_sec));
    mpz_init(&(st->st_shared));
    
    st->st_gi_in_use = 1;
    st->st_gr_in_use = 1;
    st->st_sec_in_use = 1;
    st->st_shared_in_use = 1;
    
    mpz_set(&(st->st_gi), &temp);
    
#ifdef DEBUG
    log(0, "public value received:", 0, 0, 0);
    mpz_out_str(stdout, 16, &temp);
    printf("\n");
#endif

    get_rnd_bytes(tmp, LOCALSECRETSIZE);

    mpz_set_ui(&temp, 0);
    
    for (k = 0; k < LOCALSECRETSIZE; k++)
    {
	mpz_mul_ui(&temp2, &temp, 256);
	mpz_add_ui(&temp, &temp2, tmp[k]);
    }

    mpz_set(&(st->st_sec), &temp);

    mpz_powm(&(st->st_gr), &groupgenerator, &(st->st_sec), &groupmodulo);

    mpz_powm(&(st->st_shared), &(st->st_gi), &(st->st_sec), &groupmodulo);
    
#ifdef DEBUG    
    log(0, "our secret value: ", 0, 0, 0);
    mpz_out_str(stdout, 16, &(st->st_sec));
    printf("\n");
    log(0, "our public value: ", 0, 0, 0);
    mpz_out_str(stdout, 16, &(st->st_gr));
    printf("\n");
    log(0, "shared secret: ", 0, 0, 0);
    mpz_out_str(stdout, 16, &(st->st_shared));
    printf("\n");
#endif

    /* Save Ni */
    st->st_ni = (u_char *) calloc(ntohs(isag->isag_length), sizeof(u_char));
    if (st->st_ni == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_r1()", 0, 0, 0);

    bcopy(isag, st->st_ni, ntohs(isag->isag_length));
    
    /* Create Nr */
    st->st_nr = (u_char *) calloc(DEFAULTNONCESIZE + 
				  sizeof(struct isakmp_generic), 
				  sizeof(u_char));
    if (st->st_nr == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_r1()", 0, 0, 0);
    
    isag = (struct isakmp_generic *) st->st_nr;
    isag->isag_length = ntohs(DEFAULTNONCESIZE + 
			      sizeof(struct isakmp_generic));
    
    get_rnd_bytes(st->st_nr + sizeof(struct isakmp_generic), DEFAULTNONCESIZE);

    i = sizeof(struct isakmp_hdr) + 2 * sizeof(struct isakmp_generic) + 1 +
	mpz_sizeinbase(&(st->st_gr), 16) / 2 + DEFAULTNONCESIZE;
    
    packet = (u_char *) calloc(i, sizeof(u_char));
    if (packet == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_r1()", 0, 0, 0);
    
    bcopy(isa, packet, sizeof(struct isakmp_hdr));
    
    isa = (struct isakmp_hdr *) packet;
    isag = (struct isakmp_generic *) (packet + sizeof(struct isakmp_hdr));
    isag->isag_np = ISAKMP_NEXT_NONCE;
    isa->isa_length = htonl(i);
    isag->isag_length = htons(mpz_sizeinbase(&(st->st_gr), 16) / 2 + 1 +
			      sizeof(struct isakmp_generic));
    
    mpz_set(&temp, &(st->st_gr));

    i -= (sizeof(struct isakmp_generic) + DEFAULTNONCESIZE);
    
    for (k = i - 1;
	 k >= sizeof(struct isakmp_hdr) + sizeof(struct isakmp_generic);
	 k--)
    {
	packet[k] = mpz_mdivmod_ui(&temp2, NULL, &temp, 256);
	mpz_set(&temp, &temp2);
    }
    
    mpz_clear(&temp);
    mpz_clear(&temp2);
    
    isag = (struct isakmp_generic *) st->st_nr;
    bcopy(st->st_nr, packet + i, ntohs(isag->isag_length));
    
    /* Advance state */
    st->st_state = OAKLEY_MAIN_R_2;

    /* Delete previous CLEANUP event */
    delete_event(st);

    /* Free received packet */
    free(buffer);

    /* No need to free old packet, since we didn't keep it */

    st->st_packet = packet;
    st->st_packet_len = i + DEFAULTNONCESIZE + sizeof(struct isakmp_generic);

    if (sendto(sock, packet, st->st_packet_len, 0, &sa, sizeof(sa)) !=
	st->st_packet_len)
      log(1, "sendto() failed in ipsecdoi_handle_r1()", 0, 0, 0);
#ifdef DEBUG
    else
      log(0, "transmitted %d bytes", st->st_packet_len, 0, 0);
#endif

    generate_skeyids(st);
    
#ifdef DEBUG
    log(0, "Skeyid(%d): ", st->st_skeyid_len, 0, 0);
    for (k = 0; k < st->st_skeyid_len; k++)
      printf("%02x ", st->st_skeyid[k]);
    printf("\n");
    log(0, "Skeyid_d(%d): ", st->st_skeyid_d_len, 0, 0);
    for (k = 0; k < st->st_skeyid_d_len; k++)
      printf("%02x ", st->st_skeyid_d[k]);
    printf("\n");
    log(0, "Skeyid_a(%d): ", st->st_skeyid_a_len, 0, 0);
    for (k = 0; k < st->st_skeyid_a_len; k++)
      printf("%02x ", st->st_skeyid_a[k]);
    printf("\n");
    log(0, "Skeyid_e(%d): ", st->st_skeyid_e_len, 0, 0);
    for (k = 0; k < st->st_skeyid_e_len; k++)
      printf("%02x ", st->st_skeyid_e[k]);
    printf("\n");   
    log(0, "IV(%d): ", st->st_iv_len, 0, 0);
    for (k = 0; k < st->st_iv_len; k++)
      printf("%02x ", st->st_iv[k]);
    printf("\n");
#endif

    event_schedule(EVENT_CLEANUP, EVENT_CLEANUP_DELAY, st, 0);    
}

/*
 * Handle HDR;IDii;HASH_I from initiator. Send a HDR;IDir;HASH_R back.
 */
void
ipsecdoi_handle_r2(int sock, u_char *buffer, int length, struct sockaddr sa,
		   struct state *st)
{
    struct isakmp_hdr *isa;
    struct isakmp_generic *isag;
    struct identity *id;
    u_char *hash;
    int i, k, blocksize;
    struct sockaddr_in sin;
    u_int32_t des_cbc_keys[16][2];
    
    isa = (struct isakmp_hdr *) buffer;
    if (length <= sizeof(struct isakmp_hdr) +
	2 * sizeof(struct isakmp_generic) + sizeof(struct identity))
    {
	log(0, "too short packet from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa), 0);
	free(buffer);
	/* XXX Could send notification back */
	return;
    }

    if (isa->isa_np != ISAKMP_NEXT_ID)
    {
	log(0, "bad packet received from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa));
	free(buffer);
	/* XXX Could send notification back */
	return;
    }

    i = sizeof(struct isakmp_hdr);

    /* IDii */
    isag = (struct isakmp_generic *) (buffer + i);
    if (ntohs(isag->isag_length) + i >= length)
    {
	log(0, "too short packet from %s, port %d", PRINTADDRESS(sa), 
	    get_port(sa), 0);
	free(buffer);
	/* XXX Could send notification back */
	return;
    }

    i += sizeof(struct isakmp_generic);
    if (isag->isag_np != ISAKMP_NEXT_HASH)
    {
	log(0, "bad packet received from %s, port %d", PRINTADDRESS(sa),
	    get_port(sa));
	free(buffer);
	/* XXX Could send notification back */
	return;
    }
    
    id = (struct identity *) (buffer + i);

    /* XXX Check for valid ID types ? */

    if ((id->id_port != 0) || (id->id_protoid))
    {
	log(0, "port or protocol id in ID not zero (%d/%d)", id->id_port,
	    id->id_protoid, 0);
	free(buffer);
	/* XXX Could send notification back */
	return;
    }
    
    i += sizeof(struct identity);
    st->st_peeridentity_type = id->id_type;
    st->st_peeridentity_len = ntohs(isag->isag_length) -
			      (sizeof(struct isakmp_generic) + 
			       sizeof(struct identity));
    st->st_peeridentity = (u_char *) calloc(st->st_peeridentity_len,
					    sizeof(u_char));
    if (st->st_peeridentity == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_r2()", 0, 0, 0);

    bcopy(buffer + i, st->st_peeridentity, st->st_peeridentity_len);
    i += st->st_peeridentity_len;

#ifdef DEBUG
    log(0, "IDii type is %d, length %d", id->id_type, 
	st->st_peeridentity_len, 0);
    if (id->id_type == ID_IPV4_ADDR)
    {
	bcopy(st->st_peeridentity, &(sin.sin_addr), sizeof(sin));
	log(0, "IDii is %s", inet_ntoa(sin.sin_addr), 0, 0);
    }
#endif

    /* HASH_I */
    isag = (struct isakmp_generic *) (buffer + i);
    
    if ((isag->isag_np != ISAKMP_NEXT_NONE) ||
	(ntohs(isag->isag_length) + i != length))
    {
	log(0, "bad packet from %s, port %d", PRINTADDRESS(sa), 
	    get_port(sa), 0);
	free(buffer);
	free(st->st_peeridentity);	/* Rollback */
	/* XXX Could send notification back */
	return;
    }
    
    i += sizeof(struct isakmp_generic);
    
    /* Compute HASH_I */
    hash = get_hash(st, &k, 1, 0);
    
    if (length - i != k)
    {
	log(0, "computed and received HASH_I lengths don't match (%d/%d)",
	    length - i, k, 0);
	free(buffer);
	free(hash);
	free(st->st_peeridentity);
	/* XXX Could send notification back */
	return;
    }
    
#ifdef DEBUG
    log(0, "computed HASH_I: ", 0, 0, 0);
    for (k = 0; k < length - i; k++)
      printf("%02x ", hash[k]);
    printf("\n");
    log(0, "received HASH_I: ", 0, 0, 0);
    for (k = 0; k < length - i; k++)
      printf("%02x ", buffer[i + k]);
    printf("\n");
#endif

    if (bcmp(hash, buffer + i, length - i))
    {
	log(0, "received and computed HASH_I values don't match", 0, 0, 0);
	free(buffer);
	free(hash);
	free(st->st_peeridentity);
	/* XXX Could send notification back */
	return;
    }
#ifdef DEBUG
    else
      log (0, "HASH_I verified", 0, 0, 0);
#endif

    /* Free old packet */
    free(st->st_packet);

    /* Free computed HASH_I */
    free(hash);
    
    st->st_packet_len = sizeof(struct isakmp_hdr) + 
			2 * sizeof(struct isakmp_generic) + k +
			st->st_myidentity_len + sizeof(struct identity);
    
    /* Padding */
    switch (st->st_enc)
    {
	case OAKLEY_DES_CBC:
	case OAKLEY_3DES_CBC:
	    blocksize = DES_CBC_BLOCK_SIZE;
	    break;
	    
	default:
	    exit_log("unknown/unsupported encryption algorithm %d specified",
		     st->st_enc, 0, 0);
    }
    
    k = (blocksize - 
	 (st->st_packet_len - sizeof(struct isakmp_hdr)) % blocksize);

    st->st_packet_len += k;
    
#ifdef DEBUG
    log(0, "padding is %d bytes", k, 0, 0);
#endif

    st->st_packet = (u_char *) calloc(st->st_packet_len, sizeof(u_char));
    if (st->st_packet == (u_char *) NULL)
      exit_log("calloc() failed in ipsecdoi_handle_r2()", 0, 0, 0);

    st->st_packet[st->st_packet_len - 1] = k;
    
    bcopy(buffer, st->st_packet, sizeof(struct isakmp_hdr));
    isa = (struct isakmp_hdr *) st->st_packet;
    isa->isa_length = htonl(st->st_packet_len);

    /* IDir */
    i = sizeof(struct isakmp_hdr);
    isag = (struct isakmp_generic *) (st->st_packet + i);
    i += sizeof(struct isakmp_generic);
    id = (struct identity *) (st->st_packet + i);
    i += sizeof(struct identity);
    
    isag->isag_np = ISAKMP_NEXT_HASH;
    isag->isag_length = htons(sizeof(struct isakmp_generic) +
			      sizeof(struct identity) +
			      st->st_myidentity_len);
    
    id->id_type = st->st_myidentity_type;
    bcopy(st->st_myidentity, st->st_packet + i, st->st_myidentity_len);
    i += st->st_myidentity_len;
    
    /* Get HASH_R */
    hash = get_hash(st, &k, 0, 0);

    /* HASH_R */
    isag = (struct isakmp_generic *) (st->st_packet + i);
    i += sizeof(struct isakmp_generic);
    isag->isag_length = htons(sizeof(struct isakmp_generic) + k);
    bcopy(hash, st->st_packet + i, k);

#ifdef DEBUG
    log(0, "sending HASH_R: ", 0, 0, 0);
    for (i = 0; i < k; i++)
      printf("%02x ", hash[i]);
    printf("\n");
#endif

    /* Free computed HASH_R */
    free(hash);

    /* Advance state */
    st->st_state = OAKLEY_MAIN_R_3;
    
    switch (st->st_enc)
    {
	case OAKLEY_DES_CBC:
	    /* XXX Do weak key detection */
	    des_set_key(st->st_skeyid_e, des_cbc_keys);
	    des_cbc_encrypt(st->st_packet + sizeof(struct isakmp_hdr),
			    st->st_packet + sizeof(struct isakmp_hdr),
			    st->st_packet_len - sizeof(struct isakmp_hdr),
			    des_cbc_keys, st->st_iv, 1);

	    
#ifdef DEBUG
	    log(0, "encrypting using DES CBC", 0, 0, 0);
#endif

	    /* Last block of Phase 1, kept for Phase 2 IV generation */
	    st->st_lastblock = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
						 sizeof(u_char));
	    if (st->st_lastblock == (u_char *) NULL)
	      exit_log("calloc() failed in ipsecdoi_handle_r2()", 0, 0, 0);
	    
	    
	    bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE,
		  st->st_lastblock, DES_CBC_BLOCK_SIZE);
	    st->st_lastblock_len = DES_CBC_BLOCK_SIZE;

	    /* Keep last block as IV */
	    if ((st->st_iv != (u_char *) NULL) &&
		(st->st_iv_len < DES_CBC_BLOCK_SIZE))
	    {
		free(st->st_iv);
		st->st_iv = (u_char *) NULL;
	    }
	    
	    if (st->st_iv == (u_char *) NULL)
	    {
		st->st_iv = (u_char *) calloc(DES_CBC_BLOCK_SIZE,
					      sizeof(u_char));
		if (st->st_iv == (u_char *) NULL)
		  exit_log("calloc() failed in ipsecdoi_handle_r2()",
			   0, 0, 0);
	    }
	    
	    bcopy(st->st_packet + st->st_packet_len - DES_CBC_BLOCK_SIZE , 
		  st->st_iv, DES_CBC_BLOCK_SIZE);
	    st->st_iv_len = DES_CBC_BLOCK_SIZE;
	    
	    break;
	
	    /* XXX Support more */
	default:
	    exit_log("unsupport encryption algorithm %d", st->st_enc,
		     0, 0);
    }

#ifdef DEBUG
    log(0, "last encrypted block of Phase 1: ", 0, 0, 0);
    for (k = 0; k < st->st_lastblock_len; k++)
      printf("%02x ", st->st_lastblock[k]);
    printf("\n");
    log(0, "new IV: ", 0, 0, 0);
    for (k = 0; k < st->st_iv_len; k++)
      printf("%02x ", st->st_iv[k]);
    printf("\n");
#endif

    /* Get rid of received packet */
    free(buffer);
    
    /* Delete pending CLEANUP event */
    delete_event(st);
    
    if (sendto(sock, st->st_packet, st->st_packet_len, 0, &sa, sizeof(sa))
	!= st->st_packet_len)
      log(1, "sendto() failed in ipsecdoi_handle_r2()", 0, 0, 0);
#ifdef DEBUG
    else
      log(0, "transmitted %d bytes", st->st_packet_len, 0, 0);
#endif

    /* Expiration of SA */
    if (st->st_expire != 0)
      event_schedule(EVENT_SA_EXPIRE, st->st_expire, st, 0);
    else
      event_schedule(EVENT_SA_EXPIRE, DEFAULT_SA_EXPIRE, st, 0);
}
