/**
   SSLHandshakeServer.java
 
   Copyright (C) 1999, Claymore Systems, Inc.
   All Rights Reserved.

   ekr@rtfm.com  Mon May 24 21:23:27 1999

   This package is a SSLv3/TLS implementation written by Eric Rescorla
   <ekr@rtfm.com> and licensed by Claymore Systems, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
   3. All advertising materials mentioning features or use of this software
      must display the following acknowledgement:
      This product includes software developed by Claymore Systems, Inc.
   4. Neither the name of Claymore Systems, Inc. nor the name of Eric
      Rescorla may be used to endorse or promote products derived from this
      software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.

   $Id: SSLHandshakeServer.java,v 1.5 1999/06/22 06:24:54 ekr Exp $

*/

package COM.claymoresystems.ptls;
import COM.claymoresystems.sslg.*;
import COM.claymoresystems.cert.*;
import COM.claymoresystems.util.*;
import java.io.*;
import java.util.Vector;
import java.math.BigInteger;
import java.security.PrivateKey;

class SSLHandshakeServer extends SSLHandshake
{
     public final int SSL_HS_WAIT_FOR_CLIENT_HELLO	=1;
     public final int SSL_HS_WAIT_FOR_CERTIFICATE       = 2;
     public final int SSL_HS_WAIT_FOR_CLIENT_KEY_EXCHANGE=3;
     public final int SSL_HS_WAIT_FOR_CERTIFICATE_VERIFY= 4;
     public final int SSL_HS_SEND_HELLO_REQUEST = 5;
    
     boolean resume=false;
     boolean clientAuth=false;
     SSLSessionData possibleResume=null;
     Vector offered_cipher_suites;
     Vector offered_compression_methods;

     public SSLHandshakeServer(SSLConn c){
       super(c);
       
       stateChange((_conn.write_cipher_state==null)?
	 SSL_HS_WAIT_FOR_CLIENT_HELLO:
	 SSL_HS_SEND_HELLO_REQUEST);
       
       client=false;
     }

     protected void filterCipherSuites(PrivateKey key,SSLPolicyInt policy){
       String alg=key.getAlgorithm();
       cipher_suites=new Vector();
       short[] policySuites=_conn.getPolicy().getCipherSuites();
       
       for(int i=0;i<policySuites.length;i++){
	 SSLCipherSuite cs=SSLCipherSuite.findCipherSuite(policySuites[i]);

	 if(cs==null){
	   SSLDebug.debug(SSLDebug.DEBUG_INIT,
	     "Rejecting unrecognized cipher suite" + policySuites[i]);
	   continue;
	 }

	 if(!cs.getSignatureAlgBase().equals(alg)){
	   SSLDebug.debug(SSLDebug.DEBUG_INIT,
	     "Rejecting cipher suite: " + cs.getName() +
	     " -- incompatible with signature algorithm "+ alg);
	   continue;
	 }

	 SSLDebug.debug(SSLDebug.DEBUG_INIT,"Accepting cipher suite: " +
	   cs.getName());
	 cipher_suites.addElement((Object)cs);
       }
     }
     
     public void handshakeContinue()
       throws IOException {
       InputStream is=null;
       int type=-1;

       SSLHandshakeHdr hdr=new SSLHandshakeHdr();

       if(state==SSL_HS_SEND_HELLO_REQUEST){
	 sendHelloRequest();
	 stateChange(SSL_HS_WAIT_FOR_CLIENT_HELLO);
       }
       
       is=recvHandshakeMsg(_conn,hdr);
       type=hdr.ct.value;
       _conn.debug(SSLDebug.DEBUG_STATE,
	 "Processing handshake message of type " + type);

       switch(type){
	 case SSL_HT_CLIENT_HELLO:
	   stateAssert(SSL_HS_WAIT_FOR_CLIENT_HELLO);
	   recvClientHello(is);

	   resume=false;
	   if(possibleResume!=null){
	     // If client auth was requested, and it wasn't done already,
	     // we can't resume
	     if(_conn.getPolicy().requireClientAuthP() &&
	       (possibleResume.getPeerCertificateChain()==null))
	       resume=false;
	     else{
	       restoreSession(possibleResume);
	       resume=true;
	     }
	   }

	   if(!resume){
	     session_id=new byte[32];
	     rng.nextBytes(session_id);
	     selectCipherSuite();
	   }
	     
	   sendServerHello();
	   
	   if(resume){
	     computeNextCipherStates();
	     sendChangeCipherSpec();
	     sendFinished();
	     stateChange(SSL_HS_WAIT_FOR_CHANGE_CIPHER_SPECS);
	   }
	   else{
	     sendCertificate();
	     sendServerKeyExchange(); 
	     if(_conn.getPolicy().requireClientAuthP()){
	       clientAuth=true;
	       sendCertificateRequest();
	       stateChange(SSL_HS_WAIT_FOR_CERTIFICATE);
	     }
	     else
	       stateChange(SSL_HS_WAIT_FOR_CLIENT_KEY_EXCHANGE);
	     sendServerHelloDone();
	   }
	   break;
	 case SSL_HT_CERTIFICATE:
	   stateAssert(SSL_HS_WAIT_FOR_CERTIFICATE);
	   recvCertificate(is);
	   stateChange(SSL_HS_WAIT_FOR_CLIENT_KEY_EXCHANGE);
	   break;
	 case SSL_HT_CLIENT_KEY_EXCHANGE:
	   stateAssert(SSL_HS_WAIT_FOR_CLIENT_KEY_EXCHANGE);
	   recvClientKeyExchange(is);
	   stateChange(clientAuth?SSL_HS_WAIT_FOR_CERTIFICATE_VERIFY:
	     SSL_HS_WAIT_FOR_CHANGE_CIPHER_SPECS);
	   break;
	 case SSL_HT_CERTIFICATE_VERIFY:
	   stateAssert(SSL_HS_WAIT_FOR_CERTIFICATE_VERIFY);
	   recvCertificateVerify(is);
	   stateChange(SSL_HS_WAIT_FOR_CHANGE_CIPHER_SPECS);
	   break;
	 case SSL_HT_FINISHED:
	   stateAssert(SSL_HS_WAIT_FOR_FINISHED);
	   recvFinished(is);
	   if(!resume){
	     sendChangeCipherSpec();
	     sendFinished();
	   }
	   storeSession(Util.toHex(session_id));	   
	   stateChange(SSL_HANDSHAKE_FINISHED);
	   break;
	 default:
	   _conn.alert(SSLAlertX.TLS_ALERT_HANDSHAKE_FAILURE);
       }
     }

     public void recvClientHello(InputStream is)
	 throws IOException {
	 SSLClientHello ch=new SSLClientHello();

	 ch.decode(_conn,is);

	 // Choose the highest common version
	 if(ch.client_version.value<SSL_V3_VERSION)
	   _conn.alert(SSLAlertX.TLS_ALERT_HANDSHAKE_FAILURE);
  	 _conn.ssl_version=Util.min(ch.client_version.value,_conn.ssl_version);

	 client_random=ch.random.value;

	 if(ch.session_id.value.length!=0)
	   possibleResume=findSession(Util.toHex(ch.session_id.value));

	 //Now remember the cipher_suites and compression
	 offered_cipher_suites=ch.cipher_suites.value;
	 offered_compression_methods=ch.compression_methods.value;
       }

       public void selectCipherSuite()
	 throws IOException {
	   
	 for(int i=0;i<cipher_suites.size();i++){
	   SSLCipherSuite oursuite=(SSLCipherSuite)cipher_suites.elementAt(i);
	   for(int j=0;j<offered_cipher_suites.size();j++){
	     if(oursuite.getValue()
	       ==((SSLuintX)offered_cipher_suites.elementAt(j)).value){
	       cipher_suite=oursuite;
	       SSLDebug.debug(SSLDebug.DEBUG_HANDSHAKE,"Choosing cipher" +
		 cipher_suite.getName());
	       return;
	     }
	   }
	 }

	 _conn.alert(SSLAlertX.TLS_ALERT_HANDSHAKE_FAILURE);
       }
       
       public void sendServerHello()
	 throws IOException {
	 SSLServerHello sh=new SSLServerHello();

	 sh.server_version.value=_conn.ssl_version;
	 rng.nextBytes(server_random);
	 sh.random.value=server_random;
	 sh.session_id.value=session_id;
	 sh.cipher_suite.value=cipher_suite.getValue();
	 sh.compression_method.value=0;  // The only one we support

	 sendHandshakeMsg(_conn,SSL_HT_SERVER_HELLO,sh);
       }

       public void recvClientKeyExchange(InputStream is)
         throws IOException {
	 SSLClientKeyExchange ck=new SSLClientKeyExchange();

	 ck.decode(_conn,is);			
	 computeMasterSecret();
	 computeNextCipherStates();
       }

       public void recvCertificateVerify(InputStream is)
         throws IOException {
	 SSLCertificateVerify cv=new SSLCertificateVerify(_conn,
	   this,false);

	 cv.decode(_conn,is);
       }
     
       public void sendServerKeyExchange()
	 throws IOException {
	 if(cipher_suite.requireServerKeyExchangeP(_conn.ctx.getPrivateKey())){
	   SSLServerKeyExchange sk=new SSLServerKeyExchange();
	   sendHandshakeMsg(_conn,SSL_HT_SERVER_KEY_EXCHANGE,sk);
	 }
       }
	   
       public void sendCertificateRequest()
	 throws IOException {
	 SSLCertificateRequest cr=new SSLCertificateRequest();

	 sendHandshakeMsg(_conn,SSL_HT_CERTIFICATE_REQUEST,cr);
       }
       
       public void sendServerHelloDone()
	 throws IOException {
	 sendHandshakeMsg(_conn,SSL_HT_SERVER_HELLO_DONE,new SSLServerHelloDone());
       }

       public void sendHelloRequest()
         throws IOException {
	 sendHandshakeMsg(_conn,SSL_HT_HELLO_REQUEST,new SSLHelloRequest(),false);
       }

     
}

