//
// linteger.cxx 
//
// Leonard Janke
// August 1996

#include "linteger.hxx"

LInteger::LInteger() : _magnitude(NULL), _sign(0), _size(0)
{
}

LInteger::LInteger(const unsigned int* magnitude, const int sizeInWords, 
		   const int sign=0, const int copyMag=1) 
  : _size(sizeInWords), _sign(sign) 
{
  if ( copyMag )
    {
      _magnitude=new unsigned int[_size];
      LMisc::MemCopy(magnitude,_magnitude,_size);
    }
  else
    _magnitude=magnitude;
}

LInteger::LInteger(const LInteger& a) : _size(a._size), _sign(a._sign)
{
  _magnitude=new unsigned int[_size];
  LMisc::MemCopy(a._magnitude,_magnitude,_size);
}

LInteger::LInteger(const unsigned int mag, const int sign=0) : 
  _size(1), _sign(sign)
{
  _magnitude=new unsigned int[1];

  *_magnitude=mag;
}

LInteger::LInteger(const int value) : 
  _size(1)
{
  _magnitude=new unsigned int[1];

  if ( value >= 0 )
    {
      *_magnitude=value; // implicit conversion here
      _sign=0;
    }
  else
    {
      *_magnitude=-1*value; // implicit conversion here
      _sign=1;
    }
}

LInteger::~LInteger()
{
  if ( _magnitude )
    delete[] _magnitude;
}

int LInteger::compress(unsigned int*& compressMe, const int originalLength) 
{
  int leadZeros=0;
  int scanPos=0;

  while ( ! *(compressMe+scanPos) && scanPos <originalLength-1)
    {
      leadZeros++;
      scanPos++;
    }
    
  if ( leadZeros )
    {
      unsigned int* compressedMag=new unsigned int[originalLength-leadZeros];
      
      for (int i=0; i<originalLength-leadZeros; i++)
	*(compressedMag+i)=*(compressMe+i+leadZeros);
      
      delete[] compressMe;
      compressMe=compressedMag;
    }

  return originalLength-leadZeros;
}

LInteger& LInteger::operator=(const LInteger& a) 
{
  if ( this != &a )
    {
      _size=a._size;
      _sign=a._sign;

      if ( _magnitude )
	delete[] _magnitude;

      _magnitude=new unsigned int[_size];
      LMisc::MemCopy(a._magnitude,_magnitude,_size);
    }
}

void LInteger::compress() 
{
  int leadZeros=0;
  int scanPos=0;

  while ( ! *(_magnitude+scanPos) && scanPos <_size-1)
    {
      leadZeros++;
      scanPos++;
    }
    
  if ( leadZeros )
    {
      unsigned int* compressedMag=new unsigned int[_size-leadZeros];
      
      LMisc::MemCopy(_magnitude+leadZeros,compressedMag,_size-leadZeros);
      
      if ( _magnitude )
	delete[] _magnitude;

      _magnitude=compressedMag;
      _size-=leadZeros;
    }
}

LInteger operator^(const LInteger& a, const LInteger& b)
{
  assert( !a._sign && !b._sign );

  assert( a._size == b._size );

  unsigned int* newMag;
  int newMagSize=a._size;

  newMag=new unsigned int[newMagSize];

  for (int i=0; i<newMagSize; i++)
    *(newMag+i)=*(a._magnitude+i)^*(b._magnitude+i);

  newMagSize=LInteger::compress(newMag,newMagSize);

  return  LInteger(newMag,newMagSize,0,0);
}

LInteger operator|(const LInteger& a, const LInteger& b)
{
  assert( !a._sign && !b._sign );

  assert( a._size == b._size );

  unsigned int* newMag;
  int newMagSize=a._size;

  newMag=new unsigned int[newMagSize];

  for (int i=0; i<newMagSize; i++)
    *(newMag+i)=*(a._magnitude+i)|*(b._magnitude+i);

  return  LInteger(newMag,newMagSize,0,0);
}

LInteger operator&(const LInteger& a, const LInteger& b)
{
  assert(!a._sign && !b._sign );

  assert( a._size == b._size );

  unsigned int* newMag;
  int newMagSize=a._size;

  newMag=new unsigned int[newMagSize];

  for (int i=0; i<newMagSize; i++)
    *(newMag+i)=*(a._magnitude+i)&*(b._magnitude+i);

  newMagSize=LInteger::compress(newMag,newMagSize);

  return  LInteger(newMag,newMagSize,0,0);
}

LInteger operator~(const LInteger& a)
{
  assert( !a._sign );

  unsigned int* newMag;
  int newMagSize=a._size;

  newMag=new unsigned int[newMagSize];

  for (int i=0; i<newMagSize; i++)
    *(newMag+i)=~*(a._magnitude+i);

  newMagSize=LInteger::compress(newMag,newMagSize);

  return  LInteger(newMag,newMagSize,0,0);
}

LInteger operator+(const LInteger& a, const LInteger& b) 
{
  unsigned int sizeOfC;
  unsigned int signOfC;
  unsigned int* magOfC;

  char carry;

  if ( a._sign != b._sign ) 
    {
      switch ( LInteger::CompareMagnitudes(a,b) )
	{
	case 1:   // abs(a) > abs (b)
	  signOfC=a._sign;
	  sizeOfC=a._size;
	  magOfC=new unsigned int[sizeOfC];
	  BMath::Subtract(a._magnitude, b._magnitude, magOfC,a._size,b._size);
	  sizeOfC=LInteger::compress(magOfC,sizeOfC);
	  break;
	case -1: // abs(b) < abs(a)
	  signOfC=b._sign;
	  sizeOfC=b._size;
	  magOfC=new unsigned int[sizeOfC];
	  BMath::Subtract(b._magnitude, a._magnitude, magOfC,b._size,a._size);
	  sizeOfC=LInteger::compress(magOfC,sizeOfC);
	  break;
	case 0:   // abs(a)==abs(b)
	  signOfC=0;
	  sizeOfC=1;
	  magOfC=new unsigned int[1];
	  *magOfC=0;
	  break;
	}
    }
  else // a._sign == b._sign
    {
      signOfC=a._sign;

      if ( a._size > b._size )
	{
	  sizeOfC=a._size;
	  magOfC=new unsigned int[sizeOfC];
	  BMath::Add(a._magnitude,b._magnitude, magOfC,a._size,b._size,carry);
	}
      else
	{
	  sizeOfC=b._size;
	  magOfC=new unsigned int[sizeOfC];
	  BMath::Add(b._magnitude,a._magnitude, magOfC,b._size,a._size,carry);
	}

      if ( carry )
	{
	  unsigned int* scratch;

	  scratch=new unsigned int[sizeOfC+1];

	  *scratch=1ul;
	  LMisc::MemCopy(magOfC,scratch+1,sizeOfC);

	  delete[] magOfC;
	  magOfC=scratch;
	  sizeOfC++;
	}
    }

  return LInteger(magOfC,sizeOfC,signOfC,0);
}
 
LInteger operator*(const LInteger& a, const LInteger& b) 
{
  unsigned int sizeOfC;
  unsigned int signOfC;
  unsigned int* magOfC;

  if ( a._sign==0 && a._size==1 && *a._magnitude==0 ||
       b._sign==0 && b._size==1 && *b._magnitude==0 )
    {
      signOfC=0;
      sizeOfC=1;
      magOfC=new unsigned int[1];
      *magOfC=0;
    }
  else
    {
      signOfC=a._sign^b._sign;
      sizeOfC=a._size+b._size;
      magOfC=new unsigned int[sizeOfC];
      LMisc::MemZero(magOfC,sizeOfC);
      BMath::Multiply(a._magnitude, a._size, b._magnitude, b._size, magOfC);
      sizeOfC=LInteger::compress(magOfC,sizeOfC);
    }

  return LInteger(magOfC,sizeOfC,signOfC,0);
}

LInteger operator/(const LInteger& dividend, const LInteger& divisor) 
{
  assert ( divisor!=0 );

  if ( divisor._size > dividend._size )
    if ( dividend >= 0 )
      return 0;
    else
      if ( divisor > 0 )
	return LInteger::Negative(1);
      else
	return 1;

  unsigned int* q;
  unsigned int* r;
    
  BMath::Divide(divisor._magnitude, divisor._size, 
		dividend._magnitude, dividend._size, q, r);

  if (  !dividend._sign && !divisor._sign )
    {
      delete[] r;
      int qSize=LInteger::compress(q,dividend._size-divisor._size+1);
      return LInteger(q,qSize,0,0);
    }
  else if ( !dividend._sign && divisor._sign )
    {
      delete[] r;
      int qSize=LInteger::compress(q,dividend._size-divisor._size+1);
      return LInteger::Negative(LInteger(q,qSize,0,0));
    }
  else if ( dividend._sign && !divisor._sign )
    {
      int qSize=LInteger::compress(q,dividend._size-divisor._size+1);
      int rSize=LInteger::compress(r,dividend._size+1);
      LInteger R(r,rSize,0,0);

      if ( R == 0 )
	return LInteger::Negative(LInteger(q,qSize,0,0));

      return LInteger::Negative(LInteger(q,qSize,0,0)+1);
    }
  else // dividend._sign && divisor._sign
    {
      int qSize=LInteger::compress(q,dividend._size-divisor._size+1);
      int rSize=LInteger::compress(r,dividend._size+1);
      LInteger R(r,rSize,0,0);

      if ( R == 0 )
	return LInteger(q,qSize,0,0);

      return LInteger(q,qSize,0,0)+1;
    }
}

LInteger operator%(const LInteger& dividend, const LInteger& divisor) 
{
  assert( divisor!=0 );

  if ( divisor._size > dividend._size )
    if ( dividend >= 0 )
      return dividend;
    else
      return LInteger::AbsoluteValue(divisor)+dividend;

  unsigned int* q;
  unsigned int* r;
    
  BMath::Divide(divisor._magnitude, divisor._size, 
		dividend._magnitude, dividend._size, q,r);

  delete[] q;
  int rSize=LInteger::compress(r,dividend._size+1);

  if (  !dividend._sign ) 
    return LInteger(r,rSize,0,0);
  else // dividend<0
    {
      LInteger R(r,rSize,0,0);

      if ( R == 0 )
	return LInteger(R);
      else
	return LInteger::AbsoluteValue(divisor)-R;
    }
}

LInteger operator>>(const LInteger& x, const int distance) 
{
  int n=distance/LMisc::bitsPerInt;
  unsigned int* newMag;
  int newMagSize;

  if ( x._size-n > 0 )
    {
      newMagSize=x._size-n; 
      newMag=new unsigned int[newMagSize];

      LMisc::MemCopy(x._magnitude,newMag,newMagSize);
    }
  else
    {
      newMagSize=1;
      newMag=new unsigned int[newMagSize];
      *newMag=0ul;
    }

  BMath::ShiftRight(newMag,newMagSize,char(distance%LMisc::bitsPerInt));

  newMagSize=LInteger::compress(newMag,newMagSize);
  
  return LInteger(newMag,newMagSize,x._sign,0);
}

LInteger operator<<(const LInteger& x, const int distance) 
{
  unsigned int* newMag;
  char bitPos=char(BMath::BSR(*x._magnitude));

  int newMagSize=x._size+(int(bitPos)+distance)/LMisc::bitsPerInt;

  newMag=new unsigned int[newMagSize];

  int offset=0;

  if ( (bitPos+distance%LMisc::bitsPerInt) /LMisc::bitsPerInt )
    { 
      offset++;
      *newMag=0;
    }

  LMisc::MemCopy(x._magnitude,newMag+offset,x._size);
  offset+=x._size;

  LMisc::MemZero(newMag+offset,distance/LMisc::bitsPerInt);

  BMath::ShiftLeft(newMag,newMagSize,char(distance%LMisc::bitsPerInt));

  return LInteger(newMag,newMagSize,x._sign,0);
}

int LInteger::CompareMagnitudes(const LInteger& a, const LInteger& b)
{
  // this should be rewritten in assembly and put in LMisc

  assert( !a.compressable() && !b.compressable() );

  if ( a._size > b._size )
    return 1;
  
  if ( b._size > a._size )
    return -1;

  // can assume a._size == b._size now

  int scanPos=0;

  while ( scanPos<a._size )
    if ( *(a._magnitude+scanPos) > *(b._magnitude+scanPos) )
      return 1;
    else if ( *(a._magnitude+scanPos) < *(b._magnitude+scanPos) )
      return -1;
    else
      scanPos++;

  return 0;
}

ostream& operator<<(ostream& s, const LInteger& a)
{
  if ( a._sign )
    s <<"-";
  else
    s <<"+";

  for (int i=0; i<a._size; i++)
    s <<hex<<setfill('0')<<setw(2*sizeof(unsigned int))<<*(a._magnitude+i);

  return s;
}
