/*
 *  $Id: file.c,v 1.1 1998/12/02 23:37:06 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <fist.h>
#include <cryptfs.h>


/*******************
 * File Operations *
 *******************/

// STATIC int
// cryptfs_lseek(inode_t *inode, file_t *file, off_t offset, int origin)
STATIC loff_t
cryptfs_llseek(file_t *file, loff_t offset, int origin)
{
    loff_t err;
    file_t *hidden_file = ftohf(file);

    print_entry_location();

    fist_dprint(6, "cryptfs_llseek: file=%x, offset=0x%x, origin=%d\n",
		(int) file, (int) offset, origin);

    //    fist_print_file("LLSEEK Before file", file);
    //    fist_print_file("LLSEEK Before hidden_file", hidden_file);

    // ION
    /* always set hidden position to this one */
    hidden_file->f_pos = file->f_pos;
    if (file->f_reada) { /* update readahead information if needed */
	hidden_file->f_reada = file->f_reada;
	hidden_file->f_ramax = file->f_ramax;
	hidden_file->f_raend = file->f_raend;
	hidden_file->f_ralen = file->f_ralen;
	hidden_file->f_rawin = file->f_rawin;
    }
    ASSERT(hidden_file->f_reada == file->f_reada);
    ASSERT(hidden_file->f_ramax == file->f_ramax);
    ASSERT(hidden_file->f_raend == file->f_raend);
    ASSERT(hidden_file->f_ralen == file->f_ralen);
    ASSERT(hidden_file->f_rawin == file->f_rawin);

    if (hidden_file->f_op && hidden_file->f_op->llseek) {
	err = hidden_file->f_op->llseek(hidden_file, offset, origin);
    } else {
	err = generic_llseek(hidden_file, offset, origin);
    }
    if (err < 0) {
	goto out;
    }

    if (err != file->f_pos) {
	file->f_pos = err;
	// ION maybe this?
	// 	file->f_pos = hidden_file->f_pos;
	file->f_reada = 0;
	file->f_version = ++event;
    }

out:
    //    fist_print_file("LLSEEK After file", file);
    //    fist_print_file("LLSEEK After hidden_file", hidden_file);
    print_exit_status(err);
    return err;
}


// STATIC int
// cryptfs_read(inode_t *inode, file_t *file, char *buf, int count)
STATIC ssize_t
cryptfs_read(file_t *file, char *buf, size_t count, loff_t *ppos)
{
    int err = -EINVAL;
    file_t *hidden_file = ftohf(file);
    char *hidden_buf;
    mm_segment_t old_fs;
    loff_t pos = *ppos;
    void *key;

    print_entry_location();
    if ((key = fist_get_userpass(file->f_dentry->d_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }

    if (!hidden_file->f_op || !hidden_file->f_op->read)
	goto out;
    hidden_buf = vmalloc(count);

    old_fs = get_fs();
    set_fs(KERNEL_DS);

    err = hidden_file->f_op->read(hidden_file, hidden_buf, count, &pos);

    set_fs(old_fs);
    if (err > 0) {
	cryptfs_decode_block(hidden_buf, hidden_buf, err, key);
	fist_copy_attr_atime(file->f_dentry->d_inode,
			     hidden_file->f_dentry->d_inode);
	if (copy_to_user(buf, hidden_buf, err))
	    err = -EINVAL;
    }

    // MAJOR HACK
    /*
     * because pread() does not have any way to tell us that it is
     * our caller, then we don't know for sure if we have to update
     * the file positions.  This hack relies on read() having passed us
     * the "real" pointer of its struct file's f_pos field.
     */
    if (ppos == &file->f_pos)
	hidden_file->f_pos = *ppos = pos;

    vfree(hidden_buf);

out:
    print_exit_status(err);
    return err;
}


// STATIC int
// cryptfs_write(inode_t *inode, file_t *file, const char *buf, int count)
#if 0
STATIC ssize_t
cryptfs_write(file_t *file, const char *buf, size_t count, loff_t *ppos)
{
    int err = -EINVAL;
    file_t *hidden_file = ftohf(file);
    char *hidden_buf;
    mm_segment_t old_fs;
    loff_t pos = *ppos;

    print_entry_location();

    if (!hidden_file->f_op || !hidden_file->f_op->write)
	goto out;

    if (file->f_flags & O_APPEND)
	pos = file->f_dentry->d_inode->i_size;

    hidden_buf = vmalloc(count);
    copy_from_user(hidden_buf, buf, count);
    cryptfs_encode_block(hidden_buf, hidden_buf, count);

    old_fs = get_fs();
    set_fs(KERNEL_DS);

    err = hidden_file->f_op->write(hidden_file, hidden_buf, count, &pos);

    set_fs(old_fs);

    // MAJOR HACK
    /*
     * because pread() does not have any way to tell us that it is
     * our caller, then we don't know for sure if we have to update
     * the file positions.  This hack relies on read() having passed us
     * the "real" pointer of its struct file's f_pos field.
     */
    if (ppos == &file->f_pos)
	hidden_file->f_pos = *ppos = pos;

    /* writing may increase the file's size */
    if (err > 0)
	fist_copy_attr_timesizes(file->f_dentry->d_inode,
				 hidden_file->f_dentry->d_inode);

    vfree(hidden_buf);

out:
    print_exit_status(err);
    return err;
}
#else
/*
 * Adapted from 2.1 generic_file_write
 */
// STATIC int
// cryptfs_write(inode_t *inode, file_t *file, const char *buf, int count)
STATIC ssize_t
cryptfs_write(file_t *file, const char *buf, size_t count, loff_t *ppos)
{
    file_t *hidden_file;
    page_t *page, **hash;
    unsigned long page_cache = 0;
    unsigned long pgpos, orig_pgpos, offset;
    unsigned long bytes, written;
    size_t bytes_to_write;
    loff_t pos, orig_pos, tmp_pos;
    ssize_t status;
    long didread;
    mm_segment_t old_fs;
    char *hidden_buffer;
    inode_t *inode = file->f_dentry->d_inode;
    inode_t *hidden_inode = itohi(inode);
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(file->f_dentry->d_sb)) == NULL) {
	status = -EACCES;
	goto out;
    }

    /* check that offsets are correct */
    ASSERT(file != NULL);
    ASSERT(ppos != NULL);

    pos = *ppos;
    written = 0;
    status = 0;

    old_fs = get_fs();
    hidden_file = ftohf(file);
    ASSERT(hidden_file != NULL);

    hidden_buffer = kmalloc(PAGE_SIZE, GFP_KERNEL);
    if (!hidden_buffer) {
	printk("Out of memory.\n");
	status = -ENOMEM;
	goto out;
    }
    /* we don't want anybody to do updates while we write, so lock the inode */
    down(&hidden_inode->i_sem);

    if (file->f_flags & O_APPEND)
	pos = inode->i_size;

    fist_dprint(6, "EZK: WRITE: initial values: pos %ld, count %d\n", (long) pos, count);
    fist_print_file("EZK: WRITE file IN", file);

    /* we need to handle sparse files differently */
    orig_pos = pos;
    orig_pgpos = orig_pos & PAGE_MASK;
    if (pos > inode->i_size)
	pos = inode->i_size;

    while (count) {
	fist_dprint(7, "WRITE: pos %ld, written %ld, count %d, buf 0x%lx\n",
		    (long) pos, written, count, (long) buf);
	/*
	 * Try to find the page in the cache. If it isn't there,
	 * allocate a free page.
	 */
	offset = (pos & ~PAGE_MASK);
	pgpos = pos & PAGE_MASK;

	if ((bytes = PAGE_SIZE - offset) > count && pgpos >= orig_pgpos)
	    bytes = count;

	hash = page_hash(inode, pgpos);
	if (!(page = __find_page(inode, pgpos, *hash))) {
	    if (!page_cache) {
		page_cache = __get_free_page(GFP_KERNEL);
		if (page_cache)
		    continue;
		status = -ENOMEM;
		break;
	    }
	    page = mem_map + MAP_NR(page_cache);
	    add_to_page_cache(page, inode, pgpos, hash);
	    page_cache = 0;
	}
	/*
	 * Note: setting of the PG_locked bit is handled
	 * below the i_op->xxx interface.
	 */
	didread = 0;
    page_wait:
	wait_on_page(page);
	if (PageUptodate(page))
	    goto do_update_page;

	/*
	 * The page is not up-to-date ... if we're writing less
	 * than a full page of data, we may have to read it first.
	 * But if the page is past the current end of file, we must
	 * clear it before updating.
	 */
	if (bytes < PAGE_SIZE) {
	    if (pgpos < inode->i_size) {
		status = -EIO;
		if (didread >= 2)
		    goto done_with_page;
		/* call our own readpage function */
		status = inode->i_op->readpage(file, page);
		if (status < 0)
		    goto done_with_page;
		didread++;
		goto page_wait;
	    } else {
		/* Must clear for partial writes */
		fist_dprint(7, "WRITE1: clearing page at offset 0x%x\n", (int) pgpos);
		memset((void *) page_address(page), 0, PAGE_SIZE);
	    }
	} else if (pos < orig_pos) {
	    /* Must clear for sparse files */
	    fist_dprint(7, "WRITE2: clearing page at offset 0x%x\n", (int) pgpos);
	    memset((void *) page_address(page), 0, PAGE_SIZE);
	}
	/*
	 * N.B. We should defer setting PG_uptodate at least until
	 * the data is copied. A failure in i_op->updatepage() could
	 * leave the page with garbage data.
	 */
	set_bit(PG_uptodate, &page->flags);

    do_update_page:
	/* Alright, the page is there.  Now update it. */
	fist_dprint(7, "WRITE: count %d, pos %ld, offset %ld, pgpos 0x%x, bytes %d\n",
		    count, (int) pos, (int) offset, (int) pgpos, (int) bytes);

	if (pgpos < orig_pgpos)
	    /* nothing to do, not even copy_from_user */
	    goto encode_page;
	if (pgpos == orig_pgpos) {
	    /*
	     * this is the only interesting case,
	     * we have to shift from orig_pos to pos somehow
	     */
	    pos = orig_pos;
	    offset = pos & ~PAGE_MASK;
	    if ((bytes = PAGE_SIZE - offset) > count)
		bytes = count;
	}
#if 0
	status = inode->i_op->updatepage(file, page, buf,
					 offset, bytes, sync);
#else
	fist_dprint(7, "WRITE: copying %ld bytes at offset %ld\n", bytes, offset);
	copy_from_user((char *) (page_address(page) + offset), buf, bytes);
    encode_page:
	cryptfs_encode_block((char *) page_address(page), hidden_buffer, PAGE_SIZE, key);

	hidden_file->f_pos = pgpos; /* is this needed? */
	if (inode->i_size < pgpos + PAGE_SIZE)
	    if (inode->i_size > pos + bytes)
		bytes_to_write = inode->i_size - pgpos;
	    else
		bytes_to_write = offset + bytes;
	else
	    bytes_to_write = PAGE_SIZE;

	fist_dprint(7, "WRITE: writing at pgpos %ld, size %ld\n", pgpos, bytes_to_write);
	/* switch to kernel space */
	set_fs(KERNEL_DS);
	tmp_pos = pgpos;
	status = hidden_file->f_op->write(hidden_file,
					  hidden_buffer,
					  bytes_to_write,
					  &tmp_pos);
	fist_dprint(7, "WROTE: tmp_pos %ld, status %d\n", (int) tmp_pos, (int) status);

	/* switch back to user space */
	set_fs(old_fs);
#endif

	/* the rest of the code must not see that we are writing extra bytes */
	/* do not adjust status if only filling up holes */
	if (status > 0 && pgpos >= orig_pgpos)
	    if (status < offset)
		status = 0;
	    else {
		status -= offset;
		if (status > bytes)
		    status = bytes;
	    }
    done_with_page:
	__free_page(page);
	if (status < 0)
	    break;

	/* do not adjust these variables if only filling up holes */
	if (pgpos >= orig_pgpos) {
	    written += status;
	    count -= status;
	    buf += status;
	}
	pos += status;
    }
    // MAJOR HACK
    /*
     * because pread() does not have any way to tell us that it is
     * our caller, then we don't know for sure if we have to update
     * the file positions.  This hack relies on read() having passed us
     * the "real" pointer of its struct file's f_pos field.
     */
    if (ppos == &file->f_pos)
	hidden_file->f_pos = *ppos = pos;
    //    file->f_pos = pos;

    if (pos > inode->i_size)
	inode->i_size = pos;

    up(&hidden_inode->i_sem);

    kfree_s(hidden_buffer, PAGE_SIZE);

    if (page_cache)
	free_page(page_cache);

    status = written ? written : status;

out:
    print_exit_status(status);
    return status;
}
#endif

struct cryptfs_getdents_callback {
    void *dirent;
    super_block_t *super;
    filldir_t filldir;
};

/* copied from generic filldir in fs/readir.c */
STATIC int
cryptfs_filldir(void *dirent, const char *name, int namlen, off_t offset, ino_t ino)
{
    struct cryptfs_getdents_callback *buf = (struct cryptfs_getdents_callback *) dirent;
    int err;
    char *decoded_name;
    int decoded_length;
    void *key;

    key = fist_get_userpass(buf->super);

    if ((decoded_length = cryptfs_decode_filename(name, namlen, &decoded_name, key, SKIP_DOTS)) < 0)
	return 0;			/* no error, just skip the entry */

    err = buf->filldir(buf->dirent, decoded_name, decoded_length, offset, ino);
    kfree_s(decoded_name, decoded_length);
    return err;
}


// STATIC int
// cryptfs_readdir(inode_t *inode, file_t *file, void *dirent, filldir_t filldir)
STATIC int
cryptfs_readdir(file_t *file, void *dirent, filldir_t filldir)
{
    int err = -ENOTDIR;
    file_t *hidden_file = ftohf(file);
    inode_t *inode = file->f_dentry->d_inode;
    inode_t *hidden_inode = itohi(inode);
    struct cryptfs_getdents_callback buf;

    print_entry_location();

    fist_checkinode(inode, "cryptfs_readdir");
    if (!hidden_file || !hidden_file->f_op || !hidden_file->f_op->readdir)
	goto out;

    /* prepare for callback */
    buf.dirent = dirent;
    buf.super = file->f_dentry->d_sb;
    buf.filldir = filldir;

    down(&hidden_inode->i_sem);
    err = hidden_file->f_op->readdir(hidden_file, (void *) &buf, cryptfs_filldir);
    up(&hidden_inode->i_sem);

    file->f_pos = hidden_file->f_pos;
    if (err > 0)
	fist_copy_attr_atime(file->f_dentry->d_inode,
			     hidden_file->f_dentry->d_inode);
    fist_checkinode(inode, "post cryptfs_readdir");

out:
    print_exit_status(err);
    return err;
}


// STATIC int
// cryptfs_select(inode_t *inode, file_t *file, int flag, select_table * wait)
STATIC unsigned int
cryptfs_poll(file_t *file, poll_table *wait)
{
    int err = 0;
    file_t *hidden_file = ftohf(file);

    print_entry_location();

    if (!hidden_file->f_op || !hidden_file->f_op->poll)
	goto out;

    err = hidden_file->f_op->poll(hidden_file, wait);

out:
    print_exit_status(err);
    return err;
}


STATIC int
cryptfs_ioctl(inode_t *inode, file_t *file, unsigned int cmd, unsigned long arg)
{
    int err = 0, val;
    file_t *hidden_file;
    unsigned char cbc_key[16];

    print_entry_location();

    /* check if asked for local commands */
    switch (cmd) {
#ifdef FIST_DEBUG
    case FIST_IOCTL_GET_DEBUG_VALUE:
	val = fist_get_debug_value();
	printk("IOCTL GET: got arg %d\n", val);
	err = put_user(val, (int *) arg);
	break;

    case FIST_IOCTL_SET_DEBUG_VALUE:
	err = get_user(val, (int *) arg);
	if (err)
	    break;
	fist_dprint(6, "IOCTL SET: got arg %d\n", val);
	if (val < 0 || val > 20) {
	    err = -EINVAL;
	    break;
	}
	fist_set_debug_value(val);
	break;
#endif /* FIST_DEBUG */

    case FIST_IOCTL_SET_KEY:
	if (copy_from_user((char *) cbc_key, (char *) arg, 16)) {
	    err = -EFAULT;
	    goto out;
	}
	fist_set_userpass(inode->i_sb, cbc_key);
	break;

    /* add non-debugging fist ioctl's here */

    default:
	hidden_file = ftohf(file);
	/* pass operation to hidden filesystem, and return status */
	if (hidden_file->f_op && hidden_file->f_op->ioctl)
	    err = hidden_file->f_op->ioctl(itohi(inode), hidden_file, cmd, arg);
    } /* end of switch statement */

 out:
    print_exit_status(err);
    return err;
}


// STATIC int
// cryptfs_mmap(inode_t *inode, file_t *file, vm_area_t *vma)
STATIC int
cryptfs_mmap(file_t *file, vm_area_t *vma)
{
    int err = 0;
    file_t *hidden_file = ftohf(file);
    inode_t *inode = file->f_dentry->d_inode;
    inode_t *hidden_inode = itohi(inode);
    vm_area_t *hidden_vma = (vm_area_t *) 0xdeadc0de;

    print_entry_location();

    fist_dprint(6, "MMAP1: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) inode, (int) hidden_inode, (int) inode->i_count, (int) hidden_inode->i_count);

    if (!hidden_file->f_op || !hidden_file->f_op->mmap) {
	err = -ENODEV;
	goto out;
    }

    /*
     * Most of this code comes straight from generic_file_mmap
     * in mm/filemap.c.
     */
    if ((vma->vm_flags & VM_SHARED) && (vma->vm_flags & VM_MAYWRITE)) {
	vma->vm_ops = &cryptfs_shared_vmops;
	/* share_page() can only guarantee proper page sharing if * the offsets
	 * are all page aligned. */
	if (vma->vm_offset & (PAGE_SIZE - 1)) {
	    err = -EINVAL;
	    goto out;
	}
    } else {
	vma->vm_ops = &cryptfs_private_vmops;
	if (vma->vm_offset & (hidden_inode->i_sb->s_blocksize - 1)) {
	    err = -EINVAL;
	    goto out;
	}
    }
    if (!inode->i_sb || !S_ISREG(inode->i_mode)) {
	err = -EACCES;
	goto out;
    }
    if (!hidden_inode->i_op || !hidden_inode->i_op->readpage) {
	err = -ENOEXEC;
	goto out;
    }
    UPDATE_ATIME(inode);
    vma->vm_file = file;
    file->f_count++;

    /*
     * Now we do the hidden stuff, but only for shared maps.
     */
    if ((vma->vm_flags & VM_SHARED) && (vma->vm_flags & VM_MAYWRITE)) {
	hidden_vma = kmalloc(sizeof(vm_area_t), GFP_KERNEL);
	if (!hidden_vma) {
	    printk("MMAP: Out of memory\n");
	    err = -ENOMEM;
	    goto out;
	}
	// ION, is this right?
	memcpy(hidden_vma, vma, sizeof(vm_area_t));
	vmatohvma(vma) = hidden_vma;
	err = hidden_file->f_op->mmap(hidden_file, hidden_vma);
#if 0
	/* XXX: do we still need this? */
	fput(hidden_file);
#endif
    }

    /*
     * XXX: do we need to insert_vm_struct and merge_segments as is
     * done in do_mmap()?
     */
 out:
    fist_dprint(6, "MMAP2: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) inode, (int) hidden_inode, (int) inode->i_count, (int) hidden_inode->i_count);
#if 0
    if (!err) {
	inode->i_mmap = vma;
	hidden_inode->i_mmap = hidden_vma;
    }
#endif
    print_exit_status(err);
    return err;
}


STATIC int
cryptfs_open(inode_t *inode, file_t *file)
{
    int err = -ENFILE;
    file_t *hidden_file;
    inode_t *hidden_inode = itohi(inode);
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(file->f_dentry);

    print_entry_location();

    fist_print_dentry(__FUNCTION__ " IN hidden_dentry", hidden_dentry);

    hidden_file = get_empty_filp();
    if (!hidden_file)
	goto out;

    fist_dprint(8, "cryptfs_open: got empty hidden_file\n");
    file->private_data = hidden_file; /* link two file's */

    hidden_file->f_dentry = hidden_dentry;
    dget(hidden_dentry);

    fist_dprint(6, "cryptfs_open: got f_inode\n");
    hidden_file->f_mode = file->f_mode | FMODE_READ;
    hidden_file->f_flags = file->f_flags;
    if (file->f_flags & O_APPEND) {
	fist_dprint(5, "file is opened in append-only mode!!!\n");
	hidden_file->f_flags &= ~O_APPEND;	/* turn off O_APPEND flag */
    }
    hidden_file->f_pos = file->f_pos;
    hidden_file->f_reada = file->f_reada;

    hidden_file->f_op = NULL;
    if (hidden_inode->i_op)
	hidden_file->f_op = hidden_inode->i_op->default_file_ops;

    fist_dprint(6, "cryptfs_open: pre open\n");
    if (hidden_file->f_op && hidden_file->f_op->open) {
	err = hidden_file->f_op->open(hidden_inode, hidden_file);
	if (err) {
	    printk(KERN_WARNING "cryptfs_open: error(%d)\n", err);
	    hidden_file->f_dentry = NULL;
	    put_filp(hidden_file);
	    goto out;
	}
    } else {
	err = 0;
    }

    hidden_file->f_flags &= ~(O_CREAT | O_EXCL | O_NOCTTY | O_TRUNC);
    fist_checkinode(inode, "post cryptfs_open");
    fist_dprint(6, "cryptfs_open: done (%d)\n", err);

out:
    fist_print_dentry(__FUNCTION__ " OUT hidden_dentry", hidden_dentry);
    print_exit_status(err);
    return err;
}


// newly implemented
STATIC int
cryptfs_flush(file_t *file)
{
    int err = 0;		/* assume ok (see open.c:close_fp) */
    file_t *hidden_file = ftohf(file);

    print_entry_location();

    if (!hidden_file->f_op || !hidden_file->f_op->flush)
	goto out;

    err = hidden_file->f_op->flush(hidden_file);

out:
    print_exit_status(err);
    return err;
}


// STATIC void
STATIC int
cryptfs_release(inode_t *inode, file_t *file)
{
    int err = 0;
    file_t *hidden_file = ftohf(file);
    dentry_t *hidden_dentry;

    print_entry_location();
    ASSERT(hidden_file != NULL);

    fist_print_dentry("cryptfs_release IN hidden_dentry", hidden_file->f_dentry);
    fist_checkinode(inode, "cryptfs_release");
    fist_dprint(6, "cryptfs_release IN, file->f_count=%d\n", file->f_count);
    /*
     * will decrement file refcount, and if 0, destroy the file,
     * which will call the lower file system's file release function.
     */
    hidden_dentry = hidden_file->f_dentry;
    fput(hidden_file);

    fist_dprint(6, "cryptfs_release done\n");
    fist_checkinode(inode, "post cryptfs_release");

    fist_print_dentry("cryptfs_release OUT hidden_dentry", hidden_dentry);
    print_exit_status(err);
    return err;
}


// STATIC int
// cryptfs_fsync(inode_t *inode, file_t *file)
STATIC int
cryptfs_fsync(file_t *file, dentry_t *dentry)
{
    int err = -EINVAL;
    file_t *hidden_file = ftohf(file);
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(dentry);

    print_entry_location();

    if (hidden_file->f_op && hidden_file->f_op->fsync) {
	down(&hidden_dentry->d_inode->i_sem);
	err = hidden_file->f_op->fsync(hidden_file, hidden_dentry);
	up(&hidden_dentry->d_inode->i_sem);
    }

    print_exit_status(err);
    return err;
}


// STATIC int
// cryptfs_fasync(inode_t *inode, file_t *file, int flag)
STATIC int
cryptfs_fasync(int fd, file_t *file, int flag)
{
    int err = 0;
    file_t *hidden_file = ftohf(file);

    print_entry_location();

    if (hidden_file->f_op && hidden_file->f_op->fasync)
	err = hidden_file->f_op->fasync(fd, hidden_file, flag);

    print_exit_status(err);
    return err;
}


// newly implemented
STATIC int
cryptfs_check_media_change(kdev_t dev)
{
    int err = 0;
    super_block_t *sb;

    print_entry_location();

    sb = get_super(dev);
    err = check_disk_change(stohs(sb)->s_dev);

    print_exit_status(err);
    return err;
}


#ifdef NOT_NEEDED
/*
 * This is not needed, because our check_media_change will call
 * the generic check_disk_change() which will run the revalidate
 * function of the hidden file system, if implemented.
 */
STATIC int
cryptfs_file_revalidate(kdev_t dev)
{
    return -ENOSYS;
}
#endif /* NOT_NEEDED */


// newly implemented
STATIC int
cryptfs_lock(file_t *file, int cmd, struct file_lock *fl)
{
    int err = 0;
    file_t *hidden_file = ftohf(file);

    print_entry_location();

    if (hidden_file->f_op->lock) {
	err = hidden_file->f_op->lock(hidden_file, F_GETLK, fl);
    } else {
	posix_test_lock(hidden_file, fl);
    }

    print_exit_status(err);
    return err;
}


struct file_operations cryptfs_fops =
{
    cryptfs_llseek,		/* llseek */
#if 1
    generic_file_read,		/* read */
#else
    cryptfs_read,		/* read */
#endif
#if 0
    generic_file_write,		/* write */
#else
    cryptfs_write,		/* write */
#endif
    cryptfs_readdir,		/* readdir */
    cryptfs_poll,		/* poll */
    cryptfs_ioctl,		/* ioctl */
    cryptfs_mmap,		/* mmap */
    cryptfs_open,		/* open */
    cryptfs_flush,		/* flush */
    cryptfs_release,		/* release */
    cryptfs_fsync,		/* fsync */
    cryptfs_fasync,		/* fasync */
    cryptfs_check_media_change,	/* check_media_change */
#ifdef NOT_NEEDED
    cryptfs_file_revalidate,	/* revalidate */
#else /* not NOT_NEEDED */
    NULL,			/* revalidate */
#endif /* not NOT_NEEDED */
    cryptfs_lock			/* lock */
};

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
