/*
 *  $Id: super.c,v 1.1 1998/12/02 23:37:09 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <fist.h>
#include <cryptfs.h>


STATIC void
cryptfs_read_inode(inode_t *inode)
{
    inode_t *hidden_inode;

    print_entry_location();
    fist_checkinode(inode, "cryptfs_read_inode IN");

    /*
     * XXX: calling iget from code which iget called us may result
     * in a deadlock on an SMP kernel!!!
     */
    hidden_inode = iget(stohs(inode->i_sb), inode->i_ino);
    if (!hidden_inode)
	printk(KERN_ERR "cryptfs: cryptfs_read_inode couldn't find hidden_inode\n");

    fist_copy_attr_all(inode, hidden_inode);
    inode->i_version = ++event;	/* increment inode version */
    inode->i_op = &cryptfs_iops;
    itopd(inode) = kmalloc(sizeof(struct cryptfs_inode_info), GFP_KERNEL);
    itohi(inode) = hidden_inode;

    fist_checkinode(inode, "cryptfs_read_inode OUT");
    print_exit_location();
}


#ifdef NOT_NEEDED
/*
 * Not needed.  This function will be called by sync_one(), which is
 * called by sync_list(), which is called by sync_inodes(kdev_t dev),
 * in order to sync dirty inodes of a vfs to disk.  But the higher
 * vfs code will call sync_inodes() on the lower mounted device anyway
 * so we should not have call write_inode() on the hidden_inode.
 */
STATIC void
cryptfs_write_inode(inode_t *inode)
{
    return;
}
#endif /*  NOT_NEEDED */


#ifdef NOT_NEEDED
/*
 * We don't think we need it b/c iput() will call this first, resulting
 * in decrementing our refcount, but our code will free the lower inode
 * b/c it calls iput() on it.
 * That's b/c the lower and upper f/s don't have the same refcount
 * on their inodes.
 */
STATIC void
cryptfs_put_inode(inode_t *inode)
{
    inode_t *hidden_inode = itohi(inode);

    print_entry_location();
    iput(hidden_inode);
    print_exit_location();
    return;
}
#endif /* NOT_NEEDED */


#ifdef NOT_NEEDED
/*
 * no need to do anything (maybe sync lower inode)
 * because out put_inode calls the lower put_inode, which will
 * delete the lower inode if needed.  Plus, we don't have all
 * the information needed to determine if the lower inode should
 * be deleted.  This is a decision best left to the lower f/s via
 * its iput.
 */
STATIC void
cryptfs_delete_inode(inode_t *inode)
{
    return;
}
#endif /* NOT_NEEDED */


// STATIC int cryptfs_notify_change(struct inode *inode, struct iattr *ia)
STATIC int
cryptfs_notify_change(dentry_t *dentry, struct iattr *ia)
{
    int err = 0;
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(dentry);
    inode_t *inode = dentry->d_inode;
    inode_t *hidden_inode;

    print_entry_location();
    ASSERT(hidden_dentry != NULL);
    ASSERT(inode != NULL);

    hidden_inode = itohi(inode);
    ASSERT(hidden_inode != NULL);

    fist_checkinode(inode, "cryptfs_notify_change");
    if (!hidden_inode->i_sb ||
	!hidden_inode->i_sb->s_op ||
	!hidden_inode->i_sb->s_op->notify_change) {
	err = inode_change_ok(hidden_inode, ia);
	if (!err)
	    inode_setattr(hidden_inode, ia);
	goto out_update;
    }

    err = hidden_inode->i_sb->s_op->notify_change(hidden_dentry, ia);
    if (err)
	goto out;
    fist_checkinode(inode, "post cryptfs_notify_change");

 out_update:
    err = inode_change_ok(inode, ia);
    if (!err)
	inode_setattr(inode, ia);
#if 0
    /*
     * The lower file system might have changed the attributes further,
     * so we copy the hidden_inode's attributes (and a few more) to
     * our inode.
     */
    fist_copy_attr_all(inode, hidden_inode);
#endif


out:
    print_exit_status(err);
    return err;
}


STATIC void
cryptfs_put_super(super_block_t *sb)
{
    print_entry_location();

    //    dput(stopd(sb)->s_root);

    if (stopd(sb)) {
	kfree_s(stopd(sb), sizeof(struct cryptfs_sb_info));
	stopd(sb) = NULL;
    }
    fist_dprint(6, "cryptfs: released super\n");
    fist_mod_dec_use_count();

    print_exit_location();
}


#ifdef NOT_NEEDED
/*
 * This is called in do_umount before put_super.
 * The superblock lock is not held yet.
 * We probably do not need to define this or call write_super
 * on the hidden_sb, because sync_supers() will get to hidden_sb
 * sooner or later.  But it is also called from file_fsync()...
 */
STATIC void
cryptfs_write_super(super_block_t *sb)
{
    return;
}
#endif /* NOT_NEEDED */


STATIC int
cryptfs_statfs(super_block_t *sb, struct statfs *buf, int bufsiz)
{
    int err = 0;
    super_block_t *hidden_sb = stohs(sb);

    print_entry_location();

    if (hidden_sb->s_op && hidden_sb->s_op->statfs)
	err = hidden_sb->s_op->statfs(hidden_sb, buf, bufsiz);

    print_exit_status(err);
    return err;
}


/*
 * XXX: not implemented.  This is not allowed yet.
 * Should we call this on the hidden_sb?  Probably not.
 */
STATIC int
cryptfs_remount_fs(super_block_t *sb, int *flags, char *data)
{
    return -ENOSYS;
}


/*
 * Called by iput() when the inode reference count reached zero
 * and the inode is not hashed anywhere.  Used to clear anything
 * that needs to be, before the inode is completely destroyed and put
 * on the inode free list.
 */
STATIC void
cryptfs_clear_inode(inode_t *inode)
{
    print_entry_location();

    fist_checkinode(inode, "cryptfs_clear_inode IN");
    /*
     * Decrement a reference to a hidden_inode, which was incremented
     * by our read_inode when it was created initially.
     */
    iput(itohi(inode));
    kfree_s(itopd(inode), sizeof(struct cryptfs_inode_info));
    itopd(inode) = NULL;
    fist_checkinode(inode, "cryptfs_clear_inode OUT");

    print_exit_location();
}


/*
 * Called in do_umount() if the MNT_FORCE flag was used and this
 * function is defined.  See comment in linux/fs/super.c:do_umount().
 * Used only in nfs, to kill any pending RPC tasks, so that subsequent
 * code can actually succeed and won't leave tasks that need handling.
 *
 * PS. I wonder if this is somehow useful to undo damage that was
 * left in the kernel after a user level file server (such as amd)
 * dies.
 */
STATIC void
cryptfs_umount_begin(super_block_t *sb)
{
    super_block_t *hidden_sb = stohs(sb);

    print_entry_location();

    if (hidden_sb->s_op->umount_begin)
	hidden_sb->s_op->umount_begin(hidden_sb);

    print_exit_location();
}


struct super_operations cryptfs_sops =
{
    cryptfs_read_inode,		/* read inode */

#ifdef NOT_NEEDED
    cryptfs_write_inode,		/* write inode */
    cryptfs_put_inode,		/* put inode */
    cryptfs_delete_inode,	/* delete inode */
#else /* not NOT_NEEDED */
    NULL,			/* write inode */
    NULL,			/* put inode */
    NULL,			/* delete inode */
#endif /* not NOT_NEEDED */

    cryptfs_notify_change,	/* notify change */
    cryptfs_put_super,		/* put superblock */

#ifdef NOT_NEEDED
    cryptfs_write_super,		/* write superblock */
#else /* not NOT_NEEDED */
    NULL,			/* write superblock */
#endif /* not NOT_NEEDED */

    cryptfs_statfs,		/* stat filesystem */

    cryptfs_remount_fs,		/* remount filesystem */
    cryptfs_clear_inode,		/* clear inode */
    cryptfs_umount_begin		/* umount begin */
};

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
