// pkcspad.cpp - written and placed in the public domain by Wei Dai

#include "misc.h"
#include "asn.h"

#include "pkcspad.h"

void PublicKeyUsingPKCSPadding::Encrypt(RandomNumberGenerator &rng, const byte *input, unsigned int inputLen, byte *output)
{
    assert (inputLen <= MaxPlainTextLength());

    SecByteBlock pkcsBlock(PKCSBlockLen());

    pkcsBlock[0] = 0;
    pkcsBlock[1] = 2;  // block type 2

    // pad with non-zero random bytes
    for (unsigned i = 2; i < PKCSBlockLen()-inputLen-1; i++)
        while ((pkcsBlock[i] = rng.GetByte()) == 0);

    pkcsBlock[PKCSBlockLen()-inputLen-1] = 0;     // separator
    memcpy(pkcsBlock+PKCSBlockLen()-inputLen, input, inputLen);

    Integer c;
    RawEncrypt(Integer(pkcsBlock, PKCSBlockLen()), c);
    c.Encode(output, PKCSBlockLen());
}

unsigned int PublicKeyUsingPKCSPadding::PublicDecrypt(const byte *input, byte *output)
{
    Integer m;
    RawEncrypt(Integer(input, PKCSBlockLen()), m);

    SecByteBlock pkcsBlock(PKCSBlockLen());
    m.Encode(pkcsBlock, PKCSBlockLen());

    // Require block type 1.
    if ((pkcsBlock[0] != 0) || (pkcsBlock[1] != 1))
        return 0;

    // skip past the padding until we find the seperator
    unsigned i=2;
    while (i<PKCSBlockLen() && pkcsBlock[i++])
        if (pkcsBlock[i-1] != 0xff)     // not valid padding
            return 0;
    assert(i==PKCSBlockLen() || pkcsBlock[i-1]==0);

    unsigned int outputLen = PKCSBlockLen() - i;
    if (outputLen > MaxPlainTextLength())
        return 0;

    memcpy (output, pkcsBlock+i, outputLen);
    return outputLen;
}

boolean PublicKeyUsingPKCSPadding::Verify(const byte *message, unsigned int messageLen, const byte *signature)
{
    assert(messageLen <= MaxMessageLength());
    SecByteBlock m(MaxMessageLength());
    unsigned int mLen = PublicDecrypt(signature, m);
    return (mLen==messageLen && memcmp(m, message, mLen)==0);
}

void PrivateKeyUsingPKCSPadding::PrivateEncrypt(const byte *input, unsigned int inputLen, byte *output)
{
    assert (inputLen <= MaxPlainTextLength());

    SecByteBlock pkcsBlock(PKCSBlockLen());

    pkcsBlock[0] = 0;
    pkcsBlock[1] = 1;   // block type 1
    memset(pkcsBlock+2, 0xff, PKCSBlockLen()-inputLen-3);   // padding
    pkcsBlock[PKCSBlockLen()-inputLen-1] = 0;               // separator
    memcpy(pkcsBlock+PKCSBlockLen()-inputLen, input, inputLen);

    Integer c;
    RawDecrypt(Integer(pkcsBlock, PKCSBlockLen()), c);
    c.Encode(output, PKCSBlockLen());
}

unsigned int PrivateKeyUsingPKCSPadding::Decrypt(const byte *input, byte *output)
{
    Integer m;
    RawDecrypt(Integer(input, PKCSBlockLen()), m);

    SecByteBlock pkcsBlock(PKCSBlockLen());
    m.Encode(pkcsBlock, PKCSBlockLen());

    // Require block type 2.
    if ((pkcsBlock[0] != 0) || (pkcsBlock[1] != 2))
        return 0;

    // skip past the padding until we find the seperator
    unsigned i=2;
    while (i<PKCSBlockLen() && pkcsBlock[i++]);
    assert(i==PKCSBlockLen() || pkcsBlock[i-1]==0);

    unsigned int outputLen = PKCSBlockLen() - i;
    if (outputLen > MaxPlainTextLength())
        return 0;

    memcpy (output, pkcsBlock+i, outputLen);
    return outputLen;
}
