unit Diamond;

{  diamond.pas - Encryption designed to exceed DES in security.
   This file and the Diamond2 and Diamond2 Lite Block Ciphers
   described herein are hereby dedicated to the Public Domain by the
   author and inventor, Michael Paul Johnson.  Feel free to use these
   for any purpose that is legally and morally right.  The names
   "Diamond2 Block Cipher" and "Diamond2 Lite Block Cipher" should only
   be used to describe the algorithms described in this file, to avoid
   confusion.

   Disclaimers:  the following comes with no warranty, expressed or
   implied.  You, the user, must determine the suitability of this
   information to your own uses.  You must also find out what legal
   requirements exist with respect to this data and programs using
   it, and comply with whatever valid requirements exist.

   For simplicity and speed, this implementation is fixed at 10 rounds
   and 16-byte blocks (Diamond2 instead of Diamond Lite).
 }

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, crc_32;

const
    MaxRound = 9; // Number of rounds minus one.

type
  TCipherBlock = array[0..15] of Byte;
  TSBox = array[0..MaxRound,0..15,0..255] of byte; // [round, byte of block, byte]

  TDiamond = class(TComponent)
  private
    UserKey: string;
    s,                     {Substitution boxes.}
    si: TSBox;             {Inverse substitution boxes.}
    tcbPrevCipher,         // Previous ciphertext block for CFB mode
    tcbPlain: TCipherBlock;// Current plain text block for CFB mode
    iECBPos: integer;      // Current position in tcbPlain block for CFB

    procedure permute(const x: TCipherBlock; var y: TCipherBlock);
    procedure ipermute(const x: TCipherBlock; var y: TCipherBlock);
    procedure substitute(const iRound: integer; const x: TCipherBlock;
                         var y: TCipherBlock);
    procedure isubst(const iRound: integer; const x: TCipherBlock;
                     var y: TCipherBlock);
  protected
    { Protected declarations }
  public
    { Public declarations }
    procedure SetKey(const user_key: string); { Initialize state with key. }
    procedure Burn; {Destroy key information.}
    function EncryptBlock(const PlainText: TCipherBlock): TCipherBlock;
    function DecryptBlock(const CipherText: TCipherBlock): TCipherBlock;
    procedure CFBStart(const IV: TCipherBlock); // Initialize CFB mode vars
    procedure CFBEncrypt(const iSize: integer; var Buffer: array of byte);
    procedure CFBDecrypt(const iSize: integer; var Buffer: array of byte);
    // EncryptString adds a 16-byte IV and 1-16 bytes padding, then encrypts in
    // standard CBC mode. DecryptString is the inverse function, provided that
    // Key is the same as was used to encrypt.
    function EncryptString(PlainText: string): string;
    function DecryptString(const CipherText: string): string;
    // Base64Encode is not encryption, but a way to represent a binary string
    // as a printable ASCII string.
    function Base64Encode(const s: string): string;
    function Base64Decode(const s: string): string;


  published
    property Key: string read UserKey write SetKey;
  end;

procedure Register;

implementation

const
    Base64: string = '23456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz~#%&*+-';
    UnBase64: array[0..255] of byte =
       (128,128,128,128,128,128,128,128, 128,128,128,128,128,128,128,128, //0-15
        128,128,128,128,128,128,128,128, 128,128,128,128,128,128,128,128,  //16-31
        128,128,128, 58,128, 59, 60,128, 128,128, 61, 62,128, 63,128,128,  //32-47
        128,128,  0,  1,  2,  3,  4,  5,   6,  7,128,128,128,128,128,128,  //48-63
        128,  8,  9, 10, 11, 12, 13, 14,  15,128, 16, 17, 18, 19, 20,128,  //64-79
         21, 22, 23, 24, 25, 26, 27, 28,  29, 30, 31,128,128,128,128,128,  //80-95
        128, 32, 33, 34, 35, 36, 37, 38,  39, 40, 41, 42,128, 43, 44, 45,  //96-111
         46, 47, 48, 49, 50, 51, 52, 53,  54, 55, 56,128,128,128, 57,128,  //112-127        128,128,128,128,128,128,128,128, 128,128,128,128,128,128,128,128,  //128-143
        128,128,128,128,128,128,128,128, 128,128,128,128,128,128,128,128,  //128-143
        128,128,128,128,128,128,128,128, 128,128,128,128,128,128,128,128,  //144-159
        128,128,128,128,128,128,128,128, 128,128,128,128,128,128,128,128,  //160-175
        128,128,128,128,128,128,128,128, 128,128,128,128,128,128,128,128,  //176-191
        128,128,128,128,128,128,128,128, 128,128,128,128,128,128,128,128,  //192-207
        128,128,128,128,128,128,128,128, 128,128,128,128,128,128,128,128,  //208-223
        128,128,128,128,128,128,128,128, 128,128,128,128,128,128,128,128,  //224-239
        128,128,128,128,128,128,128,128, 128,128,128,128,128,128,128,128);  //240-255


procedure Register;
begin
  RegisterComponents('MPJ', [TDiamond]);
end;

procedure TDiamond.SetKey(const user_key: string);
{  This procedure generates internal keys by filling the substitution box array
  s based on the external key given as input.  It DOES take a bit of time.  }
var iRound, iPrevRound, iBlockByte, iPrevBlockByte, k, KeyIndex: integer;
    accum: longint;

    function keyrand(const max_value: byte): byte; {Returns uniformly distributed pseudorandom}
    var i, mask: integer;
        begin                           {  value based on UserKey
        }
        result := 0;
        if max_value > 0 then
            begin
            mask := 0;      // mask is all 1's for the minimum number of bits
            i := max_value; // needed to represent max_value.
            while i > 0 do
                begin
                mask := (mask shl 1) or 1;
                i := i shr 1;
                end;
            i := 0;
            repeat
                begin
	        if (iPrevRound >= 0) then
        	    accum := crc32(accum, s[iPrevRound,iPrevBlockByte,ord(UserKey[keyindex])])
	        else
        	    accum := crc32(accum, ord(UserKey[keyindex]));
                inc(keyindex);
                if (keyindex > length(UserKey)) then
                    begin
                    keyindex := 1;   {  Recycle thru the key  }
                    accum := crc32(accum, (length(UserKey) and $FF)); // Key 'a' <> Key 'aa'
                    accum := crc32(accum, ((length(UserKey) shr 8) and $FF));
                    end;
                result := accum and mask;
                inc(i);
                if ((i>97) and (result > max_value)) then {Don't loop forever.}
                    result := result - max_value;    {Introduce negligible bias.}
                end
            until (result <= max_value); {  Discard out of range values.  }
            end
        end;

    procedure MakeOneBox;
    var n, m, p, pos: integer;
        filled: array[0..255] of boolean;
    begin {sbox is either NULL or a pointer to the previously filled array.  }
    for m := 0 to 255 do        {  The filled array is used to make sure that  }
        filled[m] := false;     {  each byte of the array is filled only once. }
    for n := 255 downto 0 do    {  n counts the number of bytes left to fill  }
        begin
        pos := keyrand(n);      {  pos is the position among the UNFILLED  }
                                {  components of the s array that the  }
                                {  number n should be placed.   }
        p := 0;
        while filled[p] do inc(p);
        for m := 0 to pos-1 do
            begin
            inc(p);
            while filled[p] do inc(p);
            end;
        s[iRound,iBlockByte,p] := n;
        filled[p] := true;
        end
    end;

  begin // SetKey
    UserKey := user_key;
    if length(UserKey) = 0 then
        UserKey := #0; // Prevent crash in case of empty key.
    keyindex := 1;
    accum := $FFFFFFFF;
    iPrevRound := -1;
    iPrevBlockByte := 0;
    for iRound := 0 to MaxRound do
        begin
        for iBlockByte := 0 to 15 do
            begin
            makeonebox;
            iPrevRound := iRound;
            iPrevBlockByte := iBlockByte;
            end
        end;
    {  Fill the inverse substitution box array si.  It is not
       necessary to do this unless the decryption mode is used.   }
    for iRound := 0 to MaxRound do
        begin
        for iBlockByte := 0 to 15 do
            begin
            for k := 0 to 255 do
                begin
                si[iRound,iBlockByte,s[iRound,iBlockByte,k]] := k;
                end
            end
        end;
    FillChar(tcbPlain, SizeOf(tcbPlain), 0);
    CFBStart(tcbPlain);
  end; // SetKey

procedure TDiamond.Burn;
var k: integer;
begin
    for k := 1 to length(UserKey) do
        UserKey[k] := #0;
    FillChar(s, SizeOf(s), 0);
    FillChar(si, SizeOf(si), 0);
    FillChar(s, SizeOf(s), 0);
    FillChar(tcbPrevCipher, SizeOf(tcbPrevCipher), 0);
    FillChar(tcbPlain, SizeOf(tcbPlain), 0);
    iECBPos := 0;
    UserKey := '';
end;

procedure TDiamond.permute(const x: TCipherBlock; var y: TCipherBlock);
{ x and y must be different.
  This procedure is designed to make each bit of the output dependent on as
  many bytes of the input as possible, especially after repeated application.
  Each output byte takes its least significant bit from the corresponding
  input byte.  The next higher bit comes from the corresponding bit of the
  next higher input byte.  This is done until all bits of the output byte
  are filled. }
    begin
    y[0] := (x[0] and 1) or (x[1] and 2) or (x[2] and 4) or
            (x[3] and 8) or (x[4] and 16) or (x[5] and 32) or
            (x[6] and 64) or (x[7] and 128);
    y[1] := (x[1] and 1) or (x[2] and 2) or (x[3] and 4) or
            (x[4] and 8) or (x[5] and 16) or (x[6] and 32) or
            (x[7] and 64) or (x[8] and 128);
    y[2] := (x[2] and 1) or (x[3] and 2) or (x[4] and 4) or
            (x[5] and 8) or (x[6] and 16) or (x[7] and 32) or
            (x[8] and 64) or (x[9] and 128);
    y[3] := (x[3] and 1) or (x[4] and 2) or (x[5] and 4) or
            (x[6] and 8) or (x[7] and 16) or (x[8] and 32) or
            (x[9] and 64) or (x[10] and 128);
    y[4] := (x[4] and 1) or (x[5] and 2) or (x[6] and 4) or
            (x[7] and 8) or (x[8] and 16) or (x[9] and 32) or
            (x[10] and 64) or (x[11] and 128);
    y[5] := (x[5] and 1) or (x[6] and 2) or (x[7] and 4) or
            (x[8] and 8) or (x[9] and 16) or (x[10] and 32) or
            (x[11] and 64) or (x[12] and 128);
    y[6] := (x[6] and 1) or (x[7] and 2) or (x[8] and 4) or
            (x[9] and 8) or (x[10] and 16) or (x[11] and 32) or
            (x[12] and 64) or (x[13] and 128);
    y[7] := (x[7] and 1) or (x[8] and 2) or (x[9] and 4) or
            (x[10] and 8) or (x[11] and 16) or (x[12] and 32) or
            (x[13] and 64) or (x[14] and 128);
    y[8] := (x[8] and 1) or (x[9] and 2) or (x[10] and 4) or
            (x[11] and 8) or (x[12] and 16) or (x[13] and 32) or
            (x[14] and 64) or (x[15] and 128);
    y[9] := (x[9] and 1) or (x[10] and 2) or (x[11] and 4) or
            (x[12] and 8) or (x[13] and 16) or (x[14] and 32) or
            (x[15] and 64) or (x[0] and 128);
    y[10] := (x[10] and 1) or (x[11] and 2) or (x[12] and 4) or
            (x[13] and 8) or (x[14] and 16) or (x[15] and 32) or
            (x[0] and 64) or (x[1] and 128);
    y[11] := (x[11] and 1) or (x[12] and 2) or (x[13] and 4) or
            (x[14] and 8) or (x[15] and 16) or (x[0] and 32) or
            (x[1] and 64) or (x[2] and 128);
    y[12] := (x[12] and 1) or (x[13] and 2) or (x[14] and 4) or
            (x[15] and 8) or (x[0] and 16) or (x[1] and 32) or
            (x[2] and 64) or (x[3] and 128);
    y[13] := (x[13] and 1) or (x[14] and 2) or (x[15] and 4) or
            (x[0] and 8) or (x[1] and 16) or (x[2] and 32) or
            (x[3] and 64) or (x[4] and 128);
    y[14] := (x[14] and 1) or (x[15] and 2) or (x[0] and 4) or
            (x[1] and 8) or (x[2] and 16) or (x[3] and 32) or
            (x[4] and 64) or (x[5] and 128);
    y[15] := (x[15] and 1) or (x[0] and 2) or (x[1] and 4) or
            (x[2] and 8) or (x[3] and 16) or (x[4] and 32) or
            (x[5] and 64) or (x[6] and 128);
    end;

procedure TDiamond.ipermute(const x: TCipherBlock; var y: TCipherBlock); {x<>y}
{  This is the inverse of the procedure permute.  }
    begin
    y[0] := (x[0] and 1) or (x[15] and 2) or (x[14] and 4) or
            (x[13] and 8) or (x[12] and 16) or (x[11] and 32) or
            (x[10] and 64) or (x[9] and 128);
    y[1] := (x[1] and 1) or (x[0] and 2) or (x[15] and 4) or
            (x[14] and 8) or (x[13] and 16) or (x[12] and 32) or
            (x[11] and 64) or (x[10] and 128);
    y[2] := (x[2] and 1) or (x[1] and 2) or (x[0] and 4) or
            (x[15] and 8) or (x[14] and 16) or (x[13] and 32) or
            (x[12] and 64) or (x[11] and 128);
    y[3] := (x[3] and 1) or (x[2] and 2) or (x[1] and 4) or
            (x[0] and 8) or (x[15] and 16) or (x[14] and 32) or
            (x[13] and 64) or (x[12] and 128);
    y[4] := (x[4] and 1) or (x[3] and 2) or (x[2] and 4) or
            (x[1] and 8) or (x[0] and 16) or (x[15] and 32) or
            (x[14] and 64) or (x[13] and 128);
    y[5] := (x[5] and 1) or (x[4] and 2) or (x[3] and 4) or
            (x[2] and 8) or (x[1] and 16) or (x[0] and 32) or
            (x[15] and 64) or (x[14] and 128);
    y[6] := (x[6] and 1) or (x[5] and 2) or (x[4] and 4) or
            (x[3] and 8) or (x[2] and 16) or (x[1] and 32) or
            (x[0] and 64) or (x[15] and 128);
    y[7] := (x[7] and 1) or (x[6] and 2) or (x[5] and 4) or
            (x[4] and 8) or (x[3] and 16) or (x[2] and 32) or
            (x[1] and 64) or (x[0] and 128);
    y[8] := (x[8] and 1) or (x[7] and 2) or (x[6] and 4) or
            (x[5] and 8) or (x[4] and 16) or (x[3] and 32) or
            (x[2] and 64) or (x[1] and 128);
    y[9] := (x[9] and 1) or (x[8] and 2) or (x[7] and 4) or
            (x[6] and 8) or (x[5] and 16) or (x[4] and 32) or
            (x[3] and 64) or (x[2] and 128);
    y[10] := (x[10] and 1) or (x[9] and 2) or (x[8] and 4) or
            (x[7] and 8) or (x[6] and 16) or (x[5] and 32) or
            (x[4] and 64) or (x[3] and 128);
    y[11] := (x[11] and 1) or (x[10] and 2) or (x[9] and 4) or
            (x[8] and 8) or (x[7] and 16) or (x[6] and 32) or
            (x[5] and 64) or (x[4] and 128);
    y[12] := (x[12] and 1) or (x[11] and 2) or (x[10] and 4) or
            (x[9] and 8) or (x[8] and 16) or (x[7] and 32) or
            (x[6] and 64) or (x[5] and 128);
    y[13] := (x[13] and 1) or (x[12] and 2) or (x[11] and 4) or
            (x[10] and 8) or (x[9] and 16) or (x[8] and 32) or
            (x[7] and 64) or (x[6] and 128);
    y[14] := (x[14] and 1) or (x[13] and 2) or (x[12] and 4) or
            (x[11] and 8) or (x[10] and 16) or (x[9] and 32) or
            (x[8] and 64) or (x[7] and 128);
    y[15] := (x[15] and 1) or (x[14] and 2) or (x[13] and 4) or
            (x[12] and 8) or (x[11] and 16) or (x[10] and 32) or
            (x[9] and 64) or (x[8] and 128);
    end;

procedure TDiamond.substitute(const iRound: integer; const x: TCipherBlock;
                              var y: TCipherBlock);
    var iBlockByte: integer;
    begin
    for iBlockByte := 0 to 15 do
        y[iBlockByte] := s[iRound, iBlockByte, x[iBlockByte]];
    end;

procedure TDiamond.isubst(const iRound: integer; const x: TCipherBlock;
                          var y: TCipherBlock);
    var iBlockByte: integer;
    begin
    for iBlockByte := 0 to 15 do
        y[iBlockByte] := si[iRound, iBlockByte, x[iBlockByte]];
    end;

function TDiamond.EncryptBlock(const PlainText: TCipherBlock): TCipherBlock;
{  Encrypt a block of 16 bytes.  }
var iRound: integer;
    z: TCipherBlock;
    begin
    substitute(0, PlainText, result);
    for iRound := 1 to MaxRound do
        begin
        permute(result, z);
        substitute(iRound, z, result);
        end
    end;

function TDiamond.DecryptBlock(const CipherText: TCipherBlock): TCipherBlock;
{  Decrypt a block of 16 bytes.  }
var iRound: integer;
    z: TCipherBlock;
    begin
    isubst(MaxRound, CipherText, result);
    for iRound := (MaxRound - 1) downto 0 do
        begin
        ipermute(result, z);
        isubst(iRound, z, result);
        end
    end;

procedure TDiamond.CFBStart(const IV: TCipherBlock); // Initialize CFB mode vars
    begin
    tcbPrevCipher := EncryptBlock(IV);
    iECBPos := 0;
    end;

procedure TDiamond.CFBEncrypt(const iSize: integer; var Buffer: array of byte);
var i, limit: integer;
    begin
    limit := iSize-1;
    if limit > high(Buffer) then
        limit := high(Buffer);
    for i := 0 to limit do
        begin
        tcbPlain[iECBPos] := Buffer[i];
        Buffer[i] := Buffer[i] xor tcbPrevCipher[iECBPos];
        inc(iECBPos);
        if iECBPos > 15 then
            begin
            tcbPrevCipher := EncryptBlock(tcbPlain);
            iECBPos := 0;
            end;
        end;
    end;

procedure TDiamond.CFBDecrypt(const iSize: integer; var Buffer: array of byte);
var i, limit: integer;
    begin
    limit := iSize-1;
    if limit > high(Buffer) then
        limit := high(Buffer);
    for i := 0 to limit do
        begin
        Buffer[i] := Buffer[i] xor tcbPrevCipher[iECBPos];
        tcbPlain[iECBPos] := Buffer[i];
        inc(iECBPos);
        if iECBPos > 15 then
            begin
            tcbPrevCipher := EncryptBlock(tcbPlain);
            iECBPos := 0;
            end;
        end;
    end;

function TDiamond.EncryptString(PlainText: string): string;
var IV, pt: TCipherBlock;
    i, j, crc, pos: longint;
    begin

    // Pad the input string to a multiple of 16 bytes.

    j := Length(PlainText) + 1;
    i := 16 - (Length(PlainText) and $F);
    SetLength(PlainText, Length(PlainText)+i);
    SetLength(Result, Length(PlainText)+16);
    while j <= Length(PlainText) do
        begin
        PlainText[j] := chr(i);
        inc(j);
        end;


    // Generate an IV from the string and time.

    FillChar(IV, SizeOf(IV), 0);
    j := 0;
    crc := $FFFFFFFF;
    for i := 1 to length(PlainText) do
        begin
        crc := crc32(crc, ord(PlainText[i]));
        j := j+ord(PlainText[i]);
        end;
    IV[0] := crc and $FF;
    IV[1] := (crc shr 8) and $FF;
    IV[2] := (crc shr 16) and $FF;
    IV[3] := (crc shr 24) and $FF;
    IV[4] := j and $FF;
    IV[5] := (j shr 8) and $FF;
    IV[6] := (j shr 16) and $FF;
    IV[7] := (j shr 24) and $FF;
    i := GetTickCount;
    IV[8] := i and $FF;
    IV[9] := (i shr 8) and $FF;
    IV[10] := (i shr 16) and $FF;
    IV[11] := (i shr 24) and $FF;
    IV := EncryptBlock(IV);

    // Store the IV in the string.

    j := 0;
    for i := 1 to 16 do
        begin
        Result[i] := chr(IV[j]);
        inc(j);
        end;

    // Encrypt the string in CBC mode.

    IV := EncryptBlock(IV);
    j := 17;
    i := 1;
    while i < length(PlainText) do
        begin
        for pos := 0 to 15 do
            begin
            pt[pos] := ord(PlainText[i]) xor IV[pos];
            inc(i);
            end;
        IV := EncryptBlock(pt);
        for pos := 0 to 15 do
            begin
            Result[j] := chr(IV[pos]);
            inc(j);
            end;
        end;
    end;

function TDiamond.DecryptString(const CipherText: string): string;
var IV, pt, ct: TCipherBlock;
    i, j, crc, pos: longint;
    s: string;
    begin

    Result := '';
    if length(CipherText) >= 32 then
        begin

        // Read in the IV.

        j := 0;
        for i := 1 to 16 do
            begin
            IV[j] := ord(CipherText[i]);
            inc(j);
            end;

        SetLength(s, Length(CipherText)-16);

        // Decrypt the string in CFB mode.

        IV := EncryptBlock(IV);
        j := 17;
        i := 1;
        while i < Length(s) do
            begin
            for pos := 0 to 15 do
                begin
                ct[pos] := ord(CipherText[j]);
                inc(j);
                end;
            pt := DecryptBlock(ct);
            for pos := 0 to 15 do
                begin
                s[i] := chr(pt[pos] xor IV[pos]);
                inc(i);
                end;
            IV := ct;
            end;

            // Unpad Plain Text string

            i := ord(s[Length(s)]); // Last byte is number of pad bytes
            if (i > 0) and (i <= 16) then
                Result := System.Copy(s, 1,Length(s) - i);
        end;

    end;

function TDiamond.Base64Encode(const s: string): string;
var s4: string;
    i, j, k: integer;
    b: byte;
    begin
    Result := '';
    SetLength(s4, 4);
    b := 0;
    i := 1;
    j := 2;
    k := 2;
    while i <= length(s) do
        begin
        b := b or ((ord(s[i]) and $C0) shr k);
        inc(k,2);
        s4[j] := Base64[(ord(s[i]) and $3F)+1];
        inc(i);
        inc(j);
        if j > 4 then
            begin
            s4[1] := Base64[b+1];
            b := 0;
            j := 2;
            k := 2;
            Result := Result + s4;
            end;
        end;
    if j <> 2 then
        begin // Flush data in s4.
        s4[j] := '.';
        s4[1] := Base64[b+1];
        Result := Result + s4;
        SetLength(Result, Length(Result) - (4 - j));
        end
    else
        Result := Result + '.';
    end;

function TDiamond.Base64Decode(const s: string): string;
var i, j, k: integer;
    b: byte;
    begin
    Result := '';
    b := 0;
    i := 1;
    j := 0;
    while (i <= length(s)) and (s[i] <> '.') do
        begin
        if j = 0 then
            begin
            b := UnBase64[ord(s[i])];
            k := 2;
            end
        else
            begin
            Result := Result + chr(UnBase64[ord(s[i])] or ((b shl k) and $C0));
            inc(k,2);
            end;
        inc(j);
        j := j and 3;
        inc(i);
        end;
    end;


initialization

end.

